using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using Ato.EN.IntegrationServices.Document.DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationCUREL
{

    public partial class CURELAddRemove2018ValidatorRemove
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="CURELAddRemove2018ValidatorRemove" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CURELAddRemove2018ValidatorRemove(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }
				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public CURELAddRemove2018Remove ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CURELAddRemove2018Remove report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------


            #region Repeating report.RP_RelationshipDetailCollection
            if (report.RP_RelationshipDetailCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (CURELAddRemove2018Remove.RP_RelationshipDetail relationshipDetail in report.RP_RelationshipDetailCollection)
                {
                    itemIndex2++;
            
                    #region VR.ATO.CUREL.000002
            
                    /*  VR.ATO.CUREL.000002
                    In the Remove service, the Relationship Start Date must be supplied where a Relationship Detail instance is provided.
    
                    Legacy Rule Format:
                    IF ((service.Interaction = curelremove) AND
                       (WHERE IN TUPLE(RelationshipDetail)  
                       ([CUREL103] = NULL))) 
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ((^CUREL302 <> NULL) AND (^CUREL103 = NULL))
            
                    Data Elements:
            
                    CUREL:^CUREL103 = :Party.AgentRelationshipStart.Date
            
                    CUREL:^CUREL302 = :RelationshipDetail
                    */
                    assertion = (report.RP_RelationshipDetailCollectionExists != false && relationshipDetail.CUREL103 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CUREL.000002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Relationship Start Date not provided.",
                            LongDescription = @"In the Remove service, the Relationship Start Date must be provided in each Relationship Detail instance supplied.",
                            Location = "/tns:CURELAddRemove/tns:RP/tns:RelationshipDetailCollection/tns:RelationshipDetail" + OccurrenceIndex(relationshipDetail.OccurrenceIndex) + "/tns:PartyAgentRelationshipStartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000002"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CUREL103", Value = GetValueOrEmpty(relationshipDetail.CUREL103) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CUREL.000002
            
                    #region VR.ATO.CUREL.000025
            
                    /*  VR.ATO.CUREL.000025
                    If Role type code is supplied then it must be 'PAYGW'.
    
                    Legacy Rule Format:
                    IF ((CUREL105 <> NULL) AND (CUREL105 <> 'PAYGW'))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ((^CUREL105 <> NULL) AND (^CUREL105 <> 'PAYGW'))
            
                    Data Elements:
            
                    CUREL:^CUREL105 = :Tax.RoleType.Code
                    */
                    assertion = (relationshipDetail.CUREL105 != null && relationshipDetail.CUREL105 != @"PAYGW");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CUREL.000025", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid Role type provided.",
                            LongDescription = @"When provided, the Role type must have a value of 'PAYGW'.",
                            Location = "/tns:CURELAddRemove/tns:RP/tns:RelationshipDetailCollection/tns:RelationshipDetail" + OccurrenceIndex(relationshipDetail.OccurrenceIndex) + "/tns:TaxRoleTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000025"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CUREL105", Value = relationshipDetail.CUREL105 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CUREL.000025
                    }
                }
        
                #endregion // Foreach loop
    
            #region VR.ATO.CUREL.000005
    
            /*  VR.ATO.CUREL.000005
            In the Remove service, Relationship Update Action must not contain a value of "Add".
    
            Legacy Rule Format:
            IF ((service.Interaction = curelremove) AND
            (ANY OCCURRENCE OF [CUREL102] = "Add"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            AnyOccurrence(^CUREL302, InSet(^CUREL102, '"Add"'))
    
            Data Elements:
    
            CUREL:^CUREL102 = :Interaction.ClientRelationshipUpdate.Code
    
            CUREL:^CUREL302 = :RelationshipDetail
            */
            assertion = (report.RP_RelationshipDetailCollection == null ? false : report.RP_RelationshipDetailCollection.Any(CUREL302Repeat => IsMatch(CUREL302Repeat.CUREL102, @"^(Add)$")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000005", Severity = ProcessMessageSeverity.Error,
                    Description = @"Relationship Update Action contains a value of ""Add"".",
                    LongDescription = @"In the Remove service, the Relationship Update Action value of ""Add"" must not be provided.
The only valid Relationship Update Action value is ""Remove"".",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:RelationshipDetailCollection/tns:RelationshipDetail/tns:InteractionClientRelationshipUpdateC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000005"} },
                };
             
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000005
    
            #region VR.ATO.CUREL.000007
    
            /*  VR.ATO.CUREL.000007
            Both Witholding Payer Number (WPN) and Excise Identification Number (EIN) must not be provided.
    
            Legacy Rule Format:
            IF ((CUREL21 <> NULL) AND (CUREL24 <> NULL))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CUREL21 <> NULL) AND (^CUREL24 <> NULL))
    
            Data Elements:
    
            CUREL:^CUREL24 = :Identifiers.ExciseIdentificationNumber.Identifier
    
            CUREL:^CUREL21 = :Identifiers.WithholdingPayerNumber.Identifier
            */
            assertion = (report.CUREL21 != null && report.CUREL24 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Both WPN and EIN provided.",
                    LongDescription = @"The Client Withholding Payer Number (WPN) and Excise Identification Number (EIN) must not both be provided.",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:ExciseIdentificationNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000007"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL21", Value = report.CUREL21 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL24", Value = report.CUREL24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000007
    
            #region VR.ATO.CUREL.000011
    
            /*  VR.ATO.CUREL.000011
            In the Remove service, Individual Date of Birth and/or Entity Name must not be provided.
    
            Legacy Rule Format:
            IF ((service.Interaction = curelremove) AND
            ([CUREL27] <> NULL) OR ([CUREL51] <> NULLORBLANK)) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CUREL27 <> NULL) OR (^CUREL51 <> BLANK))
    
            Data Elements:
    
            CUREL:^CUREL51 = :Party.EntityName.Text
    
            CUREL:^CUREL27 = :PersonDemographicDetails.Birth.Date
            */
            assertion = (report.CUREL27 != null || string.IsNullOrWhiteSpace(report.CUREL51) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual Date of Birth and/or Entity Name provided.",
                    LongDescription = @"In the Remove service, an Individual Date of Birth or Entity Name must not be provided.",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:PartyEntityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000011"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL27", Value = GetValueOrEmpty(report.CUREL27) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL51", Value = report.CUREL51 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000011
    
            #region VR.ATO.CUREL.000020
    
            /*  VR.ATO.CUREL.000020
            The Australian Business Number (ABN) of the Registered agent or Payroll provider must pass the ABN algorithm check.
    
            Legacy Rule Format:
            IF ((CUREL82 <> NULL) AND (ABNALGORITHM (CUREL82) = FALSE))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CUREL82 <> NULL) AND (FailsABNAlgorithm(^CUREL82))
    
            Data Elements:
    
            CUREL:^CUREL82 = :Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.CUREL82 != null && FailsABNAlgorithm(report.CUREL82));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000020", Severity = ProcessMessageSeverity.Error,
                    Description = @"Registered agent or Payroll provider ABN is invalid.",
                    LongDescription = @"The Australian Business Number (ABN) of the Registered agent or Payroll provider has failed the algorithm check.",
                    Location = "/tns:CURELAddRemove/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000020"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL82", Value = report.CUREL82 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000020
    
            #region VR.ATO.CUREL.000021
    
            /*  VR.ATO.CUREL.000021
            The Registered Agent Number (RAN) of the Registered agent or Payroll provider must pass the Tax Agent Number (TAN) algorithm check.
    
            Legacy Rule Format:
            IF (LENGTH([CUREL76]) = 8 AND TANALGORITHM([CUREL76]) = FALSE) OR (LENGTH([CUREL76]) = 7 AND TANALGORITHM(""0"" & [CUREL76]) = FALSE) 
              RETURN VALIDATION MESSAGE 
            ENDIF

            Technical Business Rule Format:
            (^CUREL76 <> NULL) AND (((Length(^CUREL76) = 7) AND (FailsTANAlgorithm(Concat('0', ^CUREL76)))) OR ((Length(^CUREL76) =8) AND (FailsTANAlgorithm(^CUREL76))))
    
            Data Elements:
    
            CUREL:^CUREL76 = :Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (report.CUREL76 != null && (Length(report.CUREL76) == 7 && FailsTANAlgorithm(string.Concat(@"0", report.CUREL76)) || Length(report.CUREL76) == 8 && FailsTANAlgorithm(report.CUREL76)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Registered agent or Payroll provider RAN is invalid.",
                    LongDescription = @"The Registered Agent Number (RAN) of the Registered agent or Payroll provider has failed the algorithm check.",
                    Location = "/tns:CURELAddRemove/tns:INT/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000021"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL76", Value = report.CUREL76 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000021
    
            #region VR.ATO.CUREL.000022
    
            /*  VR.ATO.CUREL.000022
            The Tax File Number (TFN) of the Client must pass the TFN algorithm check.
    
            Legacy Rule Format:
            IF ((CUREL33 <> NULL) AND (TFNALGORITHM (CUREL33) = FALSE))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CUREL33 <> NULL) AND (FailsTFNAlgorithm(^CUREL33))
    
            Data Elements:
    
            CUREL:^CUREL33 = :Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.CUREL33 != null && FailsTFNAlgorithm(report.CUREL33));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000022", Severity = ProcessMessageSeverity.Error,
                    Description = @"Client TFN is invalid.",
                    LongDescription = @"The Tax File Number (TFN) of the Client has failed the algorithm check.",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000022"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL33", Value = report.CUREL33 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000022
    
            #region VR.ATO.CUREL.000023
    
            /*  VR.ATO.CUREL.000023
            The Australian Business Number (ABN) of the Client must pass the ABN algorithm check.
    
            Legacy Rule Format:
            IF ((CUREL77 <> NULL) AND (ABNALGORITHM (CUREL77) = FALSE))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CUREL77 <> NULL) AND (FailsABNAlgorithm(^CUREL77))
    
            Data Elements:
    
            CUREL:^CUREL77 = :Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.CUREL77 != null && FailsABNAlgorithm(report.CUREL77));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Client ABN is invalid.",
                    LongDescription = @"The Australian Business Number (ABN) of the Client has failed the algorithm check.",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000023"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL77", Value = report.CUREL77 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000023
    
            #region VR.ATO.CUREL.000024
    
            /*  VR.ATO.CUREL.000024
            The Withholding Payer Number (WPN) of the Client must pass the WPN algorithm check.
    
            Legacy Rule Format:
            IF ((CUREL21 <> NULL) AND (WPNALGORITHM (CUREL21) = FALSE))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CUREL21 <> NULL) AND (FailsWPNAlgorithm(^CUREL21))
    
            Data Elements:
    
            CUREL:^CUREL21 = :Identifiers.WithholdingPayerNumber.Identifier
            */
            assertion = (report.CUREL21 != null && FailsTFNAlgorithm(report.CUREL21));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000024", Severity = ProcessMessageSeverity.Error,
                    Description = @"Client WPN is invalid.",
                    LongDescription = @"The Withholding Payer Number (WPN) of the Client has failed the algorithm check.",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:WithholdingPayerNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000024"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL21", Value = report.CUREL21 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000024
    
            #region VR.ATO.CUREL.000026
    
            /*  VR.ATO.CUREL.000026
            At least one client identifier type must be provided.
    
            Legacy Rule Format:
            IF ((CUREL33 = NULL) AND (CUREL77 = NULL) AND (CUREL21 = NULL) AND (CUREL24 = NULL) AND (CUREL104 = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CUREL33 = NULL) AND (^CUREL77 = NULL) AND (^CUREL21 = NULL) AND (^CUREL24 = NULL) AND (^CUREL104 = NULL))
    
            Data Elements:
    
            CUREL:^CUREL33 = :Identifiers.TaxFileNumber.Identifier
    
            CUREL:^CUREL77 = :Identifiers.AustralianBusinessNumber.Identifier
    
            CUREL:^CUREL21 = :Identifiers.WithholdingPayerNumber.Identifier
    
            CUREL:^CUREL24 = :Identifiers.ExciseIdentificationNumber.Identifier
    
            CUREL:^CUREL104 = :Identifiers.AustralianTaxationOfficeReferenceNumber.Identifier
            */
            assertion = (report.CUREL33 == null && report.CUREL77 == null && report.CUREL21 == null && report.CUREL24 == null && report.CUREL104 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CUREL.000030", Severity = ProcessMessageSeverity.Error,
                    Description = @"No client identifier was provided.",
                    LongDescription = @"At least one of the following client identifiers must be provided:
- Tax File Number;
- Australian Business Number;
- Withholding Payer Number;
- Excise Identification Number.
- Australian Taxation Office Reference Number",
                    Location = "/tns:CURELAddRemove/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CUREL.000026"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL33", Value = report.CUREL33 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL77", Value = report.CUREL77 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL21", Value = report.CUREL21 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL24", Value = report.CUREL24 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CUREL104", Value = report.CUREL104 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CUREL.000026

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 
