using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationFBT
{
    public partial class FBT2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;
        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;
        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;
        /// <summary>
        /// The report parameter
        /// </summary>
        private FBT2026 report;
        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref = "FBT2026Validator"/> class.
        /// </summary>
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FBT2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref = "FBT2026Validator"/> class.
        /// </summary>
        /// <param name = "report">Report variable </param>                
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FBT2026Validator(FBT2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

#region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;
            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }

        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }

            if (posB == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }

            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }

            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            return (
                from x in nodes
                select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            nodes = nodes.Where(x => x != null).ToArray();
            return !((
                from x in nodes
                select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values)
            where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;
            if (year == null || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;
            if (year == 0 || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }

        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;
        }

        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }

        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }

            return date.Value.Month;
        }

        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }

        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }

        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;
                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }

            return response;
        }

        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime? )dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;
            response = (field < (expression - range)) || (field > (expression + range));
            return response;
        }

        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;
                }
                else if (usi.Length == 9)
                {
                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }

            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;
            if (tan == null)
                return false;
            tan = tan.Trim();
            if (!decimal.TryParse(tan, out decimalTan))
                return true;
            if (tan.Length != 8)
                return true;
            decimal tanSum = 7 * int.Parse(tan.Substring(0, 1)) + 9 * int.Parse(tan.Substring(1, 1)) + 8 * int.Parse(tan.Substring(2, 1)) + 4 * int.Parse(tan.Substring(3, 1)) + 6 * int.Parse(tan.Substring(4, 1)) + 3 * int.Parse(tan.Substring(5, 1)) + 5 * int.Parse(tan.Substring(6, 1)) + 1 * int.Parse(tan.Substring(7, 1));
            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;
            if (abn == null)
                return false;
            abn = abn.Trim();
            if (!decimal.TryParse(abn, out decimalAbn))
                return true;
            if (abn.Length != 11)
                return true;
            decimal abnSum = 10 * (int.Parse(abn.Substring(0, 1)) - 1) + 1 * int.Parse(abn.Substring(1, 1)) + 3 * int.Parse(abn.Substring(2, 1)) + 5 * int.Parse(abn.Substring(3, 1)) + 7 * int.Parse(abn.Substring(4, 1)) + 9 * int.Parse(abn.Substring(5, 1)) + 11 * int.Parse(abn.Substring(6, 1)) + 13 * int.Parse(abn.Substring(7, 1)) + 15 * int.Parse(abn.Substring(8, 1)) + 17 * int.Parse(abn.Substring(9, 1)) + 19 * int.Parse(abn.Substring(10, 1));
            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;
            acn = acn.Trim();
            if (!decimal.TryParse(acn, out decimalAbn))
                return true;
            if (acn.Length != 9)
                return true;
            decimal abnSum = 8 * int.Parse(acn.Substring(0, 1)) + 7 * int.Parse(acn.Substring(1, 1)) + 6 * int.Parse(acn.Substring(2, 1)) + 5 * int.Parse(acn.Substring(3, 1)) + 4 * int.Parse(acn.Substring(4, 1)) + 3 * int.Parse(acn.Substring(5, 1)) + 2 * int.Parse(acn.Substring(6, 1)) + 1 * int.Parse(acn.Substring(7, 1));
            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;
            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;
            if (tfn == null)
                return false;
            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");
            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;
            if (tfn.Length < 8)
                return true;
            decimal tfn1To7Sum = 1 * int.Parse(tfn.Substring(0, 1)) + 4 * int.Parse(tfn.Substring(1, 1)) + 3 * int.Parse(tfn.Substring(2, 1)) + 7 * int.Parse(tfn.Substring(3, 1)) + 5 * int.Parse(tfn.Substring(4, 1)) + 8 * int.Parse(tfn.Substring(5, 1)) + 6 * int.Parse(tfn.Substring(6, 1));
            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));
            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;
                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;
                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }

        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;
            char[] characters = flags.ToCharArray();
            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name = "occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

#endregion // Functions

        public FBT2026 ConsumedReport
        {
            get
            {
                return report;
            }

            private set
            {
                report = value;
            }
        }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(FBT2026 reportIn)
        {
            //ProcessMessageDocument processMessage;
            //ProcessMessageParameter parameter;
            //bool assertion;
            this.ConsumedReport = reportIn;
            VRATOFBT500004();
            VRATOFBT500017();
            VRATOFBT500057();
            VRATOFBT500069();
            VRATOFBT500070();
            VRATOFBT500145();
            VRATOFBT500187();
            VRATOFBT500250();
            VRATOFBT500332();
            VRATOFBT500333();
            VRATOFBT500334();
            VRATOFBT500335();
            VRATOFBT500336();
            VRATOFBT500337();
            VRATOFBT500338();
            VRATOFBT500339();
            VRATOFBT500340();
            VRATOFBT500341();
            VRATOFBT500417();
            VRATOFBT500424();
            VRATOFBT500436();
            VRATOFBT500471();
            VRATOFBT500472();
            VRATOFBT500473();
            VRATOFBT500474();
            VRATOFBT500475();
            VRATOFBT500476();
            VRATOFBT500477();
            VRATOFBT500478();
            VRATOFBT500479();
            VRATOFBT500480();
            VRATOFBT500481();
            VRATOFBT500482();
            VRATOFBT500483();
            VRATOFBT500490();
            VRATOFBT500491();
            VRATOFBT500492();
            VRATOFBT500493();
            VRATOFBT500494();
            VRATOFBT500495();
            VRATOFBT500496();
            VRATOFBT500497();
            VRATOFBT500498();
            VRATOFBT500499();
            VRATOFBT500500();
            VRATOFBT500501();
            VRATOFBT500502();
            VRATOFBT500503();
            VRATOFBT500504();
            VRATOFBT500505();
            VRATOFBT500506();
            VRATOFBT500507();
            VRATOFBT500508();
            VRATOFBT500509();
            VRATOFBT500510();
            VRATOFBT500511();
            VRATOFBT500512();
            VRATOFBT500513();
            VRATOFBT500514();
            VRATOFBT500515();
            VRATOFBT500516();
            VRATOFBT500517();
            VRATOFBT500518();
            VRATOFBT500519();
            VRATOFBT500520();
            VRATOFBT500521();
            VRATOFBT500522();
            VRATOFBT500523();
            VRATOFBT500524();
            VRATOFBT500525();
            VRATOFBT500526();
            VRATOFBT500527();
            VRATOFBT500528();
            VRATOFBT500529();
            VRATOFBT500530();
            VRATOFBT500531();
            VRATOFBT500532();
            VRATOFBT500533();
            VRATOFBT500534();
            VRATOFBT500535();
            VRATOFBT500536();
            VRATOFBT500537();
            VRATOFBT500538();
            VRATOFBT500539();
            VRATOFBT500540();
            VRATOFBT500541();
            VRATOFBT500542();
            VRATOFBT500543();
            VRATOFBT500544();
            VRATOFBT500545();
            VRATOFBT500546();
            VRATOFBT500547();
            VRATOFBT500548();
            VRATOFBT500549();
            VRATOFBT500550();
            VRATOFBT500551();
            VRATOFBT500552();
            VRATOFBT500553();
            VRATOFBT500554();
            VRATOFBT500555();
            VRATOFBT500556();
            VRATOFBT500557();
            VRATOFBT500558();
            VRATOFBT500559();
            VRATOFBT500560();
            VRATOFBT500561();
            VRATOFBT500562();
            VRATOFBT500563();
            VRATOFBT500564();
            VRATOFBT500565();
            VRATOFBT500566();
            VRATOFBT500567();
            VRATOFBT500568();
            VRATOFBT500569();
            VRATOFBT500570();
            VRATOFBT500571();
            VRATOFBT500572();
            VRATOFBT500573();
            VRATOFBT500574();
            VRATOFBT500575();
            VRATOFBT500576();
            VRATOFBT500577();
            VRATOFBT500578();
            VRATOFBT500579();
            VRATOFBT500580();
            VRATOFBT500581();
            VRATOFBT500582();
            VRATOFBT500583();
            VRATOFBT500584();
            VRATOFBT500585();
            VRATOFBT500586();
            VRATOFBT500587();
            VRATOFBT500588();
            VRATOFBT500589();
            VRATOFBT500590();
            VRATOFBT500591();
            VRATOFBT500592();
            VRATOFBT500593();
            VRATOFBT500594();
            VRATOFBT500595();
            VRATOFBT500596();
            VRATOFBT500597();
            VRATOFBT500598();
            VRATOFBT500599();
            VRATOFBT500600();
            VRATOFBT500601();
            VRATOFBT500602();
            VRATOFBT500603();
            VRATOFBT500604();
            VRATOFBT500605();
            VRATOFBT500606();
            VRATOFBT500607();
            VRATOFBT500608();
            VRATOFBT500609();
            VRATOFBT500610();
            VRATOFBT500611();
            VRATOFBT500612();
            VRATOFBT500613();
            VRATOFBT500614();
            VRATOFBT500615();
            VRATOFBT500616();
            VRATOFBT500617();
            VRATOFBT500618();
            VRATOFBT500619();
            VRATOFBT500620();
            VRATOFBT500621();
            VRATOFBT500622();
            VRATOFBT500623();
            VRATOFBT500624();
            VRATOFBT500625();
            VRATOFBT500626();
            VRATOFBT500627();
            VRATOFBT500628();
            VRATOFBT500629();
            VRATOFBT500630();
            VRATOFBT500631();
            VRATOFBT500632();
            VRATOFBT500633();
            VRATOFBT500634();
            VRATOFBT500635();
            VRATOFBT500636();
            VRATOFBT500637();
            VRATOFBT500638();
            VRATOFBT500639();
            VRATOFBT500640();
            VRATOFBT500641();
            VRATOFBT500642();
            VRATOFBT500643();
            VRATOFBT500644();
            VRATOFBT500645();
            VRATOFBT500646();
            VRATOFBT500647();
            VRATOFBT500648();
            VRATOFBT500649();
            VRATOFBT500650();
            VRATOFBT500651();
            VRATOFBT500652();
            VRATOFBT500653();
            VRATOFBT500654();
            VRATOFBT500655();
            VRATOFBT500656();
            VRATOFBT500657();
            VRATOFBT500658();
            VRATOFBT500659();
            VRATOFBT500660();
            VRATOFBT500661();
            VRATOFBT500662();
            VRATOFBT500663();
            VRATOFBT500664();
            VRATOFBT536277();
            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

#region VR.ATO.FBT.500004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500004
            Both Controlling party current family name and Non-Individual Name are not provided. One of these names is required
    
            Legacy Rule Format:
            (^FBT1002 <> NULL AND (^FBT6 = BLANK AND ^FBT10 = BLANK))

            Technical Business Rule Format:
            (^FBT1002 <> NULL AND (^FBT6 = BLANK AND ^FBT10 = BLANK))
    
            Data Elements:
    
            ^FBT6 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
    
            ^FBT1002 = FBT:RP:CurrentTrusteeOrSeniorPartner
            */
            assertion = (report.RP_CurrentTrusteeOrSeniorPartnerCollectionExists != false && (string.IsNullOrWhiteSpace(report.FBT6) == true && string.IsNullOrWhiteSpace(report.FBT10) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Controlling party current family name and Non-Individual Name are not provided. One of these names is required",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500004"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT6", Value = report.FBT6 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500017
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500017()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500017
            Both Controlling party current family name and Non-Individual Name are present. Only one name is required
    
            Legacy Rule Format:
            (^FBT6 <> BLANK) AND (^FBT10 <> BLANK)

            Technical Business Rule Format:
            (^FBT6 <> BLANK) AND (^FBT10 <> BLANK)
    
            Data Elements:
    
            ^FBT6 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT6) != true && string.IsNullOrWhiteSpace(report.FBT10) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Controlling party current family name and Non-Individual Name are present. Only one name is required",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500017"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT6", Value = report.FBT6 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500057
            Payment due or credit due to you is less than zero (credit is due) and Bank Account Number is not present
    
            Legacy Rule Format:
            (^FBT77 < 0) AND (^FBT230 = NULL)

            Technical Business Rule Format:
            (^FBT77 < 0) AND (^FBT230 = NULL)
    
            Data Elements:
    
            ^FBT77 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.AdjustmentDue.Amount
    
            ^FBT230 = FBT:RP:BankDetails:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
            */
            assertion = (report.FBT77.GetValueOrDefault() < 0 && report.FBT230 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432268",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial institution details must be entered when a credit (refund) is due.",
                    LongDescription = @"Based on the return calculation details a credit or refund is due. In order for the ATO to pay this credit or refund, Australian financial institution account details must be provided. If account details are not available please phone the Tax Office on: For self lodgers 13 28 61 options 2 2 and for tax agents 13 72 86 options 1 3 1 2.",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxAdjustmentDueA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500057"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT77", Value = GetValueOrEmpty(report.FBT77) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT230", Value = report.FBT230 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500069
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500069()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500069
            Both Employers Family Name and Non-Individual Name are present . Only one name is required
    
            Legacy Rule Format:
            (^FBT12 <> BLANK) AND (^FBT16 <> BLANK)

            Technical Business Rule Format:
            (^FBT12 <> BLANK) AND (^FBT16 <> BLANK)
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT12) != true && string.IsNullOrWhiteSpace(report.FBT16) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Employers Family Name and Non-Individual Name are present . Only one name is required",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500069"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500070
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500070()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500070
            Employer Family Name or Non-individual Name is not present.
    
            Legacy Rule Format:
            (^FBT1006 <> NULL AND (^FBT12 = BLANK AND ^FBT16 = BLANK))

            Technical Business Rule Format:
            (^FBT1006 <> NULL AND (^FBT12 = BLANK AND ^FBT16 = BLANK))
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
    
            ^FBT1006 = FBT:RP:CurrentEmployer
            */
            assertion = (report.RP_CurrentEmployerCollectionExists != false && (string.IsNullOrWhiteSpace(report.FBT12) == true && string.IsNullOrWhiteSpace(report.FBT16) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000070",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Employer Family Name or Non-individual Name is not present.",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500070"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500145()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500145
            Both Employer previous family name and Employer previous non-individual name are present. Only one name is required
    
            Legacy Rule Format:
            (^FBT27 <> BLANK) AND (^FBT31 <> BLANK)

            Technical Business Rule Format:
            (^FBT27 <> BLANK) AND (^FBT31 <> BLANK)
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT27) != true && string.IsNullOrWhiteSpace(report.FBT31) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Employer previous family name and Employer previous non-individual name are present. Only one name is required.",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500145"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500187
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500187()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500187
            Current postal address and previous postal address are identical. Previous postal address is only required when an address change has occurred since the last return
    
            Legacy Rule Format:
            (^FBT32 = ^FBT17) AND (^FBT33 = ^FBT18) AND (^FBT36 = ^FBT21) AND (^FBT38 = ^FBT23) AND (^FBT37 = ^FBT22) AND (^FBT223 = ^FBT222)

            Technical Business Rule Format:
            (^FBT32 = ^FBT17) AND (^FBT33 = ^FBT18) AND (^FBT36 = ^FBT21) AND (^FBT38 = ^FBT23) AND (^FBT37 = ^FBT22) AND (^FBT223 = ^FBT222)
    
            Data Elements:
    
            ^FBT32 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line1.Text
    
            ^FBT17 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line1.Text
    
            ^FBT18 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line2.Text
    
            ^FBT21 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.LocalityName.Text
    
            ^FBT22 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FBT23 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Postcode.Text
    
            ^FBT33 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line2.Text
    
            ^FBT36 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.LocalityName.Text
    
            ^FBT37 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FBT38 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Postcode.Text
    
            ^FBT222 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT223 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FBT32 == report.FBT17 && report.FBT33 == report.FBT18 && report.FBT36 == report.FBT21 && report.FBT38 == report.FBT23 && report.FBT37 == report.FBT22 && report.FBT223 == report.FBT222);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000187",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address and previous postal address are identical. Previous postal address is only required when an address change has occurred since the last return",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500187"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT32", Value = report.FBT32 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT17", Value = report.FBT17 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT33", Value = GetValueOrEmpty(report.FBT33) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT18", Value = GetValueOrEmpty(report.FBT18) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT36", Value = report.FBT36 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT21", Value = report.FBT21 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT38", Value = GetValueOrEmpty(report.FBT38) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT23", Value = GetValueOrEmpty(report.FBT23) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT37", Value = GetValueOrEmpty(report.FBT37) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT22", Value = GetValueOrEmpty(report.FBT22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT223", Value = GetValueOrEmpty(report.FBT223) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT222", Value = GetValueOrEmpty(report.FBT222) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500250
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500250()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500250
            Both Controlling Party Previous Name Family name and Controlling Party Previous Name Non-individual name are present. Only one name is required
    
            Legacy Rule Format:
            (^FBT50 <> BLANK) AND (^FBT54 <> BLANK)

            Technical Business Rule Format:
            (^FBT50 <> BLANK) AND (^FBT54 <> BLANK)
    
            Data Elements:
    
            ^FBT50 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT50) != true && string.IsNullOrWhiteSpace(report.FBT54) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000250",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Controlling Party Previous Name Family name and Controlling Party Previous Name Non-individual name are present. Only one name is required",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500250"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT50", Value = report.FBT50 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500332
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500332()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500332
            Type 1 aggregate factored amount is incorrect
    
            Legacy Rule Format:
            (^FBT67 > ((^FBT66 * Constant('GROSS UP HIGHER RATE')) + 50)) OR (^FBT67 < ((^FBT66 * Constant('GROSS UP HIGHER RATE')) - 50))

            Technical Business Rule Format:
            (^FBT67 > ((^FBT66 * Constant('GROSS UP HIGHER RATE')) + 50)) OR (^FBT67 < ((^FBT66 * Constant('GROSS UP HIGHER RATE')) - 50))
    
            Data Elements:
    
            ^FBT67 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Type1Aggregate.Amount
    
            ^FBT66 = FBT:RP:ReturnCalculationDetails:Remuneration.FringeBenefits.Type1Aggregate.Amount
            */
            assertion = (report.FBT67.GetValueOrDefault() > report.FBT66.GetValueOrDefault() * 2.0802M + 50 || report.FBT67.GetValueOrDefault() < report.FBT66.GetValueOrDefault() * 2.0802M - 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000332",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Type 1 aggregate factored amount is incorrect",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxType1AggregateA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500332"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "GROSS UP HIGHER RATE", Value = "2.0802" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT67", Value = GetValueOrEmpty(report.FBT67) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT66", Value = GetValueOrEmpty(report.FBT66) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500333
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500333()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500333
            Type 2 aggregate factored amount is incorrect
    
            Legacy Rule Format:
            (^FBT69 > ((^FBT68 * Constant('GROSS UP LOWER RATE')) + 50)) OR (^FBT69 < ((^FBT68 * Constant('GROSS UP LOWER RATE')) - 50))

            Technical Business Rule Format:
            (^FBT69 > ((^FBT68 * Constant('GROSS UP LOWER RATE')) + 50)) OR (^FBT69 < ((^FBT68 * Constant('GROSS UP LOWER RATE')) - 50))
    
            Data Elements:
    
            ^FBT69 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Type2Aggregate.Amount
    
            ^FBT68 = FBT:RP:ReturnCalculationDetails:Remuneration.FringeBenefits.Type2Aggregate.Amount
            */
            assertion = (report.FBT69.GetValueOrDefault() > report.FBT68.GetValueOrDefault() * 1.8868M + 50 || report.FBT69.GetValueOrDefault() < report.FBT68.GetValueOrDefault() * 1.8868M - 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000333",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Type 2 aggregate factored amount is incorrect",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxType2AggregateA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500333"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "GROSS UP LOWER RATE", Value = "1.8868" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT69", Value = GetValueOrEmpty(report.FBT69) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT68", Value = GetValueOrEmpty(report.FBT68) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500334
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500334()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500334
            Fringe benefits taxable amount (A + B) or C must equal Type 1 aggregate factored amount plus Type 2 aggregate factored amount plus Aggregate Non-exempt amount (hospitals, ambulances, public benevolent institutions and health promotion charities only)
    
            Legacy Rule Format:
            OutsideRange(^FBT71, ^FBT67 + ^FBT69 + ^FBT70, 1)

            Technical Business Rule Format:
            OutsideRange(^FBT71, ^FBT67 + ^FBT69 + ^FBT70, 1)
    
            Data Elements:
    
            ^FBT71 = FBT:RP:ReturnCalculationDetails:Remuneration.FringeBenefits.TaxableTotal.Amount
    
            ^FBT67 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Type1Aggregate.Amount
    
            ^FBT69 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Type2Aggregate.Amount
    
            ^FBT70 = FBT:RP:ReturnCalculationDetails:Remuneration.FringeBenefits.AggregateNonExempt.Amount
            */
            assertion = OutsideRange(report.FBT71.GetValueOrDefault(), report.FBT67.GetValueOrDefault() + report.FBT69.GetValueOrDefault() + report.FBT70.GetValueOrDefault(), 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000334",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Fringe benefits taxable amount (A + B) or C must equal Type 1 aggregate factored amount plus Type 2 aggregate factored amount plus Aggregate Non-exempt amount (hospitals, ambulances, public benevolent institutions and health promotion charities only)",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:RemunerationFringeBenefitsTaxableTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500334"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT71", Value = GetValueOrEmpty(report.FBT71) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT67", Value = GetValueOrEmpty(report.FBT67) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT69", Value = GetValueOrEmpty(report.FBT69) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT70", Value = GetValueOrEmpty(report.FBT70) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500335
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500335()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500335
            Amount of tax payable is not equal to Fringe benefits taxable amount (A + B) or C multiplied by FBT TAX RATE
    
            Legacy Rule Format:
            (^FBT72 > ((^FBT71 * Constant('FBT TAX RATE')) + 50)) OR (^FBT72 < ((^FBT71 * Constant('FBT TAX RATE')) - 50))

            Technical Business Rule Format:
            (^FBT72 > ((^FBT71 * Constant('FBT TAX RATE')) + 50)) OR (^FBT72 < ((^FBT71 * Constant('FBT TAX RATE')) - 50))
    
            Data Elements:
    
            ^FBT72 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Payable.Amount
    
            ^FBT71 = FBT:RP:ReturnCalculationDetails:Remuneration.FringeBenefits.TaxableTotal.Amount
            */
            assertion = (report.FBT72.GetValueOrDefault() > report.FBT71.GetValueOrDefault() * 0.47M + 50 || report.FBT72.GetValueOrDefault() < report.FBT71.GetValueOrDefault() * 0.47M - 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000335",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of tax payable is not equal to Fringe benefits taxable amount (A + B) or C multiplied by FBT TAX RATE",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxPayableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500335"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "FBT TAX RATE", Value = "0.47" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT72", Value = GetValueOrEmpty(report.FBT72) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT71", Value = GetValueOrEmpty(report.FBT71) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500336
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500336()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500336
            Aggregate non rebatable amount cannot be greater than Amount of tax payable
    
            Legacy Rule Format:
            (^FBT73 <> BLANK) AND (^FBT73 > ^FBT72)

            Technical Business Rule Format:
            (^FBT73 <> BLANK) AND (^FBT73 > ^FBT72)
    
            Data Elements:
    
            ^FBT73 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.NonRebatableAggregate.Amount
    
            ^FBT72 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Payable.Amount
            */
            assertion = (report.FBT73 != null && report.FBT73.GetValueOrDefault() > report.FBT72.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000336",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregate non rebatable amount cannot be greater than Amount of tax payable",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxNonRebatableAggregateA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500336"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT73", Value = GetValueOrEmpty(report.FBT73) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT72", Value = GetValueOrEmpty(report.FBT72) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500337
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500337()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500337
            Aggregate non rebatable amount is absent and Amount of rebate is present. Enter zero if there is no non-rebatable amount
    
            Legacy Rule Format:
            (^FBT73 = NULL) AND (^FBT74 <> NULL)

            Technical Business Rule Format:
            (^FBT73 = NULL) AND (^FBT74 <> NULL)
    
            Data Elements:
    
            ^FBT73 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.NonRebatableAggregate.Amount
    
            ^FBT74 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Rebate.Amount
            */
            assertion = (report.FBT73 == null && report.FBT74 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000337",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregate non rebatable amount is absent and Amount of rebate is present. Enter zero if there is no non-rebatable amount",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxNonRebatableAggregateA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500337"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT73", Value = GetValueOrEmpty(report.FBT73) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT74", Value = GetValueOrEmpty(report.FBT74) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500338
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500338()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500338
            Aggregate non rebatable amount is present and Amount of rebate is absent. Enter zero if there is no rebate amount
    
            Legacy Rule Format:
            (^FBT73 <> NULL) AND (^FBT74 = NULL)

            Technical Business Rule Format:
            (^FBT73 <> NULL) AND (^FBT74 = NULL)
    
            Data Elements:
    
            ^FBT74 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Rebate.Amount
    
            ^FBT73 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.NonRebatableAggregate.Amount
            */
            assertion = (report.FBT73 != null && report.FBT74 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000338",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregate non rebatable amount is present and Amount of rebate is absent. Enter zero if there is no rebate amount",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxRebateA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500338"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT73", Value = GetValueOrEmpty(report.FBT73) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT74", Value = GetValueOrEmpty(report.FBT74) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500339
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500339()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500339
            Amount of rebate is greater than zero and Amount of rebate is greater than 47% of Total amount of tax calculated
    
            Legacy Rule Format:
            (^FBT74 > 0) AND (OutsideRange(^FBT74, (^FBT72 - ^FBT73) *0.47, 50))

            Technical Business Rule Format:
            (^FBT74 > 0) AND (OutsideRange(^FBT74, (^FBT72 - ^FBT73) *0.47, 50))
    
            Data Elements:
    
            ^FBT74 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Rebate.Amount
    
            ^FBT72 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Payable.Amount
    
            ^FBT73 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.NonRebatableAggregate.Amount
            */
            assertion = (report.FBT74.GetValueOrDefault() > 0 && OutsideRange(report.FBT74.GetValueOrDefault(), (report.FBT72.GetValueOrDefault() - report.FBT73.GetValueOrDefault()) * 0.47M, 50));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.100063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of rebate is greater than zero and Amount of rebate is greater than 47% of Total amount of tax calculated",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxRebateA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500339"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT74", Value = GetValueOrEmpty(report.FBT74) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT72", Value = GetValueOrEmpty(report.FBT72) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT73", Value = GetValueOrEmpty(report.FBT73) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500340
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500340()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500340
            Sub - total (Amount of tax payable less Amount of rebate) is not equal to Amount of tax payable less Amount of rebate
    
            Legacy Rule Format:
            ^FBT75 <> (^FBT72 - ^FBT74)

            Technical Business Rule Format:
            ^FBT75 <> (^FBT72 - ^FBT74)
    
            Data Elements:
    
            ^FBT75 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Liability.Amount
    
            ^FBT72 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Payable.Amount
    
            ^FBT74 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Rebate.Amount
            */
            assertion = (report.FBT75.GetValueOrDefault() != report.FBT72.GetValueOrDefault() - report.FBT74.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000340",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sub - total (Amount of tax payable less Amount of rebate) is not equal to Amount of tax payable less Amount of rebate",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxLiabilityA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500340"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT75", Value = GetValueOrEmpty(report.FBT75) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT72", Value = GetValueOrEmpty(report.FBT72) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT74", Value = GetValueOrEmpty(report.FBT74) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500341
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500341()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500341
            Payment due or credit due to you is not equal to Sub - total (Amount of tax payable less Amount of rebate) minus Less instalment amounts reported on activity statements
    
            Legacy Rule Format:
            ^FBT77 <> (^FBT75 - ^FBT76)

            Technical Business Rule Format:
            ^FBT77 <> (^FBT75 - ^FBT76)
    
            Data Elements:
    
            ^FBT77 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.AdjustmentDue.Amount
    
            ^FBT75 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.Liability.Amount
    
            ^FBT76 = FBT:RP:ReturnCalculationDetails:FringeBenefitsTax.LiabilityInstalmentsTotal.Amount
            */
            assertion = (report.FBT77.GetValueOrDefault() != report.FBT75.GetValueOrDefault() - report.FBT76.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000341",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payment due or credit due to you is not equal to Sub - total (Amount of tax payable less Amount of rebate) minus Less instalment amounts reported on activity statements",
                    Location = "/tns:FBT/tns:RP/tns:ReturnCalculationDetails/tns:FringeBenefitsTaxAdjustmentDueA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500341"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT77", Value = GetValueOrEmpty(report.FBT77) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT75", Value = GetValueOrEmpty(report.FBT75) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT76", Value = GetValueOrEmpty(report.FBT76) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500417
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500417()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500417
            Tax Agent number (TAN) is invalid
    
            Legacy Rule Format:
            (^FBT2 <> BLANK) AND (FailsTANAlgorithm(^FBT2))

            Technical Business Rule Format:
            (^FBT2 <> BLANK) AND (FailsTANAlgorithm(^FBT2))
    
            Data Elements:
    
            ^FBT2 = FBT:INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT2) != true && FailsTANAlgorithm(report.FBT2));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Agent number (TAN) is invalid",
                    Location = "/tns:FBT/tns:INT/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500417"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT2", Value = GetValueOrEmpty(report.FBT2) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500424
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500424()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500424
            Both reporting party and intermediary contact telephone numbers are missing, at least one is required
    
            Legacy Rule Format:
            (^FBT128 = BLANK) AND (^FBT61 = BLANK)

            Technical Business Rule Format:
            (^FBT128 = BLANK) AND (^FBT61 = BLANK)
    
            Data Elements:
    
            ^FBT61 = FBT:RP:ContactDetails:ElectronicContact.Telephone.Minimal.Number
    
            ^FBT128 = FBT:INT:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT128) == true && string.IsNullOrWhiteSpace(report.FBT61) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both reporting party and intermediary contact telephone numbers are missing, at least one is required",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactTelephoneMinimalN",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500424"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT128", Value = report.FBT128 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT61", Value = report.FBT61 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500436
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500436()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500436
            Agent contact name is required
    
            Legacy Rule Format:
            (^FBT2 <> BLANK) AND (^FBT129 = BLANK)

            Technical Business Rule Format:
            (^FBT2 <> BLANK) AND (^FBT129 = BLANK)
    
            Data Elements:
    
            ^FBT129 = FBT:INT:PersonUnstructuredName.FullName.Text
    
            ^FBT2 = FBT:INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT2) != true && string.IsNullOrWhiteSpace(report.FBT129) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.000436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Agent contact name is required",
                    Location = "/tns:FBT/tns:INT/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500436"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT2", Value = GetValueOrEmpty(report.FBT2) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT129", Value = report.FBT129 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500471
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500471()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500471
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT81 <> BLANK) AND (^FBT81 <> (^FBT79 - ^FBT80))

            Technical Business Rule Format:
            (^FBT81 <> BLANK) AND (^FBT81 <> (^FBT79 - ^FBT80))
    
            Data Elements:
    
            ^FBT81 = FBT:RP:CarStatutoryFormula:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT79 = FBT:RP:CarStatutoryFormula:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT80 = FBT:RP:CarStatutoryFormula:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = (report.FBT81 != null && report.FBT81.GetValueOrDefault() != report.FBT79.GetValueOrDefault() - report.FBT80.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:CarStatutoryFormula/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500471"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT81", Value = GetValueOrEmpty(report.FBT81) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT79", Value = GetValueOrEmpty(report.FBT79) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT80", Value = GetValueOrEmpty(report.FBT80) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500472
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500472()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500472
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT85 <> BLANK) AND (^FBT85 <> (^FBT83 - ^FBT84))

            Technical Business Rule Format:
            (^FBT85 <> BLANK) AND (^FBT85 <> (^FBT83 - ^FBT84))
    
            Data Elements:
    
            ^FBT85 = FBT:RP:CarOperatingCostMethod:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT83 = FBT:RP:CarOperatingCostMethod:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT84 = FBT:RP:CarOperatingCostMethod:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = (report.FBT85 != null && report.FBT85.GetValueOrDefault() != report.FBT83.GetValueOrDefault() - report.FBT84.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:CarOperatingCostMethod/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500472"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT85", Value = GetValueOrEmpty(report.FBT85) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT83", Value = GetValueOrEmpty(report.FBT83) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT84", Value = GetValueOrEmpty(report.FBT84) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500473
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500473()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500473
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT89 <> BLANK) AND (^FBT89 <> (^FBT87 - ^FBT88))

            Technical Business Rule Format:
            (^FBT89 <> BLANK) AND (^FBT89 <> (^FBT87 - ^FBT88))
    
            Data Elements:
    
            ^FBT89 = FBT:RP:LoansGranted:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT87 = FBT:RP:LoansGranted:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT88 = FBT:RP:LoansGranted:Remuneration.FringeBenefits.Reduction.Amount
            */
            assertion = (report.FBT89 != null && report.FBT89.GetValueOrDefault() != report.FBT87.GetValueOrDefault() - report.FBT88.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:LoansGranted/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500473"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT89", Value = GetValueOrEmpty(report.FBT89) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT87", Value = GetValueOrEmpty(report.FBT87) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT88", Value = GetValueOrEmpty(report.FBT88) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500474
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500474()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500474
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT91 <> BLANK) AND (^FBT91 <> ^FBT90)

            Technical Business Rule Format:
            (^FBT91 <> BLANK) AND (^FBT91 <> ^FBT90)
    
            Data Elements:
    
            ^FBT91 = FBT:RP:DebtWaiver:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT90 = FBT:RP:DebtWaiver:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
            */
            assertion = (report.FBT91 != null && report.FBT91.GetValueOrDefault() != report.FBT90.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:DebtWaiver/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500474"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT91", Value = GetValueOrEmpty(report.FBT91) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT90", Value = GetValueOrEmpty(report.FBT90) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500475
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500475()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500475
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT95 <> BLANK) AND (^FBT95 <> (^FBT92 - ^FBT93 - ^FBT94))

            Technical Business Rule Format:
            (^FBT95 <> BLANK) AND (^FBT95 <> (^FBT92 - ^FBT93 - ^FBT94))
    
            Data Elements:
    
            ^FBT95 = FBT:RP:ExpensePayments:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT92 = FBT:RP:ExpensePayments:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT93 = FBT:RP:ExpensePayments:Remuneration.FringeBenefits.EmployeeContribution.Amount
    
            ^FBT94 = FBT:RP:ExpensePayments:Remuneration.FringeBenefits.Reduction.Amount
            */
            assertion = (report.FBT95 != null && report.FBT95.GetValueOrDefault() != report.FBT92.GetValueOrDefault() - report.FBT93.GetValueOrDefault() - report.FBT94.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:ExpensePayments/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500475"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT95", Value = GetValueOrEmpty(report.FBT95) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT92", Value = GetValueOrEmpty(report.FBT92) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT93", Value = GetValueOrEmpty(report.FBT93) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT94", Value = GetValueOrEmpty(report.FBT94) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500476
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500476()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500476
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT99 <> BLANK) AND (^FBT99 <> (^FBT97 - ^FBT98))

            Technical Business Rule Format:
            (^FBT99 <> BLANK) AND (^FBT99 <> (^FBT97 - ^FBT98))
    
            Data Elements:
    
            ^FBT99 = FBT:RP:Housing:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT97 = FBT:RP:Housing:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT98 = FBT:RP:Housing:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = (report.FBT99 != null && report.FBT99.GetValueOrDefault() != report.FBT97.GetValueOrDefault() - report.FBT98.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:Housing/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500476"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT99", Value = GetValueOrEmpty(report.FBT99) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT97", Value = GetValueOrEmpty(report.FBT97) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT98", Value = GetValueOrEmpty(report.FBT98) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500477
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500477()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500477
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT103 <> BLANK) AND (^FBT103 <> (^FBT101 - ^FBT102))

            Technical Business Rule Format:
            (^FBT103 <> BLANK) AND (^FBT103 <> (^FBT101 - ^FBT102))
    
            Data Elements:
    
            ^FBT103 = FBT:RP:LAFHA:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT101 = FBT:RP:LAFHA:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT102 = FBT:RP:LAFHA:Remuneration.FringeBenefits.Reduction.Amount
            */
            assertion = (report.FBT103 != null && report.FBT103.GetValueOrDefault() != report.FBT101.GetValueOrDefault() - report.FBT102.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:LAFHA/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500477"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT103", Value = GetValueOrEmpty(report.FBT103) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT101", Value = GetValueOrEmpty(report.FBT101) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT102", Value = GetValueOrEmpty(report.FBT102) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500478
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500478()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500478
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT111 <> BLANK) AND (^FBT111 <> (^FBT108 - ^FBT109 - ^FBT110))

            Technical Business Rule Format:
            (^FBT111 <> BLANK) AND (^FBT111 <> (^FBT108 - ^FBT109 - ^FBT110))
    
            Data Elements:
    
            ^FBT111 = FBT:RP:Board:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT108 = FBT:RP:Board:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT109 = FBT:RP:Board:Remuneration.FringeBenefits.EmployeeContribution.Amount
    
            ^FBT110 = FBT:RP:Board:Remuneration.FringeBenefits.Reduction.Amount
            */
            assertion = (report.FBT111 != null && report.FBT111.GetValueOrDefault() != report.FBT108.GetValueOrDefault() - report.FBT109.GetValueOrDefault() - report.FBT110.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:Board/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500478"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT111", Value = GetValueOrEmpty(report.FBT111) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT108", Value = GetValueOrEmpty(report.FBT108) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT109", Value = GetValueOrEmpty(report.FBT109) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT110", Value = GetValueOrEmpty(report.FBT110) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500479
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500479()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500479
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT115 <> BLANK) AND (^FBT115 <> (^FBT112 - ^FBT113 - ^FBT114))

            Technical Business Rule Format:
            (^FBT115 <> BLANK) AND (^FBT115 <> (^FBT112 - ^FBT113 - ^FBT114))
    
            Data Elements:
    
            ^FBT115 = FBT:RP:Property:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT112 = FBT:RP:Property:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT113 = FBT:RP:Property:Remuneration.FringeBenefits.EmployeeContribution.Amount
    
            ^FBT114 = FBT:RP:Property:Remuneration.FringeBenefits.Reduction.Amount
            */
            assertion = (report.FBT115 != null && report.FBT115.GetValueOrDefault() != report.FBT112.GetValueOrDefault() - report.FBT113.GetValueOrDefault() - report.FBT114.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:Property/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500479"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT115", Value = GetValueOrEmpty(report.FBT115) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT112", Value = GetValueOrEmpty(report.FBT112) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT113", Value = GetValueOrEmpty(report.FBT113) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT114", Value = GetValueOrEmpty(report.FBT114) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500480
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500480()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500480
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT117 <> BLANK) AND (^FBT117 <> ^FBT116)

            Technical Business Rule Format:
            (^FBT117 <> BLANK) AND (^FBT117 <> ^FBT116)
    
            Data Elements:
    
            ^FBT117 = FBT:RP:ITExemptBodyEntertainment:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT116 = FBT:RP:ITExemptBodyEntertainment:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
            */
            assertion = (report.FBT117 != null && report.FBT117.GetValueOrDefault() != report.FBT116.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:ITExemptBodyEntertainment/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500480"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT117", Value = GetValueOrEmpty(report.FBT117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT116", Value = GetValueOrEmpty(report.FBT116) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500481
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500481()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500481
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT121 <> BLANK) AND (^FBT121 <> (^FBT118 - ^FBT119 - ^FBT120))

            Technical Business Rule Format:
            (^FBT121 <> BLANK) AND (^FBT121 <> (^FBT118 - ^FBT119 - ^FBT120))
    
            Data Elements:
    
            ^FBT121 = FBT:RP:OtherBenefits:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT118 = FBT:RP:OtherBenefits:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT119 = FBT:RP:OtherBenefits:Remuneration.FringeBenefits.EmployeeContribution.Amount
    
            ^FBT120 = FBT:RP:OtherBenefits:Remuneration.FringeBenefits.Reduction.Amount
            */
            assertion = (report.FBT121 != null && report.FBT121.GetValueOrDefault() != report.FBT118.GetValueOrDefault() - report.FBT119.GetValueOrDefault() - report.FBT120.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:OtherBenefits/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500481"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT121", Value = GetValueOrEmpty(report.FBT121) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT118", Value = GetValueOrEmpty(report.FBT118) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT119", Value = GetValueOrEmpty(report.FBT119) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT120", Value = GetValueOrEmpty(report.FBT120) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500482
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500482()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500482
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT124 <> BLANK) AND (^FBT124 <> (^FBT122 - ^FBT123))

            Technical Business Rule Format:
            (^FBT124 <> BLANK) AND (^FBT124 <> (^FBT122 - ^FBT123))
    
            Data Elements:
    
            ^FBT124 = FBT:RP:CarParking:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT122 = FBT:RP:CarParking:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
    
            ^FBT123 = FBT:RP:CarParking:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = (report.FBT124 != null && report.FBT124.GetValueOrDefault() != report.FBT122.GetValueOrDefault() - report.FBT123.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:CarParking/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500482"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT124", Value = GetValueOrEmpty(report.FBT124) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT122", Value = GetValueOrEmpty(report.FBT122) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT123", Value = GetValueOrEmpty(report.FBT123) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500483
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500483()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500483
            For a given 'Details of fringe benefits provided' type, the Taxable value of benefits must equal the Gross Taxable value of benefits minus the Employee contribution minus the Value of reductions.
    
            Legacy Rule Format:
            (^FBT126 <> BLANK) AND (^FBT126 <> ^FBT125)

            Technical Business Rule Format:
            (^FBT126 <> BLANK) AND (^FBT126 <> ^FBT125)
    
            Data Elements:
    
            ^FBT126 = FBT:RP:MealEntertainment:Remuneration.FringeBenefits.BenefitNetTaxable.Amount
    
            ^FBT125 = FBT:RP:MealEntertainment:Remuneration.FringeBenefits.BenefitGrossTaxable.Amount
            */
            assertion = (report.FBT126 != null && report.FBT126.GetValueOrDefault() != report.FBT125.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000474",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable value of benefits must equal Gross Taxable value of benefits minus Employee contribution (if applicable) minus Value of reductions (if applicable)",
                    Location = "/tns:FBT/tns:RP/tns:MealEntertainment/tns:RemunerationFringeBenefitsBenefitNetTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500483"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT126", Value = GetValueOrEmpty(report.FBT126) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT125", Value = GetValueOrEmpty(report.FBT125) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500490
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500490()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500490
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            (^FBT4 <> BLANK) AND (FailsABNAlgorithm(^FBT4))

            Technical Business Rule Format:
            (^FBT4 <> BLANK) AND (FailsABNAlgorithm(^FBT4))
    
            Data Elements:
    
            ^FBT4 = FBT:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT4) != true && FailsABNAlgorithm(report.FBT4));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:FBT/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500490"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT4", Value = GetValueOrEmpty(report.FBT4) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500491
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500491()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500491
            If the return is an amendment, sequence number, type and reason must be present
    
            Legacy Rule Format:
            (^FBT234 = TRUE) AND (^FBT235 = NULL OR ^FBT236 = NULL OR ^FBT237 = BLANK)

            Technical Business Rule Format:
            (^FBT234 = TRUE) AND (^FBT235 = NULL OR ^FBT236 = NULL OR ^FBT237 = BLANK)
    
            Data Elements:
    
            ^FBT234 = FBT:RP:Report.Amendment.Indicator
    
            ^FBT235 = FBT:RP:Report.AmendmentSequence.Number
    
            ^FBT236 = FBT:RP:Report.AmendmentType.Code
    
            ^FBT237 = FBT:RP:Report.AmendmentReason.Text
            */
            assertion = (report.FBT234 == true && (report.FBT235 == null || report.FBT236 == null || string.IsNullOrWhiteSpace(report.FBT237) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/tns:FBT/tns:RP/tns:AmendmentI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500491"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT234", Value = GetValueOrEmpty(report.FBT234) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT235", Value = report.FBT235.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT236", Value = GetValueOrEmpty(report.FBT236) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT237", Value = GetValueOrEmpty(report.FBT237) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500492
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500492()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500492
            If the return is an amendment, Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            (^FBT234 <> TRUE) AND (^FBT235 <> NULL OR ^FBT236 <> BLANK OR ^FBT237 <> BLANK)

            Technical Business Rule Format:
            (^FBT234 <> TRUE) AND (^FBT235 <> NULL OR ^FBT236 <> BLANK OR ^FBT237 <> BLANK)
    
            Data Elements:
    
            ^FBT234 = FBT:RP:Report.Amendment.Indicator
    
            ^FBT235 = FBT:RP:Report.AmendmentSequence.Number
    
            ^FBT236 = FBT:RP:Report.AmendmentType.Code
    
            ^FBT237 = FBT:RP:Report.AmendmentReason.Text
            */
            assertion = (report.FBT234 != true && (report.FBT235 != null || string.IsNullOrWhiteSpace(report.FBT236) != true || string.IsNullOrWhiteSpace(report.FBT237) != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/tns:FBT/tns:RP/tns:AmendmentI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500492"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT234", Value = GetValueOrEmpty(report.FBT234) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT235", Value = report.FBT235.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT236", Value = GetValueOrEmpty(report.FBT236) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT237", Value = GetValueOrEmpty(report.FBT237) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500493
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500493()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500493
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT6 <> BLANK) AND (NotContainsSet(^FBT6, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT6 <> BLANK) AND (NotContainsSet(^FBT6, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT6 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT6) != true && !(IsMatch(report.FBT6, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500493"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT6", Value = report.FBT6 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500494
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500494()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500494
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT6, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT6, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT6 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT6, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500494"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT6", Value = report.FBT6 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500495
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500495()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500495
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT6, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT6, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT6 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT6, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500495"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT6", Value = report.FBT6 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500496
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500496()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500496
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT8 <> BLANK) AND (NotContainsSet(^FBT8, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT8 <> BLANK) AND (NotContainsSet(^FBT8, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT8 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT8) != true && !(IsMatch(report.FBT8, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500496"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT8", Value = GetValueOrEmpty(report.FBT8) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500497
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500497()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500497
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT8, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT8, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT8 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT8, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500497"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT8", Value = GetValueOrEmpty(report.FBT8) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500498
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500498()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500498
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT8, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT8, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT8 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT8, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500498"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT8", Value = GetValueOrEmpty(report.FBT8) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500499
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500499()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500499
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT8, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT8, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT8 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT8, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500499"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT8", Value = GetValueOrEmpty(report.FBT8) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500500
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500500()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500500
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT9 <> BLANK) AND (NotContainsSet(^FBT9, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT9 <> BLANK) AND (NotContainsSet(^FBT9, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT9 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT9) != true && !(IsMatch(report.FBT9, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500500"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT9", Value = GetValueOrEmpty(report.FBT9) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500501
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500501()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500501
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT9, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT9, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT9 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT9, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500501"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT9", Value = GetValueOrEmpty(report.FBT9) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500502
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500502()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500502
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT9, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT9, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT9 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT9, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500502"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT9", Value = GetValueOrEmpty(report.FBT9) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500503
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500503()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500503
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT9, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT9, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT9 = FBT:RP:CurrentTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT9, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500503"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT9", Value = GetValueOrEmpty(report.FBT9) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500504
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500504()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500504
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FBT10, 'T/A ')) AND (EndsWithSet(^FBT10, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FBT10, 'T/A ')) AND (EndsWithSet(^FBT10, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FBT10, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.FBT10, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500504"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500505
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500505()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500505
            Non-individual name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FBT10 <> BLANK) AND (NotContainsSet(^FBT10, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FBT10 <> BLANK) AND (NotContainsSet(^FBT10, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT10) != true && !(IsMatch(report.FBT10, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500505"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500506
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500506()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500506
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FBT10, ' - ')

            Technical Business Rule Format:
            Contains(^FBT10, ' - ')
    
            Data Elements:
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT10 == null ? false : report.FBT10.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500506"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500507
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500507()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500507
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FBT10, 'P/L')

            Technical Business Rule Format:
            Contains(^FBT10, 'P/L')
    
            Data Elements:
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT10 == null ? false : report.FBT10.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500507"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500508
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500508()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500508
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT10, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT10, '"--","''","  "')
    
            Data Elements:
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT10, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500508"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500509
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500509()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500509
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^FBT10, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^FBT10, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^FBT10 = FBT:RP:CurrentTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT10, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500509"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT10", Value = GetValueOrEmpty(report.FBT10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500510
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500510()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500510
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT50 <> BLANK) AND (NotContainsSet(^FBT50, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT50 <> BLANK) AND (NotContainsSet(^FBT50, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT50 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT50) != true && !(IsMatch(report.FBT50, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500510"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT50", Value = report.FBT50 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500511
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500511()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500511
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT50, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT50, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT50 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT50, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500511"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT50", Value = report.FBT50 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500512
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500512()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500512
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT50, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT50, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT50 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT50, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500512"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT50", Value = report.FBT50 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500513
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500513()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500513
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT52 <> BLANK) AND (NotContainsSet(^FBT52, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT52 <> BLANK) AND (NotContainsSet(^FBT52, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT52 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT52) != true && !(IsMatch(report.FBT52, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500513"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT52", Value = GetValueOrEmpty(report.FBT52) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500514
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500514()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500514
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT52, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT52, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT52 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT52, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500514"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT52", Value = GetValueOrEmpty(report.FBT52) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500515
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500515()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500515
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT52, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT52, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT52 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT52, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500515"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT52", Value = GetValueOrEmpty(report.FBT52) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500516
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500516()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500516
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT52, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT52, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT52 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT52, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500516"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT52", Value = GetValueOrEmpty(report.FBT52) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500517
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500517()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500517
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT53 <> BLANK) AND (NotContainsSet(^FBT53, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT53 <> BLANK) AND (NotContainsSet(^FBT53, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT53 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT53) != true && !(IsMatch(report.FBT53, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500517"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT53", Value = GetValueOrEmpty(report.FBT53) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500518
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500518()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500518
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT53, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT53, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT53 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT53, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500518"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT53", Value = GetValueOrEmpty(report.FBT53) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500519
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500519()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500519
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT53, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT53, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT53 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT53, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500519"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT53", Value = GetValueOrEmpty(report.FBT53) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500520
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500520()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500520
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT53, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT53, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT53 = FBT:RP:PreviousTrusteeOrSeniorPartner:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT53, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500520"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT53", Value = GetValueOrEmpty(report.FBT53) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500521
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500521()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500521
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FBT54, 'T/A ')) AND (EndsWithSet(^FBT54, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FBT54, 'T/A ')) AND (EndsWithSet(^FBT54, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FBT54, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.FBT54, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500521"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500522
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500522()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500522
            Non-individual name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FBT54  <> BLANK) AND (NotContainsSet(^FBT54, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FBT54  <> BLANK) AND (NotContainsSet(^FBT54, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT54) != true && !(IsMatch(report.FBT54, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500522"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500523
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500523()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500523
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FBT54, ' - ')

            Technical Business Rule Format:
            Contains(^FBT54, ' - ')
    
            Data Elements:
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT54 == null ? false : report.FBT54.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500523"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500524
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500524()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500524
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FBT54, 'P/L')

            Technical Business Rule Format:
            Contains(^FBT54, 'P/L')
    
            Data Elements:
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT54 == null ? false : report.FBT54.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500524"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500525
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500525()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500525
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT54, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT54, '"--","''","  "')
    
            Data Elements:
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT54, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500525"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500526
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500526()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500526
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^FBT54, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^FBT54, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^FBT54 = FBT:RP:PreviousTrusteeOrSeniorPartner:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT54, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousTrusteeOrSeniorPartner/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500526"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT54", Value = GetValueOrEmpty(report.FBT54) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500527
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500527()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500527
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT12 <> BLANK) AND (NotContainsSet(^FBT12, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT12 <> BLANK) AND (NotContainsSet(^FBT12, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT12) != true && !(IsMatch(report.FBT12, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500527"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500528
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500528()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500528
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT12, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT12, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT12, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500528"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500529
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500529()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500529
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT12, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT12, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT12, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500529"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500530
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500530()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500530
            Family name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^FBT12, ' - ')

            Technical Business Rule Format:
            Contains(^FBT12, ' - ')
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FBT12 == null ? false : report.FBT12.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500530"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500531
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500531()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500531
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT12, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT12, '"--","''","  "')
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT12, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500531"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500532
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500532()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500532
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT12, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT12, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT12 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT12, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500532"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT12", Value = report.FBT12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500533
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500533()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500533
            First name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT14 <> BLANK) AND (NotContainsSet(^FBT14, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT14 <> BLANK) AND (NotContainsSet(^FBT14, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT14) != true && !(IsMatch(report.FBT14, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500533"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500534
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500534()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500534
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT14, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT14, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT14, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500534"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500535
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500535()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500535
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT14, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT14, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT14, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500535"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500536
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500536()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500536
            First name must be entered if other given name is entered
    
            Legacy Rule Format:
            (^FBT15 <> BLANK) AND (^FBT14 = BLANK)

            Technical Business Rule Format:
            (^FBT15 <> BLANK) AND (^FBT14 = BLANK)
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT15) != true && string.IsNullOrWhiteSpace(report.FBT14) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500536"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500537
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500537()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500537
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FBT14, ' - ')

            Technical Business Rule Format:
            Contains(^FBT14, ' - ')
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FBT14 == null ? false : report.FBT14.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500537"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500538
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500538()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500538
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT14, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT14, '"--","''","  "')
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT14, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500538"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500539
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500539()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500539
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT14, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT14, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT14 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT14, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500539"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT14", Value = GetValueOrEmpty(report.FBT14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500540
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500540()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500540
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT15 <> BLANK) AND (NotContainsSet(^FBT15, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT15 <> BLANK) AND (NotContainsSet(^FBT15, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT15) != true && !(IsMatch(report.FBT15, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500540"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500541
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500541()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500541
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT15, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT15, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT15, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500541"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500542
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500542()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500542
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT15, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT15, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT15, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500542"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500543
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500543()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500543
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FBT15, ' - ')

            Technical Business Rule Format:
            Contains(^FBT15, ' - ')
    
            Data Elements:
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FBT15 == null ? false : report.FBT15.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500543"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500544
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500544()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500544
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT15, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT15, '"--","''","  "')
    
            Data Elements:
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT15, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500544"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500545
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500545()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500545
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT15, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT15, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT15 = FBT:RP:CurrentEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT15, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500545"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT15", Value = GetValueOrEmpty(report.FBT15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500546
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500546()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500546
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FBT16, 'T/A ')) AND (EndsWithSet(^FBT16, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FBT16, 'T/A ')) AND (EndsWithSet(^FBT16, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FBT16, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.FBT16, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500546"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500547
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500547()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500547
            Non-individual name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FBT16  <> BLANK) AND (NotContainsSet(^FBT16, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FBT16  <> BLANK) AND (NotContainsSet(^FBT16, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT16) != true && !(IsMatch(report.FBT16, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500547"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500548
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500548()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500548
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FBT16, ' - ')

            Technical Business Rule Format:
            Contains(^FBT16, ' - ')
    
            Data Elements:
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT16 == null ? false : report.FBT16.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500548"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500549
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500549()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500549
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FBT16, 'P/L')

            Technical Business Rule Format:
            Contains(^FBT16, 'P/L')
    
            Data Elements:
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT16 == null ? false : report.FBT16.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500549"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500550
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500550()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500550
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT16, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT16, '"--","''","  "')
    
            Data Elements:
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT16, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500550"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500551
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500551()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500551
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^FBT16, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^FBT16, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^FBT16 = FBT:RP:CurrentEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT16, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500551"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT16", Value = GetValueOrEmpty(report.FBT16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500552
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500552()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500552
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            (^FBT25 = FALSE) AND (^FBT22 = NULL OR ^FBT23 = NULL)

            Technical Business Rule Format:
            (^FBT25 = FALSE) AND (^FBT22 = NULL OR ^FBT23 = NULL)
    
            Data Elements:
    
            ^FBT25 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^FBT22 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FBT23 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.FBT25 == false && (report.FBT22 == null || report.FBT23 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500552"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT25", Value = GetValueOrEmpty(report.FBT25) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT22", Value = GetValueOrEmpty(report.FBT22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT23", Value = GetValueOrEmpty(report.FBT23) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500553
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500553()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500553
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FBT17 <> BLANK) AND (StartsWithSet(^FBT17,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FBT17, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FBT17 <> BLANK) AND (StartsWithSet(^FBT17,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FBT17, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FBT17 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT17) != true && IsMatch(report.FBT17, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.FBT17, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500553"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT17", Value = report.FBT17 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500554
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500554()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500554
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            (Contains(^FBT17, 'C/-')) AND (^FBT18 = BLANK)

            Technical Business Rule Format:
            (Contains(^FBT17, 'C/-')) AND (^FBT18 = BLANK)
    
            Data Elements:
    
            ^FBT17 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line1.Text
    
            ^FBT18 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.FBT17 == null ? false : report.FBT17.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.FBT18) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500554"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT17", Value = report.FBT17 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT18", Value = GetValueOrEmpty(report.FBT18) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500555
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500555()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500555
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^FBT17, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^FBT17, '"AS ABOVE"')
    
            Data Elements:
    
            ^FBT17 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FBT17, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500555"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT17", Value = report.FBT17 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500556
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500556()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500556
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT17, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT17, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT17 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.FBT17 == null ? false : report.FBT17.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500556"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT17", Value = report.FBT17 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500557
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500557()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500557
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            (^FBT18 <> BLANK) AND (StartsWithSet(^FBT18,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FBT18, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FBT18 <> BLANK) AND (StartsWithSet(^FBT18,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FBT18, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FBT18 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT18) != true && (IsMatch(report.FBT18, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.FBT18, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500557"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT18", Value = GetValueOrEmpty(report.FBT18) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500558
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500558()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500558
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT18, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT18, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT18 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.FBT18 == null ? false : report.FBT18.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500558"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT18", Value = GetValueOrEmpty(report.FBT18) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500559
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500559()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500559
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^FBT21, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^FBT21, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^FBT21 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.FBT21, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500559"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT21", Value = report.FBT21 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500560
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500560()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500560
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT21, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT21, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT21 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.FBT21 == null ? false : report.FBT21.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500560"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT21", Value = report.FBT21 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500561
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500561()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500561
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            (^FBT25 = FALSE) AND (ContainsSet(^FBT21, '0-9'))

            Technical Business Rule Format:
            (^FBT25 = FALSE) AND (ContainsSet(^FBT21, '0-9'))
    
            Data Elements:
    
            ^FBT21 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.LocalityName.Text
    
            ^FBT25 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT25 == false && IsMatch(report.FBT21, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500561"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT25", Value = GetValueOrEmpty(report.FBT25) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT21", Value = report.FBT21 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500562
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500562()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500562
            Country code must be present for overseas address
    
            Legacy Rule Format:
            (^FBT25 = TRUE) AND (^FBT222 = NULL)

            Technical Business Rule Format:
            (^FBT25 = TRUE) AND (^FBT222 = NULL)
    
            Data Elements:
    
            ^FBT222 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT25 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT25 == true && report.FBT222 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500562"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT25", Value = GetValueOrEmpty(report.FBT25) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT222", Value = GetValueOrEmpty(report.FBT222) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500563
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500563()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500563
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            (^FBT222 = 'au') AND  (^FBT25 = TRUE)

            Technical Business Rule Format:
            (^FBT222 = 'au') AND  (^FBT25 = TRUE)
    
            Data Elements:
    
            ^FBT222 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT25 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT222 == @"au" && report.FBT25 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500563"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT222", Value = GetValueOrEmpty(report.FBT222) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT25", Value = GetValueOrEmpty(report.FBT25) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500564
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500564()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500564
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^FBT222 <> BLANK) AND (^FBT222 <> 'au') AND (^FBT25 = FALSE)

            Technical Business Rule Format:
            (^FBT222 <> BLANK) AND (^FBT222 <> 'au') AND (^FBT25 = FALSE)
    
            Data Elements:
    
            ^FBT222 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT25 = FBT:RP:CurrentEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT222) != true && report.FBT222 != @"au" && report.FBT25 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:FBT/tns:RP/tns:CurrentEmployer/tns:AddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500564"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT222", Value = GetValueOrEmpty(report.FBT222) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT25", Value = GetValueOrEmpty(report.FBT25) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500565
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500565()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500565
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT27 <> BLANK) AND (NotContainsSet(^FBT27, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT27 <> BLANK) AND (NotContainsSet(^FBT27, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT27) != true && !(IsMatch(report.FBT27, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500565"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500566
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500566()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500566
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT27, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT27, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT27, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500566"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500567
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500567()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500567
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT27, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT27, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT27, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500567"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500568
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500568()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500568
            Family name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^FBT27, ' - ')

            Technical Business Rule Format:
            Contains(^FBT27, ' - ')
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FBT27 == null ? false : report.FBT27.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500568"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500569
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500569()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500569
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT27, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT27, '"--","''","  "')
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT27, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500569"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500570
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500570()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500570
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT27, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT27, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT27 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT27, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500570"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT27", Value = report.FBT27 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500571
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500571()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500571
            Given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT29 <> BLANK) AND (NotContainsSet(^FBT29, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT29 <> BLANK) AND (NotContainsSet(^FBT29, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT29) != true && !(IsMatch(report.FBT29, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500571"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500572
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500572()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500572
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT29, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT29, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT29, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500572"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500573
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500573()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500573
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT29, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT29, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT29, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500573"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500574
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500574()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500574
            First name must be entered if other given name is entered
    
            Legacy Rule Format:
            (^FBT30 <> BLANK) AND (^FBT29 = BLANK)

            Technical Business Rule Format:
            (^FBT30 <> BLANK) AND (^FBT29 = BLANK)
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT30) != true && string.IsNullOrWhiteSpace(report.FBT29) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500574"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500575
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500575()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500575
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FBT29, ' - ')

            Technical Business Rule Format:
            Contains(^FBT29, ' - ')
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FBT29 == null ? false : report.FBT29.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500575"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500576
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500576()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500576
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT29, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT29, '"--","''","  "')
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT29, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500576"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500577
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500577()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500577
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT29, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT29, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT29 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT29, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500577"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT29", Value = GetValueOrEmpty(report.FBT29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500578
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500578()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500578
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT30 <> BLANK) AND (NotContainsSet(^FBT30, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT30 <> BLANK) AND (NotContainsSet(^FBT30, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT30) != true && !(IsMatch(report.FBT30, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500578"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500579
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500579()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500579
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT30, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT30, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT30, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500579"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500580
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500580()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500580
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT30, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT30, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT30, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500580"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500581
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500581()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500581
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FBT30, ' - ')

            Technical Business Rule Format:
            Contains(^FBT30, ' - ')
    
            Data Elements:
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FBT30 == null ? false : report.FBT30.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500581"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500582
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500582()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500582
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT30, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT30, '"--","''","  "')
    
            Data Elements:
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT30, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500582"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500583
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500583()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500583
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT30, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT30, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT30 = FBT:RP:PreviousEmployer:IndividualDetail:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT30, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:IndividualDetail/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500583"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT30", Value = GetValueOrEmpty(report.FBT30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500584
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500584()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500584
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FBT31, 'T/A ')) AND (EndsWithSet(^FBT31, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FBT31, 'T/A ')) AND (EndsWithSet(^FBT31, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FBT31, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.FBT31, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500584"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500585
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500585()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500585
            Non-individual name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FBT31  <> BLANK) AND (NotContainsSet(^FBT31, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FBT31  <> BLANK) AND (NotContainsSet(^FBT31, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT31) != true && !(IsMatch(report.FBT31, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500585"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500586
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500586()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500586
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FBT31, ' - ')

            Technical Business Rule Format:
            Contains(^FBT31, ' - ')
    
            Data Elements:
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT31 == null ? false : report.FBT31.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500586"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500587
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500587()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500587
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FBT31, 'P/L')

            Technical Business Rule Format:
            Contains(^FBT31, 'P/L')
    
            Data Elements:
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT31 == null ? false : report.FBT31.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500587"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500588
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500588()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500588
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT31, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT31, '"--","''","  "')
    
            Data Elements:
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT31, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500588"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500589
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500589()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500589
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^FBT31, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^FBT31, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^FBT31 = FBT:RP:PreviousEmployer:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT31, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500589"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT31", Value = GetValueOrEmpty(report.FBT31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500590
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500590()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500590
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            (^FBT159 = FALSE) AND (^FBT37 = NULL OR ^FBT38 = NULL)

            Technical Business Rule Format:
            (^FBT159 = FALSE) AND (^FBT37 = NULL OR ^FBT38 = NULL)
    
            Data Elements:
    
            ^FBT159 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^FBT37 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FBT38 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.FBT159 == false && (report.FBT37 == null || report.FBT38 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500590"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT159", Value = GetValueOrEmpty(report.FBT159) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT37", Value = GetValueOrEmpty(report.FBT37) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT38", Value = GetValueOrEmpty(report.FBT38) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500591
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500591()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500591
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FBT32 <> BLANK) AND (StartsWithSet(^FBT32,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FBT32, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FBT32 <> BLANK) AND (StartsWithSet(^FBT32,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FBT32, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FBT32 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT32) != true && IsMatch(report.FBT32, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.FBT32, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500591"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT32", Value = report.FBT32 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500592
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500592()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500592
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            (Contains(^FBT32, 'C/-')) AND (^FBT33 = BLANK)

            Technical Business Rule Format:
            (Contains(^FBT32, 'C/-')) AND (^FBT33 = BLANK)
    
            Data Elements:
    
            ^FBT32 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line1.Text
    
            ^FBT33 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.FBT32 == null ? false : report.FBT32.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.FBT33) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500592"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT32", Value = report.FBT32 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT33", Value = GetValueOrEmpty(report.FBT33) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500593
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500593()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500593
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^FBT32, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^FBT32, '"AS ABOVE"')
    
            Data Elements:
    
            ^FBT32 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FBT32, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500593"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT32", Value = report.FBT32 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500594
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500594()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500594
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT32, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT32, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT32 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.FBT32 == null ? false : report.FBT32.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500594"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT32", Value = report.FBT32 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500595
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500595()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500595
            Address line 2 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FBT33 <> BLANK) AND (StartsWithSet(^FBT33,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR (ContainsSet(^FBT33, '" C/- "," C/O "," C/ "," Care Of "')))

            Technical Business Rule Format:
            (^FBT33 <> BLANK) AND (StartsWithSet(^FBT33,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR (ContainsSet(^FBT33, '" C/- "," C/O "," C/ "," Care Of "')))
    
            Data Elements:
    
            ^FBT33 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT33) != true && (IsMatch(report.FBT33, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.FBT33, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500595"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT33", Value = GetValueOrEmpty(report.FBT33) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500596
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500596()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500596
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT33, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT33, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT33 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.FBT33 == null ? false : report.FBT33.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500596"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT33", Value = GetValueOrEmpty(report.FBT33) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500597
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500597()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500597
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            (FoundSet(^FBT36, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (FoundSet(^FBT36, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^FBT36 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.FBT36, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500597"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT36", Value = report.FBT36 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500598
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500598()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500598
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT36, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT36, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT36 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.FBT36 == null ? false : report.FBT36.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500598"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT36", Value = report.FBT36 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500599
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500599()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500599
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            (^FBT159 = FALSE) AND (ContainsSet(^FBT36, '0-9'))

            Technical Business Rule Format:
            (^FBT159 = FALSE) AND (ContainsSet(^FBT36, '0-9'))
    
            Data Elements:
    
            ^FBT36 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.LocalityName.Text
    
            ^FBT159 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT159 == false && IsMatch(report.FBT36, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500599"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT159", Value = GetValueOrEmpty(report.FBT159) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT36", Value = report.FBT36 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500600
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500600()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500600
            Country code must be present for overseas address
    
            Legacy Rule Format:
            (^FBT159 = TRUE) AND (^FBT223 = NULL)

            Technical Business Rule Format:
            (^FBT159 = TRUE) AND (^FBT223 = NULL)
    
            Data Elements:
    
            ^FBT223 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT159 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT159 == true && report.FBT223 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500600"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT159", Value = GetValueOrEmpty(report.FBT159) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT223", Value = GetValueOrEmpty(report.FBT223) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500601
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500601()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500601
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            (^FBT223 = 'au') AND (^FBT159 = TRUE)

            Technical Business Rule Format:
            (^FBT223 = 'au') AND (^FBT159 = TRUE)
    
            Data Elements:
    
            ^FBT223 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT159 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT223 == @"au" && report.FBT159 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500601"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT223", Value = GetValueOrEmpty(report.FBT223) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT159", Value = GetValueOrEmpty(report.FBT159) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500602
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500602()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500602
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^FBT223 <> BLANK) AND (^FBT223 <> 'au') AND (^FBT159 = FALSE)

            Technical Business Rule Format:
            (^FBT223 <> BLANK) AND (^FBT223 <> 'au') AND (^FBT159 = FALSE)
    
            Data Elements:
    
            ^FBT223 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.Country.Code
    
            ^FBT159 = FBT:RP:PreviousEmployer:AddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT223) != true && report.FBT223 != @"au" && report.FBT159 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:FBT/tns:RP/tns:PreviousEmployer/tns:AddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500602"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT223", Value = GetValueOrEmpty(report.FBT223) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT159", Value = GetValueOrEmpty(report.FBT159) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500603
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500603()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500603
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FBT40, 'T/A ')) AND (EndsWithSet(^FBT40, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FBT40, 'T/A ')) AND (EndsWithSet(^FBT40, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FBT40 = FBT:RP:BusinessTradingDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FBT40, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.FBT40, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500603"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT40", Value = report.FBT40 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500604
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500604()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500604
            Non-individual name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FBT40 <> BLANK) AND (NotContainsSet(^FBT40, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FBT40 <> BLANK) AND (NotContainsSet(^FBT40, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FBT40 = FBT:RP:BusinessTradingDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT40) != true && !(IsMatch(report.FBT40, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500604"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT40", Value = report.FBT40 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500605
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500605()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500605
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FBT40, ' - ')

            Technical Business Rule Format:
            Contains(^FBT40, ' - ')
    
            Data Elements:
    
            ^FBT40 = FBT:RP:BusinessTradingDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT40 == null ? false : report.FBT40.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500605"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT40", Value = report.FBT40 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500606
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500606()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500606
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FBT40, 'P/L')

            Technical Business Rule Format:
            Contains(^FBT40, 'P/L')
    
            Data Elements:
    
            ^FBT40 = FBT:RP:BusinessTradingDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FBT40 == null ? false : report.FBT40.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500606"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT40", Value = report.FBT40 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500607
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500607()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500607
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT40, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT40, '"--","''","  "')
    
            Data Elements:
    
            ^FBT40 = FBT:RP:BusinessTradingDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT40, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500607"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT40", Value = report.FBT40 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500608
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500608()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500608
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^FBT40, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^FBT40, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^FBT40 = FBT:RP:BusinessTradingDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FBT40, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500608"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT40", Value = report.FBT40 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500609
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500609()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500609
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            (^FBT164 = FALSE) AND (^FBT46 = NULL OR ^FBT47 = NULL)

            Technical Business Rule Format:
            (^FBT164 = FALSE) AND (^FBT46 = NULL OR ^FBT47 = NULL)
    
            Data Elements:
    
            ^FBT164 = FBT:RP:BusinessTradingDetails:AddressDetails.OverseasAddress.Indicator
    
            ^FBT46 = FBT:RP:BusinessTradingDetails:AddressDetails.StateOrTerritory.Code
    
            ^FBT47 = FBT:RP:BusinessTradingDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.FBT164 == false && (report.FBT46 == null || report.FBT47 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsOverseasAddressI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500609"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT164", Value = GetValueOrEmpty(report.FBT164) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT46", Value = GetValueOrEmpty(report.FBT46) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT47", Value = GetValueOrEmpty(report.FBT47) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500610
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500610()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500610
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^FBT41, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^FBT41, '"AS ABOVE"')
    
            Data Elements:
    
            ^FBT41 = FBT:RP:BusinessTradingDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FBT41, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500610"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT41", Value = report.FBT41 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500611
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500611()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500611
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT41, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT41, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT41 = FBT:RP:BusinessTradingDetails:AddressDetails.Line1.Text
            */
            assertion = (report.FBT41 == null ? false : report.FBT41.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500611"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT41", Value = report.FBT41 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500612
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500612()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500612
            Address line 2 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FBT42 <> BLANK) AND (StartsWithSet(^FBT42,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FBT42, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FBT42 <> BLANK) AND (StartsWithSet(^FBT42,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FBT42, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FBT42 = FBT:RP:BusinessTradingDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT42) != true && (IsMatch(report.FBT42, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.FBT42, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500612"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT42", Value = GetValueOrEmpty(report.FBT42) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500613
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500613()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500613
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT42, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT42, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT42 = FBT:RP:BusinessTradingDetails:AddressDetails.Line2.Text
            */
            assertion = (report.FBT42 == null ? false : report.FBT42.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500613"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT42", Value = GetValueOrEmpty(report.FBT42) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500614
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500614()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500614
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^FBT45, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^FBT45, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^FBT45 = FBT:RP:BusinessTradingDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.FBT45, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500614"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT45", Value = report.FBT45 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500615
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500615()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500615
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FBT45, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FBT45, 'UNKNOWN')
    
            Data Elements:
    
            ^FBT45 = FBT:RP:BusinessTradingDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.FBT45 == null ? false : report.FBT45.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500615"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT45", Value = report.FBT45 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500616
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500616()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500616
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            (^FBT164 = FALSE) AND (ContainsSet(^FBT45, '0-9'))

            Technical Business Rule Format:
            (^FBT164 = FALSE) AND (ContainsSet(^FBT45, '0-9'))
    
            Data Elements:
    
            ^FBT45 = FBT:RP:BusinessTradingDetails:AddressDetails.LocalityName.Text
    
            ^FBT164 = FBT:RP:BusinessTradingDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT164 == false && IsMatch(report.FBT45, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500616"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT164", Value = GetValueOrEmpty(report.FBT164) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT45", Value = report.FBT45 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500617
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500617()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500617
            Country code must be present for overseas address
    
            Legacy Rule Format:
            (^FBT164 = TRUE) AND (^FBT224 = NULL)

            Technical Business Rule Format:
            (^FBT164 = TRUE) AND (^FBT224 = NULL)
    
            Data Elements:
    
            ^FBT224 = FBT:RP:BusinessTradingDetails:AddressDetails.Country.Code
    
            ^FBT164 = FBT:RP:BusinessTradingDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT164 == true && report.FBT224 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500617"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT164", Value = GetValueOrEmpty(report.FBT164) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT224", Value = GetValueOrEmpty(report.FBT224) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500618
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500618()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500618
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            (^FBT224 = 'au') AND  (^FBT164 = TRUE)

            Technical Business Rule Format:
            (^FBT224 = 'au') AND  (^FBT164 = TRUE)
    
            Data Elements:
    
            ^FBT224 = FBT:RP:BusinessTradingDetails:AddressDetails.Country.Code
    
            ^FBT164 = FBT:RP:BusinessTradingDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.FBT224 == @"au" && report.FBT164 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500618"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT224", Value = GetValueOrEmpty(report.FBT224) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT164", Value = GetValueOrEmpty(report.FBT164) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500619
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500619()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500619
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^FBT224 <> BLANK) AND (^FBT224 <> 'au') AND (^FBT164 = FALSE)

            Technical Business Rule Format:
            (^FBT224 <> BLANK) AND (^FBT224 <> 'au') AND (^FBT164 = FALSE)
    
            Data Elements:
    
            ^FBT224 = FBT:RP:BusinessTradingDetails:AddressDetails.Country.Code
    
            ^FBT164 = FBT:RP:BusinessTradingDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT224) != true && report.FBT224 != @"au" && report.FBT164 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500619"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT224", Value = GetValueOrEmpty(report.FBT224) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT164", Value = GetValueOrEmpty(report.FBT164) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500620
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500620()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500620
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT56 <> BLANK) AND (NotContainsSet(^FBT56, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT56 <> BLANK) AND (NotContainsSet(^FBT56, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT56 = FBT:RP:ContactDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT56) != true && !(IsMatch(report.FBT56, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500620"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT56", Value = report.FBT56 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500621
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500621()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500621
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT56, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT56, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT56 = FBT:RP:ContactDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT56, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500621"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT56", Value = report.FBT56 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500622
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500622()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500622
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT56, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT56, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT56 = FBT:RP:ContactDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT56, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500622"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT56", Value = report.FBT56 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500623
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500623()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500623
            Family name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^FBT56, ' - ')

            Technical Business Rule Format:
            Contains(^FBT56, ' - ')
    
            Data Elements:
    
            ^FBT56 = FBT:RP:ContactDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FBT56 == null ? false : report.FBT56.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500623"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT56", Value = report.FBT56 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500624
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500624()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500624
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT56, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT56, '"--","''","  "')
    
            Data Elements:
    
            ^FBT56 = FBT:RP:ContactDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT56, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500624"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT56", Value = report.FBT56 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500625
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500625()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500625
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT56, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT56, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT56 = FBT:RP:ContactDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FBT56, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500625"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT56", Value = report.FBT56 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500626
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500626()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500626
            First name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT58 <> BLANK) AND (NotContainsSet(^FBT58, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT58 <> BLANK) AND (NotContainsSet(^FBT58, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT58) != true && !(IsMatch(report.FBT58, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500626"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500627
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500627()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500627
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT58, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT58, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT58, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500627"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500628
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500628()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500628
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT58, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT58, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT58, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500628"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500629
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500629()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500629
            First name must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            (^FBT59 <> BLANK) AND (^FBT58 = BLANK)

            Technical Business Rule Format:
            (^FBT59 <> BLANK) AND (^FBT58 = BLANK)
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT59) != true && string.IsNullOrWhiteSpace(report.FBT58) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500629"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500630
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500630()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500630
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FBT58, ' - ')

            Technical Business Rule Format:
            Contains(^FBT58, ' - ')
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FBT58 == null ? false : report.FBT58.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500630"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500631
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500631()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500631
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT58, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT58, '"--","''","  "')
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT58, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500631"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500632
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500632()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500632
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT58, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT58, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT58 = FBT:RP:ContactDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FBT58, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500632"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT58", Value = GetValueOrEmpty(report.FBT58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500633
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500633()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500633
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FBT59 <> BLANK) AND (NotContainsSet(^FBT59, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FBT59 <> BLANK) AND (NotContainsSet(^FBT59, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT59) != true && !(IsMatch(report.FBT59, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500633"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500634
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500634()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500634
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^FBT59, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^FBT59, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT59, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500634"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500635
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500635()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500635
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^FBT59, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^FBT59, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT59, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500635"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500636
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500636()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500636
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FBT59, ' - ')

            Technical Business Rule Format:
            Contains(^FBT59, ' - ')
    
            Data Elements:
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FBT59 == null ? false : report.FBT59.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500636"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500637
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500637()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500637
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^FBT59, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^FBT59, '"--","''","  "')
    
            Data Elements:
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT59, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500637"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500638
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500638()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500638
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^FBT59, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^FBT59, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^FBT59 = FBT:RP:ContactDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FBT59, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500638"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT59", Value = GetValueOrEmpty(report.FBT59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500639
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500639()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500639
            Email address must not start or end with an @ symbol
    
            Legacy Rule Format:
            (StartsWith(^FBT62, '@')) OR (EndsWith(^FBT62, '@'))

            Technical Business Rule Format:
            (StartsWith(^FBT62, '@')) OR (EndsWith(^FBT62, '@'))
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (IsMatch(report.FBT62, @"^(@)", RegexOptions.IgnoreCase) || IsMatch(report.FBT62, @"(@)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430222",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with an @ symbol",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500639"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500640
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500640()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500640
            Email address must not contain more than one @ symbol
    
            Legacy Rule Format:
            ContainsMultiple(^FBT62, '@')

            Technical Business Rule Format:
            ContainsMultiple(^FBT62, '@')
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = IsMatch(report.FBT62, @"@(.*)@", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one @ symbol",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500640"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500641
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500641()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500641
            Email address must not start or end with a dot (.)
    
            Legacy Rule Format:
            (StartsWith(^FBT62, '.')) OR (EndsWith(^FBT62, '.'))

            Technical Business Rule Format:
            (StartsWith(^FBT62, '.')) OR (EndsWith(^FBT62, '.'))
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (IsMatch(report.FBT62, @"^(\.)", RegexOptions.IgnoreCase) || IsMatch(report.FBT62, @"(\.)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with a dot (.)",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500641"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500642
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500642()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500642
            Email address must not contain more than one consecutive dots (..)
    
            Legacy Rule Format:
            Contains(^FBT62, '..')

            Technical Business Rule Format:
            Contains(^FBT62, '..')
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FBT62 == null ? false : report.FBT62.ToUpperInvariant().Contains(@".."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one consecutive dots (..)",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500642"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500643
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500643()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500643
            Email address must not contain a dot followed by a @ symbol (.@)
    
            Legacy Rule Format:
            Contains(^FBT62, '.@')

            Technical Business Rule Format:
            Contains(^FBT62, '.@')
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FBT62 == null ? false : report.FBT62.ToUpperInvariant().Contains(@".@"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430227",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a dot followed by a @ symbol (.@)",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500643"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500644
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500644()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500644
            Email address must not contain a @ symbol followed by a dot (@.)
    
            Legacy Rule Format:
            Contains(^FBT62, '@.')

            Technical Business Rule Format:
            Contains(^FBT62, '@.')
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FBT62 == null ? false : report.FBT62.ToUpperInvariant().Contains(@"@."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430228",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a @ symbol followed by a dot (@.)",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500644"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500645
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500645()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500645
            Email address must not contain an invalid character
    
            Legacy Rule Format:
            ContainsSet(^FBT62, '"(", ")", "<", ">", ",", ";", ":", "\",  """, "]", "["')

            Technical Business Rule Format:
            ContainsSet(^FBT62, '"(", ")", "<", ">", ",", ";", ":", "\",  """, "]", "["')
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = IsMatch(report.FBT62, @"\(|\)|<|>|,|;|:|\\|""|\]|\[", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430229",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain an invalid character",
                    LongDescription = @"An email address must not contain any of the following invalid characters: ( ) < > , ; : \ "" [ ]",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500645"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500646
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500646()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500646
            Email address must contain an @ symbol
    
            Legacy Rule Format:
            (^FBT62 <> BLANK) AND (NotContains(^FBT62, '@'))

            Technical Business Rule Format:
            (^FBT62 <> BLANK) AND (NotContains(^FBT62, '@'))
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT62) != true && !(report.FBT62 == null ? true : report.FBT62.ToUpperInvariant().Contains(@"@")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain an @ symbol",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500646"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500647
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500647()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500647
            An email address must not start with a space or include a space within it
    
            Legacy Rule Format:
            Contains(^FBT62, ' ')

            Technical Business Rule Format:
            Contains(^FBT62, ' ')
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FBT62 == null ? false : report.FBT62.ToUpperInvariant().Contains(@" "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430231",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An email address must not start with a space or include a space within it",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500647"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500648
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500648()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500648
            Email address must contain a dot (.)
    
            Legacy Rule Format:
            (^FBT62 <> BLANK) AND (NotContains(^FBT62, '.'))

            Technical Business Rule Format:
            (^FBT62 <> BLANK) AND (NotContains(^FBT62, '.'))
    
            Data Elements:
    
            ^FBT62 = FBT:RP:ContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT62) != true && !(report.FBT62 == null ? true : report.FBT62.ToUpperInvariant().Contains(@".")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430232",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain a dot (.)",
                    Location = "/tns:FBT/tns:RP/tns:ContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500648"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT62", Value = GetValueOrEmpty(report.FBT62) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500649
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500649()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500649
            BSB Number must be six digits and greater than 012000
    
            Legacy Rule Format:
            (^FBT229 <> BLANK) AND ((^FBT229 < 012001) OR (^FBT229 > 999999))

            Technical Business Rule Format:
            (^FBT229 <> BLANK) AND ((^FBT229 < 012001) OR (^FBT229 > 999999))
    
            Data Elements:
    
            ^FBT229 = FBT:RP:BankDetails:FinancialInstitutionAccount.BankStateBranch.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT229) != true && (AsNumeric(report.FBT229) < 12001 || AsNumeric(report.FBT229) > 999999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be six digits and greater than 012000",
                    Location = "/tns:FBT/tns:RP/tns:BankDetails/tns:FinancialInstitutionAccountBankStateBranchN",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500649"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT229", Value = report.FBT229 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500650
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500650()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500650
            Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; " ,
    
            Legacy Rule Format:
            ContainsSet(^FBT231, '"_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ","')

            Technical Business Rule Format:
            ContainsSet(^FBT231, '"_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ","')
    
            Data Elements:
    
            ^FBT231 = FBT:RP:BankDetails:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = IsMatch(report.FBT231, @"_|!|@|\$|%|\]|\[|\?|=|:|;|""|,", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; "" ,",
                    Location = "/tns:FBT/tns:RP/tns:BankDetails/tns:FinancialInstitutionAccountFinancialInstitutionAccountNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500650"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT231", Value = report.FBT231 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500651
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500651()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500651
            If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be “TrueAndCorrect”.
    
            Legacy Rule Format:
            (^FBT1030 = NULL) AND (^FBT185 <> 'TrueAndCorrect')

            Technical Business Rule Format:
            (^FBT1030 = NULL) AND (^FBT185 <> 'TrueAndCorrect')
    
            Data Elements:
    
            ^FBT185 = FBT:RP:Declaration:Declaration.StatementType.Code
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists == false && report.FBT185 != @"TrueAndCorrect");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438062",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement type code is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be ""TrueAndCorrect"".",
                    Location = "/tns:FBT/tns:RP/tns:Declaration/tns:StatementTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500651"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT185", Value = report.FBT185 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500652
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500652()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500652
            If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            (^FBT1030 = NULL) AND (^FBT215 <> TRUE)

            Technical Business Rule Format:
            (^FBT1030 = NULL) AND (^FBT215 <> TRUE)
    
            Data Elements:
    
            ^FBT215 = FBT:RP:Declaration:Declaration.StatementAccepted.Indicator
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists == false && report.FBT215 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438064",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:FBT/tns:RP/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500652"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT215", Value = GetValueOrEmpty(report.FBT215) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500653
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500653()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500653
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^FBT214 > Today()

            Technical Business Rule Format:
            ^FBT214 > Today()
    
            Data Elements:
    
            ^FBT214 = FBT:RP:Declaration:Declaration.Signature.Date
            */
            assertion = (report.FBT214.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:FBT/tns:RP/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500653"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT214", Value = GetValueOrEmptyDate(report.FBT214) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500654
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500654()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500654
            If an Intermediary is not present, Reporting Party declaration signature date must be supplied.
    
            Legacy Rule Format:
            (^FBT1030 = NULL) AND (^FBT214 = BLANK)

            Technical Business Rule Format:
            (^FBT1030 = NULL) AND (^FBT214 = BLANK)
    
            Data Elements:
    
            ^FBT214 = FBT:RP:Declaration:Declaration.Signature.Date
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists == false && report.FBT214 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signature date is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signature date must be supplied.",
                    Location = "/tns:FBT/tns:RP/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500654"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT214", Value = GetValueOrEmptyDate(report.FBT214) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500655
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500655()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500655
            If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.
    
            Legacy Rule Format:
            (^FBT1030 = NULL) AND (^FBT213 = BLANK)

            Technical Business Rule Format:
            (^FBT1030 = NULL) AND (^FBT213 = BLANK)
    
            Data Elements:
    
            ^FBT213 = FBT:RP:Declaration:Declaration.SignatoryIdentifier.Text
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists == false && string.IsNullOrWhiteSpace(report.FBT213) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438066",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signatory identifier is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.",
                    Location = "/tns:FBT/tns:RP/tns:Declaration/tns:SignatoryIdentifierT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500655"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT213", Value = report.FBT213 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500656
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500656()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500656
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^FBT129, ' - ')

            Technical Business Rule Format:
            Contains(^FBT129, ' - ')
    
            Data Elements:
    
            ^FBT129 = FBT:INT:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.FBT129 == null ? false : report.FBT129.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:FBT/tns:INT/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500656"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT129", Value = report.FBT129 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500657
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500657()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500657
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            (^FBT241 <> BLANK) and (FailsABNAlgorithm(^FBT241))

            Technical Business Rule Format:
            (^FBT241 <> BLANK) and (FailsABNAlgorithm(^FBT241))
    
            Data Elements:
    
            ^FBT241 = FBT:INT:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT241) != true && FailsABNAlgorithm(report.FBT241));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:FBT/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500657"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT241", Value = report.FBT241 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500658
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500658()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500658
            If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be “TrueAndCorrect”.
    
            Legacy Rule Format:
            (^FBT1030 <> NULL) AND (^FBT187 = NULL)

            Technical Business Rule Format:
            (^FBT1030 <> NULL) AND (^FBT187 = NULL)
    
            Data Elements:
    
            ^FBT187 = FBT:INT:Declaration:Declaration.StatementType.Code
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists != false && report.FBT187 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438061",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement type code is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be ""TrueAndCorrect"".",
                    Location = "/tns:FBT/tns:INT/tns:Declaration/tns:StatementTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500658"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT187", Value = report.FBT187 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500659
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500659()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500659
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^FBT218 > Today()

            Technical Business Rule Format:
            ^FBT218 > Today()
    
            Data Elements:
    
            ^FBT218 = FBT:INT:Declaration:Declaration.Signature.Date
            */
            assertion = (report.FBT218.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:FBT/tns:INT/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500659"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT218", Value = GetValueOrEmptyDate(report.FBT218) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500660
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500660()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500660
            If an Intermediary is present, Intermediary declaration signature date must be supplied.
    
            Legacy Rule Format:
            (^FBT1030 <> NULL) AND (^FBT218 = NULL)

            Technical Business Rule Format:
            (^FBT1030 <> NULL) AND (^FBT218 = NULL)
    
            Data Elements:
    
            ^FBT218 = FBT:INT:Declaration:Declaration.Signature.Date
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists != false && report.FBT218 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438067",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signature date is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration signature date must be supplied.",
                    Location = "/tns:FBT/tns:INT/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500660"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT218", Value = GetValueOrEmptyDate(report.FBT218) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500661
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500661()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500661
            If an Intermediary is present, Intermediary declaration signatory identifier must be supplied.
    
            Legacy Rule Format:
            (^FBT1030 <> NULL) AND (^FBT217 = BLANK)

            Technical Business Rule Format:
            (^FBT1030 <> NULL) AND (^FBT217 = BLANK)
    
            Data Elements:
    
            ^FBT217 = FBT:INT:Declaration:Declaration.SignatoryIdentifier.Text
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists != false && string.IsNullOrWhiteSpace(report.FBT217) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438065",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signatory identifier is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration signatory identifier must be supplied.",
                    Location = "/tns:FBT/tns:INT/tns:Declaration/tns:SignatoryIdentifierT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500661"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT217", Value = report.FBT217 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500662
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500662()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500662
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FBT41 <> BLANK) AND (StartsWithSet(^FBT41,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FBT41, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FBT41 <> BLANK) AND (StartsWithSet(^FBT41,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FBT41, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FBT41 = FBT:RP:BusinessTradingDetails:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FBT41) != true && IsMatch(report.FBT41, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.FBT41, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500662"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT41", Value = report.FBT41 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500663
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500663()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500663
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            (Contains(^FBT41, 'C/-')) AND (^FBT42 = BLANK)

            Technical Business Rule Format:
            (Contains(^FBT41, 'C/-')) AND (^FBT42 = BLANK)
    
            Data Elements:
    
            ^FBT41 = FBT:RP:BusinessTradingDetails:AddressDetails.Line1.Text
    
            ^FBT42 = FBT:RP:BusinessTradingDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.FBT41 == null ? false : report.FBT41.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.FBT42) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FBT/tns:RP/tns:BusinessTradingDetails/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500663"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT41", Value = report.FBT41 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT42", Value = GetValueOrEmpty(report.FBT42) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.500664
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT500664()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.500664
            If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            (^FBT1030 <> NULL) AND (^FBT219 <> TRUE)

            Technical Business Rule Format:
            (^FBT1030 <> NULL) AND (^FBT219 <> TRUE)
    
            Data Elements:
    
            ^FBT219 = FBT:INT:Declaration:Declaration.StatementAccepted.Indicator
    
            ^FBT1030 = FBT:INT
            */
            assertion = (report.IntCollectionExists != false && report.FBT219 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:FBT/tns:INT/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.500664"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT219", Value = GetValueOrEmpty(report.FBT219) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.FBT.536277
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFBT536277()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FBT.536277
            Year of return must be either the current or next (future) lodgment year associated with this FBT service. If a future year lodgment is required, then it must be lodged prior to the 1st of April of that future year, unless it is an amendment. After the 1st April of that future year, a return for that lodgment year must be submitted using the new FBT service associated to that specific lodgment year.
    
            Legacy Rule Format:
            (NotInSet(^FBT240,'"2026", "2027"')) OR (^FBT240 = 2027 AND Today() > AsDate('2027-03-31') AND ^FBT234 <> TRUE)

            Technical Business Rule Format:
            (NotInSet(^FBT240,'"2026", "2027"')) OR (^FBT240 = 2027 AND Today() > AsDate('2027-03-31') AND ^FBT234 <> TRUE)
    
            Data Elements:
    
            ^FBT240 = FBT:RP:Lodgment.PeriodYear.Year
    
            ^FBT234 = FBT:RP:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.FBT240.GetValueOrDefault().ToString(), @"^(2026|2027)$", RegexOptions.IgnoreCase)) || report.FBT240.GetValueOrDefault() == 2027 && DateTime.Now.Date > AsDate(@"2027-03-31") && report.FBT234 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FBT.100062",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this FBT service. If a future year lodgment is required, then it must be lodged prior to the 1st of April of that future year, unless it is an amendment. After the 1st April of the next lodgment year, a return for that lodgment year must be submitted using the new FBT service associated to that specific lodgment year.",
                    Location = "/tns:FBT/tns:RP/tns:LodgmentPeriodYearY",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FBT.536277"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT240", Value = GetValueOrEmpty(report.FBT240) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FBT234", Value = GetValueOrEmpty(report.FBT234) });
                response.Add(processMessage);
            }
        }
#endregion
    }
}