using DataContracts;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Schema;

namespace Ato.EN.IntegrationServices.CodeGenerationFBT
{
    /// <summary>
    /// XML Consumer for FBT
    /// </summary>
    public class FBT2026XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        //private int _parentCollectionCount;
        //private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);
#region Error Messages
        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }

            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

#endregion Error Messages
#region Embedded Schema
        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static FBT2026XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;
            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.fbt.0005.2025.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

#endregion Embedded Schema
#region Xml Reader Settings
        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };
        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }

            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);
        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();
            ProcessMessageDocument processMessage = null;
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);
                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);
                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;
                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;
            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9)
                return retval;
            var elementBeingProcessed = parts[1];
            var csvList = parts[5];
            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9)
                    return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }

            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(',')) ? new List<string>(csvList.Split(',')) : new List<string>
            {
                csvList
            };
            string lastElement = list.Last();
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count - 1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));
                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);
                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements)
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            int result;
            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);
            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList, string elementBeingProcessed)
        {
            return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }

            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;
            if (startDate == null || endDate == null)
                return;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

#endregion Xml Reader Settings
#region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }

            return emptyOrNil;
        }

#endregion IsEmptyOrNilElement
#region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

#endregion ToBoolean
#region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion MoveToContent
#region ReadToNextElement
        public bool ReadToNextElement(XmlReader reader, bool isMandatory)
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader)
        {
            bool retval = false;
            if ((reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();
                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);
                var element = GetElementDescriptor(xpath);
                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if (_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if (_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
            }

            //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();
            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0)
                return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }

            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }

            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf, string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();
            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if (numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if (numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }

                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }

            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader)
        {
            try
            {
                if (reader.EOF || _isExiting)
                    return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }

                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }

                return reader.NodeType == XmlNodeType.Element && !reader.EOF;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion ReadToNextElement
        private Stack<string> _currentXPath = new Stack<string>();
        public FBT2026 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            FBT2026 report = new FBT2026();
            // Working Variable for if can still read from the xml stream
            bool reading;
            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            //double currentDoubleValue;
            //float currentFloatValue;
            //sbyte currentsByteValue;
            //byte currentByteValue;
            //short currentShortValue;
            //ushort currentuShortValue;
            //uint currentuIntValue;
            int currentIntValue;
            //long currentLongValue;
            //ulong currentuLongValue;
            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:FBT");
            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;
            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);
            if (!MoveToContent(reader))
                return report;
            reading = !reader.EOF;
            if (ReadToNextElement(reader) && reader.LocalName == "FBT" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/fbt")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

#region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader, true) && reader.LocalName == "RP" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true;
#region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT3 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxFileNumberId
#region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT4 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region LodgmentPeriodYearY
                _currentXPath.Push("/tns:LodgmentPeriodYearY");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "LodgmentPeriodYearY" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.FBT240 = currentIntValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion LodgmentPeriodYearY
#region AmendmentI
                _currentXPath.Push("/tns:AmendmentI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.FBT234 = currentBooleanValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AmendmentI
#region AmendmentSequenceN
                _currentXPath.Push("/tns:AmendmentSequenceN");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentSequenceN" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.FBT235 = currentDecimalValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AmendmentSequenceN
#region AmendmentTypeC
                _currentXPath.Push("/tns:AmendmentTypeC");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentTypeC" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT236 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AmendmentTypeC
#region AmendmentReasonT
                _currentXPath.Push("/tns:AmendmentReasonT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentReasonT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT237 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AmendmentReasonT
#region CurrentTrusteeOrSeniorPartner
                _currentXPath.Push("/tns:CurrentTrusteeOrSeniorPartner");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CurrentTrusteeOrSeniorPartner" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CurrentTrusteeOrSeniorPartnerCollectionExists = true;
                    report.RP_CurrentTrusteeOrSeniorPartnerCollectionCount += 1;
                    _found = true;
#region IndividualDetail
                    _currentXPath.Push("/tns:IndividualDetail");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualDetail" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_CurrentTrusteeOrSeniorPartner_IndividualDetailCollectionExists = true;
                        report.RP_CurrentTrusteeOrSeniorPartner_IndividualDetailCollectionCount += 1;
                        _found = true;
#region PersonNameDetailsTitleT
                        _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT5 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                        _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT7 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                        _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT6 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT8 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT9 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
                    } // End of if IndividualDetail node exists

                    _currentXPath.Pop();
#endregion IndividualDetail
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT10 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
                } // End of if CurrentTrusteeOrSeniorPartner node exists

                _currentXPath.Pop();
#endregion CurrentTrusteeOrSeniorPartner
#region PreviousTrusteeOrSeniorPartner
                _currentXPath.Push("/tns:PreviousTrusteeOrSeniorPartner");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousTrusteeOrSeniorPartner" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_PreviousTrusteeOrSeniorPartnerCollectionExists = true;
                    report.RP_PreviousTrusteeOrSeniorPartnerCollectionCount += 1;
                    _found = true;
#region IndividualDetail
                    _currentXPath.Push("/tns:IndividualDetail");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualDetail" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_PreviousTrusteeOrSeniorPartner_IndividualDetailCollectionExists = true;
                        report.RP_PreviousTrusteeOrSeniorPartner_IndividualDetailCollectionCount += 1;
                        _found = true;
#region PersonNameDetailsTitleT
                        _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT49 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                        _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT51 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                        _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT50 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT52 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT53 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
                    } // End of if IndividualDetail node exists

                    _currentXPath.Pop();
#endregion IndividualDetail
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT54 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
                } // End of if PreviousTrusteeOrSeniorPartner node exists

                _currentXPath.Pop();
#endregion PreviousTrusteeOrSeniorPartner
#region CurrentEmployer
                _currentXPath.Push("/tns:CurrentEmployer");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CurrentEmployer" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CurrentEmployerCollectionExists = true;
                    report.RP_CurrentEmployerCollectionCount += 1;
                    _found = true;
#region IndividualDetail
                    _currentXPath.Push("/tns:IndividualDetail");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualDetail" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_CurrentEmployer_IndividualDetailCollectionExists = true;
                        report.RP_CurrentEmployer_IndividualDetailCollectionCount += 1;
                        _found = true;
#region PersonNameDetailsTitleT
                        _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT11 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                        _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT13 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                        _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT12 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT14 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT15 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
                    } // End of if IndividualDetail node exists

                    _currentXPath.Pop();
#endregion IndividualDetail
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT16 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AddressDetails
                    _currentXPath.Push("/tns:AddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_CurrentEmployer_AddressDetailsCollectionExists = true;
                        report.RP_CurrentEmployer_AddressDetailsCollectionCount += 1;
                        _found = true;
#region OverseasAddressI
                        _currentXPath.Push("/tns:OverseasAddressI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FBT25 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OverseasAddressI
#region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT17 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line1T
#region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT18 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line2T
#region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT21 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion LocalityNameT
#region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT23 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PostcodeT
#region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT22 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion StateOrTerritoryC
#region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT222 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion CountryC
                    } // End of if AddressDetails node exists

                    _currentXPath.Pop();
#endregion AddressDetails
                } // End of if CurrentEmployer node exists

                _currentXPath.Pop();
#endregion CurrentEmployer
#region PreviousEmployer
                _currentXPath.Push("/tns:PreviousEmployer");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousEmployer" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_PreviousEmployerCollectionExists = true;
                    report.RP_PreviousEmployerCollectionCount += 1;
                    _found = true;
#region IndividualDetail
                    _currentXPath.Push("/tns:IndividualDetail");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualDetail" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_PreviousEmployer_IndividualDetailCollectionExists = true;
                        report.RP_PreviousEmployer_IndividualDetailCollectionCount += 1;
                        _found = true;
#region PersonNameDetailsTitleT
                        _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT26 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                        _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT28 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                        _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT27 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT29 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT30 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
                    } // End of if IndividualDetail node exists

                    _currentXPath.Pop();
#endregion IndividualDetail
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT31 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AddressDetails
                    _currentXPath.Push("/tns:AddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_PreviousEmployer_AddressDetailsCollectionExists = true;
                        report.RP_PreviousEmployer_AddressDetailsCollectionCount += 1;
                        _found = true;
#region OverseasAddressI
                        _currentXPath.Push("/tns:OverseasAddressI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FBT159 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OverseasAddressI
#region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT32 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line1T
#region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT33 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line2T
#region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT36 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion LocalityNameT
#region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT38 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PostcodeT
#region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT37 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion StateOrTerritoryC
#region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FBT223 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion CountryC
                    } // End of if AddressDetails node exists

                    _currentXPath.Pop();
#endregion AddressDetails
                } // End of if PreviousEmployer node exists

                _currentXPath.Pop();
#endregion PreviousEmployer
#region BusinessTradingDetails
                _currentXPath.Push("/tns:BusinessTradingDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessTradingDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_BusinessTradingDetailsCollectionExists = true;
                    report.RP_BusinessTradingDetailsCollectionCount += 1;
                    _found = true;
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT40 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AddressDetailsOverseasAddressI
                    _currentXPath.Push("/tns:AddressDetailsOverseasAddressI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsOverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FBT164 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsOverseasAddressI
#region AddressDetailsLine1T
                    _currentXPath.Push("/tns:AddressDetailsLine1T");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsLine1T" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT41 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsLine1T
#region AddressDetailsLine2T
                    _currentXPath.Push("/tns:AddressDetailsLine2T");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsLine2T" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT42 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsLine2T
#region AddressDetailsLocalityNameT
                    _currentXPath.Push("/tns:AddressDetailsLocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsLocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT45 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsLocalityNameT
#region AddressDetailsPostcodeT
                    _currentXPath.Push("/tns:AddressDetailsPostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsPostcodeT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT47 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsPostcodeT
#region AddressDetailsStateOrTerritoryC
                    _currentXPath.Push("/tns:AddressDetailsStateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsStateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT46 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsStateOrTerritoryC
#region AddressDetailsCountryC
                    _currentXPath.Push("/tns:AddressDetailsCountryC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT224 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsCountryC
                } // End of if BusinessTradingDetails node exists

                _currentXPath.Pop();
#endregion BusinessTradingDetails
#region ContactDetails
                _currentXPath.Push("/tns:ContactDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ContactDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ContactDetailsCollectionExists = true;
                    report.RP_ContactDetailsCollectionCount += 1;
                    _found = true;
#region PersonNameDetailsTitleT
                    _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT55 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                    _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT57 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                    _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT56 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                    _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT58 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                    _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT59 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
#region ElectronicContactTelephoneAreaC
                    _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactTelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT60 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactTelephoneAreaC
#region ElectronicContactTelephoneMinimalN
                    _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT61 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactTelephoneMinimalN
#region ElectronicContactElectronicMailAddressT
                    _currentXPath.Push("/tns:ElectronicContactElectronicMailAddressT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactElectronicMailAddressT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT62 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactElectronicMailAddressT
                } // End of if ContactDetails node exists

                _currentXPath.Pop();
#endregion ContactDetails
#region RemunerationFringeBenefitsRecipientsCt
                _currentXPath.Push("/tns:RemunerationFringeBenefitsRecipientsCt");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsRecipientsCt" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.FBT63 = currentDecimalValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion RemunerationFringeBenefitsRecipientsCt
#region CompletionHoursN
                _currentXPath.Push("/tns:CompletionHoursN");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CompletionHoursN" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.FBT64 = currentDecimalValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion CompletionHoursN
#region LodgmentFutureReturnI
                _currentXPath.Push("/tns:LodgmentFutureReturnI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "LodgmentFutureReturnI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.FBT238 = currentBooleanValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion LodgmentFutureReturnI
#region BankDetails
                _currentXPath.Push("/tns:BankDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BankDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_BankDetailsCollectionExists = true;
                    report.RP_BankDetailsCollectionCount += 1;
                    _found = true;
#region FinancialInstitutionAccountBankStateBranchN
                    _currentXPath.Push("/tns:FinancialInstitutionAccountBankStateBranchN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialInstitutionAccountBankStateBranchN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT229 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FinancialInstitutionAccountBankStateBranchN
#region FinancialInstitutionAccountFinancialInstitutionAccountN
                    _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT230 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FinancialInstitutionAccountFinancialInstitutionAccountN
#region FinancialInstitutionAccountFinancialInstitutionAccountNameT
                    _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT231 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FinancialInstitutionAccountFinancialInstitutionAccountNameT
                } // End of if BankDetails node exists

                _currentXPath.Pop();
#endregion BankDetails
#region ReturnCalculationDetails
                _currentXPath.Push("/tns:ReturnCalculationDetails");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ReturnCalculationDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ReturnCalculationDetailsCollectionExists = true;
                    report.RP_ReturnCalculationDetailsCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsType1AggregateA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsType1AggregateA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsType1AggregateA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT66 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsType1AggregateA
#region FringeBenefitsTaxType1AggregateA
                    _currentXPath.Push("/tns:FringeBenefitsTaxType1AggregateA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxType1AggregateA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT67 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxType1AggregateA
#region RemunerationFringeBenefitsType2AggregateA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsType2AggregateA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsType2AggregateA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT68 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsType2AggregateA
#region FringeBenefitsTaxType2AggregateA
                    _currentXPath.Push("/tns:FringeBenefitsTaxType2AggregateA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxType2AggregateA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT69 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxType2AggregateA
#region RemunerationFringeBenefitsAggregateNonExemptA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsAggregateNonExemptA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsAggregateNonExemptA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT70 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsAggregateNonExemptA
#region RemunerationFringeBenefitsTaxableTotalA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsTaxableTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsTaxableTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT71 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsTaxableTotalA
#region FringeBenefitsTaxPayableA
                    _currentXPath.Push("/tns:FringeBenefitsTaxPayableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FringeBenefitsTaxPayableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT72 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxPayableA
#region FringeBenefitsTaxNonRebatableAggregateA
                    _currentXPath.Push("/tns:FringeBenefitsTaxNonRebatableAggregateA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxNonRebatableAggregateA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT73 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxNonRebatableAggregateA
#region FringeBenefitsTaxRebateA
                    _currentXPath.Push("/tns:FringeBenefitsTaxRebateA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxRebateA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT74 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxRebateA
#region FringeBenefitsTaxLiabilityA
                    _currentXPath.Push("/tns:FringeBenefitsTaxLiabilityA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FringeBenefitsTaxLiabilityA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT75 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxLiabilityA
#region FringeBenefitsTaxLiabilityInstalmentsTotalA
                    _currentXPath.Push("/tns:FringeBenefitsTaxLiabilityInstalmentsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxLiabilityInstalmentsTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT76 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxLiabilityInstalmentsTotalA
#region FringeBenefitsTaxAdjustmentDueA
                    _currentXPath.Push("/tns:FringeBenefitsTaxAdjustmentDueA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FringeBenefitsTaxAdjustmentDueA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT77 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FringeBenefitsTaxAdjustmentDueA
                } // End of if ReturnCalculationDetails node exists

                _currentXPath.Pop();
#endregion ReturnCalculationDetails
#region CarStatutoryFormula
                _currentXPath.Push("/tns:CarStatutoryFormula");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CarStatutoryFormula" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CarStatutoryFormulaCollectionExists = true;
                    report.RP_CarStatutoryFormulaCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsItemsProvidedN
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsItemsProvidedN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsItemsProvidedN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT78 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsItemsProvidedN
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT79 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT80 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT81 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if CarStatutoryFormula node exists

                _currentXPath.Pop();
#endregion CarStatutoryFormula
#region CarOperatingCostMethod
                _currentXPath.Push("/tns:CarOperatingCostMethod");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CarOperatingCostMethod" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CarOperatingCostMethodCollectionExists = true;
                    report.RP_CarOperatingCostMethodCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsItemsProvidedN
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsItemsProvidedN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsItemsProvidedN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT82 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsItemsProvidedN
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT83 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT84 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT85 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if CarOperatingCostMethod node exists

                _currentXPath.Pop();
#endregion CarOperatingCostMethod
#region LoansGranted
                _currentXPath.Push("/tns:LoansGranted");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LoansGranted" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_LoansGrantedCollectionExists = true;
                    report.RP_LoansGrantedCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsItemsProvidedN
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsItemsProvidedN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsItemsProvidedN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT86 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsItemsProvidedN
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT87 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsReductionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsReductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT88 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsReductionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT89 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if LoansGranted node exists

                _currentXPath.Pop();
#endregion LoansGranted
#region DebtWaiver
                _currentXPath.Push("/tns:DebtWaiver");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "DebtWaiver" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_DebtWaiverCollectionExists = true;
                    report.RP_DebtWaiverCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT90 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT91 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if DebtWaiver node exists

                _currentXPath.Pop();
#endregion DebtWaiver
#region ExpensePayments
                _currentXPath.Push("/tns:ExpensePayments");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ExpensePayments" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ExpensePaymentsCollectionExists = true;
                    report.RP_ExpensePaymentsCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT92 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT93 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsReductionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsReductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT94 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsReductionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT95 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if ExpensePayments node exists

                _currentXPath.Pop();
#endregion ExpensePayments
#region Housing
                _currentXPath.Push("/tns:Housing");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Housing" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_HousingCollectionExists = true;
                    report.RP_HousingCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsItemsProvidedN
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsItemsProvidedN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsItemsProvidedN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT96 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsItemsProvidedN
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT97 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT98 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT99 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if Housing node exists

                _currentXPath.Pop();
#endregion Housing
#region LAFHA
                _currentXPath.Push("/tns:LAFHA");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LAFHA" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_LAFHACollectionExists = true;
                    report.RP_LAFHACollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsItemsProvidedN
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsItemsProvidedN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsItemsProvidedN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT100 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsItemsProvidedN
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT101 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsReductionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsReductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT102 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsReductionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT103 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if LAFHA node exists

                _currentXPath.Pop();
#endregion LAFHA
#region Board
                _currentXPath.Push("/tns:Board");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Board" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_BoardCollectionExists = true;
                    report.RP_BoardCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT108 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT109 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsReductionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsReductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT110 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsReductionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT111 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if Board node exists

                _currentXPath.Pop();
#endregion Board
#region Property
                _currentXPath.Push("/tns:Property");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Property" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_PropertyCollectionExists = true;
                    report.RP_PropertyCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT112 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT113 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsReductionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsReductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT114 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsReductionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT115 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if Property node exists

                _currentXPath.Pop();
#endregion Property
#region ITExemptBodyEntertainment
                _currentXPath.Push("/tns:ITExemptBodyEntertainment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ITExemptBodyEntertainment" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ITExemptBodyEntertainmentCollectionExists = true;
                    report.RP_ITExemptBodyEntertainmentCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT116 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT117 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if ITExemptBodyEntertainment node exists

                _currentXPath.Pop();
#endregion ITExemptBodyEntertainment
#region OtherBenefits
                _currentXPath.Push("/tns:OtherBenefits");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherBenefits" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_OtherBenefitsCollectionExists = true;
                    report.RP_OtherBenefitsCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT118 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT119 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsReductionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsReductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT120 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsReductionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT121 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if OtherBenefits node exists

                _currentXPath.Pop();
#endregion OtherBenefits
#region CarParking
                _currentXPath.Push("/tns:CarParking");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CarParking" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CarParkingCollectionExists = true;
                    report.RP_CarParkingCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT122 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT123 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT124 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if CarParking node exists

                _currentXPath.Pop();
#endregion CarParking
#region MealEntertainment
                _currentXPath.Push("/tns:MealEntertainment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MealEntertainment" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_MealEntertainmentCollectionExists = true;
                    report.RP_MealEntertainmentCollectionCount += 1;
                    _found = true;
#region RemunerationFringeBenefitsBenefitGrossTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitGrossTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsBenefitGrossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT125 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitGrossTaxableA
#region RemunerationFringeBenefitsBenefitNetTaxableA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsBenefitNetTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationFringeBenefitsBenefitNetTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FBT126 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationFringeBenefitsBenefitNetTaxableA
                } // End of if MealEntertainment node exists

                _currentXPath.Pop();
#endregion MealEntertainment
#region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_DeclarationCollectionExists = true;
                    report.RP_DeclarationCollectionCount += 1;
                    _found = true;
#region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT185 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementTypeC
#region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FBT215 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementAcceptedI
#region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FBT214 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatureD
#region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT213 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatoryIdentifierT
                } // End of if Declaration node exists

                _currentXPath.Pop();
#endregion Declaration
#region RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
                _currentXPath.Push("/tns:RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT239 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
            } // End of if RP node exists

            _currentXPath.Pop();
#endregion RP
#region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader, false) && reader.LocalName == "INT" && reader.Depth == _currentXPath.Count - 1)
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true;
#region PersonUnstructuredNameFullNameT
                _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT129 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PersonUnstructuredNameFullNameT
#region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT241 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region ElectronicContactTelephoneAreaC
                _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactTelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT127 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion ElectronicContactTelephoneAreaC
#region ElectronicContactTelephoneMinimalN
                _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT128 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion ElectronicContactTelephoneMinimalN
#region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT2 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxAgentNumberId
#region TaxAgentClientReferenceT
                _currentXPath.Push("/tns:TaxAgentClientReferenceT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentClientReferenceT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FBT130 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxAgentClientReferenceT
#region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.InT_DeclarationCollectionExists = true;
                    report.InT_DeclarationCollectionCount += 1;
                    _found = true;
#region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT187 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementTypeC
#region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FBT219 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementAcceptedI
#region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FBT218 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatureD
#region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FBT217 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatoryIdentifierT
                } // End of if Declaration node exists

                _currentXPath.Pop();
#endregion Declaration
            } // End of if INT node exists

            _currentXPath.Pop();
#endregion INT
            while (reader.EOF != true)
                reader.Read();
            return report;
        }

        private string _elementCsvList = "RP#TaxFileNumberId,RP#AustralianBusinessNumberId,RP#LodgmentPeriodYearY,RP#AmendmentI,RP#AmendmentSequenceN,RP#AmendmentTypeC,RP#AmendmentReasonT,IndividualDetail#PersonNameDetailsTitleT,IndividualDetail#PersonNameDetailsNameSuffixT,IndividualDetail#PersonNameDetailsFamilyNameT,IndividualDetail#PersonNameDetailsGivenNameT,IndividualDetail#PersonNameDetailsOtherGivenNameT,IndividualDetail,CurrentTrusteeOrSeniorPartner#OrganisationNameDetailsOrganisationalNameT,CurrentTrusteeOrSeniorPartner,IndividualDetail#PersonNameDetailsTitleT,IndividualDetail#PersonNameDetailsNameSuffixT,IndividualDetail#PersonNameDetailsFamilyNameT,IndividualDetail#PersonNameDetailsGivenNameT,IndividualDetail#PersonNameDetailsOtherGivenNameT,IndividualDetail,PreviousTrusteeOrSeniorPartner#OrganisationNameDetailsOrganisationalNameT,PreviousTrusteeOrSeniorPartner,IndividualDetail#PersonNameDetailsTitleT,IndividualDetail#PersonNameDetailsNameSuffixT,IndividualDetail#PersonNameDetailsFamilyNameT,IndividualDetail#PersonNameDetailsGivenNameT,IndividualDetail#PersonNameDetailsOtherGivenNameT,IndividualDetail,CurrentEmployer#OrganisationNameDetailsOrganisationalNameT,AddressDetails#OverseasAddressI,AddressDetails#Line1T,AddressDetails#Line2T,AddressDetails#LocalityNameT,AddressDetails#PostcodeT,AddressDetails#StateOrTerritoryC,AddressDetails#CountryC,AddressDetails,CurrentEmployer,IndividualDetail#PersonNameDetailsTitleT,IndividualDetail#PersonNameDetailsNameSuffixT,IndividualDetail#PersonNameDetailsFamilyNameT,IndividualDetail#PersonNameDetailsGivenNameT,IndividualDetail#PersonNameDetailsOtherGivenNameT,IndividualDetail,PreviousEmployer#OrganisationNameDetailsOrganisationalNameT,AddressDetails#OverseasAddressI,AddressDetails#Line1T,AddressDetails#Line2T,AddressDetails#LocalityNameT,AddressDetails#PostcodeT,AddressDetails#StateOrTerritoryC,AddressDetails#CountryC,AddressDetails,PreviousEmployer,BusinessTradingDetails#OrganisationNameDetailsOrganisationalNameT,BusinessTradingDetails#AddressDetailsOverseasAddressI,BusinessTradingDetails#AddressDetailsLine1T,BusinessTradingDetails#AddressDetailsLine2T,BusinessTradingDetails#AddressDetailsLocalityNameT,BusinessTradingDetails#AddressDetailsPostcodeT,BusinessTradingDetails#AddressDetailsStateOrTerritoryC,BusinessTradingDetails#AddressDetailsCountryC,BusinessTradingDetails,ContactDetails#PersonNameDetailsTitleT,ContactDetails#PersonNameDetailsNameSuffixT,ContactDetails#PersonNameDetailsFamilyNameT,ContactDetails#PersonNameDetailsGivenNameT,ContactDetails#PersonNameDetailsOtherGivenNameT,ContactDetails#ElectronicContactTelephoneAreaC,ContactDetails#ElectronicContactTelephoneMinimalN,ContactDetails#ElectronicContactElectronicMailAddressT,ContactDetails,RP#RemunerationFringeBenefitsRecipientsCt,RP#CompletionHoursN,RP#LodgmentFutureReturnI,BankDetails#FinancialInstitutionAccountBankStateBranchN,BankDetails#FinancialInstitutionAccountFinancialInstitutionAccountN,BankDetails#FinancialInstitutionAccountFinancialInstitutionAccountNameT,BankDetails,ReturnCalculationDetails#RemunerationFringeBenefitsType1AggregateA,ReturnCalculationDetails#FringeBenefitsTaxType1AggregateA,ReturnCalculationDetails#RemunerationFringeBenefitsType2AggregateA,ReturnCalculationDetails#FringeBenefitsTaxType2AggregateA,ReturnCalculationDetails#RemunerationFringeBenefitsAggregateNonExemptA,ReturnCalculationDetails#RemunerationFringeBenefitsTaxableTotalA,ReturnCalculationDetails#FringeBenefitsTaxPayableA,ReturnCalculationDetails#FringeBenefitsTaxNonRebatableAggregateA,ReturnCalculationDetails#FringeBenefitsTaxRebateA,ReturnCalculationDetails#FringeBenefitsTaxLiabilityA,ReturnCalculationDetails#FringeBenefitsTaxLiabilityInstalmentsTotalA,ReturnCalculationDetails#FringeBenefitsTaxAdjustmentDueA,ReturnCalculationDetails,CarStatutoryFormula#RemunerationFringeBenefitsItemsProvidedN,CarStatutoryFormula#RemunerationFringeBenefitsBenefitGrossTaxableA,CarStatutoryFormula#RemunerationFringeBenefitsEmployeeContributionA,CarStatutoryFormula#RemunerationFringeBenefitsBenefitNetTaxableA,CarStatutoryFormula,CarOperatingCostMethod#RemunerationFringeBenefitsItemsProvidedN,CarOperatingCostMethod#RemunerationFringeBenefitsBenefitGrossTaxableA,CarOperatingCostMethod#RemunerationFringeBenefitsEmployeeContributionA,CarOperatingCostMethod#RemunerationFringeBenefitsBenefitNetTaxableA,CarOperatingCostMethod,LoansGranted#RemunerationFringeBenefitsItemsProvidedN,LoansGranted#RemunerationFringeBenefitsBenefitGrossTaxableA,LoansGranted#RemunerationFringeBenefitsReductionA,LoansGranted#RemunerationFringeBenefitsBenefitNetTaxableA,LoansGranted,DebtWaiver#RemunerationFringeBenefitsBenefitGrossTaxableA,DebtWaiver#RemunerationFringeBenefitsBenefitNetTaxableA,DebtWaiver,ExpensePayments#RemunerationFringeBenefitsBenefitGrossTaxableA,ExpensePayments#RemunerationFringeBenefitsEmployeeContributionA,ExpensePayments#RemunerationFringeBenefitsReductionA,ExpensePayments#RemunerationFringeBenefitsBenefitNetTaxableA,ExpensePayments,Housing#RemunerationFringeBenefitsItemsProvidedN,Housing#RemunerationFringeBenefitsBenefitGrossTaxableA,Housing#RemunerationFringeBenefitsEmployeeContributionA,Housing#RemunerationFringeBenefitsBenefitNetTaxableA,Housing,LAFHA#RemunerationFringeBenefitsItemsProvidedN,LAFHA#RemunerationFringeBenefitsBenefitGrossTaxableA,LAFHA#RemunerationFringeBenefitsReductionA,LAFHA#RemunerationFringeBenefitsBenefitNetTaxableA,LAFHA,Board#RemunerationFringeBenefitsBenefitGrossTaxableA,Board#RemunerationFringeBenefitsEmployeeContributionA,Board#RemunerationFringeBenefitsReductionA,Board#RemunerationFringeBenefitsBenefitNetTaxableA,Board,Property#RemunerationFringeBenefitsBenefitGrossTaxableA,Property#RemunerationFringeBenefitsEmployeeContributionA,Property#RemunerationFringeBenefitsReductionA,Property#RemunerationFringeBenefitsBenefitNetTaxableA,Property,ITExemptBodyEntertainment#RemunerationFringeBenefitsBenefitGrossTaxableA,ITExemptBodyEntertainment#RemunerationFringeBenefitsBenefitNetTaxableA,ITExemptBodyEntertainment,OtherBenefits#RemunerationFringeBenefitsBenefitGrossTaxableA,OtherBenefits#RemunerationFringeBenefitsEmployeeContributionA,OtherBenefits#RemunerationFringeBenefitsReductionA,OtherBenefits#RemunerationFringeBenefitsBenefitNetTaxableA,OtherBenefits,CarParking#RemunerationFringeBenefitsBenefitGrossTaxableA,CarParking#RemunerationFringeBenefitsEmployeeContributionA,CarParking#RemunerationFringeBenefitsBenefitNetTaxableA,CarParking,MealEntertainment#RemunerationFringeBenefitsBenefitGrossTaxableA,MealEntertainment#RemunerationFringeBenefitsBenefitNetTaxableA,MealEntertainment,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatureD,Declaration#SignatoryIdentifierT,Declaration,RP#RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT,RP,INT#PersonUnstructuredNameFullNameT,INT#AustralianBusinessNumberId,INT#ElectronicContactTelephoneAreaC,INT#ElectronicContactTelephoneMinimalN,INT#TaxAgentNumberId,INT#TaxAgentClientReferenceT,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatureD,Declaration#SignatoryIdentifierT,Declaration,INT";
    }
}