using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationFVSGet
{
    /// <summary>
    /// XML Consumer for FVSGetResponse
    /// </summary>
    public class FVSGetResponse2026XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static FVSGetResponse2026XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.fvs.0003.2026.get.response.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;
  
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();        
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count-1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements) 
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
           return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory) 
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader) 
        {
            bool retval = false;

            if ( (reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if(_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if(_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
             }

             //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf , string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if(numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if(numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader) 
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }
      
                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public FVSGetResponse2026 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            FVSGetResponse2026 report = new FVSGetResponse2026();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:FVSGetResponse");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "FVSGetResponse" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/fvsgetresponse")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

    
            #region FundDetails
            _currentXPath.Push("/tns:FundDetails");
            //3. use case
            if (ReadToNextElement(reader,false) && reader.LocalName == "FundDetails" && reader.Depth == _currentXPath.Count - 1)
            {
                report.FundDetailsCollectionExists = true;
                report.FundDetailsCollectionCount += 1;
                _found = true; 
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FVS15 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region OrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FVS16 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationNameDetailsOrganisationalNameT
        
                #region ProductUSIInformation
        
                _currentXPath.Push("/tns:ProductUSIInformationCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ProductUSIInformationCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.FundDetails_ProductUSIInformationCollection = new List<FVSGetResponse2026.FundDetails_ProductUSIInformation>();
                    report.FundDetails_ProductUSIInformationCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "ProductUSIInformation" ) 
                    {
                        _found = true;
        
                        FVSGetResponse2026.FundDetails_ProductUSIInformation productUSIInformation = new FVSGetResponse2026.FundDetails_ProductUSIInformation();
                        report.FundDetails_ProductUSIInformationCollection.Add(productUSIInformation);
                        report.FundDetails_ProductUSIInformationCollectionCount += 1;
                        productUSIInformation.OccurrenceIndex = report.FundDetails_ProductUSIInformationCollectionCount;
        
                        _currentXPath.Push("/tns:ProductUSIInformation[" + report.FundDetails_ProductUSIInformationCollectionCount + "]");
                
                        #region PeriodStartD
                        _currentXPath.Push("/tns:PeriodStartD");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        productUSIInformation.FVS12 = currentDateTimeValue.Date.ToString("yyyy-MM-dd");
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PeriodStartD
                
                        #region PeriodEndD
                        _currentXPath.Push("/tns:PeriodEndD");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        productUSIInformation.FVS13 = currentDateTimeValue.Date.ToString("yyyy-MM-dd");
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PeriodEndD
                
                        #region SuperannuationFundDetailsUniqueSuperannuationIdentifierId
                        _currentXPath.Push("/tns:SuperannuationFundDetailsUniqueSuperannuationIdentifierId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationFundDetailsUniqueSuperannuationIdentifierId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    productUSIInformation.FVS6 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SuperannuationFundDetailsUniqueSuperannuationIdentifierId
                
                        #region SuperannuationProductDetailsEmployerContributionRestrictionI
                        _currentXPath.Push("/tns:SuperannuationProductDetailsEmployerContributionRestrictionI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationProductDetailsEmployerContributionRestrictionI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        productUSIInformation.FVS10 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SuperannuationProductDetailsEmployerContributionRestrictionI
                
                        #region SuperannuationProductDetailsNameT
                        _currentXPath.Push("/tns:SuperannuationProductDetailsNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationProductDetailsNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    productUSIInformation.FVS11 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SuperannuationProductDetailsNameT
                
                        #region ElectronicContactWebServicesConformanceToStandardT
                        _currentXPath.Push("/tns:ElectronicContactWebServicesConformanceToStandardT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactWebServicesConformanceToStandardT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    productUSIInformation.FVS17 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ElectronicContactWebServicesConformanceToStandardT
                
                        #region SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC
                        _currentXPath.Push("/tns:SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    productUSIInformation.FVS20 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC
                
                        #region NewFundDetails
                        _currentXPath.Push("/tns:NewFundDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NewFundDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            productUSIInformation.FundDetails_ProductUSIInformation_NewFundDetailsCollectionExists = true;
                            productUSIInformation.FundDetails_ProductUSIInformation_NewFundDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region AustralianBusinessNumberId
                            _currentXPath.Push("/tns:AustralianBusinessNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS21 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AustralianBusinessNumberId
                    
                            #region SuperannuationFundDetailsUniqueSuperannuationIdentifierId
                            _currentXPath.Push("/tns:SuperannuationFundDetailsUniqueSuperannuationIdentifierId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationFundDetailsUniqueSuperannuationIdentifierId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS22 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion SuperannuationFundDetailsUniqueSuperannuationIdentifierId
                        } // End of if NewFundDetails node exists
                
                        _currentXPath.Pop();
                        #endregion NewFundDetails
                
                        #region ContributionDetails
                        _currentXPath.Push("/tns:ContributionDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ContributionDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            productUSIInformation.FundDetails_ProductUSIInformation_ContributionDetailsCollectionExists = true;
                            productUSIInformation.FundDetails_ProductUSIInformation_ContributionDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region ElectronicContactElectronicServiceAddressT
                            _currentXPath.Push("/tns:ElectronicContactElectronicServiceAddressT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactElectronicServiceAddressT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS30 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ElectronicContactElectronicServiceAddressT
                    
                            #region SuperannuationPaymentNewPaymentPlatformI
                            _currentXPath.Push("/tns:SuperannuationPaymentNewPaymentPlatformI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationPaymentNewPaymentPlatformI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            productUSIInformation.FVS35 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion SuperannuationPaymentNewPaymentPlatformI
                    
                            #region PaymentMechanismPayIDId
                            _currentXPath.Push("/tns:PaymentMechanismPayIDId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PaymentMechanismPayIDId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS36 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PaymentMechanismPayIDId
                    
                            #region FinancialInstitutionAccountBankStateBranchN
                            _currentXPath.Push("/tns:FinancialInstitutionAccountBankStateBranchN");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "FinancialInstitutionAccountBankStateBranchN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS31 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion FinancialInstitutionAccountBankStateBranchN
                    
                            #region FinancialInstitutionAccountFinancialInstitutionAccountN
                            _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountN");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS32 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion FinancialInstitutionAccountFinancialInstitutionAccountN
                    
                            #region FinancialInstitutionAccountFinancialInstitutionAccountNameT
                            _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS33 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion FinancialInstitutionAccountFinancialInstitutionAccountNameT
                    
                            #region PaymentMechanismBpayBillerC
                            _currentXPath.Push("/tns:PaymentMechanismBpayBillerC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PaymentMechanismBpayBillerC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS34 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PaymentMechanismBpayBillerC
                        } // End of if ContributionDetails node exists
                
                        _currentXPath.Pop();
                        #endregion ContributionDetails
                
                        #region RolloverDetails
                        _currentXPath.Push("/tns:RolloverDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "RolloverDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            productUSIInformation.FundDetails_ProductUSIInformation_RolloverDetailsCollectionExists = true;
                            productUSIInformation.FundDetails_ProductUSIInformation_RolloverDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region ElectronicContactElectronicServiceAddressT
                            _currentXPath.Push("/tns:ElectronicContactElectronicServiceAddressT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactElectronicServiceAddressT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS40 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ElectronicContactElectronicServiceAddressT
                    
                            #region SuperannuationPaymentNewPaymentPlatformI
                            _currentXPath.Push("/tns:SuperannuationPaymentNewPaymentPlatformI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationPaymentNewPaymentPlatformI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            productUSIInformation.FVS44 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion SuperannuationPaymentNewPaymentPlatformI
                    
                            #region PaymentMechanismPayIDId
                            _currentXPath.Push("/tns:PaymentMechanismPayIDId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PaymentMechanismPayIDId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS45 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PaymentMechanismPayIDId
                    
                            #region FinancialInstitutionAccountBankStateBranchN
                            _currentXPath.Push("/tns:FinancialInstitutionAccountBankStateBranchN");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "FinancialInstitutionAccountBankStateBranchN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS41 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion FinancialInstitutionAccountBankStateBranchN
                    
                            #region FinancialInstitutionAccountFinancialInstitutionAccountN
                            _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountN");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS42 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion FinancialInstitutionAccountFinancialInstitutionAccountN
                    
                            #region FinancialInstitutionAccountFinancialInstitutionAccountNameT
                            _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS43 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion FinancialInstitutionAccountFinancialInstitutionAccountNameT
                        } // End of if RolloverDetails node exists
                
                        _currentXPath.Pop();
                        #endregion RolloverDetails
                
                        #region TechnicalAssistanceContactDetails
                        _currentXPath.Push("/tns:TechnicalAssistanceContactDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TechnicalAssistanceContactDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            productUSIInformation.FundDetails_ProductUSIInformation_TechnicalAssistanceContactDetailsCollectionExists = true;
                            productUSIInformation.FundDetails_ProductUSIInformation_TechnicalAssistanceContactDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region PersonUnstructuredNameFullNameT
                            _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS49 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonUnstructuredNameFullNameT
                    
                            #region ElectronicContactTelephoneMinimalN
                            _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS50 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ElectronicContactTelephoneMinimalN
                    
                            #region ElectronicContactElectronicMailAddressT
                            _currentXPath.Push("/tns:ElectronicContactElectronicMailAddressT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactElectronicMailAddressT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS51 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ElectronicContactElectronicMailAddressT
                    
                            #region ElectronicContactWebUniformResourceLocatorT
                            _currentXPath.Push("/tns:ElectronicContactWebUniformResourceLocatorT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactWebUniformResourceLocatorT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        productUSIInformation.FVS52 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ElectronicContactWebUniformResourceLocatorT
                        } // End of if TechnicalAssistanceContactDetails node exists
                
                        _currentXPath.Pop();
                        #endregion TechnicalAssistanceContactDetails
                    _currentXPath.Pop();
                    } // End of while loop for ProductUSIInformation
                } // End of if ProductUSIInformationCollection node exists
        
                _currentXPath.Pop();
                #endregion ProductUSIInformation
            } // End of if FundDetails node exists
    
            _currentXPath.Pop();
            #endregion FundDetails

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "FundDetails#AustralianBusinessNumberId,FundDetails#OrganisationNameDetailsOrganisationalNameT,ProductUSIInformation#PeriodStartD,ProductUSIInformation#PeriodEndD,ProductUSIInformation#SuperannuationFundDetailsUniqueSuperannuationIdentifierId,ProductUSIInformation#SuperannuationProductDetailsEmployerContributionRestrictionI,ProductUSIInformation#SuperannuationProductDetailsNameT,ProductUSIInformation#ElectronicContactWebServicesConformanceToStandardT,ProductUSIInformation#SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC,ProductUSIInformation#AustralianBusinessNumberId,ProductUSIInformation#SuperannuationFundDetailsUniqueSuperannuationIdentifierId,NewFundDetails,ProductUSIInformation#ElectronicContactElectronicServiceAddressT,ProductUSIInformation#SuperannuationPaymentNewPaymentPlatformI,ProductUSIInformation#PaymentMechanismPayIDId,ProductUSIInformation#FinancialInstitutionAccountBankStateBranchN,ProductUSIInformation#FinancialInstitutionAccountFinancialInstitutionAccountN,ProductUSIInformation#FinancialInstitutionAccountFinancialInstitutionAccountNameT,ProductUSIInformation#PaymentMechanismBpayBillerC,ContributionDetails,ProductUSIInformation#ElectronicContactElectronicServiceAddressT,ProductUSIInformation#SuperannuationPaymentNewPaymentPlatformI,ProductUSIInformation#PaymentMechanismPayIDId,ProductUSIInformation#FinancialInstitutionAccountBankStateBranchN,ProductUSIInformation#FinancialInstitutionAccountFinancialInstitutionAccountN,ProductUSIInformation#FinancialInstitutionAccountFinancialInstitutionAccountNameT,RolloverDetails,ProductUSIInformation#PersonUnstructuredNameFullNameT,ProductUSIInformation#ElectronicContactTelephoneMinimalN,ProductUSIInformation#ElectronicContactElectronicMailAddressT,ProductUSIInformation#ElectronicContactWebUniformResourceLocatorT,TechnicalAssistanceContactDetails,ProductUSIInformation,ProductUSIInformationCollection,FundDetails";

    }
}
