using System;
using System.Runtime.CompilerServices;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationFVSSubmit
{
    public partial class FVSSubmitRequest2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;
        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;
        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;
        /// <summary>
        /// The report parameter
        /// </summary>
        private FVSSubmitRequest2026 report;
        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref = "FVSSubmitRequest2026Validator"/> class.
        /// </summary>
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FVSSubmitRequest2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref = "FVSSubmitRequest2026Validator"/> class.
        /// </summary>
        /// <param name = "report">Report variable </param>                
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FVSSubmitRequest2026Validator(FVSSubmitRequest2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;
            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }

        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        public static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        public static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        public static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        public static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        public static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }

            if (posB == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }

            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }

            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            return (
                from x in nodes
                select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            nodes = nodes.Where(x => x != null).ToArray();
            return !((
                from x in nodes
                select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values)
            where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;
            if (year == null || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;
            if (year == 0 || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }

        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;
        }

        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }

        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }

            return date.Value.Month;
        }

        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }

        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }

        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;
                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }

            return response;
        }

        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;
            response = (field < (expression - range)) || (field > (expression + range));
            return response;
        }

        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;
                }
                else if (usi.Length == 9)
                {
                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }

            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;
            if (tan == null)
                return false;
            tan = tan.Trim();
            if (!decimal.TryParse(tan, out decimalTan))
                return true;
            if (tan.Length != 8)
                return true;
            decimal tanSum = 7 * int.Parse(tan.Substring(0, 1)) + 9 * int.Parse(tan.Substring(1, 1)) + 8 * int.Parse(tan.Substring(2, 1)) + 4 * int.Parse(tan.Substring(3, 1)) + 6 * int.Parse(tan.Substring(4, 1)) + 3 * int.Parse(tan.Substring(5, 1)) + 5 * int.Parse(tan.Substring(6, 1)) + 1 * int.Parse(tan.Substring(7, 1));
            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;
            if (abn == null)
                return false;
            abn = abn.Trim();
            if (!decimal.TryParse(abn, out decimalAbn))
                return true;
            if (abn.Length != 11)
                return true;
            decimal abnSum = 10 * (int.Parse(abn.Substring(0, 1)) - 1) + 1 * int.Parse(abn.Substring(1, 1)) + 3 * int.Parse(abn.Substring(2, 1)) + 5 * int.Parse(abn.Substring(3, 1)) + 7 * int.Parse(abn.Substring(4, 1)) + 9 * int.Parse(abn.Substring(5, 1)) + 11 * int.Parse(abn.Substring(6, 1)) + 13 * int.Parse(abn.Substring(7, 1)) + 15 * int.Parse(abn.Substring(8, 1)) + 17 * int.Parse(abn.Substring(9, 1)) + 19 * int.Parse(abn.Substring(10, 1));
            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;
            acn = acn.Trim();
            if (!decimal.TryParse(acn, out decimalAbn))
                return true;
            if (acn.Length != 9)
                return true;
            decimal abnSum = 8 * int.Parse(acn.Substring(0, 1)) + 7 * int.Parse(acn.Substring(1, 1)) + 6 * int.Parse(acn.Substring(2, 1)) + 5 * int.Parse(acn.Substring(3, 1)) + 4 * int.Parse(acn.Substring(4, 1)) + 3 * int.Parse(acn.Substring(5, 1)) + 2 * int.Parse(acn.Substring(6, 1)) + 1 * int.Parse(acn.Substring(7, 1));
            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;
            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;
            if (tfn == null)
                return false;
            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");
            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;
            if (tfn.Length < 8)
                return true;
            decimal tfn1To7Sum = 1 * int.Parse(tfn.Substring(0, 1)) + 4 * int.Parse(tfn.Substring(1, 1)) + 3 * int.Parse(tfn.Substring(2, 1)) + 7 * int.Parse(tfn.Substring(3, 1)) + 5 * int.Parse(tfn.Substring(4, 1)) + 8 * int.Parse(tfn.Substring(5, 1)) + 6 * int.Parse(tfn.Substring(6, 1));
            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));
            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;
                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;
                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }

        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;
            char[] characters = flags.ToCharArray();
            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name = "occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public FVSSubmitRequest2026 ConsumedReport
        {
            get
            {
                return report;
            }

            private set
            {
                report = value;
            }
        }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(FVSSubmitRequest2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;
            this.ConsumedReport = reportIn;
            VRATOFVS000014();
            VRATOFVS000028();
            VRATOFVS000041();
            VRATOFVS000113();
            VRATOFVS000114();
            VRATOFVS000115();
            VRATOFVS000116();
            VRATOFVS000117();
            VRATOFVS000122();
            VRATOFVS000125();
            VRATOFVS000202();
            VRATOFVS000203();
            VRATOFVS000204();
            VRATOFVS000205();
            VRATOFVS000207();
            VRATOFVS000208();
            VRATOFVS000209();
            VRATOFVS000211();
            VRATOFVS000213();
            VRATOFVS000214();
            VRATOFVS000215();
            VRATOFVS000216();
            VRATOFVS000217();
            VRATOFVS000218();
            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

        #region VR.ATO.FVS.000014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000014()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000014
            USI must be either:
            - valid ABN + 3 numeric
            - SPIN (3 alpha + 4 numeric + 2 alpha)
    
            Legacy Rule Format:
            ((Length(^FVS6) <> 9 AND Length(^FVS6) <> 14) OR (Length(^FVS6) = 14 AND (FailsABNAlgorithm(Substring(^FVS6,0,11)) OR NotCharacterInSet(Substring(^FVS6,11,3), '"0-9"'))) OR
            (Length(^FVS6) = 9 AND (NotCharacterInSet(Substring(^FVS6,0,3), '"A-Z", "a-z"') OR NotCharacterInSet(Substring(^FVS6,3,4), '"0-9"') OR NotCharacterInSet(Substring(^FVS6,7,2), '"A-Z", "a-z"'))))

            Technical Business Rule Format:
            ((Length(^FVS6) <> 9 AND Length(^FVS6) <> 14) OR (Length(^FVS6) = 14 AND (FailsABNAlgorithm(Substring(^FVS6,0,11)) OR NotCharacterInSet(Substring(^FVS6,11,3), '"0-9"'))) OR
            (Length(^FVS6) = 9 AND (NotCharacterInSet(Substring(^FVS6,0,3), '"A-Z", "a-z"') OR NotCharacterInSet(Substring(^FVS6,3,4), '"0-9"') OR NotCharacterInSet(Substring(^FVS6,7,2), '"A-Z", "a-z"'))))
    
            Data Elements:
    
            ^FVS6 = FVS Submit:RP:ProductUSIInformation:SuperannuationFundDetails.UniqueSuperannuationIdentifier.Identifier
            */
            assertion = (Length(report.FVS6) != 9 && Length(report.FVS6) != 14 || Length(report.FVS6) == 14 && (FailsABNAlgorithm(Substring(report.FVS6, 0, 11)) || !(IsMatch(Substring(report.FVS6, 11, 3), @"^[0-9]*$", RegexOptions.IgnoreCase))) || Length(report.FVS6) == 9 && (!(IsMatch(Substring(report.FVS6, 0, 3), @"^[A-Za-z]*$", RegexOptions.IgnoreCase)) || !(IsMatch(Substring(report.FVS6, 3, 4), @"^[0-9]*$", RegexOptions.IgnoreCase)) || !(IsMatch(Substring(report.FVS6, 7, 2), @"^[A-Za-z]*$", RegexOptions.IgnoreCase))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An invalid USI has been provided. A valid USI must be either provider's ABN + 3 digits or a SPIN",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:SuperannuationFundDetailsUniqueSuperannuationIdentifierId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000014"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS6", Value = report.FVS6 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000028
            Web address must start with "http://" or "https://"
    
            Legacy Rule Format:
            ^FVS52 <> NULL AND NOT StartsWithSet(^FVS52,'"http://","https://"')

            Technical Business Rule Format:
            ^FVS52 <> NULL AND NOT StartsWithSet(^FVS52,'"http://","https://"')
    
            Data Elements:
    
            ^FVS52 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.Web.UniformResourceLocator.Text
            */
            assertion = (report.FVS52 != null && !(IsMatch(report.FVS52, @"^(http://|https://)", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Web address must start with ""http://"" or ""https://"".",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactWebUniformResourceLocatorT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000028"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS52", Value = GetValueOrEmpty(report.FVS52) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000041
            The standard and version the fund is certified for can only contain accepted values separated by a single space
    
            Legacy Rule Format:
            NOT RegexMatch(^FVS17, '^(ROLL2\.0-X|ROLL3\.0-X|CONT1\.0-X|CONT2\.0-X|CONT3\.0-X|GCON2\.0-X|GROL2\.0-X|GROL3\.0-X|RESP)( (ROLL2\.0-X|ROLL3\.0-X|CONT1\.0-X|CONT2\.0-X|CONT3\.0-X|GCON2\.0-X|GROL2\.0-X|GROL3\.0-X|RESP))*$')

            Technical Business Rule Format:
            NOT RegexMatch(^FVS17, '^(ROLL2\.0-X|ROLL3\.0-X|CONT1\.0-X|CONT2\.0-X|CONT3\.0-X|GCON2\.0-X|GROL2\.0-X|GROL3\.0-X|RESP)( (ROLL2\.0-X|ROLL3\.0-X|CONT1\.0-X|CONT2\.0-X|CONT3\.0-X|GCON2\.0-X|GROL2\.0-X|GROL3\.0-X|RESP))*$')
    
            Data Elements:
    
            ^FVS17 = FVS Submit:RP:ElectronicContact.WebServices.ConformanceToStandard.Text
            */
            assertion = !(RegexMatch(report.FVS17, @"^(ROLL2\.0-X|ROLL3\.0-X|CONT1\.0-X|CONT2\.0-X|CONT3\.0-X|GCON2\.0-X|GROL2\.0-X|GROL3\.0-X|RESP)( (ROLL2\.0-X|ROLL3\.0-X|CONT1\.0-X|CONT2\.0-X|CONT3\.0-X|GCON2\.0-X|GROL2\.0-X|GROL3\.0-X|RESP))*$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The standard and version details of the fund's certification is invalid",
                    LongDescription = @"Valid values for standard and version details of the fund's certification are: ROLL2.0-X, ROLL3.0-X, CONT1.0-X, CONT2.0-X, CONT3.0-X , GCON2.0-X, GROL2.0-X, GROL3.0-X, RESP",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ElectronicContactWebServicesConformanceToStandardT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000041"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS17", Value = report.FVS17 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000113
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000113()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000113
            Email address must not contain more than one @ symbol
    
            Legacy Rule Format:
            ContainsMultiple(^FVS51, '@')

            Technical Business Rule Format:
            ContainsMultiple(^FVS51, '@')
    
            Data Elements:
    
            ^FVS51 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = IsMatch(report.FVS51, @"@(.*)@", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one @ symbol",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000113"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS51", Value = report.FVS51 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000114
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000114()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000114
            Email address must not start or end with a dot (.)
    
            Legacy Rule Format:
            (StartsWith(^FVS51, '.')) OR (EndsWith(^FVS51, '.'))

            Technical Business Rule Format:
            (StartsWith(^FVS51, '.')) OR (EndsWith(^FVS51, '.'))
    
            Data Elements:
    
            ^FVS51 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (IsMatch(report.FVS51, @"^(\.)", RegexOptions.IgnoreCase) || IsMatch(report.FVS51, @"(\.)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with a dot (.)",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000114"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS51", Value = report.FVS51 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000115
            Email address must not contain more than one consecutive dots (..)
    
            Legacy Rule Format:
            Contains(^FVS51, '..')

            Technical Business Rule Format:
            Contains(^FVS51, '..')
    
            Data Elements:
    
            ^FVS51 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FVS51 == null ? false : report.FVS51.ToUpperInvariant().Contains(@".."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one consecutive dots (..)",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000115"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS51", Value = report.FVS51 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000116
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000116()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000116
            Email address must not contain a dot followed by a @ symbol (.@)
    
            Legacy Rule Format:
            Contains(^FVS51, '.@')

            Technical Business Rule Format:
            Contains(^FVS51, '.@')
    
            Data Elements:
    
            ^FVS51 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FVS51 == null ? false : report.FVS51.ToUpperInvariant().Contains(@".@"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430227",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a dot followed by a @ symbol (.@)",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000116"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS51", Value = report.FVS51 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000117
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000117()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000117
            Email address must not contain a @ symbol followed by a dot (@.)
    
            Legacy Rule Format:
            Contains(^FVS51, '@.')

            Technical Business Rule Format:
            Contains(^FVS51, '@.')
    
            Data Elements:
    
            ^FVS51 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FVS51 == null ? false : report.FVS51.ToUpperInvariant().Contains(@"@."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430228",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a @ symbol followed by a dot (@.)",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000117"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS51", Value = report.FVS51 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000122
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000122()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000122
            BSB number must be 6 digits and in range 012001-999999
    
            Legacy Rule Format:
            ^FVS31 < 012001 OR ^FVS31 > 999999

            Technical Business Rule Format:
            ^FVS31 < 012001 OR ^FVS31 > 999999
    
            Data Elements:
    
            ^FVS31 = FVS Submit:RP:ProductUSIInformation:ContributionDetails:FinancialInstitutionAccount.BankStateBranch.Number
            */
            assertion = (AsNumeric(report.FVS31) < 12001 || AsNumeric(report.FVS31) > 999999);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be six digits and greater than 012000",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:ContributionDetails/tns:FinancialInstitutionAccountBankStateBranchN",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000122"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS31", Value = report.FVS31 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000125
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000125()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000125
            BSB number must be 6 digits and in range 012001-999999
    
            Legacy Rule Format:
            ^FVS41 < 012001 OR ^FVS41 > 999999

            Technical Business Rule Format:
            ^FVS41 < 012001 OR ^FVS41 > 999999
    
            Data Elements:
    
            ^FVS41 = FVS Submit:RP:ProductUSIInformation:RolloverDetails:FinancialInstitutionAccount.BankStateBranch.Number
            */
            assertion = (AsNumeric(report.FVS41) < 12001 || AsNumeric(report.FVS41) > 999999);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be six digits and greater than 012000",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:RolloverDetails/tns:FinancialInstitutionAccountBankStateBranchN",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000125"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS41", Value = report.FVS41 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000202
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000202()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000202
            USI 'effective from' date must be at least 14 calendar days after the submission date
    
            Legacy Rule Format:
            Today() > ConvertToDate(14, 07, 2026) AND  (DaysInDuration(^FVS12 - Today())< 14)

            Technical Business Rule Format:
            Today() > ConvertToDate(14, 07, 2026) AND  (DaysInDuration(^FVS12 - Today())< 14)
    
            Data Elements:
    
            ^FVS12 = FVS Submit:RP:Period.Start.Date
            */
            assertion = (DateTime.Now.Date > ConvertToDate(14, 7, 2026) && (report.FVS12.GetValueOrDefault() - DateTime.Now.Date).Days < 14);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000202",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"USI 'effective from' date must be at least 14 calendar days after the submission date",
                    LongDescription = @"When submitting new or updated USI details on or after 15 July 2026, 'effective from' date must be at least 14 calendar days after the submission date.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000202"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS12", Value = GetValueOrEmptyDate(report.FVS12) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000203()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000203
            Each of the standard and version the fund is certified for can only exist once
    
            Legacy Rule Format:
            ContainsMultiple(^FVS17, 'CONT1.0-X') OR ContainsMultiple(^FVS17, 'CONT2.0-X') OR
            ContainsMultiple(^FVS17, 'CONT3.0-X') OR ContainsMultiple(^FVS17, 'ROLL2.0-X') OR ContainsMultiple(^FVS17, 'ROLL3.0-X') OR
            ContainsMultiple(^FVS17, 'GCON2.0-X') OR
            ContainsMultiple(^FVS17, 'GROL2.0-X') OR
            ContainsMultiple(^FVS17, 'GROL3.0-X') OR
            ContainsMultiple(^FVS17, 'RESP')

            Technical Business Rule Format:
            ContainsMultiple(^FVS17, 'CONT1.0-X') OR ContainsMultiple(^FVS17, 'CONT2.0-X') OR
            ContainsMultiple(^FVS17, 'CONT3.0-X') OR ContainsMultiple(^FVS17, 'ROLL2.0-X') OR ContainsMultiple(^FVS17, 'ROLL3.0-X') OR
            ContainsMultiple(^FVS17, 'GCON2.0-X') OR
            ContainsMultiple(^FVS17, 'GROL2.0-X') OR
            ContainsMultiple(^FVS17, 'GROL3.0-X') OR
            ContainsMultiple(^FVS17, 'RESP')
    
            Data Elements:
    
            ^FVS17 = FVS Submit:RP:ElectronicContact.WebServices.ConformanceToStandard.Text
            */
            assertion = (IsMatch(report.FVS17, @"CONT1.0-X(.*)CONT1.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"CONT2.0-X(.*)CONT2.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"CONT3.0-X(.*)CONT3.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"ROLL2.0-X(.*)ROLL2.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"ROLL3.0-X(.*)ROLL3.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"GCON2.0-X(.*)GCON2.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"GROL2.0-X(.*)GROL2.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"GROL3.0-X(.*)GROL3.0-X", RegexOptions.IgnoreCase) || IsMatch(report.FVS17, @"RESP(.*)RESP", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"One or more of the allowable standard versions has been reported more than once",
                    LongDescription = @"Duplicate certification values are not allowed. Each certification value cannot exist more than once.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ElectronicContactWebServicesConformanceToStandardT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000203"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS17", Value = report.FVS17 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000204
            A USI closing reason code must be provided
    
            Legacy Rule Format:
            ^FVS13 <> AsDate('9999-12-31') AND ^FVS20 = NULL

            Technical Business Rule Format:
            ^FVS13 <> AsDate('9999-12-31') AND ^FVS20 = NULL
    
            Data Elements:
    
            ^FVS20 = FVS Submit:RP:ProductUSIInformation:SuperannuationFundDetails.ClosedUniqueSuperannuationIdentifierReason.Code
    
            ^FVS13 = FVS Submit:RP:Period.End.Date
            */
            assertion = (report.FVS13.GetValueOrDefault() != AsDate(@"9999-12-31") && report.FVS20 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000204",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A USI closing reason code must be provided",
                    LongDescription = @"A reason code must be provided when the 'Effective To' date is not 31/12/9999. This indicates the USI is closing and requires justification.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000204"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS13", Value = GetValueOrEmptyDate(report.FVS13) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS20", Value = GetValueOrEmpty(report.FVS20) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000205()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000205
            New fund ABN must be provided when the USI is closing
    
            Legacy Rule Format:
            ^FVS20 <> NULL AND InSet(^FVS20, '"SFT","IFT"') AND ^FVS1015 = NULL

            Technical Business Rule Format:
            ^FVS20 <> NULL AND InSet(^FVS20, '"SFT","IFT"') AND ^FVS1015 = NULL
    
            Data Elements:
    
            ^FVS20 = FVS Submit:RP:ProductUSIInformation:SuperannuationFundDetails.ClosedUniqueSuperannuationIdentifierReason.Code
    
            ^FVS1015 = FVS Submit:RP:ProductUSIInformation:NewFundDetails
            */
            assertion = (report.FVS20 != null && IsMatch(report.FVS20, @"^(SFT|IFT)$") && report.RP_ProductUSIInformation_NewFundDetailsCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000205",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"New fund ABN must be provided when the USI is closing due to an SFT or IFT",
                    LongDescription = @"New fund ABN must be provided when the USI is closing due to a successor fund transfer (SFT) or intra-fund transfer (IFT).",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000205"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS20", Value = GetValueOrEmpty(report.FVS20) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000207
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000207()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000207
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            FailsABNAlgorithm(^FVS21)

            Technical Business Rule Format:
            FailsABNAlgorithm(^FVS21)
    
            Data Elements:
    
            ^FVS21 = FVS Submit:RP:ProductUSIInformation:NewFundDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = FailsABNAlgorithm(report.FVS21);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:NewFundDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000207"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS21", Value = report.FVS21 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000208()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000208
            USI must be either:
            - valid ABN + 3 numeric
            - SPIN (3 alpha + 4 numeric + 2 alpha)
    
            Legacy Rule Format:
            ^FVS22 <> NULL AND ((Length(^FVS22) <> 9 AND Length(^FVS22) <> 14) OR (Length(^FVS22) = 14 AND (FailsABNAlgorithm(Substring(^FVS22,0,11)) OR NotCharacterInSet(Substring(^FVS22,11,3), '"0-9"'))) OR
            (Length(^FVS22) = 9 AND (NotCharacterInSet(Substring(^FVS22,0,3), '"A-Z", "a-z"') OR NotCharacterInSet(Substring(^FVS22,3,4), '"0-9"') OR NotCharacterInSet(Substring(^FVS22,7,2), '"A-Z", "a-z"'))))

            Technical Business Rule Format:
            ^FVS22 <> NULL AND ((Length(^FVS22) <> 9 AND Length(^FVS22) <> 14) OR (Length(^FVS22) = 14 AND (FailsABNAlgorithm(Substring(^FVS22,0,11)) OR NotCharacterInSet(Substring(^FVS22,11,3), '"0-9"'))) OR
            (Length(^FVS22) = 9 AND (NotCharacterInSet(Substring(^FVS22,0,3), '"A-Z", "a-z"') OR NotCharacterInSet(Substring(^FVS22,3,4), '"0-9"') OR NotCharacterInSet(Substring(^FVS22,7,2), '"A-Z", "a-z"'))))
    
            Data Elements:
    
            ^FVS22 = FVS Submit:RP:ProductUSIInformation:NewFundDetails:SuperannuationFundDetails.UniqueSuperannuationIdentifier.Identifier
            */
            assertion = (report.FVS22 != null && (Length(report.FVS22) != 9 && Length(report.FVS22) != 14 || Length(report.FVS22) == 14 && (FailsABNAlgorithm(Substring(report.FVS22, 0, 11)) || !(IsMatch(Substring(report.FVS22, 11, 3), @"^[0-9]*$", RegexOptions.IgnoreCase))) || Length(report.FVS22) == 9 && (!(IsMatch(Substring(report.FVS22, 0, 3), @"^[A-Za-z]*$", RegexOptions.IgnoreCase)) || !(IsMatch(Substring(report.FVS22, 3, 4), @"^[0-9]*$", RegexOptions.IgnoreCase)) || !(IsMatch(Substring(report.FVS22, 7, 2), @"^[A-Za-z]*$", RegexOptions.IgnoreCase)))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An invalid USI has been provided. A valid USI must be either provider's ABN + 3 digits or a SPIN",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:NewFundDetails/tns:SuperannuationFundDetailsUniqueSuperannuationIdentifierId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000208"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS22", Value = GetValueOrEmpty(report.FVS22) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000209
            A contribution NPP PayID must not be provided
    
            Legacy Rule Format:
            ^FVS35 = FALSE AND ^FVS36 <> NULL

            Technical Business Rule Format:
            ^FVS35 = FALSE AND ^FVS36 <> NULL
    
            Data Elements:
    
            ^FVS35 = FVS Submit:RP:ProductUSIInformation:ContributionDetails:SuperannuationPayment.NewPaymentPlatform.Indicator
    
            ^FVS36 = FVS Submit:RP:ProductUSIInformation:ContributionDetails:PaymentMechanism.PayID.Identifier
            */
            assertion = (report.FVS35 == false && report.FVS36 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000209",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A NPP PayID must not be reported as the fund has indicated they do not accept NPP payments for contributions",
                    LongDescription = @"A contribution NPP PayID must not be provided when the fund does not accept NPP contribution payments.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:ContributionDetails/tns:SuperannuationPaymentNewPaymentPlatformI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000209"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS35", Value = GetValueOrEmpty(report.FVS35) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS36", Value = GetValueOrEmpty(report.FVS36) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000211()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000211
            A rollover NPP PayID must not be provided
    
            Legacy Rule Format:
            ^FVS44 = FALSE AND ^FVS45 <> NULL

            Technical Business Rule Format:
            ^FVS44 = FALSE AND ^FVS45 <> NULL
    
            Data Elements:
    
            ^FVS44 = FVS Submit:RP:ProductUSIInformation:RolloverDetails:SuperannuationPayment.NewPaymentPlatform.Indicator
    
            ^FVS45 = FVS Submit:RP:ProductUSIInformation:RolloverDetails:PaymentMechanism.PayID.Identifier
            */
            assertion = (report.FVS44 == false && report.FVS45 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A NPP PayID must not be reported as the fund has indicated they do not accept NPP payments for rollovers",
                    LongDescription = @"A rollover NPP PayID must not be provided when the fund does not accept NPP contribution payments.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:RolloverDetails/tns:SuperannuationPaymentNewPaymentPlatformI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000211"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS44", Value = GetValueOrEmpty(report.FVS44) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS45", Value = GetValueOrEmpty(report.FVS45) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000213
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000213()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000213
            Name field cannot contain 'space hyphen space'
    
            Legacy Rule Format:
            Contains(^FVS49, ' - ')

            Technical Business Rule Format:
            Contains(^FVS49, ' - ')
    
            Data Elements:
    
            ^FVS49 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.FVS49 == null ? false : report.FVS49.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000213",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact name cannot contain a hyphen both preceded and followed by a space",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000213"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS49", Value = report.FVS49 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000214
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000214()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000214
            RESP must be the only value submitted in the certification field for this request
    
            Legacy Rule Format:
            Contains(^FVS17, 'RESP') AND (
                Contains(^FVS17, 'CONT1.0-X') OR
                Contains(^FVS17, 'CONT2.0-X') OR
                Contains(^FVS17, 'CONT3.0-X') OR
                Contains(^FVS17, 'ROLL2.0-X') OR
                Contains(^FVS17, 'ROLL3.0-X') OR
                Contains(^FVS17, 'GCON2.0-X') OR
                Contains(^FVS17, 'GROL2.0-X') OR
                Contains(^FVS17, 'GROL3.0-X')
            )

            Technical Business Rule Format:
            Contains(^FVS17, 'RESP') AND (
                Contains(^FVS17, 'CONT1.0-X') OR
                Contains(^FVS17, 'CONT2.0-X') OR
                Contains(^FVS17, 'CONT3.0-X') OR
                Contains(^FVS17, 'ROLL2.0-X') OR
                Contains(^FVS17, 'ROLL3.0-X') OR
                Contains(^FVS17, 'GCON2.0-X') OR
                Contains(^FVS17, 'GROL2.0-X') OR
                Contains(^FVS17, 'GROL3.0-X')
            )
    
            Data Elements:
    
            ^FVS17 = FVS Submit:RP:ElectronicContact.WebServices.ConformanceToStandard.Text
            */
            assertion = ((report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"RESP")) && ((report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT1.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT2.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT3.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"ROLL2.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"ROLL3.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"GCON2.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"GROL2.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"GROL3.0-X"))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000214",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"RESP must be the only value submitted in the certification field for this request",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ElectronicContactWebServicesConformanceToStandardT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000214"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS17", Value = report.FVS17 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000215
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000215()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000215
            Reporting period start date must be prior to Reporting period end date
    
            Legacy Rule Format:
            ^FVS12 >= ^FVS13

            Technical Business Rule Format:
            ^FVS12 >= ^FVS13
    
            Data Elements:
    
            ^FVS12 = FVS Submit:RP:Period.Start.Date
    
            ^FVS13 = FVS Submit:RP:Period.End.Date
            */
            assertion = (report.FVS12.GetValueOrDefault() >= report.FVS13.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000215"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS12", Value = GetValueOrEmptyDate(report.FVS12) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS13", Value = GetValueOrEmptyDate(report.FVS13) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000216
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000216()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000216
            Each certification type can only appear once.
    
            Legacy Rule Format:
            (Contains(^FVS17, 'ROLL2.0-X') AND Contains(^FVS17, 'ROLL3.0-X')) 
            OR 
            (Contains(^FVS17, 'CONT1.0-X') AND (Contains(^FVS17, 'CONT2.0-X') OR Contains(^FVS17, 'CONT3.0-X'))) 
            OR 
            (Contains(^FVS17, 'CONT2.0-X') AND Contains(^FVS17, 'CONT3.0-X')) 
            OR 
            (Contains(^FVS17, 'GROL2.0-X') AND Contains(^FVS17, 'GROL3.0-X'))

            Technical Business Rule Format:
            (Contains(^FVS17, 'ROLL2.0-X') AND Contains(^FVS17, 'ROLL3.0-X')) 
            OR 
            (Contains(^FVS17, 'CONT1.0-X') AND (Contains(^FVS17, 'CONT2.0-X') OR Contains(^FVS17, 'CONT3.0-X'))) 
            OR 
            (Contains(^FVS17, 'CONT2.0-X') AND Contains(^FVS17, 'CONT3.0-X')) 
            OR 
            (Contains(^FVS17, 'GROL2.0-X') AND Contains(^FVS17, 'GROL3.0-X'))
    
            Data Elements:
    
            ^FVS17 = FVS Submit:RP:ElectronicContact.WebServices.ConformanceToStandard.Text
            */
            assertion = ((report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"ROLL2.0-X")) && (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"ROLL3.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT1.0-X")) && ((report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT2.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT3.0-X"))) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT2.0-X")) && (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"CONT3.0-X")) || (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"GROL2.0-X")) && (report.FVS17 == null ? false : report.FVS17.ToUpperInvariant().Contains(@"GROL3.0-X")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000216",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Each certification type can only appear once.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ElectronicContactWebServicesConformanceToStandardT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000216"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS17", Value = report.FVS17 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000217
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000217()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000217
            New Fund Details must not be provided when closure reason is WOF
    
            Legacy Rule Format:
            ^FVS20 <> NULL AND InSet(^FVS20, '"WOF"') AND  ^FVS1015 <> NULL

            Technical Business Rule Format:
            ^FVS20 <> NULL AND InSet(^FVS20, '"WOF"') AND  ^FVS1015 <> NULL
    
            Data Elements:
    
            ^FVS20 = FVS Submit:RP:ProductUSIInformation:SuperannuationFundDetails.ClosedUniqueSuperannuationIdentifierReason.Code
    
            ^FVS1015 = FVS Submit:RP:ProductUSIInformation:NewFundDetails
            */
            assertion = (report.FVS20 != null && IsMatch(report.FVS20, @"^(WOF)$") && report.RP_ProductUSIInformation_NewFundDetailsCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000217",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"New Fund Details must not be provided when closure reason is WOF",
                    LongDescription = @"New Fund Details must not be provided when closure reason is due to a wind up of fund.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:SuperannuationFundDetailsClosedUniqueSuperannuationIdentifierReasonC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000217"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS20", Value = GetValueOrEmpty(report.FVS20) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.FVS.000218
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFVS000218()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.FVS.000218
            The contact email address must be in a valid format.
    
            Legacy Rule Format:
            NOT IsValidEmail(^FVS51)

            Technical Business Rule Format:
            NOT IsValidEmail(^FVS51)
    
            Data Elements:
    
            ^FVS51 = FVS Submit:RP:ProductUSIInformation:TechnicalAssistanceContactDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = !(IsMatch(report.FVS51, @"^\S.*@.+\.\S+$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FVS.000218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The contact email address must be in a valid format.",
                    Location = "/tns:FVSSubmitRequest/tns:RP/tns:ProductUSIInformation/tns:TechnicalAssistanceContactDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.FVS.000218"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "FVS51", Value = report.FVS51 });
                response.Add(processMessage);
            }
        }
        #endregion
    }
}