using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationDDCTNS
{

    public partial class DDCTNS2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private DDCTNS2026 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response {get; private set;}

        /// <summary>
        /// Initializes a new instance of the <see cref="DDCTNS2026Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public DDCTNS2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="DDCTNS2026Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public DDCTNS2026Validator(DDCTNS2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY"):this(maxParameterNameLength,maxParameterValueLength,emptyParameterValue)
        {                   
            this.ConsumedReport = reportIn;                    
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {                   
            return datetime.GetValueOrDefault().Date;
        }

				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }
        
        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public DDCTNS2026 ConsumedReport { get {return report;} private set {report = value;}}

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();				

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(DDCTNS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            #region Repeating report.Rp_CarCollection  
            if (report.Rp_CarCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (DDCTNS2026.Rp_Car car in report.Rp_CarCollection)
                {
                    itemIndex2++;
                        VRATODDCTNS000004(car, itemIndex2);
                        VRATODDCTNS000005(car, itemIndex2);
                        VRATODDCTNS000006(car, itemIndex2);
                        VRATODDCTNS000008(car, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.Rp_SelfEducationCollection  
            if (report.Rp_SelfEducationCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (DDCTNS2026.Rp_SelfEducation selfEducation in report.Rp_SelfEducationCollection)
                {
                    itemIndex2++;
                        VRATODDCTNS000010(selfEducation, itemIndex2);
                        VRATODDCTNS000012(selfEducation, itemIndex2);
                        VRATODDCTNS000013(selfEducation, itemIndex2);
                        VRATODDCTNS000014(selfEducation, itemIndex2);
                        VRATODDCTNS000015(selfEducation, itemIndex2);
                        VRATODDCTNS000104(selfEducation, itemIndex2);
                        VRATODDCTNS000105(selfEducation, itemIndex2);
                        VRATODDCTNS000106(selfEducation, itemIndex2);
                        VRATODDCTNS000321(selfEducation, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop
                VRATODDCTNS000100();
                VRATODDCTNS000155();
                VRATODDCTNS000304();
                VRATODDCTNS000305();
                VRATODDCTNSW00007();

            #region Repeating report.Rp_PersonalSuperContributionsCollection  
            if (report.Rp_PersonalSuperContributionsCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (DDCTNS2026.Rp_PersonalSuperContributions personalSuperContributions in report.Rp_PersonalSuperContributionsCollection)
                {
                    itemIndex2++;
                        VRATODDCTNS000300(personalSuperContributions, itemIndex2);
                        VRATODDCTNS000303(personalSuperContributions, itemIndex2);
                        VRATODDCTNS000312(personalSuperContributions, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.Rp_ClothingCollection  
            if (report.Rp_ClothingCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (DDCTNS2026.Rp_Clothing clothing in report.Rp_ClothingCollection)
                {
                    itemIndex2++;
                        VRATODDCTNS000319(clothing, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
				
			
    
      #region VR.ATO.DDCTNS.000004
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000004(DDCTNS2026.Rp_Car car, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000004
                    When the Car expenses calculation method is "Cents per kilometre", logbook method details must not be provided
    
                    Legacy Rule Format:
                    (^DDCTNS102 = 'Cents per kilometre' AND (^DDCTNS104 <> NULL OR ^DDCTNS105 <> NULL OR ^DDCTNS106 <> NULL))
        
                    Technical Business Rule Format:
                    (^DDCTNS102 = 'Cents per kilometre' AND (^DDCTNS104 <> NULL OR ^DDCTNS105 <> NULL OR ^DDCTNS106 <> NULL))
            
                    Data Elements:
            
                    ^DDCTNS102 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS104 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarLogBookMethodBusinessUseRecorded.Percent
            
                    ^DDCTNS105 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarLogBookMethod.Amount
            
                    ^DDCTNS106 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarLogBookMethodDepreciation.Amount
                    */
                    assertion = (car.DDCTNS102 == @"Cents per kilometre" && (car.DDCTNS104 != null || car.DDCTNS105 != null || car.DDCTNS106 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000004", Severity = ProcessMessageSeverity.Error,
                            Description = @"Incorrect Car expenses calculation method details provided",
                            LongDescription = @"When the Car expenses calculation method is ""Cents per kilometre"", logbook method details must not be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:CarCollection/tns:Car" + OccurrenceIndex(car.OccurrenceIndex) + "/tns:IncomeTaxDeductionCalculationMethodC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000004"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS102", Value = car.DDCTNS102 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS104", Value = car.DDCTNS104.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS105", Value = GetValueOrEmpty(car.DDCTNS105) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS106", Value = GetValueOrEmpty(car.DDCTNS106) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000005
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000005(DDCTNS2026.Rp_Car car, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000005
                    When the Car expenses calculation method is "Logbook", Car expenses business kilometres travelled must not be provided
    
                    Legacy Rule Format:
                    ^DDCTNS102 = 'Logbook' AND ^DDCTNS103 <> NULL
        
                    Technical Business Rule Format:
                    ^DDCTNS102 = 'Logbook' AND ^DDCTNS103 <> NULL
            
                    Data Elements:
            
                    ^DDCTNS102 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS103 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarBusinessKilometres.Number
                    */
                    assertion = (car.DDCTNS102 == @"Logbook" && car.DDCTNS103 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000005", Severity = ProcessMessageSeverity.Error,
                            Description = @"Incorrect Car expenses calculation method details provided",
                            LongDescription = @"When the Car expenses calculation method is ""Logbook"", Car expenses business kilometres travelled must not be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:CarCollection/tns:Car" + OccurrenceIndex(car.OccurrenceIndex) + "/tns:IncomeTaxDeductionCalculationMethodC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000005"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS102", Value = car.DDCTNS102 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS103", Value = car.DDCTNS103.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000006
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000006(DDCTNS2026.Rp_Car car, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000006
                    When the Car expenses calculation method is "Cents per kilometre", Car expenses business kilometres travelled must be provided
    
                    Legacy Rule Format:
                    ^DDCTNS102 = 'Cents per kilometre' AND ^DDCTNS103 = NULL
        
                    Technical Business Rule Format:
                    ^DDCTNS102 = 'Cents per kilometre' AND ^DDCTNS103 = NULL
            
                    Data Elements:
            
                    ^DDCTNS103 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarBusinessKilometres.Number
            
                    ^DDCTNS102 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarCalculationMethod.Code
                    */
                    assertion = (car.DDCTNS102 == @"Cents per kilometre" && car.DDCTNS103 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000006", Severity = ProcessMessageSeverity.Error,
                            Description = @"Car expenses business kilometres travelled must be provided",
                            LongDescription = @"When the Car expenses calculation method is ""Cents per kilometre"", Car expenses business kilometres travelled must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:CarCollection/tns:Car" + OccurrenceIndex(car.OccurrenceIndex) + "/tns:IncomeTaxDeductionBusinessKilometresN",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000006"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS102", Value = car.DDCTNS102 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS103", Value = car.DDCTNS103.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000008(DDCTNS2026.Rp_Car car, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000008
                    When the Car expenses calculation method is "Logbook", both Car expenses percentage of business use and Car expenses amount must be provided
    
                    Legacy Rule Format:
                    (^DDCTNS102 = 'Logbook') AND (^DDCTNS104 = NULL OR ^DDCTNS105 = NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS102 = 'Logbook') AND (^DDCTNS104 = NULL OR ^DDCTNS105 = NULL)
            
                    Data Elements:
            
                    ^DDCTNS104 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarLogBookMethodBusinessUseRecorded.Percent
            
                    ^DDCTNS102 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS105 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarLogBookMethod.Amount
                    */
                    assertion = (car.DDCTNS102 == @"Logbook" && (car.DDCTNS104 == null || car.DDCTNS105 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000008", Severity = ProcessMessageSeverity.Error,
                            Description = @"Car expenses logbook details must be provided",
                            LongDescription = @"When the Car expenses calculation method is ""Logbook"", both Car expenses percentage of business use and Car expenses amount must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:CarCollection/tns:Car" + OccurrenceIndex(car.OccurrenceIndex) + "/tns:IncomeTaxDeductionLogBookMethodBusinessUseRecordedP",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000008"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS102", Value = car.DDCTNS102 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS104", Value = car.DDCTNS104.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS105", Value = GetValueOrEmpty(car.DDCTNS105) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000010(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000010
                    Self education expense category must be "Car expenses" when Self education car expenses fields are provided
    
                    Legacy Rule Format:
                    (^DDCTNS202 <> 'Car expenses') AND (^DDCTNS120 <> NULL OR ^DDCTNS121 <> NULL OR ^DDCTNS122 <> NULL OR ^DDCTNS123 <> NULL OR ^DDCTNS124 <> NULL OR ^DDCTNS125 <> NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS202 <> 'Car expenses') AND (^DDCTNS120 <> NULL OR ^DDCTNS121 <> NULL OR ^DDCTNS122 <> NULL OR ^DDCTNS123 <> NULL OR ^DDCTNS124 <> NULL OR ^DDCTNS125 <> NULL)
            
                    Data Elements:
            
                    ^DDCTNS202 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.SelfEducation.Code
            
                    ^DDCTNS120 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:Assets.CarMakeModel.Description
            
                    ^DDCTNS121 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS122 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarBusinessKilometres.Number
            
                    ^DDCTNS123 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethodBusinessUseRecorded.Percent
            
                    ^DDCTNS124 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethod.Amount
            
                    ^DDCTNS125 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethodDepreciation.Amount
                    */
                    assertion = (selfEducation.DDCTNS202 != @"Car expenses" && (selfEducation.DDCTNS120 != null || selfEducation.DDCTNS121 != null || selfEducation.DDCTNS122 != null || selfEducation.DDCTNS123 != null || selfEducation.DDCTNS124 != null || selfEducation.DDCTNS125 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000010", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education expense category must be ""Car expenses"" when Self education car expenses fields are provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:IncomeTaxDeductionC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000010"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS202", Value = selfEducation.DDCTNS202 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS120", Value = GetValueOrEmpty(selfEducation.DDCTNS120) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS121", Value = selfEducation.DDCTNS121 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS122", Value = selfEducation.DDCTNS122.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS123", Value = selfEducation.DDCTNS123.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS124", Value = GetValueOrEmpty(selfEducation.DDCTNS124) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS125", Value = GetValueOrEmpty(selfEducation.DDCTNS125) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000012
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000012(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000012
                    When the Self education car expenses calculation method is "Cents per kilometre", logbook method details must not be provided
    
                    Legacy Rule Format:
                    (^DDCTNS121 = 'Cents per kilometre') AND (^DDCTNS123 <> NULL OR ^DDCTNS124 <> NULL OR ^DDCTNS125 <> NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS121 = 'Cents per kilometre') AND (^DDCTNS123 <> NULL OR ^DDCTNS124 <> NULL OR ^DDCTNS125 <> NULL)
            
                    Data Elements:
            
                    ^DDCTNS121 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS123 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethodBusinessUseRecorded.Percent
            
                    ^DDCTNS124 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethod.Amount
            
                    ^DDCTNS125 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethodDepreciation.Amount
                    */
                    assertion = (selfEducation.DDCTNS121 == @"Cents per kilometre" && (selfEducation.DDCTNS123 != null || selfEducation.DDCTNS124 != null || selfEducation.DDCTNS125 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000012", Severity = ProcessMessageSeverity.Error,
                            Description = @"Incorrect Self education car expenses calculation method details provided",
                            LongDescription = @"When the Self education car expenses calculation method is ""Cents per kilometre"", logbook method details must not be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:SelfEducationCarExpenses/tns:IncomeTaxDeductionCarCalculationMethodC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000012"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS121", Value = selfEducation.DDCTNS121 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS123", Value = selfEducation.DDCTNS123.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS124", Value = GetValueOrEmpty(selfEducation.DDCTNS124) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS125", Value = GetValueOrEmpty(selfEducation.DDCTNS125) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000013(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000013
                    When the Self education expenses calculation method is "Logbook", Self education car expenses business kilometres travelled must not be provided
    
                    Legacy Rule Format:
                    (^DDCTNS121 = 'Logbook') AND (^DDCTNS122 <> NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS121 = 'Logbook') AND (^DDCTNS122 <> NULL)
            
                    Data Elements:
            
                    ^DDCTNS121 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS122 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarBusinessKilometres.Number
                    */
                    assertion = (selfEducation.DDCTNS121 == @"Logbook" && selfEducation.DDCTNS122 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000013", Severity = ProcessMessageSeverity.Error,
                            Description = @"Incorrect Self education car expenses calculation method details provided",
                            LongDescription = @"When the Self education expenses calculation method is ""Logbook"", Self education car expenses business kilometres travelled must not be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:SelfEducationCarExpenses/tns:IncomeTaxDeductionCarCalculationMethodC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000013"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS121", Value = selfEducation.DDCTNS121 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS122", Value = selfEducation.DDCTNS122.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000014
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000014(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000014
                    When the Self education car expenses calculation method is "Cents per kilometre", Self education car expenses business kilometres travelled must be provided
    
                    Legacy Rule Format:
                    (^DDCTNS121 = 'Cents per kilometre') AND (^DDCTNS122 = NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS121 = 'Cents per kilometre') AND (^DDCTNS122 = NULL)
            
                    Data Elements:
            
                    ^DDCTNS122 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarBusinessKilometres.Number
            
                    ^DDCTNS121 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarCalculationMethod.Code
                    */
                    assertion = (selfEducation.DDCTNS121 == @"Cents per kilometre" && selfEducation.DDCTNS122 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000014", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education car expenses business kilometres travelled must be provided",
                            LongDescription = @"When the Self education car expenses calculation method is ""Cents per kilometre"", Self education car expenses business kilometres travelled must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:SelfEducationCarExpenses/tns:IncomeTaxDeductionCarBusinessKilometresN",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000014"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS121", Value = selfEducation.DDCTNS121 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS122", Value = selfEducation.DDCTNS122.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000015
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000015(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000015
                    When the Self education car expenses calculation method is "Logbook", both Self education car expenses percentage of business use and Self education car expenses amount must be provided
    
                    Legacy Rule Format:
                    (^DDCTNS121 = 'Logbook') AND (^DDCTNS123 = NULL OR ^DDCTNS124 = NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS121 = 'Logbook') AND (^DDCTNS123 = NULL OR ^DDCTNS124 = NULL)
            
                    Data Elements:
            
                    ^DDCTNS123 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethodBusinessUseRecorded.Percent
            
                    ^DDCTNS121 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarCalculationMethod.Code
            
                    ^DDCTNS124 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses:IncomeTax.Deduction.CarLogBookMethod.Amount
                    */
                    assertion = (selfEducation.DDCTNS121 == @"Logbook" && (selfEducation.DDCTNS123 == null || selfEducation.DDCTNS124 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000015", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education car expenses logbook method details must be provided",
                            LongDescription = @"When the Self education car expenses calculation method is ""Logbook"", both Self education car expenses percentage of business use and Self education car expenses amount must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:SelfEducationCarExpenses/tns:IncomeTaxDeductionCarLogBookMethodBusinessUseRecordedP",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000015"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS121", Value = selfEducation.DDCTNS121 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS123", Value = selfEducation.DDCTNS123.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS124", Value = GetValueOrEmpty(selfEducation.DDCTNS124) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000100
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000100()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.DDCTNS.000100
            When Self education expense instances are present, the Self education reason must be provided
    
            Legacy Rule Format:
            (^DDCTNS200 = NULL AND Count(^DDCTNS1004) > 0)

            Technical Business Rule Format:
            (^DDCTNS200 = NULL AND Count(^DDCTNS1004) > 0)
    
            Data Elements:
    
            ^DDCTNS200 = DDCTNS:Rp:IncomeTax.Deduction.EducationCircumstances.Code
    
            ^DDCTNS1004 = DDCTNS:Rp:SelfEducation
            */
            assertion = (report.DDCTNS200 == null && Count(report.Rp_SelfEducationCollectionCount) > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DDCTNS.000100", Severity = ProcessMessageSeverity.Error,
                    Description = @"Self education reason must be provided",
                    LongDescription = @"When Self education expense instances are present, the Self education reason must be provided",
                    Location = "/tns:DDCTNS/tns:Rp/tns:IncomeTaxDeductionEducationCircumstancesC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS200", Value = GetValueOrEmpty(report.DDCTNS200) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.DDCTNS.000104
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000104(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000104
                    When the Self education expense/depreciation amount is present, the only valid Self education expense category codes are; "Other", "Decline in value", "Repair expenses" or "Non-deductible expenses"
    
                    Legacy Rule Format:
                    (^DDCTNS202 = 'Car expenses') AND (^DDCTNS204 <> NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS202 = 'Car expenses') AND (^DDCTNS204 <> NULL)
            
                    Data Elements:
            
                    ^DDCTNS202 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.SelfEducation.Code
            
                    ^DDCTNS204 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.Education.Amount
                    */
                    assertion = (selfEducation.DDCTNS202 == @"Car expenses" && selfEducation.DDCTNS204 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000104", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education expense category is incorrect",
                            LongDescription = @"When the Self education expense/depreciation amount is present, the only valid Self education expense category codes are; ""Other"", ""Decline in value"", ""Repair expenses"" or ""Non-deductible expenses""",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:IncomeTaxDeductionC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000104"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS202", Value = selfEducation.DDCTNS202 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS204", Value = GetValueOrEmpty(selfEducation.DDCTNS204) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000105
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000105(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000105
                    When the Self education expense description is present, the Self education expense/depreciation amount must be provided
    
                    Legacy Rule Format:
                    (^DDCTNS203 <> NULL) AND (^DDCTNS204 = NULL)
        
                    Technical Business Rule Format:
                    (^DDCTNS203 <> NULL) AND (^DDCTNS204 = NULL)
            
                    Data Elements:
            
                    ^DDCTNS203 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.EducationCategoryOther.Description
            
                    ^DDCTNS204 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.Education.Amount
                    */
                    assertion = (selfEducation.DDCTNS203 != null && selfEducation.DDCTNS204 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000105", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education expense/depreciation amount must be provided",
                            LongDescription = @"When the Self education expense description is present, the Self education expense/depreciation amount must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:IncomeTaxDeductionEducationCategoryOtherDe",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000105"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS203", Value = GetValueOrEmpty(selfEducation.DDCTNS203) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS204", Value = GetValueOrEmpty(selfEducation.DDCTNS204) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000106
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000106(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000106
                    The Self education expense/depreciation amount must be provided when the Self education expense category is "Other", "Decline in value", "Repair expenses" or "Non-deductible expenses"
    
                    Legacy Rule Format:
                    (^DDCTNS204 = NULL) AND (^DDCTNS202 <> NULL) AND (^DDCTNS202 <> 'Car expenses')
        
                    Technical Business Rule Format:
                    (^DDCTNS204 = NULL) AND (^DDCTNS202 <> NULL) AND (^DDCTNS202 <> 'Car expenses')
            
                    Data Elements:
            
                    ^DDCTNS204 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.Education.Amount
            
                    ^DDCTNS202 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.SelfEducation.Code
                    */
                    assertion = (selfEducation.DDCTNS204 == null && selfEducation.DDCTNS202 != null && selfEducation.DDCTNS202 != @"Car expenses");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000106", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education expense/depreciation amount must be provided",
                            LongDescription = @"The Self education expense/depreciation amount must be provided when the Self education expense category is ""Other"", ""Decline in value"", ""Repair expenses"" or ""Non-deductible expenses""",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:IncomeTaxDeductionA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000106"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS204", Value = GetValueOrEmpty(selfEducation.DDCTNS204) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS202", Value = selfEducation.DDCTNS202 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000155
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000155()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.DDCTNS.000155
            At least one deduction detail must be provided
    
            Legacy Rule Format:
            (^DDCTNS200 = NULL AND
            ^DDCTNS310 = NULL AND
            ^DDCTNS311 = NULL AND
            ^DDCTNS312 = NULL AND
            ^DDCTNS417 = NULL AND
            Count(^DDCTNS1001) = 0 AND
            Count(^DDCTNS1002) = 0 AND
            Count(^DDCTNS1003) = 0 AND
            Count(^DDCTNS1004) = 0 AND
            Count(^DDCTNS1006) = 0 AND
            Count(^DDCTNS1007) = 0 AND
            Count(^DDCTNS1008) = 0 AND
            Count(^DDCTNS1009) = 0 AND
            Count(^DDCTNS1010) = 0 AND
            Count(^DDCTNS1011) = 0 AND
            Count(^DDCTNS1012) = 0 AND
            Count(^DDCTNS1013) = 0 AND
            Count(^DDCTNS1014) = 0 AND
            Count(^DDCTNS1015) = 0)

            Technical Business Rule Format:
            (^DDCTNS200 = NULL AND
            ^DDCTNS310 = NULL AND
            ^DDCTNS311 = NULL AND
            ^DDCTNS312 = NULL AND
            ^DDCTNS417 = NULL AND
            Count(^DDCTNS1001) = 0 AND
            Count(^DDCTNS1002) = 0 AND
            Count(^DDCTNS1003) = 0 AND
            Count(^DDCTNS1004) = 0 AND
            Count(^DDCTNS1006) = 0 AND
            Count(^DDCTNS1007) = 0 AND
            Count(^DDCTNS1008) = 0 AND
            Count(^DDCTNS1009) = 0 AND
            Count(^DDCTNS1010) = 0 AND
            Count(^DDCTNS1011) = 0 AND
            Count(^DDCTNS1012) = 0 AND
            Count(^DDCTNS1013) = 0 AND
            Count(^DDCTNS1014) = 0 AND
            Count(^DDCTNS1015) = 0)
    
            Data Elements:
    
            ^DDCTNS200 = DDCTNS:Rp:IncomeTax.Deduction.EducationCircumstances.Code
    
            ^DDCTNS310 = DDCTNS:Rp:LowValuePool:IncomeTax.Deduction.LowValuePoolFinancialInvestments.Amount
    
            ^DDCTNS311 = DDCTNS:Rp:LowValuePool:IncomeTax.Deduction.LowValuePoolRentalProperties.Amount
    
            ^DDCTNS312 = DDCTNS:Rp:LowValuePool:IncomeTax.Deduction.LowValuePoolOther.Amount
    
            ^DDCTNS417 = DDCTNS:Rp:IncomeTax.Deduction.Election.Amount
    
            ^DDCTNS1001 = DDCTNS:Rp:Car
    
            ^DDCTNS1002 = DDCTNS:Rp:Travel
    
            ^DDCTNS1003 = DDCTNS:Rp:Clothing
    
            ^DDCTNS1004 = DDCTNS:Rp:SelfEducation
    
            ^DDCTNS1006 = DDCTNS:Rp:OtherWorkRelatedExpenses
    
            ^DDCTNS1007 = DDCTNS:Rp:GiftsorDonations
    
            ^DDCTNS1008 = DDCTNS:Rp:InterestDeductions
    
            ^DDCTNS1009 = DDCTNS:Rp:DividendDeductions
    
            ^DDCTNS1010 = DDCTNS:Rp:CostofManagingTaxAffairs
    
            ^DDCTNS1011 = DDCTNS:Rp:PersonalSuperContributions
    
            ^DDCTNS1012 = DDCTNS:Rp:ProjectPool
    
            ^DDCTNS1013 = DDCTNS:Rp:ForestryManagedInvestment
    
            ^DDCTNS1014 = DDCTNS:Rp:OtherDeductions
    
            ^DDCTNS1015 = DDCTNS:Rp:LowValuePool
            */
            assertion = (report.DDCTNS200 == null && report.DDCTNS310 == null && report.DDCTNS311 == null && report.DDCTNS312 == null && report.DDCTNS417 == null && Count(report.Rp_CarCollectionCount) == 0 && Count(report.Rp_TravelCollectionCount) == 0 && Count(report.Rp_ClothingCollectionCount) == 0 && Count(report.Rp_SelfEducationCollectionCount) == 0 && Count(report.Rp_OtherWorkRelatedExpensesCollectionCount) == 0 && Count(report.Rp_GiftsorDonationsCollectionCount) == 0 && Count(report.Rp_InterestDeductionsCollectionCount) == 0 && Count(report.Rp_DividendDeductionsCollectionCount) == 0 && Count(report.Rp_CostofManagingTaxAffairsCollectionCount) == 0 && Count(report.Rp_PersonalSuperContributionsCollectionCount) == 0 && Count(report.Rp_ProjectPoolCollectionCount) == 0 && Count(report.Rp_ForestryManagedInvestmentCollectionCount) == 0 && Count(report.Rp_OtherDeductionsCollectionCount) == 0 && Count(report.Rp_LowValuePoolCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DDCTNS.000155", Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one deduction detail must be provided",
                    Location = "/tns:DDCTNS/tns:Rp",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS200", Value = GetValueOrEmpty(report.DDCTNS200) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS310", Value = GetValueOrEmpty(report.DDCTNS310) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS311", Value = GetValueOrEmpty(report.DDCTNS311) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS312", Value = GetValueOrEmpty(report.DDCTNS312) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS417", Value = GetValueOrEmpty(report.DDCTNS417) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.DDCTNS.000300
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000300(DDCTNS2026.Rp_PersonalSuperContributions personalSuperContributions, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000300
                    To be eligible to claim a deduction for personal super contributions, an individual must provide their fund with a notice of intent and their fund must acknowledge that notice before they lodge their tax return.
    
                    Legacy Rule Format:
                    (^DDCTNS401 = NULL OR ^DDCTNS401 = FALSE) AND (^DDCTNS407 > 0)
        
                    Technical Business Rule Format:
                    (^DDCTNS401 = NULL OR ^DDCTNS401 = FALSE) AND (^DDCTNS407 > 0)
            
                    Data Elements:
            
                    ^DDCTNS401 = DDCTNS:Rp:PersonalSuperContributions:SuperannuationContribution.NOIProvidedAcknowledgmentReceived.Indicator
            
                    ^DDCTNS407 = DDCTNS:Rp:PersonalSuperContributions:IncomeTax.Deduction.SuperannuationContribution.Amount
                    */
                    assertion = ((personalSuperContributions.DDCTNS401 == null || personalSuperContributions.DDCTNS401 == false) && personalSuperContributions.DDCTNS407.GetValueOrDefault() > 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000300", Severity = ProcessMessageSeverity.Error,
                            Description = @"Answer to 'Did you provide your fund (including a retirement savings account) with a notice of intent to claim a deduction for personal superannuation contributions, and receive an acknowledgement from your fund?' must be true",
                            LongDescription = @"To be eligible to claim a deduction for personal super contributions, the response to 'Did you provide your fund (including a retirement savings account) with a notice of intent to claim a deduction for personal superannuation contributions, and receive an acknowledgement from your fund?' must be 'yes' (true)",
                            Location = "/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions" + OccurrenceIndex(personalSuperContributions.OccurrenceIndex) + "/tns:SuperannuationContributionNOIProvidedAcknowledgmentReceivedI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000300"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS401", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS401) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS407", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS407) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000303
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000303(DDCTNS2026.Rp_PersonalSuperContributions personalSuperContributions, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000303
                    An amount is present at Deduction claimed for Personal Superannuation Contributions. Either the Full superannuation fund name and Superannuation account number are not populated, or the Superannuation fund Australian Business Number (ABN) and Superannuation fund Tax File Number (TFN) are not populated.
    
                    Legacy Rule Format:
                    (^DDCTNS407 > 0 AND ((^DDCTNS402 = BLANK AND ^DDCTNS404 = BLANK) OR (^DDCTNS405 = NULL AND ^DDCTNS406 = NULL)))
        
                    Technical Business Rule Format:
                    (^DDCTNS407 > 0 AND ((^DDCTNS402 = BLANK AND ^DDCTNS404 = BLANK) OR (^DDCTNS405 = NULL AND ^DDCTNS406 = NULL)))
            
                    Data Elements:
            
                    ^DDCTNS407 = DDCTNS:Rp:PersonalSuperContributions:IncomeTax.Deduction.SuperannuationContribution.Amount
            
                    ^DDCTNS402 = DDCTNS:Rp:PersonalSuperContributions:OrganisationNameDetails.OrganisationalName.Text
            
                    ^DDCTNS404 = DDCTNS:Rp:PersonalSuperContributions:SuperannuationFundDetails.MemberAccount.Identifier
            
                    ^DDCTNS405 = DDCTNS:Rp:PersonalSuperContributions:Identifiers.AustralianBusinessNumber.Identifier
            
                    ^DDCTNS406 = DDCTNS:Rp:PersonalSuperContributions:Identifiers.TaxFileNumber.Identifier
                    */
                    assertion = (personalSuperContributions.DDCTNS407.GetValueOrDefault() > 0 && (string.IsNullOrWhiteSpace(personalSuperContributions.DDCTNS402) == true && string.IsNullOrWhiteSpace(personalSuperContributions.DDCTNS404) == true || personalSuperContributions.DDCTNS405 == null && personalSuperContributions.DDCTNS406 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000303", Severity = ProcessMessageSeverity.Error,
                            Description = @"Personal superannuation contributions information is required",
                            LongDescription = @"An amount is present at Superannuation deduction claimed for Personal superannuation contributions. Either the Full superannuation fund name and the Superannuation account number are not populated, or the Superannuation fund Australiant Business Number (ABN) and Superannuation fund Tax File Number (TFN) are not populated.",
                            Location = "/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions" + OccurrenceIndex(personalSuperContributions.OccurrenceIndex) + "/tns:IncomeTaxDeductionSuperannuationContributionA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000303"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS407", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS407) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS402", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS402) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS404", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS404) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS405", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS405) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS406", Value = GetValueOrEmpty(personalSuperContributions.DDCTNS406) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000304
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000304()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.DDCTNS.000304
            Superannuation fund Australian Business Number (ABN) has failed the ABN algorithm check.
    
            Legacy Rule Format:
            (AnyOccurrence(^DDCTNS405,(FailsABNAlgorithm(^DDCTNS405))))

            Technical Business Rule Format:
            (AnyOccurrence(^DDCTNS405,(FailsABNAlgorithm(^DDCTNS405))))
    
            Data Elements:
    
            ^DDCTNS405 = DDCTNS:Rp:PersonalSuperContributions:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.Rp_PersonalSuperContributionsCollection == null ? false : report.Rp_PersonalSuperContributionsCollection.Any(DDCTNS405Repeat => FailsABNAlgorithm(DDCTNS405Repeat.DDCTNS405)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DDCTNS.000304", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    LongDescription = @"The Superannuation fund Australian Business Number (ABN) has failed the ABN algorithm check",
                    Location = "/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000304"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS405", Value = "Count(DDCTNS405)" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.DDCTNS.000305
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000305()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.DDCTNS.000305
            The Superannuation fund Tax File Number has failed the algorithm check.
    
            Legacy Rule Format:
            (AnyOccurrence(^DDCTNS406,(FailsTFNAlgorithm(^DDCTNS406))))

            Technical Business Rule Format:
            (AnyOccurrence(^DDCTNS406,(FailsTFNAlgorithm(^DDCTNS406))))
    
            Data Elements:
    
            ^DDCTNS406 = DDCTNS:Rp:PersonalSuperContributions:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.Rp_PersonalSuperContributionsCollection == null ? false : report.Rp_PersonalSuperContributionsCollection.Any(DDCTNS406Repeat => FailsTFNAlgorithm(DDCTNS406Repeat.DDCTNS406)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DDCTNS.000305", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid TFN",
                    LongDescription = @"The Superannuation fund Tax File Number (TFN) has failed the algorithm check",
                    Location = "/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000305"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS406", Value = "Count(DDCTNS406)" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.DDCTNS.000312
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000312(DDCTNS2026.Rp_PersonalSuperContributions personalSuperContributions, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000312
                    Date of last eligble Superannuation contribution must be within the current financial year.
    
                    Legacy Rule Format:
                    (^DDCTNS403 <> NULL AND (^DDCTNS403 < ^DDCTNS300 OR ^DDCTNS403 > ^DDCTNS301))
        
                    Technical Business Rule Format:
                    (^DDCTNS403 <> NULL AND (^DDCTNS403 < ^DDCTNS300 OR ^DDCTNS403 > ^DDCTNS301))
            
                    Data Elements:
            
                    ^DDCTNS403 = DDCTNS:Rp:PersonalSuperContributions:IncomeTax.Deduction.SuperannuationContributionEligible.Date
            
                    ^DDCTNS300 = DDCTNS:Rp:Lodgment.PeriodStart.Date
            
                    ^DDCTNS301 = DDCTNS:Rp:Lodgment.PeriodEnd.Date
                    */
                    assertion = (personalSuperContributions.DDCTNS403 != null && (personalSuperContributions.DDCTNS403.GetValueOrDefault() < report.DDCTNS300.GetValueOrDefault() || personalSuperContributions.DDCTNS403.GetValueOrDefault() > report.DDCTNS301.GetValueOrDefault()));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000312", Severity = ProcessMessageSeverity.Error,
                            Description = @"Date of last eligible superannuation contribution must be within the current financial year",
                            Location = "/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions" + OccurrenceIndex(personalSuperContributions.OccurrenceIndex) + "/tns:IncomeTaxDeductionSuperannuationContributionEligibleD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000312"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS403", Value = GetValueOrEmptyDate(personalSuperContributions.DDCTNS403) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS300", Value = GetValueOrEmptyDate(report.DDCTNS300) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS301", Value = GetValueOrEmptyDate(report.DDCTNS301) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000319
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000319(DDCTNS2026.Rp_Clothing clothing, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000319
                    When an amount above zero is provided for Work related clothing expenses amount, the Work related clothing expenses claim type must be provided
    
                    Legacy Rule Format:
                    ^DDCTNS416 > 0 AND ^DDCTNS415 = NULL
        
                    Technical Business Rule Format:
                    ^DDCTNS416 > 0 AND ^DDCTNS415 = NULL
            
                    Data Elements:
            
                    ^DDCTNS415 = DDCTNS:Rp:Clothing:IncomeTax.Deduction.ClothingType.Code
            
                    ^DDCTNS416 = DDCTNS:Rp:Clothing:IncomeTax.Deduction.Clothing.Amount
                    */
                    assertion = (clothing.DDCTNS416.GetValueOrDefault() > 0 && clothing.DDCTNS415 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000319", Severity = ProcessMessageSeverity.Error,
                            Description = @"Work related clothing expenses claim type must be provided",
                            LongDescription = @"When an amount above zero is provided for Work related clothing expenses amount, the Work related clothing expenses claim type must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:ClothingCollection/tns:Clothing" + OccurrenceIndex(clothing.OccurrenceIndex) + "/tns:IncomeTaxDeductionTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000319"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS416", Value = GetValueOrEmpty(clothing.DDCTNS416) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS415", Value = GetValueOrEmpty(clothing.DDCTNS415) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.000321
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNS000321(DDCTNS2026.Rp_SelfEducation selfEducation, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.DDCTNS.000321
                    When the Self education expense category is "Car expenses", Self education car expenses details must be provided
    
                    Legacy Rule Format:
                    (^DDCTNS202 = 'Car expenses') AND (Count(^DDCTNS1005) = 0)
        
                    Technical Business Rule Format:
                    (^DDCTNS202 = 'Car expenses') AND (Count(^DDCTNS1005) = 0)
            
                    Data Elements:
            
                    ^DDCTNS202 = DDCTNS:Rp:SelfEducation:IncomeTax.Deduction.SelfEducation.Code
            
                    ^DDCTNS1005 = DDCTNS:Rp:SelfEducation:SelfEducationCarExpenses
                    */
                    assertion = (selfEducation.DDCTNS202 == @"Car expenses" && Count(selfEducation.Rp_SelfEducation_SelfEducationCarExpensesCollectionCount) == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DDCTNS.000321", Severity = ProcessMessageSeverity.Error,
                            Description = @"Self education car expenses details must be provided",
                            LongDescription = @"When the Self education expense category is ""Car expenses"", Self education car expenses details must be provided",
                            Location = "/tns:DDCTNS/tns:Rp/tns:SelfEducationCollection/tns:SelfEducation" + OccurrenceIndex(selfEducation.OccurrenceIndex) + "/tns:IncomeTaxDeductionC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.000321"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DDCTNS202", Value = selfEducation.DDCTNS202 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.DDCTNS.W00007
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATODDCTNSW00007()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.DDCTNS.W00007
            The total number of all Car expenses business kilometres travelled exceeds 10,000. Check and correct if necessary.
    
            Legacy Rule Format:
            (Sum(^DDCTNS103) > 10000)

            Technical Business Rule Format:
            (Sum(^DDCTNS103) > 10000)
    
            Data Elements:
    
            ^DDCTNS103 = DDCTNS:Rp:Car:IncomeTax.Deduction.CarBusinessKilometres.Number
            */
            assertion = ((report.Rp_CarCollection == null ? 0 : report.Rp_CarCollection.Sum(f => f.DDCTNS103.GetValueOrDefault())) > 10000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DDCTNS.W00007", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check total number of Car expenses business kilometres travelled",
                    LongDescription = @"The total number of all Car expenses business kilometres travelled exceeds 10,000. Check and correct if necessary.",
                    Location = "/tns:DDCTNS/tns:Rp/tns:CarCollection/tns:Car/tns:IncomeTaxDeductionBusinessKilometresN",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DDCTNS.W00007"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DDCTNS103", Value = "DDCTNS103" });
    
                response.Add(processMessage);
            }
      }
      #endregion 

    }
} 