﻿using Ato.EN.IntegrationServices.CodeGenerationDDCTNS;
using Ato.EN.IntegrationServices.CodeGenerationINCDTLS;
using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Ato.EN.IntegrationServices.CodeGenerationIITR
{
    public partial class IITR2026CrossformValidator
    {
        private const decimal SingleMedicareLevy = 27222;
        private const decimal TierOneFamiliesThreshold = 194000;
        private const decimal TierOneSinglesThreshold = 97000;


        /// <summary>
        /// Takes a report to validate against all DDCTNS rules and store the results in the response paramaeter object.
        /// </summary>
        /// <param name="report">Document to be validated with DDCTNS rules.</param>
        /// <param name="response">Response to add error codes to.</param>
        public static void ValidateDDCTNS(IITR2026Crossform report, List<ProcessMessageDocument> response)
        {
            int ddctnsScheduleCount = Count(report.ddctnsChildDocuments);

            AddProcessMessage(VR_ATO_IITR_300093(report), response);
            AddProcessMessage(VR_ATO_IITR_300094(report), response);
            AddProcessMessage(VR_ATO_IITR_730105(report, ddctnsScheduleCount), response);
            AddProcessMessage(VR_ATO_IITR_730166(report), response);
            AddProcessMessage(VR_ATO_IITR_300085(report), response);
            AddProcessMessage(VR_ATO_IITR_300086(report), response);
            AddProcessMessage(VR_ATO_IITR_300084(report), response);
            AddProcessMessage(VR_ATO_IITR_600024(report), response);
            AddProcessMessage(VR_ATO_IITR_600014(report), response);
            AddProcessMessage(VR_ATO_IITR_200034(report), response);
            AddProcessMessage(VR_ATO_IITR_000013(report), response);
            AddProcessMessage(VR_ATO_IITR_200015(report), response);
            AddProcessMessage(VR_ATO_IITR_100043(report), response);

            // RPLOSSSeqNumContext
            // if (report.RPLOSSSeqNumContextCollection != null)
            if (report.iitrParentDocument.RPLOSSSeqNumContextCollection != null)
            {
                int itemIndexContext2 = 0;
                foreach (IITR2026.RPLOSSSeqNumContext rplossSeqNumContext in report.iitrParentDocument.RPLOSSSeqNumContextCollection)
                {
                    itemIndexContext2++;

                    AddProcessMessage(VR_ATO_IITR_100031(report, rplossSeqNumContext), response);
                    AddProcessMessage(VR_ATO_IITR_100032(report, rplossSeqNumContext), response);
                    AddProcessMessage(VR_ATO_IITR_100033(report, rplossSeqNumContext), response);
                    AddProcessMessage(VR_ATO_IITR_100034(report, rplossSeqNumContext), response);
                }
            }

            AddProcessMessage(VR_ATO_IITR_730107(report, ddctnsScheduleCount), response);

            #region DDCTNS Child Present
            DDCTNS2026 ddctnsChildDocument = null;

            if (ddctnsScheduleCount == 1)
            {
                ddctnsChildDocument = report.ddctnsChildDocuments[0];

                AddProcessMessage(VR_ATO_IITR_600031(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_700304(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730000(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730001(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730002(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730003(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730100(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730101(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730102(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730103(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730104(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730106(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730108(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730200(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730201(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730202(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730203(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730204(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730205(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730206(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730207(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730208(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730209(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730210(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730211(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730304(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730212(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730213(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730305(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730306(report, ddctnsChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_730307(report, ddctnsChildDocument), response);

                if (ddctnsChildDocument.Rp_TravelCollectionCount > 0)
                {
                    AddProcessMessage(VR_ATO_IITR_730300(report, ddctnsChildDocument), response);
                }

                if (ddctnsChildDocument.Rp_ClothingCollectionCount > 0)
                {
                    AddProcessMessage(VR_ATO_IITR_730308(report, ddctnsChildDocument), response);
                }

                #region Repeating RPSCSequNumContext

                if (ddctnsChildDocument.Rp_PersonalSuperContributionsCollectionCount > 0)
                {
                    DateTime? dateTaxpayerTurns75 = DateTaxpayerTurns75(report);
                    bool taxpayerDidNotTurn75ThisPeriod = TaxpayerDidNotTurn75ThisPeriod(report);

                    foreach (DDCTNS2026.Rp_PersonalSuperContributions context in ddctnsChildDocument.Rp_PersonalSuperContributionsCollection)
                    {
                        AddProcessMessage(VR_ATO_IITR_730301(report, context), response);
                        AddProcessMessage(VR_ATO_IITR_730302(report, context), response);
                    }
                }

                #endregion //Repeating RPSCSequNumContext
            }

            #endregion // DDCTNS Schedule Present

            AddProcessMessage(VR_ATO_IITR_200014(report, ddctnsChildDocument), response);
            AddProcessMessage(VR_ATO_IITR_730303(report, ddctnsChildDocument), response);
        }

        public static Decimal SumDDCTNS407(IITR2026Crossform report)
        {
            return Count(report.ddctnsChildDocuments) == 1 ? (report.ddctnsChildDocuments[0].Rp_PersonalSuperContributionsCollectionCount > 0 ? Math.Truncate(report.ddctnsChildDocuments[0].Rp_PersonalSuperContributionsCollection.Sum(f => f.DDCTNS407.GetValueOrDefault())) : 0) : 0;
        }

        public static Decimal SumDDCTNS138(IITR2026Crossform report)
        {
            // Calculate the sum of the other deductions amounts
            return Count(report.ddctnsChildDocuments) == 1 ? (report.ddctnsChildDocuments[0].Rp_OtherDeductionsCollectionCount > 0 ? Math.Truncate(report.ddctnsChildDocuments[0].Rp_OtherDeductionsCollection.Sum(f => f.DDCTNS138.GetValueOrDefault())) : 0) : 0;
        }

        public static Decimal SumDDCTNS138_D(IITR2026Crossform report)
        {
            // Sum all of the values of DDCTNS138 where DDCTNS136 are equal to "D - Deductions relating to financial investments"
            return Count(report.ddctnsChildDocuments) == 1 ? (report.ddctnsChildDocuments[0].Rp_OtherDeductionsCollectionCount > 0 ? Math.Truncate(report.ddctnsChildDocuments[0].Rp_OtherDeductionsCollection.Sum(context => context.DDCTNS136 == "D - Deductions relating to financial investments" ? context.DDCTNS138.GetValueOrDefault() : 0)) : 0) : 0;
        }

        public static Decimal SumDDCTNS138_U(IITR2026Crossform report)
        {
            // Sum all of the values of DDCTNS138 where DDCTNS136 are equal to "U - Debt deductions incurred in deriving foreign rental income"
            return Count(report.ddctnsChildDocuments) == 1 ? (report.ddctnsChildDocuments[0].Rp_OtherDeductionsCollectionCount > 0 ? Math.Truncate(report.ddctnsChildDocuments[0].Rp_OtherDeductionsCollection.Sum(context => context.DDCTNS136 == "U - Debt deductions incurred in deriving foreign rental income" ? context.DDCTNS138.GetValueOrDefault() : 0)) : 0) : 0;
        }

        public static decimal GetDDCTNSValue(IITR2026Crossform reportObject, Func<DDCTNS2026, decimal> fieldSelector, bool truncateValue = true)
        {
            decimal value = 0m;

            if (Count(reportObject.ddctnsChildDocuments) == 1)
            {
                value = fieldSelector(reportObject.ddctnsChildDocuments[0]);
            }

            if (truncateValue)
                value = Math.Truncate(value);

            return value;
        }

        public static DateTime? DateTaxpayerTurns75(IITR2026Crossform report)
        {
            DateTime? dateTaxpayerTurns75 = null;

            if (report.iitrParentDocument.IITR29.HasValue)
            {
                dateTaxpayerTurns75 = report.iitrParentDocument.IITR29.Value.AddYears(75);
            }

            return dateTaxpayerTurns75;
        }

        public static bool TaxpayerDidNotTurn75ThisPeriod(IITR2026Crossform report)
        {
            bool taxpayerDidNotTurn75ThisPeriod = true;

            if (report.iitrParentDocument.IITR29.HasValue)
            {
                DateTime? dateTaxpayerTurns75 = DateTaxpayerTurns75(report);
                taxpayerDidNotTurn75ThisPeriod = report.iitrParentDocument.RPStartDate.HasValue && report.iitrParentDocument.RPEndDate.HasValue ? dateTaxpayerTurns75.Value < report.iitrParentDocument.RPStartDate.Value || dateTaxpayerTurns75.Value > report.iitrParentDocument.RPEndDate.Value : true;
            }

            return taxpayerDidNotTurn75ThisPeriod;
        }

        #region VR.ATO.IITR.300093

        /* VR.ATO.IITR.300093
            If the total of financial investment income minus financial investment deductions is less than zero then the difference must be inserted at Net financial investment loss. If the difference exceeds zero then zero must be inserted at Net financial investment loss.
    
            Legacy Rule Format:
            (^IITR13 = NULL) AND ((((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND ((^IITR204 = NULL) OR (^IITR204 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 +^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0)) - (^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)))))) OR (((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND (^IITR204 = NULL OR ^IITR204 <> 0)))

            Technical Business Rule Format:
            (^IITR13 = NULL) AND ((((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND ((^IITR204 = NULL) OR (^IITR204 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 +^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0)) - (^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)))))) OR (((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND (^IITR204 = NULL OR ^IITR204 <> 0)))
    
            Data Elements:
    
            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
            ^DDCTNS136 = DDCTNS136
    
            ^DDCTNS138 = DDCTNS138
    
            ^DDCTNS310 = DDCTNS310
    
            ^IITR13 = RP:IncomeTax.ReturnType.Code
    
            ^IITR118 = RP:Income.DividendsUnfranked.Amount
    
            ^IITR119 = RP:Income.DividendsFranked.Amount
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
            ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
    
            ^IITR249 = RP:Income.PartnershipInvestmentLessForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR261 = RP:IncomeTax.Deduction.TrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProduction.Amount
    
            ^IITR262 = RP:IncomeTax.Deduction.PartnershipExcludingForeignIncomeNonPrimaryProduction.Amount
    
            ^IITR282 = RP:Income.InvestmentNonPrimaryProductionNet.Amount
    
            ^IITR291 = RP:Tax.Losses.DeferredNonCommercialPartnershipInvestmentShare.Amount
    
            ^IITR295 = RP:Tax.Losses.DeferredNonCommercialSoleTraderInvestment.Amount
    
            ^IITR325 = RP:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount
    
            ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount
    
            ^IITR341 = RP:Income.ForestryManagedInvestmentScheme.Amount
    
            ^IITR368 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^IITR613 = RP:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount
    
            ^IITR617 = RP:IncomeTax.FinancialInvestmentsOther.Amount
    
            ^INCDTLS251 = INCDTLS251
    
            ^INCDTLS254 = INCDTLS254
    
            ^INCDTLS387 = INCDTLS387
    
            ^INCDTLS389 = INCDTLS389
    
            ^INCDTLS390 = INCDTLS390
    
            ^INCDTLS419 = INCDTLS419
    
            ^INCDTLS420 = INCDTLS420
            */
        public static ProcessMessageDocument VR_ATO_IITR_300093(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS138_D = Math.Truncate(SumDDCTNS138_D(report));
            decimal valDDCTNS310 = Math.Truncate(GetDDCTNSValue(report, x => x.DDCTNS310.GetValueOrDefault()));
            decimal valINCDTLS251 = 0;
            decimal valINCDTLS251Calc = 0;
            decimal primaryNonPrimaryCalc1 = 0;
            decimal primaryNonPrimaryCalc2 = 0;
            decimal valINCDTLS254 = 0;
            decimal valINCDTLS254Calc = 0;

            // If the INCDTLS values exist get their values to be later used in the larger sum.
            if (report.incdtlsChildDocuments != null && report.incdtlsChildDocuments.Count == 1)
            {
                if (report.incdtlsChildDocuments[0].Rp_ManagedFundsCollectionExists)
                {
                    // Calculate the value of total Primary Non Primary Production Income From Managed Fund
                    // SUM(ConditionalValue([incdtls387] = 1, [incdtls389], [incdtlS419]))
                    primaryNonPrimaryCalc1 =
                        (report.incdtlsChildDocuments[0].Rp_ManagedFundsCollection.Sum(tuple =>
                            tuple.INCDTLS387 == 1 ?
                            tuple.INCDTLS389 : tuple.INCDTLS419)
                        ).GetValueOrDefault();

                    primaryNonPrimaryCalc2 =
                        (report.incdtlsChildDocuments[0].Rp_ManagedFundsCollection.Sum(tuple =>
                            tuple.INCDTLS387 == 1 ?
                            tuple.INCDTLS390 : tuple.INCDTLS420)
                        ).GetValueOrDefault();
                }

                if (report.incdtlsChildDocuments[0].Rp_TrustIncomeCollectionExists)
                {
                    // Add INCDTLS251
                    valINCDTLS251 = report.incdtlsChildDocuments[0].Rp_TrustIncomeCollection.Sum(tuple => tuple.INCDTLS251.GetValueOrDefault());

                    // Add INCDTLS254
                    valINCDTLS254 = report.incdtlsChildDocuments[0].Rp_TrustIncomeCollection.Sum(tuple => tuple.INCDTLS254.GetValueOrDefault());
                }

                //Total up total Primary Non Primary Production Income From Managed Fund, INCDTLS251 and INCDTLS254 and truncate the result.
                valINCDTLS251Calc = (Math.Truncate(valINCDTLS251 + primaryNonPrimaryCalc1));
                valINCDTLS254Calc = (Math.Truncate(valINCDTLS254 + primaryNonPrimaryCalc2));
            }

            // Total financial investment income
            decimal cal300093_1 = report.iitrParentDocument.IITR118.GetValueOrDefault() + report.iitrParentDocument.IITR119.GetValueOrDefault() + report.iitrParentDocument.IITR120.GetValueOrDefault() + report.iitrParentDocument.IITR249.GetValueOrDefault() + report.iitrParentDocument.IITR282.GetValueOrDefault() + report.iitrParentDocument.IITR291.GetValueOrDefault() + report.iitrParentDocument.IITR295.GetValueOrDefault() + report.iitrParentDocument.IITR325.GetValueOrDefault() + report.iitrParentDocument.IITR328.GetValueOrDefault() + report.iitrParentDocument.IITR341.GetValueOrDefault() + report.iitrParentDocument.IITR613.GetValueOrDefault() + report.iitrParentDocument.IITR617.GetValueOrDefault() + valINCDTLS251Calc + valINCDTLS254Calc;

            // Total financial investment deductions
            decimal cal300093_2 = valDDCTNS310 + report.iitrParentDocument.IITR147.GetValueOrDefault() + report.iitrParentDocument.IITR261.GetValueOrDefault() + report.iitrParentDocument.IITR262.GetValueOrDefault() + report.iitrParentDocument.IITR368.GetValueOrDefault() + sumDDCTNS138_D;

            bool condition1_Calc1LessthenCalc2 = (cal300093_1 < cal300093_2 && (report.iitrParentDocument.IITR204 == null || report.iitrParentDocument.IITR204.GetValueOrDefault() != cal300093_2 - cal300093_1));
            bool condition2_calc1greaterequalsCalc2 = (cal300093_1 >= cal300093_2 && (report.iitrParentDocument.IITR204 == null || report.iitrParentDocument.IITR204 != 0));

            bool assertion = (report.iitrParentDocument.IITR13 == null && (condition1_Calc1LessthenCalc2 || condition2_calc1greaterequalsCalc2));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.301093",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check Net Financial Investment Loss amount",
                    LongDescription = @"If the total of financial investment income minus financial investment deductions is less than zero then the difference must be inserted at Net financial investment loss. If the difference exceeds zero then zero must be inserted at Net financial investment loss.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.FinancialInvestmentNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.300093" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = report.iitrParentDocument.IITR13 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = report.iitrParentDocument.IITR118.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = report.iitrParentDocument.IITR119.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = report.iitrParentDocument.IITR120.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR249", Value = report.iitrParentDocument.IITR249.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR282", Value = report.iitrParentDocument.IITR282.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR291", Value = report.iitrParentDocument.IITR291.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR295", Value = report.iitrParentDocument.IITR295.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR325", Value = report.iitrParentDocument.IITR325.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR328", Value = report.iitrParentDocument.IITR328.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR341", Value = report.iitrParentDocument.IITR341.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR613", Value = report.iitrParentDocument.IITR613.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR617", Value = report.iitrParentDocument.IITR617.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "INCDTLS251", Value = valINCDTLS251.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "INCDTLS254", Value = valINCDTLS254.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS310", Value = valDDCTNS310.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR147", Value = report.iitrParentDocument.IITR147.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR261", Value = report.iitrParentDocument.IITR261.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR262", Value = report.iitrParentDocument.IITR262.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR368", Value = report.iitrParentDocument.IITR368.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS138", Value = sumDDCTNS138_D.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Total FII Sum", Value = cal300093_1.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Total FID Sum", Value = cal300093_2.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "INCDTLS389/419 Sum", Value = primaryNonPrimaryCalc1.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "INCDTLS390/420 Sum", Value = primaryNonPrimaryCalc2.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.300094

        /* VR.ATO.IITR.300094
        If the total of Net rental property income less Net rental property deductions is less than zero then the difference must be inserted at Net Rental property loss. If the difference exceeds zero then zero must be inserted at Net Rental property loss.

        Legacy Rule Format:
            (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (((^IITR205 = NULL) OR (^IITR205 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0)) - (^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614)))))) OR (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (^IITR205 <> 0))

        Technical Business Rule Format:
            (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (((^IITR205 = NULL) OR (^IITR205 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0)) - (^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614)))))) OR (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (^IITR205 <> 0))

        Data Elements:
    
        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
        ^DDCTNS136 = DDCTNS136
    
        ^DDCTNS138 = DDCTNS138
    
        ^DDCTNS311 = DDCTNS311
    
        ^IITR250 = RP:Income.PartnershipRentalNonPrimaryProductionShare.Amount
    
        ^IITR263 = RP:IncomeTax.Deduction.PartnershipRentalNonPrimaryProduction.Amount
    
        ^IITR283 = RP:Income.RentalPropertyNonPrimaryProductionNet.Amount
    
        ^IITR292 = RP:Tax.Losses.DeferredNonCommercialPartnershipRentalShare.Amount
    
        ^IITR296 = RP:Tax.Losses.DeferredNonCommercialSoleTraderRental.Amount
    
        ^IITR324 = RP:InternationalDealings.ForeignRent.Amount
    
        ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount
    
        ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount
    
        ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount
    
        ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
        ^IITR614 = RP:Income.DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotal.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_300094(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS138_U = SumDDCTNS138_U(report);
            decimal valDDCTNS311 = GetDDCTNSValue(report, x => x.DDCTNS311.GetValueOrDefault());

            // Total net rental property income
            decimal cal300094_1 = report.iitrParentDocument.IITR250.GetValueOrDefault() + report.iitrParentDocument.IITR283.GetValueOrDefault() + report.iitrParentDocument.IITR292.GetValueOrDefault() + report.iitrParentDocument.IITR296.GetValueOrDefault() + report.iitrParentDocument.IITR324.GetValueOrDefault() + (report.iitrParentDocument.IITR334.GetValueOrDefault() - (report.iitrParentDocument.IITR335.GetValueOrDefault() + report.iitrParentDocument.IITR336.GetValueOrDefault() + report.iitrParentDocument.IITR337.GetValueOrDefault())) + report.iitrParentDocument.IITR614.GetValueOrDefault();

            // Total net rental property deductions
            decimal cal300094_2 = valDDCTNS311 + report.iitrParentDocument.IITR263.GetValueOrDefault() + sumDDCTNS138_U;

            bool condition1_Calc1LessTheCalc2 = (cal300094_1 < cal300094_2 && (report.iitrParentDocument.IITR205 == null || report.iitrParentDocument.IITR205.GetValueOrDefault() != cal300094_2 - cal300094_1));
            bool condition1_Calc1GreaterThenOrEqualsCalc2 = (cal300094_1 >= cal300094_2 && report.iitrParentDocument.IITR205.GetValueOrDefault() != 0);

            bool assertion = (condition1_Calc1LessTheCalc2 || condition1_Calc1GreaterThenOrEqualsCalc2);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.300094",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Rental Property Loss is incorrect",
                    LongDescription = @"If the total of Net rental property income less Net rental property deductions is less than zero then the difference must be inserted at Net Rental property loss. If the difference exceeds zero then zero must be inserted at Net Rental property loss.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.RentalPropertyNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.300094" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR250", Value = report.iitrParentDocument.IITR250.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR283", Value = report.iitrParentDocument.IITR283.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR292", Value = report.iitrParentDocument.IITR292.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR296", Value = report.iitrParentDocument.IITR296.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR324", Value = report.iitrParentDocument.IITR324.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR334", Value = report.iitrParentDocument.IITR334.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR335", Value = report.iitrParentDocument.IITR335.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR336", Value = report.iitrParentDocument.IITR336.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR337", Value = report.iitrParentDocument.IITR337.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR614", Value = report.iitrParentDocument.IITR614.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS311", Value = valDDCTNS311.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR263", Value = report.iitrParentDocument.IITR263.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Total net rental property income", Value = cal300094_1.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Total net rental property deductions", Value = cal300094_2.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730105

        /* VR.ATO.IITR.730105
        The number of Deductions schedules must not exceed 1

        Legacy Rule Format:
            CountDocument('DDCTNS') > 1

        Technical Business Rule Format:
            CountDocument('DDCTNS') > 1

        Data Elements:
        */
        public static ProcessMessageDocument VR_ATO_IITR_730105(IITR2026Crossform report, int ddctnsScheduleCount)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = ddctnsScheduleCount > 1;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Deductions schedules must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730105" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.300085

        /* VR.ATO.IITR.300085
                    The taxpayer has passed the ESS income test. However, the total assessable discount amount declared can not be reduced by more than the upfront discount amount declared at Discount from taxed upfront schemes - eligible for reduction.

                Legacy Rule Format:
                    (ConditionalValue((^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125) > 0, (^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125), 0) + ^IITR201 + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) <= 180000 AND ^IITR123 <= 1000 AND (^IITR124 + ^IITR125) > ^IITR127

            Technical Business Rule Format:
                    (ConditionalValue((^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125) > 0, (^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125), 0) + ^IITR201 + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) <= 180000 AND ^IITR123 <= 1000 AND (^IITR124 + ^IITR125) > ^IITR127

        Data Elements:
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
    
            ^DDCTNS407 = DDCTNS407
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^IITR157 = RP:Income.Taxable.Amount
    
            ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
            ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
            ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
            */
        public static ProcessMessageDocument VR_ATO_IITR_300085(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            IITR2026 iitrParentDocument = report.iitrParentDocument;

            decimal shareSchemeAmountsLessIncomeAndFHSS = iitrParentDocument.IITR157.GetValueOrDefault() -
                    iitrParentDocument.IITR650.GetValueOrDefault() -
                    iitrParentDocument.IITR127.GetValueOrDefault() +
                    iitrParentDocument.IITR123.GetValueOrDefault() +
                    iitrParentDocument.IITR124.GetValueOrDefault() +
                    iitrParentDocument.IITR125.GetValueOrDefault();

            decimal totalReportableFringeBenefits = iitrParentDocument.IITR490.GetValueOrDefault() + iitrParentDocument.IITR491.GetValueOrDefault();
            
            bool assertion = (
                ConditionalValue(shareSchemeAmountsLessIncomeAndFHSS > 0, shareSchemeAmountsLessIncomeAndFHSS, 0) + 
                iitrParentDocument.IITR201.GetValueOrDefault() + 
                ConditionalValue(totalReportableFringeBenefits > 3772, totalReportableFringeBenefits, 0) + 
                iitrParentDocument.IITR204.GetValueOrDefault() + iitrParentDocument.IITR205.GetValueOrDefault() + sumDDCTNS407 <= 180000 && 

                iitrParentDocument.IITR123.GetValueOrDefault() <= 1000 && 

                iitrParentDocument.IITR124.GetValueOrDefault() + iitrParentDocument.IITR125.GetValueOrDefault() > 
                iitrParentDocument.IITR127.GetValueOrDefault()
            );

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.300085",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Discount from taxed upfront schemes - eligible for reduction' amount is incorrect",
                    LongDescription = @"The taxpayer has passed the ESS income test. However, the total assessable discount amount declared can not be reduced by more than the upfront discount amount declared at Discount from taxed upfront schemes - eligible for reduction.",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.300085" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = report.iitrParentDocument.IITR127.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = report.iitrParentDocument.IITR123.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = report.iitrParentDocument.IITR124.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = report.iitrParentDocument.IITR125.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.300086

        /* VR.ATO.IITR.300086
                    Your income test amount exceeds $180,000. Consequently you are not entitled to a reduction on the upfront discount amount. The Total assessable discount amount declared must equal the sum of Discount from taxed upfront schemes - eligible for reduction, Discount from taxed upfront schemes - not eligible for reduction, and Discount from deferral schemes.

                Legacy Rule Format:
                    (ConditionalValue((^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125) > 0, (^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125),0) + ^IITR201 + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 180000 AND ((^IITR123 + ^IITR124 + ^IITR125) > ^IITR127)

            Technical Business Rule Format:
                    (ConditionalValue((^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125) > 0, (^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125),0) + ^IITR201 + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 180000 AND ((^IITR123 + ^IITR124 + ^IITR125) > ^IITR127)

        Data Elements:
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
    
            ^DDCTNS407 = DDCTNS407
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^IITR157 = RP:Income.Taxable.Amount
    
            ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
            ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
            ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
            */
        public static ProcessMessageDocument VR_ATO_IITR_300086(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            IITR2026 iitrParentDocument = report.iitrParentDocument;

            decimal shareSchemeAmountsLessIncomeAndFHSS = 
                    iitrParentDocument.IITR157.GetValueOrDefault() -
                    iitrParentDocument.IITR650.GetValueOrDefault() -
                    iitrParentDocument.IITR127.GetValueOrDefault() +
                    iitrParentDocument.IITR123.GetValueOrDefault() +
                    iitrParentDocument.IITR124.GetValueOrDefault() +
                    iitrParentDocument.IITR125.GetValueOrDefault();

            decimal totalReportableFringeBenefits = iitrParentDocument.IITR490.GetValueOrDefault() + iitrParentDocument.IITR491.GetValueOrDefault();

            bool assertion = (
                ConditionalValue(shareSchemeAmountsLessIncomeAndFHSS > 0, shareSchemeAmountsLessIncomeAndFHSS, 0)+
                
                report.iitrParentDocument.IITR201.GetValueOrDefault() + 
                
                ConditionalValue(totalReportableFringeBenefits > 3772, totalReportableFringeBenefits, 0) + 

                report.iitrParentDocument.IITR204.GetValueOrDefault() + 
                report.iitrParentDocument.IITR205.GetValueOrDefault() + sumDDCTNS407 > 180000 && 

                report.iitrParentDocument.IITR123.GetValueOrDefault() + report.iitrParentDocument.IITR124.GetValueOrDefault() + 
                report.iitrParentDocument.IITR125.GetValueOrDefault() > report.iitrParentDocument.IITR127.GetValueOrDefault()
            );

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.301086",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reduction on the upfront discount amount has been disallowed",
                    LongDescription = @"Your income test amount exceeds $180,000. Consequently you are not entitled to a reduction on the upfront discount amount. The Total assessable discount amount declared must equal the sum of Discount from taxed upfront schemes - eligible for reduction, Discount from taxed upfront schemes - not eligible for reduction, Discount from deferral schemes and Discount on ESS Interests acquired pre 1 July 2009 and 'cessation time' occurred during financial year.",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.300086" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = report.iitrParentDocument.IITR127.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = report.iitrParentDocument.IITR123.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = report.iitrParentDocument.IITR124.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = report.iitrParentDocument.IITR125.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.300084

        /* VR.ATO.IITR.300084
                    The taxpayer has passed the ESS income test. However, the total assessable discount amount cannot be reduced by more than $1,000.

                Legacy Rule Format:
                    ^IITR123 <> NULL AND (ConditionalValue((^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125) > 0, (^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125), 0) + ^IITR201 + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) <= 180000 AND (^IITR123 + ^IITR124 + ^IITR125 - 1000) > ^IITR127

            Technical Business Rule Format:
                    ^IITR123 <> NULL AND (ConditionalValue((^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125) > 0, (^IITR157 - ^IITR650 - ^IITR127 + ^IITR123 + ^IITR124 + ^IITR125), 0) + ^IITR201 + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) <= 180000 AND (^IITR123 + ^IITR124 + ^IITR125 - 1000) > ^IITR127

        Data Elements:
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^DDCTNS407 = DDCTNS407
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
    
            ^IITR157 = RP:Income.Taxable.Amount
    
            ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
            ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
            ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
            */
        public static ProcessMessageDocument VR_ATO_IITR_300084(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            bool assertion = (
                report.iitrParentDocument.IITR123 != null && 
                ConditionalValue(
                    report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault() - report.iitrParentDocument.IITR127.GetValueOrDefault() + 
                    report.iitrParentDocument.IITR123.GetValueOrDefault() + report.iitrParentDocument.IITR124.GetValueOrDefault() + report.iitrParentDocument.IITR125.GetValueOrDefault() > 0, 
                    report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault() - report.iitrParentDocument.IITR127.GetValueOrDefault() + 
                    report.iitrParentDocument.IITR123.GetValueOrDefault() + report.iitrParentDocument.IITR124.GetValueOrDefault() + report.iitrParentDocument.IITR125.GetValueOrDefault(), 
                    0
                ) + 
                
                report.iitrParentDocument.IITR201.GetValueOrDefault() + 
                
                ConditionalValue(
                    report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772, 
                    report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault(), 
                    0
                ) + 
                
                report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + sumDDCTNS407 <= 180000 && 
                report.iitrParentDocument.IITR123.GetValueOrDefault() + report.iitrParentDocument.IITR124.GetValueOrDefault() + report.iitrParentDocument.IITR125.GetValueOrDefault() - 1000 > 
                report.iitrParentDocument.IITR127.GetValueOrDefault()
            );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.300084",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total assessable discount amount is incorrect",
                    LongDescription = @"The taxpayer has passed the ESS income test. However, the total assessable discount amount cannot be reduced by more than $1,000.",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DiscountAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.300084" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = report.iitrParentDocument.IITR123.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = report.iitrParentDocument.IITR127.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = report.iitrParentDocument.IITR124.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = report.iitrParentDocument.IITR125.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.600024

        /* VR.ATO.IITR.600024
        Where Deduction claimed (Personal superannuation contributions) and Gifts or donations have created or added to a loss (i.e. after calculating total income less all deductions) before deducting total production losses (PP and non-PP) being claimed for this income year, either partial or full adjustment is required to Deduction claimed and Gifts or donations.  To determine the amount of allowable Deduction claimed and Gifts or donations, subtract all deductions except Deduction claimed, Gifts or donations, and total production losses (PP and non-PP) from Individual total income or loss amount and:  a)  if this result is zero or negative then set Deduction claimed and Gifts or donations amounts to zero; or b) if this result is positive and greater than Deduction claimed and Gifts or donations then allow both amounts in full; or c)  if this result is positive and less than or equal to Personal contributions and gifts then only that amount that reduces Taxable income or loss to zero is allowed (i.e. disallow remainder of Deduction claimed and Gifts or donations that would create a loss).

        Legacy Rule Format:
            ((^IITR157 + ^IITR154 + ^IITR156) < 0) AND ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR148) > 0)

        Technical Business Rule Format:
            ((^IITR157 + ^IITR154 + ^IITR156) < 0) AND ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR148) > 0)

        Data Elements:
    
        ^IITR148 = RP:IncomeTax.Deduction.GiftDonation.Amount
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount
    
        ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount
    
        ^IITR157 = RP:Income.Taxable.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_600024(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            bool assertion = (report.iitrParentDocument.IITR157.GetValueOrDefault() + report.iitrParentDocument.IITR154.GetValueOrDefault() + report.iitrParentDocument.IITR156.GetValueOrDefault() < 0 && sumDDCTNS407 + report.iitrParentDocument.IITR148.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.601024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Deductions claimed amount combined with Gifts or donations amount is invalid",
                    LongDescription = @"When the 'Taxable income or loss' amount, excluding the 'Primary production losses claimed this income year' and 'Non-primary production losses claimed this income year' amounts, results in a loss, the sum of amounts 'Superannuation deduction claimed' and 'Gifts or donations' must not be a positive amount. To determine the amount of allowable 'Superannuation deduction claimed' and 'Gifts or donations', subtract all deduction item amounts from 'Individual total income or loss amount', except 'Superannuation deduction claimed', 'Gifts or donations', 'Primary production losses claimed this income year', and 'Non-primary production losses claimed this income year' and: a) if this result is zero or negative then set both 'Superannuation deduction claimed' and 'Gifts or donations' amounts to zero; or b) if this result is positive and higher than the sum of 'Deduction claimed' and 'Gifts or donations', then allow both amounts in full; or c) if this result is positive and lower than, or the same as, the sum of 'Superannuation deduction claimed' and 'Gifts or donations', then only the amount that reduces 'Taxable income or loss' to zero is allowed (i.e. disallow remainder of 'Superannuation deduction claimed' and 'Gifts or donations' that would create a loss).",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.GiftDonation.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.600024" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR154", Value = report.iitrParentDocument.IITR154.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR156", Value = report.iitrParentDocument.IITR156.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR148", Value = report.iitrParentDocument.IITR148.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.600014

        /* VR.ATO.IITR.600014
        The total of all the deductions fields does not equal the amount at the Total deduction field.

        Legacy Rule Format:
            ((^IITR150 <> NULL) AND (^IITR150 <> (^IITR134 + ^IITR136 + ^IITR137 + ^IITR139 + ^IITR141 + ^IITR145 + ^IITR146 + ^IITR147 + ^IITR148 + ^IITR502 + ^IITR503 + ^IITR504 + ^IITR358 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR366 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS417, 0), 0) + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS138), 0), 0)))) OR ((^IITR150 = NULL) AND ((^IITR134 + ^IITR136 + ^IITR137 + ^IITR139 + ^IITR141 + ^IITR145 + ^IITR146 + ^IITR147 + ^IITR148 + ^IITR502 + ^IITR503 + ^IITR504 + ^IITR358 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR366 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS417, 0), 0) + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS138), 0), 0)) <> 0))

        Technical Business Rule Format:
            ((^IITR150 <> NULL) AND (^IITR150 <> (^IITR134 + ^IITR136 + ^IITR137 + ^IITR139 + ^IITR141 + ^IITR145 + ^IITR146 + ^IITR147 + ^IITR148 + ^IITR502 + ^IITR503 + ^IITR504 + ^IITR358 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR366 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS417, 0), 0) + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS138), 0), 0)))) OR ((^IITR150 = NULL) AND ((^IITR134 + ^IITR136 + ^IITR137 + ^IITR139 + ^IITR141 + ^IITR145 + ^IITR146 + ^IITR147 + ^IITR148 + ^IITR502 + ^IITR503 + ^IITR504 + ^IITR358 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR366 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS417, 0), 0) + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS138), 0), 0)) <> 0))

        Data Elements:
    
        ^IITR150 = RP:Expense.DeductionsTotal.Amount
    
        ^DDCTNS138 = DDCTNS138
    
        ^DDCTNS407 = DDCTNS407
    
        ^DDCTNS417 = DDCTNS417
    
        ^IITR134 = RP:IncomeTax.Deduction.Car.Amount
    
        ^IITR136 = RP:IncomeTax.Deduction.Travel.Amount
    
        ^IITR137 = RP:IncomeTax.Deduction.Clothing.Amount
    
        ^IITR139 = RP:IncomeTax.Deduction.EducationNet.Amount
    
        ^IITR141 = RP:IncomeTax.Deduction.WorkRelatedOther.Amount
    
        ^IITR145 = RP:IncomeTax.Deduction.LowValuePool.Amount
    
        ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount
    
        ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
    
        ^IITR148 = RP:IncomeTax.Deduction.GiftDonation.Amount
    
        ^IITR358 = RP:IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount
    
        ^IITR366 = RP:IncomeTax.Deduction.ProjectPool.Amount
    
        ^IITR368 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
        ^IITR502 = RP:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount
    
        ^IITR503 = RP:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount
    
        ^IITR504 = RP:IncomeTax.Deduction.TaxAffairManagementOther.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_600014(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS138 = SumDDCTNS138(report);
            decimal sumDDCTNS407 = SumDDCTNS407(report);

            decimal valDDCTNS417 = GetDDCTNSValue(report, x => x.DDCTNS417.GetValueOrDefault());

            // Calculate the sum of all deductions
            decimal cal600014_1 = report.iitrParentDocument.IITR134.GetValueOrDefault() + report.iitrParentDocument.IITR136.GetValueOrDefault() + report.iitrParentDocument.IITR137.GetValueOrDefault() + report.iitrParentDocument.IITR139.GetValueOrDefault() + report.iitrParentDocument.IITR141.GetValueOrDefault() + report.iitrParentDocument.IITR145.GetValueOrDefault() + report.iitrParentDocument.IITR146.GetValueOrDefault() + report.iitrParentDocument.IITR147.GetValueOrDefault() + report.iitrParentDocument.IITR148.GetValueOrDefault() + report.iitrParentDocument.IITR502.GetValueOrDefault() + report.iitrParentDocument.IITR503.GetValueOrDefault() + report.iitrParentDocument.IITR504.GetValueOrDefault() + report.iitrParentDocument.IITR358.GetValueOrDefault() + sumDDCTNS407 + report.iitrParentDocument.IITR366.GetValueOrDefault() + report.iitrParentDocument.IITR368.GetValueOrDefault() + valDDCTNS417 + sumDDCTNS138;

            bool assertion = ((report.iitrParentDocument.IITR150 != null && report.iitrParentDocument.IITR150.GetValueOrDefault() != cal600014_1)
                || (report.iitrParentDocument.IITR150 == null && cal600014_1 != 0));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.602014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual total deductions amount is incorrect",
                    LongDescription = @"The 'Individual total deductions amount' must equal the sum of all deductions claimed, which includes 'Work related car expenses', 'Work related travel expenses', 'Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses', 'Work related self-education expenses', 'Other work related expenses', 'Low value pool deduction', 'Interest deductions', 'Dividend deductions', 'Gifts or donations', 'Cost of managing tax affairs - Interest charged by the ATO', 'Cost of managing tax affairs - Litigation costs', 'Cost of managing tax affairs - Other expenses', 'Deductible amount of UPP of foreign pension or annuity', 'Superannuation deduction claimed', 'Individual - Deduction for project pool', 'Forestry managed investment scheme deduction', 'Election expenses amount' and 'Other deductions amount'",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.600014" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR150", Value = report.iitrParentDocument.IITR150.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR134", Value = report.iitrParentDocument.IITR134.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR136", Value = report.iitrParentDocument.IITR136.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR137", Value = report.iitrParentDocument.IITR137.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR139", Value = report.iitrParentDocument.IITR139.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR141", Value = report.iitrParentDocument.IITR141.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR145", Value = report.iitrParentDocument.IITR145.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR146", Value = report.iitrParentDocument.IITR146.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR147", Value = report.iitrParentDocument.IITR147.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR148", Value = report.iitrParentDocument.IITR148.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR502", Value = report.iitrParentDocument.IITR502.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR503", Value = report.iitrParentDocument.IITR503.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR504", Value = report.iitrParentDocument.IITR504.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR358", Value = report.iitrParentDocument.IITR358.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR366", Value = report.iitrParentDocument.IITR366.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR368", Value = report.iitrParentDocument.IITR368.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS417", Value = valDDCTNS417.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS138)", Value = sumDDCTNS138.ToString() });

            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.200034

        /* VR.ATO.IITR.200034
            Spouse's or de facto spouse's family name is present and the taxpayer had a spouse for the full year and had no Medicare levy surcharge. Combined Total reportable fringe benefits amounts exceed $3,772 and taxable income or loss (excluding any assessable First home super saver released amount) plus total reportable fringe benefits amounts plus net amount on which family trust has been paid plus exempt foreign employment income exceeds $27,222 and number of days not liable for MLS, spouse income details also required.

        Legacy Rule Format:
            (^IITR213 <> BLANK OR ^IITR220 = TRUE) AND ^IITR176 = FALSE AND ((^IITR157 - ^IITR650) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue((CountDocument('INCDTLS') = 1) AND (AnyOccurrence(^INCDTLS333, ^INCDTLS333 <> TRUE) AND ^IITR29 <> NULL AND (^IITR29 > ConvertToDate(30,06,(^IITR10 - 55)) OR ^IITR29 <= ConvertToDate(01,07,(^IITR10 - 61)))), Sum(^INCDTLS330), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 27222 AND (^IITR177 = NULL OR (^IITR225 = NULL AND ^IITR226 = NULL AND ^IITR227 = NULL AND ^IITR492 = NULL AND ^IITR493 = NULL AND ^IITR231 = NULL AND ^IITR234 = NULL))

        Technical Business Rule Format:
            (^IITR213 <> BLANK OR ^IITR220 = TRUE) AND ^IITR176 = FALSE AND ((^IITR157 - ^IITR650) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue((CountDocument('INCDTLS') = 1) AND (AnyOccurrence(^INCDTLS333, ^INCDTLS333 <> TRUE) AND ^IITR29 <> NULL AND (^IITR29 > ConvertToDate(30,06,(^IITR10 - 55)) OR ^IITR29 <= ConvertToDate(01,07,(^IITR10 - 61)))), Sum(^INCDTLS330), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 27222 AND (^IITR177 = NULL OR (^IITR225 = NULL AND ^IITR226 = NULL AND ^IITR227 = NULL AND ^IITR492 = NULL AND ^IITR493 = NULL AND ^IITR231 = NULL AND ^IITR234 = NULL))

        Data Elements:
    
            ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count
    
            ^DDCTNS407 = DDCTNS407
    
            ^IITR10 = RP:Report.TargetFinancial.Year
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
    
            ^IITR157 = RP:Income.Taxable.Amount
    
            ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator
    
            ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
            ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
            ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text
    
            ^IITR220 = RP:PersonDemographicDetails.SpouseFullYear.Indicator
    
            ^IITR225 = RP.SPOUSE:Income.Taxable.Amount
    
            ^IITR226 = RP.SPOUSE:IncomeTax.TrustIncomeNotIncludedInTaxableIncome.Amount
    
            ^IITR227 = RP.SPOUSE:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount
    
            ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount
    
            ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount
       
            ^IITR330 = RP:InternationalDealings.ExemptForeignEmploymentIncome.Amount
    
            ^IITR399 = RP:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount
    
            ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
    
            ^INCDTLS330 = INCDTLS330
    
            ^INCDTLS333 = INCDTLS333
        */

        public static ProcessMessageDocument VR_ATO_IITR_200034(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);
            decimal sumIITR157MinusIITR650 = report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault();
            bool incdtlsChildCountisOne = (report.incdtlsChildDocuments != null && report.incdtlsChildDocuments.Count == 1);

            List<INCDTLS2026.Rp_Income> slspTuples = new List<INCDTLS2026.Rp_Income>();

            if (Count(report.incdtlsChildDocuments) != 0 && report.incdtlsChildDocuments[0].Rp_IncomeCollection != null)
                slspTuples = report.incdtlsChildDocuments[0].Rp_IncomeCollection;

            bool assertion = ((string.IsNullOrWhiteSpace(report.iitrParentDocument.IITR213) != true || report.iitrParentDocument.IITR220 == true) && report.iitrParentDocument.IITR176 == false && (sumIITR157MinusIITR650 + ConditionalValue(report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772, report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault(), 0) + ConditionalValue(sumIITR157MinusIITR650 > 0, report.iitrParentDocument.IITR330.GetValueOrDefault(), 0) + report.iitrParentDocument.IITR399.GetValueOrDefault() - ConditionalValue(incdtlsChildCountisOne && slspTuples.Any(tuple => tuple.INCDTLS333 != true) && report.iitrParentDocument.IITR29 != null && (report.iitrParentDocument.IITR29.GetValueOrDefault() > ConvertToDate(30, 6, report.iitrParentDocument.IITR10.GetValueOrDefault() - 55) || report.iitrParentDocument.IITR29.GetValueOrDefault() <= ConvertToDate(1, 7, report.iitrParentDocument.IITR10.GetValueOrDefault() - 61)), slspTuples.Sum(tuple => tuple.INCDTLS330.GetValueOrDefault()), 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407) > SingleMedicareLevy && (report.iitrParentDocument.IITR177 == null || (report.iitrParentDocument.IITR225 == null && report.iitrParentDocument.IITR226 == null && report.iitrParentDocument.IITR227 == null && report.iitrParentDocument.IITR492 == null && report.iitrParentDocument.IITR493 == null && report.iitrParentDocument.IITR231 == null && report.iitrParentDocument.IITR234 == null )));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.208034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of days not liable for surcharge or spouse income details is required",
                    LongDescription = @"Spouse's or de facto spouse's family name is present and the taxpayer had a spouse for the full year and had no Medicare levy surcharge. Combined Total reportable fringe benefits amounts exceed $3,772 and taxable income or loss (excluding any assessable First home super saver released amount) plus total reportable fringe benefits amounts plus net amount on which family trust has been paid plus exempt foreign employment income exceeds $27,222 and number of days not liable for MLS, spouse income details also required.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count",
                    Parameters = new ProcessMessageParameters() {   new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200034" },
                        new ProcessMessageParameter { Name = "IITR213", Value = report.iitrParentDocument.IITR213 },
                        new ProcessMessageParameter { Name = "IITR220", Value = GetValueOrEmpty(report.iitrParentDocument.IITR220) },
                        new ProcessMessageParameter { Name = "IITR176", Value = GetValueOrEmpty(report.iitrParentDocument.IITR176) },
                        new ProcessMessageParameter { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR330", Value = report.iitrParentDocument.IITR330.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR399", Value = report.iitrParentDocument.IITR399.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "INCDTLS333", Value = slspTuples.Any(tuple => tuple.INCDTLS333 != true).ToString() },
                        new ProcessMessageParameter { Name = "IITR29", Value = GetValueOrEmpty(report.iitrParentDocument.IITR29) },
                        new ProcessMessageParameter { Name = "IITR10", Value = report.iitrParentDocument.IITR10.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "INCDTLS330", Value = slspTuples.Sum(tuple => tuple.INCDTLS330.GetValueOrDefault()).ToString() },
                        new ProcessMessageParameter { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "SUM(DDCTNS407)", Value = sumDDCTNS407.ToString() },
                        new ProcessMessageParameter { Name = "IITR177", Value = report.iitrParentDocument.IITR177.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR225", Value = report.iitrParentDocument.IITR225.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR226", Value = report.iitrParentDocument.IITR226.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR227", Value = report.iitrParentDocument.IITR227.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR492", Value = report.iitrParentDocument.IITR492.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR493", Value = report.iitrParentDocument.IITR493.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR231", Value = report.iitrParentDocument.IITR231.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IITR234", Value = report.iitrParentDocument.IITR234.GetValueOrDefault().ToString() },
                        }
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000013

        /* VR.ATO.IITR.000013
        Where the number of days not liable for medicare levy surcharge is less than the whole reporting period, and either income test is triggered; and the reportable employer superannuation contributions is greater than 8% of all income from salary and wages, check reportable employer superannuation contributions as MLS liability may be incurred.

        Legacy Rule Format:
            (^IITR177 = NULL OR ^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (((^IITR213 <> BLANK OR ^IITR171 <> NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum((ConditionalValue(CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, (^IITR492 + ^IITR493), 0) + ^IITR227 + ^IITR234 + ^IITR231) > (194000 + ConditionalValue(^IITR171 > 1, ((1500 * ^IITR171) - 1), 0)))) OR ((^IITR213 = BLANK) AND (^IITR171 = NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum(ConditionalValue((CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 97000))) AND (^IITR201 > (^IITR601 * 0.08))

        Technical Business Rule Format:
            (^IITR177 = NULL OR ^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (((^IITR213 <> BLANK OR ^IITR171 <> NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum((ConditionalValue(CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, (^IITR492 + ^IITR493), 0) + ^IITR227 + ^IITR234 + ^IITR231) > (194000 + ConditionalValue(^IITR171 > 1, ((1500 * ^IITR171) - 1), 0)))) OR ((^IITR213 = BLANK) AND (^IITR171 = NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum(ConditionalValue((CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 97000))) AND (^IITR201 > (^IITR601 * 0.08))

        Data Elements:

        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

        ^DDCTNS407 = DDCTNS407

        ^EndDate = EndDate

        ^IITR10 = RP:Report.TargetFinancial.Year

        ^IITR29 = RP:PersonDemographicDetails.Birth.Date

        ^IITR157 = RP:Income.Taxable.Amount

        ^IITR171 = RP:IncomeTax.MedicareLevy.DependentChildrenAndStudents.Count

        ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count

        ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

        ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text

        ^IITR225 = RP.SPOUSE:Income.Taxable.Amount

        ^IITR227 = RP.SPOUSE:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount

        ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount

        ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount

        ^IITR330 = RP:InternationalDealings.ExemptForeignEmploymentIncome.Amount

        ^IITR399 = RP:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount

        ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

        ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

        ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount

        ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount

        ^IITR601 = RP:Income.SalaryOrWages.Amount

        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount

        ^INCDTLS330 = INCDTLS330

        ^INCDTLS333 = INCDTLS333

        ^StartDate = StartDate
        */

        public static ProcessMessageDocument VR_ATO_IITR_000013(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);
            decimal sumIITR157MinusIITR650 = report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault();
            bool incdtlsChildCountisOne = (report.incdtlsChildDocuments != null && report.incdtlsChildDocuments.Count == 1);
            List<INCDTLS2026.Rp_Income> incomeTuples = new List<INCDTLS2026.Rp_Income>();

            if (Count(report.incdtlsChildDocuments) != 0 && report.incdtlsChildDocuments[0].Rp_IncomeCollection != null)
                incomeTuples = report.incdtlsChildDocuments[0].Rp_IncomeCollection;

            decimal incdtls330ConditionalSum = (ConditionalValue(incdtlsChildCountisOne, incomeTuples.Where(tuple => tuple.INCDTLS333 != true).Sum(tuple => tuple.INCDTLS330), 0));

            bool assertion = ((report.iitrParentDocument.IITR177 == null || Convert.ToDouble(report.iitrParentDocument.IITR177.GetValueOrDefault()) < (report.iitrParentDocument.RPEndDate - report.iitrParentDocument.RPStartDate).GetValueOrDefault().TotalDays + 1) && (((!string.IsNullOrWhiteSpace(report.iitrParentDocument.IITR213) || report.iitrParentDocument.IITR171 != null) && ((sumIITR157MinusIITR650 > 0 ? sumIITR157MinusIITR650 : 0) + ((report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault()) > 3772 ? (report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault()) : 0) + (sumIITR157MinusIITR650 > 0 ? report.iitrParentDocument.IITR330.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR399.GetValueOrDefault() - (report.iitrParentDocument.IITR29 > ConvertToDate(30, 6, Convert.ToInt32(report.iitrParentDocument.IITR10.GetValueOrDefault()) - 55) || report.iitrParentDocument.IITR29 <= ConvertToDate(1, 7, Convert.ToInt32(report.iitrParentDocument.IITR10.GetValueOrDefault()) - 61) ? incdtls330ConditionalSum : 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 + report.iitrParentDocument.IITR225.GetValueOrDefault() + (report.iitrParentDocument.IITR492.GetValueOrDefault() + report.iitrParentDocument.IITR493.GetValueOrDefault() > 3772 ? report.iitrParentDocument.IITR492.GetValueOrDefault() + report.iitrParentDocument.IITR493.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR227.GetValueOrDefault() + report.iitrParentDocument.IITR234.GetValueOrDefault() + report.iitrParentDocument.IITR231.GetValueOrDefault()) > (TierOneFamiliesThreshold + (report.iitrParentDocument.IITR171.GetValueOrDefault() > 1 ? 1500 * report.iitrParentDocument.IITR171.GetValueOrDefault() - 1 : 0))) || (string.IsNullOrWhiteSpace(report.iitrParentDocument.IITR213) && report.iitrParentDocument.IITR171 == null && ((sumIITR157MinusIITR650 > 0 ? sumIITR157MinusIITR650 : 0) + (report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772 ? report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() : 0) + (sumIITR157MinusIITR650 > 0 ? report.iitrParentDocument.IITR330.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR399.GetValueOrDefault() - (report.iitrParentDocument.IITR29 > ConvertToDate(30, 6, Convert.ToInt32(report.iitrParentDocument.IITR10) - 55) || report.iitrParentDocument.IITR29 <= ConvertToDate(1, 7, Convert.ToInt32(report.iitrParentDocument.IITR10) - 61) ? incdtls330ConditionalSum : 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407) > TierOneSinglesThreshold)) && (Convert.ToDouble(report.iitrParentDocument.IITR201.GetValueOrDefault()) > Convert.ToDouble(report.iitrParentDocument.IITR601) * 0.08));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000013",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check reportable employer superannuation contributions as MLS liability may be incurred",
                    LongDescription = @"Check that compulsory super contributions have not been included in reportable employer superannuation contributions at the income test section. Some reportable employer superannuation contributions are incorrectly reported on payment summaries, these amounts are generally salary sacrifice super contributions only. Incorrect reporting may cause a MLS liability.",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerReportable.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000013" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR177", Value = report.iitrParentDocument.IITR177.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR213", Value = report.iitrParentDocument.IITR213 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR171", Value = report.iitrParentDocument.IITR171.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR330", Value = report.iitrParentDocument.IITR330.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR399", Value = report.iitrParentDocument.IITR399.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmpty(report.iitrParentDocument.IITR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR10", Value = report.iitrParentDocument.IITR10.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of INCDTLS330 where INCDTLS333 = false", Value = incdtls330ConditionalSum.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR225", Value = report.iitrParentDocument.IITR225.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR492", Value = report.iitrParentDocument.IITR492.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR493", Value = report.iitrParentDocument.IITR493.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR227", Value = report.iitrParentDocument.IITR227.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR234", Value = report.iitrParentDocument.IITR234.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR231", Value = report.iitrParentDocument.IITR231.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR601", Value = report.iitrParentDocument.IITR601.GetValueOrDefault().ToString() });
            }
            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.200015

        /* VR.ATO.IITR.200015
            Where the 'Seniors and pensioners - Tax offset code' is being claimed, and the sum of income fields is greater than $87,620 where there is a spouse, or $52,759 where there is no spouse; and the Reportable employer superannuation contributions is greater than 8% of all income from salary and wages, check Reportable employer superannuation contributions as SPO entitlement may be impacted.
    
            Legacy Rule Format:
            ^IITR163 <> BLANK AND ((^IITR213 <> BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205 + ^IITR225 + ^IITR230 + ^IITR226 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR231 + ^IITR234) > 87620) OR (^IITR213 = BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205) > 52759)) AND ^IITR201 > (^IITR601 * 0.08)

            Technical Business Rule Format:
            ^IITR163 <> BLANK AND ((^IITR213 <> BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205 + ^IITR225 + ^IITR230 + ^IITR226 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR231 + ^IITR234) > 87620) OR (^IITR213 = BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205) > 52759)) AND ^IITR201 > (^IITR601 * 0.08)
    
            Data Elements:
    
            ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
            ^DDCTNS407 = DDCTNS407
    
            ^IITR157 = RP:Income.Taxable.Amount
    
            ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code
    
            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
            ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
            ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text
    
            ^IITR225 = RP.SPOUSE:Income.Taxable.Amount
    
            ^IITR226 = RP.SPOUSE:IncomeTax.TrustIncomeNotIncludedInTaxableIncome.Amount
    
            ^IITR230 = RP.SPOUSE:Pension.SpouseExempt.Amount
    
            ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount
    
            ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount
    
            ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount
    
            ^IITR601 = RP:Income.SalaryOrWages.Amount
    
            ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
        */

        public static ProcessMessageDocument VR_ATO_IITR_200015(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            decimal sumIITR157MinusIITR650 = report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault();

            bool assertion = (string.IsNullOrWhiteSpace(report.iitrParentDocument.IITR163) != true && (string.IsNullOrWhiteSpace(report.iitrParentDocument.IITR213) != true && ConditionalValue(sumIITR157MinusIITR650 > 0, sumIITR157MinusIITR650, 0) + ConditionalValue(report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772, report.iitrParentDocument.IITR490.GetValueOrDefault() * 0.530M + report.iitrParentDocument.IITR491.GetValueOrDefault(), 0) + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR225.GetValueOrDefault() + report.iitrParentDocument.IITR230.GetValueOrDefault() + report.iitrParentDocument.IITR226.GetValueOrDefault() + ConditionalValue(report.iitrParentDocument.IITR492.GetValueOrDefault() + report.iitrParentDocument.IITR493.GetValueOrDefault() > 3772, report.iitrParentDocument.IITR492.GetValueOrDefault() * 0.530M + report.iitrParentDocument.IITR493.GetValueOrDefault(), 0) + report.iitrParentDocument.IITR231.GetValueOrDefault() + report.iitrParentDocument.IITR234.GetValueOrDefault() > 87620 || string.IsNullOrWhiteSpace(report.iitrParentDocument.IITR213) == true && ConditionalValue(sumIITR157MinusIITR650 > 0, sumIITR157MinusIITR650, 0) + ConditionalValue(report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772, report.iitrParentDocument.IITR490.GetValueOrDefault() * 0.530M + report.iitrParentDocument.IITR491.GetValueOrDefault(), 0) + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() > 52759) && report.iitrParentDocument.IITR201.GetValueOrDefault() > (report.iitrParentDocument.IITR601.GetValueOrDefault()) * 0.08M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200015",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check reportable employer superannuation contributions as SPO entitlement may be impacted",
                    LongDescription = @"Check that compulsory super contributions have not been included in reportable employer superannuation contributions at the income test section. Some reportable employer superannuation contributions are incorrectly reported on payment summaries, these amounts are generally salary sacrifice super contributions only.  Incorrect reporting will result in reduced Senior and pensioner offset entitlement.",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerReportable.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200015" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR163", Value = report.iitrParentDocument.IITR163 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR213", Value = report.iitrParentDocument.IITR213 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR225", Value = report.iitrParentDocument.IITR225.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR230", Value = report.iitrParentDocument.IITR230.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR226", Value = report.iitrParentDocument.IITR226.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR492", Value = report.iitrParentDocument.IITR492.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR493", Value = report.iitrParentDocument.IITR493.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR231", Value = report.iitrParentDocument.IITR231.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR234", Value = report.iitrParentDocument.IITR234.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR601", Value = report.iitrParentDocument.IITR601.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.100043

        /* VR.ATO.IITR.100043
        Adjusted taxable income must equal the sum of (when amount above $0): Taxable income or loss income (excluding any assessable First home super saver released amount), Reportable employer superannuation contributions, Tax-free government pensions, Target foreign income, Net financial investment loss, Net rental property loss, and all Deduction claimed amounts, plus Total reportable fringe benefits amount from employers not exempt under section 57A and 53.0% of Total reportable fringe benefits amount from employers exempt under section 57A (when combined Total reportable fringe benefits amount above $3,772), less Child support you paid. If you are completing a tax return for a deceased person, this value must be divided by the number of days the taxpayer was alive during the financial year and multiplied by the total number of days in the financial year.

        Legacy Rule Format:
            ((^IITR28 = NULL OR ^IITR28 > ^RP:EndDate) AND ((^IITR208 <> NULL AND ^IITR208 <> ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1)) OR (^IITR208 = NULL AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1) <> 0))) OR ((^IITR28 > (^RP:StartDate - 1) AND ^IITR28 < (^RP:EndDate + 1)) AND ((^IITR208 <> NULL AND ^IITR208 <> (((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1)) OR (^IITR208 = NULL AND ((((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1) <> 0)))) OR (^IITR28 < ^RP:StartDate AND ^IITR208 <> 0)

        Technical Business Rule Format:
            ((^IITR28 = NULL OR ^IITR28 > ^RP:EndDate) AND ((^IITR208 <> NULL AND ^IITR208 <> ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1)) OR (^IITR208 = NULL AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1) <> 0))) OR ((^IITR28 > (^RP:StartDate - 1) AND ^IITR28 < (^RP:EndDate + 1)) AND ((^IITR208 <> NULL AND ^IITR208 <> (((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1)) OR (^IITR208 = NULL AND ((((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1) <> 0)))) OR (^IITR28 < ^RP:StartDate AND ^IITR208 <> 0)

        Data Elements:
    
        ^IITR208 = RP:Income.TaxableAdjusted.Amount
    
        ^DDCTNS407 = DDCTNS407
    
        ^EndDate = EndDate
    
        ^IITR28 = RP:PersonDemographicDetails.Death.Date
    
        ^IITR157 = RP:Income.Taxable.Amount
    
        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
        ^IITR202 = RP:Pension.TaxFree.Amount
    
        ^IITR203 = RP:InternationalDealings.ForeignIncomeTarget.Amount
    
        ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
        ^IITR206 = RP:ChildSupport.Payment.Amount
    
        ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
    
        ^StartDate = StartDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_100043(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            decimal calc1_100043 = 
                (report.iitrParentDocument.IITR157.GetValueOrDefault() - report.IITR.IITR650.GetValueOrDefault() > 0 ? report.iitrParentDocument.IITR157.GetValueOrDefault() - report.IITR.IITR650.GetValueOrDefault() : 0) + 
                (report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772 ? Math.Floor(report.iitrParentDocument.IITR490.GetValueOrDefault() * 0.53m) + report.iitrParentDocument.IITR491.GetValueOrDefault() : 0) + 
                report.iitrParentDocument.IITR201.GetValueOrDefault() + report.iitrParentDocument.IITR202.GetValueOrDefault() + report.iitrParentDocument.IITR203.GetValueOrDefault() + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + sumDDCTNS407 - report.iitrParentDocument.IITR206.GetValueOrDefault();

            decimal calc2_100043 = (report.iitrParentDocument.IITR28 != null ? calc1_100043 * Convert.ToDecimal(((report.iitrParentDocument.RPEndDate - report.iitrParentDocument.RPStartDate).Value.TotalDays + 1) / ((Convert.ToDateTime(report.iitrParentDocument.IITR28) - report.iitrParentDocument.RPStartDate).Value.TotalDays + 1)) : 0);

            bool assertion = (((report.iitrParentDocument.IITR28 == null || Convert.ToDateTime(report.iitrParentDocument.IITR28) > report.iitrParentDocument.RPEndDate) && (((report.iitrParentDocument.IITR208 != null && report.iitrParentDocument.IITR208 < (calc1_100043 - 1) || report.iitrParentDocument.IITR208 > (calc1_100043 + 1))) || (report.iitrParentDocument.IITR208 == null && (0 < (calc1_100043 - 1) || 0 > (calc1_100043 + 1))))) || (report.iitrParentDocument.IITR28 != null ? ((Convert.ToDateTime(report.iitrParentDocument.IITR28) > report.iitrParentDocument.RPStartDate.Value.AddDays(-1) && Convert.ToDateTime(report.iitrParentDocument.IITR28) < report.iitrParentDocument.RPEndDate.Value.AddDays(1)) && ((report.iitrParentDocument.IITR208 != null && (report.iitrParentDocument.IITR208.GetValueOrDefault() < (calc2_100043 - 1) || report.iitrParentDocument.IITR208.GetValueOrDefault() > (calc2_100043 + 1))) || report.iitrParentDocument.IITR208 == null && (0 < (calc2_100043 - 1) || 0 > (calc2_100043 + 1)))) : false) || ((report.iitrParentDocument.IITR28 != null ? report.iitrParentDocument.IITR28 < report.iitrParentDocument.RPStartDate : false) && (report.iitrParentDocument.IITR208 != null ? report.iitrParentDocument.IITR208 : 0) != 0));
            
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.024015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Adjusted taxable income is incorrect",
                    LongDescription = @"Adjusted taxable income must equal the sum of (when amount above $0): Taxable income or loss income (excluding any assessable First home super saver released amount), Reportable employer superannuation contributions, Tax-free government pensions, Target foreign income, Net financial investment loss, Net rental property loss, and all Deduction claimed amounts, plus Total reportable fringe benefits amount from employers not exempt under section 57A and 53.0% of Total reportable fringe benefits amount from employers exempt under section 57A (when combined Total reportable fringe benefits amount above $3,772), less Child support you paid. If you are completing a tax return for a deceased person, this value must be divided by the number of days the taxpayer was alive during the financial year and multiplied by the total number of days in the financial year.",
                    Location = "/xbrli:xbrl/tns:Income.TaxableAdjusted.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100043" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR28", Value = GetValueOrEmpty(report.iitrParentDocument.IITR28) });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR202", Value = report.iitrParentDocument.IITR202.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR203", Value = report.iitrParentDocument.IITR203.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR206", Value = report.iitrParentDocument.IITR206.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR208", Value = report.iitrParentDocument.IITR208.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "StartDate", Value = GetValueOrEmpty(report.iitrParentDocument.RPStartDate) });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "EndDate", Value = GetValueOrEmpty(report.iitrParentDocument.RPEndDate) });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.100031

        /* VR.ATO.IITR.100031
        For a resident taxpayer, where the sum of taxable income or loss and the net loss amounts that exceeds or equal to zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

        Legacy Rule Format:
            ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0))) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Technical Business Rule Format:
            ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0))) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Data Elements:
    
        ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
        ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount
    
        ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount
    
        ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount
    
        ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount
    
        ^IITR157 = RP:Income.Taxable.Amount
    
        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
        ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
        ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code
    
        ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_100031(IITR2026Crossform report, IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            List<IITR2026.RPLOSSSeqNumContext> lossContextsIITR482_100031 = (report.iitrParentDocument.RPLOSSSeqNumContextCollection.FindAll(f => !IsMatch(f.IITR482, "^(0|5|8)$")));

            bool assertion = (report.iitrParentDocument.IITR19 == true && IsMatch(rplossSeqNumContext.IITR482, "^([1-4])$") && ((report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault()) + (lossContextsIITR482_100031.Count > 0 ? lossContextsIITR482_100031.Sum(f => f.IITR487.GetValueOrDefault()) : 0)) >= 0 && ((report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault()) + (lossContextsIITR482_100031.Count > 0 ? lossContextsIITR482_100031.Sum(f => f.IITR487.GetValueOrDefault()) : 0) + (report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772 ? report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 - (report.iitrParentDocument.IITR153.GetValueOrDefault() - report.iitrParentDocument.IITR154.GetValueOrDefault()) - (report.iitrParentDocument.IITR155.GetValueOrDefault() - report.iitrParentDocument.IITR156.GetValueOrDefault())) >= 250000);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The income requirement for non-commercial losses has not been met",
                    LongDescription = @"For a resident taxpayer, where the sum of taxable income or loss and the net loss amounts that exceeds or equal to zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.iitrParentDocument.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "IITR482",
                    Value = rplossSeqNumContext.IITR482
                });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR487", Value = rplossSeqNumContext.IITR487.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR153", Value = report.iitrParentDocument.IITR153.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR154", Value = report.iitrParentDocument.IITR154.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR155", Value = report.iitrParentDocument.IITR155.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR156", Value = report.iitrParentDocument.IITR156.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.100032

        /* VR.ATO.IITR.100032
        For a resident taxpayer where the sum of taxable income or loss and the net loss amounts is less than zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

        Legacy Rule Format:
            ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0))) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Technical Business Rule Format:
            ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0))) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Data Elements:
    
        ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
        ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount
    
        ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount
    
        ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount
    
        ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount
    
        ^IITR157 = RP:Income.Taxable.Amount
    
        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
        ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
        ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code
    
        ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_100032(IITR2026Crossform report, IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            List<IITR2026.RPLOSSSeqNumContext> lossContextsIITR482_100032 = (report.iitrParentDocument.RPLOSSSeqNumContextCollection != null ? report.iitrParentDocument.RPLOSSSeqNumContextCollection.FindAll(f => !IsMatch(f.IITR482, "^(0|8)$")) : new List<IITR2026.RPLOSSSeqNumContext>());

            bool assertion = (report.iitrParentDocument.IITR19 == true && IsMatch(rplossSeqNumContext.IITR482, "^([1-4])$") && ((report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault()) + (lossContextsIITR482_100032.Count > 0 ? lossContextsIITR482_100032.Sum(f => f.IITR487.GetValueOrDefault()) : 0)) < 0 && ((report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772 ? report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 - (report.iitrParentDocument.IITR153.GetValueOrDefault() - report.iitrParentDocument.IITR154.GetValueOrDefault()) - (report.iitrParentDocument.IITR155.GetValueOrDefault() - report.iitrParentDocument.IITR156.GetValueOrDefault())) >= 250000);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100032",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The income requirement for non-commercial losses has not been met",
                    LongDescription = @"For a resident taxpayer where the sum of taxable income or loss and the net loss amounts is less than zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100032" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.iitrParentDocument.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "IITR482",
                    Value = rplossSeqNumContext.IITR482
                });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR487", Value = rplossSeqNumContext.IITR487.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR153", Value = report.iitrParentDocument.IITR153.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR154", Value = report.iitrParentDocument.IITR154.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR155", Value = report.iitrParentDocument.IITR155.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR156", Value = report.iitrParentDocument.IITR156.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.100033

        /* VR.ATO.IITR.100033
        For a non-resident taxpayer where the sum of taxable income or loss and the net loss amounts exceeds or equal to zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

        Legacy Rule Format:
            ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328 - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Technical Business Rule Format:
            ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328 - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Data Elements:
    
        ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
        ^IITR115 = RP:Income.Interest.Gross.Amount
    
        ^IITR118 = RP:Income.DividendsUnfranked.Amount
    
        ^IITR119 = RP:Income.DividendsFranked.Amount
    
        ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
        ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount
    
        ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
    
        ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount
    
        ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount
    
        ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount
    
        ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount
    
        ^IITR157 = RP:Income.Taxable.Amount
    
        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
        ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
        ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount
    
        ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code
    
        ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_100033(IITR2026Crossform report, IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            List<IITR2026.RPLOSSSeqNumContext> lossContextsIITR482_100033 = (report.iitrParentDocument.RPLOSSSeqNumContextCollection != null ? report.iitrParentDocument.RPLOSSSeqNumContextCollection.FindAll(f => !IsMatch(f.IITR482, "^(0|8)$")) : new List<IITR2026.RPLOSSSeqNumContext>());

            bool assertion = (report.iitrParentDocument.IITR19 == false && IsMatch(rplossSeqNumContext.IITR482, "^([1-4])$") && ((report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault()) + (lossContextsIITR482_100033.Count > 0 ? lossContextsIITR482_100033.Sum(f => f.IITR487.GetValueOrDefault()) : 0) + report.iitrParentDocument.IITR146.GetValueOrDefault() + report.iitrParentDocument.IITR147.GetValueOrDefault() - report.iitrParentDocument.IITR115.GetValueOrDefault() - report.iitrParentDocument.IITR118.GetValueOrDefault() - report.iitrParentDocument.IITR119.GetValueOrDefault() - report.iitrParentDocument.IITR120.GetValueOrDefault() - report.iitrParentDocument.IITR328.GetValueOrDefault()) >= 0 && ((report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault()) + (lossContextsIITR482_100033.Count > 0 ? lossContextsIITR482_100033.Sum(f => f.IITR487.GetValueOrDefault()) : 0) + (report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772 ? report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 + report.iitrParentDocument.IITR146.GetValueOrDefault() + report.iitrParentDocument.IITR147.GetValueOrDefault() - report.iitrParentDocument.IITR115.GetValueOrDefault() - report.iitrParentDocument.IITR118.GetValueOrDefault() - report.iitrParentDocument.IITR119.GetValueOrDefault() - report.iitrParentDocument.IITR120.GetValueOrDefault() - report.iitrParentDocument.IITR328.GetValueOrDefault() - (report.iitrParentDocument.IITR153.GetValueOrDefault() - report.iitrParentDocument.IITR154.GetValueOrDefault()) - (report.iitrParentDocument.IITR155.GetValueOrDefault() - report.iitrParentDocument.IITR156.GetValueOrDefault())) >= 250000);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The income requirement for non-commercial losses has not been met",
                    LongDescription = @"For a non-resident taxpayer where the sum of taxable income or loss and the net loss amounts exceeds or equal to zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100033" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.iitrParentDocument.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "IITR482",
                    Value = rplossSeqNumContext.IITR482
                });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR487", Value = rplossSeqNumContext.IITR487.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR146", Value = report.iitrParentDocument.IITR146.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR147", Value = report.iitrParentDocument.IITR147.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR115", Value = report.iitrParentDocument.IITR115.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = report.iitrParentDocument.IITR118.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = report.iitrParentDocument.IITR119.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = report.iitrParentDocument.IITR120.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR328", Value = report.iitrParentDocument.IITR328.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR153", Value = report.iitrParentDocument.IITR153.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR154", Value = report.iitrParentDocument.IITR154.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR155", Value = report.iitrParentDocument.IITR155.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR156", Value = report.iitrParentDocument.IITR156.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.100034

        /* VR.ATO.IITR.100034
        For a non-resident taxpayer where the sum of taxable income or loss and the net loss amounts is less than zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

        Legacy Rule Format:
            ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Technical Business Rule Format:
            ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

        Data Elements:
    
        ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
        ^IITR115 = RP:Income.Interest.Gross.Amount
    
        ^IITR118 = RP:Income.DividendsUnfranked.Amount
    
        ^IITR119 = RP:Income.DividendsFranked.Amount
    
        ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
        ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount
    
        ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
    
        ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount
    
        ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount
    
        ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount
    
        ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount
    
        ^IITR157 = RP:Income.Taxable.Amount
    
        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
    
        ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount
    
        ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount
    
        ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount
    
        ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code
    
        ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount
    
        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_100034(IITR2026Crossform report, IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            List<IITR2026.RPLOSSSeqNumContext> lossContextsIITR482_100034 = (report.iitrParentDocument.RPLOSSSeqNumContextCollection != null ? report.iitrParentDocument.RPLOSSSeqNumContextCollection.FindAll(f => !IsMatch(f.IITR482, "^(0|8)$")) : new List<IITR2026.RPLOSSSeqNumContext>());

            bool assertion = (report.iitrParentDocument.IITR19 == false && IsMatch(rplossSeqNumContext.IITR482, "^([1-4])$") && ((report.iitrParentDocument.IITR157.GetValueOrDefault() - report.iitrParentDocument.IITR650.GetValueOrDefault()) + (lossContextsIITR482_100034.Count > 0 ? lossContextsIITR482_100034.Sum(f => f.IITR487.GetValueOrDefault()) : 0) + report.iitrParentDocument.IITR146.GetValueOrDefault() + report.iitrParentDocument.IITR147.GetValueOrDefault() - report.iitrParentDocument.IITR115.GetValueOrDefault() - report.iitrParentDocument.IITR118.GetValueOrDefault() - report.iitrParentDocument.IITR119.GetValueOrDefault() - report.iitrParentDocument.IITR120.GetValueOrDefault() - report.iitrParentDocument.IITR328.GetValueOrDefault()) < 0 && ((report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() > 3772 ? report.iitrParentDocument.IITR490.GetValueOrDefault() + report.iitrParentDocument.IITR491.GetValueOrDefault() : 0) + report.iitrParentDocument.IITR204.GetValueOrDefault() + report.iitrParentDocument.IITR205.GetValueOrDefault() + report.iitrParentDocument.IITR201.GetValueOrDefault() + sumDDCTNS407 - (report.iitrParentDocument.IITR153.GetValueOrDefault() - report.iitrParentDocument.IITR154.GetValueOrDefault()) - (report.iitrParentDocument.IITR155.GetValueOrDefault() - report.iitrParentDocument.IITR156.GetValueOrDefault())) >= 250000);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The income requirement for non-commercial losses has not been met",
                    LongDescription = @"For a non-resident taxpayer where the sum of taxable income or loss and the net loss amounts is less than zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100034" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.iitrParentDocument.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "IITR482",
                    Value = rplossSeqNumContext.IITR482
                });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = report.iitrParentDocument.IITR157.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = report.iitrParentDocument.IITR650.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR487", Value = rplossSeqNumContext.IITR487.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR146", Value = report.iitrParentDocument.IITR146.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR147", Value = report.iitrParentDocument.IITR147.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR115", Value = report.iitrParentDocument.IITR115.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = report.iitrParentDocument.IITR118.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = report.iitrParentDocument.IITR119.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = report.iitrParentDocument.IITR120.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR328", Value = report.iitrParentDocument.IITR328.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR490", Value = report.iitrParentDocument.IITR490.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR491", Value = report.iitrParentDocument.IITR491.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR204", Value = report.iitrParentDocument.IITR204.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR205", Value = report.iitrParentDocument.IITR205.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = report.iitrParentDocument.IITR201.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR153", Value = report.iitrParentDocument.IITR153.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR154", Value = report.iitrParentDocument.IITR154.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR155", Value = report.iitrParentDocument.IITR155.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR156", Value = report.iitrParentDocument.IITR156.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730107

        /* VR.ATO.IITR.730107
        When any total deduction amounts above zero are present, the details must be provided in the Deductions schedule

        Legacy Rule Format:
            (^IITR134 > 0 OR ^IITR136 > 0 OR ^IITR137 > 0 OR ^IITR139 > 0 OR ^IITR141 > 0 OR ^IITR145 > 0 OR ^IITR146 > 0 OR ^IITR147 > 0 OR ^IITR148 > 0 OR ^IITR502 > 0 OR ^IITR503 > 0 OR ^IITR504 > 0 OR ^IITR366 > 0 OR ^IITR368 > 0) AND CountDocument('DDCTNS') = 0

        Technical Business Rule Format:
            (^IITR134 > 0 OR ^IITR136 > 0 OR ^IITR137 > 0 OR ^IITR139 > 0 OR ^IITR141 > 0 OR ^IITR145 > 0 OR ^IITR146 > 0 OR ^IITR147 > 0 OR ^IITR148 > 0 OR ^IITR502 > 0 OR ^IITR503 > 0 OR ^IITR504 > 0 OR ^IITR366 > 0 OR ^IITR368 > 0) AND CountDocument('DDCTNS') = 0

        Data Elements:
    
        ^IITR134 = RP:IncomeTax.Deduction.Car.Amount
    
        ^IITR136 = RP:IncomeTax.Deduction.Travel.Amount
    
        ^IITR137 = RP:IncomeTax.Deduction.Clothing.Amount
    
        ^IITR139 = RP:IncomeTax.Deduction.EducationNet.Amount
    
        ^IITR141 = RP:IncomeTax.Deduction.WorkRelatedOther.Amount
    
        ^IITR145 = RP:IncomeTax.Deduction.LowValuePool.Amount
    
        ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount
    
        ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
    
        ^IITR148 = RP:IncomeTax.Deduction.GiftDonation.Amount
    
        ^IITR366 = RP:IncomeTax.Deduction.ProjectPool.Amount
    
        ^IITR368 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
        ^IITR502 = RP:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount
    
        ^IITR503 = RP:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount
    
        ^IITR504 = RP:IncomeTax.Deduction.TaxAffairManagementOther.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730107(IITR2026Crossform report, int ddctnsScheduleCount)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR134.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR136.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR137.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR139.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR141.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR145.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR146.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR147.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR148.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR502.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR503.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR504.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR366.GetValueOrDefault() > 0 || report.iitrParentDocument.IITR368.GetValueOrDefault() > 0) && ddctnsScheduleCount == 0;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730107",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Deductions schedule must be provided",
                    LongDescription = @"When any total deduction amounts above zero are present, the details must be provided in the Deductions schedule",
                    Location = "/xbrli:xbrl/xbrli:context",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730107" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.600031

        /* VR.ATO.IITR.600031
        The work related self-education expenses action code is present and the amount of work related self-education expenses is missing.

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^DDCTNS200 <> NULL AND ^IITR139 = NULL

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^DDCTNS200 <> NULL AND ^IITR139 = NULL

        Data Elements:
    
        ^IITR139 = RP:IncomeTax.Deduction.EducationNet.Amount
    
        ^DDCTNS200 = DDCTNS200
        */
        public static ProcessMessageDocument VR_ATO_IITR_600031(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (ddctnsChildDocument.DDCTNS200 != null && report.iitrParentDocument.IITR139 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.601031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related self-education expenses amount must be provided",
                    LongDescription = @"When the Self education reason is present on the attached Deductions schedule, the Work related self-education expenses amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.EducationNet.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.600031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR139", Value = report.iitrParentDocument.IITR139.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DDCTNS200",
                    Value = ddctnsChildDocument.DDCTNS200
                });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.700304

        /* VR.ATO.IITR.700304
        When the taxpayer is over the age of 76 on the 1st June for the Year of return, Personal Superannuation Contributions must not be provided on the Deductions schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR29 <> NULL AND ^IITR29 < ConvertToDate(1, 6, ^IITR10 - 76)) AND RoundDown(Sum(^DDCTNS407), 0) > 0

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR29 <> NULL AND ^IITR29 < ConvertToDate(1, 6, ^IITR10 - 76)) AND RoundDown(Sum(^DDCTNS407), 0) > 0

        Data Elements:
    
        ^IITR29 = RP:PersonDemographicDetails.Birth.Date
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR10 = RP:Report.TargetFinancial.Year
        */
        public static ProcessMessageDocument VR_ATO_IITR_700304(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;
            decimal sumDDCTNS407 = SumDDCTNS407(report);

            bool assertion = report.iitrParentDocument.IITR29.HasValue && report.iitrParentDocument.IITR10.HasValue
                            && report.iitrParentDocument.IITR29.Value < new DateTime(report.iitrParentDocument.IITR10.Value - 76, 6, 1)
                            && ddctnsChildDocument.Rp_PersonalSuperContributionsCollectionCount > 0 && sumDDCTNS407 > 0;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.700304",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal Superannuation Contributions must not be provided on the Deductions schedule",
                    LongDescription = @"When the taxpayer is over the age of 76 on the 1st June for the Year of return, an amount above zero must not be provided on the Deductions schedule",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Birth.Date[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.700304" } }
                };
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR10", Value = GetValueOrEmpty(report.iitrParentDocument.IITR10) });

                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR29", Value = GetValueOrEmpty(report.iitrParentDocument.IITR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "Sum(DDCTNS407)", Value = sumDDCTNS407.ToString() });

            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730000

        /* VR.ATO.IITR.730000
        When work related car expense details are present in the attached Deductions schedule, the Work related car expenses amount must be provided

        Legacy Rule Format:
            ^IITR134 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{CarSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR134 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{CarSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR134 = RP:IncomeTax.Deduction.Car.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730000(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = ((ddctnsChildDocument.Rp_CarCollectionCount > 0) && report.iitrParentDocument.IITR134 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730000",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related car expenses amount must be provided",
                    LongDescription = @"When work related car expense details are present in the attached Deductions schedule, the Work related car expenses amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Car.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730000" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR134", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730001

        /* VR.ATO.IITR.730001
            The Work related car expenses must equal the sum of all car expense amounts claimed in the attached Deductions schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR134 <> NULL AND ^IITR134 <> RoundDown(((Sum((^DDCTNS105 + ^DDCTNS106) * (^DDCTNS104 / 100))) + (Sum(^DDCTNS103) * 0.88)), 0))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR134 <> NULL AND ^IITR134 <> RoundDown(((Sum((^DDCTNS105 + ^DDCTNS106) * (^DDCTNS104 / 100))) + (Sum(^DDCTNS103) * 0.88)), 0))

        Data Elements:
    
            ^IITR134 = RP:IncomeTax.Deduction.Car.Amount
    
            ^DDCTNS103 = DDCTNS103
    
            ^DDCTNS104 = DDCTNS104
    
            ^DDCTNS105 = DDCTNS105
    
            ^DDCTNS106 = DDCTNS106
        */

        public static ProcessMessageDocument VR_ATO_IITR_730001(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            decimal vr730001_LogbookSum = 0;
            decimal vr730001_CentsPerKmSum = 0;
            decimal vr730001_TotalSum = 0;
            if (ddctnsChildDocument.Rp_CarCollectionCount > 0)
            {
                foreach (DDCTNS2026.Rp_Car context in ddctnsChildDocument.Rp_CarCollection)
                {
                    vr730001_LogbookSum += ((context.DDCTNS105.GetValueOrDefault() + context.DDCTNS106.GetValueOrDefault()) * (context.DDCTNS104.GetValueOrDefault() / 100));
                    vr730001_CentsPerKmSum += (context.DDCTNS103.GetValueOrDefault() * 0.88M);
                }
                vr730001_TotalSum = Math.Truncate(vr730001_LogbookSum + vr730001_CentsPerKmSum);
            }
            bool assertion = (report.iitrParentDocument.IITR134 != null && report.iitrParentDocument.IITR134.GetValueOrDefault() != vr730001_TotalSum);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related car expenses amount is incorrect",
                    LongDescription = @"The Work related car expenses must equal the sum of all car expense amounts claimed in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Car.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR134", Value = report.iitrParentDocument.IITR134.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Car Logbook Total", Value = vr730001_LogbookSum.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Car Cents Per Km Total", Value = vr730001_CentsPerKmSum.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Car Deductions Total", Value = vr730001_TotalSum.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730002

        /* VR.ATO.IITR.730002
        When Dividend deductions expense details are present on the attached Deductions schedule, the Dividend deductions amount must be provided

        Legacy Rule Format:
            ^IITR147 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{DivSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR147 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{DivSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730002(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR147 == null && ddctnsChildDocument.Rp_DividendDeductionsCollectionCount > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividend deductions amount must be provided",
                    LongDescription = @"When Dividend deductions expense details are present on the attached Deductions schedule, the Dividend deductions amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Dividend.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR147", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730003

        /* VR.ATO.IITR.730003
        The Dividend deductions amount must equal the sum of all Dividend deduction amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR147 <> NULL AND ^IITR147 <> RoundDown(Sum(^DDCTNS132), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR147 <> NULL AND ^IITR147 <> RoundDown(Sum(^DDCTNS132), 0)

        Data Elements:
    
        ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount
    
        ^DDCTNS132 = DDCTNS132
        */
        public static ProcessMessageDocument VR_ATO_IITR_730003(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR147 != null && (ddctnsChildDocument.Rp_DividendDeductionsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_DividendDeductionsCollection.Sum(context => context.DDCTNS132.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR147.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividend deductions amount is incorrect",
                    LongDescription = @"The Dividend deductions amount must equal the sum of all Dividend deduction amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Dividend.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR147", Value = report.iitrParentDocument.IITR147.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS132", Value = (ddctnsChildDocument.Rp_DividendDeductionsCollectionCount > 0 ? ddctnsChildDocument.Rp_DividendDeductionsCollection.Sum(context => context.DDCTNS132.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730100

        /* VR.ATO.IITR.730100
        When project pool details are present in the attached Deductions schedule, the 'Individual - Deduction for project pool' amount must be provided

        Legacy Rule Format:
            ^IITR366 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{ProjPoolSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR366 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{ProjPoolSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR366 = RP:IncomeTax.Deduction.ProjectPool.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730100(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR366 == null && (ddctnsChildDocument.Rp_ProjectPoolCollection  != null && ddctnsChildDocument.Rp_ProjectPoolCollectionCount > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual - Deduction for project pool amount must be provided",
                    LongDescription = @"When project pool details are present in the attached Deductions schedule, the 'Individual - Deduction for project pool' amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.ProjectPool.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR366", Value = report.iitrParentDocument.IITR366.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730101

        /* VR.ATO.IITR.730101
        The 'Individual - Deduction for project pool' amount must equal the sum of all Project pool amounts in the attached Deductions schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR366 <> NULL AND ^IITR366 <> RoundDown(Sum(^DDCTNS212), 0))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR366 <> NULL AND ^IITR366 <> RoundDown(Sum(^DDCTNS212), 0))

        Data Elements:
    
        ^IITR366 = RP:IncomeTax.Deduction.ProjectPool.Amount
    
        ^DDCTNS212 = DDCTNS212
        */
        public static ProcessMessageDocument VR_ATO_IITR_730101(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            // Calculate the sum of the granular project pool amounts (if a project pool context exists, add each occurance of DDCTNS212. If not, sum equals 0)
            decimal sumDDCTNS212 = ddctnsChildDocument.Rp_ProjectPoolCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_ProjectPoolCollection.Sum(context => context.DDCTNS212.GetValueOrDefault())) : 0;

            bool assertion = (report.iitrParentDocument.IITR366 != null && sumDDCTNS212 != report.iitrParentDocument.IITR366.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730101",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual - Deduction for project pool amount is incorrect",
                    LongDescription = @"The 'Individual - Deduction for project pool' amount must equal the sum of all Project pool amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.ProjectPool.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730101" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR366", Value = report.iitrParentDocument.IITR366.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730102

        /* VR.ATO.IITR.730102
        The TFN provided for the Deductions schedule must be the same as the TFN of the parent form

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (DDCTNS:^RP:TFN <> ^RP:TFN)

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (DDCTNS:^RP:TFN <> ^RP:TFN)

        Data Elements:
    
        ^TFN = TFN
        */
        public static ProcessMessageDocument VR_ATO_IITR_730102(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (ddctnsChildDocument.DDCTNS302 != report.iitrParentDocument.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730102",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax file number invalid",
                    LongDescription = @"The TFN provided for the Deductions schedule must be the same as the TFN of the parent form",
                    Location = "/tns:DDCTNS/tns:Rp/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730102" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730103

        /* VR.ATO.IITR.730103
        Invalid context. Period start date for the Deductions schedule must be the same as the Period start date of the parent form.

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (DDCTNS:^RP:StartDate <> ^RP:StartDate)

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (DDCTNS:^RP:StartDate <> ^RP:StartDate)

        Data Elements:
    
        ^StartDate = StartDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_730103(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (ddctnsChildDocument.DDCTNS300 != report.iitrParentDocument.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730103",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date for the Deductions schedule is invalid",
                    LongDescription = @"Invalid context. Period start date for the Deductions schedule must be the same as the Period start date of the parent form.",
                    Location = "/tns:DDCTNS/tns:Rp/tns:LodgmentPeriodStartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730103" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730104

        /* VR.ATO.IITR.730104
        Invalid context. Period end date for the Deductions schedule must be the same as the Period end date of the parent form.

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (DDCTNS:^RP:EndDate <> ^RP:EndDate)

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (DDCTNS:^RP:EndDate <> ^RP:EndDate)

        Data Elements:
    
        ^EndDate = EndDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_730104(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (ddctnsChildDocument.DDCTNS301 != report.iitrParentDocument.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730104",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date for the Deductions schedule is invalid",
                    LongDescription = @"Invalid context. Period end date for the Deductions schedule must be the same as the Period end date of the parent form.",
                    Location = "/tns:DDCTNS/tns:Rp/tns:LodgmentPeriodEndD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730104" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730106

        /* VR.ATO.IITR.730106
        The Work related self education expenses must not exceed the sum of all self education amounts and self education car expenses claimed in the attached Deductions schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR139 <> NULL AND ^IITR139 > RoundDown((Sum(^DDCTNS204) + (Sum((^DDCTNS124 + ^DDCTNS125) * (^DDCTNS123 / 100))) + (Sum(^DDCTNS122) * 0.88)), 0))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND (^IITR139 <> NULL AND ^IITR139 > RoundDown((Sum(^DDCTNS204) + (Sum((^DDCTNS124 + ^DDCTNS125) * (^DDCTNS123 / 100))) + (Sum(^DDCTNS122) * 0.88)), 0))

        Data Elements:
    
            ^IITR139 = RP:IncomeTax.Deduction.EducationNet.Amount
    
            ^DDCTNS122 = DDCTNS122
    
            ^DDCTNS123 = DDCTNS123
    
            ^DDCTNS124 = DDCTNS124
    
            ^DDCTNS125 = DDCTNS125
    
            ^DDCTNS204 = DDCTNS204
        */
        public static ProcessMessageDocument VR_ATO_IITR_730106(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            decimal vr730106_LogbookSum = 0;
            decimal vr730106_CentsPerKmSum = 0;
            decimal vr730106_SelfEduAmountSum = 0;
            decimal vr730106_TotalSum = 0;
            if (ddctnsChildDocument.Rp_SelfEducationCollection != null)
            {
                foreach (DDCTNS2026.Rp_SelfEducation context in ddctnsChildDocument.Rp_SelfEducationCollection)
                {
                    vr730106_SelfEduAmountSum += (context.DDCTNS204.GetValueOrDefault());
                    vr730106_LogbookSum += ((context.DDCTNS124.GetValueOrDefault() + context.DDCTNS125.GetValueOrDefault()) * (context.DDCTNS123.GetValueOrDefault() / 100));
                    vr730106_CentsPerKmSum += (context.DDCTNS122.GetValueOrDefault() * 0.88M);
                }
                vr730106_TotalSum = Math.Truncate(vr730106_LogbookSum + vr730106_CentsPerKmSum + vr730106_SelfEduAmountSum);
            }
            bool assertion = (report.iitrParentDocument.IITR139.HasValue && report.iitrParentDocument.IITR139.Value > vr730106_TotalSum);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730106",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related self-education expenses amount is incorrect",
                    LongDescription = @"The Work related self education expenses must not exceed the sum of all self education amounts and self education car expenses claimed in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.EducationNet.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR139", Value = report.iitrParentDocument.IITR139.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Self Education Car Logbook Total", Value = vr730106_LogbookSum.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Self Education Car Cents Per Km Total", Value = vr730106_CentsPerKmSum.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Self Education Expense Total", Value = vr730106_SelfEduAmountSum.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Self Education Deductions Total", Value = vr730106_TotalSum.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730108

        /* VR.ATO.IITR.730108
        Either one or the sum of Superannuation deduction claimed amounts is the same as the amount provided for Reportable employer superannuation contributions. A deduction for contributions paid by the employer from before-tax income (including the compulsory super guarantee and salary sacrifice amounts) is not claimable, and the return may be adjusted. Check and correct if necessary.
    
        Legacy Rule Format:
        CountDocument('DDCTNS') = 1 AND ^IITR201 > 0 AND (AnyOccurrence(^IITR201, ^IITR201 = RoundDown(^DDCTNS407, 0)) OR AnyOccurrence(^IITR201, ^IITR201 = RoundDown(Sum(^DDCTNS407), 0)))

        Technical Business Rule Format:
        CountDocument('DDCTNS') = 1 AND ^IITR201 > 0 AND (AnyOccurrence(^IITR201, ^IITR201 = RoundDown(^DDCTNS407, 0)) OR AnyOccurrence(^IITR201, ^IITR201 = RoundDown(Sum(^DDCTNS407), 0)))
    
        Data Elements:
    
        ^DDCTNS407 = DDCTNS407
    
        ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730108(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS407 = SumDDCTNS407(report);

            bool assertion = report.iitrParentDocument.IITR201.GetValueOrDefault() > 0 && ((report.ddctnsChildDocuments[0].Rp_SelfEducationCollection != null ? report.ddctnsChildDocuments[0].Rp_PersonalSuperContributionsCollection.Any(f => Math.Truncate(f.DDCTNS407.GetValueOrDefault()) == report.iitrParentDocument.IITR201.GetValueOrDefault()) : false) || report.iitrParentDocument.IITR201.GetValueOrDefault() == sumDDCTNS407);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730108",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check personal superannuation deductions claimed",
                    LongDescription = @"Either one or the sum of Superannuation deduction claimed amounts is the same as the amount provided for Reportable employer superannuation contributions. A deduction for contributions paid by the employer from before-tax income (including the compulsory super guarantee and salary sacrifice amounts) is not claimable, and the return may be adjusted. Check and correct if necessary.",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerReportable.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730108" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR201", Value = GetValueOrEmpty(report.iitrParentDocument.IITR201) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(DDCTNS407)", Value = sumDDCTNS407.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730166

        /* VR.ATO.IITR.730166
        Where no income and deduction fields are provided, check if this is a nil return.
    
        Legacy Rule Format:
        (^IITR601 = NULL) AND (^IITR65 = NULL) AND (^IITR607 = NULL) AND (^IITR609 = NULL) AND (^IITR86 = NULL) AND (^IITR89 = NULL) AND (^IITR92 = NULL) AND (^IITR93 = NULL) AND (^IITR505 = NULL) AND (^IITR94 = NULL) AND (^IITR95 = NULL) AND (^IITR115 = NULL) AND (^IITR118 = NULL) AND (^IITR119 = NULL) AND (^IITR120 = NULL) AND (^IITR134 = NULL) AND (^IITR136 = NULL) AND (^IITR137 = NULL) AND (^IITR139 = NULL) AND (^IITR141 = NULL) AND (^IITR148 = NULL) AND (^IITR502 = NULL) AND (^IITR503 = NULL) AND (^IITR504 = NULL) AND (^IITR154 = NULL) AND (^IITR156 = NULL) AND (^IITR242 = NULL) AND (^IITR243 = NULL) AND (^IITR245 = NULL) AND (^IITR246 = NULL) AND (^IITR252 = NULL) AND (^IITR255 = NULL) AND (^IITR259 = NULL) AND (^IITR260 = NULL) AND (^IITR265 = NULL) AND (^IITR281 = NULL) AND (^IITR285 = NULL) AND (^IITR306 = NULL) AND (^IITR311 = NULL) AND (^IITR316 = NULL) AND (^IITR318 = NULL) AND (^IITR323 = NULL) AND (^IITR327 = NULL) AND (^IITR334 = NULL) AND (^IITR335 = NULL) AND (^IITR337 = NULL) AND (^IITR339 = NULL) AND (^IITR615 = NULL) AND (^IITR616 = NULL) AND (^IITR650 = NULL) AND (^IITR618 = NULL) AND (^IITR358 = NULL) AND CountDocument('DDCTNS') = 0

        Technical Business Rule Format:
        (^IITR601 = NULL) AND (^IITR65 = NULL) AND (^IITR607 = NULL) AND (^IITR609 = NULL) AND (^IITR86 = NULL) AND (^IITR89 = NULL) AND (^IITR92 = NULL) AND (^IITR93 = NULL) AND (^IITR505 = NULL) AND (^IITR94 = NULL) AND (^IITR95 = NULL) AND (^IITR115 = NULL) AND (^IITR118 = NULL) AND (^IITR119 = NULL) AND (^IITR120 = NULL) AND (^IITR134 = NULL) AND (^IITR136 = NULL) AND (^IITR137 = NULL) AND (^IITR139 = NULL) AND (^IITR141 = NULL) AND (^IITR148 = NULL) AND (^IITR502 = NULL) AND (^IITR503 = NULL) AND (^IITR504 = NULL) AND (^IITR154 = NULL) AND (^IITR156 = NULL) AND (^IITR242 = NULL) AND (^IITR243 = NULL) AND (^IITR245 = NULL) AND (^IITR246 = NULL) AND (^IITR252 = NULL) AND (^IITR255 = NULL) AND (^IITR259 = NULL) AND (^IITR260 = NULL) AND (^IITR265 = NULL) AND (^IITR281 = NULL) AND (^IITR285 = NULL) AND (^IITR306 = NULL) AND (^IITR311 = NULL) AND (^IITR316 = NULL) AND (^IITR318 = NULL) AND (^IITR323 = NULL) AND (^IITR327 = NULL) AND (^IITR334 = NULL) AND (^IITR335 = NULL) AND (^IITR337 = NULL) AND (^IITR339 = NULL) AND (^IITR615 = NULL) AND (^IITR616 = NULL) AND (^IITR650 = NULL) AND (^IITR618 = NULL) AND (^IITR358 = NULL) AND CountDocument('DDCTNS') = 0
    
        Data Elements:
    
        ^IITR601 = RP:Income.SalaryOrWages.Amount
    
        ^IITR65 = RP:Income.AllowancesEarningsTipsDirectorsFees.Amount
    
        ^IITR86 = RP:Income.AustralianGovernmentAllowancesAndPayments.Amount
    
        ^IITR89 = RP:Pension.Total.Amount
    
        ^IITR92 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount
    
        ^IITR93 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount
    
        ^IITR94 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
    
        ^IITR95 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
    
        ^IITR115 = RP:Income.Interest.Gross.Amount
    
        ^IITR118 = RP:Income.DividendsUnfranked.Amount
    
        ^IITR119 = RP:Income.DividendsFranked.Amount
    
        ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
        ^IITR134 = RP:IncomeTax.Deduction.Car.Amount
    
        ^IITR136 = RP:IncomeTax.Deduction.Travel.Amount
    
        ^IITR137 = RP:IncomeTax.Deduction.Clothing.Amount
    
        ^IITR139 = RP:IncomeTax.Deduction.EducationNet.Amount
    
        ^IITR141 = RP:IncomeTax.Deduction.WorkRelatedOther.Amount
    
        ^IITR148 = RP:IncomeTax.Deduction.GiftDonation.Amount
    
        ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount
    
        ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount
    
        ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount
    
        ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
        ^IITR245 = RP.Partnership:Expense.LandcareOperationsAcceleratedDepreciation.Amount
    
        ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
    
        ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount
    
        ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
        ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount
    
        ^IITR260 = RP:Expense.LandcareOperationsNonPrimaryProduction.Amount
    
        ^IITR265 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
    
        ^IITR281 = RP:Income.PrimaryProductionNet.Amount
    
        ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
    
        ^IITR306 = RP:Income.FarmManagementDepositsOrRepaymentsNet.Amount
    
        ^IITR311 = RP:Capital.Gains.Net.Amount
    
        ^IITR316 = RP:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
        ^IITR318 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
    
        ^IITR323 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeUndeductedPurchasePrice.Amount
    
        ^IITR327 = RP:Income.InternationalDealings.Net.Amount
    
        ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount
    
        ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount
    
        ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
        ^IITR339 = RP:IncomeTax.LifeInsuranceCompaniesAndFriendlySocietiesBonuses.Amount
    
        ^IITR358 = RP:IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount
    
        ^IITR502 = RP:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount
    
        ^IITR503 = RP:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount
    
        ^IITR504 = RP:IncomeTax.Deduction.TaxAffairManagementOther.Amount
    
        ^IITR505 = RP:Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount
    
        ^IITR607 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
    
        ^IITR609 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
    
        ^IITR615 = RP:IncomeTax.IncomeCategory1Other.Amount
    
        ^IITR616 = RP:IncomeTax.IncomeCategory2Total.Amount
    
        ^IITR618 = RP:IncomeTax.IncomeCategoryMiscellaneousOther.Amount
    
        ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730166(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;
            bool assertion;

            assertion = (report.iitrParentDocument.IITR601 == null && report.iitrParentDocument.IITR65 == null && report.iitrParentDocument.IITR607 == null && report.iitrParentDocument.IITR609 == null && report.iitrParentDocument.IITR86 == null && report.iitrParentDocument.IITR89 == null && report.iitrParentDocument.IITR92 == null && report.iitrParentDocument.IITR93 == null && report.iitrParentDocument.IITR505 == null && report.iitrParentDocument.IITR94 == null && report.iitrParentDocument.IITR95 == null && report.iitrParentDocument.IITR115 == null && report.iitrParentDocument.IITR118 == null && report.iitrParentDocument.IITR119 == null && report.iitrParentDocument.IITR120 == null && report.iitrParentDocument.IITR134 == null && report.iitrParentDocument.IITR136 == null && report.iitrParentDocument.IITR137 == null && report.iitrParentDocument.IITR139 == null && report.iitrParentDocument.IITR141 == null && report.iitrParentDocument.IITR148 == null && report.iitrParentDocument.IITR502 == null && report.iitrParentDocument.IITR503 == null && report.iitrParentDocument.IITR504 == null && report.iitrParentDocument.IITR154 == null && report.iitrParentDocument.IITR156 == null && report.iitrParentDocument.IITR242 == null && report.iitrParentDocument.IITR243 == null && report.iitrParentDocument.IITR245 == null && report.iitrParentDocument.IITR246 == null && report.iitrParentDocument.IITR252 == null && report.iitrParentDocument.IITR255 == null && report.iitrParentDocument.IITR259 == null && report.iitrParentDocument.IITR260 == null && report.iitrParentDocument.IITR265 == null && report.iitrParentDocument.IITR281 == null && report.iitrParentDocument.IITR285 == null && report.iitrParentDocument.IITR306 == null && report.iitrParentDocument.IITR311 == null && report.iitrParentDocument.IITR316 == null && report.iitrParentDocument.IITR318 == null && report.iitrParentDocument.IITR323 == null && report.iitrParentDocument.IITR327 == null && report.iitrParentDocument.IITR334 == null && report.iitrParentDocument.IITR335 == null && report.iitrParentDocument.IITR337 == null && report.iitrParentDocument.IITR339 == null && report.iitrParentDocument.IITR615 == null && report.iitrParentDocument.IITR616 == null && report.iitrParentDocument.IITR650 == null && report.iitrParentDocument.IITR618 == null && report.iitrParentDocument.IITR358 == null && Count(report.ddctnsChildDocuments) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730166",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"No income or deduction amounts present. Is this a nil return?",
                    Location = "/xbrli:xbrl/tns:Income.SalaryOrWages.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730166" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR601", Value = GetValueOrEmpty(report.iitrParentDocument.IITR601) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR65", Value = GetValueOrEmpty(report.iitrParentDocument.IITR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR607", Value = GetValueOrEmpty(report.iitrParentDocument.IITR607) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR609", Value = GetValueOrEmpty(report.iitrParentDocument.IITR609) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR86", Value = GetValueOrEmpty(report.iitrParentDocument.IITR86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR89", Value = GetValueOrEmpty(report.iitrParentDocument.IITR89) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR92", Value = GetValueOrEmpty(report.iitrParentDocument.IITR92) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR93", Value = GetValueOrEmpty(report.iitrParentDocument.IITR93) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR505", Value = GetValueOrEmpty(report.iitrParentDocument.IITR505) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR94", Value = GetValueOrEmpty(report.iitrParentDocument.IITR94) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR95", Value = GetValueOrEmpty(report.iitrParentDocument.IITR95) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR115", Value = GetValueOrEmpty(report.iitrParentDocument.IITR115) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = GetValueOrEmpty(report.iitrParentDocument.IITR118) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = GetValueOrEmpty(report.iitrParentDocument.IITR119) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.iitrParentDocument.IITR120) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR134", Value = GetValueOrEmpty(report.iitrParentDocument.IITR134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR136", Value = GetValueOrEmpty(report.iitrParentDocument.IITR136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR137", Value = GetValueOrEmpty(report.iitrParentDocument.IITR137) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR139", Value = GetValueOrEmpty(report.iitrParentDocument.IITR139) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR141", Value = GetValueOrEmpty(report.iitrParentDocument.IITR141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR148", Value = GetValueOrEmpty(report.iitrParentDocument.IITR148) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR502", Value = GetValueOrEmpty(report.iitrParentDocument.IITR502) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR503", Value = GetValueOrEmpty(report.iitrParentDocument.IITR503) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR504", Value = GetValueOrEmpty(report.iitrParentDocument.IITR504) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR154", Value = GetValueOrEmpty(report.iitrParentDocument.IITR154) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR156", Value = GetValueOrEmpty(report.iitrParentDocument.IITR156) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR242", Value = GetValueOrEmpty(report.iitrParentDocument.IITR242) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.iitrParentDocument.IITR243) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR245", Value = GetValueOrEmpty(report.iitrParentDocument.IITR245) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR246", Value = GetValueOrEmpty(report.iitrParentDocument.IITR246) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR252", Value = GetValueOrEmpty(report.iitrParentDocument.IITR252) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.iitrParentDocument.IITR255) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR259", Value = GetValueOrEmpty(report.iitrParentDocument.IITR259) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR260", Value = GetValueOrEmpty(report.iitrParentDocument.IITR260) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR265", Value = GetValueOrEmpty(report.iitrParentDocument.IITR265) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.iitrParentDocument.IITR281) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.iitrParentDocument.IITR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR306", Value = GetValueOrEmpty(report.iitrParentDocument.IITR306) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR311", Value = GetValueOrEmpty(report.iitrParentDocument.IITR311) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR316", Value = GetValueOrEmpty(report.iitrParentDocument.IITR316) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR318", Value = GetValueOrEmpty(report.iitrParentDocument.IITR318) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR323", Value = GetValueOrEmpty(report.iitrParentDocument.IITR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR327", Value = GetValueOrEmpty(report.iitrParentDocument.IITR327) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR334", Value = GetValueOrEmpty(report.iitrParentDocument.IITR334) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR335", Value = GetValueOrEmpty(report.iitrParentDocument.IITR335) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR337", Value = GetValueOrEmpty(report.iitrParentDocument.IITR337) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR339", Value = GetValueOrEmpty(report.iitrParentDocument.IITR339) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR615", Value = GetValueOrEmpty(report.iitrParentDocument.IITR615) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR616", Value = GetValueOrEmpty(report.iitrParentDocument.IITR616) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR650", Value = GetValueOrEmpty(report.iitrParentDocument.IITR650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR618", Value = GetValueOrEmpty(report.iitrParentDocument.IITR618) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR358", Value = GetValueOrEmpty(report.iitrParentDocument.IITR358) });

            }
            return processMessage;
        }
        #endregion // VR.ATO.IITR.730166

        #region VR.ATO.IITR.730200

        /* VR.ATO.IITR.730200
        The Cost of managing tax affairs - Interest charged by the ATO amount must equal the sum of all Interest charged by the ATO amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR502 <> NULL AND  ^IITR502 <> RoundDown(Sum(ConditionalValue(^DDCTNS304 = 'Interest charged by the ATO', ^DDCTNS306, 0)), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR502 <> NULL AND  ^IITR502 <> RoundDown(Sum(ConditionalValue(^DDCTNS304 = 'Interest charged by the ATO', ^DDCTNS306, 0)), 0)

        Data Elements:
    
        ^IITR502 = RP:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount
    
        ^DDCTNS304 = DDCTNS304
    
        ^DDCTNS306 = DDCTNS306
        */
        public static ProcessMessageDocument VR_ATO_IITR_730200(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR502 != null && (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Where(context => context.DDCTNS304.Equals("Interest charged by the ATO")).Sum(context => context.DDCTNS306.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR502.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730200",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of managing tax affairs - Interest charged by the ATO amount is incorrect",
                    LongDescription = @"The Cost of managing tax affairs - Interest charged by the ATO amount must equal the sum of all Interest charged by the ATO amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR502", Value = report.iitrParentDocument.IITR502.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS304", Value = "Interest charged by the ATO" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS306", Value = (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Where(context => context.DDCTNS304.Equals("Interest charged by the ATO")).Sum(context => context.DDCTNS306.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730201

        /* VR.ATO.IITR.730201
        The Cost of managing tax affairs - Litigation costs amount must equal the sum of all Litigation costs amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR503 <> NULL AND ^IITR503 <> RoundDown(Sum(ConditionalValue(^DDCTNS304 = 'Litigation costs', ^DDCTNS306, 0)), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR503 <> NULL AND ^IITR503 <> RoundDown(Sum(ConditionalValue(^DDCTNS304 = 'Litigation costs', ^DDCTNS306, 0)), 0)

        Data Elements:
    
        ^IITR503 = RP:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount
    
        ^DDCTNS304 = DDCTNS304
    
        ^DDCTNS306 = DDCTNS306
        */
        public static ProcessMessageDocument VR_ATO_IITR_730201(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR503 != null && (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Where(context => context.DDCTNS304.Equals("Litigation costs")).Sum(context => context.DDCTNS306.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR503.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730201",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of managing tax affairs - Litigation costs amount is incorrect",
                    LongDescription = @"The Cost of managing tax affairs - Litigation costs amount must equal the sum of all Litigation costs amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR503", Value = report.iitrParentDocument.IITR503.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS304", Value = "Litigation costs" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS306", Value = (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Where(context => context.DDCTNS304.Equals("Litigation costs")).Sum(context => context.DDCTNS306.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730202

        /* VR.ATO.IITR.730202
        The Cost of managing tax affairs - Other expenses incurred in managing your tax affairs amount must equal the sum of all Other expenses incurred in managing your tax affairs in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR504 <> NULL AND ^IITR504 <> RoundDown(Sum(ConditionalValue(^DDCTNS304 = 'Other expenses incurred in managing your tax affairs', ^DDCTNS306, 0)), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR504 <> NULL AND ^IITR504 <> RoundDown(Sum(ConditionalValue(^DDCTNS304 = 'Other expenses incurred in managing your tax affairs', ^DDCTNS306, 0)), 0)

        Data Elements:
    
        ^IITR504 = RP:IncomeTax.Deduction.TaxAffairManagementOther.Amount
    
        ^DDCTNS304 = DDCTNS304
    
        ^DDCTNS306 = DDCTNS306
        */
        public static ProcessMessageDocument VR_ATO_IITR_730202(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR504 != null && (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Where(context => context.DDCTNS304.Equals("Other expenses incurred in managing your tax affairs")).Sum(context => context.DDCTNS306.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR504.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730202",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of managing tax affairs - Other expenses amount is incorrect",
                    LongDescription = @"The Cost of managing tax affairs - Other expenses incurred in managing your tax affairs amount must equal the sum of all Other expenses incurred in managing your tax affairs in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxAffairManagementOther.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730202" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR504", Value = report.iitrParentDocument.IITR504.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS304", Value = "Litigation costs" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS306", Value = (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Where(context => context.DDCTNS304.Equals("Other expenses incurred in managing your tax affairs")).Sum(context => context.DDCTNS306.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730203

        /* VR.ATO.IITR.730203
        When Cost of managing tax affairs - Interest charged by the ATO details are present in the attached Deductions schedule, the Cost of managing tax affairs - Interest charged by the ATO amount must be provided

        Legacy Rule Format:
            ^IITR502 = NULL AND CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS304, ^DDCTNS304 = 'Interest charged by the ATO')

        Technical Business Rule Format:
            ^IITR502 = NULL AND CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS304, ^DDCTNS304 = 'Interest charged by the ATO')

        Data Elements:
    
        ^IITR502 = RP:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount
    
        ^DDCTNS304 = DDCTNS304
        */
        public static ProcessMessageDocument VR_ATO_IITR_730203(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR502 == null && (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Any(context => context.DDCTNS304.Equals("Interest charged by the ATO")) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of managing tax affairs - Interest charged by the ATO amount must be provided",
                    LongDescription = @"When Cost of managing tax affairs - Interest charged by the ATO details are present in the attached Deductions schedule, the Cost of managing tax affairs - Interest charged by the ATO amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxAffairManagementATOInterest.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR502", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS304", Value = "Interest charged by the ATO" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730204

        /* VR.ATO.IITR.730204
        When Cost of managing tax affairs - Litigation costs details are present in the attached Deductions schedule, the Cost of managing tax affairs - Litigation costs amount must be provided

        Legacy Rule Format:
            ^IITR503 = NULL AND CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS304, ^DDCTNS304 = 'Litigation costs')

        Technical Business Rule Format:
            ^IITR503 = NULL AND CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS304, ^DDCTNS304 = 'Litigation costs')

        Data Elements:
    
        ^IITR503 = RP:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount
    
        ^DDCTNS304 = DDCTNS304
        */
        public static ProcessMessageDocument VR_ATO_IITR_730204(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR503 == null && (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Any(context => context.DDCTNS304.Equals("Litigation costs")) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730204",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of managing tax affairs - Litigation costs amount must be provided",
                    LongDescription = @"When Cost of managing tax affairs - Litigation costs details are present in the attached Deductions schedule, the Cost of managing tax affairs - Litigation costs amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxAffairManagementLitigation.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR503", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS304", Value = "Litigation costs" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730205

        /* VR.ATO.IITR.730205
        When Cost of managing tax affairs - Other expenses details are present in the attached Deductions schedule, the Cost of managing tax affairs - Other expenses amount must be provided

        Legacy Rule Format:
            ^IITR504 = NULL AND CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS304, ^DDCTNS304 = 'Other expenses incurred in managing your tax affairs')

        Technical Business Rule Format:
            ^IITR504 = NULL AND CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS304, ^DDCTNS304 = 'Other expenses incurred in managing your tax affairs')

        Data Elements:
    
        ^IITR504 = RP:IncomeTax.Deduction.TaxAffairManagementOther.Amount
    
        ^DDCTNS304 = DDCTNS304
        */
        public static ProcessMessageDocument VR_ATO_IITR_730205(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR504 == null && (ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollectionCount > 0 ? ddctnsChildDocument.Rp_CostofManagingTaxAffairsCollection.Any(context => context.DDCTNS304.Equals("Other expenses incurred in managing your tax affairs")) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730205",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of managing tax affairs - Other expenses amount must be provided",
                    LongDescription = @"When Cost of managing tax affairs - Other expenses details are present in the attached Deductions schedule, the Cost of managing tax affairs - Other expenses amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxAffairManagementOther.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730205" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR504", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS304", Value = "Other expenses incurred in managing your tax affairs" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730206

        /* VR.ATO.IITR.730206
        Forestry managed investment scheme deduction amount must equal the sum of all Forestry managed investment amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR368 <> NULL AND ^IITR368 <> RoundDown(Sum(^DDCTNS309), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR368 <> NULL AND ^IITR368 <> RoundDown(Sum(^DDCTNS309), 0)

        Data Elements:
    
        ^IITR368 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
        ^DDCTNS309 = DDCTNS309
        */
        public static ProcessMessageDocument VR_ATO_IITR_730206(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR368 != null && (ddctnsChildDocument.Rp_ForestryManagedInvestmentCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_ForestryManagedInvestmentCollection.Sum(context => context.DDCTNS309.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR368.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730206",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Forestry managed investment scheme deduction amount is incorrect",
                    LongDescription = @"Forestry managed investment scheme deduction amount must equal the sum of all Forestry managed investment amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:Expense.ForestryManagedInvestmentSchemeDeduction.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730206" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR368", Value = report.iitrParentDocument.IITR368.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS309", Value = (ddctnsChildDocument.Rp_ForestryManagedInvestmentCollectionCount > 0 ? ddctnsChildDocument.Rp_ForestryManagedInvestmentCollection.Sum(context => context.DDCTNS309.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730207

        /* VR.ATO.IITR.730207
        When Forestry managed investment scheme deduction details are present in the attached Deductions schedule, the Forestry managed investment scheme deduction amount must be provided

        Legacy Rule Format:
            ^IITR368 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{ForestManInvSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR368 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{ForestManInvSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR368 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730207(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR368 == null && ddctnsChildDocument.Rp_ForestryManagedInvestmentCollectionCount > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730207",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Forestry managed investment scheme deduction amount must be provided",
                    LongDescription = @"When Forestry managed investment scheme deduction details are present in the attached Deductions schedule, the Forestry managed investment scheme deduction amount must be provided",
                    Location = "/xbrli:xbrl/tns:Expense.ForestryManagedInvestmentSchemeDeduction.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730207" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR368", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730208

        /* VR.ATO.IITR.730208
        Low value pool deduction must equal the sum of all Low value pool deduction component amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR145 <> NULL AND (^IITR145 <> RoundDown(^DDCTNS310 + ^DDCNTS311 + ^DDCTNS312, 0))

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR145 <> NULL AND (^IITR145 <> RoundDown(^DDCTNS310 + ^DDCNTS311 + ^DDCTNS312, 0))

        Data Elements:
    
        ^IITR145 = RP:IncomeTax.Deduction.LowValuePool.Amount
    
        ^DDCNTS311 = DDCNTS311
    
        ^DDCTNS310 = DDCTNS310
    
        ^DDCTNS312 = DDCTNS312
        */
        public static ProcessMessageDocument VR_ATO_IITR_730208(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR145 != null && (Math.Truncate(ddctnsChildDocument.DDCTNS310.GetValueOrDefault() + ddctnsChildDocument.DDCTNS311.GetValueOrDefault() + ddctnsChildDocument.DDCTNS312.GetValueOrDefault())) != report.iitrParentDocument.IITR145.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Low value pool deduction amount is incorrect",
                    LongDescription = @"The Low value pool deduction amount must equal the sum of all Low value pool deduction component amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.LowValuePool.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR145", Value = report.iitrParentDocument.IITR145.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS310", Value = ddctnsChildDocument.DDCTNS310.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS311", Value = ddctnsChildDocument.DDCTNS311.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS312", Value = ddctnsChildDocument.DDCTNS312.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730209

        /* VR.ATO.IITR.730209
        When Low value pool deduction component amounts are present in the attached Deductions Schedule, the Low value pool deduction amount must be provided

        Legacy Rule Format:
            ^IITR145 = NULL AND CountDocument('DDCTNS') = 1 AND ((^DDCNTS310 <> NULL) OR (^DDCTNS311 <> NULL) OR (^DDCTNS312 <> NULL))

        Technical Business Rule Format:
            ^IITR145 = NULL AND CountDocument('DDCTNS') = 1 AND ((^DDCNTS310 <> NULL) OR (^DDCTNS311 <> NULL) OR (^DDCTNS312 <> NULL))

        Data Elements:
    
        ^IITR145 = RP:IncomeTax.Deduction.LowValuePool.Amount
    
        ^DDCNTS310 = DDCNTS310
    
        ^DDCTNS311 = DDCTNS311
    
        ^DDCTNS312 = DDCTNS312
        */
        public static ProcessMessageDocument VR_ATO_IITR_730209(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR145 == null && (ddctnsChildDocument.DDCTNS310 != null || ddctnsChildDocument.DDCTNS311 != null || ddctnsChildDocument.DDCTNS312 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730209",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Low value pool deduction must be provided",
                    LongDescription = @"When Low value pool deduction component amounts are present in the attached Deductions Schedule, the Low value pool deduction amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.LowValuePool.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730209" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR145", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS310", Value = (ddctnsChildDocument.DDCTNS310 != null ? ddctnsChildDocument.DDCTNS310.GetValueOrDefault().ToString() : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS311", Value = (ddctnsChildDocument.DDCTNS311 != null ? ddctnsChildDocument.DDCTNS311.GetValueOrDefault().ToString() : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS312", Value = (ddctnsChildDocument.DDCTNS312 != null ? ddctnsChildDocument.DDCTNS312.GetValueOrDefault().ToString() : "null") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730210

        /* VR.ATO.IITR.730210
        Other work related expenses amount must equal the sum of all Other work related expenses amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR141 <> NULL AND ^IITR141 <> RoundDown(Sum(^DDCTNS315), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR141 <> NULL AND ^IITR141 <> RoundDown(Sum(^DDCTNS315), 0)

        Data Elements:
    
        ^IITR141 = RP:IncomeTax.Deduction.WorkRelatedOther.Amount
    
        ^DDCTNS315 = DDCTNS315
        */
        public static ProcessMessageDocument VR_ATO_IITR_730210(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR141 != null && (ddctnsChildDocument.Rp_OtherWorkRelatedExpensesCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_OtherWorkRelatedExpensesCollection.Sum(context => context.DDCTNS315.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR141.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730210",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other work related expenses amount is incorrect",
                    LongDescription = @"Other work related expenses amount must equal the sum of all Other work related expenses amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.WorkRelatedOther.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730210" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR141", Value = report.iitrParentDocument.IITR141.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS315", Value = (ddctnsChildDocument.Rp_ForestryManagedInvestmentCollectionCount > 0 ? ddctnsChildDocument.Rp_ForestryManagedInvestmentCollection.Sum(context => context.DDCTNS309.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730211

        /* VR.ATO.IITR.730211
        When Other work related expenses details are present in the attached Deductions schedule, the Other work related expenses amount must be provided

        Legacy Rule Format:
            ^IITR141 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{OTHWRSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR141 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{OTHWRSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR141 = RP:IncomeTax.Deduction.WorkRelatedOther.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730211(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR141 == null && ddctnsChildDocument.Rp_OtherWorkRelatedExpensesCollectionCount > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other work related expenses amount must be provided",
                    LongDescription = @"When Other work related expenses details are present in the attached Deductions schedule, the Other work related expenses amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.WorkRelatedOther.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730211" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR141", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730300

        /* VR.ATO.IITR.730300
        The Work related travel expenses amount must be equal to the sum of the Work Related Travel Amount in the attached Deduction schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND ((^IITR136 <> NULL) AND (^IITR136 <> RoundDown(Sum(^DDCTNS410),0)))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND ((^IITR136 <> NULL) AND (^IITR136 <> RoundDown(Sum(^DDCTNS410),0)))

        Data Elements:
    
        ^IITR136 = RP:IncomeTax.Deduction.Travel.Amount
    
        ^DDCTNS410 = DDCTNS410
        */
        public static ProcessMessageDocument VR_ATO_IITR_730300(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumOfTravelExpenses = Math.Truncate(ddctnsChildDocument.Rp_TravelCollection.Sum(context => context.DDCTNS410 ?? 0));

            bool assertion = report.iitrParentDocument.IITR136.HasValue && report.iitrParentDocument.IITR136.Value != sumOfTravelExpenses;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730300",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related travel expenses amount is incorrect",
                    LongDescription = @"The Work related travel expenses amount must be equal to the sum of the Work Related Travel Amount in the attached Deduction schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Travel.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730300" } }
                };
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR136", Value = report.iitrParentDocument.IITR136.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "DDCTNS410", Value = sumOfTravelExpenses.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730304

        /* VR.ATO.IITR.730304
        When work related travel expense details are present in the attached Deductions schedule, the Work related travel expenses amount must be provided

        Legacy Rule Format:
            ^IITR136 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{TrvlSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR136 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{TrvlSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR136 = RP:IncomeTax.Deduction.Travel.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730304(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = !report.iitrParentDocument.IITR136.HasValue && ddctnsChildDocument.Rp_TravelCollectionCount > 0;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730304",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related travel expenses amount must be provided",
                    LongDescription = @"When work related travel expense details are present in the attached Deductions schedule, the Work related travel expenses amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Travel.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730304" } }
                };
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR136", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730212

        /* VR.ATO.IITR.730212
        The Interest deductions amount must equal the sum of all Interest deduction amounts in the attached Deductions schedule

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR146 <> NULL AND ^IITR146 <> RoundDown(Sum(^DDCTNS318), 0)

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND ^IITR146 <> NULL AND ^IITR146 <> RoundDown(Sum(^DDCTNS318), 0)

        Data Elements:
    
        ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount
    
        ^DDCTNS318 = DDCTNS318
        */
        public static ProcessMessageDocument VR_ATO_IITR_730212(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR146 != null && (ddctnsChildDocument.Rp_InterestDeductionsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_InterestDeductionsCollection.Sum(context => context.DDCTNS318.GetValueOrDefault())) : 0) != report.iitrParentDocument.IITR146.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest deductions amount is incorrect",
                    LongDescription = @"The Interest deductions amount must equal the sum of all Interest deduction amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Interest.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730212" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR146", Value = report.iitrParentDocument.IITR146.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS318", Value = (ddctnsChildDocument.Rp_InterestDeductionsCollectionCount > 0 ? ddctnsChildDocument.Rp_InterestDeductionsCollection.Sum(context => context.DDCTNS318.GetValueOrDefault()).ToString() : "0") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730213

        /* VR.ATO.IITR.730213
        When Interest deduction are present in the attached Deductions schedule, the Interest deductions amount must be provided

        Legacy Rule Format:
            ^IITR146 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{IntDedSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR146 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{IntDedSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730213(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR146 == null && ddctnsChildDocument.Rp_InterestDeductionsCollectionCount > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730213",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest deductions amount must be provided",
                    LongDescription = @"When Interest deduction details are present in the attached Deductions schedule, the Interest deductions amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Interest.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730213" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR146", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730305

        /* VR.ATO.IITR.730305
        The Gifts or donations amount must equal the sum of all Gifts or donations amounts in the attached Deductions schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND ((^IITR148 <> NULL) AND (^IITR148 <> RoundDown(Sum(^DDCTNS413),0)))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND ((^IITR148 <> NULL) AND (^IITR148 <> RoundDown(Sum(^DDCTNS413),0)))

        Data Elements:
    
        ^IITR148 = RP:IncomeTax.Deduction.GiftDonation.Amount
    
        ^DDCTNS413 = DDCTNS413
        */
        public static ProcessMessageDocument VR_ATO_IITR_730305(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            // Calculate the sum of the granular Gifts or donations amounts (if a Gifts or donations context exists, add each Occurrence of ddctns413. If not, sum equals 0)
            decimal sumddctns413 = ddctnsChildDocument.Rp_GiftsorDonationsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_GiftsorDonationsCollection.Sum(context => context.DDCTNS413.GetValueOrDefault())) : 0;

            bool assertion = (report.iitrParentDocument.IITR148 != null && sumddctns413 != report.iitrParentDocument.IITR148.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730305",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gifts or donations amount is incorrect",
                    LongDescription = @"The Gifts or donations amount must equal the sum of all Gifts or donations amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.GiftDonation.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730305" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR148", Value = report.iitrParentDocument.IITR148.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS413", Value = ddctnsChildDocument.Rp_GiftsorDonationsCollectionCount > 0 ? Math.Truncate(ddctnsChildDocument.Rp_GiftsorDonationsCollection.Sum(context => context.DDCTNS413.GetValueOrDefault())).ToString() : "0" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730306

        /* VR.ATO.IITR.730306
        When gifts or donations details are present in the attached Deductions schedule, the Gifts or donations amount must be provided

        Legacy Rule Format:
            ^IITR148 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{GiftDonSeqNum}:Context) > 0

        Technical Business Rule Format:
            ^IITR148 = NULL AND CountDocument('DDCTNS') = 1 AND Count(^DDCTNS:RP.{GiftDonSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR148 = RP:IncomeTax.Deduction.GiftDonation.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730306(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR148 == null && ddctnsChildDocument.Rp_GiftsorDonationsCollectionCount > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730306",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gifts or donations amount must be provided",
                    LongDescription = @"When gifts or donations details are present in the attached Deductions schedule, the Gifts or donations amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.GiftDonation.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730306" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR148", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730307

        /* VR.ATO.IITR.730307
        When work related uniform, occupation specific or protective clothing, laundry and dry cleaning expense details are present in the attached Deductions schedule, the Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses amount must be provided

        Legacy Rule Format:
            (^IITR137 = NULL) AND (CountDocument('DDCTNS') = 1) AND Count(^DDCTNS:RP.{ClthngSeqNum}:Context) > 0

        Technical Business Rule Format:
            (^IITR137 = NULL) AND (CountDocument('DDCTNS') = 1) AND Count(^DDCTNS:RP.{ClthngSeqNum}:Context) > 0

        Data Elements:
    
        ^IITR137 = RP:IncomeTax.Deduction.Clothing.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_730307(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (!report.iitrParentDocument.IITR137.HasValue && ddctnsChildDocument.Rp_ClothingCollectionCount > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730307",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses must be provided",
                    LongDescription = @"When work related uniform, occupation specific or protective clothing, laundry and dry cleaning expense details are present in the attached Deductions schedule, the Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Clothing.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730307" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR137", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730308

        /* VR.ATO.IITR.730308
        The Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses amount must equal the sum of all Work related clothing expenses amounts in the attached Deductions schedule

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1) AND ((^IITR137 <> NULL) AND (^IITR137 <> RoundDown(Sum(^DDCTNS416),0)))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1) AND ((^IITR137 <> NULL) AND (^IITR137 <> RoundDown(Sum(^DDCTNS416),0)))

        Data Elements:
    
        ^IITR137 = RP:IncomeTax.Deduction.Clothing.Amount
    
        ^DDCTNS416 = DDCTNS416
        */
        public static ProcessMessageDocument VR_ATO_IITR_730308(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            decimal sumDDCTNS416 = Math.Truncate(ddctnsChildDocument.Rp_ClothingCollection.Sum(context => context.DDCTNS416 ?? 0));
            bool assertion = report.iitrParentDocument.IITR137.HasValue && report.iitrParentDocument.IITR137.Value != sumDDCTNS416;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730308",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses amount is incorrect",
                    LongDescription = @"The Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses amount must equal the sum of all Work related clothing expenses amounts in the attached Deductions schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.Clothing.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730308" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR137", Value = report.iitrParentDocument.IITR137.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS416", Value = sumDDCTNS416.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730301
        
        /* VR.ATO.IITR.730301
        The Date of last eligible superannuation contribution provided on the attached Deductions schedule must not be more than 28 days after the taxpayer's 75th birthday month

        Legacy Rule Format:
            (CountDocument('DDCTNS') = 1 AND (^DDCTNS403 > ConvertToDate(28, MonthAsInt^IITR29 + 1, Year^IITR29 + 75)))

        Technical Business Rule Format:
            (CountDocument('DDCTNS') = 1 AND (^DDCTNS403 > ConvertToDate(28, MonthAsInt^IITR29 + 1, Year^IITR29 + 75)))

        Data Elements:
    
        ^DDCTNS403 = DDCTNS403
    
        ^IITR29 = RP:PersonDemographicDetails.Birth.Date
        */
        public static ProcessMessageDocument VR_ATO_IITR_730301(IITR2026Crossform report, DDCTNS2026.Rp_PersonalSuperContributions context)
        {
            ProcessMessageDocument processMessage = null;

            DateTime? dateTaxpayerTurns75 = DateTaxpayerTurns75(report);

            bool assertion = (dateTaxpayerTurns75.HasValue && report.iitrParentDocument.IITR29.HasValue && context.DDCTNS403.HasValue && context.DDCTNS403.Value > new DateTime(dateTaxpayerTurns75.Value.Year, dateTaxpayerTurns75.Value.Month, 28).AddMonths(1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730301",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of last eligible superannuation contribution on the attached Deductions schedule is incorrect",
                    LongDescription = @"The Date of last eligible superannuation contribution provided on the attached Deductions schedule must not be more than 28 days after the taxpayer's 75th birthday month",
                    Location = $"/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions[{context.OccurrenceIndex}]/tns:IncomeTaxDeductionSuperannuationContributionEligibleD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730301" } }
                };
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR29", Value = GetValueOrEmpty(report.iitrParentDocument.IITR29) });
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "DDCTNS403", Value = GetValueOrEmpty(context.DDCTNS403) });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730302

        /* VR.ATO.IITR.730302
        Date of last eligible superannuation contribution must not be provided on the attached Deductions schedule unless the taxpayer turned 75 years of age within the financial year

        Legacy Rule Format:
            CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS403, ^DDCTNS403 <> NULL) AND (^IITR29 < ConvertToDate(Day(^RP:StartDate), MonthAsInt(^RP:StartDate), Year(^RP:StartDate) - 75) OR ^IITR29 > ConvertToDate(Day(^RP:EndDate), MonthAsInt(^RP:EndDate), Year(^RP:EndDate) - 75))

        Technical Business Rule Format:
            CountDocument('DDCTNS') = 1 AND AnyOccurrence(^DDCTNS403, ^DDCTNS403 <> NULL) AND (^IITR29 < ConvertToDate(Day(^RP:StartDate), MonthAsInt(^RP:StartDate), Year(^RP:StartDate) - 75) OR ^IITR29 > ConvertToDate(Day(^RP:EndDate), MonthAsInt(^RP:EndDate), Year(^RP:EndDate) - 75))

        Data Elements:
    
        ^DDCTNS403 = DDCTNS403
    
        ^EndDate = EndDate
    
        ^IITR29 = RP:PersonDemographicDetails.Birth.Date
    
        ^StartDate = StartDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_730302(IITR2026Crossform report, DDCTNS2026.Rp_PersonalSuperContributions context)
        {
            ProcessMessageDocument processMessage = null;

            DateTime? dateTaxpayerTurns75 = DateTaxpayerTurns75(report);
            bool taxpayerDidNotTurn75ThisPeriod = TaxpayerDidNotTurn75ThisPeriod(report);

            bool assertion = (report.iitrParentDocument.IITR29.HasValue && context.DDCTNS403.HasValue && dateTaxpayerTurns75.HasValue && taxpayerDidNotTurn75ThisPeriod);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730302",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of last eligible superannuation contribution must not be provided",
                    LongDescription = @"Date of last eligible superannuation contribution must not be provided on the attached Deductions schedule unless the taxpayer turned 75 years of age within the financial year",
                    Location = $"/tns:DDCTNS/tns:Rp/tns:PersonalSuperContributionsCollection/tns:PersonalSuperContributions[{context.OccurrenceIndex}]/tns:IncomeTaxDeductionSuperannuationContributionEligibleD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730302" } }              
                };
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR29", Value = GetValueOrEmpty(report.iitrParentDocument.IITR29) });
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "DDCTNS403", Value = GetValueOrEmpty(context.DDCTNS403) });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.200014

        /* VR.ATO.IITR.200014
        Where the 'Seniors and pensioners - Tax offset code' indicates at any time during the year the taxpayer was single or widowed, or separated; and all occurrences of Deduction claimed amounts are null, check deductible personal superannuation contributions.
    
        Legacy Rule Format:
        (^IITR163 = 'Single') AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))

        Technical Business Rule Format:
        (^IITR163 = 'Single') AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))
    
        Data Elements:
    
        ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code
    
        ^DDCTNS407 = DDCTNS407
        */
        public static ProcessMessageDocument VR_ATO_IITR_200014(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;
            int ddctnsScheduleCount = Count(report.ddctnsChildDocuments);

            bool noDDCTNS407Claimed = (ddctnsChildDocument == null || ddctnsChildDocument.Rp_PersonalSuperContributionsCollectionCount == 0) ||
                ddctnsChildDocument.Rp_PersonalSuperContributionsCollection.All(context => !context.DDCTNS407.HasValue);

            bool assertion = report.iitrParentDocument.IITR163 == @"Single" && (ddctnsScheduleCount == 0 || noDDCTNS407Claimed);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.201014",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check deductible personal superannuation contributions",
                    LongDescription = @"Rebate income is used to determine eligibility for the Seniors and pensioners tax offset. Where applicable, complete Deductible personal superannuation contributions from the Deductions schedule.",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SeniorAndPensioner.Code[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200014" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "IITR163",
                    Value = report.iitrParentDocument.IITR163
                });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DDCTNS407", Value = "null" });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730303

        /* VR.ATO.IITR.730303
        Income for surcharge purposes is used to determine if the taxpayer is liable for Medicare Levy Surcharge. Where applicable, complete Deductible personal superannuation contributions from the Deduction schedule.
    
        Legacy Rule Format:
        (^IITR176 <> TRUE) AND (^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))

        Technical Business Rule Format:
        (^IITR176 <> TRUE) AND (^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))
    
        Data Elements:
    
        ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator
    
        ^DDCTNS407 = DDCTNS407
    
        ^EndDate = EndDate
    
        ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count
    
        ^StartDate = StartDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_730303(IITR2026Crossform report, DDCTNS2026 ddctnsChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            int ddctnsScheduleCount = Count(report.ddctnsChildDocuments);

            bool noDDCTNS407Claimed = (ddctnsChildDocument == null || ddctnsChildDocument.Rp_PersonalSuperContributionsCollectionCount == 0) ||
                       ddctnsChildDocument.Rp_PersonalSuperContributionsCollection.All(context => !context.DDCTNS407.HasValue);

            bool assertion = report.iitrParentDocument.IITR176.HasValue && !report.iitrParentDocument.IITR176.Value //Indicated that not covered for whole period
                && report.iitrParentDocument.RPEndDate.HasValue && report.iitrParentDocument.RPStartDate.HasValue && report.iitrParentDocument.IITR177.HasValue //Ensure the values to be used have values
                && report.iitrParentDocument.IITR177.Value < (report.iitrParentDocument.RPEndDate.Value - report.iitrParentDocument.RPStartDate.Value).Days + 1 //Days exempt from surcharge less than whole period
                && (ddctnsScheduleCount == 0 || noDDCTNS407Claimed); //there is no DDCTNS schedule or there are no Super Contribution amounts claimed
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730303",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check deductible personal superannuation contributions",
                    LongDescription = @"Income for surcharge purposes is used to determine if the taxpayer is liable for Medicare Levy Surcharge. Where applicable, complete Deductible personal superannuation contributions from the Deduction schedule.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730303" } }
                };
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR176", Value = report.iitrParentDocument.IITR176.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "IITR177", Value = report.iitrParentDocument.IITR177.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter()
                { Name = "DDCTNS407", Value = "null" });
            }

            return processMessage;
        }

        #endregion

    }
}