using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationIITR
{

    public partial class IITR2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private IITR2026 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="IITR2026Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public IITR2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="IITR2026Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public IITR2026Validator(IITR2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());

        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public IITR2026 ConsumedReport { get { return report; } private set { report = value; } }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(IITR2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            #region Manual Rules - Rules marked for manual coding

            #region VR.ATO.IITR.000013

            /*  VR.ATO.IITR.000013
                Where the number of days not liable for medicare levy surcharge is less than the whole reporting period, and either income test is triggered; and the reportable employer superannuation contributions is greater than 8% of all income from salary and wages, check reportable employer superannuation contributions as MLS liability may be incurred.

            Legacy Rule Format:
                (^IITR177 = NULL OR ^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (((^IITR213 <> BLANK OR ^IITR171 <> NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum((ConditionalValue(CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, (^IITR492 + ^IITR493), 0) + ^IITR227 + ^IITR234 + ^IITR231) > (194000 + ConditionalValue(^IITR171 > 1, ((1500 * ^IITR171) - 1), 0)))) OR ((^IITR213 = BLANK) AND (^IITR171 = NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum(ConditionalValue((CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 97000))) AND (^IITR201 > (^IITR601 * 0.08))
            
            Technical Business Rule Format:
                (^IITR177 = NULL OR ^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (((^IITR213 <> BLANK OR ^IITR171 <> NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum((ConditionalValue(CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, (^IITR492 + ^IITR493), 0) + ^IITR227 + ^IITR234 + ^IITR231) > (194000 + ConditionalValue(^IITR171 > 1, ((1500 * ^IITR171) - 1), 0)))) OR ((^IITR213 = BLANK) AND (^IITR171 = NULL) AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue(((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) OR (^IITR29 <= ConvertToDate(1, 7, (^IITR10 - 61)))), Sum(ConditionalValue((CountDocument('INCDTLS') = 1) AND (^INCDTLS333 <> TRUE), ^INCDTLS330, 0)), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 97000))) AND (^IITR201 > (^IITR601 * 0.08))
            
            Data Elements:

            ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

            ^DDCTNS407 = DDCTNS407

            ^EndDate = EndDate

            ^IITR10 = RP:Report.TargetFinancial.Year

            ^IITR29 = RP:PersonDemographicDetails.Birth.Date

            ^IITR157 = RP:Income.Taxable.Amount

            ^IITR171 = RP:IncomeTax.MedicareLevy.DependentChildrenAndStudents.Count

            ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count

            ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

            ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

            ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text

            ^IITR225 = RP.SPOUSE:Income.Taxable.Amount

            ^IITR227 = RP.SPOUSE:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount

            ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount

            ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount

            ^IITR330 = RP:InternationalDealings.ExemptForeignEmploymentIncome.Amount

            ^IITR399 = RP:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount

            ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

            ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

            ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount

            ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount

            ^IITR601 = RP:Income.SalaryOrWages.Amount

            ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount

            ^INCDTLS330 = INCDTLS330

            ^INCDTLS333 = INCDTLS333

            ^StartDate = StartDate
            */
            VRATOIITR000013(response, report);
            #endregion // VR.ATO.IITR.000013

            #region VR.ATO.IITR.000222

            /*  VR.ATO.IITR.000222
                        Tax withheld from Australian Government allowances and payments like newstart, youth allowance and austudy payment are greater than or equal to the amount of Australian Government allowances and payments like Newstart, youth allowance and austudy payment.  Check both amounts and correct as necessary.

                    Legacy Rule Format:
                        (^IITR85 > 0) AND ((^IITR86 = NULL) OR (^IITR86 <= ^IITR85))

                Technical Business Rule Format:
                        (^IITR85 > 0) AND ((^IITR86 = NULL) OR (^IITR86 <= ^IITR85))

            Data Elements:

                ^IITR85 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentAllowancesAndPayments.Amount

                ^IITR86 = RP:Income.AustralianGovernmentAllowancesAndPayments.Amount
                */
            VRATOIITR000222(response, report);
            #endregion // VR.ATO.IITR.000222

            #region VR.ATO.IITR.000230

            /*  VR.ATO.IITR.000230
                        Other deductions relating to amounts shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts" type has been completed without an amount at Other deductions relating to amounts shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts".

                    Legacy Rule Format:
                        (^IITR265 = NULL OR ^IITR265 = 0) AND (^IITR266 <> NULL)

                Technical Business Rule Format:
                        (^IITR265 = NULL OR ^IITR265 = 0) AND (^IITR266 <> NULL)

            Data Elements:

                ^IITR265 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount

                ^IITR266 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Code
                */
            VRATOIITR000230(response, report);
            #endregion // VR.ATO.IITR.000230

            #region VR.ATO.IITR.000236

            /*  VR.ATO.IITR.000236
                        Total tax withheld does not equal the sum of all tax withheld items

                    Legacy Rule Format:
                        ((^IITR113 <> NULL) AND (^IITR113 <> (^IITR85 + ^IITR88 + ^IITR600 + ^IITR606 + ^IITR608 + ^IITR91 + ^IITR111 + ^IITR64 + ^IITR625 + ^IITR610))) OR ((^IITR113 = NULL) AND ((^IITR85 + ^IITR88 + ^IITR600 + ^IITR606 + ^IITR608 + ^IITR91 + ^IITR111 + ^IITR625 + ^IITR610) <> 0))

                Technical Business Rule Format:
                        ((^IITR113 <> NULL) AND (^IITR113 <> (^IITR85 + ^IITR88 + ^IITR600 + ^IITR606 + ^IITR608 + ^IITR91 + ^IITR111 + ^IITR64 + ^IITR625 + ^IITR610))) OR ((^IITR113 = NULL) AND ((^IITR85 + ^IITR88 + ^IITR600 + ^IITR606 + ^IITR608 + ^IITR91 + ^IITR111 + ^IITR625 + ^IITR610) <> 0))

            Data Elements:

                ^IITR113 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheld.Amount

                ^IITR64 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAllowancesEarningsTipsDirectorsFees.Amount

                ^IITR85 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentAllowancesAndPayments.Amount

                ^IITR88 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount

                ^IITR91 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount

                ^IITR111 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount

                ^IITR600 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSalaryOrWages.Amount

                ^IITR606 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount

                ^IITR608 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount

                ^IITR610 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSuperannuationLumpSumPayment.Amount

                ^IITR625 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmploymentTerminationPayment.Amount
                */
            VRATOIITR000236(response, report);
            #endregion // VR.ATO.IITR.000236

            #region VR.ATO.IITR.000237

            /*  VR.ATO.IITR.000237
                        Share of net income from trusts less capital gains, foreign income and franked distributions is greater than zero and the Share of net income from trusts less capital gains, foreign income and franked distributions action code is missing.

                    Legacy Rule Format:
                        (^IITR255 > 0) AND (^IITR256 = NULL)

                Technical Business Rule Format:
                        (^IITR255 > 0) AND (^IITR256 = NULL)

            Data Elements:

                ^IITR256 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Code

                ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
                */
            VRATOIITR000237(response, report);
            #endregion // VR.ATO.IITR.000237

            #region VR.ATO.IITR.000263

            /*  VR.ATO.IITR.000263
                        The total of all income fields does not equal the amount at the Total income or loss field.

                    Legacy Rule Format:
                        ((^IITR131 <> NULL) AND OutsideRange(^IITR131, (^IITR86 + ^IITR89 + ^IITR601 + ^IITR607 + (^IITR609*0.05) + ^IITR626 + ^IITR65 + ^IITR92 + ^IITR93 + ^IITR505 + ^IITR94 + ^IITR95 + ^IITR611 + ^IITR612 + ^IITR112 + ^IITR115 + ^IITR118 + ^IITR119 + ^IITR120 + ^IITR127 + ^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR252 + ^IITR255 + ^IITR259 - ^IITR260 - ^IITR265 + ^IITR279 + ^IITR281 + ^IITR285 + ^IITR299 + ^IITR300 + ^IITR306 + ^IITR311 + ^IITR316 + ^IITR318 + ^IITR321 + ^IITR322 + ^IITR323 + ^IITR324 + ^IITR327 + ^IITR328 + ^IITR334 - ^IITR335 - ^IITR336 - ^IITR337 + ^IITR339 + ^IITR341 + ^IITR615 + ^IITR616 + ^IITR650 + ^IITR618), 1)) OR ((^IITR131 = NULL) AND OutsideRange(0, (^IITR86 + ^IITR89 + ^IITR601 + ^IITR607 + (^IITR609*0.05) + ^IITR626 + ^IITR65 + ^IITR92 + ^IITR93 + ^IITR505 + ^IITR94 + ^IITR95 + ^IITR611 + ^IITR612 + ^IITR112 + ^IITR115 + ^IITR118 + ^IITR119 + ^IITR120 + ^IITR127 + ^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR252 + ^IITR255 + ^IITR259 - ^IITR260 - ^IITR265 + ^IITR279 + ^IITR281 + ^IITR285 + ^IITR299 + ^IITR300 + ^IITR306 + ^IITR311 + ^IITR316 + ^IITR318 + ^IITR321 + ^IITR322 + ^IITR323 + ^IITR324 + ^IITR327 + ^IITR328 + ^IITR334 - ^IITR335 - ^IITR336 - ^IITR337 + ^IITR339 + ^IITR341 + ^IITR615 + ^IITR616 + ^IITR650 + ^IITR618), 1))

                Technical Business Rule Format:
                        ((^IITR131 <> NULL) AND OutsideRange(^IITR131, (^IITR86 + ^IITR89 + ^IITR601 + ^IITR607 + (^IITR609*0.05) + ^IITR626 + ^IITR65 + ^IITR92 + ^IITR93 + ^IITR505 + ^IITR94 + ^IITR95 + ^IITR611 + ^IITR612 + ^IITR112 + ^IITR115 + ^IITR118 + ^IITR119 + ^IITR120 + ^IITR127 + ^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR252 + ^IITR255 + ^IITR259 - ^IITR260 - ^IITR265 + ^IITR279 + ^IITR281 + ^IITR285 + ^IITR299 + ^IITR300 + ^IITR306 + ^IITR311 + ^IITR316 + ^IITR318 + ^IITR321 + ^IITR322 + ^IITR323 + ^IITR324 + ^IITR327 + ^IITR328 + ^IITR334 - ^IITR335 - ^IITR336 - ^IITR337 + ^IITR339 + ^IITR341 + ^IITR615 + ^IITR616 + ^IITR650 + ^IITR618), 1)) OR ((^IITR131 = NULL) AND OutsideRange(0, (^IITR86 + ^IITR89 + ^IITR601 + ^IITR607 + (^IITR609*0.05) + ^IITR626 + ^IITR65 + ^IITR92 + ^IITR93 + ^IITR505 + ^IITR94 + ^IITR95 + ^IITR611 + ^IITR612 + ^IITR112 + ^IITR115 + ^IITR118 + ^IITR119 + ^IITR120 + ^IITR127 + ^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR252 + ^IITR255 + ^IITR259 - ^IITR260 - ^IITR265 + ^IITR279 + ^IITR281 + ^IITR285 + ^IITR299 + ^IITR300 + ^IITR306 + ^IITR311 + ^IITR316 + ^IITR318 + ^IITR321 + ^IITR322 + ^IITR323 + ^IITR324 + ^IITR327 + ^IITR328 + ^IITR334 - ^IITR335 - ^IITR336 - ^IITR337 + ^IITR339 + ^IITR341 + ^IITR615 + ^IITR616 + ^IITR650 + ^IITR618), 1))

            Data Elements:

                ^IITR131 = RP:Income.Net.Amount

                ^IITR65 = RP:Income.AllowancesEarningsTipsDirectorsFees.Amount

                ^IITR86 = RP:Income.AustralianGovernmentAllowancesAndPayments.Amount

                ^IITR89 = RP:Pension.Total.Amount

                ^IITR92 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount

                ^IITR93 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount

                ^IITR94 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount

                ^IITR95 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount

                ^IITR112 = RP:Income.PersonalServicesIncome.Total.Amount

                ^IITR115 = RP:Income.Interest.Gross.Amount

                ^IITR118 = RP:Income.DividendsUnfranked.Amount

                ^IITR119 = RP:Income.DividendsFranked.Amount

                ^IITR120 = RP:IncomeTax.FrankingCredits.Amount

                ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount

                ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount

                ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount

                ^IITR245 = RP.Partnership:Expense.LandcareOperationsAcceleratedDepreciation.Amount

                ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount

                ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount

                ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount

                ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount

                ^IITR260 = RP:Expense.LandcareOperationsNonPrimaryProduction.Amount

                ^IITR265 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount

                ^IITR279 = RP:Income.PersonalServicesIncome.Net.Amount

                ^IITR281 = RP:Income.PrimaryProductionNet.Amount

                ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount

                ^IITR299 = RP:Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount

                ^IITR300 = RP:Tax.Losses.DeferredNonCommercialNonPrimaryProduction.Amount

                ^IITR306 = RP:Income.FarmManagementDepositsOrRepaymentsNet.Amount

                ^IITR311 = RP:Capital.Gains.Net.Amount

                ^IITR316 = RP:Income.ControlledForeignCompaniesAttributableIncome.Amount

                ^IITR318 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount

                ^IITR321 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount

                ^IITR322 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeWithoutUndeductedPurchasePrice.Amount

                ^IITR323 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeUndeductedPurchasePrice.Amount

                ^IITR324 = RP:InternationalDealings.ForeignRent.Amount

                ^IITR327 = RP:Income.InternationalDealings.Net.Amount

                ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount

                ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount

                ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount

                ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount

                ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount

                ^IITR339 = RP:IncomeTax.LifeInsuranceCompaniesAndFriendlySocietiesBonuses.Amount

                ^IITR341 = RP:Income.ForestryManagedInvestmentScheme.Amount

                ^IITR505 = RP:Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount

                ^IITR601 = RP:Income.SalaryOrWages.Amount

                ^IITR607 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount

                ^IITR609 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount

                ^IITR611 = RP:Income.Superannuation.LumpSumPaymentTaxableComponentTaxedElement.Amount

                ^IITR612 = RP:Income.Superannuation.LumpSumPaymentTaxableComponentUntaxedElement.Amount

                ^IITR615 = RP:IncomeTax.IncomeCategory1Other.Amount

                ^IITR616 = RP:IncomeTax.IncomeCategory2Total.Amount

                ^IITR618 = RP:IncomeTax.IncomeCategoryMiscellaneousOther.Amount

                ^IITR626 = RP:Income.EmploymentTerminationPaymentTaxable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
                */
            VRATOIITR000263(response, report);
            #endregion // VR.ATO.IITR.000263

            #region VR.ATO.IITR.000604

            /*  VR.ATO.IITR.000604
                        Private health insurance policy has an entry with Tax claim code "Claiming your spouses share of rebate" then there must be a matching entry with Tax claim code "Claiming your share of rebate". The number of lines with tax claim code "Claiming your spouses share of rebate" must be equal to the number of lines with tax claim code "Claiming your share of rebate" for a particular policy, but this cannot be enforced due to technical constraints.

                    Legacy Rule Format:
                        ^IITR185 = 'Claiming your spouses share of rebate' AND CountOccurrence(^IITR185, ^IITR185 = 'Claiming your share of rebate' WHERE (HasDuplicateValues(^IITR180) AND HasDuplicateValues(^IITR181))) = 0

                Technical Business Rule Format:
                        ^IITR185 = 'Claiming your spouses share of rebate' AND CountOccurrence(^IITR185, ^IITR185 = 'Claiming your share of rebate' WHERE (HasDuplicateValues(^IITR180) AND HasDuplicateValues(^IITR181))) = 0

            Data Elements:

                ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code

                ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier

                ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier
                */
            VRATOIITR000604(response, report);
            #endregion // VR.ATO.IITR.000604

            #region VR.ATO.IITR.000645

            /*  VR.ATO.IITR.000645
                        Unable to confirm whether the correct Non-resident foreign income amount has been provided for the simple self-assessment method, as the Foreign occupation code provided on the Non-Resident Foreign Income (NRFI) schedule is not recognised. Record the details of the message and contact the agency service desk.

                    Legacy Rule Format:
                        CountDocument('NRFI') = 1 AND ^NRFI3 = 'Simple' AND ^NRFI2 <> NULL AND NotInDomain(Substring(^NRFI2,1,4), 'OCCUPATION PREFIX CODES')

                Technical Business Rule Format:
                        CountDocument('NRFI') = 1 AND ^NRFI3 = 'Simple' AND ^NRFI2 <> NULL AND NotInDomain(Substring(^NRFI2,1,4), 'OCCUPATION PREFIX CODES')

            Data Elements:

                ^NRFI2 = NRFI2

                ^NRFI3 = NRFI3
                */
            VRATOIITR000645(response, report);
            #endregion // VR.ATO.IITR.000645

            #region VR.ATO.IITR.100031

            /*  VR.ATO.IITR.100031
                        For a resident taxpayer, where the sum of taxable income or loss and the net loss amounts that exceeds or equal to zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

                    Legacy Rule Format:
                        ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0))) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

                Technical Business Rule Format:
                        ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0))) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","5","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

            Data Elements:

                ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount

                ^DDCTNS407 = DDCTNS407

                ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator

                ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount

                ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount

                ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount

                ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
                */
            VRATOIITR100031(response, report);
            #endregion // VR.ATO.IITR.100031

            #region VR.ATO.IITR.100032

            /*  VR.ATO.IITR.100032
                        For a resident taxpayer where the sum of taxable income or loss and the net loss amounts is less than zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

                    Legacy Rule Format:
                        ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0))) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

                Technical Business Rule Format:
                        ^IITR19 = TRUE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0))) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

            Data Elements:

                ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount

                ^DDCTNS407 = DDCTNS407

                ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator

                ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount

                ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount

                ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount

                ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
                */
            VRATOIITR100032(response, report);
            #endregion // VR.ATO.IITR.100032

            #region VR.ATO.IITR.100033

            /*  VR.ATO.IITR.100033
                        For a non-resident taxpayer where the sum of taxable income or loss and the net loss amounts exceeds or equal to zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

                    Legacy Rule Format:
                        ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328 - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

                Technical Business Rule Format:
                        ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) >= 0 AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328 - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

            Data Elements:

                ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount

                ^DDCTNS407 = DDCTNS407

                ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator

                ^IITR115 = RP:Income.Interest.Gross.Amount

                ^IITR118 = RP:Income.DividendsUnfranked.Amount

                ^IITR119 = RP:Income.DividendsFranked.Amount

                ^IITR120 = RP:IncomeTax.FrankingCredits.Amount

                ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount

                ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount

                ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount

                ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount

                ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount

                ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount

                ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
                */
            VRATOIITR100033(response, report);
            #endregion // VR.ATO.IITR.100033

            #region VR.ATO.IITR.100034

            /*  VR.ATO.IITR.100034
                        For a non-resident taxpayer where the sum of taxable income or loss and the net loss amounts is less than zero, the reported income for the purposes of the non-commercial losses income test is equal to $250,000 or more. Ensure loss codes used and/or amounts reported at the relevant labels are correct. A copy of the formula used to calculate income for this test can be found at www.ato.gov.au/ncl

                    Legacy Rule Format:
                        ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

                Technical Business Rule Format:
                        ^IITR19 = FALSE AND AnyOccurrence(^IITR482, InSet(^IITR482, '"1","2","3","4"')) AND ((^IITR157 - ^IITR650) + Sum(ConditionalValue(NotInSet(^IITR482, '"0","8"'), ^IITR487, 0)) + ^IITR146 + ^IITR147 - ^IITR115 - ^IITR118 - ^IITR119 - ^IITR120 - ^IITR328) < 0 AND (ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - (^IITR153 - ^IITR154) - (^IITR155 - ^IITR156)) >= 250000

            Data Elements:

                ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount

                ^DDCTNS407 = DDCTNS407

                ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator

                ^IITR115 = RP:Income.Interest.Gross.Amount

                ^IITR118 = RP:Income.DividendsUnfranked.Amount

                ^IITR119 = RP:Income.DividendsFranked.Amount

                ^IITR120 = RP:IncomeTax.FrankingCredits.Amount

                ^IITR146 = RP:IncomeTax.Deduction.Interest.Amount

                ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount

                ^IITR153 = RP:IncomeTax.LossesBroughtForwardPrimaryProduction.Amount

                ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount

                ^IITR155 = RP:IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount

                ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount

                ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
                */
            VRATOIITR100034(response, report);
            #endregion // VR.ATO.IITR.100034

            #region VR.ATO.IITR.100043

            /*  VR.ATO.IITR.100043
                        Adjusted taxable income must equal the sum of (when amount above $0): Taxable income or loss income (excluding any assessable First home super saver released amount), Reportable employer superannuation contributions, Tax-free government pensions, Target foreign income, Net financial investment loss, Net rental property loss, and all Deduction claimed amounts, plus Total reportable fringe benefits amount from employers not exempt under section 57A and 53.0% of Total reportable fringe benefits amount from employers exempt under section 57A (when combined Total reportable fringe benefits amount above $3,772), less Child support you paid. If you are completing a tax return for a deceased person, this value must be divided by the number of days the taxpayer was alive during the financial year and multiplied by the total number of days in the financial year.

                    Legacy Rule Format:
                        ((^IITR28 = NULL OR ^IITR28 > ^RP:EndDate) AND ((^IITR208 <> NULL AND ^IITR208 <> ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1)) OR (^IITR208 = NULL AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1) <> 0))) OR ((^IITR28 > (^RP:StartDate - 1) AND ^IITR28 < (^RP:EndDate + 1)) AND ((^IITR208 <> NULL AND ^IITR208 <> (((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1)) OR (^IITR208 = NULL AND ((((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1) <> 0)))) OR (^IITR28 < ^RP:StartDate AND ^IITR208 <> 0)

                Technical Business Rule Format:
                        ((^IITR28 = NULL OR ^IITR28 > ^RP:EndDate) AND ((^IITR208 <> NULL AND ^IITR208 <> ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1)) OR (^IITR208 = NULL AND ((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) +/- 1) <> 0))) OR ((^IITR28 > (^RP:StartDate - 1) AND ^IITR28 < (^RP:EndDate + 1)) AND ((^IITR208 <> NULL AND ^IITR208 <> (((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1)) OR (^IITR208 = NULL AND ((((ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ^IITR202 + ^IITR203 + ^IITR204 + ^IITR205 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) - ^IITR206) * (^RP:EndDate - ^RP:StartDate + 1) / (^IITR28 - ^RP:StartDate + 1)) +/- 1) <> 0)))) OR (^IITR28 < ^RP:StartDate AND ^IITR208 <> 0)

            Data Elements:

                ^IITR208 = RP:Income.TaxableAdjusted.Amount

                ^DDCTNS407 = DDCTNS407

                ^EndDate = EndDate

                ^IITR28 = RP:PersonDemographicDetails.Death.Date

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR202 = RP:Pension.TaxFree.Amount

                ^IITR203 = RP:InternationalDealings.ForeignIncomeTarget.Amount

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR206 = RP:ChildSupport.Payment.Amount

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount

                ^StartDate = StartDate
                */
            VRATOIITR100043(response, report);
            #endregion // VR.ATO.IITR.100043

            #region VR.ATO.IITR.200014

            /*  VR.ATO.IITR.200014
                        Where the 'Seniors and pensioners - Tax offset code' indicates at any time during the year the taxpayer was single or widowed, or separated; and all occurrences of Deduction claimed amounts are null, check deductible personal superannuation contributions.

                    Legacy Rule Format:
                        (^IITR163 = 'Single') AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))

                Technical Business Rule Format:
                        (^IITR163 = 'Single') AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))

            Data Elements:

                ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code

                ^DDCTNS407 = DDCTNS407
                */
            VRATOIITR200014(response, report);
            #endregion // VR.ATO.IITR.200014

            #region VR.ATO.IITR.200015

            /*  VR.ATO.IITR.200015
                        Where the 'Seniors and pensioners - Tax offset code' is being claimed, and the sum of income fields is greater than $83,580 where there is a spouse, or $50,119 where there is no spouse; and the Reportable employer superannuation contributions is greater than 8% of all income from salary and wages, check Reportable employer superannuation contributions as SPO entitlement may be impacted.

                    Legacy Rule Format:
                        ^IITR163 <> BLANK AND ((^IITR213 <> BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205 + ^IITR225 + ^IITR230 + ^IITR226 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR231 + ^IITR234) > 83580) OR (^IITR213 = BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205) > 50119)) AND ^IITR201 > (^IITR601 * 0.08)

                Technical Business Rule Format:
                        ^IITR163 <> BLANK AND ((^IITR213 <> BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205 + ^IITR225 + ^IITR230 + ^IITR226 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR231 + ^IITR234) > 83580) OR (^IITR213 = BLANK AND (ConditionalValue((^IITR157 - ^IITR650) > 0, (^IITR157 - ^IITR650), 0) + ConditionalValue((^IITR490 + ^IITR491) > 3772, ((^IITR490 * 0.530) + ^IITR491), 0) + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0) + ^IITR204 + ^IITR205) > 50119)) AND ^IITR201 > (^IITR601 * 0.08)

            Data Elements:

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^DDCTNS407 = DDCTNS407

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text

                ^IITR225 = RP.SPOUSE:Income.Taxable.Amount

                ^IITR226 = RP.SPOUSE:IncomeTax.TrustIncomeNotIncludedInTaxableIncome.Amount

                ^IITR230 = RP.SPOUSE:Pension.SpouseExempt.Amount

                ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount

                ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR601 = RP:Income.SalaryOrWages.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount
                */
            VRATOIITR200015(response, report);
            #endregion // VR.ATO.IITR.200015

            #region VR.ATO.IITR.200034

            /*  VR.ATO.IITR.200034
                        Spouse's or de facto spouse's family name is present and the taxpayer had a spouse for the full year and had no Medicare levy surcharge. Combined Total reportable fringe benefits amounts exceed $3,772 and taxable income or loss (excluding any assessable First home super saver released amount) plus total reportable fringe benefits amounts plus net amount on which family trust has been paid plus exempt foreign employment income exceeds $27,222 and number of days not liable for MLS, spouse income details also required.

                    Legacy Rule Format:
                        (^IITR213 <> BLANK OR ^IITR220 = TRUE) AND ^IITR176 = FALSE AND ((^IITR157 - ^IITR650) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue((CountDocument('INCDTLS') = 1) AND (AnyOccurrence(^INCDTLS333, ^INCDTLS333 <> TRUE) AND ^IITR29 <> NULL AND (^IITR29 > ConvertToDate(30,06,(^IITR10 - 55)) OR ^IITR29 <= ConvertToDate(01,07,(^IITR10 - 61)))), Sum(^INCDTLS330), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 27222 AND (^IITR177 = NULL OR (^IITR225 = NULL AND ^IITR226 = NULL AND ^IITR227 = NULL AND ^IITR492 = NULL AND ^IITR493 = NULL AND ^IITR231 = NULL AND ^IITR234 = NULL))

                Technical Business Rule Format:
                        (^IITR213 <> BLANK OR ^IITR220 = TRUE) AND ^IITR176 = FALSE AND ((^IITR157 - ^IITR650) + ConditionalValue((^IITR490 + ^IITR491) > 3772, (^IITR490 + ^IITR491), 0) + ConditionalValue((^IITR157 - ^IITR650) > 0, ^IITR330, 0) + ^IITR399 - ConditionalValue((CountDocument('INCDTLS') = 1) AND (AnyOccurrence(^INCDTLS333, ^INCDTLS333 <> TRUE) AND ^IITR29 <> NULL AND (^IITR29 > ConvertToDate(30,06,(^IITR10 - 55)) OR ^IITR29 <= ConvertToDate(01,07,(^IITR10 - 61)))), Sum(^INCDTLS330), 0) + ^IITR204 + ^IITR205 + ^IITR201 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(^DDCTNS407), 0), 0)) > 27222 AND (^IITR177 = NULL OR (^IITR225 = NULL AND ^IITR226 = NULL AND ^IITR227 = NULL AND ^IITR492 = NULL AND ^IITR493 = NULL AND ^IITR231 = NULL AND ^IITR234 = NULL))

            Data Elements:

                ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count

                ^DDCTNS407 = DDCTNS407

                ^IITR10 = RP:Report.TargetFinancial.Year

                ^IITR29 = RP:PersonDemographicDetails.Birth.Date

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text

                ^IITR220 = RP:PersonDemographicDetails.SpouseFullYear.Indicator

                ^IITR225 = RP.SPOUSE:Income.Taxable.Amount

                ^IITR226 = RP.SPOUSE:IncomeTax.TrustIncomeNotIncludedInTaxableIncome.Amount

                ^IITR227 = RP.SPOUSE:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount

                ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount

                ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount

                ^IITR330 = RP:InternationalDealings.ExemptForeignEmploymentIncome.Amount

                ^IITR399 = RP:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount

                ^IITR490 = RP.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR491 = RP.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount

                ^IITR650 = RP:IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount

                ^INCDTLS330 = INCDTLS330

                ^INCDTLS333 = INCDTLS333
                */
            VRATOIITR200034(response, report);
            #endregion // VR.ATO.IITR.200034

            #region VR.ATO.IITR.210034

            /*  VR.ATO.IITR.210034
                        If two or more rows have been entered for one policy, tax claim codes must be the same on all rows except where the tax claim code is "Claiming your spouses share of rebate" or "Covered as dependent child" in any of those rows. Tax claim code(s) must be corrected.

                    Legacy Rule Format:
                        ^IITR185 <> NULL AND NotInSet(^IITR185, '"Claiming your spouses share of rebate","Covered as dependent child"') AND (NotSameValues(^IITR180) WHERE NotInSet(^IITR185, '"Claiming your spouses share of rebate","Covered as dependent child"') OR NotSameValues(^IITR181) WHERE NotInSet(^IITR185, '"Claiming your spouses share of rebate","Covered as dependent child"'))

                Technical Business Rule Format:
                        ^IITR185 <> NULL AND NotInSet(^IITR185, '"Claiming your spouses share of rebate","Covered as dependent child"') AND (NotSameValues(^IITR180) WHERE NotInSet(^IITR185, '"Claiming your spouses share of rebate","Covered as dependent child"') OR NotSameValues(^IITR181) WHERE NotInSet(^IITR185, '"Claiming your spouses share of rebate","Covered as dependent child"'))

            Data Elements:

                ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code

                ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier

                ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier
                */
            VRATOIITR210034(response, report);
            #endregion // VR.ATO.IITR.210034

            #region VR.ATO.IITR.210035

            /*  VR.ATO.IITR.210035
                        You will need the private health insurance statement to complete the private health insurance policy details. Ensure that you have only entered the amounts shown on the statement at Your premiums eligible for Australian Government rebate and Your Australian Government rebate received. These values are dollar amounts.

                    Legacy Rule Format:
                        (InSet(^IITR185, '"No dependants", "Dependants", "Claiming your share of rebate", "Claiming your spouses share of rebate"')) AND (^IITR183 > 0) AND ((^IITR182 <> NULL AND ^IITR183 >= ^IITR182) OR (^IITR182 = NULL))

                Technical Business Rule Format:
                        (InSet(^IITR185, '"No dependants", "Dependants", "Claiming your share of rebate", "Claiming your spouses share of rebate"')) AND (^IITR183 > 0) AND ((^IITR182 <> NULL AND ^IITR183 >= ^IITR182) OR (^IITR182 = NULL))

            Data Elements:

                ^IITR182 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.RebatableComponent.Amount

                ^IITR183 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.GovernmentRebateReceived.Amount

                ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code
                */
            VRATOIITR210035(response, report);
            #endregion // VR.ATO.IITR.210035

            #region VR.ATO.IITR.300005

            /*  VR.ATO.IITR.300005
                        Where the sum of the taxpayers adjusted taxable income and their spouse's adjusted taxable income is greater than $117,194; and the reportable employer superannuation contributions is greater than 8% of Salary or wages gross income, check reportable employer superannuation contributions.

                    Legacy Rule Format:
                        (^IITR208 + ^IITR237) > 112578 AND ^IITR201 > (^IITR601 * 0.08)

                Technical Business Rule Format:
                        (^IITR208 + ^IITR237) > 112578 AND ^IITR201 > (^IITR601 * 0.08)

            Data Elements:

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount

                ^IITR208 = RP:Income.TaxableAdjusted.Amount

                ^IITR237 = RP.SPOUSE:Income.TaxableAdjusted.Amount

                ^IITR601 = RP:Income.SalaryOrWages.Amount
                */
            VRATOIITR300005(response, report);
            #endregion // VR.ATO.IITR.300005

            #region VR.ATO.IITR.300039

            /*  VR.ATO.IITR.300039
                        A loss amount has been entered at primary production business income or non-primary production business income or Net Personal Services Income, but there are no Business loss activity details.

                    Legacy Rule Format:
                        (^IITR281 < 0 OR ^IITR285 < 0 OR ^IITR279 < 0) AND ^IITR479 = BLANK

                Technical Business Rule Format:
                        (^IITR281 < 0 OR ^IITR285 < 0 OR ^IITR279 < 0) AND ^IITR479 = BLANK

            Data Elements:

                ^IITR479 = RP.{LOSSSeqNum}:OrganisationDetails.MainIncomeActivity.Description

                ^IITR279 = RP:Income.PersonalServicesIncome.Net.Amount

                ^IITR281 = RP:Income.PrimaryProductionNet.Amount

                ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
                */
            VRATOIITR300039(response, report);
            #endregion // VR.ATO.IITR.300039

            #region VR.ATO.IITR.300040

            /*  VR.ATO.IITR.300040
                        A loss has been entered at PP distribution from partnerships or non-PP distribution from partnerships and business loss activity details have not been entered.

                    Legacy Rule Format:
                        (^IITR242 < 0 OR ^IITR252 < 0) AND (COUNT(RP.{LOSSSeqNum}) = 0 OR (^IITR573 = 1 AND ^IITR479 = BLANK))

                Technical Business Rule Format:
                        (^IITR242 < 0 OR ^IITR252 < 0) AND (COUNT(RP.{LOSSSeqNum}) = 0 OR (^IITR573 = 1 AND ^IITR479 = BLANK))

            Data Elements:

                ^IITR479 = RP.{LOSSSeqNum}:OrganisationDetails.MainIncomeActivity.Description

                ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount

                ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount

                ^IITR573 = RP.{LOSSSeqNum}:Report.ItemOrder.Number
                */
            VRATOIITR300040(response, report);
            #endregion // VR.ATO.IITR.300040

            #region VR.ATO.IITR.300087

            /*  VR.ATO.IITR.300087
                        Taxpayer is ineligible to make deposits and claim deductions for deductible deposits because taxpayer's non-primary production income exceeded the threshold amount.

                    Legacy Rule Format:
                        (^IITR302 > 0) AND ((^IITR157 + ^IITR154 + ^IITR156 - ^IITR311 - (^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR281 + ^IITR299 + ^IITR306) - ConditionalValue(CountDocument('INCDTLS') = 1,Sum(ConditionalValue(InSet(^INCDTLS138, '"D","B","N"'),^INCDTLS140,0)),0) - Sum(ConditionalValue((CountDocument('INCDTLS') = 1) AND (^INCDTLS333 = FALSE),(^INCDTLS330 + ^INCDTLS331),0))) > 100000)

                Technical Business Rule Format:
                        (^IITR302 > 0) AND ((^IITR157 + ^IITR154 + ^IITR156 - ^IITR311 - (^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR281 + ^IITR299 + ^IITR306) - ConditionalValue(CountDocument('INCDTLS') = 1,Sum(ConditionalValue(InSet(^INCDTLS138, '"D","B","N"'),^INCDTLS140,0)),0) - Sum(ConditionalValue((CountDocument('INCDTLS') = 1) AND (^INCDTLS333 = FALSE),(^INCDTLS330 + ^INCDTLS331),0))) > 100000)

            Data Elements:

                ^IITR302 = RP:IncomeTax.Deduction.FarmManagementDeposits.Amount

                ^IITR154 = RP:IncomeTax.LossesClaimedPrimaryProduction.Amount

                ^IITR156 = RP:IncomeTax.LossesClaimedNonPrimaryProduction.Amount

                ^IITR157 = RP:Income.Taxable.Amount

                ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount

                ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount

                ^IITR245 = RP.Partnership:Expense.LandcareOperationsAcceleratedDepreciation.Amount

                ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount

                ^IITR281 = RP:Income.PrimaryProductionNet.Amount

                ^IITR299 = RP:Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount

                ^IITR306 = RP:Income.FarmManagementDepositsOrRepaymentsNet.Amount

                ^IITR311 = RP:Capital.Gains.Net.Amount

                ^INCDTLS138 = INCDTLS138

                ^INCDTLS140 = INCDTLS140

                ^INCDTLS330 = INCDTLS330

                ^INCDTLS331 = INCDTLS331

                ^INCDTLS333 = INCDTLS333
                */
            VRATOIITR300087(response, report);
            #endregion // VR.ATO.IITR.300087

            #region VR.ATO.IITR.300093

            /*  VR.ATO.IITR.300093
                        If the total of financial investment income minus financial investment deductions is less than zero then the difference must be inserted at Net financial investment loss. If the difference exceeds zero then zero must be inserted at Net financial investment loss.

                    Legacy Rule Format:
                        (^IITR13 = NULL) AND ((((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND ((^IITR204 = NULL) OR (^IITR204 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 +^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0)) - (^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)))))) OR (((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND (^IITR204 = NULL OR ^IITR204 <> 0)))

                Technical Business Rule Format:
                        (^IITR13 = NULL) AND ((((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND ((^IITR204 = NULL) OR (^IITR204 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 +^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0)) - (^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)))))) OR (((^IITR118 + ^IITR119 + ^IITR120 + ^IITR249 + ^IITR282 + ^IITR291 + ^IITR295 + ^IITR325 + ^IITR328+ ^IITR341 + ^IITR613 + ^IITR617 + ConditionalValue(CountDocument('INCDTLS') = 1, (RoundDown((Sum(^INCDTLS251) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS389, ^INCDTLS419))), 0) + RoundDown((Sum(^INCDTLS254) + Sum(ConditionalValue(^INCDTLS387 = 1, ^INCDTLS390, ^INCDTLS420))), 0)), 0)) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS310, 0), 0) + ^IITR147 + ^IITR261 + ^IITR262 + ^IITR368 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'D - Deductions relating to financial investments', ^DDCTNS138, 0)), 0), 0))) AND (^IITR204 = NULL OR ^IITR204 <> 0)))

            Data Elements:

                ^IITR204 = RP:Tax.Losses.FinancialInvestmentNet.Amount

                ^DDCTNS136 = DDCTNS136

                ^DDCTNS138 = DDCTNS138

                ^DDCTNS310 = DDCTNS310

                ^IITR13 = RP:IncomeTax.ReturnType.Code

                ^IITR118 = RP:Income.DividendsUnfranked.Amount

                ^IITR119 = RP:Income.DividendsFranked.Amount

                ^IITR120 = RP:IncomeTax.FrankingCredits.Amount

                ^IITR147 = RP:IncomeTax.Deduction.Dividend.Amount

                ^IITR249 = RP:Income.PartnershipInvestmentLessForeignIncomeNonPrimaryProductionShare.Amount

                ^IITR261 = RP:IncomeTax.Deduction.TrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProduction.Amount

                ^IITR262 = RP:IncomeTax.Deduction.PartnershipExcludingForeignIncomeNonPrimaryProduction.Amount

                ^IITR282 = RP:Income.InvestmentNonPrimaryProductionNet.Amount

                ^IITR291 = RP:Tax.Losses.DeferredNonCommercialPartnershipInvestmentShare.Amount

                ^IITR295 = RP:Tax.Losses.DeferredNonCommercialSoleTraderInvestment.Amount

                ^IITR325 = RP:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount

                ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount

                ^IITR341 = RP:Income.ForestryManagedInvestmentScheme.Amount

                ^IITR368 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount

                ^IITR613 = RP:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount

                ^IITR617 = RP:IncomeTax.FinancialInvestmentsOther.Amount

                ^INCDTLS251 = INCDTLS251

                ^INCDTLS254 = INCDTLS254

                ^INCDTLS387 = INCDTLS387

                ^INCDTLS389 = INCDTLS389

                ^INCDTLS390 = INCDTLS390

                ^INCDTLS419 = INCDTLS419

                ^INCDTLS420 = INCDTLS420
                */
            VRATOIITR300093(response, report);
            #endregion // VR.ATO.IITR.300093

            #region VR.ATO.IITR.300094

            /*  VR.ATO.IITR.300094
                        If the total of Net rental property income less Net rental property deductions is less than zero then the difference must be inserted at Net Rental property loss. If the difference exceeds zero then zero must be inserted at Net Rental property loss.

                    Legacy Rule Format:
                        (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (((^IITR205 = NULL) OR (^IITR205 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0)) - (^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614)))))) OR (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (^IITR205 <> 0))

                Technical Business Rule Format:
                        (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) < (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (((^IITR205 = NULL) OR (^IITR205 <> ((ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0)) - (^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614)))))) OR (((^IITR250 + ^IITR283 + ^IITR292 + ^IITR296 + ^IITR324 + (^IITR334 - (^IITR335 + ^IITR336 + ^IITR337)) + ^IITR614) >= (ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(^DDCTNS311, 0), 0) + ^IITR263 + ConditionalValue(CountDocument('DDCTNS') = 1, RoundDown(Sum(ConditionalValue(^DDCTNS136 = 'U - Debt deductions incurred in deriving foreign rental income', ^DDCTNS138, 0)), 0), 0))) AND (^IITR205 <> 0))

            Data Elements:

                ^IITR205 = RP:Tax.Losses.RentalPropertyNet.Amount

                ^DDCTNS136 = DDCTNS136

                ^DDCTNS138 = DDCTNS138

                ^DDCTNS311 = DDCTNS311

                ^IITR250 = RP:Income.PartnershipRentalNonPrimaryProductionShare.Amount

                ^IITR263 = RP:IncomeTax.Deduction.PartnershipRentalNonPrimaryProduction.Amount

                ^IITR283 = RP:Income.RentalPropertyNonPrimaryProductionNet.Amount

                ^IITR292 = RP:Tax.Losses.DeferredNonCommercialPartnershipRentalShare.Amount

                ^IITR296 = RP:Tax.Losses.DeferredNonCommercialSoleTraderRental.Amount

                ^IITR324 = RP:InternationalDealings.ForeignRent.Amount

                ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount

                ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount

                ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount

                ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount

                ^IITR614 = RP:Income.DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotal.Amount
                */
            VRATOIITR300094(response, report);
            #endregion // VR.ATO.IITR.300094

            #region VR.ATO.IITR.310027

            /*  VR.ATO.IITR.310027
                        The answer to "Satisfied the results test" is No, and no answer has been provided to the "Received a personal services business determination(s) for the whole of the period PSI earned" question.

                    Legacy Rule Format:
                        (^IITR406 = NULL) AND (^IITR404 = TRUE) AND (^IITR405 = FALSE)

                Technical Business Rule Format:
                        (^IITR406 = NULL) AND (^IITR404 = TRUE) AND (^IITR405 = FALSE)

            Data Elements:

                ^IITR406 = RP:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator

                ^IITR404 = RP:Income.PersonalServicesIncomeIncluded.Indicator

                ^IITR405 = RP:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
                */
            VRATOIITR310027(response, report);
            #endregion // VR.ATO.IITR.310027

            #region VR.ATO.IITR.400063

            /*  VR.ATO.IITR.400063
                        The Sequence Container within the Reporting Party Loss context must be assigned sequentially beginning at 1 and must not exceed 10

                    Legacy Rule Format:
                        ((^IITR573 < 1 OR ^IITR573 > 10)) OR (^IITR573 > Count(^RP.{LOSSSeqNum}:Context))

                Technical Business Rule Format:
                        ((^IITR573 < 1 OR ^IITR573 > 10)) OR (^IITR573 > Count(^RP.{LOSSSeqNum}:Context))

            Data Elements:

                ^IITR573 = RP.{LOSSSeqNum}:Report.ItemOrder.Number
                */
            VRATOIITR400063(response, report);
            #endregion // VR.ATO.IITR.400063

            #region VR.ATO.IITR.410018

            /*  VR.ATO.IITR.410018
                        Primary production deferred losses and/or non-primary production deferred losses have been entered without Business loss activity details being completed.

                    Legacy Rule Format:
                        (^IITR299 + ^IITR300) > 0 AND (Count(^RP.{LOSSSeqNum}:Context) = 0 OR AllOccurrences(^IITR479, ^IITR479 = BLANK))

                Technical Business Rule Format:
                        (^IITR299 + ^IITR300) > 0 AND (Count(^RP.{LOSSSeqNum}:Context) = 0 OR AllOccurrences(^IITR479, ^IITR479 = BLANK))

            Data Elements:

                ^IITR479 = RP.{LOSSSeqNum}:OrganisationDetails.MainIncomeActivity.Description

                ^IITR299 = RP:Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount

                ^IITR300 = RP:Tax.Losses.DeferredNonCommercialNonPrimaryProduction.Amount
                */
            VRATOIITR410018(response, report);
            #endregion // VR.ATO.IITR.410018

            #region VR.ATO.IITR.410034

            /*  VR.ATO.IITR.410034
                        For income test purposes, spouse's adjusted taxable income (ATI) is the sum of spouse's taxable income, spouse's adjusted fringe benefit (Your spouse's total reportable fringe benefits amount from employers exempt under section 57A x 0.530 plus Your spouse's total reportable fringe benefits amounts from employers not exempt under section 57A), spouse's target foreign income, spouse's tax-free pensions (exempt pensions plus tax-free pensions paid under Military Rehabilitation and Compensation Act), spouse's total net investment loss, spouse's reportable superannuation contributions less child support the spouse paid. For the benefit of other calculations, ensure the ATI amount is correct.

                    Legacy Rule Format:
                        ^IITR223 <> TRUE AND ((^IITR237 = NULL AND OutsideRange(0, (^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR233 + ^IITR230 + ^IITR232 + ^IITR234 + ^IITR231 - ^IITR235), 1)) OR (^IITR237 <> NULL AND OutsideRange(^IITR237, (^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR233 + ^IITR230 + ^IITR232 + ^IITR234 + ^IITR231 - ^IITR235), 1)))

                Technical Business Rule Format:
                        ^IITR223 <> TRUE AND ((^IITR237 = NULL AND OutsideRange(0, (^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR233 + ^IITR230 + ^IITR232 + ^IITR234 + ^IITR231 - ^IITR235), 1)) OR (^IITR237 <> NULL AND OutsideRange(^IITR237, (^IITR225 + ConditionalValue((^IITR492 + ^IITR493) > 3772, ((^IITR492 * 0.530) + ^IITR493), 0) + ^IITR233 + ^IITR230 + ^IITR232 + ^IITR234 + ^IITR231 - ^IITR235), 1)))

            Data Elements:

                ^IITR223 = RP.SPOUSE:PersonDemographicDetails.Deceased.Indicator

                ^IITR225 = RP.SPOUSE:Income.Taxable.Amount

                ^IITR230 = RP.SPOUSE:Pension.SpouseExempt.Amount

                ^IITR231 = RP.SPOUSE:SuperannuationContribution.EmployerReportable.Amount

                ^IITR232 = RP.SPOUSE:Pension.TaxFree.Amount

                ^IITR233 = RP.SPOUSE:InternationalDealings.ForeignIncomeTarget.Amount

                ^IITR234 = RP.SPOUSE:Tax.Losses.InvestmentNet.Amount

                ^IITR235 = RP.SPOUSE:ChildSupport.Payment.Amount

                ^IITR237 = RP.SPOUSE:Income.TaxableAdjusted.Amount

                ^IITR492 = RP.SPOUSE.Eligible:Income.FringeBenefitsReportable.Amount

                ^IITR493 = RP.SPOUSE.Ineligible:Income.FringeBenefitsReportable.Amount
                */
            VRATOIITR410034(response, report);
            #endregion // VR.ATO.IITR.410034

            #region VR.ATO.IITR.410068

            /*  VR.ATO.IITR.410068
                        Business loss activity - Partnership or sole trader codes are: '161' - Partnerships; '006' - Sole Trader.

                    Legacy Rule Format:
                        ^IITR481 <> NULL AND NotInSet(^IITR481, '"006", "161"')

                Technical Business Rule Format:
                        ^IITR481 <> NULL AND NotInSet(^IITR481, '"006", "161"')

            Data Elements:

                ^IITR481 = RP.{LOSSSeqNum}:OrganisationDetails.OrganisationType.Code
                */
            VRATOIITR410068(response, report);
            #endregion // VR.ATO.IITR.410068

            #region VR.ATO.IITR.410083

            /*  VR.ATO.IITR.410083
                        Distributions to your spouse on which family trust distribution tax has been paid has a maximum field length of 11

                    Legacy Rule Format:
                        NotMonetary(^IITR227, 'U', 11, 0)

                Technical Business Rule Format:
                        NotMonetary(^IITR227, 'U', 11, 0)

            Data Elements:

                ^IITR227 = RP.SPOUSE:IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount
                */
            VRATOIITR410083(response, report);
            #endregion // VR.ATO.IITR.410083

            #region VR.ATO.IITR.410085

            /*  VR.ATO.IITR.410085
                        Amount of any Australian Government pensions and allowances received by your spouse in the year of the return has a maximum field length of 11

                    Legacy Rule Format:
                        NotMonetary(^IITR229, 'U', 11, 0)

                Technical Business Rule Format:
                        NotMonetary(^IITR229, 'U', 11, 0)

            Data Elements:

                ^IITR229 = RP.SPOUSE:Pension.Total.Amount
                */
            VRATOIITR410085(response, report);
            #endregion // VR.ATO.IITR.410085

            #region VR.ATO.IITR.410086

            /*  VR.ATO.IITR.410086
                        Amount of any exempt pension income received by your spouse in the year of the return has a maximum field length of 11

                    Legacy Rule Format:
                        NotMonetary(^IITR230, 'U', 11, 0)

                Technical Business Rule Format:
                        NotMonetary(^IITR230, 'U', 11, 0)

            Data Elements:

                ^IITR230 = RP.SPOUSE:Pension.SpouseExempt.Amount
                */
            VRATOIITR410086(response, report);
            #endregion // VR.ATO.IITR.410086

            #region VR.ATO.IITR.410116

            /*  VR.ATO.IITR.410116
                        Small business entity simplified depreciation - Deduction for certain assets has a maximum field length of 11

                    Legacy Rule Format:
                        NotMonetary(^IITR507, 'U', 11, 0)

                Technical Business Rule Format:
                        NotMonetary(^IITR507, 'U', 11, 0)

            Data Elements:

                ^IITR507 = RP:IncomeTax.Deduction.LowCostAssetDepreciation.Amount
                */
            VRATOIITR410116(response, report);
            #endregion // VR.ATO.IITR.410116

            #region VR.ATO.IITR.410117

            /*  VR.ATO.IITR.410117
                        Small business entity simplified depreciation - Deduction for general small business pool has a maximum field length of 11

                    Legacy Rule Format:
                        NotMonetary(^IITR508, 'U', 11, 0)

                Technical Business Rule Format:
                        NotMonetary(^IITR508, 'U', 11, 0)

            Data Elements:

                ^IITR508 = RP:IncomeTax.Deduction.SmallBusinessPoolDepreciation.Amount
                */
            VRATOIITR410117(response, report);
            #endregion // VR.ATO.IITR.410117

            #region VR.ATO.IITR.600051

            /*  VR.ATO.IITR.600051
                        Work related uniform, occupation specific or protective clothing, laundry and dry cleaning expenses has a maximum field length of 11

                    Legacy Rule Format:
                        NotMonetary(^IITR137, 'U', 11, 0)

                Technical Business Rule Format:
                        NotMonetary(^IITR137, 'U', 11, 0)

            Data Elements:

                ^IITR137 = RP:IncomeTax.Deduction.Clothing.Amount
                */
            VRATOIITR600051(response, report);
            #endregion // VR.ATO.IITR.600051

            #region VR.ATO.IITR.630005

            /*  VR.ATO.IITR.630005
                        Check Total Tax Withheld amount is greater than or equal to the sum of the Income amounts.

                    Legacy Rule Format:
                        ^IITR113 >= (^IITR601 + ^IITR65 + ^IITR607 + ^IITR609 + ^IITR626 + ^IITR86 + ^IITR89 + ^IITR92 + ^IITR93 + ^IITR505 + ^IITR94 + ^IITR95 + ^IITR611 + ^IITR612 + ^IITR112)

                Technical Business Rule Format:
                        ^IITR113 >= (^IITR601 + ^IITR65 + ^IITR607 + ^IITR609 + ^IITR626 + ^IITR86 + ^IITR89 + ^IITR92 + ^IITR93 + ^IITR505 + ^IITR94 + ^IITR95 + ^IITR611 + ^IITR612 + ^IITR112)

            Data Elements:

                ^IITR113 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheld.Amount

                ^IITR65 = RP:Income.AllowancesEarningsTipsDirectorsFees.Amount

                ^IITR86 = RP:Income.AustralianGovernmentAllowancesAndPayments.Amount

                ^IITR89 = RP:Pension.Total.Amount

                ^IITR92 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount

                ^IITR93 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount

                ^IITR94 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount

                ^IITR95 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount

                ^IITR112 = RP:Income.PersonalServicesIncome.Total.Amount

                ^IITR505 = RP:Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount

                ^IITR601 = RP:Income.SalaryOrWages.Amount

                ^IITR607 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount

                ^IITR609 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount

                ^IITR611 = RP:Income.Superannuation.LumpSumPaymentTaxableComponentTaxedElement.Amount

                ^IITR612 = RP:Income.Superannuation.LumpSumPaymentTaxableComponentUntaxedElement.Amount

                ^IITR626 = RP:Income.EmploymentTerminationPaymentTaxable.Amount
                */
            VRATOIITR630005(response, report);
            #endregion // VR.ATO.IITR.630005

            #region VR.ATO.IITR.730108

            /*  VR.ATO.IITR.730108
                        Either one or the sum of Superannuation deduction claimed amounts is the same as the amount provided for Reportable employer superannuation contributions. A deduction for contributions paid by the employer from before-tax income (including the compulsory super guarantee and salary sacrifice amounts) is not claimable, and the return may be adjusted. Check and correct if necessary.

                    Legacy Rule Format:
                        CountDocument('DDCTNS') = 1 AND ^IITR201 > 0 AND (AnyOccurrence(^IITR201, ^IITR201 = RoundDown(^DDCTNS407, 0)) OR AnyOccurrence(^IITR201, ^IITR201 = RoundDown(Sum(^DDCTNS407), 0)))

                Technical Business Rule Format:
                        CountDocument('DDCTNS') = 1 AND ^IITR201 > 0 AND (AnyOccurrence(^IITR201, ^IITR201 = RoundDown(^DDCTNS407, 0)) OR AnyOccurrence(^IITR201, ^IITR201 = RoundDown(Sum(^DDCTNS407), 0)))

            Data Elements:

                ^DDCTNS407 = DDCTNS407

                ^IITR201 = RP:SuperannuationContribution.EmployerReportable.Amount
                */
            VRATOIITR730108(response, report);
            #endregion // VR.ATO.IITR.730108

            #region VR.ATO.IITR.730163

            /*  VR.ATO.IITR.730163
                        When an amount above zero is provided for "Foreign income tax offset", the "Net foreign employment income - payment summary" amount must not exceed the sum of "Salary or wages gross income", "Salary or wages lump sum A" and "Tax withheld - lump sum payments in arrears"

                    Legacy Rule Format:
                        (^IITR331 > 0) AND (^IITR329 > (^IITR601 + ^IITR607 + ^IITR354))

                Technical Business Rule Format:
                        (^IITR331 > 0) AND (^IITR329 > (^IITR601 + ^IITR607 + ^IITR354))

            Data Elements:

                ^IITR331 = RP:Income.InternationalDealings.TaxOffset.Amount

                ^IITR329 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount

                ^IITR354 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldLumpSumPaymentsInArrears.Amount

                ^IITR601 = RP:Income.SalaryOrWages.Amount

                ^IITR607 = RP:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
                */
            VRATOIITR730163(response, report);
            #endregion // VR.ATO.IITR.730163


            #region VR.ATO.IITR.730190

            /*  VR.ATO.IITR.730190
                        Where the age of tax payer is less than 55 on 30 June of the reporting period; and Total tax withheld from SLS payments, from the same Payer ABN, is more than the sum of First $1,780,000 of the total SLS untaxed amounts, from the same payer ABN, * 32% (30% SLS rate 3 + 2% Medicare), Plus First $1,780,000 of the total SLS untaxed amounts, from the same payer ABN, above $1,780,000 * 47% (45% SLS rate 4 + 2% Medicare), check Australian superannuation lump sum payments "Tax withheld amount"

                    Legacy Rule Format:
                        CountDocument('INCDTLS') = 1 AND AnyOccurrence(^INCDTLS327, ^INCDTLS327 <> NULL) AND ((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) AND (^INCDTLS307 <> NULL) AND AnyOccurrence(^INCDTLS330, ^INCDTLS330 = NULL OR ^INCDTLS330 = 0) AND (((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) * 0.32))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (1780000 * 0.32 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 1780000) * 0.47))))))

                Technical Business Rule Format:
                        CountDocument('INCDTLS') = 1 AND AnyOccurrence(^INCDTLS327, ^INCDTLS327 <> NULL) AND ((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 55))) AND (^INCDTLS307 <> NULL) AND AnyOccurrence(^INCDTLS330, ^INCDTLS330 = NULL OR ^INCDTLS330 = 0) AND (((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) * 0.32))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (1780000 * 0.32 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 1780000) * 0.47))))))

            Data Elements:

                ^IITR10 = RP:Report.TargetFinancial.Year

                ^IITR29 = RP:PersonDemographicDetails.Birth.Date

                ^INCDTLS307 = INCDTLS307

                ^INCDTLS327 = INCDTLS327

                ^INCDTLS329 = INCDTLS329

                ^INCDTLS330 = INCDTLS330

                ^INCDTLS331 = INCDTLS331
                */
            VRATOIITR730190(response, report);
            #endregion // VR.ATO.IITR.730190

            #region VR.ATO.IITR.730191

            /*  VR.ATO.IITR.730191
                        Where the age of the tax payer is greater than 60 on 30 June of the reporting period; and Total tax withheld from SLS payments, from the same Payer ABN, is more than the sum of First $1,780,000  of the total SLS untaxed amounts, from the same payer ABN, * 17% (15% SLS rate 1 + 2% Medicare), Plus First $1,780,000 of the total SLS untaxed amounts, from the same payer ABN, above $1,780,000 * 47% (45% SLS rate 4 + 2% Medicare), check Australian superannuation lump sum payments "Tax withheld amount"

                    Legacy Rule Format:
                        CountDocument('INCDTLS') = 1 AND AnyOccurrence(^INCDTLS327, ^INCDTLS327 <> NULL) AND ((^IITR29 <> NULL AND ^IITR29 <= ConvertToDate(30, 6, (^IITR10 - 60))) AND (^INCDTLS307 <> NULL) AND AnyOccurrence(^INCDTLS330, ^INCDTLS330 = NULL OR ^INCDTLS330 = 0) AND (((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) * 0.17))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (1780000 * 0.17 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 1780000) * 0.47))))))

                Technical Business Rule Format:
                        CountDocument('INCDTLS') = 1 AND AnyOccurrence(^INCDTLS327, ^INCDTLS327 <> NULL) AND ((^IITR29 <> NULL AND ^IITR29 <= ConvertToDate(30, 6, (^IITR10 - 60))) AND (^INCDTLS307 <> NULL) AND AnyOccurrence(^INCDTLS330, ^INCDTLS330 = NULL OR ^INCDTLS330 = 0) AND (((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) * 0.17))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (1780000 * 0.17 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 1780000) * 0.47))))))

            Data Elements:

                ^IITR10 = RP:Report.TargetFinancial.Year

                ^IITR29 = RP:PersonDemographicDetails.Birth.Date

                ^INCDTLS307 = INCDTLS307

                ^INCDTLS327 = INCDTLS327

                ^INCDTLS329 = INCDTLS329

                ^INCDTLS330 = INCDTLS330

                ^INCDTLS331 = INCDTLS331
                */
            VRATOIITR730191(response, report);
            #endregion // VR.ATO.IITR.730191

            #region VR.ATO.IITR.730192

            /*  VR.ATO.IITR.730192
                        Where the age of the tax payer is greater than 55 and less than 60 on 30 June of the reporting period; and Total tax withheld from SLS payments, from the same Payer ABN, is more than the sum of First $245,000 of the total SLS untaxed amounts, from the same payer ABN, * 17% (15% sls rate 1 + 2% Medicare), Plus amount between $245,000 and up to $1,780,000 of the total SLS untaxed amounts, from the same payer ABN, * 32% (30% sls rate 3 + 2% Medicare), Plus First $1,780,000 of the total SLS untaxed amounts, from the same payer ABN, above $1,780,000 * 47% (45% sls rate 4 + 2% Medicare), check Australian superannuation lump sum payments "Tax withheld amount"

                    Legacy Rule Format:
                        CountDocument('INCDTLS') = 1 AND AnyOccurrence(^INCDTLS327, ^INCDTLS327 <> NULL) AND ((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 60))) AND (^IITR29 <= ConvertToDate(30, 6, (^IITR10 - 55))) AND (^INCDTLS307 <> NULL) AND AnyOccurrence(^INCDTLS330, ^INCDTLS330 = NULL OR ^INCDTLS330 = 0) AND (((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 245000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) * 0.17))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 245000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (245000 * 0.17 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 245000) * 0.32)))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (245000 * 0.17 + (1780000 - 245000) * 0.32 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 1780000) * 0.47))))))

                Technical Business Rule Format:
                        CountDocument('INCDTLS') = 1 AND AnyOccurrence(^INCDTLS327, ^INCDTLS327 <> NULL) AND ((^IITR29 > ConvertToDate(30, 6, (^IITR10 - 60))) AND (^IITR29 <= ConvertToDate(30, 6, (^IITR10 - 55))) AND (^INCDTLS307 <> NULL) AND AnyOccurrence(^INCDTLS330, ^INCDTLS330 = NULL OR ^INCDTLS330 = 0) AND (((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 245000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) * 0.17))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 245000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) <= 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (245000 * 0.17 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 245000) * 0.32)))) OR ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) > 1780000) AND (Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS329,0)) > (245000 * 0.17 + (1780000 - 245000) * 0.32 + ((Sum(ConditionalValue(HasDuplicateValues(^INCDTLS307) WHERE (^INCDTLS330 = NULL OR ^INCDTLS330 = 0),^INCDTLS331,0)) - 1780000) * 0.47))))))

            Data Elements:

                ^IITR10 = RP:Report.TargetFinancial.Year

                ^IITR29 = RP:PersonDemographicDetails.Birth.Date

                ^INCDTLS307 = INCDTLS307

                ^INCDTLS327 = INCDTLS327

                ^INCDTLS329 = INCDTLS329

                ^INCDTLS330 = INCDTLS330

                ^INCDTLS331 = INCDTLS331
                */
            VRATOIITR730192(response, report);
            #endregion // VR.ATO.IITR.730192

            #region VR.ATO.IITR.730264

            /*  VR.ATO.IITR.730264
                        Working holiday maker net income has been declared as greater than the sum of all Salary or wages gross amount amounts where the Did you earn this income while you held a Working Holiday visa (subclass 417) or Work and Holiday visa (subclass 462)? indicator is true. Check that all sources of working holiday maker income have been declared.

                    Legacy Rule Format:
                        (^IITR499 <> NULL) AND ((CountDocument('INCDTLS') = 0) OR (^IITR499 > Sum(ConditionalValue(^INCDTLS310 = TRUE, ^INCDTLS312, 0))))

                Technical Business Rule Format:
                        (^IITR499 <> NULL) AND ((CountDocument('INCDTLS') = 0) OR (^IITR499 > Sum(ConditionalValue(^INCDTLS310 = TRUE, ^INCDTLS312, 0))))

            Data Elements:

                ^IITR499 = RP.WHM:Income.Net.Amount

                ^INCDTLS310 = INCDTLS310

                ^INCDTLS312 = INCDTLS312
                */
            VRATOIITR730264(response, report);
            #endregion // VR.ATO.IITR.730264

            #region VR.ATO.IITR.730303

            /*  VR.ATO.IITR.730303
                        Income for surcharge purposes is used to determine if the taxpayer is liable for Medicare Levy Surcharge. Where applicable, complete Deductible personal superannuation contributions from the Deduction schedule.

                    Legacy Rule Format:
                        (^IITR176 <> TRUE) AND (^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))

                Technical Business Rule Format:
                        (^IITR176 <> TRUE) AND (^IITR177 < (^RP:EndDate - ^RP:StartDate + 1)) AND (CountDocument('DDCTNS') = 0 OR AllOccurrences(^DDCTNS407, ^DDCTNS407 = NULL))

            Data Elements:

                ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator

                ^DDCTNS407 = DDCTNS407

                ^EndDate = EndDate

                ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count

                ^StartDate = StartDate
                */
            VRATOIITR730303(response, report);
            #endregion // VR.ATO.IITR.730303

            #region VR.ATO.IITR.730435

            /*  VR.ATO.IITR.730435
                        The sum of Primary production deferred losses and Non-primary production deferred losses must be equal to the sum of Business loss activity - Net loss amount where the Business loss activity - Type of Loss is "8" (loss is to be deferred)

                    Legacy Rule Format:
                        (^IITR299 <> NULL OR ^IITR300 <> NULL) AND (^IITR299 + ^IITR300 <> Sum(ConditionalValue(^IITR482 = '8', ^IITR487, 0)))

                Technical Business Rule Format:
                        (^IITR299 <> NULL OR ^IITR300 <> NULL) AND (^IITR299 + ^IITR300 <> Sum(ConditionalValue(^IITR482 = '8', ^IITR487, 0)))

            Data Elements:

                ^IITR299 = RP:Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount

                ^IITR300 = RP:Tax.Losses.DeferredNonCommercialNonPrimaryProduction.Amount

                ^IITR482 = RP.{LOSSSeqNum}:IncomeTax.LossesType.Code

                ^IITR487 = RP.{LOSSSeqNum}:Tax.Losses.Net.Amount
                */
            VRATOIITR730435(response, report);
            #endregion // VR.ATO.IITR.730435
            #endregion Manual Rules - Rules marked for manual coding
            VRATOGEN000209();
            VRATOGEN000458();
            VRATOGEN000459();
            VRATOGEN001023();
            VRATOGEN001023_3();
            VRATOGEN001023_4();
            VRATOGEN001023_5();
            VRATOGEN001023_6();
            VRATOGEN001023_7();
            VRATOGEN001023_8();
            VRATOGEN001023_9();
            VRATOGEN001023_10();
            VRATOGEN001023_11();
            VRATOGEN001023_12();
            VRATOGEN001023_13();
            VRATOGEN001023_14();
            VRATOGEN001023_15();
            VRATOGEN001023_16();
            VRATOGEN001023_17();
            VRATOGEN001023_18();
            VRATOGEN001023_19();
            VRATOGEN001023_20();
            VRATOGEN001023_21();
            VRATOGEN001023_22();
            VRATOGEN001023_23();
            VRATOGEN428247();
            VRATOGEN430230();
            VRATOGEN430231();
            VRATOGEN430232();
            VRATOGEN432041();
            VRATOGEN438019();
            VRATOGEN438021();
            VRATOGEN438058();
            VRATOGEN438059();
            VRATOGEN438070();
            VRATOGEN438076();
            VRATOGEN438078();
            VRATOGEN438081();
            VRATOGEN438082();
            VRATOGEN500009();
            VRATOGEN500030();
            VRATOGEN500030_3();
            VRATOGEN500030_4();
            VRATOGEN500030_5();
            VRATOGEN500030_6();
            VRATOGEN500030_7();
            VRATOGEN500030_8();
            VRATOGEN500030_9();
            VRATOGEN500030_10();
            VRATOGEN500030_11();
            VRATOGEN500030_12();
            VRATOGEN500030_13();
            VRATOGEN500030_14();
            VRATOGEN500030_15();
            VRATOGEN500030_16();
            VRATOGEN500030_17();
            VRATOGEN500030_18();
            VRATOGEN500030_19();
            VRATOGEN500030_20();
            VRATOGEN500030_21();
            VRATOGEN500033();
            VRATOGEN500033_3();
            VRATOGEN500033_4();
            VRATOGEN500033_5();
            VRATOGEN500033_6();
            VRATOGEN500033_7();
            VRATOGEN500033_8();
            VRATOGEN500033_9();
            VRATOGEN500033_10();
            VRATOGEN500033_11();
            VRATOGEN500033_12();
            VRATOGEN500033_13();
            VRATOGEN500033_14();
            VRATOGEN500033_15();
            VRATOGEN500033_16();
            VRATOGEN500033_17();
            VRATOGEN500033_18();
            VRATOGEN500033_19();
            VRATOGEN500033_20();
            VRATOGEN500033_21();
            VRATOGEN500037();
            VRATOGEN500037_1();
            VRATOGEN500046();
            VRATOGEN500051();
            VRATOGEN500053();
            VRATOGEN500054();
            VRATOGEN500055();
            VRATOGEN500055_1();
            VRATOGEN500058();
            VRATOGEN500058_1();
            VRATOGEN500059();
            VRATOGEN500059_1();
            VRATOGEN500064();
            VRATOGEN500066();
            VRATOGEN500066_1();
            VRATOGEN500069();
            VRATOGEN500069_1();
            VRATOGEN500072();
            VRATOGEN500073();
            VRATOGEN500074();
            VRATOGEN500075();
            VRATOGEN500077();
            VRATOGEN500088();
            VRATOGEN500089();
            VRATOGEN500090();
            VRATOGEN500094();
            VRATOGEN500100();
            VRATOGEN500103();
            VRATOGEN500146();
            VRATOGEN500147();
            VRATOGEN500148();
            VRATOGEN500149();
            VRATOGEN500150();
            VRATOIITR000001();
            VRATOIITR000002();
            VRATOIITR000003();
            VRATOIITR000005();
            VRATOIITR000006();
            VRATOIITR000007();
            VRATOIITR000008();
            VRATOIITR000009();
            VRATOIITR000010();
            VRATOIITR000012();
            VRATOIITR000014();
            VRATOIITR000015();
            VRATOIITR000016();
            VRATOIITR000018();
            VRATOIITR000019();
            VRATOIITR000020();
            VRATOIITR000021();
            VRATOIITR000022();
            VRATOIITR000023();
            VRATOIITR000025();
            VRATOIITR000026();
            VRATOIITR000027();
            VRATOIITR000031();
            VRATOIITR000032();
            VRATOIITR000033();
            VRATOIITR000044();
            VRATOIITR000049();
            VRATOIITR000050();
            VRATOIITR000051();
            VRATOIITR000052();
            VRATOIITR000054();
            VRATOIITR000056();
            VRATOIITR000061();
            VRATOIITR000071();
            VRATOIITR000080();
            VRATOIITR000081();
            VRATOIITR000100();
            VRATOIITR000101();
            VRATOIITR000102();
            VRATOIITR000104();
            VRATOIITR000105();
            VRATOIITR000106();
            VRATOIITR000108();
            VRATOIITR000109();
            VRATOIITR000110();
            VRATOIITR000111();
            VRATOIITR000112();
            VRATOIITR000113();
            VRATOIITR000114();
            VRATOIITR000115();
            VRATOIITR000116();
            VRATOIITR000117();
            VRATOIITR000118();
            VRATOIITR000121();
            VRATOIITR000122();
            VRATOIITR000123();
            VRATOIITR000125();
            VRATOIITR000128();
            VRATOIITR000129();
            VRATOIITR000132();
            VRATOIITR000133();
            VRATOIITR000134();
            VRATOIITR000135();
            VRATOIITR000137();
            VRATOIITR000139();
            VRATOIITR000140();
            VRATOIITR000141();
            VRATOIITR000145();
            VRATOIITR000146();
            VRATOIITR000147();
            VRATOIITR000151();
            VRATOIITR000153();
            VRATOIITR000154();
            VRATOIITR000156();
            VRATOIITR000159();
            VRATOIITR000160();
            VRATOIITR000162();
            VRATOIITR000163();
            VRATOIITR000164();
            VRATOIITR000168();
            VRATOIITR000169();
            VRATOIITR000170();
            VRATOIITR000178();
            VRATOIITR000185();
            VRATOIITR000187();
            VRATOIITR000188();
            VRATOIITR000190();
            VRATOIITR000198();
            VRATOIITR000199();
            VRATOIITR000200();
            VRATOIITR000201();
            VRATOIITR000202();
            VRATOIITR000203();
            VRATOIITR000204();
            VRATOIITR000207();
            VRATOIITR000208();
            VRATOIITR000209();
            VRATOIITR000210();
            VRATOIITR000211();
            VRATOIITR000214();
            VRATOIITR000216();
            VRATOIITR000219();
            VRATOIITR000220();
            VRATOIITR000224();
            VRATOIITR000225();
            VRATOIITR000226();
            VRATOIITR000227();
            VRATOIITR000234();
            VRATOIITR000235();
            VRATOIITR000238();
            VRATOIITR000241();
            VRATOIITR000242();
            VRATOIITR000243();
            VRATOIITR000244();
            VRATOIITR000245();
            VRATOIITR000246();
            VRATOIITR000247();
            VRATOIITR000248();
            VRATOIITR000249();
            VRATOIITR000250();
            VRATOIITR000251();
            VRATOIITR000252();
            VRATOIITR000253();
            VRATOIITR000254();
            VRATOIITR000255();
            VRATOIITR000256();
            VRATOIITR000257();
            VRATOIITR000258();
            VRATOIITR000259();
            VRATOIITR000260();
            VRATOIITR000261();
            VRATOIITR000262();
            VRATOIITR000266();
            VRATOIITR000267();
            VRATOIITR000268();
            VRATOIITR000269();
            VRATOIITR000270();
            VRATOIITR000271();
            VRATOIITR000272();
            VRATOIITR000277();
            VRATOIITR000280();
            VRATOIITR000283();
            VRATOIITR000284();
            VRATOIITR000294();
            VRATOIITR000295();
            VRATOIITR000300();
            VRATOIITR000309();
            VRATOIITR000312();
            VRATOIITR000314();
            VRATOIITR000316();
            VRATOIITR000317();
            VRATOIITR000318();
            VRATOIITR000319();
            VRATOIITR000320();
            VRATOIITR000321();
            VRATOIITR000322();
            VRATOIITR000323();
            VRATOIITR000324();
            VRATOIITR000325();
            VRATOIITR000326();
            VRATOIITR000327();
            VRATOIITR000329();
            VRATOIITR000330();
            VRATOIITR000331();
            VRATOIITR000332();
            VRATOIITR000333();
            VRATOIITR000334();
            VRATOIITR000336();
            VRATOIITR000337();
            VRATOIITR000338();
            VRATOIITR000339();
            VRATOIITR000340();
            VRATOIITR000343();
            VRATOIITR000346();
            VRATOIITR000347();
            VRATOIITR000348();
            VRATOIITR000349();
            VRATOIITR000350();
            VRATOIITR000351();
            VRATOIITR000353();
            VRATOIITR000354();
            VRATOIITR000355();
            VRATOIITR000356();
            VRATOIITR000357();
            VRATOIITR000358();
            VRATOIITR000359();
            VRATOIITR000360();
            VRATOIITR000361();
            VRATOIITR000362();
            VRATOIITR000363();
            VRATOIITR000364();
            VRATOIITR000365();
            VRATOIITR000366();
            VRATOIITR000367();
            VRATOIITR000368();
            VRATOIITR000369();
            VRATOIITR000370();
            VRATOIITR000371();
            VRATOIITR000372();
            VRATOIITR000373();
            VRATOIITR000380();
            VRATOIITR000501();
            VRATOIITR000502();
            VRATOIITR000503();
            VRATOIITR000504();
            VRATOIITR000505();
            VRATOIITR000506();
            VRATOIITR000507();
            VRATOIITR000510();
            VRATOIITR000511();
            VRATOIITR000512();
            VRATOIITR000513();
            VRATOIITR000514();
            VRATOIITR000515();
            VRATOIITR000516();
            VRATOIITR000518();
            VRATOIITR000608();
            VRATOIITR000609();
            VRATOIITR000613();
            VRATOIITR000614();
            VRATOIITR000621();
            VRATOIITR000622();
            VRATOIITR000623();
            VRATOIITR000625();
            VRATOIITR000626();
            VRATOIITR000629();
            VRATOIITR000631();
            VRATOIITR000632();
            VRATOIITR000636();
            VRATOIITR000637();
            VRATOIITR000638();
            VRATOIITR000640();
            VRATOIITR000646();
            VRATOIITR000647();
            VRATOIITR000648();
            VRATOIITR000649();
            VRATOIITR000652();
            VRATOIITR000653();
            VRATOIITR000654();
            VRATOIITR000655();
            VRATOIITR000656();
            VRATOIITR000657();
            VRATOIITR000658();
            VRATOIITR000659();
            VRATOIITR000662();
            VRATOIITR000665();
            VRATOIITR000666();
            VRATOIITR000669();
            VRATOIITR000670();
            VRATOIITR000671();
            VRATOIITR000672();
            VRATOIITR000673();
            VRATOIITR000674();
            VRATOIITR000675();
            VRATOIITR000676();
            VRATOIITR100009();
            VRATOIITR100010();
            VRATOIITR100011();
            VRATOIITR100029();
            VRATOIITR100030();
            VRATOIITR100046();
            VRATOIITR100063();
            VRATOIITR100070();
            VRATOIITR100072();
            VRATOIITR100073();
            VRATOIITR100074();
            VRATOIITR100075();
            VRATOIITR100076();
            VRATOIITR100077();
            VRATOIITR100078();
            VRATOIITR100079();
            VRATOIITR100081();
            VRATOIITR100083();
            VRATOIITR100084();
            VRATOIITR100085();
            VRATOIITR100086();
            VRATOIITR100087();
            VRATOIITR100088();
            VRATOIITR100089();
            VRATOIITR100094();
            VRATOIITR100095();
            VRATOIITR130070();
            VRATOIITR130089();
            VRATOIITR200001();
            VRATOIITR200002();
            VRATOIITR200003();
            VRATOIITR200016();
            VRATOIITR200018();
            VRATOIITR200021();
            VRATOIITR200024();
            VRATOIITR200026();
            VRATOIITR200030();
            VRATOIITR210014();
            VRATOIITR210015();
            VRATOIITR210016();
            VRATOIITR210017();
            VRATOIITR210018();
            VRATOIITR210019();
            VRATOIITR210022();
            VRATOIITR210023();
            VRATOIITR210046();
            VRATOIITR210047();
            VRATOIITR210048();
            VRATOIITR210050();
            VRATOIITR210051();
            VRATOIITR300000();
            VRATOIITR300001();
            VRATOIITR300003();
            VRATOIITR300004();
            VRATOIITR300008();
            VRATOIITR300010();
            VRATOIITR300011();
            VRATOIITR300012();
            VRATOIITR300030();
            VRATOIITR300031();
            VRATOIITR300032();
            VRATOIITR300056();
            VRATOIITR300057();
            VRATOIITR300101();
            VRATOIITR300102();
            VRATOIITR300110();
            VRATOIITR300111();
            VRATOIITR300112();
            VRATOIITR310000();
            VRATOIITR310001();
            VRATOIITR310002();
            VRATOIITR310003();
            VRATOIITR310004();
            VRATOIITR310005();
            VRATOIITR310006();
            VRATOIITR310017();
            VRATOIITR310018();
            VRATOIITR310026();
            VRATOIITR310030();
            VRATOIITR310031();
            VRATOIITR310032();
            VRATOIITR310033();
            VRATOIITR310034();
            VRATOIITR310051();
            VRATOIITR310052();
            VRATOIITR310053();
            VRATOIITR310054();
            VRATOIITR310062();
            VRATOIITR310064();
            VRATOIITR310065();
            VRATOIITR310074();
            VRATOIITR310075();
            VRATOIITR310076();
            VRATOIITR310079();
            VRATOIITR310080();
            VRATOIITR310081();
            VRATOIITR310082();
            VRATOIITR310092();
            VRATOIITR310143();
            VRATOIITR310144();
            VRATOIITR310145();
            VRATOIITR310146();
            VRATOIITR310147();
            VRATOIITR310148();
            VRATOIITR310149();
            VRATOIITR310150();
            VRATOIITR310151();
            VRATOIITR310152();
            VRATOIITR310153();
            VRATOIITR310155();
            VRATOIITR310156();
            VRATOIITR310157();
            VRATOIITR310161();
            VRATOIITR310162();
            VRATOIITR310163();
            VRATOIITR310165();
            VRATOIITR310167();
            VRATOIITR310168();
            VRATOIITR310169();
            VRATOIITR310170();
            VRATOIITR310171();
            VRATOIITR310172();
            VRATOIITR310173();
            VRATOIITR310174();
            VRATOIITR310175();
            VRATOIITR310176();
            VRATOIITR310177();
            VRATOIITR310178();
            VRATOIITR310179();
            VRATOIITR310181();
            VRATOIITR310182();
            VRATOIITR310183();
            VRATOIITR310184();
            VRATOIITR310185();
            VRATOIITR310186();
            VRATOIITR310198();
            VRATOIITR310199();
            VRATOIITR310207();
            VRATOIITR310208();
            VRATOIITR310209();
            VRATOIITR310210();
            VRATOIITR310211();
            VRATOIITR310212();
            VRATOIITR310213();
            VRATOIITR310214();
            VRATOIITR310215();
            VRATOIITR310216();
            VRATOIITR310217();
            VRATOIITR310218();
            VRATOIITR310219();
            VRATOIITR310220();
            VRATOIITR310222();
            VRATOIITR310224();
            VRATOIITR310225();
            VRATOIITR310227();
            VRATOIITR310228();
            VRATOIITR310229();
            VRATOIITR310230();
            VRATOIITR400006();
            VRATOIITR400007();
            VRATOIITR400008();
            VRATOIITR400009();
            VRATOIITR400010();
            VRATOIITR400011();
            VRATOIITR400012();
            VRATOIITR400013();
            VRATOIITR400014();
            VRATOIITR400015();
            VRATOIITR400016();
            VRATOIITR400017();
            VRATOIITR400018();
            VRATOIITR400019();
            VRATOIITR400020();
            VRATOIITR400021();
            VRATOIITR400022();
            VRATOIITR400023();
            VRATOIITR400024();
            VRATOIITR400025();
            VRATOIITR400026();
            VRATOIITR400027();
            VRATOIITR400028();
            VRATOIITR400029();
            VRATOIITR400030();
            VRATOIITR400031();
            VRATOIITR400033();
            VRATOIITR400034();
            VRATOIITR400035();
            VRATOIITR400036();
            VRATOIITR400037();
            VRATOIITR400038();
            VRATOIITR400039();
            VRATOIITR400040();
            VRATOIITR400041();
            VRATOIITR400042();
            VRATOIITR400047();
            VRATOIITR400048();
            VRATOIITR400049();
            VRATOIITR400050();
            VRATOIITR400051();
            VRATOIITR400052();
            VRATOIITR400054();
            VRATOIITR400055();
            VRATOIITR400056();
            VRATOIITR400065();
            VRATOIITR400066();
            VRATOIITR400070();
            VRATOIITR400073();
            VRATOIITR400077();
            VRATOIITR400078();
            VRATOIITR400079();
            VRATOIITR400080();
            VRATOIITR400081();
            VRATOIITR400082();
            VRATOIITR400083();
            VRATOIITR400084();
            VRATOIITR400085();
            VRATOIITR400086();
            VRATOIITR400087();
            VRATOIITR400088();
            VRATOIITR400096();
            VRATOIITR400097();
            VRATOIITR400100();
            VRATOIITR400151();
            VRATOIITR410000();
            VRATOIITR410001();
            VRATOIITR410002();
            VRATOIITR410003();
            VRATOIITR410005();
            VRATOIITR410007();
            VRATOIITR410008();
            VRATOIITR410009();
            VRATOIITR410026();
            VRATOIITR410064();
            VRATOIITR410066();
            VRATOIITR410067();
            VRATOIITR410075();
            VRATOIITR410081();
            VRATOIITR410082();
            VRATOIITR410087();
            VRATOIITR410088();
            VRATOIITR410089();
            VRATOIITR410090();
            VRATOIITR410091();
            VRATOIITR410093();
            VRATOIITR410094();
            VRATOIITR410100();
            VRATOIITR410101();
            VRATOIITR410102();
            VRATOIITR410118();
            VRATOIITR410119();
            VRATOIITR410120();
            VRATOIITR410122();
            VRATOIITR410123();
            VRATOIITR410124();
            VRATOIITR410125();
            VRATOIITR410126();
            VRATOIITR410152();
            VRATOIITR410153();
            VRATOIITR410154();
            VRATOIITR410155();
            VRATOIITR410156();
            VRATOIITR410157();
            VRATOIITR410158();
            VRATOIITR410159();
            VRATOIITR410189();
            VRATOIITR410192();
            VRATOIITR410193();
            VRATOIITR410206();
            VRATOIITR410207();
            VRATOIITR600001();
            VRATOIITR600012();
            VRATOIITR600013();
            VRATOIITR600015();
            VRATOIITR600016();
            VRATOIITR600025();
            VRATOIITR600026();
            VRATOIITR600032();
            VRATOIITR600033();
            VRATOIITR600049();
            VRATOIITR600050();
            VRATOIITR600052();
            VRATOIITR600053();
            VRATOIITR600057();
            VRATOIITR600058();
            VRATOIITR600059();
            VRATOIITR600060();
            VRATOIITR600062();
            VRATOIITR600063();
            VRATOIITR600064();
            VRATOIITR600065();
            VRATOIITR600066();
            VRATOIITR600067();
            VRATOIITR600068();
            VRATOIITR600069();
            VRATOIITR600070();
            VRATOIITR600071();
            VRATOIITR600089();
            VRATOIITR600090();
            VRATOIITR600091();
            VRATOIITR600093();
            VRATOIITR600094();
            VRATOIITR600098();
            VRATOIITR600100();
            VRATOIITR600101();
            VRATOIITR600102();
            VRATOIITR600103();
            VRATOIITR610005();
            VRATOIITR610006();
            VRATOIITR610007();
            VRATOIITR610008();
            VRATOIITR610047();
            VRATOIITR610054();
            VRATOIITR610055();
            VRATOIITR610076();
            VRATOIITR610078();
            VRATOIITR610129();
            VRATOIITR610149();
            VRATOIITR630002();
            VRATOIITR630003();
            VRATOIITR630006();
            VRATOIITR630007();
            VRATOIITR630014();
            VRATOIITR630015();
            VRATOIITR630016();
            VRATOIITR700002();
            VRATOIITR700003();
            VRATOIITR700004();
            VRATOIITR730075();
            VRATOIITR730160();
            VRATOIITR730161();
            VRATOIITR730162();
            VRATOIITR730263();
            VRATOIITR730266();
            VRATOIITR730267();
            VRATOIITR730268();
            VRATOIITR730269();
            VRATOIITR730270();
            VRATOIITR730271();
            VRATOIITR730272();
            VRATOIITR730273();
            VRATOIITR730320();
            VRATOIITR730321();
            VRATOIITR730322();
            VRATOIITR730323();
            VRATOIITR730360();
            VRATOIITR730361();
            VRATOIITR730362();
            VRATOIITR730363();
            VRATOIITR730364();
            VRATOIITR730365();
            VRATOIITR730366();
            VRATOIITR730367();
            VRATOIITR730368();
            VRATOIITR730369();
            VRATOIITR730370();
            VRATOIITR730371();
            VRATOIITR730372();
            VRATOIITR730373();
            VRATOIITR730489();
            VRATOIITR730507();
            VRATOIITR730508();
            VRATOIITR730509();
            VRATOIITR800013();
			VRATOIITR800014();
            VRATOIITRW00001();

            #region Repeating RPLOSSSeqNumContext  
            if (report.RPLOSSSeqNumContextCollection != null)
            {
                int itemIndexContext2 = 0;
                foreach (IITR2026.RPLOSSSeqNumContext rplossSeqNumContext in report.RPLOSSSeqNumContextCollection)
                {
                    itemIndexContext2++;
                    VRATOGEN001023_1(rplossSeqNumContext);
                    VRATOGEN500030_1(rplossSeqNumContext);
                    VRATOGEN500033_1(rplossSeqNumContext);
                    VRATOIITR300037(rplossSeqNumContext);
                    VRATOIITR310023(rplossSeqNumContext);
                    VRATOIITR310042(rplossSeqNumContext);
                    VRATOIITR310045(rplossSeqNumContext);
                    VRATOIITR310048(rplossSeqNumContext);
                    VRATOIITR400053(rplossSeqNumContext);
                    VRATOIITR410006(rplossSeqNumContext);
                    VRATOIITR410011(rplossSeqNumContext);
                    VRATOIITR410014(rplossSeqNumContext);
                    VRATOIITR410016(rplossSeqNumContext);
                    VRATOIITR410071(rplossSeqNumContext);
                    VRATOIITR410106(rplossSeqNumContext);
                    VRATOIITR410111(rplossSeqNumContext);
                    VRATOIITR410113(rplossSeqNumContext);
                    VRATOIITR410114(rplossSeqNumContext);
                    VRATOIITR410115(rplossSeqNumContext);
                    VRATOIITR410190(rplossSeqNumContext);
                    VRATOIITR410199(rplossSeqNumContext);
                    VRATOIITR410203(rplossSeqNumContext);
                    VRATOIITR410205(rplossSeqNumContext);
                    VRATOIITR410208(rplossSeqNumContext);
                    VRATOIITR600083(rplossSeqNumContext);
                    VRATOIITR700001(rplossSeqNumContext);
					VRATOIITR800015(rplossSeqNumContext);
                }
            }

            #endregion // Foreach loop

            #region Repeating RPPHISeqNumContext  
            if (report.RPPHISeqNumContextCollection != null)
            {
                int itemIndexContext2 = 0;
                foreach (IITR2026.RPPHISeqNumContext rpphiSeqNumContext in report.RPPHISeqNumContextCollection)
                {
                    itemIndexContext2++;
                    VRATOGEN001023_2(rpphiSeqNumContext);
                    VRATOGEN500030_2(rpphiSeqNumContext);
                    VRATOGEN500033_2(rpphiSeqNumContext);
                    VRATOIITR000600(rpphiSeqNumContext);
                    VRATOIITR000601(rpphiSeqNumContext);
                    VRATOIITR000602(rpphiSeqNumContext);
                    VRATOIITR000603(rpphiSeqNumContext);
                    VRATOIITR000605(rpphiSeqNumContext);
                    VRATOIITR000618(rpphiSeqNumContext);
                    VRATOIITR000619(rpphiSeqNumContext);
                    VRATOIITR000620(rpphiSeqNumContext);
                    VRATOIITR000630(rpphiSeqNumContext);
                    VRATOIITR210003(rpphiSeqNumContext);
                    VRATOIITR400058(rpphiSeqNumContext);
                    VRATOIITR600085(rpphiSeqNumContext);
                }
            }

            #endregion // Foreach loop

            #region Repeating IITR310Collection  
            if (report.IITR310Collection != null)
            {
                int itemIndex1 = 0;
                foreach (string IITR310 in report.IITR310Collection)
                {
                    itemIndex1++;
                    VRATOGEN500076(IITR310, itemIndex1);
                }
            }

            #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }



        #region VR.ATO.GEN.000209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            Count(^RP:Context) <> 1

            Technical Business Rule Format:
            Count(^RP:Context) <> 1
    
            Data Elements:
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000458
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000458()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000458
            The Trustee Suffix code is not a valid code
    
            Legacy Rule Format:
            (^IITR23 <> BLANK) AND (NotInDomain(^IITR23, 'SUFFIX CODES'))

            Technical Business Rule Format:
            (^IITR23 <> BLANK) AND (NotInDomain(^IITR23, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^IITR23 = RP:PersonNameDetails.NameSuffix.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR23) != true && !(IsMatch(report.IITR23, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR23", Value = GetValueOrEmpty(report.IITR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000459
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000459()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000459
            The Title code is not a valid code
    
            Legacy Rule Format:
            (^IITR21 <> BLANK) AND (NotInDomain(^IITR21, 'TITLE CODES'))

            Technical Business Rule Format:
            (^IITR21 <> BLANK) AND (NotInDomain(^IITR21, 'TITLE CODES'))
    
            Data Elements:
    
            ^IITR21 = RP:PersonNameDetails.Title.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR21) != true && !(IsMatch(report.IITR21, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR21", Value = GetValueOrEmpty(report.IITR21) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP(Instant):Context <> NULL AND ^RP(Instant):TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP(Instant):Context <> NULL AND ^RP(Instant):TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPInstantExists != false && report.RPInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_1(IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party

            Legacy Rule Format:
            ^RP.{LOSSSeqNum}:Context <> NULL AND ^RP.{LOSSSeqNum}:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.{LOSSSeqNum}:Context <> NULL AND ^RP.{LOSSSeqNum}:TFN <> ^RP:TFN

            Data Elements:
            */
            assertion = (rplossSeqNumContext.Exists != false && rplossSeqNumContext.IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rplossSeqNumContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_2(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party

            Legacy Rule Format:
            ^RP.{PHISeqNum}:Context <> NULL AND ^RP.{PHISeqNum}:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.{PHISeqNum}:Context <> NULL AND ^RP.{PHISeqNum}:TFN <> ^RP:TFN

            Data Elements:
            */
            assertion = (rpphiSeqNumContext.Exists != false && rpphiSeqNumContext.IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rpphiSeqNumContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.BUS:Context <> NULL AND ^RP.BUS:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.BUS:Context <> NULL AND ^RP.BUS:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPBUSExists != false && report.RPBUSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPBUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Eligible:Context <> NULL AND ^RP.Eligible:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Eligible:Context <> NULL AND ^RP.Eligible:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPEligibleExists != false && report.RPEligibleIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPEligibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Ineligible:Context <> NULL AND ^RP.Ineligible:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Ineligible:Context <> NULL AND ^RP.Ineligible:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPIneligibleExists != false && report.RPIneligibleIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIneligibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Investor:Context <> NULL AND ^RP.Investor:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Investor:Context <> NULL AND ^RP.Investor:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPInvestorExists != false && report.RPInvestorIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Investor.YPrevious:Context <> NULL AND ^RP.Investor.YPrevious:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Investor.YPrevious:Context <> NULL AND ^RP.Investor.YPrevious:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPInvestorYPreviousExists != false && report.RPInvestorYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.JFOREIGN:Context <> NULL AND ^RP.JFOREIGN:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.JFOREIGN:Context <> NULL AND ^RP.JFOREIGN:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.LimitedPartners:Context <> NULL AND ^RP.LimitedPartners:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.LimitedPartners:Context <> NULL AND ^RP.LimitedPartners:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPLimitedPartnersExists != false && report.RPLimitedPartnersIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.LimitedPartners.YPrevious:Context <> NULL AND ^RP.LimitedPartners.YPrevious:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.LimitedPartners.YPrevious:Context <> NULL AND ^RP.LimitedPartners.YPrevious:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPLimitedPartnersYPreviousExists != false && report.RPLimitedPartnersYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.NonPrim:Context <> NULL AND ^RP.NonPrim:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.NonPrim:Context <> NULL AND ^RP.NonPrim:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPNonPrimExists != false && report.RPNonPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.NonPrim(Instant):Context <> NULL AND ^RP.NonPrim(Instant):TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.NonPrim(Instant):Context <> NULL AND ^RP.NonPrim(Instant):TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPNonPrimInstantExists != false && report.RPNonPrimInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Partnership:Context <> NULL AND ^RP.Partnership:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Partnership:Context <> NULL AND ^RP.Partnership:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPPartnershipExists != false && report.RPPartnershipIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPartnershipOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.POS:Context <> NULL AND ^RP.POS:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.POS:Context <> NULL AND ^RP.POS:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPPOSExists != false && report.RPPOSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPOSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Prim:Context <> NULL AND ^RP.Prim:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Prim:Context <> NULL AND ^RP.Prim:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPPrimExists != false && report.RPPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Prim(Instant):Context <> NULL AND ^RP.Prim(Instant):TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Prim(Instant):Context <> NULL AND ^RP.Prim(Instant):TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPPrimInstantExists != false && report.RPPrimInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.RES:Context <> NULL AND ^RP.RES:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.RES:Context <> NULL AND ^RP.RES:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPRESExists != false && report.RPRESIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRESOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.SPOUSE:Context <> NULL AND ^RP.SPOUSE:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.SPOUSE:Context <> NULL AND ^RP.SPOUSE:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPSPOUSEExists != false && report.RPSPOUSEIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.SPOUSE.Eligible:Context <> NULL AND ^RP.SPOUSE.Eligible:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.SPOUSE.Eligible:Context <> NULL AND ^RP.SPOUSE.Eligible:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPSPOUSEEligibleExists != false && report.RPSPOUSEEligibleIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEEligibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.SPOUSE.Ineligible:Context <> NULL AND ^RP.SPOUSE.Ineligible:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.SPOUSE.Ineligible:Context <> NULL AND ^RP.SPOUSE.Ineligible:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPSPOUSEIneligibleExists != false && report.RPSPOUSEIneligibleIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEIneligibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.TrueAndCorrect:Context <> NULL AND ^RP.TrueAndCorrect:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.TrueAndCorrect:Context <> NULL AND ^RP.TrueAndCorrect:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPTrueAndCorrectExists != false && report.RPTrueAndCorrectIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.Trust:Context <> NULL AND ^RP.Trust:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.Trust:Context <> NULL AND ^RP.Trust:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPTrustExists != false && report.RPTrustIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrustOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_23()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            ^RP.WHM:Context <> NULL AND ^RP.WHM:TFN <> ^RP:TFN

            Technical Business Rule Format:
            ^RP.WHM:Context <> NULL AND ^RP.WHM:TFN <> ^RP:TFN
    
            Data Elements:
            */
            assertion = (report.RPWHMExists != false && report.RPWHMIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPWHMOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428247
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428247()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428247
            Your Tax Agent number has failed the Tax Agent registration algorithm check. Input the correct Tax Agent registration number.
    
            Legacy Rule Format:
            (^IITR542 <> BLANK) AND (FailsTANAlgorithm(^IITR542))

            Technical Business Rule Format:
            (^IITR542 <> BLANK) AND (FailsTANAlgorithm(^IITR542))
    
            Data Elements:
    
            ^IITR542 = INT.TrueAndCorrect:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR542) != true && FailsTANAlgorithm(report.IITR542));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR542", Value = report.IITR542 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430230
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430230()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430230
            The e-mail address must contain an @ symbol
    
            Legacy Rule Format:
            (^IITR51 <> BLANK) AND (NotContains(^IITR51, '@'))

            Technical Business Rule Format:
            (^IITR51 <> BLANK) AND (NotContains(^IITR51, '@'))
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR51) != true && !(report.IITR51 == null ? true : report.IITR51.ToUpperInvariant().Contains(@"@")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain an @ symbol",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430230" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430231
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430231()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430231
            An email address must not start with a space or include a space within it
    
            Legacy Rule Format:
            Contains(^IITR51, ' ')

            Technical Business Rule Format:
            Contains(^IITR51, ' ')
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.IITR51 == null ? false : report.IITR51.ToUpperInvariant().Contains(@" "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430231",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An email address must not start with a space or include a space within it",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430231" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430232
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430232()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430232
            E-mail address must contain a dot (.)
    
            Legacy Rule Format:
            (^IITR51 <> BLANK) AND (NotContains(^IITR51, '.'))

            Technical Business Rule Format:
            (^IITR51 <> BLANK) AND (NotContains(^IITR51, '.'))
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR51) != true && !(report.IITR51 == null ? true : report.IITR51.ToUpperInvariant().Contains(@".")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430232",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain a dot (.)",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430232" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.432041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN432041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.432041
            Bank account name must not be longer than 32 characters
    
            Legacy Rule Format:
            Length(^IITR55) > 32

            Technical Business Rule Format:
            Length(^IITR55) > 32
    
            Data Elements:
    
            ^IITR55 = RP:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = (Length(report.IITR55) > 32);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not be longer than 32 characters",
                    LongDescription = @"32 characters is the Banking Standard and set by the Banking Industry",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.432041" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR55", Value = GetValueOrEmpty(report.IITR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438019
            If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied.
    
            Legacy Rule Format:
            (^IITR309 = TRUE) AND (Count(^IITR310) = 0)

            Technical Business Rule Format:
            (^IITR309 = TRUE) AND (Count(^IITR310) = 0)
    
            Data Elements:
    
            ^IITR309 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^IITR310 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (report.IITR309.GetValueOrDefault() && Count(report.IITR310Collection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code not supplied",
                    LongDescription = @"If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438019" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR309", Value = GetValueOrEmpty(report.IITR309) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438021
            If an exemption or rollover code(s) is provided then the exemption or rollover applied indicator must be true
    
            Legacy Rule Format:
            (Count(^IITR310) > 0) AND (^IITR309 <> TRUE)

            Technical Business Rule Format:
            (Count(^IITR310) > 0) AND (^IITR309 <> TRUE)
    
            Data Elements:
    
            ^IITR310 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
    
            ^IITR309 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
            */
            assertion = (Count(report.IITR310Collection) > 0 && report.IITR309 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover applied indicator must be true",
                    LongDescription = @"If an exemption or rollover code(s) is provided then the exemption or rollover applied indicator must be true",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR309", Value = GetValueOrEmpty(report.IITR309) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438058
            The number of Reporting Party Investor contexts must not exceed 1
    
            Legacy Rule Format:
            Count(^RP.Investor:Context) > 1

            Technical Business Rule Format:
            Count(^RP.Investor:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPInvestorCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Investor contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438058" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438059
            The number of Reporting Party Limited Partners contexts must not exceed 1
    
            Legacy Rule Format:
            Count(^RP.LimitedPartners:Context) > 1

            Technical Business Rule Format:
            Count(^RP.LimitedPartners:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPLimitedPartnersCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Limited Partners contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438059" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438070
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438070()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438070
            If an Intermediary is present, Intermediary Declarer contact name must be provided.
    
            Legacy Rule Format:
            ((^INT.TrueAndCorrect:Context <> NULL) AND (^IITR538 = BLANK))

            Technical Business Rule Format:
            ((^INT.TrueAndCorrect:Context <> NULL) AND (^IITR538 = BLANK))
    
            Data Elements:
    
            ^IITR538 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.IITR538) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438070",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer contact name must be provided",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer contact name must be provided.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438070" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR538", Value = report.IITR538 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438076
            If an Intermediary is present, Intermediary Declarer Telephone number must be provided.
    
            Legacy Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR541 = BLANK)

            Technical Business Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR541 = BLANK)
    
            Data Elements:
    
            ^IITR541 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.IITR541) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438076",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact telephone details must be provided",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer Telephone number must be provided.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438076" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR541", Value = report.IITR541 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438078
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438078()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438078
            If an Intermediary is present, Intermediary Declarer Signature Date must be present.
    
            Legacy Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR543 = NULL)

            Technical Business Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR543 = NULL)
    
            Data Elements:
    
            ^IITR543 = INT.TrueAndCorrect:Declaration.Signature.Date
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.IITR543 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438078",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signature Date is required",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer Signature Date must be present.",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438078" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR543", Value = GetValueOrEmptyDate(report.IITR543) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438081
            Invalid context. The number of Reporting Party Investor YPrevious contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP.Investor.Yprevious:Context) > 1

            Technical Business Rule Format:
            Count(^RP.Investor.Yprevious:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPInvestorYPreviousCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438081",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:Investor:YPrevious contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438081" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438082
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438082()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438082
            Invalid context. The number of Reporting Party Limited Partners YPrevious contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP.LimitedPartners.Yprevious:Context) > 1

            Technical Business Rule Format:
            Count(^RP.LimitedPartners.Yprevious:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPLimitedPartnersYPreviousCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438082",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:Limited Partners:YPrevious contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438082" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500009
            The TFN has failed the algorithm check.
    
            Legacy Rule Format:
            (FailsTFNAlgorithm(^RP:TFN))

            Technical Business Rule Format:
            (FailsTFNAlgorithm(^RP:TFN))
    
            Data Elements:
            */
            assertion = FailsTFNAlgorithm(report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500009" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^INT.TrueAndCorrect:StartDate <> NULL AND ^INT.TrueAndCorrect:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^INT.TrueAndCorrect:StartDate <> NULL AND ^INT.TrueAndCorrect:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.INTTrueAndCorrectExists ) && (report.INTTrueAndCorrectStartDate != null && report.INTTrueAndCorrectStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_1(IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.

            Legacy Rule Format:
            ^RP.{LOSSSeqNum}:StartDate <> NULL AND ^RP.{LOSSSeqNum}:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.{LOSSSeqNum}:StartDate <> NULL AND ^RP.{LOSSSeqNum}:StartDate <> ^RP:StartDate

            Data Elements:
            */
            assertion = (rplossSeqNumContext.StartDate != null && rplossSeqNumContext.StartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rplossSeqNumContext.OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_2(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.

            Legacy Rule Format:
            ^RP.{PHISeqNum}:StartDate <> NULL AND ^RP.{PHISeqNum}:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.{PHISeqNum}:StartDate <> NULL AND ^RP.{PHISeqNum}:StartDate <> ^RP:StartDate

            Data Elements:
            */
            assertion = (rpphiSeqNumContext.StartDate != null && rpphiSeqNumContext.StartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rpphiSeqNumContext.OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.BUS:StartDate <> NULL AND ^RP.BUS:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.BUS:StartDate <> NULL AND ^RP.BUS:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = (report.RPBUSExists && (report.RPBUSStartDate != null && report.RPBUSStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPBUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Eligible:StartDate <> NULL AND ^RP.Eligible:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Eligible:StartDate <> NULL AND ^RP.Eligible:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = (report.RPEligibleExists && (report.RPEligibleStartDate != null && report.RPEligibleStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPEligibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Ineligible:StartDate <> NULL AND ^RP.Ineligible:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Ineligible:StartDate <> NULL AND ^RP.Ineligible:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPIneligibleExists ) && (report.RPIneligibleStartDate != null && report.RPIneligibleStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIneligibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Investor:StartDate <> NULL AND ^RP.Investor:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Investor:StartDate <> NULL AND ^RP.Investor:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPInvestorExists ) && (report.RPInvestorStartDate != null && report.RPInvestorStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Investor.YPrevious:StartDate <> NULL AND ^RP.Investor.YPrevious:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Investor.YPrevious:StartDate <> NULL AND ^RP.Investor.YPrevious:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPInvestorYPreviousExists ) && (report.RPInvestorYPreviousStartDate != null && report.RPInvestorYPreviousStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.JFOREIGN:StartDate <> NULL AND ^RP.JFOREIGN:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.JFOREIGN:StartDate <> NULL AND ^RP.JFOREIGN:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPJFOREIGNExists ) && (report.RPJFOREIGNStartDate != null && report.RPJFOREIGNStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.LimitedPartners:StartDate <> NULL AND ^RP.LimitedPartners:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.LimitedPartners:StartDate <> NULL AND ^RP.LimitedPartners:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPLimitedPartnersExists ) && (report.RPLimitedPartnersStartDate != null && report.RPLimitedPartnersStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.LimitedPartners.YPrevious:StartDate <> NULL AND ^RP.LimitedPartners.YPrevious:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.LimitedPartners.YPrevious:StartDate <> NULL AND ^RP.LimitedPartners.YPrevious:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPLimitedPartnersYPreviousExists ) && (report.RPLimitedPartnersYPreviousStartDate != null && report.RPLimitedPartnersYPreviousStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.NonPrim:StartDate <> NULL AND ^RP.NonPrim:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.NonPrim:StartDate <> NULL AND ^RP.NonPrim:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPNonPrimExists ) && (report.RPNonPrimStartDate != null && report.RPNonPrimStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Partnership:StartDate <> NULL AND ^RP.Partnership:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Partnership:StartDate <> NULL AND ^RP.Partnership:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPPartnershipExists ) && (report.RPPartnershipStartDate != null && report.RPPartnershipStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPartnershipOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.POS:StartDate <> NULL AND ^RP.POS:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.POS:StartDate <> NULL AND ^RP.POS:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPPOSExists ) && (report.RPPOSStartDate != null && report.RPPOSStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPOSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Prim:StartDate <> NULL AND ^RP.Prim:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Prim:StartDate <> NULL AND ^RP.Prim:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPPrimExists ) && (report.RPPrimStartDate != null && report.RPPrimStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.RES:StartDate <> NULL AND ^RP.RES:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.RES:StartDate <> NULL AND ^RP.RES:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPRESExists ) && (report.RPRESStartDate != null && report.RPRESStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRESOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.SPOUSE:StartDate <> NULL AND ^RP.SPOUSE:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.SPOUSE:StartDate <> NULL AND ^RP.SPOUSE:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPSPOUSEExists ) && (report.RPSPOUSEStartDate != null && report.RPSPOUSEStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.SPOUSE.Eligible:StartDate <> NULL AND ^RP.SPOUSE.Eligible:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.SPOUSE.Eligible:StartDate <> NULL AND ^RP.SPOUSE.Eligible:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPSPOUSEEligibleExists ) && (report.RPSPOUSEEligibleStartDate != null && report.RPSPOUSEEligibleStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEEligibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.SPOUSE.Ineligible:StartDate <> NULL AND ^RP.SPOUSE.Ineligible:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.SPOUSE.Ineligible:StartDate <> NULL AND ^RP.SPOUSE.Ineligible:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPSPOUSEIneligibleExists ) && (report.RPSPOUSEIneligibleStartDate != null && report.RPSPOUSEIneligibleStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEIneligibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.TrueAndCorrect:StartDate <> NULL AND ^RP.TrueAndCorrect:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.TrueAndCorrect:StartDate <> NULL AND ^RP.TrueAndCorrect:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPTrueAndCorrectExists ) && (report.RPTrueAndCorrectStartDate != null && report.RPTrueAndCorrectStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Trust:StartDate <> NULL AND ^RP.Trust:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.Trust:StartDate <> NULL AND ^RP.Trust:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPTrustExists ) && (report.RPTrustStartDate != null && report.RPTrustStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrustOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500030_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.WHM:StartDate <> NULL AND ^RP.WHM:StartDate <> ^RP:StartDate

            Technical Business Rule Format:
            ^RP.WHM:StartDate <> NULL AND ^RP.WHM:StartDate <> ^RP:StartDate
    
            Data Elements:
            */
            assertion = ((report.RPWHMExists ) && (report.RPWHMStartDate != null && report.RPWHMStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPWHMOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^INT.TrueAndCorrect:EndDate <> NULL AND ^INT.TrueAndCorrect:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^INT.TrueAndCorrect:EndDate <> NULL AND ^INT.TrueAndCorrect:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.INTTrueAndCorrectExists ) && (report.INTTrueAndCorrectEndDate != null && report.INTTrueAndCorrectEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_1(IITR2026.RPLOSSSeqNumContext rplossSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.

            Legacy Rule Format:
            ^RP.{LOSSSeqNum}:EndDate <> NULL AND ^RP.{LOSSSeqNum}:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.{LOSSSeqNum}:EndDate <> NULL AND ^RP.{LOSSSeqNum}:EndDate <> ^RP:EndDate

            Data Elements:
            */
            assertion = (rplossSeqNumContext.EndDate != null && rplossSeqNumContext.EndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rplossSeqNumContext.OccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_2(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.

            Legacy Rule Format:
            ^RP.{PHISeqNum}:EndDate <> NULL AND ^RP.{PHISeqNum}:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.{PHISeqNum}:EndDate <> NULL AND ^RP.{PHISeqNum}:EndDate <> ^RP:EndDate

            Data Elements:
            */
            assertion = (rpphiSeqNumContext.EndDate != null && rpphiSeqNumContext.EndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rpphiSeqNumContext.OccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.BUS:EndDate <> NULL AND ^RP.BUS:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.BUS:EndDate <> NULL AND ^RP.BUS:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPBUSExists ) && (report.RPBUSEndDate != null && report.RPBUSEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPBUSOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Eligible:EndDate <> NULL AND ^RP.Eligible:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Eligible:EndDate <> NULL AND ^RP.Eligible:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPEligibleExists ) && (report.RPEligibleEndDate != null && report.RPEligibleEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPEligibleOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Ineligible:EndDate <> NULL AND ^RP.Ineligible:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Ineligible:EndDate <> NULL AND ^RP.Ineligible:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPIneligibleExists ) && (report.RPIneligibleEndDate != null && report.RPIneligibleEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIneligibleOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Investor:EndDate <> NULL AND ^RP.Investor:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Investor:EndDate <> NULL AND ^RP.Investor:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPInvestorExists ) && (report.RPInvestorEndDate != null && report.RPInvestorEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Investor.YPrevious:EndDate <> NULL AND ^RP.Investor.YPrevious:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Investor.YPrevious:EndDate <> NULL AND ^RP.Investor.YPrevious:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPInvestorYPreviousExists ) && (report.RPInvestorYPreviousEndDate != null && report.RPInvestorYPreviousEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.JFOREIGN:EndDate <> NULL AND ^RP.JFOREIGN:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.JFOREIGN:EndDate <> NULL AND ^RP.JFOREIGN:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPJFOREIGNExists ) && (report.RPJFOREIGNEndDate != null && report.RPJFOREIGNEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.LimitedPartners:EndDate <> NULL AND ^RP.LimitedPartners:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.LimitedPartners:EndDate <> NULL AND ^RP.LimitedPartners:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPLimitedPartnersExists ) && (report.RPLimitedPartnersEndDate != null && report.RPLimitedPartnersEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.LimitedPartners.YPrevious:EndDate <> NULL AND ^RP.LimitedPartners.YPrevious:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.LimitedPartners.YPrevious:EndDate <> NULL AND ^RP.LimitedPartners.YPrevious:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPLimitedPartnersYPreviousExists ) && (report.RPLimitedPartnersYPreviousEndDate != null && report.RPLimitedPartnersYPreviousEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.NonPrim:EndDate <> NULL AND ^RP.NonPrim:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.NonPrim:EndDate <> NULL AND ^RP.NonPrim:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPNonPrimExists ) && (report.RPNonPrimEndDate != null && report.RPNonPrimEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Partnership:EndDate <> NULL AND ^RP.Partnership:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Partnership:EndDate <> NULL AND ^RP.Partnership:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPPartnershipExists ) && (report.RPPartnershipEndDate != null && report.RPPartnershipEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPartnershipOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.POS:EndDate <> NULL AND ^RP.POS:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.POS:EndDate <> NULL AND ^RP.POS:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPPOSExists ) && (report.RPPOSEndDate != null && report.RPPOSEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPOSOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Prim:EndDate <> NULL AND ^RP.Prim:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Prim:EndDate <> NULL AND ^RP.Prim:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPPrimExists ) && (report.RPPrimEndDate != null && report.RPPrimEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.RES:EndDate <> NULL AND ^RP.RES:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.RES:EndDate <> NULL AND ^RP.RES:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPRESExists ) && (report.RPRESEndDate != null && report.RPRESEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRESOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.SPOUSE:EndDate <> NULL AND ^RP.SPOUSE:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.SPOUSE:EndDate <> NULL AND ^RP.SPOUSE:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPSPOUSEExists ) && (report.RPSPOUSEEndDate != null && report.RPSPOUSEEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.SPOUSE.Eligible:EndDate <> NULL AND ^RP.SPOUSE.Eligible:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.SPOUSE.Eligible:EndDate <> NULL AND ^RP.SPOUSE.Eligible:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPSPOUSEEligibleExists ) && (report.RPSPOUSEEligibleEndDate != null && report.RPSPOUSEEligibleEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEEligibleOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.SPOUSE.Ineligible:EndDate <> NULL AND ^RP.SPOUSE.Ineligible:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.SPOUSE.Ineligible:EndDate <> NULL AND ^RP.SPOUSE.Ineligible:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPSPOUSEIneligibleExists ) && (report.RPSPOUSEIneligibleEndDate != null && report.RPSPOUSEIneligibleEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSPOUSEIneligibleOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.TrueAndCorrect:EndDate <> NULL AND ^RP.TrueAndCorrect:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.TrueAndCorrect:EndDate <> NULL AND ^RP.TrueAndCorrect:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPTrueAndCorrectExists ) && (report.RPTrueAndCorrectEndDate != null && report.RPTrueAndCorrectEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.Trust:EndDate <> NULL AND ^RP.Trust:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.Trust:EndDate <> NULL AND ^RP.Trust:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPTrustExists ) && (report.RPTrustEndDate != null && report.RPTrustEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrustOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500033_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500033
            Invalid context. Period end date must be the same across all duration contexts.
    
            Legacy Rule Format:
            ^RP.WHM:EndDate <> NULL AND ^RP.WHM:EndDate <> ^RP:EndDate

            Technical Business Rule Format:
            ^RP.WHM:EndDate <> NULL AND ^RP.WHM:EndDate <> ^RP:EndDate
    
            Data Elements:
            */
            assertion = ((report.RPWHMExists ) && (report.RPWHMEndDate != null && report.RPWHMEndDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPWHMOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500033" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500037()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500037
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            ContainsSet(^IITR538, '"{","}","|","_","\","#"')

            Technical Business Rule Format:
            ContainsSet(^IITR538, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^IITR538 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = IsMatch(report.IITR538, @"\{|\}|\||_|\\|#", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500037" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR538", Value = report.IITR538 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500037_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500037
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            ContainsSet(^IITR550, '"{","}","|","_","\","#"')

            Technical Business Rule Format:
            ContainsSet(^IITR550, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^IITR550 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = IsMatch(report.IITR550, @"\{|\}|\||_|\\|#", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500037" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR550", Value = GetValueOrEmpty(report.IITR550) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500046
            Invalid context. The number of RP.TrueAndCorrect contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP.TrueAndCorrect:Context) > 1

            Technical Business Rule Format:
            Count(^RP.TrueAndCorrect:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPTrueAndCorrectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.TrueAndCorrect contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500046" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500051()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500051
            Scheme for Intermediary.TrueAndCorrect must be set to http://www.abr.gov.au/abn
    
            Legacy Rule Format:
            ^INT.TrueAndCorrect:IdentifierScheme <> 'http://www.abr.gov.au/abn'

            Technical Business Rule Format:
            ^INT.TrueAndCorrect:IdentifierScheme <> 'http://www.abr.gov.au/abn'
    
            Data Elements:
            */
            assertion = ((report.INTTrueAndCorrectExists ) && (report.INTTrueAndCorrectIdentifierScheme != @"http://www.abr.gov.au/abn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500051",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Intermediary.TrueAndCorrect must be ""http://www.abr.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500051" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500053()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500053
            Scheme for RP.TrueAndCorrect must be set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            ^RP.TrueAndCorrect:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.TrueAndCorrect:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPTrueAndCorrectExists ) && (report.RPTrueAndCorrectIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty.TrueAndCorrect must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500054
            ABN in the context definition must pass the ABN algorithm check
    
            Legacy Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (FailsABNAlgorithm(^INT.TrueAndCorrect:ABN))

            Technical Business Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (FailsABNAlgorithm(^INT.TrueAndCorrect:ABN))
    
            Data Elements:
            */
            assertion = (report.INTTrueAndCorrectExists != false && FailsABNAlgorithm(report.INTTrueAndCorrectIdentifierABN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary declarer Australian Business Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500055
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500055()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500055
            Declarer Contact Name has a maximum field length of 200
    
            Legacy Rule Format:
            Length(^IITR538) > 200

            Technical Business Rule Format:
            Length(^IITR538) > 200
    
            Data Elements:
    
            ^IITR538 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (Length(report.IITR538) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500055",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name must not exceed 200 characters",
                    LongDescription = @"For Declarer Contact Name, the maximum allowable characters for this form is 200",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500055" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR538", Value = report.IITR538 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500055
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500055_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500055
            Declarer Contact Name has a maximum field length of 200
    
            Legacy Rule Format:
            Length(^IITR550) > 200

            Technical Business Rule Format:
            Length(^IITR550) > 200
    
            Data Elements:
    
            ^IITR550 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (Length(report.IITR550) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500055",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name must not exceed 200 characters",
                    LongDescription = @"For Declarer Contact Name, the maximum allowable characters for this form is 200",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500055" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR550", Value = GetValueOrEmpty(report.IITR550) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500058
            Declarer Position held of person making the declaration has a maximum field length of 50
    
            Legacy Rule Format:
            Length(^IITR539) > 50

            Technical Business Rule Format:
            Length(^IITR539) > 50
    
            Data Elements:
    
            ^IITR539 = INT.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (Length(report.IITR539) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position must not exceed 50 characters",
                    LongDescription = @"For Declarer position, the maximum allowable characters for this form is 50",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500058" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR539", Value = GetValueOrEmpty(report.IITR539) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500058_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500058
            Declarer Position held of person making the declaration has a maximum field length of 50
    
            Legacy Rule Format:
            Length(^IITR551) > 50

            Technical Business Rule Format:
            Length(^IITR551) > 50
    
            Data Elements:
    
            ^IITR551 = RP.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (Length(report.IITR551) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position must not exceed 50 characters",
                    LongDescription = @"For Declarer position, the maximum allowable characters for this form is 50",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500058" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR551", Value = GetValueOrEmpty(report.IITR551) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR539 <> BLANK) AND (NotCharacterInSet(^IITR539, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR539 <> BLANK) AND (NotCharacterInSet(^IITR539, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR539 = INT.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR539) != true && !(IsMatch(report.IITR539, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position held of person making the declaration contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR539", Value = GetValueOrEmpty(report.IITR539) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500059_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR551 <> BLANK) AND (NotCharacterInSet(^IITR551, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR551 <> BLANK) AND (NotCharacterInSet(^IITR551, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR551 = RP.TrueAndCorrect:PersonNameDetails.Position.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR551) != true && !(IsMatch(report.IITR551, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position held of person making the declaration contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR551", Value = GetValueOrEmpty(report.IITR551) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500064
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500064()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500064
            Tax agent number is required when the declarer is a tax agent
    
            Legacy Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR542 = NULL)

            Technical Business Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR542 = NULL)
    
            Data Elements:
    
            ^IITR542 = INT.TrueAndCorrect:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.IITR542 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500064",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer tax agent number must be present",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500064" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR542", Value = report.IITR542 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500066
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500066()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500066
            Signature date must not be later than today
    
            Legacy Rule Format:
            (^IITR543 > Today())

            Technical Business Rule Format:
            (^IITR543 > Today())
    
            Data Elements:
    
            ^IITR543 = INT.TrueAndCorrect:Declaration.Signature.Date
            */
            assertion = (report.IITR543.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500066" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR543", Value = GetValueOrEmptyDate(report.IITR543) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500066
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500066_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500066
            Signature date must not be later than today
    
            Legacy Rule Format:
            (^IITR554 > Today())

            Technical Business Rule Format:
            (^IITR554 > Today())
    
            Data Elements:
    
            ^IITR554 = RP.TrueAndCorrect:Declaration.Signature.Date
            */
            assertion = (report.IITR554.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500066" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR554", Value = GetValueOrEmptyDate(report.IITR554) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500069
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500069()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500069
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = _ [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR545 <> BLANK) AND (NotCharacterInSet(^IITR545, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR545 <> BLANK) AND (NotCharacterInSet(^IITR545, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR545 = INT.TrueAndCorrect:Declaration.SignatoryIdentifier.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR545) != true && !(IsMatch(report.IITR545, @"^[a-zA-Z0-9!@\$%&\*\(\)\-_=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer identifier contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500069" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR545", Value = report.IITR545 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500069
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500069_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500069
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = _ [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR556 <> BLANK) AND (NotCharacterInSet(^IITR556, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR556 <> BLANK) AND (NotCharacterInSet(^IITR556, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR556 = RP.TrueAndCorrect:Declaration.SignatoryIdentifier.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR556) != true && !(IsMatch(report.IITR556, @"^[a-zA-Z0-9!@\$%&\*\(\)\-_=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer identifier contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500069" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR556", Value = GetValueOrEmpty(report.IITR556) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500072
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500072()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500072
            Invalid context. The number of RP.RES contexts must equal 1.
    
            Legacy Rule Format:
            Count(^RP.RES:Context) <> 1

            Technical Business Rule Format:
            Count(^RP.RES:Context) <> 1
    
            Data Elements:
            */
            assertion = (Count(report.RPRESCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500072",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.RES contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRESLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500072" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500073
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500073()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500073
            Scheme for RP.POS must be set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            ^RP.POS:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.POS:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPPOSExists ) && (report.RPPOSIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for RP.POS must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPOSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500073" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500074
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500074()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500074
            Scheme for RP.RES must be set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            ^RP.RES:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.RES:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPRESExists ) && (report.RPRESIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for RP.RES must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRESOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500074" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500075
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPExists ) && (report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500075" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500076(string IITR310, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500076
            The same exemption or rollover code has been entered more than once.

            Legacy Rule Format:
            (^IITR310 <> BLANK) AND HasDuplicateValues(^IITR310)

            Technical Business Rule Format:
            (^IITR310 <> BLANK) AND HasDuplicateValues(^IITR310)

            Data Elements:

            ^IITR310 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (string.IsNullOrWhiteSpace(IITR310) != true && HasDuplicateValues(report.IITR310Collection));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500076",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code must be unique",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code" + OccurrenceIndex(DuplicateValueIndex(report.IITR310Collection) + 1) + "",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500076" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR310", Value = GetValueOrEmpty(IITR310) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500077
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500077()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500077
            A maximum of 25 exemption or rollover codes are permitted
    
            Legacy Rule Format:
            Count(^IITR310) > 25

            Technical Business Rule Format:
            Count(^IITR310) > 25
    
            Data Elements:
    
            ^IITR310 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (Count(report.IITR310Collection) > 25);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500077",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The maximum number of exemption or rollover codes permitted is 25",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500077" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500088
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500088()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500088
            Invalid context. The number of Reporting Party (Instant) contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP(Instant):Context) > 1

            Technical Business Rule Format:
            Count(^RP(Instant):Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500088",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party (Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500088" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500089
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500089()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500089
            Entity identifier scheme for Reporting Party (Instant) must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP(Instant):IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP(Instant):IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPInstantExists ) && (report.RPInstantIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500089",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party (Instant) must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500089" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500090
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500090()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500090
            Invalid context. Period date of Reporting Party (Instant) must match period start date of Reporting Party.
    
            Legacy Rule Format:
            IF (RP(Instant):period.instantDate <> RP:period.startDate)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^InstantDate1 <> ^StartDate1
    
            Data Elements:
    
            ^StartDate1 = RP
    
            ^InstantDate1 = RP(Instant)
            */
            assertion = ((report.RPInstantExists ) && (report.RPInstantInstantDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500090",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party (Instant) period date is invalid",
                    LongDescription = @"Invalid context. Period date of Reporting Party (Instant) must match period start date of Reporting Party.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500090" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500094
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500094()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500094
            Intermediary Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR544 <> TRUE)

            Technical Business Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR544 <> TRUE)
    
            Data Elements:
    
            ^IITR544 = INT.TrueAndCorrect:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.IITR544 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500094",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500094" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR544", Value = GetValueOrEmpty(report.IITR544) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500100
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500100()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500100
            Reporting Party Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^RP.TrueAndCorrect:Context <> NULL) AND (^IITR555 <> TRUE)

            Technical Business Rule Format:
            (^RP.TrueAndCorrect:Context <> NULL) AND (^IITR555 <> TRUE)
    
            Data Elements:
    
            ^IITR555 = RP.TrueAndCorrect:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.RPTrueAndCorrectExists != false && report.IITR555 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR555", Value = GetValueOrEmpty(report.IITR555) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500103
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500103()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500103
            Invalid context. The number of RP.POS contexts must equal 1.
    
            Legacy Rule Format:
            Count(^RP.POS:Context) <> 1

            Technical Business Rule Format:
            Count(^RP.POS:Context) <> 1
    
            Data Elements:
            */
            assertion = (Count(report.RPPOSCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500103",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.POS contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPOSLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500103" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500146
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500146()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500146
            Invalid context. The number of Intermediary.TrueAndCorrect contexts must equal 1.
    
            Legacy Rule Format:
            Count(^INT.TrueAndCorrect:Context) <> 1

            Technical Business Rule Format:
            Count(^INT.TrueAndCorrect:Context) <> 1
    
            Data Elements:
            */
            assertion = (Count(report.INTTrueAndCorrectCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500146",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Intermediary.TrueAndCorrect contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500146" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500147
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500147()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500147
            Telephone number is required, if an area code is provided
    
            Legacy Rule Format:
            ^IITR553 = BLANK AND ^IITR580 <> BLANK

            Technical Business Rule Format:
            ^IITR553 = BLANK AND ^IITR580 <> BLANK
    
            Data Elements:
    
            ^IITR580 = RP.TrueAndCorrect:ElectronicContact.Telephone.Country.Code
    
            ^IITR553 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR553)  && string.IsNullOrWhiteSpace(report.IITR580) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500061",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"Declarer telephone number is required when Declarer telephone area code is provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500147" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR553", Value = GetValueOrEmpty(report.IITR553) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR580", Value = GetValueOrEmpty(report.IITR580) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500148
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500148()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500148
            Area code is required, if a Telephone number is provided
    
            Legacy Rule Format:
            ^IITR553 <> BLANK AND ^IITR580 = BLANK

            Technical Business Rule Format:
            ^IITR553 <> BLANK AND ^IITR580 = BLANK
    
            Data Elements:
    
            ^IITR580 = RP.TrueAndCorrect:ElectronicContact.Telephone.Country.Code
    
            ^IITR553 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR553) != true && string.IsNullOrWhiteSpace(report.IITR580) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500062",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"Declarer daytime phone area code is required when Declarer daytime phone number is provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500148" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR553", Value = GetValueOrEmpty(report.IITR553) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR580", Value = GetValueOrEmpty(report.IITR580) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500149()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500149
            If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.
    
            Legacy Rule Format:
            ^IITR541 = BLANK AND ^IITR581 <> BLANK

            Technical Business Rule Format:
            ^IITR541 = BLANK AND ^IITR581 <> BLANK
    
            Data Elements:
    
            ^IITR581 = INT.TrueAndCorrect:ElectronicContact.Telephone.Country.Code
    
            ^IITR541 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR541)  && string.IsNullOrWhiteSpace(report.IITR581) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438072",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500149" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR541", Value = report.IITR541 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR581", Value = report.IITR581 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500150
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500150()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500150
            If a telephone number is supplied and Intermediary is present, Intermediary area code is required.
    
            Legacy Rule Format:
            ^IITR541 <> BLANK AND ^IITR581 = BLANK

            Technical Business Rule Format:
            ^IITR541 <> BLANK AND ^IITR581 = BLANK
    
            Data Elements:
    
            ^IITR581 = INT.TrueAndCorrect:ElectronicContact.Telephone.Country.Code
    
            ^IITR541 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR541) != true && string.IsNullOrWhiteSpace(report.IITR581) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is present, Intermediary area code is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500150" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR541", Value = report.IITR541 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR581", Value = report.IITR581 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000001
            Current postal address - Address line 1 field contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-.
    
            Legacy Rule Format:
            (^IITR33 <> BLANK) AND (StartsWithSet(^IITR33,'"CO "') OR FoundSet(^IITR33, '"C/O","C/","Care Of"'))

            Technical Business Rule Format:
            (^IITR33 <> BLANK) AND (StartsWithSet(^IITR33,'"CO "') OR FoundSet(^IITR33, '"C/O","C/","Care Of"'))
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR33) != true && (IsMatch(report.IITR33, @"^(CO )", RegexOptions.IgnoreCase) || IsMatch(report.IITR33, @"(^(C/O|C/|Care Of)$)|(^(.*)( C/O | C/ | Care Of )(.*)$)|(^(C/O |C/ |Care Of )(.*)$)|(^(.*)( C/O| C/| Care Of)$)", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 1 is incorrect",
                    LongDescription = @"Current postal address - Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000002
            Current postal address - Address line 2 field cannot contain 'care of' reference.
    
            Legacy Rule Format:
            (^IITR34 <> BLANK) AND (StartsWithSet(^IITR34,'"CO "') OR FoundSet(^IITR34, '"C/O","C/","Care Of","C/-"'))

            Technical Business Rule Format:
            (^IITR34 <> BLANK) AND (StartsWithSet(^IITR34,'"CO "') OR FoundSet(^IITR34, '"C/O","C/","Care Of","C/-"'))
    
            Data Elements:
    
            ^IITR34 = RP.POS:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR34) != true && (IsMatch(report.IITR34, @"^(CO )", RegexOptions.IgnoreCase) || IsMatch(report.IITR34, @"(^(C/O|C/|Care Of|C/-)$)|(^(.*)( C/O | C/ | Care Of | C/- )(.*)$)|(^(C/O |C/ |Care Of |C/- )(.*)$)|(^(.*)( C/O| C/| Care Of| C/-)$)", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 2 must not contain a 'care of' reference",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR34", Value = GetValueOrEmpty(report.IITR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000003
            AS ABOVE' is not acceptable in the field.
    
            Legacy Rule Format:
            ((FoundSet(^IITR41, '"AS ABOVE"')) OR (FoundSet(^IITR42, '"AS ABOVE"')))

            Technical Business Rule Format:
            ((FoundSet(^IITR41, '"AS ABOVE"')) OR (FoundSet(^IITR42, '"AS ABOVE"')))
    
            Data Elements:
    
            ^IITR41 = RP.RES:AddressDetails.Line1.Text
    
            ^IITR42 = RP.RES:AddressDetails.Line2.Text
            */
            assertion = (IsMatch(report.IITR41, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase) || IsMatch(report.IITR42, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address must not contain ""AS ABOVE""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR41", Value = report.IITR41 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR42", Value = GetValueOrEmpty(report.IITR42) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000005
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000005()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000005
            Address line 1 contains 'Care of' reference and no street address in Address line 2. If 'Care of' reference is on Address line 1, it must be followed by a name then Address line 2 must contain the street address.
    
            Legacy Rule Format:
            ((Contains(^IITR33, 'C/-')) AND (^IITR34 = BLANK))

            Technical Business Rule Format:
            ((Contains(^IITR33, 'C/-')) AND (^IITR34 = BLANK))
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
    
            ^IITR34 = RP.POS:AddressDetails.Line2.Text
            */
            assertion = ((report.IITR33 == null ? false : report.IITR33.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.IITR34) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000005",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"C/- must be followed by a Name and the Address on line 2 for Current postal address",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000005" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR34", Value = GetValueOrEmpty(report.IITR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000006
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            ContainsSet(^IITR22, '"{","}","|","_","\","#"')

            Technical Business Rule Format:
            ContainsSet(^IITR22, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IITR22, @"\{|\}|\||_|\\|#", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000006",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000006" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000007
            This is an annual accounting submission. The dates provided should be for your annual accounting period, commencing on the 1st day of the month. For normal balancers, this will be the period 1 July - 30 June
    
            Legacy Rule Format:
            (((Year(^RP:EndDate) - Year(^RP:StartDate)) * 12) + MonthAsInt(^RP:EndDate) - MonthAsInt(^RP:StartDate) + 1) > 23

            Technical Business Rule Format:
            (((Year(^RP:EndDate) - Year(^RP:StartDate)) * 12) + MonthAsInt(^RP:EndDate) - MonthAsInt(^RP:StartDate) + 1) > 23
    
            Data Elements:
    
            ^EndDate = EndDate
    
            ^StartDate = StartDate
            */
            assertion = ((Year(report.RPEndDate) - Year(report.RPStartDate)) * 12 + MonthAsInt(report.RPEndDate) - MonthAsInt(report.RPStartDate) + 1 > 23);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The account period dates must not exceed 23 months for an approved transitional period",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000007" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000008
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used
    
            Legacy Rule Format:
            (FoundSet(^IITR22, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IITR22, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IITR22, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000009
            Year of return is mandatory
    
            Legacy Rule Format:
            ^IITR10 = NULL

            Technical Business Rule Format:
            ^IITR10 = NULL
    
            Data Elements:
    
            ^IITR10 = RP:Report.TargetFinancial.Year
            */
            assertion = (report.IITR10 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return must be present",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000009" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR10", Value = GetValueOrEmpty(report.IITR10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000010
            Australian residency indicator is mandatory
    
            Legacy Rule Format:
            ^IITR19 = NULL

            Technical Business Rule Format:
            ^IITR19 = NULL
    
            Data Elements:
    
            ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
            */
            assertion = (report.IITR19 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian residency indicator must be present",
                    Location = "/xbrli:xbrl/tns:Residency.TaxPurposesPersonStatus.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.IITR19) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000012
            Family Name field contains all numeric characters with any combination of spaces; at least one alpha must be present
    
            Legacy Rule Format:
            (^IITR22 <> BLANK) AND (NotContainsSet(^IITR22, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IITR22 <> BLANK) AND (NotContainsSet(^IITR22, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR22) != true && !(IsMatch(report.IITR22, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name field cannot contain all numeric characters including any spaces",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000012" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000014()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000014
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present
    
            Legacy Rule Format:
            (^IITR33 <> BLANK) AND (NotContainsSet(^IITR33, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IITR33 <> BLANK) AND (NotContainsSet(^IITR33, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR33) != true && !(IsMatch(report.IITR33, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 1 must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000014" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000015
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000015()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000015
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present
    
            Legacy Rule Format:
            (^IITR34 <> BLANK) AND (NotContainsSet(^IITR34, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IITR34 <> BLANK) AND (NotContainsSet(^IITR34, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IITR34 = RP.POS:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR34) != true && !(IsMatch(report.IITR34, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 2 must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000015" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR34", Value = GetValueOrEmpty(report.IITR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000016
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^IITR22, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^IITR22, '"--","''","  "'))
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IITR22, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name field contains repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000016" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000018
            Name field cannot contain 'space hyphen space'
    
            Legacy Rule Format:
            Contains(^IITR22, ' - ')

            Technical Business Rule Format:
            Contains(^IITR22, ' - ')
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.IITR22 == null ? false : report.IITR22.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000018",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name field must not have space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000018" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000019
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used in the individual name field
    
            Legacy Rule Format:
            (FoundSet(^IITR22, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IITR22, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IITR22, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000019" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000020()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000020
            A valid Amendment Sequence Number must be provided when Amendment indicator is set. Valid values are 1-9
    
            Legacy Rule Format:
            (^IITR16 = TRUE) AND ((^IITR56 < 1 OR ^IITR56 > 9))

            Technical Business Rule Format:
            (^IITR16 = TRUE) AND ((^IITR56 < 1 OR ^IITR56 > 9))
    
            Data Elements:
    
            ^IITR56 = RP:Report.AmendmentSequence.Number
    
            ^IITR16 = RP:Report.Amendment.Indicator
            */
            assertion = (report.IITR16.GetValueOrDefault() && (report.IITR56.GetValueOrDefault() < 1 || report.IITR56.GetValueOrDefault() > 9));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where the lodgment is an amendment, a valid sequence number of 1 to 9 must be provided",
                    LongDescription = @"A valid Amendment Sequence Number must be provided when Amendment indicator is set. Valid values are 1-9.",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentSequence.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000020" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR16", Value = GetValueOrEmpty(report.IITR16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR56", Value = report.IITR56.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000021
            Individual name field contains suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used in the individual name field
    
            Legacy Rule Format:
            (FoundSet(^IITR22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IITR22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IITR22, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name field contains ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR as a suffix",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000022
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR31 <> BLANK) AND (NotCharacterInSet(^IITR31, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR31 <> BLANK) AND (NotCharacterInSet(^IITR31, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR31 = RP:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR31) != true && !(IsMatch(report.IITR31, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Additional Information contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000022" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR31", Value = GetValueOrEmpty(report.IITR31) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000023
            Date of death is present and title is not Executor for (EF) or Representative for (RF) or Trustee for (TF). Only these titles are allowed for a deceased taxpayer
    
            Legacy Rule Format:
            ((^IITR28 <> NULL) AND (NotInSet(^IITR21, '"EF", "RF", "TF"')))

            Technical Business Rule Format:
            ((^IITR28 <> NULL) AND (NotInSet(^IITR21, '"EF", "RF", "TF"')))
    
            Data Elements:
    
            ^IITR28 = RP:PersonDemographicDetails.Death.Date
    
            ^IITR21 = RP:PersonNameDetails.Title.Text
            */
            assertion = (report.IITR28 != null && !(IsMatch(report.IITR21, @"^(EF|RF|TF)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check deceased taxpayer's title",
                    LongDescription = @"Date of death is present and title is not Executor for (EF) or Representative for (RF) or Trustee for (TF). Only these titles are allowed for a deceased taxpayer.",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Death.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR28", Value = GetValueOrEmptyDate(report.IITR28) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR21", Value = GetValueOrEmpty(report.IITR21) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000025()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000025
            Family name is mandatory
    
            Legacy Rule Format:
            ^IITR22 = BLANK

            Technical Business Rule Format:
            ^IITR22 = BLANK
    
            Data Elements:
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR22) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Surname required",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.FamilyName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000025" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000026
            Name changed indicator is mandatory
    
            Legacy Rule Format:
            ^IITR26 = NULL

            Technical Business Rule Format:
            ^IITR26 = NULL
    
            Data Elements:
    
            ^IITR26 = RP:Party.ChangeName.Indicator
            */
            assertion = (report.IITR26 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000026",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The ""Has name or title changed since last tax return lodged"" question must be answered",
                    Location = "/xbrli:xbrl/tns:Party.ChangeName.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000026" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR26", Value = GetValueOrEmpty(report.IITR26) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000027
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000027()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000027
            Date of birth is mandatory
    
            Legacy Rule Format:
            ^IITR29 = NULL

            Technical Business Rule Format:
            ^IITR29 = NULL
    
            Data Elements:
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
            */
            assertion = (report.IITR29 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of birth required",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Birth.Date[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000027" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmptyDate(report.IITR29) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000031
            Final tax return is mandatory
    
            Legacy Rule Format:
            ^IITR30 = NULL

            Technical Business Rule Format:
            ^IITR30 = NULL
    
            Data Elements:
    
            ^IITR30 = RP:Lodgment.FinalReturn.Indicator
            */
            assertion = (report.IITR30 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Final tax return must be present",
                    Location = "/xbrli:xbrl/tns:Lodgment.FinalReturn.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR30", Value = GetValueOrEmpty(report.IITR30) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000032
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000032()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000032
            If requesting an amendment, the Amendment indicator, Amendment type code, Amendment Sequence Number and Reason for amendment must be completed. If not lodging an amendment, the Amendment indicator, Amendment type code, Amendment Sequence Number and Reason for amendment are not required.
    
            Legacy Rule Format:
            (^IITR16 = TRUE AND (^IITR17 = BLANK OR ^IITR18 = BLANK OR ^IITR56 = NULL)) OR (^IITR16 <> TRUE AND (^IITR17 <> BLANK OR ^IITR18 <> BLANK OR ^IITR56 <> NULL))

            Technical Business Rule Format:
            (^IITR16 = TRUE AND (^IITR17 = BLANK OR ^IITR18 = BLANK OR ^IITR56 = NULL)) OR (^IITR16 <> TRUE AND (^IITR17 <> BLANK OR ^IITR18 <> BLANK OR ^IITR56 <> NULL))
    
            Data Elements:
    
            ^IITR16 = RP:Report.Amendment.Indicator
    
            ^IITR17 = RP:Report.AmendmentType.Code
    
            ^IITR18 = RP:Report.AmendmentReason.Text
    
            ^IITR56 = RP:Report.AmendmentSequence.Number
            */
            assertion = (report.IITR16.GetValueOrDefault() && (string.IsNullOrWhiteSpace(report.IITR17)  || string.IsNullOrWhiteSpace(report.IITR18)  || report.IITR56 == null) || report.IITR16 != true && (string.IsNullOrWhiteSpace(report.IITR17) != true || string.IsNullOrWhiteSpace(report.IITR18) != true || report.IITR56 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000032",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"When requesting an amendment, all amendment details must be provided",
                    LongDescription = @"When requesting an amendment to the Individual Income Tax Return, the Amendment type code, Amendment Sequence Number and Reason for amendment must be completed, with the Amendment indicator set to 'yes' (true). If not lodging an amendment, the Amendment indicator, Amendment type code, Amendment Sequence Number and Reason for amendment are not required.",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000032" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR16", Value = GetValueOrEmpty(report.IITR16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR17", Value = GetValueOrEmpty(report.IITR17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR18", Value = GetValueOrEmpty(report.IITR18) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR56", Value = report.IITR56.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000033
            Accounting period start date must commence on or after 1st January in the previous year
    
            Legacy Rule Format:
            ^RP:StartDate < ConvertToDate(1,1, (^IITR10 - 1))

            Technical Business Rule Format:
            ^RP:StartDate < ConvertToDate(1,1, (^IITR10 - 1))
    
            Data Elements:
    
            ^IITR10 = RP:Report.TargetFinancial.Year
    
            ^StartDate = StartDate
            */
            assertion = ((report.RPExists ) && (report.RPStartDate < ConvertToDate(1, 1, report.IITR10.GetValueOrDefault() - 1)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date financial year from date must commence on or after 1st January in the previous year",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000033" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR10", Value = GetValueOrEmpty(report.IITR10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000044
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000044()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000044
            When Family name is provided, First name must also be provided.
    
            Legacy Rule Format:
            (^IITR24 = BLANK) AND (^IITR22 <> BLANK)

            Technical Business Rule Format:
            (^IITR24 = BLANK) AND (^IITR22 <> BLANK)
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
    
            ^IITR22 = RP:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR24)  && string.IsNullOrWhiteSpace(report.IITR22) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000044",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check name is complete where First name field is blank",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000044" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR22", Value = report.IITR22 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000049
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000049()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000049
            Current postal address - Address line 1 has a maximum field length of 38
    
            Legacy Rule Format:
            Length(^IITR33) > 38

            Technical Business Rule Format:
            Length(^IITR33) > 38
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
            */
            assertion = (Length(report.IITR33) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000049",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 1 must not exceed 38 characters",
                    LongDescription = @"For Current postal address - Address line 1, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000049" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000050
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000050()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000050
            Current postal address - Address line 2 has a maximum field length of 38
    
            Legacy Rule Format:
            Length(^IITR34) > 38

            Technical Business Rule Format:
            Length(^IITR34) > 38
    
            Data Elements:
    
            ^IITR34 = RP.POS:AddressDetails.Line2.Text
            */
            assertion = (Length(report.IITR34) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000050",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 2 must not exceed 38 characters",
                    LongDescription = @"For Current postal address - Address line 2, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000050" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR34", Value = GetValueOrEmpty(report.IITR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000051()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000051
            Current postal address - Suburb/Town has a maximum field length of 27
    
            Legacy Rule Format:
            Length(^IITR35) > 27

            Technical Business Rule Format:
            Length(^IITR35) > 27
    
            Data Elements:
    
            ^IITR35 = RP.POS:AddressDetails.LocalityName.Text
            */
            assertion = (Length(report.IITR35) > 27);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000051",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Suburb/Town must not exceed 27 characters",
                    LongDescription = @"For Current postal address - Suburb/Town, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000051" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR35", Value = report.IITR35 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000052
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000052()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000052
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character
    
            Legacy Rule Format:
            (^IITR18 <> BLANK) AND (NotCharacterInSet(^IITR18, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR18 <> BLANK) AND (NotCharacterInSet(^IITR18, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR18 = RP:Report.AmendmentReason.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR18) != true && !(IsMatch(report.IITR18, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000052",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reason for amendment contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentReason.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000052" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR18", Value = GetValueOrEmpty(report.IITR18) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000054
            Either the taxpayer's Daytime phone number or Your mobile phone number must be provided.
    
            Legacy Rule Format:
            (^IITR579 = BLANK OR ^IITR50 = BLANK) AND (^IITR48 = BLANK)

            Technical Business Rule Format:
            (^IITR579 = BLANK OR ^IITR50 = BLANK) AND (^IITR48 = BLANK)
    
            Data Elements:
    
            ^IITR48 = RP:ElectronicContact.Telephone.Mobile.Number
    
            ^IITR50 = RP:ElectronicContact.Telephone.Minimal.Number
    
            ^IITR579 = RP:ElectronicContact.Telephone.Country.Code
            */
            assertion = ((string.IsNullOrWhiteSpace(report.IITR579)  || string.IsNullOrWhiteSpace(report.IITR50) ) && string.IsNullOrWhiteSpace(report.IITR48) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001054",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check taxpayer phone contact details",
                    LongDescription = @"Neither the taxpayer's daytime phone number or mobile phone number have been provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Mobile.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000054" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR579", Value = GetValueOrEmpty(report.IITR579) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR50", Value = GetValueOrEmpty(report.IITR50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR48", Value = GetValueOrEmpty(report.IITR48) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000056
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000056()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000056
            The dates provided should be the 1st of the month.
    
            Legacy Rule Format:
            (Day(^RP:StartDate) <> 1)

            Technical Business Rule Format:
            (Day(^RP:StartDate) <> 1)
    
            Data Elements:
            */
            assertion = (Day(report.RPStartDate) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000056",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The period start date must be the first of the month",
                    LongDescription = @"This is an annual accounting submission. The dates provided should be for your annual accounting period, commencing on the 1st day of the month. For normal balancers, this will be the period 1 July - 30 June.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000056" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000061
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000061()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000061
            Check that account name details are correct where numerics have been entered.
    
            Legacy Rule Format:
            ContainsSet(^IITR55, '0-9')

            Technical Business Rule Format:
            ContainsSet(^IITR55, '0-9')
    
            Data Elements:
    
            ^IITR55 = RP:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = IsMatch(report.IITR55, @"[0-9]", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000061",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check account name where field contains numerics",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000061" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR55", Value = GetValueOrEmpty(report.IITR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000071
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000071()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000071
            The number of days full levy exemption and/or the number of days half levy exemption exceed accounting period.
    
            Legacy Rule Format:
            (^IITR172 + ^IITR174) > (DaysInDuration(^RP:EndDate - ^RP:StartDate) + 1)

            Technical Business Rule Format:
            (^IITR172 + ^IITR174) > (DaysInDuration(^RP:EndDate - ^RP:StartDate) + 1)
    
            Data Elements:
    
            ^IITR172 = RP:IncomeTax.MedicareLevy.ExemptionFullDays.Count
    
            ^EndDate = EndDate
    
            ^IITR174 = RP:IncomeTax.MedicareLevy.ExemptionHalfDays.Count
    
            ^StartDate = StartDate
            */
            assertion = ((report.RPExists ) && (report.IITR172.GetValueOrDefault() + report.IITR174.GetValueOrDefault() > (report.RPEndDate.GetValueOrDefault() - report.RPStartDate.GetValueOrDefault()).Days + 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000071",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The combined days for full and half Medicare levy exemption must be within income year.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.ExemptionFullDays.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000071" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR172", Value = report.IITR172.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR174", Value = report.IITR174.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000080
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000080()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000080
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR33 <> BLANK) AND (NotCharacterInSet(^IITR33, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR33 <> BLANK) AND (NotCharacterInSet(^IITR33, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR33) != true && !(IsMatch(report.IITR33, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000080",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000080" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000081
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR34 <> BLANK) AND (NotCharacterInSet(^IITR34, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR34 <> BLANK) AND (NotCharacterInSet(^IITR34, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR34 = RP.POS:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR34) != true && !(IsMatch(report.IITR34, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000081",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000081" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR34", Value = GetValueOrEmpty(report.IITR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000100
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000100()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000100
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            Contains(^IITR33, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IITR33, 'UNKNOWN')
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
            */
            assertion = (report.IITR33 == null ? false : report.IITR33.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000101
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000101()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000101
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            Contains(^IITR34, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IITR34, 'UNKNOWN')
    
            Data Elements:
    
            ^IITR34 = RP.POS:AddressDetails.Line2.Text
            */
            assertion = (report.IITR34 == null ? false : report.IITR34.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000101",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000101" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR34", Value = GetValueOrEmpty(report.IITR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000102
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000102()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000102
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            Contains(^IITR35, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IITR35, 'UNKNOWN')
    
            Data Elements:
    
            ^IITR35 = RP.POS:AddressDetails.LocalityName.Text
            */
            assertion = (report.IITR35 == null ? false : report.IITR35.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000102",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000102" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR35", Value = report.IITR35 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000104
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000104()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000104
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            Contains(^IITR41, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IITR41, 'UNKNOWN')
    
            Data Elements:
    
            ^IITR41 = RP.RES:AddressDetails.Line1.Text
            */
            assertion = (report.IITR41 == null ? false : report.IITR41.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000104",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000104" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR41", Value = report.IITR41 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000105
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000105()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000105
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            Contains(^IITR42, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IITR42, 'UNKNOWN')
    
            Data Elements:
    
            ^IITR42 = RP.RES:AddressDetails.Line2.Text
            */
            assertion = (report.IITR42 == null ? false : report.IITR42.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000105" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR42", Value = GetValueOrEmpty(report.IITR42) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000106
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000106()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000106
            Unknown cannot be entered as an Address.  Tax Agents who do not know client's current postal address will need to contact Registry Division by fax or e-mail to make changes to their client's record.
    
            Legacy Rule Format:
            Contains(^IITR43, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IITR43, 'UNKNOWN')
    
            Data Elements:
    
            ^IITR43 = RP.RES:AddressDetails.LocalityName.Text
            */
            assertion = (report.IITR43 == null ? false : report.IITR43.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000106",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR43", Value = report.IITR43 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000108
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000108()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000108
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            ((^IITR38 = NULL) OR (^IITR38 = 'au')) AND (ContainsSet(^IITR35, '0-9'))

            Technical Business Rule Format:
            ((^IITR38 = NULL) OR (^IITR38 = 'au')) AND (ContainsSet(^IITR35, '0-9'))
    
            Data Elements:
    
            ^IITR35 = RP.POS:AddressDetails.LocalityName.Text
    
            ^IITR38 = RP.POS:AddressDetails.Country.Code
            */
            assertion = ((report.IITR38 == null || report.IITR38 == @"au") && IsMatch(report.IITR35, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000108",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address must not contain numeric characters when an Australian address is provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000108" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR38", Value = GetValueOrEmpty(report.IITR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR35", Value = report.IITR35 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000109
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000109()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000109
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            ((^IITR52 = NULL) OR (^IITR52 = 'au')) AND (ContainsSet(^IITR43, '0-9'))

            Technical Business Rule Format:
            ((^IITR52 = NULL) OR (^IITR52 = 'au')) AND (ContainsSet(^IITR43, '0-9'))
    
            Data Elements:
    
            ^IITR43 = RP.RES:AddressDetails.LocalityName.Text
    
            ^IITR52 = RP.RES:AddressDetails.Country.Code
            */
            assertion = ((report.IITR52 == null || report.IITR52 == @"au") && IsMatch(report.IITR43, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000109",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address must not contain numeric characters when an Australian address is provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000109" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR52", Value = GetValueOrEmpty(report.IITR52) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR43", Value = report.IITR43 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000110
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000110()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000110
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            ((^IITR38 = NULL) OR (^IITR38 = 'au')) AND (FoundSet(^IITR35, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            ((^IITR38 = NULL) OR (^IITR38 = 'au')) AND (FoundSet(^IITR35, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IITR35 = RP.POS:AddressDetails.LocalityName.Text
    
            ^IITR38 = RP.POS:AddressDetails.Country.Code
            */
            assertion = ((report.IITR38 == null || report.IITR38 == @"au") && IsMatch(report.IITR35, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000110",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000110" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR38", Value = GetValueOrEmpty(report.IITR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR35", Value = report.IITR35 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000111
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000111()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000111
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            ((^IITR52 = NULL) OR (^IITR52 = 'au')) AND (FoundSet(^IITR43, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            ((^IITR52 = NULL) OR (^IITR52 = 'au')) AND (FoundSet(^IITR43, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IITR43 = RP.RES:AddressDetails.LocalityName.Text
    
            ^IITR52 = RP.RES:AddressDetails.Country.Code
            */
            assertion = ((report.IITR52 == null || report.IITR52 == @"au") && IsMatch(report.IITR43, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000111",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000111" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR52", Value = GetValueOrEmpty(report.IITR52) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR43", Value = report.IITR43 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000112
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000112()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000112
            State, postcode, or country are missing from the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            (^IITR36 = NULL) AND (^IITR37 = NULL) AND ((^IITR38 = NULL) OR (^IITR38 = 'au'))

            Technical Business Rule Format:
            (^IITR36 = NULL) AND (^IITR37 = NULL) AND ((^IITR38 = NULL) OR (^IITR38 = 'au'))
    
            Data Elements:
    
            ^IITR36 = RP.POS:AddressDetails.StateOrTerritory.Code
    
            ^IITR37 = RP.POS:AddressDetails.Postcode.Text
    
            ^IITR38 = RP.POS:AddressDetails.Country.Code
            */
            assertion = (report.IITR36 == null && report.IITR37 == null && (report.IITR38 == null || report.IITR38 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000112",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address state and postcode are required for Australian addresses",
                    LongDescription = @"State, postcode, or country are required in the Current postal address field.  Determine whether the address is Australian or overseas and correct accordingly",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000112" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR36", Value = GetValueOrEmpty(report.IITR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR37", Value = GetValueOrEmpty(report.IITR37) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR38", Value = GetValueOrEmpty(report.IITR38) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000113
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000113()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000113
            State, postcode, or country are missing from the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            (^IITR44 = NULL) AND (^IITR45 = NULL) AND ((^IITR52 = NULL) OR (^IITR52 = 'au'))

            Technical Business Rule Format:
            (^IITR44 = NULL) AND (^IITR45 = NULL) AND ((^IITR52 = NULL) OR (^IITR52 = 'au'))
    
            Data Elements:
    
            ^IITR44 = RP.RES:AddressDetails.StateOrTerritory.Code
    
            ^IITR45 = RP.RES:AddressDetails.Postcode.Text
    
            ^IITR52 = RP.RES:AddressDetails.Country.Code
            */
            assertion = (report.IITR44 == null && report.IITR45 == null && (report.IITR52 == null || report.IITR52 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000113",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address state and postcode are required for Australian addresses",
                    LongDescription = @"State, postcode, or country are required in the Home postal address field.  Determine whether the address is Australian or overseas and correct accordingly",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000113" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR44", Value = GetValueOrEmpty(report.IITR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR45", Value = GetValueOrEmpty(report.IITR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR52", Value = GetValueOrEmpty(report.IITR52) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000114
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000114()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000114
            State is present in the address field without a postcode.
    
            Legacy Rule Format:
            (^IITR36 <> NULL) AND (^IITR37 = NULL)

            Technical Business Rule Format:
            (^IITR36 <> NULL) AND (^IITR37 = NULL)
    
            Data Elements:
    
            ^IITR36 = RP.POS:AddressDetails.StateOrTerritory.Code
    
            ^IITR37 = RP.POS:AddressDetails.Postcode.Text
            */
            assertion = (report.IITR36 != null && report.IITR37 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000114",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The postcode from the Current postal address must be present if State is present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000114" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR36", Value = GetValueOrEmpty(report.IITR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR37", Value = GetValueOrEmpty(report.IITR37) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000115
            Country name is present with State or postcode in the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            (^IITR38 <> NULL) AND (^IITR38 <> 'au') AND ((^IITR36 <> NULL) OR (^IITR37 <> NULL))

            Technical Business Rule Format:
            (^IITR38 <> NULL) AND (^IITR38 <> 'au') AND ((^IITR36 <> NULL) OR (^IITR37 <> NULL))
    
            Data Elements:
    
            ^IITR38 = RP.POS:AddressDetails.Country.Code
    
            ^IITR36 = RP.POS:AddressDetails.StateOrTerritory.Code
    
            ^IITR37 = RP.POS:AddressDetails.Postcode.Text
            */
            assertion = (report.IITR38 != null && report.IITR38 != @"au" && (report.IITR36 != null || report.IITR37 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000115",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australia must be entered as the country where an Australian state or territory has been entered.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000115" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR38", Value = GetValueOrEmpty(report.IITR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR36", Value = GetValueOrEmpty(report.IITR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR37", Value = GetValueOrEmpty(report.IITR37) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000116
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000116()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000116
            Country name is present with State or postcode in the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            ((^IITR44 <> NULL) OR (^IITR45 <> NULL)) AND ((^IITR52 <> NULL) AND (^IITR52 <> 'au'))

            Technical Business Rule Format:
            ((^IITR44 <> NULL) OR (^IITR45 <> NULL)) AND ((^IITR52 <> NULL) AND (^IITR52 <> 'au'))
    
            Data Elements:
    
            ^IITR52 = RP.RES:AddressDetails.Country.Code
    
            ^IITR44 = RP.RES:AddressDetails.StateOrTerritory.Code
    
            ^IITR45 = RP.RES:AddressDetails.Postcode.Text
            */
            assertion = ((report.IITR44 != null || report.IITR45 != null) && (report.IITR52 != null && report.IITR52 != @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000116",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and postcode must not be present for non-Australian addresses",
                    LongDescription = @"Country name is present with State or Postcode in the Home address field.  Determine whether the address is Australian or overseas and correct accordingly",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000116" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR44", Value = GetValueOrEmpty(report.IITR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR45", Value = GetValueOrEmpty(report.IITR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR52", Value = GetValueOrEmpty(report.IITR52) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000117
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000117()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000117
            Postcode is present in the address field without a State code.
    
            Legacy Rule Format:
            (^IITR37 <> BLANK) AND (^IITR36 = BLANK)

            Technical Business Rule Format:
            (^IITR37 <> BLANK) AND (^IITR36 = BLANK)
    
            Data Elements:
    
            ^IITR36 = RP.POS:AddressDetails.StateOrTerritory.Code
    
            ^IITR37 = RP.POS:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR37) != true && string.IsNullOrWhiteSpace(report.IITR36) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000117",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - State must be present if Postcode is present",
                    LongDescription = @"If the Current postal address is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000117" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR37", Value = GetValueOrEmpty(report.IITR37) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR36", Value = GetValueOrEmpty(report.IITR36) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000118
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000118()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000118
            Postcode is present in the address field without a State code.
    
            Legacy Rule Format:
            (^IITR45 <> BLANK) AND (^IITR44 = BLANK)

            Technical Business Rule Format:
            (^IITR45 <> BLANK) AND (^IITR44 = BLANK)
    
            Data Elements:
    
            ^IITR44 = RP.RES:AddressDetails.StateOrTerritory.Code
    
            ^IITR45 = RP.RES:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR45) != true && string.IsNullOrWhiteSpace(report.IITR44) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000118",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - State must be present if Postcode is present",
                    LongDescription = @"If the Home address is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000118" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR45", Value = GetValueOrEmpty(report.IITR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR44", Value = GetValueOrEmpty(report.IITR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000121
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000121()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000121
            Daytime phone area code is present without daytime phone number.
    
            Legacy Rule Format:
            (^IITR50 = BLANK) AND (^IITR579 <> BLANK)

            Technical Business Rule Format:
            (^IITR50 = BLANK) AND (^IITR579 <> BLANK)
    
            Data Elements:
    
            ^IITR579 = RP:ElectronicContact.Telephone.Country.Code
    
            ^IITR50 = RP:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR50)  && string.IsNullOrWhiteSpace(report.IITR579) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000121",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"Daytime phone number is required when daytime phone area code is provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000121" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR50", Value = GetValueOrEmpty(report.IITR50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR579", Value = GetValueOrEmpty(report.IITR579) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000122
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000122()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000122
            State is present in the address field without a postcode.
    
            Legacy Rule Format:
            (^IITR44 <> NULL) AND (^IITR45 = NULL)

            Technical Business Rule Format:
            (^IITR44 <> NULL) AND (^IITR45 = NULL)
    
            Data Elements:
    
            ^IITR44 = RP.RES:AddressDetails.StateOrTerritory.Code
    
            ^IITR45 = RP.RES:AddressDetails.Postcode.Text
            */
            assertion = (report.IITR44 != null && report.IITR45 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000122",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The postcode from the Home address must be present if State is present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000122" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR44", Value = GetValueOrEmpty(report.IITR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR45", Value = GetValueOrEmpty(report.IITR45) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000123
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000123()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000123
            Information has been provided at BSB number, or account number, or account name, however, not all of the information is present.  Check BSB number, account number, and account name and input the missing information.
    
            Legacy Rule Format:
            (^IITR53 <> NULL OR ^IITR54 <> BLANK OR ^IITR55 <> BLANK) AND (^IITR53 = NULL OR ^IITR54 = BLANK OR ^IITR55 = BLANK)

            Technical Business Rule Format:
            (^IITR53 <> NULL OR ^IITR54 <> BLANK OR ^IITR55 <> BLANK) AND (^IITR53 = NULL OR ^IITR54 = BLANK OR ^IITR55 = BLANK)
    
            Data Elements:
    
            ^IITR53 = RP:FinancialInstitutionAccount.BankStateBranch.Number
    
            ^IITR54 = RP:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
    
            ^IITR55 = RP:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = ((report.IITR53 != null || string.IsNullOrWhiteSpace(report.IITR54) != true || string.IsNullOrWhiteSpace(report.IITR55) != true) && (report.IITR53 == null || string.IsNullOrWhiteSpace(report.IITR54)  || string.IsNullOrWhiteSpace(report.IITR55) ));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000123",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where either BSB number, account number or account name is provided, all must be present",
                    LongDescription = @"Financial institution details are needed for the Tax office to pay any refund owing the taxpayer. All details must be provided, even if provided in the past.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.BankStateBranch.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000123" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR53", Value = GetValueOrEmpty(report.IITR53) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR54", Value = GetValueOrEmpty(report.IITR54) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR55", Value = GetValueOrEmpty(report.IITR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000125
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000125()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000125
            The email address starts with an @ symbol, or ends with an @ symbol, or contains more than one @ symbol. Email addresses must be in the format 'name@domain.sub-domain'.
    
            Legacy Rule Format:
            ((StartsWith(^IITR51, '@')) OR (EndsWith(^IITR51, '@')) OR (ContainsMultiple(^IITR51, '@')))

            Technical Business Rule Format:
            ((StartsWith(^IITR51, '@')) OR (EndsWith(^IITR51, '@')) OR (ContainsMultiple(^IITR51, '@')))
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (IsMatch(report.IITR51, @"^(@)", RegexOptions.IgnoreCase) || IsMatch(report.IITR51, @"(@)$", RegexOptions.IgnoreCase) || IsMatch(report.IITR51, @"@(.*)@", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000125",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid @ symbol in email address",
                    LongDescription = @"The email address starts with an @ symbol, or ends with an @ symbol, or contains more than one @ symbol. Email addresses must be in the format 'name@domain.sub-domain'",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000125" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000128
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000128()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000128
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            (^IITR37 <> BLANK) AND (^IITR37 < 0200 OR ^IITR37 > 9999)

            Technical Business Rule Format:
            (^IITR37 <> BLANK) AND (^IITR37 < 0200 OR ^IITR37 > 9999)
    
            Data Elements:
    
            ^IITR37 = RP.POS:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR37) != true && (AsNumeric(report.IITR37) < 200 || AsNumeric(report.IITR37) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000128",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, Current postal address - Postcode is invalid. Valid range is 0200 - 9999.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000128" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR37", Value = GetValueOrEmpty(report.IITR37) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000129
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000129()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000129
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            (^IITR45 <> BLANK) AND (^IITR45 < 0200 OR ^IITR45 > 9999)

            Technical Business Rule Format:
            (^IITR45 <> BLANK) AND (^IITR45 < 0200 OR ^IITR45 > 9999)
    
            Data Elements:
    
            ^IITR45 = RP.RES:AddressDetails.Postcode.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR45) != true && (AsNumeric(report.IITR45) < 200 || AsNumeric(report.IITR45) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000129",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, Home address - Postcode is invalid. Valid range is 0200 - 9999.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000129" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR45", Value = GetValueOrEmpty(report.IITR45) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000132()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000132
            The Bank/State/Branch number is present and it is either less than six characters in length or the number is less than 012000 or greater than 999999
    
            Legacy Rule Format:
            ^IITR53 <> BLANK AND (^IITR53 < 012000 OR ^IITR53 > 999999)

            Technical Business Rule Format:
            ^IITR53 <> BLANK AND (^IITR53 < 012000 OR ^IITR53 > 999999)
    
            Data Elements:
    
            ^IITR53 = RP:FinancialInstitutionAccount.BankStateBranch.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR53) != true && (AsNumeric(report.IITR53) < 12000 || AsNumeric(report.IITR53) > 999999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank/State/Branch number invalid. Valid range is 012000 - 999999.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.BankStateBranch.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR53", Value = GetValueOrEmpty(report.IITR53) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000133()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000133
            The email address starts with a dot, or ends with a dot, or contains adjacent dots. Email addresses must be in the format 'name@domain.sub-domain'.
    
            Legacy Rule Format:
            (Contains(^IITR51, '..')  OR (StartsWith(^IITR51, '.')) OR (EndsWith(^IITR51,'.')))

            Technical Business Rule Format:
            (Contains(^IITR51, '..')  OR (StartsWith(^IITR51, '.')) OR (EndsWith(^IITR51,'.')))
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = ((report.IITR51 == null ? false : report.IITR51.ToUpperInvariant().Contains(@"..")) || IsMatch(report.IITR51, @"^(\.)", RegexOptions.IgnoreCase) || IsMatch(report.IITR51, @"(\.)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid dot in email address",
                    LongDescription = @"The email address starts with a dot, or ends with a dot, or contains adjacent dots. Email addresses must be in the format 'name@domain.sub-domain'",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000133" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000134()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000134
            The email address field has a dot character adjacent to an @ symbol. Email addresses must be in the format 'name@domain.sub-domain'.
    
            Legacy Rule Format:
            (Contains(^IITR51, '.@') OR Contains(^IITR51,'@.'))

            Technical Business Rule Format:
            (Contains(^IITR51, '.@') OR Contains(^IITR51,'@.'))
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = ((report.IITR51 == null ? false : report.IITR51.ToUpperInvariant().Contains(@".@")) || (report.IITR51 == null ? false : report.IITR51.ToUpperInvariant().Contains(@"@.")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000134",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid dot and @ symbol in email address",
                    LongDescription = @"The email address field has a dot character adjacent to an @ symbol. Email addresses must be in the format 'name@domain.sub-domain'",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000134" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000135()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000135
            Reason for amendment has a maximum field length of 3000
    
            Legacy Rule Format:
            Length(^IITR18) > 3000

            Technical Business Rule Format:
            Length(^IITR18) > 3000
    
            Data Elements:
    
            ^IITR18 = RP:Report.AmendmentReason.Text
            */
            assertion = (Length(report.IITR18) > 3000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reason for amendment must not exceed 3000 characters",
                    LongDescription = @"For Reason for amendment, the maximum allowable characters for this form is 3000",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentReason.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR18", Value = GetValueOrEmpty(report.IITR18) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000137()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000137
            Daytime phone number is present without daytime phone area code.
    
            Legacy Rule Format:
            (^IITR50 <> BLANK) AND (^IITR579 = BLANK)

            Technical Business Rule Format:
            (^IITR50 <> BLANK) AND (^IITR579 = BLANK)
    
            Data Elements:
    
            ^IITR579 = RP:ElectronicContact.Telephone.Country.Code
    
            ^IITR50 = RP:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR50) != true && string.IsNullOrWhiteSpace(report.IITR579) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"Daytime phone area code is required when a Daytime phone number is provided",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000137" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR50", Value = GetValueOrEmpty(report.IITR50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR579", Value = GetValueOrEmpty(report.IITR579) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000139
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000139()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000139
            Your Australian mobile phone number should start with 04 and followed by 8 numeric characters.
    
            Legacy Rule Format:
            ((^IITR48 <> BLANK) AND NOT (StartsWith(^IITR48, '04')))

            Technical Business Rule Format:
            ((^IITR48 <> BLANK) AND NOT (StartsWith(^IITR48, '04')))
    
            Data Elements:
    
            ^IITR48 = RP:ElectronicContact.Telephone.Mobile.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR48) != true && !(IsMatch(report.IITR48, @"^(04)", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000139",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your mobile phone number is incorrect",
                    LongDescription = @"Your Australian mobile phone number should start with 04 and followed by 8 numeric characters",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Mobile.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000139" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR48", Value = GetValueOrEmpty(report.IITR48) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000140
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000140()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000140
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            (^IITR36 <> BLANK) AND (NotInSet(^IITR36, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^IITR36 <> BLANK) AND (NotInSet(^IITR36, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IITR36 = RP.POS:AddressDetails.StateOrTerritory.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR36) != true && !(IsMatch(report.IITR36, @"^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000140",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid State code",
                    LongDescription = @"The State code in the Current postal address is not Valid (Valid States are ""QLD"",""NSW"",""VIC"",""SA"",""WA"",""NT"",""ACT"",""TAS"")",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000140" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR36", Value = GetValueOrEmpty(report.IITR36) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000141()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000141
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            (^IITR44 <> BLANK) AND (NotInSet(^IITR44, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^IITR44 <> BLANK) AND (NotInSet(^IITR44, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IITR44 = RP.RES:AddressDetails.StateOrTerritory.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR44) != true && !(IsMatch(report.IITR44, @"^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000141",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid State code",
                    LongDescription = @"The State code in the Home address is not Valid (Valid States are ""QLD"",""NSW"",""VIC"",""SA"",""WA"",""NT"",""ACT"",""TAS"")",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR44", Value = GetValueOrEmpty(report.IITR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000145()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000145
            Home address - Address line 1 has a maximum field length of 38
    
            Legacy Rule Format:
            Length(^IITR41) > 38

            Technical Business Rule Format:
            Length(^IITR41) > 38
    
            Data Elements:
    
            ^IITR41 = RP.RES:AddressDetails.Line1.Text
            */
            assertion = (Length(report.IITR41) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 1 must not exceed 38 characters",
                    LongDescription = @"For Home address - Address line 1, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000145" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR41", Value = report.IITR41 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000146
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000146()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000146
            Home address - Address line 2 has a maximum field length of 38
    
            Legacy Rule Format:
            Length(^IITR42) > 38

            Technical Business Rule Format:
            Length(^IITR42) > 38
    
            Data Elements:
    
            ^IITR42 = RP.RES:AddressDetails.Line2.Text
            */
            assertion = (Length(report.IITR42) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000146",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 2 must not exceed 38 characters",
                    LongDescription = @"For Home address - Address line 2, the maximum allowable characters for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000146" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR42", Value = GetValueOrEmpty(report.IITR42) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000147
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000147()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000147
            Home address - Suburb/Town has a maximum field length of 27
    
            Legacy Rule Format:
            Length(^IITR43) > 27

            Technical Business Rule Format:
            Length(^IITR43) > 27
    
            Data Elements:
    
            ^IITR43 = RP.RES:AddressDetails.LocalityName.Text
            */
            assertion = (Length(report.IITR43) > 27);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Suburb/Town must not exceed 27 characters",
                    LongDescription = @"For Home address - Suburb/Town, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000147" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR43", Value = report.IITR43 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000151()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000151
            Your mobile phone number has a maximum field length of 10
    
            Legacy Rule Format:
            Length(^IITR48) > 10

            Technical Business Rule Format:
            Length(^IITR48) > 10
    
            Data Elements:
    
            ^IITR48 = RP:ElectronicContact.Telephone.Mobile.Number
            */
            assertion = (Length(report.IITR48) > 10);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000151",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your mobile phone number must not exceed 10 characters",
                    LongDescription = @"For Your mobile phone number, the maximum allowable characters for this form is 10",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Mobile.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR48", Value = GetValueOrEmpty(report.IITR48) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000153()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000153
            Daytime phone number has a maximum field length of 15
    
            Legacy Rule Format:
            Length(^IITR50) > 15

            Technical Business Rule Format:
            Length(^IITR50) > 15
    
            Data Elements:
    
            ^IITR50 = RP:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.IITR50) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Daytime phone number must not exceed 15 characters",
                    LongDescription = @"For Daytime phone number, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR50", Value = GetValueOrEmpty(report.IITR50) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000154
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000154()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000154
            Contact E-mail address has a maximum field length of 76
    
            Legacy Rule Format:
            Length(^IITR51) > 76

            Technical Business Rule Format:
            Length(^IITR51) > 76
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (Length(report.IITR51) > 76);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000154",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact E-mail address must not exceed 76 characters",
                    LongDescription = @"For Contact E-mail address, the maximum allowable characters for this form is 76",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000154" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000156
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000156()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000156
            Account number has a maximum field length of 9
    
            Legacy Rule Format:
            Length(^IITR54) > 9

            Technical Business Rule Format:
            Length(^IITR54) > 9
    
            Data Elements:
    
            ^IITR54 = RP:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
            */
            assertion = (Length(report.IITR54) > 9);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000156",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account number must not exceed 9 characters",
                    LongDescription = @"For Account number, the maximum allowable characters for this form is 9",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.FinancialInstitutionAccount.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000156" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR54", Value = GetValueOrEmpty(report.IITR54) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000159
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000159()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000159
            Spouse's Customer reference number has a maximum field length of 10
    
            Legacy Rule Format:
            Length(^IITR533) > 10

            Technical Business Rule Format:
            Length(^IITR533) > 10
    
            Data Elements:
    
            ^IITR533 = RP.SPOUSE:Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier
            */
            assertion = (Length(report.IITR533) > 10);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000159",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Spouse's Customer reference number must not exceed 10 characters",
                    LongDescription = @"For Spouse's Customer reference number, the maximum allowable characters for this form is 10",
                    Location = "/xbrli:xbrl/tns:Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000159" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR533", Value = GetValueOrEmpty(report.IITR533) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000160
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000160()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000160
            Client's reference has a maximum field length of 12
    
            Legacy Rule Format:
            Length(^IITR536) > 12

            Technical Business Rule Format:
            Length(^IITR536) > 12
    
            Data Elements:
    
            ^IITR536 = INT.TrueAndCorrect:Identifiers.TaxAgentClientReference.Text
            */
            assertion = (Length(report.IITR536) > 12);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001160",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Client's reference must not exceed 12 characters",
                    LongDescription = @"For Client's reference, the maximum allowable characters for this form is 12",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentClientReference.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000160" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR536", Value = GetValueOrEmpty(report.IITR536) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000162
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000162()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000162
            Home address - Address line 1 is mandatory
    
            Legacy Rule Format:
            ^IITR41 = BLANK

            Technical Business Rule Format:
            ^IITR41 = BLANK
    
            Data Elements:
    
            ^IITR41 = RP.RES:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR41) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000162",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 1 must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000162" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR41", Value = report.IITR41 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000163
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000163()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000163
            Home address - Suburb/Town is mandatory
    
            Legacy Rule Format:
            ^IITR43 = BLANK

            Technical Business Rule Format:
            ^IITR43 = BLANK
    
            Data Elements:
    
            ^IITR43 = RP.RES:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR43) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000163",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Suburb/Town must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000163" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR43", Value = report.IITR43 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000164
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000164()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000164
            Numeric field, values nnn where n is a digit 0 to 9 only to be used.
    
            Legacy Rule Format:
            (^IITR48 <> BLANK) AND (NotCharacterInSet(^IITR48, '0-9'))

            Technical Business Rule Format:
            (^IITR48 <> BLANK) AND (NotCharacterInSet(^IITR48, '0-9'))
    
            Data Elements:
    
            ^IITR48 = RP:ElectronicContact.Telephone.Mobile.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR48) != true && !(IsMatch(report.IITR48, @"^[0-9]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000164",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your mobile phone number must be numeric",
                    LongDescription = @"Your mobile number must be a numeric field, values nnn where n is a digit 0 to 9 only to be used.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Mobile.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000164" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR48", Value = GetValueOrEmpty(report.IITR48) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000168
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000168()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000168
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR41 <> BLANK) AND (NotCharacterInSet(^IITR41, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR41 <> BLANK) AND (NotCharacterInSet(^IITR41, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR41 = RP.RES:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR41) != true && !(IsMatch(report.IITR41, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000168",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 1 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000168" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR41", Value = report.IITR41 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000169
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000169()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000169
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR42 <> BLANK) AND (NotCharacterInSet(^IITR42, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR42 <> BLANK) AND (NotCharacterInSet(^IITR42, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR42 = RP.RES:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR42) != true && !(IsMatch(report.IITR42, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000169",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Address line 2 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000169" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR42", Value = GetValueOrEmpty(report.IITR42) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000170
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000170()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000170
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR43 <> BLANK) AND (NotCharacterInSet(^IITR43, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR43 <> BLANK) AND (NotCharacterInSet(^IITR43, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR43 = RP.RES:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR43) != true && !(IsMatch(report.IITR43, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000170",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Home address - Suburb/Town contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPRESId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000170" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR43", Value = report.IITR43 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000178
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000178()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000178
            A text character must be one of the following: A to Z a to z 0 to 9 & * ( ) - ' , . / or a space character.
    
            Legacy Rule Format:
            (^IITR55 <> BLANK) AND (NotCharacterInSet(^IITR55, '"a-z","A-Z","0-9","&","*","(",")","-","'",",",".","/"," "'))

            Technical Business Rule Format:
            (^IITR55 <> BLANK) AND (NotCharacterInSet(^IITR55, '"a-z","A-Z","0-9","&","*","(",")","-","'",",",".","/"," "'))
    
            Data Elements:
    
            ^IITR55 = RP:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR55) != true && !(IsMatch(report.IITR55, @"^[a-zA-Z0-9&\*\(\)\-',\./ ]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000178",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 & * ( ) - ' , . / or a space character.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000178" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR55", Value = GetValueOrEmpty(report.IITR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000185
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000185()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000185
            The account name is preceded by a space, or contains only a space. Remove all leading spaces.
    
            Legacy Rule Format:
            StartsWith(^IITR55, ' ')

            Technical Business Rule Format:
            StartsWith(^IITR55, ' ')
    
            Data Elements:
    
            ^IITR55 = RP:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = IsMatch(report.IITR55, @"^( )", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000185",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account name contains a leading space",
                    LongDescription = @"The account name is preceded by a space, or contains only a space. Remove all leading spaces",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000185" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR55", Value = GetValueOrEmpty(report.IITR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000187
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000187()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000187
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR35 <> BLANK) AND (NotCharacterInSet(^IITR35, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR35 <> BLANK) AND (NotCharacterInSet(^IITR35, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR35 = RP.POS:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR35) != true && !(IsMatch(report.IITR35, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000187",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000187" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR35", Value = report.IITR35 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000188
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000188()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000188
            Additional Information has a maximum field length of 3000
    
            Legacy Rule Format:
            Length(^IITR31) > 3000

            Technical Business Rule Format:
            Length(^IITR31) > 3000
    
            Data Elements:
    
            ^IITR31 = RP:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
            */
            assertion = (Length(report.IITR31) > 3000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000188",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Additional Information must not exceed 3000 characters",
                    LongDescription = @"Additional Information, the maximum allowable characters for this form is 3000",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000188" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR31", Value = GetValueOrEmpty(report.IITR31) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000190
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000190()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000190
            Declaration Signatory identifier text must be present
    
            Legacy Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR545 = BLANK)

            Technical Business Rule Format:
            (^INT.TrueAndCorrect:Context <> NULL) AND (^IITR545 = BLANK)
    
            Data Elements:
    
            ^IITR545 = INT.TrueAndCorrect:Declaration.SignatoryIdentifier.Text
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.IITR545) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000190",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signatory identifier text must be present",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000190" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR545", Value = report.IITR545 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000198
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000198()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000198
            Locality name is mandatory
    
            Legacy Rule Format:
            ^IITR35 = BLANK

            Technical Business Rule Format:
            ^IITR35 = BLANK
    
            Data Elements:
    
            ^IITR35 = RP.POS:AddressDetails.LocalityName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR35) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000198",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Locality name must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000198" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR35", Value = report.IITR35 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000199
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000199()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000199
            Current postal address - Address line 1 is mandatory
    
            Legacy Rule Format:
            ^IITR33 = BLANK

            Technical Business Rule Format:
            ^IITR33 = BLANK
    
            Data Elements:
    
            ^IITR33 = RP.POS:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR33) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000199",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address - Address line 1 must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + report.RPPOSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000199" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR33", Value = report.IITR33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000200()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000200
            TFN amounts deducted on interest is greater than 50% of gross interest.
    
            Legacy Rule Format:
            (^IITR116 > 0) AND (((^IITR115 *0.5) + 1) < ^IITR116)

            Technical Business Rule Format:
            (^IITR116 > 0) AND (((^IITR115 *0.5) + 1) < ^IITR116)
    
            Data Elements:
    
            ^IITR116 = RP:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^IITR115 = RP:Income.Interest.Gross.Amount
            */
            assertion = (report.IITR116.GetValueOrDefault() > 0 && report.IITR115.GetValueOrDefault() * 0.5M + 1 < report.IITR116.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001200",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from gross interest is incorrect",
                    LongDescription = @"TFN amounts deducted on interest is greater than 50% of gross interest.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR116", Value = GetValueOrEmpty(report.IITR116) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR115", Value = GetValueOrEmpty(report.IITR115) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000201()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000201
            TFN amounts withheld from dividends is greater than 50% of the unfranked amount.
    
            Legacy Rule Format:
            (^IITR121 > 0) AND (((^IITR118 *0.5) + 1) < ^IITR121)

            Technical Business Rule Format:
            (^IITR121 > 0) AND (((^IITR118 *0.5) + 1) < ^IITR121)
    
            Data Elements:
    
            ^IITR121 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
    
            ^IITR118 = RP:Income.DividendsUnfranked.Amount
            */
            assertion = (report.IITR121.GetValueOrDefault() > 0 && report.IITR118.GetValueOrDefault() * 0.5M + 1 < report.IITR121.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000201",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must be less than 50% of the unfranked amount.",
                    LongDescription = @"If no TFN was provided, the amounts withheld from dividends cannot be greater than 50% of the unfranked amount.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR121", Value = GetValueOrEmpty(report.IITR121) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = GetValueOrEmpty(report.IITR118) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000202
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000202()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000202
            Tax withheld from attributed personal services income is greater than 50% of Attributed personal services income
    
            Legacy Rule Format:
            (^IITR111 > 0) AND (^IITR111 > (^IITR112 * 0.5))

            Technical Business Rule Format:
            (^IITR111 > 0) AND (^IITR111 > (^IITR112 * 0.5))
    
            Data Elements:
    
            ^IITR111 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount
    
            ^IITR112 = RP:Income.PersonalServicesIncome.Total.Amount
            */
            assertion = (report.IITR111.GetValueOrDefault() > 0 && report.IITR111.GetValueOrDefault() > report.IITR112.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000202",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld from attributed PSI incorrect",
                    LongDescription = @"Tax withheld from attributed personal services income is greater than 50% of Attributed personal services income",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000202" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR111", Value = GetValueOrEmpty(report.IITR111) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR112", Value = GetValueOrEmpty(report.IITR112) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000203()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000203
            Other deductions relating to amounts shown type has been completed without an amount at Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts.  When the deduction amount includes any deferred non-commercial business losses from a prior year, the code must be given to describe the business loss.
    
            Legacy Rule Format:
            (^IITR246 = NULL OR ^IITR246 = 0) AND (^IITR247 <> NULL)

            Technical Business Rule Format:
            (^IITR246 = NULL OR ^IITR246 = 0) AND (^IITR247 <> NULL)
    
            Data Elements:
    
            ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
    
            ^IITR247 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Code
            */
            assertion = ((report.IITR246 == null || report.IITR246 == 0) && report.IITR247 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other deductions relating to amounts shown type is invalid",
                    LongDescription = @"Other deductions relating to amounts shown type has been completed without an amount at Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts.  When the deduction amount includes any deferred non-commercial business losses from a prior year, the code must be given to describe the business loss.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR246", Value = GetValueOrEmpty(report.IITR246) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR247", Value = GetValueOrEmpty(report.IITR247) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000204
            Tax withheld for personal services cannot be present without personal services income present.
    
            Legacy Rule Format:
            (^IITR111 > 0) AND (^IITR112 = NULL OR ^IITR112 = 0)

            Technical Business Rule Format:
            (^IITR111 > 0) AND (^IITR112 = NULL OR ^IITR112 = 0)
    
            Data Elements:
    
            ^IITR112 = RP:Income.PersonalServicesIncome.Total.Amount
    
            ^IITR111 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount
            */
            assertion = (report.IITR111.GetValueOrDefault() > 0 && (report.IITR112 == null || report.IITR112 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000204",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed personal services income is incorrect",
                    LongDescription = @"When an amount above zero is provided for 'Tax withheld from attributed personal services income', an amount above zero for 'Attributed personal services income' is required",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncome.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR111", Value = GetValueOrEmpty(report.IITR111) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR112", Value = GetValueOrEmpty(report.IITR112) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000207
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000207()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000207
            Where Discount from taxed upfront schemes - eligible for reduction is greater than $1,000 the Total Assessable discount amount must be present.
    
            Legacy Rule Format:
            (^IITR123 > 1000) AND (^IITR127 = NULL OR ^IITR127 = 0)

            Technical Business Rule Format:
            (^IITR123 > 1000) AND (^IITR127 = NULL OR ^IITR127 = 0)
    
            Data Elements:
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
            */
            assertion = (report.IITR123.GetValueOrDefault() > 1000 && (report.IITR127 == null || report.IITR127 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000207",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Assessable discount amount is required",
                    LongDescription = @"Where Discount from taxed upfront schemes - eligible for reduction is greater than $1,000 the Total Assessable discount amount must be present",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DiscountAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000207" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = GetValueOrEmpty(report.IITR123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = GetValueOrEmpty(report.IITR127) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000208()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000208
            Where a positive Total Assessable discount amount is present, at least one positive amount must be provided at Discount from taxed upfront schemes - eligible for reduction, Discount from taxed upfront schemes - not eligible for reduction, or Discount from deferral schemes
    
            Legacy Rule Format:
            (^IITR127 > 0) AND (^IITR123 = NULL OR ^IITR123 = 0) AND (^IITR124 = NULL OR ^IITR124 = 0) AND (^IITR125 = NULL OR ^IITR125 = 0)

            Technical Business Rule Format:
            (^IITR127 > 0) AND (^IITR123 = NULL OR ^IITR123 = 0) AND (^IITR124 = NULL OR ^IITR124 = 0) AND (^IITR125 = NULL OR ^IITR125 = 0)
    
            Data Elements:
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
            */
            assertion = (report.IITR127.GetValueOrDefault() > 0 && (report.IITR123 == null || report.IITR123 == 0) && (report.IITR124 == null || report.IITR124 == 0) && (report.IITR125 == null || report.IITR125 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A Discount amount is required",
                    LongDescription = @"Where a positive Total Assessable discount amount is present, at least one positive amount must be provided at Discount from taxed upfront schemes - eligible for reduction, Discount from taxed upfront schemes - not eligible for reduction, or Discount from deferral schemes",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DiscountAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = GetValueOrEmpty(report.IITR127) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = GetValueOrEmpty(report.IITR123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = GetValueOrEmpty(report.IITR124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = GetValueOrEmpty(report.IITR125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000209
            Where there is no value present for Discount from taxed upfront schemes - eligible for reduction, the Total Assessable discount amount must equal the sum of Discount from taxed upfront schemes - not eligible for reduction and Discount from deferral schemes
    
            Legacy Rule Format:
            (^IITR123 = NULL OR ^IITR123 = 0) AND ((^IITR127 <> NULL AND (^IITR124 + ^IITR125) <> ^IITR127) OR (^IITR127 = NULL AND (^IITR124 + ^IITR125) <> 0))

            Technical Business Rule Format:
            (^IITR123 = NULL OR ^IITR123 = 0) AND ((^IITR127 <> NULL AND (^IITR124 + ^IITR125) <> ^IITR127) OR (^IITR127 = NULL AND (^IITR124 + ^IITR125) <> 0))
    
            Data Elements:
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
            */
            assertion = ((report.IITR123 == null || report.IITR123 == 0) && (report.IITR127 != null && report.IITR124.GetValueOrDefault() + report.IITR125.GetValueOrDefault() != report.IITR127.GetValueOrDefault() || report.IITR127 == null && report.IITR124.GetValueOrDefault() + report.IITR125.GetValueOrDefault() != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001209",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Assessable discount amount is incorrect",
                    LongDescription = @"Where there is no value present for Discount from taxed upfront schemes - eligible for reduction, the Total Assessable discount amount must equal the sum of Discount from taxed upfront schemes - not eligible for reduction and Discount from deferral schemes",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DiscountAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000209" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = GetValueOrEmpty(report.IITR123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = GetValueOrEmpty(report.IITR127) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = GetValueOrEmpty(report.IITR124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = GetValueOrEmpty(report.IITR125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000210
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000210()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000210
            Foreign source discounts cannot be greater than the Total Assessable discount amount
    
            Legacy Rule Format:
            (^IITR127 <> NULL AND ^IITR129 > ^IITR127) OR (^IITR127 = NULL AND ^IITR129 > 0)

            Technical Business Rule Format:
            (^IITR127 <> NULL AND ^IITR129 > ^IITR127) OR (^IITR127 = NULL AND ^IITR129 > 0)
    
            Data Elements:
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
    
            ^IITR129 = RP:Income.EmployeeShareScheme.ForeignSourceDiscount.Amount
            */
            assertion = (report.IITR127 != null && report.IITR129.GetValueOrDefault() > report.IITR127.GetValueOrDefault() || report.IITR127 == null && report.IITR129.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000210",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Employee share scheme foreign source discounts must be less than total assessable discount amount.",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DiscountAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000210" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = GetValueOrEmpty(report.IITR127) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR129", Value = GetValueOrEmpty(report.IITR129) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000211()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000211
            Either Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts or Landcare operations and deduction for decline in value of water facility is present and an amount of Distribution from partnerships or Share of net income from trusts is required
    
            Legacy Rule Format:
            (^IITR246 > 0 OR ^IITR245 > 0) AND (^IITR242 = NULL AND ^IITR243 = NULL)

            Technical Business Rule Format:
            (^IITR246 > 0 OR ^IITR245 > 0) AND (^IITR242 = NULL AND ^IITR243 = NULL)
    
            Data Elements:
    
            ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
            ^IITR245 = RP.Partnership:Expense.LandcareOperationsAcceleratedDepreciation.Amount
    
            ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            */
            assertion = ((report.IITR246.GetValueOrDefault() > 0 || report.IITR245.GetValueOrDefault() > 0) && (report.IITR242 == null && report.IITR243 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from partnerships is invalid",
                    LongDescription = @"When an amount above zero for either 'Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts' or 'Landcare operations and deduction for decline in value of water facility' is provided, at least one amount for 'Distribution from partnerships' or 'Share of net income from trusts' must be provided",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000211" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR246", Value = GetValueOrEmpty(report.IITR246) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR245", Value = GetValueOrEmpty(report.IITR245) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR242", Value = GetValueOrEmpty(report.IITR242) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000214
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000214()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000214
            Addition of Net income or loss from carrying on a business of investing plus Net income or loss from carrying on a rental property business plus Other income or loss relating to Other income or loss relating to Net income or loss from business is not equal to the amount of Net non-Primary Production income/loss.
    
            Legacy Rule Format:
            ((^IITR285 <> NULL) AND (^IITR285 <> (^IITR282 + ^IITR283 + ^IITR284))) OR ((^IITR285 = NULL) AND ((^IITR282 + ^IITR283 + ^IITR284) <> 0))

            Technical Business Rule Format:
            ((^IITR285 <> NULL) AND (^IITR285 <> (^IITR282 + ^IITR283 + ^IITR284))) OR ((^IITR285 = NULL) AND ((^IITR282 + ^IITR283 + ^IITR284) <> 0))
    
            Data Elements:
    
            ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
    
            ^IITR282 = RP:Income.InvestmentNonPrimaryProductionNet.Amount
    
            ^IITR283 = RP:Income.RentalPropertyNonPrimaryProductionNet.Amount
    
            ^IITR284 = RP:Income.NonPrimaryProductionOtherNet.Amount
            */
            assertion = (report.IITR285 != null && report.IITR285.GetValueOrDefault() != report.IITR282.GetValueOrDefault() + report.IITR283.GetValueOrDefault() + report.IITR284.GetValueOrDefault() || report.IITR285 == null && report.IITR282.GetValueOrDefault() + report.IITR283.GetValueOrDefault() + report.IITR284.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000214",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-Primary Production income/loss amount is incorrect",
                    LongDescription = @"The sum of 'Net income or loss from carrying on a business of investing', 'Net income or loss from carrying on a rental property business' and 'Other income or loss relating to Other income or loss relating to Net income or loss from business', must equal the amount provided for 'Net non-Primary Production income/loss'",
                    Location = "/xbrli:xbrl/tns:Income.NonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000214" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.IITR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR282", Value = GetValueOrEmpty(report.IITR282) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR283", Value = GetValueOrEmpty(report.IITR283) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR284", Value = GetValueOrEmpty(report.IITR284) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000216
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000216()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000216
            Discount from deferral schemes has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR125, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR125, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
            */
            assertion = NotMonetary(report.IITR125, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000216",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Discount from deferral schemes is not in a valid monetary format",
                    LongDescription = @"Discount from deferral schemes must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DeferralDiscount.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000216" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = GetValueOrEmpty(report.IITR125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000219
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000219()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000219
            Dividends - Franking credit cannot be greater than 48% of Dividends - Franked amount.
    
            Legacy Rule Format:
            (^IITR120 > 0) AND ((^IITR119*0.48) < (^IITR120 - 1))

            Technical Business Rule Format:
            (^IITR120 > 0) AND ((^IITR119*0.48) < (^IITR120 - 1))
    
            Data Elements:
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
            ^IITR119 = RP:Income.DividendsFranked.Amount
            */
            assertion = (report.IITR120.GetValueOrDefault() > 0 && report.IITR119.GetValueOrDefault() * 0.48M < report.IITR120.GetValueOrDefault() - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000219",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends - Franking credit amount is invalid",
                    LongDescription = @"Dividends - Franking credit cannot be greater than 48% of Dividends - Franked amount",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000219" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.IITR120) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = GetValueOrEmpty(report.IITR119) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000220
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000220()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000220
            Dividends - Franked amount has a value and Dividends - Franking credit has no value.
    
            Legacy Rule Format:
            (^IITR119 > 0) AND (^IITR120 = NULL)

            Technical Business Rule Format:
            (^IITR119 > 0) AND (^IITR120 = NULL)
    
            Data Elements:
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
            ^IITR119 = RP:Income.DividendsFranked.Amount
            */
            assertion = (report.IITR119.GetValueOrDefault() > 0 && report.IITR120 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000220",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends - Franking credit is invalid",
                    LongDescription = @"When an amount above zero is provided for 'Dividends - Franked amount', an amount for 'Dividends - Franking credit' must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000220" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = GetValueOrEmpty(report.IITR119) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.IITR120) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000224
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000224()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000224
            Tax withheld amounts from Australian government pensions or allowances are greater than or equal to the amount of Australian government pensions.  Check both amounts and correct as necessary.
    
            Legacy Rule Format:
            (^IITR88 > 0) AND ((^IITR89 = NULL) OR (^IITR89 <= ^IITR88))

            Technical Business Rule Format:
            (^IITR88 > 0) AND ((^IITR89 = NULL) OR (^IITR89 <= ^IITR88))
    
            Data Elements:
    
            ^IITR88 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount
    
            ^IITR89 = RP:Pension.Total.Amount
            */
            assertion = (report.IITR88.GetValueOrDefault() > 0 && (report.IITR89 == null || report.IITR89.GetValueOrDefault() <= report.IITR88.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld from Australian Government pensions and allowances incorrect",
                    LongDescription = @"Tax withheld amounts from Australian government pensions or allowances are greater than or equal to the amount of Australian government pensions. Both amounts must be correct.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000224" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR88", Value = GetValueOrEmpty(report.IITR88) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR89", Value = GetValueOrEmpty(report.IITR89) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000225
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000225()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000225
            Allowances, earnings, tips, director's fees etc must be greater than the Tax withheld from allowances, earnings, tips, director’s fees etc
    
            Legacy Rule Format:
            (^IITR64 > 0) AND ((^IITR65 = NULL) OR (^IITR65 <= ^IITR64))

            Technical Business Rule Format:
            (^IITR64 > 0) AND ((^IITR65 = NULL) OR (^IITR65 <= ^IITR64))
    
            Data Elements:
    
            ^IITR64 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAllowancesEarningsTipsDirectorsFees.Amount
    
            ^IITR65 = RP:Income.AllowancesEarningsTipsDirectorsFees.Amount
            */
            assertion = (report.IITR64.GetValueOrDefault() > 0 && (report.IITR65 == null || report.IITR65.GetValueOrDefault() <= report.IITR64.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001225",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld from allowances, earnings, tips, director's fees etc amount incorrect",
                    LongDescription = @"Allowances, earnings, tips, director's fees etc must be greater than the Tax withheld from allowances, earnings, tips, director's fees etc",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAllowancesEarningsTipsDirectorsFees.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000225" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR64", Value = GetValueOrEmpty(report.IITR64) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR65", Value = GetValueOrEmpty(report.IITR65) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000226
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000226()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000226
            Share of net income from trusts is greater than zero and Share of net income from trusts action code has not been completed.
    
            Legacy Rule Format:
            (^IITR243 > 0) AND (^IITR244 = NULL)

            Technical Business Rule Format:
            (^IITR243 > 0) AND (^IITR244 = NULL)
    
            Data Elements:
    
            ^IITR244 = RP:Income.TrustPrimaryProductionNetShare.Code
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
            */
            assertion = (report.IITR243.GetValueOrDefault() > 0 && report.IITR244 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts action code is required",
                    LongDescription = @"When an amount above zero is provided for 'Share of net income from trusts', 'Share of net income from trusts action code' is required",
                    Location = "/xbrli:xbrl/tns:Income.TrustPrimaryProductionNetShare.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000226" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR244", Value = GetValueOrEmpty(report.IITR244) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000227
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000227()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000227
            Share of net income from trusts code is present and the income amount is less than or equal to zero. A code is not required for negative or zero amounts.
    
            Legacy Rule Format:
            (^IITR244 <> NULL) AND (^IITR243 = NULL OR ^IITR243 <= 0)

            Technical Business Rule Format:
            (^IITR244 <> NULL) AND (^IITR243 = NULL OR ^IITR243 <= 0)
    
            Data Elements:
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
            ^IITR244 = RP:Income.TrustPrimaryProductionNetShare.Code
            */
            assertion = (report.IITR244 != null && (report.IITR243 == null || report.IITR243.GetValueOrDefault() <= 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000227",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts is invalid",
                    LongDescription = @"When 'Share of net income from trusts code' is present, an amount above zero is required for 'Share of net income from trusts'. If 'Share of net income from trusts' is zero or negative, 'Share of net income from trusts code' must not be provided.",
                    Location = "/xbrli:xbrl/tns:Income.TrustPrimaryProductionNetShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000227" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR244", Value = GetValueOrEmpty(report.IITR244) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000234
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000234()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000234
            There are amounts for either Other deductions relating to amounts shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts", Landcare operations expenses, Share of franking credit from franked dividends or Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions or Credit for TFN amounts withheld from payments from closely held trusts without any Distribution from partnerships less foreign income and Share of net income from trusts less capital gains, foreign income and franked distributions and Franked distributions from trusts being present.
    
            Legacy Rule Format:
            (^IITR265 > 0 OR ^IITR260 > 0 OR ^IITR269 > 0 OR ^IITR270 > 0 OR ^IITR271 > 0) AND (^IITR252 = NULL AND ^IITR255 = NULL AND ^IITR259 = NULL)

            Technical Business Rule Format:
            (^IITR265 > 0 OR ^IITR260 > 0 OR ^IITR269 > 0 OR ^IITR270 > 0 OR ^IITR271 > 0) AND (^IITR252 = NULL AND ^IITR255 = NULL AND ^IITR259 = NULL)
    
            Data Elements:
    
            ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount
    
            ^IITR260 = RP:Expense.LandcareOperationsNonPrimaryProduction.Amount
    
            ^IITR265 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
    
            ^IITR269 = RP:IncomeTax.FrankingCreditsShare.Amount
    
            ^IITR270 = RP:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
    
            ^IITR271 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount
            */
            assertion = ((report.IITR265.GetValueOrDefault() > 0 || report.IITR260.GetValueOrDefault() > 0 || report.IITR269.GetValueOrDefault() > 0 || report.IITR270.GetValueOrDefault() > 0 || report.IITR271.GetValueOrDefault() > 0) && (report.IITR252 == null && report.IITR255 == null && report.IITR259 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000234",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP distribution or share of net income amounts are incorrect",
                    LongDescription = @"There are amounts for either Other deductions relating to amounts shown at ""Distribution from partnerships less foreign income"", ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts"", Landcare operations expenses, Share of franking credit from franked dividends or Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions or Credit for TFN amounts withheld from payments from closely held trusts without any Distribution from partnerships less foreign income and Share of net income from trusts less capital gains, foreign income and franked distributions and Franked distributions from trusts being present",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000234" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR265", Value = GetValueOrEmpty(report.IITR265) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR260", Value = GetValueOrEmpty(report.IITR260) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR269", Value = GetValueOrEmpty(report.IITR269) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR270", Value = GetValueOrEmpty(report.IITR270) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR271", Value = GetValueOrEmpty(report.IITR271) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR252", Value = GetValueOrEmpty(report.IITR252) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.IITR255) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR259", Value = GetValueOrEmpty(report.IITR259) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000235
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000235()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000235
            Share of credit for tax withheld where Australian business number not quoted is greater than zero without any Distribution from partnerships and Distribution from partnerships less foreign income and Distribution from partnerships less foreign income and Share of net income from trusts less capital gains, foreign income and franked distributions or Franked distributions from trusts
    
            Legacy Rule Format:
            (^IITR268 > 0) AND (^IITR242 = NULL AND ^IITR243 = NULL AND ^IITR252 = NULL AND ^IITR255 = NULL AND ^IITR259 = NULL)

            Technical Business Rule Format:
            (^IITR268 > 0) AND (^IITR242 = NULL AND ^IITR243 = NULL AND ^IITR252 = NULL AND ^IITR255 = NULL AND ^IITR259 = NULL)
    
            Data Elements:
    
            ^IITR268 = RP:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
            ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount
            */
            assertion = (report.IITR268.GetValueOrDefault() > 0 && (report.IITR242 == null && report.IITR243 == null && report.IITR252 == null && report.IITR255 == null && report.IITR259 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000235",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution and share of net income amounts incorrect",
                    LongDescription = @"Share of credit for tax withheld where Australian business number not quoted is greater than zero without any Distribution from partnerships and Distribution from partnerships less foreign income and Distribution from partnerships less foreign income and Share of net income from trusts less capital gains, foreign income and franked distributions or Franked distributions from trusts",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000235" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR268", Value = GetValueOrEmpty(report.IITR268) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR242", Value = GetValueOrEmpty(report.IITR242) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR252", Value = GetValueOrEmpty(report.IITR252) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.IITR255) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR259", Value = GetValueOrEmpty(report.IITR259) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000238
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000238()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000238
            Share of net income from trusts less capital gains, foreign income and franked distributions action code is present and the amount of Share of net income from trusts less capital gains, foreign income and franked distributions is less than or equal to zero.  An action code is not required for negative or zero amounts.
    
            Legacy Rule Format:
            (^IITR256 <> NULL) AND (^IITR255 = NULL OR ^IITR255 <= 0)

            Technical Business Rule Format:
            (^IITR256 <> NULL) AND (^IITR255 = NULL OR ^IITR255 <= 0)
    
            Data Elements:
    
            ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR256 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Code
            */
            assertion = (report.IITR256 != null && (report.IITR255 == null || report.IITR255.GetValueOrDefault() <= 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000238",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts less capital gains, foreign income and franked distributions is invalid",
                    LongDescription = @"When 'Share of net income from trusts less capital gains, foreign income and franked distributions action code' is present, an amount above zero is required for 'Share of net income from trusts less capital gains, foreign income and franked distributions'. If 'Share of net income from trusts less capital gains, foreign income and franked distributions' is zero or negative, 'Share of net income from trusts less capital gains, foreign income and franked distributions action code' must not be provided.",
                    Location = "/xbrli:xbrl/tns:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000238" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR256", Value = GetValueOrEmpty(report.IITR256) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.IITR255) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000241
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000241()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000241
            ESS TFN amounts withheld from discounts cannot be greater than 50% of the sum of Discount from taxed upfront schemes - eligible for reduction, Discount from taxed upfront schemes - not eligible for reduction, and Discount from deferral schemes.
    
            Legacy Rule Format:
            (^IITR128 > (^IITR123 + ^IITR124 + ^IITR125)*0.5)

            Technical Business Rule Format:
            (^IITR128 > (^IITR123 + ^IITR124 + ^IITR125)*0.5)
    
            Data Elements:
    
            ^IITR128 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
    
            ^IITR125 = RP:Income.EmployeeShareScheme.DeferralDiscount.Amount
            */
            assertion = (report.IITR128.GetValueOrDefault() > (report.IITR123.GetValueOrDefault() + report.IITR124.GetValueOrDefault() + report.IITR125.GetValueOrDefault()) * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000241",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ESS TFN amounts withheld from discounts is incorrect",
                    LongDescription = @"ESS TFN amounts withheld from discounts cannot be greater than 50% of the sum of Discount from taxed upfront schemes - eligible for reduction, Discount from taxed upfront schemes - not eligible for reduction, and Discount from deferral schemes",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000241" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR128", Value = GetValueOrEmpty(report.IITR128) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = GetValueOrEmpty(report.IITR123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = GetValueOrEmpty(report.IITR124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR125", Value = GetValueOrEmpty(report.IITR125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000242
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000242()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000242
            The amount entered for PSI Tax withheld where ABN not quoted is greater than 50% of PSI where ABN not quoted.
    
            Legacy Rule Format:
            (^IITR277 > 0) AND (^IITR277 > (^IITR412 * 0.5))

            Technical Business Rule Format:
            (^IITR277 > 0) AND (^IITR277 > (^IITR412 * 0.5))
    
            Data Elements:
    
            ^IITR277 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount
    
            ^IITR412 = RP:Income.PersonalServicesIncome.ABNNotQuoted.Amount
            */
            assertion = (report.IITR277.GetValueOrDefault() > 0 && report.IITR277.GetValueOrDefault() > report.IITR412.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000242",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted is incorrect",
                    LongDescription = @"The amount entered for Tax withheld where ABN not quoted, must not be higher than 50% of PSI where ABN not quoted",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000242" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR277", Value = GetValueOrEmpty(report.IITR277) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR412", Value = GetValueOrEmpty(report.IITR412) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000243
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000243()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000243
            An amount greater than zero has been entered at PSI where ABN not quoted, and Tax withheld where ABN not quoted is absent or equal to zero.
    
            Legacy Rule Format:
            ((^IITR277 = NULL) OR (^IITR277 = 0)) AND (^IITR412 > 0)

            Technical Business Rule Format:
            ((^IITR277 = NULL) OR (^IITR277 = 0)) AND (^IITR412 > 0)
    
            Data Elements:
    
            ^IITR277 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount
    
            ^IITR412 = RP:Income.PersonalServicesIncome.ABNNotQuoted.Amount
            */
            assertion = ((report.IITR277 == null || report.IITR277 == 0) && report.IITR412.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000243",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted is incorrect",
                    LongDescription = @"Where an amount above zero has been entered at PSI where ABN not quoted, a Tax withheld where ABN not quoted amount above zero must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000243" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR277", Value = GetValueOrEmpty(report.IITR277) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR412", Value = GetValueOrEmpty(report.IITR412) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000244
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000244()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000244
            Net income/loss from business tax withheld where Australian business number not quoted is greater than 50% of the sum of PP - Gross payments where Australian Business Number not quoted and Non-PP - Gross payments where Australian Business Number not quoted
    
            Legacy Rule Format:
            (^IITR287 > 0) AND (^IITR287 > ((^IITR436 + ^IITR437)*0.50))

            Technical Business Rule Format:
            (^IITR287 > 0) AND (^IITR287 > ((^IITR436 + ^IITR437)*0.50))
    
            Data Elements:
    
            ^IITR287 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount
    
            ^IITR436 = RP:Income.ABNNotQuotedPrimaryProductionPaymentGross.Amount
    
            ^IITR437 = RP:Income.ABNNotQuotedNonPrimaryProductionPaymentGross.Amount
            */
            assertion = (report.IITR287.GetValueOrDefault() > 0 && report.IITR287.GetValueOrDefault() > (report.IITR436.GetValueOrDefault() + report.IITR437.GetValueOrDefault()) * 0.50M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000244",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted is incorrect",
                    LongDescription = @"Net income/loss from business tax withheld where Australian business number not quoted is greater than 50% of the sum of PP - Gross payments where Australian Business Number not quoted and Non-PP - Gross payments where Australian Business Number not quoted",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000244" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR287", Value = GetValueOrEmpty(report.IITR287) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR436", Value = GetValueOrEmpty(report.IITR436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR437", Value = GetValueOrEmpty(report.IITR437) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000245
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000245()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000245
            The sum of PP - Gross payments where Australian Business Number not quoted and Non-PP - Gross payments where Australian Business Number not quoted is greater than zero, and Net income/loss from business tax withheld where Australian business number not quoted is absent.
    
            Legacy Rule Format:
            ((^IITR436 + ^IITR437) > 0) AND (^IITR287 = NULL)

            Technical Business Rule Format:
            ((^IITR436 + ^IITR437) > 0) AND (^IITR287 = NULL)
    
            Data Elements:
    
            ^IITR287 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount
    
            ^IITR436 = RP:Income.ABNNotQuotedPrimaryProductionPaymentGross.Amount
    
            ^IITR437 = RP:Income.ABNNotQuotedNonPrimaryProductionPaymentGross.Amount
            */
            assertion = (report.IITR436.GetValueOrDefault() + report.IITR437.GetValueOrDefault() > 0 && report.IITR287 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000245",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted is required",
                    LongDescription = @"The sum of PP - Gross payments where Australian Business Number not quoted and Non-PP - Gross payments where Australian Business Number not quoted is greater than zero, and Net income/loss from business tax withheld where Australian business number not quoted is absent",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000245" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR436", Value = GetValueOrEmpty(report.IITR436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR437", Value = GetValueOrEmpty(report.IITR437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR287", Value = GetValueOrEmpty(report.IITR287) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000246
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000246()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000246
            The amount at Tax withheld - voluntary agreement is greater than 50% of PSI - voluntary agreement
    
            Legacy Rule Format:
            (^IITR276 > 0) AND (^IITR276 > (^IITR411*0.50))

            Technical Business Rule Format:
            (^IITR276 > 0) AND (^IITR276 > (^IITR411*0.50))
    
            Data Elements:
    
            ^IITR276 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount
    
            ^IITR411 = RP:Income.PersonalServicesIncome.VoluntaryAgreement.Amount
            */
            assertion = (report.IITR276.GetValueOrDefault() > 0 && report.IITR276.GetValueOrDefault() > report.IITR411.GetValueOrDefault() * 0.50M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000246",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld - voluntary agreement is incorrect",
                    LongDescription = @"The amount at Tax withheld - voluntary agreement is greater than 50% of PSI - voluntary agreement",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000246" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR276", Value = GetValueOrEmpty(report.IITR276) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR411", Value = GetValueOrEmpty(report.IITR411) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000247
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000247()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000247
            An amount greater than zero has been entered at PSI - voluntary agreement and Tax withheld - voluntary agreement is missing or equal to zero
    
            Legacy Rule Format:
            ^IITR411 > 0 AND (^IITR276 = NULL OR ^IITR276 = 0)

            Technical Business Rule Format:
            ^IITR411 > 0 AND (^IITR276 = NULL OR ^IITR276 = 0)
    
            Data Elements:
    
            ^IITR276 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount
    
            ^IITR411 = RP:Income.PersonalServicesIncome.VoluntaryAgreement.Amount
            */
            assertion = (report.IITR411.GetValueOrDefault() > 0 && (report.IITR276 == null || report.IITR276 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000247",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld - voluntary agreement is incorrect",
                    LongDescription = @"An amount greater than zero has been entered at PSI - voluntary agreement and Tax withheld - voluntary agreement is required or equal to zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000247" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR411", Value = GetValueOrEmpty(report.IITR411) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR276", Value = GetValueOrEmpty(report.IITR276) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000248
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000248()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000248
            The amount entered at Tax withheld - labour hire or other specified payments is greater than 50% of PSI - labour hire or other specified payments
    
            Legacy Rule Format:
            (^IITR278 > 0) AND (^IITR278 > (^IITR413*0.50))

            Technical Business Rule Format:
            (^IITR278 > 0) AND (^IITR278 > (^IITR413*0.50))
    
            Data Elements:
    
            ^IITR278 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount
    
            ^IITR413 = RP:Income.PersonalServicesIncome.LabourHireOrOtherSpecifiedPayment.Amount
            */
            assertion = (report.IITR278.GetValueOrDefault() > 0 && report.IITR278.GetValueOrDefault() > report.IITR413.GetValueOrDefault() * 0.50M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000248",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld labour hire or other specified payments incorrect",
                    LongDescription = @"The amount entered at Tax withheld - labour hire or other specified payments is greater than 50% of PSI - labour hire or other specified payments",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000248" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR278", Value = GetValueOrEmpty(report.IITR278) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR413", Value = GetValueOrEmpty(report.IITR413) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000249
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000249()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000249
            An amount greater than zero has been entered at PSI - labour hire or other specified payments and Tax withheld - labour hire or other specified payments is missing
    
            Legacy Rule Format:
            ^IITR413 > 0 AND (^IITR278 = NULL)

            Technical Business Rule Format:
            ^IITR413 > 0 AND (^IITR278 = NULL)
    
            Data Elements:
    
            ^IITR278 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount
    
            ^IITR413 = RP:Income.PersonalServicesIncome.LabourHireOrOtherSpecifiedPayment.Amount
            */
            assertion = (report.IITR413.GetValueOrDefault() > 0 && report.IITR278 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000249",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld labour hire or other specified payments is required",
                    LongDescription = @"When a positive amount has been provided for 'PSI - labour hire or other specified payments', an amount for 'Tax withheld - labour hire or other specified payments' is required",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000249" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR413", Value = GetValueOrEmpty(report.IITR413) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR278", Value = GetValueOrEmpty(report.IITR278) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000250
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000250()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000250
            Amounts have been entered for PP -  Gross payments voluntary agreement or Non-PP - Gross payments voluntary agreement and Net income/loss from business tax withheld voluntary agreement is missing
    
            Legacy Rule Format:
            ((^IITR439 + ^IITR440) > 0) AND (^IITR286 = NULL)

            Technical Business Rule Format:
            ((^IITR439 + ^IITR440) > 0) AND (^IITR286 = NULL)
    
            Data Elements:
    
            ^IITR286 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount
    
            ^IITR439 = RP:Income.VoluntaryAgreementPrimaryProductionGross.Amount
    
            ^IITR440 = RP:Income.VoluntaryAgreementNonPrimaryProductionGross.Amount
            */
            assertion = (report.IITR439.GetValueOrDefault() + report.IITR440.GetValueOrDefault() > 0 && report.IITR286 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000250",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld voluntary agreement required",
                    LongDescription = @"Amounts have been entered for PP -  Gross payments voluntary agreement or Non-PP - Gross payments voluntary agreement and Net income/loss from business tax withheld voluntary agreement is required",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000250" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR439", Value = GetValueOrEmpty(report.IITR439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR440", Value = GetValueOrEmpty(report.IITR440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR286", Value = GetValueOrEmpty(report.IITR286) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000251
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000251()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000251
            Amount entered at Net income/loss from business tax withheld voluntary agreement is greater than 50% of PP -  Gross payments voluntary agreement and Non-PP - Gross payments voluntary agreement
    
            Legacy Rule Format:
            (^IITR286 > 0) AND (^IITR286 > (^IITR439 + ^IITR440)*0.50)

            Technical Business Rule Format:
            (^IITR286 > 0) AND (^IITR286 > (^IITR439 + ^IITR440)*0.50)
    
            Data Elements:
    
            ^IITR286 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount
    
            ^IITR439 = RP:Income.VoluntaryAgreementPrimaryProductionGross.Amount
    
            ^IITR440 = RP:Income.VoluntaryAgreementNonPrimaryProductionGross.Amount
            */
            assertion = (report.IITR286.GetValueOrDefault() > 0 && report.IITR286.GetValueOrDefault() > (report.IITR439.GetValueOrDefault() + report.IITR440.GetValueOrDefault()) * 0.50M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000251",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld voluntary agreement incorrect",
                    LongDescription = @"Amount entered at Net income/loss from business tax withheld voluntary agreement is greater than 50% of PP -  Gross payments voluntary agreement and Non-PP - Gross payments voluntary agreement",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000251" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR286", Value = GetValueOrEmpty(report.IITR286) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR439", Value = GetValueOrEmpty(report.IITR439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR440", Value = GetValueOrEmpty(report.IITR440) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000252
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000252()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000252
            Amounts have been entered at PP - Gross payments labour hire or other specified payments and Non-PP - Gross payments labour hire or other specified payments and Net income/loss from business tax withheld - labour hire or other specified payments is missing
    
            Legacy Rule Format:
            ((^IITR441 + ^IITR442) > 0) AND (^IITR289 = NULL)

            Technical Business Rule Format:
            ((^IITR441 + ^IITR442) > 0) AND (^IITR289 = NULL)
    
            Data Elements:
    
            ^IITR289 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount
    
            ^IITR441 = RP:Income.LabourHireArrangementPrimaryProductionPaymentGross.Amount
    
            ^IITR442 = RP:Income.LabourHireArrangementNonPrimaryProductionPaymentGross.Amount
            */
            assertion = (report.IITR441.GetValueOrDefault() + report.IITR442.GetValueOrDefault() > 0 && report.IITR289 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000252",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld labour hire required",
                    LongDescription = @"Amounts have been entered at PP - Gross payments labour hire or other specified payments and Non-PP - Gross payments labour hire or other specified payments and Net income/loss from business tax withheld - labour hire or other specified payments is required",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000252" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR441", Value = GetValueOrEmpty(report.IITR441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR442", Value = GetValueOrEmpty(report.IITR442) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR289", Value = GetValueOrEmpty(report.IITR289) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000253
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000253()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000253
            Amounts have been entered at Net income/loss from business tax withheld - labour hire or other specified payments and PP - Gross payments labour hire or other specified payments and Non-PP - Gross payments labour hire or other specified payments are missing or equal to zero
    
            Legacy Rule Format:
            (^IITR289 > 0) AND ((^IITR441 + ^IITR442) = 0)

            Technical Business Rule Format:
            (^IITR289 > 0) AND ((^IITR441 + ^IITR442) = 0)
    
            Data Elements:
    
            ^IITR289 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount
    
            ^IITR441 = RP:Income.LabourHireArrangementPrimaryProductionPaymentGross.Amount
    
            ^IITR442 = RP:Income.LabourHireArrangementNonPrimaryProductionPaymentGross.Amount
            */
            assertion = (report.IITR289.GetValueOrDefault() > 0 && report.IITR441.GetValueOrDefault() + report.IITR442.GetValueOrDefault() == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000253",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross payments labour hire required",
                    LongDescription = @"Amounts have been entered at Net income/loss from business tax withheld - labour hire or other specified payments and PP - Gross payments labour hire or other specified payments and Non-PP - Gross payments labour hire or other specified payments are required or equal to zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000253" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR289", Value = GetValueOrEmpty(report.IITR289) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR441", Value = GetValueOrEmpty(report.IITR441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR442", Value = GetValueOrEmpty(report.IITR442) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000254
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000254()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000254
            Foreign income tax offset is greater than or equal to total foreign source income. Total foreign source income is the sum of foreign source discounts from the employee share schemes section on the main return form plus Total current year capital gains plus CFC income plus transferor trust income plus other net foreign employment income plus Net foreign pension or annuity income without an undeducted purchase price plus Net foreign pension or annuity income with an undeducted purchase price plus Net foreign rent plus Other net foreign source income plus Net foreign employment income - payment summary plus Primary production deferred losses plus Non-primary production deferred losses.
    
            Legacy Rule Format:
            (^IITR331 > 1000) AND (^IITR331 >=  (^IITR129 + ^IITR312 + ^IITR316 + ^IITR318 + ^IITR321 + ^IITR322 + ^IITR323 + ^IITR324 + ^IITR327 + ^IITR329 + ^IITR299 + ^IITR300))

            Technical Business Rule Format:
            (^IITR331 > 1000) AND (^IITR331 >=  (^IITR129 + ^IITR312 + ^IITR316 + ^IITR318 + ^IITR321 + ^IITR322 + ^IITR323 + ^IITR324 + ^IITR327 + ^IITR329 + ^IITR299 + ^IITR300))
    
            Data Elements:
    
            ^IITR331 = RP:Income.InternationalDealings.TaxOffset.Amount
    
            ^IITR129 = RP:Income.EmployeeShareScheme.ForeignSourceDiscount.Amount
    
            ^IITR299 = RP:Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount
    
            ^IITR300 = RP:Tax.Losses.DeferredNonCommercialNonPrimaryProduction.Amount
    
            ^IITR312 = RP:Capital.Gains.Total.Amount
    
            ^IITR316 = RP:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^IITR318 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
    
            ^IITR321 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount
    
            ^IITR322 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeWithoutUndeductedPurchasePrice.Amount
    
            ^IITR323 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeUndeductedPurchasePrice.Amount
    
            ^IITR324 = RP:InternationalDealings.ForeignRent.Amount
    
            ^IITR327 = RP:Income.InternationalDealings.Net.Amount
    
            ^IITR329 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
            */
            assertion = (report.IITR331.GetValueOrDefault() > 1000 && report.IITR331.GetValueOrDefault() >= report.IITR129.GetValueOrDefault() + report.IITR312.GetValueOrDefault() + report.IITR316.GetValueOrDefault() + report.IITR318.GetValueOrDefault() + report.IITR321.GetValueOrDefault() + report.IITR322.GetValueOrDefault() + report.IITR323.GetValueOrDefault() + report.IITR324.GetValueOrDefault() + report.IITR327.GetValueOrDefault() + report.IITR329.GetValueOrDefault() + report.IITR299.GetValueOrDefault() + report.IITR300.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000254",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income tax offset must be less than total foreign income",
                    LongDescription = @"Foreign income tax offset must be less than the sum of total current year capital gains, CFC income, Transferor trust income, Other net foreign employment income, Net foreign pension or annuity income without an undeducted purchase price, Net foreign pension or annuity income with an UPP, Net foreign rent, Other net foreign source income, Net foreign employment income - payment summary, Primary production deferred losses and Non-Primary production deferred losses.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.TaxOffset.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000254" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR331", Value = GetValueOrEmpty(report.IITR331) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR129", Value = GetValueOrEmpty(report.IITR129) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR312", Value = GetValueOrEmpty(report.IITR312) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR316", Value = GetValueOrEmpty(report.IITR316) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR318", Value = GetValueOrEmpty(report.IITR318) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR321", Value = GetValueOrEmpty(report.IITR321) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR322", Value = GetValueOrEmpty(report.IITR322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR323", Value = GetValueOrEmpty(report.IITR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR324", Value = GetValueOrEmpty(report.IITR324) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR327", Value = GetValueOrEmpty(report.IITR327) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR329", Value = GetValueOrEmpty(report.IITR329) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR299", Value = GetValueOrEmpty(report.IITR299) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR300", Value = GetValueOrEmpty(report.IITR300) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000255()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000255
            Amount entered at Net income/loss from business tax withheld - labour hire or other specified payments is greater than 50% of PP - Gross payments labour hire or other specified payments AND Non-PP - Gross payments labour hire or other specified payments
    
            Legacy Rule Format:
            (^IITR289 > 0) AND (^IITR289 > (^IITR441 + ^IITR442)*0.50)

            Technical Business Rule Format:
            (^IITR289 > 0) AND (^IITR289 > (^IITR441 + ^IITR442)*0.50)
    
            Data Elements:
    
            ^IITR289 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount
    
            ^IITR441 = RP:Income.LabourHireArrangementPrimaryProductionPaymentGross.Amount
    
            ^IITR442 = RP:Income.LabourHireArrangementNonPrimaryProductionPaymentGross.Amount
            */
            assertion = (report.IITR289.GetValueOrDefault() > 0 && report.IITR289.GetValueOrDefault() > (report.IITR441.GetValueOrDefault() + report.IITR442.GetValueOrDefault()) * 0.50M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld labour hire incorrect",
                    LongDescription = @"Amount entered at Net income/loss from business tax withheld - labour hire or other specified payments is greater than 50% of PP - Gross payments labour hire or other specified payments AND Non-PP - Gross payments labour hire or other specified payments",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000255" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR289", Value = GetValueOrEmpty(report.IITR289) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR441", Value = GetValueOrEmpty(report.IITR441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR442", Value = GetValueOrEmpty(report.IITR442) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000256
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000256()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000256
            PP - Net income or loss from business this year less PP - Deferred non-commercial losses from a prior year does not equal Net Primary Production income/loss
    
            Legacy Rule Format:
            (^IITR281 <> NULL AND (^IITR281 <> (^IITR474 - ^IITR476))) OR (^IITR281 = NULL AND ((^IITR474 - ^IITR476) <> 0))

            Technical Business Rule Format:
            (^IITR281 <> NULL AND (^IITR281 <> (^IITR474 - ^IITR476))) OR (^IITR281 = NULL AND ((^IITR474 - ^IITR476) <> 0))
    
            Data Elements:
    
            ^IITR281 = RP:Income.PrimaryProductionNet.Amount
    
            ^IITR474 = RP:Income.BusinessPrimaryProductionNet.Amount
    
            ^IITR476 = RP:Tax.Losses.BusinessDeferredNonCommercialLossPriorYearPrimaryProduction.Amount
            */
            assertion = (report.IITR281 != null && report.IITR281.GetValueOrDefault() != report.IITR474.GetValueOrDefault() - report.IITR476.GetValueOrDefault() || report.IITR281 == null && report.IITR474.GetValueOrDefault() - report.IITR476.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000256",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Primary Production income/loss is incorrect",
                    LongDescription = @"PP - Net income or loss from business this year less PP - Deferred non-commercial losses from a prior year does not equal Net Primary Production income/loss",
                    Location = "/xbrli:xbrl/tns:Income.PrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000256" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.IITR281) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR474", Value = GetValueOrEmpty(report.IITR474) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR476", Value = GetValueOrEmpty(report.IITR476) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000257
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000257()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000257
            Non-PP - Net income or loss from business this year less Non-PP - Deferred non-commercial losses from a prior year does not equal Net non-Primary Production income/loss
    
            Legacy Rule Format:
            (^IITR285 <> NULL AND (^IITR285 <> (^IITR475 - ^IITR477))) OR (^IITR285 = NULL AND ((^IITR475 - ^IITR477) <> 0))

            Technical Business Rule Format:
            (^IITR285 <> NULL AND (^IITR285 <> (^IITR475 - ^IITR477))) OR (^IITR285 = NULL AND ((^IITR475 - ^IITR477) <> 0))
    
            Data Elements:
    
            ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
    
            ^IITR475 = RP:Income.BusinessNonPrimaryProductionNet.Amount
    
            ^IITR477 = RP:Tax.Losses.BusinessDeferredNonCommercialLossPriorYearNonPrimaryProduction.Amount
            */
            assertion = (report.IITR285 != null && report.IITR285.GetValueOrDefault() != report.IITR475.GetValueOrDefault() - report.IITR477.GetValueOrDefault() || report.IITR285 == null && report.IITR475.GetValueOrDefault() - report.IITR477.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000257",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-Primary Production income/loss is incorrect",
                    LongDescription = @"Non-PP - Net income or loss from business this year less Non-PP - Deferred non-commercial losses from a prior year does not equal Net non-Primary Production income/loss",
                    Location = "/xbrli:xbrl/tns:Income.NonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000257" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.IITR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR475", Value = GetValueOrEmpty(report.IITR475) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR477", Value = GetValueOrEmpty(report.IITR477) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000258
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000258()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000258
            The answer to the question 'Received 80% or more of PSI from one source' is 'yes'; however there are no income amounts provided at Net Personal Services Income and Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains)
    
            Legacy Rule Format:
            (^IITR407 = TRUE) AND (^IITR279 = NULL) AND (^IITR438 = NULL)

            Technical Business Rule Format:
            (^IITR407 = TRUE) AND (^IITR279 = NULL) AND (^IITR438 = NULL)
    
            Data Elements:
    
            ^IITR407 = RP:IncomeTax.PersonalServicesIncome.OneSourceBenchmark.Indicator
    
            ^IITR279 = RP:Income.PersonalServicesIncome.Net.Amount
    
            ^IITR438 = RP:Income.ForeignResidentWithholdingNonPrimaryProduction.Amount
            */
            assertion = (report.IITR407.GetValueOrDefault() && report.IITR279 == null && report.IITR438 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001258",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Personal Services Income and Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains) are required",
                    LongDescription = @"The answer to the question 'Received 80% or more of PSI from one source' is 'yes'; however there are no income amounts provided at Net Personal Services Income and Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PersonalServicesIncome.OneSourceBenchmark.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000258" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR407", Value = GetValueOrEmpty(report.IITR407) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR279", Value = GetValueOrEmpty(report.IITR279) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR438", Value = GetValueOrEmpty(report.IITR438) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000259()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000259
            Net Personal Services Income is not equal to the sum of all PSI income less deductions.
    
            Legacy Rule Format:
            ((^IITR279 <> NULL) AND (^IITR279 <> (^IITR411 + ^IITR412 + ^IITR413 + ^IITR414 - ^IITR415 - ^IITR416))) OR ((^IITR279 = NULL) AND ((^IITR411 + ^IITR412 + ^IITR413 + ^IITR414 - ^IITR415 - ^IITR416) <> 0))

            Technical Business Rule Format:
            ((^IITR279 <> NULL) AND (^IITR279 <> (^IITR411 + ^IITR412 + ^IITR413 + ^IITR414 - ^IITR415 - ^IITR416))) OR ((^IITR279 = NULL) AND ((^IITR411 + ^IITR412 + ^IITR413 + ^IITR414 - ^IITR415 - ^IITR416) <> 0))
    
            Data Elements:
    
            ^IITR279 = RP:Income.PersonalServicesIncome.Net.Amount
    
            ^IITR411 = RP:Income.PersonalServicesIncome.VoluntaryAgreement.Amount
    
            ^IITR412 = RP:Income.PersonalServicesIncome.ABNNotQuoted.Amount
    
            ^IITR413 = RP:Income.PersonalServicesIncome.LabourHireOrOtherSpecifiedPayment.Amount
    
            ^IITR414 = RP:Income.PersonalServicesIncome.Other.Amount
    
            ^IITR415 = RP:Expense.PersonalServicesIncome.PrincipalWorkPaymentsToAssociates.Amount
    
            ^IITR416 = RP:Expense.PersonalServicesIncome.OtherDeductionsTotal.Amount
            */
            assertion = (report.IITR279 != null && report.IITR279.GetValueOrDefault() != report.IITR411.GetValueOrDefault() + report.IITR412.GetValueOrDefault() + report.IITR413.GetValueOrDefault() + report.IITR414.GetValueOrDefault() - report.IITR415.GetValueOrDefault() - report.IITR416.GetValueOrDefault() || report.IITR279 == null && report.IITR411.GetValueOrDefault() + report.IITR412.GetValueOrDefault() + report.IITR413.GetValueOrDefault() + report.IITR414.GetValueOrDefault() - report.IITR415.GetValueOrDefault() - report.IITR416.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000259",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Personal Services Income is incorrect",
                    LongDescription = @"The sum of 'PSI - voluntary agreement', 'PSI where Australian business number not quoted', 'PSI - labour hire or other specified payments', 'PSI - other', 'Deductions for payments to associates for principal work' and 'Total amount of other deductions against PSI', must equal the amount provided for 'Net Personal Services Income'",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncome.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000259" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR279", Value = GetValueOrEmpty(report.IITR279) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR411", Value = GetValueOrEmpty(report.IITR411) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR412", Value = GetValueOrEmpty(report.IITR412) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR413", Value = GetValueOrEmpty(report.IITR413) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR414", Value = GetValueOrEmpty(report.IITR414) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR415", Value = GetValueOrEmpty(report.IITR415) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR416", Value = GetValueOrEmpty(report.IITR416) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000260
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000260()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000260
            Share of credit for foreign resident withholding amounts (excluding capital gains) is greater than zero without any Distribution from partnerships or Share of net income from trusts or Distribution from partnerships less foreign income or an amount at Franked distributions from trusts
    
            Legacy Rule Format:
            (^IITR273 > 0) AND (^IITR242 = NULL AND ^IITR243 = NULL AND ^IITR252 = NULL AND ^IITR255 = NULL AND ^IITR259 = NULL)

            Technical Business Rule Format:
            (^IITR273 > 0) AND (^IITR242 = NULL AND ^IITR243 = NULL AND ^IITR252 = NULL AND ^IITR255 = NULL AND ^IITR259 = NULL)
    
            Data Elements:
    
            ^IITR273 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
    
            ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
            ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount
            */
            assertion = (report.IITR273.GetValueOrDefault() > 0 && (report.IITR242 == null && report.IITR243 == null && report.IITR252 == null && report.IITR255 == null && report.IITR259 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001260",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution or share amounts incorrect",
                    LongDescription = @"Share of credit for foreign resident withholding amounts (excluding capital gains) is greater than zero without any Distribution from partnerships or Share of net income from trusts or Distribution from partnerships less foreign income or an amount at Franked distributions from trusts",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000260" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR273", Value = GetValueOrEmpty(report.IITR273) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR242", Value = GetValueOrEmpty(report.IITR242) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR252", Value = GetValueOrEmpty(report.IITR252) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.IITR255) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR259", Value = GetValueOrEmpty(report.IITR259) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000261
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000261()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000261
            When an amount above zero has been provided for 'Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains)', an amount for 'Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains)' must be provided
    
            Legacy Rule Format:
            (^IITR438 > 0) AND (^IITR288 = NULL)

            Technical Business Rule Format:
            (^IITR438 > 0) AND (^IITR288 = NULL)
    
            Data Elements:
    
            ^IITR288 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount
    
            ^IITR438 = RP:Income.ForeignResidentWithholdingNonPrimaryProduction.Amount
            */
            assertion = (report.IITR438.GetValueOrDefault() > 0 && report.IITR288 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001261",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains) amount required",
                    LongDescription = @"When an amount above zero has been provided for 'Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains)', an amount for 'Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains)' must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000261" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR438", Value = GetValueOrEmpty(report.IITR438) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR288", Value = GetValueOrEmpty(report.IITR288) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000262()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000262
            When an amount is provided for 'Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains)', it must not be greater than 50% of the 'Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains)' amount
    
            Legacy Rule Format:
            (^IITR288 > 0) AND (^IITR288 > (^IITR438*0.50) + 1)

            Technical Business Rule Format:
            (^IITR288 > 0) AND (^IITR288 > (^IITR438*0.50) + 1)
    
            Data Elements:
    
            ^IITR288 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount
    
            ^IITR438 = RP:Income.ForeignResidentWithholdingNonPrimaryProduction.Amount
            */
            assertion = (report.IITR288.GetValueOrDefault() > 0 && report.IITR288.GetValueOrDefault() > report.IITR438.GetValueOrDefault() * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001262",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains) amount is incorrect",
                    LongDescription = @"When an amount is provided for 'Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains)', it must not be greater than 50% of the 'Non-PP - Gross payments subject to foreign resident withholding (excluding capital gains)' amount",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000262" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR288", Value = GetValueOrEmpty(report.IITR288) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR438", Value = GetValueOrEmpty(report.IITR438) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000266()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000266
            Net primary production income/loss amount is present without total business income from primary production (Gross payments, Assessable government industry payments, Other business income).
    
            Legacy Rule Format:
            (^IITR281 <> NULL) AND (^IITR436 = NULL AND ^IITR439 = NULL AND ^IITR441 = NULL AND ^IITR443 = NULL AND ^IITR447 = NULL)

            Technical Business Rule Format:
            (^IITR281 <> NULL) AND (^IITR436 = NULL AND ^IITR439 = NULL AND ^IITR441 = NULL AND ^IITR443 = NULL AND ^IITR447 = NULL)
    
            Data Elements:
    
            ^IITR281 = RP:Income.PrimaryProductionNet.Amount
    
            ^IITR436 = RP:Income.ABNNotQuotedPrimaryProductionPaymentGross.Amount
    
            ^IITR439 = RP:Income.VoluntaryAgreementPrimaryProductionGross.Amount
    
            ^IITR441 = RP:Income.LabourHireArrangementPrimaryProductionPaymentGross.Amount
    
            ^IITR443 = RP:Income.GovernmentIndustryPrimaryProductionPaymentsAssessable.Amount
    
            ^IITR447 = RP:Income.BusinessPrimaryProductionOther.Amount
            */
            assertion = (report.IITR281 != null && (report.IITR436 == null && report.IITR439 == null && report.IITR441 == null && report.IITR443 == null && report.IITR447 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000266",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Primary Production business income required",
                    LongDescription = @"Net primary production income/loss amount is present without total business income from primary production (Gross payments, Assessable government industry payments, Other business income)",
                    Location = "/xbrli:xbrl/tns:Income.PrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000266" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.IITR281) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR436", Value = GetValueOrEmpty(report.IITR436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR439", Value = GetValueOrEmpty(report.IITR439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR441", Value = GetValueOrEmpty(report.IITR441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR443", Value = GetValueOrEmpty(report.IITR443) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR447", Value = GetValueOrEmpty(report.IITR447) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000267()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000267
            Total business income from primary production (Gross payments, Assessable government industry payments, other business income) is present without net primary production income/loss amount.
    
            Legacy Rule Format:
            (^IITR436 <> NULL OR ^IITR439 <> NULL OR ^IITR441 <> NULL OR ^IITR443 <> NULL OR ^IITR447 <> NULL) AND (^IITR281 = NULL)

            Technical Business Rule Format:
            (^IITR436 <> NULL OR ^IITR439 <> NULL OR ^IITR441 <> NULL OR ^IITR443 <> NULL OR ^IITR447 <> NULL) AND (^IITR281 = NULL)
    
            Data Elements:
    
            ^IITR281 = RP:Income.PrimaryProductionNet.Amount
    
            ^IITR436 = RP:Income.ABNNotQuotedPrimaryProductionPaymentGross.Amount
    
            ^IITR439 = RP:Income.VoluntaryAgreementPrimaryProductionGross.Amount
    
            ^IITR441 = RP:Income.LabourHireArrangementPrimaryProductionPaymentGross.Amount
    
            ^IITR443 = RP:Income.GovernmentIndustryPrimaryProductionPaymentsAssessable.Amount
    
            ^IITR447 = RP:Income.BusinessPrimaryProductionOther.Amount
            */
            assertion = ((report.IITR436 != null || report.IITR439 != null || report.IITR441 != null || report.IITR443 != null || report.IITR447 != null) && report.IITR281 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000267",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Primary Production income/loss must be provided",
                    LongDescription = @"Total business income from primary production (Gross payments, Assessable government industry payments, other business income) is present without net primary production income/loss amount",
                    Location = "/xbrli:xbrl/tns:Income.PrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR436", Value = GetValueOrEmpty(report.IITR436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR439", Value = GetValueOrEmpty(report.IITR439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR441", Value = GetValueOrEmpty(report.IITR441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR443", Value = GetValueOrEmpty(report.IITR443) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR447", Value = GetValueOrEmpty(report.IITR447) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.IITR281) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000268
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000268()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000268
            Net non-primary production income/loss amount is present without total business income non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Gross payments voluntary agreement, Gross payments-labour hire or other payments, Assessable government industry payments, Other business income)
    
            Legacy Rule Format:
            (^IITR285 <> NULL) AND (^IITR437 = NULL AND ^IITR438 = NULL AND ^IITR440 = NULL AND ^IITR442 = NULL AND ^IITR445 = NULL AND ^IITR448 = NULL)

            Technical Business Rule Format:
            (^IITR285 <> NULL) AND (^IITR437 = NULL AND ^IITR438 = NULL AND ^IITR440 = NULL AND ^IITR442 = NULL AND ^IITR445 = NULL AND ^IITR448 = NULL)
    
            Data Elements:
    
            ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
    
            ^IITR437 = RP:Income.ABNNotQuotedNonPrimaryProductionPaymentGross.Amount
    
            ^IITR438 = RP:Income.ForeignResidentWithholdingNonPrimaryProduction.Amount
    
            ^IITR440 = RP:Income.VoluntaryAgreementNonPrimaryProductionGross.Amount
    
            ^IITR442 = RP:Income.LabourHireArrangementNonPrimaryProductionPaymentGross.Amount
    
            ^IITR445 = RP:Income.GovernmentIndustryNonPrimaryProductionPaymentsAssessable.Amount
    
            ^IITR448 = RP:Income.BusinessNonPrimaryProductionOther.Amount
            */
            assertion = (report.IITR285 != null && (report.IITR437 == null && report.IITR438 == null && report.IITR440 == null && report.IITR442 == null && report.IITR445 == null && report.IITR448 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001268",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-Primary Production business income required",
                    LongDescription = @"Net non-primary production income/loss amount is present without total business income non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Gross payments voluntary agreement, Gross payments-labour hire or other payments, Assessable government industry payments, Other business income)",
                    Location = "/xbrli:xbrl/tns:Income.NonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000268" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.IITR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR437", Value = GetValueOrEmpty(report.IITR437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR438", Value = GetValueOrEmpty(report.IITR438) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR440", Value = GetValueOrEmpty(report.IITR440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR442", Value = GetValueOrEmpty(report.IITR442) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR445", Value = GetValueOrEmpty(report.IITR445) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR448", Value = GetValueOrEmpty(report.IITR448) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000269
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000269()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000269
            Total business income from non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Gross payments voluntary agreement, Gross payments-labour hire or other payments, Assessable government industry payments, Other business income) is present without net non-primary production income/loss amount
    
            Legacy Rule Format:
            (^IITR285 = NULL) AND (^IITR437 <> NULL OR ^IITR438 <> NULL OR ^IITR440 <> NULL OR ^IITR442 <> NULL OR ^IITR445 <> NULL OR ^IITR448 <> NULL)

            Technical Business Rule Format:
            (^IITR285 = NULL) AND (^IITR437 <> NULL OR ^IITR438 <> NULL OR ^IITR440 <> NULL OR ^IITR442 <> NULL OR ^IITR445 <> NULL OR ^IITR448 <> NULL)
    
            Data Elements:
    
            ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
    
            ^IITR437 = RP:Income.ABNNotQuotedNonPrimaryProductionPaymentGross.Amount
    
            ^IITR438 = RP:Income.ForeignResidentWithholdingNonPrimaryProduction.Amount
    
            ^IITR440 = RP:Income.VoluntaryAgreementNonPrimaryProductionGross.Amount
    
            ^IITR442 = RP:Income.LabourHireArrangementNonPrimaryProductionPaymentGross.Amount
    
            ^IITR445 = RP:Income.GovernmentIndustryNonPrimaryProductionPaymentsAssessable.Amount
    
            ^IITR448 = RP:Income.BusinessNonPrimaryProductionOther.Amount
            */
            assertion = (report.IITR285 == null && (report.IITR437 != null || report.IITR438 != null || report.IITR440 != null || report.IITR442 != null || report.IITR445 != null || report.IITR448 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001269",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-Primary Production income/loss is required",
                    LongDescription = @"Total business income from non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Gross payments voluntary agreement, Gross payments-labour hire or other payments, Assessable government industry payments, Other business income) is present without net non-primary production income/loss amount",
                    Location = "/xbrli:xbrl/tns:Income.NonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000269" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.IITR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR437", Value = GetValueOrEmpty(report.IITR437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR438", Value = GetValueOrEmpty(report.IITR438) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR440", Value = GetValueOrEmpty(report.IITR440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR442", Value = GetValueOrEmpty(report.IITR442) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR445", Value = GetValueOrEmpty(report.IITR445) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR448", Value = GetValueOrEmpty(report.IITR448) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000270
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000270()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000270
            When an amount has been provided for 'Net income/loss from business tax withheld voluntary agreement', 'Net income/loss from business tax withheld where Australian business number not quoted', 'Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains)', or 'Net income/loss from business tax withheld - labour hire or other specified payments', an amount must be provided for 'Net Primary Production income/loss' or 'Net non-Primary Production income/loss'
    
            Legacy Rule Format:
            (^IITR286 > 0 OR ^IITR287 > 0 OR ^IITR288 > 0 OR ^IITR289 > 0) AND (^IITR281 = NULL AND ^IITR285 = NULL)

            Technical Business Rule Format:
            (^IITR286 > 0 OR ^IITR287 > 0 OR ^IITR288 > 0 OR ^IITR289 > 0) AND (^IITR281 = NULL AND ^IITR285 = NULL)
    
            Data Elements:
    
            ^IITR281 = RP:Income.PrimaryProductionNet.Amount
    
            ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
    
            ^IITR286 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount
    
            ^IITR287 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount
    
            ^IITR288 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount
    
            ^IITR289 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount
            */
            assertion = ((report.IITR286.GetValueOrDefault() > 0 || report.IITR287.GetValueOrDefault() > 0 || report.IITR288.GetValueOrDefault() > 0 || report.IITR289.GetValueOrDefault() > 0) && (report.IITR281 == null && report.IITR285 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001270",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income or loss from business is required",
                    LongDescription = @"When an amount has been provided for 'Net income/loss from business tax withheld voluntary agreement', 'Net income/loss from business tax withheld where Australian business number not quoted', 'Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains)', or 'Net income/loss from business tax withheld - labour hire or other specified payments', an amount must be provided for 'Net Primary Production income/loss' or 'Net non-Primary Production income/loss'",
                    Location = "/xbrli:xbrl/tns:Income.PrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000270" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR286", Value = GetValueOrEmpty(report.IITR286) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR287", Value = GetValueOrEmpty(report.IITR287) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR288", Value = GetValueOrEmpty(report.IITR288) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR289", Value = GetValueOrEmpty(report.IITR289) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.IITR281) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.IITR285) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000271
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000271()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000271
            Taxpayer's Net farm management deposits or repayments must equal the sum of Early repayments - natural disaster and drought and Other repayments, less Deductible deposits
    
            Legacy Rule Format:
            ^IITR306 <> NULL AND ^IITR306 <> (^IITR304 + ^IITR305 - ^IITR302)

            Technical Business Rule Format:
            ^IITR306 <> NULL AND ^IITR306 <> (^IITR304 + ^IITR305 - ^IITR302)
    
            Data Elements:
    
            ^IITR306 = RP:Income.FarmManagementDepositsOrRepaymentsNet.Amount
    
            ^IITR302 = RP:IncomeTax.Deduction.FarmManagementDeposits.Amount
    
            ^IITR304 = RP:Income.FarmManagementEarlyRepaymentsNaturalDisaster.Amount
    
            ^IITR305 = RP:Income.FarmManagementOtherRepayments.Amount
            */
            assertion = (report.IITR306 != null && report.IITR306.GetValueOrDefault() != report.IITR304.GetValueOrDefault() + report.IITR305.GetValueOrDefault() - report.IITR302.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.021002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net farm management deposits or repayments is incorrect",
                    LongDescription = @"Taxpayer's Net farm management deposits or repayments must equal the sum of Early repayments - natural disaster and drought and Other repayments, less Deductible deposits",
                    Location = "/xbrli:xbrl/tns:Income.FarmManagementDepositsOrRepaymentsNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000271" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR306", Value = GetValueOrEmpty(report.IITR306) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR304", Value = GetValueOrEmpty(report.IITR304) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR305", Value = GetValueOrEmpty(report.IITR305) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR302", Value = GetValueOrEmpty(report.IITR302) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000272
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000272()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000272
            Landcare operations expenses has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR260, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR260, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR260 = RP:Expense.LandcareOperationsNonPrimaryProduction.Amount
            */
            assertion = NotMonetary(report.IITR260, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000272",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Landcare operations expenses is not in a valid monetary format",
                    LongDescription = @"Landcare operations expenses must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.LandcareOperationsNonPrimaryProduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000272" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR260", Value = GetValueOrEmpty(report.IITR260) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000277
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000277()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000277
            There is an amount entered in Deductible deposits and it should not exceed taxable primary production income.
    
            Legacy Rule Format:
            ^IITR302 > 0 AND ^IITR302 > (^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR281 + ^IITR299 + ^IITR304 + ^IITR305)

            Technical Business Rule Format:
            ^IITR302 > 0 AND ^IITR302 > (^IITR242 + ^IITR243 - ^IITR245 - ^IITR246 + ^IITR281 + ^IITR299 + ^IITR304 + ^IITR305)
    
            Data Elements:
    
            ^IITR302 = RP:IncomeTax.Deduction.FarmManagementDeposits.Amount
    
            ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
            ^IITR245 = RP.Partnership:Expense.LandcareOperationsAcceleratedDepreciation.Amount
    
            ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
    
            ^IITR281 = RP:Income.PrimaryProductionNet.Amount
    
            ^IITR299 = RP:Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount
    
            ^IITR304 = RP:Income.FarmManagementEarlyRepaymentsNaturalDisaster.Amount
    
            ^IITR305 = RP:Income.FarmManagementOtherRepayments.Amount
            */
            assertion = (report.IITR302.GetValueOrDefault() > 0 && report.IITR302.GetValueOrDefault() > report.IITR242.GetValueOrDefault() + report.IITR243.GetValueOrDefault() - report.IITR245.GetValueOrDefault() - report.IITR246.GetValueOrDefault() + report.IITR281.GetValueOrDefault() + report.IITR299.GetValueOrDefault() + report.IITR304.GetValueOrDefault() + report.IITR305.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.021001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Deductible deposits amount has exceeded the taxable primary production amount",
                    LongDescription = @"When an amount has been entered for 'Deductible deposits', it must not exceed the total amount of taxable primary production income. Taxable primary production income is the sum of 'Distribution from partnerships', 'Share of net income from trusts', 'Net Primary Production income/loss', 'Primary production deferred losses', 'Early repayments - natural disaster and drought' and 'Other repayments', less 'Landcare operations and deduction for decline in value of water facility' and 'Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts'.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.FarmManagementDeposits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000277" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR302", Value = GetValueOrEmpty(report.IITR302) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR242", Value = GetValueOrEmpty(report.IITR242) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR245", Value = GetValueOrEmpty(report.IITR245) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR246", Value = GetValueOrEmpty(report.IITR246) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.IITR281) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR299", Value = GetValueOrEmpty(report.IITR299) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR304", Value = GetValueOrEmpty(report.IITR304) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR305", Value = GetValueOrEmpty(report.IITR305) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000280
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000280()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000280
            Addition of Distribution from partnerships relating to financial investments, less foreign income plus Share of net rental property income or loss from partnerships plus Other distributions from partnerships is not equal to the amount of Distribution from partnerships less foreign income.
    
            Legacy Rule Format:
            ((^IITR252 <> NULL) AND (^IITR252 <> (^IITR249 + ^IITR250 + ^IITR251))) OR ((^IITR252 = NULL) AND ((^IITR249 + ^IITR250 + ^IITR251) <> 0))

            Technical Business Rule Format:
            ((^IITR252 <> NULL) AND (^IITR252 <> (^IITR249 + ^IITR250 + ^IITR251))) OR ((^IITR252 = NULL) AND ((^IITR249 + ^IITR250 + ^IITR251) <> 0))
    
            Data Elements:
    
            ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR249 = RP:Income.PartnershipInvestmentLessForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR250 = RP:Income.PartnershipRentalNonPrimaryProductionShare.Amount
    
            ^IITR251 = RP:Income.PartnershipNonPrimaryProductionOtherShare.Amount
            */
            assertion = (report.IITR252 != null && report.IITR252.GetValueOrDefault() != report.IITR249.GetValueOrDefault() + report.IITR250.GetValueOrDefault() + report.IITR251.GetValueOrDefault() || report.IITR252 == null && report.IITR249.GetValueOrDefault() + report.IITR250.GetValueOrDefault() + report.IITR251.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000280",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from partnerships less foreign income amount is incorrect",
                    LongDescription = @"The sum of 'Distribution from partnerships relating to financial investments, less foreign income', 'Share of net rental property income or loss from partnerships', 'Other distributions from partnerships' must equal the amount for 'Distribution from partnerships less foreign income'",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000280" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR252", Value = GetValueOrEmpty(report.IITR252) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR249", Value = GetValueOrEmpty(report.IITR249) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR250", Value = GetValueOrEmpty(report.IITR250) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR251", Value = GetValueOrEmpty(report.IITR251) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000283
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000283()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000283
            Addition of Managed investment scheme deductions relating to amounts shown at "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts" and Partnership deductions relating to financial investment amounts shown at "Distribution from partnerships less foreign income" and Partnership deductions relating to rental property income or loss shown at "Distribution from partnerships less foreign income" and Other deductions relating to distributions shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts" is not equal to the amount of Other deductions relating to amounts shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts".
    
            Legacy Rule Format:
            (^IITR265 <> NULL AND (^IITR265 <> (^IITR261 + ^IITR262 + ^IITR263 + ^IITR264))) OR (^IITR265 = NULL AND ((^IITR261 + ^IITR262 + ^IITR263 + ^IITR264) <> 0))

            Technical Business Rule Format:
            (^IITR265 <> NULL AND (^IITR265 <> (^IITR261 + ^IITR262 + ^IITR263 + ^IITR264))) OR (^IITR265 = NULL AND ((^IITR261 + ^IITR262 + ^IITR263 + ^IITR264) <> 0))
    
            Data Elements:
    
            ^IITR265 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
    
            ^IITR261 = RP:IncomeTax.Deduction.TrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProduction.Amount
    
            ^IITR262 = RP:IncomeTax.Deduction.PartnershipExcludingForeignIncomeNonPrimaryProduction.Amount
    
            ^IITR263 = RP:IncomeTax.Deduction.PartnershipRentalNonPrimaryProduction.Amount
    
            ^IITR264 = RP:IncomeTax.Deduction.TrustsPartnershipsOtherAllowableNonPrimaryProduction.Amount
            */
            assertion = (report.IITR265 != null && report.IITR265.GetValueOrDefault() != report.IITR261.GetValueOrDefault() + report.IITR262.GetValueOrDefault() + report.IITR263.GetValueOrDefault() + report.IITR264.GetValueOrDefault() || report.IITR265 == null && report.IITR261.GetValueOrDefault() + report.IITR262.GetValueOrDefault() + report.IITR263.GetValueOrDefault() + report.IITR264.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000283",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other deductions relating to amounts shown is incorrect",
                    LongDescription = @"Addition of Managed investment scheme deductions relating to amounts shown at ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts"" and Partnership deductions relating to financial investment amounts shown at ""Distribution from partnerships less foreign income"" and Partnership deductions relating to rental property income or loss shown at ""Distribution from partnerships less foreign income"" and Other deductions relating to distributions shown at ""Distribution from partnerships less foreign income"", ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts"" is not equal to the amount of Other deductions relating to amounts shown at ""Distribution from partnerships less foreign income"", ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts""",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000283" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR265", Value = GetValueOrEmpty(report.IITR265) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR261", Value = GetValueOrEmpty(report.IITR261) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR262", Value = GetValueOrEmpty(report.IITR262) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR263", Value = GetValueOrEmpty(report.IITR263) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR264", Value = GetValueOrEmpty(report.IITR264) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000284
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000284()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000284
            The sum of Share of net income from trusts plus Share of net income from trusts less capital gains, foreign income and franked distributions plus Franked distributions from trusts plus Total current year capital gains plus foreign source income is less than zero which indicates that an overall trust loss is being distributed from a trust to an individual. Adjust distributed amounts at Share of net income from trusts and Share of net income from trusts less capital gains, foreign income and franked distributions until the overall negative amount no longer exists (i.e. make the sum equal to zero). Please note, where there is a negative amount at any of the foreign source income fields, the amount of zero will be used in that field.
    
            Legacy Rule Format:
            (^IITR243 + ^IITR255 + ^IITR259 + ^IITR312 + ConditionalValue(^IITR321 > 0,^IITR321,0) + ConditionalValue(^IITR322 > 0,^IITR322,0) + ConditionalValue(^IITR323 > 0, ^IITR323,0) + ConditionalValue(^IITR324 > 0, ^IITR324,0) + ConditionalValue(^IITR327 > 0, ^IITR327,0) + ^IITR328 + ConditionalValue(^IITR329 > 0, ^IITR329,0) < 0)

            Technical Business Rule Format:
            (^IITR243 + ^IITR255 + ^IITR259 + ^IITR312 + ConditionalValue(^IITR321 > 0,^IITR321,0) + ConditionalValue(^IITR322 > 0,^IITR322,0) + ConditionalValue(^IITR323 > 0, ^IITR323,0) + ConditionalValue(^IITR324 > 0, ^IITR324,0) + ConditionalValue(^IITR327 > 0, ^IITR327,0) + ^IITR328 + ConditionalValue(^IITR329 > 0, ^IITR329,0) < 0)
    
            Data Elements:
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
    
            ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount
    
            ^IITR312 = RP:Capital.Gains.Total.Amount
    
            ^IITR321 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount
    
            ^IITR322 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeWithoutUndeductedPurchasePrice.Amount
    
            ^IITR323 = RP:InternationalDealings.ForeignPensionOrAnnuityIncomeUndeductedPurchasePrice.Amount
    
            ^IITR324 = RP:InternationalDealings.ForeignRent.Amount
    
            ^IITR327 = RP:Income.InternationalDealings.Net.Amount
    
            ^IITR328 = RP:IncomeTax.FrankingCreditsNewZealand.Amount
    
            ^IITR329 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
            */
            assertion = (report.IITR243.GetValueOrDefault() + report.IITR255.GetValueOrDefault() + report.IITR259.GetValueOrDefault() + report.IITR312.GetValueOrDefault() + ConditionalValue(report.IITR321.GetValueOrDefault() > 0, report.IITR321.GetValueOrDefault(), 0) + ConditionalValue(report.IITR322.GetValueOrDefault() > 0, report.IITR322.GetValueOrDefault(), 0) + ConditionalValue(report.IITR323.GetValueOrDefault() > 0, report.IITR323.GetValueOrDefault(), 0) + ConditionalValue(report.IITR324.GetValueOrDefault() > 0, report.IITR324.GetValueOrDefault(), 0) + ConditionalValue(report.IITR327.GetValueOrDefault() > 0, report.IITR327.GetValueOrDefault(), 0) + report.IITR328.GetValueOrDefault() + ConditionalValue(report.IITR329.GetValueOrDefault() > 0, report.IITR329.GetValueOrDefault(), 0) < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000284",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An overall trust loss cannot be distributed to an individual",
                    LongDescription = @"When the sum of 'Share of net income from trusts', 'Share of net income from trusts less capital gains, foreign income and franked distributions', 'Franked distributions from trusts', 'Total current year capital gains', 'Other net foreign employment income', 'Net foreign pension or annuity income without an undeducted purchase price', 'Net foreign pension or annuity income with an undeducted purchase price', 'Net foreign rent', 'Other net foreign source income', 'Australian franking credits from a New Zealand franking company', and 'Net foreign employment income - payment summary' is negative, which indicates that an overall trust loss is being distributed from a trust to an individual, adjust distributed amounts at 'Share of net income from trusts' and 'Share of net income from trusts less capital gains, foreign income and franked distributions' until the overall negative amount no longer exists (i.e. make the sum equal to zero). Note: Where there is a negative amount at any of the foreign source income fields, an amount of zero is used in the summation of that field.",
                    Location = "/xbrli:xbrl/tns:Income.TrustPrimaryProductionNetShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000284" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.IITR255) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR259", Value = GetValueOrEmpty(report.IITR259) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR312", Value = GetValueOrEmpty(report.IITR312) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR321", Value = GetValueOrEmpty(report.IITR321) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR322", Value = GetValueOrEmpty(report.IITR322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR323", Value = GetValueOrEmpty(report.IITR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR324", Value = GetValueOrEmpty(report.IITR324) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR327", Value = GetValueOrEmpty(report.IITR327) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR328", Value = GetValueOrEmpty(report.IITR328) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR329", Value = GetValueOrEmpty(report.IITR329) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000294
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000294()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000294
            Main salary or wage occupation description has a maximum length of 100
    
            Legacy Rule Format:
            Length(^IITR58) > 100

            Technical Business Rule Format:
            Length(^IITR58) > 100
    
            Data Elements:
    
            ^IITR58 = RP:PersonDemographicDetails.Occupation.Description
            */
            assertion = (Length(report.IITR58) > 100);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000294",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Main salary or wage occupation description must not exceed 100 characters",
                    LongDescription = @"For Main salary or wage occupation description, the maximum allowable characters for this form is 100",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000294" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR58", Value = GetValueOrEmpty(report.IITR58) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000295
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000295()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000295
            Individual total income or loss amount is mandatory
    
            Legacy Rule Format:
            ^IITR131 = NULL

            Technical Business Rule Format:
            ^IITR131 = NULL
    
            Data Elements:
    
            ^IITR131 = RP:Income.Net.Amount
            */
            assertion = (report.IITR131 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000295",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual total income or loss amount must be present",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000295" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR131", Value = GetValueOrEmpty(report.IITR131) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000300
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000300()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000300
            Amount of allowances, earnings, tips, director's fees, etc has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR65, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR65, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR65 = RP:Income.AllowancesEarningsTipsDirectorsFees.Amount
            */
            assertion = NotMonetary(report.IITR65, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000300",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of allowances, earnings, tips, director's fees, etc is not in a valid monetary format",
                    LongDescription = @"Amount of allowances, earnings, tips, director's fees, etc must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.AllowancesEarningsTipsDirectorsFees.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000300" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR65", Value = GetValueOrEmpty(report.IITR65) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000309
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000309()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000309
            Attributed personal services income has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR112, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR112, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR112 = RP:Income.PersonalServicesIncome.Total.Amount
            */
            assertion = NotMonetary(report.IITR112, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000309",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed personal services income is not in a valid monetary format",
                    LongDescription = @"Attributed personal services income must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncome.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000309" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR112", Value = GetValueOrEmpty(report.IITR112) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000312
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000312()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000312
            Australian Government allowances and payments like Newstart, youth allowance and austudy payment has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR86, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR86, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR86 = RP:Income.AustralianGovernmentAllowancesAndPayments.Amount
            */
            assertion = NotMonetary(report.IITR86, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000312",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Australian Government allowances and payments..."" is not in a valid monetary format",
                    LongDescription = @"Australian Government allowances and payments like Newstart, youth allowance and austudy payment must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.AustralianGovernmentAllowancesAndPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000312" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR86", Value = GetValueOrEmpty(report.IITR86) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000314
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000314()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000314
            Amount of Australian Government pensions and allowances has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR89, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR89, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR89 = RP:Pension.Total.Amount
            */
            assertion = NotMonetary(report.IITR89, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of Australian Government pensions and allowances is not in a valid monetary format",
                    LongDescription = @"Amount of Australian Government pensions and allowances must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Pension.Total.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000314" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR89", Value = GetValueOrEmpty(report.IITR89) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000316
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000316()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000316
            Australian annuities and superannuation income streams Taxable component "Taxed element" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR92, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR92, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR92 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount
            */
            assertion = NotMonetary(report.IITR92, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000316",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable ""Taxed element"" of Australian annuities and superannuation is not in a valid monetary format",
                    LongDescription = @"Australian annuities and superannuation income streams Taxable component ""Taxed element"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000316" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR92", Value = GetValueOrEmpty(report.IITR92) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000317
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000317()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000317
            Australian annuities and superannuation income streams Taxable component "Untaxed element" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR93, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR93, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR93 = RP:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount
            */
            assertion = NotMonetary(report.IITR93, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000317",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable ""Untaxed element"" of Australian annuities and superannuation is not in a valid monetary format",
                    LongDescription = @"Australian annuities and superannuation income streams Taxable component ""Untaxed element"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000317" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR93", Value = GetValueOrEmpty(report.IITR93) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000318
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000318()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000318
            Australian annuities and superannuation income streams Lump sum in arrears "Taxed element" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR94, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR94, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR94 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
            */
            assertion = NotMonetary(report.IITR94, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000318",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Lump sum ""Taxed element"" of Australian annuities and superannuation is not in a valid monetary format",
                    LongDescription = @"Australian annuities and superannuation income streams Lump sum in arrears ""Taxed element"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000318" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR94", Value = GetValueOrEmpty(report.IITR94) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000319
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000319()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000319
            Australian annuities and superannuation income streams Lump sum in arrears "Untaxed element" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR95, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR95, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR95 = RP:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
            */
            assertion = NotMonetary(report.IITR95, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000319",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Lump sum ""Untaxed element"" of Australian annuities and superannuation is not in a valid monetary format",
                    LongDescription = @"Australian annuities and superannuation income streams Lump sum in arrears ""Untaxed element"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000319" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR95", Value = GetValueOrEmpty(report.IITR95) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000320
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000320()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000320
            Gross interest has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR115, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR115, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR115 = RP:Income.Interest.Gross.Amount
            */
            assertion = NotMonetary(report.IITR115, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000320",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest is not in a valid monetary format",
                    LongDescription = @"Gross interest must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Interest.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000320" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR115", Value = GetValueOrEmpty(report.IITR115) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000321
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000321()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000321
            TFN amounts withheld from gross interest has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR116, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR116, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR116 = RP:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            */
            assertion = NotMonetary(report.IITR116, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000321",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from gross interest is not in a valid monetary format",
                    LongDescription = @"TFN amounts withheld from gross interest must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000321" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR116", Value = GetValueOrEmpty(report.IITR116) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000322
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000322()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000322
            Dividends - Unfranked amount has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR118, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR118, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR118 = RP:Income.DividendsUnfranked.Amount
            */
            assertion = NotMonetary(report.IITR118, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000322",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends - Unfranked amount is not in a valid monetary format",
                    LongDescription = @"Dividends - Unfranked amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.DividendsUnfranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000322" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = GetValueOrEmpty(report.IITR118) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000323
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000323()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000323
            Dividends - Franked amount has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR119, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR119, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR119 = RP:Income.DividendsFranked.Amount
            */
            assertion = NotMonetary(report.IITR119, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000323",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends - Franked amount is not in a valid monetary format",
                    LongDescription = @"Dividends - Franked amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.DividendsFranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000323" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = GetValueOrEmpty(report.IITR119) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000324
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000324()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000324
            Dividends - Franking credit has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR120, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR120, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
            */
            assertion = NotMonetary(report.IITR120, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000324",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends - Franking credit is not in a valid monetary format",
                    LongDescription = @"Dividends - Franking credit must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000324" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.IITR120) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000325
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000325()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000325
            TFN amounts withheld from dividends has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR121, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR121, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR121 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            */
            assertion = NotMonetary(report.IITR121, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000325",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends is not in a valid monetary format",
                    LongDescription = @"TFN amounts withheld from dividends must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000325" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR121", Value = GetValueOrEmpty(report.IITR121) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000326
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000326()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000326
            Discount from taxed upfront schemes - eligible for reduction has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR123, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR123, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR123 = RP:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
            */
            assertion = NotMonetary(report.IITR123, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000326",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Discount from taxed upfront schemes - eligible for reduction is not in a valid monetary format",
                    LongDescription = @"Discount from taxed upfront schemes - eligible for reduction must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000326" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR123", Value = GetValueOrEmpty(report.IITR123) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000327
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000327()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000327
            Discount from taxed upfront schemes - not eligible for reduction has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR124, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR124, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR124 = RP:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
            */
            assertion = NotMonetary(report.IITR124, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000327",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Discount from taxed upfront schemes - not eligible for reduction is not in a valid monetary format",
                    LongDescription = @"Discount from taxed upfront schemes - not eligible for reduction must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000327" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR124", Value = GetValueOrEmpty(report.IITR124) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000329
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000329()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000329
            Total Assessable discount amount has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR127, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR127, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR127 = RP:Income.EmployeeShareScheme.DiscountAssessable.Amount
            */
            assertion = NotMonetary(report.IITR127, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000329",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Assessable discount amount is not in a valid monetary format",
                    LongDescription = @"Total Assessable discount amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.DiscountAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000329" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR127", Value = GetValueOrEmpty(report.IITR127) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000330
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000330()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000330
            ESS TFN amounts withheld from discounts has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR128, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR128, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR128 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount
            */
            assertion = NotMonetary(report.IITR128, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000330",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ESS TFN amounts withheld from discounts is not in a valid monetary format",
                    LongDescription = @"ESS TFN amounts withheld from discounts must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000330" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR128", Value = GetValueOrEmpty(report.IITR128) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000331
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000331()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000331
            Foreign source discounts has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR129, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR129, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR129 = RP:Income.EmployeeShareScheme.ForeignSourceDiscount.Amount
            */
            assertion = NotMonetary(report.IITR129, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000331",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign source discounts is not in a valid monetary format",
                    LongDescription = @"Foreign source discounts must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.EmployeeShareScheme.ForeignSourceDiscount.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000331" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR129", Value = GetValueOrEmpty(report.IITR129) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000332
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000332()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000332
            Individual total income or loss amount has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR131, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR131, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR131 = RP:Income.Net.Amount
            */
            assertion = NotMonetary(report.IITR131, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000332",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual total income or loss amount is not in a valid monetary format",
                    LongDescription = @"Individual total income or loss amount may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000332" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR131", Value = GetValueOrEmpty(report.IITR131) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000333
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000333()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000333
            Distribution from partnerships has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR242, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR242, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR242 = RP:Income.PartnershipPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR242, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000333",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from partnerships is not in a valid monetary format",
                    LongDescription = @"Distribution from partnerships may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000333" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR242", Value = GetValueOrEmpty(report.IITR242) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000334
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000334()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000334
            Share of net income from trusts has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR243, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR243, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR243 = RP:Income.TrustPrimaryProductionNetShare.Amount
            */
            assertion = NotMonetary(report.IITR243, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000334",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts is not in a valid monetary format",
                    LongDescription = @"Share of net income from trusts may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.TrustPrimaryProductionNetShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000334" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR243", Value = GetValueOrEmpty(report.IITR243) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000336
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000336()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000336
            Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR246, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR246, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR246 = RP:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR246, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000336",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Deductions relating to amounts shown at Distribution from partnerships..."" is not in a valid monetary format",
                    LongDescription = @"Deductions relating to amounts shown at Distribution from partnerships and Share of net income from trusts must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000336" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR246", Value = GetValueOrEmpty(report.IITR246) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000337
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000337()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000337
            Distribution from partnerships relating to financial investments, less foreign income has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR249, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR249, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR249 = RP:Income.PartnershipInvestmentLessForeignIncomeNonPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR249, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000337",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Distribution from partnerships relating to financial investments..."" is not in a valid monetary format",
                    LongDescription = @"Distribution from partnerships relating to financial investments, less foreign income may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipInvestmentLessForeignIncomeNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000337" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR249", Value = GetValueOrEmpty(report.IITR249) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000338
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000338()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000338
            Share of net rental property income or loss from partnerships has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR250, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR250, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR250 = RP:Income.PartnershipRentalNonPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR250, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000338",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net rental property income or loss from partnerships is not in a valid monetary format",
                    LongDescription = @"Share of net rental property income or loss from partnerships may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipRentalNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000338" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR250", Value = GetValueOrEmpty(report.IITR250) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000339
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000339()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000339
            Other distributions from partnerships has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR251, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR251, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR251 = RP:Income.PartnershipNonPrimaryProductionOtherShare.Amount
            */
            assertion = NotMonetary(report.IITR251, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000339",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other distributions from partnerships is not in a valid monetary format",
                    LongDescription = @"Other distributions from partnerships may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipNonPrimaryProductionOtherShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000339" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR251", Value = GetValueOrEmpty(report.IITR251) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000340
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000340()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000340
            Distribution from partnerships less foreign income has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR252, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR252, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR252 = RP:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR252, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000340",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from partnerships less foreign income is not in a valid monetary format",
                    LongDescription = @"Distribution from partnerships less foreign income may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000340" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR252", Value = GetValueOrEmpty(report.IITR252) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000343
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000343()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000343
            Share of net income from trusts less capital gains, foreign income and franked distributions has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR255, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR255, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR255 = RP:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR255, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000343",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Share of net income from trusts less... franked distributions"" is not in a valid monetary format",
                    LongDescription = @"Share of net income from trusts less capital gains, foreign income and franked distributions may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000343" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR255", Value = GetValueOrEmpty(report.IITR255) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000346
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000346()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000346
            Franked distributions from trusts has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR259, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR259, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR259 = RP:Income.TrustFrankedNonPrimaryProductionShare.Amount
            */
            assertion = NotMonetary(report.IITR259, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000346",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franked distributions from trusts is not in a valid monetary format",
                    LongDescription = @"Franked distributions from trusts must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.TrustFrankedNonPrimaryProductionShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000346" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR259", Value = GetValueOrEmpty(report.IITR259) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000347
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000347()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000347
            Managed investment scheme deductions relating to amounts shown at "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR261, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR261, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR261 = RP:IncomeTax.Deduction.TrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProduction.Amount
            */
            assertion = NotMonetary(report.IITR261, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000347",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Managed investment scheme deductions relating to amounts shown is not in a valid monetary format",
                    LongDescription = @"Managed investment scheme deductions relating to amounts shown at ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000347" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR261", Value = GetValueOrEmpty(report.IITR261) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000348
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000348()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000348
            Partnership deductions relating to financial investment amounts shown at "Distribution from partnerships less foreign income" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR262, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR262, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR262 = RP:IncomeTax.Deduction.PartnershipExcludingForeignIncomeNonPrimaryProduction.Amount
            */
            assertion = NotMonetary(report.IITR262, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000348",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Partnership deductions relating to financial investment amounts..."" is not in a valid monetary format",
                    LongDescription = @"Partnership deductions relating to financial investment amounts shown at ""Distribution from partnerships less foreign income"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.PartnershipExcludingForeignIncomeNonPrimaryProduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000348" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR262", Value = GetValueOrEmpty(report.IITR262) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000349
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000349()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000349
            Partnership deductions relating to rental property income or loss shown at "Distribution from partnerships less foreign income" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR263, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR263, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR263 = RP:IncomeTax.Deduction.PartnershipRentalNonPrimaryProduction.Amount
            */
            assertion = NotMonetary(report.IITR263, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000349",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Partnership deductions relating to rental property income..."" is not in a valid monetary format",
                    LongDescription = @"Partnership deductions relating to rental property income or loss shown at ""Distribution from partnerships less foreign income"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.PartnershipRentalNonPrimaryProduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000349" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR263", Value = GetValueOrEmpty(report.IITR263) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000350
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000350()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000350
            Other deductions relating to distributions shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR264, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR264, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR264 = RP:IncomeTax.Deduction.TrustsPartnershipsOtherAllowableNonPrimaryProduction.Amount
            */
            assertion = NotMonetary(report.IITR264, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000350",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Other deductions relating to distributions shown at..."" is not in a valid monetary format",
                    LongDescription = @"Other deductions relating to distributions shown at ""Distribution from partnerships less foreign income"", ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TrustsPartnershipsOtherAllowableNonPrimaryProduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000350" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR264", Value = GetValueOrEmpty(report.IITR264) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000351
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000351()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000351
            Other deductions relating to amounts shown at "Distribution from partnerships less foreign income", "Share of net income from trusts less capital gains, foreign income and franked distributions" and "Franked distributions from trusts" has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR265, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR265, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR265 = RP:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
            */
            assertion = NotMonetary(report.IITR265, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000351",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Other deductions relating to amounts shown at..."" is not in a valid monetary format",
                    LongDescription = @"Other deductions relating to amounts shown at ""Distribution from partnerships less foreign income"", ""Share of net income from trusts less capital gains, foreign income and franked distributions"" and ""Franked distributions from trusts"" must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000351" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR265", Value = GetValueOrEmpty(report.IITR265) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000353
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000353()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000353
            Share of credit for tax withheld where Australian business number not quoted has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR268, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR268, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR268 = RP:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            */
            assertion = NotMonetary(report.IITR268, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000353",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of credit for tax withheld where ABN not quoted is not in a valid monetary format",
                    LongDescription = @"Share of credit for tax withheld where Australian business number not quoted must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000353" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR268", Value = GetValueOrEmpty(report.IITR268) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000354
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000354()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000354
            Share of franking credit from franked dividends has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR269, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR269, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR269 = RP:IncomeTax.FrankingCreditsShare.Amount
            */
            assertion = NotMonetary(report.IITR269, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000354",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of franking credit from franked dividends is not in a valid monetary format",
                    LongDescription = @"Share of franking credit from franked dividends must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCreditsShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000354" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR269", Value = GetValueOrEmpty(report.IITR269) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000355
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000355()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000355
            Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR270, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR270, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR270 = RP:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
            */
            assertion = NotMonetary(report.IITR270, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Share of credit for TFN amounts withheld from interest..."" is not in a valid monetary format",
                    LongDescription = @"Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000355" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR270", Value = GetValueOrEmpty(report.IITR270) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000356
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000356()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000356
            Credit for TFN amounts withheld from payments from closely held trusts has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR271, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR271, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR271 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount
            */
            assertion = NotMonetary(report.IITR271, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000356",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for TFN amounts withheld from payments from closely held trusts is not in a valid monetary format",
                    LongDescription = @"Credit for TFN amounts withheld from payments from closely held trusts must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000356" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR271", Value = GetValueOrEmpty(report.IITR271) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000357
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000357()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000357
            Share of credit for tax paid by trustee has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR272, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR272, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR272 = RP:IncomeTax.CreditTrusteeTaxPaidShare.Amount
            */
            assertion = NotMonetary(report.IITR272, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000357",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of credit for tax paid by trustee is not in a valid monetary format",
                    LongDescription = @"Share of credit for tax paid by trustee must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.CreditTrusteeTaxPaidShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000357" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR272", Value = GetValueOrEmpty(report.IITR272) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000358
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000358()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000358
            Share of credit for foreign resident withholding amounts (excluding capital gains) has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR273, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR273, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR273 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            */
            assertion = NotMonetary(report.IITR273, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001358",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of credit for foreign resident withholding amounts (excluding capital gains) is not in a valid monetary format",
                    LongDescription = @"Share of credit for foreign resident withholding amounts (excluding capital gains) must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000358" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR273", Value = GetValueOrEmpty(report.IITR273) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000359
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000359()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000359
            Share of National rental affordability scheme tax offset has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR274, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR274, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR274 = RP:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            */
            assertion = NotMonetary(report.IITR274, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000359",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of National rental affordability scheme tax offset is not in a valid monetary format",
                    LongDescription = @"Share of National rental affordability scheme tax offset must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000359" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR274", Value = GetValueOrEmpty(report.IITR274) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000360
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000360()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000360
            Tax withheld - voluntary agreement has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR276, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR276, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR276 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount
            */
            assertion = NotMonetary(report.IITR276, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000360",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld - voluntary agreement is not in a valid monetary format",
                    LongDescription = @"Tax withheld - voluntary agreement must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000360" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR276", Value = GetValueOrEmpty(report.IITR276) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000361
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000361()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000361
            Tax withheld where Australian business number not quoted has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR277, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR277, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR277 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount
            */
            assertion = NotMonetary(report.IITR277, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000361",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where Australian business number not quoted is not in a valid monetary format",
                    LongDescription = @"Tax withheld where Australian business number not quoted must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000361" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR277", Value = GetValueOrEmpty(report.IITR277) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000362
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000362()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000362
            Tax withheld - labour hire or other specified payments has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR278, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR278, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR278 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount
            */
            assertion = NotMonetary(report.IITR278, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000362",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld - labour hire or other specified payments is not in a valid monetary format",
                    LongDescription = @"Tax withheld - labour hire or other specified payments must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000362" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR278", Value = GetValueOrEmpty(report.IITR278) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000363
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000363()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000363
            Net Personal Services Income has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR279, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR279, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR279 = RP:Income.PersonalServicesIncome.Net.Amount
            */
            assertion = NotMonetary(report.IITR279, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000363",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Personal Services Income is not in a valid monetary format",
                    LongDescription = @"Net Personal Services Income may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncome.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000363" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR279", Value = GetValueOrEmpty(report.IITR279) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000364
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000364()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000364
            Net Primary Production income/loss has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR281, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR281, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR281 = RP:Income.PrimaryProductionNet.Amount
            */
            assertion = NotMonetary(report.IITR281, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000364",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Primary Production income/loss is not in a valid monetary format",
                    LongDescription = @"Net Primary Production income/loss may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000364" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR281", Value = GetValueOrEmpty(report.IITR281) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000365
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000365()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000365
            Net income or loss from carrying on a business of investing has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR282, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR282, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR282 = RP:Income.InvestmentNonPrimaryProductionNet.Amount
            */
            assertion = NotMonetary(report.IITR282, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000365",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income or loss from carrying on a business of investing is not in a valid monetary format",
                    LongDescription = @"Net income or loss from carrying on a business of investing may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.InvestmentNonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000365" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR282", Value = GetValueOrEmpty(report.IITR282) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000366
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000366()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000366
            Net income or loss from carrying on a rental property business has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR283, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR283, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR283 = RP:Income.RentalPropertyNonPrimaryProductionNet.Amount
            */
            assertion = NotMonetary(report.IITR283, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000366",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income or loss from carrying on a rental property business is not in a valid monetary format",
                    LongDescription = @"Net income or loss from carrying on a rental property business may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.RentalPropertyNonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000366" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR283", Value = GetValueOrEmpty(report.IITR283) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000367
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000367()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000367
            Other income or loss relating to Other income or loss relating to Net income or loss from business has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR284, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR284, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR284 = RP:Income.NonPrimaryProductionOtherNet.Amount
            */
            assertion = NotMonetary(report.IITR284, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000367",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Other income or loss relating to Other income or loss..."" is not in a valid monetary format",
                    LongDescription = @"Other income or loss relating to Other income or loss relating to Net income or loss from business may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.NonPrimaryProductionOtherNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000367" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR284", Value = GetValueOrEmpty(report.IITR284) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000368
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000368()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000368
            Net non-Primary Production income/loss has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR285, 'S', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR285, 'S', 11, 0)
    
            Data Elements:
    
            ^IITR285 = RP:Income.NonPrimaryProductionNet.Amount
            */
            assertion = NotMonetary(report.IITR285, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000368",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-Primary Production income/loss is not in a valid monetary format",
                    LongDescription = @"Net non-Primary Production income/loss may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.NonPrimaryProductionNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000368" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR285", Value = GetValueOrEmpty(report.IITR285) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000369
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000369()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000369
            Net income/loss from business tax withheld voluntary agreement has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR286, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR286, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR286 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount
            */
            assertion = NotMonetary(report.IITR286, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000369",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income/loss from business tax withheld voluntary agreement is not in a valid monetary format",
                    LongDescription = @"Net income/loss from business tax withheld voluntary agreement must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000369" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR286", Value = GetValueOrEmpty(report.IITR286) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000370
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000370()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000370
            Net income/loss from business tax withheld where Australian business number not quoted has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR287, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR287, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR287 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount
            */
            assertion = NotMonetary(report.IITR287, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000370",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income/loss from business tax withheld where ABN not quoted is not in a valid monetary format",
                    LongDescription = @"Net income/loss from business tax withheld where Australian business number not quoted must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000370" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR287", Value = GetValueOrEmpty(report.IITR287) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000371
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000371()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000371
            Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains) has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR288, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR288, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR288 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount
            */
            assertion = NotMonetary(report.IITR288, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001371",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains) is not in a valid monetary format",
                    LongDescription = @"Net income/loss from business tax withheld - foreign resident withholding (excluding capital gains) must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000371" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR288", Value = GetValueOrEmpty(report.IITR288) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000372
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000372()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000372
            Net income/loss from business tax withheld - labour hire or other specified payments has a maximum field length of 13
    
            Legacy Rule Format:
            NotMonetary(^IITR289, 'U', 13, 2)

            Technical Business Rule Format:
            NotMonetary(^IITR289, 'U', 13, 2)
    
            Data Elements:
    
            ^IITR289 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount
            */
            assertion = NotMonetary(report.IITR289, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000372",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Net income/loss from business tax withheld... other specified payments"" is not in a valid monetary format",
                    LongDescription = @"Net income/loss from business tax withheld - labour hire or other specified payments must be unsigned, contain 2 decimals and not exceed 11 significant digits; no other characters are allowed except for the decimal point",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000372" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR289", Value = GetValueOrEmpty(report.IITR289) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000373
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000373()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000373
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR58 <> BLANK) AND (NotCharacterInSet(^IITR58, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR58 <> BLANK) AND (NotCharacterInSet(^IITR58, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR58 = RP:PersonDemographicDetails.Occupation.Description
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR58) != true && !(IsMatch(report.IITR58, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000373",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Main salary or wage occupation description contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000373" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR58", Value = GetValueOrEmpty(report.IITR58) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000380
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000380()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000380
            The Applied exemption, rollover or additional discount code of "T - Small Business Restructure Roll-over" is only valid for Australian residents unless otherwise eligible under certain circumstances
    
            Legacy Rule Format:
            AnyOccurrence(^IITR310, ^IITR310 = 'T - Small Business Restructure Roll-over') AND ^IITR19 <> TRUE

            Technical Business Rule Format:
            AnyOccurrence(^IITR310, ^IITR310 = 'T - Small Business Restructure Roll-over') AND ^IITR19 <> TRUE
    
            Data Elements:
    
            ^IITR310 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
    
            ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
            */
            assertion = ((report.IITR310Collection == null ? false : report.IITR310Collection.Any(IITR310Repeat => IITR310Repeat == @"T - Small Business Restructure Roll-over")) && report.IITR19 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001380",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check the Applied exemption, rollover or additional discount code",
                    LongDescription = @"The Applied exemption, rollover or additional discount code of ""T - Small Business Restructure Roll-over"" is only valid for Australian residents unless otherwise eligible under certain circumstances",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000380" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR310", Value = "IITR310" });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.IITR19) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000501
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000501()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000501
            The number of full days medicare levy exemption must be present with the action code.
    
            Legacy Rule Format:
            (^IITR173 <> NULL) AND ((^IITR172 = NULL) OR (^IITR172 = 0))

            Technical Business Rule Format:
            (^IITR173 <> NULL) AND ((^IITR172 = NULL) OR (^IITR172 = 0))
    
            Data Elements:
    
            ^IITR173 = RP:IncomeTax.MedicareLevy.ExemptionFullDays.Code
    
            ^IITR172 = RP:IncomeTax.MedicareLevy.ExemptionFullDays.Count
            */
            assertion = (report.IITR173 != null && (report.IITR172 == null || report.IITR172 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000501",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of full days medicare levy exemption required",
                    LongDescription = @"The number of full days medicare levy exemption must be present with the action code",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.ExemptionFullDays.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000501" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR173", Value = GetValueOrEmpty(report.IITR173) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR172", Value = report.IITR172.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000502
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000502()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000502
            When it has been indicated that the taxpayer and their dependants have not been covered by private patient hospital cover for the entire income year, the number of days not liable for surcharge must be provided.
    
            Legacy Rule Format:
            ^IITR176 <> TRUE AND ^IITR177 = NULL

            Technical Business Rule Format:
            ^IITR176 <> TRUE AND ^IITR177 = NULL
    
            Data Elements:
    
            ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator
    
            ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count
            */
            assertion = (report.IITR176 != true && report.IITR177 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000502",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Enter number of days NOT liable for Medicare levy surcharge",
                    LongDescription = @"When it has been indicated that the taxpayer and their dependants have not been covered by private patient hospital cover for the entire income year, the number of days not liable for surcharge must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000502" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR176", Value = GetValueOrEmpty(report.IITR176) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR177", Value = report.IITR177.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000503
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000503()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000503
            When taxpayer and their dependants have been covered by private patient hospital cover for the entire income year, their Private health insurance details - Health insurer ID, Membership number, and correct Tax claim code must also be provided.
    
            Legacy Rule Format:
            ^IITR176 = TRUE AND (Count(^RP.{PHISeqNum}:Context) = 0 OR AnyOccurrence(^IITR181, ^IITR181 = BLANK) OR AnyOccurrence(^IITR180, ^IITR180 = BLANK) OR AnyOccurrence(^IITR185, ^IITR185 = NULL))

            Technical Business Rule Format:
            ^IITR176 = TRUE AND (Count(^RP.{PHISeqNum}:Context) = 0 OR AnyOccurrence(^IITR181, ^IITR181 = BLANK) OR AnyOccurrence(^IITR180, ^IITR180 = BLANK) OR AnyOccurrence(^IITR185, ^IITR185 = NULL))
    
            Data Elements:
    
            ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator
    
            ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier
    
            ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier
    
            ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code
            */
            assertion = (report.IITR176.GetValueOrDefault() && (Count(report.RPPHISeqNumContextCollection) == 0 || (report.RPPHISeqNumContextCollection == null ? false : report.RPPHISeqNumContextCollection.Any(IITR181Repeat => string.IsNullOrWhiteSpace(IITR181Repeat.IITR181) )) || (report.RPPHISeqNumContextCollection == null ? false : report.RPPHISeqNumContextCollection.Any(IITR180Repeat => string.IsNullOrWhiteSpace(IITR180Repeat.IITR180) )) || (report.RPPHISeqNumContextCollection == null ? false : report.RPPHISeqNumContextCollection.Any(IITR185Repeat => IITR185Repeat.IITR185 == null))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000503",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Private health insurance policy details must be completed",
                    LongDescription = @"When taxpayer and their dependants have been covered by private patient hospital cover for the entire income year, their Private health insurance details - Health insurer ID, Membership number, and correct Tax claim code must also be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000503" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR176", Value = GetValueOrEmpty(report.IITR176) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR181", Value = "IITR181" });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR180", Value = "IITR180" });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR185", Value = "IITR185" });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000504
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000504()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000504
            If private patient hospital cover has been retained for the whole year, the number of days not liable for Medicare levy surcharge must be blank or a valid accounting period.
    
            Legacy Rule Format:
            ^IITR176 = TRUE AND (^IITR177 <> NULL AND ^IITR177 <> (DaysInDuration(^RP:EndDate - ^RP:StartDate) + 1))

            Technical Business Rule Format:
            ^IITR176 = TRUE AND (^IITR177 <> NULL AND ^IITR177 <> (DaysInDuration(^RP:EndDate - ^RP:StartDate) + 1))
    
            Data Elements:
    
            ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator
    
            ^EndDate = EndDate
    
            ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count
    
            ^StartDate = StartDate
            */
            assertion = ((report.RPExists ) && (report.IITR176.GetValueOrDefault() && (report.IITR177 != null && report.IITR177.GetValueOrDefault() != (report.RPEndDate.GetValueOrDefault() - report.RPStartDate.GetValueOrDefault()).Days + 1)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000504",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect value for number of days not liable for Medicare levy surcharge",
                    LongDescription = @"If private patient hospital cover has been retained for the whole year, the number of days not liable for Medicare levy surcharge must be blank or a valid accounting period",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000504" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR176", Value = GetValueOrEmpty(report.IITR176) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR177", Value = report.IITR177.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000505
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000505()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000505
            The number of days full levy exemption and/or the number of days half levy exemption exceed accounting period.
    
            Legacy Rule Format:
            ^IITR177 > (DaysInDuration(^RP:EndDate - ^RP:StartDate) + 1)

            Technical Business Rule Format:
            ^IITR177 > (DaysInDuration(^RP:EndDate - ^RP:StartDate) + 1)
    
            Data Elements:
    
            ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count
    
            ^EndDate = EndDate
    
            ^StartDate = StartDate
            */
            assertion = ((report.RPExists ) && (report.IITR177.GetValueOrDefault() > (report.RPEndDate.GetValueOrDefault() - report.RPStartDate.GetValueOrDefault()).Days + 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000505",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Levy exemption days exceeds limit",
                    LongDescription = @"The number of days full levy exemption and/or the number of days half levy exemption exceed accounting period",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000505" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR177", Value = report.IITR177.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000506
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000506()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000506
            Seniors and pensioners veteran code is present and Seniors and pensioners tax offset code is required
    
            Legacy Rule Format:
            ^IITR163 = BLANK AND ^IITR164 <> BLANK

            Technical Business Rule Format:
            ^IITR163 = BLANK AND ^IITR164 <> BLANK
    
            Data Elements:
    
            ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code
    
            ^IITR164 = RP:TaxOffsetClaim.Veteran.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR163)  && string.IsNullOrWhiteSpace(report.IITR164) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000506",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Seniors and pensioners tax offset indicator must be present",
                    LongDescription = @"Seniors and pensioners veteran code is present and Seniors and pensioners tax offset code is required",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SeniorAndPensioner.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000506" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR163", Value = GetValueOrEmpty(report.IITR163) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR164", Value = GetValueOrEmpty(report.IITR164) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000507
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000507()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000507
            Total tax offsets do not equal the sum of all tax offsets including Australian superannuation income stream, Superannuation contributions on behalf of your spouse tax offset, Zone or overseas forces tax offset, Invalid and Invalid Carer tax offset, Landcare and water facility tax offset brought forward from prior years, current and carried forward Early stage venture capital limited partnership (ESVCLP) tax offsets, current and carried forward Early stage investor tax offsets, and Other refundable tax offsets
    
            Legacy Rule Format:
            (^IITR168 <> NULL AND ^IITR168 <> (^IITR166 + ^IITR379  + ^IITR381 + ^IITR386 + ^IITR388 + ^IITR395 + ^IITR495 + ^IITR500 + ^IITR496 + ^IITR501)) OR (^IITR168 = NULL AND (^IITR166 + ^IITR379 + ^IITR381 + ^IITR386 + ^IITR388 + ^IITR395 + ^IITR495 + ^IITR500 + ^IITR496 + ^IITR501) <> 0)

            Technical Business Rule Format:
            (^IITR168 <> NULL AND ^IITR168 <> (^IITR166 + ^IITR379  + ^IITR381 + ^IITR386 + ^IITR388 + ^IITR395 + ^IITR495 + ^IITR500 + ^IITR496 + ^IITR501)) OR (^IITR168 = NULL AND (^IITR166 + ^IITR379 + ^IITR381 + ^IITR386 + ^IITR388 + ^IITR395 + ^IITR495 + ^IITR500 + ^IITR496 + ^IITR501) <> 0)
    
            Data Elements:
    
            ^IITR168 = RP:TaxOffsetClaim.Total.Amount
    
            ^IITR166 = RP:TaxOffsetClaim.SuperannuationIncomeStream.Amount
    
            ^IITR379 = RP:TaxOffsetClaim.SuperannuationContributionSpouse.Amount
    
            ^IITR381 = RP:TaxOffsetClaim.ZoneOrOverseasForces.Amount
    
            ^IITR386 = RP:TaxOffsetClaim.DependentInvalidAndCarer.Amount
    
            ^IITR388 = RP:TaxOffsetClaim.LandcareAndWaterFacilityBroughtForward.Amount
    
            ^IITR395 = RP:IncomeTax.TaxOffsetRefundableOther.Amount
    
            ^IITR495 = RP.LimitedPartners:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^IITR496 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^IITR500 = RP.LimitedPartners.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^IITR501 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = (report.IITR168 != null && report.IITR168.GetValueOrDefault() != report.IITR166.GetValueOrDefault() + report.IITR379.GetValueOrDefault() + report.IITR381.GetValueOrDefault() + report.IITR386.GetValueOrDefault() + report.IITR388.GetValueOrDefault() + report.IITR395.GetValueOrDefault() + report.IITR495.GetValueOrDefault() + report.IITR500.GetValueOrDefault() + report.IITR496.GetValueOrDefault() + report.IITR501.GetValueOrDefault() || report.IITR168 == null && report.IITR166.GetValueOrDefault() + report.IITR379.GetValueOrDefault() + report.IITR381.GetValueOrDefault() + report.IITR386.GetValueOrDefault() + report.IITR388.GetValueOrDefault() + report.IITR395.GetValueOrDefault() + report.IITR495.GetValueOrDefault() + report.IITR500.GetValueOrDefault() + report.IITR496.GetValueOrDefault() + report.IITR501.GetValueOrDefault() != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.002507",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total tax offsets incorrect",
                    LongDescription = @"Total tax offsets do not equal the sum of all tax offsets including Australian superannuation income stream, Superannuation contributions on behalf of your spouse tax offset, Zone or overseas forces tax offset, Invalid and Invalid Carer tax offset, Landcare and water facility tax offset brought forward from prior years, current and carried forward Early stage venture capital limited partnership (ESVCLP) tax offsets, current and carried forward Early stage investor tax offsets, and Other refundable tax offsets",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000507" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR168", Value = GetValueOrEmpty(report.IITR168) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR166", Value = GetValueOrEmpty(report.IITR166) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR379", Value = GetValueOrEmpty(report.IITR379) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR381", Value = GetValueOrEmpty(report.IITR381) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR386", Value = GetValueOrEmpty(report.IITR386) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR388", Value = GetValueOrEmpty(report.IITR388) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR395", Value = GetValueOrEmpty(report.IITR395) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR495", Value = GetValueOrEmpty(report.IITR495) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR500", Value = GetValueOrEmpty(report.IITR500) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR496", Value = GetValueOrEmpty(report.IITR496) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR501", Value = GetValueOrEmpty(report.IITR501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000510
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000510()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000510
            Australian superannuation income stream has a maximum length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR166, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR166, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR166 = RP:TaxOffsetClaim.SuperannuationIncomeStream.Amount
            */
            assertion = NotMonetary(report.IITR166, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000510",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian superannuation income stream is not in a valid monetary format",
                    LongDescription = @"For Australian superannuation income stream must be unsigned but must not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SuperannuationIncomeStream.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000510" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR166", Value = GetValueOrEmpty(report.IITR166) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000511
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000511()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000511
            Total tax offsets has a maximum length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR168, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR168, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR168 = RP:TaxOffsetClaim.Total.Amount
            */
            assertion = NotMonetary(report.IITR168, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000511",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total tax offsets is not in a valid monetary format",
                    LongDescription = @"For Total tax offsets must be unsigned but must not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000511" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR168", Value = GetValueOrEmpty(report.IITR168) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000512
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000512()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000512
            Number of dependent children and students has a maximum length of 2
    
            Legacy Rule Format:
            Length(^IITR171) > 2

            Technical Business Rule Format:
            Length(^IITR171) > 2
    
            Data Elements:
    
            ^IITR171 = RP:IncomeTax.MedicareLevy.DependentChildrenAndStudents.Count
            */
            assertion = (Length(report.IITR171.GetValueOrDefault()) > 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000512",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of dependent children and students must not exceed 2 characters",
                    LongDescription = @"For the Number of dependent children and students, the maximum allowable characters for this form is 2",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.DependentChildrenAndStudents.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000512" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR171", Value = report.IITR171.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000513
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000513()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000513
            Exemption - number of days full levy exemption has a maximum length of 3
    
            Legacy Rule Format:
            Length(^IITR172) > 3

            Technical Business Rule Format:
            Length(^IITR172) > 3
    
            Data Elements:
    
            ^IITR172 = RP:IncomeTax.MedicareLevy.ExemptionFullDays.Count
            */
            assertion = (Length(report.IITR172.GetValueOrDefault()) > 3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000513",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption - number of days full levy exemption must not exceed 3 characters",
                    LongDescription = @"For Exemption - number of days full levy exemption, the maximum allowable characters for this form is 3",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.ExemptionFullDays.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000513" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR172", Value = report.IITR172.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000514
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000514()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000514
            Exemption - number of days half levy exemption has a maximum length of 3
    
            Legacy Rule Format:
            Length(^IITR174) > 3

            Technical Business Rule Format:
            Length(^IITR174) > 3
    
            Data Elements:
    
            ^IITR174 = RP:IncomeTax.MedicareLevy.ExemptionHalfDays.Count
            */
            assertion = (Length(report.IITR174.GetValueOrDefault()) > 3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000514",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption - number of days half levy exemption must not exceed 3 characters",
                    LongDescription = @"For Exemption - number of days half levy exemption, the maximum allowable characters for this form is 3",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.ExemptionHalfDays.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000514" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR174", Value = report.IITR174.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000515
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000515()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000515
            It is mandatory to specify whether you and all your dependants covered by private patient hospital cover for the whole financial year
    
            Legacy Rule Format:
            ^IITR176 = NULL

            Technical Business Rule Format:
            ^IITR176 = NULL
    
            Data Elements:
    
            ^IITR176 = RP:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator
            */
            assertion = (report.IITR176 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000515",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"You and all your dependants covered by private patient hospital cover...' must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000515" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR176", Value = GetValueOrEmpty(report.IITR176) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000516
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000516()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000516
            Number of days not liable for surcharge has a maximum length of 3
    
            Legacy Rule Format:
            Length(^IITR177) > 3

            Technical Business Rule Format:
            Length(^IITR177) > 3
    
            Data Elements:
    
            ^IITR177 = RP:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count
            */
            assertion = (Length(report.IITR177.GetValueOrDefault()) > 3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000516",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of days not liable for surcharge must not exceed 3 characters",
                    LongDescription = @"For Number of days not liable for surcharge, the maximum allowable characters for this form is 3",
                    Location = "/xbrli:xbrl/tns:IncomeTax.MedicareLevy.SurchargeExemptionDays.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000516" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR177", Value = report.IITR177.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000518
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000518()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000518
            The Sequence Container within the Reporting Party Private Health Insurance Sequence Number context instance must not be duplicated
    
            Legacy Rule Format:
            HasDuplicateValues(^IITR570)

            Technical Business Rule Format:
            HasDuplicateValues(^IITR570)
    
            Data Elements:
    
            ^IITR570 = RP.{PHISeqNum}:Report.ItemOrder.Number
            */
            assertion = HasDuplicateValues(report.RPPHISeqNumContextCollection == null ? null : report.RPPHISeqNumContextCollection.Select(f => f.IITR570).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000518",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sequence Container must not be duplicated",
                    LongDescription = @"The Sequence Container within the Reporting Party Private Health Insurance Sequence Number context instance must not be duplicated",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.RPPHISeqNumContextCollection[DuplicateValueIndex(report.RPPHISeqNumContextCollection.Select(f => f.IITR570))].Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000518" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000600
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000600(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000600
            When providing Tax claim code as "No dependants", "Dependants", "Claiming your share of rebate" or" Claiming your spouses share of rebate", then Health insurer ID, Membership number, Your premiums eligible for Australian Government rebate, Your Australian Government rebate received, and Benefit code must be provided.

            Legacy Rule Format:
            InSet(^IITR185, '"No dependants","Dependants","Claiming your share of rebate","Claiming your spouses share of rebate"') AND (^IITR181 = BLANK OR ^IITR180 = NULL OR ^IITR182 = NULL OR ^IITR183 = NULL OR ^IITR184 = NULL)

            Technical Business Rule Format:
            InSet(^IITR185, '"No dependants","Dependants","Claiming your share of rebate","Claiming your spouses share of rebate"') AND (^IITR181 = BLANK OR ^IITR180 = NULL OR ^IITR182 = NULL OR ^IITR183 = NULL OR ^IITR184 = NULL)

            Data Elements:

            ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code

            ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier

            ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier

            ^IITR182 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.RebatableComponent.Amount

            ^IITR183 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.GovernmentRebateReceived.Amount

            ^IITR184 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.Benefit.Code
            */
            assertion = (IsMatch(rpphiSeqNumContext.IITR185, @"^(No dependants|Dependants|Claiming your share of rebate|Claiming your spouses share of rebate)$") && (string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR181)  || rpphiSeqNumContext.IITR180 == null || rpphiSeqNumContext.IITR182 == null || rpphiSeqNumContext.IITR183 == null || rpphiSeqNumContext.IITR184 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000600",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All private health insurance policy details required.",
                    LongDescription = @"All private health insurance policy details must be completed where tax claim code is ""No dependants"", ""Dependants"", ""Claiming your share of rebate"", or ""Claiming your spouses share of rebate"".",
                    Location = "/xbrli:xbrl/tns:IncomeTax.HealthInsurance.TaxClaim.Code[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000600" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR185", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR185) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR181", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR181) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR180", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR180) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR182", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR182) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR183", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR183) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR184", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR184) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000601
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000601(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000601
            The health insurer ID code of (other) has been entered but details of the health insurer have not been provided in the Additional Information field. The required details which describe the health insurer must be provided.

            Legacy Rule Format:
            ((^IITR180 <> BLANK)) AND ((^IITR180 = 'OTH')) AND ((^IITR31 = BLANK))

            Technical Business Rule Format:
            ((^IITR180 <> BLANK)) AND ((^IITR180 = 'OTH')) AND ((^IITR31 = BLANK))

            Data Elements:

            ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier

            ^IITR31 = RP:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
            */
            assertion = (string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR180) != true && rpphiSeqNumContext.IITR180 == @"OTH" && string.IsNullOrWhiteSpace(report.IITR31) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000601",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Details of health insurer ID is required",
                    LongDescription = @"When the health insurer ID of (other) has been entered but details of the health insurer have not been provided in the Additional Information field, the required details which describe the health insurer must be provided",
                    Location = "/xbrli:xbrl/tns:Identifiers.HealthInsuranceProvider.Identifier[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000601" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR180", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR180) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR31", Value = GetValueOrEmpty(report.IITR31) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000602
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000602(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000602
            The health insurer ID code is invalid.

            Legacy Rule Format:
            (^IITR180 <> NULL) AND (NotInDomain(^IITR180, 'Health Insurer ID Codes'))

            Technical Business Rule Format:
            (^IITR180 <> NULL) AND (NotInDomain(^IITR180, 'Health Insurer ID Codes'))

            Data Elements:

            ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier
            */
            assertion = (rpphiSeqNumContext.IITR180 != null && !(IsMatch(rpphiSeqNumContext.IITR180, @"^(ACA|AHB|AHM|AMA|AUF|BUP|CBC|CBH|CDH|CPS|FAI|GMH|HBF|HCF|HCI|HIF|LHM|LHS|MBP|MDH|MYO|NHB|NIB|NTF|OMF|OTH|PWA|QCH|QTU|RBH|RTE|SLM|SPE|SPS|WFD)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001602",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Health insurer ID invalid",
                    LongDescription = @"Health insurer ID must be a code as assigned by the Australian Taxation Office",
                    Location = "/xbrli:xbrl/tns:Identifiers.HealthInsuranceProvider.Identifier[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000602" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR180", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR180) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000603
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000603(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000603
            Private health insurance policy Tax claim code is "Spouse claiming your share of rebate" or "Covered as dependent child" and Private health insurance policy's Health insurer ID and Membership number are missing.

            Legacy Rule Format:
            ((^IITR185 <> BLANK)) AND ((InSet(^IITR185, '"Spouse claiming your share of rebate","Covered as dependent child"')) AND (^IITR180 = BLANK) AND (^IITR181 = BLANK))

            Technical Business Rule Format:
            ((^IITR185 <> BLANK)) AND ((InSet(^IITR185, '"Spouse claiming your share of rebate","Covered as dependent child"')) AND (^IITR180 = BLANK) AND (^IITR181 = BLANK))

            Data Elements:

            ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code

            ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier

            ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR185) != true && (IsMatch(rpphiSeqNumContext.IITR185, @"^(Spouse claiming your share of rebate|Covered as dependent child)$") && string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR180)  && string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR181) ));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000603",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Private health insurance policy details are incomplete",
                    LongDescription = @"When providing Private health insurance policy's Tax claim code as ""Spouse claiming your share of rebate "" or ""Covered as dependent child"" then Private health insurance policy's Health insurer ID and Membership number must also be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.HealthInsurance.TaxClaim.Code[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000603" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR185", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR185) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR180", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR180) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR181", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000605
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000605(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000605
            Where there are any Private Health Insurance policy details that have been completed a Tax claim code must be present.

            Legacy Rule Format:
            (^IITR180 <> NULL OR ^IITR181 <> BLANK OR ^IITR182 <> NULL OR ^IITR183 <> NULL OR ^IITR184 <> NULL) AND (^IITR185 = NULL)

            Technical Business Rule Format:
            (^IITR180 <> NULL OR ^IITR181 <> BLANK OR ^IITR182 <> NULL OR ^IITR183 <> NULL OR ^IITR184 <> NULL) AND (^IITR185 = NULL)

            Data Elements:

            ^IITR180 = RP.{PHISeqNum}:Identifiers.HealthInsuranceProvider.Identifier

            ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier

            ^IITR182 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.RebatableComponent.Amount

            ^IITR183 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.GovernmentRebateReceived.Amount

            ^IITR184 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.Benefit.Code

            ^IITR185 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.TaxClaim.Code
            */
            assertion = ((rpphiSeqNumContext.IITR180 != null || string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR181) != true || rpphiSeqNumContext.IITR182 != null || rpphiSeqNumContext.IITR183 != null || rpphiSeqNumContext.IITR184 != null) && rpphiSeqNumContext.IITR185 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000605",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax claim code is required",
                    LongDescription = @"Where there are any Private Health Insurance policy details that have been completed a Tax claim code must be present",
                    Location = "/xbrli:xbrl/tns:Identifiers.HealthInsuranceProvider.Identifier[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000605" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR180", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR180) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR181", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR181) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR182", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR182) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR183", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR183) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR184", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR184) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR185", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR185) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000608
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000608()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000608
            When a Contributions paid amount is present, a Superannuation contributions on behalf of your spouse tax offset amount above 0 must be provided.
    
            Legacy Rule Format:
            (^IITR378 <> NULL) AND (^IITR378 > 0) AND ((^IITR379 = NULL) OR (^IITR379 = 0))

            Technical Business Rule Format:
            (^IITR378 <> NULL) AND (^IITR378 > 0) AND ((^IITR379 = NULL) OR (^IITR379 = 0))
    
            Data Elements:
    
            ^IITR378 = RP:SuperannuationContribution.SpouseContributions.Amount
    
            ^IITR379 = RP:TaxOffsetClaim.SuperannuationContributionSpouse.Amount
            */
            assertion = (report.IITR378 != null && report.IITR378.GetValueOrDefault() > 0 && (report.IITR379 == null || report.IITR379 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000608",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation contributions on behalf of your spouse's tax offset is required",
                    LongDescription = @"When a Contributions paid amount is present, a Superannuation contributions on behalf of your spouse tax offset amount above 0 must be provided",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.SpouseContributions.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000608" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR378", Value = GetValueOrEmpty(report.IITR378) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR379", Value = GetValueOrEmpty(report.IITR379) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000609
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000609()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000609
            When a Superannuation contributions paid on behalf of your spouse's tax offset amount is present, a Contributions paid amount above 0 must be provided.
    
            Legacy Rule Format:
            (^IITR379 <> NULL) AND (^IITR379 > 0) AND ((^IITR378 = NULL) OR (^IITR378 = 0))

            Technical Business Rule Format:
            (^IITR379 <> NULL) AND (^IITR379 > 0) AND ((^IITR378 = NULL) OR (^IITR378 = 0))
    
            Data Elements:
    
            ^IITR379 = RP:TaxOffsetClaim.SuperannuationContributionSpouse.Amount
    
            ^IITR378 = RP:SuperannuationContribution.SpouseContributions.Amount
            */
            assertion = (report.IITR379 != null && report.IITR379.GetValueOrDefault() > 0 && (report.IITR378 == null || report.IITR378 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000609",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contributions paid amount is required",
                    LongDescription = @"When a Superannuation contributions paid on behalf of your spouse's tax offset amount is present, a Contributions paid amount above 0 must be provided",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SuperannuationContributionSpouse.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000609" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR379", Value = GetValueOrEmpty(report.IITR379) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR378", Value = GetValueOrEmpty(report.IITR378) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000613
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000613()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000613
            When the Other refundable tax offsets amount is above 0, the Other refundable tax offsets code must be present.
    
            Legacy Rule Format:
            (^IITR395 <> NULL) AND (^IITR395 > 0) AND (^IITR396 = NULL)

            Technical Business Rule Format:
            (^IITR395 <> NULL) AND (^IITR395 > 0) AND (^IITR396 = NULL)
    
            Data Elements:
    
            ^IITR395 = RP:IncomeTax.TaxOffsetRefundableOther.Amount
    
            ^IITR396 = RP:IncomeTax.TaxOffsetRefundableOther.Code
            */
            assertion = (report.IITR395 != null && report.IITR395.GetValueOrDefault() > 0 && report.IITR396 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000613",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other refundable tax offsets code is required",
                    LongDescription = @"When the Other refundable tax offsets amount is above 0, the Other refundable tax offsets code must be present",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetRefundableOther.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000613" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR395", Value = GetValueOrEmpty(report.IITR395) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR396", Value = GetValueOrEmpty(report.IITR396) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000614
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000614()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000614
            When the Other refundable tax offsets code is present, the amount of Other refundable tax offsets must be above 0. An Other refundable tax offsets code is not required for zero amounts.
    
            Legacy Rule Format:
            (^IITR396 <> NULL) AND ((^IITR395 = NULL)  OR (^IITR395 = 0))

            Technical Business Rule Format:
            (^IITR396 <> NULL) AND ((^IITR395 = NULL)  OR (^IITR395 = 0))
    
            Data Elements:
    
            ^IITR396 = RP:IncomeTax.TaxOffsetRefundableOther.Code
    
            ^IITR395 = RP:IncomeTax.TaxOffsetRefundableOther.Amount
            */
            assertion = (report.IITR396 != null && (report.IITR395 == null || report.IITR395 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000614",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other refundable tax offsets amount is invalid",
                    LongDescription = @"When the Other refundable tax offsets code is present, the amount of Other refundable tax offsets must be above 0. An Other refundable tax offsets code is not required for zero amounts.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetRefundableOther.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000614" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR396", Value = GetValueOrEmpty(report.IITR396) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR395", Value = GetValueOrEmpty(report.IITR395) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000618
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000618(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000618
            Private health insurance - Membership number has a maximum field length of 15

            Legacy Rule Format:
            Length(^IITR181) > 15

            Technical Business Rule Format:
            Length(^IITR181) > 15

            Data Elements:

            ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier
            */
            assertion = (Length(rpphiSeqNumContext.IITR181) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000618",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Private health insurance - Membership number must not exceed 15 characters",
                    LongDescription = @"For Private health insurance - Membership number, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:Identifiers.HealthInsuranceMembership.Identifier[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000618" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR181", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000619
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000619(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000619
            Your premiums eligible for Australian Government rebate has a maximum field length of 11

            Legacy Rule Format:
            NotMonetary(^IITR182, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR182, 'U', 11, 0)

            Data Elements:

            ^IITR182 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.RebatableComponent.Amount
            */
            assertion = NotMonetary(rpphiSeqNumContext.IITR182, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000619",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your premiums eligible for Australian Government rebate is not in a valid monetary format",
                    LongDescription = @"Your premiums eligible for Australian Government rebate must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.HealthInsurance.RebatableComponent.Amount[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000619" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR182", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR182) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000620
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000620(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000620
            Your Australian Government rebate received has a maximum field length of 11

            Legacy Rule Format:
            NotMonetary(^IITR183, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR183, 'U', 11, 0)

            Data Elements:

            ^IITR183 = RP.{PHISeqNum}:IncomeTax.HealthInsurance.GovernmentRebateReceived.Amount
            */
            assertion = NotMonetary(rpphiSeqNumContext.IITR183, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000620",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your Australian Government rebate is not in a valid monetary format",
                    LongDescription = @"Your Australian Government rebate must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.HealthInsurance.GovernmentRebateReceived.Amount[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000620" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR183", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR183) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000621
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000621()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000621
            Contributions paid has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR378, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR378, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR378 = RP:SuperannuationContribution.SpouseContributions.Amount
            */
            assertion = NotMonetary(report.IITR378, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000621",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contributions paid is not in a valid monetary format",
                    LongDescription = @"Contributions paid must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.SpouseContributions.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000621" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR378", Value = GetValueOrEmpty(report.IITR378) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000622
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000622()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000622
            Superannuation contributions on behalf of your spouse tax offset has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR379, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR379, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR379 = RP:TaxOffsetClaim.SuperannuationContributionSpouse.Amount
            */
            assertion = NotMonetary(report.IITR379, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000622",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation contributions on behalf of your spouse tax offset is not in a valid monetary format",
                    LongDescription = @"Superannuation contributions on behalf of your spouse tax offset must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SuperannuationContributionSpouse.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000622" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR379", Value = GetValueOrEmpty(report.IITR379) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000623
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000623()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000623
            Zone or overseas forces tax offset has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR381, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR381, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR381 = RP:TaxOffsetClaim.ZoneOrOverseasForces.Amount
            */
            assertion = NotMonetary(report.IITR381, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000623",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Zone or overseas forces tax offset is not in a valid monetary format",
                    LongDescription = @"Zone or overseas forces tax offset must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.ZoneOrOverseasForces.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000623" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR381", Value = GetValueOrEmpty(report.IITR381) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000625
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000625()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000625
            Invalid and Invalid Carer tax offset has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR386, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR386, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR386 = RP:TaxOffsetClaim.DependentInvalidAndCarer.Amount
            */
            assertion = NotMonetary(report.IITR386, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.020020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid and Invalid Carer tax offset is not in a valid monetary format",
                    LongDescription = @"For Invalid and Invalid Carer tax offset must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.DependentInvalidAndCarer.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000625" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR386", Value = GetValueOrEmpty(report.IITR386) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000626
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000626()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000626
            Landcare and water facility tax offset brought forward from prior years has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR388, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR388, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR388 = RP:TaxOffsetClaim.LandcareAndWaterFacilityBroughtForward.Amount
            */
            assertion = NotMonetary(report.IITR388, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000626",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Landcare and water facility tax offset brought forward from prior years is not in a valid monetary format",
                    LongDescription = @"Landcare and water facility tax offset brought forward from prior years must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.LandcareAndWaterFacilityBroughtForward.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000626" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR388", Value = GetValueOrEmpty(report.IITR388) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000629
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000629()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000629
            Other refundable tax offsets has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR395, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR395, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR395 = RP:IncomeTax.TaxOffsetRefundableOther.Amount
            */
            assertion = NotMonetary(report.IITR395, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000629",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other refundable tax offsets is not in a valid monetary format",
                    LongDescription = @"Other refundable tax offsets must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetRefundableOther.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000629" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR395", Value = GetValueOrEmpty(report.IITR395) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000630
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000630(IITR2026.RPPHISeqNumContext rpphiSeqNumContext)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000630
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.

            Legacy Rule Format:
            (^IITR181 <> BLANK) AND (NotCharacterInSet(^IITR181, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR181 <> BLANK) AND (NotCharacterInSet(^IITR181, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Data Elements:

            ^IITR181 = RP.{PHISeqNum}:Identifiers.HealthInsuranceMembership.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(rpphiSeqNumContext.IITR181) != true && !(IsMatch(rpphiSeqNumContext.IITR181, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000630",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Private health insurance - Membership number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:Identifiers.HealthInsuranceMembership.Identifier[@contextRef='" + rpphiSeqNumContext.Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000630" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR181", Value = GetValueOrEmpty(rpphiSeqNumContext.IITR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000631
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000631()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000631
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            (^IITR379 <> NULL) AND (^IITR379 < 1 OR ^IITR379 > 540)

            Technical Business Rule Format:
            (^IITR379 <> NULL) AND (^IITR379 < 1 OR ^IITR379 > 540)
    
            Data Elements:
    
            ^IITR379 = RP:TaxOffsetClaim.SuperannuationContributionSpouse.Amount
            */
            assertion = (report.IITR379 != null && (report.IITR379.GetValueOrDefault() < 1 || report.IITR379.GetValueOrDefault() > 540));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000631",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation contributions on behalf of your spouse tax offset is invalid",
                    LongDescription = @"Valid range for Superannuation contributions on behalf of your spouse tax offset is 1 - 540",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SuperannuationContributionSpouse.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000631" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR379", Value = GetValueOrEmpty(report.IITR379) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000632
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000632()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000632
            Value entered is not a valid code/value for that field.
    
            Legacy Rule Format:
            (^IITR388 <> NULL) AND (^IITR388 < 1 OR ^IITR388 > 13100)

            Technical Business Rule Format:
            (^IITR388 <> NULL) AND (^IITR388 < 1 OR ^IITR388 > 13100)
    
            Data Elements:
    
            ^IITR388 = RP:TaxOffsetClaim.LandcareAndWaterFacilityBroughtForward.Amount
            */
            assertion = (report.IITR388 != null && (report.IITR388.GetValueOrDefault() < 1 || report.IITR388.GetValueOrDefault() > 13100));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000632",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Landcare and water facility tax offset is invalid",
                    LongDescription = @"Valid range for Landcare and water facility tax offset is 1 - 13100",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.LandcareAndWaterFacilityBroughtForward.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000632" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR388", Value = GetValueOrEmpty(report.IITR388) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000636
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000636()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000636
            Bank Account Details have not been completed. Any refund due may be delayed.
    
            Legacy Rule Format:
            ^IITR53 = NULL AND ^IITR54 = BLANK AND ^IITR55 = BLANK

            Technical Business Rule Format:
            ^IITR53 = NULL AND ^IITR54 = BLANK AND ^IITR55 = BLANK
    
            Data Elements:
    
            ^IITR53 = RP:FinancialInstitutionAccount.BankStateBranch.Number
    
            ^IITR54 = RP:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
    
            ^IITR55 = RP:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = (report.IITR53 == null && string.IsNullOrWhiteSpace(report.IITR54)  && string.IsNullOrWhiteSpace(report.IITR55) );
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000636",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Bank Account Details have not been completed. Any refund due may be delayed.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.BankStateBranch.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000636" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR53", Value = GetValueOrEmpty(report.IITR53) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR54", Value = GetValueOrEmpty(report.IITR54) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR55", Value = GetValueOrEmpty(report.IITR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000637
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000637()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000637
            Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.
    
            Legacy Rule Format:
            NotInSet(^IITR10, '"2026","2027"') OR (^IITR10 = 2027 AND Today() > AsDate('2027-06-15') AND ^IITR16 <> TRUE)

            Technical Business Rule Format:
            NotInSet(^IITR10, '"2026","2027"') OR (^IITR10 = 2027 AND Today() > AsDate('2027-06-15') AND ^IITR16 <> TRUE)
    
            Data Elements:
    
            ^IITR10 = RP:Report.TargetFinancial.Year
    
            ^IITR16 = RP:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.IITR10.GetValueOrDefault().ToString(), @"^(2026|2027)$", RegexOptions.IgnoreCase)) || report.IITR10.GetValueOrDefault() == 2027 && DateTime.Now.Date > AsDate(@"2027-06-15") && report.IITR16 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500127",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000637" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR10", Value = GetValueOrEmpty(report.IITR10) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR16", Value = GetValueOrEmpty(report.IITR16) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000638
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000638()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000638
            Non-resident foreign income amount has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR494, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR494, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR494 = RP.JFOREIGN:Income.Net.Amount
            */
            assertion = NotMonetary(report.IITR494, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000638",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident foreign income amount is not in a valid monetary format",
                    LongDescription = @"Non-resident foreign income amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters. When amount is negative, enter zero (0).",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000638" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR494", Value = GetValueOrEmpty(report.IITR494) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000640
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000640()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000640
            For an Australian resident, Non-resident foreign income amount must not be provided.
    
            Legacy Rule Format:
            (^IITR19 = TRUE AND (^IITR666 = NULL AND ^IITR667 = NULL) AND ^IITR494 <> NULL)

            Technical Business Rule Format:
            (^IITR19 = TRUE AND (^IITR666 = NULL AND ^IITR667 = NULL) AND ^IITR494 <> NULL)
    
            Data Elements:
    
            ^IITR494 = RP.JFOREIGN:Income.Net.Amount
    
            ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
            ^IITR666 = RP:Residency.TaxPurposesStart.Date
    
            ^IITR667 = RP:Residency.TaxPurposesEnd.Date
            */
            assertion = (report.IITR19.GetValueOrDefault() && (report.IITR666 == null && report.IITR667 == null) && report.IITR494 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000640",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian resident, Non-resident foreign income amount must not be provided",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000640" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR666", Value = GetValueOrEmptyDate(report.IITR666) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR667", Value = GetValueOrEmptyDate(report.IITR667) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR494", Value = GetValueOrEmpty(report.IITR494) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000646
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000646()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000646
            Early stage venture capital limited partnership (ESVCLP) - Current year tax offset has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR495, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR495, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR495 = RP.LimitedPartners:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.IITR495, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001646",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage venture capital limited partnership (ESVCLP) - Current year tax offset is not in a valid monetary format",
                    LongDescription = @"Early stage venture capital limited partnership (ESVCLP) - Current year tax offset must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPLimitedPartnersId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000646" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR495", Value = GetValueOrEmpty(report.IITR495) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000647
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000647()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000647
            Early stage investor - Current year tax offset has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR496, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR496, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR496 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.IITR496, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001647",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage investor - Current year tax offset is not in a valid monetary format",
                    LongDescription = @"Early stage investor - Current year tax offset must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000647" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR496", Value = GetValueOrEmpty(report.IITR496) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000648
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000648()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000648
            Only taxpayers who have been a sophisticated investor for at least one early stage innovation company (ESIC) investment in the year, may claim an Early stage investor tax offset amount above $10,000
    
            Legacy Rule Format:
            (^IITR496 + ^IITR501) > 10000 AND (^IITR496 + ^IITR501) < 200001

            Technical Business Rule Format:
            (^IITR496 + ^IITR501) > 10000 AND (^IITR496 + ^IITR501) < 200001
    
            Data Elements:
    
            ^IITR496 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^IITR501 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = (report.IITR496.GetValueOrDefault() + report.IITR501.GetValueOrDefault() > 10000 && report.IITR496.GetValueOrDefault() + report.IITR501.GetValueOrDefault() < 200001);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000648",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Early stage investor tax offset claim amount exceeds $10,000",
                    LongDescription = @"Only taxpayers who have been a sophisticated investor for at least one early stage innovation company (ESIC) investment in the year, may claim an Early stage investor tax offset amount above $10,000",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000648" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR496", Value = GetValueOrEmpty(report.IITR496) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR501", Value = GetValueOrEmpty(report.IITR501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000649
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000649()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000649
            The Early stage investor tax offset is capped at $200,000 per investor and their affiliates combined each income year. This $200,000 limit includes the sum of the current year tax offset entitlement and the amount of early stage investor tax offset carried forward from the previous year.
    
            Legacy Rule Format:
            (^IITR496 + ^IITR501) > 200000

            Technical Business Rule Format:
            (^IITR496 + ^IITR501) > 200000
    
            Data Elements:
    
            ^IITR496 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^IITR501 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = (report.IITR496.GetValueOrDefault() + report.IITR501.GetValueOrDefault() > 200000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.001649",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage investor tax offset claim amount must not exceed $200,000",
                    LongDescription = @"The Early stage investor tax offset is capped at $200,000 per investor and their affiliates combined each income year. This $200,000 limit includes the sum of the current year tax offset entitlement and the amount of early stage investor tax offset carried forward from the previous year.",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000649" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR496", Value = GetValueOrEmpty(report.IITR496) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR501", Value = GetValueOrEmpty(report.IITR501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000652
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000652()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000652
            The date entered is a future date. Date of birth cannot be any day in advance of the form submission date.
    
            Legacy Rule Format:
            (^IITR29 <> NULL) AND (^IITR29 > Today())

            Technical Business Rule Format:
            (^IITR29 <> NULL) AND (^IITR29 > Today())
    
            Data Elements:
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
            */
            assertion = (report.IITR29 != null && report.IITR29.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000652",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of birth must be prior to the form submission date",
                    LongDescription = @"The date entered is a future date. Date of birth cannot be any day in advance of the form submission date.",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Birth.Date[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000652" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmptyDate(report.IITR29) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000653
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000653()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000653
            Early stage venture capital limited partnership (ESVCLP) - Tax offset carried forward from previous year has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR500, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR500, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR500 = RP.LimitedPartners.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.IITR500, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000653",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage venture capital limited partnership (ESVCLP) - Tax offset carried forward from previous year is not in a valid monetary format",
                    LongDescription = @"Early stage venture capital limited partnership (ESVCLP) - Tax offset carried forward from previous year must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPLimitedPartnersYPreviousId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000653" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR500", Value = GetValueOrEmpty(report.IITR500) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000654
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000654()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000654
            Early stage investor - Tax offset carried forward from previous year has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR501, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR501, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR501 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.IITR501, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000654",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage investor - Tax offset carried forward from previous year is not in a valid monetary format",
                    LongDescription = @"Early stage investor - Tax offset carried forward from previous year must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorYPreviousId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000654" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR501", Value = GetValueOrEmpty(report.IITR501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000655
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000655()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000655
            A text character must be one of the following: A to Z a to z or 0 to 9
    
            Legacy Rule Format:
            ^IITR54 <> BLANK AND NotCharacterInSet(^IITR54, '"a-z","A-Z","0-9"')

            Technical Business Rule Format:
            ^IITR54 <> BLANK AND NotCharacterInSet(^IITR54, '"a-z","A-Z","0-9"')
    
            Data Elements:
    
            ^IITR54 = RP:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR54) != true && !(IsMatch(report.IITR54, @"^[a-zA-Z0-9]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000655",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z or 0 to 9",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount.FinancialInstitutionAccount.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000655" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR54", Value = GetValueOrEmpty(report.IITR54) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000656
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000656()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000656
            Scheme for Reporting Party Investor must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.Investor:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.Investor:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPInvestorExists ) && (report.RPInvestorIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000656",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Investor must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000656" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000657
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000657()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000657
            Scheme for Reporting Party Investor YPrevious must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.Investor.YPrevious:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.Investor.YPrevious:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPInvestorYPreviousExists ) && (report.RPInvestorYPreviousIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000657",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Investor YPrevious must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000657" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000658
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000658()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000658
            Scheme for Reporting Party Limited Partners must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.LimitedPartners:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.LimitedPartners:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPLimitedPartnersExists ) && (report.RPLimitedPartnersIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000658",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Limited Partners must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000658" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000659
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000659()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000659
            Scheme for Reporting Party Limited Partners YPrevious must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.LimitedPartners.YPrevious:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.LimitedPartners.YPrevious:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPLimitedPartnersYPreviousExists ) && (report.RPLimitedPartnersYPreviousIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000659",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Limited Partners YPrevious must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000659" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000662
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000662()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000662
            Assessable amount from capped defined benefit income stream has a maximum field length of 11
    
            Legacy Rule Format:
            NotMonetary(^IITR505, 'U', 11, 0)

            Technical Business Rule Format:
            NotMonetary(^IITR505, 'U', 11, 0)
    
            Data Elements:
    
            ^IITR505 = RP:Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount
            */
            assertion = NotMonetary(report.IITR505, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000662",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessable amount from capped defined benefit income stream is not in a valid monetary format",
                    LongDescription = @"Assessable amount from capped defined benefit income stream must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000662" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR505", Value = GetValueOrEmpty(report.IITR505) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000665
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000665()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000665
            Where the age of the taxpayer is 61 or over on 30 June of the reporting period, then the Australian superannuation income stream offset amount must not exceed $11,875
    
            Legacy Rule Format:
            ^IITR166 > 11875 AND (^IITR29 <> NULL AND ^IITR29 <= ConvertToDate(30,6, ^IITR10 - 61))

            Technical Business Rule Format:
            ^IITR166 > 11875 AND (^IITR29 <> NULL AND ^IITR29 <= ConvertToDate(30,6, ^IITR10 - 61))
    
            Data Elements:
    
            ^IITR166 = RP:TaxOffsetClaim.SuperannuationIncomeStream.Amount
    
            ^IITR10 = RP:Report.TargetFinancial.Year
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
            */
            assertion = (report.IITR166.GetValueOrDefault() > 11875 && (report.IITR29 != null && report.IITR29.GetValueOrDefault() <= ConvertToDate(30, 6, report.IITR10.GetValueOrDefault() - 61)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.002665",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian superannuation income stream offset amount exceeds limit",
                    LongDescription = @"Where the age of the taxpayer is 61 or over on 30 June of the reporting period, then the Australian superannuation income stream offset amount must not exceed $11,875",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SuperannuationIncomeStream.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000665" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR166", Value = GetValueOrEmpty(report.IITR166) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmptyDate(report.IITR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR10", Value = GetValueOrEmpty(report.IITR10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000666
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000666()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000666
            Where the age of the taxpayer is 61 or over on 30 June of the reporting period, and the Assessable amount from capped defined benefit income stream provided is greater than zero, then the Australian superannuation income stream offset amount must not be provided
    
            Legacy Rule Format:
            ^IITR166 > 0 AND (^IITR29 <> NULL AND ^IITR29 <= ConvertToDate(30,6, ^IITR10 - 61)) AND ^IITR505 > 0

            Technical Business Rule Format:
            ^IITR166 > 0 AND (^IITR29 <> NULL AND ^IITR29 <= ConvertToDate(30,6, ^IITR10 - 61)) AND ^IITR505 > 0
    
            Data Elements:
    
            ^IITR166 = RP:TaxOffsetClaim.SuperannuationIncomeStream.Amount
    
            ^IITR10 = RP:Report.TargetFinancial.Year
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
    
            ^IITR505 = RP:Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount
            */
            assertion = (report.IITR166.GetValueOrDefault() > 0 && (report.IITR29 != null && report.IITR29.GetValueOrDefault() <= ConvertToDate(30, 6, report.IITR10.GetValueOrDefault() - 61)) && report.IITR505.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000666",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian superannuation income stream offset amount must not be provided",
                    LongDescription = @"Where the age of the taxpayer is 61 or over on 30 June of the reporting period, and the Assessable amount from capped defined benefit income stream provided is greater than zero, then the Australian superannuation income stream offset amount must not be provided",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.SuperannuationIncomeStream.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000666" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR166", Value = GetValueOrEmpty(report.IITR166) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmptyDate(report.IITR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR10", Value = GetValueOrEmpty(report.IITR10) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR505", Value = GetValueOrEmpty(report.IITR505) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000669
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000669()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000669
            Scheme for Reporting Party Working Holiday Maker must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.WHM:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.WHM:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPWHMExists ) && (report.RPWHMIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000669",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Working Holiday Makers must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPWHMOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000669" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000670
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000670()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000670
            Scheme for Reporting Party Partnership must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.Partnership:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.Partnership:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPPartnershipExists ) && (report.RPPartnershipIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000670",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Partnership must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPartnershipOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000670" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000671
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000671()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000671
            Scheme for Reporting Party Trust must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.Trust:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.Trust:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPTrustExists ) && (report.RPTrustIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000671",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Trust must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrustOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000671" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000672
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000672()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000672
            Scheme for Reporting Party Primary Production must be set to "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            ^RP.Prim:IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Technical Business Rule Format:
            ^RP.Prim:IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
            */
            assertion = ((report.RPPrimExists ) && (report.RPPrimIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000672",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party Primary Production must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000672" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000673
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000673()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000673
            Invalid context. The number of Reporting Party Partnership contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP.Partnership:Context) > 1

            Technical Business Rule Format:
            Count(^RP.Partnership:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPPartnershipCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000673",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Partnership contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPartnershipLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000673" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000674
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000674()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000674
            Invalid context. The number of Reporting Party Trust contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP.Trust:Context) > 1

            Technical Business Rule Format:
            Count(^RP.Trust:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPTrustCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000674",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Trust contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrustLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000674" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000675
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000675()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000675
            Invalid context. The number of Reporting Party Primary Production contexts must not exceed 1.
    
            Legacy Rule Format:
            Count(^RP.Prim:Context) > 1

            Technical Business Rule Format:
            Count(^RP.Prim:Context) > 1
    
            Data Elements:
            */
            assertion = (Count(report.RPPrimCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000675",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Primary Production contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000675" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.000676
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR000676()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.000676
            Daytime phone area code has a maximum field length of 4.
    
            Legacy Rule Format:
            Length(^IITR579) > 4

            Technical Business Rule Format:
            Length(^IITR579) > 4
    
            Data Elements:
    
            ^IITR579 = RP:ElectronicContact.Telephone.Country.Code
            */
            assertion = (Length(report.IITR579) > 4);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000676",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Daytime phone area code must not exceed 4 characters",
                    LongDescription = @"For Daytime phone area code, the maximum allowable characters for this form is 4",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000676" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR579", Value = GetValueOrEmpty(report.IITR579) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100009
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character
    
            Legacy Rule Format:
            (^IITR533 <> BLANK) AND (NotCharacterInSet(^IITR533, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR533 <> BLANK) AND (NotCharacterInSet(^IITR533, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR533 = RP.SPOUSE:Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR533) != true && !(IsMatch(report.IITR533, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Spouse's Customer reference number contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100009" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR533", Value = GetValueOrEmpty(report.IITR533) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100010
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * - = _ ' . ? / or a space character
    
            Legacy Rule Format:
            (^IITR51 <> BLANK) AND (NotCharacterInSet(^IITR51, 'a-z,A-Z,0-9,"!","@","$","%","&","*","-","_","=","'",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR51 <> BLANK) AND (NotCharacterInSet(^IITR51, 'a-z,A-Z,0-9,"!","@","$","%","&","*","-","_","=","'",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR51 = RP:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR51) != true && !(IsMatch(report.IITR51, @"^[a-z,A-Z,0-9,""!@\$%&\*\-_='\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact E-mail address contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * - = _ ' . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR51", Value = GetValueOrEmpty(report.IITR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100011
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100011()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100011
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character
    
            Legacy Rule Format:
            (^IITR536 <> BLANK) AND (NotCharacterInSet(^IITR536, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR536 <> BLANK) AND (NotCharacterInSet(^IITR536, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR536 = INT.TrueAndCorrect:Identifiers.TaxAgentClientReference.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR536) != true && !(IsMatch(report.IITR536, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Client's reference contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentClientReference.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100011" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR536", Value = GetValueOrEmpty(report.IITR536) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100029
            When the taxpayer is claiming a refund of franking credits for a short form Income Tax Return Type, the taxable income or loss amount must be less than $18,201, the taxpayer must be an Australian resident for tax purposes, the amount of franking credits must be less than $5,461, and the total tax offset amount must be zero.
    
            Legacy Rule Format:
            (^IITR13 <> NULL) AND ((^IITR157 > 18200) OR (^IITR168 > 0) OR (^IITR19 <> TRUE) OR (^IITR120 > 5460))

            Technical Business Rule Format:
            (^IITR13 <> NULL) AND ((^IITR157 > 18200) OR (^IITR168 > 0) OR (^IITR19 <> TRUE) OR (^IITR120 > 5460))
    
            Data Elements:
    
            ^IITR13 = RP:IncomeTax.ReturnType.Code
    
            ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
            ^IITR157 = RP:Income.Taxable.Amount
    
            ^IITR168 = RP:TaxOffsetClaim.Total.Amount
            */
            assertion = (report.IITR13 != null && (report.IITR157.GetValueOrDefault() > 18200 || report.IITR168.GetValueOrDefault() > 0 || report.IITR19 != true || report.IITR120.GetValueOrDefault() > 5460));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Criteria for a taxpayer to claim a refund of franking credits are not correct",
                    LongDescription = @"When the 'Income Tax Return Type' is provided, implying that the return is to be treated as a short Individual return, and the taxpayer is claiming a refund of franking credits, the 'Taxable income or loss' amount must be below $18,201, the 'Australian resident' indicator must be set to 'yes' (true), the 'Dividends - Franking credit' amount must be below $5,461, and the 'Total tax offsets' amount must be zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ReturnType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100029" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = GetValueOrEmpty(report.IITR13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = GetValueOrEmpty(report.IITR157) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR168", Value = GetValueOrEmpty(report.IITR168) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.IITR120) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100030
            When the taxpayer is claiming a refund of franking credits for a short form Income Tax Return Type, and the taxpayer is younger than 18 years old, they must be an Australian resident for tax purposes, the amount of franking credits must be less than $126, and the taxable income or loss amount must be less than $417.
    
            Legacy Rule Format:
            (^IITR13 <> NULL) AND (^IITR29 > ConvertToDate(Day(^RP:EndDate), MonthAsInt(^RP:EndDate), Year(^RP:EndDate) - 18)) AND ((^IITR19 <> TRUE) OR (^IITR120 > 125) OR (^IITR157 > 416))

            Technical Business Rule Format:
            (^IITR13 <> NULL) AND (^IITR29 > ConvertToDate(Day(^RP:EndDate), MonthAsInt(^RP:EndDate), Year(^RP:EndDate) - 18)) AND ((^IITR19 <> TRUE) OR (^IITR120 > 125) OR (^IITR157 > 416))
    
            Data Elements:
    
            ^IITR13 = RP:IncomeTax.ReturnType.Code
    
            ^EndDate = EndDate
    
            ^IITR19 = RP:Residency.TaxPurposesPersonStatus.Indicator
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
            ^IITR157 = RP:Income.Taxable.Amount
            */
            assertion = (report.IITR13 != null && report.IITR29.GetValueOrDefault() > ConvertToDate(Day(report.RPEndDate), MonthAsInt(report.RPEndDate), Year(report.RPEndDate) - 18) && (report.IITR19 != true || report.IITR120.GetValueOrDefault() > 125 || report.IITR157.GetValueOrDefault() > 416));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Criteria for a taxpayer to claim a refund of franking credits are not correct",
                    LongDescription = @"When the 'Income Tax Return Type' is provided, implying that the return is to be treated as a short Individual return, the taxpayer is claiming a refund of franking credits, and the taxpayer is younger than 18 years old, the 'Australian resident' indicator must be set to 'yes' (true), the 'Dividends - Franking credit' amount must be below $126, and the 'Taxable income or loss' amount must be below $417",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ReturnType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = GetValueOrEmpty(report.IITR13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmptyDate(report.IITR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR19", Value = GetValueOrEmpty(report.IITR19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.IITR120) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR157", Value = GetValueOrEmpty(report.IITR157) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100046
            The taxpayer was under 18 years of age on the last day of the financial year of the return and under 18 income is missing at Adjustments. If this amount is not completed the taxpayer may be taxed at a higher rate than necessary.
    
            Legacy Rule Format:
            (^IITR29 > ConvertToDate(Day(^RP:EndDate), MonthAsInt(^RP:EndDate), Year(^RP:EndDate) - 18)) AND (^IITR188 = NULL)

            Technical Business Rule Format:
            (^IITR29 > ConvertToDate(Day(^RP:EndDate), MonthAsInt(^RP:EndDate), Year(^RP:EndDate) - 18)) AND (^IITR188 = NULL)
    
            Data Elements:
    
            ^IITR188 = RP:IncomeTax.ExceptedNetIncomeUnder18.Amount
    
            ^EndDate = EndDate
    
            ^IITR29 = RP:PersonDemographicDetails.Birth.Date
            */
            assertion = (report.IITR29.GetValueOrDefault() > ConvertToDate(Day(report.RPEndDate), MonthAsInt(report.RPEndDate), Year(report.RPEndDate) - 18) && report.IITR188 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Under 18 excepted net income amount must be provided",
                    LongDescription = @"When the taxpayer is under 18 years of age on the last day of the financial year of the return, the 'Under 18 ' excepted net income amount must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ExceptedNetIncomeUnder18.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100046" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR29", Value = GetValueOrEmptyDate(report.IITR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR188", Value = GetValueOrEmpty(report.IITR188) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100063()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100063
            Where the Spouse's Customer reference number is present, then the Spouse's family name, first name, date of birth and gender code are required.
    
            Legacy Rule Format:
            ^IITR533 <> BLANK AND (^IITR213 = BLANK OR ^IITR214 = BLANK OR ^IITR217 = NULL OR ^IITR218 = NULL)

            Technical Business Rule Format:
            ^IITR533 <> BLANK AND (^IITR213 = BLANK OR ^IITR214 = BLANK OR ^IITR217 = NULL OR ^IITR218 = NULL)
    
            Data Elements:
    
            ^IITR533 = RP.SPOUSE:Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier
    
            ^IITR213 = RP.SPOUSE:PersonNameDetails.FamilyName.Text
    
            ^IITR214 = RP.SPOUSE:PersonNameDetails.GivenName.Text
    
            ^IITR217 = RP.SPOUSE:PersonDemographicDetails.Birth.Date
    
            ^IITR218 = RP.SPOUSE:PersonDemographicDetails.Sex.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR533) != true && (string.IsNullOrWhiteSpace(report.IITR213)  || string.IsNullOrWhiteSpace(report.IITR214)  || report.IITR217 == null || report.IITR218 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.130063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Spouse details are required",
                    LongDescription = @"Where the Spouse's Customer reference number is present, then the Spouse's or de facto spouse's family name, first name, date of birth and gender code are required",
                    Location = "/xbrli:xbrl/tns:Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR533", Value = GetValueOrEmpty(report.IITR533) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR213", Value = GetValueOrEmpty(report.IITR213) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR214", Value = GetValueOrEmpty(report.IITR214) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR217", Value = GetValueOrEmptyDate(report.IITR217) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR218", Value = GetValueOrEmpty(report.IITR218) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100070
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100070()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100070
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            Length(^IITR541) > 15

            Technical Business Rule Format:
            Length(^IITR541) > 15
    
            Data Elements:
    
            ^IITR541 = INT.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.IITR541) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100070",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Telephone number must not exceed 15 characters",
                    LongDescription = @"For Declarer Telephone number, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100070" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR541", Value = report.IITR541 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100072
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100072()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100072
            Other given name/s present without the first name.
    
            Legacy Rule Format:
            (^IITR24 = BLANK) AND (^IITR25 <> BLANK)

            Technical Business Rule Format:
            (^IITR24 = BLANK) AND (^IITR25 <> BLANK)
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR24)  && string.IsNullOrWhiteSpace(report.IITR25) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000472",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name is required",
                    LongDescription = @"Other given name is provided without First name",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100072" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100073
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100073()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100073
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            (FoundSet(^IITR24, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IITR24, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IITR24, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100073" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100074
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100074()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100074
            Individual name field contains 'Exec for', 'Rep for' or 'Trustee for' at the beginning of the name field with a space after, in the field with a space before and after or at the end of the field with a space before. These abbreviations cannot be used.
    
            Legacy Rule Format:
            (FoundSet(^IITR25, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IITR25, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IITR25, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100074" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100075
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            (^IITR25 <> BLANK) AND (NotContainsSet(^IITR25, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IITR25 <> BLANK) AND (NotContainsSet(^IITR25, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR25) != true && !(IsMatch(report.IITR25, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100075" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100076
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            (ContainsSet(^IITR25, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^IITR25, '"--","''","  "'))
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IITR25, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100076" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100077
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100077()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100077
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^IITR25, ' - ')

            Technical Business Rule Format:
            Contains(^IITR25, ' - ')
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.IITR25 == null ? false : report.IITR25.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100077" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100078
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100078()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100078
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used in the individual name field.
    
            Legacy Rule Format:
            (FoundSet(^IITR25, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IITR25, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IITR25, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100078" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100079
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100079()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100079
            Individual name field contains suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used in the individual name field.
    
            Legacy Rule Format:
            (FoundSet(^IITR25, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IITR25, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IITR25, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100079" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100081
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            (^IITR25 <> BLANK) AND (NotCharacterInSet(^IITR25, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Technical Business Rule Format:
            (^IITR25 <> BLANK) AND (NotCharacterInSet(^IITR25, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^IITR25 = RP:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR25) != true && !(IsMatch(report.IITR25, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500042",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual other given names contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.OtherGivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100081" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR25", Value = GetValueOrEmpty(report.IITR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100083
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100083()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100083
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            ContainsSet(^IITR24, '"{","}","|","_","\","#"')

            Technical Business Rule Format:
            ContainsSet(^IITR24, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IITR24, @"\{|\}|\||_|\\|#", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual First given name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100083" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100084
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100084()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100084
            Field contains all numeric characters with any combination of spaces; at least one alpha must be present.
    
            Legacy Rule Format:
            (^IITR24 <> BLANK) AND (NotContainsSet(^IITR24, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IITR24 <> BLANK) AND (NotContainsSet(^IITR24, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IITR24) != true && !(IsMatch(report.IITR24, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100084" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100085
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100085()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100085
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            (ContainsSet(^IITR24, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^IITR24, '"--","''","  "'))
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IITR24, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100085" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100086
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100086()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100086
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^IITR24, ' - ')

            Technical Business Rule Format:
            Contains(^IITR24, ' - ')
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = (report.IITR24 == null ? false : report.IITR24.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100086" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100087
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100087()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100087
            Individual name field contains titles MR, MRS, MISS or MS at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used in the individual name field.
    
            Legacy Rule Format:
            (FoundSet(^IITR24, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IITR24, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IITR24, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100087" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100088
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100088()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100088
            Individual name field contains suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR at the beginning of the field with a space after, in the field with a space before and after, at the end of the field with a space before or alone in the field. These abbreviations cannot be used in the individual name field.
    
            Legacy Rule Format:
            (FoundSet(^IITR24, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IITR24, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IITR24 = RP:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IITR24, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.GivenName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100088" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR24", Value = GetValueOrEmpty(report.IITR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100089
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100089()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100089
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^IITR538, ' - ')

            Technical Business Rule Format:
            Contains(^IITR538, ' - ')
    
            Data Elements:
    
            ^IITR538 = INT.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.IITR538 == null ? false : report.IITR538.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100089" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR538", Value = report.IITR538 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100094
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100094()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100094
            Reporting Party Declarer Telephone area code has a maximum field length of 4.
    
            Legacy Rule Format:
            Length(^IITR580) > 4

            Technical Business Rule Format:
            Length(^IITR580) > 4
    
            Data Elements:
    
            ^IITR580 = RP.TrueAndCorrect:ElectronicContact.Telephone.Country.Code
            */
            assertion = (Length(report.IITR580) > 4);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100094",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Declarer Telephone area code must not exceed 4 characters",
                    LongDescription = @"For Reporting Party Declarer Telephone area code, the maximum allowable characters for this form is 4.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100094" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR580", Value = GetValueOrEmpty(report.IITR580) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.100095
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR100095()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.100095
            Intermediary Declarer Telephone area code has a maximum field length of 4.
    
            Legacy Rule Format:
            Length(^IITR581) > 4

            Technical Business Rule Format:
            Length(^IITR581) > 4
    
            Data Elements:
    
            ^IITR581 = INT.TrueAndCorrect:ElectronicContact.Telephone.Country.Code
            */
            assertion = (Length(report.IITR581) > 4);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100095",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declarer Telephone area code must not exceed 4 characters",
                    LongDescription = @"For Intermediary Declarer Telephone area code, the maximum allowable characters for this form is 4.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Country.Code[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.100095" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR581", Value = report.IITR581 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.130070
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR130070()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.130070
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            Length(^IITR553) > 15

            Technical Business Rule Format:
            Length(^IITR553) > 15
    
            Data Elements:
    
            ^IITR553 = RP.TrueAndCorrect:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.IITR553) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.100070",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Telephone number must not exceed 15 characters",
                    LongDescription = @"For Declarer Telephone number, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.130070" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR553", Value = GetValueOrEmpty(report.IITR553) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.130089
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR130089()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.130089
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^IITR550, ' - ')

            Technical Business Rule Format:
            Contains(^IITR550, ' - ')
    
            Data Elements:
    
            ^IITR550 = RP.TrueAndCorrect:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.IITR550 == null ? false : report.IITR550.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.130089" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR550", Value = GetValueOrEmpty(report.IITR550) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200001
            When there is no electronic claim form indicator present, implying that the return is to be treated as a full Individual return, questions relating to capital gains, foreign entities and foreign source income must be answered.
    
            Legacy Rule Format:
            (^IITR13 = NULL) AND ((^IITR308 = NULL) OR (^IITR315 = NULL) OR (^IITR317 = NULL) OR (^IITR332 = NULL))

            Technical Business Rule Format:
            (^IITR13 = NULL) AND ((^IITR308 = NULL) OR (^IITR315 = NULL) OR (^IITR317 = NULL) OR (^IITR332 = NULL))
    
            Data Elements:
    
            ^IITR13 = RP:IncomeTax.ReturnType.Code
    
            ^IITR308 = RP:CapitalGainsTax.Event.Indicator
    
            ^IITR315 = RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
    
            ^IITR317 = RP:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
    
            ^IITR332 = RP:InternationalDealings.AssetsLocatedOutsideAustralia.Indicator
            */
            assertion = (report.IITR13 == null && (report.IITR308 == null || report.IITR315 == null || report.IITR317 == null || report.IITR332 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Questions relating to the full individual return form must be answered",
                    LongDescription = @"When the 'Income Tax Return Type' is not provided, implying that the return is to be treated as a full Individual return, the questions 'Did you have a CGT event during the year', 'Had a direct or indirect interest in a controlled foreign company (CFC)', 'Have you EVER, either directly or indirectly, caused the transfer of property (incl. money) or services to a non-resident trust estate' and 'Owned, or had interest in, assets located outside Australia with total value of AUD $50,000 or more' must all be answered",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ReturnType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = GetValueOrEmpty(report.IITR13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR308", Value = GetValueOrEmpty(report.IITR308) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR315", Value = GetValueOrEmpty(report.IITR315) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR317", Value = GetValueOrEmpty(report.IITR317) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR332", Value = GetValueOrEmpty(report.IITR332) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200002
            Where the short form Income Tax Return Type is being used, for the claim/application to be valid, certain questions in the return form are not to be answered.  They are: Did you have a CGT event during the year?, Did you have either a direct or indirect interest in a controlled foreign company (CFC)?, Have you ever, either directly or indirectly, caused the transfer of property (incl. money) or services to a non-resident trust estate?, and During the year did you own, or have an interest in, assets located outside Australia which had a total value of AUD $50,000 or more?
    
            Legacy Rule Format:
            (^IITR13 <> NULL) AND ((^IITR308 <> NULL) OR (^IITR315 <> NULL) OR (^IITR317 <> NULL) OR (^IITR332 <> NULL))

            Technical Business Rule Format:
            (^IITR13 <> NULL) AND ((^IITR308 <> NULL) OR (^IITR315 <> NULL) OR (^IITR317 <> NULL) OR (^IITR332 <> NULL))
    
            Data Elements:
    
            ^IITR13 = RP:IncomeTax.ReturnType.Code
    
            ^IITR308 = RP:CapitalGainsTax.Event.Indicator
    
            ^IITR315 = RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
    
            ^IITR317 = RP:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
    
            ^IITR332 = RP:InternationalDealings.AssetsLocatedOutsideAustralia.Indicator
            */
            assertion = (report.IITR13 != null && (report.IITR308 != null || report.IITR315 != null || report.IITR317 != null || report.IITR332 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Questions relating to a full individual return must not be answered on the short individual return",
                    LongDescription = @"When the 'Income Tax Return Type' is provided, implying that the return is to be treated as a short Individual return, the questions 'Did you have a CGT event during the year', 'Had a direct or indirect interest in a controlled foreign company (CFC)', 'Have you ever, either directly or indirectly, caused the transfer of property (incl. money) or services to a non-resident trust estate' and 'Owned, or had interest in, assets located outside Australia with total value of AUD $50,000 or more' must not be answered",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ReturnType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = GetValueOrEmpty(report.IITR13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR308", Value = GetValueOrEmpty(report.IITR308) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR315", Value = GetValueOrEmpty(report.IITR315) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR317", Value = GetValueOrEmpty(report.IITR317) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR332", Value = GetValueOrEmpty(report.IITR332) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200003
            The short form Income Tax Return Type is being used, which implies that the return is for a refund of franking credits, and yet the needed amounts for the dividends have not been provided. Unfranked dividends, franked dividends, franking credit or TFN amounts withheld from dividends amounts are needed to make a valid claim.
    
            Legacy Rule Format:
            (^IITR13 <> NULL AND (^IITR118 = NULL OR ^IITR118 = 0) AND (^IITR119 = NULL OR ^IITR119 = 0) AND (^IITR120 = NULL OR ^IITR120 = 0) AND (^IITR121 = NULL OR ^IITR121 = 0))

            Technical Business Rule Format:
            (^IITR13 <> NULL AND (^IITR118 = NULL OR ^IITR118 = 0) AND (^IITR119 = NULL OR ^IITR119 = 0) AND (^IITR120 = NULL OR ^IITR120 = 0) AND (^IITR121 = NULL OR ^IITR121 = 0))
    
            Data Elements:
    
            ^IITR13 = RP:IncomeTax.ReturnType.Code
    
            ^IITR118 = RP:Income.DividendsUnfranked.Amount
    
            ^IITR119 = RP:Income.DividendsFranked.Amount
    
            ^IITR120 = RP:IncomeTax.FrankingCredits.Amount
    
            ^IITR121 = RP:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            */
            assertion = (report.IITR13 != null && (report.IITR118 == null || report.IITR118 == 0) && (report.IITR119 == null || report.IITR119 == 0) && (report.IITR120 == null || report.IITR120 == 0) && (report.IITR121 == null || report.IITR121 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividend amounts for the application for the refund of franking credits have not been provided",
                    LongDescription = @"When the 'Income Tax Return Type' is provided, implying that the return is to be treated as a short Individual return, the 'Dividends - Unfranked amount', 'Dividends - Franked amount', 'Dividends - Franking credit' or 'TFN amounts withheld from dividends' amounts must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ReturnType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = GetValueOrEmpty(report.IITR13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR118", Value = GetValueOrEmpty(report.IITR118) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR119", Value = GetValueOrEmpty(report.IITR119) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR120", Value = GetValueOrEmpty(report.IITR120) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR121", Value = GetValueOrEmpty(report.IITR121) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200016
            Where the 'Seniors and pensioners - Tax offset code' indicates the taxpayer had a spouse during the year, and Spouse's taxable income, Amount of any exempt pension received by your spouse, or Your spouse's tax-free government pensions amounts is zero, check spouse's income for 'Seniors and pensioners - Tax offset code'.
    
            Legacy Rule Format:
            (InSet(^IITR163, '"Eligible spouse lived apart", "Non eligible spouse lived apart", "Eligible spouse lived together", "Non eligible spouse lived together"')) AND ((^IITR225 <> NULL AND ^IITR225 = 0) OR (^IITR230 <> NULL AND ^IITR230 = 0) OR (^IITR232 <> NULL AND ^IITR232 = 0))

            Technical Business Rule Format:
            (InSet(^IITR163, '"Eligible spouse lived apart", "Non eligible spouse lived apart", "Eligible spouse lived together", "Non eligible spouse lived together"')) AND ((^IITR225 <> NULL AND ^IITR225 = 0) OR (^IITR230 <> NULL AND ^IITR230 = 0) OR (^IITR232 <> NULL AND ^IITR232 = 0))
    
            Data Elements:
    
            ^IITR225 = RP.SPOUSE:Income.Taxable.Amount
    
            ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code
    
            ^IITR230 = RP.SPOUSE:Pension.SpouseExempt.Amount
    
            ^IITR232 = RP.SPOUSE:Pension.TaxFree.Amount
            */
            assertion = (IsMatch(report.IITR163, @"^(Eligible spouse lived apart|Non eligible spouse lived apart|Eligible spouse lived together|Non eligible spouse lived together)$") && (report.IITR225 != null && report.IITR225 == 0 || report.IITR230 != null && report.IITR230 == 0 || report.IITR232 != null && report.IITR232 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200016",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check spouse's income for senior and pensioner tax offset",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount[@contextRef='" + report.RPSPOUSEId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200016" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR163", Value = GetValueOrEmpty(report.IITR163) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR225", Value = GetValueOrEmpty(report.IITR225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR230", Value = GetValueOrEmpty(report.IITR230) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR232", Value = GetValueOrEmpty(report.IITR232) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200018
            Rebate income is required to determine eligibility for Seniors and pensioners tax offset. Spouse's taxable income is part of the rebate income amount and therefore must be present when the Seniors and pensioners tax offset is equal to "Eligible spouse lived apart", "Non eligible spouse lived apart", "Eligible spouse lived together" or "Non eligible spouse lived together."
    
            Legacy Rule Format:
            (InSet(^IITR163, '"Eligible spouse lived apart", "Non eligible spouse lived apart", "Eligible spouse lived together", "Non eligible spouse lived together"')) AND (^IITR225 = NULL)

            Technical Business Rule Format:
            (InSet(^IITR163, '"Eligible spouse lived apart", "Non eligible spouse lived apart", "Eligible spouse lived together", "Non eligible spouse lived together"')) AND (^IITR225 = NULL)
    
            Data Elements:
    
            ^IITR225 = RP.SPOUSE:Income.Taxable.Amount
    
            ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code
            */
            assertion = (IsMatch(report.IITR163, @"^(Eligible spouse lived apart|Non eligible spouse lived apart|Eligible spouse lived together|Non eligible spouse lived together)$") && report.IITR225 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200018",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable income of spouse must be present for claim of Seniors and pensioners tax offset",
                    LongDescription = @"Rebate income is required to determine eligibility for Seniors and pensioners tax offset. Spouse's taxable income is part of the rebate income amount and therefore must be present when the Seniors and pensioners tax offset is equal to ""Eligible spouse lived apart"", ""Non eligible spouse lived apart"", ""Eligible spouse lived together"" or ""Non eligible spouse lived together.""",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount[@contextRef='" + report.RPSPOUSEId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200018" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR163", Value = GetValueOrEmpty(report.IITR163) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR225", Value = GetValueOrEmpty(report.IITR225) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200021
            Where the taxpayer is claiming for the Seniors and pensioners tax offset, and the tax offset code is indicative of the taxpayer having a spouse, the spouse's gender must be provided in the spouse details section of the return form.
    
            Legacy Rule Format:
            ^IITR163 <> NULL AND ^IITR163 <> 'Single' AND ^IITR218 = NULL

            Technical Business Rule Format:
            ^IITR163 <> NULL AND ^IITR163 <> 'Single' AND ^IITR218 = NULL
    
            Data Elements:
    
            ^IITR218 = RP.SPOUSE:PersonDemographicDetails.Sex.Code
    
            ^IITR163 = RP:TaxOffsetClaim.SeniorAndPensioner.Code
            */
            assertion = (report.IITR163 != null && report.IITR163 != @"Single" && report.IITR218 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.230021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For the Seniors and pensioners tax offset claim, provide the spouse's gender",
                    LongDescription = @"Where the taxpayer is claiming for the Seniors and pensioners tax offset, and the tax offset code is indicative of the taxpayer having a spouse, the spouse's gender must be provided in the spouse details section of the return form.",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Sex.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR163", Value = GetValueOrEmpty(report.IITR163) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR218", Value = GetValueOrEmpty(report.IITR218) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200024()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200024
            If Seniors and pensioners - Veteran code is "Spouse Veteran War Widow Widower" or "Both", then details of the taxpayer's spouse is required including Spouse's taxable income, Amount of any Australian Government pensions and allowances received by your spouse in the year of return and Amount of any exempt pension income received by your spouse in the year of the return
    
            Legacy Rule Format:
            (InSet(^IITR164, '"Spouse Veteran War Widow Widower", "Both"')) AND (^IITR225 = NULL OR ^IITR229 = NULL OR ^IITR230 = NULL)

            Technical Business Rule Format:
            (InSet(^IITR164, '"Spouse Veteran War Widow Widower", "Both"')) AND (^IITR225 = NULL OR ^IITR229 = NULL OR ^IITR230 = NULL)
    
            Data Elements:
    
            ^IITR225 = RP.SPOUSE:Income.Taxable.Amount
    
            ^IITR164 = RP:TaxOffsetClaim.Veteran.Code
    
            ^IITR229 = RP.SPOUSE:Pension.Total.Amount
    
            ^IITR230 = RP.SPOUSE:Pension.SpouseExempt.Amount
            */
            assertion = (IsMatch(report.IITR164, @"^(Spouse Veteran War Widow Widower|Both)$") && (report.IITR225 == null || report.IITR229 == null || report.IITR230 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.200024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Spouse details required",
                    LongDescription = @"If Seniors and pensioners - Veteran code is ""Spouse Veteran War Widow Widower"" or ""Both"", then details of the taxpayer's spouse is required including Spouse's taxable income, Amount of any Australian Government pensions and allowances received by your spouse in the year of return and Amount of any exempt pension income received by your spouse in the year of the return",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount[@contextRef='" + report.RPSPOUSEId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.200024" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR164", Value = GetValueOrEmpty(report.IITR164) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR225", Value = GetValueOrEmpty(report.IITR225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR229", Value = GetValueOrEmpty(report.IITR229) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR230", Value = GetValueOrEmpty(report.IITR230) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IITR.200026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIITR200026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IITR.200026
            Australian superannuation income stream tax offset amount is greater than zero and the Australian annuities and superannuation income streams income amounts are either missing or equal to zero.
    
            Legacy Rule Format:
            (^IITR166 > 0) AND ((^IITR92 = NULL) OR (^IITR92 = 0)) AND ((^IITR93 = NULL) OR (^IITR93 = 0)) AND ((^IITR94 = NULL) OR (^IITR94 = 0)) AND ((^IITR95 = NULL) OR (^IITR95 = 0))

            Technical Business Rule Format:
            (^IITR166 > 0) AND ((^IITR92 = NULL) OR (^IITR92 = 0)) AND ((^IITR93 = NULL) OR (^I