using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationNRFI
{

    public partial class NRFI2024Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private NRFI2024 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="NRFI2024Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public NRFI2024Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="NRFI2024Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public NRFI2024Validator(NRFI2024 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());

        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public NRFI2024 ConsumedReport { get { return report; } private set { report = value; } }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(NRFI2024 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            #region Manual Rules - Rules marked for manual coding

            #region VR.ATO.NRFI.100018

            /*  VR.ATO.NRFI.100018
                Foreign tax return period must be exactly one year when the Foreign income assessment method is "Overseas"

            Legacy Rule Format:
                IF [NRFI8] <> NULL AND [NRFI9] <> NULL AND [NRFI9] <> ConvertToDate(Day([NRFI8]) - 1, MonthAsInt([NRFI8]), Year([NRFI8]) + 1)
                    RETURN VALIDATION MESSAGE
                ENDIF

            Technical Business Rule Format:
                ^NRFI8 <> NULL AND ^NRFI9 <> NULL AND ^NRFI9 <> ConvertToDate(Day(^NRFI8) - 1, MonthAsInt(^NRFI8), Year(^NRFI8) + 1)

            Data Elements:

            ^NRFI9 = RP:Period.End.Date

            ^NRFI8 = RP:Period.Start.Date
            */
            VRATONRFI100018(response, report);
            #endregion // VR.ATO.NRFI.100018
            #endregion Manual Rules - Rules marked for manual coding
            VRATOGEN500075();
            VRATONRFI100001();
            VRATONRFI100002();
            VRATONRFI100003();
            VRATONRFI100004();
            VRATONRFI100005();
            VRATONRFI100006();
            VRATONRFI100007();
            VRATONRFI100008();
            VRATONRFI100011();
            VRATONRFI100012();
            VRATONRFI100013();
            VRATONRFI100017();
            VRATONRFI100019();
            VRATONRFI100020();
            VRATONRFI100021();
            VRATONRFI100022();
            VRATONRFI100023();
            VRATONRFI100024();
            VRATONRFI100025();
            VRATONRFI100026();
            VRATONRFI100027();
            VRATONRFI100028();
            VRATONRFI100029();
            VRATONRFI100030();
            VRATONRFI100031();
            VRATONRFI100032();
            VRATONRFI100033();
            VRATONRFI100034();
            VRATONRFI100035();
            VRATONRFI100036();
            VRATONRFI100037();
            VRATONRFI100038();
            VRATONRFI100039();
            VRATONRFI100040();
            VRATONRFI100041();
            VRATONRFI100042();
            VRATONRFI100043();
            VRATONRFI100044();
            VRATONRFI100045();
            VRATONRFI100046();
            VRATONRFI100047();

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

        #region VR.ATO.GEN.500075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500075
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RP:entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'
    
            Data Elements:
    
            ^IdentifierScheme = RP
            */
            assertion = ((report.RPExists == true) && (report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500075" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100001
            Invalid context. The number of Reporting party contexts must equal 1.
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100001" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100002
            At least one Foreign income source country code must be provided
    
            Legacy Rule Format:
            IF COUNT([NRFI1]) = 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^NRFI1) = 0
    
            Data Elements:
    
            ^NRFI1 = RP:AddressDetails.Country.Code
            */
            assertion = (Count(report.NRFI1Collection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one Foreign income source country code must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100002" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100003
            Foreign income source country code must not be "Australia", "Christmas Island", "Norfolk Island" or "Cocos (Keeling) Islands"
    
            Legacy Rule Format:
            IF ANY OCCURRENCE OF([NRFI1]) = SET("au","cc","cx","nf")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            AnyOccurrence(^NRFI1, InSet(^NRFI1, '"au","cc","cx","nf"'))
    
            Data Elements:
    
            ^NRFI1 = RP:AddressDetails.Country.Code
            */
            assertion = (report.NRFI1Collection == null ? false : report.NRFI1Collection.Any(NRFI1Repeat => IsMatch(NRFI1Repeat, @"^(au|cc|cx|nf)$")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income source country code must not be ""Australia"", ""Christmas Island"", ""Norfolk Island"" or ""Cocos (Keeling) Islands""",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI1", Value = "Count(NRFI1)" });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100004
            Foreign occupation code is mandatory
    
            Legacy Rule Format:
            IF [NRFI2] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI2 = NULL
    
            Data Elements:
    
            ^NRFI2 = RP:PersonDemographicDetails.Occupation.Code
            */
            assertion = (report.NRFI2 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.110004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign occupation code must be present",
                    LongDescription = @"Foreign occupation code must be present. Where the individual does not have an occupation, or a relevant occupation code is not available, enter '999000'.",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100004" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI2", Value = report.NRFI2 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100005
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100005()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100005
            Foreign income assessment method is mandatory
    
            Legacy Rule Format:
            IF [NRFI3] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = NULL
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
            */
            assertion = (report.NRFI3 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100005",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income assessment method must be present",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100005" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100006
            When the Foreign income assessment method is "Comprehensive" then at least one income field amount must be provided
    
            Legacy Rule Format:
            IF [NRFI3] = "Comprehensive" AND ([NRFI11] = NULL AND [NRFI12] = NULL AND [NRFI13] = NULL AND [NRFI14] = NULL AND [NRFI15] = NULL AND [NRFI16] = NULL AND [NRFI17] = NULL AND [NRFI18] = NULL AND [NRFI19] = NULL AND [NRFI20] = NULL AND [NRFI21] = NULL AND [NRFI22] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = 'Comprehensive' AND (^NRFI11 = NULL AND ^NRFI12 = NULL AND ^NRFI13 = NULL AND ^NRFI14 = NULL AND ^NRFI15 = NULL AND ^NRFI16 = NULL AND ^NRFI17 = NULL AND ^NRFI18 = NULL AND ^NRFI19 = NULL AND ^NRFI20 = NULL AND ^NRFI21 = NULL AND ^NRFI22 = NULL)
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
    
            ^NRFI11 = RP:Income.SalaryOrWages.Amount
    
            ^NRFI12 = RP:Remuneration.EmploymentAllowances.Amount
    
            ^NRFI13 = RP:Payment.Government.Amount
    
            ^NRFI14 = RP:Payment.Taxable.Amount
    
            ^NRFI15 = RP:Income.Net.Amount
    
            ^NRFI16 = RP:Income.PersonalServicesIncome.Total.Amount
    
            ^NRFI17 = RP:Income.Interest.Gross.Amount
    
            ^NRFI18 = RP:Income.DividendsTotal.Amount
    
            ^NRFI19 = RP:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount
    
            ^NRFI20 = RP:Capital.Gains.Total.Amount
    
            ^NRFI21 = RP:InternationalDealings.ForeignRent.Amount
    
            ^NRFI22 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount
            */
            assertion = (report.NRFI3 == @"Comprehensive" && (report.NRFI11 == null && report.NRFI12 == null && report.NRFI13 == null && report.NRFI14 == null && report.NRFI15 == null && report.NRFI16 == null && report.NRFI17 == null && report.NRFI18 == null && report.NRFI19 == null && report.NRFI20 == null && report.NRFI21 == null && report.NRFI22 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100006",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one income field must be present if the Foreign income assessment method is ""Comprehensive""",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100006" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI11", Value = GetValueOrEmpty(report.NRFI11) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI12", Value = GetValueOrEmpty(report.NRFI12) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI13", Value = GetValueOrEmpty(report.NRFI13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI14", Value = GetValueOrEmpty(report.NRFI14) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI15", Value = GetValueOrEmpty(report.NRFI15) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI16", Value = GetValueOrEmpty(report.NRFI16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI17", Value = GetValueOrEmpty(report.NRFI17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI18", Value = GetValueOrEmpty(report.NRFI18) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI19", Value = GetValueOrEmpty(report.NRFI19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI20", Value = GetValueOrEmpty(report.NRFI20) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI21", Value = GetValueOrEmpty(report.NRFI21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI22", Value = GetValueOrEmpty(report.NRFI22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100007
            Gross foreign income amount is mandatory if the Foreign income assessment method is "Simple"
    
            Legacy Rule Format:
            IF [NRFI3] = "Simple" AND [NRFI4] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = 'Simple' AND ^NRFI4 = NULL
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
    
            ^NRFI4 = RP:InternationalDealings.ForeignIncomeGross.Amount
            */
            assertion = (report.NRFI3 == @"Simple" && report.NRFI4 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income amount must be provided",
                    LongDescription = @"Gross foreign income amount must be present if the Foreign income assessment method provided is ""Simple""",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100007" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI4", Value = GetValueOrEmpty(report.NRFI4) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100008
            When the Foreign income assessment method is "Overseas", all foreign tax return details must be provided
    
            Legacy Rule Format:
            IF [NRFI3] = "Overseas" AND ([NRFI5] = NULLORBLANK OR [NRFI6] = NULL OR [NRFI7] = NULL OR [NRFI8] = NULL OR [NRFI9] = NULL OR [NRFI10] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = 'Overseas' AND (^NRFI5 = BLANK OR ^NRFI6 = NULL OR ^NRFI7 = NULL OR ^NRFI8 = NULL OR ^NRFI9 = NULL OR ^NRFI10 = NULL)
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
    
            ^NRFI5 = RP:Identifiers.TaxIdentificationNumber.Identifier
    
            ^NRFI6 = RP:Residency.TaxPurposesCountry.Code
    
            ^NRFI7 = RP:Report.TargetFinancial.Year
    
            ^NRFI8 = RP:Period.Start.Date
    
            ^NRFI9 = RP:Period.End.Date
    
            ^NRFI10 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
            */
            assertion = (report.NRFI3 == @"Overseas" && (string.IsNullOrWhiteSpace(report.NRFI5) == true || report.NRFI6 == null || report.NRFI7 == null || report.NRFI8 == null || report.NRFI9 == null || report.NRFI10 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All foreign tax return details must be provided",
                    LongDescription = @"When the Foreign income assessment method is ""Overseas"", all foreign tax return details must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI5", Value = GetValueOrEmpty(report.NRFI5) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI6", Value = GetValueOrEmpty(report.NRFI6) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI7", Value = GetValueOrEmpty(report.NRFI7) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI8", Value = GetValueOrEmptyDate(report.NRFI8) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI9", Value = GetValueOrEmptyDate(report.NRFI9) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI10", Value = GetValueOrEmpty(report.NRFI10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100011
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100011()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100011
            The number of Foreign income source country codes must not exceed 10
    
            Legacy Rule Format:
            IF COUNT([NRFI1]) > 10
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^NRFI1) > 10
    
            Data Elements:
    
            ^NRFI1 = RP:AddressDetails.Country.Code
            */
            assertion = (Count(report.NRFI1Collection) > 10);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Foreign income source country codes must not exceed 10",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100011" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100012
            Gross foreign income amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI4] <> NULL AND [NRFI4] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI4, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI4 = RP:InternationalDealings.ForeignIncomeGross.Amount
            */
            assertion = NotMonetary(report.NRFI4, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income amount is not in a valid monetary format",
                    LongDescription = @"Gross foreign income amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100012" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI4", Value = GetValueOrEmpty(report.NRFI4) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100013
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character
    
            Legacy Rule Format:
            IF [NRFI5] <> NULLORBLANK AND ANY CHARACTER OF [NRFI5] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI5 <> BLANK AND NotCharacterInSet(^NRFI5, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "')
    
            Data Elements:
    
            ^NRFI5 = RP:Identifiers.TaxIdentificationNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.NRFI5) != true && !(IsMatch(report.NRFI5, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign tax return Tax File Number equivalent contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxIdentificationNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100013" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI5", Value = GetValueOrEmpty(report.NRFI5) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100017
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100017()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100017
            The Foreign tax return period must overlap with the tax return year
    
            Legacy Rule Format:
            IF ([NRFI8] <> NULL AND [NRFI8] > RP:period.endDate) OR ([NRFI9] <> NULL AND [NRFI9] < RP:period.startDate)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^NRFI8 <> NULL AND ^NRFI8 > ^EndDate) OR (^NRFI9 <> NULL AND ^NRFI9 < ^StartDate1)
    
            Data Elements:
    
            ^NRFI8 = RP:Period.Start.Date
    
            ^EndDate = RP
    
            ^StartDate1 = RP
    
            ^NRFI9 = RP:Period.End.Date
            */
            assertion = (report.NRFI8 != null && report.NRFI8.GetValueOrDefault() > report.RPEndDate || report.NRFI9 != null && report.NRFI9.GetValueOrDefault() < report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Foreign tax return period must overlap with the tax return year",
                    Location = "/xbrli:xbrl/tns:Period.Start.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100017" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI8", Value = GetValueOrEmptyDate(report.NRFI8) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI9", Value = GetValueOrEmptyDate(report.NRFI9) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100019
            Foreign tax return taxable income has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI10] <> NULL AND [NRFI10] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI10, 'S', 11, 0)
    
            Data Elements:
    
            ^NRFI10 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
            */
            assertion = NotMonetary(report.NRFI10, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign tax return taxable income amount is not in a valid monetary format",
                    LongDescription = @"Foreign tax return taxable income may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100019" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI10", Value = GetValueOrEmpty(report.NRFI10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100020()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100020
            Total salary or wages amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI11] <> NULL AND [NRFI11] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI11, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI11 = RP:Income.SalaryOrWages.Amount
            */
            assertion = NotMonetary(report.NRFI11, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary or wages amount is not in a valid monetary format",
                    LongDescription = @"Total salary or wages amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.SalaryOrWages.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100020" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI11", Value = GetValueOrEmpty(report.NRFI11) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100021
            Total allowance amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI12] <> NULL AND [NRFI12] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI12, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI12 = RP:Remuneration.EmploymentAllowances.Amount
            */
            assertion = NotMonetary(report.NRFI12, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total allowance amount is not in a valid monetary format",
                    LongDescription = @"Total allowance amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Remuneration.EmploymentAllowances.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI12", Value = GetValueOrEmpty(report.NRFI12) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100022
            Total government allowances amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI13] <> NULL AND [NRFI13] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI13, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI13 = RP:Payment.Government.Amount
            */
            assertion = NotMonetary(report.NRFI13, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total government allowances amount is not in a valid monetary format",
                    LongDescription = @"Total government allowances amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Payment.Government.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100022" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI13", Value = GetValueOrEmpty(report.NRFI13) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100023
            Total pension amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI14] <> NULL AND [NRFI14] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI14, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI14 = RP:Payment.Taxable.Amount
            */
            assertion = NotMonetary(report.NRFI14, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total pension amount is not in a valid monetary format",
                    LongDescription = @"Total pension amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Payment.Taxable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI14", Value = GetValueOrEmpty(report.NRFI14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100024()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100024
            Net business income has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI15] <> NULL AND [NRFI15] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI15, 'S', 11, 0)
    
            Data Elements:
    
            ^NRFI15 = RP:Income.Net.Amount
            */
            assertion = NotMonetary(report.NRFI15, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net business income amount is not in a valid monetary format",
                    LongDescription = @"Net business income may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100024" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI15", Value = GetValueOrEmpty(report.NRFI15) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100025()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100025
            Total personal services income has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI16] <> NULL AND [NRFI16] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI16, 'S', 11, 0)
    
            Data Elements:
    
            ^NRFI16 = RP:Income.PersonalServicesIncome.Total.Amount
            */
            assertion = NotMonetary(report.NRFI16, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total personal services income amount is not in a valid monetary format",
                    LongDescription = @"Total personal services income may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncome.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100025" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI16", Value = GetValueOrEmpty(report.NRFI16) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100026
            Total interest amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI17] <> NULL AND [NRFI17] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI17, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI17 = RP:Income.Interest.Gross.Amount
            */
            assertion = NotMonetary(report.NRFI17, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100026",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total interest amount is not in a valid monetary format",
                    LongDescription = @"Total interest amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Interest.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100026" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI17", Value = GetValueOrEmpty(report.NRFI17) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100027
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100027()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100027
            Total dividend amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI18] <> NULL AND [NRFI18] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI18, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI18 = RP:Income.DividendsTotal.Amount
            */
            assertion = NotMonetary(report.NRFI18, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total dividend amount is not in a valid monetary format",
                    LongDescription = @"Total dividend amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.DividendsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100027" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI18", Value = GetValueOrEmpty(report.NRFI18) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100028
            Net partnership and trust income amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI19] <> NULL AND [NRFI19] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI19, 'S', 11, 0)
    
            Data Elements:
    
            ^NRFI19 = RP:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount
            */
            assertion = NotMonetary(report.NRFI19, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net partnership and trust income amount is not in a valid monetary format",
                    LongDescription = @"Net partnership and trust income amount may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100028" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI19", Value = GetValueOrEmpty(report.NRFI19) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100029
            Net capital gains amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI20] <> NULL AND [NRFI20] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI20, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI20 = RP:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.NRFI20, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gains amount is not in a valid monetary format",
                    LongDescription = @"Net capital gains amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100029" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI20", Value = GetValueOrEmpty(report.NRFI20) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100030
            Net rent amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI21] <> NULL AND [NRFI21] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI21, 'S', 11, 0)
    
            Data Elements:
    
            ^NRFI21 = RP:InternationalDealings.ForeignRent.Amount
            */
            assertion = NotMonetary(report.NRFI21, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net rent amount is not in a valid monetary format",
                    LongDescription = @"Net rent amount may be signed but must not exceed 11 digits; no other characters are allowed except for the preceding sign",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignRent.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI21", Value = GetValueOrEmpty(report.NRFI21) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100031
            Total other income amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI22] <> NULL AND [NRFI22] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI22, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI22 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount
            */
            assertion = NotMonetary(report.NRFI22, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total other income amount is not in a valid monetary format",
                    LongDescription = @"Total other income amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignEmploymentIncomeOther.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI22", Value = GetValueOrEmpty(report.NRFI22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100032
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100032()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100032
            Total employee work related deduction amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI23] <> NULL AND [NRFI23] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI23, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI23 = RP:Expense.DeductionsTotal.Amount
            */
            assertion = NotMonetary(report.NRFI23, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100032",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total employee work related deduction amount is not in a valid monetary format",
                    LongDescription = @"Total employee work related deduction amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100032" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI23", Value = GetValueOrEmpty(report.NRFI23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100033
            Total depreciation amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI24] <> NULL AND [NRFI24] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI24, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI24 = RP:Expense.DepreciationAllowableDeduction.Amount
            */
            assertion = NotMonetary(report.NRFI24, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total depreciation amount is not in a valid monetary format",
                    LongDescription = @"Total depreciation amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100033" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI24", Value = GetValueOrEmpty(report.NRFI24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100034
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100034()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100034
            Total interest and dividend amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI25] <> NULL AND [NRFI25] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI25, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI25 = RP:IncomeTax.Deduction.InterestDividendClaimed.Amount
            */
            assertion = NotMonetary(report.NRFI25, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total interest and dividend amount is not in a valid monetary format",
                    LongDescription = @"Total interest and dividend amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.InterestDividendClaimed.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100034" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI25", Value = GetValueOrEmpty(report.NRFI25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100035
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100035()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100035
            Total undeducted purchase price of a foreign pension amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI26] <> NULL AND [NRFI26] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI26, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI26 = RP:IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount
            */
            assertion = NotMonetary(report.NRFI26, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total undeducted purchase price of a foreign pension amount is not in a valid monetary format",
                    LongDescription = @"Total undeducted purchase price of a foreign pension amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100035" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI26", Value = GetValueOrEmpty(report.NRFI26) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100036
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100036()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100036
            Total personal super contributions for business has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI27] <> NULL AND [NRFI27] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI27, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI27 = RP:IncomeTax.Deduction.SuperannuationContribution.Amount
            */
            assertion = NotMonetary(report.NRFI27, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total personal super contributions for business amount is not in a valid monetary format",
                    LongDescription = @"Total personal super contributions for business must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.SuperannuationContribution.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100036" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI27", Value = GetValueOrEmpty(report.NRFI27) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100037()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100037
            Total other deduction amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI28] <> NULL AND [NRFI28] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI28, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI28 = RP:Expense.DeductibleOtherTotal.Amount
            */
            assertion = NotMonetary(report.NRFI28, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total other deduction amount is not in a valid monetary format",
                    LongDescription = @"Total other deduction amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DeductibleOtherTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100037" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI28", Value = GetValueOrEmpty(report.NRFI28) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100038()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100038
            Prior year tax losses amount has a maximum field length of 11
    
            Legacy Rule Format:
            IF [NRFI32] <> NULL AND [NRFI32] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^NRFI32, 'U', 11, 0)
    
            Data Elements:
    
            ^NRFI32 = RP:Tax.Losses.Deducted.Amount
            */
            assertion = NotMonetary(report.NRFI32, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Prior year tax losses amount is not in a valid monetary format",
                    LongDescription = @"Prior year tax losses amount must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100038" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI32", Value = GetValueOrEmpty(report.NRFI32) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100039()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100039
            Foreign taxation return country code must not be "Australia", "Christmas Island", "Norfolk Island" or "Cocos (Keeling) Islands"
    
            Legacy Rule Format:
            IF [NRFI6] = SET("au","cc","cx","nf")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            InSet(^NRFI6, '"au","cc","cx","nf"')
    
            Data Elements:
    
            ^NRFI6 = RP:Residency.TaxPurposesCountry.Code
            */
            assertion = IsMatch(report.NRFI6, @"^(au|cc|cx|nf)$");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign taxation return country code must not be ""Australia"", ""Christmas Island"", ""Norfolk Island"" or ""Cocos (Keeling) Islands""",
                    Location = "/xbrli:xbrl/tns:Residency.TaxPurposesCountry.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100039" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI6", Value = GetValueOrEmpty(report.NRFI6) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100040
            Foreign tax return Tax File Number equivalent has a maximum field length of 20
    
            Legacy Rule Format:
            IF [NRFI5] <> NULLORBLANK AND LENGTH([NRFI5]) > 20
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^NRFI5) > 20
    
            Data Elements:
    
            ^NRFI5 = RP:Identifiers.TaxIdentificationNumber.Identifier
            */
            assertion = (Length(report.NRFI5) > 20);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign tax return Tax File Number equivalent must not exceed 20 characters",
                    LongDescription = @"For Foreign tax return Tax File Number equivalent, the maximum allowable characters for this form is 20",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxIdentificationNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100040" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI5", Value = GetValueOrEmpty(report.NRFI5) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100041
            When the Foreign income assessment method is "Simple", overseas assessed and comprehensive assessment details must not be provided
    
            Legacy Rule Format:
            IF [NRFI3] =  "Simple" AND ([NRFI5] <> NULLORBLANK OR [NRFI6] <> NULL OR [NRFI7] <> NULL OR [NRFI8] <> NULL OR [NRFI9] <> NULL OR [NRFI10] <> NULL OR [NRFI11] <> NULL OR [NRFI12] <> NULL OR [NRFI13] <> NULL OR [NRFI14] <> NULL OR [NRFI15] <> NULL OR [NRFI16] <> NULL OR [NRFI17] <> NULL OR [NRFI18] <> NULL OR [NRFI19] <> NULL OR [NRFI20] <> NULL OR [NRFI21] <> NULL OR [NRFI22] <> NULL OR [NRFI23] <> NULL OR [NRFI24] <> NULL OR [NRFI25] <> NULL OR [NRFI26] <> NULL OR [NRFI27] <> NULL OR [NRFI28] <> NULL OR [NRFI32] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = 'Simple' AND (^NRFI5 <> BLANK OR ^NRFI6 <> NULL OR ^NRFI7 <> NULL OR ^NRFI8 <> NULL OR ^NRFI9 <> NULL OR ^NRFI10 <> NULL OR ^NRFI11 <> NULL OR ^NRFI12 <> NULL OR ^NRFI13 <> NULL OR ^NRFI14 <> NULL OR ^NRFI15 <> NULL OR ^NRFI16 <> NULL OR ^NRFI17 <> NULL OR ^NRFI18 <> NULL OR ^NRFI19 <> NULL OR ^NRFI20 <> NULL OR ^NRFI21 <> NULL OR ^NRFI22 <> NULL OR ^NRFI23 <> NULL OR ^NRFI24 <> NULL OR ^NRFI25 <> NULL OR ^NRFI26 <> NULL OR ^NRFI27 <> NULL OR ^NRFI28 <> NULL OR ^NRFI32 <> NULL)
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
    
            ^NRFI5 = RP:Identifiers.TaxIdentificationNumber.Identifier
    
            ^NRFI6 = RP:Residency.TaxPurposesCountry.Code
    
            ^NRFI7 = RP:Report.TargetFinancial.Year
    
            ^NRFI8 = RP:Period.Start.Date
    
            ^NRFI9 = RP:Period.End.Date
    
            ^NRFI10 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
    
            ^NRFI11 = RP:Income.SalaryOrWages.Amount
    
            ^NRFI12 = RP:Remuneration.EmploymentAllowances.Amount
    
            ^NRFI13 = RP:Payment.Government.Amount
    
            ^NRFI14 = RP:Payment.Taxable.Amount
    
            ^NRFI15 = RP:Income.Net.Amount
    
            ^NRFI16 = RP:Income.PersonalServicesIncome.Total.Amount
    
            ^NRFI17 = RP:Income.Interest.Gross.Amount
    
            ^NRFI18 = RP:Income.DividendsTotal.Amount
    
            ^NRFI19 = RP:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount
    
            ^NRFI20 = RP:Capital.Gains.Total.Amount
    
            ^NRFI21 = RP:InternationalDealings.ForeignRent.Amount
    
            ^NRFI22 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount
    
            ^NRFI23 = RP:Expense.DeductionsTotal.Amount
    
            ^NRFI24 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^NRFI25 = RP:IncomeTax.Deduction.InterestDividendClaimed.Amount
    
            ^NRFI26 = RP:IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount
    
            ^NRFI27 = RP:IncomeTax.Deduction.SuperannuationContribution.Amount
    
            ^NRFI28 = RP:Expense.DeductibleOtherTotal.Amount
    
            ^NRFI32 = RP:Tax.Losses.Deducted.Amount
            */
            assertion = (report.NRFI3 == @"Simple" && (string.IsNullOrWhiteSpace(report.NRFI5) != true || report.NRFI6 != null || report.NRFI7 != null || report.NRFI8 != null || report.NRFI9 != null || report.NRFI10 != null || report.NRFI11 != null || report.NRFI12 != null || report.NRFI13 != null || report.NRFI14 != null || report.NRFI15 != null || report.NRFI16 != null || report.NRFI17 != null || report.NRFI18 != null || report.NRFI19 != null || report.NRFI20 != null || report.NRFI21 != null || report.NRFI22 != null || report.NRFI23 != null || report.NRFI24 != null || report.NRFI25 != null || report.NRFI26 != null || report.NRFI27 != null || report.NRFI28 != null || report.NRFI32 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect Foreign income assessment method details provided",
                    LongDescription = @"When the Foreign income assessment method is ""Simple"", overseas assessed and comprehensive assessment details must not be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100041" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI5", Value = GetValueOrEmpty(report.NRFI5) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI6", Value = GetValueOrEmpty(report.NRFI6) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI7", Value = GetValueOrEmpty(report.NRFI7) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI8", Value = GetValueOrEmptyDate(report.NRFI8) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI9", Value = GetValueOrEmptyDate(report.NRFI9) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI10", Value = GetValueOrEmpty(report.NRFI10) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI11", Value = GetValueOrEmpty(report.NRFI11) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI12", Value = GetValueOrEmpty(report.NRFI12) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI13", Value = GetValueOrEmpty(report.NRFI13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI14", Value = GetValueOrEmpty(report.NRFI14) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI15", Value = GetValueOrEmpty(report.NRFI15) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI16", Value = GetValueOrEmpty(report.NRFI16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI17", Value = GetValueOrEmpty(report.NRFI17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI18", Value = GetValueOrEmpty(report.NRFI18) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI19", Value = GetValueOrEmpty(report.NRFI19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI20", Value = GetValueOrEmpty(report.NRFI20) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI21", Value = GetValueOrEmpty(report.NRFI21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI22", Value = GetValueOrEmpty(report.NRFI22) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI23", Value = GetValueOrEmpty(report.NRFI23) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI24", Value = GetValueOrEmpty(report.NRFI24) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI25", Value = GetValueOrEmpty(report.NRFI25) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI26", Value = GetValueOrEmpty(report.NRFI26) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI27", Value = GetValueOrEmpty(report.NRFI27) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI28", Value = GetValueOrEmpty(report.NRFI28) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI32", Value = GetValueOrEmpty(report.NRFI32) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100042
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100042()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100042
            When the Foreign income assessment method is "Overseas", simple self assessment and comprehensive assessment details must not be provided
    
            Legacy Rule Format:
            IF [NRFI3] =  "Overseas" AND ([NRFI4] <> NULL OR [NRFI11] <> NULL OR [NRFI12] <> NULL OR [NRFI13] <> NULL OR [NRFI14] <> NULL OR [NRFI15] <> NULL OR [NRFI16] <> NULL OR [NRFI17] <> NULL OR [NRFI18] <> NULL OR [NRFI19] <> NULL OR [NRFI20] <> NULL OR [NRFI21] <> NULL OR [NRFI22] <> NULL OR [NRFI23] <> NULL OR [NRFI24] <> NULL OR [NRFI25] <> NULL OR [NRFI26] <> NULL OR [NRFI27] <> NULL OR [NRFI28] <> NULL OR [NRFI32] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = 'Overseas' AND (^NRFI4 <> NULL OR ^NRFI11 <> NULL OR ^NRFI12 <> NULL OR ^NRFI13 <> NULL OR ^NRFI14 <> NULL OR ^NRFI15 <> NULL OR ^NRFI16 <> NULL OR ^NRFI17 <> NULL OR ^NRFI18 <> NULL OR ^NRFI19 <> NULL OR ^NRFI20 <> NULL OR ^NRFI21 <> NULL OR ^NRFI22 <> NULL OR ^NRFI23 <> NULL OR ^NRFI24 <> NULL OR ^NRFI25 <> NULL OR ^NRFI26 <> NULL OR ^NRFI27 <> NULL OR ^NRFI28 <> NULL OR ^NRFI32 <> NULL)
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
    
            ^NRFI4 = RP:InternationalDealings.ForeignIncomeGross.Amount
    
            ^NRFI11 = RP:Income.SalaryOrWages.Amount
    
            ^NRFI12 = RP:Remuneration.EmploymentAllowances.Amount
    
            ^NRFI13 = RP:Payment.Government.Amount
    
            ^NRFI14 = RP:Payment.Taxable.Amount
    
            ^NRFI15 = RP:Income.Net.Amount
    
            ^NRFI16 = RP:Income.PersonalServicesIncome.Total.Amount
    
            ^NRFI17 = RP:Income.Interest.Gross.Amount
    
            ^NRFI18 = RP:Income.DividendsTotal.Amount
    
            ^NRFI19 = RP:InternationalDealings.ForeignSourceCompanyIncomeOther.Amount
    
            ^NRFI20 = RP:Capital.Gains.Total.Amount
    
            ^NRFI21 = RP:InternationalDealings.ForeignRent.Amount
    
            ^NRFI22 = RP:InternationalDealings.ForeignEmploymentIncomeOther.Amount
    
            ^NRFI23 = RP:Expense.DeductionsTotal.Amount
    
            ^NRFI24 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^NRFI25 = RP:IncomeTax.Deduction.InterestDividendClaimed.Amount
    
            ^NRFI26 = RP:IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount
    
            ^NRFI27 = RP:IncomeTax.Deduction.SuperannuationContribution.Amount
    
            ^NRFI28 = RP:Expense.DeductibleOtherTotal.Amount
    
            ^NRFI32 = RP:Tax.Losses.Deducted.Amount
            */
            assertion = (report.NRFI3 == @"Overseas" && (report.NRFI4 != null || report.NRFI11 != null || report.NRFI12 != null || report.NRFI13 != null || report.NRFI14 != null || report.NRFI15 != null || report.NRFI16 != null || report.NRFI17 != null || report.NRFI18 != null || report.NRFI19 != null || report.NRFI20 != null || report.NRFI21 != null || report.NRFI22 != null || report.NRFI23 != null || report.NRFI24 != null || report.NRFI25 != null || report.NRFI26 != null || report.NRFI27 != null || report.NRFI28 != null || report.NRFI32 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100042",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect Foreign income assessment method details provided",
                    LongDescription = @"When the Foreign income assessment method is ""Overseas"", simple self assessment and comprehensive assessment details must not be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100042" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI4", Value = GetValueOrEmpty(report.NRFI4) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI11", Value = GetValueOrEmpty(report.NRFI11) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI12", Value = GetValueOrEmpty(report.NRFI12) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI13", Value = GetValueOrEmpty(report.NRFI13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI14", Value = GetValueOrEmpty(report.NRFI14) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI15", Value = GetValueOrEmpty(report.NRFI15) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI16", Value = GetValueOrEmpty(report.NRFI16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI17", Value = GetValueOrEmpty(report.NRFI17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI18", Value = GetValueOrEmpty(report.NRFI18) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI19", Value = GetValueOrEmpty(report.NRFI19) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI20", Value = GetValueOrEmpty(report.NRFI20) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI21", Value = GetValueOrEmpty(report.NRFI21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI22", Value = GetValueOrEmpty(report.NRFI22) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI23", Value = GetValueOrEmpty(report.NRFI23) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI24", Value = GetValueOrEmpty(report.NRFI24) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI25", Value = GetValueOrEmpty(report.NRFI25) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI26", Value = GetValueOrEmpty(report.NRFI26) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI27", Value = GetValueOrEmpty(report.NRFI27) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI28", Value = GetValueOrEmpty(report.NRFI28) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI32", Value = GetValueOrEmpty(report.NRFI32) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100043
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100043()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100043
            When the Foreign income assessment method is "Comprehensive", simple self assessment and overseas assessed details must not be provided
    
            Legacy Rule Format:
            IF [NRFI3] =  "Comprehensive" AND ([NRFI4] <> NULL OR [NRFI5] <> NULLORBLANK OR [NRFI6] <> NULL OR [NRFI7] <> NULL OR [NRFI8] <> NULL OR [NRFI9] <> NULL OR [NRFI10] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI3 = 'Comprehensive' AND (^NRFI4 <> NULL OR ^NRFI5 <> BLANK OR ^NRFI6 <> NULL OR ^NRFI7 <> NULL OR ^NRFI8 <> NULL OR ^NRFI9 <> NULL OR ^NRFI10 <> NULL)
    
            Data Elements:
    
            ^NRFI3 = RP:IncomeTax.NonResidentAssessment.Code
    
            ^NRFI4 = RP:InternationalDealings.ForeignIncomeGross.Amount
    
            ^NRFI5 = RP:Identifiers.TaxIdentificationNumber.Identifier
    
            ^NRFI6 = RP:Residency.TaxPurposesCountry.Code
    
            ^NRFI7 = RP:Report.TargetFinancial.Year
    
            ^NRFI8 = RP:Period.Start.Date
    
            ^NRFI9 = RP:Period.End.Date
    
            ^NRFI10 = RP:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
            */
            assertion = (report.NRFI3 == @"Comprehensive" && (report.NRFI4 != null || string.IsNullOrWhiteSpace(report.NRFI5) != true || report.NRFI6 != null || report.NRFI7 != null || report.NRFI8 != null || report.NRFI9 != null || report.NRFI10 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100043",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect Foreign income assessment method details provided",
                    LongDescription = @"When the Foreign income assessment method is ""Comprehensive"", simple self assessment and overseas assessed details must not be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.NonResidentAssessment.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100043" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI3", Value = report.NRFI3 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI4", Value = GetValueOrEmpty(report.NRFI4) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI5", Value = GetValueOrEmpty(report.NRFI5) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI6", Value = GetValueOrEmpty(report.NRFI6) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI7", Value = GetValueOrEmpty(report.NRFI7) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI8", Value = GetValueOrEmptyDate(report.NRFI8) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI9", Value = GetValueOrEmptyDate(report.NRFI9) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI10", Value = GetValueOrEmpty(report.NRFI10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100044
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100044()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100044
            Foreign occupation description is mandatory
    
            Legacy Rule Format:
            IF [NRFI33] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^NRFI33 = BLANK
    
            Data Elements:
    
            ^NRFI33 = RP:PersonDemographicDetails.Occupation.Description
            */
            assertion = (string.IsNullOrWhiteSpace(report.NRFI33) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100044",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign occupation description must be present",
                    LongDescription = @"Foreign occupation description must be present. Where the individual does not have an occupation, or a relevant occupation is not available, enter 'Occupation not listed'.",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100044" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI33", Value = report.NRFI33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100045
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100045()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100045
            Foreign occupation description has a maximum length of 100
    
            Legacy Rule Format:
            IF LENGTH([NRFI33]) > 100
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^NRFI33) > 100
    
            Data Elements:
    
            ^NRFI33 = RP:PersonDemographicDetails.Occupation.Description
            */
            assertion = (Length(report.NRFI33) > 100);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100045",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign occupation description must not exceed 100 characters",
                    LongDescription = @"For Foreign occupation description, the maximum allowable characters for this form is 100",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100045" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI33", Value = report.NRFI33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100046
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character
    
            Legacy Rule Format:
            IF ([NRFI33] <> NULLORBLANK) AND (ANY CHARACTER OF [NRFI33] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^NRFI33 <> BLANK) AND (NotCharacterInSet(^NRFI33, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^NRFI33 = RP:PersonDemographicDetails.Occupation.Description
            */
            assertion = (string.IsNullOrWhiteSpace(report.NRFI33) != true && !(IsMatch(report.NRFI33, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign occupation description contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100046" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI33", Value = report.NRFI33 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.NRFI.100047
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATONRFI100047()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.NRFI.100047
            Occupation code is not a valid code
    
            Legacy Rule Format:
            IF ([NRFI2] <> NULLORBLANK) AND ([NRFI2] <> SET (DOMAIN(OCCUPATION CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^NRFI2 <> BLANK) AND (NotInDomain(^NRFI2, 'OCCUPATION CODES'))
    
            Data Elements:
    
            ^NRFI2 = RP:PersonDemographicDetails.Occupation.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.NRFI2) != true && !(IsMatch(report.NRFI2, @"^(111111|111211|111212|111311|111312|111399|121111|121311|121312|121313|121314|121315|121316|121318|121321|121322|121323|121399|121511|121512|121513|121599|121611|121612|121613|121614|121615|121616|121617|121699|121711|121799|131112|131113|131114|132111|132211|132311|132411|132511|133111|133112|133211|133311|133312|133411|133511|133512|133513|133611|133612|134111|134211|134212|134213|134214|134299|134311|134411|134412|134499|135111|135112|135199|139111|139112|139113|139211|139911|139912|139913|139915|139916|139917|139999|141111|141211|141311|141411|141911|141912|141999|142111|142112|142113|142114|142115|142116|149111|149112|149113|149211|149212|149311|149411|149412|149413|149911|149912|149913|149914|149915|149999|211111|211112|211113|211199|211211|211212|211213|211214|211299|211311|211411|211412|211413|211499|212111|212112|212113|212114|212211|212212|212311|212312|212313|212314|212315|212316|212317|212318|212399|212411|212412|212413|212414|212415|212416|212499|221111|221112|221113|221211|221212|221213|221214|222111|222112|222113|222199|222211|222212|222213|222299|222311|222312|223111|223112|223113|223311|224111|224112|224114|224115|224116|224211|224212|224213|224214|224311|224411|224412|224511|224512|224611|224712|224713|224714|224911|224912|224913|224914|224999|225111|225112|225113|225114|225115|225211|225212|225213|225311|225411|225412|225499|231111|231112|231113|231114|231199|231211|231212|231213|231214|231215|231299|232111|232112|232212|232213|232214|232311|232312|232313|232411|232412|232413|232414|232511|232611|233111|233112|233211|233212|233213|233214|233215|233311|233411|233511|233512|233513|233611|233612|233911|233912|233913|233914|233915|233916|233999|234111|234113|234114|234115|234116|234211|234212|234213|234311|234312|234313|234314|234399|234411|234412|234413|234511|234513|234514|234515|234516|234517|234521|234522|234599|234611|234612|234711|234911|234912|234913|234914|234915|234999|241111|241213|241311|241411|241511|241512|241513|241599|241999|242111|242112|242211|249111|249112|249211|249212|249213|249214|249299|249311|251111|251112|251211|251212|251213|251214|251311|251312|251411|251412|251511|251512|251513|251911|251912|251999|252111|252112|252211|252212|252213|252214|252299|252311|252312|252411|252511|252611|252711|252712|253111|253112|253211|253311|253312|253313|253314|253315|253316|253317|253318|253321|253322|253323|253324|253399|253411|253511|253512|253513|253514|253515|253516|253517|253518|253521|253911|253912|253913|253914|253915|253917|253918|253999|254111|254211|254212|254311|254411|254412|254413|254414|254415|254416|254417|254418|254421|254422|254423|254424|254425|254499|261111|261112|261113|261211|261212|261311|261312|261313|261314|261315|261316|261317|262111|262113|262114|262115|262116|262117|262118|263111|263112|263113|263211|263212|263213|263299|263311|263312|271111|271211|271212|271213|271214|271299|271311|272111|272112|272113|272114|272115|272199|272211|272311|272312|272313|272314|272399|272411|272412|272413|272414|272499|272511|272611|272612|272613|311112|311113|311114|311115|311211|311212|311213|311214|311215|311216|311217|311299|311311|311312|311313|311314|311399|311411|311412|311413|311414|311499|312111|312112|312113|312114|312115|312116|312199|312211|312212|312311|312312|312411|312412|312511|312512|312611|312911|312912|312913|312914|312999|313111|313112|313113|313199|313211|313212|313213|313214|321111|321211|321212|321213|321214|322111|322112|322113|322114|322115|322211|322311|322312|322313|323111|323112|323113|323211|323212|323213|323214|323215|323311|323312|323313|323314|323315|323316|323411|323412|324111|324211|324212|324311|331111|331112|331211|331212|331213|332111|332211|333111|333211|333212|333311|333411|334112|334113|334114|334115|334116|334117|341111|341113|342111|342211|342212|342311|342312|342313|342314|342411|342412|342413|342414|351111|351112|351211|351311|351411|361111|361112|361113|361114|361115|361116|361199|361311|362111|362312|362313|362411|362511|362512|362611|362711|362712|363111|363112|363113|363114|363115|363116|363199|363211|363212|363213|363214|363215|363299|363311|363312|391111|392111|392112|392211|392311|392312|393111|393112|393113|393114|393211|393212|393213|393299|393311|394112|394113|394211|394212|394213|394214|394299|399111|399112|399211|399212|399213|399311|399312|399411|399511|399512|399513|399514|399515|399516|399517|399599|399611|399911|399912|399913|399914|399915|399916|399918|399999|411111|411211|411212|411213|411214|411311|411411|411412|411511|411611|411711|411712|411713|411714|411715|411716|421111|421112|421113|421114|422111|422112|422115|422116|423111|423211|423311|423312|423313|423314|423411|423412|423413|431111|431112|431211|431311|431411|431511|431911|431912|431999|441111|441211|441212|441311|441312|442111|442211|442212|442213|442214|442215|442216|442217|442299|451111|451211|451311|451399|451411|451412|451511|451512|451513|451611|451612|451711|451799|451811|451812|451813|451814|451815|451816|451899|452111|452211|452212|452213|452214|452215|452216|452217|452299|452311|452312|452313|452314|452315|452316|452317|452318|452321|452322|452323|452411|452412|452413|452414|452450|452451|452452|452453|452454|452455|452456|452457|452458|452459|452460|452499|511111|511112|512111|512211|512299|521111|521211|521212|531000|531111|532111|532112|532113|541111|541112|541211|542111|542112|542113|542114|551111|551112|551211|551311|552111|552211|552311|552312|552313|552314|561000|561111|561112|561113|561199|561211|561212|561311|561411|561412|561511|561611|561911|561912|561913|561999|591112|591113|591115|591116|591117|591211|591212|599111|599112|599211|599212|599213|599214|599215|599311|599411|599511|599512|599513|599514|599515|599516|599517|599518|599521|599599|599611|599612|599613|599711|599912|599913|599914|599915|599916|599999|611111|611112|611211|611311|611312|611313|611314|611399|612111|612112|612113|612114|612115|621111|621211|621311|621411|621511|621611|621711|621712|621713|621911|621912|621999|631111|631112|639111|639112|639211|639212|639311|639411|639412|639511|711111|711112|711113|711114|711211|711311|711313|711314|711411|711511|711512|711513|711514|711515|711516|711611|711711|711712|711713|711714|711715|711716|711911|711912|711913|711914|711999|712111|712211|712212|712213|712311|712911|712912|712913|712914|712915|712916|712917|712918|712921|712922|712999|721111|721112|721211|721212|721213|721214|721215|721216|721311|721911|721912|721913|721914|721915|721916|721999|731111|731112|731211|731212|731213|731311|731312|732111|733111|733112|733113|733114|733115|741111|811111|811211|811311|811411|811412|811511|811512|811513|811611|811612|811699|821111|821112|821113|821114|821211|821311|821411|821412|821511|821611|821711|821712|821713|821714|821911|821912|821913|821914|821915|831111|831112|831113|831114|831115|831116|831117|831118|831199|831211|831212|831311|831312|831313|832111|832112|832113|832114|832115|832199|832211|839111|839211|839212|839311|839312|839313|839411|839412|839413|839911|839912|839913|839914|839915|839916|839917|839918|839999|842111|842211|842212|842213|842214|842215|842216|842217|842218|842221|842222|842299|842311|842312|842313|842314|842315|842316|842317|842318|842321|842399|842411|842499|843111|843112|843113|843211|843311|843411|843911|843912|843999|851111|851211|851311|891111|891112|891113|891211|899111|899211|899212|899311|899411|899412|899413|899414|899415|899511|899512|899611|899711|899911|899912|899913|899914|899915|899916|899917|899918|899921|899922|899923|899999|913301|922101|922201|922301|922302|922303|922401|922501|922502|922503|923202|923301|923302|923303|923304|923305|923306|923307|923401|925001|925101|925401|926101|931101|931102|931301|931302|932101|932102|932201|932301|932302|932303|932401|933101|933102|933201|933202|933301|933302|933401|934101|934201|934202|934203|935101|935102|935103|935104|936102|936103|936104|936201|936202|936203|936204|939101|939201|939301|939302|939303|939401|939901|939902|939903|939904|939905|941001|942001|943101|944201|944202|945101|945102|945103|945104|945201|945203|951001|953001|955001|955201|959901|959902|961101|961202|962102|971201|973001|974101|981101|983101|984101|984102|984103|989901|989902|999000)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.NRFI.100047",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Occupation code is not a valid code",
                    Location = "/xbrli:xbrl/tns:PersonDemographicDetails.Occupation.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.NRFI.100047" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "NRFI2", Value = report.NRFI2 });

                response.Add(processMessage);
            }
        }
        #endregion
    }
}