using System;
using System.Collections.Generic;
using System.Linq;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationRNTLPRPTY
{
    public partial class RNTLPRPTY2019Validator
    {
        #region Manual Rules - Rules marked for manual coding

        #region Helper methods

        /// <summary>
        /// Provides an abstract method for validating against rules where the 'Your Share' amount must be provided.
        /// 
        /// Legacy Rule Format (abstract):
        /// (^[YourShareAmount] = NULL) AND (^[TotalAmount] <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)
        /// 
        /// </summary>
        /// <param name="ruleId">The rule ID for the rule being validated e.g. VR.ATO.RNTLPRPTY.000200</param>
        /// <param name="msgCode">The code for the message that this rule will add to the response if validation fails e.g. CMN.ATO.RNTLPRPTY.000200</param>
        /// <param name="shortDesc">The short error message</param>
        /// <param name="longDesc">The long error message</param>
        /// <param name="locationFieldName">The XBRL location of the field where validation is considered to have failed e.g. tns:Expense.RealEstateProperty.Marketing.Amount</param>
        /// <param name="yourShareParamName">The alias for the Your Share amount field e.g. RNTLPRPTY138</param>
        /// <param name="yourShareParamSelector">A lambda expression that returns the value of the Your Share amount field named in yourShareParamSelector</param>
        /// <param name="totalParamName">The alias for the Total amount field e.g. RNTLPRPTY116</param>
        /// <param name="totalParamSelector">A lambda expression that returns the value of the Total amount field named in totalParamSelector</param>
        protected void ValidateYourShareMustBeProvided(List<ProcessMessageDocument> response, RNTLPRPTY2019 report, string ruleId, string msgCode, string shortDesc, string longDesc, string locationFieldName, string yourShareParamName, Func<RNTLPRPTY2019.RPPrptySeqNumPortionContext, decimal?> yourShareParamSelector, string totalParamName, Func<RNTLPRPTY2019.RPPrptySeqNumTotalContext, decimal?> totalParamSelector)
        {
            if (report.RPPrptySeqNumPortionContextCollection == null)
                return;

            ProcessMessageDocument processMessage;
            bool assertion;

            RNTLPRPTY2019.RPPrptySeqNumContext rpPrptySeqNumContext = null;
            RNTLPRPTY2019.RPPrptySeqNumTotalContext rpPrptySeqNumTotalContext = null;

            foreach (RNTLPRPTY2019.RPPrptySeqNumPortionContext rpPrptySeqNumPortionContext in report.RPPrptySeqNumPortionContextCollection)
            {
                // Find the associated Base and Total contexts (if they exist)
                rpPrptySeqNumContext = Count(report.RPPrptySeqNumContextCollection) > 0 ? report.RPPrptySeqNumContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == rpPrptySeqNumPortionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null;
                rpPrptySeqNumTotalContext = Count(report.RPPrptySeqNumTotalContextCollection) > 0 ? report.RPPrptySeqNumTotalContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == rpPrptySeqNumPortionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null;

                // Only validate if both Portion and Total contexts exist
                if (rpPrptySeqNumContext != null && rpPrptySeqNumTotalContext != null)
                {
                    // If Your Share is null AND Total exists AND RNTLPRPTY104 exists and is less than 100%
                    assertion = ((!yourShareParamSelector(rpPrptySeqNumPortionContext).HasValue)
                                && (totalParamSelector(rpPrptySeqNumTotalContext).HasValue)
                                && (rpPrptySeqNumContext.RNTLPRPTY104.HasValue && rpPrptySeqNumContext.RNTLPRPTY104.Value < 100));

                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = msgCode,
                            Severity = ProcessMessageSeverity.Error,
                            Description = shortDesc,
                            LongDescription = longDesc,
                            Location = string.Format("/xbrli:xbrl/{0}[@contextRef='{1}']", locationFieldName, rpPrptySeqNumPortionContext.Id),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = ruleId } },
                        };

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = yourShareParamName, Value = GetValueOrEmpty(rpPrptySeqNumPortionContext != null ? yourShareParamSelector(rpPrptySeqNumPortionContext) : null) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = totalParamName, Value = GetValueOrEmpty(rpPrptySeqNumTotalContext != null ? totalParamSelector(rpPrptySeqNumTotalContext) : null) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "RNTLPRPTY104", Value = rpPrptySeqNumContext.RNTLPRPTY104.GetValueOrDefault().ToString() });

                        response.Add(processMessage);
                    }
                }
            }
        }

        /// <summary>
        /// Provides an abstract method for validating against rules where the 'Your Share' amount must not exceed the 'Total' amount
        /// 
        /// Legacy Rule Format (abstract):
        /// (^[YourShareAmount] <> NULL) AND (^[TotalAmount] = NULL OR ^[YourShareAmount] > ^[TotalAmount])
        /// 
        /// </summary>
        /// <param name="ruleId">The rule ID for the rule being validated e.g. VR.ATO.RNTLPRPTY.000200</param>
        /// <param name="msgCode">The code for the message that this rule will add to the response if validation fails e.g. CMN.ATO.RNTLPRPTY.000200</param>
        /// <param name="shortDesc">The short error message</param>
        /// <param name="longDesc">The long error message</param>
        /// <param name="locationFieldName">The XBRL location of the field where validation is considered to have failed e.g. tns:Expense.RealEstateProperty.Marketing.Amount</param>
        /// <param name="yourShareParamName">The alias for the Your Share amount field e.g. RNTLPRPTY138</param>
        /// <param name="yourShareParamSelector">A lambda expression that returns the value of the Your Share amount field named in yourShareParamSelector</param>
        /// <param name="totalParamName">The alias for the Total amount field e.g. RNTLPRPTY116</param>
        /// <param name="totalParamSelector">A lambda expression that returns the value of the Total amount field named in totalParamSelector</param>
        protected void ValidateYourShareMustNotExceedTotal(List<ProcessMessageDocument> response, RNTLPRPTY2019 report, string ruleId, string msgCode, string shortDesc, string longDesc, string locationFieldName, string yourShareParamName, Func<RNTLPRPTY2019.RPPrptySeqNumPortionContext, decimal?> yourShareParamSelector, string totalParamName, Func<RNTLPRPTY2019.RPPrptySeqNumTotalContext, decimal?> totalParamSelector)
        {
            if (report.RPPrptySeqNumPortionContextCollection == null)
                return;

            ProcessMessageDocument processMessage;
            bool assertion;

            RNTLPRPTY2019.RPPrptySeqNumTotalContext rpPrptySeqNumTotalContext = null;

            foreach (RNTLPRPTY2019.RPPrptySeqNumPortionContext rpPrptySeqNumPortionContext in report.RPPrptySeqNumPortionContextCollection)
            {
                // Find the associated total context (if it exists)
                rpPrptySeqNumTotalContext = Count(report.RPPrptySeqNumTotalContextCollection) > 0 ? report.RPPrptySeqNumTotalContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == rpPrptySeqNumPortionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null;

                // If Your share is provided AND either total is not provided OR your share is greater than total
                assertion = (yourShareParamSelector(rpPrptySeqNumPortionContext).HasValue &&
                            (rpPrptySeqNumTotalContext == null ||
                                    (!totalParamSelector(rpPrptySeqNumTotalContext).HasValue ||
                                    yourShareParamSelector(rpPrptySeqNumPortionContext).Value > totalParamSelector(rpPrptySeqNumTotalContext).Value)));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = msgCode,
                        Severity = ProcessMessageSeverity.Error,
                        Description = shortDesc,
                        LongDescription = longDesc,
                        Location = string.Format("/xbrli:xbrl/{0}[@contextRef='{1}']", locationFieldName, rpPrptySeqNumPortionContext.Id),
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = ruleId } },
                    };

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = yourShareParamName, Value = GetValueOrEmpty(yourShareParamSelector(rpPrptySeqNumPortionContext)) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = totalParamName, Value = GetValueOrEmpty(rpPrptySeqNumTotalContext != null ? totalParamSelector(rpPrptySeqNumTotalContext) : null) });

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000001

        /*  VR.ATO.RNTLPRPTY.000001
                    When the Ownership percentage is less than 100% and (total) Land tax is present, Your share of land tax must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY147 = NULL) AND (^RNTLPRPTY125 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY147 = NULL) AND (^RNTLPRPTY125 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY147 = RP.{PrptySeqNum}.Portion:Expense.LandTax.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY125 = RP.{PrptySeqNum}.Total:Expense.LandTax.Amount
            */
        protected void VRATORNTLPRPTY000001(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000001", "CMN.ATO.RNTLPRPTY.000001",
                @"Your share of land tax must be provided",
                @"When the Ownership percentage is less than 100% and (total) Land tax is present, Your share of land tax must be provided",
                "tns:Expense.LandTax.Amount",
                "RNTLPRPTY147", (context => context.RNTLPRPTY147),
                "RNTLPRPTY125", (context => context.RNTLPRPTY125));
        }

        #endregion // VR.ATO.RNTLPRPTY.000001

        #region VR.ATO.RNTLPRPTY.000002

        /*  VR.ATO.RNTLPRPTY.000002
                    When Your share of land tax is provided, the amount must not exceed (total) Land tax

                Legacy Rule Format:
                    (^RNTLPRPTY147 <> NULL) AND (^RNTLPRPTY125 = NULL OR ^RNTLPRPTY147 > ^RNTLPRPTY125)

            Technical Business Rule Format:
                    (^RNTLPRPTY147 <> NULL) AND (^RNTLPRPTY125 = NULL OR ^RNTLPRPTY147 > ^RNTLPRPTY125)

        Data Elements:
    
            ^RNTLPRPTY147 = RP.{PrptySeqNum}.Portion:Expense.LandTax.Amount
    
            ^RNTLPRPTY125 = RP.{PrptySeqNum}.Total:Expense.LandTax.Amount
            */
        protected void VRATORNTLPRPTY000002(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000002", "CMN.ATO.RNTLPRPTY.000002",
                @"Your share of land tax is incorrect",
                @"When Your share of land tax is provided, the amount must not exceed (total) Land tax",
                "tns:Expense.LandTax.Amount",
                "RNTLPRPTY147", (context => context.RNTLPRPTY147),
                "RNTLPRPTY125", (context => context.RNTLPRPTY125));
        }

        #endregion // VR.ATO.RNTLPRPTY.000002

        #region VR.ATO.RNTLPRPTY.000003

        /*  VR.ATO.RNTLPRPTY.000003
            When the Ownership percentage is less than 100% and (total) Legal fees is present, Your share of legal fees must be provided

        Legacy Rule Format:
            (^RNTLPRPTY148 = NULL) AND (^RNTLPRPTY126 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Technical Business Rule Format:
            (^RNTLPRPTY148 = NULL) AND (^RNTLPRPTY126 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
        ^RNTLPRPTY148 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LegalFees.Amount
    
        ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
        ^RNTLPRPTY126 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LegalFees.Amount
        */

        protected void VRATORNTLPRPTY000003(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000003", "CMN.ATO.RNTLPRPTY.000003",
                @"Your share of legal fees must be provided",
                @"When the Ownership percentage is less than 100% and (total) Legal fees is present, Your share of legal fees must be provided",
                "tns:Expense.RealEstateProperty.LegalFees.Amount",
                "RNTLPRPTY148", (context => context.RNTLPRPTY148),
                "RNTLPRPTY126", (context => context.RNTLPRPTY126));
        }
        #endregion // VR.ATO.RNTLPRPTY.000003

        #region VR.ATO.RNTLPRPTY.000004

        /*  VR.ATO.RNTLPRPTY.000004
            When Your share of legal fees is provided, the amount must not exceed (total) Legal fees

        Legacy Rule Format:
            (^RNTLPRPTY148 <> NULL) AND (^RNTLPRPTY126 = NULL OR ^RNTLPRPTY148 > ^RNTLPRPTY126)

        Technical Business Rule Format:
            (^RNTLPRPTY148 <> NULL) AND (^RNTLPRPTY126 = NULL OR ^RNTLPRPTY148 > ^RNTLPRPTY126)

        Data Elements:
    
        ^RNTLPRPTY148 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LegalFees.Amount
    
        ^RNTLPRPTY126 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LegalFees.Amount
        */
        protected void VRATORNTLPRPTY000004(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000004", "CMN.ATO.RNTLPRPTY.000004",
                @"Your share of legal fees is incorrect",
                @"When Your share of legal fees is provided, the amount must not exceed (total) Legal fees",
                "tns:Expense.RealEstateProperty.LegalFees.Amount",
                "RNTLPRPTY148", (context => context.RNTLPRPTY148),
                "RNTLPRPTY126", (context => context.RNTLPRPTY126));
        }
        #endregion // VR.ATO.RNTLPRPTY.000004

        #region VR.ATO.RNTLPRPTY.000005

        /*  VR.ATO.RNTLPRPTY.000005
            When the Ownership percentage is less than 100% and (total) Pest control is present, Your share of pest control must be provided

        Legacy Rule Format:
            (^RNTLPRPTY149 = NULL) AND (^RNTLPRPTY127 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Technical Business Rule Format:
            (^RNTLPRPTY149 = NULL) AND (^RNTLPRPTY127 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
        ^RNTLPRPTY149 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.PestControl.Amount
    
        ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
        ^RNTLPRPTY127 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.PestControl.Amount
        */
        protected void VRATORNTLPRPTY000005(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000005", "CMN.ATO.RNTLPRPTY.000005",
                @"Your share of pest control must be provided",
                @"When the Ownership percentage is less than 100% and (total) Pest control is present, Your share of pest control must be provided",
                "tns:Expense.RealEstateProperty.PestControl.Amount",
                "RNTLPRPTY149", (context => context.RNTLPRPTY149),
                "RNTLPRPTY127", (context => context.RNTLPRPTY127));
        }
        #endregion // VR.ATO.RNTLPRPTY.000003

        #region VR.ATO.RNTLPRPTY.000006

        /*  VR.ATO.RNTLPRPTY.000006
            When Your share of pest control is provided, the amount must not exceed (total) Pest control

        Legacy Rule Format:
            (^RNTLPRPTY149 <> NULL) AND (^RNTLPRPTY127 = NULL OR ^RNTLPRPTY149 > ^RNTLPRPTY127)

        Technical Business Rule Format:
            (^RNTLPRPTY149 <> NULL) AND (^RNTLPRPTY127 = NULL OR ^RNTLPRPTY149 > ^RNTLPRPTY127)

        Data Elements:
    
        ^RNTLPRPTY149 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.PestControl.Amount
    
        ^RNTLPRPTY127 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.PestControl.Amount
        */
        protected void VRATORNTLPRPTY000006(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000006", "CMN.ATO.RNTLPRPTY.000006",
                @"Your share of pest control is incorrect",
                @"When Your share of pest control is provided, the amount must not exceed (total) Pest control",
                "tns:Expense.RealEstateProperty.PestControl.Amount",
                "RNTLPRPTY149", (context => context.RNTLPRPTY149),
                "RNTLPRPTY127", (context => context.RNTLPRPTY127));
        }
        #endregion // VR.ATO.RNTLPRPTY.000006

        #region VR.ATO.RNTLPRPTY.000007

        /*  VR.ATO.RNTLPRPTY.000007
                    When the Ownership percentage is less than 100% and (total) Property agent fees/commission is present, Your share of property agent fees/commission must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY150 = NULL) AND (^RNTLPRPTY128 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY150 = NULL) AND (^RNTLPRPTY128 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY150 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.AgentFeesCommission.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY128 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.AgentFeesCommission.Amount
            */
        protected void VRATORNTLPRPTY000007(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000007", "CMN.ATO.RNTLPRPTY.000007",
                @"Your share of property agent fees/commission must be provided",
                @"When the Ownership percentage is less than 100% and (total) Property agent fees/commission is present, Your share of property agent fees/commission must be provided",
                "tns:Expense.RealEstateProperty.AgentFeesCommission.Amount",
                "RNTLPRPTY150", (context => context.RNTLPRPTY150),
                "RNTLPRPTY128", (context => context.RNTLPRPTY128));
        }
        #endregion // VR.ATO.RNTLPRPTY.000007

        #region VR.ATO.RNTLPRPTY.000008

        /*  VR.ATO.RNTLPRPTY.000008
                    When Your share of property agent fees/commission is provided, the amount must not exceed (total) Property agent fees/commission

                Legacy Rule Format:
                    (^RNTLPRPTY150 <> NULL) AND (^RNTLPRPTY128 = NULL OR ^RNTLPRPTY150 > ^RNTLPRPTY128)

            Technical Business Rule Format:
                    (^RNTLPRPTY150 <> NULL) AND (^RNTLPRPTY128 = NULL OR ^RNTLPRPTY150 > ^RNTLPRPTY128)

        Data Elements:
    
            ^RNTLPRPTY150 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.AgentFeesCommission.Amount
    
            ^RNTLPRPTY128 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.AgentFeesCommission.Amount
            */
        protected void VRATORNTLPRPTY000008(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000008", "CMN.ATO.RNTLPRPTY.000008",
                @"Your share of property agent fees/commission is incorrect",
                @"When Your share of property agent fees/commission is provided, the amount must not exceed (total) Property agent fees/commission",
                "tns:Expense.RealEstateProperty.AgentFeesCommission.Amount",
                "RNTLPRPTY150", (context => context.RNTLPRPTY150),
                "RNTLPRPTY128", (context => context.RNTLPRPTY128));
        }
        #endregion // VR.ATO.RNTLPRPTY.000008

        #region VR.ATO.RNTLPRPTY.000009

        /*  VR.ATO.RNTLPRPTY.000009
                    When the Ownership percentage is less than 100% and (total) Repairs and maintenance is present, Your share of repairs and maintenance must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY151 = NULL) AND (^RNTLPRPTY129 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY151 = NULL) AND (^RNTLPRPTY129 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY151 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY129 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
            */
        protected void VRATORNTLPRPTY000009(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000009", "CMN.ATO.RNTLPRPTY.000009",
                @"Your share of repairs and maintenance must be provided",
                @"When the Ownership percentage is less than 100% and (total) Repairs and maintenance is present, Your share of repairs and maintenance must be provided",
                "tns:Expense.RealEstateProperty.RepairsAndMaintenance.Amount",
                "RNTLPRPTY151", (context => context.RNTLPRPTY151),
                "RNTLPRPTY129", (context => context.RNTLPRPTY129));
        }
        #endregion // VR.ATO.RNTLPRPTY.000009

        #region VR.ATO.RNTLPRPTY.000010

        /*  VR.ATO.RNTLPRPTY.000010
                    When Your share of repairs and maintenance is provided, the amount must not exceed (total) Repairs and maintenance

                Legacy Rule Format:
                    (^RNTLPRPTY151 <> NULL) AND (^RNTLPRPTY129 = NULL OR ^RNTLPRPTY151 > ^RNTLPRPTY129)

            Technical Business Rule Format:
                    (^RNTLPRPTY151 <> NULL) AND (^RNTLPRPTY129 = NULL OR ^RNTLPRPTY151 > ^RNTLPRPTY129)

        Data Elements:
    
            ^RNTLPRPTY151 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
    
            ^RNTLPRPTY129 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
            */
        protected void VRATORNTLPRPTY000010(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000010", "CMN.ATO.RNTLPRPTY.000010",
                @"Your share of repairs and maintenance is incorrect",
                @"When Your share of repairs and maintenance is provided, the amount must not exceed (total) Repairs and maintenance",
                "tns:Expense.RealEstateProperty.RepairsAndMaintenance.Amount",
                "RNTLPRPTY151", (context => context.RNTLPRPTY151),
                "RNTLPRPTY129", (context => context.RNTLPRPTY129));
        }
        #endregion // VR.ATO.RNTLPRPTY.000010

        #region VR.ATO.RNTLPRPTY.000015

        /*  VR.ATO.RNTLPRPTY.000015
                    When the Ownership percentage is less than 100% and (total) Water charges is present, Your share of water charges must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY155 = NULL) AND (^RNTLPRPTY133 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY155 = NULL) AND (^RNTLPRPTY133 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY155 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Water.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY133 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Water.Amount
            */
        protected void VRATORNTLPRPTY000015(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000015", "CMN.ATO.RNTLPRPTY.000015",
            @"Your share of water charges must be provided",
            @"When the Ownership percentage is less than 100% and (total) Water charges is present, Your share of water charges must be provided",
            "tns:Expense.RealEstateProperty.Water.Amount",
            "RNTLPRPTY155", (context => context.RNTLPRPTY155),
            "RNTLPRPTY133", (context => context.RNTLPRPTY133));
        }
        #endregion // VR.ATO.RNTLPRPTY.000015

        #region VR.ATO.RNTLPRPTY.000016

        /*  VR.ATO.RNTLPRPTY.000016
                    When Your share of water charges is provided, the amount must not exceed (total) Water charges

                Legacy Rule Format:
                    (^RNTLPRPTY155 <> NULL) AND (^RNTLPRPTY133 = NULL OR ^RNTLPRPTY155 > ^RNTLPRPTY133)

            Technical Business Rule Format:
                    (^RNTLPRPTY155 <> NULL) AND (^RNTLPRPTY133 = NULL OR ^RNTLPRPTY155 > ^RNTLPRPTY133)

        Data Elements:
    
            ^RNTLPRPTY155 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Water.Amount
    
            ^RNTLPRPTY133 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Water.Amount
            */
        protected void VRATORNTLPRPTY000016(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000016", "CMN.ATO.RNTLPRPTY.000016",
            @"Your share of water charges is incorrect",
            @"When Your share of water charges is provided, the amount must not exceed (total) Water charges",
            "tns:Expense.RealEstateProperty.Water.Amount",
            "RNTLPRPTY155", (context => context.RNTLPRPTY155),
            "RNTLPRPTY133", (context => context.RNTLPRPTY133));
        }

        #endregion // VR.ATO.RNTLPRPTY.000016

        #region VR.ATO.RNTLPRPTY.000019

        /*  VR.ATO.RNTLPRPTY.000019
                    Entity identifier scheme must be "http://www.ato.gov.au/tfn" across all contexts

                Legacy Rule Format:
                    (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')

            Technical Business Rule Format:
                    ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'

        Data Elements:
    
            ^IdentifierScheme = IdentifierScheme
            */
        protected void VRATORNTLPRPTY000019(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;

            //Each of these loops checks the 3 contexts collections of RNTLPRPTY. This allows a more focused message of the exact context that has an error

            if (report.RPPrptySeqNumContextCollection != null)
            {
                foreach (RNTLPRPTY2019.RPPrptySeqNumContext context in report.RPPrptySeqNumContextCollection)
                {
                    if (context.IdentifierScheme != "http://www.ato.gov.au/tfn")
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000019",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Entity identifier scheme must be ""http://www.ato.gov.au/tfn"" across all contexts",
                            Location = string.Format("/xbrli:xbrl/xbrli:context[@id='{0}']/xbrli:entity/xbrli:identifier/@scheme", context.Id),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000019" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IdentifierScheme", Value = context.IdentifierScheme });
                        response.Add(processMessage);
                    }
                }
            }

            if (report.RPPrptySeqNumPortionContextCollection != null)
            {
                foreach (RNTLPRPTY2019.RPPrptySeqNumPortionContext context in report.RPPrptySeqNumPortionContextCollection)
                {
                    if (context.IdentifierScheme != "http://www.ato.gov.au/tfn")
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000019",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Entity identifier scheme must be ""http://www.ato.gov.au/tfn"" across all contexts",
                            Location = string.Format("/xbrli:xbrl/xbrli:context[@id='{0}']/xbrli:entity/xbrli:identifier/@scheme", context.Id),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000019" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IdentifierScheme", Value = context.IdentifierScheme });
                        response.Add(processMessage);
                    }
                }
            }

            if (report.RPPrptySeqNumTotalContextCollection != null)
            {
                foreach (RNTLPRPTY2019.RPPrptySeqNumTotalContext context in report.RPPrptySeqNumTotalContextCollection)
                {
                    if (context.IdentifierScheme != "http://www.ato.gov.au/tfn")
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000019",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Entity identifier scheme must be ""http://www.ato.gov.au/tfn"" across all contexts",
                            Location = string.Format("/xbrli:xbrl/xbrli:context[@id='{0}']/xbrli:entity/xbrli:identifier/@scheme", context.Id),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000019" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IdentifierScheme", Value = context.IdentifierScheme });
                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000019

        #region VR.ATO.RNTLPRPTY.000020

        /*  VR.ATO.RNTLPRPTY.000020
                    Entity identifier TFN must be the same across all contexts

                Legacy Rule Format:
                    (NotSameValues(^TFN))

            Technical Business Rule Format:
                    (NotSameValues(^TFN))

        Data Elements:
    
            ^TFN = TFN
            */
        protected void VRATORNTLPRPTY000020(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            IEnumerable<string> ContextTFNs = new List<string>();

            if (report.RPPrptySeqNumContextCollection != null)
                ContextTFNs = ContextTFNs.Concat(report.RPPrptySeqNumContextCollection.Select(context => context.IdentifierTFN));
            if (report.RPPrptySeqNumPortionContextCollection != null)
                ContextTFNs = ContextTFNs.Concat(report.RPPrptySeqNumPortionContextCollection.Select(context => context.IdentifierTFN));
            if (report.RPPrptySeqNumTotalContextCollection != null)
                ContextTFNs = ContextTFNs.Concat(report.RPPrptySeqNumTotalContextCollection.Select(context => context.IdentifierTFN));

            assertion = NotSameValues(ContextTFNs);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Entity identifier TFN must be the same across all contexts.",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000020" } },
                };

                /*  Parameters are commented out or non-existant for manual rules 

                */
                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000020

        #region VR.ATO.RNTLPRPTY.000021

        /*  VR.ATO.RNTLPRPTY.000021
                    Period start date must be the same across all duration contexts

                Legacy Rule Format:
                    (NotSameValues(^StartDate))

            Technical Business Rule Format:
                    (NotSameValues(^StartDate))

        Data Elements:
    
            ^StartDate = StartDate
            */
        protected void VRATORNTLPRPTY000021(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            //Date values are being converted to string as esr NotSameValues function is not using the correct overload on non-string types
            IEnumerable<string> ContextStartDates = new List<string>();

            if (report.RPPrptySeqNumContextCollection != null)
                ContextStartDates = ContextStartDates.Concat(report.RPPrptySeqNumContextCollection.Select(context => context.StartDate.Value.ToShortDateString()));
            if (report.RPPrptySeqNumPortionContextCollection != null)
                ContextStartDates = ContextStartDates.Concat(report.RPPrptySeqNumPortionContextCollection.Select(context => context.StartDate.Value.ToShortDateString()));
            if (report.RPPrptySeqNumTotalContextCollection != null)
                ContextStartDates = ContextStartDates.Concat(report.RPPrptySeqNumTotalContextCollection.Select(context => context.StartDate.Value.ToShortDateString()));

            assertion = NotSameValues(ContextStartDates);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000021" } },
                };

                /*  Parameters are commented out or non-existant for manual rules 

                */
                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000021

        #region VR.ATO.RNTLPRPTY.000022

        /*  VR.ATO.RNTLPRPTY.000022
                    Period end date must be the same across all duration contexts

                Legacy Rule Format:
                    (NotSameValues(^EndDate))

            Technical Business Rule Format:
                    (NotSameValues(^EndDate))

        Data Elements:
    
            ^EndDate = EndDate
            */
        protected void VRATORNTLPRPTY000022(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            //Date values are being converted to string as esr NotSameValues function is not using the correct overload on non-string types
            IEnumerable<string> ContextEndDates = new List<string>();

            if (report.RPPrptySeqNumContextCollection != null)
                ContextEndDates = ContextEndDates.Concat(report.RPPrptySeqNumContextCollection.Select(context => context.EndDate.Value.ToShortDateString()));
            if (report.RPPrptySeqNumPortionContextCollection != null)
                ContextEndDates = ContextEndDates.Concat(report.RPPrptySeqNumPortionContextCollection.Select(context => context.EndDate.Value.ToShortDateString()));
            if (report.RPPrptySeqNumTotalContextCollection != null)
                ContextEndDates = ContextEndDates.Concat(report.RPPrptySeqNumTotalContextCollection.Select(context => context.EndDate.Value.ToShortDateString()));

            assertion = NotSameValues(ContextEndDates);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000022" } },
                };

                /*  Parameters are commented out or non-existant for manual rules 

                */
                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000022

        #region VR.ATO.RNTLPRPTY.000011

        /*  VR.ATO.RNTLPRPTY.000011
                    When the Ownership percentage is less than 100% and (total) Stationery, telephone, and postage is present, Your share of stationery, telephone, and postage must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY153 = NULL) AND (^RNTLPRPTY131 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY153 = NULL) AND (^RNTLPRPTY131 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY153 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.OfficeSupplies.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY131 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.OfficeSupplies.Amount
            */
        protected void VRATORNTLPRPTY000011(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000011", "CMN.ATO.RNTLPRPTY.000011",
                @"Your share of stationery, telephone, and postage must be provided",
                @"When the Ownership percentage is less than 100% and (total) Stationery, telephone, and postage is present, Your share of stationery, telephone, and postage must be provided",
                "tns:Expense.RealEstateProperty.OfficeSupplies.Amount",
                "RNTLPRPTY153", (context => context.RNTLPRPTY153),
                "RNTLPRPTY131", (context => context.RNTLPRPTY131));
        }
        #endregion // VR.ATO.RNTLPRPTY.000011

        #region VR.ATO.RNTLPRPTY.000012

        /*  VR.ATO.RNTLPRPTY.000012
                    When Your share of stationery, telephone, and postage is provided, the amount must not exceed (total) Stationery, telephone, and postage

                Legacy Rule Format:
                    (^RNTLPRPTY153 <> NULL) AND (^RNTLPRPTY131 = NULL OR ^RNTLPRPTY153 > ^RNTLPRPTY131)

            Technical Business Rule Format:
                    (^RNTLPRPTY153 <> NULL) AND (^RNTLPRPTY131 = NULL OR ^RNTLPRPTY153 > ^RNTLPRPTY131)

        Data Elements:
    
            ^RNTLPRPTY153 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.OfficeSupplies.Amount
    
            ^RNTLPRPTY131 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.OfficeSupplies.Amount
            */
        protected void VRATORNTLPRPTY000012(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000012", "CMN.ATO.RNTLPRPTY.000012",
                @"Your share of stationery, telephone, and postage is incorrect",
                @"When Your share of stationery, telephone, and postage is provided, the amount must not exceed (total) Stationery, telephone, and postage",
                "tns:Expense.RealEstateProperty.OfficeSupplies.Amount",
                "RNTLPRPTY153", (context => context.RNTLPRPTY153),
                "RNTLPRPTY131", (context => context.RNTLPRPTY131));
        }
        #endregion // VR.ATO.RNTLPRPTY.000012

        #region VR.ATO.RNTLPRPTY.000013

        /*  VR.ATO.RNTLPRPTY.000013
                When the Ownership percentage is less than 100% and (total) Travel expenses is present, Your share of travel expenses must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY154 = NULL) AND (^RNTLPRPTY132 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                    (^RNTLPRPTY154 = NULL) AND (^RNTLPRPTY132 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Data Elements:
    
                    ^RNTLPRPTY154 = RP.{PrptySeqNum}.Portion:Expense.Travel.Amount
    
                    ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
                    ^RNTLPRPTY132 = RP.{PrptySeqNum}.Total:Expense.Travel.Amount
            */
        protected void VRATORNTLPRPTY000013(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000013", "CMN.ATO.RNTLPRPTY.000013",
                @"Your share of travel expenses must be provided",
                @"When the Ownership percentage is less than 100% and (total) Travel expenses is present, Your share of travel expenses must be provided",
                "tns:Expense.Travel.Amount",
                "RNTLPRPTY154", (context => context.RNTLPRPTY154),
                "RNTLPRPTY132", (context => context.RNTLPRPTY132));
        }
        #endregion // VR.ATO.RNTLPRPTY.000013

        #region VR.ATO.RNTLPRPTY.000014

        /*  VR.ATO.RNTLPRPTY.000014
                When Your share of travel expenses is provided, the amount must not exceed (total) Travel expenses

                Legacy Rule Format:
                    (^RNTLPRPTY154 <> NULL) AND (^RNTLPRPTY132 = NULL OR ^RNTLPRPTY154 > ^RNTLPRPTY132)

                Technical Business Rule Format:
                    (^RNTLPRPTY154 <> NULL) AND (^RNTLPRPTY132 = NULL OR ^RNTLPRPTY154 > ^RNTLPRPTY132)

                 Data Elements:
    
                    ^RNTLPRPTY154 = RP.{PrptySeqNum}.Portion:Expense.Travel.Amount
    
                    ^RNTLPRPTY132 = RP.{PrptySeqNum}.Total:Expense.Travel.Amount
            */
        protected void VRATORNTLPRPTY000014(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000014", "CMN.ATO.RNTLPRPTY.000014",
                @"Your share of travel expenses is incorrect",
                @"When Your share of travel expenses is provided, the amount must not exceed (total) Travel expenses",
                "tns:Expense.Travel.Amount",
                "RNTLPRPTY154", (context => context.RNTLPRPTY154),
                "RNTLPRPTY132", (context => context.RNTLPRPTY132));
        }
        #endregion // VR.ATO.RNTLPRPTY.000014

        #region VR.ATO.RNTLPRPTY.000017

        /*  VR.ATO.RNTLPRPTY.000017
                    When the Ownership percentage is less than 100% and (total) Sundry rental expenses is present, Your share of sundry rental expenses must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY156 = NULL) AND (^RNTLPRPTY134 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY156 = NULL) AND (^RNTLPRPTY134 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY156 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Sundry.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY134 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Sundry.Amount
            */
        protected void VRATORNTLPRPTY000017(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000017", "CMN.ATO.RNTLPRPTY.000017",
            @"Your share of sundry rental expenses must be provided",
            @"When the Ownership percentage is less than 100% and (total) Sundry rental expenses is present, Your share of sundry rental expenses must be provided",
            "tns:Expense.RealEstateProperty.Sundry.Amount",
            "RNTLPRPTY156", (context => context.RNTLPRPTY156),
            "RNTLPRPTY134", (context => context.RNTLPRPTY134));
        }
        #endregion // VR.ATO.RNTLPRPTY.000017

        #region VR.ATO.RNTLPRPTY.000018

        /*  VR.ATO.RNTLPRPTY.000018
                    When Your share of sundry rental expenses is provided, the amount must not exceed (total) Sundry rental expenses

                Legacy Rule Format:
                    (^RNTLPRPTY156 <> NULL) AND (^RNTLPRPTY134 = NULL OR ^RNTLPRPTY156 > ^RNTLPRPTY134)

            Technical Business Rule Format:
                    (^RNTLPRPTY156 <> NULL) AND (^RNTLPRPTY134 = NULL OR ^RNTLPRPTY156 > ^RNTLPRPTY134)

        Data Elements:
    
            ^RNTLPRPTY156 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Sundry.Amount
    
            ^RNTLPRPTY134 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Sundry.Amount
            */
        protected void VRATORNTLPRPTY000018(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000018", "CMN.ATO.RNTLPRPTY.000018",
            @"Your share of sundry rental expenses is incorrect",
            @"When Your share of sundry rental expenses is provided, the amount must not exceed (total) Sundry rental expenses",
            "tns:Expense.RealEstateProperty.Sundry.Amount",
            "RNTLPRPTY156", (context => context.RNTLPRPTY156),
            "RNTLPRPTY134", (context => context.RNTLPRPTY134));
        }
        #endregion // VR.ATO.RNTLPRPTY.000018

        #region VR.ATO.RNTLPRPTY.000026

        /*  VR.ATO.RNTLPRPTY.000026
        Each Reporting Party Rental Property Total Financial detail context instance must have a matching (parent) Reporting Party Rental Property context

        Legacy Rule Format:
        (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY113) = 0)

        Technical Business Rule Format:
        (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY113) = 0)

        Data Elements:

        ^RNTLPRPTY113 = RP.{PrptySeqNum}.Total:Report.ItemOrder.Number

        ^RNTLPRPTY103 = RP.{PrptySeqNum}:Report.ItemOrder.Number
        */
        protected void VRATORNTLPRPTY000026(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;

            if (report.RPPrptySeqNumTotalContextCollection != null)
            {
                foreach (RNTLPRPTY2019.RPPrptySeqNumTotalContext context in report.RPPrptySeqNumTotalContextCollection)
                {
                    bool assertion = false;
                    string totalSequenceNumber = context.RNTLPRPTY113;

                    //Check if there are no parents (error due to no matching parent context records). Else if there are parents, attempt to find matching parent for the total record.
                    if (report.RPPrptySeqNumContextCollection == null)
                        assertion = true;
                    else if (report.RPPrptySeqNumContextCollection.FindAll(x => x.RNTLPRPTY103 == totalSequenceNumber).Count == 0)
                        assertion = true;

                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000026",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Each Rental Property Total Financial detail context instance must have a matching parent",
                            LongDescription = @"Each Reporting Party Rental Property Total Financial detail context instance must have a matching (parent) Reporting Party Rental Property context",
                            Location = string.Format("/xbrli:xbrl/xbrli:context[@id='{0}']", context.Id),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000026" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IdentifierScheme", Value = context.IdentifierScheme });
                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000209

        #region VR.ATO.RNTLPRPTY.000027

        /*  VR.ATO.RNTLPRPTY.000027
        When a Reporting Party Rental Property Total Financial detail context instance has been provided, and the Ownership percentage is less than 100%, then a paired Reporting Party Rental Property Portion Financial detail context instance must be provided

        Legacy Rule Format:
        (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100 AND CountOccurrence(^RNTLPRPTY135, ^RNTLPRPTY135 = ^RNTLPRPTY113) = 0)

        Technical Business Rule Format:
        (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100 AND CountOccurrence(^RNTLPRPTY135, ^RNTLPRPTY135 = ^RNTLPRPTY113) = 0)

        Data Elements:

        ^RNTLPRPTY113 = RP.{PrptySeqNum}.Total:Report.ItemOrder.Number

        ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

        ^RNTLPRPTY135 = RP.{PrptySeqNum}.Portion:Report.ItemOrder.Number
        */

        protected void VRATORNTLPRPTY000027(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;

            if (report.RPPrptySeqNumTotalContextCollection != null)
            {
                foreach (RNTLPRPTY2019.RPPrptySeqNumTotalContext context in report.RPPrptySeqNumTotalContextCollection)
                {
                    bool assertion = false;

                    // Check parent context's exist, this is due to some unit tests removing the parent but keeping the Total (Invalid), causing a null refrence exception here.
                    if (report.RPPrptySeqNumContextCollection != null)
                    {
                        // Find the parent associated with the Total context, this is so the Ownership Percentage can be taken
                        RNTLPRPTY2019.RPPrptySeqNumContext parentContext = report.RPPrptySeqNumContextCollection.Find(x => x.RNTLPRPTY103 == context.RNTLPRPTY113);

                        decimal? ownershipPercent = null;
                        if (parentContext != null)
                            ownershipPercent = parentContext.RNTLPRPTY104;

                        // Determine if a corresponding partial context is needed based on Ownership Percentage. Then make sure there is a corresponding Partial context
                        if (ownershipPercent != null && ownershipPercent < 100)
                        {
                            if (report.RPPrptySeqNumPortionContextCollection == null)
                                assertion = true;
                            else
                                if (report.RPPrptySeqNumPortionContextCollection.FindAll(x => x.RNTLPRPTY135 == context.RNTLPRPTY113).Count() == 0)
                                assertion = true;
                        }

                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.RNTLPRPTY.000027",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Rental Property Portion Financial detail context instance must be provided",
                                LongDescription = @"When a Reporting Party Rental Property Total Financial detail context instance has been provided, and the Ownership percentage is less than 100%, then a paired Reporting Party Rental Property Portion Financial detail context instance must be provided",
                                Location = string.Format("/xbrli:xbrl/tns:RealEstateProperty.Ownership.Percent[@contextRef='{0}']", parentContext.Id),
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000027" } },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IdentifierScheme", Value = context.IdentifierScheme });
                            response.Add(processMessage);
                        }
                    }
                }
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000209

        #region VR.ATO.RNTLPRPTY.000029

        /*  VR.ATO.RNTLPRPTY.000029
        Each Reporting Party Rental Property Portion Financial detail context instance must have a matching (parent) Reporting Party Rental Property context

        Legacy Rule Format:
        (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY135) = 0)

        Technical Business Rule Format:
        (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY135) = 0)

        Data Elements:

        ^RNTLPRPTY135 = RP.{PrptySeqNum}.Portion:Report.ItemOrder.Number

        ^RNTLPRPTY103 = RP.{PrptySeqNum}:Report.ItemOrder.Number
        */
        protected void VRATORNTLPRPTY000029(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ProcessMessageDocument processMessage;

            if (report.RPPrptySeqNumPortionContextCollection != null)
            {
                foreach (RNTLPRPTY2019.RPPrptySeqNumPortionContext context in report.RPPrptySeqNumPortionContextCollection)
                {
                    bool assertion = false;
                    string totalSequenceNumber = context.RNTLPRPTY135;

                    //Check if there are no parents (error due to no matching parent context records). Else if there are parents, attempt to find matching parent for the Partial record.
                    if (report.RPPrptySeqNumContextCollection == null)
                        assertion = true;
                    else if (report.RPPrptySeqNumContextCollection.FindAll(x => x.RNTLPRPTY103 == totalSequenceNumber).Count == 0)
                        assertion = true;

                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000029",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Each Rental Property Portion Financial detail context instance must have a matching parent",
                            LongDescription = @"Each Reporting Party Rental Property Portion Financial detail context instance must have a matching (parent) Reporting Party Rental Property context",
                            Location = string.Format("/xbrli:xbrl/xbrli:context[@id='{0}']", context.Id),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000029" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IdentifierScheme", Value = context.IdentifierScheme });
                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000209

        #region VR.ATO.RNTLPRPTY.000107

        /*  VR.ATO.RNTLPRPTY.000107
        When Your share of rental income is provided, the amount must not exceed (total) Rental income

        Legacy Rule Format:
        (^RNTLPRPTY136 <> NULL) AND (^RNTLPRPTY136 > ^RNTLPRPTY114)

        Technical Business Rule Format:
        (^RNTLPRPTY136 <> NULL) AND (^RNTLPRPTY136 > ^RNTLPRPTY114)

        Data Elements:

        ^RNTLPRPTY136 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.Rental.Amount

        ^RNTLPRPTY114 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.Rental.Amount
        */
        protected void VRATORNTLPRPTY000107(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000107", "CMN.ATO.RNTLPRPTY.000107",
                @"Your share of rental income is incorrect",
                @"When Your share of rental income is provided, the amount must not exceed (total) Rental income",
                "tns:Income.RealEstateProperty.Rental.Amount",
                "RNTLPRPTY136", (context => context.RNTLPRPTY136),
                "RNTLPRPTY114", (context => context.RNTLPRPTY114));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000108

        /*  VR.ATO.RNTLPRPTY.000108
        When the Ownership percentage is less than 100% and (total) Other rental related income is present, Your share of other rental related income must be provided
    
        Legacy Rule Format:
        (^RNTLPRPTY137 = NULL) AND (^RNTLPRPTY115 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)
        
        Technical Business Rule Format:
        (^RNTLPRPTY137 = NULL) AND (^RNTLPRPTY115 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)
            
        Data Elements:
            
        ^RNTLPRPTY137 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.RentalRelatedOther.Amount
            
        ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
            
        ^RNTLPRPTY115 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.RentalRelatedOther.Amount
        */
        protected void VRATORNTLPRPTY000108(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000108", "CMN.ATO.RNTLPRPTY.000108",
                @"Your share of other rental related income must be provided",
                @"When the Ownership percentage is less than 100% and (total) Other rental related income is present, Your share of other rental related income must be provided",
                "tns:Income.RealEstateProperty.RentalRelatedOther.Amount",
                "RNTLPRPTY137", (context => context.RNTLPRPTY137),
                "RNTLPRPTY115", (context => context.RNTLPRPTY115));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000109

        /*  VR.ATO.RNTLPRPTY.000109
        When Your share of other rental related income is provided, the amount must not exceed (total) Other rental related income
    
        Legacy Rule Format:
        (^RNTLPRPTY137 <> NULL) AND (^RNTLPRPTY115 = NULL OR ^RNTLPRPTY137 > ^RNTLPRPTY115)
        
        Technical Business Rule Format:
        (^RNTLPRPTY137 <> NULL) AND (^RNTLPRPTY115 = NULL OR ^RNTLPRPTY137 > ^RNTLPRPTY115)
            
        Data Elements:
            
        ^RNTLPRPTY137 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.RentalRelatedOther.Amount
            
        ^RNTLPRPTY115 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.RentalRelatedOther.Amount
        */
        protected void VRATORNTLPRPTY000109(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000109", "CMN.ATO.RNTLPRPTY.000109",
                @"Your share of other rental related income is incorrect",
                @"When Your share of other rental related income is provided, the amount must not exceed (total) Other rental related income",
                "tns:Income.RealEstateProperty.RentalRelatedOther.Amount",
                "RNTLPRPTY137", (context => context.RNTLPRPTY137),
                "RNTLPRPTY115", (context => context.RNTLPRPTY115));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000110

        /*  VR.ATO.RNTLPRPTY.000110
        When the Ownership percentage is less than 100% and (total) Interest on loans is present, Your share of interest on loans must be provided
    
        Legacy Rule Format:
        (^RNTLPRPTY146 = NULL) AND (^RNTLPRPTY124 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)
        
        Technical Business Rule Format:
        (^RNTLPRPTY146 = NULL) AND (^RNTLPRPTY124 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)
            
        Data Elements:
            
        ^RNTLPRPTY146 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LoanInterest.Amount
            
        ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
            
        ^RNTLPRPTY124 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LoanInterest.Amount
        */
        protected void VRATORNTLPRPTY000110(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000110", "CMN.ATO.RNTLPRPTY.000110",
                @"Your share of interest on loans must be provided",
                @"When the Ownership percentage is less than 100% and (total) Interest on loans is present, Your share of interest on loans must be provided",
                "tns:Expense.RealEstateProperty.LoanInterest.Amount",
                "RNTLPRPTY146", (context => context.RNTLPRPTY146),
                "RNTLPRPTY124", (context => context.RNTLPRPTY124));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000111

        /*  VR.ATO.RNTLPRPTY.000111
        When Your share of interest on loans is provided, the amount must not exceed (total) Interest on loans
    
        Legacy Rule Format:
        (^RNTLPRPTY146 <> NULL) AND (^RNTLPRPTY124 = NULL OR ^RNTLPRPTY146 > ^RNTLPRPTY124)
        
        Technical Business Rule Format:
        (^RNTLPRPTY146 <> NULL) AND (^RNTLPRPTY124 = NULL OR ^RNTLPRPTY146 > ^RNTLPRPTY124)
            
        Data Elements:
            
        ^RNTLPRPTY146 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LoanInterest.Amount
            
        ^RNTLPRPTY124 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LoanInterest.Amount
        */
        protected void VRATORNTLPRPTY000111(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000111", "CMN.ATO.RNTLPRPTY.000111",
                @"Your share of interest on loans is incorrect",
                @"When Your share of interest on loans is provided, the amount must not exceed (total) Interest on loans",
                "tns:Expense.RealEstateProperty.LoanInterest.Amount",
                "RNTLPRPTY146", (context => context.RNTLPRPTY146),
                "RNTLPRPTY124", (context => context.RNTLPRPTY124));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000112

        /*  VR.ATO.RNTLPRPTY.000112
        When the Ownership percentage is less than 100% and (total) Capital works deductions is present, Your share of capital works deductions must be provided

        Legacy Rule Format:
        (^RNTLPRPTY152 = NULL) AND (^RNTLPRPTY130 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Technical Business Rule Format:
        (^RNTLPRPTY152 = NULL) AND (^RNTLPRPTY130 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
        ^RNTLPRPTY152 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
    
        ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
        ^RNTLPRPTY130 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
        */
        protected void VRATORNTLPRPTY000112(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000112", "CMN.ATO.RNTLPRPTY.000112",
                @"Your share of capital works deductions must be provided",
                @"When the Ownership percentage is less than 100% and (total) Capital works deductions is present, Your share of capital works deductions must be provided",
                "tns:Expense.RealEstateProperty.CapitalWorksDeduction.Amount",
                "RNTLPRPTY152", (context => context.RNTLPRPTY152),
                "RNTLPRPTY130", (context => context.RNTLPRPTY130));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000113

        /*  VR.ATO.RNTLPRPTY.000113
        When Your share of capital works deductions is provided, the amount must not exceed (total) Capital works deductions

        Legacy Rule Format:
        (^RNTLPRPTY152 <> NULL) AND (^RNTLPRPTY130 = NULL OR ^RNTLPRPTY152 > ^RNTLPRPTY130)

        Technical Business Rule Format:
        (^RNTLPRPTY152 <> NULL) AND (^RNTLPRPTY130 = NULL OR ^RNTLPRPTY152 > ^RNTLPRPTY130)

        Data Elements:
         * 
    
        ^RNTLPRPTY152 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
    
        ^RNTLPRPTY130 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
        */
        protected void VRATORNTLPRPTY000113(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000113", "CMN.ATO.RNTLPRPTY.000113",
                @"Your share of capital works deductions is incorrect",
                @"When Your share of capital works deductions is provided, the amount must not exceed (total) Capital works deductions",
                "tns:Expense.RealEstateProperty.CapitalWorksDeduction.Amount",
                "RNTLPRPTY152", (context => context.RNTLPRPTY152),
                "RNTLPRPTY130", (context => context.RNTLPRPTY130));
        }

        #endregion

        #region VR.ATO.RNTLPRPTY.000120

        /*  VR.ATO.RNTLPRPTY.000120
        Date property first earned rental income must be prior to form submission end date

        Legacy Rule Format:
        (AnyOccurrence(^RNTLPRPTY105, ^RNTLPRPTY105 > ^EndDate))

        Technical Business Rule Format:
        (AnyOccurrence(^RNTLPRPTY105, ^RNTLPRPTY105 > ^EndDate))

        Data Elements:

        ^RNTLPRPTY105 = RP.{PrptySeqNum}:RealEstateProperty.RentalIncomeFirstEarned.Date

        ^EndDate = EndDate
        */
        protected void VRATORNTLPRPTY000120(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            bool assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY105Repeat => RNTLPRPTY105Repeat.RNTLPRPTY105 > RNTLPRPTY105Repeat.EndDate));
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000120",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date property first earned rental income must be prior to form submission end date",
                    LongDescription = @"The date entered is a future date. Date property first earned rental income cannot be later than the form submission end date.",
                    Location = "/xbrli:xbrl/tns:RealEstateProperty.RentalIncomeFirstEarned.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000120" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY105", Value = (report.RPPrptySeqNumContextCollection == null ? "null" : report.RPPrptySeqNumContextCollection.Find(RNTLPRPTY105Repeat => RNTLPRPTY105Repeat.RNTLPRPTY105 > RNTLPRPTY105Repeat.EndDate).RNTLPRPTY105.ToString()) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.000120

        #region VR.ATO.RNTLPRPTY.000200

        /*  VR.ATO.RNTLPRPTY.000200
                    When the Ownership percentage is less than 100% and (total) Advertising for tenants is present, Your share of advertising for tenants must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY138 = NULL) AND (^RNTLPRPTY116 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY138 = NULL) AND (^RNTLPRPTY116 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY138 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Marketing.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY116 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Marketing.Amount
            */
        protected void VRATORNTLPRPTY000200(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000200", "CMN.ATO.RNTLPRPTY.000200",
                @"Your share of advertising for tenants must be provided",
                @"When the Ownership percentage is less than 100% and (total) Advertising for tenants is present, Your share of advertising for tenants must be provided",
                "tns:Expense.RealEstateProperty.Marketing.Amount",
                "RNTLPRPTY138", (context => context.RNTLPRPTY138),
                "RNTLPRPTY116", (context => context.RNTLPRPTY116));
        }
        #endregion // VR.ATO.RNTLPRPTY.000200

        #region VR.ATO.RNTLPRPTY.000201

        /*  VR.ATO.RNTLPRPTY.000201
                    When Your share of advertising for tenants is provided, the amount must not exceed (total) Advertising for tenants

                Legacy Rule Format:
                    (^RNTLPRPTY138 <> NULL) AND (^RNTLPRPTY116 = NULL OR ^RNTLPRPTY138 > ^RNTLPRPTY116)

            Technical Business Rule Format:
                    (^RNTLPRPTY138 <> NULL) AND (^RNTLPRPTY116 = NULL OR ^RNTLPRPTY138 > ^RNTLPRPTY116)

        Data Elements:
    
            ^RNTLPRPTY138 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Marketing.Amount
    
            ^RNTLPRPTY116 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Marketing.Amount
            */
        protected void VRATORNTLPRPTY000201(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000201", "CMN.ATO.RNTLPRPTY.000201",
                @"Your share of advertising for tenants is incorrect",
                @"When Your share of advertising for tenants is provided, the amount must not exceed (total) Advertising for tenants",
                "tns:Expense.RealEstateProperty.Marketing.Amount",
                "RNTLPRPTY138", (context => context.RNTLPRPTY138),
                "RNTLPRPTY116", (context => context.RNTLPRPTY116));
        }
        #endregion // VR.ATO.RNTLPRPTY.000201

        #region VR.ATO.RNTLPRPTY.000202

        /*  VR.ATO.RNTLPRPTY.000202
                    When the Ownership percentage is less than 100% and (total) Body corporate fees is present, Your share of body corporate fees must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY139 = NULL) AND (^RNTLPRPTY117 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY139 = NULL) AND (^RNTLPRPTY117 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY139 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.BodyCorporate.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY117 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.BodyCorporate.Amount
            */
        protected void VRATORNTLPRPTY000202(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000202", "CMN.ATO.RNTLPRPTY.000202",
                @"Your share of body corporate fees must be provided",
                @"When the Ownership percentage is less than 100% and (total) Body corporate fees is present, Your share of body corporate fees must be provided",
                "tns:Expense.RealEstateProperty.BodyCorporate.Amount",
                "RNTLPRPTY139", (context => context.RNTLPRPTY139),
                "RNTLPRPTY117", (context => context.RNTLPRPTY117));
        }
        #endregion // VR.ATO.RNTLPRPTY.000202

        #region VR.ATO.RNTLPRPTY.000203

        /*  VR.ATO.RNTLPRPTY.000203
                    When Your share of body corporate fees is provided, the amount must not exceed (total) Body corporate fees

                Legacy Rule Format:
                    (^RNTLPRPTY139 <> NULL) AND (^RNTLPRPTY117 = NULL OR ^RNTLPRPTY139 > ^RNTLPRPTY117)

            Technical Business Rule Format:
                    (^RNTLPRPTY139 <> NULL) AND (^RNTLPRPTY117 = NULL OR ^RNTLPRPTY139 > ^RNTLPRPTY117)

        Data Elements:
    
            ^RNTLPRPTY139 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.BodyCorporate.Amount
    
            ^RNTLPRPTY117 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.BodyCorporate.Amount
            */
        protected void VRATORNTLPRPTY000203(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000203", "CMN.ATO.RNTLPRPTY.000203",
                @"Your share of body corporate fees is incorrect",
                @"When Your share of body corporate fees is provided, the amount must not exceed (total) Body corporate fees",
                "tns:Expense.RealEstateProperty.BodyCorporate.Amount",
                "RNTLPRPTY139", (context => context.RNTLPRPTY139),
                "RNTLPRPTY117", (context => context.RNTLPRPTY117));
        }
        #endregion // VR.ATO.RNTLPRPTY.000203

        #region VR.ATO.RNTLPRPTY.000204

        /*  VR.ATO.RNTLPRPTY.000204
                    When the Ownership percentage is less than 100% and (total) Borrowing expenses is present, Your share of borrowing expenses must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY140 = NULL) AND (^RNTLPRPTY118 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY140 = NULL) AND (^RNTLPRPTY118 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY140 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Borrowing.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY118 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Borrowing.Amount
            */
        protected void VRATORNTLPRPTY000204(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000204", "CMN.ATO.RNTLPRPTY.000204",
                @"Your share of borrowing expenses must be provided",
                @"When the Ownership percentage is less than 100% and (total) Borrowing expenses is present, Your share of borrowing expenses must be provided",
                "tns:Expense.RealEstateProperty.Borrowing.Amount",
                "RNTLPRPTY140", (context => context.RNTLPRPTY140),
                "RNTLPRPTY118", (context => context.RNTLPRPTY118));
        }
        #endregion // VR.ATO.RNTLPRPTY.000204

        #region VR.ATO.RNTLPRPTY.000205

        /*  VR.ATO.RNTLPRPTY.000205
                    When Your share of borrowing expenses is provided, the amount must not exceed (total) Borrowing expenses

                Legacy Rule Format:
                    (^RNTLPRPTY140 <> NULL) AND (^RNTLPRPTY118 = NULL OR ^RNTLPRPTY140 > ^RNTLPRPTY118)

            Technical Business Rule Format:
                    (^RNTLPRPTY140 <> NULL) AND (^RNTLPRPTY118 = NULL OR ^RNTLPRPTY140 > ^RNTLPRPTY118)

        Data Elements:
    
            ^RNTLPRPTY140 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Borrowing.Amount
    
            ^RNTLPRPTY118 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Borrowing.Amount
            */
        protected void VRATORNTLPRPTY000205(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000205", "CMN.ATO.RNTLPRPTY.000205",
                @"Your share of borrowing expenses is incorrect",
                @"When Your share of borrowing expenses is provided, the amount must not exceed (total) Borrowing expenses",
                "tns:Expense.RealEstateProperty.Borrowing.Amount",
                "RNTLPRPTY140", (context => context.RNTLPRPTY140),
                "RNTLPRPTY118", (context => context.RNTLPRPTY118));
        }
        #endregion // VR.ATO.RNTLPRPTY.000205

        #region VR.ATO.RNTLPRPTY.000206

        /*  VR.ATO.RNTLPRPTY.000206
                    When the Ownership percentage is less than 100% and (total) Cleaning is present, Your share of cleaning must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY141 = NULL) AND (^RNTLPRPTY119 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY141 = NULL) AND (^RNTLPRPTY119 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY141 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Cleaning.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY119 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Cleaning.Amount
            */
        protected void VRATORNTLPRPTY000206(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000206", "CMN.ATO.RNTLPRPTY.000206",
                @"Your share of cleaning must be provided",
                @"When the Ownership percentage is less than 100% and (total) Cleaning is present, Your share of cleaning must be provided",
                "tns:Expense.RealEstateProperty.Cleaning.Amount",
                "RNTLPRPTY141", (context => context.RNTLPRPTY141),
                "RNTLPRPTY119", (context => context.RNTLPRPTY119));
        }
        #endregion // VR.ATO.RNTLPRPTY.000206

        #region VR.ATO.RNTLPRPTY.000207

        /*  VR.ATO.RNTLPRPTY.000207
                    When Your share of cleaning is provided, the amount must not exceed (total) Cleaning

                Legacy Rule Format:
                    (^RNTLPRPTY141 <> NULL) AND (^RNTLPRPTY119 = NULL OR ^RNTLPRPTY141 > ^RNTLPRPTY119)

            Technical Business Rule Format:
                    (^RNTLPRPTY141 <> NULL) AND (^RNTLPRPTY119 = NULL OR ^RNTLPRPTY141 > ^RNTLPRPTY119)

        Data Elements:
    
            ^RNTLPRPTY141 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Cleaning.Amount
    
            ^RNTLPRPTY119 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Cleaning.Amount
            */
        protected void VRATORNTLPRPTY000207(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000207", "CMN.ATO.RNTLPRPTY.000207",
                @"Your share of cleaning is incorrect",
                @"When Your share of cleaning is provided, the amount must not exceed (total) Cleaning",
                "tns:Expense.RealEstateProperty.Cleaning.Amount",
                "RNTLPRPTY141", (context => context.RNTLPRPTY141),
                "RNTLPRPTY119", (context => context.RNTLPRPTY119));
        }
        #endregion // VR.ATO.RNTLPRPTY.000207

        #region VR.ATO.RNTLPRPTY.000208

        /*  VR.ATO.RNTLPRPTY.000208
                    When the Ownership percentage is less than 100% and (total) Council rates is present, Your share of council rates must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY142 = NULL) AND (^RNTLPRPTY120 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY142 = NULL) AND (^RNTLPRPTY120 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY142 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CouncilRates.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY120 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CouncilRates.Amount
            */
        protected void VRATORNTLPRPTY000208(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000208", "CMN.ATO.RNTLPRPTY.000208",
                @"Your share of council rates must be provided",
                @"When the Ownership percentage is less than 100% and (total) Council rates is present, Your share of council rates must be provided",
                "tns:Expense.RealEstateProperty.CouncilRates.Amount",
                "RNTLPRPTY142", (context => context.RNTLPRPTY142),
                "RNTLPRPTY120", (context => context.RNTLPRPTY120));
        }
        #endregion // VR.ATO.RNTLPRPTY.000208

        #region VR.ATO.RNTLPRPTY.000209

        /*  VR.ATO.RNTLPRPTY.000209
                    When Your share of council rates is provided, the amount must not exceed (total) Council rates

                Legacy Rule Format:
                    (^RNTLPRPTY142 <> NULL) AND (^RNTLPRPTY120 = NULL OR ^RNTLPRPTY142 > ^RNTLPRPTY120)

            Technical Business Rule Format:
                    (^RNTLPRPTY142 <> NULL) AND (^RNTLPRPTY120 = NULL OR ^RNTLPRPTY142 > ^RNTLPRPTY120)

        Data Elements:
    
            ^RNTLPRPTY142 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CouncilRates.Amount
    
            ^RNTLPRPTY120 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CouncilRates.Amount
            */
        protected void VRATORNTLPRPTY000209(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000209", "CMN.ATO.RNTLPRPTY.000209",
                @"Your share of council rates is incorrect",
                @"When Your share of council rates is provided, the amount must not exceed (total) Council rates",
                "tns:Expense.RealEstateProperty.CouncilRates.Amount",
                "RNTLPRPTY142", (context => context.RNTLPRPTY142),
                "RNTLPRPTY120", (context => context.RNTLPRPTY120));
        }
        #endregion // VR.ATO.RNTLPRPTY.000209

        #region VR.ATO.RNTLPRPTY.000210

        /*  VR.ATO.RNTLPRPTY.000210
                    When the Ownership percentage is less than 100% and (total) Capital allowances (depreciation on plant) is present, Your share of capital allowances (depreciation on plant) must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY143 = NULL) AND (^RNTLPRPTY121 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY143 = NULL) AND (^RNTLPRPTY121 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY143 = RP.{PrptySeqNum}.Portion:Expense.DepreciationandAmortisation.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY121 = RP.{PrptySeqNum}.Total:Expense.DepreciationandAmortisation.Amount
            */
        protected void VRATORNTLPRPTY000210(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000210", "CMN.ATO.RNTLPRPTY.000210",
                @"Your share of capital allowances (depreciation on plant) must be provided",
                @"When the Ownership percentage is less than 100% and (total) Capital allowances (depreciation on plant) is present, Your share of capital allowances (depreciation on plant) must be provided",
                "tns:Expense.DepreciationandAmortisation.Amount",
                "RNTLPRPTY143", (context => context.RNTLPRPTY143),
                "RNTLPRPTY121", (context => context.RNTLPRPTY121));
        }
        #endregion // VR.ATO.RNTLPRPTY.000210

        #region VR.ATO.RNTLPRPTY.000211

        /*  VR.ATO.RNTLPRPTY.000211
                    When Your share of capital allowances (depreciation on plant) is provided, the amount must not exceed (total) Capital allowances (depreciation on plant)

                Legacy Rule Format:
                    (^RNTLPRPTY143 <> NULL) AND (^RNTLPRPTY121 = NULL OR ^RNTLPRPTY143 > ^RNTLPRPTY121)

            Technical Business Rule Format:
                    (^RNTLPRPTY143 <> NULL) AND (^RNTLPRPTY121 = NULL OR ^RNTLPRPTY143 > ^RNTLPRPTY121)

        Data Elements:
    
            ^RNTLPRPTY143 = RP.{PrptySeqNum}.Portion:Expense.DepreciationandAmortisation.Amount
    
            ^RNTLPRPTY121 = RP.{PrptySeqNum}.Total:Expense.DepreciationandAmortisation.Amount
            */
        protected void VRATORNTLPRPTY000211(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000211", "CMN.ATO.RNTLPRPTY.000211",
                @"Your share of capital allowances (depreciation on plant) is incorrect",
                @"When Your share of capital allowances (depreciation on plant) is provided, the amount must not exceed (total) Capital allowances (depreciation on plant)",
                "tns:Expense.DepreciationandAmortisation.Amount",
                "RNTLPRPTY143", (context => context.RNTLPRPTY143),
                "RNTLPRPTY121", (context => context.RNTLPRPTY121));
        }
        #endregion // VR.ATO.RNTLPRPTY.000211

        #region VR.ATO.RNTLPRPTY.000212

        /*  VR.ATO.RNTLPRPTY.000212
                    When the Ownership percentage is less than 100% and (total) Gardening/lawn mowing is present, Your share of gardening/lawn mowing must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY144 = NULL) AND (^RNTLPRPTY122 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY144 = NULL) AND (^RNTLPRPTY122 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY144 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Gardening.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY122 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Gardening.Amount
            */
        protected void VRATORNTLPRPTY000212(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000212", "CMN.ATO.RNTLPRPTY.000212",
                @"Your share of gardening/lawn mowing must be provided",
                @"When the Ownership percentage is less than 100% and (total) Gardening/lawn mowing is present, Your share of gardening/lawn mowing must be provided",
                "tns:Expense.RealEstateProperty.Gardening.Amount",
                "RNTLPRPTY144", (context => context.RNTLPRPTY144),
                "RNTLPRPTY122", (context => context.RNTLPRPTY122));
        }
        #endregion // VR.ATO.RNTLPRPTY.000212

        #region VR.ATO.RNTLPRPTY.000213

        /*  VR.ATO.RNTLPRPTY.000213
                    When Your share of gardening/lawn mowing is provided, the amount must not exceed (total) Gardening/lawn mowing

                Legacy Rule Format:
                    (^RNTLPRPTY144 <> NULL) AND (^RNTLPRPTY122 = NULL OR ^RNTLPRPTY144 > ^RNTLPRPTY122)

            Technical Business Rule Format:
                    (^RNTLPRPTY144 <> NULL) AND (^RNTLPRPTY122 = NULL OR ^RNTLPRPTY144 > ^RNTLPRPTY122)

        Data Elements:
    
            ^RNTLPRPTY144 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Gardening.Amount
    
            ^RNTLPRPTY122 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Gardening.Amount
            */
        protected void VRATORNTLPRPTY000213(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000213", "CMN.ATO.RNTLPRPTY.000213",
                @"Your share of gardening/lawn mowing is incorrect",
                @"When Your share of gardening/lawn mowing is provided, the amount must not exceed (total) Gardening/lawn mowing",
                "tns:Expense.RealEstateProperty.Gardening.Amount",
                "RNTLPRPTY144", (context => context.RNTLPRPTY144),
                "RNTLPRPTY122", (context => context.RNTLPRPTY122));
        }
        #endregion // VR.ATO.RNTLPRPTY.000213

        #region VR.ATO.RNTLPRPTY.000214

        /*  VR.ATO.RNTLPRPTY.000214
                    When the Ownership percentage is less than 100% and (total) Insurance is present, Your share of insurance must be provided

                Legacy Rule Format:
                    (^RNTLPRPTY145 = NULL) AND (^RNTLPRPTY123 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                    (^RNTLPRPTY145 = NULL) AND (^RNTLPRPTY123 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

        Data Elements:
    
            ^RNTLPRPTY145 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Insurance.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY123 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Insurance.Amount
            */
        protected void VRATORNTLPRPTY000214(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustBeProvided(response, report, "VR.ATO.RNTLPRPTY.000214", "CMN.ATO.RNTLPRPTY.000214",
                  @"Your share of insurance must be provided",
                  @"When the Ownership percentage is less than 100% and (total) Insurance is present, Your share of insurance must be provided",
                  "tns:Expense.RealEstateProperty.Insurance.Amount",
                  "RNTLPRPTY145", (context => context.RNTLPRPTY145),
                  "RNTLPRPTY123", (context => context.RNTLPRPTY123));
        }
        #endregion // VR.ATO.RNTLPRPTY.000214

        #region VR.ATO.RNTLPRPTY.000215

        /*  VR.ATO.RNTLPRPTY.000215
                    When Your share of insurance is provided, the amount must not exceed (total) Insurance

                Legacy Rule Format:
                    (^RNTLPRPTY145 <> NULL) AND (^RNTLPRPTY123 = NULL OR ^RNTLPRPTY145 > ^RNTLPRPTY123)

            Technical Business Rule Format:
                    (^RNTLPRPTY145 <> NULL) AND (^RNTLPRPTY123 = NULL OR ^RNTLPRPTY145 > ^RNTLPRPTY123)

        Data Elements:
    
            ^RNTLPRPTY145 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Insurance.Amount
    
            ^RNTLPRPTY123 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Insurance.Amount
            */
        protected void VRATORNTLPRPTY000215(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            ValidateYourShareMustNotExceedTotal(response, report, "VR.ATO.RNTLPRPTY.000215", "CMN.ATO.RNTLPRPTY.000215",
                @"Your share of insurance is incorrect",
                @"When Your share of insurance is provided, the amount must not exceed (total) Insurance",
                "tns:Expense.RealEstateProperty.Insurance.Amount",
                "RNTLPRPTY145", (context => context.RNTLPRPTY145),
                "RNTLPRPTY123", (context => context.RNTLPRPTY123));
        }
        #endregion // VR.ATO.RNTLPRPTY.000215

        #region VR.ATO.RNTLPRPTY.W00216

        /*  VR.ATO.RNTLPRPTY.W00216
                    Only deductions for repairs and maintenance may be claimed that relate directly to wear and tear or other damage to rental properties and must not include capital works or depreciation

                Legacy Rule Format:
                    ((^RNTLPRPTY104 = 100) AND (^RNTLPRPTY129 <> NULL AND ^RNTLPRPTY129 > 3500)) OR ((^RNTLPRPTY104 < 100) AND (^RNTLPRPTY151 <> NULL AND ^RNTLPRPTY151 > 3500))

            Technical Business Rule Format:
                    ((^RNTLPRPTY104 = 100) AND (^RNTLPRPTY129 <> NULL AND ^RNTLPRPTY129 > 3500)) OR ((^RNTLPRPTY104 < 100) AND (^RNTLPRPTY151 <> NULL AND ^RNTLPRPTY151 > 3500))

        Data Elements:
    
            ^RNTLPRPTY129 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY151 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
            */
        protected void VRATORNTLPRPTYW00216(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            if (report.RPPrptySeqNumTotalContextCollection == null)
                return;

            // Evaluate the rule for all Property Total contexts as we can assume that each Portion context should have a related Total context
            foreach (RNTLPRPTY2019.RPPrptySeqNumTotalContext rpPrptySeqNumTotalContext in report.RPPrptySeqNumTotalContextCollection)
            {
                // Find the related Property parent and Portion contexts for this Total context
                RNTLPRPTY2019.RPPrptySeqNumContext rpPrptySeqNumContext = Count(report.RPPrptySeqNumContextCollection) > 0 ? report.RPPrptySeqNumContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == rpPrptySeqNumTotalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null;
                RNTLPRPTY2019.RPPrptySeqNumPortionContext rpPrptySeqNumPortionContext = Count(report.RPPrptySeqNumPortionContextCollection) > 0 ? report.RPPrptySeqNumPortionContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == rpPrptySeqNumTotalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null;

                // If this Total context doesn't have a related Property context then continue to the next in the collection, though this should never be the case given VR.ATO.RNTLPRPTY.000026
                if (rpPrptySeqNumContext == null)
                    continue;

                bool assertion = (rpPrptySeqNumContext.RNTLPRPTY104.GetValueOrDefault() == 100 && (rpPrptySeqNumTotalContext.RNTLPRPTY129.HasValue && rpPrptySeqNumTotalContext.RNTLPRPTY129.Value > 3500)
                                 || rpPrptySeqNumContext.RNTLPRPTY104.GetValueOrDefault() < 100 && (rpPrptySeqNumPortionContext != null && rpPrptySeqNumPortionContext.RNTLPRPTY151.HasValue && rpPrptySeqNumPortionContext.RNTLPRPTY151.Value > 3500));
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.RNTLPRPTY.W00216",
                        Severity = ProcessMessageSeverity.Warning,
                        Description = @"Check repairs and maintenance expenses",
                        LongDescription = @"Only deductions for repairs and maintenance may be claimed that relate directly to wear and tear or other damage to rental properties and must not include capital works or depreciation",
                        Location = "/xbrli:xbrl/tns:Expense.RealEstateProperty.RepairsAndMaintenance.Amount[@contextRef='" + rpPrptySeqNumTotalContext.Id + "']",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.W00216" } },
                    };

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "RNTLPRPTY104", Value = rpPrptySeqNumContext.RNTLPRPTY104.GetValueOrDefault().ToString() });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "RNTLPRPTY129", Value = GetValueOrEmpty(rpPrptySeqNumTotalContext.RNTLPRPTY129) });

                    // Only add RNTLPRPTY151 to parameters if the related Portion context was found
                    if (rpPrptySeqNumPortionContext != null)
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "RNTLPRPTY151", Value = GetValueOrEmpty(rpPrptySeqNumPortionContext.RNTLPRPTY151) });

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.W00216

        #region VR.ATO.RNTLPRPTY.W00217

        /*  VR.ATO.RNTLPRPTY.W00217
                    Total borrowing expenses claimed is very high. Borrowing expenses must be claimed over five years from when the loan was established and must not include the loan balance, interest or stamp duty on the transfer of the property.

                Legacy Rule Format:
                    (AnyOccurrence(^RNTLPRPTY118, ^RNTLPRPTY118 > 10000))

            Technical Business Rule Format:
                    (AnyOccurrence(^RNTLPRPTY118, ^RNTLPRPTY118 > 10000))

        Data Elements:
    
            ^RNTLPRPTY118 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Borrowing.Amount
            */
        protected void VRATORNTLPRPTYW00217(List<ProcessMessageDocument> response, RNTLPRPTY2019 report)
        {
            if (report.RPPrptySeqNumTotalContextCollection == null)
                return;

            // Evaluate the rule for each Property Total context so that we can raise the message with the context and actual field value
            foreach (RNTLPRPTY2019.RPPrptySeqNumTotalContext rpPrptySeqNumTotalContext in report.RPPrptySeqNumTotalContextCollection)
            {
                bool assertion = (rpPrptySeqNumTotalContext.RNTLPRPTY118.GetValueOrDefault() > 10000);
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.RNTLPRPTY.W00217",
                        Severity = ProcessMessageSeverity.Warning,
                        Description = @"Check (total) Borrowing expenses",
                        LongDescription = @"Total borrowing expenses claimed is very high. Borrowing expenses must be claimed over five years from when the loan was established and must not include the loan balance, interest or stamp duty on the transfer of the property.",
                        Location = "/xbrli:xbrl/tns:Expense.RealEstateProperty.Borrowing.Amount[@contextRef='" + rpPrptySeqNumTotalContext.Id + "']",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.W00217" } },
                    };

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "RNTLPRPTY118", Value = GetValueOrEmpty(rpPrptySeqNumTotalContext.RNTLPRPTY118) });

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.RNTLPRPTY.W00217

        #endregion Manual Rules - Rules marked for manual coding
    }
}