
using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationCGLS
{

    public partial class CGLS2017Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="CGLS2017Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CGLS2017Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : day.ToString(), month == 0 ? null : month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }


        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public CGLS2017 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CGLS2017 report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;


    
            #region VR.ATO.CGLS.404000
    
            /*  VR.ATO.CGLS.404000
            If any labels have been completed that relate to tax losses or net capital losses that have satisfied the business continuity test, then an amount must be entered against at least one of the financial years listed showing the year ownership test failed.
    
            Legacy Rule Format:
            IF ([CGLS6] > 0 OR [CGLS9] > 0 OR [CGLS21] > 0 OR [CGLS24] > 0) AND ([CGLS39] = NULL OR [CGLS39] = 0) AND ([CGLS40] = NULL OR [CGLS40] = 0) AND ([CGLS41] = NULL OR [CGLS41] = 0) AND ([CGLS42] = NULL OR [CGLS42] = 0) AND ([CGLS43] = NULL OR [CGLS43] = 0)
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS6] = CGLS:RP.COTF:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount
            [CGLS9] = CGLS:RP.COTF:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            [CGLS21] = CGLS:RP.COTF:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount
            [CGLS24] = CGLS:RP.COTF:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            [CGLS39] = CGLS:RP.Y0:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount
            [CGLS40] = CGLS:RP.Y0-1:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount
            [CGLS41] = CGLS:RP.Y0-2:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount
            [CGLS42] = CGLS:RP.Y0-3:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount
            [CGLS43] = CGLS:RP.Y0-4P:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount

            Technical Business Rule Format:
            (^CGLS6 > 0 OR ^CGLS9 > 0 OR ^CGLS21 > 0 OR ^CGLS24 > 0) AND (^CGLS39 = NULL OR ^CGLS39 = 0) AND (^CGLS40 = NULL OR ^CGLS40 = 0) AND (^CGLS41 = NULL OR ^CGLS41 = 0) AND (^CGLS42 = NULL OR ^CGLS42 = 0) AND (^CGLS43 = NULL OR ^CGLS43 = 0)
    
            Data Elements:
    
            RP.COTF:^CGLS6 = tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount
    
            RP.COTF:^CGLS9 = tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
    
            RP.COTF:^CGLS21 = tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount
    
            RP.COTF:^CGLS24 = tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
    
            RP.Y0:^CGLS39 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
    
            RP.Y0-1:^CGLS40 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
    
            RP.Y0-2:^CGLS41 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
    
            RP.Y0-3:^CGLS42 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
    
            RP.Y0-4P:^CGLS43 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
            */
            assertion = ((report.CGLS6.GetValueOrDefault() > 0 || report.CGLS9.GetValueOrDefault() > 0 || report.CGLS21.GetValueOrDefault() > 0 || report.CGLS24.GetValueOrDefault() > 0) && (report.CGLS39 == null || report.CGLS39 == 0) && (report.CGLS40 == null || report.CGLS40 == 0) && (report.CGLS41 == null || report.CGLS41 == 0) && (report.CGLS42 == null || report.CGLS42 == 0) && (report.CGLS43 == null || report.CGLS43 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404235", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital loss transferred from joining entities present has been applied incorrect.",
                    LongDescription = @"If any labels have been completed that relate to tax losses or net capital losses that have satisfied the business continuity test, then an amount must be entered against at least one of the financial years listed showing the year ownership test failed.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount[@contextRef='" + report.RPCOTFId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404000"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS6", Value = report.CGLS6.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS9", Value = report.CGLS9.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS21", Value = report.CGLS21.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS24", Value = report.CGLS24.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS39", Value = report.CGLS39.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS40", Value = report.CGLS40.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS41", Value = report.CGLS41.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS42", Value = report.CGLS42.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS43", Value = report.CGLS43.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404000
    
            #region VR.ATO.CGLS.404008
    
            /*  VR.ATO.CGLS.404008
            If individual transferor TFNs,  available fractions and amounts have been entered for transferred net capital losses deducted and transferred tax losses deducted then the question "Were the apportionment rules applied?" must be answered with either "yes" or "no".
    
            Legacy Rule Format:
            IF (ANY OCCURRENCE OF [CGLS109] > 0 OR ANY OCCURRENCE OF [CGLS114] > 0) AND [CGLS35] = NULL
                  RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS35] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredLossUtilisedApportionmentRulesApplied.Indicator
            [CGLS109] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Amount
            [CGLS114] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Amount

            Technical Business Rule Format:
            (AnyOccurrence(^CGLS109, ^CGLS109 > 0) OR AnyOccurrence(^CGLS114, ^CGLS114 > 0)) AND ^CGLS35 = NULL
    
            Data Elements:
    
            TRFR:^CGLS109 = tns:Tax.Losses.TransferredDeducted.Amount
    
            RP:^CGLS35 = tns:Tax.Losses.TransferredLossUtilisedApportionmentRulesApplied.Indicator
    
            TRFR:^CGLS114 = tns:Capital.Losses.TransferredApplied.Amount
            */
            assertion = (((report.TRFRContextCollection == null ? false : report.TRFRContextCollection.Any(CGLS109Repeat => CGLS109Repeat.CGLS109.GetValueOrDefault() > 0)) || (report.TRFRContextCollection == null ? false : report.TRFRContextCollection.Any(CGLS114Repeat => CGLS114Repeat.CGLS114.GetValueOrDefault() > 0))) && report.CGLS35 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404222", Severity = ProcessMessageSeverity.Error,
                    Description = @"Details of transferred tax/net capital losses deducted/applied. Apportionment rules applied not indicated",
                    LongDescription = @"If transferred tax losses deducted or transferred net capital losses applied fields completed then (Were the apportionment rules applied?) must equal yes or no",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS109", Value = "CGLS109" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS114", Value = "CGLS114" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS35", Value = GetValueOrEmpty(report.CGLS35) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404008
    
            #region VR.ATO.CGLS.404009
    
            /*  VR.ATO.CGLS.404009
            The question "Were the apportionment rules applied?" has been answered with either "yes" or "no" however individual transferor TFNs, available fractions and amounts for both transferred net capital losses applied and transferred tax losses deducted have not been completed.
    
            Legacy Rule Format:
            IF [CGLS35] <> NULL AND
             COUNT(CONTEXT(TRFR))=0
              RETURN VALIDATION MESSAGE
             ENDIF
            
            [CGLS35] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredLossUtilisedApportionmentRulesApplied.Indicator

            Technical Business Rule Format:
            ^CGLS35 <> NULL AND Count(^Context)=0
    
            Data Elements:
    
            RP:^CGLS35 = tns:Tax.Losses.TransferredLossUtilisedApportionmentRulesApplied.Indicator
    
            RP:^Context = TRFR
            */
            assertion = (report.CGLS35 != null && Count(report.TRFRContextCollection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404223", Severity = ProcessMessageSeverity.Error,
                    Description = @"Apportionment rules applied indicated. Details of transferred tax/net capital losses deducted absent",
                    LongDescription = @"(Were the apportionment rules applied?) equals yes or no however transferred tax losses deducted or transferred net capital losses applied fields have not been completed",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404009"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS35", Value = GetValueOrEmpty(report.CGLS35) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404009
    
            #region VR.ATO.CGLS.404010
    
            /*  VR.ATO.CGLS.404010
            The sum of group tax losses carried forward to later income years and transferred tax losses carried forward to later income years must equal the total tax losses carried forward to later income years.
    
            Legacy Rule Format:
            IF [CGLS17] + [CGLS18] <> [CGLS19]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS17] = CGLS:RP.GRP:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [CGLS18] = CGLS:RP.TRFD:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [CGLS19] = CGLS:RP.TOT:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            ^CGLS17 + ^CGLS18 <> ^CGLS19
    
            Data Elements:
    
            RP.GRP:^CGLS17 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.TRFD:^CGLS18 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.TOT:^CGLS19 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = (report.CGLS17.GetValueOrDefault() + report.CGLS18.GetValueOrDefault() != report.CGLS19.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Sum of tax losses carried forward must equal total tax losses carried forward",
                    LongDescription = @"The sum of (Tax losses carried forward to later income years - Group) and (Tax losses carried forward to later income years - Transferred) must equal (Tax losses carried forward to later income years - Total)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPGRPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS17", Value = report.CGLS17.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS18", Value = report.CGLS18.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS19", Value = report.CGLS19.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404010
    
            #region VR.ATO.CGLS.404012
    
            /*  VR.ATO.CGLS.404012
            The amount of tax losses carried forward to later income years for which the business continuity test must be satisfied before they can be deducted cannot exceed the total amount of tax losses carried forward to later income years.
    
            Legacy Rule Format:
            IF [CGLS117] <> NULL AND [CGLS117] > [CGLS19]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS19] = CGLS:RP.TOT:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [CGLS117] = CGLS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            ^CGLS117 <> NULL AND ^CGLS117 > ^CGLS19
    
            Data Elements:
    
            RP.TOT:^CGLS19 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP:^CGLS117 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = (report.CGLS117 != null && report.CGLS117.GetValueOrDefault() > report.CGLS19.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404231", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses carried forward to later income years incorrect.",
                    LongDescription = @"The amount of tax losses carried forward to later income years for which the business continuity test must be satisfied before they can be deducted cannot exceed the total amount of tax losses carried forward to later income years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404012"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS117", Value = report.CGLS117.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS19", Value = report.CGLS19.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404012
    
            #region VR.ATO.CGLS.404013
    
            /*  VR.ATO.CGLS.404013
            Tax losses carried forward to later income years-Total must equal the Total tax losses carried forward to later years.
    
            Legacy Rule Format:
            IF ([CGLS19] <> NULL OR [CGLS93] <> NULL)
            AND [CGLS19] <> [CGLS93]  
            RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS19] = CGLS:RP.TOT:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [CGLS93] = CGLS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount

            Technical Business Rule Format:
            (^CGLS19 <> NULL OR ^CGLS93 <> NULL) AND ^CGLS19 <> ^CGLS93
    
            Data Elements:
    
            RP.TOT:^CGLS19 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP:^CGLS93 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = ((report.CGLS19 != null || report.CGLS93 != null) && report.CGLS19.GetValueOrDefault() != report.CGLS93.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404013", Severity = ProcessMessageSeverity.Error,
                    Description = @"(Tax losses carried forward to later income years - Total) must equal Part F label K",
                    LongDescription = @"(Tax losses carried forward to later income years - Total) must be equal to (Tax losses reconciliation for consolidated groups - Total tax losses carried forward to later income years)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS19", Value = report.CGLS19.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS93", Value = report.CGLS93.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404013
    
            #region VR.ATO.CGLS.404019
    
            /*  VR.ATO.CGLS.404019
            The sum of group net capital losses carried forward to later income years and transferred net capital losses carried forward to later income years must equal the total net capital losses carried forward to later income years.
    
            Legacy Rule Format:
            IF [CGLS32] + [CGLS33] <> [CGLS34]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS32] = CGLS:RP.GRP(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [CGLS33] = CGLS:RP.TRFD(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [CGLS34] = CGLS:RP.TOT(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            ^CGLS32 + ^CGLS33 <> ^CGLS34
    
            Data Elements:
    
            RP.GRP(Instant):^CGLS32 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.TRFD(Instant):^CGLS33 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.TOT(Instant):^CGLS34 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = (report.CGLS32.GetValueOrDefault() + report.CGLS33.GetValueOrDefault() != report.CGLS34.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404019", Severity = ProcessMessageSeverity.Error,
                    Description = @"Sum of net capital losses carried forward must equal total net capital losses carried forward",
                    LongDescription = @"The sum of (Net capital losses carried forward to later income years - Group) and (Net capital losses carried forward to later income years - Transferred) must equal (Net capital losses carried forward to later income years - Total)",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPGRPInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404019"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS32", Value = report.CGLS32.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS33", Value = report.CGLS33.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS34", Value = report.CGLS34.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404019
    
            #region VR.ATO.CGLS.404021
    
            /*  VR.ATO.CGLS.404021
            Amount of net capital losses carried forward to later income years for which the business continuity test must be satisfied before they can be applied, cannot exceed the total amount of net capital losses carried forward to later income years.
    
            Legacy Rule Format:
            IF [CGLS118] <> NULL AND [CGLS118] > [CGLS34]
                  RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS34] = CGLS:RP.TOT(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [CGLS118] = CGLS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            ^CGLS118 <> NULL AND ^CGLS118 > ^CGLS34
    
            Data Elements:
    
            RP.TOT(Instant):^CGLS34 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP:^CGLS118 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = (report.CGLS118 != null && report.CGLS118.GetValueOrDefault() > report.CGLS34.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404232", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses carried forward to later income years is incorrect.",
                    LongDescription = @"Amount of net capital losses carried forward to later income years for which the business continuity test must be satisfied before they can be applied, cannot exceed the total amount of net capital losses carried forward to later income years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPTOTInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404021"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS118", Value = report.CGLS118.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS34", Value = report.CGLS34.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404021
    
            #region VR.ATO.CGLS.404022
    
            /*  VR.ATO.CGLS.404022
            If the question "Has the head company cancelled the transfer of a loss?" has been answered with "yes" then the details of at least one individual joining entity TFN and amount  must be entered.
    
            Legacy Rule Format:
            IF [CGLS36] = TRUE AND
             COUNT CONTEXT JE = 0 
                    RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS36] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator

            Technical Business Rule Format:
            ^CGLS36 = TRUE AND Count(^Context) = 0
    
            Data Elements:
    
            RP:^CGLS36 = tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator
    
            RP:^Context = JE
            */
            assertion = (report.CGLS36 == true && Count(report.JEContextCollection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404022", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company has cancelled the transfer of a loss. Details of cancellation are absent or incomplete",
                    LongDescription = @"The head company has cancelled the transfer of a loss however (Details of cancellation of transfer of losses - Joining entity TFN) and (Details of cancellation of transfer of losses - Amount) are absent or incomplete",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404022"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS36", Value = GetValueOrEmpty(report.CGLS36) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404022
    
            #region VR.ATO.CGLS.404038
    
            /*  VR.ATO.CGLS.404038
            Ensure the number of transferor details that can be supplied do not exceed a count of 12
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(TRFR)) > 12
               RETURN VALIDATION MESSAGE 
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 12
    
            Data Elements:
    
            ^Context = TRFR
            */
            assertion = (Count(report.TRFRContextCollection) > 12);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404038", Severity = ProcessMessageSeverity.Error,
                    Description = @"The total number of transferor details that can be supplied is 12",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.TRFRContextCollection == null ? 0 : report.TRFRContextCollection.Last().LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404038"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404038
    
            #region VR.ATO.CGLS.404039
    
            /*  VR.ATO.CGLS.404039
            Ensure the number of Joining Entity details that can be supplied do not exceed a count of 4
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(JE)) > 4
               RETURN VALIDATION MESSAGE 
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 4
    
            Data Elements:
    
            ^Context = JE
            */
            assertion = (Count(report.JEContextCollection) > 4);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404039", Severity = ProcessMessageSeverity.Error,
                    Description = @"The total number of joining entity details that can be supplied is 4",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.JEContextCollection == null ? 0 : report.JEContextCollection.Last().LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404039"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404039
    
            #region VR.ATO.CGLS.404040
    
            /*  VR.ATO.CGLS.404040
            The total number RP.COTT supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.COTT)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.COTT
            */
            assertion = (Count(report.RPCOTTCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCOTTLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404040"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404040
    
            #region VR.ATO.CGLS.404041
    
            /*  VR.ATO.CGLS.404041
            The total number RP.COTP supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.COTP)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.COTP
            */
            assertion = (Count(report.RPCOTPCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCOTPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404041"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404041
    
            #region VR.ATO.CGLS.404042
    
            /*  VR.ATO.CGLS.404042
            The total number RP.COTF supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.COTF)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.COTF
            */
            assertion = (Count(report.RPCOTFCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCOTFLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404042"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404042
    
            #region VR.ATO.CGLS.404043
    
            /*  VR.ATO.CGLS.404043
            The total number RP.GRP supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.GRP)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.GRP
            */
            assertion = (Count(report.RPGRPCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGRPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404043"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404043
    
            #region VR.ATO.CGLS.404044
    
            /*  VR.ATO.CGLS.404044
            The total number RP.TFRD supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TRFD)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TRFD
            */
            assertion = (Count(report.RPTRFDCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTRFDLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404044"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404044
    
            #region VR.ATO.CGLS.404045
    
            /*  VR.ATO.CGLS.404045
            The total number RP.TOT supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TOT)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TOT
            */
            assertion = (Count(report.RPTOTCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOTLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404045"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404045
    
            #region VR.ATO.CGLS.404047
    
            /*  VR.ATO.CGLS.404047
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS5] <> NULL AND [CGLS5] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS5] = CGLS:RP.COTP:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS5, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTP:^CGLS5 = tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount
            */
            assertion = NotMonetary(report.CGLS5, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount[@contextRef='" + report.RPCOTPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404047"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS5", Value = report.CGLS5.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404047
    
            #region VR.ATO.CGLS.404048
    
            /*  VR.ATO.CGLS.404048
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS6] <> NULL AND [CGLS6] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS6] = CGLS:RP.COTF:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS6, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTF:^CGLS6 = tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount
            */
            assertion = NotMonetary(report.CGLS6, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount[@contextRef='" + report.RPCOTFId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404048"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS6", Value = report.CGLS6.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404048
    
            #region VR.ATO.CGLS.404049
    
            /*  VR.ATO.CGLS.404049
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS7] <> NULL AND [CGLS7] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS7] = CGLS:RP.COTT:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS7, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTT:^CGLS7 = tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount
            */
            assertion = NotMonetary(report.CGLS7, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesIncludingHeadCompany.Amount[@contextRef='" + report.RPCOTTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404049"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS7", Value = report.CGLS7.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404049
    
            #region VR.ATO.CGLS.404050
    
            /*  VR.ATO.CGLS.404050
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS8] <> NULL AND [CGLS8] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS8] = CGLS:RP.COTP:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS8, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTP:^CGLS8 = tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            */
            assertion = NotMonetary(report.CGLS8, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount[@contextRef='" + report.RPCOTPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404050"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS8", Value = report.CGLS8.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404050
    
            #region VR.ATO.CGLS.404051
    
            /*  VR.ATO.CGLS.404051
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS9] <> NULL AND [CGLS9] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS9] = CGLS:RP.COTF:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS9, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTF:^CGLS9 = tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            */
            assertion = NotMonetary(report.CGLS9, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount[@contextRef='" + report.RPCOTFId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404051"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS9", Value = report.CGLS9.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404051
    
            #region VR.ATO.CGLS.404052
    
            /*  VR.ATO.CGLS.404052
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS10] <> NULL AND [CGLS10] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS10] = CGLS:RP.COTT:rvctc3.02.34:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS10, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTT:^CGLS10 = tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            */
            assertion = NotMonetary(report.CGLS10, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount[@contextRef='" + report.RPCOTTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404052"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS10", Value = report.CGLS10.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404052
    
            #region VR.ATO.CGLS.404057
    
            /*  VR.ATO.CGLS.404057
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS17] <> NULL AND [CGLS17] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS17] = CGLS:RP.GRP:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS17, 'U', 11, 0)
    
            Data Elements:
    
            RP.GRP:^CGLS17 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.CGLS17, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPGRPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS17", Value = report.CGLS17.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404057
    
            #region VR.ATO.CGLS.404058
    
            /*  VR.ATO.CGLS.404058
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS18] <> NULL AND [CGLS18] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS18] = CGLS:RP.TRFD:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS18, 'U', 11, 0)
    
            Data Elements:
    
            RP.TRFD:^CGLS18 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.CGLS18, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS18", Value = report.CGLS18.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404058
    
            #region VR.ATO.CGLS.404059
    
            /*  VR.ATO.CGLS.404059
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS19] <> NULL AND [CGLS19] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS19] = CGLS:RP.TOT:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS19, 'U', 11, 0)
    
            Data Elements:
    
            RP.TOT:^CGLS19 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.CGLS19, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS19", Value = report.CGLS19.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404059
    
            #region VR.ATO.CGLS.404060
    
            /*  VR.ATO.CGLS.404060
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS20] <> NULL AND [CGLS20] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS20] = CGLS:RP.COTP:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS20, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTP:^CGLS20 = tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount
            */
            assertion = NotMonetary(report.CGLS20, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount[@contextRef='" + report.RPCOTPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404060"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS20", Value = report.CGLS20.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404060
    
            #region VR.ATO.CGLS.404061
    
            /*  VR.ATO.CGLS.404061
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS21] <> NULL AND [CGLS21] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS21] = CGLS:RP.COTF:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS21, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTF:^CGLS21 = tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount
            */
            assertion = NotMonetary(report.CGLS21, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount[@contextRef='" + report.RPCOTFId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404061"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS21", Value = report.CGLS21.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404061
    
            #region VR.ATO.CGLS.404062
    
            /*  VR.ATO.CGLS.404062
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS22] <> NULL AND [CGLS22] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS22] = CGLS:RP.COTT:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS22, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTT:^CGLS22 = tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount
            */
            assertion = NotMonetary(report.CGLS22, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredFromJoiningEntitiesIncludingHeadCompanyNet.Amount[@contextRef='" + report.RPCOTTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404062"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS22", Value = report.CGLS22.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404062
    
            #region VR.ATO.CGLS.404063
    
            /*  VR.ATO.CGLS.404063
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS23] <> NULL AND [CGLS23] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS23] = CGLS:RP.COTP:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS23, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTP:^CGLS23 = tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            */
            assertion = NotMonetary(report.CGLS23, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount[@contextRef='" + report.RPCOTPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS23", Value = report.CGLS23.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404063
    
            #region VR.ATO.CGLS.404064
    
            /*  VR.ATO.CGLS.404064
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS24] <> NULL AND [CGLS24] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS24] = CGLS:RP.COTF:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS24, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTF:^CGLS24 = tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            */
            assertion = NotMonetary(report.CGLS24, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount[@contextRef='" + report.RPCOTFId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404064"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS24", Value = report.CGLS24.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404064
    
            #region VR.ATO.CGLS.404065
    
            /*  VR.ATO.CGLS.404065
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS25] <> NULL AND [CGLS25] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS25] = CGLS:RP.COTT:rvctc3.02.34:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS25, 'U', 11, 0)
    
            Data Elements:
    
            RP.COTT:^CGLS25 = tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount
            */
            assertion = NotMonetary(report.CGLS25, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredFromJoiningEntitiesAfterConsolidation.Amount[@contextRef='" + report.RPCOTTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404065"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS25", Value = report.CGLS25.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404065
    
            #region VR.ATO.CGLS.404070
    
            /*  VR.ATO.CGLS.404070
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS32] <> NULL AND [CGLS32] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS32] = CGLS:RP.GRP(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS32, 'U', 11, 0)
    
            Data Elements:
    
            RP.GRP(Instant):^CGLS32 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.CGLS32, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPGRPInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404070"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS32", Value = report.CGLS32.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404070
    
            #region VR.ATO.CGLS.404071
    
            /*  VR.ATO.CGLS.404071
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS33] <> NULL AND [CGLS33] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS33] = CGLS:RP.TRFD(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS33, 'U', 11, 0)
    
            Data Elements:
    
            RP.TRFD(Instant):^CGLS33 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.CGLS33, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPTRFDInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404071"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS33", Value = report.CGLS33.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404071
    
            #region VR.ATO.CGLS.404072
    
            /*  VR.ATO.CGLS.404072
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS34] <> NULL AND [CGLS34] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS34] = CGLS:RP.TOT(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS34, 'U', 11, 0)
    
            Data Elements:
    
            RP.TOT(Instant):^CGLS34 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.CGLS34, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPTOTInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS34", Value = report.CGLS34.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404072
    
            #region VR.ATO.CGLS.404074
    
            /*  VR.ATO.CGLS.404074
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS39] <> NULL AND [CGLS39] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS39] = CGLS:RP.Y0:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS39, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0:^CGLS39 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
            */
            assertion = NotMonetary(report.CGLS39, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredContinuityTestFail.Amount[@contextRef='" + report.RPY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS39", Value = report.CGLS39.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404074
    
            #region VR.ATO.CGLS.404075
    
            /*  VR.ATO.CGLS.404075
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS40] <> NULL AND [CGLS40] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS40] = CGLS:RP.Y0-1:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS40, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-1:^CGLS40 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
            */
            assertion = NotMonetary(report.CGLS40, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredContinuityTestFail.Amount[@contextRef='" + report.RPY0Minus1Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS40", Value = report.CGLS40.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404075
    
            #region VR.ATO.CGLS.404076
    
            /*  VR.ATO.CGLS.404076
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS41] <> NULL AND [CGLS41] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS41] = CGLS:RP.Y0-2:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS41, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-2:^CGLS41 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
            */
            assertion = NotMonetary(report.CGLS41, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredContinuityTestFail.Amount[@contextRef='" + report.RPY0Minus2Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS41", Value = report.CGLS41.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404076
    
            #region VR.ATO.CGLS.404077
    
            /*  VR.ATO.CGLS.404077
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS42] <> NULL AND [CGLS42] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS42] = CGLS:RP.Y0-3:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS42, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-3:^CGLS42 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
            */
            assertion = NotMonetary(report.CGLS42, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredContinuityTestFail.Amount[@contextRef='" + report.RPY0Minus3Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404077"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS42", Value = report.CGLS42.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404077
    
            #region VR.ATO.CGLS.404078
    
            /*  VR.ATO.CGLS.404078
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS43] <> NULL AND [CGLS43] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS43] = CGLS:RP.Y0-4P:rvctc3.02.34:Tax.Losses.TransferredContinuityTestFail.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS43, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-4P:^CGLS43 = tns:Tax.Losses.TransferredContinuityTestFail.Amount
            */
            assertion = NotMonetary(report.CGLS43, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredContinuityTestFail.Amount[@contextRef='" + report.RPY0Minus4PId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS43", Value = report.CGLS43.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404078
    
            #region VR.ATO.CGLS.404089
    
            /*  VR.ATO.CGLS.404089
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS69] <> NULL AND [CGLS69] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS69] = CGLS:RP:rvctc3.02.02:Tax.Losses.ControlledForeignCompanyDeducted.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS69, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS69 = tns:Tax.Losses.ControlledForeignCompanyDeducted.Amount
            */
            assertion = NotMonetary(report.CGLS69, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ControlledForeignCompanyDeducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404089"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS69", Value = report.CGLS69.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404089
    
            #region VR.ATO.CGLS.404090
    
            /*  VR.ATO.CGLS.404090
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS70] <> NULL AND [CGLS70] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS70] = CGLS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.ControlledForeignCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS70, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS70 = tns:Tax.Losses.CarriedForward.ControlledForeignCompany.Amount
            */
            assertion = NotMonetary(report.CGLS70, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.ControlledForeignCompany.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404090"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS70", Value = report.CGLS70.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404090
    
            #region VR.ATO.CGLS.404100
    
            /*  VR.ATO.CGLS.404100
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS82] <> NULL AND [CGLS82] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS82] = CGLS:RP:rvctc3.02.04:Tax.Losses.ControlledForeignCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS82, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS82 = tns:Tax.Losses.ControlledForeignCompany.Amount
            */
            assertion = NotMonetary(report.CGLS82, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ControlledForeignCompany.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS82", Value = report.CGLS82.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404100
    
            #region VR.ATO.CGLS.404102
    
            /*  VR.ATO.CGLS.404102
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS85] <> NULL AND [CGLS85] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS85] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS85, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS85 = tns:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount
            */
            assertion = NotMonetary(report.CGLS85, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404102"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS85", Value = report.CGLS85.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404102
    
            #region VR.ATO.CGLS.404103
    
            /*  VR.ATO.CGLS.404103
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS86] <> NULL AND [CGLS86] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS86] = CGLS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS86, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS86 = tns:Tax.Losses.ForgivenDebtNet.Amount
            */
            assertion = NotMonetary(report.CGLS86, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ForgivenDebtNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404103"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS86", Value = report.CGLS86.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404103
    
            #region VR.ATO.CGLS.404104
    
            /*  VR.ATO.CGLS.404104
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS87] <> NULL AND [CGLS87] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS87] = CGLS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS87, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS87 = tns:Tax.Losses.Incurred.Amount
            */
            assertion = NotMonetary(report.CGLS87, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Incurred.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404104"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS87", Value = report.CGLS87.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404104
    
            #region VR.ATO.CGLS.404105
    
            /*  VR.ATO.CGLS.404105
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS88] <> NULL AND [CGLS88] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS88] = CGLS:RP:rvctc3.02.02:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS88, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS88 = tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
            */
            assertion = NotMonetary(report.CGLS88, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404105"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS88", Value = report.CGLS88.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404105
    
            #region VR.ATO.CGLS.404106
    
            /*  VR.ATO.CGLS.404106
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS89] <> NULL AND [CGLS89] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS89] = CGLS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS89, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS89 = tns:Tax.Losses.ExemptIncomeNet.Amount
            */
            assertion = NotMonetary(report.CGLS89, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ExemptIncomeNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404106"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS89", Value = report.CGLS89.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404106
    
            #region VR.ATO.CGLS.404107
    
            /*  VR.ATO.CGLS.404107
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS90] <> NULL AND [CGLS90] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS90] = CGLS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS90, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS90 = tns:Tax.Losses.Forgone.Amount
            */
            assertion = NotMonetary(report.CGLS90, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Forgone.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404107"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS90", Value = report.CGLS90.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404107
    
            #region VR.ATO.CGLS.404108
    
            /*  VR.ATO.CGLS.404108
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS91] <> NULL AND [CGLS91] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS91] = CGLS:RP:rvctc3.02.00:Tax.Losses.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS91, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS91 = tns:Tax.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGLS91, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404108"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS91", Value = report.CGLS91.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404108
    
            #region VR.ATO.CGLS.404109
    
            /*  VR.ATO.CGLS.404109
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS92] <> NULL AND [CGLS92] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS92] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredOut.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS92, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS92 = tns:Tax.Losses.TransferredOut.Amount
            */
            assertion = NotMonetary(report.CGLS92, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredOut.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404109"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS92", Value = report.CGLS92.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404109
    
            #region VR.ATO.CGLS.404110
    
            /*  VR.ATO.CGLS.404110
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS93] <> NULL AND [CGLS93] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS93] = CGLS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS93, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS93 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = NotMonetary(report.CGLS93, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404110"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS93", Value = report.CGLS93.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404110
    
            #region VR.ATO.CGLS.404113
    
            /*  VR.ATO.CGLS.404113
            The total number RP.Y0 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0
            */
            assertion = (Count(report.RPY0Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404113"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404113
    
            #region VR.ATO.CGLS.404114
    
            /*  VR.ATO.CGLS.404114
            The total number RP.Y0-1 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-1)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-1
            */
            assertion = (Count(report.RPY0Minus1Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404114"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404114
    
            #region VR.ATO.CGLS.404115
    
            /*  VR.ATO.CGLS.404115
            The total number RP.Y0-2 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-2)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-2
            */
            assertion = (Count(report.RPY0Minus2Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404115"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404115
    
            #region VR.ATO.CGLS.404116
    
            /*  VR.ATO.CGLS.404116
            The total number RP.Y0-3 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-3)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-3
            */
            assertion = (Count(report.RPY0Minus3Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404116"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404116
    
            #region VR.ATO.CGLS.404117
    
            /*  VR.ATO.CGLS.404117
            The total number RP.Y0-4P supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-4P)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-4P
            */
            assertion = (Count(report.RPY0Minus4PCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4PLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404117"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404117
    
            #region VR.ATO.CGLS.404123
    
            /*  VR.ATO.CGLS.404123
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS94] <> NULL AND [CGLS94] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS94] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS94, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS94 = tns:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount
            */
            assertion = NotMonetary(report.CGLS94, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404123"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS94", Value = report.CGLS94.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404123
    
            #region VR.ATO.CGLS.404124
    
            /*  VR.ATO.CGLS.404124
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS83] <> NULL AND [CGLS83] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS83] = CGLS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS83, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS83 = tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            */
            assertion = NotMonetary(report.CGLS83, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404124"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS83", Value = report.CGLS83.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404124
    
            #region VR.ATO.CGLS.404130
    
            /*  VR.ATO.CGLS.404130
            The question "Has the head company cancelled the transfer of a loss?" must be answered
    
            Legacy Rule Format:
            IF [CGLS36]= NULL                                                                                                                                                                                                                           
            RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS36] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator

            Technical Business Rule Format:
            ^CGLS36= NULL
    
            Data Elements:
    
            RP:^CGLS36 = tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator
            */
            assertion = (report.CGLS36 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404130"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS36", Value = GetValueOrEmpty(report.CGLS36) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404130
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.COTF
    
            ^TFN = RP
            */
            assertion = ((report.RPCOTFExists == true) && (report.RPCOTFIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.COTP
    
            ^TFN = RP
            */
            assertion = ((report.RPCOTPExists == true) && (report.RPCOTPIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.COTT
    
            ^TFN = RP
            */
            assertion = ((report.RPCOTTExists == true) && (report.RPCOTTIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.GRP
    
            ^TFN = RP
            */
            assertion = ((report.RPGRPExists == true) && (report.RPGRPIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP.GRP(Instant)
    
            ^TFN = RP
            */
            assertion = ((report.RPGRPInstantExists == true) && (report.RPGRPInstantIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.TOT
    
            ^TFN = RP
            */
            assertion = ((report.RPTOTExists == true) && (report.RPTOTIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP.TOT(Instant)
    
            ^TFN = RP
            */
            assertion = ((report.RPTOTInstantExists == true) && (report.RPTOTInstantIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.TRFD
    
            ^TFN = RP
            */
            assertion = ((report.RPTRFDExists == true) && (report.RPTRFDIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP.TRFD(Instant)
    
            ^TFN = RP
            */
            assertion = ((report.RPTRFDInstantExists == true) && (report.RPTRFDInstantIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.Y0
    
            ^TFN = RP
            */
            assertion = ((report.RPY0Exists == true) && (report.RPY0IdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.Y0-1
    
            ^TFN = RP
            */
            assertion = ((report.RPY0Minus1Exists == true) && (report.RPY0Minus1IdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.Y0-2
    
            ^TFN = RP
            */
            assertion = ((report.RPY0Minus2Exists == true) && (report.RPY0Minus2IdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.Y0-3
    
            ^TFN = RP
            */
            assertion = ((report.RPY0Minus3Exists == true) && (report.RPY0Minus3IdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404134
    
            /*  VR.ATO.CGLS.404134
            TFN does not match the reporting party
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.TFN <> RP:entity.identifier.TFN)
                      RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^OtherTFN = RP:RP.Y0-4P
    
            ^TFN = RP
            */
            assertion = ((report.RPY0Minus4PExists == true) && (report.RPY0Minus4PIdentifierTFN != report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404134"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404134
    
            #region VR.ATO.CGLS.404139
    
            /*  VR.ATO.CGLS.404139
            Transferor context TFN cannot be duplicated. Unique TFN is required
    
            Legacy Rule Format:
            IF TRFR:entity.identifier = ANY OTHER OCCURRENCE(TRFR:entity.identifier.TFN)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^TFN)
    
            Data Elements:
    
            ^TFN = TRFR
            */
            assertion = HasDuplicateValues(report.TRFRContextCollection == null ? null : report.TRFRContextCollection.Select(f => f.IdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404053", Severity = ProcessMessageSeverity.Error,
                    Description = @"TRFR context TFN duplicated. Unique TFN required",
                    LongDescription = @"TRFR context TFN duplicated. Unique TFN required",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.TRFRContextCollection[DuplicateValueIndex(report.TRFRContextCollection.Select(f => f.IdentifierTFN))].Id + "']" + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404139"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404139
    
            #region VR.ATO.CGLS.404140
    
            /*  VR.ATO.CGLS.404140
            Joining Entity context TFN cannot be duplicated. Unique TFN is required
    
            Legacy Rule Format:
            IF JE:entity.identifier = ANY OTHER OCCURRENCE(JE:entity.identifier.TFN)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^TFN)
    
            Data Elements:
    
            ^TFN = JE
            */
            assertion = HasDuplicateValues(report.JEContextCollection == null ? null : report.JEContextCollection.Select(f => f.IdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404054", Severity = ProcessMessageSeverity.Error,
                    Description = @"JE context TFN duplicated. Unique TFN required",
                    LongDescription = @"JE context TFN duplicated. Unique TFN required",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.JEContextCollection[DuplicateValueIndex(report.JEContextCollection.Select(f => f.IdentifierTFN))].Id + "']" + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404140"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404140
    
            #region VR.ATO.CGLS.404145
    
            /*  VR.ATO.CGLS.404145
            In the tax losses reconciliation the sum of the added amounts must be equal to or greater than the sum of the subtracted amounts
    
            Legacy Rule Format:
            IF ([CGLS94] <> NULL OR [CGLS86] <> NULL OR [CGLS89] <> NULL OR [CGLS90] <> NULL OR [CGLS91] <> NULL OR [CGLS92] <> NULL OR  [CGLS104] <> NULL) AND ([CGLS94]+[CGLS86]+[CGLS89]+[CGLS90]+[CGLS91]+[CGLS92]> [CGLS104]+ [CGLS83]+[CGLS85]+[CGLS87]+[CGLS88])
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS83] = CGLS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            [CGLS85] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount
            [CGLS86] = CGLS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount
            [CGLS87] = CGLS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount
            [CGLS88] = CGLS:RP:rvctc3.02.02:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
            [CGLS89] = CGLS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount
            [CGLS90] = CGLS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount
            [CGLS91] = CGLS:RP:rvctc3.02.00:Tax.Losses.Total.Amount
            [CGLS92] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredOut.Amount
            [CGLS94] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount
            [CGLS104] = CGLS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount

            Technical Business Rule Format:
            (^CGLS94 <> NULL OR ^CGLS86 <> NULL OR ^CGLS89 <> NULL OR ^CGLS90 <> NULL OR ^CGLS91 <> NULL OR ^CGLS92 <> NULL OR  ^CGLS104 <> NULL) AND (^CGLS94+^CGLS86+^CGLS89+^CGLS90+^CGLS91+^CGLS92> ^CGLS104+ ^CGLS83+^CGLS85+^CGLS87+^CGLS88)
    
            Data Elements:
    
            RP:^CGLS83 = tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
    
            RP:^CGLS85 = tns:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount
    
            RP:^CGLS86 = tns:Tax.Losses.ForgivenDebtNet.Amount
    
            RP:^CGLS87 = tns:Tax.Losses.Incurred.Amount
    
            RP:^CGLS88 = tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
    
            RP:^CGLS89 = tns:Tax.Losses.ExemptIncomeNet.Amount
    
            RP:^CGLS90 = tns:Tax.Losses.Forgone.Amount
    
            RP:^CGLS91 = tns:Tax.Losses.Total.Amount
    
            RP:^CGLS92 = tns:Tax.Losses.TransferredOut.Amount
    
            RP:^CGLS94 = tns:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount
    
            RP:^CGLS104 = tns:Tax.Losses.Uplift.Amount
            */
            assertion = ((report.CGLS94 != null || report.CGLS86 != null || report.CGLS89 != null || report.CGLS90 != null || report.CGLS91 != null || report.CGLS92 != null || report.CGLS104 != null) && report.CGLS94.GetValueOrDefault() + report.CGLS86.GetValueOrDefault() + report.CGLS89.GetValueOrDefault() + report.CGLS90.GetValueOrDefault() + report.CGLS91.GetValueOrDefault() + report.CGLS92.GetValueOrDefault() > report.CGLS104.GetValueOrDefault() + report.CGLS83.GetValueOrDefault() + report.CGLS85.GetValueOrDefault() + report.CGLS87.GetValueOrDefault() + report.CGLS88.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404145", Severity = ProcessMessageSeverity.Error,
                    Description = @"Amounts in the tax losses reconciliation statement for consolidated groups are incorrect",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404145"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS94", Value = report.CGLS94.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS86", Value = report.CGLS86.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS89", Value = report.CGLS89.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS90", Value = report.CGLS90.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS91", Value = report.CGLS91.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS92", Value = report.CGLS92.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS104", Value = report.CGLS104.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS83", Value = report.CGLS83.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS85", Value = report.CGLS85.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS87", Value = report.CGLS87.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS88", Value = report.CGLS88.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404145
    
            #region VR.ATO.CGLS.404146
    
            /*  VR.ATO.CGLS.404146
            The tax losses reconciliation for consolidated group losses must equal the Total tax losses carried forward to later years.
    
            Legacy Rule Format:
            IF ([CGLS83] <> NULL OR [CGLS104] <> NULL OR [CGLS85] <> NULL OR [CGLS94] <> NULL OR [CGLS86] <> NULL OR [CGLS87] <> NULL OR [CGLS88] <> NULL OR [CGLS89] <> NULL OR [CGLS90] <> NULL OR [CGLS91] <> NULL OR [CGLS92] <> NULL OR [CGLS93] <> NULL) AND ([CGLS93]<> [CGLS83]+[CGLS104]+[CGLS85]-[CGLS94]-[CGLS86]+[CGLS87]+[CGLS88]-[CGLS89]-[CGLS90]-[CGLS91]-[CGLS92]) 
            RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS83] = CGLS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            [CGLS85] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount
            [CGLS86] = CGLS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount
            [CGLS87] = CGLS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount
            [CGLS88] = CGLS:RP:rvctc3.02.02:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
            [CGLS89] = CGLS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount
            [CGLS90] = CGLS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount
            [CGLS91] = CGLS:RP:rvctc3.02.00:Tax.Losses.Total.Amount
            [CGLS92] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredOut.Amount
            [CGLS93] = CGLS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            [CGLS94] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount
            [CGLS104] = CGLS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount

            Technical Business Rule Format:
            (^CGLS83 <> NULL OR ^CGLS104 <> NULL OR ^CGLS85 <> NULL OR ^CGLS94 <> NULL OR ^CGLS86 <> NULL OR ^CGLS87 <> NULL OR ^CGLS88 <> NULL OR ^CGLS89 <> NULL OR ^CGLS90 <> NULL OR ^CGLS91 <> NULL OR ^CGLS92 <> NULL OR ^CGLS93 <> NULL) AND (^CGLS93 <> ^CGLS83+^CGLS104+^CGLS85-^CGLS94-^CGLS86+^CGLS87+^CGLS88-^CGLS89-^CGLS90-^CGLS91-^CGLS92)
    
            Data Elements:
    
            RP:^CGLS83 = tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
    
            RP:^CGLS85 = tns:Tax.Losses.TransferredFromJoiningEntitiesUnderSubdivision707A.Amount
    
            RP:^CGLS86 = tns:Tax.Losses.ForgivenDebtNet.Amount
    
            RP:^CGLS87 = tns:Tax.Losses.Incurred.Amount
    
            RP:^CGLS88 = tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
    
            RP:^CGLS89 = tns:Tax.Losses.ExemptIncomeNet.Amount
    
            RP:^CGLS90 = tns:Tax.Losses.Forgone.Amount
    
            RP:^CGLS91 = tns:Tax.Losses.Total.Amount
    
            RP:^CGLS92 = tns:Tax.Losses.TransferredOut.Amount
    
            RP:^CGLS93 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            RP:^CGLS94 = tns:Tax.Losses.TransferredWithNilAvailableFractionApplied.Amount
    
            RP:^CGLS104 = tns:Tax.Losses.Uplift.Amount
            */
            assertion = ((report.CGLS83 != null || report.CGLS104 != null || report.CGLS85 != null || report.CGLS94 != null || report.CGLS86 != null || report.CGLS87 != null || report.CGLS88 != null || report.CGLS89 != null || report.CGLS90 != null || report.CGLS91 != null || report.CGLS92 != null || report.CGLS93 != null) && report.CGLS93.GetValueOrDefault() != report.CGLS83.GetValueOrDefault() + report.CGLS104.GetValueOrDefault() + report.CGLS85.GetValueOrDefault() - report.CGLS94.GetValueOrDefault() - report.CGLS86.GetValueOrDefault() + report.CGLS87.GetValueOrDefault() + report.CGLS88.GetValueOrDefault() - report.CGLS89.GetValueOrDefault() - report.CGLS90.GetValueOrDefault() - report.CGLS91.GetValueOrDefault() - report.CGLS92.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404146", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total tax losses carried forward to later income years is incorrect",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404146"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS83", Value = report.CGLS83.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS104", Value = report.CGLS104.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS85", Value = report.CGLS85.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS94", Value = report.CGLS94.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS86", Value = report.CGLS86.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS87", Value = report.CGLS87.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS88", Value = report.CGLS88.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS89", Value = report.CGLS89.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS90", Value = report.CGLS90.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS91", Value = report.CGLS91.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS92", Value = report.CGLS92.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS93", Value = report.CGLS93.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404146
    
            #region VR.ATO.CGLS.404200
    
            /*  VR.ATO.CGLS.404200
            The total number supplied cannot exceed 6
    
            Legacy Rule Format:
            IF COUNT([CGLS108]) > 6
            RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction

            Technical Business Rule Format:
            Count(^CGLS108) > 6
    
            Data Elements:
    
            TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
            */
            assertion = (Count(report.TRFRContextCollection == null ? null : report.TRFRContextCollection.Select(f => f.CGLS108)) > 6);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404200", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transferor TFNs supplied for transferred tax losses deducted cannot exceed 6",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Fraction[@contextRef='" + report.TRFRContextCollection.Last().Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404200"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404200
    
            #region VR.ATO.CGLS.404201
    
            /*  VR.ATO.CGLS.404201
            The total number supplied cannot exceed 6
    
            Legacy Rule Format:
            IF COUNT([CGLS113]) > 6
            RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction

            Technical Business Rule Format:
            Count(^CGLS113) > 6
    
            Data Elements:
    
            TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
            */
            assertion = (Count(report.TRFRContextCollection == null ? null : report.TRFRContextCollection.Select(f => f.CGLS113)) > 6);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404201", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transferor TFNs supplied for transferred capital losses applied cannot exceed 6",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredApplied.Fraction[@contextRef='" + report.TRFRContextCollection.Last().Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404201"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404201
    
            #region VR.ATO.CGLS.404202
    
            /*  VR.ATO.CGLS.404202
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS115] <> NULL AND [CGLS115] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS115] = CGLS:RP:rvctc3.02.34:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS115, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS115 = tns:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            */
            assertion = NotMonetary(report.CGLS115, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404202"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS115", Value = report.CGLS115.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404202
    
            #region VR.ATO.CGLS.404203
    
            /*  VR.ATO.CGLS.404203
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS116] <> NULL AND [CGLS116] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS116] = CGLS:RP:rvctc3.02.34:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS116, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS116 = tns:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            */
            assertion = NotMonetary(report.CGLS116, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS116", Value = report.CGLS116.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404203
    
            #region VR.ATO.CGLS.404204
    
            /*  VR.ATO.CGLS.404204
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS117] <> NULL AND [CGLS117] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS117] = CGLS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS117, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS117 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = NotMonetary(report.CGLS117, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404204"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS117", Value = report.CGLS117.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404204
    
            #region VR.ATO.CGLS.404205
    
            /*  VR.ATO.CGLS.404205
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS118] <> NULL AND [CGLS118] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS118] = CGLS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS118, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS118 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = NotMonetary(report.CGLS118, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS118", Value = report.CGLS118.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404205
    
            #region VR.ATO.CGLS.404206
    
            /*  VR.ATO.CGLS.404206
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS104] <> NULL AND [CGLS104] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS104] = CGLS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS104, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS104 = tns:Tax.Losses.Uplift.Amount
            */
            assertion = NotMonetary(report.CGLS104, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Uplift.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404206"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS104", Value = report.CGLS104.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404206
    
            #region VR.ATO.CGLS.404207
    
            /*  VR.ATO.CGLS.404207
            The sum of group tax losses deducted and transferred tax losses decucted must equal total tax losses deducted.
    
            Legacy Rule Format:
            IF ([CGLS105] + [CGLS106]) <> [CGLS107]
                     RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS105] = CGLS:RP.GRP:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [CGLS106] = CGLS:RP.TRFD:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [CGLS107] = CGLS:RP.TOT:rvctc3.02.15:Tax.Losses.Deducted.Amount

            Technical Business Rule Format:
            (^CGLS105 + ^CGLS106) <> ^CGLS107
    
            Data Elements:
    
            RP.GRP:^CGLS105 = tns:Tax.Losses.Deducted.Amount
    
            RP.TRFD:^CGLS106 = tns:Tax.Losses.Deducted.Amount
    
            RP.TOT:^CGLS107 = tns:Tax.Losses.Deducted.Amount
            */
            assertion = (report.CGLS105.GetValueOrDefault() + report.CGLS106.GetValueOrDefault() != report.CGLS107.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404207", Severity = ProcessMessageSeverity.Error,
                    Description = @"Sum of group and transferred tax losses deducted must equal tax losses deducted total",
                    LongDescription = @"The sum of (Tax losses deducted - Group) and (Tax losses deducted - Transferred) must equal (Tax losses deducted - Total)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPGRPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS105", Value = report.CGLS105.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS106", Value = report.CGLS106.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS107", Value = report.CGLS107.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404207
    
            #region VR.ATO.CGLS.404208
    
            /*  VR.ATO.CGLS.404208
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF rvctc.xx.xx:Tax.Losses.Deducted.Amount <> NULL AND rvctc.xx.xx:Tax.Losses.Deducted.Amount <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGLS105, 'U', 11, 0)
    
            Data Elements:
    
            RP.GRP:^CGLS105 = tns:Tax.Losses.Deducted.Amount
            */
            assertion = NotMonetary(report.CGLS105, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPGRPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS105", Value = report.CGLS105.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404208
    
            #region VR.ATO.CGLS.404208
    
            /*  VR.ATO.CGLS.404208
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF rvctc.xx.xx:Tax.Losses.Deducted.Amount <> NULL AND rvctc.xx.xx:Tax.Losses.Deducted.Amount <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGLS107, 'U', 11, 0)
    
            Data Elements:
    
            RP.TOT:^CGLS107 = tns:Tax.Losses.Deducted.Amount
            */
            assertion = NotMonetary(report.CGLS107, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS107", Value = report.CGLS107.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404208
    
            #region VR.ATO.CGLS.404208
    
            /*  VR.ATO.CGLS.404208
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF rvctc.xx.xx:Tax.Losses.Deducted.Amount <> NULL AND rvctc.xx.xx:Tax.Losses.Deducted.Amount <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGLS106, 'U', 11, 0)
    
            Data Elements:
    
            RP.TRFD:^CGLS106 = tns:Tax.Losses.Deducted.Amount
            */
            assertion = NotMonetary(report.CGLS106, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS106", Value = report.CGLS106.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404208
    
            #region VR.ATO.CGLS.404209
    
            /*  VR.ATO.CGLS.404209
            The sum of  group net capital losses applied and transferred net capital losses applied must equal total net capital losses applied.
    
            Legacy Rule Format:
            IF ([CGLS110] + [CGLS111]) <> [CGLS112]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS110] = CGLS:RP.GRP:rvctc3.02.15:Capital.Losses.AppliedNet.Amount
            [CGLS111] = CGLS:RP.TRFD:rvctc3.02.15:Capital.Losses.AppliedNet.Amount
            [CGLS112] = CGLS:RP.TOT:rvctc3.02.15:Capital.Losses.AppliedNet.Amount

            Technical Business Rule Format:
            (^CGLS110 + ^CGLS111) <> ^CGLS112
    
            Data Elements:
    
            RP.GRP:^CGLS110 = tns:Capital.Losses.AppliedNet.Amount
    
            RP.TRFD:^CGLS111 = tns:Capital.Losses.AppliedNet.Amount
    
            RP.TOT:^CGLS112 = tns:Capital.Losses.AppliedNet.Amount
            */
            assertion = (report.CGLS110.GetValueOrDefault() + report.CGLS111.GetValueOrDefault() != report.CGLS112.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404209", Severity = ProcessMessageSeverity.Error,
                    Description = @"The sum of net capital losses applied fields must equal the total of net capital losses applied",
                    LongDescription = @"The sum of (Net capital losses applied - Group) and (Net capital losses applied - Transferred) must equal (Net capital losses applied - Total)",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPGRPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS110", Value = report.CGLS110.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS111", Value = report.CGLS111.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS112", Value = report.CGLS112.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404209
    
            #region VR.ATO.CGLS.404210
    
            /*  VR.ATO.CGLS.404210
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF rvctc.xx.xx:Capital.Losses.AppliedNet.Amount <> NULL AND rvctc3.02.14:Capital.Losses.AppliedNet.Amount <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGLS110, 'U', 11, 0)
    
            Data Elements:
    
            RP.GRP:^CGLS110 = tns:Capital.Losses.AppliedNet.Amount
            */
            assertion = NotMonetary(report.CGLS110, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPGRPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS110", Value = report.CGLS110.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404210
    
            #region VR.ATO.CGLS.404210
    
            /*  VR.ATO.CGLS.404210
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF rvctc.xx.xx:Capital.Losses.AppliedNet.Amount <> NULL AND rvctc3.02.14:Capital.Losses.AppliedNet.Amount <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGLS112, 'U', 11, 0)
    
            Data Elements:
    
            RP.TOT:^CGLS112 = tns:Capital.Losses.AppliedNet.Amount
            */
            assertion = NotMonetary(report.CGLS112, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS112", Value = report.CGLS112.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404210
    
            #region VR.ATO.CGLS.404210
    
            /*  VR.ATO.CGLS.404210
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF rvctc.xx.xx:Capital.Losses.AppliedNet.Amount <> NULL AND rvctc3.02.14:Capital.Losses.AppliedNet.Amount <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGLS111, 'U', 11, 0)
    
            Data Elements:
    
            RP.TRFD:^CGLS111 = tns:Capital.Losses.AppliedNet.Amount
            */
            assertion = NotMonetary(report.CGLS111, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS111", Value = report.CGLS111.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404210
    
            #region VR.ATO.CGLS.404211
    
            /*  VR.ATO.CGLS.404211
            Amount of tax losses deducted after consolidation cannot exceed total amount of tax losses deducted.
    
            Legacy Rule Format:
            IF [CGLS115] <> NULL AND [CGLS115] > [CGLS107]
                     RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS107] = CGLS:RP.TOT:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [CGLS115] = CGLS:RP:rvctc3.02.34:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount

            Technical Business Rule Format:
            ^CGLS115 <> NULL AND ^CGLS115 > ^CGLS107
    
            Data Elements:
    
            RP.TOT:^CGLS107 = tns:Tax.Losses.Deducted.Amount
    
            RP:^CGLS115 = tns:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            */
            assertion = (report.CGLS115 != null && report.CGLS115.GetValueOrDefault() > report.CGLS107.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404234", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses deducted after consolidation applied incorrect.",
                    LongDescription = @"Amount of tax losses deducted after consolidation which failed the continuity of ownership test but satisfied the business continuity test cannot exceed the Total Tax losses deducted.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS115", Value = report.CGLS115.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS107", Value = report.CGLS107.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404211
    
            #region VR.ATO.CGLS.404213
    
            /*  VR.ATO.CGLS.404213
            Amount of net capital losses applied after consolidation cannot exceed total amount of net capital losses applied.
    
            Legacy Rule Format:
            IF [CGLS116] <> NULL AND [CGLS116] > [CGLS112]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS112] = CGLS:RP.TOT:rvctc3.02.15:Capital.Losses.AppliedNet.Amount
            [CGLS116] = CGLS:RP:rvctc3.02.34:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount

            Technical Business Rule Format:
            ^CGLS116 <> NULL AND ^CGLS116 > ^CGLS112
    
            Data Elements:
    
            RP.TOT:^CGLS112 = tns:Capital.Losses.AppliedNet.Amount
    
            RP:^CGLS116 = tns:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            */
            assertion = (report.CGLS116 != null && report.CGLS116.GetValueOrDefault() > report.CGLS112.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404233", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses applied after consolidation incorrect.",
                    LongDescription = @"Amount of net capital losses applied after consolidation (which the continuity of ownership test failed but satisfied the business continuity test) cannot exceed total amount of net capital losses applied.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPTOTId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404213"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS116", Value = report.CGLS116.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS112", Value = report.CGLS112.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404213
    
            #region VR.ATO.CGLS.404214
    
            /*  VR.ATO.CGLS.404214
            If an amount has been entered at transferred tax losses deducted then the details of a least one transferred tax loss deducted including transferor TFN, available fraction and amount (greater than zero) must be completed.
    
            Legacy Rule Format:
            IF [CGLS106] > 0 AND COUNT([CGLS108]) = 0
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS106] = CGLS:RP.TRFD:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction

            Technical Business Rule Format:
            ^CGLS106 > 0 AND Count(^CGLS108) = 0
    
            Data Elements:
    
            RP.TRFD:^CGLS106 = tns:Tax.Losses.Deducted.Amount
    
            TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
            */
            assertion = (report.CGLS106.GetValueOrDefault() > 0 && Count(report.TRFRContextCollection == null ? null : report.TRFRContextCollection.Select(f => f.CGLS108)) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404214", Severity = ProcessMessageSeverity.Error,
                    Description = @"(Tax losses deducted - Transferred) greater than zero. Transferor details required.",
                    LongDescription = @"The sum of (Tax losses deducted - Group) and (Tax losses deducted - Transferred) must equal (Tax losses deducted - Total)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404214"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS106", Value = report.CGLS106.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404214
    
            #region VR.ATO.CGLS.404215
    
            /*  VR.ATO.CGLS.404215
            The individual details of transferred tax losses deducted including transferor TFN, available fraction and amount are only required to be completed if an amount is entered against transferred tax losses deducted.
    
            Legacy Rule Format:
            IF ANY OCCURRENCE OF [CGLS108] <> NULL AND ([CGLS106] = NULL OR [CGLS106] = 0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS106] = CGLS:RP.TRFD:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction

            Technical Business Rule Format:
            AnyOccurrence(^CGLS108, ^CGLS108 <> NULL) AND (^CGLS106 = NULL OR ^CGLS106 = 0)
    
            Data Elements:
    
            RP.TRFD:^CGLS106 = tns:Tax.Losses.Deducted.Amount
    
            TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
            */
            assertion = ((report.TRFRContextCollection == null ? false : report.TRFRContextCollection.Any(CGLS108Repeat => CGLS108Repeat.CGLS108 != null)) && (report.CGLS106 == null || report.CGLS106 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404215", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transferor details present. Tax losses deducted - transferred is absent or equal to zero",
                    LongDescription = @"Details of transferor TFN, available fraction and amount are present however an amount for (Tax losses deducted - Transferred) is absent or equal to zero",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404215"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS108", Value = "CGLS108" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS106", Value = report.CGLS106.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404215
    
            #region VR.ATO.CGLS.404216
    
            /*  VR.ATO.CGLS.404216
            The sum of all transferred tax losses deducted, listed against individual transferror TFN's, cannot exceed  transferred tax losses deducted amount
    
            Legacy Rule Format:
            IF SUM([CGLS109]) > [CGLS106]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS106] = CGLS:RP.TRFD:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [CGLS109] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Amount

            Technical Business Rule Format:
            Sum(^CGLS109) > ^CGLS106
    
            Data Elements:
    
            RP.TRFD:^CGLS106 = tns:Tax.Losses.Deducted.Amount
    
            TRFR:^CGLS109 = tns:Tax.Losses.TransferredDeducted.Amount
            */
            assertion = ((report.TRFRContextCollection == null ? 0 : report.TRFRContextCollection.Sum(f => f.CGLS109.GetValueOrDefault())) > report.CGLS106.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404216", Severity = ProcessMessageSeverity.Error,
                    Description = @"The sum of transferred tax losses deducted - amounts cannot exceed tax losses deducted - transferred",
                    LongDescription = @"The sum of (Transferred tax losses deducted - Amounts) cannot exceed (Tax losses deducted - Transferred)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404216"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS109", Value = "CGLS109" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS106", Value = report.CGLS106.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404216
    
            #region VR.ATO.CGLS.404217
    
            /*  VR.ATO.CGLS.404217
            If an amount has been entered at transferred net capital losses applied then the details of a least one transferred net capital loss applied, including transferor TFN, available fraction and amount (greater than zero) must be completed.
    
            Legacy Rule Format:
            IF [CGLS111] > 0 AND COUNT([CGLS113]) = 0
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS111] = CGLS:RP.TRFD:rvctc3.02.15:Capital.Losses.AppliedNet.Amount
            [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction

            Technical Business Rule Format:
            ^CGLS111 > 0 AND Count(^CGLS113) = 0
    
            Data Elements:
    
            RP.TRFD:^CGLS111 = tns:Capital.Losses.AppliedNet.Amount
    
            TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
            */
            assertion = (report.CGLS111.GetValueOrDefault() > 0 && Count(report.TRFRContextCollection == null ? null : report.TRFRContextCollection.Select(f => f.CGLS113)) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404217", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses applied - transferred completed. Transferor details absent or incomplete",
                    LongDescription = @"(Net capital losses applied - Transferred) completed and (Transferred net capital losses applied - Transferor TFN), (Transferred net capital losses applied - Available fraction) and (Transferred net capital losses applied - Amount) absent or incomplete",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404217"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS111", Value = report.CGLS111.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404217
    
            #region VR.ATO.CGLS.404218
    
            /*  VR.ATO.CGLS.404218
            The individual details of transferred net capital losses applied, including transferor TFN, available fraction and amount are only required to be completed if an amount is entered against transferred net capital losses applied.
    
            Legacy Rule Format:
            IF ANY OCCURRENCE OF [CGLS113] <> NULL AND ([CGLS111] = NULL OR [CGLS111] = 0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS111] = CGLS:RP.TRFD:rvctc3.02.15:Capital.Losses.AppliedNet.Amount
            [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction

            Technical Business Rule Format:
            AnyOccurrence(^CGLS113, ^CGLS113 <> NULL) AND (^CGLS111 = NULL OR ^CGLS111 = 0)
    
            Data Elements:
    
            RP.TRFD:^CGLS111 = tns:Capital.Losses.AppliedNet.Amount
    
            TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
            */
            assertion = ((report.TRFRContextCollection == null ? false : report.TRFRContextCollection.Any(CGLS113Repeat => CGLS113Repeat.CGLS113 != null)) && (report.CGLS111 == null || report.CGLS111 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404218", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transferor details present. Net capital losses applied - transferred absent or equal to zero",
                    LongDescription = @"(Transferred net capital losses applied - Transferor TFN), (Transferred net capital losses applied - Available fraction) and (Transferred net capital losses applied - Amount) present and (Net capital losses applied - Transferred) absent or equal to zero",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404218"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS113", Value = "CGLS113" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS111", Value = report.CGLS111.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404218
    
            #region VR.ATO.CGLS.404219
    
            /*  VR.ATO.CGLS.404219
            The sum of all transferred net capital losses applied, listed against individual transferror TFN's, cannot exceed  transferred net capital losses applied amount.
    
            Legacy Rule Format:
            IF SUM([CGLS114]) > [CGLS111]
             RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS111] = CGLS:RP.TRFD:rvctc3.02.15:Capital.Losses.AppliedNet.Amount
            [CGLS114] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Amount

            Technical Business Rule Format:
            Sum(^CGLS114) > ^CGLS111
    
            Data Elements:
    
            RP.TRFD:^CGLS111 = tns:Capital.Losses.AppliedNet.Amount
    
            TRFR:^CGLS114 = tns:Capital.Losses.TransferredApplied.Amount
            */
            assertion = ((report.TRFRContextCollection == null ? 0 : report.TRFRContextCollection.Sum(f => f.CGLS114.GetValueOrDefault())) > report.CGLS111.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGLS.404219", Severity = ProcessMessageSeverity.Error,
                    Description = @"The sum of transferred net capital losses applied cannot exceed net capital losses applied",
                    LongDescription = @"The sum of all transferred net capital losses applied cannot exceed (Net capital losses applied - Transferred)",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.AppliedNet.Amount[@contextRef='" + report.RPTRFDId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404219"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS114", Value = "CGLS114" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS111", Value = report.CGLS111.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404219
    
            #region VR.ATO.CGLS.404270
    
            /*  VR.ATO.CGLS.404270
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS119] <> NULL AND [CGLS119] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS119] = CGLS:RP:rvctc3.02.22:Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS119, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS119 = tns:Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount
            */
            assertion = NotMonetary(report.CGLS119, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404270"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS119", Value = report.CGLS119.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404270
    
            #region VR.ATO.CGLS.404271
    
            /*  VR.ATO.CGLS.404271
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [CGLS120] <> NULL AND [CGLS120] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [CGLS120] = CGLS:RP:rvctc3.02.22:Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount

            Technical Business Rule Format:
            NotMonetary(^CGLS120, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGLS120 = tns:Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount
            */
            assertion = NotMonetary(report.CGLS120, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404271"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGLS120", Value = report.CGLS120.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404271
    
            #region VR.ATO.CGLS.404300
    
            /*  VR.ATO.CGLS.404300
            The total number RP.GRP(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.GRP(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.GRP(Instant)
            */
            assertion = (Count(report.RPGRPInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGRPInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404300"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404300
    
            #region VR.ATO.CGLS.404301
    
            /*  VR.ATO.CGLS.404301
            The total number RP.TFRD(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TRFD(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.TRFD(Instant)
            */
            assertion = (Count(report.RPTRFDInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTRFDInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404301"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404301
    
            #region VR.ATO.CGLS.404302
    
            /*  VR.ATO.CGLS.404302
            The total number RP.TOT(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TOT(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.TOT(Instant)
            */
            assertion = (Count(report.RPTOTInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOTInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404302"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGLS.404302
    
            #region VR.ATO.GEN.000208
    
            /*  VR.ATO.GEN.000208
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000208"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000208
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.COTF
    
            ^Context = RP:RP.COTF
            */
            assertion = (report.RPCOTFExists != false && report.RPCOTFIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCOTFOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.COTP
    
            ^Context = RP:RP.COTP
            */
            assertion = (report.RPCOTPExists != false && report.RPCOTPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCOTPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.COTT
    
            ^Context = RP:RP.COTT
            */
            assertion = (report.RPCOTTExists != false && report.RPCOTTIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCOTTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.GRP
    
            ^Context = RP:RP.GRP
            */
            assertion = (report.RPGRPExists != false && report.RPGRPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGRPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.GRP(Instant)
    
            ^Context = RP.GRP(Instant)
            */
            assertion = (report.RPGRPInstantExists != false && report.RPGRPInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGRPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TOT
    
            ^Context = RP:RP.TOT
            */
            assertion = (report.RPTOTExists != false && report.RPTOTIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.TOT(Instant)
    
            ^Context = RP.TOT(Instant)
            */
            assertion = (report.RPTOTInstantExists != false && report.RPTOTInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOTInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TRFD
    
            ^Context = RP:RP.TRFD
            */
            assertion = (report.RPTRFDExists != false && report.RPTRFDIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTRFDOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.TRFD(Instant)
    
            ^Context = RP.TRFD(Instant)
            */
            assertion = (report.RPTRFDInstantExists != false && report.RPTRFDInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTRFDInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0
    
            ^Context = RP:RP.Y0
            */
            assertion = (report.RPY0Exists != false && report.RPY0IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-1
    
            ^Context = RP:RP.Y0-1
            */
            assertion = (report.RPY0Minus1Exists != false && report.RPY0Minus1IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-2
    
            ^Context = RP:RP.Y0-2
            */
            assertion = (report.RPY0Minus2Exists != false && report.RPY0Minus2IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-3
    
            ^Context = RP:RP.Y0-3
            */
            assertion = (report.RPY0Minus3Exists != false && report.RPY0Minus3IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-4P
    
            ^Context = RP:RP.Y0-4P
            */
            assertion = (report.RPY0Minus4PExists != false && report.RPY0Minus4PIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4POccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021

            #region Repeating JEContext
            if (report.JEContextCollection != null)
            {
                int itemIndexContext1 = 0;
                foreach (CGLS2017.JEContext jeContext in report.JEContextCollection)
                {
                    itemIndexContext1++;
            
                    #region VR.ATO.CGLS.404023
            
                    /*  VR.ATO.CGLS.404023
                    If the question "Has the head company cancelled the transfer of a loss?" has been answered with "no" then the details of the joining entity TFN and amount must not be completed.
    
                    Legacy Rule Format:
                    IF [CGLS36] = FALSE AND
                     (CGLS:JE:entity.identifier.TFN <> NULLORBLANK OR [CGLS38] <> NULL)
                            RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS36] = CGLS:RP:rvctc3.02.02:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator
                    [CGLS38] = CGLS:JE:rvctc3.02.02:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount
        
                    Technical Business Rule Format:
                    ^CGLS36 = FALSE AND (^TFN <> BLANK OR ^CGLS38 <> NULL)
            
                    Data Elements:
            
                    RP:^CGLS36 = tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator
            
                    JE:^CGLS38 = tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount
            
                    JE:^TFN = JE
                    */
                    assertion = (report.CGLS36 == false && (string.IsNullOrWhiteSpace(jeContext.IdentifierTFN) != true || jeContext.CGLS38 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404023", Severity = ProcessMessageSeverity.Error,
                            Description = @"Head company has not cancelled the transfer of a loss. Details of cancellation completed",
                            LongDescription = @"The head company has not cancelled the transfer of a loss. (Details of cancellation of transfer of losses - joining entity TFN) and (Details of cancellation of transfer of losses - Amount) must not be completed",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404023"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS36", Value = GetValueOrEmpty(report.CGLS36) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS38", Value = jeContext.CGLS38.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404023
            
                    #region VR.ATO.CGLS.404073
            
                    /*  VR.ATO.CGLS.404073
                    If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
                    Legacy Rule Format:
                    IF [CGLS38] <> NULL AND [CGLS38] <> MONETARY(U,11,0) 
                        RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS38] = CGLS:JE:rvctc3.02.02:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount
        
                    Technical Business Rule Format:
                    NotMonetary(^CGLS38, 'U', 11, 0)
            
                    Data Elements:
            
                    JE:^CGLS38 = tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount
                    */
                    assertion = NotMonetary(jeContext.CGLS38, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount[@contextRef='" + jeContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404073"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS38", Value = jeContext.CGLS38.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404073
            
                    #region VR.ATO.CGLS.404133
            
                    /*  VR.ATO.CGLS.404133
                    This validation is to check Tax file number of the joining entity against the TFN algorithm.
    
                    Legacy Rule Format:
                    IF (ALGORITHM([CGLS:JE:entity.identifier.TFN]) = FALSE)
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    FailsTFNAlgorithm(^TFN)
            
                    Data Elements:
            
                    ^TFN = JE
                    */
                    assertion = FailsTFNAlgorithm(jeContext.IdentifierTFN);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428016", Severity = ProcessMessageSeverity.Error,
                            Description = @"Tax File Number has failed the algorithm check",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(jeContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404133"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404133
            
                    #region VR.ATO.CGLS.404143
            
                    /*  VR.ATO.CGLS.404143
                    If the head company has cancelled the transfer of a loss then the amount must be shown
    
                    Legacy Rule Format:
                    IN CONTEXT(JE)
                    IF [CGLS38] = NULL
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS38] = CGLS:JE:rvctc3.02.02:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount
        
                    Technical Business Rule Format:
                    ^CGLS38 = NULL
            
                    Data Elements:
            
                    JE:^CGLS38 = tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount
                    */
                    assertion = (jeContext.CGLS38 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404057", Severity = ProcessMessageSeverity.Error,
                            Description = @"Cancelled transfer of loss amount must be supplied",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredLossCancelledByHeadCompany.Amount[@contextRef='" + jeContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404143"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS38", Value = jeContext.CGLS38.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404143
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating TRFRContext
            if (report.TRFRContextCollection != null)
            {
                int itemIndexContext1 = 0;
                foreach (CGLS2017.TRFRContext trfrContext in report.TRFRContextCollection)
                {
                    itemIndexContext1++;
            
                    #region VR.ATO.CGLS.404037
            
                    /*  VR.ATO.CGLS.404037
                    If a valid Transferor TFN is supplied, then a valid available fraction must be supplied
    
                    Legacy Rule Format:
                    IN CONTEXT(TRFR)
                    IF [CGLS108] = NULL AND [CGLS113] = NULL
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction
                    [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction
        
                    Technical Business Rule Format:
                    ^CGLS108 = NULL AND ^CGLS113 = NULL
            
                    Data Elements:
            
                    TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
            
                    TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
                    */
                    assertion = (trfrContext.CGLS108 == null && trfrContext.CGLS113 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404037", Severity = ProcessMessageSeverity.Error,
                            Description = @"A valid available fraction and amount must be supplied with the TFN",
                            LongDescription = @"If a valid Transferor TFN is supplied, then a valid available fraction and amount must be supplied",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404037"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS108", Value = trfrContext.CGLS108.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS113", Value = trfrContext.CGLS113.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404037
            
                    #region VR.ATO.CGLS.404131
            
                    /*  VR.ATO.CGLS.404131
                    This validation is to check Tax file number of the transferor entity against the TFN algorithm.
    
                    Legacy Rule Format:
                    IF (ALGORITHM([CGLS:TRFR:entity.identifier.TFN]) = FALSE)
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    FailsTFNAlgorithm(^TFN)
            
                    Data Elements:
            
                    ^TFN = TRFR
                    */
                    assertion = FailsTFNAlgorithm(trfrContext.IdentifierTFN);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428016", Severity = ProcessMessageSeverity.Error,
                            Description = @"Tax File Number has failed the algorithm check",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(trfrContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404131"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404131
            
                    #region VR.ATO.CGLS.404220
            
                    /*  VR.ATO.CGLS.404220
                    If a transferred tax loss deducted amount is entered then a corresponding available fraction must also be entered
    
                    Legacy Rule Format:
                    IN CONTEXT(TRFR)
                    IF [CGLS108] = NULL AND [CGLS109] <> NULL
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction
                    [CGLS109] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Amount
        
                    Technical Business Rule Format:
                    ^CGLS108 = NULL AND ^CGLS109 <> NULL
            
                    Data Elements:
            
                    TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
            
                    TRFR:^CGLS109 = tns:Tax.Losses.TransferredDeducted.Amount
                    */
                    assertion = (trfrContext.CGLS108 == null && trfrContext.CGLS109 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404220", Severity = ProcessMessageSeverity.Error,
                            Description = @"Each transferred amount of tax loss deducted must have a corresponding available fraction",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404220"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS108", Value = trfrContext.CGLS108.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS109", Value = trfrContext.CGLS109.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404220
            
                    #region VR.ATO.CGLS.404224
            
                    /*  VR.ATO.CGLS.404224
                    If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
                    Legacy Rule Format:
                    IF [CGLS109] <> NULL AND [CGLS109] <> MONETARY(U,11,0) 
                        RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS109] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Amount
        
                    Technical Business Rule Format:
                    NotMonetary(^CGLS109, 'U', 11, 0)
            
                    Data Elements:
            
                    TRFR:^CGLS109 = tns:Tax.Losses.TransferredDeducted.Amount
                    */
                    assertion = NotMonetary(trfrContext.CGLS109, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Amount[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404224"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS109", Value = trfrContext.CGLS109.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404224
            
                    #region VR.ATO.CGLS.404225
            
                    /*  VR.ATO.CGLS.404225
                    If a transferred tax loss deducted available fraction is entered then a corresponding transferred tax loss deducted amount must be entered.
    
                    Legacy Rule Format:
                    IN CONTEXT(TRFR)
                    IF [CGLS108] <> NULL AND [CGLS109] = NULL
                    RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction
                    [CGLS109] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Amount
        
                    Technical Business Rule Format:
                    ^CGLS108 <> NULL AND ^CGLS109 = NULL
            
                    Data Elements:
            
                    TRFR:^CGLS109 = tns:Tax.Losses.TransferredDeducted.Amount
            
                    TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
                    */
                    assertion = (trfrContext.CGLS108 != null && trfrContext.CGLS109 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404225", Severity = ProcessMessageSeverity.Error,
                            Description = @"Each available fraction must have a corresponding transferred tax loss deducted amount",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Amount[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404225"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS108", Value = trfrContext.CGLS108.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS109", Value = trfrContext.CGLS109.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404225
            
                    #region VR.ATO.CGLS.404226
            
                    /*  VR.ATO.CGLS.404226
                    If available fraction field populated, then entered figure must be a single digit followed by 3 decimal places.
    
                    Legacy Rule Format:
                    IF [CGLS113] <> NULL AND [CGLS113] <> NUMBER (U,4,3) 
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction
        
                    Technical Business Rule Format:
                    NotNumeric(^CGLS113, 'U', 4, 3)
            
                    Data Elements:
            
                    TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
                    */
                    assertion = NotNumeric(trfrContext.CGLS113, @"U", 4, 3);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404046", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredApplied.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404226"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS113", Value = trfrContext.CGLS113.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404226
            
                    #region VR.ATO.CGLS.404227
            
                    /*  VR.ATO.CGLS.404227
                    If a transferred net capital loss applied amount is entered then a corresponding available fraction must also be entered
    
                    Legacy Rule Format:
                    IN CONTEXT(TRFR)
                    IF [CGLS113] = NULL AND [CGLS114] <> NULL
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction
                    [CGLS114] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Amount
        
                    Technical Business Rule Format:
                    ^CGLS113 = NULL AND ^CGLS114 <> NULL
            
                    Data Elements:
            
                    TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
            
                    TRFR:^CGLS114 = tns:Capital.Losses.TransferredApplied.Amount
                    */
                    assertion = (trfrContext.CGLS113 == null && trfrContext.CGLS114 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404227", Severity = ProcessMessageSeverity.Error,
                            Description = @"Each transferred net capital loss applied amount must have a corresponding available fraction",
                            Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredApplied.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404227"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS113", Value = trfrContext.CGLS113.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS114", Value = trfrContext.CGLS114.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404227
            
                    #region VR.ATO.CGLS.404228
            
                    /*  VR.ATO.CGLS.404228
                    The available fraction used for transferred net capital losses applied cannot equal zero
    
                    Legacy Rule Format:
                    IF [CGLS113] = 0
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction
        
                    Technical Business Rule Format:
                    ^CGLS113 = 0
            
                    Data Elements:
            
                    TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
                    */
                    assertion = (trfrContext.CGLS113 == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404228", Severity = ProcessMessageSeverity.Error,
                            Description = @"Fraction amount for transferred net capital losses applied must be greater than zero",
                            Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredApplied.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404228"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS113", Value = trfrContext.CGLS113.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404228
            
                    #region VR.ATO.CGLS.404229
            
                    /*  VR.ATO.CGLS.404229
                    If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
                    Legacy Rule Format:
                    IF [CGLS114] <> NULL AND [CGLS114] <> MONETARY(U,11,0) 
                        RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS114] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Amount
        
                    Technical Business Rule Format:
                    NotMonetary(^CGLS114, 'U', 11, 0)
            
                    Data Elements:
            
                    TRFR:^CGLS114 = tns:Capital.Losses.TransferredApplied.Amount
                    */
                    assertion = NotMonetary(trfrContext.CGLS114, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredApplied.Amount[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404229"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS114", Value = trfrContext.CGLS114.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404229
            
                    #region VR.ATO.CGLS.404230
            
                    /*  VR.ATO.CGLS.404230
                    If a transferred net capital loss applied available fraction is entered then a corresponding transferred net capital loss applied amount must be entered.
    
                    Legacy Rule Format:
                    IN CONTEXT(TRFR)
                    IF [CGLS113] <> NULL AND [CGLS114] = NULL
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction
                    [CGLS114] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Amount
        
                    Technical Business Rule Format:
                    ^CGLS113 <> NULL AND ^CGLS114 = NULL
            
                    Data Elements:
            
                    TRFR:^CGLS114 = tns:Capital.Losses.TransferredApplied.Amount
            
                    TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
                    */
                    assertion = (trfrContext.CGLS113 != null && trfrContext.CGLS114 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404230", Severity = ProcessMessageSeverity.Error,
                            Description = @"Each available fraction must have a corresponding transferred net capital loss applied amount",
                            Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredApplied.Amount[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404230"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS113", Value = trfrContext.CGLS113.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS114", Value = trfrContext.CGLS114.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404230
            
                    #region VR.ATO.CGLS.404231
            
                    /*  VR.ATO.CGLS.404231
                    Each available fraction label amount completed for transferred tax losses deducted or transferred net capital losses deducted must not exceed 1.000.
    
                    Legacy Rule Format:
                    IN CONTEXT(TRFR)
                    IF [CGLS108] > 1.000 OR [CGLS113] > 1.000
                          RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction
                    [CGLS113] = CGLS:TRFR:rvctc3.02.15:Capital.Losses.TransferredApplied.Fraction
        
                    Technical Business Rule Format:
                    ^CGLS108 > 1.000 OR ^CGLS113 > 1.000
            
                    Data Elements:
            
                    TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
            
                    TRFR:^CGLS113 = tns:Capital.Losses.TransferredApplied.Fraction
                    */
                    assertion = (trfrContext.CGLS108.GetValueOrDefault() > 1.000M || trfrContext.CGLS113.GetValueOrDefault() > 1.000M);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404007", Severity = ProcessMessageSeverity.Error,
                            Description = @"Each occurrence of available fraction must not exceed 1.000",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404231"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS108", Value = trfrContext.CGLS108.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS113", Value = trfrContext.CGLS113.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404231
            
                    #region VR.ATO.CGLS.404232
            
                    /*  VR.ATO.CGLS.404232
                    If available fraction field populated, then entered figure must be a single digit followed by 3 decimal places.
    
                    Legacy Rule Format:
                    IF [CGLS108] <> NULL AND [CGLS108] <> NUMBER(U,4,3)
                         RETURN VALIDATION ERROR
                    ENDIF
                    
                    [CGLS108] = CGLS:TRFR:rvctc3.02.15:Tax.Losses.TransferredDeducted.Fraction
        
                    Technical Business Rule Format:
                    NotNumeric(^CGLS108, 'U', 4, 3)
            
                    Data Elements:
            
                    TRFR:^CGLS108 = tns:Tax.Losses.TransferredDeducted.Fraction
                    */
                    assertion = NotNumeric(trfrContext.CGLS108, @"U", 4, 3);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGLS.404046", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredDeducted.Fraction[@contextRef='" + trfrContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGLS.404232"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGLS108", Value = trfrContext.CGLS108.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGLS.404232
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 