
using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationIDS
{

    public partial class IDS2025Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private IDS2025 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="IDS2025Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public IDS2025Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="IDS2025Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public IDS2025Validator(IDS2025 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());

        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public IDS2025 ConsumedReport { get { return report; } private set { report = value; } }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(IDS2025 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            #region Manual Rules - Rules marked for manual coding

            #region VR.ATO.IDS.440005

            /*  VR.ATO.IDS.440005
                If an expenditure or revenue amount is completed at 'Did you have dealings with international related parties, apart from the dealings with related parties located in specified countries?' or 'Did you have dealings with international related parties located in specified countries?', then there must be at least one expenditure or revenue amount greater than zero at one of Questions 5, 6, 7, 8, 9, 10, 11 or 12.

            Legacy Rule Format:
                (AnyOccurrence(^IDS5, ^IDS5 > 0) OR AnyOccurrence(^IDS6, ^IDS6 > 0) OR AnyOccurrence(^IDS12, ^IDS12 > 0) OR AnyOccurrence(^IDS13, ^IDS13 > 0)) AND ((^IDS17 = NULL OR ^IDS17 = 0) AND (^IDS18 = NULL OR ^IDS18 = 0) AND (^IDS22 = NULL OR ^IDS22 = 0) AND (^IDS23 = NULL OR ^IDS23 = 0) AND (^IDS26 = NULL OR ^IDS26 = 0) AND (^IDS27 = NULL OR ^IDS27 = 0) AND (^IDS31 = NULL OR ^IDS31 = 0) AND (^IDS32 = NULL OR ^IDS32 = 0) AND (^IDS36 = NULL OR ^IDS36 = 0) AND (^IDS37 = NULL OR ^IDS37 = 0) AND (^IDS40 = NULL OR ^IDS40 = 0) AND (^IDS41 = NULL OR ^IDS41 = 0) AND (^IDS44 = NULL OR ^IDS44 = 0) AND (^IDS45 = NULL OR ^IDS45 = 0) AND (^IDS48 = NULL OR ^IDS48 = 0) AND (^IDS49 = NULL OR ^IDS49 = 0) AND (^IDS52 = NULL OR ^IDS52 = 0) AND (^IDS53 = NULL OR ^IDS53 = 0) AND (^IDS56 = NULL OR ^IDS56 = 0) AND (^IDS57 = NULL OR ^IDS57 = 0) AND (^IDS60 = NULL OR ^IDS60 = 0) AND (^IDS61 = NULL OR ^IDS61 = 0) AND (^IDS64 = NULL OR ^IDS64 = 0) AND (^IDS65 = NULL OR ^IDS65 = 0) AND (^IDS68 = NULL OR ^IDS68 = 0) AND (^IDS69 = NULL OR ^IDS69 = 0) AND (^IDS72 = NULL OR ^IDS72 = 0) AND (^IDS73 = NULL OR ^IDS73 = 0) AND (^IDS76 = NULL OR ^IDS76 = 0) AND (^IDS77 = NULL OR ^IDS77 = 0) AND (^IDS82 = NULL OR ^IDS82 = 0) AND (^IDS83 = NULL OR ^IDS83 = 0) AND (^IDS89 = NULL OR ^IDS89 = 0) AND (^IDS90 = NULL OR ^IDS90 = 0) AND (^IDS93 = NULL OR ^IDS93 = 0) AND (^IDS94 = NULL OR ^IDS94 = 0) AND (^IDS102 = NULL OR ^IDS102 = 0) AND (^IDS103 = NULL OR ^IDS103 = 0) AND (^IDS106 = NULL OR ^IDS106 = 0) AND (^IDS107 = NULL OR ^IDS107 = 0) AND (^IDS110 = NULL OR ^IDS110 = 0) AND (^IDS111 = NULL OR ^IDS111 = 0) AND (^IDS114 = NULL OR ^IDS114 = 0) AND (^IDS115 = NULL OR ^IDS115 = 0) AND ALL OCCURRENCES OF(^IDS336) = SET(0, NULL) AND ALL OCCURRENCES OF(^IDS337) = SET(0, NULL) AND ALL OCCURRENCES OF(^IDS338) = SET(0, NULL) AND ALL OCCURRENCES OF(^IDS339) = SET(0, NULL) AND (^IDS118 = NULL OR ^IDS118 = 0) AND (^IDS119 = NULL OR ^IDS119 = 0) AND (^IDS124 = NULL OR ^IDS124 = 0) AND (^IDS125 = NULL OR ^IDS125 = 0))

            Technical Business Rule Format:
                (AnyOccurrence(^IDS5, ^IDS5 > 0) OR AnyOccurrence(^IDS6, ^IDS6 > 0) OR AnyOccurrence(^IDS12, ^IDS12 > 0) OR AnyOccurrence(^IDS13, ^IDS13 > 0)) AND ((^IDS17 = NULL OR ^IDS17 = 0) AND (^IDS18 = NULL OR ^IDS18 = 0) AND (^IDS22 = NULL OR ^IDS22 = 0) AND (^IDS23 = NULL OR ^IDS23 = 0) AND (^IDS26 = NULL OR ^IDS26 = 0) AND (^IDS27 = NULL OR ^IDS27 = 0) AND (^IDS31 = NULL OR ^IDS31 = 0) AND (^IDS32 = NULL OR ^IDS32 = 0) AND (^IDS36 = NULL OR ^IDS36 = 0) AND (^IDS37 = NULL OR ^IDS37 = 0) AND (^IDS40 = NULL OR ^IDS40 = 0) AND (^IDS41 = NULL OR ^IDS41 = 0) AND (^IDS44 = NULL OR ^IDS44 = 0) AND (^IDS45 = NULL OR ^IDS45 = 0) AND (^IDS48 = NULL OR ^IDS48 = 0) AND (^IDS49 = NULL OR ^IDS49 = 0) AND (^IDS52 = NULL OR ^IDS52 = 0) AND (^IDS53 = NULL OR ^IDS53 = 0) AND (^IDS56 = NULL OR ^IDS56 = 0) AND (^IDS57 = NULL OR ^IDS57 = 0) AND (^IDS60 = NULL OR ^IDS60 = 0) AND (^IDS61 = NULL OR ^IDS61 = 0) AND (^IDS64 = NULL OR ^IDS64 = 0) AND (^IDS65 = NULL OR ^IDS65 = 0) AND (^IDS68 = NULL OR ^IDS68 = 0) AND (^IDS69 = NULL OR ^IDS69 = 0) AND (^IDS72 = NULL OR ^IDS72 = 0) AND (^IDS73 = NULL OR ^IDS73 = 0) AND (^IDS76 = NULL OR ^IDS76 = 0) AND (^IDS77 = NULL OR ^IDS77 = 0) AND (^IDS82 = NULL OR ^IDS82 = 0) AND (^IDS83 = NULL OR ^IDS83 = 0) AND (^IDS89 = NULL OR ^IDS89 = 0) AND (^IDS90 = NULL OR ^IDS90 = 0) AND (^IDS93 = NULL OR ^IDS93 = 0) AND (^IDS94 = NULL OR ^IDS94 = 0) AND (^IDS102 = NULL OR ^IDS102 = 0) AND (^IDS103 = NULL OR ^IDS103 = 0) AND (^IDS106 = NULL OR ^IDS106 = 0) AND (^IDS107 = NULL OR ^IDS107 = 0) AND (^IDS110 = NULL OR ^IDS110 = 0) AND (^IDS111 = NULL OR ^IDS111 = 0) AND (^IDS114 = NULL OR ^IDS114 = 0) AND (^IDS115 = NULL OR ^IDS115 = 0) AND ALL OCCURRENCES OF(^IDS336) = SET(0, NULL) AND ALL OCCURRENCES OF(^IDS337) = SET(0, NULL) AND ALL OCCURRENCES OF(^IDS338) = SET(0, NULL) AND ALL OCCURRENCES OF(^IDS339) = SET(0, NULL) AND (^IDS118 = NULL OR ^IDS118 = 0) AND (^IDS119 = NULL OR ^IDS119 = 0) AND (^IDS124 = NULL OR ^IDS124 = 0) AND (^IDS125 = NULL OR ^IDS125 = 0))

            Data Elements:

            ^IDS5 = IDS:RP:ForeignCountry:ForeignDealings:Expense.Total.Amount

            ^IDS6 = IDS:RP:ForeignCountry:ForeignDealings:Revenue.Total.Amount

            ^IDS12 = IDS:RP:SpecifiedCountry:SpecifiedDealings:Expense.Total.Amount

            ^IDS13 = IDS:RP:SpecifiedCountry:SpecifiedDealings:Revenue.Total.Amount

            ^IDS17 = IDS:RP:TangiblePropertyRevenue:Expense.Total.Amount

            ^IDS18 = IDS:RP:TangiblePropertyRevenue:Revenue.Total.Amount

            ^IDS22 = IDS:RP:Royalties:Expense.DeductionsTotal.Amount

            ^IDS23 = IDS:RP:Royalties:Income.IncludedInAssessableTotal.Amount

            ^IDS26 = IDS:RP:LicenceFees:Expense.DeductionsTotal.Amount

            ^IDS27 = IDS:RP:LicenceFees:Income.IncludedInAssessableTotal.Amount

            ^IDS31 = IDS:RP:RentOrLeasing:Expense.Total.Amount

            ^IDS32 = IDS:RP:RentOrLeasing:Revenue.Total.Amount

            ^IDS36 = IDS:RP:ServiceArrangements:TreasuryServices:Expense.Total.Amount

            ^IDS37 = IDS:RP:ServiceArrangements:TreasuryServices:Revenue.Total.Amount

            ^IDS40 = IDS:RP:ServiceArrangements:ManagementAndAdministrationServices:Expense.Total.Amount

            ^IDS41 = IDS:RP:ServiceArrangements:ManagementAndAdministrationServices:Revenue.Total.Amount

            ^IDS44 = IDS:RP:ServiceArrangements:InsuranceServices:Expense.Total.Amount

            ^IDS45 = IDS:RP:ServiceArrangements:InsuranceServices:Revenue.Total.Amount

            ^IDS48 = IDS:RP:ServiceArrangements:ReinsuranceServices:Expense.Total.Amount

            ^IDS49 = IDS:RP:ServiceArrangements:ReinsuranceServices:Revenue.Total.Amount

            ^IDS52 = IDS:RP:ServiceArrangements:RAndDServices:Expense.Total.Amount

            ^IDS53 = IDS:RP:ServiceArrangements:RAndDServices:Revenue.Total.Amount

            ^IDS56 = IDS:RP:ServiceArrangements:SalesAndMarketingServices:Expense.Total.Amount

            ^IDS57 = IDS:RP:ServiceArrangements:SalesAndMarketingServices:Revenue.Total.Amount

            ^IDS60 = IDS:RP:ServiceArrangements:SoftwareAndITServices:Expense.Total.Amount

            ^IDS61 = IDS:RP:ServiceArrangements:SoftwareAndITServices:Revenue.Total.Amount

            ^IDS64 = IDS:RP:ServiceArrangements:TechnicalServices:Expense.Total.Amount

            ^IDS65 = IDS:RP:ServiceArrangements:TechnicalServices:Revenue.Total.Amount

            ^IDS68 = IDS:RP:ServiceArrangements:LogisticsServices:Expense.Total.Amount

            ^IDS69 = IDS:RP:ServiceArrangements:LogisticsServices:Revenue.Total.Amount

            ^IDS72 = IDS:RP:ServiceArrangements:AssetManagementServices:Expense.Total.Amount

            ^IDS73 = IDS:RP:ServiceArrangements:AssetManagementServices:Revenue.Total.Amount

            ^IDS76 = IDS:RP:ServiceArrangements:OtherServices:Expense.Total.Amount

            ^IDS77 = IDS:RP:ServiceArrangements:OtherServices:Revenue.Total.Amount

            ^IDS82 = IDS:RP:Derivatives:Expense.Total.Amount

            ^IDS83 = IDS:RP:Derivatives:Revenue.Total.Amount

            ^IDS89 = IDS:RP:DebtFactoring:Expense.BookValue.Amount

            ^IDS90 = IDS:RP:DebtFactoring:Expense.Consideration.Amount

            ^IDS93 = IDS:RP:Securitisation:Expense.BookValue.Amount

            ^IDS94 = IDS:RP:Securitisation:Expense.Consideration.Amount

            ^IDS102 = IDS:RP:FinancialDealings:Interest:Expense.Total.Amount

            ^IDS103 = IDS:RP:FinancialDealings:Interest:Revenue.Total.Amount

            ^IDS106 = IDS:RP:FinancialDealings:Guarantees:Expense.Total.Amount

            ^IDS107 = IDS:RP:FinancialDealings:Guarantees:Revenue.Total.Amount

            ^IDS110 = IDS:RP:FinancialDealings:Insurance:Expense.Total.Amount

            ^IDS111 = IDS:RP:FinancialDealings:Insurance:Revenue.Total.Amount

            ^IDS114 = IDS:RP:FinancialDealings:Reinsurance:Expense.Total.Amount

            ^IDS115 = IDS:RP:FinancialDealings:Reinsurance:Revenue.Total.Amount

            ^IDS118 = IDS:RP:FinancialDealings:OtherFinancial:Expense.Total.Amount

            ^IDS119 = IDS:RP:FinancialDealings:OtherFinancial:Revenue.Total.Amount

            ^IDS124 = IDS:RP:RevenueDealings:Expense.Total.Amount

            ^IDS125 = IDS:RP:RevenueDealings:Revenue.Total.Amount

            ^IDS336 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeLossesHighestNominal.Amount

            ^IDS337 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeGainsHighestNominal.Amount

            ^IDS338 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeLossesOther.Amount

            ^IDS339 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeGainsOther.Amount
            */
            VRATOIDS440005(response, report);
            #endregion // VR.ATO.IDS.440005
            #endregion Manual Rules - Rules marked for manual coding
            VRATOIDS440001();
            VRATOIDS440003();
            VRATOIDS440004();
            VRATOIDS440010();
            VRATOIDS440011();
            VRATOIDS440012();
            VRATOIDS440013();
            VRATOIDS440014();
            VRATOIDS440015();
            VRATOIDS440020();
            VRATOIDS440021();
            VRATOIDS440022();
            VRATOIDS440023();
            VRATOIDS440024();
            VRATOIDS440025();
            VRATOIDS440026();
            VRATOIDS440027();
            VRATOIDS440028();
            VRATOIDS440029();
            VRATOIDS440030();
            VRATOIDS440031();
            VRATOIDS440032();
            VRATOIDS440033();
            VRATOIDS440034();
            VRATOIDS440035();
            VRATOIDS440036();
            VRATOIDS440037();
            VRATOIDS440038();
            VRATOIDS440047();
            VRATOIDS440052();
            VRATOIDS440053();
            VRATOIDS440054();
            VRATOIDS440055();
            VRATOIDS440056();
            VRATOIDS440057();
            VRATOIDS440058();
            VRATOIDS440059();
            VRATOIDS440060();
            VRATOIDS440061();
            VRATOIDS440064();
            VRATOIDS440067();
            VRATOIDS440068();
            VRATOIDS440073();
            VRATOIDS440074();
            VRATOIDS440075();
            VRATOIDS440076();
            VRATOIDS440080();
            VRATOIDS440081();
            VRATOIDS440083();
            VRATOIDS440095();
            VRATOIDS440096();
            VRATOIDS440102();
            VRATOIDS440103();
            VRATOIDS440105();
            VRATOIDS440106();
            VRATOIDS440111();
            VRATOIDS440115();
            VRATOIDS440128();
            VRATOIDS440129();
            VRATOIDS440132();
            VRATOIDS440133();
            VRATOIDS440135();
            VRATOIDS440137();
            VRATOIDS440141();
            VRATOIDS440142();
            VRATOIDS440160();
            VRATOIDS440183();
            VRATOIDS440197();
            VRATOIDS440198();
            VRATOIDS440203();
            VRATOIDS440204();
            VRATOIDS440295();
            VRATOIDS440296();
            VRATOIDS440297();
            VRATOIDS440308();
            VRATOIDS440309();
            VRATOIDS440310();
            VRATOIDS440420();
            VRATOIDS440421();
            VRATOIDS440425();
            VRATOIDS440436();
            VRATOIDS440439();
            VRATOIDS440442();
            VRATOIDS440448();
            VRATOIDS440506();
            VRATOIDS440507();
            VRATOIDS440508();
            VRATOIDS440509();
            VRATOIDS440510();
            VRATOIDS440511();
            VRATOIDS440512();
            VRATOIDS440513();
            VRATOIDS440515();
            VRATOIDS440516();
            VRATOIDS440524();
            VRATOIDS440525();
            VRATOIDS440533();
            VRATOIDS440534();
            VRATOIDS440543();
            VRATOIDS440544();
            VRATOIDS440545();
            VRATOIDS440560();
            VRATOIDS440561();
            VRATOIDS440562();
            VRATOIDS440563();
            VRATOIDS440574();
            VRATOIDS440577();
            VRATOIDS440578();
            VRATOIDS440583();
            VRATOIDS440587();
            VRATOIDS440588();
            VRATOIDS440589();
            VRATOIDS440593();
            VRATOIDS440594();
            VRATOIDS440602();
            VRATOIDS440603();
            VRATOIDS440604();
            VRATOIDS440608();
            VRATOIDS440609();
            VRATOIDS440618();
            VRATOIDS440619();
            VRATOIDS440620();
            VRATOIDS440621();
            VRATOIDS440622();
            VRATOIDS440623();
            VRATOIDS440634();
            VRATOIDS440635();
            VRATOIDS440636();
            VRATOIDS440637();
            VRATOIDS440638();
            VRATOIDS440639();
            VRATOIDS440640();
            VRATOIDS440641();
            VRATOIDS440642();
            VRATOIDS440655();
            VRATOIDS440657();
            VRATOIDS440675();
            VRATOIDS440678();
            VRATOIDS440681();
            VRATOIDS440682();
            VRATOIDS440691();
            VRATOIDS440692();
            VRATOIDS440693();
            VRATOIDS440694();
            VRATOIDS440695();
            VRATOIDS440696();
            VRATOIDS440700();
            VRATOIDS440750();
            VRATOIDS440751();
            VRATOIDS440756();
            VRATOIDS440757();
            VRATOIDS440758();
            VRATOIDS440783();
            VRATOIDS440793();
            VRATOIDS440795();
            VRATOIDS440796();
            VRATOIDS440800();
            VRATOIDS440801();
            VRATOIDS440802();
            VRATOIDS440803();
            VRATOIDS440813();
            VRATOIDS440814();
            VRATOIDS440815();
            VRATOIDS440816();
            VRATOIDS440821();
            VRATOIDS440822();
            VRATOIDS440823();
            VRATOIDS440825();
            VRATOIDS440826();
            VRATOIDS440827();
            VRATOIDS440828();
            VRATOIDS440832();
            VRATOIDS440836();
            VRATOIDS440843();
            VRATOIDS440848();
            VRATOIDS440849();
            VRATOIDS440855();
            VRATOIDS440859();
            VRATOIDS440863();
            VRATOIDS440865();
            VRATOIDS440870();
            VRATOIDS440871();
            VRATOIDS440881();
            VRATOIDS440882();
            VRATOIDS440883();
            VRATOIDS440885();
            VRATOIDS440889();
            VRATOIDS440891();
            VRATOIDS440897();
            VRATOIDS440898();
            VRATOIDS440900();
            VRATOIDS440902();
            VRATOIDS440904();
            VRATOIDS440906();
            VRATOIDS440908();
            VRATOIDS440910();
            VRATOIDS440915();
            VRATOIDS440916();
            VRATOIDS440917();
            VRATOIDS440918();
            VRATOIDS440920();
            VRATOIDS440921();
            VRATOIDS440922();
            VRATOIDS440923();
            VRATOIDS440924();
            VRATOIDS440925();
            VRATOIDS440926();
            VRATOIDS440927();
            VRATOIDS440928();
            VRATOIDS440930();
            VRATOIDS440932();
            VRATOIDS440933();
            VRATOIDS440934();
            VRATOIDS440935();
            VRATOIDS440936();
            VRATOIDS440938();
            VRATOIDS440939();
            VRATOIDS440941();
            VRATOIDS440942();
            VRATOIDS440943();
            VRATOIDS440945();
            VRATOIDS440946();
            VRATOIDS440948();
            VRATOIDS440949();
            VRATOIDS440950();
            VRATOIDS440952();
            VRATOIDS440954();
            VRATOIDS440956();
            VRATOIDS440957();
            VRATOIDS440958();
            VRATOIDS440960();
            VRATOIDS440961();
            VRATOIDS440962();
            VRATOIDS440963();
            VRATOIDS440964();
            VRATOIDS440965();
            VRATOIDS440966();
            VRATOIDS440967();
            VRATOIDS440968();
            VRATOIDS440969();
            VRATOIDSW00001();
            VRATOIDSW00002();
            VRATOIDSW00003();
            VRATOIDSW00004();
            VRATOIDSW00005();
            VRATOIDSW00006();
            VRATOIDSW00007();
            VRATOIDSW00009();
            VRATOIDSW00016();
            VRATOIDSW00017();
            VRATOIDSW00018();
            VRATOIDSW00019();

            #region Repeating report.RP_ForeignExchangeCollection  
            if (report.RP_ForeignExchangeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_ForeignExchange foreignExchange in report.RP_ForeignExchangeCollection)
                {
                    itemIndex2++;

                    #region Repeating foreignExchange.RP_ForeignExchange_GainOrLossCollection  
                    if (foreignExchange.RP_ForeignExchange_GainOrLossCollection != null)
                    {
                        int itemIndex3 = 0;
                        foreach (IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss gainOrLoss in foreignExchange.RP_ForeignExchange_GainOrLossCollection)
                        {
                            itemIndex3++;
                            VRATOIDS440045(foreignExchange, gainOrLoss, itemIndex3);
                            VRATOIDS440046(foreignExchange, gainOrLoss, itemIndex3);
                            VRATOIDS440529(foreignExchange, gainOrLoss, itemIndex3);
                        }
                    }

                    #endregion // Foreach loop
                }
            }

            #endregion // Foreach loop

            #region Repeating IDS178Collection  
            if (report.IDS178Collection != null)
            {
                int itemIndex2 = 0;
                foreach (string IDS178 in report.IDS178Collection)
                {
                    itemIndex2++;
                    VRATOIDS440294(IDS178, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_ForeignExchangeCollection  
            if (report.RP_ForeignExchangeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_ForeignExchange foreignExchange in report.RP_ForeignExchangeCollection)
                {
                    itemIndex2++;
                    VRATOIDS440554(foreignExchange, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_SpecifiedCountryCollection  
            if (report.RP_SpecifiedCountryCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_SpecifiedCountry specifiedCountry in report.RP_SpecifiedCountryCollection)
                {
                    itemIndex2++;
                    VRATOIDS440555(specifiedCountry, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_ForeignCountryCollection  
            if (report.RP_ForeignCountryCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_ForeignCountry foreignCountry in report.RP_ForeignCountryCollection)
                {
                    itemIndex2++;
                    VRATOIDS440556(foreignCountry, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_MaterialArrangementsCollection  
            if (report.RP_MaterialArrangementsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_MaterialArrangements materialArrangements in report.RP_MaterialArrangementsCollection)
                {
                    itemIndex2++;
                    VRATOIDS440644(materialArrangements, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_HybridPayerMismatchCollection  
            if (report.RP_HybridPayerMismatchCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_HybridPayerMismatch hybridPayerMismatch in report.RP_HybridPayerMismatchCollection)
                {
                    itemIndex2++;
                    VRATOIDS440654(hybridPayerMismatch, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection  
            if (report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_RestructuringEventCurrentOrPriorIncomeYear restructuringEventCurrentOrPriorIncomeYear in report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection)
                {
                    itemIndex2++;
                    VRATOIDS440831(restructuringEventCurrentOrPriorIncomeYear, itemIndex2);
                    VRATOIDS440953(restructuringEventCurrentOrPriorIncomeYear, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_PriorIncomeYearTopMaterialArrangementsCollection  
            if (report.RP_PriorIncomeYearTopMaterialArrangementsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_PriorIncomeYearTopMaterialArrangements priorIncomeYearTopMaterialArrangements in report.RP_PriorIncomeYearTopMaterialArrangementsCollection)
                {
                    itemIndex2++;
                    VRATOIDS440899(priorIncomeYearTopMaterialArrangements, itemIndex2);
                    VRATOIDS440901(priorIncomeYearTopMaterialArrangements, itemIndex2);
                    VRATOIDS440903(priorIncomeYearTopMaterialArrangements, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection  
            if (report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement in report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection)
                {
                    itemIndex2++;
                    VRATOIDS440905(priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex2);
                    VRATOIDS440907(priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex2);
                    VRATOIDS440909(priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex2);
                    VRATOIDS440911(priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex2);
                    VRATOIDS440914(priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_ArrangementsCollection  
            if (report.RP_ArrangementsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IDS2025.RP_Arrangements arrangements in report.RP_ArrangementsCollection)
                {
                    itemIndex2++;
                    VRATOIDS440929(arrangements, itemIndex2);
                    VRATOIDS440937(arrangements, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection  
            if (report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection != null)
            {
                int itemIndex3 = 0;
                foreach (IDS2025.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement in report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection)
                {
                    itemIndex3++;
                    VRATOIDS440944(currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex3);
                    VRATOIDS440947(currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, itemIndex3);
                }
            }

            #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }



        #region VR.ATO.IDS.440001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440001
            If there is a FALSE response to 'Is the total of your international related party dealings over $2 million?', then no information must be provided at International related party dealings Section A Questions 3 to 16 of the schedule.
    
            Legacy Rule Format:
            ^IDS1 = FALSE AND (^IDS2 <> NULL OR ^IDS9 <> NULL OR ^IDS16 <> NULL OR ^IDS21 <> NULL OR ^IDS30 <> NULL OR ^IDS35 <> NULL OR ^IDS81 <> NULL OR ^IDS88 <> NULL OR ^IDS97 <> NULL OR ^IDS333 <> NULL OR ^IDS123 <> NULL OR ^IDS129 <> NULL OR ^IDS138 <> NULL OR ^IDS147 <> NULL OR ^IDS150 <> NULL OR ^IDS426 <> NULL)

            Technical Business Rule Format:
            ^IDS1 = FALSE AND (^IDS2 <> NULL OR ^IDS9 <> NULL OR ^IDS16 <> NULL OR ^IDS21 <> NULL OR ^IDS30 <> NULL OR ^IDS35 <> NULL OR ^IDS81 <> NULL OR ^IDS88 <> NULL OR ^IDS97 <> NULL OR ^IDS333 <> NULL OR ^IDS123 <> NULL OR ^IDS129 <> NULL OR ^IDS138 <> NULL OR ^IDS147 <> NULL OR ^IDS150 <> NULL OR ^IDS426 <> NULL)
    
            Data Elements:
    
            ^IDS1 = IDS:RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
            ^IDS2 = IDS:RP:InternationalDealings.RelatedPartiesTransactions.Indicator
    
            ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
    
            ^IDS16 = IDS:RP:InternationalDealings.TransactionsTangiblePropertyRevenue.Indicator
    
            ^IDS21 = IDS:RP:InternationalDealings.TransactionsRoyaltiesLicenceFees.Indicator
    
            ^IDS30 = IDS:RP:InternationalDealings.TransactionsRentLeasing.Indicator
    
            ^IDS35 = IDS:RP:InternationalDealings.TransactionsServiceArrangements.Indicator
    
            ^IDS81 = IDS:RP:InternationalDealings.DerivativeTransactions.Indicator
    
            ^IDS88 = IDS:RP:InternationalDealings.DebtFactoringSecuritisation.Indicator
    
            ^IDS97 = IDS:RP:InternationalDealings.OtherFinancialDealings.Indicator
    
            ^IDS123 = IDS:RP:InternationalDealings.OtherRevenue.Indicator
    
            ^IDS129 = IDS:RP:Miscellaneous.CapitalNature.Indicator
    
            ^IDS138 = IDS:RP:Miscellaneous.ReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsideration.Indicator
    
            ^IDS147 = IDS:RP:InternationalDealings.ShareBasedEmployeeRemuneration.Indicator
    
            ^IDS150 = IDS:RP:InternationalDealings.CostContributionArrangements.Indicator
    
            ^IDS333 = IDS:RP:InternationalDealings.ForeignExchangeGainLoss.Indicator
    
            ^IDS426 = IDS:RP:Miscellaneous.ResearchAndDevelopmentCostPlusRemuneration.Indicator
            */
            assertion = (report.IDS1 == false && (report.IDS2 != null || report.IDS9 != null || report.IDS16 != null || report.IDS21 != null || report.IDS30 != null || report.IDS35 != null || report.IDS81 != null || report.IDS88 != null || report.IDS97 != null || report.IDS333 != null || report.IDS123 != null || report.IDS129 != null || report.IDS138 != null || report.IDS147 != null || report.IDS150 != null || report.IDS426 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information is not required",
                    LongDescription = @"If there is a 'False' response to 'Is the total of your international related party dealings over $2 million?' then no information should be entered for International related party dealings",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS1", Value = GetValueOrEmpty(report.IDS1) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS2", Value = GetValueOrEmpty(report.IDS2) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS9", Value = GetValueOrEmpty(report.IDS9) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS16", Value = GetValueOrEmpty(report.IDS16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS21", Value = GetValueOrEmpty(report.IDS21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS30", Value = GetValueOrEmpty(report.IDS30) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS35", Value = GetValueOrEmpty(report.IDS35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS81", Value = GetValueOrEmpty(report.IDS81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS88", Value = GetValueOrEmpty(report.IDS88) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS97", Value = GetValueOrEmpty(report.IDS97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS333", Value = GetValueOrEmpty(report.IDS333) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS123", Value = GetValueOrEmpty(report.IDS123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS129", Value = GetValueOrEmpty(report.IDS129) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS138", Value = GetValueOrEmpty(report.IDS138) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS147", Value = GetValueOrEmpty(report.IDS147) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS150", Value = GetValueOrEmpty(report.IDS150) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS426", Value = GetValueOrEmpty(report.IDS426) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440003
            If foreign country code(s) have been completed, then the response to 'Did you have dealings with international related parties, apart from the dealings with related parties located in specified countries?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS2 <> TRUE AND Count(^IDS1002) <> 0

            Technical Business Rule Format:
            ^IDS2 <> TRUE AND Count(^IDS1002) <> 0
    
            Data Elements:
    
            ^IDS2 = IDS:RP:InternationalDealings.RelatedPartiesTransactions.Indicator
    
            ^IDS1002 = IDS:RP:ForeignCountry
            */
            assertion = (report.IDS2 != true && Count(report.RP_ForeignCountryCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If the foreign country codes have been completed then the response to 'Did you have dealings with international related parties, apart from dealings with related parties located in specified countries?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRelatedPartiesTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS2", Value = GetValueOrEmpty(report.IDS2) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440004
            If there is a TRUE response to 'Did you have dealings with international related parties, apart from the dealings with related parties located in specified countries?', then foreign country code(s) must be completed.
    
            Legacy Rule Format:
            ^IDS2 = TRUE AND Count(^IDS1002) = 0

            Technical Business Rule Format:
            ^IDS2 = TRUE AND Count(^IDS1002) = 0
    
            Data Elements:
    
            ^IDS2 = IDS:RP:InternationalDealings.RelatedPartiesTransactions.Indicator
    
            ^IDS1002 = IDS:RP:ForeignCountry
            */
            assertion = (report.IDS2 == true && Count(report.RP_ForeignCountryCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign country code required",
                    LongDescription = @"If there is a 'True' response to 'Did you have dealings with international related parties, apart from dealings with related parties located in specified countries?' then a foreign country code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRelatedPartiesTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440004" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS2", Value = GetValueOrEmpty(report.IDS2) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440010
            If specified country code(s) have been completed, then the response to 'Did you have dealings with international related parties located in specified countries?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS9 <> TRUE AND Count(^IDS1004) <> 0

            Technical Business Rule Format:
            ^IDS9 <> TRUE AND Count(^IDS1004) <> 0
    
            Data Elements:
    
            ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
    
            ^IDS1004 = IDS:RP:SpecifiedCountry
            */
            assertion = (report.IDS9 != true && Count(report.RP_SpecifiedCountryCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If the specified country codes have been completed then the response to 'Did you have dealings with international related parties located in specified countries?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsSpecifiedCountriesTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS9", Value = GetValueOrEmpty(report.IDS9) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440011
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440011()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440011
            If there is a TRUE response to 'Did you have dealings with international related parties located in specified countries?', then specified country code(s) must be completed.
    
            Legacy Rule Format:
            ^IDS9 = TRUE AND Count(^IDS1004) = 0

            Technical Business Rule Format:
            ^IDS9 = TRUE AND Count(^IDS1004) = 0
    
            Data Elements:
    
            ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
    
            ^IDS1004 = IDS:RP:SpecifiedCountry
            */
            assertion = (report.IDS9 == true && Count(report.RP_SpecifiedCountryCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified country code required",
                    LongDescription = @"If there is a 'True' response to 'Did you have dealings with international related parties located in specified countries?' then the specified country code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsSpecifiedCountriesTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440011" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS9", Value = GetValueOrEmpty(report.IDS9) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440012
            If international related party dealings information has been completed for Tangible property of a revenue nature, then the response to 'Did you have any international related party dealings involving tangible property of a revenue nature, including trading stock and raw materials?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS16 <> TRUE AND ^IDS1006 <> NULL

            Technical Business Rule Format:
            ^IDS16 <> TRUE AND ^IDS1006 <> NULL
    
            Data Elements:
    
            ^IDS16 = IDS:RP:InternationalDealings.TransactionsTangiblePropertyRevenue.Indicator
    
            ^IDS1006 = IDS:RP:TangiblePropertyRevenue
            */
            assertion = (report.IDS16 != true && report.RP_TangiblePropertyRevenueCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for Tangible property of a revenue nature then the response to 'Did you have any international related party dealings involving tangible property of a revenue nature, including trading stock and raw materials?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsTangiblePropertyRevenueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440012" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS16", Value = GetValueOrEmpty(report.IDS16) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440013
            If there is a TRUE response to 'Did you have any international related party dealings involving tangible property of a revenue nature, including trading stock and raw materials?', then Tangible property expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS16 = TRUE AND ^IDS17 = NULL AND ^IDS18 = NULL

            Technical Business Rule Format:
            ^IDS16 = TRUE AND ^IDS17 = NULL AND ^IDS18 = NULL
    
            Data Elements:
    
            ^IDS16 = IDS:RP:InternationalDealings.TransactionsTangiblePropertyRevenue.Indicator
    
            ^IDS17 = IDS:RP:TangiblePropertyRevenue:Expense.Total.Amount
    
            ^IDS18 = IDS:RP:TangiblePropertyRevenue:Revenue.Total.Amount
            */
            assertion = (report.IDS16 == true && report.IDS17 == null && report.IDS18 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tangible property of a revenue nature information incomplete",
                    LongDescription = @"If there is a 'True' response to 'Did you have any international related party dealings involving tangible property of a revenue nature, including trading stock and raw materials?' then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsTangiblePropertyRevenueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440013" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS16", Value = GetValueOrEmpty(report.IDS16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS17", Value = GetValueOrEmpty(report.IDS17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS18", Value = GetValueOrEmpty(report.IDS18) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440014()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440014
            If international related party dealings information has been completed for Royalties or Licence fees, then the response to 'Did you have any international related party dealings involving royalties or licence fees?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS21 <> TRUE AND (^IDS1007 <> NULL OR ^IDS1008 <> NULL)

            Technical Business Rule Format:
            ^IDS21 <> TRUE AND (^IDS1007 <> NULL OR ^IDS1008 <> NULL)
    
            Data Elements:
    
            ^IDS21 = IDS:RP:InternationalDealings.TransactionsRoyaltiesLicenceFees.Indicator
    
            ^IDS1007 = IDS:RP:Royalties
    
            ^IDS1008 = IDS:RP:LicenceFees
            */
            assertion = (report.IDS21 != true && (report.RP_RoyaltiesCollectionExists != false || report.RP_LicenceFeesCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for royalties or licence fees then the response to 'Did you have international related party dealings involving royalties or licence fees?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsRoyaltiesLicenceFeesI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440014" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS21", Value = GetValueOrEmpty(report.IDS21) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440015
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440015()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440015
            If there is a TRUE response to 'Did you have any international related party dealings involving royalties or licence fees?', then international related party dealings information must be completed for Royalties and/or Licence fees and at least one of the expenditure or revenue amounts must be completed.
    
            Legacy Rule Format:
            ^IDS21 = TRUE AND ((^IDS1007 = NULL AND ^IDS1008 = NULL) OR (^IDS1007 <> NULL AND ^IDS22 = NULL AND ^IDS23 = NULL) OR (^IDS1008 <> NULL AND ^IDS26 = NULL AND ^IDS27 = NULL))

            Technical Business Rule Format:
            ^IDS21 = TRUE AND ((^IDS1007 = NULL AND ^IDS1008 = NULL) OR (^IDS1007 <> NULL AND ^IDS22 = NULL AND ^IDS23 = NULL) OR (^IDS1008 <> NULL AND ^IDS26 = NULL AND ^IDS27 = NULL))
    
            Data Elements:
    
            ^IDS21 = IDS:RP:InternationalDealings.TransactionsRoyaltiesLicenceFees.Indicator
    
            ^IDS22 = IDS:RP:Royalties:Expense.DeductionsTotal.Amount
    
            ^IDS23 = IDS:RP:Royalties:Income.IncludedInAssessableTotal.Amount
    
            ^IDS26 = IDS:RP:LicenceFees:Expense.DeductionsTotal.Amount
    
            ^IDS27 = IDS:RP:LicenceFees:Income.IncludedInAssessableTotal.Amount
    
            ^IDS1007 = IDS:RP:Royalties
    
            ^IDS1008 = IDS:RP:LicenceFees
            */
            assertion = (report.IDS21 == true && (report.RP_RoyaltiesCollectionExists == false && report.RP_LicenceFeesCollectionExists == false || report.RP_RoyaltiesCollectionExists != false && report.IDS22 == null && report.IDS23 == null || report.RP_LicenceFeesCollectionExists != false && report.IDS26 == null && report.IDS27 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440689",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required.",
                    LongDescription = @"If there is a TRUE response to 'Did you have any international related party dealings involving royalties or licence fees?', then international related party dealings information must be completed for Royalties and/or Licence fees and at least one of the expenditure or revenue amounts must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsRoyaltiesLicenceFeesI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440015" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS21", Value = GetValueOrEmpty(report.IDS21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS22", Value = GetValueOrEmpty(report.IDS22) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS23", Value = GetValueOrEmpty(report.IDS23) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS26", Value = GetValueOrEmpty(report.IDS26) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS27", Value = GetValueOrEmpty(report.IDS27) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440020()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440020
            If international related party dealings information has been completed for Rent/leasing, then the response to 'Did you have any international related party dealings involving rent or leasing?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS30 <> TRUE AND ^IDS1009 <> NULL

            Technical Business Rule Format:
            ^IDS30 <> TRUE AND ^IDS1009 <> NULL
    
            Data Elements:
    
            ^IDS30 = IDS:RP:InternationalDealings.TransactionsRentLeasing.Indicator
    
            ^IDS1009 = IDS:RP:RentOrLeasing
            */
            assertion = (report.IDS30 != true && report.RP_RentOrLeasingCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for rent or leasing then the response to 'Did you have any international related party dealings involving rent or leasing?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsRentLeasingI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440020" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS30", Value = GetValueOrEmpty(report.IDS30) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440021
            If there is a TRUE response to 'Did you have any international related party dealings involving rent or leasing?', then Rent/leasing expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS30 = TRUE AND ^IDS31 = NULL AND ^IDS32 = NULL

            Technical Business Rule Format:
            ^IDS30 = TRUE AND ^IDS31 = NULL AND ^IDS32 = NULL
    
            Data Elements:
    
            ^IDS30 = IDS:RP:InternationalDealings.TransactionsRentLeasing.Indicator
    
            ^IDS31 = IDS:RP:RentOrLeasing:Expense.Total.Amount
    
            ^IDS32 = IDS:RP:RentOrLeasing:Revenue.Total.Amount
            */
            assertion = (report.IDS30 == true && report.IDS31 == null && report.IDS32 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Rent or leasing information incomplete",
                    LongDescription = @"If there is a 'True' response to 'Did you have any international related party dealings involving rent or leasing?' then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsRentLeasingI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS30", Value = GetValueOrEmpty(report.IDS30) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS31", Value = GetValueOrEmpty(report.IDS31) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS32", Value = GetValueOrEmpty(report.IDS32) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440022
            If international related party dealings information has been completed for any service arrangements, then the response to 'Did you have any service arrangements with international related parties?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS35 <> TRUE AND ^IDS2001 <> NULL

            Technical Business Rule Format:
            ^IDS35 <> TRUE AND ^IDS2001 <> NULL
    
            Data Elements:
    
            ^IDS35 = IDS:RP:InternationalDealings.TransactionsServiceArrangements.Indicator
    
            ^IDS2001 = IDS:RP:ServiceArrangements
            */
            assertion = (report.IDS35 != true && report.RP_ServiceArrangementsCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for any service arrangements then the response to 'Did you have any service arrangements with international related parties?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsServiceArrangementsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440022" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS35", Value = GetValueOrEmpty(report.IDS35) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440023
            If there is a TRUE response to 'Did you have any service arrangements with international related parties?', then service arrangements information must be completed.
    
            Legacy Rule Format:
            ^IDS35 = TRUE AND ^IDS2001 = NULL

            Technical Business Rule Format:
            ^IDS35 = TRUE AND ^IDS2001 = NULL
    
            Data Elements:
    
            ^IDS35 = IDS:RP:InternationalDealings.TransactionsServiceArrangements.Indicator
    
            ^IDS2001 = IDS:RP:ServiceArrangements
            */
            assertion = (report.IDS35 == true && report.RP_ServiceArrangementsCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required",
                    LongDescription = @"If there is a 'True' response to 'Did you have any service arrangements with international related parties?' then service arrangements information must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTransactionsServiceArrangementsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS35", Value = GetValueOrEmpty(report.IDS35) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440024()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440024
            If service arrangements information has been completed for Treasury related services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1010 <> NULL AND ^IDS36 = NULL AND ^IDS37 = NULL

            Technical Business Rule Format:
            ^IDS1010 <> NULL AND ^IDS36 = NULL AND ^IDS37 = NULL
    
            Data Elements:
    
            ^IDS36 = IDS:RP:ServiceArrangements:TreasuryServices:Expense.Total.Amount
    
            ^IDS37 = IDS:RP:ServiceArrangements:TreasuryServices:Revenue.Total.Amount
    
            ^IDS1010 = IDS:RP:ServiceArrangements:TreasuryServices
            */
            assertion = (report.RP_ServiceArrangements_TreasuryServicesCollectionExists != false && report.IDS36 == null && report.IDS37 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Treasury related services information incomplete",
                    LongDescription = @"If service arrangements information has been completed for treasury related services then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:TreasuryServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440024" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS36", Value = GetValueOrEmpty(report.IDS36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS37", Value = GetValueOrEmpty(report.IDS37) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440025()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440025
            If service arrangements information has been completed for Management and administration services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1011 <> NULL AND ^IDS40 = NULL AND ^IDS41 = NULL

            Technical Business Rule Format:
            ^IDS1011 <> NULL AND ^IDS40 = NULL AND ^IDS41 = NULL
    
            Data Elements:
    
            ^IDS40 = IDS:RP:ServiceArrangements:ManagementAndAdministrationServices:Expense.Total.Amount
    
            ^IDS41 = IDS:RP:ServiceArrangements:ManagementAndAdministrationServices:Revenue.Total.Amount
    
            ^IDS1011 = IDS:RP:ServiceArrangements:ManagementAndAdministrationServices
            */
            assertion = (report.RP_ServiceArrangements_ManagementAndAdministrationServicesCollectionExists != false && report.IDS40 == null && report.IDS41 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Management and administration services information incomplete",
                    LongDescription = @"If service arrangements information has been completed for management and administration services then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:ManagementAndAdministrationServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440025" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS40", Value = GetValueOrEmpty(report.IDS40) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS41", Value = GetValueOrEmpty(report.IDS41) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440026
            If service arrangements information has been completed for Insurance services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1012 <> NULL AND ^IDS44 = NULL AND ^IDS45 = NULL

            Technical Business Rule Format:
            ^IDS1012 <> NULL AND ^IDS44 = NULL AND ^IDS45 = NULL
    
            Data Elements:
    
            ^IDS44 = IDS:RP:ServiceArrangements:InsuranceServices:Expense.Total.Amount
    
            ^IDS45 = IDS:RP:ServiceArrangements:InsuranceServices:Revenue.Total.Amount
    
            ^IDS1012 = IDS:RP:ServiceArrangements:InsuranceServices
            */
            assertion = (report.RP_ServiceArrangements_InsuranceServicesCollectionExists != false && report.IDS44 == null && report.IDS45 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440026",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Insurance information incomplete",
                    LongDescription = @"If service arrangements information has been completed for insurance then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:InsuranceServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440026" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS44", Value = GetValueOrEmpty(report.IDS44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS45", Value = GetValueOrEmpty(report.IDS45) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440027
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440027()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440027
            If service arrangements information has been completed for Reinsurance services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1013 <> NULL AND ^IDS48 = NULL AND ^IDS49 = NULL

            Technical Business Rule Format:
            ^IDS1013 <> NULL AND ^IDS48 = NULL AND ^IDS49 = NULL
    
            Data Elements:
    
            ^IDS48 = IDS:RP:ServiceArrangements:ReinsuranceServices:Expense.Total.Amount
    
            ^IDS49 = IDS:RP:ServiceArrangements:ReinsuranceServices:Revenue.Total.Amount
    
            ^IDS1013 = IDS:RP:ServiceArrangements:ReinsuranceServices
            */
            assertion = (report.RP_ServiceArrangements_ReinsuranceServicesCollectionExists != false && report.IDS48 == null && report.IDS49 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reinsurance information incomplete",
                    LongDescription = @"If service arrangements information has been completed for reinsurance then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:ReinsuranceServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440027" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS48", Value = GetValueOrEmpty(report.IDS48) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS49", Value = GetValueOrEmpty(report.IDS49) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440028
            If service arrangements information has been completed for Research and development services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1014 <> NULL AND ^IDS52 = NULL AND ^IDS53 = NULL

            Technical Business Rule Format:
            ^IDS1014 <> NULL AND ^IDS52 = NULL AND ^IDS53 = NULL
    
            Data Elements:
    
            ^IDS52 = IDS:RP:ServiceArrangements:RAndDServices:Expense.Total.Amount
    
            ^IDS53 = IDS:RP:ServiceArrangements:RAndDServices:Revenue.Total.Amount
    
            ^IDS1014 = IDS:RP:ServiceArrangements:RAndDServices
            */
            assertion = (report.RP_ServiceArrangements_RAndDServicesCollectionExists != false && report.IDS52 == null && report.IDS53 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Research and development information incomplete",
                    LongDescription = @"If service arrangements information has been completed for research and development then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:RAndDServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440028" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS52", Value = GetValueOrEmpty(report.IDS52) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS53", Value = GetValueOrEmpty(report.IDS53) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440029
            If service arrangements information has been completed for Sales and marketing services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1015 <> NULL AND ^IDS56 = NULL AND ^IDS57 = NULL

            Technical Business Rule Format:
            ^IDS1015 <> NULL AND ^IDS56 = NULL AND ^IDS57 = NULL
    
            Data Elements:
    
            ^IDS56 = IDS:RP:ServiceArrangements:SalesAndMarketingServices:Expense.Total.Amount
    
            ^IDS57 = IDS:RP:ServiceArrangements:SalesAndMarketingServices:Revenue.Total.Amount
    
            ^IDS1015 = IDS:RP:ServiceArrangements:SalesAndMarketingServices
            */
            assertion = (report.RP_ServiceArrangements_SalesAndMarketingServicesCollectionExists != false && report.IDS56 == null && report.IDS57 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sales and marketing services information incomplete",
                    LongDescription = @"If service arrangements information has been completed for sales and marketing services then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:SalesAndMarketingServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440029" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS56", Value = GetValueOrEmpty(report.IDS56) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS57", Value = GetValueOrEmpty(report.IDS57) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440030
            If service arrangements information has been completed for Software and information technology services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1016 <> NULL AND ^IDS60 = NULL AND ^IDS61 = NULL

            Technical Business Rule Format:
            ^IDS1016 <> NULL AND ^IDS60 = NULL AND ^IDS61 = NULL
    
            Data Elements:
    
            ^IDS60 = IDS:RP:ServiceArrangements:SoftwareAndITServices:Expense.Total.Amount
    
            ^IDS61 = IDS:RP:ServiceArrangements:SoftwareAndITServices:Revenue.Total.Amount
    
            ^IDS1016 = IDS:RP:ServiceArrangements:SoftwareAndITServices
            */
            assertion = (report.RP_ServiceArrangements_SoftwareAndITServicesCollectionExists != false && report.IDS60 == null && report.IDS61 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Software and information technology services information incomplete",
                    LongDescription = @"If service arrangements information has been completed for software and information technology services then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:SoftwareAndITServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS60", Value = GetValueOrEmpty(report.IDS60) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS61", Value = GetValueOrEmpty(report.IDS61) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440031
            If service arrangements information has been completed for Technical services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1017 <> NULL AND ^IDS64 = NULL AND ^IDS65 = NULL

            Technical Business Rule Format:
            ^IDS1017 <> NULL AND ^IDS64 = NULL AND ^IDS65 = NULL
    
            Data Elements:
    
            ^IDS64 = IDS:RP:ServiceArrangements:TechnicalServices:Expense.Total.Amount
    
            ^IDS65 = IDS:RP:ServiceArrangements:TechnicalServices:Revenue.Total.Amount
    
            ^IDS1017 = IDS:RP:ServiceArrangements:TechnicalServices
            */
            assertion = (report.RP_ServiceArrangements_TechnicalServicesCollectionExists != false && report.IDS64 == null && report.IDS65 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Technical services information incomplete",
                    LongDescription = @"If service arrangements information has been completed for technical services then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:TechnicalServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS64", Value = GetValueOrEmpty(report.IDS64) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS65", Value = GetValueOrEmpty(report.IDS65) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440032
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440032()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440032
            If service arrangements information has been completed for Logistics services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1018 <> NULL AND ^IDS68 = NULL AND ^IDS69 = NULL

            Technical Business Rule Format:
            ^IDS1018 <> NULL AND ^IDS68 = NULL AND ^IDS69 = NULL
    
            Data Elements:
    
            ^IDS68 = IDS:RP:ServiceArrangements:LogisticsServices:Expense.Total.Amount
    
            ^IDS69 = IDS:RP:ServiceArrangements:LogisticsServices:Revenue.Total.Amount
    
            ^IDS1018 = IDS:RP:ServiceArrangements:LogisticsServices
            */
            assertion = (report.RP_ServiceArrangements_LogisticsServicesCollectionExists != false && report.IDS68 == null && report.IDS69 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440032",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Logistics information incomplete",
                    LongDescription = @"If service arrangements information has been completed for logistics then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:LogisticsServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440032" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS68", Value = GetValueOrEmpty(report.IDS68) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS69", Value = GetValueOrEmpty(report.IDS69) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440033
            If service arrangements information has been completed for Asset management services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1019 <> NULL AND ^IDS72 = NULL AND ^IDS73 = NULL

            Technical Business Rule Format:
            ^IDS1019 <> NULL AND ^IDS72 = NULL AND ^IDS73 = NULL
    
            Data Elements:
    
            ^IDS72 = IDS:RP:ServiceArrangements:AssetManagementServices:Expense.Total.Amount
    
            ^IDS73 = IDS:RP:ServiceArrangements:AssetManagementServices:Revenue.Total.Amount
    
            ^IDS1019 = IDS:RP:ServiceArrangements:AssetManagementServices
            */
            assertion = (report.RP_ServiceArrangements_AssetManagementServicesCollectionExists != false && report.IDS72 == null && report.IDS73 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Asset management information incomplete",
                    LongDescription = @"If service arrangements information has been completed for asset management then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:AssetManagementServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440033" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS72", Value = GetValueOrEmpty(report.IDS72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS73", Value = GetValueOrEmpty(report.IDS73) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440034
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440034()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440034
            If service arrangements information has been completed for Other services, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1020 <> NULL AND ^IDS76 = NULL AND ^IDS77 = NULL

            Technical Business Rule Format:
            ^IDS1020 <> NULL AND ^IDS76 = NULL AND ^IDS77 = NULL
    
            Data Elements:
    
            ^IDS76 = IDS:RP:ServiceArrangements:OtherServices:Expense.Total.Amount
    
            ^IDS77 = IDS:RP:ServiceArrangements:OtherServices:Revenue.Total.Amount
    
            ^IDS1020 = IDS:RP:ServiceArrangements:OtherServices
            */
            assertion = (report.RP_ServiceArrangements_OtherServicesCollectionExists != false && report.IDS76 == null && report.IDS77 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other services information incomplete",
                    LongDescription = @"If service arrangements information has been completed for asset management then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ServiceArrangements/tns:OtherServices/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440034" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS76", Value = GetValueOrEmpty(report.IDS76) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS77", Value = GetValueOrEmpty(report.IDS77) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440035
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440035()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440035
            If international related party dealings information has been completed for derivative transactions, then the response to 'Did you have any derivative transactions with international related parties?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS81 <> TRUE AND ^IDS1021 <> NULL

            Technical Business Rule Format:
            ^IDS81 <> TRUE AND ^IDS1021 <> NULL
    
            Data Elements:
    
            ^IDS81 = IDS:RP:InternationalDealings.DerivativeTransactions.Indicator
    
            ^IDS1021 = IDS:RP:Derivatives
            */
            assertion = (report.IDS81 != true && report.RP_DerivativesCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for derivative transactions then the response to 'Did you have any derivative transactions with international related parties?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsDerivativeTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440035" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS81", Value = GetValueOrEmpty(report.IDS81) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440036
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440036()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440036
            If there is a TRUE response to 'Did you have any derivative transactions with international related parties?', then the derivative transactions expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS81 = TRUE AND ^IDS82 = NULL AND ^IDS83 = NULL

            Technical Business Rule Format:
            ^IDS81 = TRUE AND ^IDS82 = NULL AND ^IDS83 = NULL
    
            Data Elements:
    
            ^IDS81 = IDS:RP:InternationalDealings.DerivativeTransactions.Indicator
    
            ^IDS82 = IDS:RP:Derivatives:Expense.Total.Amount
    
            ^IDS83 = IDS:RP:Derivatives:Revenue.Total.Amount
            */
            assertion = (report.IDS81 == true && report.IDS82 == null && report.IDS83 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Derivative transactions information incomplete",
                    LongDescription = @"If there is a 'True' response to 'Did you have any derivative transactions with international related parties?' then expenditure or revenue, main pricing methodology code, percentage of dealings with documentation code and principal derivative type code must be completed, and a response must be entered at 'Did you engage in the trading of derivatives globally through a trading structure in which you shared global profits from these activities with international related parties?'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsDerivativeTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440036" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS81", Value = GetValueOrEmpty(report.IDS81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS82", Value = GetValueOrEmpty(report.IDS82) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS83", Value = GetValueOrEmpty(report.IDS83) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440037()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440037
            If international related party dealings information has been completed for debt factoring or securitisation arrangements, then the response to 'Did you enter into any debt factoring or securitisation arrangements with international related parties?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS88 <> TRUE AND (^IDS1022 <> NULL OR ^IDS1023 <> NULL)

            Technical Business Rule Format:
            ^IDS88 <> TRUE AND (^IDS1022 <> NULL OR ^IDS1023 <> NULL)
    
            Data Elements:
    
            ^IDS88 = IDS:RP:InternationalDealings.DebtFactoringSecuritisation.Indicator
    
            ^IDS1022 = IDS:RP:DebtFactoring
    
            ^IDS1023 = IDS:RP:Securitisation
            */
            assertion = (report.IDS88 != true && (report.RP_DebtFactoringCollectionExists != false || report.RP_SecuritisationCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for debt factoring or securitisation then the response to 'Did you enter into any debt factoring or securitisation arrangements with international related parties?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsDebtFactoringSecuritisationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440037" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS88", Value = GetValueOrEmpty(report.IDS88) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440038()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440038
            If there is a TRUE response to 'Did you have any international related party dealings involving royalties or licence fees?', then international related party dealings information must be completed for Debt factoring and/or Securitisation and at least one of the Book value or Consideration amounts must be completed.
    
            Legacy Rule Format:
            ^IDS88 = TRUE AND ((^IDS1022 = NULL AND ^IDS1023 = NULL) OR (^IDS1022 <> NULL AND ^IDS89 = NULL AND ^IDS90 = NULL) OR (^IDS1023 <> NULL AND ^IDS93 = NULL AND ^IDS94 = NULL))

            Technical Business Rule Format:
            ^IDS88 = TRUE AND ((^IDS1022 = NULL AND ^IDS1023 = NULL) OR (^IDS1022 <> NULL AND ^IDS89 = NULL AND ^IDS90 = NULL) OR (^IDS1023 <> NULL AND ^IDS93 = NULL AND ^IDS94 = NULL))
    
            Data Elements:
    
            ^IDS88 = IDS:RP:InternationalDealings.DebtFactoringSecuritisation.Indicator
    
            ^IDS89 = IDS:RP:DebtFactoring:Expense.BookValue.Amount
    
            ^IDS90 = IDS:RP:DebtFactoring:Expense.Consideration.Amount
    
            ^IDS93 = IDS:RP:Securitisation:Expense.BookValue.Amount
    
            ^IDS94 = IDS:RP:Securitisation:Expense.Consideration.Amount
    
            ^IDS1022 = IDS:RP:DebtFactoring
    
            ^IDS1023 = IDS:RP:Securitisation
            */
            assertion = (report.IDS88 == true && (report.RP_DebtFactoringCollectionExists == false && report.RP_SecuritisationCollectionExists == false || report.RP_DebtFactoringCollectionExists != false && report.IDS89 == null && report.IDS90 == null || report.RP_SecuritisationCollectionExists != false && report.IDS93 == null && report.IDS94 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440690",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required.",
                    LongDescription = @"If there is a TRUE response to 'Did you have any international related party dealings involving royalties or licence fees?', then international related party dealings information must be completed for Debt factoring and/or Securitisation and at least one of the Book value or Consideration amounts must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsDebtFactoringSecuritisationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440038" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS88", Value = GetValueOrEmpty(report.IDS88) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS89", Value = GetValueOrEmpty(report.IDS89) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS90", Value = GetValueOrEmpty(report.IDS90) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS93", Value = GetValueOrEmpty(report.IDS93) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS94", Value = GetValueOrEmpty(report.IDS94) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440045
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440045(IDS2025.RP_ForeignExchange foreignExchange, IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss gainOrLoss, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440045
            If international related party dealings information has been completed for financial dealings for amounts borrowed, amounts loaned, interest, guarantees, insurance, reinsurance, foreign exchange or other financial dealings, then the response to 'Did you have any dealings of a financial nature other than those listed under derivative transactions or debt factoring or securitisation arrangements with international related parties?' must be TRUE.

            Legacy Rule Format:
            ^IDS97 <> TRUE AND (^IDS2002 <> NULL OR ^IDS336 <> NULL OR ^IDS337 <> NULL OR ^IDS338 <> NULL OR ^IDS339 <> NULL)

            Technical Business Rule Format:
            ^IDS97 <> TRUE AND (^IDS2002 <> NULL OR ^IDS336 <> NULL OR ^IDS337 <> NULL OR ^IDS338 <> NULL OR ^IDS339 <> NULL)

            Data Elements:

            ^IDS97 = IDS:RP:InternationalDealings.OtherFinancialDealings.Indicator

            ^IDS336 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeLossesHighestNominal.Amount

            ^IDS337 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeGainsHighestNominal.Amount

            ^IDS338 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeLossesOther.Amount

            ^IDS339 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeGainsOther.Amount

            ^IDS2002 = IDS:RP:FinancialDealings
            */
            assertion = (report.IDS97 != true && (report.RP_FinancialDealingsCollectionExists != false || gainOrLoss.IDS336 != null || gainOrLoss.IDS337 != null || foreignExchange.IDS338 != null || foreignExchange.IDS339 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440685",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect.",
                    LongDescription = @"If international related party dealings information has been completed for financial dealings for amounts borrowed, amounts loaned, interest, guarantees, insurance, reinsurance, foreign exchange or other financial dealings, then the response to 'Did you have any dealings of a financial nature other than those listed under derivative transactions or debt factoring or securitisation arrangements with international related parties?' must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsOtherFinancialDealingsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440045" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS97", Value = GetValueOrEmpty(report.IDS97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS336", Value = GetValueOrEmpty(gainOrLoss.IDS336) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS337", Value = GetValueOrEmpty(gainOrLoss.IDS337) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS338", Value = GetValueOrEmpty(foreignExchange.IDS338) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS339", Value = GetValueOrEmpty(foreignExchange.IDS339) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440046(IDS2025.RP_ForeignExchange foreignExchange, IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss gainOrLoss, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440046
            If there is a TRUE response to 'Did you have any dealings of a financial nature other than those listed under derivative transactions or debt factoring or securitisation arrangements with international related parties?', then the international related party dealings information must be completed for financial dealings for amounts borrowed, amounts loaned, interest, guarantees, insurance, reinsurance, foreign exchange or other financial dealings.

            Legacy Rule Format:
            ^IDS97 = TRUE AND ^IDS2002 = NULL AND ^IDS336 = NULL AND ^IDS337 = NULL AND ^IDS338 = NULL AND ^IDS339 = NULL

            Technical Business Rule Format:
            ^IDS97 = TRUE AND ^IDS2002 = NULL AND ^IDS336 = NULL AND ^IDS337 = NULL AND ^IDS338 = NULL AND ^IDS339 = NULL

            Data Elements:

            ^IDS97 = IDS:RP:InternationalDealings.OtherFinancialDealings.Indicator

            ^IDS336 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeLossesHighestNominal.Amount

            ^IDS337 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeGainsHighestNominal.Amount

            ^IDS338 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeLossesOther.Amount

            ^IDS339 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeGainsOther.Amount

            ^IDS2002 = IDS:RP:FinancialDealings
            */
            assertion = (report.IDS97 == true && report.RP_FinancialDealingsCollectionExists == false && gainOrLoss.IDS336 == null && gainOrLoss.IDS337 == null && foreignExchange.IDS338 == null && foreignExchange.IDS339 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440699",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required.",
                    LongDescription = @"If there is a TRUE response to 'Did you have any dealings of a financial nature other than those listed under derivative transactions or debt factoring or securitisation arrangements with international related parties?', then the international related party dealings information must be completed for financial dealings for amounts borrowed, amounts loaned, interest, guarantees, insurance, reinsurance, foreign exchange or other financial dealings.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsOtherFinancialDealingsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440046" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS97", Value = GetValueOrEmpty(report.IDS97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS336", Value = GetValueOrEmpty(gainOrLoss.IDS336) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS337", Value = GetValueOrEmpty(gainOrLoss.IDS337) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS338", Value = GetValueOrEmpty(foreignExchange.IDS338) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS339", Value = GetValueOrEmpty(foreignExchange.IDS339) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440047
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440047()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440047
            If average balance of amounts borrowed or amounts loaned is greater than zero for interest bearing loans, then interest expenditure or interest revenue must be completed.
    
            Legacy Rule Format:
            (^IDS98 > 0 OR ^IDS100 > 0) AND ^IDS102 = NULL AND ^IDS103 = NULL

            Technical Business Rule Format:
            (^IDS98 > 0 OR ^IDS100 > 0) AND ^IDS102 = NULL AND ^IDS103 = NULL
    
            Data Elements:
    
            ^IDS98 = IDS:RP:FinancialDealings:InterestBearingLoans:Liabilities.BorrowedAverageBalance.Amount
    
            ^IDS100 = IDS:RP:FinancialDealings:InterestBearingLoans:Assets.LoanedAverageBalance.Amount
    
            ^IDS102 = IDS:RP:FinancialDealings:Interest:Expense.Total.Amount
    
            ^IDS103 = IDS:RP:FinancialDealings:Interest:Revenue.Total.Amount
            */
            assertion = ((report.IDS98.GetValueOrDefault() > 0 || report.IDS100.GetValueOrDefault() > 0) && report.IDS102 == null && report.IDS103 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440047",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest bearing loans information incomplete",
                    LongDescription = @"If dealings of a financial nature information has been completed for interest bearing loans, either amounts borrowed or amounts loaned, then an expenditure or revenue amount must be completed for interest",
                    Location = "/tns:IDS/tns:RP/tns:FinancialDealings/tns:InterestBearingLoans/tns:LiabilitiesBorrowedAverageBalanceA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440047" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS98", Value = GetValueOrEmpty(report.IDS98) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS100", Value = GetValueOrEmpty(report.IDS100) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS102", Value = GetValueOrEmpty(report.IDS102) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS103", Value = GetValueOrEmpty(report.IDS103) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440052
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440052()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440052
            If dealings of a financial nature information has been completed for Interest, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1026 <> NULL AND ^IDS102 = NULL AND ^IDS103 = NULL

            Technical Business Rule Format:
            ^IDS1026 <> NULL AND ^IDS102 = NULL AND ^IDS103 = NULL
    
            Data Elements:
    
            ^IDS102 = IDS:RP:FinancialDealings:Interest:Expense.Total.Amount
    
            ^IDS103 = IDS:RP:FinancialDealings:Interest:Revenue.Total.Amount
    
            ^IDS1026 = IDS:RP:FinancialDealings:Interest
            */
            assertion = (report.RP_FinancialDealings_InterestCollectionExists != false && report.IDS102 == null && report.IDS103 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440052",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest information incomplete",
                    LongDescription = @"If dealings of a financial nature information has been completed for interest then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialDealings/tns:Interest/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440052" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS102", Value = GetValueOrEmpty(report.IDS102) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS103", Value = GetValueOrEmpty(report.IDS103) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440053()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440053
            If dealings of a financial nature information has been completed for Guarantees, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1027 <> NULL AND ^IDS106 = NULL AND ^IDS107 = NULL

            Technical Business Rule Format:
            ^IDS1027 <> NULL AND ^IDS106 = NULL AND ^IDS107 = NULL
    
            Data Elements:
    
            ^IDS106 = IDS:RP:FinancialDealings:Guarantees:Expense.Total.Amount
    
            ^IDS107 = IDS:RP:FinancialDealings:Guarantees:Revenue.Total.Amount
    
            ^IDS1027 = IDS:RP:FinancialDealings:Guarantees
            */
            assertion = (report.RP_FinancialDealings_GuaranteesCollectionExists != false && report.IDS106 == null && report.IDS107 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Guarantees information incomplete",
                    LongDescription = @"If dealings of a financial nature information has been completed for guarantees then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialDealings/tns:Guarantees/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440053" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS106", Value = GetValueOrEmpty(report.IDS106) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS107", Value = GetValueOrEmpty(report.IDS107) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440054
            If dealings of a financial nature information has been completed for Insurance, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1028 <> NULL AND ^IDS110 = NULL AND ^IDS111 = NULL

            Technical Business Rule Format:
            ^IDS1028 <> NULL AND ^IDS110 = NULL AND ^IDS111 = NULL
    
            Data Elements:
    
            ^IDS110 = IDS:RP:FinancialDealings:Insurance:Expense.Total.Amount
    
            ^IDS111 = IDS:RP:FinancialDealings:Insurance:Revenue.Total.Amount
    
            ^IDS1028 = IDS:RP:FinancialDealings:Insurance
            */
            assertion = (report.RP_FinancialDealings_InsuranceCollectionExists != false && report.IDS110 == null && report.IDS111 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Insurance information incomplete",
                    LongDescription = @"If dealings of a financial nature information has been completed for insurance then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialDealings/tns:Insurance/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440054" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS110", Value = GetValueOrEmpty(report.IDS110) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS111", Value = GetValueOrEmpty(report.IDS111) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440055
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440055()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440055
            If dealings of a financial nature information has been completed for Reinsurance, then expenditure or revenue amount must be completed.
    
            Legacy Rule Format:
            ^IDS1029 <> NULL AND ^IDS114 = NULL AND ^IDS115 = NULL

            Technical Business Rule Format:
            ^IDS1029 <> NULL AND ^IDS114 = NULL AND ^IDS115 = NULL
    
            Data Elements:
    
            ^IDS114 = IDS:RP:FinancialDealings:Reinsurance:Expense.Total.Amount
    
            ^IDS115 = IDS:RP:FinancialDealings:Reinsurance:Revenue.Total.Amount
    
            ^IDS1029 = IDS:RP:FinancialDealings:Reinsurance
            */
            assertion = (report.RP_FinancialDealings_ReinsuranceCollectionExists != false && report.IDS114 == null && report.IDS115 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440055",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reinsurance information incomplete",
                    LongDescription = @"If dealings of a financial nature information has been completed for reinsurance then expenditure or revenue, main pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialDealings/tns:Reinsurance/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440055" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS114", Value = GetValueOrEmpty(report.IDS114) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS115", Value = GetValueOrEmpty(report.IDS115) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440056
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440056()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440056
            If dealings of a financial nature information has been completed for other financial dealings then expenditure or revenue must be completed.
    
            Legacy Rule Format:
            ^IDS1032 <> NULL AND ^IDS118 = NULL AND ^IDS119 = NULL

            Technical Business Rule Format:
            ^IDS1032 <> NULL AND ^IDS118 = NULL AND ^IDS119 = NULL
    
            Data Elements:
    
            ^IDS118 = IDS:RP:FinancialDealings:OtherFinancial:Expense.Total.Amount
    
            ^IDS119 = IDS:RP:FinancialDealings:OtherFinancial:Revenue.Total.Amount
    
            ^IDS1032 = IDS:RP:FinancialDealings:OtherFinancial
            */
            assertion = (report.RP_FinancialDealings_OtherFinancialCollectionExists != false && report.IDS118 == null && report.IDS119 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440056",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other financial dealings information incomplete",
                    LongDescription = @"If dealings of a financial nature information has been completed for other financial dealings then expenditure or revenue, main pricing methodology code, percentage of dealings with documentation code and description must be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialDealings/tns:OtherFinancial/tns:ExpenseTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440056" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS118", Value = GetValueOrEmpty(report.IDS118) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS119", Value = GetValueOrEmpty(report.IDS119) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440057
            If international related party dealings information that could not be classified under any of the specific categories have been completed for other dealings of a revenue nature, then the response to 'Did you have any other international related party dealings of a revenue nature not reported elsewhere?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS123 <> TRUE AND ^IDS1033 <> NULL

            Technical Business Rule Format:
            ^IDS123 <> TRUE AND ^IDS1033 <> NULL
    
            Data Elements:
    
            ^IDS123 = IDS:RP:InternationalDealings.OtherRevenue.Indicator
    
            ^IDS1033 = IDS:RP:RevenueDealings
            */
            assertion = (report.IDS123 != true && report.RP_RevenueDealingsCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for other dealings of a revenue nature then the response to 'Did you have any other international related party dealings of a revenue nature?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsOtherRevenueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440057" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS123", Value = GetValueOrEmpty(report.IDS123) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440058
            If there is a TRUE response to 'Did you have any other international related party dealings of a revenue nature not reported elsewhere?', then expenditure or revenue must be completed for those other dealings.
    
            Legacy Rule Format:
            ^IDS123 = TRUE AND ^IDS124 = NULL AND ^IDS125 = NULL

            Technical Business Rule Format:
            ^IDS123 = TRUE AND ^IDS124 = NULL AND ^IDS125 = NULL
    
            Data Elements:
    
            ^IDS123 = IDS:RP:InternationalDealings.OtherRevenue.Indicator
    
            ^IDS124 = IDS:RP:RevenueDealings:Expense.Total.Amount
    
            ^IDS125 = IDS:RP:RevenueDealings:Revenue.Total.Amount
            */
            assertion = (report.IDS123 == true && report.IDS124 == null && report.IDS125 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other dealings of a revenue nature information incomplete",
                    LongDescription = @"If there is a 'True' response to 'Did you have any other international related party dealings of a revenue nature?' then expenditure or revenue, main pricing methodology code, percentage of dealings with documentation code and description must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsOtherRevenueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440058" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS123", Value = GetValueOrEmpty(report.IDS123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS124", Value = GetValueOrEmpty(report.IDS124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS125", Value = GetValueOrEmpty(report.IDS125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440059
            If international related party dealings information has been completed for any tangible or intangible property, then the response to 'Did you dispose of or acquire any tangible/intangible property of a non-revenue (capital) nature to or from international related parties not reported elsewhere?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS129 <> TRUE AND ^IDS2003 <> NULL

            Technical Business Rule Format:
            ^IDS129 <> TRUE AND ^IDS2003 <> NULL
    
            Data Elements:
    
            ^IDS129 = IDS:RP:Miscellaneous.CapitalNature.Indicator
    
            ^IDS2003 = IDS:RP:DisposalOrAcquisition
            */
            assertion = (report.IDS129 != true && report.RP_DisposalOrAcquisitionCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440457",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for tangible, assignment of intellectual property, assignment of shares or other equity interests, assignment of loans or debts or intangible property then the response to 'Did you dispose of or acquire any tangible/intangible property of a non-revenue (capital) nature to or from international related parties?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousCapitalNatureI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440059" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS129", Value = GetValueOrEmpty(report.IDS129) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440060
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440060()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440060
            If there is a TRUE response to 'Did you dispose of or acquire any tangible/intangible property of a non-revenue (capital) nature to or from international related parties not reported elsewhere?', then the disposal or acquisition information must be completed.
    
            Legacy Rule Format:
            ^IDS129 = TRUE AND ^IDS2003 = NULL

            Technical Business Rule Format:
            ^IDS129 = TRUE AND ^IDS2003 = NULL
    
            Data Elements:
    
            ^IDS129 = IDS:RP:Miscellaneous.CapitalNature.Indicator
    
            ^IDS2003 = IDS:RP:DisposalOrAcquisition
            */
            assertion = (report.IDS129 == true && report.RP_DisposalOrAcquisitionCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440686",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required.",
                    LongDescription = @"If there is a TRUE response to 'Did you dispose of or acquire any tangible/intangible property of a non-revenue (capital) nature to or from international related parties not reported elsewhere?', then the disposal or acquisition information must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousCapitalNatureI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440060" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS129", Value = GetValueOrEmpty(report.IDS129) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440061
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440061()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440061
            If there has been any disposal or acquisition of tangible property, then the consideration paid or consideration received amount must be completed.
    
            Legacy Rule Format:
            ^IDS1034 <> NULL AND ^IDS130 = NULL AND ^IDS131 = NULL

            Technical Business Rule Format:
            ^IDS1034 <> NULL AND ^IDS130 = NULL AND ^IDS131 = NULL
    
            Data Elements:
    
            ^IDS130 = IDS:RP:DisposalOrAcquisition:TangiblePropertyNonRevenue:Assets.ConsiderationPaid.Amount
    
            ^IDS131 = IDS:RP:DisposalOrAcquisition:TangiblePropertyNonRevenue:Assets.ConsiderationReceived.Amount
    
            ^IDS1034 = IDS:RP:DisposalOrAcquisition:TangiblePropertyNonRevenue
            */
            assertion = (report.RP_DisposalOrAcquisition_TangiblePropertyNonRevenueCollectionExists != false && report.IDS130 == null && report.IDS131 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440061",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tangible property information incomplete",
                    LongDescription = @"If tangible property information has been completed then consideration paid or consideration received, main capital asset pricing methodology code and percentage of dealings with documentation code must be completed",
                    Location = "/tns:IDS/tns:RP/tns:DisposalOrAcquisition/tns:TangiblePropertyNonRevenue/tns:AssetsConsiderationPaidA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440061" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS130", Value = GetValueOrEmpty(report.IDS130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS131", Value = GetValueOrEmpty(report.IDS131) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440064
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440064()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440064
            If there has been any other disposal or acquisition of intangible property, then the consideration paid or consideration received amount must be completed.
    
            Legacy Rule Format:
            ^IDS1038 <> NULL AND ^IDS134 = NULL AND ^IDS135 = NULL

            Technical Business Rule Format:
            ^IDS1038 <> NULL AND ^IDS134 = NULL AND ^IDS135 = NULL
    
            Data Elements:
    
            ^IDS134 = IDS:RP:DisposalOrAcquisition:IntangibleProperty:Assets.ConsiderationPaid.Amount
    
            ^IDS135 = IDS:RP:DisposalOrAcquisition:IntangibleProperty:Assets.ConsiderationReceived.Amount
    
            ^IDS1038 = IDS:RP:DisposalOrAcquisition:IntangibleProperty
            */
            assertion = (report.RP_DisposalOrAcquisition_IntangiblePropertyCollectionExists != false && report.IDS134 == null && report.IDS135 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440443",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other intangible property information incomplete",
                    Location = "/tns:IDS/tns:RP/tns:DisposalOrAcquisition/tns:IntangibleProperty/tns:AssetsConsiderationPaidA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440064" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS134", Value = GetValueOrEmpty(report.IDS134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS135", Value = GetValueOrEmpty(report.IDS135) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440067
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440067()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440067
            If international related party dealings information has been completed for no payment or non-monetary payment, then the response to 'Did you provide any services or assets or liabilities or property – tangible or intangible (eg. intellectual property) or processes or rights or obligations for no payment or a non-monetary payment, or receive any of the same for a non-monetary payment to/from an international related party?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS138 <> TRUE AND (^IDS1040 <> NULL OR ^IDS1041 <> NULL OR ^IDS1042 <> NULL OR ^IDS1043 <> NULL)

            Technical Business Rule Format:
            ^IDS138 <> TRUE AND (^IDS1040 <> NULL OR ^IDS1041 <> NULL OR ^IDS1042 <> NULL OR ^IDS1043 <> NULL)
    
            Data Elements:
    
            ^IDS138 = IDS:RP:Miscellaneous.ReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsideration.Indicator
    
            ^IDS1040 = IDS:RP:CapitalNoPayment
    
            ^IDS1041 = IDS:RP:RevenueNoPayment
    
            ^IDS1042 = IDS:RP:CapitalNonMonetaryPayment
    
            ^IDS1043 = IDS:RP:RevenueNonMonetaryPayment
            */
            assertion = (report.IDS138 != true && (report.RP_CapitalNoPaymentCollectionExists != false || report.RP_RevenueNoPaymentCollectionExists != false || report.RP_CapitalNonMonetaryPaymentCollectionExists != false || report.RP_RevenueNonMonetaryPaymentCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440067",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for No payment or non-monetary payment then the response to 'Did you provide any of the following for no payment or a non-monetary payment, or receive any of the following for a non-monetary payment to/from an international related party?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440067" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS138", Value = GetValueOrEmpty(report.IDS138) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440068
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440068()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440068
            If there is a TRUE response to 'Did you provide any services or assets or liabilities or property – tangible or intangible (eg. intellectual property) or processes or rights or obligations for no payment or a non-monetary payment, or receive any of the same for a non-monetary payment to/from an international related party?', then no payment or non-monetary payment information must be completed.
    
            Legacy Rule Format:
            ^IDS138 = TRUE AND ^IDS1040 = NULL AND ^IDS1041 = NULL AND ^IDS1042 = NULL AND ^IDS1043 = NULL

            Technical Business Rule Format:
            ^IDS138 = TRUE AND ^IDS1040 = NULL AND ^IDS1041 = NULL AND ^IDS1042 = NULL AND ^IDS1043 = NULL
    
            Data Elements:
    
            ^IDS138 = IDS:RP:Miscellaneous.ReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsideration.Indicator
    
            ^IDS1040 = IDS:RP:CapitalNoPayment
    
            ^IDS1041 = IDS:RP:RevenueNoPayment
    
            ^IDS1042 = IDS:RP:CapitalNonMonetaryPayment
    
            ^IDS1043 = IDS:RP:RevenueNonMonetaryPayment
            */
            assertion = (report.IDS138 == true && report.RP_CapitalNoPaymentCollectionExists == false && report.RP_RevenueNoPaymentCollectionExists == false && report.RP_CapitalNonMonetaryPaymentCollectionExists == false && report.RP_RevenueNonMonetaryPaymentCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required",
                    LongDescription = @"If there is a 'True' response to 'Did you provide any of the following for no payment or a non-monetary payment, or receive any of the following for a non-monetary payment to/from an international related party?' then No payment or non-monetary payment information must be completed",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440068" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS138", Value = GetValueOrEmpty(report.IDS138) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440073
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440073()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440073
            If international related party dealings information has been completed for recharge amounts, then the response to 'Did you provide or receive share-based employee remuneration to or from an international related party?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS147 <> TRUE AND ^IDS1044 <> NULL

            Technical Business Rule Format:
            ^IDS147 <> TRUE AND ^IDS1044 <> NULL
    
            Data Elements:
    
            ^IDS147 = IDS:RP:InternationalDealings.ShareBasedEmployeeRemuneration.Indicator
    
            ^IDS1044 = IDS:RP:ShareBasedRemuneration
            */
            assertion = (report.IDS147 != true && report.RP_ShareBasedRemunerationCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect",
                    LongDescription = @"If International related party dealings information has been completed for recharge amounts then the response to 'Did you provide or receive share-based employee remuneration to or from an international related party?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsShareBasedEmployeeRemunerationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440073" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS147", Value = GetValueOrEmpty(report.IDS147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440074
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440074()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440074
            If there is a TRUE response to 'Did you provide or receive share-based employee remuneration to or from an international related party?', then recharge amounts must be completed.
    
            Legacy Rule Format:
            ^IDS147 = TRUE AND ^IDS1044 = NULL

            Technical Business Rule Format:
            ^IDS147 = TRUE AND ^IDS1044 = NULL
    
            Data Elements:
    
            ^IDS147 = IDS:RP:InternationalDealings.ShareBasedEmployeeRemuneration.Indicator
    
            ^IDS1044 = IDS:RP:ShareBasedRemuneration
            */
            assertion = (report.IDS147 == true && report.RP_ShareBasedRemunerationCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information required",
                    LongDescription = @"If there is a 'True' response to 'Did you provide or receive share-based employee remuneration to or from an international related party?' then recharge amounts information must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsShareBasedEmployeeRemunerationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440074" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS147", Value = GetValueOrEmpty(report.IDS147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440075
            If Restructuring event information has been provided, then the response to 'Did a restructuring event(s) occur in the current year involving international related parties or your branch operations?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS151 <> TRUE AND Count(^IDS1045) <> 0

            Technical Business Rule Format:
            ^IDS151 <> TRUE AND Count(^IDS1045) <> 0
    
            Data Elements:
    
            ^IDS151 = IDS:RP:InternationalDealings.RestructuringEvents.Indicator
    
            ^IDS1045 = IDS:RP:RestructuringEvent
            */
            assertion = (report.IDS151 != true && Count(report.RP_RestructuringEventCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440704",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings question incorrect.",
                    LongDescription = @"If Restructuring event information has been provided, then the response to 'Did a restructuring event(s) occur in the current year involving international related parties or your branch operations?' must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRestructuringEventsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440075" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS151", Value = GetValueOrEmpty(report.IDS151) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440076
            If there is a TRUE response to 'Did a restructuring event(s) occur in the current year involving international related parties or your branch operations?', then restructuring event information must be completed.
    
            Legacy Rule Format:
            ^IDS151 = TRUE AND Count(^IDS1045) = 0

            Technical Business Rule Format:
            ^IDS151 = TRUE AND Count(^IDS1045) = 0
    
            Data Elements:
    
            ^IDS151 = IDS:RP:InternationalDealings.RestructuringEvents.Indicator
    
            ^IDS1045 = IDS:RP:RestructuringEvent
            */
            assertion = (report.IDS151 == true && Count(report.RP_RestructuringEventCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440076",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Branch operations restructuring event information is required",
                    LongDescription = @"If there is a 'True' response to 'Did a restructuring event occur in the current year involving international related parties or your branch operations?' then restructuring event information must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRestructuringEventsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440076" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS151", Value = GetValueOrEmpty(report.IDS151) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440080
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440080()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440080
            If there is a FALSE response to 'Do you have any branch operations (including Australian branch operations if you are a non-resident)?', then amounts claimed or returned for internally recorded dealings with branch operations must not be completed.
    
            Legacy Rule Format:
            ^IDS160 = FALSE AND (^IDS1046 <> NULL OR ^IDS1047 <> NULL OR ^IDS1048 <> NULL OR ^IDS1049 <> NULL)

            Technical Business Rule Format:
            ^IDS160 = FALSE AND (^IDS1046 <> NULL OR ^IDS1047 <> NULL OR ^IDS1048 <> NULL OR ^IDS1049 <> NULL)
    
            Data Elements:
    
            ^IDS160 = IDS:RP:InternationalDealings.BranchOperations.Indicator
    
            ^IDS1046 = IDS:RP:BranchOperationsInterestBearingLoans
    
            ^IDS1047 = IDS:RP:BranchOperationsInterestFreeLoans
    
            ^IDS1048 = IDS:RP:InternalTrading
    
            ^IDS1049 = IDS:RP:BranchOperationsOther
            */
            assertion = (report.IDS160 == false && (report.RP_BranchOperationsInterestBearingLoansCollectionExists != false || report.RP_BranchOperationsInterestFreeLoansCollectionExists != false || report.RP_InternalTradingCollectionExists != false || report.RP_BranchOperationsOtherCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440697",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Branch operations information is not required.",
                    LongDescription = @"If there is a FALSE response to 'Do you have any branch operations (including Australian branch operations if you are a non-resident)?', then total amount claimed or returned for other internally recorded dealings with branch operations must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsBranchOperationsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440080" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS160", Value = GetValueOrEmpty(report.IDS160) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440081
            If there is a TRUE response to 'Do you have any branch operations (including Australian branch operations if you are a non-resident)?', then amounts claimed or returned for internally recorded dealings with branch operations must be completed.
    
            Legacy Rule Format:
            ^IDS160 = TRUE AND ^IDS1046 = NULL AND ^IDS1047 = NULL AND ^IDS1048 = NULL AND ^IDS1049 = NULL

            Technical Business Rule Format:
            ^IDS160 = TRUE AND ^IDS1046 = NULL AND ^IDS1047 = NULL AND ^IDS1048 = NULL AND ^IDS1049 = NULL
    
            Data Elements:
    
            ^IDS160 = IDS:RP:InternationalDealings.BranchOperations.Indicator
    
            ^IDS1046 = IDS:RP:BranchOperationsInterestBearingLoans
    
            ^IDS1047 = IDS:RP:BranchOperationsInterestFreeLoans
    
            ^IDS1048 = IDS:RP:InternalTrading
    
            ^IDS1049 = IDS:RP:BranchOperationsOther
            */
            assertion = (report.IDS160 == true && report.RP_BranchOperationsInterestBearingLoansCollectionExists == false && report.RP_BranchOperationsInterestFreeLoansCollectionExists == false && report.RP_InternalTradingCollectionExists == false && report.RP_BranchOperationsOtherCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440698",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Branch operations information is required.",
                    LongDescription = @"If there is a TRUE response to 'Do you have any branch operations (including Australian branch operations if you are a non-resident)?', then total amount claimed or returned for other internally recorded dealings with branch operations must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsBranchOperationsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440081" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS160", Value = GetValueOrEmpty(report.IDS160) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440083
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440083()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440083
            If average balance of amounts borrowed is provided for interest bearing loans, then interest of amounts borrowed must be completed and vice-versa.
    
            Legacy Rule Format:
            (^IDS161 <> NULL AND ^IDS163 = NULL) OR (^IDS163 <> NULL AND ^IDS161 = NULL)

            Technical Business Rule Format:
            (^IDS161 <> NULL AND ^IDS163 = NULL) OR (^IDS163 <> NULL AND ^IDS161 = NULL)
    
            Data Elements:
    
            ^IDS161 = IDS:RP:BranchOperationsInterestBearingLoans:Liabilities.BorrowedAverageBalance.Amount
    
            ^IDS163 = IDS:RP:BranchOperationsInterestBearingLoans:Expense.InterestBorrowedTotal.Amount
            */
            assertion = (report.IDS161 != null && report.IDS163 == null || report.IDS163 != null && report.IDS161 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440083",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest bearing loans Amounts borrowed information incomplete",
                    LongDescription = @"If internally recorded dealings with branch operations information has been completed for interest bearing loans, either amounts borrowed average balance or amounts borrowed interest, then both of these must be completed",
                    Location = "/tns:IDS/tns:RP/tns:BranchOperationsInterestBearingLoans/tns:LiabilitiesBorrowedAverageBalanceA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440083" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS161", Value = GetValueOrEmpty(report.IDS161) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS163", Value = GetValueOrEmpty(report.IDS163) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440095
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440095()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440095
            If there is a FALSE response to 'Did you hold any financing arrangement during the income year?', then average quarterly balance of debt interests or equity interests information must not be completed.
    
            Legacy Rule Format:
            ^IDS172 = FALSE AND ^IDS1050 <> NULL

            Technical Business Rule Format:
            ^IDS172 = FALSE AND ^IDS1050 <> NULL
    
            Data Elements:
    
            ^IDS172 = IDS:RP:InternationalDealings.FinancialArrangements.Indicator
    
            ^IDS1050 = IDS:RP:FinancialArrangements
            */
            assertion = (report.IDS172 == false && report.RP_FinancialArrangementsCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440095",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial arrangements information not required",
                    LongDescription = @"If there is a 'False' response to 'Did you hold any financing arrangements during the income year?' then average quarterly balance of debt interests or equity interests information should not be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsFinancialArrangementsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440095" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS172", Value = GetValueOrEmpty(report.IDS172) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440096
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440096()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440096
            If there is a TRUE response to 'Did you hold any financing arrangement during the income year?', then average quarterly balance of debt interests or equity interests information must be completed.
    
            Legacy Rule Format:
            ^IDS172 = TRUE AND ^IDS1050 = NULL

            Technical Business Rule Format:
            ^IDS172 = TRUE AND ^IDS1050 = NULL
    
            Data Elements:
    
            ^IDS172 = IDS:RP:InternationalDealings.FinancialArrangements.Indicator
    
            ^IDS1050 = IDS:RP:FinancialArrangements
            */
            assertion = (report.IDS172 == true && report.RP_FinancialArrangementsCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440096",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial arrangements information required",
                    LongDescription = @"If there is a 'True' response to 'Did you hold any financing arrangements during the income year?' then average quarterly balance of debt interests or equity interests information must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsFinancialArrangementsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440096" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS172", Value = GetValueOrEmpty(report.IDS172) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440102
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440102()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440102
            If there is a FALSE response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?', then tax timing method elections made must not be completed.
    
            Legacy Rule Format:
            ^IDS177 = FALSE AND Count(^IDS178) <> 0

            Technical Business Rule Format:
            ^IDS177 = FALSE AND Count(^IDS178) <> 0
    
            Data Elements:
    
            ^IDS177 = IDS:RP:InternationalDealings.TaxationOfFinancialArrangementsRulesApplied.Indicator
    
            ^IDS178 = IDS:RP:InternationalDealings.TaxTimingMethodElections.Code
            */
            assertion = (report.IDS177 == false && Count(report.IDS178Collection) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440102",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial arrangements information not required",
                    LongDescription = @"If there is a 'False' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' then the tax timing method elections you have made should not be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440102" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS177", Value = GetValueOrEmpty(report.IDS177) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440103
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440103()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440103
            If there is a TRUE response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?', then tax timing method elections made must be completed.
    
            Legacy Rule Format:
            ^IDS177 = TRUE AND Count(^IDS178) = 0

            Technical Business Rule Format:
            ^IDS177 = TRUE AND Count(^IDS178) = 0
    
            Data Elements:
    
            ^IDS177 = IDS:RP:InternationalDealings.TaxationOfFinancialArrangementsRulesApplied.Indicator
    
            ^IDS178 = IDS:RP:InternationalDealings.TaxTimingMethodElections.Code
            */
            assertion = (report.IDS177 == true && Count(report.IDS178Collection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440103",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial arrangements information required",
                    LongDescription = @"If there is a 'True' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' then the tax timing method elections you have made must be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440103" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS177", Value = GetValueOrEmpty(report.IDS177) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440105
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440105()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440105
            If there is a FALSE response to 'Did you have any interests in Controlled Foreign Companies (CFCs) or Controlled Foreign Trusts (CFTs)?', then CFC and CFT information must not be completed.
    
            Legacy Rule Format:
            ^IDS184 = FALSE AND (^IDS367 <> NULL OR ^IDS371 <> NULL OR ^IDS1052 <> NULL OR ^IDS1053 <> NULL OR ^IDS1054 <> NULL OR ^IDS433 <> NULL OR ^IDS434 <> NULL OR ^IDS435 <> NULL OR ^IDS1055 <> NULL OR ^IDS1056 <> NULL OR ^IDS1057 <> NULL OR ^IDS439 <> NULL OR ^IDS191 <> NULL OR ^IDS192 <> NULL OR ^IDS193 <> NULL OR ^IDS446 <> NULL OR ^IDS450 <> NULL OR ^IDS388 <> NULL OR ^IDS454 <> NULL)

            Technical Business Rule Format:
            ^IDS184 = FALSE AND (^IDS367 <> NULL OR ^IDS371 <> NULL OR ^IDS1052 <> NULL OR ^IDS1053 <> NULL OR ^IDS1054 <> NULL OR ^IDS433 <> NULL OR ^IDS434 <> NULL OR ^IDS435 <> NULL OR ^IDS1055 <> NULL OR ^IDS1056 <> NULL OR ^IDS1057 <> NULL OR ^IDS439 <> NULL OR ^IDS191 <> NULL OR ^IDS192 <> NULL OR ^IDS193 <> NULL OR ^IDS446 <> NULL OR ^IDS450 <> NULL OR ^IDS388 <> NULL OR ^IDS454 <> NULL)
    
            Data Elements:
    
            ^IDS184 = IDS:RP:Miscellaneous.InterestInControlledForeignCompaniesAndControlledForeignTrusts.Indicator
    
            ^IDS191 = IDS:RP:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
    
            ^IDS192 = IDS:RP:Income.ControlledForeignCompaniesChangeOfResidenceIncome.Amount
    
            ^IDS193 = IDS:RP:Income.ControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotal.Amount
    
            ^IDS367 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Indicator
    
            ^IDS371 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Indicator
    
            ^IDS388 = IDS:RP:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS433 = IDS:RP:Miscellaneous.ControlledForeignCompaniesExcludedTaintedInterestIncome.Indicator
    
            ^IDS434 = IDS:RP:Miscellaneous.ControlledForeignCompaniesBankingLicences.Indicator
    
            ^IDS435 = IDS:RP:Miscellaneous.ControlledForeignCompaniesIncomeMoneyLending.Indicator
    
            ^IDS439 = IDS:RP:Income.ControlledForeignCompaniesExcludedTaintedInterestIncomeTotal.Amount
    
            ^IDS446 = IDS:RP:Income.GrossRevenueActiveIncomeTestSatisfiedTotal.Amount
    
            ^IDS450 = IDS:RP:Income.GrossRevenueActiveIncomeTestNotSatisfiedTotal.Amount
    
            ^IDS454 = IDS:RP:Income.ControlledForeignCompaniesNotionalAllowableDeductionsTotal.Amount
    
            ^IDS1052 = IDS:RP:Listed
    
            ^IDS1053 = IDS:RP:Specified
    
            ^IDS1054 = IDS:RP:Unlisted
    
            ^IDS1055 = IDS:RP:ListedCFCAmounts
    
            ^IDS1056 = IDS:RP:SpecifiedCFCAmounts
    
            ^IDS1057 = IDS:RP:UnlistedCFCAmounts
            */
            assertion = (report.IDS184 == false && (report.IDS367 != null || report.IDS371 != null || report.RP_ListedCollectionExists != false || report.RP_SpecifiedCollectionExists != false || report.RP_UnlistedCollectionExists != false || report.IDS433 != null || report.IDS434 != null || report.IDS435 != null || report.RP_ListedCFCAmountsCollectionExists != false || report.RP_SpecifiedCFCAmountsCollectionExists != false || report.RP_UnlistedCFCAmountsCollectionExists != false || report.IDS439 != null || report.IDS191 != null || report.IDS192 != null || report.IDS193 != null || report.IDS446 != null || report.IDS450 != null || report.IDS388 != null || report.IDS454 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities information not required",
                    LongDescription = @"If there is a 'False' response to 'Did you have any interests in controlled foreign companies (CFCs) or controlled foreign trusts (CFTs)?' then the number of CFCs and CFTs which you had an interest in and amounts included in assessable income under sections 456, 457 or 459A should not be completed, and a response at 'Did you have a CFC that was a resident of an unlisted country that provided a benefit (as defined in section 47A of the ITAA 1936), either directly or indirectly, to you or any of your related entities at any time during the income year?' should not be completed",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440105" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS184", Value = GetValueOrEmpty(report.IDS184) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS367", Value = GetValueOrEmpty(report.IDS367) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS371", Value = GetValueOrEmpty(report.IDS371) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS433", Value = GetValueOrEmpty(report.IDS433) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS434", Value = GetValueOrEmpty(report.IDS434) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS435", Value = GetValueOrEmpty(report.IDS435) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS439", Value = GetValueOrEmpty(report.IDS439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS191", Value = GetValueOrEmpty(report.IDS191) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS192", Value = GetValueOrEmpty(report.IDS192) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS193", Value = GetValueOrEmpty(report.IDS193) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS446", Value = GetValueOrEmpty(report.IDS446) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS450", Value = GetValueOrEmpty(report.IDS450) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS388", Value = GetValueOrEmpty(report.IDS388) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS454", Value = GetValueOrEmpty(report.IDS454) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440106
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440106()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440106
            If there is a TRUE response to 'Did you have any interests in Controlled Foreign Companies (CFCs) or Controlled Foreign Trusts (CFTs)?', then CFC and CFT information must be completed.
    
            Legacy Rule Format:
            ^IDS184 = TRUE AND ((^IDS185 = NULL AND ^IDS186 = NULL AND ^IDS187 = NULL) OR (^IDS188 = NULL AND ^IDS189 = NULL AND ^IDS190 = NULL AND ^IDS191 = NULL AND ^IDS192 = NULL AND ^IDS193 = NULL AND ^IDS367 = NULL AND ^IDS371 = NULL AND ^IDS375 = NULL AND ^IDS376 = NULL AND ^IDS377 = NULL AND ^IDS433 = NULL AND ^IDS378 = NULL AND ^IDS379 = NULL AND ^IDS380 = NULL AND ^IDS381 = NULL AND ^IDS382 = NULL AND ^IDS383 = NULL AND ^IDS384 = NULL AND ^IDS385 = NULL AND ^IDS386 = NULL AND ^IDS387 = NULL AND ^IDS388 = NULL AND ^IDS451 = NULL AND ^IDS452 = NULL AND ^IDS453 = NULL AND ^IDS454 = NULL))

            Technical Business Rule Format:
            ^IDS184 = TRUE AND ((^IDS185 = NULL AND ^IDS186 = NULL AND ^IDS187 = NULL) OR (^IDS188 = NULL AND ^IDS189 = NULL AND ^IDS190 = NULL AND ^IDS191 = NULL AND ^IDS192 = NULL AND ^IDS193 = NULL AND ^IDS367 = NULL AND ^IDS371 = NULL AND ^IDS375 = NULL AND ^IDS376 = NULL AND ^IDS377 = NULL AND ^IDS433 = NULL AND ^IDS378 = NULL AND ^IDS379 = NULL AND ^IDS380 = NULL AND ^IDS381 = NULL AND ^IDS382 = NULL AND ^IDS383 = NULL AND ^IDS384 = NULL AND ^IDS385 = NULL AND ^IDS386 = NULL AND ^IDS387 = NULL AND ^IDS388 = NULL AND ^IDS451 = NULL AND ^IDS452 = NULL AND ^IDS453 = NULL AND ^IDS454 = NULL))
    
            Data Elements:
    
            ^IDS184 = IDS:RP:Miscellaneous.InterestInControlledForeignCompaniesAndControlledForeignTrusts.Indicator
    
            ^IDS185 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompaniesTrustsInterest.Count
    
            ^IDS186 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompaniesTrustsInterest.Count
    
            ^IDS187 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompaniesTrustsInterest.Count
    
            ^IDS188 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^IDS189 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^IDS190 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^IDS191 = IDS:RP:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
    
            ^IDS192 = IDS:RP:Income.ControlledForeignCompaniesChangeOfResidenceIncome.Amount
    
            ^IDS193 = IDS:RP:Income.ControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotal.Amount
    
            ^IDS367 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Indicator
    
            ^IDS371 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Indicator
    
            ^IDS375 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
    
            ^IDS376 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
    
            ^IDS377 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
    
            ^IDS378 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedConcessionIncome.Amount
    
            ^IDS379 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedIncomeNotDerived.Amount
    
            ^IDS380 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesOtherNotionalAssessableIncome.Amount
    
            ^IDS381 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS382 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedIncome.Amount
    
            ^IDS383 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesOtherNotionalAssessableIncome.Amount
    
            ^IDS384 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS385 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedIncome.Amount
    
            ^IDS386 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesOtherNotionalAssessableIncome.Amount
    
            ^IDS387 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS388 = IDS:RP:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS433 = IDS:RP:Miscellaneous.ControlledForeignCompaniesExcludedTaintedInterestIncome.Indicator
    
            ^IDS451 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesNotionalAllowableDeductions.Amount
    
            ^IDS452 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesNotionalAllowableDeductions.Amount
    
            ^IDS453 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesNotionalAllowableDeductions.Amount
    
            ^IDS454 = IDS:RP:Income.ControlledForeignCompaniesNotionalAllowableDeductionsTotal.Amount
            */
            assertion = (report.IDS184 == true && (report.IDS185 == null && report.IDS186 == null && report.IDS187 == null || report.IDS188 == null && report.IDS189 == null && report.IDS190 == null && report.IDS191 == null && report.IDS192 == null && report.IDS193 == null && report.IDS367 == null && report.IDS371 == null && report.IDS375 == null && report.IDS376 == null && report.IDS377 == null && report.IDS433 == null && report.IDS378 == null && report.IDS379 == null && report.IDS380 == null && report.IDS381 == null && report.IDS382 == null && report.IDS383 == null && report.IDS384 == null && report.IDS385 == null && report.IDS386 == null && report.IDS387 == null && report.IDS388 == null && report.IDS451 == null && report.IDS452 == null && report.IDS453 == null && report.IDS454 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440106",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities information required",
                    LongDescription = @"If there is a 'True' response to 'Did you have any interests in controlled foreign companies (CFCs) or controlled foreign trusts (CFTs)?' then the number of CFCs and CFTs which you had an interest in and amounts included in assessable income under sections 456, 457 or 459A must be completed, and a response at 'Did you have a CFC that was a resident of an unlisted country that provided a benefit (as defined in section 47A of the ITAA 1936), either directly or indirectly, to you or any of your related entities at any time during the income year?' must be completed",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS184", Value = GetValueOrEmpty(report.IDS184) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS185", Value = GetValueOrEmpty(report.IDS185) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS186", Value = GetValueOrEmpty(report.IDS186) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS187", Value = GetValueOrEmpty(report.IDS187) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS188", Value = GetValueOrEmpty(report.IDS188) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS189", Value = GetValueOrEmpty(report.IDS189) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS190", Value = GetValueOrEmpty(report.IDS190) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS191", Value = GetValueOrEmpty(report.IDS191) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS192", Value = GetValueOrEmpty(report.IDS192) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS193", Value = GetValueOrEmpty(report.IDS193) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS367", Value = GetValueOrEmpty(report.IDS367) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS371", Value = GetValueOrEmpty(report.IDS371) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS375", Value = GetValueOrEmpty(report.IDS375) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS376", Value = GetValueOrEmpty(report.IDS376) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS377", Value = GetValueOrEmpty(report.IDS377) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS433", Value = GetValueOrEmpty(report.IDS433) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS378", Value = GetValueOrEmpty(report.IDS378) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS379", Value = GetValueOrEmpty(report.IDS379) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS380", Value = GetValueOrEmpty(report.IDS380) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS381", Value = GetValueOrEmpty(report.IDS381) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS382", Value = GetValueOrEmpty(report.IDS382) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS383", Value = GetValueOrEmpty(report.IDS383) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS384", Value = GetValueOrEmpty(report.IDS384) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS385", Value = GetValueOrEmpty(report.IDS385) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS386", Value = GetValueOrEmpty(report.IDS386) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS387", Value = GetValueOrEmpty(report.IDS387) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS388", Value = GetValueOrEmpty(report.IDS388) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS451", Value = GetValueOrEmpty(report.IDS451) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS452", Value = GetValueOrEmpty(report.IDS452) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS453", Value = GetValueOrEmpty(report.IDS453) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS454", Value = GetValueOrEmpty(report.IDS454) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440111
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440111()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440111
            Section 456 - Total CFCs attributable income must be equal to the sum of corresponding amount for Listed, Specified and Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS191 <> (^IDS188 + ^IDS189 + ^IDS190)

            Technical Business Rule Format:
            ^IDS191 <> (^IDS188 + ^IDS189 + ^IDS190)
    
            Data Elements:
    
            ^IDS191 = IDS:RP:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
    
            ^IDS188 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^IDS189 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^IDS190 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
            */
            assertion = (report.IDS191.GetValueOrDefault() != report.IDS188.GetValueOrDefault() + report.IDS189.GetValueOrDefault() + report.IDS190.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440111",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total of Section 456 CFCs attributable income incorrect",
                    LongDescription = @"Section 456 CFCs attributable income Total must equal the sum of listed countries, specified countries and other unlisted countries",
                    Location = "/tns:IDS/tns:RP/tns:IncomeControlledForeignCompaniesAttributableIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440111" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS191", Value = GetValueOrEmpty(report.IDS191) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS188", Value = GetValueOrEmpty(report.IDS188) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS189", Value = GetValueOrEmpty(report.IDS189) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS190", Value = GetValueOrEmpty(report.IDS190) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440115
            If there is a FALSE response to 'Do you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts?', then following must not be completed:
            - Section 23AH Foreign branch income or Section 23AI Amounts paid out of attributed CFC income or Subdivision 768-A foreign equity distributions for Listed, Specified or Other unlisted countries
            - Non-deductible expenses incurred in deriving section 23AH non-assessable income
            - Response to the question 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?'
            - Response to the question 'Did you have any Capital Gains Tax (CGT) events in relation to your interest in a foreign company?'
    
            Legacy Rule Format:
            ^IDS195 = FALSE AND (^IDS1058 <> NULL OR ^IDS1059 <> NULL OR ^IDS1060 <> NULL OR ^IDS199 <> NULL OR ^IDS206 <> NULL OR ^IDS208 <> NULL)

            Technical Business Rule Format:
            ^IDS195 = FALSE AND (^IDS1058 <> NULL OR ^IDS1059 <> NULL OR ^IDS1060 <> NULL OR ^IDS199 <> NULL OR ^IDS206 <> NULL OR ^IDS208 <> NULL)
    
            Data Elements:
    
            ^IDS195 = IDS:RP:Income.ForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrusts.Indicator
    
            ^IDS199 = IDS:RP:Expense.NonDeductibleExemptIncome.Amount
    
            ^IDS206 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncome.Indicator
    
            ^IDS208 = IDS:RP:CapitalGainsTax.Event.Indicator
    
            ^IDS1058 = IDS:RP:ListedForeignBranchOperations
    
            ^IDS1059 = IDS:RP:SpecifiedForeignBranchOperations
    
            ^IDS1060 = IDS:RP:UnlistedForeignBranchOperations
            */
            assertion = (report.IDS195 == false && (report.RP_ListedForeignBranchOperationsCollectionExists != false || report.RP_SpecifiedForeignBranchOperationsCollectionExists != false || report.RP_UnlistedForeignBranchOperationsCollectionExists != false || report.IDS199 != null || report.IDS206 != null || report.IDS208 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440688",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities information not required.",
                    LongDescription = @"If there is a FALSE response to 'Do you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts?', then following must not be completed:
- Section 23AH Foreign branch income or Section 23AI Amounts paid out of attributed CFC income or Subdivision 768-A foreign equity distributions for Listed, Specified or Other unlisted countries
- Non-deductible expenses incurred in deriving section 23AH non-assessable income
- Response to the question 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?'
- Response to the question 'Did you have any Capital Gains Tax (CGT) events in relation to your interest in a foreign company?'",
                    Location = "/tns:IDS/tns:RP/tns:IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440115" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS195", Value = GetValueOrEmpty(report.IDS195) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS199", Value = GetValueOrEmpty(report.IDS199) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS206", Value = GetValueOrEmpty(report.IDS206) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS208", Value = GetValueOrEmpty(report.IDS208) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440128
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440128()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440128
            If there has been debt deductions in earning non-assessable foreign income claimed under s25-90, then the response to 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS206 <> TRUE AND ^IDS207 <> NULL

            Technical Business Rule Format:
            ^IDS206 <> TRUE AND ^IDS207 <> NULL
    
            Data Elements:
    
            ^IDS206 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncome.Indicator
    
            ^IDS207 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncomeTotal.Amount
            */
            assertion = (report.IDS206 != true && report.IDS207 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440303",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities information completed incorrectly",
                    LongDescription = @"If 'Debt deductions in earning non-assessable non-exempt foreign income claimed' is greater than zero, a 'true' response is required for 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?'.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440128" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS206", Value = GetValueOrEmpty(report.IDS206) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS207", Value = GetValueOrEmpty(report.IDS207) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440129
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440129()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440129
            If there is a TRUE response to 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?', then 'Debt deductions in earning non-assessable non-exempt foreign income claimed' must be completed.
    
            Legacy Rule Format:
            ^IDS206 = TRUE AND ^IDS207 = NULL

            Technical Business Rule Format:
            ^IDS206 = TRUE AND ^IDS207 = NULL
    
            Data Elements:
    
            ^IDS206 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncome.Indicator
    
            ^IDS207 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncomeTotal.Amount
            */
            assertion = (report.IDS206 == true && report.IDS207 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440304",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities information required",
                    LongDescription = @"IIf there is a 'true' response to 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?' then 'Debt deductions in earning non-assessable non-exempt foreign income claimed' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440129" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS206", Value = GetValueOrEmpty(report.IDS206) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS207", Value = GetValueOrEmpty(report.IDS207) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440132()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440132
            If capital gain/loss amounts or capital gain/loss reductions has been provided, then the response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS208 <> TRUE AND ^IDS1061 <> NULL

            Technical Business Rule Format:
            ^IDS208 <> TRUE AND ^IDS1061 <> NULL
    
            Data Elements:
    
            ^IDS208 = IDS:RP:CapitalGainsTax.Event.Indicator
    
            ^IDS1061 = IDS:RP:CGTEvents
            */
            assertion = (report.IDS208 != true && report.RP_CGTEventsCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities question incorrect",
                    LongDescription = @"If Interests in foreign entities information has been completed for capital gain/loss amounts or capital gain/loss reductions then the response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:CapitalGainsTaxEventI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS208", Value = GetValueOrEmpty(report.IDS208) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440133()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440133
            If there is a TRUE response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?', then capital gain/loss amounts or capital gain/loss reductions must be completed.
    
            Legacy Rule Format:
            ^IDS208 = TRUE AND ^IDS1061 = NULL

            Technical Business Rule Format:
            ^IDS208 = TRUE AND ^IDS1061 = NULL
    
            Data Elements:
    
            ^IDS208 = IDS:RP:CapitalGainsTax.Event.Indicator
    
            ^IDS1061 = IDS:RP:CGTEvents
            */
            assertion = (report.IDS208 == true && report.RP_CGTEventsCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities information required",
                    LongDescription = @"If there is a 'True' response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?' then capital gain/loss amounts or capital gain/loss reductions must be completed",
                    Location = "/tns:IDS/tns:RP/tns:CapitalGainsTaxEventI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440133" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS208", Value = GetValueOrEmpty(report.IDS208) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440135()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440135
            Capital gain reductions must be less than or equal to Capital gain amounts.
    
            Legacy Rule Format:
            ^IDS210 > 0 AND ^IDS210 > ^IDS209

            Technical Business Rule Format:
            ^IDS210 > 0 AND ^IDS210 > ^IDS209
    
            Data Elements:
    
            ^IDS210 = IDS:RP:CGTEvents:IncomeTax.ActiveForeignBusinessCapitalGainsReduction.Amount
    
            ^IDS209 = IDS:RP:CGTEvents:CapitalGainsTax.ActiveForeignBusinessCapitalGains.Amount
            */
            assertion = (report.IDS210.GetValueOrDefault() > 0 && report.IDS210.GetValueOrDefault() > report.IDS209.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains reductions must not be greater than the Capital gains amounts",
                    LongDescription = @"If interests in foreign entities information has been completed for capital gains tax events, then the capital gains reductions must not be greater than the capital gains amounts",
                    Location = "/tns:IDS/tns:RP/tns:CGTEvents/tns:IncomeTaxActiveForeignBusinessCapitalGainsReductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS210", Value = GetValueOrEmpty(report.IDS210) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS209", Value = GetValueOrEmpty(report.IDS209) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440137()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440137
            Capital loss reductions must be less than or equal to Capital loss amounts.
    
            Legacy Rule Format:
            ^IDS212 > 0 AND ^IDS212 > ^IDS211

            Technical Business Rule Format:
            ^IDS212 > 0 AND ^IDS212 > ^IDS211
    
            Data Elements:
    
            ^IDS211 = IDS:RP:CGTEvents:CapitalGainsTax.ActiveForeignBusinessCapitalLosses.Amount
    
            ^IDS212 = IDS:RP:CGTEvents:IncomeTax.ActiveForeignBusinessCapitalLossesReduction.Amount
            */
            assertion = (report.IDS212.GetValueOrDefault() > 0 && report.IDS212.GetValueOrDefault() > report.IDS211.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital loss reductions must not be greater than the Capital loss amounts",
                    LongDescription = @"If interests in foreign entities information has been completed for capital gains tax events, then the capital loss reductions must not be greater than the capital loss amounts",
                    Location = "/tns:IDS/tns:RP/tns:CGTEvents/tns:CapitalGainsTaxActiveForeignBusinessCapitalLossesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440137" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS212", Value = GetValueOrEmpty(report.IDS212) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS211", Value = GetValueOrEmpty(report.IDS211) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440141()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440141
            If Interests in foreign entities information has been completed for FHLPs or FHCs, then the response to 'Were you a partner in a foreign hybrid limited partnership (FHLP) or shareholder in a foreign hybrid company (FHC)?' must be TRUE.
    
            Legacy Rule Format:
            (^IDS217 <> TRUE) AND (^IDS218 <> NULL OR ^IDS219 <> NULL)

            Technical Business Rule Format:
            (^IDS217 <> TRUE) AND (^IDS218 <> NULL OR ^IDS219 <> NULL)
    
            Data Elements:
    
            ^IDS217 = IDS:RP:Miscellaneous.ForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholder.Indicator
    
            ^IDS218 = IDS:RP:Miscellaneous.ForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestIn.Count
    
            ^IDS219 = IDS:RP:Assets.Investment.ForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotal.Amount
            */
            assertion = (report.IDS217 != true && (report.IDS218 != null || report.IDS219 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440141",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities question incorrect",
                    LongDescription = @"If Interests in foreign entities information has been completed for FHLPs or FHCs then the response to 'Were you a partner in a foreign hybrid limited partnership (FHLP) or shareholder in a foreign hybrid company (FHC)?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS217", Value = GetValueOrEmpty(report.IDS217) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS218", Value = GetValueOrEmpty(report.IDS218) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS219", Value = GetValueOrEmpty(report.IDS219) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440142()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440142
            If there is a TRUE response to 'Were you a partner in a foreign hybrid limited partnership (FHLP) or shareholder in a foreign hybrid company (FHC)?', then Number of FHLPs or FHCs you had an interest in and Total amount of your share of FHLP/FHCs net income/profit must be completed.
    
            Legacy Rule Format:
            ^IDS217 = TRUE AND (^IDS218 = NULL OR ^IDS219 = NULL)

            Technical Business Rule Format:
            ^IDS217 = TRUE AND (^IDS218 = NULL OR ^IDS219 = NULL)
    
            Data Elements:
    
            ^IDS217 = IDS:RP:Miscellaneous.ForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholder.Indicator
    
            ^IDS218 = IDS:RP:Miscellaneous.ForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestIn.Count
    
            ^IDS219 = IDS:RP:Assets.Investment.ForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotal.Amount
            */
            assertion = (report.IDS217 == true && (report.IDS218 == null || report.IDS219 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FHLP/FHC information incomplete",
                    LongDescription = @"If there is a 'True' response to 'Were you a partner in a foreign hybrid limited partnership (FHLP) or shareholder in a foreign hybrid company (FHC)?' then both the number of FHLPs or FHCs you had an interest in and the total amount of your share of net income/distribution of profit must be completed",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS217", Value = GetValueOrEmpty(report.IDS217) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS218", Value = GetValueOrEmpty(report.IDS218) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS219", Value = GetValueOrEmpty(report.IDS219) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440160
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440160()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440160
            If there is an amount greater than zero at 'Debt deductions on debt from related non-residents', then it must be less than or equal to General information - Debt deductions amount.
    
            Legacy Rule Format:
            (^IDS227 > 0) AND (^IDS227 > ^IDS226)

            Technical Business Rule Format:
            (^IDS227 > 0) AND (^IDS227 > ^IDS226)
    
            Data Elements:
    
            ^IDS227 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
            */
            assertion = (report.IDS227.GetValueOrDefault() > 0 && report.IDS227.GetValueOrDefault() > report.IDS226.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440160",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt deductions incorrect",
                    LongDescription = @"If there is an amount at 'General information - Debt deductions on debt from related non-residents'  it must be less than or equal to the amount at 'General information - Debt deductions'",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesRelatedNonResidentsDebtDeductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440160" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS227", Value = GetValueOrEmpty(report.IDS227) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440183
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440183()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440183
            If All non-ADI entities - Excess debt amount is greater than zero, then
            Debt deductions and Amount of debt deduction disallowed must be provided.
    
            Legacy Rule Format:
            ^IDS245 > 0 AND (^IDS226 = NULL OR ^IDS228 = NULL)

            Technical Business Rule Format:
            ^IDS245 > 0 AND (^IDS226 = NULL OR ^IDS228 = NULL)
    
            Data Elements:
    
            ^IDS245 = IDS:RP:ThinCapitalisation:AustralianPlantationForestryEntity:Liabilities.ThinCapitalisation.ExcessDebt.Amount
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
            */
            assertion = (report.IDS245.GetValueOrDefault() > 0 && (report.IDS226 == null || report.IDS228 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440183",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt deductions and/or amount of debt deduction disallowed required",
                    LongDescription = @"If 'All non-ADI entities - excess debt amount' is greater than zero then 'General information - debt deductions' and/or 'General information - amount of debt deduction disallowed' must be supplied.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:AustralianPlantationForestryEntity/tns:LiabilitiesExcessDebtA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440183" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS245", Value = GetValueOrEmpty(report.IDS245) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440197
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440197()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440197
            If thin capitalisation information has been completed for Worldwide ADIs or Worldwide Non-ADIs, then the response to 'Did you rely on worldwide gearing debt/capital tests?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS257 <> TRUE AND (^IDS1071 <> NULL OR ^IDS1072 <> NULL)

            Technical Business Rule Format:
            ^IDS257 <> TRUE AND (^IDS1071 <> NULL OR ^IDS1072 <> NULL)
    
            Data Elements:
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
    
            ^IDS1071 = IDS:RP:ThinCapitalisation:ADI
    
            ^IDS1072 = IDS:RP:ThinCapitalisation:NonADI
            */
            assertion = (report.IDS257 != true && (report.RP_ThinCapitalisation_ADICollectionExists != false || report.RP_ThinCapitalisation_NonADICollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440197",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation question incorrect",
                    LongDescription = @"If Thin capitalisation information has been completed for worldwide gearing debt/capital tests then the response to 'Did you rely on worldwide gearing debt/capital tests?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesGearingDebtCapitalTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440197" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440198
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440198()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440198
            If there is a TRUE response to 'Did you rely on worldwide gearing debt/capital tests?', then applicable thin capitalisation information must be completed for Worldwide ADIs and Worldwide Non-ADIs.
    
            Legacy Rule Format:
            ^IDS257 = TRUE AND ^IDS1071 = NULL AND ^IDS1072 = NULL

            Technical Business Rule Format:
            ^IDS257 = TRUE AND ^IDS1071 = NULL AND ^IDS1072 = NULL
    
            Data Elements:
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
    
            ^IDS1071 = IDS:RP:ThinCapitalisation:ADI
    
            ^IDS1072 = IDS:RP:ThinCapitalisation:NonADI
            */
            assertion = (report.IDS257 == true && report.RP_ThinCapitalisation_ADICollectionExists == false && report.RP_ThinCapitalisation_NonADICollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440198",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation information required",
                    LongDescription = @"If there is a 'True' response to 'Did you rely on worldwide gearing debt/capital tests?' then all applicable worldwide gearing debt/capital test information must be completed",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesGearingDebtCapitalTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440198" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440203()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440203
            If there is a FALSE response to 'Are you a foreign bank or other qualifying financial entity that has not elected out of Part IIIB of the ITAA 1936?', then amounts under sections 160ZZZ, 160ZZZA, 160ZZZJ, 160ZZZE or 160ZZZF must not be completed.
    
            Legacy Rule Format:
            ^IDS263 = FALSE AND (^IDS1073 <> NULL OR ^IDS1074 <> NULL)

            Technical Business Rule Format:
            ^IDS263 = FALSE AND (^IDS1073 <> NULL OR ^IDS1074 <> NULL)
    
            Data Elements:
    
            ^IDS263 = IDS:RP:FinancialServices.ForeignBankOrQualifyingFinancialEntity.Indicator
    
            ^IDS1073 = IDS:RP:NotionalAmount
    
            ^IDS1074 = IDS:RP:NotionalDerivativeAndForeignExchange
            */
            assertion = (report.IDS263 == false && (report.RP_NotionalAmountCollectionExists != false || report.RP_NotionalDerivativeAndForeignExchangeCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.441203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial services entities information not required",
                    LongDescription = @"If there is a 'False' response to 'Are you a foreign bank or other qualifying financial entity that has not elected out of Part IIIB of the ITAA 1936?' then amounts under sections 160ZZZ, 160ZZZA, 160ZZZJ, 160ZZZE or 160ZZZF should not be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialServicesForeignBankOrQualifyingFinancialEntityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS263", Value = GetValueOrEmpty(report.IDS263) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440204
            If there is a TRUE response to 'Are you a foreign bank or other qualifying financial entity that has not elected out of Part IIIB of the ITAA 1936?', then amounts under sections 160ZZZ, 160ZZZA, 160ZZZJ, 160ZZZE and 160ZZZF must be completed.
    
            Legacy Rule Format:
            ^IDS263 = TRUE AND (^IDS1073 = NULL OR ^IDS1074 = NULL)

            Technical Business Rule Format:
            ^IDS263 = TRUE AND (^IDS1073 = NULL OR ^IDS1074 = NULL)
    
            Data Elements:
    
            ^IDS263 = IDS:RP:FinancialServices.ForeignBankOrQualifyingFinancialEntity.Indicator
    
            ^IDS1073 = IDS:RP:NotionalAmount
    
            ^IDS1074 = IDS:RP:NotionalDerivativeAndForeignExchange
            */
            assertion = (report.IDS263 == true && (report.RP_NotionalAmountCollectionExists == false || report.RP_NotionalDerivativeAndForeignExchangeCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.441204",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial services entities information required",
                    LongDescription = @"If there is a 'True' response to 'Are you a foreign bank or other qualifying financial entity that has not elected out of Part IIIB of the ITAA 1936?' then amounts under sections 160ZZZ, 160ZZZA, 160ZZZJ, 160ZZZE and 160ZZZF must be completed",
                    Location = "/tns:IDS/tns:RP/tns:FinancialServicesForeignBankOrQualifyingFinancialEntityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS263", Value = GetValueOrEmpty(report.IDS263) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440294
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440294(string IDS178, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440294
            If tax timing method election made code is 'G = No elections made', then no other codes must be completed.

            Legacy Rule Format:
            ^IDS178 = 'G = No elections made' AND (Count(^IDS178) > 1)

            Technical Business Rule Format:
            ^IDS178 = 'G = No elections made' AND (Count(^IDS178) > 1)

            Data Elements:

            ^IDS178 = IDS:RP:InternationalDealings.TaxTimingMethodElections.Code
            */
            assertion = (IDS178 == @"G = No elections made" && Count(report.IDS178Collection) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440294",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TOFA tax timing method elections incorrect",
                    LongDescription = @"If the tax timing method election for 'No elections made' is selected then no other tax timing method election should be completed",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTaxTimingMethodElectionsCCollection/tns:InternationalDealingsTaxTimingMethodElectionsC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440294" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS178", Value = GetValueOrEmpty(IDS178) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440295
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440295()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440295
            Tax timing method election made code must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS178)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS178)
    
            Data Elements:
    
            ^IDS178 = IDS:RP:InternationalDealings.TaxTimingMethodElections.Code
            */
            assertion = HasDuplicateValues(report.IDS178Collection);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440295",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TOFA tax timing method elections incorrect",
                    LongDescription = @"The tax timing method election code must be unique",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsTaxTimingMethodElectionsCCollection/tns:InternationalDealingsTaxTimingMethodElectionsC" + OccurrenceIndex(DuplicateValueIndex(report.IDS178Collection) + 1) + "",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440295" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440296
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440296()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440296
            If transfer amount(s) or appendix 8 exemption code(s) have been completed, then the response to 'During the last three income years (including the current income year) did you directly or indirectly transfer property, money or services to a non-resident trust, where that non-resident trust was still in existence during that income year?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS213 <> TRUE AND Count(^IDS1062) <> 0

            Technical Business Rule Format:
            ^IDS213 <> TRUE AND Count(^IDS1062) <> 0
    
            Data Elements:
    
            ^IDS213 = IDS:RP:Miscellaneous.TransferPropertyMoneyOrServices.Indicator
    
            ^IDS1062 = IDS:RP:Transfer
            */
            assertion = (report.IDS213 != true && Count(report.RP_TransferCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440296",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interests in foreign entities question incorrect",
                    LongDescription = @"If Interests in foreign entities information has been completed for transfers of property, money or services then the response to 'During the last three income years, including the current income year, did you directly or indirectly transfer property, money or services to a non-resident trust, where that non-resident trust was still in existence during that income year?' must be 'True'",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousTransferPropertyMoneyOrServicesI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440296" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS213", Value = GetValueOrEmpty(report.IDS213) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440297
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440297()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440297
            If there is a TRUE response to 'During the last three income years (including the current income year) did you directly or indirectly transfer property, money or services to a non-resident trust, where that non-resident trust was still in existence during that income year?', then transfer amounts and Appendix 8 exemption code fields for the three years must be completed.
    
            Legacy Rule Format:
            ^IDS213 = TRUE AND Count(^IDS1062) <> 3

            Technical Business Rule Format:
            ^IDS213 = TRUE AND Count(^IDS1062) <> 3
    
            Data Elements:
    
            ^IDS213 = IDS:RP:Miscellaneous.TransferPropertyMoneyOrServices.Indicator
    
            ^IDS1062 = IDS:RP:Transfer
            */
            assertion = (report.IDS213 == true && Count(report.RP_TransferCollectionCount) != 3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440701",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Transfers information incomplete",
                    LongDescription = @"If there is a TRUE response to 'During the last three income years (including the current income year) did you directly or indirectly transfer property, money or services to a non-resident trust, where that non-resident trust was still in existence during that income year?', then transfer amounts and Appendix 8 exemption code fields for the three years must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousTransferPropertyMoneyOrServicesI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440297" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS213", Value = GetValueOrEmpty(report.IDS213) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440308
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440308()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440308
            If there is a TRUE response to 'Did you have a Conduit Foreign Income balance and/or distribute amounts of Conduit Foreign Income during the income year?', then 'Balance of conduit foreign income' or 'Conduit foreign income distributed during income year' must be completed.
    
            Legacy Rule Format:
            ^IDS309 = TRUE AND ^IDS1102 = NULL

            Technical Business Rule Format:
            ^IDS309 = TRUE AND ^IDS1102 = NULL
    
            Data Elements:
    
            ^IDS309 = IDS:RP:InternationalDealings.ConduitForeignIncome.Indicator
    
            ^IDS1102 = IDS:RP:ConduitForeignIncome
            */
            assertion = (report.IDS309 == true && report.RP_ConduitForeignIncomeCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440308",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Balance of conduit foreign income' or 'Conduit foreign income distributed during income year' is required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsConduitForeignIncomeI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440308" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS309", Value = GetValueOrEmpty(report.IDS309) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440309
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440309()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440309
            If there is a TRUE response to 'Did you pay amounts of interest exempt from withholding tax during the income year?', then at least one of the following must have an amount greater than zero:
            - Interest to financial institution exempt from withholding under a DTA
            - Section 128F exempt interest paid
            - Section 128FA exempt interest paid
    
            Legacy Rule Format:
            ^IDS315 = TRUE AND ((^IDS316 = NULL OR ^IDS316 = 0) AND (^IDS318 = NULL OR ^IDS318 = 0) AND (^IDS319 = NULL OR ^IDS319 = 0))

            Technical Business Rule Format:
            ^IDS315 = TRUE AND ((^IDS316 = NULL OR ^IDS316 = 0) AND (^IDS318 = NULL OR ^IDS318 = 0) AND (^IDS319 = NULL OR ^IDS319 = 0))
    
            Data Elements:
    
            ^IDS315 = IDS:RP:InternationalDealings.WithholdingTaxExemptNonResidentInterestPaid.Indicator
    
            ^IDS316 = IDS:RP:InternationalDealings.WithholdingTaxExemptNonResidentInterestPaid.Amount
    
            ^IDS318 = IDS:RP:128F:InternationalDealings.WithholdingTaxExemptNonResidentInterestPaid.Amount
    
            ^IDS319 = IDS:RP:128FA:InternationalDealings.WithholdingTaxExemptNonResidentInterestPaid.Amount
            */
            assertion = (report.IDS315 == true && ((report.IDS316 == null || report.IDS316 == 0) && (report.IDS318 == null || report.IDS318 == 0) && (report.IDS319 == null || report.IDS319 == 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440309",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest to financial institution exempt from withholding under a DTA', 'Section 128F exempt interest paid' or 'Section 128FA exempt interest paid' must have a value.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440309" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS315", Value = GetValueOrEmpty(report.IDS315) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS316", Value = GetValueOrEmpty(report.IDS316) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS318", Value = GetValueOrEmpty(report.IDS318) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS319", Value = GetValueOrEmpty(report.IDS319) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440310
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440310()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440310
            DTA country code is required when 'Interest to financial institution exempt from withholding under a DTA' is provided.
    
            Legacy Rule Format:
            ^IDS316 <> NULL AND ^IDS317 = NULL

            Technical Business Rule Format:
            ^IDS316 <> NULL AND ^IDS317 = NULL
    
            Data Elements:
    
            ^IDS317 = IDS:RP:InternationalDealings.DoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountry.Code
    
            ^IDS316 = IDS:RP:InternationalDealings.WithholdingTaxExemptNonResidentInterestPaid.Amount
            */
            assertion = (report.IDS316 != null && report.IDS317 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440310",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"DTA Country Code' is required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsDoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440310" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS316", Value = GetValueOrEmpty(report.IDS316) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS317", Value = GetValueOrEmpty(report.IDS317) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440420
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440420()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440420
            If there is a TRUE response to 'Does Country-by-Country reporting apply to you?', then 'Have you lodged Part A of your local file at the same time as your income tax return?' must be completed.
    
            Legacy Rule Format:
            ^IDS330 = TRUE AND ^IDS331 = NULL

            Technical Business Rule Format:
            ^IDS330 = TRUE AND ^IDS331 = NULL
    
            Data Elements:
    
            ^IDS330 = IDS:RP:InternationalDealings.CountryByCountryReportApplicability.Indicator
    
            ^IDS331 = IDS:RP:InternationalDealings.LocalFilePartAVoluntaryLodgment.Indicator
            */
            assertion = (report.IDS330 == true && report.IDS331 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440420",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsCountryByCountryReportApplicabilityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440420" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS330", Value = GetValueOrEmpty(report.IDS330) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS331", Value = GetValueOrEmpty(report.IDS331) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440421
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440421()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440421
            If there is a FALSE response to 'Have you lodged Part A of your local file at the same time as your income tax return?', then 'Is the total of your international related party dealings over $2 million?' must be completed.
    
            Legacy Rule Format:
            ^IDS331 = FALSE AND ^IDS1 = NULL

            Technical Business Rule Format:
            ^IDS331 = FALSE AND ^IDS1 = NULL
    
            Data Elements:
    
            ^IDS331 = IDS:RP:InternationalDealings.LocalFilePartAVoluntaryLodgment.Indicator
    
            ^IDS1 = IDS:RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
            */
            assertion = (report.IDS331 == false && report.IDS1 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440421",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsLocalFilePartAVoluntaryLodgmentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440421" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS331", Value = GetValueOrEmpty(report.IDS331) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS1", Value = GetValueOrEmpty(report.IDS1) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440425
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440425()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440425
            Principle derivative types - Derivative code must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS332)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS332)
    
            Data Elements:
    
            ^IDS332 = IDS:RP:Derivatives:InternationalDealings.RelatedPartyTransactionType.Code
            */
            assertion = HasDuplicateValues(report.IDS332Collection);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440291",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Duplicate derivative code present for derivative transactions",
                    LongDescription = @"In derivative transactions information the same principal derivative type code has been entered more than once. Each derivative code can only appear once",
                    Location = "/tns:IDS/tns:RP/tns:Derivatives/tns:InternationalDealingsRelatedPartyTransactionTypeCCollection/tns:InternationalDealingsRelatedPartyTransactionTypeC" + OccurrenceIndex(DuplicateValueIndex(report.IDS332Collection) + 1) + "",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440425" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440436
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440436()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440436
            If there has been any assignment of intellectual property, then the consideration paid or consideration received amount must be completed.
    
            Legacy Rule Format:
            ^IDS1035 <> NULL AND ^IDS340 = NULL AND ^IDS341 = NULL

            Technical Business Rule Format:
            ^IDS1035 <> NULL AND ^IDS340 = NULL AND ^IDS341 = NULL
    
            Data Elements:
    
            ^IDS340 = IDS:RP:DisposalOrAcquisition:IntellectualProperty:Assets.ConsiderationPaid.Amount
    
            ^IDS341 = IDS:RP:DisposalOrAcquisition:IntellectualProperty:Assets.ConsiderationReceived.Amount
    
            ^IDS1035 = IDS:RP:DisposalOrAcquisition:IntellectualProperty
            */
            assertion = (report.RP_DisposalOrAcquisition_IntellectualPropertyCollectionExists != false && report.IDS340 == null && report.IDS341 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assignment of intellectual property information incomplete",
                    Location = "/tns:IDS/tns:RP/tns:DisposalOrAcquisition/tns:IntellectualProperty/tns:AssetsConsiderationPaidA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440436" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS340", Value = GetValueOrEmpty(report.IDS340) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS341", Value = GetValueOrEmpty(report.IDS341) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440439
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440439()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440439
            If there has been any assignment of shares or other equity interests, then the consideration paid or consideration received amount must be completed.
    
            Legacy Rule Format:
            ^IDS1036 <> NULL AND ^IDS344 = NULL AND ^IDS345 = NULL

            Technical Business Rule Format:
            ^IDS1036 <> NULL AND ^IDS344 = NULL AND ^IDS345 = NULL
    
            Data Elements:
    
            ^IDS344 = IDS:RP:DisposalOrAcquisition:SharesOrOtherEquityInterests:Assets.ConsiderationPaid.Amount
    
            ^IDS345 = IDS:RP:DisposalOrAcquisition:SharesOrOtherEquityInterests:Assets.ConsiderationReceived.Amount
    
            ^IDS1036 = IDS:RP:DisposalOrAcquisition:SharesOrOtherEquityInterests
            */
            assertion = (report.RP_DisposalOrAcquisition_SharesOrOtherEquityInterestsCollectionExists != false && report.IDS344 == null && report.IDS345 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assignment of shares or other equity interests information incomplete",
                    Location = "/tns:IDS/tns:RP/tns:DisposalOrAcquisition/tns:SharesOrOtherEquityInterests/tns:AssetsConsiderationPaidA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440439" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS344", Value = GetValueOrEmpty(report.IDS344) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS345", Value = GetValueOrEmpty(report.IDS345) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440442
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440442()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440442
            If there has been any assignment of loans or debts (not liabilities), then the consideration paid or consideration received amount must be completed.
    
            Legacy Rule Format:
            ^IDS1037 <> NULL AND ^IDS348 = NULL AND ^IDS349 = NULL

            Technical Business Rule Format:
            ^IDS1037 <> NULL AND ^IDS348 = NULL AND ^IDS349 = NULL
    
            Data Elements:
    
            ^IDS348 = IDS:RP:DisposalOrAcquisition:LoansOrDebts:Assets.ConsiderationPaid.Amount
    
            ^IDS349 = IDS:RP:DisposalOrAcquisition:LoansOrDebts:Assets.ConsiderationReceived.Amount
    
            ^IDS1037 = IDS:RP:DisposalOrAcquisition:LoansOrDebts
            */
            assertion = (report.RP_DisposalOrAcquisition_LoansOrDebtsCollectionExists != false && report.IDS348 == null && report.IDS349 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440687",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If there has been any assignment of loans or debts (not liabilities), then the consideration paid or consideration received amount must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:DisposalOrAcquisition/tns:LoansOrDebts/tns:AssetsConsiderationPaidA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440442" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS348", Value = GetValueOrEmpty(report.IDS348) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS349", Value = GetValueOrEmpty(report.IDS349) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440448
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440448()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440448
            If there is a TRUE response to 'Have you lodged Part A of your local file at the same time as your income tax return?', then international related party dealings information is not required.
    
            Legacy Rule Format:
            ^IDS331 = TRUE AND (^IDS1 <> NULL OR ^IDS151 <> NULL OR ^IDS159 <> NULL)

            Technical Business Rule Format:
            ^IDS331 = TRUE AND (^IDS1 <> NULL OR ^IDS151 <> NULL OR ^IDS159 <> NULL)
    
            Data Elements:
    
            ^IDS331 = IDS:RP:InternationalDealings.LocalFilePartAVoluntaryLodgment.Indicator
    
            ^IDS1 = IDS:RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
            ^IDS151 = IDS:RP:InternationalDealings.RestructuringEvents.Indicator
    
            ^IDS159 = IDS:RP:InternationalDealings.TransactionsRestructuringEventsAssetRevaluation.Indicator
            */
            assertion = (report.IDS331 == true && (report.IDS1 != null || report.IDS151 != null || report.IDS159 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information is not required",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsLocalFilePartAVoluntaryLodgmentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440448" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS331", Value = GetValueOrEmpty(report.IDS331) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS1", Value = GetValueOrEmpty(report.IDS1) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS151", Value = GetValueOrEmpty(report.IDS151) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS159", Value = GetValueOrEmpty(report.IDS159) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440506
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440506()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440506
            If there is a FALSE response to 'Did you acquire any interests in CFCs or CFTs during the income year?', then the number of acquired CFCs and CFTs for Listed, Specified or Other unlisted countries must not be completed.
    
            Legacy Rule Format:
            ^IDS367 = FALSE AND (^IDS368 <> NULL OR ^IDS369 <> NULL OR ^IDS370 <> NULL)

            Technical Business Rule Format:
            ^IDS367 = FALSE AND (^IDS368 <> NULL OR ^IDS369 <> NULL OR ^IDS370 <> NULL)
    
            Data Elements:
    
            ^IDS367 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Indicator
    
            ^IDS368 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Count
    
            ^IDS369 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Count
    
            ^IDS370 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Count
            */
            assertion = (report.IDS367 == false && (report.IDS368 != null || report.IDS369 != null || report.IDS370 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440506",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs and CFTs information not required",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompanyTrustInterestAcquiredI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440506" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS367", Value = GetValueOrEmpty(report.IDS367) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS368", Value = GetValueOrEmpty(report.IDS368) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS369", Value = GetValueOrEmpty(report.IDS369) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS370", Value = GetValueOrEmpty(report.IDS370) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440507
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440507()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440507
            If there is a TRUE response to 'Did you acquire any interests in CFCs or CFTs during the income year?', then the number of acquired CFCs and CFTs for Listed, Specified or Other unlisted countries must be completed.
    
            Legacy Rule Format:
            ^IDS367 = TRUE AND ^IDS368 = NULL AND ^IDS369 = NULL AND ^IDS370 = NULL

            Technical Business Rule Format:
            ^IDS367 = TRUE AND ^IDS368 = NULL AND ^IDS369 = NULL AND ^IDS370 = NULL
    
            Data Elements:
    
            ^IDS367 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Indicator
    
            ^IDS368 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Count
    
            ^IDS369 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Count
    
            ^IDS370 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompanyTrustInterestAcquired.Count
            */
            assertion = (report.IDS367 == true && report.IDS368 == null && report.IDS369 == null && report.IDS370 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440507",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs and CFTs information required",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompanyTrustInterestAcquiredI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440507" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS367", Value = GetValueOrEmpty(report.IDS367) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS368", Value = GetValueOrEmpty(report.IDS368) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS369", Value = GetValueOrEmpty(report.IDS369) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS370", Value = GetValueOrEmpty(report.IDS370) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440508
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440508()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440508
            If there is a FALSE response to 'Did you dispose of any interests in CFCs or CFTs during the income year?', then the number of disposed CFCs and CFTs for Listed, Specified or Other unlisted countries must not be completed.
    
            Legacy Rule Format:
            ^IDS371 = FALSE AND (^IDS372 <> NULL OR ^IDS373 <> NULL OR ^IDS374 <> NULL)

            Technical Business Rule Format:
            ^IDS371 = FALSE AND (^IDS372 <> NULL OR ^IDS373 <> NULL OR ^IDS374 <> NULL)
    
            Data Elements:
    
            ^IDS371 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Indicator
    
            ^IDS372 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Count
    
            ^IDS373 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Count
    
            ^IDS374 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Count
            */
            assertion = (report.IDS371 == false && (report.IDS372 != null || report.IDS373 != null || report.IDS374 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440506",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs and CFTs information not required",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompanyTrustInterestDisposedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440508" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS371", Value = GetValueOrEmpty(report.IDS371) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS372", Value = GetValueOrEmpty(report.IDS372) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS373", Value = GetValueOrEmpty(report.IDS373) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS374", Value = GetValueOrEmpty(report.IDS374) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440509
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440509()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440509
            If there is a TRUE response to 'Did you dispose of any interests in CFCs or CFTs during the income year?', then the number of disposed CFCs and CFTs for Listed, Specified or Other unlisted countries must be completed.
    
            Legacy Rule Format:
            ^IDS371 = TRUE AND ^IDS372 = NULL AND ^IDS373 = NULL AND ^IDS374 = NULL

            Technical Business Rule Format:
            ^IDS371 = TRUE AND ^IDS372 = NULL AND ^IDS373 = NULL AND ^IDS374 = NULL
    
            Data Elements:
    
            ^IDS371 = IDS:RP:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Indicator
    
            ^IDS372 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Count
    
            ^IDS373 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Count
    
            ^IDS374 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompanyTrustInterestDisposed.Count
            */
            assertion = (report.IDS371 == true && report.IDS372 == null && report.IDS373 == null && report.IDS374 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440507",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs and CFTs information required",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompanyTrustInterestDisposedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440509" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS371", Value = GetValueOrEmpty(report.IDS371) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS372", Value = GetValueOrEmpty(report.IDS372) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS373", Value = GetValueOrEmpty(report.IDS373) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS374", Value = GetValueOrEmpty(report.IDS374) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440510
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440510()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440510
            Subtotal notional assessable income - Listed countries must be equal to the sum of Adjusted tainted income that is eligible designated concession income - Listed countries, Adjusted tainted income not treated as derived from sources in listed countries - Listed countries and Other notional assessable income - Listed countries.
    
            Legacy Rule Format:
            ^IDS381 <> NULL AND OutsideRange(^IDS381, (^IDS378 + ^IDS379 + ^IDS380),1)

            Technical Business Rule Format:
            ^IDS381 <> NULL AND OutsideRange(^IDS381, (^IDS378 + ^IDS379 + ^IDS380),1)
    
            Data Elements:
    
            ^IDS381 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS378 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedConcessionIncome.Amount
    
            ^IDS379 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedIncomeNotDerived.Amount
    
            ^IDS380 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesOtherNotionalAssessableIncome.Amount
            */
            assertion = (report.IDS381 != null && OutsideRange(report.IDS381.GetValueOrDefault(), report.IDS378.GetValueOrDefault() + report.IDS379.GetValueOrDefault() + report.IDS380.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440510",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal notional assessable income is not equal to the sum of Amounts of notional assessable income for listed country CFC",
                    Location = "/tns:IDS/tns:RP/tns:ListedCFCAmounts/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440510" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS381", Value = GetValueOrEmpty(report.IDS381) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS378", Value = GetValueOrEmpty(report.IDS378) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS379", Value = GetValueOrEmpty(report.IDS379) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS380", Value = GetValueOrEmpty(report.IDS380) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440511
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440511()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440511
            Subtotal notional assessable income - Specified countries must be equal to the sum of Adjusted tainted income - Specified countries and Other notional assessable income - Specified countries.
    
            Legacy Rule Format:
            ^IDS384 <> NULL AND OutsideRange(^IDS384,(^IDS382 + ^IDS383),1)

            Technical Business Rule Format:
            ^IDS384 <> NULL AND OutsideRange(^IDS384,(^IDS382 + ^IDS383),1)
    
            Data Elements:
    
            ^IDS384 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS382 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedIncome.Amount
    
            ^IDS383 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesOtherNotionalAssessableIncome.Amount
            */
            assertion = (report.IDS384 != null && OutsideRange(report.IDS384.GetValueOrDefault(), report.IDS382.GetValueOrDefault() + report.IDS383.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440511",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal notional assessable income is not equal to the sum of Amounts of notional assessable income for specified country CFC",
                    Location = "/tns:IDS/tns:RP/tns:SpecifiedCFCAmounts/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440511" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS384", Value = GetValueOrEmpty(report.IDS384) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS382", Value = GetValueOrEmpty(report.IDS382) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS383", Value = GetValueOrEmpty(report.IDS383) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440512
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440512()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440512
            Subtotal notional assessable income - Other unlisted countries must be equal to the sum of Adjusted tainted income - Other unlisted countries and Other notional assessable income - Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS387 <> NULL AND OutsideRange(^IDS387,(^IDS385 + ^IDS386),1)

            Technical Business Rule Format:
            ^IDS387 <> NULL AND OutsideRange(^IDS387,(^IDS385 + ^IDS386),1)
    
            Data Elements:
    
            ^IDS387 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS385 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesAdjustedTaintedIncome.Amount
    
            ^IDS386 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesOtherNotionalAssessableIncome.Amount
            */
            assertion = (report.IDS387 != null && OutsideRange(report.IDS387.GetValueOrDefault(), report.IDS385.GetValueOrDefault() + report.IDS386.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440512",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal notional assessable income is not equal to the sum of Amounts of notional assessable income for other unlisted country CFC",
                    Location = "/tns:IDS/tns:RP/tns:UnlistedCFCAmounts/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440512" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS387", Value = GetValueOrEmpty(report.IDS387) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS385", Value = GetValueOrEmpty(report.IDS385) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS386", Value = GetValueOrEmpty(report.IDS386) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440513
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440513()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440513
            Total notional assessable income must be equal to the sum of Subtotal notional assessable income for Listed, Specified and Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS388 <> NULL AND OutsideRange(^IDS388,(^IDS381 + ^IDS384 + ^IDS387),1)

            Technical Business Rule Format:
            ^IDS388 <> NULL AND OutsideRange(^IDS388,(^IDS381 + ^IDS384 + ^IDS387),1)
    
            Data Elements:
    
            ^IDS388 = IDS:RP:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS381 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS384 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
    
            ^IDS387 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesNotionalAssessableIncomeTotal.Amount
            */
            assertion = (report.IDS388 != null && OutsideRange(report.IDS388.GetValueOrDefault(), report.IDS381.GetValueOrDefault() + report.IDS384.GetValueOrDefault() + report.IDS387.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440513",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total notional assessable income is not equal to the sum of, amounts of notional assessable income for listed, specified and other unlisted country CFC",
                    Location = "/tns:IDS/tns:RP/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440513" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS388", Value = GetValueOrEmpty(report.IDS388) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS381", Value = GetValueOrEmpty(report.IDS381) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS384", Value = GetValueOrEmpty(report.IDS384) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS387", Value = GetValueOrEmpty(report.IDS387) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440515
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440515()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440515
            If there is a FALSE response to 'Do any of the schedules within the Practical Compliance Guideline PCG 2017/1 apply to your offshore dealings?', then Hub arrangement information must not be provided.
    
            Legacy Rule Format:
            ^IDS393 = FALSE AND Count(^IDS1063) <> 0

            Technical Business Rule Format:
            ^IDS393 = FALSE AND Count(^IDS1063) <> 0
    
            Data Elements:
    
            ^IDS393 = IDS:RP:Miscellaneous.OffshoreDealingsCompliance.Indicator
    
            ^IDS1063 = IDS:RP:HubArrangement
            */
            assertion = (report.IDS393 == false && Count(report.RP_HubArrangementCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440515",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident trust and foreign hubs information not required",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousOffshoreDealingsComplianceI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440515" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS393", Value = GetValueOrEmpty(report.IDS393) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440516
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440516()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440516
            If there is a TRUE response to 'Do any of the schedules within the Practical Compliance Guideline PCG 2017/1 apply to your offshore dealings?', then Hub arrangement information must be provided.
    
            Legacy Rule Format:
            ^IDS393 = TRUE AND Count(^IDS1063) = 0

            Technical Business Rule Format:
            ^IDS393 = TRUE AND Count(^IDS1063) = 0
    
            Data Elements:
    
            ^IDS393 = IDS:RP:Miscellaneous.OffshoreDealingsCompliance.Indicator
    
            ^IDS1063 = IDS:RP:HubArrangement
            */
            assertion = (report.IDS393 == true && Count(report.RP_HubArrangementCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440516",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Provide the types of hub arrangement(s) that have the highest dollar value of property or services imported or exported to or from Australia.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousOffshoreDealingsComplianceI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440516" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS393", Value = GetValueOrEmpty(report.IDS393) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440524
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440524()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440524
            If there is a FALSE response to 'Are you a foreign bank or other qualifying financial entity that has elected out of Part IIIB of the ITAA 1936?', then Notional amounts of financial entity that has elected out of Part IIIB must not be completed.
    
            Legacy Rule Format:
            ^IDS412 = FALSE AND ^IDS1075 <> NULL

            Technical Business Rule Format:
            ^IDS412 = FALSE AND ^IDS1075 <> NULL
    
            Data Elements:
    
            ^IDS412 = IDS:RP:FinancialServices.ForeignBankOrQualifyingFinancialEntityElectedOut.Indicator
    
            ^IDS1075 = IDS:RP:NotionalAmountElectedOut
            */
            assertion = (report.IDS412 == false && report.RP_NotionalAmountElectedOutCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440524",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial services entities information not required",
                    Location = "/tns:IDS/tns:RP/tns:FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440524" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS412", Value = GetValueOrEmpty(report.IDS412) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440525
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440525()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440525
            If there is a TRUE response to 'Are you a foreign bank or other qualifying financial entity that has elected out of Part IIIB of the ITAA 1936?', then Notional amounts of financial entity that has elected out of Part IIIB must be completed.
    
            Legacy Rule Format:
            ^IDS412 = TRUE AND ^IDS1075 = NULL

            Technical Business Rule Format:
            ^IDS412 = TRUE AND ^IDS1075 = NULL
    
            Data Elements:
    
            ^IDS412 = IDS:RP:FinancialServices.ForeignBankOrQualifyingFinancialEntityElectedOut.Indicator
    
            ^IDS1075 = IDS:RP:NotionalAmountElectedOut
            */
            assertion = (report.IDS412 == true && report.RP_NotionalAmountElectedOutCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440525",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial services entities information required",
                    Location = "/tns:IDS/tns:RP/tns:FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440525" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS412", Value = GetValueOrEmpty(report.IDS412) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440529
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440529(IDS2025.RP_ForeignExchange foreignExchange, IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss gainOrLoss, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440529
            Foreign exchange losses deducted or Foreign exchange gains returned must be provided for each currency reported.

            Legacy Rule Format:
            ^IDS1031 <> NULL AND ^IDS336 = NULL AND ^IDS337 = NULL

            Technical Business Rule Format:
            ^IDS1031 <> NULL AND ^IDS336 = NULL AND ^IDS337 = NULL

            Data Elements:

            ^IDS336 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeLossesHighestNominal.Amount

            ^IDS337 = IDS:RP:ForeignExchange:GainOrLoss:InternationalDealings.ForeignExchangeGainsHighestNominal.Amount

            ^IDS1031 = IDS:RP:ForeignExchange:GainOrLoss
            */
            assertion = (foreignExchange.RP_ForeignExchange_GainOrLossCollectionExists != false && gainOrLoss.IDS336 == null && gainOrLoss.IDS337 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440433",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A foreign exchange gain or loss must be reported",
                    Location = "/tns:IDS/tns:RP/tns:ForeignExchangeCollection/tns:ForeignExchange" + OccurrenceIndex(foreignExchange.OccurrenceIndex) + "/tns:GainOrLossCollection/tns:GainOrLoss" + OccurrenceIndex(gainOrLoss.OccurrenceIndex) + "/tns:MiscellaneousFunctionalCurrencyC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440529" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS336", Value = GetValueOrEmpty(gainOrLoss.IDS336) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS337", Value = GetValueOrEmpty(gainOrLoss.IDS337) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440533
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440533()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440533
            If Foreign exchange - Currency/s have been completed, then the response to 'Have you returned a foreign exchange gain or deducted a foreign exchange loss in relation to dealings with international related parties?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS333 <> TRUE AND Count(^IDS1030) <> 0

            Technical Business Rule Format:
            ^IDS333 <> TRUE AND Count(^IDS1030) <> 0
    
            Data Elements:
    
            ^IDS333 = IDS:RP:InternationalDealings.ForeignExchangeGainLoss.Indicator
    
            ^IDS1030 = IDS:RP:ForeignExchange
            */
            assertion = (report.IDS333 != true && Count(report.RP_ForeignExchangeCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440533",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect response for foreign exchange gain or loss in relation to dealings with international related parties.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsForeignExchangeGainLossI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440533" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS333", Value = GetValueOrEmpty(report.IDS333) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440534
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440534()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440534
            If there is a TRUE response to 'Have you returned a foreign exchange gain or deducted a foreign exchange loss in relation to dealings with international related parties?', then at least one transaction type must be completed.
    
            Legacy Rule Format:
            ^IDS333 = TRUE AND Count(^IDS1030) = 0

            Technical Business Rule Format:
            ^IDS333 = TRUE AND Count(^IDS1030) = 0
    
            Data Elements:
    
            ^IDS333 = IDS:RP:InternationalDealings.ForeignExchangeGainLoss.Indicator
    
            ^IDS1030 = IDS:RP:ForeignExchange
            */
            assertion = (report.IDS333 == true && Count(report.RP_ForeignExchangeCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440534",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign exchange information incomplete",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsForeignExchangeGainLossI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440534" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS333", Value = GetValueOrEmpty(report.IDS333) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440543
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440543()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440543
            Foreign country code must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS418)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS418)
    
            Data Elements:
    
            ^IDS418 = IDS:RP:ForeignCountry:OrganisationDetails.InternationalRelatedPartyCountry.Code
            */
            assertion = HasDuplicateValues(report.RP_ForeignCountryCollection == null ? null : report.RP_ForeignCountryCollection.Select(f => f.IDS418).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440284",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign country code has occurred more than once. Delete the duplicate code and input an alternative code",
                    Location = "/tns:IDS/tns:RP/tns:ForeignCountryCollection/tns:ForeignCountry[" + report.RP_ForeignCountryCollection[DuplicateValueIndex(report.RP_ForeignCountryCollection.Select(f => f.IDS418))].OccurrenceIndex + "]/tns:OrganisationDetailsInternationalRelatedPartyCountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440543" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440544
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440544()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440544
            Specified country code must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS420)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS420)
    
            Data Elements:
    
            ^IDS420 = IDS:RP:SpecifiedCountry:OrganisationDetails.InternationalRelatedPartyCountry.Code
            */
            assertion = HasDuplicateValues(report.RP_SpecifiedCountryCollection == null ? null : report.RP_SpecifiedCountryCollection.Select(f => f.IDS420).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440287",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified country code has occurred more than once. Delete the duplicate code and input an alternative code",
                    Location = "/tns:IDS/tns:RP/tns:SpecifiedCountryCollection/tns:SpecifiedCountry[" + report.RP_SpecifiedCountryCollection[DuplicateValueIndex(report.RP_SpecifiedCountryCollection.Select(f => f.IDS420))].OccurrenceIndex + "]/tns:OrganisationDetailsInternationalRelatedPartyCountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440544" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440545
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440545()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440545
            Foreign exchange - Transaction type code must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS422)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS422)
    
            Data Elements:
    
            ^IDS422 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeTransactionType.Code
            */
            assertion = HasDuplicateValues(report.RP_ForeignExchangeCollection == null ? null : report.RP_ForeignExchangeCollection.Select(f => f.IDS422).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Transaction type code has occurred more than once. Delete the duplicate code and input an alternative code",
                    Location = "/tns:IDS/tns:RP/tns:ForeignExchangeCollection/tns:ForeignExchange[" + report.RP_ForeignExchangeCollection[DuplicateValueIndex(report.RP_ForeignExchangeCollection.Select(f => f.IDS422))].OccurrenceIndex + "]/tns:InternationalDealingsTransactionTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440545" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440554
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440554(IDS2025.RP_ForeignExchange foreignExchange, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440554
            If foreign exchange transaction type is present, then foreign exchange gain or loss details must be completed.

            Legacy Rule Format:
            ^IDS1030 <> NULL AND ^IDS1031 = NULL AND ^IDS338 = NULL AND ^IDS339 = NULL

            Technical Business Rule Format:
            ^IDS1030 <> NULL AND ^IDS1031 = NULL AND ^IDS338 = NULL AND ^IDS339 = NULL

            Data Elements:

            ^IDS338 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeLossesOther.Amount

            ^IDS339 = IDS:RP:ForeignExchange:InternationalDealings.ForeignExchangeGainsOther.Amount

            ^IDS1030 = IDS:RP:ForeignExchange

            ^IDS1031 = IDS:RP:ForeignExchange:GainOrLoss
            */
            assertion = (report.RP_ForeignExchangeCollectionExists != false && foreignExchange.RP_ForeignExchange_GainOrLossCollectionExists == false && foreignExchange.IDS338 == null && foreignExchange.IDS339 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440554",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign exchange information incomplete",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsForeignExchangeGainLossI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440554" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS338", Value = GetValueOrEmpty(foreignExchange.IDS338) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS339", Value = GetValueOrEmpty(foreignExchange.IDS339) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440555
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440555(IDS2025.RP_SpecifiedCountry specifiedCountry, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440555
            If specified country code(s) have been completed, then details of specified country dealings must be completed.

            Legacy Rule Format:
            ^IDS1004 <> NULL AND ^IDS1005 = NULL AND ^IDS14 = NULL AND ^IDS15 = NULL

            Technical Business Rule Format:
            ^IDS1004 <> NULL AND ^IDS1005 = NULL AND ^IDS14 = NULL AND ^IDS15 = NULL

            Data Elements:

            ^IDS14 = IDS:RP:SpecifiedCountry:Expense.Other.Amount

            ^IDS15 = IDS:RP:SpecifiedCountry:Revenue.Other.Amount

            ^IDS1004 = IDS:RP:SpecifiedCountry

            ^IDS1005 = IDS:RP:SpecifiedCountry:SpecifiedDealings
            */
            assertion = (report.RP_SpecifiedCountryCollectionExists != false && specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollectionExists == false && specifiedCountry.IDS14 == null && specifiedCountry.IDS15 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440684",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified country dealings information required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsSpecifiedCountriesTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440555" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS14", Value = GetValueOrEmpty(specifiedCountry.IDS14) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS15", Value = GetValueOrEmpty(specifiedCountry.IDS15) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440556
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440556(IDS2025.RP_ForeignCountry foreignCountry, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440556
            If foreign country code(s) have been completed, then details of foreign country dealings must be completed.

            Legacy Rule Format:
            ^IDS1002 <> NULL AND ^IDS1003 = NULL AND ^IDS7 = NULL AND ^IDS8 = NULL

            Technical Business Rule Format:
            ^IDS1002 <> NULL AND ^IDS1003 = NULL AND ^IDS7 = NULL AND ^IDS8 = NULL

            Data Elements:

            ^IDS7 = IDS:RP:ForeignCountry:Expense.Other.Amount

            ^IDS8 = IDS:RP:ForeignCountry:Revenue.Other.Amount

            ^IDS1002 = IDS:RP:ForeignCountry

            ^IDS1003 = IDS:RP:ForeignCountry:ForeignDealings
            */
            assertion = (report.RP_ForeignCountryCollectionExists != false && foreignCountry.RP_ForeignCountry_ForeignDealingsCollectionExists == false && foreignCountry.IDS7 == null && foreignCountry.IDS8 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440683",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign country dealings information required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRelatedPartiesTransactionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440556" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS7", Value = GetValueOrEmpty(foreignCountry.IDS7) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS8", Value = GetValueOrEmpty(foreignCountry.IDS8) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440560
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440560()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440560
            If there is a TRUE response to 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?', then no information must be provided at International related party dealings Section A Questions 3 to 17 of the schedule.
    
            Legacy Rule Format:
            ^IDS425 = TRUE AND (^IDS2 <> NULL OR ^IDS9 <> NULL OR ^IDS16 <> NULL OR ^IDS21 <> NULL OR ^IDS30 <> NULL OR ^IDS35 <> NULL OR ^IDS81 <> NULL OR ^IDS88 <> NULL OR ^IDS97 <> NULL OR ^IDS333 <> NULL OR ^IDS123 <> NULL OR ^IDS129 <> NULL OR ^IDS426 <> NULL OR ^IDS138 <> NULL OR ^IDS147 <> NULL OR ^IDS150 <> NULL OR ^IDS151 <> NULL OR ^IDS159 <> NULL)

            Technical Business Rule Format:
            ^IDS425 = TRUE AND (^IDS2 <> NULL OR ^IDS9 <> NULL OR ^IDS16 <> NULL OR ^IDS21 <> NULL OR ^IDS30 <> NULL OR ^IDS35 <> NULL OR ^IDS81 <> NULL OR ^IDS88 <> NULL OR ^IDS97 <> NULL OR ^IDS333 <> NULL OR ^IDS123 <> NULL OR ^IDS129 <> NULL OR ^IDS426 <> NULL OR ^IDS138 <> NULL OR ^IDS147 <> NULL OR ^IDS150 <> NULL OR ^IDS151 <> NULL OR ^IDS159 <> NULL)
    
            Data Elements:
    
            ^IDS425 = IDS:RP:InternationalDealings.SmallBusinessConcessionsEligibility.Indicator
    
            ^IDS2 = IDS:RP:InternationalDealings.RelatedPartiesTransactions.Indicator
    
            ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
    
            ^IDS16 = IDS:RP:InternationalDealings.TransactionsTangiblePropertyRevenue.Indicator
    
            ^IDS21 = IDS:RP:InternationalDealings.TransactionsRoyaltiesLicenceFees.Indicator
    
            ^IDS30 = IDS:RP:InternationalDealings.TransactionsRentLeasing.Indicator
    
            ^IDS35 = IDS:RP:InternationalDealings.TransactionsServiceArrangements.Indicator
    
            ^IDS81 = IDS:RP:InternationalDealings.DerivativeTransactions.Indicator
    
            ^IDS88 = IDS:RP:InternationalDealings.DebtFactoringSecuritisation.Indicator
    
            ^IDS97 = IDS:RP:InternationalDealings.OtherFinancialDealings.Indicator
    
            ^IDS123 = IDS:RP:InternationalDealings.OtherRevenue.Indicator
    
            ^IDS129 = IDS:RP:Miscellaneous.CapitalNature.Indicator
    
            ^IDS138 = IDS:RP:Miscellaneous.ReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsideration.Indicator
    
            ^IDS147 = IDS:RP:InternationalDealings.ShareBasedEmployeeRemuneration.Indicator
    
            ^IDS150 = IDS:RP:InternationalDealings.CostContributionArrangements.Indicator
    
            ^IDS151 = IDS:RP:InternationalDealings.RestructuringEvents.Indicator
    
            ^IDS159 = IDS:RP:InternationalDealings.TransactionsRestructuringEventsAssetRevaluation.Indicator
    
            ^IDS333 = IDS:RP:InternationalDealings.ForeignExchangeGainLoss.Indicator
    
            ^IDS426 = IDS:RP:Miscellaneous.ResearchAndDevelopmentCostPlusRemuneration.Indicator
            */
            assertion = (report.IDS425 == true && (report.IDS2 != null || report.IDS9 != null || report.IDS16 != null || report.IDS21 != null || report.IDS30 != null || report.IDS35 != null || report.IDS81 != null || report.IDS88 != null || report.IDS97 != null || report.IDS333 != null || report.IDS123 != null || report.IDS129 != null || report.IDS426 != null || report.IDS138 != null || report.IDS147 != null || report.IDS150 != null || report.IDS151 != null || report.IDS159 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440560",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information is not required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsSmallBusinessConcessionsEligibilityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440560" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS425", Value = GetValueOrEmpty(report.IDS425) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS2", Value = GetValueOrEmpty(report.IDS2) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS9", Value = GetValueOrEmpty(report.IDS9) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS16", Value = GetValueOrEmpty(report.IDS16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS21", Value = GetValueOrEmpty(report.IDS21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS30", Value = GetValueOrEmpty(report.IDS30) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS35", Value = GetValueOrEmpty(report.IDS35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS81", Value = GetValueOrEmpty(report.IDS81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS88", Value = GetValueOrEmpty(report.IDS88) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS97", Value = GetValueOrEmpty(report.IDS97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS333", Value = GetValueOrEmpty(report.IDS333) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS123", Value = GetValueOrEmpty(report.IDS123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS129", Value = GetValueOrEmpty(report.IDS129) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS426", Value = GetValueOrEmpty(report.IDS426) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS138", Value = GetValueOrEmpty(report.IDS138) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS147", Value = GetValueOrEmpty(report.IDS147) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS150", Value = GetValueOrEmpty(report.IDS150) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS151", Value = GetValueOrEmpty(report.IDS151) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS159", Value = GetValueOrEmpty(report.IDS159) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440561
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440561()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440561
            If there is a FALSE response to 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?', then all International related party dealings questions in Section A of the schedule must be completed.
    
            Legacy Rule Format:
            ^IDS425 = FALSE AND (^IDS2 = NULL OR ^IDS9 = NULL OR ^IDS16 = NULL OR ^IDS21 = NULL OR ^IDS30 = NULL OR ^IDS35 = NULL OR ^IDS81 = NULL OR ^IDS88 = NULL OR ^IDS97 = NULL OR ^IDS333 = NULL OR ^IDS123 = NULL OR ^IDS129 = NULL OR ^IDS426 = NULL OR ^IDS138 = NULL OR ^IDS147 = NULL OR ^IDS150 = NULL OR ^IDS151 = NULL OR ^IDS159 = NULL)

            Technical Business Rule Format:
            ^IDS425 = FALSE AND (^IDS2 = NULL OR ^IDS9 = NULL OR ^IDS16 = NULL OR ^IDS21 = NULL OR ^IDS30 = NULL OR ^IDS35 = NULL OR ^IDS81 = NULL OR ^IDS88 = NULL OR ^IDS97 = NULL OR ^IDS333 = NULL OR ^IDS123 = NULL OR ^IDS129 = NULL OR ^IDS426 = NULL OR ^IDS138 = NULL OR ^IDS147 = NULL OR ^IDS150 = NULL OR ^IDS151 = NULL OR ^IDS159 = NULL)
    
            Data Elements:
    
            ^IDS425 = IDS:RP:InternationalDealings.SmallBusinessConcessionsEligibility.Indicator
    
            ^IDS2 = IDS:RP:InternationalDealings.RelatedPartiesTransactions.Indicator
    
            ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
    
            ^IDS16 = IDS:RP:InternationalDealings.TransactionsTangiblePropertyRevenue.Indicator
    
            ^IDS21 = IDS:RP:InternationalDealings.TransactionsRoyaltiesLicenceFees.Indicator
    
            ^IDS30 = IDS:RP:InternationalDealings.TransactionsRentLeasing.Indicator
    
            ^IDS35 = IDS:RP:InternationalDealings.TransactionsServiceArrangements.Indicator
    
            ^IDS81 = IDS:RP:InternationalDealings.DerivativeTransactions.Indicator
    
            ^IDS88 = IDS:RP:InternationalDealings.DebtFactoringSecuritisation.Indicator
    
            ^IDS97 = IDS:RP:InternationalDealings.OtherFinancialDealings.Indicator
    
            ^IDS123 = IDS:RP:InternationalDealings.OtherRevenue.Indicator
    
            ^IDS129 = IDS:RP:Miscellaneous.CapitalNature.Indicator
    
            ^IDS138 = IDS:RP:Miscellaneous.ReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsideration.Indicator
    
            ^IDS147 = IDS:RP:InternationalDealings.ShareBasedEmployeeRemuneration.Indicator
    
            ^IDS150 = IDS:RP:InternationalDealings.CostContributionArrangements.Indicator
    
            ^IDS151 = IDS:RP:InternationalDealings.RestructuringEvents.Indicator
    
            ^IDS159 = IDS:RP:InternationalDealings.TransactionsRestructuringEventsAssetRevaluation.Indicator
    
            ^IDS333 = IDS:RP:InternationalDealings.ForeignExchangeGainLoss.Indicator
    
            ^IDS426 = IDS:RP:Miscellaneous.ResearchAndDevelopmentCostPlusRemuneration.Indicator
            */
            assertion = (report.IDS425 == false && (report.IDS2 == null || report.IDS9 == null || report.IDS16 == null || report.IDS21 == null || report.IDS30 == null || report.IDS35 == null || report.IDS81 == null || report.IDS88 == null || report.IDS97 == null || report.IDS333 == null || report.IDS123 == null || report.IDS129 == null || report.IDS426 == null || report.IDS138 == null || report.IDS147 == null || report.IDS150 == null || report.IDS151 == null || report.IDS159 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440561",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information is required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsSmallBusinessConcessionsEligibilityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440561" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS425", Value = GetValueOrEmpty(report.IDS425) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS2", Value = GetValueOrEmpty(report.IDS2) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS9", Value = GetValueOrEmpty(report.IDS9) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS16", Value = GetValueOrEmpty(report.IDS16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS21", Value = GetValueOrEmpty(report.IDS21) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS30", Value = GetValueOrEmpty(report.IDS30) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS35", Value = GetValueOrEmpty(report.IDS35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS81", Value = GetValueOrEmpty(report.IDS81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS88", Value = GetValueOrEmpty(report.IDS88) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS97", Value = GetValueOrEmpty(report.IDS97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS333", Value = GetValueOrEmpty(report.IDS333) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS123", Value = GetValueOrEmpty(report.IDS123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS129", Value = GetValueOrEmpty(report.IDS129) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS426", Value = GetValueOrEmpty(report.IDS426) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS138", Value = GetValueOrEmpty(report.IDS138) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS147", Value = GetValueOrEmpty(report.IDS147) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS150", Value = GetValueOrEmpty(report.IDS150) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS151", Value = GetValueOrEmpty(report.IDS151) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS159", Value = GetValueOrEmpty(report.IDS159) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440562
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440562()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440562
            If there is a FALSE response to 'Did you receive or were entitled to receive remuneration from international related parties on a cost-plus basis for the performance of R&D type services?', then R&D type services cost information must not be completed.
    
            Legacy Rule Format:
            ^IDS426 = FALSE AND ^IDS1039 <> NULL

            Technical Business Rule Format:
            ^IDS426 = FALSE AND ^IDS1039 <> NULL
    
            Data Elements:
    
            ^IDS426 = IDS:RP:Miscellaneous.ResearchAndDevelopmentCostPlusRemuneration.Indicator
    
            ^IDS1039 = IDS:RP:ResearchAndDevelopment
            */
            assertion = (report.IDS426 == false && report.RP_ResearchAndDevelopmentCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440562",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost information of R&D type services is not required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousResearchAndDevelopmentCostPlusRemunerationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440562" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS426", Value = GetValueOrEmpty(report.IDS426) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440563
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440563()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440563
            If there is a TRUE response to 'Did you receive or were entitled to receive remuneration from international related parties on a cost-plus basis for the performance of R&D type services?', then R&D type services - Costs on which margin calculated and R&D type services - Cost-plus margin remuneration must be completed.
    
            Legacy Rule Format:
            ^IDS426 = TRUE AND ^IDS1039 = NULL

            Technical Business Rule Format:
            ^IDS426 = TRUE AND ^IDS1039 = NULL
    
            Data Elements:
    
            ^IDS426 = IDS:RP:Miscellaneous.ResearchAndDevelopmentCostPlusRemuneration.Indicator
    
            ^IDS1039 = IDS:RP:ResearchAndDevelopment
            */
            assertion = (report.IDS426 == true && report.RP_ResearchAndDevelopmentCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440563",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost information of R&D type services is required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousResearchAndDevelopmentCostPlusRemunerationI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440563" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS426", Value = GetValueOrEmpty(report.IDS426) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440574
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440574()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440574
            If there is a TRUE response to 'Did you exclude tainted interest income from the passive income of a CFC which was an AFI subsidiary?', then response must be provided to the questions, 'Did your AFI subsidiary CFCs have banking licences in their countries?' and 'Did your AFI subsidiary CFCs principally derive their income from the lending of money?'.
    
            Legacy Rule Format:
            ^IDS433 = TRUE AND (^IDS434 = NULL OR ^IDS435 = NULL)

            Technical Business Rule Format:
            ^IDS433 = TRUE AND (^IDS434 = NULL OR ^IDS435 = NULL)
    
            Data Elements:
    
            ^IDS433 = IDS:RP:Miscellaneous.ControlledForeignCompaniesExcludedTaintedInterestIncome.Indicator
    
            ^IDS434 = IDS:RP:Miscellaneous.ControlledForeignCompaniesBankingLicences.Indicator
    
            ^IDS435 = IDS:RP:Miscellaneous.ControlledForeignCompaniesIncomeMoneyLending.Indicator
            */
            assertion = (report.IDS433 == true && (report.IDS434 == null || report.IDS435 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440574",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial intermediary business information is required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompaniesExcludedTaintedInterestIncomeI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440574" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS433", Value = GetValueOrEmpty(report.IDS433) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS434", Value = GetValueOrEmpty(report.IDS434) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS435", Value = GetValueOrEmpty(report.IDS435) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440577
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440577()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440577
            If there is a FALSE response to 'Did your AFI subsidiary CFCs principally derive their income from the lending of money?', then tainted interest income excluded from passive income of CFCs (excluding CFCs with banking licences in their country) or the number of CFCs which excluded tainted interest income from passive income for Listed, Specified or Other unlisted countries must not be provided.
    
            Legacy Rule Format:
            ^IDS435 = FALSE AND (^IDS436 <> NULL OR ^IDS437 <> NULL OR ^IDS438 <> NULL OR ^IDS439 <> NULL OR ^IDS440 <> NULL OR ^IDS441 <> NULL OR ^IDS442 <> NULL)

            Technical Business Rule Format:
            ^IDS435 = FALSE AND (^IDS436 <> NULL OR ^IDS437 <> NULL OR ^IDS438 <> NULL OR ^IDS439 <> NULL OR ^IDS440 <> NULL OR ^IDS441 <> NULL OR ^IDS442 <> NULL)
    
            Data Elements:
    
            ^IDS435 = IDS:RP:Miscellaneous.ControlledForeignCompaniesIncomeMoneyLending.Indicator
    
            ^IDS436 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS437 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS438 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS439 = IDS:RP:Income.ControlledForeignCompaniesExcludedTaintedInterestIncomeTotal.Amount
    
            ^IDS440 = IDS:RP:ListedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS441 = IDS:RP:SpecifiedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS442 = IDS:RP:UnlistedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
            */
            assertion = (report.IDS435 == false && (report.IDS436 != null || report.IDS437 != null || report.IDS438 != null || report.IDS439 != null || report.IDS440 != null || report.IDS441 != null || report.IDS442 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440577",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial intermediary business information is not required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440577" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS435", Value = GetValueOrEmpty(report.IDS435) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS436", Value = GetValueOrEmpty(report.IDS436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS437", Value = GetValueOrEmpty(report.IDS437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS438", Value = GetValueOrEmpty(report.IDS438) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS439", Value = GetValueOrEmpty(report.IDS439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS440", Value = GetValueOrEmpty(report.IDS440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS441", Value = GetValueOrEmpty(report.IDS441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS442", Value = GetValueOrEmpty(report.IDS442) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440578
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440578()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440578
            If there is a TRUE response to 'Did your AFI subsidiary CFCs principally derive their income from the lending of money?', then tainted interest income excluded from passive income of CFCs (excluding CFCs with banking licences in their country) and the number of CFCs which excluded tainted interest income from passive income for Listed, Specified and Other unlisted countries must be provided.
    
            Legacy Rule Format:
            ^IDS435 = TRUE AND (^IDS436 = NULL OR ^IDS437 = NULL OR ^IDS438 = NULL OR ^IDS439 = NULL OR ^IDS440 = NULL OR ^IDS441 = NULL OR ^IDS442 = NULL)

            Technical Business Rule Format:
            ^IDS435 = TRUE AND (^IDS436 = NULL OR ^IDS437 = NULL OR ^IDS438 = NULL OR ^IDS439 = NULL OR ^IDS440 = NULL OR ^IDS441 = NULL OR ^IDS442 = NULL)
    
            Data Elements:
    
            ^IDS435 = IDS:RP:Miscellaneous.ControlledForeignCompaniesIncomeMoneyLending.Indicator
    
            ^IDS436 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS437 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS438 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS439 = IDS:RP:Income.ControlledForeignCompaniesExcludedTaintedInterestIncomeTotal.Amount
    
            ^IDS440 = IDS:RP:ListedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS441 = IDS:RP:SpecifiedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS442 = IDS:RP:UnlistedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
            */
            assertion = (report.IDS435 == true && (report.IDS436 == null || report.IDS437 == null || report.IDS438 == null || report.IDS439 == null || report.IDS440 == null || report.IDS441 == null || report.IDS442 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440578",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial intermediary business information is required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440578" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS435", Value = GetValueOrEmpty(report.IDS435) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS436", Value = GetValueOrEmpty(report.IDS436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS437", Value = GetValueOrEmpty(report.IDS437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS438", Value = GetValueOrEmpty(report.IDS438) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS439", Value = GetValueOrEmpty(report.IDS439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS440", Value = GetValueOrEmpty(report.IDS440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS441", Value = GetValueOrEmpty(report.IDS441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS442", Value = GetValueOrEmpty(report.IDS442) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440583
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440583()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440583
            Total amount of tainted interest income excluded from passive income of CFCs (excluding CFCs with banking licences in their country) must be equal to the sum of corresponding amount for Listed, Specified and Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS439 <> NULL AND OutsideRange(^IDS439, ^IDS436 + ^IDS437 + ^IDS438, 1)

            Technical Business Rule Format:
            ^IDS439 <> NULL AND OutsideRange(^IDS439, ^IDS436 + ^IDS437 + ^IDS438, 1)
    
            Data Elements:
    
            ^IDS439 = IDS:RP:Income.ControlledForeignCompaniesExcludedTaintedInterestIncomeTotal.Amount
    
            ^IDS436 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS437 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
    
            ^IDS438 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
            */
            assertion = (report.IDS439 != null && OutsideRange(report.IDS439.GetValueOrDefault(), report.IDS436.GetValueOrDefault() + report.IDS437.GetValueOrDefault() + report.IDS438.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440583",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total excluded tainted interest income is not equal to the sum of, tainted interest income excluded from passive income of your financial intermediary subsidiary CFCs for Listed, Specified and Other unlisted countries.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440583" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS439", Value = GetValueOrEmpty(report.IDS439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS436", Value = GetValueOrEmpty(report.IDS436) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS437", Value = GetValueOrEmpty(report.IDS437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS438", Value = GetValueOrEmpty(report.IDS438) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440587
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440587()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440587
            Number of CFCs which excluded tainted interest income from passive income - Other unlisted countries must be provided when Total amount of tainted interest income excluded from passive income of CFCs - Other unlisted countries is present.
    
            Legacy Rule Format:
            ^IDS442 = NULL AND ^IDS438 <> NULL

            Technical Business Rule Format:
            ^IDS442 = NULL AND ^IDS438 <> NULL
    
            Data Elements:
    
            ^IDS442 = IDS:RP:UnlistedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS438 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
            */
            assertion = (report.IDS442 == null && report.IDS438 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440587",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs - Other unlisted countries must be present.",
                    Location = "/tns:IDS/tns:RP/tns:UnlistedCFCAmounts/tns:MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440587" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS442", Value = GetValueOrEmpty(report.IDS442) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS438", Value = GetValueOrEmpty(report.IDS438) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440588
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440588()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440588
            Number of CFCs which excluded tainted interest income from passive income - Specified countries must be provided when Total amount of tainted interest income excluded from passive income of CFCs - Specified countries is present.
    
            Legacy Rule Format:
            ^IDS441 = NULL AND ^IDS437 <> NULL

            Technical Business Rule Format:
            ^IDS441 = NULL AND ^IDS437 <> NULL
    
            Data Elements:
    
            ^IDS441 = IDS:RP:SpecifiedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS437 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
            */
            assertion = (report.IDS441 == null && report.IDS437 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440588",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs - Specified countries must be present.",
                    Location = "/tns:IDS/tns:RP/tns:SpecifiedCFCAmounts/tns:MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440588" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS441", Value = GetValueOrEmpty(report.IDS441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS437", Value = GetValueOrEmpty(report.IDS437) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440589
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440589()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440589
            Number of CFCs which excluded tainted interest income from passive income - Listed countries must be provided when Total amount of tainted interest income excluded from passive income of CFCs - Listed countries is present.
    
            Legacy Rule Format:
            ^IDS440 = NULL AND ^IDS436 <> NULL

            Technical Business Rule Format:
            ^IDS440 = NULL AND ^IDS436 <> NULL
    
            Data Elements:
    
            ^IDS440 = IDS:RP:ListedCFCAmounts:Miscellaneous.ControlledForeignCompanyExcludedTaintedInterestIncome.Count
    
            ^IDS436 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesExcludedTaintedInterestIncome.Amount
            */
            assertion = (report.IDS440 == null && report.IDS436 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440589",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of CFCs - Listed countries must be present.",
                    Location = "/tns:IDS/tns:RP/tns:ListedCFCAmounts/tns:MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440589" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS440", Value = GetValueOrEmpty(report.IDS440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS436", Value = GetValueOrEmpty(report.IDS436) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440593
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440593()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440593
            Gross revenue of CFCs that have satisfied the active income test - Total must be equal to the sum of corresponding amount for Listed, Specified and Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS446 <> NULL AND OutsideRange(^IDS446, ^IDS443 + ^IDS444 + ^IDS445, 1)

            Technical Business Rule Format:
            ^IDS446 <> NULL AND OutsideRange(^IDS446, ^IDS443 + ^IDS444 + ^IDS445, 1)
    
            Data Elements:
    
            ^IDS446 = IDS:RP:Income.GrossRevenueActiveIncomeTestSatisfiedTotal.Amount
    
            ^IDS443 = IDS:RP:ListedCFCAmounts:Income.GrossRevenueActiveIncomeTestSatisfied.Amount
    
            ^IDS444 = IDS:RP:SpecifiedCFCAmounts:Income.GrossRevenueActiveIncomeTestSatisfied.Amount
    
            ^IDS445 = IDS:RP:UnlistedCFCAmounts:Income.GrossRevenueActiveIncomeTestSatisfied.Amount
            */
            assertion = (report.IDS446 != null && OutsideRange(report.IDS446.GetValueOrDefault(), report.IDS443.GetValueOrDefault() + report.IDS444.GetValueOrDefault() + report.IDS445.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440593",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total gross revenue included in the gross turnover of your CFCs that have satisfied the active income test is not equal to the sum of amounts listed at the gross revenue for Listed, Specified and Other unlisted countries.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440593" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS446", Value = GetValueOrEmpty(report.IDS446) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS443", Value = GetValueOrEmpty(report.IDS443) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS444", Value = GetValueOrEmpty(report.IDS444) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS445", Value = GetValueOrEmpty(report.IDS445) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440594
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440594()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440594
            Gross revenue of CFCs that have not satisfied the active income test - Total must be equal to the sum of corresponding amount for Listed, Specified and Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS450 <> NULL AND OutsideRange(^IDS450, ^IDS447 + ^IDS448 + ^IDS449, 1)

            Technical Business Rule Format:
            ^IDS450 <> NULL AND OutsideRange(^IDS450, ^IDS447 + ^IDS448 + ^IDS449, 1)
    
            Data Elements:
    
            ^IDS450 = IDS:RP:Income.GrossRevenueActiveIncomeTestNotSatisfiedTotal.Amount
    
            ^IDS447 = IDS:RP:ListedCFCAmounts:Income.GrossRevenueActiveIncomeTestNotSatisfied.Amount
    
            ^IDS448 = IDS:RP:SpecifiedCFCAmounts:Income.GrossRevenueActiveIncomeTestNotSatisfied.Amount
    
            ^IDS449 = IDS:RP:UnlistedCFCAmounts:Income.GrossRevenueActiveIncomeTestNotSatisfied.Amount
            */
            assertion = (report.IDS450 != null && OutsideRange(report.IDS450.GetValueOrDefault(), report.IDS447.GetValueOrDefault() + report.IDS448.GetValueOrDefault() + report.IDS449.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440594",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total gross revenue included in the gross turnover of your CFCs that have not satisfied the active income test is not equal to the sum of amounts listed at the gross revenue for Listed, Specified and Other unlisted countries.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440594" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS450", Value = GetValueOrEmpty(report.IDS450) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS447", Value = GetValueOrEmpty(report.IDS447) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS448", Value = GetValueOrEmpty(report.IDS448) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS449", Value = GetValueOrEmpty(report.IDS449) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440602
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440602()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440602
            Total notional allowable deductions must be equal to the sum of corresponding amount for Listed, Specified and Other unlisted countries.
    
            Legacy Rule Format:
            ^IDS454 <> NULL AND OutsideRange(^IDS454, ^IDS451 + ^IDS452 + ^IDS453, 1)

            Technical Business Rule Format:
            ^IDS454 <> NULL AND OutsideRange(^IDS454, ^IDS451 + ^IDS452 + ^IDS453, 1)
    
            Data Elements:
    
            ^IDS454 = IDS:RP:Income.ControlledForeignCompaniesNotionalAllowableDeductionsTotal.Amount
    
            ^IDS451 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesNotionalAllowableDeductions.Amount
    
            ^IDS452 = IDS:RP:SpecifiedCFCAmounts:Income.ControlledForeignCompaniesNotionalAllowableDeductions.Amount
    
            ^IDS453 = IDS:RP:UnlistedCFCAmounts:Income.ControlledForeignCompaniesNotionalAllowableDeductions.Amount
            */
            assertion = (report.IDS454 != null && OutsideRange(report.IDS454.GetValueOrDefault(), report.IDS451.GetValueOrDefault() + report.IDS452.GetValueOrDefault() + report.IDS453.GetValueOrDefault(), 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440602",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total notional allowable deductions is not equal to the sum of amounts listed at notional allowable deductions for Listed countries, Specified countries and Other unlisted countries.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeControlledForeignCompaniesNotionalAllowableDeductionsTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440602" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS454", Value = GetValueOrEmpty(report.IDS454) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS451", Value = GetValueOrEmpty(report.IDS451) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS452", Value = GetValueOrEmpty(report.IDS452) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS453", Value = GetValueOrEmpty(report.IDS453) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440603
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440603()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440603
            If there is a FALSE response to 'Did you have amounts denied under section 160ZZZL of the ITAA 1936?', then no value must be provided at 'Amount denied under section 160ZZZL of the ITAA 1936'.
    
            Legacy Rule Format:
            ^IDS455 = FALSE AND ^IDS456 <> NULL

            Technical Business Rule Format:
            ^IDS455 = FALSE AND ^IDS456 <> NULL
    
            Data Elements:
    
            ^IDS455 = IDS:RP:FinancialServices.NotionalPaymentDenied.Indicator
    
            ^IDS456 = IDS:RP:FinancialServices.NotionalPaymentDenied.Amount
            */
            assertion = (report.IDS455 == false && report.IDS456 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440603",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial services entities information is not required.",
                    Location = "/tns:IDS/tns:RP/tns:FinancialServicesNotionalPaymentDeniedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440603" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS455", Value = GetValueOrEmpty(report.IDS455) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS456", Value = GetValueOrEmpty(report.IDS456) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440604
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440604()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440604
            If there is a TRUE response to 'Did you have amounts denied under section 160ZZZL of the ITAA 1936?', then 'Amount denied under section 160ZZZL of the ITAA 1936' must be provided.
    
            Legacy Rule Format:
            ^IDS455 = TRUE AND ^IDS456 = NULL

            Technical Business Rule Format:
            ^IDS455 = TRUE AND ^IDS456 = NULL
    
            Data Elements:
    
            ^IDS455 = IDS:RP:FinancialServices.NotionalPaymentDenied.Indicator
    
            ^IDS456 = IDS:RP:FinancialServices.NotionalPaymentDenied.Amount
            */
            assertion = (report.IDS455 == true && report.IDS456 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440604",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial services entities information is required.",
                    Location = "/tns:IDS/tns:RP/tns:FinancialServicesNotionalPaymentDeniedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440604" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS455", Value = GetValueOrEmpty(report.IDS455) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS456", Value = GetValueOrEmpty(report.IDS456) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440608
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440608()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440608
            If there is a FALSE response to 'Have you made or received a payment at any time during the income year which gave rise to a deduction / non-inclusion or a deduction / deduction mismatch?' then no information must be provided at the following:
            - Total amount of payments which gave rise to deduction/non-inclusion mismatches and deduction/deduction mismatches
            - Material arrangements which gave rise to mismatch.
    
            Legacy Rule Format:
            ^IDS458 = FALSE AND (^IDS482 <> NULL OR Count(^IDS1106) <> 0)

            Technical Business Rule Format:
            ^IDS458 = FALSE AND (^IDS482 <> NULL OR Count(^IDS1106) <> 0)
    
            Data Elements:
    
            ^IDS458 = IDS:RP:Miscellaneous.HybridMismatchDeductionNonInclusion.Indicator
    
            ^IDS482 = IDS:RP:Miscellaneous.HybridMismatchPayments.Amount
    
            ^IDS1106 = IDS:RP:MaterialArrangements
            */
            assertion = (report.IDS458 == false && (report.IDS482 != null || Count(report.RP_MaterialArrangementsCollectionCount) != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440608",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information not required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchDeductionNonInclusionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440608" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS458", Value = GetValueOrEmpty(report.IDS458) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS482", Value = GetValueOrEmpty(report.IDS482) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440609
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440609()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440609
            If there is a TRUE response to 'Have you made or received a payment at any time during the income year which gave rise to a deduction / non-inclusion or a deduction / deduction mismatch?', then 'Total amount of payments which gave rise to deduction/non-inclusion mismatches and deduction/deduction mismatches' must be provided.
    
            Legacy Rule Format:
            ^IDS458 = TRUE AND ^IDS482 = NULL

            Technical Business Rule Format:
            ^IDS458 = TRUE AND ^IDS482 = NULL
    
            Data Elements:
    
            ^IDS458 = IDS:RP:Miscellaneous.HybridMismatchDeductionNonInclusion.Indicator
    
            ^IDS482 = IDS:RP:Miscellaneous.HybridMismatchPayments.Amount
            */
            assertion = (report.IDS458 == true && report.IDS482 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440609",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information is required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchDeductionNonInclusionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440609" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS458", Value = GetValueOrEmpty(report.IDS458) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS482", Value = GetValueOrEmpty(report.IDS482) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440618
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440618()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440618
            If all or some of the CFCs in Listed countries have satisfied the active income test, then Gross revenue included in the gross turnover of CFCs that have satisfied the active income test - Listed countries must be completed.
    
            Legacy Rule Format:
            ^IDS443 = NULL AND InSet(^IDS375, '"Yes","Both"')

            Technical Business Rule Format:
            ^IDS443 = NULL AND InSet(^IDS375, '"Yes","Both"')
    
            Data Elements:
    
            ^IDS443 = IDS:RP:ListedCFCAmounts:Income.GrossRevenueActiveIncomeTestSatisfied.Amount
    
            ^IDS375 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
            */
            assertion = (report.IDS443 == null && IsMatch(report.IDS375, @"^(Yes|Both)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440618",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross revenue of CFCs that have satisfied the active income test - Listed countries is required.",
                    Location = "/tns:IDS/tns:RP/tns:ListedCFCAmounts/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440618" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS443", Value = GetValueOrEmpty(report.IDS443) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS375", Value = GetValueOrEmpty(report.IDS375) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440619
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440619()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440619
            If all or some of the CFCs in Specified countries have satisfied the active income test, then Gross revenue included in the gross turnover of CFCs that have satisfied the active income test - Specified countries must be completed.
    
            Legacy Rule Format:
            ^IDS444 = NULL AND InSet(^IDS376, '"Yes","Both"')

            Technical Business Rule Format:
            ^IDS444 = NULL AND InSet(^IDS376, '"Yes","Both"')
    
            Data Elements:
    
            ^IDS444 = IDS:RP:SpecifiedCFCAmounts:Income.GrossRevenueActiveIncomeTestSatisfied.Amount
    
            ^IDS376 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
            */
            assertion = (report.IDS444 == null && IsMatch(report.IDS376, @"^(Yes|Both)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440619",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross revenue of CFCs that have satisfied the active income test - Specified countries is required.",
                    Location = "/tns:IDS/tns:RP/tns:SpecifiedCFCAmounts/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440619" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS444", Value = GetValueOrEmpty(report.IDS444) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS376", Value = GetValueOrEmpty(report.IDS376) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440620
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440620()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440620
            If all or some of the CFCs in Other unlisted countries have satisfied the active income test, then Gross revenue included in the gross turnover of CFCs that have satisfied the active income test - Other unlisted countries must be completed.
    
            Legacy Rule Format:
            ^IDS445 = NULL AND InSet(^IDS377, '"Yes","Both"')

            Technical Business Rule Format:
            ^IDS445 = NULL AND InSet(^IDS377, '"Yes","Both"')
    
            Data Elements:
    
            ^IDS445 = IDS:RP:UnlistedCFCAmounts:Income.GrossRevenueActiveIncomeTestSatisfied.Amount
    
            ^IDS377 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
            */
            assertion = (report.IDS445 == null && IsMatch(report.IDS377, @"^(Yes|Both)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440620",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross revenue of CFCs that have satisfied the active income test - Other unlisted countries is required.",
                    Location = "/tns:IDS/tns:RP/tns:UnlistedCFCAmounts/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440620" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS445", Value = GetValueOrEmpty(report.IDS445) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS377", Value = GetValueOrEmpty(report.IDS377) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440621
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440621()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440621
            If none or some of the CFCs in Listed countries have not satisfied the active income test, then Gross revenue included in the gross turnover of CFCs that have not satisfied the active income test - Listed countries must be completed.
    
            Legacy Rule Format:
            ^IDS447 = NULL AND InSet(^IDS375, '"No","Both"')

            Technical Business Rule Format:
            ^IDS447 = NULL AND InSet(^IDS375, '"No","Both"')
    
            Data Elements:
    
            ^IDS447 = IDS:RP:ListedCFCAmounts:Income.GrossRevenueActiveIncomeTestNotSatisfied.Amount
    
            ^IDS375 = IDS:RP:Listed:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
            */
            assertion = (report.IDS447 == null && IsMatch(report.IDS375, @"^(No|Both)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440621",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross revenue of CFCs that have not satisfied the active income test - Listed countries is required.",
                    Location = "/tns:IDS/tns:RP/tns:ListedCFCAmounts/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440621" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS447", Value = GetValueOrEmpty(report.IDS447) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS375", Value = GetValueOrEmpty(report.IDS375) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440622
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440622()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440622
            If none or some of the CFCs in Specified countries have not satisfied the active income test, then Gross revenue included in the gross turnover of CFCs that have not satisfied the active income test - Specified countries must be completed.
    
            Legacy Rule Format:
            ^IDS448 = NULL AND InSet(^IDS376, '"No","Both"')

            Technical Business Rule Format:
            ^IDS448 = NULL AND InSet(^IDS376, '"No","Both"')
    
            Data Elements:
    
            ^IDS448 = IDS:RP:SpecifiedCFCAmounts:Income.GrossRevenueActiveIncomeTestNotSatisfied.Amount
    
            ^IDS376 = IDS:RP:Specified:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
            */
            assertion = (report.IDS448 == null && IsMatch(report.IDS376, @"^(No|Both)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440622",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross revenue of CFCs that have not satisfied the active income test - Specified countries is required.",
                    Location = "/tns:IDS/tns:RP/tns:SpecifiedCFCAmounts/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440622" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS448", Value = GetValueOrEmpty(report.IDS448) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS376", Value = GetValueOrEmpty(report.IDS376) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440623
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440623()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440623
            If none or some of the CFCs in Other unlisted countries have not satisfied the active income test, then Gross revenue included in the gross turnover of CFCs that have not satisfied the active income test - Other unlisted countries must be completed.
    
            Legacy Rule Format:
            ^IDS449 = NULL AND InSet(^IDS377, '"No","Both"')

            Technical Business Rule Format:
            ^IDS449 = NULL AND InSet(^IDS377, '"No","Both"')
    
            Data Elements:
    
            ^IDS449 = IDS:RP:UnlistedCFCAmounts:Income.GrossRevenueActiveIncomeTestNotSatisfied.Amount
    
            ^IDS377 = IDS:RP:Unlisted:Miscellaneous.ControlledForeignCompaniesActiveIncomeTestSatisfied.Code
            */
            assertion = (report.IDS449 == null && IsMatch(report.IDS377, @"^(No|Both)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440623",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross revenue of CFCs that have not satisfied the active income test - Other unlisted countries is required.",
                    Location = "/tns:IDS/tns:RP/tns:UnlistedCFCAmounts/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440623" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS449", Value = GetValueOrEmpty(report.IDS449) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS377", Value = GetValueOrEmpty(report.IDS377) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440634
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440634()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440634
            If there is a TRUE response to 'Did you pay an amount of interest or an amount under a derivative financial arrangement to an international related party which was not taxed or taxed at 10% or less?', then 'Total amount of deductions subject to subdivision 832-J' and Hybrid payer mismatch information must be provided.
    
            Legacy Rule Format:
            ^IDS468 = TRUE AND (^IDS483 = NULL OR Count(^IDS1107) = 0)

            Technical Business Rule Format:
            ^IDS468 = TRUE AND (^IDS483 = NULL OR Count(^IDS1107) = 0)
    
            Data Elements:
    
            ^IDS468 = IDS:RP:Miscellaneous.PaymentNotTaxedOrTaxed10PercentOrLess.Indicator
    
            ^IDS483 = IDS:RP:Miscellaneous.HybridMismatchDeductions.Amount
    
            ^IDS1107 = IDS:RP:HybridPayerMismatch
            */
            assertion = (report.IDS468 == true && (report.IDS483 == null || Count(report.RP_HybridPayerMismatchCollectionCount) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440634",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440634" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS468", Value = GetValueOrEmpty(report.IDS468) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS483", Value = GetValueOrEmpty(report.IDS483) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440635
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440635()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440635
            If there is a FALSE response to 'Did you pay an amount of interest or an amount under a derivative financial arrangement to an international related party which was not taxed or taxed at 10% or less?', then 'Total amount of deductions subject to subdivision 832-J' or Hybrid payer mismatch information must not be provided.
    
            Legacy Rule Format:
            ^IDS468 = FALSE AND (^IDS483 <> NULL OR Count(^IDS1107) <> 0)

            Technical Business Rule Format:
            ^IDS468 = FALSE AND (^IDS483 <> NULL OR Count(^IDS1107) <> 0)
    
            Data Elements:
    
            ^IDS468 = IDS:RP:Miscellaneous.PaymentNotTaxedOrTaxed10PercentOrLess.Indicator
    
            ^IDS483 = IDS:RP:Miscellaneous.HybridMismatchDeductions.Amount
    
            ^IDS1107 = IDS:RP:HybridPayerMismatch
            */
            assertion = (report.IDS468 == false && (report.IDS483 != null || Count(report.RP_HybridPayerMismatchCollectionCount) != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440635",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information not required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440635" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS468", Value = GetValueOrEmpty(report.IDS468) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS483", Value = GetValueOrEmpty(report.IDS483) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440636
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440636()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440636
            If there is a TRUE response to 'Did you or your international related parties restructure or replace an arrangement in the current or prior year which would have been subject to any of the hybrid mismatch rules if the arrangement was still in place?', then at least one restructuring event information must be provided.
    
            Legacy Rule Format:
            ^IDS474 = TRUE AND Count(^IDS1108) = 0

            Technical Business Rule Format:
            ^IDS474 = TRUE AND Count(^IDS1108) = 0
    
            Data Elements:
    
            ^IDS474 = IDS:RP:Miscellaneous.HybridArrangementRestructureReplacement.Indicator
    
            ^IDS1108 = IDS:RP:HybridArrangementRestructure
            */
            assertion = (report.IDS474 == true && Count(report.RP_HybridArrangementRestructureCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440636",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Restructing event information required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridArrangementRestructureReplacementI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440636" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS474", Value = GetValueOrEmpty(report.IDS474) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440637
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440637()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440637
            If there is a FALSE response to 'Did you or your international related parties restructure or replace an arrangement in the current or prior year which would have been subject to any of the hybrid mismatch rules if the arrangement was still in place?', then restructuring event information must not be provided.
    
            Legacy Rule Format:
            ^IDS474 = FALSE AND Count(^IDS1108) <> 0

            Technical Business Rule Format:
            ^IDS474 = FALSE AND Count(^IDS1108) <> 0
    
            Data Elements:
    
            ^IDS474 = IDS:RP:Miscellaneous.HybridArrangementRestructureReplacement.Indicator
    
            ^IDS1108 = IDS:RP:HybridArrangementRestructure
            */
            assertion = (report.IDS474 == false && Count(report.RP_HybridArrangementRestructureCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440637",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Restructing event information is not required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridArrangementRestructureReplacementI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440637" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS474", Value = GetValueOrEmpty(report.IDS474) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440638
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440638()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440638
            If there is a TRUE response to 'If you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts, then did you receive a foreign equity distribution that gave rise to a foreign income tax deduction?', then 'Amount that is not non-assessable non-exempt under section 768-7' must be provided.
    
            Legacy Rule Format:
            ^IDS477 = TRUE AND ^IDS478 = NULL

            Technical Business Rule Format:
            ^IDS477 = TRUE AND ^IDS478 = NULL
    
            Data Elements:
    
            ^IDS477 = IDS:RP:Income.ForeignEquityDistributionForeignTaxDeduction.Indicator
    
            ^IDS478 = IDS:RP:Income.ForeignDeductionNotNonAssessableNonExempt.Amount
            */
            assertion = (report.IDS477 == true && report.IDS478 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440638",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information required.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeForeignEquityDistributionForeignTaxDeductionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440638" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS477", Value = GetValueOrEmpty(report.IDS477) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS478", Value = GetValueOrEmpty(report.IDS478) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440639
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440639()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440639
            If there is a FALSE response to 'If you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts, then did you receive a foreign equity distribution that gave rise to a foreign income tax deduction?', then 'Amount that is not non-assessable non-exempt under section 768-7' must not be provided.
    
            Legacy Rule Format:
            ^IDS477 = FALSE AND ^IDS478 <> NULL

            Technical Business Rule Format:
            ^IDS477 = FALSE AND ^IDS478 <> NULL
    
            Data Elements:
    
            ^IDS477 = IDS:RP:Income.ForeignEquityDistributionForeignTaxDeduction.Indicator
    
            ^IDS478 = IDS:RP:Income.ForeignDeductionNotNonAssessableNonExempt.Amount
            */
            assertion = (report.IDS477 == false && report.IDS478 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440639",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information not required.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeForeignEquityDistributionForeignTaxDeductionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440639" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS477", Value = GetValueOrEmpty(report.IDS477) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS478", Value = GetValueOrEmpty(report.IDS478) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440640
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440640()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440640
            If there is a TRUE response to 'If you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts, then did you derive branch hybrid mismatch income?', then 'Amounts that are not non-assessable non-exempt under subsection 23AH(4A)' must be provided.
    
            Legacy Rule Format:
            ^IDS479 = TRUE AND ^IDS480 = NULL

            Technical Business Rule Format:
            ^IDS479 = TRUE AND ^IDS480 = NULL
    
            Data Elements:
    
            ^IDS479 = IDS:RP:Income.DerivedBranchHybridMismatchIncome.Indicator
    
            ^IDS480 = IDS:RP:Income.ForeignHybridIncomeNotNonAssessableNonExempt.Amount
            */
            assertion = (report.IDS479 == true && report.IDS480 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440640",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information is required.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeDerivedBranchHybridMismatchIncomeI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440640" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS479", Value = GetValueOrEmpty(report.IDS479) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS480", Value = GetValueOrEmpty(report.IDS480) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440641
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440641()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440641
            If there is a FALSE response to 'If you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts, then did you derive branch hybrid mismatch income?', then 'Amounts that are not non-assessable non-exempt under subsection 23AH(4A)' must not be provided.
    
            Legacy Rule Format:
            ^IDS479 = FALSE AND ^IDS480 <> NULL

            Technical Business Rule Format:
            ^IDS479 = FALSE AND ^IDS480 <> NULL
    
            Data Elements:
    
            ^IDS479 = IDS:RP:Income.DerivedBranchHybridMismatchIncome.Indicator
    
            ^IDS480 = IDS:RP:Income.ForeignHybridIncomeNotNonAssessableNonExempt.Amount
            */
            assertion = (report.IDS479 == false && report.IDS480 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440641",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information is not required.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeDerivedBranchHybridMismatchIncomeI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440641" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS479", Value = GetValueOrEmpty(report.IDS479) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS480", Value = GetValueOrEmpty(report.IDS480) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440642
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440642()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440642
            If there is a TRUE response to 'Is the total of your international related party dealings over $2 million?', then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?' must be completed.
    
            Legacy Rule Format:
            ^IDS1 = TRUE AND ^IDS425 = NULL

            Technical Business Rule Format:
            ^IDS1 = TRUE AND ^IDS425 = NULL
    
            Data Elements:
    
            ^IDS1 = IDS:RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
            ^IDS425 = IDS:RP:InternationalDealings.SmallBusinessConcessionsEligibility.Indicator
            */
            assertion = (report.IDS1 == true && report.IDS425 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440642",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International related party dealings information is required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440642" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS1", Value = GetValueOrEmpty(report.IDS1) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS425", Value = GetValueOrEmpty(report.IDS425) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440644
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440644(IDS2025.RP_MaterialArrangements materialArrangements, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440644
            If Total amount of payments made during income year is greater than Total amount of deductions denied or amount of income inclusion, then Reason for difference must be completed.

            Legacy Rule Format:
            (^IDS462 > ^IDS463) AND ^IDS464 = NULL

            Technical Business Rule Format:
            (^IDS462 > ^IDS463) AND ^IDS464 = NULL

            Data Elements:

            ^IDS464 = IDS:RP:MaterialArrangements:Miscellaneous.HybridMismatchDifferenceReason.Code

            ^IDS462 = IDS:RP:MaterialArrangements:Miscellaneous.HybridMismatchPayments.Amount

            ^IDS463 = IDS:RP:MaterialArrangements:Miscellaneous.HybridMismatchDeductionsDeniedOrIncomeInclusion.Amount
            */
            assertion = (materialArrangements.IDS462.GetValueOrDefault() > materialArrangements.IDS463.GetValueOrDefault() && materialArrangements.IDS464 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440644",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reason for difference is required.",
                    Location = "/tns:IDS/tns:RP/tns:MaterialArrangementsCollection/tns:MaterialArrangements" + OccurrenceIndex(materialArrangements.OccurrenceIndex) + "/tns:MiscellaneousHybridMismatchDifferenceReasonC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440644" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS462", Value = GetValueOrEmpty(materialArrangements.IDS462) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS463", Value = GetValueOrEmpty(materialArrangements.IDS463) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS464", Value = GetValueOrEmpty(materialArrangements.IDS464) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440654
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440654(IDS2025.RP_HybridPayerMismatch hybridPayerMismatch, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440654
            If Amount of deductions denied is zero, then Reason - Hybrid payer mismatch exception must be provided.

            Legacy Rule Format:
            ^IDS472 = 0 AND ^IDS473 = NULL

            Technical Business Rule Format:
            ^IDS472 = 0 AND ^IDS473 = NULL

            Data Elements:

            ^IDS473 = IDS:RP:HybridPayerMismatch:Miscellaneous.HybridMismatchExceptionReason.Code

            ^IDS472 = IDS:RP:HybridPayerMismatch:Miscellaneous.HybridMismatchDeductionsDenied.Amount
            */
            assertion = (hybridPayerMismatch.IDS472 == 0 && hybridPayerMismatch.IDS473 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440654",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information required.",
                    Location = "/tns:IDS/tns:RP/tns:HybridPayerMismatchCollection/tns:HybridPayerMismatch" + OccurrenceIndex(hybridPayerMismatch.OccurrenceIndex) + "/tns:MiscellaneousHybridMismatchExceptionReasonC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440654" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS472", Value = GetValueOrEmpty(hybridPayerMismatch.IDS472) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS473", Value = GetValueOrEmpty(hybridPayerMismatch.IDS473) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440655
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440655()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440655
            The reported codes for the type of hub arrangements that have the highest aggregate dollar value of property or services imported to or exported from Australia, must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS394)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS394)
    
            Data Elements:
    
            ^IDS394 = IDS:RP:HubArrangement:Miscellaneous.HubArrangement.Code
            */
            assertion = HasDuplicateValues(report.RP_HubArrangementCollection == null ? null : report.RP_HubArrangementCollection.Select(f => f.IDS394).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440655",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The code reported at 'Type of hub' must only have one occurrence.",
                    Location = "/tns:IDS/tns:RP/tns:HubArrangementCollection/tns:HubArrangement[" + report.RP_HubArrangementCollection[DuplicateValueIndex(report.RP_HubArrangementCollection.Select(f => f.IDS394))].OccurrenceIndex + "]/tns:MiscellaneousC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440655" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440657
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440657()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440657
            If there is a FALSE response to 'Did the hybrid mismatch rules apply to you?', then no response must be provided to the following questions:
            - Have you made or received a payment at any time during the income year which gave rise to a deduction / non-inclusion or a deduction / deduction mismatch?
            - Have you made an importing payment under a structured arrangement?
            - Do you have any other offshore hybrid mismatches within your Division 832 control groups?
            - Did you pay an amount of interest or an amount under a derivative financial arrangement to an international related party which was not taxed or taxed at 10% or less?
    
            Legacy Rule Format:
            ^IDS481 = FALSE AND (^IDS458 <> NULL OR ^IDS501 <> NULL OR ^IDS505 <> NULL OR ^IDS468 <> NULL)

            Technical Business Rule Format:
            ^IDS481 = FALSE AND (^IDS458 <> NULL OR ^IDS501 <> NULL OR ^IDS505 <> NULL OR ^IDS468 <> NULL)
    
            Data Elements:
    
            ^IDS481 = IDS:RP:Miscellaneous.HybridMismatch.Indicator
    
            ^IDS458 = IDS:RP:Miscellaneous.HybridMismatchDeductionNonInclusion.Indicator
    
            ^IDS468 = IDS:RP:Miscellaneous.PaymentNotTaxedOrTaxed10PercentOrLess.Indicator
    
            ^IDS501 = IDS:RP:Miscellaneous.HybridMismatchPayment.Indicator
    
            ^IDS505 = IDS:RP:Miscellaneous.OffshoreHybridMismatch.Indicator
            */
            assertion = (report.IDS481 == false && (report.IDS458 != null || report.IDS501 != null || report.IDS505 != null || report.IDS468 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440657",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information not required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440657" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS481", Value = GetValueOrEmpty(report.IDS481) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS458", Value = GetValueOrEmpty(report.IDS458) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS501", Value = GetValueOrEmpty(report.IDS501) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS505", Value = GetValueOrEmpty(report.IDS505) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS468", Value = GetValueOrEmpty(report.IDS468) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440675
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440675()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440675
            If there is a TRUE response to 'Do you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts?', then following must be completed:
            - Section 23AH Foreign branch income or Section 23AI Amounts paid out of attributed CFC income or Subdivision 768-A foreign equity distributions for Listed, Specified or Other unlisted countries or Non-deductible expenses incurred in deriving section 23AH non-assessable income
            - Response to the question 'Did you have any debt deductions in earning non-assessable non-exempt foreign income?'
            - Response to the question 'Did you have any Capital Gains Tax (CGT) events in relation to your interest in a foreign company?'
            - Response to the question 'If you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts, then did you receive a foreign equity distribution that gave rise to a foreign income tax deduction?'
            - Response to the question 'If you have foreign branch operations or any direct or indirect interests in foreign companies or foreign trusts, then did you derive branch hybrid mismatch income?'
    
            Legacy Rule Format:
            ^IDS195 = TRUE AND ((^IDS1058 = NULL AND ^IDS1059 = NULL AND ^IDS1060 = NULL AND ^IDS199 = NULL) OR ^IDS206 = NULL OR ^IDS208 = NULL OR ^IDS477 = NULL OR ^IDS479 = NULL)

            Technical Business Rule Format:
            ^IDS195 = TRUE AND ((^IDS1058 = NULL AND ^IDS1059 = NULL AND ^IDS1060 = NULL AND ^IDS199 = NULL) OR ^IDS206 = NULL OR ^IDS208 = NULL OR ^IDS477 = NULL OR ^IDS479 = NULL)
    
            Data Elements:
    
            ^IDS195 = IDS:RP:Income.ForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrusts.Indicator
    
            ^IDS199 = IDS:RP:Expense.NonDeductibleExemptIncome.Amount
    
            ^IDS206 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncome.Indicator
    
            ^IDS208 = IDS:RP:CapitalGainsTax.Event.Indicator
    
            ^IDS477 = IDS:RP:Income.ForeignEquityDistributionForeignTaxDeduction.Indicator
    
            ^IDS479 = IDS:RP:Income.DerivedBranchHybridMismatchIncome.Indicator
    
            ^IDS1058 = IDS:RP:ListedForeignBranchOperations
    
            ^IDS1059 = IDS:RP:SpecifiedForeignBranchOperations
    
            ^IDS1060 = IDS:RP:UnlistedForeignBranchOperations
            */
            assertion = (report.IDS195 == true && (report.RP_ListedForeignBranchOperationsCollectionExists == false && report.RP_SpecifiedForeignBranchOperationsCollectionExists == false && report.RP_UnlistedForeignBranchOperationsCollectionExists == false && report.IDS199 == null || report.IDS206 == null || report.IDS208 == null || report.IDS477 == null || report.IDS479 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440675",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Information regarding foreign branch operations or interests in foreign companies or foreign trusts is required.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440675" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS195", Value = GetValueOrEmpty(report.IDS195) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS199", Value = GetValueOrEmpty(report.IDS199) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS206", Value = GetValueOrEmpty(report.IDS206) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS208", Value = GetValueOrEmpty(report.IDS208) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS477", Value = GetValueOrEmpty(report.IDS477) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS479", Value = GetValueOrEmpty(report.IDS479) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440678
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440678()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440678
            If there is a TRUE response to 'Did the hybrid mismatch rules apply to you?', then response must be provided to the following questions:
            - Have you made or received a payment at any time during the income year which gave rise to a deduction / non-inclusion or a deduction / deduction mismatch?
            - Have you made an importing payment under a structured arrangement?
            - Do you have any other offshore hybrid mismatches within your Division 832 control groups?
            - Did you pay an amount of interest or an amount under a derivative financial arrangement to an international related party which was not taxed or taxed at 10% or less?
    
            Legacy Rule Format:
            ^IDS481 = TRUE AND (^IDS458 = NULL OR ^IDS501 = NULL OR ^IDS505 = NULL OR ^IDS468 = NULL)

            Technical Business Rule Format:
            ^IDS481 = TRUE AND (^IDS458 = NULL OR ^IDS501 = NULL OR ^IDS505 = NULL OR ^IDS468 = NULL)
    
            Data Elements:
    
            ^IDS481 = IDS:RP:Miscellaneous.HybridMismatch.Indicator
    
            ^IDS458 = IDS:RP:Miscellaneous.HybridMismatchDeductionNonInclusion.Indicator
    
            ^IDS468 = IDS:RP:Miscellaneous.PaymentNotTaxedOrTaxed10PercentOrLess.Indicator
    
            ^IDS501 = IDS:RP:Miscellaneous.HybridMismatchPayment.Indicator
    
            ^IDS505 = IDS:RP:Miscellaneous.OffshoreHybridMismatch.Indicator
            */
            assertion = (report.IDS481 == true && (report.IDS458 == null || report.IDS501 == null || report.IDS505 == null || report.IDS468 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440678",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch information required.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440678" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS481", Value = GetValueOrEmpty(report.IDS481) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS458", Value = GetValueOrEmpty(report.IDS458) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS501", Value = GetValueOrEmpty(report.IDS501) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS505", Value = GetValueOrEmpty(report.IDS505) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS468", Value = GetValueOrEmpty(report.IDS468) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440681
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440681()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440681
            If there is a TRUE response to 'Do you have an unfranked non portfolio dividend account (refer section 46FB ITAA )?', then 'Balance of unfranked non-portfolio dividend account at year end' and 'Conduit foreign income included in balance of unfranked non-portfolio dividend account at year end - code' must be completed.
    
            Legacy Rule Format:
            ^IDS312 = TRUE AND ^IDS1103 = NULL

            Technical Business Rule Format:
            ^IDS312 = TRUE AND ^IDS1103 = NULL
    
            Data Elements:
    
            ^IDS312 = IDS:RP:InternationalDealings.UnfrankedNonPortfolioDividendAccount.Indicator
    
            ^IDS1103 = IDS:RP:Unfranked
            */
            assertion = (report.IDS312 == true && report.RP_UnfrankedCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440681",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unfranked non portfolio dividend account information is required.",
                    Location = "/tns:IDS/tns:RP/tns:InternationalDealingsUnfrankedNonPortfolioDividendAccountI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440681" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS312", Value = GetValueOrEmpty(report.IDS312) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440682
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440682()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440682
            ABN is invalid.
    
            Legacy Rule Format:
            (^IDS222 <> NULL) and (FailsABNAlgorithm(^IDS222))

            Technical Business Rule Format:
            (^IDS222 <> NULL) and (FailsABNAlgorithm(^IDS222))
    
            Data Elements:
    
            ^IDS222 = IDS:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.IDS222 != null && FailsABNAlgorithm(report.IDS222));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:IDS/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440682" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS222", Value = GetValueOrEmpty(report.IDS222) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440691
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440691()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440691
            If there is a TRUE response to "Have you made an importing payment under a structured arrangement?", then total amount of importing payments made under the structured arrangements, total amount of offshore hybrid mismatch under the structured arrangements and total amount of deductions disallowed for structured arrangements must be completed.
    
            Legacy Rule Format:
            ^IDS501 = TRUE AND ^IDS1109 = NULL

            Technical Business Rule Format:
            ^IDS501 = TRUE AND ^IDS1109 = NULL
    
            Data Elements:
    
            ^IDS501 = IDS:RP:Miscellaneous.HybridMismatchPayment.Indicator
    
            ^IDS1109 = IDS:RP:StructuredArrangement
            */
            assertion = (report.IDS501 == true && report.RP_StructuredArrangementCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440691",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Structured arrangement hybrid mismatch information required.",
                    LongDescription = @"If there is a TRUE response to ""Have you made an importing payment under a structured arrangement?"", then total amount of importing payments made under the structured arrangements, total amount of offshore hybrid mismatch under the structured arrangements and total amount of deductions disallowed for structured arrangements must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchPaymentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440691" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS501", Value = GetValueOrEmpty(report.IDS501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440692
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440692()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440692
            If total amount of importing payments made under the structured arrangements, total amount of offshore hybrid mismatch under the structured arrangements and total amount of deductions disallowed for structured arrangements are provided, then the response to "Have you made an importing payment under a structured arrangement?" must be TRUE.
    
            Legacy Rule Format:
            ^IDS501 <> TRUE AND ^IDS1109 <> NULL

            Technical Business Rule Format:
            ^IDS501 <> TRUE AND ^IDS1109 <> NULL
    
            Data Elements:
    
            ^IDS501 = IDS:RP:Miscellaneous.HybridMismatchPayment.Indicator
    
            ^IDS1109 = IDS:RP:StructuredArrangement
            */
            assertion = (report.IDS501 != true && report.RP_StructuredArrangementCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440692",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Structured arrangement hybrid mismatch question incorrect.",
                    LongDescription = @"If total amount of importing payments made under the structured arrangements, total amount of offshore hybrid mismatch under the structured arrangements and total amount of deductions disallowed for structured arrangements are provided, then the response to ""Have you made an importing payment under a structured arrangement?"" must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchPaymentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440692" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS501", Value = GetValueOrEmpty(report.IDS501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440693
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440693()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440693
            If there is a TRUE response to either "Have you made an importing payment under a structured arrangement?" or "Do you have any other offshore hybrid mismatches within your Division 832 control groups?", then details of material offshore hybrid mismatch/es must be completed.
    
            Legacy Rule Format:
            (^IDS501 = TRUE OR ^IDS505 = TRUE) AND Count(^IDS1111) = 0

            Technical Business Rule Format:
            (^IDS501 = TRUE OR ^IDS505 = TRUE) AND Count(^IDS1111) = 0
    
            Data Elements:
    
            ^IDS501 = IDS:RP:Miscellaneous.HybridMismatchPayment.Indicator
    
            ^IDS505 = IDS:RP:Miscellaneous.OffshoreHybridMismatch.Indicator
    
            ^IDS1111 = IDS:RP:MaterialOffshoreHybridMismatch
            */
            assertion = ((report.IDS501 == true || report.IDS505 == true) && Count(report.RP_MaterialOffshoreHybridMismatchCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440693",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Material offshore hybrid mismatch information required.",
                    LongDescription = @"If there is a TRUE response to either ""Have you made an importing payment under a structured arrangement?"" or ""Do you have any other offshore hybrid mismatches within your Division 832 control groups?"", then details of material offshore hybrid mismatch/es must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchPaymentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440693" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS501", Value = GetValueOrEmpty(report.IDS501) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS505", Value = GetValueOrEmpty(report.IDS505) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440694
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440694()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440694
            If details of material offshore hybrid mismatch/es have been provided, then the response to "Have you made an importing payment under a structured arrangement?" or "Do you have any other offshore hybrid mismatches within your Division 832 control groups?" must be TRUE.
    
            Legacy Rule Format:
            ^IDS501 <> TRUE AND ^IDS505 <> TRUE AND Count(^IDS1111) <> 0

            Technical Business Rule Format:
            ^IDS501 <> TRUE AND ^IDS505 <> TRUE AND Count(^IDS1111) <> 0
    
            Data Elements:
    
            ^IDS501 = IDS:RP:Miscellaneous.HybridMismatchPayment.Indicator
    
            ^IDS505 = IDS:RP:Miscellaneous.OffshoreHybridMismatch.Indicator
    
            ^IDS1111 = IDS:RP:MaterialOffshoreHybridMismatch
            */
            assertion = (report.IDS501 != true && report.IDS505 != true && Count(report.RP_MaterialOffshoreHybridMismatchCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440694",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hybrid mismatch question/s incorrect.",
                    LongDescription = @"If details of material offshore hybrid mismatch/es have been provided, then the response to ""Have you made an importing payment under a structured arrangement?"" or ""Do you have any other offshore hybrid mismatches within your Division 832 control groups?"" must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousHybridMismatchPaymentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440694" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS501", Value = GetValueOrEmpty(report.IDS501) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS505", Value = GetValueOrEmpty(report.IDS505) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440695
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440695()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440695
            If there is a TRUE response to "Do you have any other offshore hybrid mismatches within your Division 832 control groups?", then total amount of offshore hybrid mismatch under non-structured arrangements and total amount of deductions disallowed for non-structured arrangements must be completed.
    
            Legacy Rule Format:
            ^IDS505 = TRUE AND ^IDS1110 = NULL

            Technical Business Rule Format:
            ^IDS505 = TRUE AND ^IDS1110 = NULL
    
            Data Elements:
    
            ^IDS505 = IDS:RP:Miscellaneous.OffshoreHybridMismatch.Indicator
    
            ^IDS1110 = IDS:RP:NonStructuredArrangement
            */
            assertion = (report.IDS505 == true && report.RP_NonStructuredArrangementCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440695",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-structured arrangement hybrid mismatch information required.",
                    LongDescription = @"If there is a TRUE response to ""Do you have any other offshore hybrid mismatches within your Division 832 control groups?"", then total amount of offshore hybrid mismatch under non-structured arrangements and total amount of deductions disallowed for non-structured arrangements must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousOffshoreHybridMismatchI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440695" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS505", Value = GetValueOrEmpty(report.IDS505) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440696
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440696()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440696
            If total amount of offshore hybrid mismatch under non-structured arrangements and total amount of deductions disallowed for non-structured arrangements are provided, then the response to "Do you have any other offshore hybrid mismatches within your Division 832 control groups?" must be TRUE.
    
            Legacy Rule Format:
            ^IDS505 <> TRUE AND ^IDS1110 <> NULL

            Technical Business Rule Format:
            ^IDS505 <> TRUE AND ^IDS1110 <> NULL
    
            Data Elements:
    
            ^IDS505 = IDS:RP:Miscellaneous.OffshoreHybridMismatch.Indicator
    
            ^IDS1110 = IDS:RP:NonStructuredArrangement
            */
            assertion = (report.IDS505 != true && report.RP_NonStructuredArrangementCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440696",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-structured arrangement hybrid mismatch question incorrect.",
                    LongDescription = @"If total amount of offshore hybrid mismatch under non-structured arrangements and total amount of deductions disallowed for non-structured arrangements are provided, then the response to ""Do you have any other offshore hybrid mismatches within your Division 832 control groups?"" must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:MiscellaneousOffshoreHybridMismatchI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440696" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS505", Value = GetValueOrEmpty(report.IDS505) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440700
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440700()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440700
            Total amount claimed or returned for other internally recorded dealings must be provided.
    
            Legacy Rule Format:
            ^IDS1049 <> NULL AND ^IDS169 = NULL AND ^IDS170 = NULL

            Technical Business Rule Format:
            ^IDS1049 <> NULL AND ^IDS169 = NULL AND ^IDS170 = NULL
    
            Data Elements:
    
            ^IDS169 = IDS:RP:BranchOperationsOther:Miscellaneous.InternallyRecordedDealingsAmountsClaimedTotal.Amount
    
            ^IDS170 = IDS:RP:BranchOperationsOther:Miscellaneous.InternallyRecordedDealingsAmountsReturnedTotal.Amount
    
            ^IDS1049 = IDS:RP:BranchOperationsOther
            */
            assertion = (report.RP_BranchOperationsOtherCollectionExists != false && report.IDS169 == null && report.IDS170 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440700",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total amount claimed or returned for other internally recorded dealings must be provided.",
                    Location = "/tns:IDS/tns:RP/tns:BranchOperationsOther/tns:MiscellaneousInternallyRecordedDealingsAmountsClaimedTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440700" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS169", Value = GetValueOrEmpty(report.IDS169) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS170", Value = GetValueOrEmpty(report.IDS170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440750
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440750()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440750
            If there is a FALSE response to 'Were the thin capitalisation rules applicable to you?' then a response must be provided at 'Did you rely on the $2 million threshold exemption?'
    
            Legacy Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = NULL

            Technical Business Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = NULL
    
            Data Elements:
    
            ^IDS400 = IDS:RP:Liabilities.ThinCapitalisation.2MillionThresholdTest.Indicator
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
            */
            assertion = (report.IDS220 == false && report.IDS400 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440750",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A response to 'Did you rely on the $2 million threshold exemption?' is required.",
                    LongDescription = @"If there is a FALSE response to 'Were the thin capitalisation rules applicable to you?' then a response must be provided at 'Did you rely on the $2 million threshold exemption?'",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesThinCapitalisation2MillionThresholdTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440750" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS400", Value = GetValueOrEmpty(report.IDS400) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440751
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440751()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440751
            If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' then a response must be provided at 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?'
    
            Legacy Rule Format:
            ^IDS220 = TRUE AND ^IDS221 = NULL

            Technical Business Rule Format:
            ^IDS220 = TRUE AND ^IDS221 = NULL
    
            Data Elements:
    
            ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
            */
            assertion = (report.IDS220 == true && report.IDS221 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440751",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation information required.",
                    LongDescription = @"If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' then a response must be provided at 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?'",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesThinCapitalisationAustralianResidentCompanyElectedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440751" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS221", Value = GetValueOrEmpty(report.IDS221) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440756
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440756()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440756
            If there is a FALSE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?', then entity Australian Business Number (ABN) must not be completed and 'Entity type code for the income year' must be completed.
    
            Legacy Rule Format:
            ^IDS221 = FALSE AND (^IDS222 <> NULL OR ^IDS223 = NULL)

            Technical Business Rule Format:
            ^IDS221 = FALSE AND (^IDS222 <> NULL OR ^IDS223 = NULL)
    
            Data Elements:
    
            ^IDS222 = IDS:RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
            */
            assertion = (report.IDS221 == false && (report.IDS222 != null || report.IDS223 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440756",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity ABN is not required but Entity type code for the income year must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?', then entity Australian Business Number (ABN) must not be completed and 'Entity type code for the income year' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440756" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS221", Value = GetValueOrEmpty(report.IDS221) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS222", Value = GetValueOrEmpty(report.IDS222) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440757
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440757()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440757
            If there is a TRUE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes? then provide the information at 'ABN of the electing Australian company' and no further information is required in Section D - Thin capitalisation of the schedule.
    
            Legacy Rule Format:
            ^IDS221 = TRUE AND (^IDS222 = NULL OR ^IDS1064 <> NULL)

            Technical Business Rule Format:
            ^IDS221 = TRUE AND (^IDS222 = NULL OR ^IDS1064 <> NULL)
    
            Data Elements:
    
            ^IDS222 = IDS:RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
    
            ^IDS1064 = IDS:RP:ThinCapitalisation
            */
            assertion = (report.IDS221 == true && (report.IDS222 == null || report.RP_ThinCapitalisationCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440757",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Provide the information at ABN of the electing Australian company.",
                    LongDescription = @"If there is a TRUE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes? then provide the information at 'ABN of the electing Australian company' and no further information is required in Section D - Thin capitalisation of the schedule.",
                    Location = "/tns:IDS/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440757" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS221", Value = GetValueOrEmpty(report.IDS221) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS222", Value = GetValueOrEmpty(report.IDS222) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440758
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440758()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440758
            If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' and there is a FALSE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes? and 'Entity type code for the income year' is equal to '2' or '3' then 'Financial entity type code' must be completed.
    
            Legacy Rule Format:
            ^IDS220 = TRUE and ^IDS221 = FALSE and InSet(^IDS223, '"2","3"') AND ^IDS560 = NULL

            Technical Business Rule Format:
            ^IDS220 = TRUE and ^IDS221 = FALSE and InSet(^IDS223, '"2","3"') AND ^IDS560 = NULL
    
            Data Elements:
    
            ^IDS560 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.FinancialEntityType.Code
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
            */
            assertion = (report.IDS220 == true && report.IDS221 == false && IsMatch(report.IDS223, @"^(2|3)$") && report.IDS560 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440758",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial entity type code must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' and there is a FALSE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes? and 'Entity type code for the income year' is equal to '2' or '3' then 'Financial entity type code' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesFinancialEntityTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440758" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS221", Value = GetValueOrEmpty(report.IDS221) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS560", Value = GetValueOrEmpty(report.IDS560) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440783
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440783()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440783
            If 'Entity type code for the income year' is equal to '1', '2' or '3' and there is a TRUE response to 'Did you rely on the third party debt test?', then a response is required at the following labels.
            - Third party debt test choice made (Code)
            - Third party earnings limit
            - Did the holder of a debt interest issued by you that satisfies the third party debt conditions, have recourse to Australian assets that were rights that satisfy paragraph 820-427A(5)(a) and (b)?
            - Did the holder of a debt interest issued by you that satisfies the third party debt conditions have recourse to Australian assets held by another member of the obligor group in relation to the debt interest?
            - Did the holder of a debt interest issued by you have recourse to assets that were membership interests in you that satisfy paragraph 820-427A(4)(b)?
            - Did the holder of a debt interest, that satisfies the third party debt conditions, have recourse to minor or insignificant assets that were disregarded pursuant to paragraph 820-427A(3)(c)?
            - Did you have a conduit financing arrangement?
            - Were you a party to one or more cross-staple arrangements in effect during the income year?
            - Were you a member of an obligor group in relation to a debt interest not issued by you?
    
            Legacy Rule Format:
            InSet(^IDS223, '"1","2","3"') AND ^IDS589 = TRUE AND (^IDS590 = NULL OR ^IDS591 = NULL OR ^IDS592 = NULL OR ^IDS593 = NULL OR ^IDS594 = NULL OR ^IDS595 = NULL OR ^IDS597 = NULL OR ^IDS604 = NULL OR ^IDS605 = NULL)

            Technical Business Rule Format:
            InSet(^IDS223, '"1","2","3"') AND ^IDS589 = TRUE AND (^IDS590 = NULL OR ^IDS591 = NULL OR ^IDS592 = NULL OR ^IDS593 = NULL OR ^IDS594 = NULL OR ^IDS595 = NULL OR ^IDS597 = NULL OR ^IDS604 = NULL OR ^IDS605 = NULL)
    
            Data Elements:
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS590 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.ThirdPartyDebtTestChoice.Code
    
            ^IDS591 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Capital.ThinCapitalisation.EntityThirdPartyEarningsLimit.Amount
    
            ^IDS592 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.Subsections820Dash427A.Indicator
    
            ^IDS593 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.AssetsHeldByAnotherMemberOfTheObligorGroup.Indicator
    
            ^IDS594 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.MembershipInterests820Dash427A.Indicator
    
            ^IDS595 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.MinorOrInsignificantAssetsDisregardedPursuant820Dash427A.Indicator
    
            ^IDS597 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.ConduitFinancingArrangement.Indicator
    
            ^IDS604 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.CrossStapleArrangements.Indicator
    
            ^IDS605 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.MemberOfAnObligorGroupNotIssuedByTheEntity.Indicator
            */
            assertion = (IsMatch(report.IDS223, @"^(1|2|3)$") && report.IDS589 == true && (report.IDS590 == null || report.IDS591 == null || report.IDS592 == null || report.IDS593 == null || report.IDS594 == null || report.IDS595 == null || report.IDS597 == null || report.IDS604 == null || report.IDS605 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440783",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation information required.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '1', '2' or '3' and there is a TRUE response to 'Did you rely on the third party debt test?', then a response is required at the following labels.
- Third party debt test choice made (Code)
- Third party earnings limit
- Did the holder of a debt interest issued by you that satisfies the third party debt conditions, have recourse to Australian assets that were rights that satisfy paragraph 820-427A(5)(a) and (b)?
- Did the holder of a debt interest issued by you that satisfies the third party debt conditions have recourse to Australian assets held by another member of the obligor group in relation to the debt interest?
- Did the holder of a debt interest issued by you have recourse to assets that were membership interests in you that satisfy paragraph 820-427A(4)(b)?
- Did the holder of a debt interest, that satisfies the third party debt conditions, have recourse to minor or insignificant assets that were disregarded pursuant to paragraph 820-427A(3)(c)?
- Did you have a conduit financing arrangement?
- Were you a party to one or more cross-staple arrangements in effect during the income year?
- Were you a member of an obligor group in relation to a debt interest not issued by you?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesThirdPartyDebtTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440783" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS590", Value = GetValueOrEmpty(report.IDS590) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS591", Value = GetValueOrEmpty(report.IDS591) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS592", Value = GetValueOrEmpty(report.IDS592) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS593", Value = GetValueOrEmpty(report.IDS593) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS594", Value = GetValueOrEmpty(report.IDS594) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS595", Value = GetValueOrEmpty(report.IDS595) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS597", Value = GetValueOrEmpty(report.IDS597) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS604", Value = GetValueOrEmpty(report.IDS604) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS605", Value = GetValueOrEmpty(report.IDS605) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440793
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440793()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440793
            If 'Entity type code for the income year' is not equal to '2' or '3', then 'Financial entity type code' must not be provided.
    
            Legacy Rule Format:
            NotInSet(^IDS223, '"2","3"') AND ^IDS560 <> NULL

            Technical Business Rule Format:
            NotInSet(^IDS223, '"2","3"') AND ^IDS560 <> NULL
    
            Data Elements:
    
            ^IDS560 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.FinancialEntityType.Code
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
            */
            assertion = (!(IsMatch(report.IDS223, @"^(2|3)$", RegexOptions.IgnoreCase)) && report.IDS560 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440793",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial entity type code must not be provided.",
                    LongDescription = @"If 'Entity type code for the income year' is not equal to '2' or '3', then 'Financial entity type code' must not be provided.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesFinancialEntityTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440793" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS560", Value = GetValueOrEmpty(report.IDS560) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440795
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440795()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440795
            If 'Entity type code for the income year' is equal to '4' or '5'(ADI) and there is a TRUE response to 'Did you rely on worldwide gearing debt/capital tests?', then labels under 'ADI' must be provided.
    
            Legacy Rule Format:
            InSet(^IDS223, '"4","5"') AND ^IDS257 = TRUE AND ^IDS1071 = NULL

            Technical Business Rule Format:
            InSet(^IDS223, '"4","5"') AND ^IDS257 = TRUE AND ^IDS1071 = NULL
    
            Data Elements:
    
            ^IDS1071 = IDS:RP:ThinCapitalisation:ADI
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
            */
            assertion = (IsMatch(report.IDS223, @"^(4|5)$") && report.IDS257 == true && report.RP_ThinCapitalisation_ADICollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440795",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ADI fields must be provided.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '4' or '5'(ADI) and there is a TRUE response to 'Did you rely on worldwide gearing debt/capital tests?', then labels under 'ADI' must be provided.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:ADI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440795" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440796
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440796()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440796
            If 'Entity type code for the income year' is equal to '2' or '3'(non-ADI) and there is a TRUE response to 'Did you rely on worldwide gearing debt/capital tests?', then labels under 'Non-ADI' must be provided.
    
            Legacy Rule Format:
            InSet(^IDS223, '"2","3"') AND ^IDS257 = TRUE AND ^IDS1072 = NULL

            Technical Business Rule Format:
            InSet(^IDS223, '"2","3"') AND ^IDS257 = TRUE AND ^IDS1072 = NULL
    
            Data Elements:
    
            ^IDS1072 = IDS:RP:ThinCapitalisation:NonADI
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
            */
            assertion = (IsMatch(report.IDS223, @"^(2|3)$") && report.IDS257 == true && report.RP_ThinCapitalisation_NonADICollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440796",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-ADI fields must be provided.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '2' or '3'(non-ADI) and there is a TRUE response to 'Did you rely on worldwide gearing debt/capital tests?', then labels under 'Non-ADI' must be provided.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:NonADI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440796" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440800
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440800()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440800
            If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' and there is an amount greater than zero at 'Debt deductions in earning non-assessable non-exempt foreign income claimed', then it must be less than or equal to 'Debt deductions'.
    
            Legacy Rule Format:
            ^IDS220 = TRUE AND ^IDS207 > 0 AND ^IDS207 > ^IDS226

            Technical Business Rule Format:
            ^IDS220 = TRUE AND ^IDS207 > 0 AND ^IDS207 > ^IDS226
    
            Data Elements:
    
            ^IDS207 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncomeTotal.Amount
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
            */
            assertion = (report.IDS220 == true && report.IDS207.GetValueOrDefault() > 0 && report.IDS207.GetValueOrDefault() > report.IDS226.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440800",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt deductions in earning non-assessable non-exempt foreign income claimed amount incorrect.",
                    LongDescription = @"If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' and there is an amount greater than zero at 'Debt deductions in earning non-assessable non-exempt foreign income claimed', then it must be less than or equal to 'Debt deductions'.",
                    Location = "/tns:IDS/tns:RP/tns:IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440800" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS207", Value = GetValueOrEmpty(report.IDS207) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440801
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440801()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440801
            If the response at 'Number of GR group members' is less than or equal to three, then the occurrence of 'GR Group members with the highest entity EBITDA' information must match with the response at 'Number of GR group members'.
    
            Legacy Rule Format:
            ^IDS576 <= 3 AND Count(^IDS2023) <> ^IDS576

            Technical Business Rule Format:
            ^IDS576 <= 3 AND Count(^IDS2023) <> ^IDS576
    
            Data Elements:
    
            ^IDS2023 = IDS:RP:ThinCapitalisation:GroupRatioTest:GRGroupMembers
    
            ^IDS576 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.GroupEntities.Count
            */
            assertion = (report.IDS576.GetValueOrDefault() <= 3 && Count(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionCount) != report.IDS576.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440801",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"GR Group members with the highest entity EBITDA information required.",
                    LongDescription = @"If the response at 'Number of GR group members' is less than or equal to three, then the occurrence of 'GR Group members with the highest entity EBITDA' information must match with the response at 'Number of GR group members'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest/tns:GRGroupMembersCollection/tns:GRGroupMembers" + OccurrenceIndex(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection == null ? 0 : (report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Count() == 0 ? 0 : report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Last().OccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440801" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS576", Value = GetValueOrEmpty(report.IDS576) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440802
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440802()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440802
            If the response at 'Number of GR group members' is greater than three, then three occurrences of 'GR Group members with the highest entity EBITDA' information must be completed.
    
            Legacy Rule Format:
            ^IDS576 > 3 AND (Count(^IDS2023) <> 3)

            Technical Business Rule Format:
            ^IDS576 > 3 AND (Count(^IDS2023) <> 3)
    
            Data Elements:
    
            ^IDS2023 = IDS:RP:ThinCapitalisation:GroupRatioTest:GRGroupMembers
    
            ^IDS576 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.GroupEntities.Count
            */
            assertion = (report.IDS576.GetValueOrDefault() > 3 && Count(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionCount) != 3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440802",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"GR Group members with the highest entity EBITDA information required.",
                    LongDescription = @"If the response at 'Number of GR group members' is greater than three, then three occurrences of 'GR Group members with the highest entity EBITDA' information must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest/tns:GRGroupMembersCollection/tns:GRGroupMembers" + OccurrenceIndex(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection == null ? 0 : (report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Count() == 0 ? 0 : report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Last().OccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440802" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS576", Value = GetValueOrEmpty(report.IDS576) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440803
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440803()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440803
            If 'Financial entity type code' is provided and there is an amount greater than zero at any of the following labels, then 'Entity type code for the income year' must be '2' or '3'.
            - Financial entity - Average zero capital amount
            - Financial entity - Average on-lent amount
            - Investing financial entity - non-ADI - Average zero capital amount
            - Investing financial entity - non-ADI - Average on-lent amount
    
            Legacy Rule Format:
            ^IDS560 <> NULL AND (^IDS251 > 0 OR ^IDS252 > 0 OR ^IDS622 > 0 OR ^IDS623 > 0) AND NotInSet(^IDS223, '"2","3"')

            Technical Business Rule Format:
            ^IDS560 <> NULL AND (^IDS251 > 0 OR ^IDS252 > 0 OR ^IDS622 > 0 OR ^IDS623 > 0) AND NotInSet(^IDS223, '"2","3"')
    
            Data Elements:
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS251 = IDS:RP:ThinCapitalisation:FinancialEntity:Capital.ThinCapitalisation.ZeroCapitalAverage.Amount
    
            ^IDS252 = IDS:RP:ThinCapitalisation:FinancialEntity:Liabilities.ThinCapitalisation.OnLentAverage.Amount
    
            ^IDS560 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.FinancialEntityType.Code
    
            ^IDS622 = IDS:RP:ThinCapitalisation:InvestingFinancialEntityNonADI:Capital.ThinCapitalisation.ZeroCapitalAverage.Amount
    
            ^IDS623 = IDS:RP:ThinCapitalisation:InvestingFinancialEntityNonADI:Liabilities.ThinCapitalisation.OnLentAverage.Amount
            */
            assertion = (report.IDS560 != null && (report.IDS251.GetValueOrDefault() > 0 || report.IDS252.GetValueOrDefault() > 0 || report.IDS622.GetValueOrDefault() > 0 || report.IDS623.GetValueOrDefault() > 0) && !(IsMatch(report.IDS223, @"^(2|3)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440803",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity type code for the income year must be '2' or '3'.",
                    LongDescription = @"If 'Financial entity type code' is provided and there is an amount greater than zero at any of the following labels, then 'Entity type code for the income year' must be '2' or '3'.
- Financial entity - Average zero capital amount
- Financial entity - Average on-lent amount
- Investing financial entity - non-ADI - Average zero capital amount
- Investing financial entity - non-ADI - Average on-lent amount",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesEntityTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440803" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS560", Value = GetValueOrEmpty(report.IDS560) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS251", Value = GetValueOrEmpty(report.IDS251) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS252", Value = GetValueOrEmpty(report.IDS252) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS622", Value = GetValueOrEmpty(report.IDS622) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS623", Value = GetValueOrEmpty(report.IDS623) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440813
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440813()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440813
            If any of the following labels are provided, then the response to 'Did you rely on the group ratio test?' must be TRUE.
            - GR group net third party interest expense
            - GR group net profit (disregarding tax expenses)
            - GR group adjusted net third party interest expense
            - GR group depreciation & amortisation expenses
            - GR group EBITDA
            - Group ratio
            - Group ratio earnings limit
    
            Legacy Rule Format:
            (^IDS578 <> NULL OR ^IDS579 <> NULL OR ^IDS580 <> NULL OR ^IDS581 <> NULL OR ^IDS582 <> NULL OR ^IDS583 <> NULL OR ^IDS584 <> NULL) AND ^IDS575 <> TRUE

            Technical Business Rule Format:
            (^IDS578 <> NULL OR ^IDS579 <> NULL OR ^IDS580 <> NULL OR ^IDS581 <> NULL OR ^IDS582 <> NULL OR ^IDS583 <> NULL OR ^IDS584 <> NULL) AND ^IDS575 <> TRUE
    
            Data Elements:
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS578 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.NetThirdPartyInterestExpense.Amount
    
            ^IDS579 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.NetProfit.Amount
    
            ^IDS580 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.NetThirdPartyInterestExpenseAdjusted.Amount
    
            ^IDS581 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.DepreciationAndAmortisationExpenses.Amount
    
            ^IDS582 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.GroupTaxEBITDA.Amount
    
            ^IDS583 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.Ratio.Amount
    
            ^IDS584 = IDS:RP:ThinCapitalisation:GroupRatioTest:Capital.ThinCapitalisation.GroupRatioEarningsLimit.Amount
            */
            assertion = ((report.IDS578 != null || report.IDS579 != null || report.IDS580 != null || report.IDS581 != null || report.IDS582 != null || report.IDS583 != null || report.IDS584 != null) && report.IDS575 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440813",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The response to 'Did you rely on the group ratio test?' question must be TRUE.",
                    LongDescription = @"If any of the following labels are provided, then the response to 'Did you rely on the group ratio test?' must be TRUE.
- GR group net third party interest expense
- GR group net profit (disregarding tax expenses)
- GR group adjusted net third party interest expense
- GR group depreciation & amortisation expenses
- GR group EBITDA
- Group ratio
- Group ratio earnings limit",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesGroupRatioTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440813" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS578", Value = GetValueOrEmpty(report.IDS578) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS579", Value = GetValueOrEmpty(report.IDS579) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS580", Value = GetValueOrEmpty(report.IDS580) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS581", Value = GetValueOrEmpty(report.IDS581) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS582", Value = GetValueOrEmpty(report.IDS582) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS583", Value = GetValueOrEmpty(report.IDS583) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS584", Value = GetValueOrEmpty(report.IDS584) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440814
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440814()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440814
            If there is an amount greater than zero at 'Third party earnings limit', then the response to 'Did you rely on the third party debt test?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS591 > 0 AND ^IDS589 <> TRUE

            Technical Business Rule Format:
            ^IDS591 > 0 AND ^IDS589 <> TRUE
    
            Data Elements:
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
    
            ^IDS591 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Capital.ThinCapitalisation.EntityThirdPartyEarningsLimit.Amount
            */
            assertion = (report.IDS591.GetValueOrDefault() > 0 && report.IDS589 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440814",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The response to 'Did you rely on the third party debt test?' question must be TRUE.",
                    LongDescription = @"If there is an amount greater than zero at 'Third party earnings limit', then the response to 'Did you rely on the third party debt test?' must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesThirdPartyDebtTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440814" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS591", Value = GetValueOrEmpty(report.IDS591) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440815
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440815()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440815
            ABN is invalid.
    
            Legacy Rule Format:
            ^IDS599 <> NULL and (FailsABNAlgorithm(^IDS599))

            Technical Business Rule Format:
            ^IDS599 <> NULL and (FailsABNAlgorithm(^IDS599))
    
            Data Elements:
    
            ^IDS599 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.IDS599 != null && FailsABNAlgorithm(report.IDS599));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440815",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:ThirdPartyDebtTest/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440815" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS599", Value = GetValueOrEmpty(report.IDS599) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440816
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440816()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440816
            Name of GR group member must be unique.
    
            Legacy Rule Format:
            HasDuplicateValues(^IDS585)

            Technical Business Rule Format:
            HasDuplicateValues(^IDS585)
    
            Data Elements:
    
            ^IDS585 = IDS:RP:ThinCapitalisation:GroupRatioTest:GRGroupMembers:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = HasDuplicateValues(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection == null ? null : report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Select(f => f.IDS585).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440816",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of GR group member cannot be the same.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest/tns:GRGroupMembersCollection/tns:GRGroupMembers[" + report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection[DuplicateValueIndex(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Select(f => f.IDS585))].OccurrenceIndex + "]/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440816" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440821
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440821()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440821
            If there is a TRUE response to 'Were the debt deduction creation rules applicable to you?' then 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' must be completed.
    
            Legacy Rule Format:
            ^IDS640 = TRUE AND ^IDS628 = NULL

            Technical Business Rule Format:
            ^IDS640 = TRUE AND ^IDS628 = NULL
    
            Data Elements:
    
            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
            */
            assertion = (report.IDS640 == true && report.IDS628 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440821",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Restructuring event information required.",
                    LongDescription = @"If there is a TRUE response to 'Were the debt deduction creation rules applicable to you?' then 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440821" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440822
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440822()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440822
            If there is a TRUE response to 'Certain special purpose entities' then 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' must be TRUE.
    
            Legacy Rule Format:
            ^IDS402 = TRUE AND ^IDS606 <> TRUE

            Technical Business Rule Format:
            ^IDS402 = TRUE AND ^IDS606 <> TRUE
    
            Data Elements:
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
    
            ^IDS402 = IDS:RP:Liabilities.ThinCapitalisation.ExemptionSpecialPurposeEntities.Indicator
            */
            assertion = (report.IDS402 == true && report.IDS606 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440822",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The response to Special purpose entity question must be TRUE.",
                    LongDescription = @"If there is a TRUE response to 'Certain special purpose entities' then a TRUE response must be provided at 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesSpecialPurposeEntityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440822" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS402", Value = GetValueOrEmpty(report.IDS402) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440823
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440823()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440823
            If 'Entity type code for the income year' is equal to '1' then a response must be provided to the following questions:
            - Debt deductions
            - Debt deductions on debt from related non-residents
            - Amount of debt deduction disallowed
            - Adjusted average debt
            - Interest income and other amounts covered by paragraph 820-50(3)(b)
            - Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
            - Tax-EBITDA
            - Did you rely on the group ratio test?
            - Did you rely on the third party debt test?
            - Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?
            - Do you have a FRT disallowed amount from a prior income year?
    
            Legacy Rule Format:
            ^IDS223 = "1" AND (^IDS226 = NULL OR ^IDS227 = NULL OR ^IDS228 = NULL OR ^IDS229 = NULL OR ^IDS563 = NULL OR ^IDS564 = NULL OR ^IDS2020 = NULL OR ^IDS575 = NULL OR ^IDS589 = NULL OR ^IDS606 = NULL OR ^IDS631 = NULL)

            Technical Business Rule Format:
            ^IDS223 = '1' AND (^IDS226 = NULL OR ^IDS227 = NULL OR ^IDS228 = NULL OR ^IDS229 = NULL OR ^IDS563 = NULL OR ^IDS564 = NULL OR ^IDS2020 = NULL OR ^IDS575 = NULL OR ^IDS589 = NULL OR ^IDS606 = NULL OR ^IDS631 = NULL)
    
            Data Elements:
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS227 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS229 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DebtAdjustedAverage.Amount
    
            ^IDS563 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.InterestIncomeAndOtherAmounts.Amount
    
            ^IDS564 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.DerivedNonResidentInterestIncomeAndOtherAmounts.Amount
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
    
            ^IDS631 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmount.Indicator
    
            ^IDS2020 = IDS:RP:ThinCapitalisation:EBITDA
            */
            assertion = (report.IDS223 == @"1" && (report.IDS226 == null || report.IDS227 == null || report.IDS228 == null || report.IDS229 == null || report.IDS563 == null || report.IDS564 == null || report.RP_ThinCapitalisation_EBITDACollectionExists == false || report.IDS575 == null || report.IDS589 == null || report.IDS606 == null || report.IDS631 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440823",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation information required.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '1' then a response must be provided to the following questions: 
- Debt deductions
- Debt deductions on debt from related non-residents
- Amount of debt deduction disallowed
- Adjusted average debt
- Interest income and other amounts covered by paragraph 820-50(3)(b)
- Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
- Tax-EBITDA
- Did you rely on the group ratio test?
- Did you rely on the third party debt test?
- Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?
- Do you have a FRT disallowed amount from a prior income year?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesDeductionDebtA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440823" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS227", Value = GetValueOrEmpty(report.IDS227) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS229", Value = GetValueOrEmpty(report.IDS229) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS563", Value = GetValueOrEmpty(report.IDS563) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS564", Value = GetValueOrEmpty(report.IDS564) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS631", Value = GetValueOrEmpty(report.IDS631) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440825
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440825()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440825
            If 'Entity type code for the income year' is equal to '2' or '3' and 'Financial entity type code' is equal to 'a', 'b', 'c' or 'd' then a response must be provided to the following questions.
            - Average values calculation method code
            - Debt deductions
            - Debt deductions on debt from related non-residents
            - Amount of debt deduction disallowed
            - Adjusted average debt
            - Interest income and other amounts covered by paragraph 820-50(3)(b)
            - Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
            - Did you rely on the third party debt test?
            - Investing financial entity - non-ADI - Average value of assets
            - Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?
            - Investing financial entity - non-ADI
            - Outward investing financial entity
            - Did you rely on worldwide gearing debt/capital tests?
    
            Legacy Rule Format:
            InSet(^IDS223, '"2","3"') 
            AND InSet(^IDS560, '"a","b","c","d"')
            AND (^IDS225 = NULL 
            OR ^IDS226 = NULL 
            OR ^IDS227 = NULL 
            OR ^IDS228 = NULL 
            OR ^IDS229 = NULL 
            OR ^IDS563 = NULL 
            OR ^IDS564 = NULL
            OR ^IDS589 = NULL
            OR ^IDS606 = NULL
            OR ^IDS2026 = NULL 
            OR ^IDS2027 = NULL 
             OR ^IDS257 = NULL)

            Technical Business Rule Format:
            InSet(^IDS223, '"2","3"') 
            AND InSet(^IDS560, '"a","b","c","d"')
            AND (^IDS225 = NULL 
            OR ^IDS226 = NULL 
            OR ^IDS227 = NULL 
            OR ^IDS228 = NULL 
            OR ^IDS229 = NULL 
            OR ^IDS563 = NULL 
            OR ^IDS564 = NULL
            OR ^IDS589 = NULL
            OR ^IDS606 = NULL
            OR ^IDS2026 = NULL 
            OR ^IDS2027 = NULL 
             OR ^IDS257 = NULL)
    
            Data Elements:
    
            ^IDS225 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.AveragingMethodAdopted.Code
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS227 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS229 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DebtAdjustedAverage.Amount
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
    
            ^IDS560 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.FinancialEntityType.Code
    
            ^IDS563 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.InterestIncomeAndOtherAmounts.Amount
    
            ^IDS564 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.DerivedNonResidentInterestIncomeAndOtherAmounts.Amount
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
    
            ^IDS2026 = IDS:RP:ThinCapitalisation:InvestingFinancialEntityNonADI
    
            ^IDS2027 = IDS:RP:ThinCapitalisation:OutwardInvestingFinancialEntity
            */
            assertion = (IsMatch(report.IDS223, @"^(2|3)$") && IsMatch(report.IDS560, @"^(a|b|c|d)$") && (report.IDS225 == null || report.IDS226 == null || report.IDS227 == null || report.IDS228 == null || report.IDS229 == null || report.IDS563 == null || report.IDS564 == null || report.IDS589 == null || report.IDS606 == null || report.RP_ThinCapitalisation_InvestingFinancialEntityNonADICollectionExists == false || report.RP_ThinCapitalisation_OutwardInvestingFinancialEntityCollectionExists == false || report.IDS257 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440825",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation, Investing financial entity - non-ADI and Outward investing financial entity information required.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '2' or '3' and 'Financial entity type code' is equal to 'a', 'b', 'c' or 'd' then a response must be provided to the following questions.
- Average values calculation method code
- Debt deductions
- Debt deductions on debt from related non-residents
- Amount of debt deduction disallowed
- Adjusted average debt
- Interest income and other amounts covered by paragraph 820-50(3)(b)
- Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
- Did you rely on the third party debt test?
- Investing financial entity - non-ADI - Average value of assets
- Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?
- Investing financial entity - non-ADI
- Outward investing financial entity
- Did you rely on worldwide gearing debt/capital tests?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesAveragingMethodAdoptedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440825" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS560", Value = GetValueOrEmpty(report.IDS560) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS225", Value = GetValueOrEmpty(report.IDS225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS227", Value = GetValueOrEmpty(report.IDS227) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS229", Value = GetValueOrEmpty(report.IDS229) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS563", Value = GetValueOrEmpty(report.IDS563) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS564", Value = GetValueOrEmpty(report.IDS564) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440826
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440826()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440826
            If 'Entity type code for the income year' is equal to '4' or '5' then a response must be provided to the following questions.
            - Average values calculation method code
            - Debt deductions
            - Debt deductions on debt from related non-residents
            - Amount of debt deduction disallowed
            - Interest income and other amounts covered by paragraph 820-50(3)(b)
            - Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
            - Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?
            - All ADI entities - (Adjusted) average equity capital
            - Did you rely on worldwide gearing debt/capital tests?
    
            Legacy Rule Format:
            InSet(^IDS223, '"4","5"')
            AND (
            ^IDS225 = NULL
            OR ^IDS226 = NULL
            OR ^IDS227 = NULL
            OR ^IDS228 = NULL
            OR ^IDS563 = NULL
            OR ^IDS564 = NULL
            OR ^IDS606 = NULL
            OR ^IDS1065 = NULL
            OR ^IDS1066 = NULL
            OR ^IDS1067 = NULL
            OR ^IDS257 = NULL)

            Technical Business Rule Format:
            InSet(^IDS223, '"4","5"')
            AND (
            ^IDS225 = NULL
            OR ^IDS226 = NULL
            OR ^IDS227 = NULL
            OR ^IDS228 = NULL
            OR ^IDS563 = NULL
            OR ^IDS564 = NULL
            OR ^IDS606 = NULL
            OR ^IDS1065 = NULL
            OR ^IDS1066 = NULL
            OR ^IDS1067 = NULL
            OR ^IDS257 = NULL)
    
            Data Elements:
    
            ^IDS225 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.AveragingMethodAdopted.Code
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS227 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
    
            ^IDS563 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.InterestIncomeAndOtherAmounts.Amount
    
            ^IDS564 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.DerivedNonResidentInterestIncomeAndOtherAmounts.Amount
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
    
            ^IDS1065 = IDS:RP:ThinCapitalisation:AllADI
    
            ^IDS1066 = IDS:RP:ThinCapitalisation:OutwardInvestingADI
    
            ^IDS1067 = IDS:RP:ThinCapitalisation:InwardInvestingADI
            */
            assertion = (IsMatch(report.IDS223, @"^(4|5)$") && (report.IDS225 == null || report.IDS226 == null || report.IDS227 == null || report.IDS228 == null || report.IDS563 == null || report.IDS564 == null || report.IDS606 == null || report.RP_ThinCapitalisation_AllADICollectionExists == false || report.RP_ThinCapitalisation_OutwardInvestingADICollectionExists == false || report.RP_ThinCapitalisation_InwardInvestingADICollectionExists == false || report.IDS257 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440826",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation, General Information, all ADI entities, Outward investing ADI, and Inward investing ADI information required.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '4' or '5' then a response must be provided to the following questions.
- Average values calculation method code
- Debt deductions
- Debt deductions on debt from related non-residents
- Amount of debt deduction disallowed
- Interest income and other amounts covered by paragraph 820-50(3)(b)
- Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
- Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?
- All ADI entities - (Adjusted) average equity capital
- Did you rely on worldwide gearing debt/capital tests?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesAveragingMethodAdoptedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440826" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS225", Value = GetValueOrEmpty(report.IDS225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS227", Value = GetValueOrEmpty(report.IDS227) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS563", Value = GetValueOrEmpty(report.IDS563) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS564", Value = GetValueOrEmpty(report.IDS564) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440827
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440827()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440827
            If 'Entity type code for the income year' is equal to '6' then responses for the following labels/questions must be supplied:
            - Average values calculations method code
            - Debt deductions
            - Debt deductions on debt from related non-residents
            - Amount of debt deduction disallowed
            - Adjusted average debt
            - Interest income and other amounts covered by paragraph 820-50(3)(b)
            - Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
            - If you were an Australian plantation forestry entity, provide the following info where applicable
            - Did you rely on worldwide gearing debt/capital tests?
    
            Legacy Rule Format:
            ^IDS223 = "6"
            AND (^IDS225 = NULL
            OR ^IDS226 = NULL
            OR ^IDS227 = NULL
            OR ^IDS228 = NULL
            OR ^IDS229 = NULL
            OR ^IDS563 = NULL
            OR ^IDS564 = NULL
            OR ^IDS606 = NULL
            OR ^IDS1068 = NULL 
            OR ^IDS1069 = NULL 
            OR ^IDS1070 = NULL
            OR ^IDS257 = NULL)

            Technical Business Rule Format:
            ^IDS223 = '6'
            AND (^IDS225 = NULL
            OR ^IDS226 = NULL
            OR ^IDS227 = NULL
            OR ^IDS228 = NULL
            OR ^IDS229 = NULL
            OR ^IDS563 = NULL
            OR ^IDS564 = NULL
            OR ^IDS606 = NULL
            OR ^IDS1068 = NULL 
            OR ^IDS1069 = NULL 
            OR ^IDS1070 = NULL
            OR ^IDS257 = NULL)
    
            Data Elements:
    
            ^IDS225 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.AveragingMethodAdopted.Code
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS227 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS229 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DebtAdjustedAverage.Amount
    
            ^IDS257 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GearingDebtCapitalTest.Indicator
    
            ^IDS563 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.InterestIncomeAndOtherAmounts.Amount
    
            ^IDS564 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.DerivedNonResidentInterestIncomeAndOtherAmounts.Amount
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
    
            ^IDS1068 = IDS:RP:ThinCapitalisation:AustralianPlantationForestryEntity
    
            ^IDS1069 = IDS:RP:ThinCapitalisation:FinancialEntity
    
            ^IDS1070 = IDS:RP:ThinCapitalisation:OutwardInvestingEntity
            */
            assertion = (report.IDS223 == @"6" && (report.IDS225 == null || report.IDS226 == null || report.IDS227 == null || report.IDS228 == null || report.IDS229 == null || report.IDS563 == null || report.IDS564 == null || report.IDS606 == null || report.RP_ThinCapitalisation_AustralianPlantationForestryEntityCollectionExists == false || report.RP_ThinCapitalisation_FinancialEntityCollectionExists == false || report.RP_ThinCapitalisation_OutwardInvestingEntityCollectionExists == false || report.IDS257 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440827",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation, Australian plantation forestry entity, Financial entity, Outward investing entity information and a response to 'Did you rely on worldwide gearing debt/capital tests?' is required.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '6' then responses for the following labels/questions must be supplied:
- Average values calculations method code
- Debt deductions
- Debt deductions on debt from related non-residents
- Amount of debt deduction disallowed
- Adjusted average debt
- Interest income and other amounts covered by paragraph 820-50(3)(b)
- Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
- If you were an Australian plantation forestry entity, provide the following info where applicable
- Did you rely on worldwide gearing debt/capital tests?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesAveragingMethodAdoptedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440827" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS225", Value = GetValueOrEmpty(report.IDS225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS227", Value = GetValueOrEmpty(report.IDS227) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS229", Value = GetValueOrEmpty(report.IDS229) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS563", Value = GetValueOrEmpty(report.IDS563) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS564", Value = GetValueOrEmpty(report.IDS564) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS257", Value = GetValueOrEmpty(report.IDS257) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440828
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440828()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440828
            If there is a TRUE response to 'Certain special purpose entities' then a TRUE response must be provided at 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' and a response is required at the following questions.
            - Debt deductions
            - Debt deductions on debt from related non-residents
            - Average debt amount
            - Interest income and other amounts covered by paragraph 820-50(3)(b)
            - Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
    
            Legacy Rule Format:
            ^IDS402 = TRUE AND (^IDS606 <> TRUE OR ^IDS607 = NULL OR ^IDS608 = NULL OR ^IDS609 = NULL OR ^IDS610 = NULL OR ^IDS611 = NULL)

            Technical Business Rule Format:
            ^IDS402 = TRUE AND (^IDS606 <> TRUE OR ^IDS607 = NULL OR ^IDS608 = NULL OR ^IDS609 = NULL OR ^IDS610 = NULL OR ^IDS611 = NULL)
    
            Data Elements:
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
    
            ^IDS402 = IDS:RP:Liabilities.ThinCapitalisation.ExemptionSpecialPurposeEntities.Indicator
    
            ^IDS607 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS608 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS609 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Liabilities.ThinCapitalisation.DebtAverage.Amount
    
            ^IDS610 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Capital.ThinCapitalisation.InterestIncomeAndOtherAmounts.Amount
    
            ^IDS611 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Capital.ThinCapitalisation.DerivedNonResidentInterestIncomeAndOtherAmounts.Amount
            */
            assertion = (report.IDS402 == true && (report.IDS606 != true || report.IDS607 == null || report.IDS608 == null || report.IDS609 == null || report.IDS610 == null || report.IDS611 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440828",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Special entity information is required.",
                    LongDescription = @"If there is a TRUE response to 'Certain special purpose entities' then a TRUE response must be provided at 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' and a response is required at the following questions.
- Debt deductions
- Debt deductions on debt from related non-residents
- Average debt amount
- Interest income and other amounts covered by paragraph 820-50(3)(b)
- Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesSpecialPurposeEntityI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440828" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS402", Value = GetValueOrEmpty(report.IDS402) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS607", Value = GetValueOrEmpty(report.IDS607) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS608", Value = GetValueOrEmpty(report.IDS608) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS609", Value = GetValueOrEmpty(report.IDS609) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS610", Value = GetValueOrEmpty(report.IDS610) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS611", Value = GetValueOrEmpty(report.IDS611) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440831
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440831(IDS2025.RP_RestructuringEventCurrentOrPriorIncomeYear restructuringEventCurrentOrPriorIncomeYear, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440831
            If there is a TRUE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' then a response is required at 'Restructuring event - Description'.

            Legacy Rule Format:
            ^IDS628 = TRUE AND ^IDS629 = NULL

            Technical Business Rule Format:
            ^IDS628 = TRUE AND ^IDS629 = NULL

            Data Elements:

            ^IDS629 = IDS:RP:RestructuringEventCurrentOrPriorIncomeYear:Liabilities.ThinCapitalisation.RestructuringEventsDescription.Text

            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator
            */
            assertion = (report.IDS628 == true && restructuringEventCurrentOrPriorIncomeYear.IDS629 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440831",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Restructuring event - Description must be provided.",
                    LongDescription = @"If there is a TRUE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' then a response is required at 'Restructuring event - Description'.",
                    Location = "/tns:IDS/tns:RP/tns:RestructuringEventCurrentOrPriorIncomeYearCollection/tns:RestructuringEventCurrentOrPriorIncomeYear" + OccurrenceIndex(restructuringEventCurrentOrPriorIncomeYear.OccurrenceIndex) + "/tns:LiabilitiesThinCapitalisationRestructuringEventsDescriptionT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440831" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS629", Value = restructuringEventCurrentOrPriorIncomeYear.IDS629 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440832
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440832()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440832
            If there is an amount greater than zero at 'All ADI entities - (Adjusted) average equity capital' and then 'Entity type code for the income year' must be '4' or '5'.
    
            Legacy Rule Format:
            ^IDS231 > 0 
            AND NotInSet(^IDS223, '"4","5"')

            Technical Business Rule Format:
            ^IDS231 > 0 
            AND NotInSet(^IDS223, '"4","5"')
    
            Data Elements:
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS231 = IDS:RP:ThinCapitalisation:AllADI:Liabilities.ThinCapitalisation.EquityCapitalAdjustedAverage.Amount
            */
            assertion = (report.IDS231.GetValueOrDefault() > 0 && !(IsMatch(report.IDS223, @"^(4|5)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440832",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity type code for the income year must be '4' or '5'.",
                    LongDescription = @"If there is an amount greater than zero at 'All ADI entities - (Adjusted) average equity capital' then 'Entity type code for the income year' must be '4' or '5'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesEntityTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440832" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS231", Value = GetValueOrEmpty(report.IDS231) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440836
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440836()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440836
            If there is a TRUE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' then 'Special purpose entity' must be completed.
    
            Legacy Rule Format:
            ^IDS606 = TRUE
            AND ^IDS2025 = NULL

            Technical Business Rule Format:
            ^IDS606 = TRUE
            AND ^IDS2025 = NULL
    
            Data Elements:
    
            ^IDS2025 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
            */
            assertion = (report.IDS606 == true && report.RP_ThinCapitalisation_SpecialPurposeEntityCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440836",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Special purpose entity is required.",
                    LongDescription = @"If there is a TRUE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' then 'Special purpose entity' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:SpecialPurposeEntity",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440836" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440843
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440843()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440843
            If 'Entity type code for the income year' is equal to '2' or '3' and a FALSE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' then 'Were the debt deduction creation rules applicable to you?' must be completed.
    
            Legacy Rule Format:
            InSet(^IDS223, '"2","3"')
            AND ^IDS606 = FALSE
            AND ^IDS640 = NULL

            Technical Business Rule Format:
            InSet(^IDS223, '"2","3"')
            AND ^IDS606 = FALSE
            AND ^IDS640 = NULL
    
            Data Elements:
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
            */
            assertion = (IsMatch(report.IDS223, @"^(2|3)$") && report.IDS606 == false && report.IDS640 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440843",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt deduction creation rules applicable to you must be completed.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '2' or '3' and a FALSE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' then 'Were the debt deduction creation rules applicable to you?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440843" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440848
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440848()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440848
            If there is a FALSE response to 'Did you rely on the group ratio test?', then 'Group ratio test' must not be completed.
    
            Legacy Rule Format:
            ^IDS575 = FALSE AND ^IDS2022 <> NULL

            Technical Business Rule Format:
            ^IDS575 = FALSE AND ^IDS2022 <> NULL
    
            Data Elements:
    
            ^IDS2022 = IDS:RP:ThinCapitalisation:GroupRatioTest
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
            */
            assertion = (report.IDS575 == false && report.RP_ThinCapitalisation_GroupRatioTestCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440848",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Group ratio test must not be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did you rely on the group ratio test?', then 'Group ratio test' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440848" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440849
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440849()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440849
            If there is a TRUE response to 'Did you rely on the third party debt test?' then 'Third party debt test' must be completed.
    
            Legacy Rule Format:
            ^IDS589 = TRUE AND ^IDS2024 = NULL

            Technical Business Rule Format:
            ^IDS589 = TRUE AND ^IDS2024 = NULL
    
            Data Elements:
    
            ^IDS2024 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
            */
            assertion = (report.IDS589 == true && report.RP_ThinCapitalisation_ThirdPartyDebtTestCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440849",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Third party debt test is required.",
                    LongDescription = @"If there is a TRUE response to 'Did you rely on the third party debt test?' then 'Third party debt test' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:ThirdPartyDebtTest",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440849" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440855
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440855()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440855
            If there is a FALSE response to 'Were the thin capitalisation rules applicable to you?' and a TRUE response to 'Did you rely on the $2 million threshold exemption?', then 'Were the debt deduction creation rules applicable to you?' must be completed.
    
            Legacy Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = TRUE AND ^IDS640 = NULL

            Technical Business Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = TRUE AND ^IDS640 = NULL
    
            Data Elements:
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS400 = IDS:RP:Liabilities.ThinCapitalisation.2MillionThresholdTest.Indicator
            */
            assertion = (report.IDS220 == false && report.IDS400 == true && report.IDS640 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440855",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt deduction creation rules applicable to you must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were the thin capitalisation rules applicable to you?' and a TRUE response to 'Did you rely on the $2 million threshold exemption?', then 'Were the debt deduction creation rules applicable to you?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440855" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS400", Value = GetValueOrEmpty(report.IDS400) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440859
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440859()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440859
            If there are FALSE responses to 'Were the thin capitalisation rules applicable to you?', 'Did you rely on the $2 million threshold exemption?', '90% asset threshold' and 'Certain special purpose entities' and a TRUE response to 'Australian resident entity that is neither an inward nor outward investing entity' then 'Were the debt deduction creation rules applicable to you?' must be completed.
    
            Legacy Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = FALSE AND ^IDS401 = FALSE AND ^IDS402 = FALSE AND ^IDS630 = TRUE AND ^IDS640 = NULL

            Technical Business Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = FALSE AND ^IDS401 = FALSE AND ^IDS402 = FALSE AND ^IDS630 = TRUE AND ^IDS640 = NULL
    
            Data Elements:
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS400 = IDS:RP:Liabilities.ThinCapitalisation.2MillionThresholdTest.Indicator
    
            ^IDS401 = IDS:RP:Liabilities.ThinCapitalisation.90PercentAssetThresholdTest.Indicator
    
            ^IDS402 = IDS:RP:Liabilities.ThinCapitalisation.ExemptionSpecialPurposeEntities.Indicator
    
            ^IDS630 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentEntityNonInOutInvestingEntity.Indicator
            */
            assertion = (report.IDS220 == false && report.IDS400 == false && report.IDS401 == false && report.IDS402 == false && report.IDS630 == true && report.IDS640 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440859",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt deduction creation rules applicable to you must be completed.",
                    LongDescription = @"If there are FALSE responses to 'Were the thin capitalisation rules applicable to you?', 'Did you rely on the $2 million threshold exemption?', '90% asset threshold' and 'Certain special purpose entities' and a TRUE response to 'Australian resident entity that is neither an inward nor outward investing entity' then 'Were the debt deduction creation rules applicable to you?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440859" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS400", Value = GetValueOrEmpty(report.IDS400) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS401", Value = GetValueOrEmpty(report.IDS401) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS402", Value = GetValueOrEmpty(report.IDS402) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS630", Value = GetValueOrEmpty(report.IDS630) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440863
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440863()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440863
            If there is a TRUE response to '90% asset threshold' and a FALSE response to 'Were the debt deduction creation rules applicable to you?' then 'Debt Deduction Creation Rules choice (Code)' must be '3'.
    
            Legacy Rule Format:
            ^IDS401 = TRUE AND ^IDS640 = FALSE AND ^IDS641 <> "3"

            Technical Business Rule Format:
            ^IDS401 = TRUE AND ^IDS640 = FALSE AND ^IDS641 <> '3'
    
            Data Elements:
    
            ^IDS641 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtChoice.Code
    
            ^IDS401 = IDS:RP:Liabilities.ThinCapitalisation.90PercentAssetThresholdTest.Indicator
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
            */
            assertion = (report.IDS401 == true && report.IDS640 == false && report.IDS641 != @"3");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440863",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Debt Deduction Creation Rules choice (Code) must be '3'.",
                    LongDescription = @"If there is a TRUE response to '90% asset threshold' and a FALSE response to 'Were the debt deduction creation rules applicable to you?' then 'Debt Deduction Creation Rules choice (Code)' must be '3'.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440863" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS401", Value = GetValueOrEmpty(report.IDS401) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS641", Value = GetValueOrEmpty(report.IDS641) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440865
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440865()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440865
            If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?', then the following must not be completed:
            - Did you rely on the $2 million threshold exemption?
            - 90% asset threshold
            - Certain special purpose entities
            - You were an Australian resident entity that is neither an inward nor outward investing entity and it is exempted from the thin capitalisation rules.
    
            Legacy Rule Format:
            ^IDS220 = TRUE AND 
            (^IDS400 <> NULL OR
            ^IDS401 <> NULL OR
            ^IDS402 <> NULL OR
            ^IDS630 <> NULL)

            Technical Business Rule Format:
            ^IDS220 = TRUE AND 
            (^IDS400 <> NULL OR
            ^IDS401 <> NULL OR
            ^IDS402 <> NULL OR
            ^IDS630 <> NULL)
    
            Data Elements:
    
            ^IDS400 = IDS:RP:Liabilities.ThinCapitalisation.2MillionThresholdTest.Indicator
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS401 = IDS:RP:Liabilities.ThinCapitalisation.90PercentAssetThresholdTest.Indicator
    
            ^IDS402 = IDS:RP:Liabilities.ThinCapitalisation.ExemptionSpecialPurposeEntities.Indicator
    
            ^IDS630 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentEntityNonInOutInvestingEntity.Indicator
            */
            assertion = (report.IDS220 == true && (report.IDS400 != null || report.IDS401 != null || report.IDS402 != null || report.IDS630 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440865",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The reponses to the following are not required:
- Did you rely on the $2 million threshold exemption?
- 90% asset threshold
- Certain special purpose entities
- Australian resident entity that is neither an inward nor outward investing entity",
                    LongDescription = @"If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?', then the following must not be completed:
- Did you rely on the $2 million threshold exemption?
- 90% asset threshold
- Certain special purpose entities
- You were an Australian resident entity that is neither an inward nor outward investing entity and it is exempted from the thin capitalisation rules.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesThinCapitalisation2MillionThresholdTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440865" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS400", Value = GetValueOrEmpty(report.IDS400) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS401", Value = GetValueOrEmpty(report.IDS401) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS402", Value = GetValueOrEmpty(report.IDS402) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS630", Value = GetValueOrEmpty(report.IDS630) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440870
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440870()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440870
            If there are FALSE responses to 'Were the thin capitalisation rules applicable to you?' and 'Did you rely on the $2 million threshold exemption?' then at least one TRUE response is expected for '90% asset threshold', 'Certain special purpose entities' and 'You were an Australian resident entity that is neither an inward nor outward investing entity and it is exempted from the thin capitalisation rules.'
    
            Legacy Rule Format:
            ^IDS220 = FALSE
            AND ^IDS400 = FALSE
            AND (^IDS401 <> TRUE
            AND ^IDS402 <> TRUE
            AND ^IDS630 <> TRUE)

            Technical Business Rule Format:
            ^IDS220 = FALSE
            AND ^IDS400 = FALSE
            AND (^IDS401 <> TRUE
            AND ^IDS402 <> TRUE
            AND ^IDS630 <> TRUE)
    
            Data Elements:
    
            ^IDS401 = IDS:RP:Liabilities.ThinCapitalisation.90PercentAssetThresholdTest.Indicator
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS400 = IDS:RP:Liabilities.ThinCapitalisation.2MillionThresholdTest.Indicator
    
            ^IDS402 = IDS:RP:Liabilities.ThinCapitalisation.ExemptionSpecialPurposeEntities.Indicator
    
            ^IDS630 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentEntityNonInOutInvestingEntity.Indicator
            */
            assertion = (report.IDS220 == false && report.IDS400 == false && (report.IDS401 != true && report.IDS402 != true && report.IDS630 != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440870",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least a TRUE response is required at either '90% asset threshold' or 'Certain special purpose entities', or 'You were an Australian resident entity that is neither an inward nor outward investing entity and it is exempted from the thin capitalisation rules.'",
                    LongDescription = @"If there are FALSE responses to 'Were the thin capitalisation rules applicable to you?' and 'Did you rely on the $2 million threshold exemption?'then at least one TRUE response is expected for '90% asset threshold', 'Certain special purpose entities' and 'You were an Australian resident entity that is neither an inward nor outward investing entity and it is exempted from the thin capitalisation rules.'",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesThinCapitalisation90PercentAssetThresholdTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440870" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS400", Value = GetValueOrEmpty(report.IDS400) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS401", Value = GetValueOrEmpty(report.IDS401) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS402", Value = GetValueOrEmpty(report.IDS402) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS630", Value = GetValueOrEmpty(report.IDS630) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440871
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440871()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440871
            If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' and a FALSE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?' then the following must be completed:
            - Debt deductions
            - Debt deductions on debt from related non-residents
            - Amount of debt deduction disallowed
            - Interest income and other amounts covered by paragraph 820-50(3)(b)
            - Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents
    
            Legacy Rule Format:
            ^IDS220 = TRUE
            AND ^IDS221 = FALSE
            AND (^IDS226 = NULL
            OR ^IDS227 = NULL
            OR ^IDS228 = NULL
            OR ^IDS563 = NULL
            OR ^IDS564 = NULL)

            Technical Business Rule Format:
            ^IDS220 = TRUE
            AND ^IDS221 = FALSE
            AND (^IDS226 = NULL
            OR ^IDS227 = NULL
            OR ^IDS228 = NULL
            OR ^IDS563 = NULL
            OR ^IDS564 = NULL)
    
            Data Elements:
    
            ^IDS226 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebt.Amount
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
    
            ^IDS227 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS563 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.InterestIncomeAndOtherAmounts.Amount
    
            ^IDS564 = IDS:RP:ThinCapitalisation:Capital.ThinCapitalisation.DerivedNonResidentInterestIncomeAndOtherAmounts.Amount
            */
            assertion = (report.IDS220 == true && report.IDS221 == false && (report.IDS226 == null || report.IDS227 == null || report.IDS228 == null || report.IDS563 == null || report.IDS564 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440871",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation information required.",
                    LongDescription = @"If there is a TRUE response to 'Were the thin capitalisation rules applicable to you?' and a FALSE response to 'Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?' then the following must be completed:
- Debt deductions
- Debt deductions on debt from related non-residents
- Amount of debt deduction disallowed
- Interest income and other amounts covered by paragraph 820-50(3)(b)
- Interest income and other amounts covered by paragraph 820-50(3)(b) derived from related non-residents",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesDeductionDebtA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440871" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS221", Value = GetValueOrEmpty(report.IDS221) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS226", Value = GetValueOrEmpty(report.IDS226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS227", Value = GetValueOrEmpty(report.IDS227) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS563", Value = GetValueOrEmpty(report.IDS563) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS564", Value = GetValueOrEmpty(report.IDS564) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440881
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440881()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440881
            If there is a FALSE response to 'Do you have a FRT disallowed amount from a prior income year?' then the 'Fixed Ratio Test disallowed code' must be either '1', '2', or '3'.
    
            Legacy Rule Format:
            ^IDS631 = FALSE AND NotInSet(^IDS632, '"1","2","3"')

            Technical Business Rule Format:
            ^IDS631 = FALSE AND NotInSet(^IDS632, '"1","2","3"')
    
            Data Elements:
    
            ^IDS632 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowed.Code
    
            ^IDS631 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmount.Indicator
            */
            assertion = (report.IDS631 == false && !(IsMatch(report.IDS632, @"^(1|2|3)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440881",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Fixed ratio test disallowed code must be '1', '2' or '3'.",
                    LongDescription = @"If there is a FALSE response to 'Do you have a FRT disallowed amount from a prior income year?' then the 'Fixed Ratio Test disallowed code' must be either '1', '2', or '3'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440881" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS631", Value = GetValueOrEmpty(report.IDS631) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS632", Value = GetValueOrEmpty(report.IDS632) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440882
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440882()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440882
            If there is a FALSE response to 'Do you have a FRT disallowed amount from a prior income year?' then the following must not be completed:
            - FRT disallowed amount applied in this income year
            - FRT disallowed amount to be carried forward to future income year
            - Were you the head company of a tax consolidated group or MEC group?
            - Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?
            - Provide the amount calculated according to subsection 705-112(2)
    
            Legacy Rule Format:
            ^IDS631 = FALSE AND (
            ^IDS633 <> NULL OR
            ^IDS634 <> NULL OR
            ^IDS635 <> NULL OR
            ^IDS636 <> NULL OR
            ^IDS637 <> NULL)

            Technical Business Rule Format:
            ^IDS631 = FALSE AND (
            ^IDS633 <> NULL OR
            ^IDS634 <> NULL OR
            ^IDS635 <> NULL OR
            ^IDS636 <> NULL OR
            ^IDS637 <> NULL)
    
            Data Elements:
    
            ^IDS633 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAppliedIncomeYear.Amount
    
            ^IDS631 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmount.Indicator
    
            ^IDS634 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedCarriedForward.Amount
    
            ^IDS635 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.TaxConsolidatedGroupMultipleEntryConsolidated.Indicator
    
            ^IDS636 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmountsTransferred.Indicator
    
            ^IDS637 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedCalculated.Amount
            */
            assertion = (report.IDS631 == false && (report.IDS633 != null || report.IDS634 != null || report.IDS635 != null || report.IDS636 != null || report.IDS637 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440882",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FRT disallowed amounts along with related information must not be completed.",
                    LongDescription = @"If there is a FALSE response to 'Do you have a FRT disallowed amount from a prior income year?' then the following must not be completed:
- FRT disallowed amount applied in this income year
- FRT disallowed amount to be carried forward to future income year
- Were you the head company of a tax consolidated group or MEC group?
- Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?
- Provide the amount calculated according to subsection 705-112(2)",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440882" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS631", Value = GetValueOrEmpty(report.IDS631) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS633", Value = GetValueOrEmpty(report.IDS633) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS634", Value = GetValueOrEmpty(report.IDS634) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS635", Value = GetValueOrEmpty(report.IDS635) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS636", Value = GetValueOrEmpty(report.IDS636) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS637", Value = GetValueOrEmpty(report.IDS637) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440883
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440883()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440883
            If there is a TRUE response to 'Do you have a FRT disallowed amount from a prior income year?' then the following labels must be provided:
            - FRT disallowed amount applied in this income year 
            - FRT disallowed amount to be carried forward to future income year
            - Were you the head company of a tax consolidated group or MEC group?
    
            Legacy Rule Format:
            ^IDS631 = TRUE
            AND (^IDS633 = NULL
            OR ^IDS634 = NULL
            OR ^IDS635 = NULL)

            Technical Business Rule Format:
            ^IDS631 = TRUE
            AND (^IDS633 = NULL
            OR ^IDS634 = NULL
            OR ^IDS635 = NULL)
    
            Data Elements:
    
            ^IDS633 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAppliedIncomeYear.Amount
    
            ^IDS631 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmount.Indicator
    
            ^IDS634 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedCarriedForward.Amount
    
            ^IDS635 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.TaxConsolidatedGroupMultipleEntryConsolidated.Indicator
            */
            assertion = (report.IDS631 == true && (report.IDS633 == null || report.IDS634 == null || report.IDS635 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440883",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following labels must be provided:
- FRT disallowed amount applied in this income year 
- FRT disallowed amount to be carried forward to future income year
- Were you the head company of a tax consolidated group or MEC group?",
                    LongDescription = @"If there is a TRUE response to 'Do you have a FRT disallowed amount from a prior income year?' then the following labels must be provided:
- FRT disallowed amount applied in this income year 
- FRT disallowed amount to be carried forward to future income year
- Were you the head company of a tax consolidated group or MEC group?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440883" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS631", Value = GetValueOrEmpty(report.IDS631) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS633", Value = GetValueOrEmpty(report.IDS633) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS634", Value = GetValueOrEmpty(report.IDS634) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS635", Value = GetValueOrEmpty(report.IDS635) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440885
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440885()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440885
            If the response to 'Do you have a FRT disallowed amount from a prior income year?' is TRUE then 'Fixed Ratio Test disallowed code' must not be completed.
    
            Legacy Rule Format:
            ^IDS631 = TRUE AND ^IDS632 <> NULL

            Technical Business Rule Format:
            ^IDS631 = TRUE AND ^IDS632 <> NULL
    
            Data Elements:
    
            ^IDS632 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowed.Code
    
            ^IDS631 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmount.Indicator
            */
            assertion = (report.IDS631 == true && report.IDS632 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440885",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Fixed ratio test disallowed code must not be completed.",
                    LongDescription = @"If the response to 'Do you have a FRT disallowed amount from a prior income year?' is TRUE then 'Fixed Ratio Test disallowed code' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440885" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS631", Value = GetValueOrEmpty(report.IDS631) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS632", Value = GetValueOrEmpty(report.IDS632) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440889
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440889()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440889
            If the response to 'Were you the head company of a tax consolidated group or MEC group?' is FALSE, then the following must not be completed:
            - Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?
            - Provide the amount calculated according to subsection 705-112(2)
    
            Legacy Rule Format:
            ^IDS635 = FALSE AND
            (^IDS636 <> NULL 
            OR ^IDS637 <> NULL)

            Technical Business Rule Format:
            ^IDS635 = FALSE AND
            (^IDS636 <> NULL 
            OR ^IDS637 <> NULL)
    
            Data Elements:
    
            ^IDS636 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmountsTransferred.Indicator
    
            ^IDS635 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.TaxConsolidatedGroupMultipleEntryConsolidated.Indicator
    
            ^IDS637 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedCalculated.Amount
            */
            assertion = (report.IDS635 == false && (report.IDS636 != null || report.IDS637 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440889",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following labels must not be provided:
- Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?
- Provide the amount calculated according to subsection 705-112(2)",
                    LongDescription = @"If the response to 'Were you the head company of a tax consolidated group or MEC group?' is FALSE, then the following must not be completed:
- Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?
- Provide the amount calculated according to subsection 705-112(2)",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedAmountsTransferredI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440889" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS635", Value = GetValueOrEmpty(report.IDS635) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS636", Value = GetValueOrEmpty(report.IDS636) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS637", Value = GetValueOrEmpty(report.IDS637) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440891
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440891()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440891
            If there is a TRUE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' then 'Restructuring Event' must be completed.
    
            Legacy Rule Format:
            ^IDS628 = TRUE
            AND Count(^IDS2030) = 0

            Technical Business Rule Format:
            ^IDS628 = TRUE
            AND Count(^IDS2030) = 0
    
            Data Elements:
    
            ^IDS2030 = IDS:RP:RestructuringEventCurrentOrPriorIncomeYear
    
            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator
            */
            assertion = (report.IDS628 == true && Count(report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440891",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Restructuring event is required.",
                    LongDescription = @"If there is a TRUE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' then 'Restructuring Event' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:RestructuringEventCurrentOrPriorIncomeYearCollection/tns:RestructuringEventCurrentOrPriorIncomeYear" + OccurrenceIndex(report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection == null ? 0 : (report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection.Count() == 0 ? 0 : report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection.Last().OccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440891" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440897
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440897()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440897
            If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?', then 'Total debt deductions disallowed due to subsection 820-423A(2)' and 'Prior Income Year Top Material Arrangements' must be completed.
    
            Legacy Rule Format:
            ^IDS645 = TRUE AND (^IDS646 = NULL OR Count(^IDS2031) = 0)

            Technical Business Rule Format:
            ^IDS645 = TRUE AND (^IDS646 = NULL OR Count(^IDS2031) = 0)
    
            Data Elements:
    
            ^IDS646 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearCGT.Amount
    
            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator
    
            ^IDS2031 = IDS:RP:PriorIncomeYearTopMaterialArrangements
            */
            assertion = (report.IDS645 == true && (report.IDS646 == null || Count(report.RP_PriorIncomeYearTopMaterialArrangementsCollectionCount) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440897",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Total debt deductions disallowed due to subsection 820-423A(2)' and the Prior Income Year Top Material Arrangements section is required.",
                    LongDescription = @"If there is a response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?' and 'Prior Income Year Top Material Arrangements', then there should be at least one occurrence of 'Prior Income Year Top Material Arrangements' and no more than three.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440897" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS646", Value = GetValueOrEmpty(report.IDS646) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440898
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440898()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440898
            If there is a FALSE response to 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS643 = FALSE AND ^IDS650 = NULL

            Technical Business Rule Format:
            ^IDS643 = FALSE AND ^IDS650 = NULL
    
            Data Elements:
    
            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator
    
            ^IDS643 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairPriorIncomeYear.Indicator
            */
            assertion = (report.IDS643 == false && report.IDS650 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440898",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440898" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS643", Value = GetValueOrEmpty(report.IDS643) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440899
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440899(IDS2025.RP_PriorIncomeYearTopMaterialArrangements priorIncomeYearTopMaterialArrangements, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440899
            If there is a FALSE response to 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?', then the following must not be completed:
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?
            - Total debt deductions disallowed due to subsection 820-423A(2)
            - Prior Income Year Top Material Arrangements

            Legacy Rule Format:
            ^IDS643 = FALSE AND (
            ^IDS644 <> NULL OR
            ^IDS645 <> NULL OR
            ^IDS646 <> NULL OR
            ^IDS2031 <> NULL)

            Technical Business Rule Format:
            ^IDS643 = FALSE AND (
            ^IDS644 <> NULL OR
            ^IDS645 <> NULL OR
            ^IDS646 <> NULL OR
            ^IDS2031 <> NULL)

            Data Elements:

            ^IDS644 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssetAcquisition.Indicator

            ^IDS643 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairPriorIncomeYear.Indicator

            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator

            ^IDS646 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearCGT.Amount

            ^IDS2031 = IDS:RP:PriorIncomeYearTopMaterialArrangements
            */
            assertion = (report.IDS643 == false && (report.IDS644 != null || report.IDS645 != null || report.IDS646 != null || report.RP_PriorIncomeYearTopMaterialArrangementsCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440899",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a FALSE response to 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?', then the following must not be completed:
- Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed due to subsection 820-423A(2)
- Prior Income Year Top Material Arrangements",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440899" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS643", Value = GetValueOrEmpty(report.IDS643) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS644", Value = GetValueOrEmpty(report.IDS644) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS646", Value = GetValueOrEmpty(report.IDS646) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440900
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440900()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440900
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS644 = FALSE AND ^IDS650 = NULL

            Technical Business Rule Format:
            ^IDS644 = FALSE AND ^IDS650 = NULL
    
            Data Elements:
    
            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator
    
            ^IDS644 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssetAcquisition.Indicator
            */
            assertion = (report.IDS644 == false && report.IDS650 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440900",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440900" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS644", Value = GetValueOrEmpty(report.IDS644) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440901
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440901(IDS2025.RP_PriorIncomeYearTopMaterialArrangements priorIncomeYearTopMaterialArrangements, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440901
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?' then the following must not be completed:
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?
            - Total debt deductions disallowed due to subsection 820-423A(2)
            - Prior Income Year Top Material Arrangements

            Legacy Rule Format:
            ^IDS644 = FALSE AND (^IDS645 <> NULL OR ^IDS646 <> NULL OR ^IDS2031 <> NULL)

            Technical Business Rule Format:
            ^IDS644 = FALSE AND (^IDS645 <> NULL OR ^IDS646 <> NULL OR ^IDS2031 <> NULL)

            Data Elements:

            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator

            ^IDS644 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssetAcquisition.Indicator

            ^IDS646 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearCGT.Amount

            ^IDS2031 = IDS:RP:PriorIncomeYearTopMaterialArrangements
            */
            assertion = (report.IDS644 == false && (report.IDS645 != null || report.IDS646 != null || report.RP_PriorIncomeYearTopMaterialArrangementsCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440901",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must not be completed:
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed due to subsection 820-423A(2)
- Prior Income Year Top Material Arrangements section",
                    LongDescription = @"If there is a TRUE response to 'In a prior income year, did you directly or indirectly acquire a GCT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?' and 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?', and a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440901" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS644", Value = GetValueOrEmpty(report.IDS644) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS646", Value = GetValueOrEmpty(report.IDS646) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440902
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440902()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440902
            If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS645 = FALSE AND ^IDS650 = NULL

            Technical Business Rule Format:
            ^IDS645 = FALSE AND ^IDS650 = NULL
    
            Data Elements:
    
            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator
    
            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator
            */
            assertion = (report.IDS645 == false && report.IDS650 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440902",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?', then 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440902" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440903
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440903(IDS2025.RP_PriorIncomeYearTopMaterialArrangements priorIncomeYearTopMaterialArrangements, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440903
            If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?', then the following must not be completed:
            - Total debt deductions disallowed due to subsection 820-423A(2)
            - Prior Income Year Top Material Arrangements

            Legacy Rule Format:
            ^IDS645 = FALSE AND (^IDS646 <> NULL OR ^IDS2031 <> NULL)

            Technical Business Rule Format:
            ^IDS645 = FALSE AND (^IDS646 <> NULL OR ^IDS2031 <> NULL)

            Data Elements:

            ^IDS646 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearCGT.Amount

            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator

            ^IDS2031 = IDS:RP:PriorIncomeYearTopMaterialArrangements
            */
            assertion = (report.IDS645 == false && (report.IDS646 != null || report.RP_PriorIncomeYearTopMaterialArrangementsCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440903",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must not be completed:
- Total debt deductions disallowed due to subsection 820-423A(2)
- Prior Income Year Top Material Arrangements section",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?', then the following must not be completed:
- Total debt deductions disallowed due to subsection 820-423A(2)
- Prior Income Year Top Material Arrangements",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440903" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS646", Value = GetValueOrEmpty(report.IDS646) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440904
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440904()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440904
            If there is a FALSE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS650 = FALSE AND ^IDS658 = NULL

            Technical Business Rule Format:
            ^IDS650 = FALSE AND ^IDS658 = NULL
    
            Data Elements:
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
    
            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator
            */
            assertion = (report.IDS650 == false && report.IDS658 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440904",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440904" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440905
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440905(IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440905
            If there is a FALSE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?', then the following must not be completed:
            - Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)? 
            - Total debt deductions disallowed by ss820-423A(5)
            - Prior Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS650 = FALSE AND
            (^IDS651 <> NULL OR
            ^IDS652 <> NULL OR
            ^IDS653 <> NULL OR
            ^IDS654 <> NULL OR
            ^IDS2032 <> NULL)

            Technical Business Rule Format:
            ^IDS650 = FALSE AND
            (^IDS651 <> NULL OR
            ^IDS652 <> NULL OR
            ^IDS653 <> NULL OR
            ^IDS654 <> NULL OR
            ^IDS2032 <> NULL)

            Data Elements:

            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator

            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator

            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator

            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator

            ^IDS654 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearDistributions.Amount

            ^IDS2032 = IDS:RP:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS650 == false && (report.IDS651 != null || report.IDS652 != null || report.IDS653 != null || report.IDS654 != null || report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440905",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a FALSE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?', then the following must not be completed:
- Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?
- Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)? 
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440905" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS654", Value = GetValueOrEmpty(report.IDS654) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440906
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440906()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440906
            If there is a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS651 = FALSE AND ^IDS658 = NULL

            Technical Business Rule Format:
            ^IDS651 = FALSE AND ^IDS658 = NULL
    
            Data Elements:
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
    
            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator
            */
            assertion = (report.IDS651 == false && report.IDS658 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440906",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440906" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440907
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440907(IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440907
            If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' and a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', then the following must not be completed:
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)? 
            - Total debt deductions disallowed by ss820-423A(5)
            - Prior Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS650 = TRUE AND ^IDS651 = FALSE AND 
            (^IDS652 <> NULL OR
            ^IDS653 <> NULL OR
            ^IDS654 <> NULL OR
            ^IDS2032 <> NULL)

            Technical Business Rule Format:
            ^IDS650 = TRUE AND ^IDS651 = FALSE AND 
            (^IDS652 <> NULL OR
            ^IDS653 <> NULL OR
            ^IDS654 <> NULL OR
            ^IDS2032 <> NULL)

            Data Elements:

            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator

            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator

            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator

            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator

            ^IDS654 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearDistributions.Amount

            ^IDS2032 = IDS:RP:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS650 == true && report.IDS651 == false && (report.IDS652 != null || report.IDS653 != null || report.IDS654 != null || report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440907",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' and a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', then the following must not be completed:
- Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)? 
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440907" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS654", Value = GetValueOrEmpty(report.IDS654) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440908
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440908()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440908
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS652 = FALSE 
            AND ^IDS658 = NULL

            Technical Business Rule Format:
            ^IDS652 = FALSE 
            AND ^IDS658 = NULL
    
            Data Elements:
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
    
            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator
            */
            assertion = (report.IDS652 == false && report.IDS658 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440908",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440908" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440909
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440909(IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440909
            If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' and 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', and a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then the following must not be completed:
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)? 
            - Total debt deductions disallowed by ss820-423A(5)
            - Prior Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS650 = TRUE 
            AND ^IDS651 = TRUE 
            AND ^IDS652 = FALSE 
            AND (^IDS653 <> NULL 
            OR ^IDS654 <> NULL 
            OR ^IDS2032 <> NULL)

            Technical Business Rule Format:
            ^IDS650 = TRUE 
            AND ^IDS651 = TRUE 
            AND ^IDS652 = FALSE 
            AND (^IDS653 <> NULL 
            OR ^IDS654 <> NULL 
            OR ^IDS2032 <> NULL)

            Data Elements:

            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator

            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator

            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator

            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator

            ^IDS654 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearDistributions.Amount

            ^IDS2032 = IDS:RP:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS650 == true && report.IDS651 == true && report.IDS652 == false && (report.IDS653 != null || report.IDS654 != null || report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440909",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' and 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', and a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then the following must not be completed:
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)? 
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440909" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS654", Value = GetValueOrEmpty(report.IDS654) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440910
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440910()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440910
            If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS653 = FALSE 
            AND ^IDS658 = NULL

            Technical Business Rule Format:
            ^IDS653 = FALSE 
            AND ^IDS658 = NULL
    
            Data Elements:
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
    
            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator
            */
            assertion = (report.IDS653 == false && report.IDS658 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440910",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?', then 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440910" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440911
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440911(IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440911
            If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?', 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?', and a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?', then the following must not be completed:
            - Total debt deductions disallowed by ss820-423A(5)
            - Prior Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS650 = TRUE 
            AND ^IDS651 = TRUE 
            AND ^IDS652 = TRUE 
            AND ^IDS653 = FALSE 
            AND (^IDS654 <> NULL OR ^IDS2032 <> NULL)

            Technical Business Rule Format:
            ^IDS650 = TRUE 
            AND ^IDS651 = TRUE 
            AND ^IDS652 = TRUE 
            AND ^IDS653 = FALSE 
            AND (^IDS654 <> NULL OR ^IDS2032 <> NULL)

            Data Elements:

            ^IDS654 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearDistributions.Amount

            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator

            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator

            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator

            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator

            ^IDS2032 = IDS:RP:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS650 == true && report.IDS651 == true && report.IDS652 == true && report.IDS653 == false && (report.IDS654 != null || report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440911",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must not be provided:
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    LongDescription = @"If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?', 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?', and a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?', then the following must not be completed:
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440911" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS654", Value = GetValueOrEmpty(report.IDS654) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440914
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440914(IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440914
            If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?' then the following must be completed: 
            - Total debt deductions disallowed by ss820-423A(5)
            - Prior Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS653 = TRUE 
            AND (^IDS654 = NULL
            OR ^IDS2032 = NULL)

            Technical Business Rule Format:
            ^IDS653 = TRUE 
            AND (^IDS654 = NULL
            OR ^IDS2032 = NULL)

            Data Elements:

            ^IDS654 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearDistributions.Amount

            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator

            ^IDS2032 = IDS:RP:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS653 == true && (report.IDS654 == null || report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440914",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must be completed: 
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?' then the following must be completed: 
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440914" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS654", Value = GetValueOrEmpty(report.IDS654) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440915
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440915()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440915
            If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?' then 'Total debt deductions disallowed by subsection 820-423A(2)' and 'Arrangements' must be completed.
    
            Legacy Rule Format:
            ^IDS663 = TRUE 
            AND (^IDS664 = NULL OR Count(^IDS2034) = 0)

            Technical Business Rule Format:
            ^IDS663 = TRUE 
            AND (^IDS664 = NULL OR Count(^IDS2034) = 0)
    
            Data Elements:
    
            ^IDS664 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearCGT.Amount
    
            ^IDS663 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearCGT.Indicator
    
            ^IDS2034 = IDS:RP:Arrangements
            */
            assertion = (report.IDS663 == true && (report.IDS664 == null || Count(report.RP_ArrangementsCollectionCount) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440915",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must be completed:
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?' then 'Total debt deductions disallowed by subsection 820-423A(2)' and 'Arrangements' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440915" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS663", Value = GetValueOrEmpty(report.IDS663) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS664", Value = GetValueOrEmpty(report.IDS664) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440916
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440916()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440916
            If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?' then 'Total debt deductions disallowed due to section 820-423A(5)' and 'Current Income Year Top Payments or Distributions to Associated Pair Arrangement' must be completed.
    
            Legacy Rule Format:
            ^IDS673 = TRUE 
            AND (^IDS674 = NULL OR Count(^IDS2036) = 0)

            Technical Business Rule Format:
            ^IDS673 = TRUE 
            AND (^IDS674 = NULL OR Count(^IDS2036) = 0)
    
            Data Elements:
    
            ^IDS674 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearDistributions.Amount
    
            ^IDS673 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearDistributions.Indicator
    
            ^IDS2036 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS673 == true && (report.IDS674 == null || Count(report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440916",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must be completed:
- Total debt deductions disallowed due to subsection 820-423A(5)
- Current Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?' then 'Total debt deductions disallowed due to section 820-423A(5)' and 'Current Income Year Top Payments or Distributions to Associated Pair Arrangement' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440916" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS673", Value = GetValueOrEmpty(report.IDS673) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS674", Value = GetValueOrEmpty(report.IDS674) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440917
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440917()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440917
            If there is a TRUE response to 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' then 'Total debt deductions disallowed' and 'Associate Pairs' must be provided.
    
            Legacy Rule Format:
            ^IDS679 = TRUE 
            AND (^IDS680 = NULL OR Count(^IDS2037) = 0)

            Technical Business Rule Format:
            ^IDS679 = TRUE 
            AND (^IDS680 = NULL OR Count(^IDS2037) = 0)
    
            Data Elements:
    
            ^IDS680 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentYearPayments.Amount
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
    
            ^IDS2037 = IDS:RP:AssociatePairs
            */
            assertion = (report.IDS679 == true && (report.IDS680 == null || Count(report.RP_AssociatePairsCollectionCount) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440917",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must be completed:
- Total debt deductions disallowed
- Associate Pairs",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' then 'Total debt deductions disallowed' and 'Associate Pairs' must be provided.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440917" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS680", Value = GetValueOrEmpty(report.IDS680) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440918
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440918()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440918
            If there is a FALSE response to 'Were the debt deduction creation rules applicable to you?' then the 'Debt Deduction Creation Rules choice (Code)' must be provided.
    
            Legacy Rule Format:
            ^IDS640 = FALSE AND
            ^IDS641 = NULL

            Technical Business Rule Format:
            ^IDS640 = FALSE AND
            ^IDS641 = NULL
    
            Data Elements:
    
            ^IDS641 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtChoice.Code
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
            */
            assertion = (report.IDS640 == false && report.IDS641 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440918",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The 'Debt Deduction Creation Rules choice (Code)' must be provided.",
                    LongDescription = @"If there is a FALSE response to 'Were the debt deduction creation rules applicable to you?' then the 'Debt Deduction Creation Rules choice (Code)' must be provided.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440918" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS641", Value = GetValueOrEmpty(report.IDS641) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440920
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440920()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440920
            If there is a TRUE response to 'Were the debt deduction creation rules applicable to you?', then 'Debt Deduction Creation Rules choice (Code)' must not be completed.
    
            Legacy Rule Format:
            ^IDS640 = TRUE 
            AND ^IDS641 <> NULL

            Technical Business Rule Format:
            ^IDS640 = TRUE 
            AND ^IDS641 <> NULL
    
            Data Elements:
    
            ^IDS641 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtChoice.Code
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
            */
            assertion = (report.IDS640 == true && report.IDS641 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440920",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The 'Debt Deduction Creation Rules choice (Code)' must not be provided.",
                    LongDescription = @"If there is a TRUE response to 'Were the debt deduction creation rules applicable to you?', then 'Debt Deduction Creation Rules choice (Code)' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440920" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS641", Value = GetValueOrEmpty(report.IDS641) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440921
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440921()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440921
            If there is a FALSE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' then 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?' must be completed.
    
            Legacy Rule Format:
            ^IDS628 = FALSE AND
            ^IDS643 = NULL

            Technical Business Rule Format:
            ^IDS628 = FALSE AND
            ^IDS643 = NULL
    
            Data Elements:
    
            ^IDS643 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairPriorIncomeYear.Indicator
    
            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator
            */
            assertion = (report.IDS628 == false && report.IDS643 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440921",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.' then 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairPriorIncomeYearI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440921" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS643", Value = GetValueOrEmpty(report.IDS643) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440922
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440922()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440922
            If there is a FALSE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.', then 'Restructuring Event' must not be completed.
    
            Legacy Rule Format:
            ^IDS628 = FALSE 
            AND Count(^IDS2030) <> 0

            Technical Business Rule Format:
            ^IDS628 = FALSE 
            AND Count(^IDS2030) <> 0
    
            Data Elements:
    
            ^IDS2030 = IDS:RP:RestructuringEventCurrentOrPriorIncomeYear
    
            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator
            */
            assertion = (report.IDS628 == false && Count(report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionCount) != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440922",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Restructuring Event must not be completed",
                    LongDescription = @"If there is a FALSE response to 'Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.', then 'Restructuring Event' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:RestructuringEventCurrentOrPriorIncomeYearCollection/tns:RestructuringEventCurrentOrPriorIncomeYear" + OccurrenceIndex(report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection == null ? 0 : (report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection.Count() == 0 ? 0 : report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection.Last().OccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440922" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440923
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440923()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440923
            If there is a TRUE response to 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?' must be completed.
    
            Legacy Rule Format:
            ^IDS643 = TRUE 
            AND ^IDS644 = NULL

            Technical Business Rule Format:
            ^IDS643 = TRUE 
            AND ^IDS644 = NULL
    
            Data Elements:
    
            ^IDS644 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssetAcquisition.Indicator
    
            ^IDS643 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairPriorIncomeYear.Indicator
            */
            assertion = (report.IDS643 == true && report.IDS644 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440923",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440923" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS643", Value = GetValueOrEmpty(report.IDS643) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS644", Value = GetValueOrEmpty(report.IDS644) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440924
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440924()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440924
            If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?' then 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?' must be completed.
    
            Legacy Rule Format:
            ^IDS644 = TRUE 
            AND ^IDS645 = NULL

            Technical Business Rule Format:
            ^IDS644 = TRUE 
            AND ^IDS645 = NULL
    
            Data Elements:
    
            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator
    
            ^IDS644 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssetAcquisition.Indicator
            */
            assertion = (report.IDS644 == true && report.IDS645 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440924",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The lable 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?' then 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440924" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS644", Value = GetValueOrEmpty(report.IDS644) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440925
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440925()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440925
            If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' then 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' must be completed.
    
            Legacy Rule Format:
            ^IDS650 = TRUE 
            AND ^IDS651 = NULL

            Technical Business Rule Format:
            ^IDS650 = TRUE 
            AND ^IDS651 = NULL
    
            Data Elements:
    
            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator
    
            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator
            */
            assertion = (report.IDS650 == true && report.IDS651 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440925",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' then 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440925" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440926
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440926()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440926
            If there is a TRUE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' must be completed.
    
            Legacy Rule Format:
            ^IDS651 = TRUE 
            AND ^IDS652 = NULL

            Technical Business Rule Format:
            ^IDS651 = TRUE 
            AND ^IDS652 = NULL
    
            Data Elements:
    
            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator
    
            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator
            */
            assertion = (report.IDS651 == true && report.IDS652 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440926",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440926" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440927
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440927()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440927
            If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?' must be completed.
    
            Legacy Rule Format:
            ^IDS652 = TRUE 
            AND ^IDS653 = NULL

            Technical Business Rule Format:
            ^IDS652 = TRUE 
            AND ^IDS653 = NULL
    
            Data Elements:
    
            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator
    
            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator
            */
            assertion = (report.IDS652 == true && report.IDS653 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440927",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then 'Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440927" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440928
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440928()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440928
            If there is a FALSE response to 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' then 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS658 = FALSE 
            AND ^IDS669 = NULL

            Technical Business Rule Format:
            ^IDS658 = FALSE 
            AND ^IDS669 = NULL
    
            Data Elements:
    
            ^IDS669 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.CurrentYearDistributionsAssociatePair.Indicator
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
            */
            assertion = (report.IDS658 == false && report.IDS669 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440928",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In the current income year did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' then 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440928" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS669", Value = GetValueOrEmpty(report.IDS669) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440929
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440929(IDS2025.RP_Arrangements arrangements, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440929
            If there is a FALSE response to 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?', then the following must not be completed:
            - Total value of associate pair acquisitions for the current income year
            - Did any of the exceptions in section 820-423AA apply to the assets your acquired? 
            - Exceptions for acquisition of certain CGT assets
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?
            - Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
            - Total debt deductions disallowed by subsection 820-423A(2)
            - Arrangements

            Legacy Rule Format:
            ^IDS658 = FALSE AND
            (^IDS659 <> NULL OR
            ^IDS660 <> NULL OR
            ^IDS661 <> NULL OR
            ^IDS662 <> NULL OR
            ^IDS663 <> NULL OR
            ^IDS664 <> NULL OR
            ^IDS2034 <> NULL)

            Technical Business Rule Format:
            ^IDS658 = FALSE AND
            (^IDS659 <> NULL OR
            ^IDS660 <> NULL OR
            ^IDS661 <> NULL OR
            ^IDS662 <> NULL OR
            ^IDS663 <> NULL OR
            ^IDS664 <> NULL OR
            ^IDS2034 <> NULL)

            Data Elements:

            ^IDS659 = IDS:RP:Interaction.Transaction.Amount

            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator

            ^IDS660 = IDS:RP:Liabilities.DebtDeductionCreationRules.AssetException.Indicator

            ^IDS661 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionException.Code

            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator

            ^IDS663 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearCGT.Indicator

            ^IDS664 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearCGT.Amount

            ^IDS2034 = IDS:RP:Arrangements
            */
            assertion = (report.IDS658 == false && (report.IDS659 != null || report.IDS660 != null || report.IDS661 != null || report.IDS662 != null || report.IDS663 != null || report.IDS664 != null || report.RP_ArrangementsCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440929",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a FALSE response to 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?', then the following must not be completed:
- Total value of associate pair acquisitions for the current income year
- Did any of the exceptions in section 820-423AA apply to the assets your acquired? 
- Exceptions for acquisition of certain CGT assets
- Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?
- Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements",
                    Location = "/tns:IDS/tns:RP/tns:InteractionTransactionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440929" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS659", Value = GetValueOrEmpty(report.IDS659) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS660", Value = GetValueOrEmpty(report.IDS660) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS661", Value = GetValueOrEmpty(report.IDS661) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS663", Value = GetValueOrEmpty(report.IDS663) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS664", Value = GetValueOrEmpty(report.IDS664) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440930
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440930()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440930
            If there is a TRUE response to 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' then 'Total value of associate pair acquisitions for the current income year' must be completed.
    
            Legacy Rule Format:
            ^IDS658 = TRUE 
            AND ^IDS659 = NULL

            Technical Business Rule Format:
            ^IDS658 = TRUE 
            AND ^IDS659 = NULL
    
            Data Elements:
    
            ^IDS659 = IDS:RP:Interaction.Transaction.Amount
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
            */
            assertion = (report.IDS658 == true && report.IDS659 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440930",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Total value of associate pair acquisitions for the current income year' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?' then 'Total value of associate pair acquisitions for the current income year' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:InteractionTransactionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440930" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS659", Value = GetValueOrEmpty(report.IDS659) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440932
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440932()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440932
            If there is a FALSE response to 'Did any of the exceptions in section 820-423AA apply to the assets your acquired?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' must be completed.
    
            Legacy Rule Format:
            ^IDS660 = FALSE 
            AND ^IDS662 = NULL

            Technical Business Rule Format:
            ^IDS660 = FALSE 
            AND ^IDS662 = NULL
    
            Data Elements:
    
            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator
    
            ^IDS660 = IDS:RP:Liabilities.DebtDeductionCreationRules.AssetException.Indicator
            */
            assertion = (report.IDS660 == false && report.IDS662 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440932",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did any of the exceptions in section 820-423AA apply to the assets your acquired?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440932" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS660", Value = GetValueOrEmpty(report.IDS660) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440933
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440933()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440933
            If there is a FALSE response to 'Did any of the exceptions in section 820-423AA apply to the assets your acquired?', then 'Exceptions for acquisition of certain CGT assets' must not be completed.
    
            Legacy Rule Format:
            ^IDS660 = FALSE 
            AND ^IDS661 <> NULL

            Technical Business Rule Format:
            ^IDS660 = FALSE 
            AND ^IDS661 <> NULL
    
            Data Elements:
    
            ^IDS661 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionException.Code
    
            ^IDS660 = IDS:RP:Liabilities.DebtDeductionCreationRules.AssetException.Indicator
            */
            assertion = (report.IDS660 == false && report.IDS661 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440933",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Exceptions for acquisition of certain CGT assets' must not be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did any of the exceptions in section 820-423AA apply to the assets your acquired?', then 'Exceptions for acquisition of certain CGT assets' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440933" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS660", Value = GetValueOrEmpty(report.IDS660) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS661", Value = GetValueOrEmpty(report.IDS661) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440934
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440934()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440934
            If there is a TRUE response to 'Did any of the exceptions in section 820-423AA apply to the assets your acquired?' then 'Exceptions for acquisition of certain CGT assets' must be completed.
    
            Legacy Rule Format:
            ^IDS660 = TRUE 
            AND ^IDS661 = NULL

            Technical Business Rule Format:
            ^IDS660 = TRUE 
            AND ^IDS661 = NULL
    
            Data Elements:
    
            ^IDS661 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionException.Code
    
            ^IDS660 = IDS:RP:Liabilities.DebtDeductionCreationRules.AssetException.Indicator
            */
            assertion = (report.IDS660 == true && report.IDS661 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440934",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Exceptions for acquisition of certain CGT assets' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Did any of the exceptions in section 820-423AA apply to the assets your acquired?' then 'Exceptions for acquisition of certain CGT assets' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440934" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS660", Value = GetValueOrEmpty(report.IDS660) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS661", Value = GetValueOrEmpty(report.IDS661) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440935
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440935()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440935
            If there is a response to 'Exceptions for acquisition of certain CGT assets' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' must be completed.
    
            Legacy Rule Format:
            ^IDS661 <> NULL 
            AND ^IDS662 = NULL

            Technical Business Rule Format:
            ^IDS661 <> NULL 
            AND ^IDS662 = NULL
    
            Data Elements:
    
            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator
    
            ^IDS661 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionException.Code
            */
            assertion = (report.IDS661 != null && report.IDS662 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440935",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' must be completed.",
                    LongDescription = @"If there is a response to 'Exceptions for acquisition of certain CGT assets' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440935" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS661", Value = GetValueOrEmpty(report.IDS661) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440936
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440936()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440936
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' then 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.
    
            Legacy Rule Format:
            ^IDS662 = FALSE 
            AND ^IDS669 = NULL

            Technical Business Rule Format:
            ^IDS662 = FALSE 
            AND ^IDS669 = NULL
    
            Data Elements:
    
            ^IDS669 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.CurrentYearDistributionsAssociatePair.Indicator
    
            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator
            */
            assertion = (report.IDS662 == false && report.IDS669 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440936",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'In the current income year did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' then 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440936" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS669", Value = GetValueOrEmpty(report.IDS669) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440937
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440937(IDS2025.RP_Arrangements arrangements, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440937
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?', then the following must not be completed:
            - Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
            - Total debt deductions disallowed by subsection 820-423A(2)
            - Arrangements

            Legacy Rule Format:
            ^IDS662 = FALSE AND
            (^IDS663 <> NULL 
            OR ^IDS664 <> NULL 
            OR ^IDS2034 <> NULL)

            Technical Business Rule Format:
            ^IDS662 = FALSE AND
            (^IDS663 <> NULL 
            OR ^IDS664 <> NULL 
            OR ^IDS2034 <> NULL)

            Data Elements:

            ^IDS663 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearCGT.Indicator

            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator

            ^IDS664 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearCGT.Amount

            ^IDS2034 = IDS:RP:Arrangements
            */
            assertion = (report.IDS662 == false && (report.IDS663 != null || report.IDS664 != null || report.RP_ArrangementsCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440937",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must not be completed:
- Were any of those debt deductions referable to an amount paid or payable either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?', then the following must not be completed:
- Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440937" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS663", Value = GetValueOrEmpty(report.IDS663) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS664", Value = GetValueOrEmpty(report.IDS664) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440938
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440938()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440938
            If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' then 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?' must be completed.
    
            Legacy Rule Format:
            ^IDS662 = TRUE 
            AND ^IDS663 = NULL

            Technical Business Rule Format:
            ^IDS662 = TRUE 
            AND ^IDS663 = NULL
    
            Data Elements:
    
            ^IDS663 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearCGT.Indicator
    
            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator
            */
            assertion = (report.IDS662 == true && report.IDS663 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440938",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of those debt deductions referable to an amount paid or payable either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?' then 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440938" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS663", Value = GetValueOrEmpty(report.IDS663) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440939
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440939()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440939
            If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?', then the following must not be completed:
            - Total debt deductions disallowed by subsection 820-423A(2)
            - Arrangements
    
            Legacy Rule Format:
            ^IDS663 = FALSE 
            AND (^IDS664 <> NULL OR Count(^IDS2034) <> 0)

            Technical Business Rule Format:
            ^IDS663 = FALSE 
            AND (^IDS664 <> NULL OR Count(^IDS2034) <> 0)
    
            Data Elements:
    
            ^IDS664 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearCGT.Amount
    
            ^IDS663 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearCGT.Indicator
    
            ^IDS2034 = IDS:RP:Arrangements
            */
            assertion = (report.IDS663 == false && (report.IDS664 != null || Count(report.RP_ArrangementsCollectionCount) != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440939",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following must not be completed:
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?', then the following must not be completed:
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440939" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS663", Value = GetValueOrEmpty(report.IDS663) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS664", Value = GetValueOrEmpty(report.IDS664) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440941
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440941()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440941
            If there is a FALSE response to 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.
    
            Legacy Rule Format:
            ^IDS669 = FALSE AND
            ^IDS679 = NULL

            Technical Business Rule Format:
            ^IDS669 = FALSE AND
            ^IDS679 = NULL
    
            Data Elements:
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
    
            ^IDS669 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.CurrentYearDistributionsAssociatePair.Indicator
            */
            assertion = (report.IDS669 == false && report.IDS679 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440941",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440941" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS669", Value = GetValueOrEmpty(report.IDS669) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440942
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440942()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440942
            If there is a TRUE response to 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' then 'Total value of prescribed payments for the income year' must be completed.
    
            Legacy Rule Format:
            ^IDS669 = TRUE AND
            ^IDS670 = NULL

            Technical Business Rule Format:
            ^IDS669 = TRUE AND
            ^IDS670 = NULL
    
            Data Elements:
    
            ^IDS670 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:PaymentRecord.Payment.Amount
    
            ^IDS669 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.CurrentYearDistributionsAssociatePair.Indicator
            */
            assertion = (report.IDS669 == true && report.IDS670 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440942",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Total value of prescribed payments for the income year' must be completed",
                    LongDescription = @"If there is a TRUE response to 'In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?' then 'Total value of prescribed payments for the income year' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:PaymentRecordA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440942" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS669", Value = GetValueOrEmpty(report.IDS669) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS670", Value = GetValueOrEmpty(report.IDS670) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440943
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440943()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440943
            If there is a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.
    
            Legacy Rule Format:
            ^IDS671 = FALSE AND
            ^IDS679 = NULL

            Technical Business Rule Format:
            ^IDS671 = FALSE AND
            ^IDS679 = NULL
    
            Data Elements:
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
    
            ^IDS671 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.FinancialArrangementFunding.Indicator
            */
            assertion = (report.IDS671 == false && report.IDS679 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440943",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440943" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS671", Value = GetValueOrEmpty(report.IDS671) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440944
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440944(IDS2025.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440944
            If there is a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', then the following must not be completed:
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
            - Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? 
            - Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? 
            - Total debt deductions disallowed due to section 820-423A(5)
            - Current Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS671 = FALSE AND 
            (^IDS672 <> NULL OR
            ^IDS673 <> NULL OR
            ^IDS674 <> NULL OR
            ^IDS2036 <> NULL)

            Technical Business Rule Format:
            ^IDS671 = FALSE AND 
            (^IDS672 <> NULL OR
            ^IDS673 <> NULL OR
            ^IDS674 <> NULL OR
            ^IDS2036 <> NULL)

            Data Elements:

            ^IDS672 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtCurrentIncomeYearDistributions.Indicator

            ^IDS671 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.FinancialArrangementFunding.Indicator

            ^IDS673 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearDistributions.Indicator

            ^IDS674 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearDistributions.Amount

            ^IDS2036 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS671 == false && (report.IDS672 != null || report.IDS673 != null || report.IDS674 != null || report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440944",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a FALSE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?', then the following must not be completed:
- Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
- Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? 
- Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? 
- Total debt deductions disallowed due to section 820-423A(5)
- Current Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440944" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS671", Value = GetValueOrEmpty(report.IDS671) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS672", Value = GetValueOrEmpty(report.IDS672) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS673", Value = GetValueOrEmpty(report.IDS673) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS674", Value = GetValueOrEmpty(report.IDS674) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440945
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440945()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440945
            If there is a TRUE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' must be completed.
    
            Legacy Rule Format:
            ^IDS671 = TRUE AND
            ^IDS672 = NULL

            Technical Business Rule Format:
            ^IDS671 = TRUE AND
            ^IDS672 = NULL
    
            Data Elements:
    
            ^IDS672 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtCurrentIncomeYearDistributions.Indicator
    
            ^IDS671 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.FinancialArrangementFunding.Indicator
            */
            assertion = (report.IDS671 == true && report.IDS672 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440945",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' must be completed.",
                    LongDescription = @"If there is a TRUE response to 'Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?' then 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440945" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS671", Value = GetValueOrEmpty(report.IDS671) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS672", Value = GetValueOrEmpty(report.IDS672) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440946
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440946()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440946
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.
    
            Legacy Rule Format:
            ^IDS672 = FALSE AND
            ^IDS679 = NULL

            Technical Business Rule Format:
            ^IDS672 = FALSE AND
            ^IDS679 = NULL
    
            Data Elements:
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
    
            ^IDS672 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtCurrentIncomeYearDistributions.Indicator
            */
            assertion = (report.IDS672 == false && report.IDS679 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440946",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440946" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS672", Value = GetValueOrEmpty(report.IDS672) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440947
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440947(IDS2025.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440947
            If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?', then the following should not be completed:
            - Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? 
            - Total debt deductions disallowed due to section 820-423A(5)
            - Current Income Year Top Payments or Distributions to Associated Pair Arrangement

            Legacy Rule Format:
            ^IDS672 = FALSE AND 
            (^IDS673 <>NULL OR 
            ^IDS674 <> NULL OR 
            ^IDS2036 <> NULL)

            Technical Business Rule Format:
            ^IDS672 = FALSE AND 
            (^IDS673 <>NULL OR 
            ^IDS674 <> NULL OR 
            ^IDS2036 <> NULL)

            Data Elements:

            ^IDS673 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearDistributions.Indicator

            ^IDS672 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtCurrentIncomeYearDistributions.Indicator

            ^IDS674 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearDistributions.Amount

            ^IDS2036 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS672 == false && (report.IDS673 != null || report.IDS674 != null || report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440947",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?', then the following should not be completed:
- Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? 
- Total debt deductions disallowed due to section 820-423A(5)
- Current Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440947" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS672", Value = GetValueOrEmpty(report.IDS672) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS673", Value = GetValueOrEmpty(report.IDS673) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS674", Value = GetValueOrEmpty(report.IDS674) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440948
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440948()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440948
            If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?' must be completed.
    
            Legacy Rule Format:
            ^IDS672 = TRUE AND
            ^IDS673 = NULL

            Technical Business Rule Format:
            ^IDS672 = TRUE AND
            ^IDS673 = NULL
    
            Data Elements:
    
            ^IDS673 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearDistributions.Indicator
    
            ^IDS672 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtCurrentIncomeYearDistributions.Indicator
            */
            assertion = (report.IDS672 == true && report.IDS673 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440948",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)? ' must be completed",
                    LongDescription = @"If there is a TRUE response to 'Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?' then 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440948" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS672", Value = GetValueOrEmpty(report.IDS672) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS673", Value = GetValueOrEmpty(report.IDS673) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440949
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440949()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440949
            If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be answered.
    
            Legacy Rule Format:
            ^IDS673 = FALSE AND
            ^IDS679 = NULL

            Technical Business Rule Format:
            ^IDS673 = FALSE AND
            ^IDS679 = NULL
    
            Data Elements:
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
    
            ^IDS673 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearDistributions.Indicator
            */
            assertion = (report.IDS673 == false && report.IDS679 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440949",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?' then 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?' must be answered.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440949" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS673", Value = GetValueOrEmpty(report.IDS673) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440950
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440950()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440950
            If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?', then the following must not be completed:
            - Total debt deductions disallowed due to section 820-423A(5)
            - Current Income Year Top Payments or Distributions to Associated Pair Arrangement
    
            Legacy Rule Format:
            ^IDS673 = FALSE AND 
            (^IDS674 <> NULL OR Count(^IDS2036) <> 0)

            Technical Business Rule Format:
            ^IDS673 = FALSE AND 
            (^IDS674 <> NULL OR Count(^IDS2036) <> 0)
    
            Data Elements:
    
            ^IDS674 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearDistributions.Amount
    
            ^IDS673 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearDistributions.Indicator
    
            ^IDS2036 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
            */
            assertion = (report.IDS673 == false && (report.IDS674 != null || Count(report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount) != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440950",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Certain questions must not be answered. Please see the Long Description for specific information.",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(5)(e)?', then the following must not be completed:
- Total debt deductions disallowed due to section 820-423A(5)
- Current Income Year Top Payments or Distributions to Associated Pair Arrangement",
                    Location = "/tns:IDS/tns:RP/tns:CurrentIncomeYearPaymentsorDistributionsSelection/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440950" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS673", Value = GetValueOrEmpty(report.IDS673) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS674", Value = GetValueOrEmpty(report.IDS674) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440952
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440952()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440952
            If there is a FALSE response to 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?', then the following should not be completed:
            - Total debt deductions disallowed 
            - Associate pairs
    
            Legacy Rule Format:
            ^IDS679 = FALSE
            AND (^IDS680 <> NULL OR Count(^IDS2037) <> 0)

            Technical Business Rule Format:
            ^IDS679 = FALSE
            AND (^IDS680 <> NULL OR Count(^IDS2037) <> 0)
    
            Data Elements:
    
            ^IDS680 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentYearPayments.Amount
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
    
            ^IDS2037 = IDS:RP:AssociatePairs
            */
            assertion = (report.IDS679 == false && (report.IDS680 != null || Count(report.RP_AssociatePairsCollectionCount) != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440952",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The following should not be completed:
- Total debt deductions disallowed 
- Associate pairs",
                    LongDescription = @"If there is a FALSE response to 'Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?', then the following should not be completed:
- Total debt deductions disallowed 
- Associate pairs",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440952" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS680", Value = GetValueOrEmpty(report.IDS680) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440953
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440953(IDS2025.RP_RestructuringEventCurrentOrPriorIncomeYear restructuringEventCurrentOrPriorIncomeYear, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440953
            If there is a FALSE response to 'Were the debt deduction creation rules applicable to you?', then the following must not be completed:
            - Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.
            - Restructuring Event
            - In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?
            - Total debt deductions disallowed due to subsection 820-423A(2)
            - Prior Income Year Top Material Arrangements
            - In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?
            - Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
            - Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?
            - Total debt deductions disallowed by ss820-423A(5)
            - Prior Income Year Top Payments or Distributions to Associated Pair Arrangement
            - In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?
            - Total value of associate pair acquisitions for the current income year
            - Did any of the exceptions in section 820-423AA apply to the assets your acquired?
            - Exceptions for acquisition of certain CGT assets
            - Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?
            - Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
            - Total debt deductions disallowed by subsection 820-423A(2)
            - Arrangements
            - Current Income Year Payments or Distributions Selection
            - Current Income Year Top Payments or Distributions to Associated Pair Arrangement
            - Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b)an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?
            - Total debt deductions disallowed
            - Associate Pairs

            Legacy Rule Format:
            ^IDS640 = FALSE AND
            (^IDS628 <> NULL
            OR ^IDS2030 <> NULL
            OR ^IDS643 <> NULL
            OR ^IDS644 <> NULL
            OR ^IDS645 <> NULL
            OR ^IDS646 <> NULL
            OR ^IDS2031 <> NULL
            OR ^IDS650 <> NULL
            OR ^IDS651 <> NULL
            OR ^IDS652 <> NULL
            OR ^IDS653 <> NULL
            OR ^IDS654 <> NULL
            OR ^IDS2032 <> NULL
            OR ^IDS658 <> NULL
            OR ^IDS659 <> NULL
            OR ^IDS660 <> NULL
            OR ^IDS661 <> NULL
            OR ^IDS662 <> NULL
            OR ^IDS663 <> NULL
            OR ^IDS664 <> NULL
            OR ^IDS2034 <> NULL
            OR ^IDS2035 <> NULL 
            OR ^IDS2036 <> NULL
            OR ^IDS679 <> NULL
            OR ^IDS680 <> NULL
            OR ^IDS2037 <> NULL)

            Technical Business Rule Format:
            ^IDS640 = FALSE AND
            (^IDS628 <> NULL
            OR ^IDS2030 <> NULL
            OR ^IDS643 <> NULL
            OR ^IDS644 <> NULL
            OR ^IDS645 <> NULL
            OR ^IDS646 <> NULL
            OR ^IDS2031 <> NULL
            OR ^IDS650 <> NULL
            OR ^IDS651 <> NULL
            OR ^IDS652 <> NULL
            OR ^IDS653 <> NULL
            OR ^IDS654 <> NULL
            OR ^IDS2032 <> NULL
            OR ^IDS658 <> NULL
            OR ^IDS659 <> NULL
            OR ^IDS660 <> NULL
            OR ^IDS661 <> NULL
            OR ^IDS662 <> NULL
            OR ^IDS663 <> NULL
            OR ^IDS664 <> NULL
            OR ^IDS2034 <> NULL
            OR ^IDS2035 <> NULL 
            OR ^IDS2036 <> NULL
            OR ^IDS679 <> NULL
            OR ^IDS680 <> NULL
            OR ^IDS2037 <> NULL)

            Data Elements:

            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator

            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator

            ^IDS643 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairPriorIncomeYear.Indicator

            ^IDS644 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssetAcquisition.Indicator

            ^IDS645 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearCGT.Indicator

            ^IDS646 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearCGT.Amount

            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator

            ^IDS651 = IDS:RP:Liabilities.DebtDeductionCreationRules.FinancialArrangementsFunding.Indicator

            ^IDS652 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDistributionsPriorIncomeYearRelated.Indicator

            ^IDS653 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairPriorIncomeYearDistributions.Indicator

            ^IDS654 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedPriorIncomeYearDistributions.Amount

            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator

            ^IDS659 = IDS:RP:Interaction.Transaction.Amount

            ^IDS660 = IDS:RP:Liabilities.DebtDeductionCreationRules.AssetException.Indicator

            ^IDS661 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionException.Code

            ^IDS662 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtCGTAssets.Indicator

            ^IDS663 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtAssociatePairCurrentIncomeYearCGT.Indicator

            ^IDS664 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentIncomeYearCGT.Amount

            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator

            ^IDS680 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedCurrentYearPayments.Amount

            ^IDS2030 = IDS:RP:RestructuringEventCurrentOrPriorIncomeYear

            ^IDS2031 = IDS:RP:PriorIncomeYearTopMaterialArrangements

            ^IDS2032 = IDS:RP:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement

            ^IDS2034 = IDS:RP:Arrangements

            ^IDS2035 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection

            ^IDS2036 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement

            ^IDS2037 = IDS:RP:AssociatePairs
            */
            assertion = (report.IDS640 == false && (report.IDS628 != null || report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionExists != false || report.IDS643 != null || report.IDS644 != null || report.IDS645 != null || report.IDS646 != null || report.RP_PriorIncomeYearTopMaterialArrangementsCollectionExists != false || report.IDS650 != null || report.IDS651 != null || report.IDS652 != null || report.IDS653 != null || report.IDS654 != null || report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false || report.IDS658 != null || report.IDS659 != null || report.IDS660 != null || report.IDS661 != null || report.IDS662 != null || report.IDS663 != null || report.IDS664 != null || report.RP_ArrangementsCollectionExists != false || report.RP_CurrentIncomeYearPaymentsorDistributionsSelectionCollectionExists != false || report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists != false || report.IDS679 != null || report.IDS680 != null || report.RP_AssociatePairsCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440953",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Section H: Debt deduction creation rules must not be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were the debt deduction creation rules applicable to you?', then the following must not be completed:
- Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.
- Restructuring Event
- In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?
- Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal equitable obligation(s)?
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed due to subsection 820-423A(2)
- Prior Income Year Top Material Arrangements
- In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?
- Did you use a financial arrangement to fund or facilitate the funding of any of those payments or distributions?
- Were any of your debt deductions for the current income year, wholly or partly, in relation to those financial arrangements?
- Were any of your debt deductions referable to an amount paid or payable either directly or indirectly to, an associate pair covered by paragraph 820-423A(5)(e)?
- Total debt deductions disallowed by ss820-423A(5)
- Prior Income Year Top Payments or Distributions to Associated Pair Arrangement
- In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?
- Total value of associate pair acquisitions for the current income year
- Did any of the exceptions in section 820-423AA apply to the assets your acquired?
- Exceptions for acquisition of certain CGT assets
- Were any of your debt deductions for the current income year, wholly or partly, in relation to the acquisition or holding of the CGT asset(s), or legal or equitable obligation(s) in the current year, other than a CGT asset covered by an exception in section 820-423AA?
- Were any of your debt deductions referable to an amount paid or payable, either directly or indirectly, to an associate pair covered by paragraph 820-423A(2)(e)?
- Total debt deductions disallowed by subsection 820-423A(2)
- Arrangements
- Current Income Year Payments or Distributions Selection
- Current Income Year Top Payments or Distributions to Associated Pair Arrangement
- Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b)an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?
- Total debt deductions disallowed
- Associate Pairs",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440953" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS643", Value = GetValueOrEmpty(report.IDS643) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS644", Value = GetValueOrEmpty(report.IDS644) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS645", Value = GetValueOrEmpty(report.IDS645) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS646", Value = GetValueOrEmpty(report.IDS646) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS651", Value = GetValueOrEmpty(report.IDS651) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS652", Value = GetValueOrEmpty(report.IDS652) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS653", Value = GetValueOrEmpty(report.IDS653) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS654", Value = GetValueOrEmpty(report.IDS654) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS659", Value = GetValueOrEmpty(report.IDS659) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS660", Value = GetValueOrEmpty(report.IDS660) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS661", Value = GetValueOrEmpty(report.IDS661) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS662", Value = GetValueOrEmpty(report.IDS662) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS663", Value = GetValueOrEmpty(report.IDS663) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS664", Value = GetValueOrEmpty(report.IDS664) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS680", Value = GetValueOrEmpty(report.IDS680) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440954
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440954()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440954
            If 'Entity type code for the income year' is equal to '1', then the following must not be completed:
            - Financial entity type code
            - Average values calculation method code
    
            Legacy Rule Format:
            ^IDS223 = "1" AND
            (^IDS560 <> NULL 
            OR ^IDS225 <> NULL)

            Technical Business Rule Format:
            ^IDS223 = '1' AND
            (^IDS560 <> NULL 
            OR ^IDS225 <> NULL)
    
            Data Elements:
    
            ^IDS560 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.FinancialEntityType.Code
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS225 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.AveragingMethodAdopted.Code
            */
            assertion = (report.IDS223 == @"1" && (report.IDS560 != null || report.IDS225 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440954",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial entity type code and Average values calculation method code must not be completed.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '1', then the following must not be completed:
- Financial entity type code
- Average values calculation method code",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesFinancialEntityTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440954" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS560", Value = GetValueOrEmpty(report.IDS560) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS225", Value = GetValueOrEmpty(report.IDS225) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440956
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440956()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440956
            If there is a FALSE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?', then 'Special purpose entity' must not be completed.
    
            Legacy Rule Format:
            ^IDS606 = FALSE AND ^IDS2025 <> NULL

            Technical Business Rule Format:
            ^IDS606 = FALSE AND ^IDS2025 <> NULL
    
            Data Elements:
    
            ^IDS2025 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
            */
            assertion = (report.IDS606 == false && report.RP_ThinCapitalisation_SpecialPurposeEntityCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440956",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Special purpose entity must not be completed.",
                    LongDescription = @"If there is a FALSE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?', then 'Special purpose entity' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:SpecialPurposeEntity",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440956" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440957
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440957()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440957
            If 'Entity type code for the income year' is equal to '4' or '5', then 'Adjusted average debt' must not be completed.
    
            Legacy Rule Format:
            InSet(^IDS223, '"4","5"') AND ^IDS229 <> NULL

            Technical Business Rule Format:
            InSet(^IDS223, '"4","5"') AND ^IDS229 <> NULL
    
            Data Elements:
    
            ^IDS229 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DebtAdjustedAverage.Amount
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
            */
            assertion = (IsMatch(report.IDS223, @"^(4|5)$") && report.IDS229 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440957",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Adjusted average debt must not be completed.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '4' or '5', then 'Adjusted average debt' must not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesDebtAdjustedAverageA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440957" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS229", Value = GetValueOrEmpty(report.IDS229) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440958
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440958()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440958
            If there is a FALSE response to 'Did you rely on the third party debt test?' then 'Third party debt test' should not be completed.
    
            Legacy Rule Format:
            ^IDS589 = FALSE AND ^IDS2024 <> NULL

            Technical Business Rule Format:
            ^IDS589 = FALSE AND ^IDS2024 <> NULL
    
            Data Elements:
    
            ^IDS2024 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
            */
            assertion = (report.IDS589 == false && report.RP_ThinCapitalisation_ThirdPartyDebtTestCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440958",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Third party debt test should not be completed.",
                    LongDescription = @"If there is a FALSE response to 'Did you rely on the third party debt test?' then 'Third party debt test' should not be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:ThirdPartyDebtTest",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440958" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440960
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440960()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440960
            If there is a TRUE response to 'Were you the head company of a tax consolidated group or MEC group?' then 'Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?' must be completed.
    
            Legacy Rule Format:
            ^IDS635 = TRUE AND ^IDS636 = NULL

            Technical Business Rule Format:
            ^IDS635 = TRUE AND ^IDS636 = NULL
    
            Data Elements:
    
            ^IDS636 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmountsTransferred.Indicator
    
            ^IDS635 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.TaxConsolidatedGroupMultipleEntryConsolidated.Indicator
            */
            assertion = (report.IDS635 == true && report.IDS636 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440960",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year required.",
                    LongDescription = @"If there is a TRUE response to 'Were you the head company of a tax consolidated group or MEC group?' then 'Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedAmountsTransferredI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440960" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS635", Value = GetValueOrEmpty(report.IDS635) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS636", Value = GetValueOrEmpty(report.IDS636) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440961
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440961()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440961
            If there is a TRUE response to 'Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?', then 'Provide the amount calculated according to subsection 705-112(2)' must be completed.
    
            Legacy Rule Format:
            ^IDS636 = TRUE AND ^IDS637 = NULL

            Technical Business Rule Format:
            ^IDS636 = TRUE AND ^IDS637 = NULL
    
            Data Elements:
    
            ^IDS637 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedCalculated.Amount
    
            ^IDS636 = IDS:RP:ThinCapitalisation:FRTDisallowedamount:Liabilities.ThinCapitalisation.FixedRatioTestDisallowedAmountsTransferred.Indicator
            */
            assertion = (report.IDS636 == true && report.IDS637 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440961",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount calculated to subsection 705-112(2) required.",
                    LongDescription = @"If there is a TRUE response to 'Were FRT disallowed amounts transferred to you, as a head company, by joining entities under section 820-590, during the income year?', then 'Provide the amount calculated according to subsection 705-112(2)' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:FRTDisallowedamount/tns:LiabilitiesFixedRatioTestDisallowedCalculatedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440961" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS636", Value = GetValueOrEmpty(report.IDS636) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS637", Value = GetValueOrEmpty(report.IDS637) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440962
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440962()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440962
            If there is a FALSE response to 'Were the thin capitalisation rules applicable to you?' and there is a TRUE response to 'Did you rely on the $2 million threshold exemption?', then no further information is required in Section D - Thin capitalisation of the schedule.
    
            Legacy Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = TRUE AND (^IDS401 <> NULL OR ^IDS402 <> NULL OR ^IDS630 <> NULL OR ^IDS221 <> NULL OR ^IDS222 <> NULL OR ^IDS1064 <> NULL)

            Technical Business Rule Format:
            ^IDS220 = FALSE AND ^IDS400 = TRUE AND (^IDS401 <> NULL OR ^IDS402 <> NULL OR ^IDS630 <> NULL OR ^IDS221 <> NULL OR ^IDS222 <> NULL OR ^IDS1064 <> NULL)
    
            Data Elements:
    
            ^IDS1064 = IDS:RP:ThinCapitalisation
    
            ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
    
            ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
    
            ^IDS222 = IDS:RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^IDS400 = IDS:RP:Liabilities.ThinCapitalisation.2MillionThresholdTest.Indicator
    
            ^IDS401 = IDS:RP:Liabilities.ThinCapitalisation.90PercentAssetThresholdTest.Indicator
    
            ^IDS402 = IDS:RP:Liabilities.ThinCapitalisation.ExemptionSpecialPurposeEntities.Indicator
    
            ^IDS630 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentEntityNonInOutInvestingEntity.Indicator
            */
            assertion = (report.IDS220 == false && report.IDS400 == true && (report.IDS401 != null || report.IDS402 != null || report.IDS630 != null || report.IDS221 != null || report.IDS222 != null || report.RP_ThinCapitalisationCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440962",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Thin capitalisation information not required.",
                    LongDescription = @"If there is a FALSE response to 'Were the thin capitalisation rules applicable to you?' and there is a TRUE response to 'Did you rely on the $2 million threshold exemption?', then no further information is required in Section D - Thin capitalisation of the schedule.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440962" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS220", Value = GetValueOrEmpty(report.IDS220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS400", Value = GetValueOrEmpty(report.IDS400) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS401", Value = GetValueOrEmpty(report.IDS401) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS402", Value = GetValueOrEmpty(report.IDS402) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS630", Value = GetValueOrEmpty(report.IDS630) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS221", Value = GetValueOrEmpty(report.IDS221) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS222", Value = GetValueOrEmpty(report.IDS222) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440963
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440963()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440963
            If 'Entity type code for the income year' is equal to '1', '2', '3' or '6', then 'Adjusted Average Debt' must be provided.
    
            Legacy Rule Format:
            InSet(^IDS223, '"1","2","3","6"') AND ^IDS229 = NULL

            Technical Business Rule Format:
            InSet(^IDS223, '"1","2","3","6"') AND ^IDS229 = NULL
    
            Data Elements:
    
            ^IDS229 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DebtAdjustedAverage.Amount
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
            */
            assertion = (IsMatch(report.IDS223, @"^(1|2|3|6)$") && report.IDS229 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440963",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Adjusted Average Debt must be provided.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '1', '2', '3' or '6', then 'Adjusted Average Debt' must be provided.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesDebtAdjustedAverageA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440963" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS229", Value = GetValueOrEmpty(report.IDS229) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440964
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440964()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440964
            If there is a TRUE response to 'Did you rely on the third party debt test?' then the response to 'Did you rely on the group ratio test?' must not be TRUE.
    
            Legacy Rule Format:
            ^IDS589 = TRUE AND ^IDS575 = TRUE

            Technical Business Rule Format:
            ^IDS589 = TRUE AND ^IDS575 = TRUE
    
            Data Elements:
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
            */
            assertion = (report.IDS589 == true && report.IDS575 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440964",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The response to 'Did you rely on the group ratio test?' must not be TRUE.",
                    LongDescription = @"If there is a TRUE response to 'Did you rely on the third party debt test?' then the response to 'Did you rely on the group ratio test?' must not be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesGroupRatioTestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440964" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440965
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440965()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440965
            If there is a TRUE response to '90% asset threshold' then 'Were the debt deduction creation rules applicable to you?' must be completed.
    
            Legacy Rule Format:
            ^IDS401 = TRUE AND ^IDS640 = NULL

            Technical Business Rule Format:
            ^IDS401 = TRUE AND ^IDS640 = NULL
    
            Data Elements:
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
    
            ^IDS401 = IDS:RP:Liabilities.ThinCapitalisation.90PercentAssetThresholdTest.Indicator
            */
            assertion = (report.IDS401 == true && report.IDS640 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440965",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The label 'Were the debt deduction creation rules applicable to you?' must be completed.",
                    LongDescription = @"If there is a TRUE response to '90% asset threshold' then 'Were the debt deduction creation rules applicable to you?' must be completed.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440965" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS401", Value = GetValueOrEmpty(report.IDS401) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440966
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440966()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440966
            If there is a 'TRUE' response to 'Did you rely on the group ratio test?' then, information must be completed at 'Tax-EBITDA' and 'Group ratio test'(with at-least two 'GR Group members with the highest entity EBITDA'). 
            The response at 'Did you rely on the third party debt test?' must also be 'FALSE'.
    
            Legacy Rule Format:
            ^IDS575 = TRUE AND (^IDS2020 = NULL OR ^IDS2022 = NULL OR Count(^IDS2023) < 2 OR ^IDS589 <> FALSE)

            Technical Business Rule Format:
            ^IDS575 = TRUE AND (^IDS2020 = NULL OR ^IDS2022 = NULL OR Count(^IDS2023) < 2 OR ^IDS589 <> FALSE)
    
            Data Elements:
    
            ^IDS2020 = IDS:RP:ThinCapitalisation:EBITDA
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS589 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.ThirdPartyDebtTest.Indicator
    
            ^IDS2022 = IDS:RP:ThinCapitalisation:GroupRatioTest
    
            ^IDS2023 = IDS:RP:ThinCapitalisation:GroupRatioTest:GRGroupMembers
            */
            assertion = (report.IDS575 == true && (report.RP_ThinCapitalisation_EBITDACollectionExists == false || report.RP_ThinCapitalisation_GroupRatioTestCollectionExists == false || Count(report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionCount) < 2 || report.IDS589 != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440966",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Information must be completed at 'Tax-EBITDA' and 'Group ratio test' (with at-least two 'GR Group members with the highest entity EBITDA'). The response at 'Did you rely on the third party debt test?' must also be 'FALSE'.",
                    LongDescription = @"If there is a 'TRUE' response to 'Did you rely on the group ratio test?' then, information must be completed at 'Tax-EBITDA' and 'Group ratio test'(with at-least two 'GR Group members with the highest entity EBITDA'). 
The response at 'Did you rely on the third party debt test?' must also be 'FALSE'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:EBITDA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440966" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS589", Value = GetValueOrEmpty(report.IDS589) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440967
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440967()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440967
            If there is a TRUE response to 'Did you have a conduit financing arrangement?' and there is a FALSE response to 'Were you the conduit financer?' then a response is required at the following labels.
            - Provide the conduit financer's ABN
            - If you were not the conduit financer, did the terms of the relevant debt interest issued by you, to the extent that those terms relate to costs incurred, differ from the terms of the ultimate debt interest?
    
            Legacy Rule Format:
            ^IDS597 = TRUE and ^IDS598 = FALSE AND (^IDS599 = NULL OR ^IDS602 = NULL)

            Technical Business Rule Format:
            ^IDS597 = TRUE and ^IDS598 = FALSE AND (^IDS599 = NULL OR ^IDS602 = NULL)
    
            Data Elements:
    
            ^IDS599 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Identifiers.AustralianBusinessNumber.Identifier
    
            ^IDS597 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.ConduitFinancingArrangement.Indicator
    
            ^IDS598 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.ConduitFinancer.Indicator
    
            ^IDS602 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.NotConduitFinancerCostsIncurredAndDifferFromTerms.Indicator
            */
            assertion = (report.IDS597 == true && report.IDS598 == false && (report.IDS599 == null || report.IDS602 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440967",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Responses to the following label must be provided:
- Provide the conduit financer's ABN
- If you were not the conduit financer, did the terms of the relevant debt interest issued by you, to the extent that those terms relate to costs incurred, differ from the terms of the ultimate debt interest?",
                    LongDescription = @"If there is a TRUE response to 'Did you have a conduit financing arrangement?' and there is a FALSE response to 'Were you the conduit financer?' then a response is required at the following labels.
- Provide the conduit financer's ABN
- If you were not the conduit financer, did the terms of the relevant debt interest issued by you, to the extent that those terms relate to costs incurred, differ from the terms of the ultimate debt interest?",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:ThirdPartyDebtTest/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440967" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS597", Value = GetValueOrEmpty(report.IDS597) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS598", Value = GetValueOrEmpty(report.IDS598) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS599", Value = GetValueOrEmpty(report.IDS599) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS602", Value = GetValueOrEmpty(report.IDS602) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440968
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440968()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440968
            If there is a TRUE response to 'Were you the conduit financer?' then a response is required at 'If you were the conduit financer, did the terms of each relevant debt interest held by you, to the extent that those terms relate to costs incurred, differ from the terms of the ultimate debt interest?'
    
            Legacy Rule Format:
            ^IDS598 = TRUE AND ^IDS600 = NULL

            Technical Business Rule Format:
            ^IDS598 = TRUE AND ^IDS600 = NULL
    
            Data Elements:
    
            ^IDS600 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.ConduitFinancerCostsIncurredAndDifferFromTerms.Indicator
    
            ^IDS598 = IDS:RP:ThinCapitalisation:ThirdPartyDebtTest:Liabilities.ThinCapitalisation.ConduitFinancer.Indicator
            */
            assertion = (report.IDS598 == true && report.IDS600 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440968",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A response is required to the following label:
'If you were the conduit financer, did the terms of each relevant debt interest held by you, to the extent that those terms relate to costs incurred, differ from the terms of the ultimate debt interest?'",
                    LongDescription = @"If there is a TRUE response to 'Were you the conduit financer?' then a response is required at 'If you were the conduit financer, did the terms of each relevant debt interest held by you, to the extent that those terms relate to costs incurred, differ from the terms of the ultimate debt interest?'",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:ThirdPartyDebtTest/tns:LiabilitiesConduitFinancerCostsIncurredAndDifferFromTermsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440968" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS598", Value = GetValueOrEmpty(report.IDS598) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS600", Value = GetValueOrEmpty(report.IDS600) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.440969
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440969()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.440969
            If there is a TRUE response to 'Did you rely on the group ratio test?', then the following labels must be provided.
            - Modified Taxable income or loss
            - Net debt deductions 
            - Modified depreciation and forestry costs
            - Excess tax EBITDA amount
            - Tax EBITDA
            - Fixed ratio earnings limit
            - Total disregarded amounts
    
            Legacy Rule Format:
            ^IDS575 = TRUE AND (^IDS565 = NULL OR ^IDS566 = NULL OR ^IDS567 = NULL OR ^IDS568 = NULL OR ^IDS569 = NULL OR ^IDS570 = NULL OR ^IDS571 = NULL)

            Technical Business Rule Format:
            ^IDS575 = TRUE AND (^IDS565 = NULL OR ^IDS566 = NULL OR ^IDS567 = NULL OR ^IDS568 = NULL OR ^IDS569 = NULL OR ^IDS570 = NULL OR ^IDS571 = NULL)
    
            Data Elements:
    
            ^IDS565 = IDS:RP:ThinCapitalisation:EBITDA:IncomeTax.ModifiedTaxableIncomeOrLoss.Amount
    
            ^IDS566 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.NetDeductionDebt.Amount
    
            ^IDS567 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ModifiedDepreciationAndForestryCosts.Amount
    
            ^IDS568 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ExcessTaxEBITDA.Amount
    
            ^IDS569 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.EntityTaxEBITDA.Amount
    
            ^IDS570 = IDS:RP:ThinCapitalisation:EBITDA:Capital.ThinCapitalisation.EarningsLimit.Amount
    
            ^IDS571 = IDS:RP:ThinCapitalisation:EBITDA:Capital.ThinCapitalisation.TotalDisregardedAmounts.Amount
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
            */
            assertion = (report.IDS575 == true && (report.IDS565 == null || report.IDS566 == null || report.IDS567 == null || report.IDS568 == null || report.IDS569 == null || report.IDS570 == null || report.IDS571 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.440969",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Responses are required for the Group ratio test.",
                    LongDescription = @"If there is a TRUE response to 'Did you rely on the group ratio test?', then the following labels must be provided.
- Modified Taxable income or loss
- Net debt deductions 
- Modified depreciation and forestry costs
- Excess tax EBITDA amount
- Tax EBITDA
- Fixed ratio earnings limit
- Total disregarded amounts",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:EBITDA/tns:IncomeTaxModifiedTaxableIncomeOrLossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440969" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS565", Value = GetValueOrEmpty(report.IDS565) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS566", Value = GetValueOrEmpty(report.IDS566) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS567", Value = GetValueOrEmpty(report.IDS567) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS568", Value = GetValueOrEmpty(report.IDS568) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS569", Value = GetValueOrEmpty(report.IDS569) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS570", Value = GetValueOrEmpty(report.IDS570) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS571", Value = GetValueOrEmpty(report.IDS571) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00001
            A value of zero is expected at 'Tax EBITDA' if the sum of the following labels is less than or equal to zero.
            - Modified Taxable income or loss
            - Net debt deductions
            - Modified depreciation and forestry costs
            - Excess tax EBITDA amount
    
            Legacy Rule Format:
            ^IDS2020 <> NULL AND (^IDS565 + ^IDS566 + ^IDS567 + ^IDS568) <= 0 AND ^IDS569 <> 0

            Technical Business Rule Format:
            ^IDS2020 <> NULL AND (^IDS565 + ^IDS566 + ^IDS567 + ^IDS568) <= 0 AND ^IDS569 <> 0
    
            Data Elements:
    
            ^IDS569 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.EntityTaxEBITDA.Amount
    
            ^IDS565 = IDS:RP:ThinCapitalisation:EBITDA:IncomeTax.ModifiedTaxableIncomeOrLoss.Amount
    
            ^IDS566 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.NetDeductionDebt.Amount
    
            ^IDS567 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ModifiedDepreciationAndForestryCosts.Amount
    
            ^IDS568 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ExcessTaxEBITDA.Amount
    
            ^IDS2020 = IDS:RP:ThinCapitalisation:EBITDA
            */
            assertion = (report.RP_ThinCapitalisation_EBITDACollectionExists != false && report.IDS565.GetValueOrDefault() + report.IDS566.GetValueOrDefault() + report.IDS567.GetValueOrDefault() + report.IDS568.GetValueOrDefault() <= 0 && report.IDS569 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00001",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"A value of zero is expected at Tax EBITDA.",
                    LongDescription = @"A value of zero is expected at 'Tax EBITDA' if the sum of the following labels is less than or equal to zero.
- Modified Taxable income or loss
- Net debt deductions
- Modified depreciation and forestry costs
- Excess tax EBITDA amount",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:EBITDA/tns:LiabilitiesEntityTaxEBITDAA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS565", Value = GetValueOrEmpty(report.IDS565) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS566", Value = GetValueOrEmpty(report.IDS566) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS567", Value = GetValueOrEmpty(report.IDS567) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS568", Value = GetValueOrEmpty(report.IDS568) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS569", Value = GetValueOrEmpty(report.IDS569) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00002
            If the sum of the following labels is greater than zero then 'Tax EBITDA' should equal to the sum of the following labels.
            - Modified Taxable income or loss
            - Net debt deductions
            - Modified depreciation and forestry costs
            - Excess tax EBITDA amount
    
            Legacy Rule Format:
            ^IDS2020 <> NULL AND (^IDS565+ ^IDS566 + ^IDS567 + ^IDS568) > 0 AND ^IDS569 <> (^IDS565+ ^IDS566 + ^IDS567 + ^IDS568)

            Technical Business Rule Format:
            ^IDS2020 <> NULL AND (^IDS565+ ^IDS566 + ^IDS567 + ^IDS568) > 0 AND ^IDS569 <> (^IDS565+ ^IDS566 + ^IDS567 + ^IDS568)
    
            Data Elements:
    
            ^IDS569 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.EntityTaxEBITDA.Amount
    
            ^IDS565 = IDS:RP:ThinCapitalisation:EBITDA:IncomeTax.ModifiedTaxableIncomeOrLoss.Amount
    
            ^IDS566 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.NetDeductionDebt.Amount
    
            ^IDS567 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ModifiedDepreciationAndForestryCosts.Amount
    
            ^IDS568 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ExcessTaxEBITDA.Amount
    
            ^IDS2020 = IDS:RP:ThinCapitalisation:EBITDA
            */
            assertion = (report.RP_ThinCapitalisation_EBITDACollectionExists != false && report.IDS565.GetValueOrDefault() + report.IDS566.GetValueOrDefault() + report.IDS567.GetValueOrDefault() + report.IDS568.GetValueOrDefault() > 0 && report.IDS569.GetValueOrDefault() != report.IDS565.GetValueOrDefault() + report.IDS566.GetValueOrDefault() + report.IDS567.GetValueOrDefault() + report.IDS568.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00002",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Tax EBITDA should equal to the sum of  Modified Taxable income or loss, Net debt deductions, Modified depreciation and forestry costs and Excess tax EBITDA amount.",
                    LongDescription = @"If the sum of the following labels is greater than zero then 'Tax EBITDA' should equal to the sum of the following labels.
- Modified Taxable income or loss
- Net debt deductions
- Modified depreciation and forestry costs
- Excess tax EBITDA amount",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:EBITDA/tns:LiabilitiesEntityTaxEBITDAA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS565", Value = GetValueOrEmpty(report.IDS565) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS566", Value = GetValueOrEmpty(report.IDS566) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS567", Value = GetValueOrEmpty(report.IDS567) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS568", Value = GetValueOrEmpty(report.IDS568) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS569", Value = GetValueOrEmpty(report.IDS569) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00003
            Fixed ratio earnings limit' should equal to 30% of 'Tax EBITDA'.
    
            Legacy Rule Format:
            ^IDS570 <> NULL AND ^IDS570 <> (0.3 * ^IDS569)

            Technical Business Rule Format:
            ^IDS570 <> NULL AND ^IDS570 <> (0.3 * ^IDS569)
    
            Data Elements:
    
            ^IDS570 = IDS:RP:ThinCapitalisation:EBITDA:Capital.ThinCapitalisation.EarningsLimit.Amount
    
            ^IDS569 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.EntityTaxEBITDA.Amount
            */
            assertion = (report.IDS570 != null && report.IDS570.GetValueOrDefault() != 0.3M * report.IDS569.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00003",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Fixed ration earnings limit is incorrect. The limit should equal to 30% of Tax EBITDA.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:EBITDA/tns:EarningsLimitA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS570", Value = GetValueOrEmpty(report.IDS570) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS569", Value = GetValueOrEmpty(report.IDS569) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00004
            If 'Fixed Ratio Earnings Limit' is greater than or equal to 'Net Debt Deductions' then 'Amount of Debt Deduction Disallowed' should equal to zero.
    
            Legacy Rule Format:
            ^IDS570 <> NULL AND (^IDS570 >= ^IDS566 AND ^IDS228 <> 0)

            Technical Business Rule Format:
            ^IDS570 <> NULL AND (^IDS570 >= ^IDS566 AND ^IDS228 <> 0)
    
            Data Elements:
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS566 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.NetDeductionDebt.Amount
    
            ^IDS570 = IDS:RP:ThinCapitalisation:EBITDA:Capital.ThinCapitalisation.EarningsLimit.Amount
            */
            assertion = (report.IDS570 != null && (report.IDS570.GetValueOrDefault() >= report.IDS566.GetValueOrDefault() && report.IDS228 != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00004",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"A value of zero is expected at Amount of Debt Deduction Disallowed.",
                    LongDescription = @"If 'Fixed Ratio Earnings Limit' is greater than or equal to 'Net Debt Deductions' then 'Amount of Debt Deduction Disallowed' should equal to zero.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesDeductionDebtDisallowedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00004" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS570", Value = GetValueOrEmpty(report.IDS570) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS566", Value = GetValueOrEmpty(report.IDS566) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00005
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00005()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00005
            If 'Fixed Ratio Earnings Limit' is less than 'Net Debt Deductions' then 'Amount of Debt Deduction Disallowed' should equal to 'Net Debt Deductions' minus 'Fixed Ratio Earnings Limit'.
    
            Legacy Rule Format:
            ^IDS570 <> NULL AND ^IDS570 < ^IDS566 AND ^IDS228 <> (^IDS566 - ^IDS570)

            Technical Business Rule Format:
            ^IDS570 <> NULL AND ^IDS570 < ^IDS566 AND ^IDS228 <> (^IDS566 - ^IDS570)
    
            Data Elements:
    
            ^IDS228 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.DeductionDebtDisallowed.Amount
    
            ^IDS566 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.NetDeductionDebt.Amount
    
            ^IDS570 = IDS:RP:ThinCapitalisation:EBITDA:Capital.ThinCapitalisation.EarningsLimit.Amount
            */
            assertion = (report.IDS570 != null && report.IDS570.GetValueOrDefault() < report.IDS566.GetValueOrDefault() && report.IDS228.GetValueOrDefault() != report.IDS566.GetValueOrDefault() - report.IDS570.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00005",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Amount of Debt Deduction Disallowed should equal to Net Debt Deductions minus Fixed Ratio Earnings Limit.",
                    LongDescription = @"If 'Fixed Ratio Earnings Limit' is less than 'Net Debt Deductions' then 'Amount of Debt Deduction Disallowed' should equal to 'Net Debt Deductions' minus 'Fixed Ratio Earnings Limit'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:LiabilitiesDeductionDebtDisallowedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00005" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS570", Value = GetValueOrEmpty(report.IDS570) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS566", Value = GetValueOrEmpty(report.IDS566) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS228", Value = GetValueOrEmpty(report.IDS228) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00006
            If 'Excess tax EBITDA amount' is greater than zero then the label 'Highest excess tax EBITDA amounts transferred from a controlled entity' should be completed.
    
            Legacy Rule Format:
            ^IDS568 > 0 AND Count(^IDS2021) = 0

            Technical Business Rule Format:
            ^IDS568 > 0 AND Count(^IDS2021) = 0
    
            Data Elements:
    
            ^IDS568 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.ExcessTaxEBITDA.Amount
    
            ^IDS2021 = IDS:RP:ThinCapitalisation:EBITDA:HighestExcessEBITDA
            */
            assertion = (report.IDS568.GetValueOrDefault() > 0 && Count(report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00006",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"It is expected to complete the label Highest excess tax EBITDA amounts transferred from a controlled entity.",
                    LongDescription = @"If 'Excess tax EBITDA amount' is greater than zero then the label 'Highest excess tax EBITDA amounts transferred from a controlled entity' should be completed.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:EBITDA/tns:HighestExcessEBITDACollection/tns:HighestExcessEBITDA" + OccurrenceIndex(report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollection == null ? 0 : (report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollection.Count() == 0 ? 0 : report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollection.Last().OccurrenceIndex)) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00006" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS568", Value = GetValueOrEmpty(report.IDS568) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00007
            If there is a TRUE response to 'Did you rely on the group ratio test?' then
            'GR group EBITDA' should equal to the sum of the following labels.
            - GR group net profit (disregarding tax expenses)
            - GR group adjusted net third party interest expense
            - GR group depreciation & amortisation expenses
    
            Legacy Rule Format:
            ^IDS575 = TRUE AND ^IDS582 <> NULL AND ^IDS582 <>  (^IDS579 + ^IDS580+ ^IDS581)

            Technical Business Rule Format:
            ^IDS575 = TRUE AND ^IDS582 <> NULL AND ^IDS582 <>  (^IDS579 + ^IDS580+ ^IDS581)
    
            Data Elements:
    
            ^IDS582 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.GroupTaxEBITDA.Amount
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS579 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.NetProfit.Amount
    
            ^IDS580 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.NetThirdPartyInterestExpenseAdjusted.Amount
    
            ^IDS581 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.DepreciationAndAmortisationExpenses.Amount
            */
            assertion = (report.IDS575 == true && report.IDS582 != null && report.IDS582.GetValueOrDefault() != report.IDS579.GetValueOrDefault() + report.IDS580.GetValueOrDefault() + report.IDS581.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00007",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"GR group EBITDA should equal to the sum of the GR group net profit (disregarding tax expenses), GR group adjusted net third party interest expense and GR group depreciation & amortisation expenses.",
                    LongDescription = @"If there is a TRUE response to 'Did you rely on the group ratio test?' then
'GR group EBITDA' should equal to the sum of the following labels.
- GR group net profit (disregarding tax expenses)
- GR group adjusted net third party interest expense
- GR group depreciation & amortisation expenses",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest/tns:LiabilitiesGroupTaxEBITDAA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00007" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS582", Value = GetValueOrEmpty(report.IDS582) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS579", Value = GetValueOrEmpty(report.IDS579) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS580", Value = GetValueOrEmpty(report.IDS580) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS581", Value = GetValueOrEmpty(report.IDS581) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00009
            If there is a TRUE response to 'Did you rely on the group ratio test?' then it is expected thtat 'Group ratio earnings limit' is the multiplication of 'Tax EBITDA' and 'Group ratio'.
    
            Legacy Rule Format:
            ^IDS575 = TRUE AND ^IDS584 <> NULL AND ^IDS584 <> (^IDS569 * ^IDS583)

            Technical Business Rule Format:
            ^IDS575 = TRUE AND ^IDS584 <> NULL AND ^IDS584 <> (^IDS569 * ^IDS583)
    
            Data Elements:
    
            ^IDS584 = IDS:RP:ThinCapitalisation:GroupRatioTest:Capital.ThinCapitalisation.GroupRatioEarningsLimit.Amount
    
            ^IDS569 = IDS:RP:ThinCapitalisation:EBITDA:Liabilities.ThinCapitalisation.EntityTaxEBITDA.Amount
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS583 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.Ratio.Amount
            */
            assertion = (report.IDS575 == true && report.IDS584 != null && report.IDS584.GetValueOrDefault() != report.IDS569.GetValueOrDefault() * report.IDS583.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00009",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"It is expected that Group ratio earnings limit is the multiplication of Tax EBITDA and Group ratio.",
                    LongDescription = @"If there is a TRUE response to 'Did you rely on the group ratio test?' then it is expected thtat 'Group ratio earnings limit' is the multiplication of 'Tax EBITDA' and 'Group ratio'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest/tns:GroupRatioEarningsLimitA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00009" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS584", Value = GetValueOrEmpty(report.IDS584) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS569", Value = GetValueOrEmpty(report.IDS569) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS583", Value = GetValueOrEmpty(report.IDS583) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00016
            If there is a TRUE response to 'Did you rely on the group ratio test?' then it is expected that 'Group ratio' is the division of 'GR group net third party interest expense' by 'GR group EBITDA'.
    
            Legacy Rule Format:
            ^IDS575 = TRUE AND ^IDS583 <> NULL AND ^IDS583 <> ((^IDS578 / ^IDS582))

            Technical Business Rule Format:
            ^IDS575 = TRUE AND ^IDS583 <> NULL AND ^IDS583 <> ((^IDS578 / ^IDS582))
    
            Data Elements:
    
            ^IDS583 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.Ratio.Amount
    
            ^IDS575 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.GroupRatioTest.Indicator
    
            ^IDS578 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.NetThirdPartyInterestExpense.Amount
    
            ^IDS582 = IDS:RP:ThinCapitalisation:GroupRatioTest:Liabilities.ThinCapitalisation.GroupTaxEBITDA.Amount
            */
            assertion = (report.IDS575 == true && report.IDS583 != null && report.IDS583.GetValueOrDefault() != report.IDS578.GetValueOrDefault() / report.IDS582.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00016",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"It is expected that Group ratio is the division of GR group net third party interest expense by GR group EBITDA.",
                    LongDescription = @"If there is a TRUE response to 'Did you rely on the group ratio test?' then it is expected that 'Group ratio' is the division of 'GR group net third party interest expense' by 'GR group EBITDA'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:GroupRatioTest/tns:LiabilitiesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00016" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS575", Value = GetValueOrEmpty(report.IDS575) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS583", Value = GetValueOrEmpty(report.IDS583) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS578", Value = GetValueOrEmpty(report.IDS578) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS582", Value = GetValueOrEmpty(report.IDS582) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00017
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00017()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00017
            If there is an amount greater than zero at 'Debt deductions on debt from related non-residents', then it must be less than or equal to 'Debt deductions'.
    
            Legacy Rule Format:
            ^IDS608 > 0 AND ^IDS608 > ^IDS607

            Technical Business Rule Format:
            ^IDS608 > 0 AND ^IDS608 > ^IDS607
    
            Data Elements:
    
            ^IDS608 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Liabilities.ThinCapitalisation.RelatedNonResidentsDebtDeduction.Amount
    
            ^IDS607 = IDS:RP:ThinCapitalisation:SpecialPurposeEntity:Liabilities.ThinCapitalisation.DeductionDebt.Amount
            */
            assertion = (report.IDS608.GetValueOrDefault() > 0 && report.IDS608.GetValueOrDefault() > report.IDS607.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00017",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Special Purpose Entity - Debt deductions on debt from related non-residents amount incorrect.",
                    LongDescription = @"If there is an amount greater than zero at 'Debt deductions on debt from related non-residents', then it must be less than or equal to 'Debt deductions'.",
                    Location = "/tns:IDS/tns:RP/tns:ThinCapitalisation/tns:SpecialPurposeEntity/tns:LiabilitiesRelatedNonResidentsDebtDeductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00017" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS608", Value = GetValueOrEmpty(report.IDS608) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS607", Value = GetValueOrEmpty(report.IDS607) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00018
            If there is a TRUE response to 'Were the debt deduction creation rules applicable to you?', then the following must be completed: 
            - Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.
            - In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?
            - In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?
            - In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?
            - In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?
            - Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?
    
            Legacy Rule Format:
            ^IDS640 = TRUE
            AND (^IDS628 = NULL
            OR ^IDS643 = NULL
            OR ^IDS650 = NULL
            OR ^IDS658 = NULL
            OR ^IDS669 = NULL
            OR ^IDS679 = NULL)

            Technical Business Rule Format:
            ^IDS640 = TRUE
            AND (^IDS628 = NULL
            OR ^IDS643 = NULL
            OR ^IDS650 = NULL
            OR ^IDS658 = NULL
            OR ^IDS669 = NULL
            OR ^IDS679 = NULL)
    
            Data Elements:
    
            ^IDS628 = IDS:RP:Liabilities.DebtDeductionCreationRules.RestructuringEventsDescription.Indicator
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
    
            ^IDS643 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairPriorIncomeYear.Indicator
    
            ^IDS650 = IDS:RP:Liabilities.DebtDeductionCreationRules.PriorYearDistributionsAssociatePair.Indicator
    
            ^IDS658 = IDS:RP:Liabilities.DebtDeductionCreationRules.CGTAssetAquisitionAssociatePairCurrentIncomeYear.Indicator
    
            ^IDS669 = IDS:RP:CurrentIncomeYearPaymentsorDistributionsSelection:Liabilities.DebtDeductionCreationRules.CurrentYearDistributionsAssociatePair.Indicator
    
            ^IDS679 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtDisallowedAssociatePairTransaction.Indicator
            */
            assertion = (report.IDS640 == true && (report.IDS628 == null || report.IDS643 == null || report.IDS650 == null || report.IDS658 == null || report.IDS669 == null || report.IDS679 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00018",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"You have answered TRUE to 'Were the debt deduction creation rules applicable to you?', and missing critical information within Section H. Please see the Long Descrption for more information.",
                    LongDescription = @"If there is a TRUE response to 'Were the debt deduction creation rules applicable to you?', then the following must be completed: 
- Did you restructure or replace an arrangement during the current or prior income year which would have satisfied the conditions in subsection 820-423A(2) or (5) if the arrangement was still in place and the debt deduction creation rules were applicable? Disregard paragraph 820-423A(2)(g) and (5)(f) when answering this question.
- In a prior income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair other than a CGT asset covered by section 820-423AA?
- In a prior income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?
- In the current income year, did you directly or indirectly acquire a CGT asset, or legal or equitable obligation, from an associate pair?
- In the current income year, did you make one or more payments or distributions covered by subsection 820-423A(5A), directly or indirectly, to an associate pair?
- Were any of your debt deductions, for the current or prior income year, disallowed by subsection 820-423A(1) because you were either: (a) an associate pair of an acquirer or of an associate disposer for a transaction covered by subsection 820-423A(2); or (b) an associate pair of the payer or of an associate recipient for a transaction covered by subsection 820-423A(5)?",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00018" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS628", Value = GetValueOrEmpty(report.IDS628) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS643", Value = GetValueOrEmpty(report.IDS643) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS650", Value = GetValueOrEmpty(report.IDS650) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS658", Value = GetValueOrEmpty(report.IDS658) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS669", Value = GetValueOrEmpty(report.IDS669) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS679", Value = GetValueOrEmpty(report.IDS679) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.IDS.W00019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDSW00019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.IDS.W00019
            If 'Entity type code for the income year' is equal to '1' and there is a FALSE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' then 'Were the debt deduction creation rules applicable to you? must be TRUE.
    
            Legacy Rule Format:
            ^IDS223 = "1"
            AND ^IDS606 = FALSE
            AND ^IDS640 <> TRUE

            Technical Business Rule Format:
            ^IDS223 = '1'
            AND ^IDS606 = FALSE
            AND ^IDS640 <> TRUE
    
            Data Elements:
    
            ^IDS640 = IDS:RP:Liabilities.DebtDeductionCreationRules.DeductionDebtApplicable.Indicator
    
            ^IDS223 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.EntityType.Code
    
            ^IDS606 = IDS:RP:ThinCapitalisation:Liabilities.ThinCapitalisation.SpecialPurposeEntity.Indicator
            */
            assertion = (report.IDS223 == @"1" && report.IDS606 == false && report.IDS640 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IDS.W00019",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"The label 'Were the debt deduction creation rules applicable to you?' must be TRUE.",
                    LongDescription = @"If 'Entity type code for the income year' is equal to '1' and there is a FALSE response to 'Were you, or a member of your tax consolidated group, a special purpose entity subject to section 820-39 of the ITAA 1997?' then 'Were the debt deduction creation rules applicable to you? must be TRUE.",
                    Location = "/tns:IDS/tns:RP/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.W00019" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS223", Value = GetValueOrEmpty(report.IDS223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS606", Value = GetValueOrEmpty(report.IDS606) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IDS640", Value = GetValueOrEmpty(report.IDS640) });

                response.Add(processMessage);
            }
        }
        #endregion

    }
}