using System;
using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using VaTS;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;
using Ato.EN.IntegrationServices.CodeGenerationPTR;
using DataContracts;

namespace Ato.CD.Inbound.PTR202502
{
    public class CrossFormValidatorDISTBENTRT : ICrossFormValidator
    {
        private PTR2025 ParentDocument { get; }

        private DISTBENTRT2024 ChildDocument { get; }

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> Response { get; private set; }

        public CrossFormValidatorDISTBENTRT(PTR2025 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (DISTBENTRT2024)childDocument.ConsumedReport;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOPTR440035();
            VRATOPTR440036();
            VRATOPTR440037();
            VRATOPTRW00013();

            return Response;
        }


        #region VR.ATO.PTR.440035

        /*  
            VR.ATO.PTR.440035
            Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form

            Legacy Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR500 <> ^DISTBENTRT1

            Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR500 <> ^DISTBENTRT1

            Data Elements:

            ^PTR500 = PTR:RP:Period.Start.Date

            ^DISTBENTRT1 = DISTBENTRT:RP:Period.Start.Date
        */
        public void VRATOPTR440035()
        {
            bool assertion = ParentDocument.PTR500 != ChildDocument.DISTBENTRT1;
            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form",
                    Location = "/tns:PTR/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440035" } ,
                        new ProcessMessageParameter { Name = "PTR500", Value = ParentDocument.PTR500.HasValue ? ParentDocument.PTR500.Value.ToString("yyyy-MM-dd"):"null" },
                        new ProcessMessageParameter { Name = "DISTBENTRT1", Value = ChildDocument.DISTBENTRT1.HasValue ?  ChildDocument.DISTBENTRT1.Value.ToString("yyyy-MM-dd"):"null" },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion VR.ATO.PTR.440035

        #region VR.ATO.PTR.440036

        /*  
            VR.ATO.PTR.440036
            Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form

            Legacy Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR501 <> ^DISTBENTRT2

            Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR501 <> ^DISTBENTRT2

            Data Elements:
    
            ^PTR501 = PTR:RP:Period.End.Date
    
            ^DISTBENTRT2 = DISTBENTRT:RP:Period.End.Date
        */
        public void VRATOPTR440036()
        {
            bool assertion = ParentDocument.PTR501 != ChildDocument.DISTBENTRT2;
            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form",
                    Location = "/tns:PTR/tns:RP/tns:PeriodEndD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter{ Name = "RuleIdentifier", Value = "VR.ATO.PTR.440036" },
                        new ProcessMessageParameter { Name = "PTR501", Value = ParentDocument.PTR501.HasValue ? ParentDocument.PTR501.Value.ToString("yyyy-MM-dd"):"Null" },
                        new ProcessMessageParameter { Name = "DISTBENTRT2", Value =ChildDocument.DISTBENTRT2.HasValue ? ChildDocument.DISTBENTRT2.Value.ToString("yyyy-MM-dd") :"Null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion VR.ATO.PTR.440036

        #region VR.ATO.PTR.440037

        /*  
            VR.ATO.PTR.440037
            The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form

            Legacy Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR5 <> ^DISTBENTRT3

            Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR5 <> ^DISTBENTRT3

            Data Elements:

            ^PTR5 = PTR:RP:Identifiers.TaxFileNumber.Identifier

            ^DISTBENTRT3 = DISTBENTRT:RP:Identifiers.TaxFileNumber.Identifier
        */
        public void VRATOPTR440037()
        {
            bool assertion = ParentDocument.PTR5 != ChildDocument.DISTBENTRT3;
            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form",
                    Location = "/tns:PTR/tns:RP/tns:TaxFileNumberId",

                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440037" },
                        new ProcessMessageParameter { Name = "PTR5", Value = ParentDocument.PTR5 },
                        new ProcessMessageParameter { Name = "DISTBENTRT3", Value = ChildDocument.DISTBENTRT3 },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }


        #endregion VR.ATO.PTR.440037

        #region VR.ATO.PTR.W00013

        /*  
            VR.ATO.PTR.W00013
            Distributions to Beneficiaries of Trust schedule is provided but Trust distributions - Income amounts or credits are not declared

            Legacy Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR66 = NULL AND ^PTR70 = NULL AND ^PTR322 = NULL

            Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^PTR66 = NULL AND ^PTR70 = NULL AND ^PTR322 = NULL

            Data Elements:

            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR70 = PTR:RP:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR322 = PTR:RP:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
        */
        public void VRATOPTRW00013()
        {
            bool assertion = (!ParentDocument.PTR66.HasValue) && (!ParentDocument.PTR70.HasValue) && (!ParentDocument.PTR322.HasValue);
            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.W00013",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure Trust distributions - Income amounts or credits from trusts income are declared",
                    LongDescription = @"Ensure Trust distributions - Income amounts or credits are declared when Distributions to Beneficiaries of Trust schedule is provided",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00013" },
                        new ProcessMessageParameter { Name = "PTR66", Value = "null" },
                        new ProcessMessageParameter { Name = "PTR70", Value = "null" },
                        new ProcessMessageParameter { Name = "PTR322", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }

        #endregion VR.ATO.PTR.W00013

    }
}