
using System;
using System.Collections.Generic;
using Au.Gov.Sbr.Xbrl.Document;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationCGTS
{
    public class CGTS2018Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public CGTS2018 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public CGTS2018 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            CGTS2018 report;
            report = new CGTS2018();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrances being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) ));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.CGTS1 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}IncomeTax.AttributionManagedInvestmentTrustClass.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS153 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS118 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS119 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS127 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS128 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.SmallBusinessRollover.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS129 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS130 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Income.CapitalGainsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS131 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS150 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Miscellaneous.RequestedAmendment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS151 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS138 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.SmallBusinessExemptionAssetOrEvent.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS139 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.ForeignResidentExemption.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS140 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.ScriptForScriptRolloverDeferral.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS45 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.InterCompanyAssetRolloverDeferral.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS46 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.DemergingEntityExemption.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS47 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPEarnoutArrangementGroupNumberContext
            IEnumerable<SBRContext> sbrRPEarnoutArrangementGroupNumberContexts = null;
            SBRContext sbrRPEarnoutArrangementGroupNumberContext = null;
    
            sbrRPEarnoutArrangementGroupNumberContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.TYPED && dd.Dimension.EndsWith("}EarnoutArrangementGroupItemDimension")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPEarnoutArrangementGroupNumberContexts != null && sbrRPEarnoutArrangementGroupNumberContexts.Count() > 0)
            {
        
                report.RPEarnoutArrangementGroupNumberContextCollection = new List<CGTS2018.RPEarnoutArrangementGroupNumberContext>();
                for (int i = 0; i < sbrRPEarnoutArrangementGroupNumberContexts.Count(); i++)
                {
                    sbrRPEarnoutArrangementGroupNumberContext = sbrRPEarnoutArrangementGroupNumberContexts.ElementAt(i);
                    CGTS2018.RPEarnoutArrangementGroupNumberContext rpEarnoutArrangementGroupNumberContext = new CGTS2018.RPEarnoutArrangementGroupNumberContext();
                    report.RPEarnoutArrangementGroupNumberContextCollection.Add(rpEarnoutArrangementGroupNumberContext);
            
                    rpEarnoutArrangementGroupNumberContext.Id = sbrRPEarnoutArrangementGroupNumberContext.Id;
                    rpEarnoutArrangementGroupNumberContext.Identifier = sbrRPEarnoutArrangementGroupNumberContext.EntityIdentifier;
                    rpEarnoutArrangementGroupNumberContext.IdentifierScheme = sbrRPEarnoutArrangementGroupNumberContext.EntityScheme;
            
                    rpEarnoutArrangementGroupNumberContext.Count = sbrRPEarnoutArrangementGroupNumberContexts.Count();
                    rpEarnoutArrangementGroupNumberContext.Exists = true;
                    rpEarnoutArrangementGroupNumberContext.OccurrenceIndex = Array.IndexOf(contextArray, sbrRPEarnoutArrangementGroupNumberContext) + 1;
                    rpEarnoutArrangementGroupNumberContext.LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPEarnoutArrangementGroupNumberContexts.Last()) + 1;
            
    
                    if (rpEarnoutArrangementGroupNumberContext.IdentifierScheme == "http://www.ato.gov.au/tfn") rpEarnoutArrangementGroupNumberContext.IdentifierTFN = rpEarnoutArrangementGroupNumberContext.Identifier;
        
                    rpEarnoutArrangementGroupNumberContext.StartDate = ((SBRDurationPeriod)sbrRPEarnoutArrangementGroupNumberContext.Period).DurationStartAsDateObject;
                    rpEarnoutArrangementGroupNumberContext.EndDate = ((SBRDurationPeriod)sbrRPEarnoutArrangementGroupNumberContext.Period).DurationEndAsDateObject;
            
                    currentSegment = sbrRPEarnoutArrangementGroupNumberContext.Segment.Segment;
            
                    currentNamespaceManager = new XmlNamespaceManager(currentSegment.OwnerDocument.NameTable);
                    currentNamespaceManager.AddNamespace("xbrldi", "http://xbrl.org/2006/xbrldi");
            
                    currentNode = currentSegment.SelectSingleNode("xbrldi:typedMember[contains(@dimension, 'EarnoutArrangementGroupItemDimension')]", currentNamespaceManager);
            
                    if (currentNode != null)
                    {
                        rpEarnoutArrangementGroupNumberContext.EarnoutArrangementGroupItemDimensionValue = currentNode.InnerText.Trim();
                    }
            
                    sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPEarnoutArrangementGroupNumberContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    rpEarnoutArrangementGroupNumberContext.CGTS160 = rpEarnoutArrangementGroupNumberContext.EarnoutArrangementGroupItemDimensionValue;
            
                    if (sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Miscellaneous.EarnoutArrangementEligibility.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpEarnoutArrangementGroupNumberContext.CGTS133 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Miscellaneous.EarnoutArrangementDuration.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpEarnoutArrangementGroupNumberContext.CGTS134 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Miscellaneous.EarnoutArrangementApplicableYear.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpEarnoutArrangementGroupNumberContext.CGTS135 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}CapitalGainsTax.EarnoutArrangementEstimatedCapitalProceedsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpEarnoutArrangementGroupNumberContext.CGTS136 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPEarnoutArrangementGroupNumberElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}CapitalGainsTax.CapitalGainsOrLossesOnNonQualifyingEarnoutArrangementNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpEarnoutArrangementGroupNumberContext.CGTS152 = decimal.Parse(currentValues[0].Value);
                    }
        } 
            } 
            #endregion End of Context sbrRPEarnoutArrangementGroupNumberContext
    
            #region sbrRPAppliedContext
            IEnumerable<SBRContext> sbrRPAppliedContexts = null;
            SBRContext sbrRPAppliedContext = null;
    
            sbrRPAppliedContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AccountingTreatmentMethodDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Applied")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAppliedElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAppliedContexts != null && sbrRPAppliedContexts.Count() > 0)
            {
                sbrRPAppliedContext = sbrRPAppliedContexts.First();
        
                report.RPAppliedId = sbrRPAppliedContext.Id;
                report.RPAppliedIdentifier = sbrRPAppliedContext.EntityIdentifier;
                report.RPAppliedIdentifierScheme = sbrRPAppliedContext.EntityScheme;
        
                report.RPAppliedCount = sbrRPAppliedContexts.Count();
                report.RPAppliedExists = true;
                report.RPAppliedOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAppliedContext) + 1;
                report.RPAppliedLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAppliedContexts.Last()) + 1;
        

                if (report.RPAppliedIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAppliedIdentifierTFN = report.RPAppliedIdentifier;
    
                report.RPAppliedStartDate = ((SBRDurationPeriod)sbrRPAppliedContext.Period).DurationStartAsDateObject;
                report.RPAppliedEndDate = ((SBRDurationPeriod)sbrRPAppliedContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAppliedElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAppliedContexts.Count(); i++)
                {
                    sbrRPAppliedContext = sbrRPAppliedContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAppliedContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAppliedElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAppliedElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAppliedElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAppliedElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.TransferredTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS122 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAppliedElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS123 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAppliedElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}TaxConcession.CapitalGains.DiscountTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS126 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAppliedContext
    
            #region sbrRPAppliedY0Context
            IEnumerable<SBRContext> sbrRPAppliedY0Contexts = null;
            SBRContext sbrRPAppliedY0Context = null;
    
            sbrRPAppliedY0Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AccountingTreatmentMethodDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Applied"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0")) ))));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAppliedY0ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAppliedY0Contexts != null && sbrRPAppliedY0Contexts.Count() > 0)
            {
                sbrRPAppliedY0Context = sbrRPAppliedY0Contexts.First();
        
                report.RPAppliedY0Id = sbrRPAppliedY0Context.Id;
                report.RPAppliedY0Identifier = sbrRPAppliedY0Context.EntityIdentifier;
                report.RPAppliedY0IdentifierScheme = sbrRPAppliedY0Context.EntityScheme;
        
                report.RPAppliedY0Count = sbrRPAppliedY0Contexts.Count();
                report.RPAppliedY0Exists = true;
                report.RPAppliedY0OccurrenceIndex = Array.IndexOf(contextArray, sbrRPAppliedY0Context) + 1;
                report.RPAppliedY0LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAppliedY0Contexts.Last()) + 1;
        

                if (report.RPAppliedY0IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAppliedY0IdentifierTFN = report.RPAppliedY0Identifier;
    
                report.RPAppliedY0StartDate = ((SBRDurationPeriod)sbrRPAppliedY0Context.Period).DurationStartAsDateObject;
                report.RPAppliedY0EndDate = ((SBRDurationPeriod)sbrRPAppliedY0Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAppliedY0ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAppliedY0Contexts.Count(); i++)
                {
                    sbrRPAppliedY0Context = sbrRPAppliedY0Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAppliedY0Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAppliedY0ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAppliedY0ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAppliedY0ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAppliedY0ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS120 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAppliedY0Context
    
            #region sbrRPAppliedY0MinusContext
            IEnumerable<SBRContext> sbrRPAppliedY0MinusContexts = null;
            SBRContext sbrRPAppliedY0MinusContext = null;
    
            sbrRPAppliedY0MinusContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AccountingTreatmentMethodDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Applied"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus")) ))));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAppliedY0MinusElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAppliedY0MinusContexts != null && sbrRPAppliedY0MinusContexts.Count() > 0)
            {
                sbrRPAppliedY0MinusContext = sbrRPAppliedY0MinusContexts.First();
        
                report.RPAppliedY0MinusId = sbrRPAppliedY0MinusContext.Id;
                report.RPAppliedY0MinusIdentifier = sbrRPAppliedY0MinusContext.EntityIdentifier;
                report.RPAppliedY0MinusIdentifierScheme = sbrRPAppliedY0MinusContext.EntityScheme;
        
                report.RPAppliedY0MinusCount = sbrRPAppliedY0MinusContexts.Count();
                report.RPAppliedY0MinusExists = true;
                report.RPAppliedY0MinusOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAppliedY0MinusContext) + 1;
                report.RPAppliedY0MinusLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAppliedY0MinusContexts.Last()) + 1;
        

                if (report.RPAppliedY0MinusIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAppliedY0MinusIdentifierTFN = report.RPAppliedY0MinusIdentifier;
    
                report.RPAppliedY0MinusStartDate = ((SBRDurationPeriod)sbrRPAppliedY0MinusContext.Period).DurationStartAsDateObject;
                report.RPAppliedY0MinusEndDate = ((SBRDurationPeriod)sbrRPAppliedY0MinusContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAppliedY0MinusElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAppliedY0MinusContexts.Count(); i++)
                {
                    sbrRPAppliedY0MinusContext = sbrRPAppliedY0MinusContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAppliedY0MinusContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAppliedY0MinusElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAppliedY0MinusElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAppliedY0MinusElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAppliedY0MinusElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.NetTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS43 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAppliedY0MinusContext
    
            #region sbrRPListedCompanySharesContext
            IEnumerable<SBRContext> sbrRPListedCompanySharesContexts = null;
            SBRContext sbrRPListedCompanySharesContext = null;
    
            sbrRPListedCompanySharesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ListedCompanyShares")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPListedCompanySharesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPListedCompanySharesContexts != null && sbrRPListedCompanySharesContexts.Count() > 0)
            {
                sbrRPListedCompanySharesContext = sbrRPListedCompanySharesContexts.First();
        
                report.RPListedCompanySharesId = sbrRPListedCompanySharesContext.Id;
                report.RPListedCompanySharesIdentifier = sbrRPListedCompanySharesContext.EntityIdentifier;
                report.RPListedCompanySharesIdentifierScheme = sbrRPListedCompanySharesContext.EntityScheme;
        
                report.RPListedCompanySharesCount = sbrRPListedCompanySharesContexts.Count();
                report.RPListedCompanySharesExists = true;
                report.RPListedCompanySharesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPListedCompanySharesContext) + 1;
                report.RPListedCompanySharesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPListedCompanySharesContexts.Last()) + 1;
        

                if (report.RPListedCompanySharesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPListedCompanySharesIdentifierTFN = report.RPListedCompanySharesIdentifier;
    
                report.RPListedCompanySharesStartDate = ((SBRDurationPeriod)sbrRPListedCompanySharesContext.Period).DurationStartAsDateObject;
                report.RPListedCompanySharesEndDate = ((SBRDurationPeriod)sbrRPListedCompanySharesContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPListedCompanySharesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPListedCompanySharesContexts.Count(); i++)
                {
                    sbrRPListedCompanySharesContext = sbrRPListedCompanySharesContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPListedCompanySharesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPListedCompanySharesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPListedCompanySharesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPListedCompanySharesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPListedCompanySharesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS95 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPListedCompanySharesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS96 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPListedCompanySharesContext
    
            #region sbrRPOtherSharesContext
            IEnumerable<SBRContext> sbrRPOtherSharesContexts = null;
            SBRContext sbrRPOtherSharesContext = null;
    
            sbrRPOtherSharesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}OtherShares")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOtherSharesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOtherSharesContexts != null && sbrRPOtherSharesContexts.Count() > 0)
            {
                sbrRPOtherSharesContext = sbrRPOtherSharesContexts.First();
        
                report.RPOtherSharesId = sbrRPOtherSharesContext.Id;
                report.RPOtherSharesIdentifier = sbrRPOtherSharesContext.EntityIdentifier;
                report.RPOtherSharesIdentifierScheme = sbrRPOtherSharesContext.EntityScheme;
        
                report.RPOtherSharesCount = sbrRPOtherSharesContexts.Count();
                report.RPOtherSharesExists = true;
                report.RPOtherSharesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOtherSharesContext) + 1;
                report.RPOtherSharesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOtherSharesContexts.Last()) + 1;
        

                if (report.RPOtherSharesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOtherSharesIdentifierTFN = report.RPOtherSharesIdentifier;
    
                report.RPOtherSharesStartDate = ((SBRDurationPeriod)sbrRPOtherSharesContext.Period).DurationStartAsDateObject;
                report.RPOtherSharesEndDate = ((SBRDurationPeriod)sbrRPOtherSharesContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPOtherSharesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOtherSharesContexts.Count(); i++)
                {
                    sbrRPOtherSharesContext = sbrRPOtherSharesContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOtherSharesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOtherSharesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOtherSharesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOtherSharesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOtherSharesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS97 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOtherSharesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS98 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOtherSharesContext
    
            #region sbrRPListedUnitTrustsContext
            IEnumerable<SBRContext> sbrRPListedUnitTrustsContexts = null;
            SBRContext sbrRPListedUnitTrustsContext = null;
    
            sbrRPListedUnitTrustsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ListedUnitTrusts")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPListedUnitTrustsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPListedUnitTrustsContexts != null && sbrRPListedUnitTrustsContexts.Count() > 0)
            {
                sbrRPListedUnitTrustsContext = sbrRPListedUnitTrustsContexts.First();
        
                report.RPListedUnitTrustsId = sbrRPListedUnitTrustsContext.Id;
                report.RPListedUnitTrustsIdentifier = sbrRPListedUnitTrustsContext.EntityIdentifier;
                report.RPListedUnitTrustsIdentifierScheme = sbrRPListedUnitTrustsContext.EntityScheme;
        
                report.RPListedUnitTrustsCount = sbrRPListedUnitTrustsContexts.Count();
                report.RPListedUnitTrustsExists = true;
                report.RPListedUnitTrustsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPListedUnitTrustsContext) + 1;
                report.RPListedUnitTrustsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPListedUnitTrustsContexts.Last()) + 1;
        

                if (report.RPListedUnitTrustsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPListedUnitTrustsIdentifierTFN = report.RPListedUnitTrustsIdentifier;
    
                report.RPListedUnitTrustsStartDate = ((SBRDurationPeriod)sbrRPListedUnitTrustsContext.Period).DurationStartAsDateObject;
                report.RPListedUnitTrustsEndDate = ((SBRDurationPeriod)sbrRPListedUnitTrustsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPListedUnitTrustsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPListedUnitTrustsContexts.Count(); i++)
                {
                    sbrRPListedUnitTrustsContext = sbrRPListedUnitTrustsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPListedUnitTrustsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPListedUnitTrustsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPListedUnitTrustsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPListedUnitTrustsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPListedUnitTrustsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS99 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPListedUnitTrustsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS100 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPListedUnitTrustsContext
    
            #region sbrRPOtherUnitsContext
            IEnumerable<SBRContext> sbrRPOtherUnitsContexts = null;
            SBRContext sbrRPOtherUnitsContext = null;
    
            sbrRPOtherUnitsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}OtherUnits")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOtherUnitsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOtherUnitsContexts != null && sbrRPOtherUnitsContexts.Count() > 0)
            {
                sbrRPOtherUnitsContext = sbrRPOtherUnitsContexts.First();
        
                report.RPOtherUnitsId = sbrRPOtherUnitsContext.Id;
                report.RPOtherUnitsIdentifier = sbrRPOtherUnitsContext.EntityIdentifier;
                report.RPOtherUnitsIdentifierScheme = sbrRPOtherUnitsContext.EntityScheme;
        
                report.RPOtherUnitsCount = sbrRPOtherUnitsContexts.Count();
                report.RPOtherUnitsExists = true;
                report.RPOtherUnitsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOtherUnitsContext) + 1;
                report.RPOtherUnitsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOtherUnitsContexts.Last()) + 1;
        

                if (report.RPOtherUnitsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOtherUnitsIdentifierTFN = report.RPOtherUnitsIdentifier;
    
                report.RPOtherUnitsStartDate = ((SBRDurationPeriod)sbrRPOtherUnitsContext.Period).DurationStartAsDateObject;
                report.RPOtherUnitsEndDate = ((SBRDurationPeriod)sbrRPOtherUnitsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPOtherUnitsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOtherUnitsContexts.Count(); i++)
                {
                    sbrRPOtherUnitsContext = sbrRPOtherUnitsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOtherUnitsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOtherUnitsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOtherUnitsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOtherUnitsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOtherUnitsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS101 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOtherUnitsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS102 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOtherUnitsContext
    
            #region sbrRPAustralianRealPropertyContext
            IEnumerable<SBRContext> sbrRPAustralianRealPropertyContexts = null;
            SBRContext sbrRPAustralianRealPropertyContext = null;
    
            sbrRPAustralianRealPropertyContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}RealPropertyAUS")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAustralianRealPropertyElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAustralianRealPropertyContexts != null && sbrRPAustralianRealPropertyContexts.Count() > 0)
            {
                sbrRPAustralianRealPropertyContext = sbrRPAustralianRealPropertyContexts.First();
        
                report.RPAustralianRealPropertyId = sbrRPAustralianRealPropertyContext.Id;
                report.RPAustralianRealPropertyIdentifier = sbrRPAustralianRealPropertyContext.EntityIdentifier;
                report.RPAustralianRealPropertyIdentifierScheme = sbrRPAustralianRealPropertyContext.EntityScheme;
        
                report.RPAustralianRealPropertyCount = sbrRPAustralianRealPropertyContexts.Count();
                report.RPAustralianRealPropertyExists = true;
                report.RPAustralianRealPropertyOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAustralianRealPropertyContext) + 1;
                report.RPAustralianRealPropertyLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAustralianRealPropertyContexts.Last()) + 1;
        

                if (report.RPAustralianRealPropertyIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAustralianRealPropertyIdentifierTFN = report.RPAustralianRealPropertyIdentifier;
    
                report.RPAustralianRealPropertyStartDate = ((SBRDurationPeriod)sbrRPAustralianRealPropertyContext.Period).DurationStartAsDateObject;
                report.RPAustralianRealPropertyEndDate = ((SBRDurationPeriod)sbrRPAustralianRealPropertyContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAustralianRealPropertyElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAustralianRealPropertyContexts.Count(); i++)
                {
                    sbrRPAustralianRealPropertyContext = sbrRPAustralianRealPropertyContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAustralianRealPropertyContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAustralianRealPropertyElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAustralianRealPropertyElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAustralianRealPropertyElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAustralianRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS103 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAustralianRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS110 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAustralianRealPropertyContext
    
            #region sbrRPOverseasRealPropertyContext
            IEnumerable<SBRContext> sbrRPOverseasRealPropertyContexts = null;
            SBRContext sbrRPOverseasRealPropertyContext = null;
    
            sbrRPOverseasRealPropertyContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}OverseasRealProperty")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOverseasRealPropertyElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOverseasRealPropertyContexts != null && sbrRPOverseasRealPropertyContexts.Count() > 0)
            {
                sbrRPOverseasRealPropertyContext = sbrRPOverseasRealPropertyContexts.First();
        
                report.RPOverseasRealPropertyId = sbrRPOverseasRealPropertyContext.Id;
                report.RPOverseasRealPropertyIdentifier = sbrRPOverseasRealPropertyContext.EntityIdentifier;
                report.RPOverseasRealPropertyIdentifierScheme = sbrRPOverseasRealPropertyContext.EntityScheme;
        
                report.RPOverseasRealPropertyCount = sbrRPOverseasRealPropertyContexts.Count();
                report.RPOverseasRealPropertyExists = true;
                report.RPOverseasRealPropertyOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOverseasRealPropertyContext) + 1;
                report.RPOverseasRealPropertyLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOverseasRealPropertyContexts.Last()) + 1;
        

                if (report.RPOverseasRealPropertyIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOverseasRealPropertyIdentifierTFN = report.RPOverseasRealPropertyIdentifier;
    
                report.RPOverseasRealPropertyStartDate = ((SBRDurationPeriod)sbrRPOverseasRealPropertyContext.Period).DurationStartAsDateObject;
                report.RPOverseasRealPropertyEndDate = ((SBRDurationPeriod)sbrRPOverseasRealPropertyContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPOverseasRealPropertyElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOverseasRealPropertyContexts.Count(); i++)
                {
                    sbrRPOverseasRealPropertyContext = sbrRPOverseasRealPropertyContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOverseasRealPropertyContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOverseasRealPropertyElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOverseasRealPropertyElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOverseasRealPropertyElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOverseasRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS111 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOverseasRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS112 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOverseasRealPropertyContext
    
            #region sbrRPCapitalGainsTrustDistributionsContext
            IEnumerable<SBRContext> sbrRPCapitalGainsTrustDistributionsContexts = null;
            SBRContext sbrRPCapitalGainsTrustDistributionsContext = null;
    
            sbrRPCapitalGainsTrustDistributionsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}CapitalGainsTrustDistributions")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCapitalGainsTrustDistributionsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCapitalGainsTrustDistributionsContexts != null && sbrRPCapitalGainsTrustDistributionsContexts.Count() > 0)
            {
                sbrRPCapitalGainsTrustDistributionsContext = sbrRPCapitalGainsTrustDistributionsContexts.First();
        
                report.RPCapitalGainsTrustDistributionsId = sbrRPCapitalGainsTrustDistributionsContext.Id;
                report.RPCapitalGainsTrustDistributionsIdentifier = sbrRPCapitalGainsTrustDistributionsContext.EntityIdentifier;
                report.RPCapitalGainsTrustDistributionsIdentifierScheme = sbrRPCapitalGainsTrustDistributionsContext.EntityScheme;
        
                report.RPCapitalGainsTrustDistributionsCount = sbrRPCapitalGainsTrustDistributionsContexts.Count();
                report.RPCapitalGainsTrustDistributionsExists = true;
                report.RPCapitalGainsTrustDistributionsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCapitalGainsTrustDistributionsContext) + 1;
                report.RPCapitalGainsTrustDistributionsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCapitalGainsTrustDistributionsContexts.Last()) + 1;
        

                if (report.RPCapitalGainsTrustDistributionsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCapitalGainsTrustDistributionsIdentifierTFN = report.RPCapitalGainsTrustDistributionsIdentifier;
    
                report.RPCapitalGainsTrustDistributionsStartDate = ((SBRDurationPeriod)sbrRPCapitalGainsTrustDistributionsContext.Period).DurationStartAsDateObject;
                report.RPCapitalGainsTrustDistributionsEndDate = ((SBRDurationPeriod)sbrRPCapitalGainsTrustDistributionsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPCapitalGainsTrustDistributionsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCapitalGainsTrustDistributionsContexts.Count(); i++)
                {
                    sbrRPCapitalGainsTrustDistributionsContext = sbrRPCapitalGainsTrustDistributionsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCapitalGainsTrustDistributionsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCapitalGainsTrustDistributionsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCapitalGainsTrustDistributionsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCapitalGainsTrustDistributionsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCapitalGainsTrustDistributionsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS113 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCapitalGainsTrustDistributionsContext
    
            #region sbrRPCollectablesContext
            IEnumerable<SBRContext> sbrRPCollectablesContexts = null;
            SBRContext sbrRPCollectablesContext = null;
    
            sbrRPCollectablesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Collectables")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCollectablesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCollectablesContexts != null && sbrRPCollectablesContexts.Count() > 0)
            {
                sbrRPCollectablesContext = sbrRPCollectablesContexts.First();
        
                report.RPCollectablesId = sbrRPCollectablesContext.Id;
                report.RPCollectablesIdentifier = sbrRPCollectablesContext.EntityIdentifier;
                report.RPCollectablesIdentifierScheme = sbrRPCollectablesContext.EntityScheme;
        
                report.RPCollectablesCount = sbrRPCollectablesContexts.Count();
                report.RPCollectablesExists = true;
                report.RPCollectablesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCollectablesContext) + 1;
                report.RPCollectablesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCollectablesContexts.Last()) + 1;
        

                if (report.RPCollectablesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCollectablesIdentifierTFN = report.RPCollectablesIdentifier;
    
                report.RPCollectablesStartDate = ((SBRDurationPeriod)sbrRPCollectablesContext.Period).DurationStartAsDateObject;
                report.RPCollectablesEndDate = ((SBRDurationPeriod)sbrRPCollectablesContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPCollectablesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCollectablesContexts.Count(); i++)
                {
                    sbrRPCollectablesContext = sbrRPCollectablesContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCollectablesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCollectablesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCollectablesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCollectablesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCollectablesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS114 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPCollectablesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS115 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCollectablesContext
    
            #region sbrRPOtherCapitalAssetsContext
            IEnumerable<SBRContext> sbrRPOtherCapitalAssetsContexts = null;
            SBRContext sbrRPOtherCapitalAssetsContext = null;
    
            sbrRPOtherCapitalAssetsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}CGTOther")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOtherCapitalAssetsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOtherCapitalAssetsContexts != null && sbrRPOtherCapitalAssetsContexts.Count() > 0)
            {
                sbrRPOtherCapitalAssetsContext = sbrRPOtherCapitalAssetsContexts.First();
        
                report.RPOtherCapitalAssetsId = sbrRPOtherCapitalAssetsContext.Id;
                report.RPOtherCapitalAssetsIdentifier = sbrRPOtherCapitalAssetsContext.EntityIdentifier;
                report.RPOtherCapitalAssetsIdentifierScheme = sbrRPOtherCapitalAssetsContext.EntityScheme;
        
                report.RPOtherCapitalAssetsCount = sbrRPOtherCapitalAssetsContexts.Count();
                report.RPOtherCapitalAssetsExists = true;
                report.RPOtherCapitalAssetsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOtherCapitalAssetsContext) + 1;
                report.RPOtherCapitalAssetsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOtherCapitalAssetsContexts.Last()) + 1;
        

                if (report.RPOtherCapitalAssetsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOtherCapitalAssetsIdentifierTFN = report.RPOtherCapitalAssetsIdentifier;
    
                report.RPOtherCapitalAssetsStartDate = ((SBRDurationPeriod)sbrRPOtherCapitalAssetsContext.Period).DurationStartAsDateObject;
                report.RPOtherCapitalAssetsEndDate = ((SBRDurationPeriod)sbrRPOtherCapitalAssetsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPOtherCapitalAssetsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOtherCapitalAssetsContexts.Count(); i++)
                {
                    sbrRPOtherCapitalAssetsContext = sbrRPOtherCapitalAssetsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOtherCapitalAssetsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOtherCapitalAssetsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOtherCapitalAssetsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOtherCapitalAssetsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOtherCapitalAssetsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS116 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOtherCapitalAssetsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS117 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOtherCapitalAssetsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}CapitalGainsTax.DeferredRelief.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS163 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOtherCapitalAssetsContext
    
            #region sbrRPCollectablesUnappliedInstantContext
            IEnumerable<SBRContext> sbrRPCollectablesUnappliedInstantContexts = null;
            SBRContext sbrRPCollectablesUnappliedInstantContext = null;
    
            sbrRPCollectablesUnappliedInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Collectables"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AccountingTreatmentMethodDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Unapplied")) ))));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCollectablesUnappliedInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCollectablesUnappliedInstantContexts != null && sbrRPCollectablesUnappliedInstantContexts.Count() > 0)
            {
                sbrRPCollectablesUnappliedInstantContext = sbrRPCollectablesUnappliedInstantContexts.First();
        
                report.RPCollectablesUnappliedInstantId = sbrRPCollectablesUnappliedInstantContext.Id;
                report.RPCollectablesUnappliedInstantIdentifier = sbrRPCollectablesUnappliedInstantContext.EntityIdentifier;
                report.RPCollectablesUnappliedInstantIdentifierScheme = sbrRPCollectablesUnappliedInstantContext.EntityScheme;
        
                report.RPCollectablesUnappliedInstantCount = sbrRPCollectablesUnappliedInstantContexts.Count();
                report.RPCollectablesUnappliedInstantExists = true;
                report.RPCollectablesUnappliedInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCollectablesUnappliedInstantContext) + 1;
                report.RPCollectablesUnappliedInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCollectablesUnappliedInstantContexts.Last()) + 1;
        

                if (report.RPCollectablesUnappliedInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCollectablesUnappliedInstantIdentifierTFN = report.RPCollectablesUnappliedInstantIdentifier;
    
                report.RPCollectablesUnappliedInstantInstantDate = ((SBRInstantPeriod)sbrRPCollectablesUnappliedInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPCollectablesUnappliedInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCollectablesUnappliedInstantContexts.Count(); i++)
                {
                    sbrRPCollectablesUnappliedInstantContext = sbrRPCollectablesUnappliedInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCollectablesUnappliedInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCollectablesUnappliedInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCollectablesUnappliedInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCollectablesUnappliedInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCollectablesUnappliedInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS124 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCollectablesUnappliedInstantContext
    
            #region sbrRPCGTOtherUnappliedInstantContext
            IEnumerable<SBRContext> sbrRPCGTOtherUnappliedInstantContexts = null;
            SBRContext sbrRPCGTOtherUnappliedInstantContext = null;
    
            sbrRPCGTOtherUnappliedInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}CGTOther"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AccountingTreatmentMethodDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Unapplied")) ))));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCGTOtherUnappliedInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCGTOtherUnappliedInstantContexts != null && sbrRPCGTOtherUnappliedInstantContexts.Count() > 0)
            {
                sbrRPCGTOtherUnappliedInstantContext = sbrRPCGTOtherUnappliedInstantContexts.First();
        
                report.RPCGTOtherUnappliedInstantId = sbrRPCGTOtherUnappliedInstantContext.Id;
                report.RPCGTOtherUnappliedInstantIdentifier = sbrRPCGTOtherUnappliedInstantContext.EntityIdentifier;
                report.RPCGTOtherUnappliedInstantIdentifierScheme = sbrRPCGTOtherUnappliedInstantContext.EntityScheme;
        
                report.RPCGTOtherUnappliedInstantCount = sbrRPCGTOtherUnappliedInstantContexts.Count();
                report.RPCGTOtherUnappliedInstantExists = true;
                report.RPCGTOtherUnappliedInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCGTOtherUnappliedInstantContext) + 1;
                report.RPCGTOtherUnappliedInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCGTOtherUnappliedInstantContexts.Last()) + 1;
        

                if (report.RPCGTOtherUnappliedInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCGTOtherUnappliedInstantIdentifierTFN = report.RPCGTOtherUnappliedInstantIdentifier;
    
                report.RPCGTOtherUnappliedInstantInstantDate = ((SBRInstantPeriod)sbrRPCGTOtherUnappliedInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPCGTOtherUnappliedInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCGTOtherUnappliedInstantContexts.Count(); i++)
                {
                    sbrRPCGTOtherUnappliedInstantContext = sbrRPCGTOtherUnappliedInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCGTOtherUnappliedInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCGTOtherUnappliedInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCGTOtherUnappliedInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCGTOtherUnappliedInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCGTOtherUnappliedInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS125 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCGTOtherUnappliedInstantContext
    
            #region sbrRPTransfersToOtherClassesContext
            IEnumerable<SBRContext> sbrRPTransfersToOtherClassesContexts = null;
            SBRContext sbrRPTransfersToOtherClassesContext = null;
    
            sbrRPTransfersToOtherClassesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}TransfersToOtherClasses")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPTransfersToOtherClassesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPTransfersToOtherClassesContexts != null && sbrRPTransfersToOtherClassesContexts.Count() > 0)
            {
                sbrRPTransfersToOtherClassesContext = sbrRPTransfersToOtherClassesContexts.First();
        
                report.RPTransfersToOtherClassesId = sbrRPTransfersToOtherClassesContext.Id;
                report.RPTransfersToOtherClassesIdentifier = sbrRPTransfersToOtherClassesContext.EntityIdentifier;
                report.RPTransfersToOtherClassesIdentifierScheme = sbrRPTransfersToOtherClassesContext.EntityScheme;
        
                report.RPTransfersToOtherClassesCount = sbrRPTransfersToOtherClassesContexts.Count();
                report.RPTransfersToOtherClassesExists = true;
                report.RPTransfersToOtherClassesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTransfersToOtherClassesContext) + 1;
                report.RPTransfersToOtherClassesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTransfersToOtherClassesContexts.Last()) + 1;
        

                if (report.RPTransfersToOtherClassesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPTransfersToOtherClassesIdentifierTFN = report.RPTransfersToOtherClassesIdentifier;
    
                report.RPTransfersToOtherClassesStartDate = ((SBRDurationPeriod)sbrRPTransfersToOtherClassesContext.Period).DurationStartAsDateObject;
                report.RPTransfersToOtherClassesEndDate = ((SBRDurationPeriod)sbrRPTransfersToOtherClassesContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPTransfersToOtherClassesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPTransfersToOtherClassesContexts.Count(); i++)
                {
                    sbrRPTransfersToOtherClassesContext = sbrRPTransfersToOtherClassesContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPTransfersToOtherClassesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPTransfersToOtherClassesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPTransfersToOtherClassesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPTransfersToOtherClassesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPTransfersToOtherClassesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS154 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTransfersToOtherClassesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/cgts}Capital.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CGTS155 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPTransfersToOtherClassesContext
            return report;

        } // Of Consume Method
    } // Of Class
} // Of Namespace