using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationCGNFT
{

    public partial class CGNFT2018Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="CGNFT2018Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CGNFT2018Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }
				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public CGNFT2018 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CGNFT2018 report, bool isStandalone)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------

    
            #region VR.ATO.CGNFT.000001
    
            /*  VR.ATO.CGNFT.000001
            The Australian Business Number (ABN) has failed the ABN algorithm check.
    
            Legacy Rule Format:
            IF (ABNALGORITHM(RP:entity.identifier.ABN) = FALSE)
             RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FailsABNAlgorithm(^ABN))
    
            Data Elements:
    
            ^ABN = RP
            */
            assertion = FailsABNAlgorithm(report.RPIdentifierABN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    LongDescription = @"The Head company Australian business number has failed the ABN algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000001"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000001
    
            #region VR.ATO.CGNFT.000009
    
            /*  VR.ATO.CGNFT.000009
            The chosen Date of consolidation should not be completed when submitting a Notification of members joining and/or leaving the group request. If you are notifying of consolidation please submit a Members Joining/Leaving a consolidated group schedule with the Company tax return.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF ([cgnft6] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft6] = CGNFT:RP:pyde.02.17:OrganisationDetails.TaxConsolidation.Date

            Technical Business Rule Format:
            ^CGNFT6 <> NULL
    
            Data Elements:
    
            RP:^CGNFT6 = tns:OrganisationDetails.TaxConsolidation.Date
            */
            if (isStandalone)
            {
                assertion = (report.CGNFT6 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000009", Severity = ProcessMessageSeverity.Error,
                        Description = @"Date of consolidation date is not required",
                        LongDescription = @"The chosen Date of consolidation should not be completed when submitting a Notification of members joining and/or leaving the group request. If you are notifying the commissioner of a formation of a consolidated group, please submit a Members Joining/Leaving a consolidated group schedule with the Company tax return.",
                        Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidation.Date",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000009"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT6", Value = GetValueOrEmpty(report.CGNFT6) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000009
    
            #region VR.ATO.CGNFT.000010
    
            /*  VR.ATO.CGNFT.000010
            The Multiple entry consolidated group indicator must not be set when submitting a Members Joining/Leaving a consolidated group schedule with the Notification of members joining and/or leaving the group request. If you are notifying of consolidation please submit a Members Joining/Leaving a consolidated group schedule with the Company tax return.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF ([cgnft7] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft7] = CGNFT:RP:pyde.02.17:OrganisationDetails.TaxConsolidationMultipleEntryConsolidatedGroup.Indicator

            Technical Business Rule Format:
            ^CGNFT7 <> NULL
    
            Data Elements:
    
            RP:^CGNFT7 = tns:OrganisationDetails.TaxConsolidationMultipleEntryConsolidatedGroup.Indicator
            */
            if (isStandalone)
            {
                assertion = (report.CGNFT7 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000010", Severity = ProcessMessageSeverity.Error,
                        Description = @"Multiple entry consolidated group indicator is not required",
                        LongDescription = @"The Multiple entry consolidated group indicator must not be set when submitting a Members Joining/Leaving a consolidated group schedule with the Notification of members joining and/or leaving the group request. If you are notifying the commissioner of a formation of a consolidated group, please submit a Members Joining/Leaving a consolidated group schedule with the Company tax return.",
                        Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationMultipleEntryConsolidatedGroup.Indicator",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000010"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT7", Value = GetValueOrEmpty(report.CGNFT7) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000010


            #region Repeating CPPEContext
            if (report.CPPEContextCollection != null)
            {
                int itemIndexContext1 = 0;
                foreach (CGNFT2018.CPPEContext cppeContext in report.CPPEContextCollection)
                {
                    itemIndexContext1++;


                    #region VR.ATO.CGNFT.000018

                    /*  VR.ATO.CGNFT.000018
                    Scheme for ConsolidatedParty.PurchasingEntity must be set to http://www.ato.gov.au/tfn

                    Legacy Rule Format:
                    IF (CP.PE:entity.identifier.scheme <> “http://www.ato.gov.au/tfn”)
                        RETURN VALIDATION MESSAGE
                    ENDIF

                    Technical Business Rule Format:
                    ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'

                    Data Elements:

                    ^IdentifierScheme = CP:CP.PE
                    */
                    assertion = ((cppeContext.CPPEExists == true) && (cppeContext.CPPEIdentifierScheme != @"http://www.ato.gov.au/tfn"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000018",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Entity identifier scheme for ConsolidatedParty.PurchasingEntity must be ""http://www.ato.gov.au/tfn""",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cppeContext.CPPEOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000018" } },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000018


                    #region VR.ATO.CGNFT.000028

                    /*  VR.ATO.CGNFT.000028
                    A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.

                    Legacy Rule Format:
                    IF ([cgnft19] <> NULLORBLANK) AND ([cgnft19] CONTAINS SET("{","}","|","_","\","#"))
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

                    Technical Business Rule Format:
                    ContainsSet(^CGNFT19, '"{","}","|","_","\","#"')

                    Data Elements:

                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(cppeContext.CGNFT19, @"\{|\}|\||_|\\|#", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000028",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity legal name contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000028" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT19", Value = cppeContext.CGNFT19 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000028

                    #region VR.ATO.CGNFT.000032

                    /*  VR.ATO.CGNFT.000032
                    Non-individual name field only contains characters other than alphas and numerics, at least one alpha or numeric must be present.

                    Legacy Rule Format:
                    IF ([cgnft19] <> NULLORBLANK) AND ([cgnft19] DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

                    Technical Business Rule Format:
                    (^CGNFT19 <> BLANK) AND (NotContainsSet(^CGNFT19, '"a-z", "A-Z", "0-9"'))

                    Data Elements:

                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(cppeContext.CGNFT19) != true && !(IsMatch(cppeContext.CGNFT19, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000032",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity legal name must contain at least one alpha or numeric character",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000032" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT19", Value = cppeContext.CGNFT19 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000032
                    
                    //VR.ATO.CGNFT.000044 removed for CR2900 | INC000029748406
                 
                    #region VR.ATO.CGNFT.000054

                    /*  VR.ATO.CGNFT.000054
                    Name field cannot contain 'space hyphen space'.

                    Legacy Rule Format:
                    IF ([cgnft19] CONTAINS  " - " )
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

                    Technical Business Rule Format:
                    Contains(^CGNFT19, ' - ')

                    Data Elements:

                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (cppeContext.CGNFT19 == null ? false : cppeContext.CGNFT19.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000054",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity legal name cannot contain space hyphen space combinations",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000054" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT19", Value = cppeContext.CGNFT19 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000054

                    #region VR.ATO.CGNFT.000066

                    /*  VR.ATO.CGNFT.000066
                    Non-individual name field contains 'P/L' which is not a valid abbreviation. This abbreviation cannot be used.

                    Legacy Rule Format:
                    IF ([cgnft19] CONTAINS "P/L")
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

                    Technical Business Rule Format:
                    Contains(^CGNFT19, 'P/L')

                    Data Elements:

                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (cppeContext.CGNFT19 == null ? false : cppeContext.CGNFT19.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000066",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity legal name cannot contain ""P/L""",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000066" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT19", Value = cppeContext.CGNFT19 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000066

                    #region VR.ATO.CGNFT.000071

                    /*  VR.ATO.CGNFT.000071
                    Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.

                    Legacy Rule Format:
                    IF ([cgnft19] CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

                    Technical Business Rule Format:
                    (ContainsSet(^CGNFT19, '"--","''","  "'))

                    Data Elements:

                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(cppeContext.CGNFT19, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000071",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity legal name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000071" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT19", Value = cppeContext.CGNFT19 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000071

                    #region VR.ATO.CGNFT.000084

                    /*  VR.ATO.CGNFT.000084
                    The Australian Business Number (ABN) has failed the ABN algorithm check.

                    Legacy Rule Format:
                    IF ([cgnft20] <> NULLORBLANK) and (ABNALGORITHM([cgnft20]) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft20] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier

                    Technical Business Rule Format:
                    (^CGNFT20 <> BLANK) and (FailsABNAlgorithm(^CGNFT20))

                    Data Elements:

                    CP.PE:^CGNFT20 = tns:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(cppeContext.CGNFT20) != true && FailsABNAlgorithm(cppeContext.CGNFT20));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000084",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ABN",
                            LongDescription = @"The Purchasing entity Australian business number has failed the ABN algorithm check",
                            Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000084" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT20", Value = cppeContext.CGNFT20 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000084

                    #region VR.ATO.CGNFT.000094

                    /*  VR.ATO.CGNFT.000094
                    The ACN/ARBN/SMSF auditor number has failed the algorithm check.

                    Legacy Rule Format:
                    IF ([cgnft27] <> NULLORBLANK) AND (ARBNALGORITHM([cgnft27]) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    [cgnft27] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianRegisteredBodyNumber.Identifier

                    Technical Business Rule Format:
                    (^CGNFT27 <> BLANK) AND (FailsARBNAlgorithm(^CGNFT27))

                    Data Elements:

                    CP.PE:^CGNFT27 = tns:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(cppeContext.CGNFT27) != true && FailsACNAlgorithm(cppeContext.CGNFT27));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000094",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ARBN",
                            LongDescription = @"The Purchasing entity Australian registered body number has failed the ARBN algorithm check",
                            Location = "/xbrli:xbrl/tns:Identifiers.AustralianRegisteredBodyNumber.Identifier[@contextRef='" + cppeContext.CPPEId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000094" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT27", Value = cppeContext.CGNFT27 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000094

                    #region VR.ATO.CGNFT.000096

                    /*  VR.ATO.CGNFT.000096
                    The TFN of the Consolidated Party Purchasing Entity should be the same as the TFN of the Consolidated Party

                    Legacy Rule Format:
                    FOR EACH CP.PE
                    IF (CP.PE:entity.identifier.TFN <> ANY OCCURRENCE OF CP:entity.identifier.TFN)
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    Technical Business Rule Format:
                    (AnyOccurrence(^TFN, ^TFN <> ^OtherTFN))

                    Data Elements:

                    ^OtherTFN = CP:CP.PE

                    ^TFN = CP
                    */
                    assertion = report.CPContextCollection == null || !(report.CPContextCollection.Any(cpContext => cpContext.IdentifierTFN == cppeContext.CPPEIdentifierTFN));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000096",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"The Consolidated Party Purchasing Entity Tax file number should be the same as the associated Consolidated Party TFN",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cppeContext.CPPEOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000096" } },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000096

                    #region VR.ATO.CGNFT.000196

                    /*  VR.ATO.CGNFT.000196
                    Invalid context. Period start date must be the same as the Period end date for this form.

                    Legacy Rule Format:
                    WHERE PARENT RETURN DOES NOT EXIST
                    IF period.startDate <> period.endDate
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    Technical Business Rule Format:
                    ^StartDate <> ^EndDate

                    Data Elements:

                    ^StartDate = CP:CP.PE

                    ^EndDate = CP:CP.PE
                    */
                    if (isStandalone)
                    {
                        assertion = ((cppeContext.CPPEExists == true) && (cppeContext.CPPEStartDate != cppeContext.CPPEEndDate));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.CGNFT.000196",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Invalid context. Period start date must be the same as the Period end date for this form.",
                                Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cppeContext.CPPEOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000196" } },
                            };
                            response.Add(processMessage);
                        }
                    }
                    #endregion // VR.ATO.CGNFT.000196

                    #region VR.ATO.CGNFT.000197

                    /*  VR.ATO.CGNFT.000197
                    Invalid context. Period start date cannot be any day in the future.

                    Legacy Rule Format:
                    WHERE PARENT RETURN DOES NOT EXIST
                    IF period.startDate > DATE(TODAY)
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    Technical Business Rule Format:
                    ^StartDate > Today()

                    Data Elements:

                    ^StartDate = CP:CP.PE
                    */
                    if (isStandalone)
                    {
                        assertion = ((cppeContext.CPPEExists == true) && (cppeContext.CPPEStartDate > DateTime.Now.Date));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.CGNFT.000197",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Invalid context. Period start date cannot be any day in the future.",
                                Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cppeContext.CPPEOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000197" } },
                            };
                            response.Add(processMessage);
                        }
                    }
                    #endregion // VR.ATO.CGNFT.000197

                    #region VR.ATO.GEN.500030

                    /*  VR.ATO.GEN.500030
                    Invalid context. Period start date must be the same across all duration contexts.

                    Legacy Rule Format:
                    IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                        RETURN VALIDATION MESSAGE
                    ENDIF

                    Technical Business Rule Format:
                    ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate

                    Data Elements:

                    ^StartDate = RP

                    ^StartDate1 = CP:CP.PE
                    */
                    assertion = ((cppeContext.CPPEExists == true) && (cppeContext.CPPEStartDate != null && cppeContext.CPPEStartDate != report.RPStartDate));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500030",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030" } },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.500030
                }
            }
            #endregion Repeating CPPEContext
            
            #region VR.ATO.CGNFT.000020

            /*  VR.ATO.CGNFT.000020
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([cgnft4] <> NULLORBLANK) AND ([cgnft4] CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft4] = CGNFT:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            ContainsSet(^CGNFT4, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            RP:^CGNFT4 = tns:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CGNFT4, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000020", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Legal name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000020"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT4", Value = report.CGNFT4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000020

            #region VR.ATO.CGNFT.000023

            /*  VR.ATO.CGNFT.000023
            Declarer Position held of person making the declaration is mandatory
    
            Legacy Rule Format:
            IF (CONTEXT(RP.TrueAndCorrect) <> NULL AND RP.TrueAndCorrect:pyde.xx.xx:PersonNameDetails.Position.Text = NULLORBLANK) OR (CONTEXT(INT.TrueAndCorrect) <> NULL AND INT.TrueAndCorrect:pyde.xx.xx:PersonNameDetails.Position.Text = NULLORBLANK) 
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL AND ^CGNFT24 = BLANK) OR (^Context <> NULL AND ^CGNFT37 = BLANK)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT24 = tns:PersonNameDetails.Position.Text
    
            INT.TrueAndCorrect:^CGNFT37 = tns:PersonNameDetails.Position.Text
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
    
            INT.TrueAndCorrect:^Context = RP:RP.TrueAndCorrect
            */
            assertion = (report.RPTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT24) == true || report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT37) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Position held of person making the declaration must be provided",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "CGNFT24", Value = report.CGNFT24 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "CGNFT37", Value = report.CGNFT37 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000023

            #region VR.ATO.CGNFT.000029

            /*  VR.ATO.CGNFT.000029
            Non-individual name field only contains characters other than alphas and numerics, at least one alpha or numeric must be present.
    
            Legacy Rule Format:
            IF ([cgnft4] <> NULLORBLANK) AND ([cgnft4] DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft4] = CGNFT:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (^CGNFT4 <> BLANK) AND (NotContainsSet(^CGNFT4, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            RP:^CGNFT4 = tns:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT4) != true && !(IsMatch(report.CGNFT4, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Legal name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000029"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT4", Value = report.CGNFT4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000029


            //VR.ATO.CGNFT.000033 removed for CR2900 | INC000029748406
           
            #region VR.ATO.CGNFT.000038
    
            /*  VR.ATO.CGNFT.000038
            The Head company Tax file number is mandatory
    
            Legacy Rule Format:
            IF ([cgnft3] = NULLORBLANK) 
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft3] = CGNFT:RP:pyid.02.00:Identifiers.TaxFileNumber.Identifier

            Technical Business Rule Format:
            ^CGNFT3 = BLANK
    
            Data Elements:
    
            RP:^CGNFT3 = tns:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT3) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Tax file number must be provided",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxFileNumber.Identifier[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT3", Value = report.CGNFT3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000038
    
            #region VR.ATO.CGNFT.000043
    
            /*  VR.ATO.CGNFT.000043
            The Head of company Legal name is mandatory
    
            Legacy Rule Format:
            IF ([cgnft4] = NULLORBLANK) 
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft4] = CGNFT:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            ^CGNFT4 = BLANK
    
            Data Elements:
    
            RP:^CGNFT4 = tns:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT4) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000043", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Legal name must be provided",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000043"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT4", Value = report.CGNFT4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000043
    
            
    
            #region VR.ATO.CGNFT.000049
    
            /*  VR.ATO.CGNFT.000049
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF ([cgnft4] CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft4] = CGNFT:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            Contains(^CGNFT4, ' - ')
    
            Data Elements:
    
            RP:^CGNFT4 = tns:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CGNFT4 == null ? false : report.CGNFT4.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000049", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Legal name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000049"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT4", Value = report.CGNFT4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000049
    
            #region VR.ATO.CGNFT.000052
    
            /*  VR.ATO.CGNFT.000052
            The chosen date of consolidation must be completed when a Members joining/leaving consolidated group schedule is lodged with the income tax return of the head company
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF ([cgnft6] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft6] = CGNFT:RP:pyde.02.17:OrganisationDetails.TaxConsolidation.Date

            Technical Business Rule Format:
            ^CGNFT6 = NULL
    
            Data Elements:
    
            RP:^CGNFT6 = tns:OrganisationDetails.TaxConsolidation.Date
            */
            if (!isStandalone)
            {
                assertion = (report.CGNFT6 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000052", Severity = ProcessMessageSeverity.Error,
                        Description = @"The Date of consolidation is required when lodging the income tax return of the head company",
                        LongDescription = @"The Date of consolidation is required when a Members joining/leaving a consolidated group schedule is lodged with the income tax return of the head company",
                        Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidation.Date",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000052"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT6", Value = GetValueOrEmpty(report.CGNFT6) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000052
    
            
    
            #region VR.ATO.CGNFT.000055
    
            /*  VR.ATO.CGNFT.000055
            The date the subsidiary member joined the group must not be earlier than the chosen date of consolidation.
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF ANY OCCURRENCE OF (([cgnft14] <> NULL) AND ([cgnft14] < [cgnft6]))
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft6] = CGNFT:RP:pyde.02.17:OrganisationDetails.TaxConsolidation.Date
            [cgnft14] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date

            Technical Business Rule Format:
            AnyOccurrence(^CGNFT14, (^CGNFT14 <> NULL) AND (^CGNFT14 < ^CGNFT6))
    
            Data Elements:
    
            CP:^CGNFT14 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
    
            RP:^CGNFT6 = tns:OrganisationDetails.TaxConsolidation.Date
            */
            if (!isStandalone)
            {
                assertion = (report.CPContextCollection == null ? false : report.CPContextCollection.Any(CGNFT14Repeat => CGNFT14Repeat.CGNFT14 != null && CGNFT14Repeat.CGNFT14.GetValueOrDefault() < report.CGNFT6.GetValueOrDefault()));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000055", Severity = ProcessMessageSeverity.Error,
                        Description = @"The Subsidiary member entry date must not be earlier than the chosen Date of consolidation",
                        Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000055"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT14", Value = "CGNFT14" });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT6", Value = "CGNFT6" });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000055
    
            #region VR.ATO.CGNFT.000059
    
            /*  VR.ATO.CGNFT.000059
            The Multiple Entry Consolidated group indicator must be set when a Members joining/leaving a consolidated group schedule is lodged with the income tax return of the head company.
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF ([cgnft7] = NULL)
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft7] = CGNFT:RP:pyde.02.17:OrganisationDetails.TaxConsolidationMultipleEntryConsolidatedGroup.Indicator

            Technical Business Rule Format:
            ^CGNFT7 = NULL
    
            Data Elements:
    
            RP:^CGNFT7 = tns:OrganisationDetails.TaxConsolidationMultipleEntryConsolidatedGroup.Indicator
            */
            if (!isStandalone)
            {
                assertion = (report.CGNFT7 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000059", Severity = ProcessMessageSeverity.Error,
                        Description = @"Multiple entry consolidated group indicator must be set",
                        LongDescription = @"The Multiple Entry Consolidated group indicator must be set when a Members joining/leaving a consolidated group schedule is lodged with the income tax return of the head company.",
                        Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationMultipleEntryConsolidatedGroup.Indicator",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000059"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CGNFT7", Value = GetValueOrEmpty(report.CGNFT7) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000059
    
            #region VR.ATO.CGNFT.000060
    
            /*  VR.ATO.CGNFT.000060
            Non-individual name field contains 'P/L' which is not a valid abbreviation. This abbreviation cannot be used.
    
            Legacy Rule Format:
            IF ([cgnft4] CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft4] = CGNFT:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            Contains(^CGNFT4, 'P/L')
    
            Data Elements:
    
            RP:^CGNFT4 = tns:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.CGNFT4 == null ? false : report.CGNFT4.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000060", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Legal name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000060"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT4", Value = report.CGNFT4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000060
    
           
    
            #region VR.ATO.CGNFT.000067
    
            /*  VR.ATO.CGNFT.000067
            Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            IF ([cgnft4] CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [cgnft4] = CGNFT:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text

            Technical Business Rule Format:
            (ContainsSet(^CGNFT4, '"--","''","  "'))
    
            Data Elements:
    
            RP:^CGNFT4 = tns:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.CGNFT4, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000067", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Legal name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000067"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT4", Value = report.CGNFT4 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000067
    
            #region VR.ATO.CGNFT.000068
    
            /*  VR.ATO.CGNFT.000068
            Invalid context. The minimum number of Consolidated Party contexts is 1.
    
            Legacy Rule Format:
            IF COUNT(CP) = 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) = 0
    
            Data Elements:
    
            ^Context = CP
            */
            assertion = (Count(report.CPContextCollection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000068", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The minimum number of Consolidated Party contexts is 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.CPContextCollection == null ? 0 : (report.CPContextCollection.Count() == 0 ? 0 : report.CPContextCollection.Last().LastOccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000068"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000068
    
            
    
            
    
            //#region VR.ATO.CGNFT.000087
    
            ///*  VR.ATO.CGNFT.000087
            //Declaration Signatory Identifier text must be present
    
            //Legacy Rule Format:
            //IF (pyin.xx.xx:Declaration.SignatoryIdentifier.Text = NULLORBLANK) 
            //     RETURN VALIDATION MESSAGE
            //ENDIF

            //Technical Business Rule Format:
            //^CGNFT31 = BLANK
    
            //Data Elements:
    
            //RP.TrueAndCorrect:^CGNFT31 = tns:Declaration.SignatoryIdentifier.Text
            //*/
            //assertion = (string.IsNullOrWhiteSpace(report.CGNFT31) == true);
            //if (assertion)
            //{
            //    processMessage = new ProcessMessageDocument()
            //    {
            //        Code = "CMN.ATO.GEN.410114", Severity = ProcessMessageSeverity.Error,
            //        Description = @"Declaration Signatory Identifier text must be present",
            //        Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
            //        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000087"} },
            //    };
            //    processMessage.Parameters.Add(new ProcessMessageParameter
            //        { Name = "CGNFT31", Value = report.CGNFT31 });
    
            //    response.Add(processMessage);
            //}
            //#endregion // VR.ATO.CGNFT.000087
    
            
    
            
    
            #region VR.ATO.CGNFT.000098
    
            /*  VR.ATO.CGNFT.000098
            The ABN of the Declarer Reporting Party should be the same as the ABN of the Reporting Party
    
            Legacy Rule Format:
            IF (CONTEXT(RP.TrueAndCorrect)<>NULL) AND (RP.TrueAndCorrect:entity.identifier.ABN <> RP:entity.identifier.ABN)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^ABN <> ^ABN)
    
            Data Elements:
    
            ^Context = RP:RP.TrueAndCorrect
    
            ^ABN = RP:RP.TrueAndCorrect
    
            ^ABN = RP
            */
            assertion = (report.RPTrueAndCorrectExists != false && report.RPTrueAndCorrectIdentifierABN != report.RPIdentifierABN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000098", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Declarer Reporting Party Australian Business number should be the same as the Reporting Party ABN",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000098"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000098
    
            #region VR.ATO.CGNFT.000102
    
            /*  VR.ATO.CGNFT.000102
            The TFN has failed the algorithm check.
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier <> NULLORBLANK) AND (TFNALGORITHM (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier) = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT3 <> BLANK) AND (FailsTFNAlgorithm(^CGNFT3))
    
            Data Elements:
    
            RP:^CGNFT3 = tns:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT3) != true && FailsTFNAlgorithm(report.CGNFT3));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000102", Severity = ProcessMessageSeverity.Error,
                    Description = @"Head company Tax file number must be a valid TFN",
                    LongDescription = @"The Head company Tax file number has failed the Tax File Number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxFileNumber.Identifier[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000102"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT3", Value = report.CGNFT3 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000102
    
            #region VR.ATO.CGNFT.000103
    
            /*  VR.ATO.CGNFT.000103
            Invalid context. The number of Consolidated Party contexts must not exceed 45.
    
            Legacy Rule Format:
            IF COUNT(CP) > 45
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 45
    
            Data Elements:
    
            ^Context = CP
            */
            assertion = (Count(report.CPContextCollection) > 45);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000103", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Consolidated Party contexts must not exceed 45.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.CPContextCollection == null ? 0 : (report.CPContextCollection.Count() == 0 ? 0 : report.CPContextCollection.Last().LastOccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000103"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000103
    
            #region VR.ATO.CGNFT.000104
    
            /*  VR.ATO.CGNFT.000104
            Declaration Statement Accepted Indicator must be present
    
            Legacy Rule Format:
            IF (INT.TrueAndCorrect:pyin.xx.xx:Declaration.StatementAccepted.Indicator = NULL) AND (RP.TrueAndCorrect:pyin.xx.xx:Declaration.StatementAccepted.Indicator = NULL)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT42 = NULL) AND (^CGNFT30 = NULL)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT30 = tns:Declaration.StatementAccepted.Indicator
    
            INT.TrueAndCorrect:^CGNFT42 = tns:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.CGNFT42 == null && report.CGNFT30 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410078", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be present",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000104"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT42", Value = GetValueOrEmpty(report.CGNFT42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT30", Value = GetValueOrEmpty(report.CGNFT30) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000104
    
            #region VR.ATO.CGNFT.000191
    
            /*  VR.ATO.CGNFT.000191
            Declarer Contact Name has a maximum field length of 100
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 100
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CGNFT23) > 100
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT23 = tns:PersonUnstructuredName.FullName.Text
            */
            assertion = (Length(report.CGNFT23) > 100);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name must not exceed 100 characters",
                    LongDescription = @"Declarer Contact Name has a maximum field length of 100",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT23", Value = report.CGNFT23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000191
    
            #region VR.ATO.CGNFT.000191
    
            /*  VR.ATO.CGNFT.000191
            Declarer Contact Name has a maximum field length of 100
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 100
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CGNFT36) > 100
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT36 = tns:PersonUnstructuredName.FullName.Text
            */
            assertion = (Length(report.CGNFT36) > 100);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name must not exceed 100 characters",
                    LongDescription = @"Declarer Contact Name has a maximum field length of 100",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT36", Value = report.CGNFT36 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGNFT.000191
    
            
    
            #region VR.ATO.CGNFT.000196
    
            /*  VR.ATO.CGNFT.000196
            Invalid context. Period start date must be the same as the Period end date for this form.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF period.startDate <> period.endDate
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate <> ^EndDate
    
            Data Elements:
    
            ^StartDate = INT.TrueAndCorrect
    
            ^EndDate = INT.TrueAndCorrect
            */
            if (isStandalone)
            {
                assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectStartDate != report.INTTrueAndCorrectEndDate));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000196", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. Period start date must be the same as the Period end date for this form.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000196"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000196
    
            #region VR.ATO.CGNFT.000196
    
            /*  VR.ATO.CGNFT.000196
            Invalid context. Period start date must be the same as the Period end date for this form.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF period.startDate <> period.endDate
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate <> ^EndDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^EndDate = RP
            */
            if (isStandalone)
            {
                assertion = ((report.RPExists == true) && (report.RPStartDate != report.RPEndDate));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000196", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. Period start date must be the same as the Period end date for this form.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000196"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000196
    
            #region VR.ATO.CGNFT.000196
    
            /*  VR.ATO.CGNFT.000196
            Invalid context. Period start date must be the same as the Period end date for this form.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF period.startDate <> period.endDate
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate <> ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.TrueAndCorrect
    
            ^EndDate = RP:RP.TrueAndCorrect
            */
            if (isStandalone)
            {
                assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectStartDate != report.RPTrueAndCorrectEndDate));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000196", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. Period start date must be the same as the Period end date for this form.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000196"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000196
    
            
    
            #region VR.ATO.CGNFT.000197
    
            /*  VR.ATO.CGNFT.000197
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = INT.TrueAndCorrect
            */
            if (isStandalone)
            {
                assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectStartDate > DateTime.Now.Date));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000197", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. Period start date cannot be any day in the future.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000197"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000197
    
            #region VR.ATO.CGNFT.000197
    
            /*  VR.ATO.CGNFT.000197
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = RP
            */
            if (isStandalone)
            {
                assertion = ((report.RPExists == true) && (report.RPStartDate > DateTime.Now.Date));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000197", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. Period start date cannot be any day in the future.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000197"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000197
    
            #region VR.ATO.CGNFT.000197
    
            /*  VR.ATO.CGNFT.000197
            Invalid context. Period start date cannot be any day in the future.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF period.startDate > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate > Today()
    
            Data Elements:
    
            ^StartDate = RP:RP.TrueAndCorrect
            */
            if (isStandalone)
            {
                assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectStartDate > DateTime.Now.Date));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.CGNFT.000197", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. Period start date cannot be any day in the future.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000197"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.CGNFT.000197
    
            #region VR.ATO.GEN.000209
    
            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000209
    
            #region VR.ATO.GEN.410128
    
            /*  VR.ATO.GEN.410128
            Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.StatementAccepted.Indicator <> NULLORBLANK) AND (pyin.xx.xx:Declaration.StatementAccepted.Indicator = FALSE)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT30 <> BLANK) AND (^CGNFT30 = FALSE)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT30 = tns:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.CGNFT30 != null && report.CGNFT30 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410128", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410128"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT30", Value = GetValueOrEmpty(report.CGNFT30) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410128
    
            #region VR.ATO.GEN.410128
    
            /*  VR.ATO.GEN.410128
            Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.StatementAccepted.Indicator <> NULLORBLANK) AND (pyin.xx.xx:Declaration.StatementAccepted.Indicator = FALSE)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT42 <> BLANK) AND (^CGNFT42 = FALSE)
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT42 = tns:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.CGNFT42 != null && report.CGNFT42 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410128", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/xbrli:xbrl/tns:Declaration.StatementAccepted.Indicator[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410128"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT42", Value = GetValueOrEmpty(report.CGNFT42) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410128
    
            #region VR.ATO.GEN.428247
    
            /*  VR.ATO.GEN.428247
            Your Tax Agent number has failed the Tax Agent registration algorithm check. Input the correct Tax Agent registration number.
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT28 <> BLANK) AND (FailsTANAlgorithm(^CGNFT28))
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT28 = tns:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT28) != true && FailsTANAlgorithm(report.CGNFT28));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT28", Value = report.CGNFT28 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428247
    
            #region VR.ATO.GEN.428247
    
            /*  VR.ATO.GEN.428247
            Your Tax Agent number has failed the Tax Agent registration algorithm check. Input the correct Tax Agent registration number.
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT40 <> BLANK) AND (FailsTANAlgorithm(^CGNFT40))
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT40 = tns:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT40) != true && FailsTANAlgorithm(report.CGNFT40));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT40", Value = report.CGNFT40 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428247
    
            #region VR.ATO.GEN.438070
    
            /*  VR.ATO.GEN.438070
            If an Intermediary is present, Intermediary Declarer contact name must be provided.
    
            Legacy Rule Format:
            IF ((CONTEXT(INT.TrueAndCorrect)<>NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^Context <> NULL) AND (^CGNFT36 = BLANK))
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT36 = tns:PersonUnstructuredName.FullName.Text
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT36) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438070", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer contact name must be provided",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer contact name must be provided.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438070"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT36", Value = report.CGNFT36 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438070
    
            #region VR.ATO.GEN.438071
    
            /*  VR.ATO.GEN.438071
            If an Intermediary is not present, Reporting Party Declarer contact name must be provided.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)=NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context=NULL) AND (^CGNFT23 = BLANK)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT23 = tns:PersonUnstructuredName.FullName.Text
    
            RP.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && string.IsNullOrWhiteSpace(report.CGNFT23) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438071", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer contact name must be provided",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer contact name must be provided.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438071"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT23", Value = report.CGNFT23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438071
    
            #region VR.ATO.GEN.438072
    
            /*  VR.ATO.GEN.438072
            If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) <> NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT39 = BLANK) AND (^CGNFT38 <> BLANK)
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT38 = tns:ElectronicContact.Telephone.Area.Code
    
            INT.TrueAndCorrect:^CGNFT39 = tns:ElectronicContact.Telephone.Minimal.Number
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT39) == true && string.IsNullOrWhiteSpace(report.CGNFT38) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438072", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT39", Value = report.CGNFT39 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT38", Value = report.CGNFT38 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438072
    
            #region VR.ATO.GEN.438073
    
            /*  VR.ATO.GEN.438073
            If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) = NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code <> NULLORBLANK) OR (CONTEXT(INT.TrueAndCorrect) <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context = NULL) AND (^CGNFT26 = BLANK) AND (^CGNFT25 <> BLANK) OR (^Context <> NULL)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT25 = tns:ElectronicContact.Telephone.Area.Code
    
            RP.TrueAndCorrect:^CGNFT26 = tns:ElectronicContact.Telephone.Minimal.Number
    
            RP.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && string.IsNullOrWhiteSpace(report.CGNFT26) == true && string.IsNullOrWhiteSpace(report.CGNFT25) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438073", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438073"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT26", Value = report.CGNFT26 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT25", Value = report.CGNFT25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438073
    
            #region VR.ATO.GEN.438074
    
            /*  VR.ATO.GEN.438074
            If a telephone number is supplied and Intermediary is present, Intermediary area code is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) <> NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number <> NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT39 <> NULL) AND (^CGNFT38 = BLANK)
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT38 = tns:ElectronicContact.Telephone.Area.Code
    
            INT.TrueAndCorrect:^CGNFT39 = tns:ElectronicContact.Telephone.Minimal.Number
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.CGNFT39 != null && string.IsNullOrWhiteSpace(report.CGNFT38) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438074", Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is present, Intermediary area code is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT39", Value = report.CGNFT39 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT38", Value = report.CGNFT38 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438074
    
            #region VR.ATO.GEN.438075
    
            /*  VR.ATO.GEN.438075
            If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) = NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number <> NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Area.Code = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context = NULL) AND (^CGNFT26 <> NULL) AND (^CGNFT25 = BLANK)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT25 = tns:ElectronicContact.Telephone.Area.Code
    
            RP.TrueAndCorrect:^CGNFT26 = tns:ElectronicContact.Telephone.Minimal.Number
    
            RP.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && report.CGNFT26 != null && string.IsNullOrWhiteSpace(report.CGNFT25) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438075", Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Area.Code[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT26", Value = report.CGNFT26 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT25", Value = report.CGNFT25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438075
    
            #region VR.ATO.GEN.438076
    
            /*  VR.ATO.GEN.438076
            If an Intermediary is present, Intermediary Declarer Telephone number must be provided.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)<>NULL) AND (INT.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT39 = BLANK)
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT39 = tns:ElectronicContact.Telephone.Minimal.Number
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT39) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438076", Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact telephone details must be provided",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer Telephone number must be provided.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT39", Value = report.CGNFT39 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438076
    
            #region VR.ATO.GEN.438077
    
            /*  VR.ATO.GEN.438077
            Declarer Telephone number is mandatory
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)=NULL) AND (RP.TrueAndCorrect:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context=NULL) AND (^CGNFT26 = BLANK)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT26 = tns:ElectronicContact.Telephone.Minimal.Number
    
            RP.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && string.IsNullOrWhiteSpace(report.CGNFT26) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438077", Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact telephone details must be provided",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer Telephone number must be provided.",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438077"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT26", Value = report.CGNFT26 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438077
    
            #region VR.ATO.GEN.438078
    
            /*  VR.ATO.GEN.438078
            If an Intermediary is present, Intermediary Declarer Signature Date must be present.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)<>NULL) AND (INT.TrueAndCorrect:pyin.xx.xx:Declaration.Signature.Date = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT41 = NULL)
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT41 = tns:Declaration.Signature.Date
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && report.CGNFT41 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438078", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signature Date is required",
                    LongDescription = @"If an Intermediary is present, Intermediary Declarer Signature Date must be present.",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT41", Value = GetValueOrEmpty(report.CGNFT41) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438078
    
            #region VR.ATO.GEN.438079
    
            /*  VR.ATO.GEN.438079
            If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)=NULL) AND (RP.TrueAndCorrect:pyin.xx.xx:Declaration.Signature.Date = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context=NULL) AND (^CGNFT29 = NULL)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT29 = tns:Declaration.Signature.Date
    
            RP.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && report.CGNFT29 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438079", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signature Date is required",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438079"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT29", Value = GetValueOrEmpty(report.CGNFT29) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438079
    
            #region VR.ATO.GEN.500030
    
            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^StartDate1 = INT.TrueAndCorrect
            */
            assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectStartDate != null && report.INTTrueAndCorrectStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500030
    
            #region VR.ATO.GEN.500030
    
            /*  VR.ATO.GEN.500030
            Invalid context. Period start date must be the same across all duration contexts.
    
            Legacy Rule Format:
            IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^StartDate1 = RP:RP.TrueAndCorrect
            */
            assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectStartDate != null && report.RPTrueAndCorrectStartDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500030
    
            #region VR.ATO.GEN.500037
    
            /*  VR.ATO.GEN.500037
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CGNFT23, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT23 = tns:PersonUnstructuredName.FullName.Text
            */
            assertion = IsMatch(report.CGNFT23, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500037", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500037"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT23", Value = report.CGNFT23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500037
    
            #region VR.ATO.GEN.500037
    
            /*  VR.ATO.GEN.500037
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET("{","}","|","_","\","#"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^CGNFT36, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT36 = tns:PersonUnstructuredName.FullName.Text
            */
            assertion = IsMatch(report.CGNFT36, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500037", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500037"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT36", Value = report.CGNFT36 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500037
    
            #region VR.ATO.GEN.500046
    
            /*  VR.ATO.GEN.500046
            Invalid context. The number of RP.TrueAndCorrect contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.TrueAndCorrect) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TrueAndCorrect
            */
            assertion = (Count(report.RPTrueAndCorrectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500046", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.TrueAndCorrect contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500046"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500046
    
            #region VR.ATO.GEN.500047
    
            /*  VR.ATO.GEN.500047
            Invalid context. The number of Intermediary.TrueAndCorrect contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(INT.TrueAndCorrect) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (Count(report.INTTrueAndCorrectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500047", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Intermediary.TrueAndCorrect contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500047"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500047
    
            #region VR.ATO.GEN.500049
    
            /*  VR.ATO.GEN.500049
            Scheme for Reporting Party must be set to http://www.abr.gov.au/abn
    
            Legacy Rule Format:
            IF (RP:entity.identifier.scheme <> "http://www.abr.gov.au/abn")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.abr.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = RP
            */
            assertion = ((report.RPExists == true) && (report.RPIdentifierScheme != @"http://www.abr.gov.au/abn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500049", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Reporting Party must be ""http://www.abr.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500049"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500049
    
            #region VR.ATO.GEN.500050
    
            /*  VR.ATO.GEN.500050
            Ensure at least one declaration is provided for Intermediary or Reporting Party
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) = NULL) AND (CONTEXT(RP.TrueAndCorrect) = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context = NULL) AND (^Context = NULL)
    
            Data Elements:
    
            ^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists == false && report.RPTrueAndCorrectExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500050", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. At least one declaration is required from either Intermediary or Reporting Party.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500050"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500050
    
            #region VR.ATO.GEN.500051
    
            /*  VR.ATO.GEN.500051
            Scheme for Intermediary.TrueAndCorrect must be set to http://www.abr.gov.au/abn
    
            Legacy Rule Format:
            IF (INT.TrueAndCorrect:entity.identifier.scheme <> "http://www.abr.gov.au/abn")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.abr.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = INT.TrueAndCorrect
            */
            assertion = ((report.INTTrueAndCorrectExists == true) && (report.INTTrueAndCorrectIdentifierScheme != @"http://www.abr.gov.au/abn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500051", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Intermediary.TrueAndCorrect must be ""http://www.abr.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500051"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500051
    
            #region VR.ATO.GEN.500052
    
            /*  VR.ATO.GEN.500052
            Scheme for ReportingParty.TrueAndCorrect must be set to http://www.abr.gov.au/abn
    
            Legacy Rule Format:
            IF (RP.TrueAndCorrect:entity.identifier.scheme <> "http://www.abr.gov.au/abn")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^IdentifierScheme <> 'http://www.abr.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TrueAndCorrect
            */
            assertion = ((report.RPTrueAndCorrectExists == true) && (report.RPTrueAndCorrectIdentifierScheme != @"http://www.abr.gov.au/abn"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500052", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty.TrueAndCorrect must be ""http://www.abr.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500052"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500052
    
            #region VR.ATO.GEN.500054
    
            /*  VR.ATO.GEN.500054
            ABN in the context definition must pass the ABN algorithm check
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect)<>NULL) AND (ABNALGORITHM(INT.TrueAndCorrect:entity.identifier.ABN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (FailsABNAlgorithm(^ABN))
    
            Data Elements:
    
            ^Context = INT.TrueAndCorrect
    
            ^ABN = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && FailsABNAlgorithm(report.INTTrueAndCorrectIdentifierABN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary declarer Australian Business Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTTrueAndCorrectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500054"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500054
    
            #region VR.ATO.GEN.500057
    
            /*  VR.ATO.GEN.500057
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CGNFT23, ' - ')
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT23 = tns:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.CGNFT23 == null ? false : report.CGNFT23.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT23", Value = report.CGNFT23 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500057
    
            #region VR.ATO.GEN.500057
    
            /*  VR.ATO.GEN.500057
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CGNFT36, ' - ')
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT36 = tns:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.CGNFT36 == null ? false : report.CGNFT36.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName.FullName.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT36", Value = report.CGNFT36 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500057
    
            #region VR.ATO.GEN.500058
    
            /*  VR.ATO.GEN.500058
            Declarer Position held of person making the declaration has a maximum field length of 50
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.Position.Text) > 50
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CGNFT24) > 50
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT24 = tns:PersonNameDetails.Position.Text
            */
            assertion = (Length(report.CGNFT24) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500058", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position must not exceed 50 characters",
                    LongDescription = @"For Declarer position, the maximum allowable characters for this form is 50",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT24", Value = report.CGNFT24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500058
    
            #region VR.ATO.GEN.500058
    
            /*  VR.ATO.GEN.500058
            Declarer Position held of person making the declaration has a maximum field length of 50
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonNameDetails.Position.Text) > 50
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CGNFT37) > 50
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT37 = tns:PersonNameDetails.Position.Text
            */
            assertion = (Length(report.CGNFT37) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500058", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position must not exceed 50 characters",
                    LongDescription = @"For Declarer position, the maximum allowable characters for this form is 50",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT37", Value = report.CGNFT37 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500058
    
            #region VR.ATO.GEN.500059
    
            /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Position.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:PersonNameDetails.Position.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT24 <> BLANK) AND (NotCharacterInSet(^CGNFT24, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT24 = tns:PersonNameDetails.Position.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT24) != true && !(IsMatch(report.CGNFT24, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500059", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position held of person making the declaration contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT24", Value = report.CGNFT24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500059
    
            #region VR.ATO.GEN.500059
    
            /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Position.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:PersonNameDetails.Position.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT37 <> BLANK) AND (NotCharacterInSet(^CGNFT37, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT37 = tns:PersonNameDetails.Position.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT37) != true && !(IsMatch(report.CGNFT37, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500059", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer position held of person making the declaration contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails.Position.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT37", Value = report.CGNFT37 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500059
    
            #region VR.ATO.GEN.500063
    
            /*  VR.ATO.GEN.500063
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CGNFT26) > 15
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT26 = tns:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.CGNFT26) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Telephone number must not exceed 15 characters",
                    LongDescription = @"For Declarer Telephone number, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT26", Value = report.CGNFT26 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500063
    
            #region VR.ATO.GEN.500063
    
            /*  VR.ATO.GEN.500063
            Declarer Telephone number has a maximum field length of 15
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number) > 15
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CGNFT39) > 15
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT39 = tns:ElectronicContact.Telephone.Minimal.Number
            */
            assertion = (Length(report.CGNFT39) > 15);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Telephone number must not exceed 15 characters",
                    LongDescription = @"For Declarer Telephone number, the maximum allowable characters for this form is 15",
                    Location = "/xbrli:xbrl/tns:ElectronicContact.Telephone.Minimal.Number[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT39", Value = report.CGNFT39 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500063
    
            #region VR.ATO.GEN.500065
    
            /*  VR.ATO.GEN.500065
            Tax agent number is not provided for reporting party as only applicable for Tax agent (Intermediary)
    
            Legacy Rule Format:
            IF (CONTEXT(RP.TrueAndCorrect) <> NULL) AND (RP.TrueAndCorrect:pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT28 <> NULL)
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT28 = tns:Identifiers.TaxAgentNumber.Identifier
    
            RP.TrueAndCorrect:^Context = RP:RP.TrueAndCorrect
            */
            assertion = (report.RPTrueAndCorrectExists != false && report.CGNFT28 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500065", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer tax agent number must not be present for Reporting Party",
                    LongDescription = @"Declarer tax agent number is only applicable for an Intermediary tax agent",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500065"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT28", Value = report.CGNFT28 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500065
    
            #region VR.ATO.GEN.500066
    
            /*  VR.ATO.GEN.500066
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT29 > Today())
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT29 = tns:Declaration.Signature.Date
            */
            assertion = (report.CGNFT29.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT29", Value = GetValueOrEmpty(report.CGNFT29) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500066
    
            #region VR.ATO.GEN.500066
    
            /*  VR.ATO.GEN.500066
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT41 > Today())
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT41 = tns:Declaration.Signature.Date
            */
            assertion = (report.CGNFT41.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration.Signature.Date[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT41", Value = GetValueOrEmpty(report.CGNFT41) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500066
    
            #region VR.ATO.GEN.500069
    
            /*  VR.ATO.GEN.500069
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT31 <> BLANK) AND (NotCharacterInSet(^CGNFT31, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            RP.TrueAndCorrect:^CGNFT31 = tns:Declaration.SignatoryIdentifier.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT31) != true && !(IsMatch(report.CGNFT31, @"^[a-zA-Z0-9!@\$%&\*\(\)\-_=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500069", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer identifier contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT31", Value = report.CGNFT31 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500069
    
            #region VR.ATO.GEN.500069
    
            /*  VR.ATO.GEN.500069
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyin.xx.xx:Declaration.SignatoryIdentifier.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGNFT43 <> BLANK) AND (NotCharacterInSet(^CGNFT43, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","_","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT43 = tns:Declaration.SignatoryIdentifier.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.CGNFT43) != true && !(IsMatch(report.CGNFT43, @"^[a-zA-Z0-9!@\$%&\*\(\)\-_=\[\];:'"",\.\?/ ]*$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500069", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer identifier contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT43", Value = report.CGNFT43 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500069
    
            #region VR.ATO.GEN.500095
    
            /*  VR.ATO.GEN.500095
            Declaration Signatory identifier text must be present
    
            Legacy Rule Format:
            IF (CONTEXT(INT.TrueAndCorrect) <> NULL) AND (INT.TrueAndCorrect:pyin.xx.xx:Declaration.SignatoryIdentifier.Text = NULLORBLANK) 
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT43 = BLANK)
    
            Data Elements:
    
            INT.TrueAndCorrect:^CGNFT43 = tns:Declaration.SignatoryIdentifier.Text
    
            INT.TrueAndCorrect:^Context = INT.TrueAndCorrect
            */
            assertion = (report.INTTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT43) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500095", Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Signatory identifier text must be present",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.INTTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500095"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGNFT43", Value = report.CGNFT43 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500095

            #region VR.ATO.GEN.500101

            /*  VR.ATO.GEN.500101
            Declaration Signatory identifier text must be present
    
            Legacy Rule Format:
            IF (CONTEXT(RP.TrueAndCorrect) <> NULL) AND (RP.TrueAndCorrect:pyin.xx.xx:Declaration.SignatoryIdentifier.Text = NULLORBLANK) 
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^CGNFT31 = BLANK)
    
            Data Elements:
    
            ^CGNFT31 = RP.TrueAndCorrect:Declaration.SignatoryIdentifier.Text
    
            ^Context = RP:RP.TrueAndCorrect
            */
            assertion = (report.RPTrueAndCorrectExists != false && string.IsNullOrWhiteSpace(report.CGNFT31) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500101",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Declaration Signatory identifier text must be present",
                    Location = "/xbrli:xbrl/tns:Declaration.SignatoryIdentifier.Text[@contextRef='" + report.RPTrueAndCorrectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500101" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "CGNFT31", Value = report.CGNFT31 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500101

            #region Repeating CPContext
            if (report.CPContextCollection != null)
            {
                int itemIndexContext1 = 0;
                foreach (CGNFT2018.CPContext cpContext in report.CPContextCollection)
                {
                    itemIndexContext1++;
            
                    #region VR.ATO.CGNFT.000011
            
                    /*  VR.ATO.CGNFT.000011
                    The TFN has failed the algorithm check.
    
                    Legacy Rule Format:
                    IF (TFNALGORITHM(CP:entity.identifier.TFN) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FailsTFNAlgorithm(^TFN))
            
                    Data Elements:
            
                    ^TFN = CP
                    */
                    assertion = FailsTFNAlgorithm(cpContext.IdentifierTFN);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary Tax file number must be a valid TFN",
                            LongDescription = @"The Subsidiary Tax file number has failed the Tax File Number algorithm check",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cpContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000011"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000011

                    #region VR.ATO.CGNFT.000082

                    /*  VR.ATO.CGNFT.000082
                    Invalid context. The number of ConsolidationParty.PurchasingEntity contexts must not exceed 1 for each ConsolidationParty context.

                    Legacy Rule Format:
                    FOR EACH CP
                    IF (COUNT(CP.PE WHERE CP.PE:entity.identifier = CP:entity.identifier ) > 1)
                       RETURN VALIDATION MESSAGE
                    ENDIF

                    Technical Business Rule Format:
                    CountOccurrence(^Identifier1, ^Identifier1 = ^Identifier2) > 1

                    Data Elements:

                    ^Identifier1 = CP

                    ^Identifier2 = CP:CP.PE
                    */
                    assertion = ((report.CPPEContextCollection == null ? 0 : report.CPPEContextCollection.Count(Identifier1Repeat => Identifier1Repeat.CPPEIdentifier == cpContext.Identifier)) > 1);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000082",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid context. The number of ConsolidationParty.PurchasingEntity contexts must not exceed 1 for each ConsolidationParty context.",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.CPContextCollection == null ? 0 : (report.CPContextCollection.Count() == 0 ? 0 : report.CPContextCollection.Last().LastOccurrenceIndex)) + "/xbrli:entity/xbrli:identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000082" } },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000082

                    #region VR.ATO.CGNFT.000015

                    /*  VR.ATO.CGNFT.000015
                    Subsidiary member exit reason description has a maximum length of 200
    
                    Legacy Rule Format:
                    IF LENGTH([cgnft17]) > 200
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft17] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
        
                    Technical Business Rule Format:
                    Length(^CGNFT17) > 200
            
                    Data Elements:
            
                    CP:^CGNFT17 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
                    */
                    assertion = (Length(cpContext.CGNFT17) > 200);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000015", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary member exit reason description must not exceed 200 characters",
                            LongDescription = @"For Subsidiary member exit reason description, the maximum allowable characters for this form is 200",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000015"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT17", Value = cpContext.CGNFT17 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000015
            
                    #region VR.ATO.CGNFT.000017
            
                    /*  VR.ATO.CGNFT.000017
                    Scheme for Consolidated Party must be set to http://www.ato.gov.au/tfn
    
                    Legacy Rule Format:
                    IF (CP:entity.identifier.scheme <> “http://www.ato.gov.au/tfn”)
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'
            
                    Data Elements:
            
                    ^IdentifierScheme = CP
                    */
                    assertion = (cpContext.IdentifierScheme != @"http://www.ato.gov.au/tfn");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000017", Severity = ProcessMessageSeverity.Error,
                            Description = @"Entity identifier scheme for Consolidated Party must be ""http://www.ato.gov.au/tfn""",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cpContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000017"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000017
            
                    #region VR.ATO.CGNFT.000024
            
                    /*  VR.ATO.CGNFT.000024
                    A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
                    Legacy Rule Format:
                    IF ([cgnft10] <> NULLORBLANK) AND ([cgnft10] CONTAINS SET("{","}","|","_","\","#"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
        
                    Technical Business Rule Format:
                    ContainsSet(^CGNFT10, '"{","}","|","_","\","#"')
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(cpContext.CGNFT10, @"\{|\}|\||_|\\|#",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000024", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary legal name contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000024"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000024
            
                    #region VR.ATO.CGNFT.000025
            
                    /*  VR.ATO.CGNFT.000025
                    A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
                    Legacy Rule Format:
                    IF ([cgnft17] <> NULLORBLANK) AND (ANY CHARACTER OF [cgnft17] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft17] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
        
                    Technical Business Rule Format:
                    (^CGNFT17 <> BLANK) AND (NotCharacterInSet(^CGNFT17, 'a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
            
                    Data Elements:
            
                    CP:^CGNFT17 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT17) != true && !(IsMatch(cpContext.CGNFT17, @"^[a-z,A-Z,0-9,""!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ""]*$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000025", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary member exit reason description contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000025"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT17", Value = cpContext.CGNFT17 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000025
            
                    #region VR.ATO.CGNFT.000031
            
                    /*  VR.ATO.CGNFT.000031
                    Non-individual name field only contains characters other than alphas and numerics, at least one alpha or numeric must be present.
    
                    Legacy Rule Format:
                    IF ([cgnft10] <> NULLORBLANK) AND ([cgnft10] DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
        
                    Technical Business Rule Format:
                    (^CGNFT10 <> BLANK) AND (NotContainsSet(^CGNFT10, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT10) != true && !(IsMatch(cpContext.CGNFT10, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000031", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary legal name must contain at least one alpha or numeric character",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000031"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000031
            
                    #region VR.ATO.CGNFT.000036
            
                    /*  VR.ATO.CGNFT.000036
                    The ABN of the subsidiary member joining or leaving the consolidated group must differ from the ABN of the head company.
    
                    Legacy Rule Format:
                    IF ([cgnft11] <> NULLORBLANK) AND ([cgnft11] = RP:entity.identifier.ABN)
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft11] = CGNFT:CP:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
        
                    Technical Business Rule Format:
                    (^CGNFT11 <> BLANK) AND (^CGNFT11 = ^ABN)
            
                    Data Elements:
            
                    CP:^CGNFT11 = tns:Identifiers.AustralianBusinessNumber.Identifier
            
                    CP:^ABN = RP
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT11) != true && cpContext.CGNFT11 == report.RPIdentifierABN);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000036", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary ABN must not be the same  as the Head company ABN",
                            LongDescription = @"The ABN of the subsidiary member joining or leaving the consolidated group must differ from the ABN of the head company",
                            Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000036"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT11", Value = cpContext.CGNFT11 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000036
            
                    #region VR.ATO.CGNFT.000040
            
                    /*  VR.ATO.CGNFT.000040
                    The TFN of the subsidiary member joining or leaving the consolidated group must differ from the TFN of the head company.
    
                    Legacy Rule Format:
                    IF (CP:entity.identifier.TFN <> NULL) AND (CP:entity.identifier.TFN = [cgnft3])
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft3] = CGNFT:RP:pyid.02.00:Identifiers.TaxFileNumber.Identifier
        
                    Technical Business Rule Format:
                    (^TFN <> NULL) AND (^TFN = ^CGNFT3)
            
                    Data Elements:
            
                    RP:^CGNFT3 = tns:Identifiers.TaxFileNumber.Identifier
            
                    RP:^TFN = CP
                    */
                    assertion = (cpContext.IdentifierTFN != null && cpContext.IdentifierTFN == report.CGNFT3);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000040", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary TFN must not be the same  as the Head company TFN",
                            LongDescription = @"The TFN of the subsidiary member joining or leaving the consolidated group must differ from the TFN of the head company",
                            Location = "/xbrli:xbrl/tns:Identifiers.TaxFileNumber.Identifier[@contextRef='" + report.RPId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000040"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT3", Value = report.CGNFT3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000040

                    //VR.ATO.CGNFT.000033 removed for CR2900 | INC000029748406
            
                    #region VR.ATO.CGNFT.000051
            
                    /*  VR.ATO.CGNFT.000051
                    Name field cannot contain 'space hyphen space'.
    
                    Legacy Rule Format:
                    IF ([cgnft10] CONTAINS  " - " )
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
        
                    Technical Business Rule Format:
                    Contains(^CGNFT10, ' - ')
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (cpContext.CGNFT10 == null ? false : cpContext.CGNFT10.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000051", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary legal name cannot contain space hyphen space combinations",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000051"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000051
            
                    #region VR.ATO.CGNFT.000056
            
                    /*  VR.ATO.CGNFT.000056
                    The date the subsidiary member left the group cannot be earlier than the chosen date of consolidation.
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    IF ([cgnft15] <> NULL) AND ([cgnft15] < [cgnft6])
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft6] = CGNFT:RP:pyde.02.17:OrganisationDetails.TaxConsolidation.Date
                    [cgnft15] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
        
                    Technical Business Rule Format:
                    (^CGNFT15 <> NULL) AND (^CGNFT15 < ^CGNFT6)
            
                    Data Elements:
            
                    CP:^CGNFT15 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
            
                    RP:^CGNFT6 = tns:OrganisationDetails.TaxConsolidation.Date
                    */
                    if (!isStandalone)
                    {
                        assertion = (cpContext.CGNFT15 != null && cpContext.CGNFT15.GetValueOrDefault() < report.CGNFT6.GetValueOrDefault());
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.CGNFT.000056", Severity = ProcessMessageSeverity.Error,
                                Description = @"Subsidiary member exit date must not be earlier than the Date of consolidation",
                                Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date[@contextRef='" + cpContext.Id + "']",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000056"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "CGNFT15", Value = GetValueOrEmpty(cpContext.CGNFT15) });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "CGNFT6", Value = GetValueOrEmpty(report.CGNFT6) });
                
                            response.Add(processMessage);
                        }
                        }
                    #endregion // VR.ATO.CGNFT.000056
            
                    #region VR.ATO.CGNFT.000062
            
                    /*  VR.ATO.CGNFT.000062
                    Non-individual name field contains 'P/L' which is not a valid abbreviation. This abbreviation cannot be used.
    
                    Legacy Rule Format:
                    IF ([cgnft10] CONTAINS "P/L")
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
        
                    Technical Business Rule Format:
                    Contains(^CGNFT10, 'P/L')
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (cpContext.CGNFT10 == null ? false : cpContext.CGNFT10.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000062", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary legal name cannot contain ""P/L""",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000062"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000062

                    #region VR.ATO.CGNFT.000065

                    /*  VR.ATO.CGNFT.000065
                    If any subsidiary member joining or leaving fields have been completed then the subsidiary member legal name and TFN must be completed.
    
                    Legacy Rule Format:
                    IF (([cgnft10] <> NULLORBLANK) OR ([cgnft11] <> NULLORBLANK) OR (([cgnft12] <> NULLORBLANK) OR ([cgnft13] <> NULLORBLANK)) OR (CP:entity.identifier.TFN <> NULL) OR ([cgnft14] <> NULL) OR ([cgnft15] <> NULL) OR ([cgnft16] <> NULLORBLANK) OR ([cgnft17] <> NULLORBLANK) OR ([cgnft19] <> NULLORBLANK) OR ([cgnft20] <> NULLORBLANK)) AND (([cgnft10] = NULLORBLANK) OR (CP:entity.identifier.TFN = NULL))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
                    [cgnft11] = CGNFT:CP:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
                    [cgnft12] = CGNFT:CP:pyid.02.00:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    [cgnft13] = CGNFT:CP:pyid.02.00:Identifiers.AustralianCompanyNumber.Identifier
                    [cgnft14] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
                    [cgnft15] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
                    [cgnft16] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
                    [cgnft17] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
                    [cgnft20] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
        
                    Technical Business Rule Format:
                    ((^CGNFT10 <> BLANK) OR (^CGNFT11 <> BLANK) OR ((^CGNFT12 <> BLANK) OR (^CGNFT13 <> BLANK)) OR (^TFN <> NULL) OR (^CGNFT14 <> NULL) OR (^CGNFT15 <> NULL) OR (^CGNFT16 <> BLANK) OR (^CGNFT17 <> BLANK) OR (^CGNFT19 <> BLANK) OR (^CGNFT20 <> BLANK)) AND ((^CGNFT10 = BLANK) OR (^TFN = NULL))
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
            
                    CP:^CGNFT11 = tns:Identifiers.AustralianBusinessNumber.Identifier
            
                    CP:^CGNFT12 = tns:Identifiers.AustralianRegisteredBodyNumber.Identifier
            
                    CP:^CGNFT13 = tns:Identifiers.AustralianCompanyNumber.Identifier
            
                    CP:^CGNFT14 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
            
                    CP:^CGNFT15 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
            
                    CP:^CGNFT16 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
            
                    CP:^CGNFT17 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
            
                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
            
                    CP.PE:^CGNFT20 = tns:Identifiers.AustralianBusinessNumber.Identifier
            
                    CP.PE:^TFN = CP
                    */
                    assertion = 
                    (
                        (
                            string.IsNullOrWhiteSpace(cpContext.CGNFT10) != true ||
                            string.IsNullOrWhiteSpace(cpContext.CGNFT11) != true ||
                            (
                                string.IsNullOrWhiteSpace(cpContext.CGNFT12) != true ||
                                string.IsNullOrWhiteSpace(cpContext.CGNFT13) != true
                            ) ||
                            cpContext.IdentifierTFN != null ||
                            cpContext.CGNFT14 != null ||
                            cpContext.CGNFT15 != null ||
                            string.IsNullOrWhiteSpace(cpContext.CGNFT16) != true ||
                            string.IsNullOrWhiteSpace(cpContext.CGNFT17) != true ||
                            string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT19) != true ||
                            string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT20) != true
                        )
                        && 
                        (
                            string.IsNullOrWhiteSpace(cpContext.CGNFT10) == true ||
                            cpContext.IdentifierTFN == null
                        )
                    );

                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000065", Severity = ProcessMessageSeverity.Error,
                            Description = @"The Subsidiary legal name and Subsidiary Tax file number must be provided",
                            LongDescription = @"The Subsidiary legal name and Subsidiary Tax file number must be provided, when the details of any subsidiary member joining or leaving have been completed  - Tax file number, Legal Name, Australian business number, Australian registered body number, Member entry date, Member exit date, Member exit reason, Member exit reason description, Purchasing entity legal name or Purchasing entity Australian business number",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000065"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT11", Value = cpContext.CGNFT11 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT12", Value = cpContext.CGNFT12 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT13", Value = cpContext.CGNFT13 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT14", Value = GetValueOrEmpty(cpContext.CGNFT14) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT15", Value = GetValueOrEmpty(cpContext.CGNFT15) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT16", Value = cpContext.CGNFT16 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT17", Value = cpContext.CGNFT17 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT19", Value = cpContext.CPPEContext.CGNFT19 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT20", Value = cpContext.CPPEContext.CGNFT20 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000065
            
                    #region VR.ATO.CGNFT.000069
            
                    /*  VR.ATO.CGNFT.000069
                    The Subsidiary legal name is mandatory
    
                    Legacy Rule Format:
                    IF ([cgnft10] = NULLORBLANK) 
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
        
                    Technical Business Rule Format:
                    ^CGNFT10 = BLANK
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT10) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000069", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary legal name must be provided",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000069"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000069
            
                    #region VR.ATO.CGNFT.000070
            
                    /*  VR.ATO.CGNFT.000070
                    Individual or non-individual name field contains repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    IF ([cgnft10] CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft10] = CGNFT:CP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
        
                    Technical Business Rule Format:
                    (ContainsSet(^CGNFT10, '"--","''","  "'))
            
                    Data Elements:
            
                    CP:^CGNFT10 = tns:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(cpContext.CGNFT10, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000070", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary legal name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000070"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT10", Value = cpContext.CGNFT10 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000070
            
                    #region VR.ATO.CGNFT.000076
            
                    /*  VR.ATO.CGNFT.000076
                    The Australian Business Number (ABN) has failed the ABN algorithm check.
    
                    Legacy Rule Format:
                    IF ([cgnft11] <> NULLORBLANK) and (ABNALGORITHM([cgnft11]) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft11] = CGNFT:CP:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
        
                    Technical Business Rule Format:
                    (^CGNFT11 <> BLANK) and (FailsABNAlgorithm(^CGNFT11))
            
                    Data Elements:
            
                    CP:^CGNFT11 = tns:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT11) != true && FailsABNAlgorithm(cpContext.CGNFT11));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000076", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ABN",
                            LongDescription = @"The Subsidiary Australian business number has failed the ABN algorithm check",
                            Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000076"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT11", Value = cpContext.CGNFT11 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000076
            
                    #region VR.ATO.CGNFT.000078
            
                    /*  VR.ATO.CGNFT.000078
                    If leaving code 'Other' is selected then the exit reason description field must be completed.
    
                    Legacy Rule Format:
                    IF ([cgnft16] = "Other") AND ([cgnft17] = NULLORBLANK)
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft16] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
                    [cgnft17] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
        
                    Technical Business Rule Format:
                    (^CGNFT16 = 'Other') AND (^CGNFT17 = BLANK)
            
                    Data Elements:
            
                    CP:^CGNFT17 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
            
                    CP:^CGNFT16 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
                    */
                    assertion = (cpContext.CGNFT16 == @"Other" && string.IsNullOrWhiteSpace(cpContext.CGNFT17) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000078", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary member exit reason description must be provided",
                            LongDescription = @"When the Subsidiary member exit reason 'Other' is selected, the Subsidiary member exit reason description must be completed",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000078"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT16", Value = cpContext.CGNFT16 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT17", Value = cpContext.CGNFT17 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000078
            
                    #region VR.ATO.CGNFT.000079
            
                    /*  VR.ATO.CGNFT.000079
                    If the purchasing entity legal name or ABN has been completed then the leaving code must equal 'Full-sold'.
    
                    Legacy Rule Format:
                    IF ([cgnft16] <> "Full-sold") AND (([cgnft19] <> NULLORBLANK) OR (([cgnft20] <> NULLORBLANK) OR ([cgnft27] <> NULLORBLANK)))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft16] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
                    [cgnft20] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
                    [cgnft27] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianRegisteredBodyNumber.Identifier
        
                    Technical Business Rule Format:
                    (^CGNFT16 <> 'Full-sold') AND ((^CGNFT19 <> BLANK) OR ((^CGNFT20 <> BLANK) OR (^CGNFT27 <> BLANK)))
            
                    Data Elements:
            
                    CP:^CGNFT16 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
            
                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
            
                    CP.PE:^CGNFT20 = tns:Identifiers.AustralianBusinessNumber.Identifier
            
                    CP.PE:^CGNFT27 = tns:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (cpContext.CGNFT16 != @"Full-sold" && (string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT19) != true || (string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT20) != true || string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT27) != true)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000079", Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity details present and Subsidiary member exit reason not set to 'Full-Sold'",
                            LongDescription = @"When Purchasing entity legal name or Purchasing entity Australian business number are provided, the Subsidiary member exit reason must be set to 'Full-Sold'",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000079"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT16", Value = cpContext.CGNFT16 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT19", Value = cpContext.CPPEContext.CGNFT19 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT20", Value = cpContext.CPPEContext.CGNFT20 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT27", Value = cpContext.CPPEContext.CGNFT27 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000079
            
                    #region VR.ATO.CGNFT.000080
            
                    /*  VR.ATO.CGNFT.000080
                    If the leaving code equals 'Full-sold' then the purchasing entity legal name or ABN must be completed.
    
                    Legacy Rule Format:
                    IF ([cgnft16] = "Full-sold") AND (([cgnft19] = NULLORBLANK) AND (([cgnft20] = NULLORBLANK) AND ([cgnft27] = NULLORBLANK)))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft16] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
                    [cgnft20] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
                    [cgnft27] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianRegisteredBodyNumber.Identifier
        
                    Technical Business Rule Format:
                    (^CGNFT16 = 'Full-sold') AND ((^CGNFT19 = BLANK) AND ((^CGNFT20 = BLANK) AND (^CGNFT27 = BLANK)))
            
                    Data Elements:
            
                    CP:^CGNFT16 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
            
                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
            
                    CP.PE:^CGNFT20 = tns:Identifiers.AustralianBusinessNumber.Identifier
            
                    CP.PE:^CGNFT27 = tns:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (cpContext.CGNFT16 == @"Full-sold" && (string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT19) == true && (string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT20) == true && string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT27) == true)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000080", Severity = ProcessMessageSeverity.Error,
                            Description = @"Purchasing entity details absent and Subsidiary member exit reason set to 'Full-Sold'",
                            LongDescription = @"When Subsidiary member exit reason is set to 'Full-Sold', Purchasing entity legal name and Purchasing entity Australian business number must be provided",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000080"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT16", Value = cpContext.CGNFT16 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT19", Value = cpContext.CPPEContext.CGNFT19 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT20", Value = cpContext.CPPEContext.CGNFT20 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT27", Value = cpContext.CPPEContext.CGNFT27 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000080
            
                    #region VR.ATO.CGNFT.000089
            
                    /*  VR.ATO.CGNFT.000089
                    The date the subsidiary member joined the consolidated group must not be prior to 01/07/2002 and cannot be a future date.
    
                    Legacy Rule Format:
                    IF ([cgnft14] <> NULL) AND (([cgnft14] < 2002-07-01) OR ([cgnft14] > DATE(TODAY)))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft14] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
        
                    Technical Business Rule Format:
                    (^CGNFT14 <> NULL) AND ((^CGNFT14 < AsDate('2002-07-01')) OR (^CGNFT14 > Today()))
            
                    Data Elements:
            
                    CP:^CGNFT14 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
                    */
                    assertion = (cpContext.CGNFT14 != null && (cpContext.CGNFT14.GetValueOrDefault() < AsDate(@"2002-07-01") || cpContext.CGNFT14.GetValueOrDefault() > DateTime.Now.Date));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000089", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary member entry date is invalid",
                            LongDescription = @"The date the subsidiary member joined the consolidated group must not be prior to 01/07/2002 and cannot be a future date",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000089"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT14", Value = GetValueOrEmpty(cpContext.CGNFT14) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000089
            
                    #region VR.ATO.CGNFT.000090
            
                    /*  VR.ATO.CGNFT.000090
                    The Subsidiary member entry date must be completed if the Members joining/leaving a consolidated group schedule is lodged with the income tax return of the head company.
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    IF (([cgnft14] = NULL))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft14] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
        
                    Technical Business Rule Format:
                    ((^CGNFT14 = NULL))
            
                    Data Elements:
            
                    CP:^CGNFT14 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
                    */
                    if (!isStandalone)
                    {
                        assertion = (cpContext.CGNFT14 == null);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.CGNFT.000090", Severity = ProcessMessageSeverity.Error,
                                Description = @"Subsidiary member entry date must be provided",
                                LongDescription = @"The Subsidiary member entry date must be completed if the Members joining/leaving a consolidated group schedule is lodged with the income tax return of the head company",
                                Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date[@contextRef='" + cpContext.Id + "']",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000090"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "CGNFT14", Value = GetValueOrEmpty(cpContext.CGNFT14) });
                
                            response.Add(processMessage);
                        }
                        }
                    #endregion // VR.ATO.CGNFT.000090
            
                    #region VR.ATO.CGNFT.000091
            
                    /*  VR.ATO.CGNFT.000091
                    The ACN/ARBN/SMSF auditor number has failed the algorithm check.
    
                    Legacy Rule Format:
                    IF ([cgnft12] <> NULLORBLANK) AND (ARBNALGORITHM([cgnft12]) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft12] = CGNFT:CP:pyid.02.00:Identifiers.AustralianRegisteredBodyNumber.Identifier
        
                    Technical Business Rule Format:
                    (^CGNFT12 <> BLANK) AND (FailsARBNAlgorithm(^CGNFT12))
            
                    Data Elements:
            
                    CP:^CGNFT12 = tns:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT12) != true && FailsACNAlgorithm(cpContext.CGNFT12));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000091", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ARBN",
                            LongDescription = @"The Subsidiary Australian registered body number has failed the ARBN algorithm check",
                            Location = "/xbrli:xbrl/tns:Identifiers.AustralianRegisteredBodyNumber.Identifier[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000091"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT12", Value = cpContext.CGNFT12 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000091
            
                    #region VR.ATO.CGNFT.000092
            
                    /*  VR.ATO.CGNFT.000092
                    The date the subsidiary member left the group cannot be earlier that the date joined the group.
    
                    Legacy Rule Format:
                    IF ([cgnft14] <> NULL) AND ([cgnft15] <> NULL) AND ([cgnft15] < [cgnft14])
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft14] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
                    [cgnft15] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
        
                    Technical Business Rule Format:
                    (^CGNFT14 <> NULL) AND (^CGNFT15 <> NULL) AND (^CGNFT15 < ^CGNFT14)
            
                    Data Elements:
            
                    CP:^CGNFT15 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
            
                    CP:^CGNFT14 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberEntry.Date
                    */
                    assertion = (cpContext.CGNFT14 != null && cpContext.CGNFT15 != null && cpContext.CGNFT15.GetValueOrDefault() < cpContext.CGNFT14.GetValueOrDefault());
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000092", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary member exit date must not be earlier than the entry date",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000092"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT14", Value = GetValueOrEmpty(cpContext.CGNFT14) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT15", Value = GetValueOrEmpty(cpContext.CGNFT15) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000092
            
                    #region VR.ATO.CGNFT.000097
            
                    /*  VR.ATO.CGNFT.000097
                    If any subsidiary member leaving fields have been completed then the date left the group and leaving code must be completed.
    
                    Legacy Rule Format:
                    IF (([cgnft15] <> NULL) OR ([cgnft16] <> NULLORBLANK) OR ([cgnft17] <> NULLORBLANK) OR ([cgnft19] <> NULLORBLANK) OR (([cgnft20] <> NULLORBLANK) OR ([cgnft27] <> NULLORBLANK))) AND (([cgnft15] = NULL) OR ([cgnft16] = NULLORBLANK))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft15] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
                    [cgnft16] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
                    [cgnft17] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
                    [cgnft19] = CGNFT:CP.PE:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text
                    [cgnft20] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier
                    [cgnft27] = CGNFT:CP.PE:pyid.02.00:Identifiers.AustralianRegisteredBodyNumber.Identifier
        
                    Technical Business Rule Format:
                    ((^CGNFT15 <> NULL) OR (^CGNFT16 <> BLANK) OR (^CGNFT17 <> BLANK) OR (^CGNFT19 <> BLANK) OR ((^CGNFT20 <> BLANK) OR (^CGNFT27 <> BLANK))) AND ((^CGNFT15 = NULL) OR (^CGNFT16 = BLANK))
            
                    Data Elements:
            
                    CP:^CGNFT15 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
            
                    CP:^CGNFT16 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Code
            
                    CP:^CGNFT17 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Description
            
                    CP.PE:^CGNFT19 = tns:OrganisationNameDetails.OrganisationalName.Text
            
                    CP.PE:^CGNFT20 = tns:Identifiers.AustralianBusinessNumber.Identifier
            
                    CP.PE:^CGNFT27 = tns:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = ((cpContext.CGNFT15 != null || string.IsNullOrWhiteSpace(cpContext.CGNFT16) != true || string.IsNullOrWhiteSpace(cpContext.CGNFT17) != true || string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT19) != true || (string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT20) != true || string.IsNullOrWhiteSpace(cpContext.CPPEContext.CGNFT27) != true)) && (cpContext.CGNFT15 == null || string.IsNullOrWhiteSpace(cpContext.CGNFT16) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000097", Severity = ProcessMessageSeverity.Error,
                            Description = @"The Subsidiary member exit date and Subsidiary member exit reason must be provided",
                            LongDescription = @"The Subsidiary member exit date and Subsidiary member exit reason must be provided, when the details of any subsidiary member leaving have been completed  - Member exit date, Member exit reason, Member exit reason description, Purchasing entity legal name or Purchasing entity Australian business number",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000097"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT15", Value = GetValueOrEmpty(cpContext.CGNFT15) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT16", Value = cpContext.CGNFT16 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT17", Value = cpContext.CGNFT17 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT19", Value = cpContext.CPPEContext.CGNFT19 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT20", Value = cpContext.CPPEContext.CGNFT20 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT27", Value = cpContext.CPPEContext.CGNFT27 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000097
            
                    #region VR.ATO.CGNFT.000099
            
                    /*  VR.ATO.CGNFT.000099
                    The date the subsidiary member left the consolidated group cannot be a future date.
    
                    Legacy Rule Format:
                    IF ([cgnft15] > DATE(TODAY))
                         RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [cgnft15] = CGNFT:CP:pyde.02.17:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
        
                    Technical Business Rule Format:
                    (^CGNFT15 > Today())
            
                    Data Elements:
            
                    CP:^CGNFT15 = tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date
                    */
                    assertion = (cpContext.CGNFT15.GetValueOrDefault() > DateTime.Now.Date);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGNFT.000099", Severity = ProcessMessageSeverity.Error,
                            Description = @"Subsidiary member exit date cannot be a future date",
                            Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationSubsidiaryMemberExit.Date[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000099"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT15", Value = GetValueOrEmpty(cpContext.CGNFT15) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGNFT.000099
            
                    #region VR.ATO.CGNFT.000196
            
                    /*  VR.ATO.CGNFT.000196
                    Invalid context. Period start date must be the same as the Period end date for this form.
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN DOES NOT EXIST
                    IF period.startDate <> period.endDate
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^StartDate <> ^EndDate
            
                    Data Elements:
            
                    ^StartDate = CP
            
                    ^EndDate = CP
                    */
                    if (isStandalone)
                    {
                        assertion = (cpContext.StartDate != cpContext.EndDate);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.CGNFT.000196", Severity = ProcessMessageSeverity.Error,
                                Description = @"Invalid context. Period start date must be the same as the Period end date for this form.",
                                Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cpContext.OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000196"} },
                            };
                            response.Add(processMessage);
                        }
                        }
                    #endregion // VR.ATO.CGNFT.000196
            
                    #region VR.ATO.CGNFT.000197
            
                    /*  VR.ATO.CGNFT.000197
                    Invalid context. Period start date cannot be any day in the future.
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN DOES NOT EXIST
                    IF period.startDate > DATE(TODAY)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^StartDate > Today()
            
                    Data Elements:
            
                    ^StartDate = CP
                    */
                    if (isStandalone)
                    {
                        assertion = (cpContext.StartDate > DateTime.Now.Date);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.CGNFT.000197", Severity = ProcessMessageSeverity.Error,
                                Description = @"Invalid context. Period start date cannot be any day in the future.",
                                Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(cpContext.OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000197"} },
                            };
                            response.Add(processMessage);
                        }
                        }
                    #endregion // VR.ATO.CGNFT.000197
            
                    #region VR.ATO.GEN.428021
            
                    /*  VR.ATO.GEN.428021
                    The ACN/ARBN/SMSF auditor number has failed the algorithm check.
    
                    Legacy Rule Format:
                    IF (pyid.xx.xx:Identifiers.AustralianCompanyNumber.Identifier <> NULLORBLANK) AND (ACNALGORITHM(pyid.xx.xx:Identifiers.AustralianCompanyNumber.Identifier) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGNFT13 <> BLANK) AND (FailsACNAlgorithm(^CGNFT13))
            
                    Data Elements:
            
                    CP:^CGNFT13 = tns:Identifiers.AustralianCompanyNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(cpContext.CGNFT13) != true && FailsACNAlgorithm(cpContext.CGNFT13));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428021", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ACN",
                            LongDescription = @"The Australian Company Number (ACN) has failed the algorithm check",
                            Location = "/xbrli:xbrl/tns:Identifiers.AustralianCompanyNumber.Identifier[@contextRef='" + cpContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428021"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGNFT13", Value = cpContext.CGNFT13 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428021
            
                    #region VR.ATO.GEN.500030
            
                    /*  VR.ATO.GEN.500030
                    Invalid context. Period start date must be the same across all duration contexts.
    
                    Legacy Rule Format:
                    IF period.startDate WHERE CONTEXT(ALL) <> RP:period.startDate
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^StartDate1 <> NULL AND ^StartDate1 <> ^StartDate
            
                    Data Elements:
            
                    ^StartDate = RP
            
                    ^StartDate1 = CP
                    */
                    assertion = (cpContext.StartDate != null && cpContext.StartDate != report.RPStartDate);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500030", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid context. Period start date must be the same across all duration contexts.",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500030"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.500030
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 
