using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationCGNFT;
using Ato.EN.IntegrationServices.CodeGenerationCTR;
using VaTS;
using DataContracts;
using System;
using System.Collections.Generic;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.CTR202602
{
    internal class CrossFormValidatorCGNFT : ICrossFormValidator
    {
        private CTR2026 ParentDocument { get; }
        private CGNFT2018 ChildDocument { get; }
        public List<ProcessMessageDocument> Response { get; private set; }

        internal CrossFormValidatorCGNFT(CTR2026 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (CGNFT2018)childDocument.ConsumedReport;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOCGNFT000027();
            VRATOCGNFT000030();
            VRATOCGNFT000057();

            return Response;
        }

        #region VR.ATO.CGNFT.000027

        /*  VR.ATO.CGNFT.000027
        Invalid context. Period start date must be the same as the Period start date of the parent form.

        Legacy Rule Format:
        IF (PARENT RETURN <> NULL) AND (RP:period.startDate <> CTR:RP:period.startDate)
             RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR Start Date = CTR Start Date
        ^CGNFT Start Date = CGNFT Start Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGNFT000027()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.RPStartDate.GetValueOrDefault() != ChildDocument.RPStartDate.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period start date must be the same as the Period start date of the parent form.",
                    Location = ChildDocument.RPOccurrenceIndex > 1 ? "/xbrli:xbrl/xbrli:context[" + ChildDocument.RPOccurrenceIndex.ToString() + "]xbrli:period/xbrli:startDate"
                    : "/xbrli:xbrl/xbrli:context/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000027" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR Start Date", Value = CTR2026Validator.GetValueOrEmptyDate(ParentDocument.RPStartDate) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CGNFT Start Date", Value = CTR2026Validator.GetValueOrEmptyDate(ChildDocument.RPStartDate) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGNFT.000027

        #region VR.ATO.CGNFT.000030

        /*  VR.ATO.CGNFT.000030
         Invalid context. Period end date must be the same as the Period end date of the parent form.

         Legacy Rule Format:
         IF (PARENT RETURN <> NULL) AND (RP:period.endDate <> CTR:RP:period.endDate)
             RETURN VALIDATION MESSAGE
         ENDIF

         Data Elements:

         ^CTR End Date = CTR End Date
         ^CGNFT End Date = CGNFT End Date
         */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGNFT000030()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.RPEndDate.GetValueOrDefault() != ChildDocument.RPEndDate.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. Period end date must be the same as the Period end date of the parent form.",
                    Location = ChildDocument.RPOccurrenceIndex > 1 ? "/xbrli:xbrl/xbrli:context[" + ChildDocument.RPOccurrenceIndex.ToString() + "]xbrli:period/xbrli:endDate"
                     : "/xbrli:xbrl/xbrli:context/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000030" } },
                };
              
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR End Date", Value = CTR2026Validator.GetValueOrEmptyDate(ParentDocument.RPEndDate) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CGNFT End Date", Value = CTR2026Validator.GetValueOrEmptyDate(ChildDocument.RPEndDate) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGNFT.000030

        #region VR.ATO.CGNFT.000057

        /*  VR.ATO.CGNFT.000057
        The chosen date of consolidation must not be prior to 01/07/2002 and must be within the financial year of income applying to the income tax return of the head company.

        Legacy Rule Format:
        IF (PARENT RETURN <> NULL) AND ((CTR:RP.Period.Start.Date <> NULL) AND 
        (CTR:RP.Period.End.Date <> NULL)) AND
        (([cgnft6] < 2002-07-01) OR ([cgnft6] < CTR:RP.Period.Start.Date) OR ([cgnft6] > CTR:RP.Period.End.Date))
            RETURN VALIDATION MESSAGE
        ENDIF
        
         Data Elements:

        ^CGNFT6 = CGNFT:RP:OrganisationDetails.TaxConsolidation.Date
        ^CTR Start Date = CTR Start Date
        ^CTR End Date = CTR End Date
         */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGNFT000057()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.RPEndDate.HasValue && ParentDocument.RPStartDate.HasValue &&
                  (ChildDocument.CGNFT6.GetValueOrDefault() < DateTime.Parse(@"2002-07-01") ||
                     ChildDocument.CGNFT6.GetValueOrDefault() > ParentDocument.RPEndDate.GetValueOrDefault() ||
                     ChildDocument.CGNFT6.GetValueOrDefault() < ParentDocument.RPStartDate.GetValueOrDefault()
                  );

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of consolidation is invalid",
                    LongDescription = @"The Date of consolidation must not be prior to 01/07/2002 and must be within the financial year of income applying to the income tax return of the head company",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidation.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000057" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CGNFT6", Value = CTR2026Validator.GetValueOrEmptyDate(ChildDocument.CGNFT6) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR Start Date", Value = CTR2026Validator.GetValueOrEmptyDate(ParentDocument.RPStartDate) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR End Date", Value = CTR2026Validator.GetValueOrEmptyDate(ParentDocument.RPEndDate) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGNFT.000057
    }
}
