using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationCTR;
using Ato.EN.IntegrationServices.CodeGenerationIDS;
using VaTS;
using DataContracts;
using System.Collections.Generic;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.CTR202602
{
    internal class CrossFormValidatorIDS : ICrossFormValidator
    {
        private CTR2026 ParentDocument { get; }
        private IDS2025 ChildDocument { get; }
        public List<ProcessMessageDocument> Response { get; private set; }

        internal CrossFormValidatorIDS(CTR2026 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (IDS2025)childDocument.ConsumedReport;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //Parent Cross From Validation Rules
            VRATOCTR428405();
            VRATOCTR428406();
            VRATOCTR428408();
            VRATOCTR428409();
            VRATOCTR428410();
            VRATOCTR428412();
            VRATOCTR428413();
            VRATOCTR428414();
            VRATOCTR428415();
            VRATOCTR428416();
            VRATOCTR428417();
            VRATOCTR428418();
            VRATOCTR428419();
            VRATOCTR428420();
            VRATOCTR428421();
            VRATOCTR428422();
            VRATOCTR428426();
            VRATOCTR428427();
            VRATOCTR438050();

            //Child Cross From Validation Rules
            VRATOIDS440670();

            return Response;
        }

        #region VR.ATO.CTR.428405

        /*  VR.ATO.CTR.428405
        If there is a 'Yes' response to 'Were the thin capitalisation rules applicable to you?' on the IDS then a 'Yes' response is required at 'Were the thin capitalisation or debt deduction creation rules applicable to you?' on the company return form
       
        Legacy Rule Format:
        IF ([CTR196] = FALSE) AND ([IDS220] = TRUE)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR196 = CTR:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        ^IDS220 = IDS:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428405()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.CTR196.GetValueOrDefault() && ChildDocument.IDS220.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check response to Thin Capitalisation question",
                    LongDescription = @"If there is a 'Yes' response to 'Were the thin capitalisation rules applicable to you?' on the IDS then a 'Yes' response is required at 'Were the thin capitalisation or debt deduction creation rules applicable to you?' on the company return form",
                    Location = "/xbrli:xbrl/tns:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428405" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR196", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR196) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS220", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS220) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428405

        #region VR.ATO.CTR.428406

        /*  VR.ATO.CTR.428406
        Where there is a Royalties deduction amount present on the IDS, there must also be an amount at Royalty expenses overseas, or Gross payments subject to foreign resident withholding (excluding capital gains) on the company return form.

        Legacy Rule Format:
        IF ([CTR57] = NULL OR [CTR57] = 0) AND (([CTR366] = NULL OR [CTR366] = 0) AND ([IDS22] > 0))
            RETURN VALIDATION MESSAGE
        ENDIF
        
        Data Elements:

        ^CTR57 = CTR:RP:Remuneration.PaymentToForeignResidentGross.Amount
        ^CTR366 = CTR:RP.JFOREIGN:Expense.Royalties.Amount
        ^IDS22 = IDS:RP.Royalties:Expense.DeductionsTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428406()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (!ParentDocument.CTR57.HasValue || ParentDocument.CTR57.Value == 0) &&
                            ((!ParentDocument.CTR366.HasValue || ParentDocument.CTR366.Value == 0) && ChildDocument.IDS22 > 0);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Royalty expenses overseas or Foreign resident withholding expenses (excluding capital gains) required.",
                    LongDescription = @"If a Royalties deduction amount has been completed on the IDS then Foreign resident withholding (excluding capital gains) expenses or Royalty expenses overseas must be greater than zero on the company return form.",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428406" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR366", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR366) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS22", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS22) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428406

        #region VR.ATO.CTR.428408

        /*  VR.ATO.CTR.428408
        If there is an amount at question 6a, label C on the IDS it must be less than or equal to the amount at question 6 label U on the company return form

        Legacy Rule Format:
        IF ([CTR366] >  0) AND ([CTR366] < [IDS22])
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR366 = CTR:RP.JFOREIGN:Expense.Royalties.Amount
        ^IDS22 = IDS:RP.Royalties:Expense.DeductionsTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428408()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR366.GetValueOrDefault() > 0 &&
                            ParentDocument.CTR366.GetValueOrDefault() < ChildDocument.IDS22.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Royalties deductions incorrect",
                    LongDescription = @"If Royalties deductions has been completed on the IDS then it must be less than or equal to Royalty expenses overseas on the Company return form",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount[@contextRef='" + ParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428408" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR366", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR366) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS22", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS22) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428408

        #region VR.ATO.CTR.428409

        /*  VR.ATO.CTR.428409
        Where there is a Rent/leasing expenditure amount present on the IDS, there must also be an amount at either Lease expenses overseas or Rent expenses on the company return form

        Legacy Rule Format:
        IF (([CTR78] = NULL OR [CTR78] = 0) AND ([CTR79] = NULL OR [CTR79] = 0)) AND ([IDS31] > 0) 
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR78 = CTR:RP:Expense.Operating.Lease.Amount
        ^CTR79 = CTR:RP:Expense.Operating.Rent.Amount
        ^IDS31 = IDS:RP:Expense.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428409()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (!ParentDocument.CTR78.HasValue || ParentDocument.CTR78.Value == 0) &&
                             (!ParentDocument.CTR79.HasValue || ParentDocument.CTR79.Value == 0) &&
                             (ChildDocument.IDS31.HasValue && ChildDocument.IDS31 > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Lease expenses overseas or Rent expenses are required",
                    LongDescription = @"Where there is a Rent/leasing expenditure amount present on the IDS, there must also be an amount at either Lease expenses overseas or Rent expenses on the company return form",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Lease.Amount[@contextRef='" + ParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428409" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR78", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR78) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR79", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR79) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS31", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS31) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428409

        #region VR.ATO.CTR.428410

        /*  VR.ATO.CTR.428410
        Where there is a Rent/leasing revenue amount present on the IDS, there must also be an amount at Gross rent and other leasing and hiring income amount on the company return form.

        Legacy Rule Format:
        IF ([CTR65] = NULL OR [CTR65] = 0) AND ([IDS32] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR65 = CTR:RP:Income.RentLeasingHiringGross.Amount
        ^IDS32 = IDS:RP:Revenue.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428410()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (!ParentDocument.CTR65.HasValue || ParentDocument.CTR65.Value == 0) && (ChildDocument.IDS32.HasValue && ChildDocument.IDS32 > 0);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent and other leasing and hiring income required",
                    LongDescription = @"If Rent/leasing revenue has been completed on the IDS then Gross rent and other leasing and hiring income must be greater than zero on the company return form",
                    Location = "/xbrli:xbrl/tns:Income.RentLeasingHiringGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428410" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR65", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR65) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS32", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS32) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428410

        #region VR.ATO.CTR.428412

        /*  VR.ATO.CTR.428412
        Where there is an interest revenue amount present on the IDS, there must also be an amount at Gross Interest or Gross payments subject to foreign resident withholding (excluding capital gains) on the company return form.

        Legacy Rule Format:
        IF (([CTR57] = NULL OR [CTR57] = 0) AND ([CTR64] = NULL OR [CTR64] = 0)) AND ([IDS103] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR57 = CTR:RP:Remuneration.PaymentToForeignResidentGross.Amount
        ^CTR64 = CTR:RP:Income.Interest.Gross.Amount
        ^IDS103 = IDS:RP.OtherInterest:Revenue.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428412()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ((!ParentDocument.CTR57.HasValue || ParentDocument.CTR57.Value == 0) && (!ParentDocument.CTR64.HasValue || ParentDocument.CTR64.Value == 0)) && ChildDocument.IDS103 > 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438070",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest or Gross payments subject to foreign resident withholding (excluding capital gains) required.",
                    LongDescription = @"If Interest revenue has been completed on the IDS then Gross interest or Gross payments subject to foreign resident withholding (excluding capital gains) must be greater than zero on the company return form.",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428412" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR64", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR64) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS103", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS103) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428412

        #region VR.ATO.CTR.428413

        /*  VR.ATO.CTR.428413
        The sum of amounts at attributable income of controlled foreign companies and controlled foreign trusts on the IDS should be equal to or greater than the sum of amounts at Listed country and Unlisted country and equal to or less than the sum of amounts at Listed country, Unlisted country and Transferor trust on the company return form.
       
        Legacy Rule Format:
        IF ([CTR156] > 0 OR [CTR158] > 0 OR [CTR159] > 0) AND ((([IDS191] + [IDS192] + [IDS193]) < ([CTR156] + [CTR158])) OR (([IDS191] + [IDS192] + [IDS193]) > ([CTR156] + [CTR158] + [CTR159])))
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR156 = CTR:RP:Income.InternationalDealings.Attributed.Amount
        ^CTR158 = CTR:RP:Income.InternationalDealings.Attributed.Amount
        ^CTR159 = CTR:RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
        ^IDS191 = IDS:RP:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
        ^IDS192 = IDS:RP:Income.ControlledForeignCompaniesChangeOfResidenceIncome.Amount
        ^IDS193 = IDS:RP:Income.ControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428413()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (ParentDocument.CTR156.GetValueOrDefault() > 0 || ParentDocument.CTR158.GetValueOrDefault() > 0 || ParentDocument.CTR159.GetValueOrDefault() > 0) &&
                          (
                            (ChildDocument.IDS191.GetValueOrDefault() + ChildDocument.IDS192.GetValueOrDefault() + ChildDocument.IDS193.GetValueOrDefault()) <
                            (ParentDocument.CTR156.GetValueOrDefault() + ParentDocument.CTR158.GetValueOrDefault()) ||
                            (ChildDocument.IDS191.GetValueOrDefault() + ChildDocument.IDS192.GetValueOrDefault() + ChildDocument.IDS193.GetValueOrDefault()) >
                            (ParentDocument.CTR156.GetValueOrDefault() + ParentDocument.CTR158.GetValueOrDefault() + ParentDocument.CTR159.GetValueOrDefault())
                           );

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430222",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income incorrect",
                    LongDescription = @"The sum of amounts at attributable income of controlled foreign companies and controlled foreign trusts on the International Dealing Schedule should be equal to or greater than the sum of amounts at Listed country, Unlisted country and equal to or less than the sum of amounts at Listed country,Unlisted country and Transferor trust on the company return form.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428413" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR156) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR158", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR158) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR159", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR159) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS191", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS191) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS192", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS192) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS193", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS193) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428413

        #region VR.ATO.CTR.428414

        /*  VR.ATO.CTR.428414
        CFCs attributable income - Listed countries amount on the International Dealings Schedule (IDS) must be equal to or less than Attributed foreign income - Listed country on the parent Company Tax Return (CTR)
       
        Legacy Rule Format:
        IF ([CTR156] > 0) AND ([IDS188] > [CTR156])
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR156 = CTR:RP.CFCLC:Income.InternationalDealings.Attributed.Amount
        ^IDS188 = IDS:RP.CFCLC:Income.ControlledForeignCompaniesAttributableIncome.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428414()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR156.GetValueOrDefault() > 0 && ChildDocument.IDS188.GetValueOrDefault() > ParentDocument.CTR156.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income amount is incorrect",
                    LongDescription = @"CFCs attributable income - Listed countries amount on the International Dealings Schedule (IDS) must be equal to or less than Attributed foreign income - Listed country on the parent Company Tax Return (CTR)",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount[@contextRef='" + ParentDocument.RPCFCLCId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428414" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR156) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS188", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS188) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428414

        #region VR.ATO.CTR.428415

        /*  VR.ATO.CTR.428415
        Amounts are showing at attributable income of controlled foreign companies and controlled foreign trusts question 22 on IDS but there is no 'yes' response to the question concerning whether you have an overseas branch or a direct or indirect interest in a foreign trust, foreign company, controlled foreign entity or transferor trust on the company return form (overseas interests label Z).

        Legacy Rule Format:
        IF [CTR195] = FALSE AND ([IDS191] > 0 OR [IDS192] > 0 OR [IDS193] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR195 = CTR:RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
        ^IDS191 = IDS:RP:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
        ^IDS192 = IDS:RP:Income.ControlledForeignCompaniesChangeOfResidenceIncome.Amount
        ^IDS193 = IDS:RP:Income.ControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428415()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.CTR195.GetValueOrDefault() &&
                             (ChildDocument.IDS191.GetValueOrDefault() > 0 || ChildDocument.IDS192.GetValueOrDefault() > 0 || ChildDocument.IDS193.GetValueOrDefault() > 0);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428415",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Response required at overseas interests question",
                    LongDescription = @"If assessable income of controlled foreign companies and controlled foreign trusts under sections 456, 457 or 459A has been completed on the IDS then the response to the Overseas interests question on the company return form must be 'True'.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428415" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR195", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR195) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS191", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS191) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS192", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS192) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS193", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS193) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428415

        #region VR.ATO.CTR.428416

        /*  VR.ATO.CTR.428416
        There is an amount at Listed country or Unlisted country or Transferor trust on the company tax return form and the number of controlled foreign companies or trusts on the IDS has not been supplied.

        Legacy Rule Format:
        IF ([CTR156] > 0 OR [CTR158] > 0 OR [CTR159] > 0) AND ([IDS185] = NULL AND [IDS186] = NULL AND [IDS187] = NULL)
            RETURN VALIDATION MESSAGE
        ENDIF

         Data Elements:

        ^CTR156 = CTR:RP:Income.InternationalDealings.Attributed.Amount
        ^CTR158 = CTR:RP:Income.InternationalDealings.Attributed.Amount
        ^CTR159 = CTR:RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
        ^IDS185 = IDS:RP:Miscellaneous.ControlledForeignCompaniesTrustsInterest.Count
        ^IDS186 = IDS:RP:Miscellaneous.ControlledForeignCompaniesTrustsInterest.Count
        ^IDS187 = IDS:RP:Miscellaneous.ControlledForeignCompaniesTrustsInterest.Count
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428416()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (ParentDocument.CTR156.GetValueOrDefault() > 0 || ParentDocument.CTR158.GetValueOrDefault() > 0 || ParentDocument.CTR159.GetValueOrDefault() > 0) &&
                             (!ChildDocument.IDS185.HasValue && !ChildDocument.IDS186.HasValue && !ChildDocument.IDS187.HasValue);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of controlled foreign companies or trusts on International Dealing Schedule is required",
                    LongDescription = @"Attributed foreign income has been completed on the company return form and the number of controlled foreign companies or trusts on the IDS has not been supplied.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount[@contextRef='" + ParentDocument.RPCFCLCId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428416" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR156) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR158", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR158) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR159", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR159) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS185", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS185) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS186", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS186) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS187", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS187) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428416

        #region VR.ATO.CTR.428417

        /*  VR.ATO.CTR.428417
        If there is a yes response to question 'Did you have dealings with international related parties located in specified countries?' within the International Dealings Schedule (IDS), there must also be a yes response to 'Overseas transactions - Transactions with specified countries' on the company return form.

        Legacy Rule Format:
        IF [CTR198] = FALSE AND [IDS9] = TRUE
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR198 = CTR:RP:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator
        ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428417()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.CTR198.GetValueOrDefault() && ChildDocument.IDS9.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Transactions with specified countries question incorrect",
                    LongDescription = @"If there is a 'True' response at 'Did you have dealings with international related parties located in specified countries?' on the International Dealings Schedule (IDS), then the response at 'Overseas transactions - Transactions with specified countries' on the company return must be 'True'.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428417" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR198", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR198) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS9", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS9) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428417

        #region VR.ATO.CTR.428418

        /*  VR.ATO.CTR.428418
        An amount is showing at question 5 Label C on IDS and there is no amount at cost of sales (Item 6 label A on the company return form.

        Legacy Rule Format:
        IF [CTR73] = NULL AND ([IDS17] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR73 = CTR:RP:Expense.Operating.CostOfSales.Amount
        ^IDS17 = IDS:RP:Expense.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428418()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.CTR73.HasValue && ChildDocument.IDS17 > 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428418",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Cost of sales required",
                    LongDescription = @"If an expenditure amount has been completed at 'Tangible property of a revenue nature including trading stock and raw materials' on the IDS then Cost of sales must be completed on the company return form.",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.CostOfSales.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428418" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR73", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR73) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS17", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS17) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428418

        #region VR.ATO.CTR.428419

        /*  VR.ATO.CTR.428419
        An amount is showing at question 5 Label D on IDS and there is no amount at Other sales of goods and services (Item 6 label C on the company return form

        Legacy Rule Format:
        IF ([CTR59] = NULL OR [CTR59] = 0) AND [IDS18] > 0
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR59 = CTR:RP:Income.SaleOfGoodsAndServices.Other.Amount
        ^IDS18 = IDS:RP:Revenue.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428419()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (!ParentDocument.CTR59.HasValue || ParentDocument.CTR59.Value == 0) && ChildDocument.IDS18 > 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428419",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other sales of goods and services required",
                    LongDescription = @"If a revenue amount has been completed at 'Tangible property of a revenue nature including trading stock and raw materials' on the IDS then Other sales of goods and services must be greater than zero on the company return form.",
                    Location = "/xbrli:xbrl/tns:Income.SaleOfGoodsAndServices.Other.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428419" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR59", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR59) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS18", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS18) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428419

        #region VR.ATO.CTR.428420

        /*  VR.ATO.CTR.428420
        An amount is showing at debt deductions in earning non-assessable non-exempt foreign income claimed under s25-90 or 230-15(3) and there is no amount at Interest expenses within Australia or Interest expenses overseas on the company return.

        Legacy Rule Format:
        IF ([CTR80] = NULL OR [CTR80] = 0 ) AND ([CTR81] = NULL OR [CTR81] = 0 ) AND (COUNT(SCHEDULE = "IDS") = 1) AND ([IDS207] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR80 = CTR:RP:Expense.Operating.Interest.Amount
        ^CTR81 = CTR:RP:Expense.Operating.Interest.Amount
        ^IDS207 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncomeTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428420()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (!ParentDocument.CTR80.HasValue || ParentDocument.CTR80.Value == 0) &&
                           (!ParentDocument.CTR81.HasValue || ParentDocument.CTR81.Value == 0) &&
                           (ChildDocument.IDS207.HasValue && ChildDocument.IDS207 > 0);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest expenses required",
                    LongDescription = @"If Debt deductions in earning non-assessable non-exempt foreign income claimed under s25-90 or 230-15(3) has been completed on the International dealings schedule (IDS) then Interest expenses within Australia or Interest expenses overseas must be greater than zero on the company return",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Interest.Amount[@contextRef='" + ParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428420" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR80", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR80) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR81", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR81) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS207", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS207) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428420

        #region VR.ATO.CTR.428421

        /*  VR.ATO.CTR.428421
        An amount is showing at Total amount of your share of net income/distribution of profit (question 29 Label C on IDS) and there is no amount at Item 7 other assessable income, Label B on the company return form.

        Legacy Rule Format:
        IF ([CTR95] = NULL OR [CTR95] = 0) AND ([IDS219] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF
        
        Data Elements:

        ^CTR95 = CTR:RP:Income.AssessableOther.Amount
        ^IDS219 = IDS:RP:Assets.Investment.ForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428421()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (!ParentDocument.CTR95.HasValue || ParentDocument.CTR95.Value == 0) &&
                             (ChildDocument.IDS219.HasValue && ChildDocument.IDS219 > 0);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428421",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other assessable income required",
                    LongDescription = @"If Total amount of your share of net income/distribution of profit has been completed on the IDS then Other assessable income must be completed on the company return form.",
                    Location = "/xbrli:xbrl/tns:Income.AssessableOther.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428421" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR95", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR95) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS219", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS219) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428421

        #region VR.ATO.CTR.428422

        /*  VR.ATO.CTR.428422
        An amount is showing at question 26 on IDS and there is no amount at net capital gain (Item 7 label A on the company return form, Item 21 label A on the partnership and trust return forms).

        Legacy Rule Format:
        IF (([CTR91] = NULL) AND ([IDS209] > 0))
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR91 = CTR:RP:Income.CapitalGainsNet.Amount
        ^IDS209 = IDS:RP:CapitalGainsTax.ActiveForeignBusinessCapitalGains.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428422()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.CTR91.HasValue && ChildDocument.IDS209 > 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428422",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gain required",
                    LongDescription = @"If Capital gain amounts has been completed on the IDS then Net capital gain must be completed on the main return.",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount[@contextRef='" + ParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428422" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR91", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR91) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS209", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS209) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428422

        #region VR.ATO.CTR.428426

        /*  VR.ATO.CTR.428426
        If the response at "Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?" is 'true' , then the status of the company at Non-resident with permanent establishment on the company return.

        Legacy Rule Format:
        IF ([CTR51] <> "NonResidentPermEstab") AND ([IDS221] = TRUE)
            RETURN VALIDATION MESSAGE
        ENDIF
 
        Data Elements:

        ^CTR51 = CTR:RP:Residency.TaxPurposesOrganisationStatus.Code
        ^IDS221 = IDS:RP:Liabilities.ThinCapitalisation.AustralianResidentCompanyElected.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428426()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR51 != @"NonResidentPermEstab" && ChildDocument.IDS221.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Status of company is incorrect",
                    LongDescription = @"If there is a 'True' response to ""Has an Australian resident company elected under subdivision 820-FB of the ITAA 1997 to treat your qualifying Australian branch operations as part of a consolidated group, MEC group or a single company for thin capitalisation purposes?"" on the IDS then the residency status at Status of company must be Non-resident with permanent establishment on the company return",
                    Location = "/xbrli:xbrl/tns:Residency.TaxPurposesOrganisationStatus.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428426" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR51", Value = ParentDocument.CTR51 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS221", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS221) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428426

        #region VR.ATO.CTR.428427

        /*  VR.ATO.CTR.428427
        Amounts are showing at question 26 on IDS but there is no 'yes' response to the question concerning whether you have a CGT event during the year (Item 7 label G on the company return form; item 21 label G on the partnership and trust return forms).
      
        Legacy Rule Format:
        IF ([CTR89] = FALSE) AND ([IDS208] = TRUE)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR89 = CTR:RP:CapitalGainsTax.Event.Indicator
        ^IDS208 = IDS:RP:CapitalGainsTax.Event.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428427()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR89.GetValueOrDefault() == false && ChildDocument.IDS208.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check response to CGT event question",
                    LongDescription = @"If there is a 'True' response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?' On the IDS then a 'True' response is required at 'Did you have a CGT event during the year?' on the company return.",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.Event.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428427" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR89", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR89) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS208", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS208) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428427

        #region VR.ATO.CTR.438050

        /*  VR.ATO.CTR.438050
        If there is an amount at either 'Total TOFA gains' or 'Total TOFA Losses' that indicates you made a gain or loss from a financial arrangement subject to the TOFA rules, then there must be a 'True' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' on the IDS

        Legacy Rule Format:
        IF ([CTR310] > 0 OR [CTR311] > 0) 
        AND (COUNT(SCHEDULE = "IDS") = 1 AND [IDS177] <> TRUE)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR310 = CTR:RP:Income.GainsTotal.Amount
        ^CTR311 = IDS:RP:Expense.LossesTotal.Amount
        ^IDS177 = IDS:RP:InternationalDealings.TaxationOfFinancialArrangementsRulesApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438050()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (ParentDocument.CTR310.GetValueOrDefault() > 0 || ParentDocument.CTR311.GetValueOrDefault() > 0) &&
                             !ChildDocument.IDS177.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438050",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Response to 'Are you subject to the TOFA rules contained in Division 230 of the ITAA 1997' is incorrect",
                    LongDescription = @"If there is an amount at either 'Total TOFA gains' or 'Total TOFA Losses' that indicates you made a gain or loss from a financial arrangement subject to the TOFA rules, then there must be a 'True' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' on the IDS",
                    Location = "/xbrli:xbrl/tns:Income.GainsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.438050" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR310", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR310) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR311", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR311) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS177", Value = CTR2026Validator.GetValueOrEmpty(ChildDocument.IDS177) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.438050

        #region VR.ATO.IDS.440670

        /* VR.ATO.IDS.440670
        Your supplied TFN does not match the TFN supplied on the form it was submitted with

        Technical Rule Format:
        ^CTR TFN = CTR:RP:Identifiers.TaxFileNumber.Identifier
        ^IDS500 = IDS:RP:Identifiers.TaxFileNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440670()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.RPIdentifierTFN.Equals(ChildDocument.IDS500);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440670" } }
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR TFN", Value = ParentDocument.RPIdentifierTFN });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IEE50", Value = ChildDocument.IDS500 });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.IDS.440670
    }
}
