using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using Au.Gov.Sbr.Xbrl.Document;
using DataContracts;
using Au.Gov.Sbr.Xbrl.Metadata;

namespace Ato.EN.IntegrationServices.CodeGenerationDIS
{
    public class DIS2018Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public DIS2018 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public DIS2018 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            DIS2018 report;
            report = new DIS2018();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrences being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) ));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.DIS22 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.DIS42 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.DIS23 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                        report.OrganisationNameDetailsCollectionExists = true;
                        report.OrganisationNameDetailsCollectionCount = sbrOrganisationNameDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_RPCollection;
                        sbrOrganisationNameDetailsCollection_MN_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_RPCollection != null && sbrOrganisationNameDetailsCollection_MN_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_RP;
                            sbrOrganisationNameDetailsCollection_MN_RP = sbrOrganisationNameDetailsCollection_MN_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_RPTupleElementMap = sbrOrganisationNameDetailsCollection_MN_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_MN_RPExists = true;
                            report.OrganisationNameDetailsCollection_MN_RPCount = sbrOrganisationNameDetailsCollection_MN_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS34 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS24 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName", out currentValue))
                        sbrPersonUnstructuredNameCollection = currentValue.Occurrences;
                    else
                        sbrPersonUnstructuredNameCollection = null;
            
                    #region sbrPersonUnstructuredNameCollection
            
                    if (sbrPersonUnstructuredNameCollection != null)
                    {
                        report.PersonUnstructuredNameCollectionExists = true;
                        report.PersonUnstructuredNameCollectionCount = sbrPersonUnstructuredNameCollection.Count();
                
                        IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection_Contact_RPCollection;
                        sbrPersonUnstructuredNameCollection_Contact_RPCollection = sbrPersonUnstructuredNameCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "Contact" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrPersonUnstructuredNameCollection_Contact_RPCollection != null && sbrPersonUnstructuredNameCollection_Contact_RPCollection.Count() > 0)
                        {
                            SBRElement sbrPersonUnstructuredNameCollection_Contact_RP;
                            sbrPersonUnstructuredNameCollection_Contact_RP = sbrPersonUnstructuredNameCollection_Contact_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrPersonUnstructuredNameCollection_Contact_RPTupleElementMap;
                            sbrPersonUnstructuredNameCollection_Contact_RPTupleElementMap = sbrPersonUnstructuredNameCollection_Contact_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.PersonUnstructuredNameCollection_Contact_RPExists = true;
                            report.PersonUnstructuredNameCollection_Contact_RPCount = sbrPersonUnstructuredNameCollection_Contact_RPCollection.Count();
                            report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_RP.OccurrenceIndex + 1;
                            report.PersonUnstructuredNameCollection_Contact_RPLastOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS36 = currentValue.Value;
                            }
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS25 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrPersonUnstructuredNameCollection
                    if (validationMode) ValidateRPPersonUnstructuredNameCollection(errors, sbrPersonUnstructuredNameCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;
            
                    #region sbrElectronicContactTelephoneCollection
            
                    if (sbrElectronicContactTelephoneCollection != null)
                    {
                        report.ElectronicContactTelephoneCollectionExists = true;
                        report.ElectronicContactTelephoneCollectionCount = sbrElectronicContactTelephoneCollection.Count();
                
                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_RPCollection;
                        sbrElectronicContactTelephoneCollection_03_RPCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_RPCollection != null && sbrElectronicContactTelephoneCollection_03_RPCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_RP;
                            sbrElectronicContactTelephoneCollection_03_RP = sbrElectronicContactTelephoneCollection_03_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_RPTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_RPTupleElementMap = sbrElectronicContactTelephoneCollection_03_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ElectronicContactTelephoneCollection_03_RPExists = true;
                            report.ElectronicContactTelephoneCollection_03_RPCount = sbrElectronicContactTelephoneCollection_03_RPCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_RPOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_RP.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_RPLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS32 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS33 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS26 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS27 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateRPElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrAddressDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails", out currentValue))
                        sbrAddressDetailsCollection = currentValue.Occurrences;
                    else
                        sbrAddressDetailsCollection = null;
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                        report.AddressDetailsCollectionExists = true;
                        report.AddressDetailsCollectionCount = sbrAddressDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_BUS_C_RPCollection;
                        sbrAddressDetailsCollection_BUS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "BUS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_BUS_C_RPCollection != null && sbrAddressDetailsCollection_BUS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_BUS_C_RP;
                            sbrAddressDetailsCollection_BUS_C_RP = sbrAddressDetailsCollection_BUS_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_BUS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_BUS_C_RPTupleElementMap = sbrAddressDetailsCollection_BUS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_BUS_C_RPExists = true;
                            report.AddressDetailsCollection_BUS_C_RPCount = sbrAddressDetailsCollection_BUS_C_RPCollection.Count();
                            report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_BUS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_BUS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_BUS_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS45 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS46 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS47 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS48 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS49 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS50 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS51 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS52 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS56 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS57 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.CountryName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS58 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS59 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_C_RPCollection;
                        sbrAddressDetailsCollection_POS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_C_RPCollection != null && sbrAddressDetailsCollection_POS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_C_RP;
                            sbrAddressDetailsCollection_POS_C_RP = sbrAddressDetailsCollection_POS_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_POS_C_RPTupleElementMap = sbrAddressDetailsCollection_POS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_POS_C_RPExists = true;
                            report.AddressDetailsCollection_POS_C_RPCount = sbrAddressDetailsCollection_POS_C_RPCollection.Count();
                            report.AddressDetailsCollection_POS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS60 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS61 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS62 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS64 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS65 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS66 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS67 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS69 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS70 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS71 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.CountryName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS72 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS73 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection1(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrInvestorCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Investor", out currentValue))
                        sbrInvestorCollection = currentValue.Occurrences;
                    else
                        sbrInvestorCollection = null;
            
                    #region sbrInvestorCollection
            
                    if (sbrInvestorCollection != null)
                    {
                        report.InvestorCollectionExists = true;
                        report.InvestorCollectionCount = sbrInvestorCollection.Count();
                        if (sbrInvestorCollection != null && sbrInvestorCollection.Count() > 0)
                        {
                            report.InvestorCollection = new List<DIS2018.Investor>();
                            for (int tupleIndex = 0; tupleIndex < sbrInvestorCollection.Count(); tupleIndex++)
                            {
                                SBRElement sbrInvestor;
                                sbrInvestor = sbrInvestorCollection.ElementAt(tupleIndex);
                
                                IDictionary<string, SBRElement> sbrInvestorTupleElementMap;
                                sbrInvestorTupleElementMap = sbrInvestor.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                                DIS2018.Investor investor = new DIS2018.Investor();
                                report.InvestorCollection.Add(investor);
                                investor.OccurrenceIndex = tupleIndex + 1;
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Identifiers.TaxFileNumber.Identifier", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS1 = currentValue.Value;
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS2 = currentValue.Value;
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonDemographicDetails.Birth.Date", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS13 = DateTime.Parse(currentValue.Value);
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS16 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Equity.Dividends.FrankedPaid.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS17 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Expense.Interest.Paid.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS20 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Party.Type.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS15 = currentValue.Value;
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonDemographicDetails.Deceased.Indicator", out currentValue))
                                {
                                    if (!currentValue.IsNil) investor.DIS44 = ToBoolean(currentValue.Value);
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Equity.Dividends.UnfrankedPaid.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}ConduitForeignIncome") || p.Domain.EndsWith(":ConduitForeignIncome"))>-1)
                                                {
                                                    investor.DIS19 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Equity.Dividends.UnfrankedPaid.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}NonConduitForeignIncome") || p.Domain.EndsWith(":NonConduitForeignIncome"))>-1)
                                                {
                                                    investor.DIS18 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                    
                                IEnumerable<SBRElement> sbrInvestor_PersonNameDetailsCollection;
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonNameDetails", out currentValue))
                                    sbrInvestor_PersonNameDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrInvestor_PersonNameDetailsCollection = null;
                        
                                #region sbrInvestor_PersonNameDetailsCollection
                        
                                if (sbrInvestor_PersonNameDetailsCollection != null)
                                {
                                    investor.Investor_PersonNameDetailsCollectionExists = true;
                                    investor.Investor_PersonNameDetailsCollectionCount = sbrInvestor_PersonNameDetailsCollection.Count();
                                    SBRElement sbrInvestor_PersonNameDetails;
                                    sbrInvestor_PersonNameDetails = sbrInvestor_PersonNameDetailsCollection.ElementAt(0);
                            
                                    IDictionary<string, SBRElement> sbrInvestor_PersonNameDetailsTupleElementMap;
                                    sbrInvestor_PersonNameDetailsTupleElementMap = sbrInvestor_PersonNameDetails.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                    if (sbrInvestor_PersonNameDetailsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonNameDetails.FamilyName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) investor.DIS4 = currentValue.Value;
                                    }
                            
                                    if (sbrInvestor_PersonNameDetailsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonNameDetails.GivenName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) investor.DIS5 = currentValue.Value;
                                    }
                            
                                    if (sbrInvestor_PersonNameDetailsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonNameDetails.OtherGivenName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) investor.DIS6 = currentValue.Value;
                                    }
                            
                                    if (sbrInvestor_PersonNameDetailsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonNameDetails.Position.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) investor.DIS55 = currentValue.Value;
                                    }
                                } 
                                #endregion End of Tuple sbrInvestor_PersonNameDetailsCollection
                    
                                IEnumerable<SBRElement> sbrInvestor_OrganisationNameDetailsCollection;
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails", out currentValue))
                                    sbrInvestor_OrganisationNameDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrInvestor_OrganisationNameDetailsCollection = null;
                        
                                #region sbrInvestor_OrganisationNameDetailsCollection
                        
                                if (sbrInvestor_OrganisationNameDetailsCollection != null)
                                {
                                    investor.Investor_OrganisationNameDetailsCollectionExists = true;
                                    investor.Investor_OrganisationNameDetailsCollectionCount = sbrInvestor_OrganisationNameDetailsCollection.Count();
                            
                                    IEnumerable<SBRElement> sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection;
                                    sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection = sbrInvestor_OrganisationNameDetailsCollection.Where(
                                        ce => ce.ChildElements != null && 
                                        ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection != null && sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrInvestor_OrganisationNameDetailsCollection_MN_RP;
                                        sbrInvestor_OrganisationNameDetailsCollection_MN_RP = sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection.ElementAt(0);
                            
                                        IDictionary<string, SBRElement> sbrInvestor_OrganisationNameDetailsCollection_MN_RPTupleElementMap;
                                        sbrInvestor_OrganisationNameDetailsCollection_MN_RPTupleElementMap = sbrInvestor_OrganisationNameDetailsCollection_MN_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                        investor.Investor_OrganisationNameDetailsCollection_MN_RPExists = true;
                                        investor.Investor_OrganisationNameDetailsCollection_MN_RPCount = sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection.Count();
                                        investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex = sbrInvestor_OrganisationNameDetailsCollection_MN_RP.OccurrenceIndex + 1;
                                        investor.Investor_OrganisationNameDetailsCollection_MN_RPLastOccurrenceIndex = sbrInvestor_OrganisationNameDetailsCollection_MN_RPCollection.Last().OccurrenceIndex + 1;
                                
                                        if (sbrInvestor_OrganisationNameDetailsCollection_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS3 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_OrganisationNameDetailsCollection_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS53 = currentValue.Value;
                                        }
                                    }  
                                } 
                                #endregion End of Tuple sbrInvestor_OrganisationNameDetailsCollection
                                if (validationMode) ValidateRPInvestor_OrganisationNameDetailsCollection(errors, sbrInvestor_OrganisationNameDetailsCollection, sbrRPContexts, report);
                        
                    
                                IEnumerable<SBRElement> sbrInvestor_AddressDetailsCollection;
                                if (sbrInvestorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails", out currentValue))
                                    sbrInvestor_AddressDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrInvestor_AddressDetailsCollection = null;
                        
                                #region sbrInvestor_AddressDetailsCollection
                        
                                if (sbrInvestor_AddressDetailsCollection != null)
                                {
                                    investor.Investor_AddressDetailsCollectionExists = true;
                                    investor.Investor_AddressDetailsCollectionCount = sbrInvestor_AddressDetailsCollection.Count();
                            
                                    IEnumerable<SBRElement> sbrInvestor_AddressDetailsCollection_POS_C_RPCollection;
                                    sbrInvestor_AddressDetailsCollection_POS_C_RPCollection = sbrInvestor_AddressDetailsCollection.Where(
                                        ce => ce.ChildElements != null && 
                                        ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                                        ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrInvestor_AddressDetailsCollection_POS_C_RPCollection != null && sbrInvestor_AddressDetailsCollection_POS_C_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrInvestor_AddressDetailsCollection_POS_C_RP;
                                        sbrInvestor_AddressDetailsCollection_POS_C_RP = sbrInvestor_AddressDetailsCollection_POS_C_RPCollection.ElementAt(0);
                            
                                        IDictionary<string, SBRElement> sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap;
                                        sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap = sbrInvestor_AddressDetailsCollection_POS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                        investor.Investor_AddressDetailsCollection_POS_C_RPExists = true;
                                        investor.Investor_AddressDetailsCollection_POS_C_RPCount = sbrInvestor_AddressDetailsCollection_POS_C_RPCollection.Count();
                                        investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex = sbrInvestor_AddressDetailsCollection_POS_C_RP.OccurrenceIndex + 1;
                                        investor.Investor_AddressDetailsCollection_POS_C_RPLastOccurrenceIndex = sbrInvestor_AddressDetailsCollection_POS_C_RPCollection.Last().OccurrenceIndex + 1;
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.OverseasAddress.Indicator", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS41 = ToBoolean(currentValue.Value);
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Usage.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS37 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Currency.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS54 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line1.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS7 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line2.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS8 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line3.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS38 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line4.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS39 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.LocalityName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS9 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Postcode.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS11 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.StateOrTerritory.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS10 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.CountryName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS12 = currentValue.Value;
                                        }
                                
                                        if (sbrInvestor_AddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Country.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) investor.DIS40 = currentValue.Value;
                                        }
                                    }  
                                } 
                                #endregion End of Tuple sbrInvestor_AddressDetailsCollection
                                if (validationMode) ValidateRPInvestor_AddressDetailsCollection(errors, sbrInvestor_AddressDetailsCollection, sbrRPContexts, report);
                        
                    } 
                    #endregion End of Tuple sbrInvestorCollection
                        } 
                    } 
        
                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;
            
                    #region sbrDeclarationCollection
            
                    if (sbrDeclarationCollection != null)
                    {
                        report.DeclarationCollectionExists = true;
                        report.DeclarationCollectionCount = sbrDeclarationCollection.Count();
                
                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPCollection;
                        sbrDeclarationCollection_TrueAndCorrect_RPCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_RPCollection != null && sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_RP;
                            sbrDeclarationCollection_TrueAndCorrect_RP = sbrDeclarationCollection_TrueAndCorrect_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.DeclarationCollection_TrueAndCorrect_RPExists = true;
                            report.DeclarationCollection_TrueAndCorrect_RPCount = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RP.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_RPLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS74 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS75 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS76 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS77 = DateTime.Parse(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS78 = currentValue.Value;
                            }
                
                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;
                    
                            #region sbrDeclaration_PersonUnstructuredNameCollection
                    
                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                                report.Declaration_PersonUnstructuredNameCollectionExists = true;
                                report.Declaration_PersonUnstructuredNameCollectionCount = sbrDeclaration_PersonUnstructuredNameCollection.Count();
                        
                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null && 
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrRPContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.ElementAt(0);
                        
                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                        
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Last().OccurrenceIndex + 1;
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.DIS79 = currentValue.Value;
                                    }
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.DIS80 = currentValue.Value;
                                    }
                                }  
                            } 
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateRPDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrRPContexts, report);
                    
                        }  
                    } 
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateRPDeclarationCollection(errors, sbrDeclarationCollection, sbrRPContexts, report);
            
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPCFIContext
            IEnumerable<SBRContext> sbrRPCFIContexts = null;
            SBRContext sbrRPCFIContext = null;
    
            sbrRPCFIContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssessableForeignIncomeClassificationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ConduitForeignIncome")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCFIElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCFIContexts != null && sbrRPCFIContexts.Count() > 0)
            {
                sbrRPCFIContext = sbrRPCFIContexts.First();
        
                report.RPCFIId = sbrRPCFIContext.Id;
                report.RPCFIIdentifier = sbrRPCFIContext.EntityIdentifier;
                report.RPCFIIdentifierScheme = sbrRPCFIContext.EntityScheme;
        
                report.RPCFICount = sbrRPCFIContexts.Count();
                report.RPCFIExists = true;
                report.RPCFIOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCFIContext) + 1;
                report.RPCFILastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCFIContexts.Last()) + 1;
        

                if (report.RPCFIIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCFIIdentifierTFN = report.RPCFIIdentifier;
    
                report.RPCFIStartDate = ((SBRDurationPeriod)sbrRPCFIContext.Period).DurationStartAsDateObject;
                report.RPCFIEndDate = ((SBRDurationPeriod)sbrRPCFIContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPCFIElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCFIContexts.Count(); i++)
                {
                    sbrRPCFIContext = sbrRPCFIContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCFIContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCFIElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCFIElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCFIElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                } 
            } 
            #endregion End of Context sbrRPCFIContext
    
            #region sbrRPNONCFIContext
            IEnumerable<SBRContext> sbrRPNONCFIContexts = null;
            SBRContext sbrRPNONCFIContext = null;
    
            sbrRPNONCFIContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssessableForeignIncomeClassificationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonConduitForeignIncome")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNONCFIElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNONCFIContexts != null && sbrRPNONCFIContexts.Count() > 0)
            {
                sbrRPNONCFIContext = sbrRPNONCFIContexts.First();
        
                report.RPNONCFIId = sbrRPNONCFIContext.Id;
                report.RPNONCFIIdentifier = sbrRPNONCFIContext.EntityIdentifier;
                report.RPNONCFIIdentifierScheme = sbrRPNONCFIContext.EntityScheme;
        
                report.RPNONCFICount = sbrRPNONCFIContexts.Count();
                report.RPNONCFIExists = true;
                report.RPNONCFIOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNONCFIContext) + 1;
                report.RPNONCFILastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNONCFIContexts.Last()) + 1;
        

                if (report.RPNONCFIIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNONCFIIdentifierTFN = report.RPNONCFIIdentifier;
    
                report.RPNONCFIStartDate = ((SBRDurationPeriod)sbrRPNONCFIContext.Period).DurationStartAsDateObject;
                report.RPNONCFIEndDate = ((SBRDurationPeriod)sbrRPNONCFIContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPNONCFIElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNONCFIContexts.Count(); i++)
                {
                    sbrRPNONCFIContext = sbrRPNONCFIContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNONCFIContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNONCFIElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNONCFIElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNONCFIElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                } 
            } 
            #endregion End of Context sbrRPNONCFIContext
    
            #region sbrINTContext
            IEnumerable<SBRContext> sbrINTContexts = null;
            SBRContext sbrINTContext = null;
    
            sbrINTContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Intermediary")) ));
            Dictionary<string, IList<SBRElement>> sbrPopulatedINTElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrINTContexts != null && sbrINTContexts.Count() > 0)
            {
                sbrINTContext = sbrINTContexts.First();
        
                report.INTId = sbrINTContext.Id;
                report.INTIdentifier = sbrINTContext.EntityIdentifier;
                report.INTIdentifierScheme = sbrINTContext.EntityScheme;
        
                report.INTCount = sbrINTContexts.Count();
                report.INTExists = true;
                report.INTOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContext) + 1;
                report.INTLastOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContexts.Last()) + 1;
        

                if (report.INTIdentifierScheme == "http://www.ato.gov.au/abn") report.INTIdentifierABN = report.INTIdentifier;
    
                report.INTStartDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationStartAsDateObject;
                report.INTEndDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedINTElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrINTContexts.Count(); i++)
                {
                    sbrINTContext = sbrINTContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrINTContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedINTElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedINTElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedINTElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.DIS63 = report.INTIdentifier;
        
                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;
            
                    #region sbrDeclarationCollection
            
                    if (sbrDeclarationCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTCollection;
                        sbrDeclarationCollection_TrueAndCorrect_INTCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_INTCollection != null && sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_INT;
                            sbrDeclarationCollection_TrueAndCorrect_INT = sbrDeclarationCollection_TrueAndCorrect_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.DeclarationCollection_TrueAndCorrect_INTExists = true;
                            report.DeclarationCollection_TrueAndCorrect_INTCount = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INT.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_INTLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS81 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS82 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS83 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS84 = DateTime.Parse(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS85 = currentValue.Value;
                            }
                
                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;
                    
                            #region sbrDeclaration_PersonUnstructuredNameCollection
                    
                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                        
                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null && 
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrINTContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.ElementAt(0);
                        
                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                        
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Last().OccurrenceIndex + 1;
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.DIS86 = currentValue.Value;
                                    }
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.DIS87 = currentValue.Value;
                                    }
                                }  
                            } 
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateINTDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrINTContexts, report);
                    
                        }  
                    } 
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateINTDeclarationCollection(errors, sbrDeclarationCollection, sbrINTContexts, report);
            
            
                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Identifiers.TaxAgentNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.DIS68 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_INTCollection;
                        sbrOrganisationNameDetailsCollection_MN_INTCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_INTCollection != null && sbrOrganisationNameDetailsCollection_MN_INTCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_INT;
                            sbrOrganisationNameDetailsCollection_MN_INT = sbrOrganisationNameDetailsCollection_MN_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_INTTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_INTTupleElementMap = sbrOrganisationNameDetailsCollection_MN_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_MN_INTExists = true;
                            report.OrganisationNameDetailsCollection_MN_INTCount = sbrOrganisationNameDetailsCollection_MN_INTCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_INT.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_INTLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_MN_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS90 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS91 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateINTOrganisationNameDetailsCollection(errors, sbrOrganisationNameDetailsCollection, sbrINTContexts, report);
            
        
                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;
            
                    #region sbrElectronicContactTelephoneCollection
            
                    if (sbrElectronicContactTelephoneCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_INTCollection;
                        sbrElectronicContactTelephoneCollection_03_INTCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_INTCollection != null && sbrElectronicContactTelephoneCollection_03_INTCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_INT;
                            sbrElectronicContactTelephoneCollection_03_INT = sbrElectronicContactTelephoneCollection_03_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_INTTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_INTTupleElementMap = sbrElectronicContactTelephoneCollection_03_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ElectronicContactTelephoneCollection_03_INTExists = true;
                            report.ElectronicContactTelephoneCollection_03_INTCount = sbrElectronicContactTelephoneCollection_03_INTCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INT.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_INTLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS92 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS93 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS94 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS95 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateINTElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrINTContexts, report);
            
        
                    IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName", out currentValue))
                        sbrPersonUnstructuredNameCollection = currentValue.Occurrences;
                    else
                        sbrPersonUnstructuredNameCollection = null;
            
                    #region sbrPersonUnstructuredNameCollection
            
                    if (sbrPersonUnstructuredNameCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection_Contact_INTCollection;
                        sbrPersonUnstructuredNameCollection_Contact_INTCollection = sbrPersonUnstructuredNameCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "Contact" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrPersonUnstructuredNameCollection_Contact_INTCollection != null && sbrPersonUnstructuredNameCollection_Contact_INTCollection.Count() > 0)
                        {
                            SBRElement sbrPersonUnstructuredNameCollection_Contact_INT;
                            sbrPersonUnstructuredNameCollection_Contact_INT = sbrPersonUnstructuredNameCollection_Contact_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap;
                            sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap = sbrPersonUnstructuredNameCollection_Contact_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.PersonUnstructuredNameCollection_Contact_INTExists = true;
                            report.PersonUnstructuredNameCollection_Contact_INTCount = sbrPersonUnstructuredNameCollection_Contact_INTCollection.Count();
                            report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_INT.OccurrenceIndex + 1;
                            report.PersonUnstructuredNameCollection_Contact_INTLastOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS88 = currentValue.Value;
                            }
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.DIS89 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrPersonUnstructuredNameCollection
                    if (validationMode) ValidateINTPersonUnstructuredNameCollection(errors, sbrPersonUnstructuredNameCollection, sbrINTContexts, report);
            
                } 
            } 
            #endregion End of Context sbrINTContext
            return report;

        } // Of Consume Method
        #region ValidateRPOrganisationNameDetailsCollection
        protected virtual void ValidateRPOrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement DIS34Value = null;
                    string DIS34 = null;
                    SBRElement DIS24Value = null;
                    string DIS24 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalNameType.Code", out DIS34Value))
                    {
                        if (!DIS34Value.IsNil) DIS34 = DIS34Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalName.Text", out DIS24Value))
                    {
                        if (!DIS24Value.IsNil) DIS24 = DIS24Value.Value;
                    }
        
                    #region VR.ATO.GEN.410049

        /*  VR.ATO.GEN.410049
            Organisation Name Type Code must be 'MN'

        Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS34 <> BLANK) AND (^DIS34 <> 'MN')

        Data Elements:

        ^DIS34 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
        */
                    assertion = (string.IsNullOrWhiteSpace(DIS34) != true && DIS34 != @"MN");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410049", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type Code must be 'MN'",
                            Location = DIS34Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalNameType.Code" : DeriveSbrElementLocation(DIS34Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410049"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS34", Value = DIS34 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPPersonUnstructuredNameCollection
        protected virtual void ValidateRPPersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement DIS36Value = null;
                    string DIS36 = null;
                    bool PersonUnstructuredNameCollection_Contact_RPExists = true;

                    SBRElement DIS25Value = null;
                    string DIS25 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out DIS36Value))
                    {
                        if (!DIS36Value.IsNil) DIS36 = DIS36Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out DIS25Value))
                    {
                        if (!DIS25Value.IsNil) DIS25 = DIS25Value.Value;
                    }
        
                    #region VR.ATO.DIS.406035

        /*  VR.ATO.DIS.406035
            The unstructured person name for the tax agent or intermediary contact must include a Usage Code of 'Contact'.

        Legacy Rule Format:
            WHERE IN TUPLE (xbrli\personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF ((pyde.xx.xx:PersonUnstructuredName.Usage.Code ) <> NULLORBLANK) AND ((pyde.xx.xx:PersonUnstructuredName.Usage.Code) <> "Contact")
            RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS36 <> BLANK) AND (^DIS36 <> 'Contact')

        Data Elements:

        ^DIS36 = RP:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='Contact']
        */
                    assertion = (string.IsNullOrWhiteSpace(DIS36) != true && DIS36 != @"Contact");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406035", Severity = ProcessMessageSeverity.Error,
                            Description = @"Usage code for contact name must be ""Contact""",
                            LongDescription = @"The unstructured person name for the tax agent or intermediary contact must include a Usage Code of 'Contact'.",
                            Location = DIS36Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(DIS36Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406035"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS36", Value = DIS36 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.000243

        /*  VR.ATO.GEN.000243
            Both person unstructured name usage code and name must be provided

        Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code = NULLORBLANK OR pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^DIS36 = BLANK OR ^DIS25 = BLANK)

        Data Elements:

        ^DIS36 = RP:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='Contact']

        ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']

        ^PersonUnstructuredName = RP:PersonUnstructuredName
        */
                    assertion = (PersonUnstructuredNameCollection_Contact_RPExists != false && (string.IsNullOrWhiteSpace(DIS36) == true || string.IsNullOrWhiteSpace(DIS25) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000243", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both person unstructured name usage code and name must be provided",
                            Location = DIS36Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(DIS36Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000243"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS36", Value = DIS36 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS25", Value = DIS25 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPElectronicContactTelephoneCollection
        protected virtual void ValidateRPElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection
        protected virtual void ValidateRPAddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement DIS46Value = null;
                    string DIS46 = null;

                    SBRElement DIS47Value = null;
                    string DIS47 = null;
                    bool AddressDetailsCollection_BUS_C_RPExists = true;
                    SBRElement DIS45Value = null;
                    bool? DIS45 = null;
                    SBRElement DIS48Value = null;
                    string DIS48 = null;
                    SBRElement DIS49Value = null;
                    string DIS49 = null;
                    SBRElement DIS50Value = null;
                    string DIS50 = null;
                    SBRElement DIS51Value = null;
                    string DIS51 = null;
                    SBRElement DIS52Value = null;
                    string DIS52 = null;
                    SBRElement DIS56Value = null;
                    string DIS56 = null;
                    SBRElement DIS57Value = null;
                    string DIS57 = null;
                    SBRElement DIS58Value = null;
                    string DIS58 = null;
                    SBRElement DIS59Value = null;
                    string DIS59 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Usage.Code", out DIS46Value))
                    {
                        if (!DIS46Value.IsNil) DIS46 = DIS46Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Currency.Code", out DIS47Value))
                    {
                        if (!DIS47Value.IsNil) DIS47 = DIS47Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.OverseasAddress.Indicator", out DIS45Value))
                    {
                        if (!DIS45Value.IsNil) DIS45 = ToBoolean(DIS45Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line1.Text", out DIS48Value))
                    {
                        if (!DIS48Value.IsNil) DIS48 = DIS48Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line2.Text", out DIS49Value))
                    {
                        if (!DIS49Value.IsNil) DIS49 = DIS49Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line3.Text", out DIS50Value))
                    {
                        if (!DIS50Value.IsNil) DIS50 = DIS50Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line4.Text", out DIS51Value))
                    {
                        if (!DIS51Value.IsNil) DIS51 = DIS51Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.LocalityName.Text", out DIS52Value))
                    {
                        if (!DIS52Value.IsNil) DIS52 = DIS52Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Postcode.Text", out DIS56Value))
                    {
                        if (!DIS56Value.IsNil) DIS56 = DIS56Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.StateOrTerritory.Code", out DIS57Value))
                    {
                        if (!DIS57Value.IsNil) DIS57 = DIS57Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.CountryName.Text", out DIS58Value))
                    {
                        if (!DIS58Value.IsNil) DIS58 = DIS58Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Country.Code", out DIS59Value))
                    {
                        if (!DIS59Value.IsNil) DIS59 = DIS59Value.Value;
                    }
        
                    #region VR.ATO.DIS.406028

        /*  VR.ATO.DIS.406028
            Address usage code must be 'BUS' or 'POS' , and Currency Code must be 'C'

        Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Usage.Code <>  SET("BUS","POS")) OR (pyde.xx.xx:AddressDetails.Currency.Code <> "C") 
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (NotInSet(^DIS46, '"BUS","POS"')) OR (^DIS47 <> 'C')

        Data Elements:

        ^DIS46 = RP:AddressDetails:AddressDetails.Usage.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']

        ^DIS47 = RP:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
        */
                    assertion = (!(IsMatch(DIS46, @"^(BUS|POS)$",RegexOptions.IgnoreCase)) || DIS47 != @"C");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406028", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address usage code must be 'BUS' or 'POS' and Currency Code must be 'C'",
                            LongDescription = @"Address usage code must be 'BUS' (Business) or 'POS' (Postal) and Currency Code must be 'C' (Current)",
                            Location = DIS46Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Usage.Code" : DeriveSbrElementLocation(DIS46Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406028"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.410148

        /*  VR.ATO.GEN.410148
            Address Currency Code must be present if an Address is present.

        Legacy Rule Format:
            IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Currency.Code = NULLORBLANK
              RETURN VALIDATION MESSAGE
            END IF
    
        Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS47 = BLANK)

        Data Elements:

        ^DIS47 = RP:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']

        ^AddressDetails = RP:AddressDetails
        */
                    assertion = (AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(DIS47) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410148", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Currency Code must be present if an Address is present.",
                            Location = DIS47Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Currency.Code" : DeriveSbrElementLocation(DIS47Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410148"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.428202

        /*  VR.ATO.GEN.428202
            Address currency code must be "C" or "P"

        Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Currency.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Currency.Code <> SET("C","P"))
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS47 <> BLANK) AND (NotInSet(^DIS47, '"C","P"'))

        Data Elements:

        ^DIS47 = RP:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
        */
                    assertion = (string.IsNullOrWhiteSpace(DIS47) != true && !(IsMatch(DIS47, @"^(C|P)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428202", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address currency code must be ""C"" or ""P""",
                            Location = DIS47Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Currency.Code" : DeriveSbrElementLocation(DIS47Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428202"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection1
        protected virtual void ValidateRPAddressDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPInvestor_OrganisationNameDetailsCollection
        protected virtual void ValidateRPInvestor_OrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPInvestor_AddressDetailsCollection
        protected virtual void ValidateRPInvestor_AddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement DIS37Value = null;
                    string DIS37 = null;

                    SBRElement DIS54Value = null;
                    string DIS54 = null;
                    bool Investor_AddressDetailsCollection_POS_C_RPExists = true;
                    SBRElement DIS41Value = null;
                    bool? DIS41 = null;
                    SBRElement DIS7Value = null;
                    string DIS7 = null;
                    SBRElement DIS8Value = null;
                    string DIS8 = null;
                    SBRElement DIS38Value = null;
                    string DIS38 = null;
                    SBRElement DIS39Value = null;
                    string DIS39 = null;
                    SBRElement DIS9Value = null;
                    string DIS9 = null;
                    SBRElement DIS11Value = null;
                    string DIS11 = null;
                    SBRElement DIS10Value = null;
                    string DIS10 = null;
                    SBRElement DIS12Value = null;
                    string DIS12 = null;
                    SBRElement DIS40Value = null;
                    string DIS40 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Usage.Code", out DIS37Value))
                    {
                        if (!DIS37Value.IsNil) DIS37 = DIS37Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Currency.Code", out DIS54Value))
                    {
                        if (!DIS54Value.IsNil) DIS54 = DIS54Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.OverseasAddress.Indicator", out DIS41Value))
                    {
                        if (!DIS41Value.IsNil) DIS41 = ToBoolean(DIS41Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line1.Text", out DIS7Value))
                    {
                        if (!DIS7Value.IsNil) DIS7 = DIS7Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line2.Text", out DIS8Value))
                    {
                        if (!DIS8Value.IsNil) DIS8 = DIS8Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line3.Text", out DIS38Value))
                    {
                        if (!DIS38Value.IsNil) DIS38 = DIS38Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Line4.Text", out DIS39Value))
                    {
                        if (!DIS39Value.IsNil) DIS39 = DIS39Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.LocalityName.Text", out DIS9Value))
                    {
                        if (!DIS9Value.IsNil) DIS9 = DIS9Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Postcode.Text", out DIS11Value))
                    {
                        if (!DIS11Value.IsNil) DIS11 = DIS11Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.StateOrTerritory.Code", out DIS10Value))
                    {
                        if (!DIS10Value.IsNil) DIS10 = DIS10Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.CountryName.Text", out DIS12Value))
                    {
                        if (!DIS12Value.IsNil) DIS12 = DIS12Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}AddressDetails.Country.Code", out DIS40Value))
                    {
                        if (!DIS40Value.IsNil) DIS40 = DIS40Value.Value;
                    }
        
                    #region VR.ATO.DIS.406048

        /*  VR.ATO.DIS.406048
            Address usage code must be 'POS' , and Currency Code must be 'C'

        Legacy Rule Format:
            WHERE IN TUPLE (addressdetails2.xx.xx:AddressDetails) IN TUPLE (dis.0003.lodge.req.xx.xx:Investor)
            IF (pyde.xx.xx:AddressDetails.Usage.Code <> "POS") OR (pyde.xx.xx:AddressDetails.Currency.Code <> "C") 
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS37 <> 'POS') OR (^DIS54 <> 'C')

        Data Elements:

        ^DIS37 = RP:Investor:AddressDetails:AddressDetails.Usage.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']

        ^DIS54 = RP:Investor:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
        */
                    assertion = (DIS37 != @"POS" || DIS54 != @"C");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406048", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address usage code must be 'POS' and Currency Code must be 'C'",
                            Location = DIS37Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Usage.Code" : DeriveSbrElementLocation(DIS37Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406048"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS37", Value = GetValueOrEmpty(DIS37) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS54", Value = GetValueOrEmpty(DIS54) });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.410148

        /*  VR.ATO.GEN.410148
            Address Currency Code must be present if an Address is present.

        Legacy Rule Format:
            IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Currency.Code = NULLORBLANK
              RETURN VALIDATION MESSAGE
            END IF
    
        Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS54 = BLANK)

        Data Elements:

        ^DIS54 = RP:Investor:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']

        ^AddressDetails = RP:Investor:AddressDetails
        */
                    assertion = (Investor_AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(DIS54) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410148", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Currency Code must be present if an Address is present.",
                            Location = DIS54Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Currency.Code" : DeriveSbrElementLocation(DIS54Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410148"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS54", Value = GetValueOrEmpty(DIS54) });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.428202

        /*  VR.ATO.GEN.428202
            Address currency code must be "C" or "P"

        Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Currency.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Currency.Code <> SET("C","P"))
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS54 <> BLANK) AND (NotInSet(^DIS54, '"C","P"'))

        Data Elements:

        ^DIS54 = RP:Investor:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
        */
                    assertion = (string.IsNullOrWhiteSpace(DIS54) != true && !(IsMatch(DIS54, @"^(C|P)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428202", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address currency code must be ""C"" or ""P""",
                            Location = DIS54Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Currency.Code" : DeriveSbrElementLocation(DIS54Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428202"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS54", Value = GetValueOrEmpty(DIS54) });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateRPDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists = true;

                    SBRElement DIS79Value = null;
                    string DIS79 = null;

                    SBRElement DIS80Value = null;
                    string DIS80 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out DIS79Value))
                    {
                        if (!DIS79Value.IsNil) DIS79 = DIS79Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out DIS80Value))
                    {
                        if (!DIS80Value.IsNil) DIS80 = DIS80Value.Value;
                    }
        
                    #region VR.ATO.GEN.000243

        /*  VR.ATO.GEN.000243
            Both person unstructured name usage code and name must be provided

        Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code = NULLORBLANK OR pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^DIS79 = BLANK OR ^DIS80 = BLANK)

        Data Elements:

        ^DIS79 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^DIS80 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = RP:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists != false && (string.IsNullOrWhiteSpace(DIS79) == true || string.IsNullOrWhiteSpace(DIS80) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000243", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both person unstructured name usage code and name must be provided",
                            Location = DIS79Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(DIS79Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000243"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS79", Value = DIS79 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS80", Value = DIS80 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.436279

        /*  VR.ATO.GEN.436279
            The unstructured person name tuple must include the Usage Code 'DeclarationSignatory'.

        Legacy Rule Format:
            WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^DIS79 <> 'DeclarationSignatory')

        Data Elements:

        ^DIS79 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = RP:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists != false && DIS79 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = DIS80Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.FullName.Text" : DeriveSbrElementLocation(DIS80Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS79", Value = DIS79 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPDeclarationCollection
        protected virtual void ValidateRPDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool DeclarationCollection_TrueAndCorrect_RPExists = true;

                    SBRElement DIS74Value = null;
                    string DIS74 = null;

                    SBRElement DIS75Value = null;
                    bool? DIS75 = null;

                    SBRElement DIS78Value = null;
                    string DIS78 = null;

                    SBRElement DIS77Value = null;
                    DateTime? DIS77 = null;
                    SBRElement DIS76Value = null;
                    string DIS76 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementType.Code", out DIS74Value))
                    {
                        if (!DIS74Value.IsNil) DIS74 = DIS74Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementAccepted.Indicator", out DIS75Value))
                    {
                        if (!DIS75Value.IsNil) DIS75 = ToBoolean(DIS75Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.SignatoryIdentifier.Text", out DIS78Value))
                    {
                        if (!DIS78Value.IsNil) DIS78 = DIS78Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Signature.Date", out DIS77Value))
                    {
                        if (!DIS77Value.IsNil) DIS77 = DateTime.Parse(DIS77Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Statement.Text", out DIS76Value))
                    {
                        if (!DIS76Value.IsNil) DIS76 = DIS76Value.Value;
                    }
        
                    #region VR.ATO.GEN.430250

        /*  VR.ATO.GEN.430250
            The Declaration must be complete

        Legacy Rule Format:
            WHERE IN TUPLE(Declaration2.xx.xx:Declaration)
            IF pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect" OR pyin.xx.xx:Declaration.StatementAccepted.Indicator = FALSE OR pyin.xx.xx:Declaration.SignatoryIdentifier.Text = NULLORBLANK OR pyin.xx.xx:Declaration.Signature.Date = NULL 
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^Declaration <> NULL) AND (^DIS74 <> 'TrueAndCorrect' OR ^DIS75 = FALSE OR ^DIS78 = BLANK OR ^DIS77 = NULL)

        Data Elements:

        ^DIS75 = RP:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^DIS74 = RP:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^DIS78 = RP:Declaration:Declaration.SignatoryIdentifier.Text WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^DIS77 = RP:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^Declaration = RP:Declaration
        */
                    assertion = (DeclarationCollection_TrueAndCorrect_RPExists != false && (DIS74 != @"TrueAndCorrect" || DIS75 == false || string.IsNullOrWhiteSpace(DIS78) == true || DIS77 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430250", Severity = ProcessMessageSeverity.Error,
                            Description = @"The Declaration must be complete",
                            LongDescription = @"Within the Declaration, the Statement Type must be 'TrueAndCorrect', Statement Accepted must be 'Yes' (true), and the Signatory Identifier and Signature Date must be present.",
                            Location = DIS75Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementAccepted.Indicator" : DeriveSbrElementLocation(DIS75Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430250"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS74", Value = DIS74 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS75", Value = GetValueOrEmpty(DIS75) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS78", Value = DIS78 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS77", Value = GetValueOrEmpty(DIS77) });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateINTDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;

                    SBRElement DIS86Value = null;
                    string DIS86 = null;

                    SBRElement DIS87Value = null;
                    string DIS87 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out DIS86Value))
                    {
                        if (!DIS86Value.IsNil) DIS86 = DIS86Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out DIS87Value))
                    {
                        if (!DIS87Value.IsNil) DIS87 = DIS87Value.Value;
                    }
        
                    #region VR.ATO.GEN.000243

        /*  VR.ATO.GEN.000243
            Both person unstructured name usage code and name must be provided

        Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code = NULLORBLANK OR pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^DIS86 = BLANK OR ^DIS87 = BLANK)

        Data Elements:

        ^DIS86 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^DIS87 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && (string.IsNullOrWhiteSpace(DIS86) == true || string.IsNullOrWhiteSpace(DIS87) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000243", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both person unstructured name usage code and name must be provided",
                            Location = DIS86Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(DIS86Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000243"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS86", Value = DIS86 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS87", Value = DIS87 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.436279

        /*  VR.ATO.GEN.436279
            The unstructured person name tuple must include the Usage Code 'DeclarationSignatory'.

        Legacy Rule Format:
            WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^DIS86 <> 'DeclarationSignatory')

        Data Elements:

        ^DIS86 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && DIS86 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = DIS87Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.FullName.Text" : DeriveSbrElementLocation(DIS87Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS86", Value = DIS86 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTDeclarationCollection
        protected virtual void ValidateINTDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool DeclarationCollection_TrueAndCorrect_INTExists = true;

                    SBRElement DIS81Value = null;
                    string DIS81 = null;

                    SBRElement DIS82Value = null;
                    bool? DIS82 = null;

                    SBRElement DIS85Value = null;
                    string DIS85 = null;
                    SBRElement DIS83Value = null;
                    string DIS83 = null;
                    SBRElement DIS84Value = null;
                    DateTime? DIS84 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementType.Code", out DIS81Value))
                    {
                        if (!DIS81Value.IsNil) DIS81 = DIS81Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.StatementAccepted.Indicator", out DIS82Value))
                    {
                        if (!DIS82Value.IsNil) DIS82 = ToBoolean(DIS82Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.SignatoryIdentifier.Text", out DIS85Value))
                    {
                        if (!DIS85Value.IsNil) DIS85 = DIS85Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Statement.Text", out DIS83Value))
                    {
                        if (!DIS83Value.IsNil) DIS83 = DIS83Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}Declaration.Signature.Date", out DIS84Value))
                    {
                        if (!DIS84Value.IsNil) DIS84 = DateTime.Parse(DIS84Value.Value);
                    }
        
                    #region VR.ATO.GEN.430250

        /*  VR.ATO.GEN.430250
            The Declaration must be complete

        Legacy Rule Format:
            WHERE IN TUPLE(Declaration2.xx.xx:Declaration)
            IF pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect" OR pyin.xx.xx:Declaration.StatementAccepted.Indicator = FALSE OR pyin.xx.xx:Declaration.SignatoryIdentifier.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^Declaration <> NULL) AND (^DIS81 <> 'TrueAndCorrect' OR ^DIS82 = FALSE OR ^DIS85 = BLANK)

        Data Elements:

        ^DIS82 = INT:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^DIS81 = INT:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^DIS85 = INT:Declaration:Declaration.SignatoryIdentifier.Text WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^Declaration = INT:Declaration
        */
                    assertion = (DeclarationCollection_TrueAndCorrect_INTExists != false && (DIS81 != @"TrueAndCorrect" || DIS82 == false || string.IsNullOrWhiteSpace(DIS85) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430250", Severity = ProcessMessageSeverity.Error,
                            Description = @"The Declaration must be complete",
                            LongDescription = @"Within the Declaration, the Statement Type must be 'TrueAndCorrect', Statement Accepted must be 'Yes' (true), and the Signatory Identifier and Signature Date must be present.",
                            Location = DIS82Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementAccepted.Indicator" : DeriveSbrElementLocation(DIS82Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430250"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS81", Value = DIS81 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS82", Value = GetValueOrEmpty(DIS82) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS85", Value = DIS85 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTOrganisationNameDetailsCollection
        protected virtual void ValidateINTOrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement DIS90Value = null;
                    string DIS90 = null;
                    SBRElement DIS91Value = null;
                    string DIS91 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalNameType.Code", out DIS90Value))
                    {
                        if (!DIS90Value.IsNil) DIS90 = DIS90Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}OrganisationNameDetails.OrganisationalName.Text", out DIS91Value))
                    {
                        if (!DIS91Value.IsNil) DIS91 = DIS91Value.Value;
                    }
        
                    #region VR.ATO.GEN.410049

        /*  VR.ATO.GEN.410049
            Organisation Name Type Code must be 'MN'

        Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS90 <> BLANK) AND (^DIS90 <> 'MN')

        Data Elements:

        ^DIS90 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
        */
                    assertion = (string.IsNullOrWhiteSpace(DIS90) != true && DIS90 != @"MN");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410049", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type Code must be 'MN'",
                            Location = DIS90Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalNameType.Code" : DeriveSbrElementLocation(DIS90Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410049"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS90", Value = DIS90 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTElectronicContactTelephoneCollection
        protected virtual void ValidateINTElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateINTPersonUnstructuredNameCollection
        protected virtual void ValidateINTPersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, DIS2018 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement DIS88Value = null;
                    string DIS88 = null;
                    bool PersonUnstructuredNameCollection_Contact_INTExists = true;

                    SBRElement DIS89Value = null;
                    string DIS89 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.Usage.Code", out DIS88Value))
                    {
                        if (!DIS88Value.IsNil) DIS88 = DIS88Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/dis}PersonUnstructuredName.FullName.Text", out DIS89Value))
                    {
                        if (!DIS89Value.IsNil) DIS89 = DIS89Value.Value;
                    }
        
                    #region VR.ATO.DIS.406035

        /*  VR.ATO.DIS.406035
            The unstructured person name for the tax agent or intermediary contact must include a Usage Code of 'Contact'.

        Legacy Rule Format:
            WHERE IN TUPLE (xbrli\personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF ((pyde.xx.xx:PersonUnstructuredName.Usage.Code ) <> NULLORBLANK) AND ((pyde.xx.xx:PersonUnstructuredName.Usage.Code) <> "Contact")
            RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^DIS88 <> BLANK) AND (^DIS88 <> 'Contact')

        Data Elements:

        ^DIS88 = INT:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='Contact']
        */
                    assertion = (string.IsNullOrWhiteSpace(DIS88) != true && DIS88 != @"Contact");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406035", Severity = ProcessMessageSeverity.Error,
                            Description = @"Usage code for contact name must be ""Contact""",
                            LongDescription = @"The unstructured person name for the tax agent or intermediary contact must include a Usage Code of 'Contact'.",
                            Location = DIS89Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.FullName.Text" : DeriveSbrElementLocation(DIS89Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406035"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS88", Value = DIS88 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.000243

        /*  VR.ATO.GEN.000243
            Both person unstructured name usage code and name must be provided

        Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code = NULLORBLANK OR pyde.xx.xx:PersonUnstructuredName.FullName.Text = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^DIS88 = BLANK OR ^DIS89 = BLANK)

        Data Elements:

        ^DIS88 = INT:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='Contact']

        ^DIS89 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']

        ^PersonUnstructuredName = INT:PersonUnstructuredName
        */
                    assertion = (PersonUnstructuredNameCollection_Contact_INTExists != false && (string.IsNullOrWhiteSpace(DIS88) == true || string.IsNullOrWhiteSpace(DIS89) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000243", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both person unstructured name usage code and name must be provided",
                            Location = DIS88Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(DIS88Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000243"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS88", Value = DIS88 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS89", Value = DIS89 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

    } // Of Class
} // Of Namespace

