using System.Linq;
using System.Collections.Generic;
using VaTS;
using DataContracts;
using Ato.EN.IntegrationServices.CodeGenerationFITR;

namespace Ato.CD.Inbound.FITR202402
{
    internal class CrossFormValidator
    {
        private FITR2024 ParentDocument { get; }
        private BusinessDocumentCollection ChildDocuments { get; }
        public List<ProcessMessageDocument> response { get; private set; }

        public CrossFormValidator(FITR2024 report, BusinessDocumentCollection childDocuments)
        {
            ParentDocument = report;
            ChildDocuments = childDocuments ?? new BusinessDocumentCollection();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            response = new List<ProcessMessageDocument>();

            VRATOFITR434034(response);
            VRATOFITR434085(response);
            VRATOFITR434300(response);
            VRATOFITR434614(response);

            return response;
        }

        public bool ScheduledChildrenExists(string scheduledName)
        {
            return ChildDocuments.Any(docs => docs.DocumentName.Equals(scheduledName));
        }

        #region VR.ATO.FITR.434034

        /*  
            VR.ATO.FITR.434034	CMN.ATO.GEN.434034
            IF [FITR23] > 10000 AND  (COUNT(SCHEDULE = ""CGTS"") = 0)  
                RETURN VALIDATION MESSAGE
            ENDIF
            [FITR23] = FITR:RP:bafpr1.02.00:Income.CapitalGainsNet.Amount
        */
        private void VRATOFITR434034(List<ProcessMessageDocument> response)
        {
            bool assertion = !ScheduledChildrenExists(Schedule.CGTS.ToString()) && ParentDocument.FITR23.HasValue && ParentDocument.FITR23.Value > 10000;
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "If Net capital gain is more than $10,000, a CGT schedule must be present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsNetA",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434034" },
                        new ProcessMessageParameter { Name = "FITR23", Value = ParentDocument.FITR23.Value.ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434034

        #region VR.ATO.FITR.434085

        /*  VR.ATO.FITR.434085
            Losses schedule must be attached if losses carried forward exceeds $100,001

            Legacy Rule Format:
            IF ([FITR76] + [FITR207]) > 100000 AND (COUNT(SCHEDULE = "LS") = 0) 
                RETURN VALIDATION MESSAGE
            ENDIF

            [FITR76] = FITR:RP.Closing:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            [FITR207] = FITR:RP.Closing(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            (^FITR76 + ^FITR207) > 100000 AND (CountDocument('LS') = 0)

            Data Elements:

            RP.Closing:^FITR76 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
   
            RP.Closing(Instant):^FITR207 = tns:Capital.Losses.CarriedForward.Net.Amount
        */
        protected void VRATOFITR434085(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR76.GetValueOrDefault() + ParentDocument.FITR207.GetValueOrDefault() > 100000 && !ScheduledChildrenExists(Schedule.LS.ToString());

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434085",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Losses schedule must be attached if losses carried forward exceeds $100,000",
                    LongDescription = "If the sum of 'Tax losses carried forward to later income years' and 'Net capital losses carried forward to later income years' is greater than $100000, a Losses schedule must be present",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:TaxCarriedForwardLaterIncomeYearsTotalA",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434085" },
                        new ProcessMessageParameter { Name = "FITR76", Value = ParentDocument.FITR76.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "FITR207", Value = ParentDocument.FITR207.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.FITR.434085

        #region VR.ATO.FITR.434300

        /*  VR.ATO.FITR.434300
            Payment summary schedule must be present when Payments where ABN not quoted amount is present

            Legacy Rule Format:
            IF [FITR32] > 0 AND  (COUNT(SCHEDULE = "PSS") = 0)  
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [FITR32] = FITR:RP:lrla.02.00:Remuneration.ABNNotQuotedPaymentGross.Amount

            Technical Business Rule Format:
            ^FITR32 > 0 AND  (CountDocument('PSS') = 0)

            Data Elements:
            
            RP:^FITR32 = tns:Remuneration.ABNNotQuotedPaymentGross.Amount
        */
        protected void VRATOFITR434300(List<ProcessMessageDocument> response)
        {
            bool assertion = !ScheduledChildrenExists(Schedule.PSS.ToString()) && ParentDocument.FITR32.GetValueOrDefault() > 0;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434044",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Payment summary schedule must be present when Payments where ABN not quoted amount is present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434300" },
                        new ProcessMessageParameter { Name = "FITR32", Value = ParentDocument.FITR32.GetValueOrDefault().ToString() }
                    }
                });
            }
        }

        #endregion // VR.ATO.FITR.434300

        #region VR.ATO.FITR.434614

        /*  VR.ATO.FITR.434614
            Distributions to Beneficiaries of Trust schedule must be present when the type of fund or trust is a Small APRA fund and Trust distributions - Income amounts or credits are provided

            Technical Business Rule:
            CountDocument('DISTBENTRT') = 0 AND ^FITR14 = '022' AND (^FITR37 > 0 OR ^FITR38 > 0 OR ^FITR39 > 0 OR ^FITR40 > 0)
        */
        private void VRATOFITR434614(List<ProcessMessageDocument> response)
        {
            bool assertion = !ScheduledChildrenExists(Schedule.DISTBENTRT.ToString()) && ParentDocument.FITR14.Equals("022") && (ParentDocument.FITR37.GetValueOrDefault() > 0 || ParentDocument.FITR38.GetValueOrDefault() > 0 || ParentDocument.FITR39.GetValueOrDefault() > 0 || ParentDocument.FITR40.GetValueOrDefault() > 0);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.434614",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distributions to Beneficiaries of Trust schedule must be present",
                    LongDescription = @"Distributions to Beneficiaries of Trust schedule must be present when the type of fund or trust is a Small APRA fund and Trust distributions - Income amounts or credits are provided",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:PartyTypeC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434614" },
                        new ProcessMessageParameter { Name = "FITR14", Value = ParentDocument.FITR14 },
                        new ProcessMessageParameter { Name = "FITR37", Value = ParentDocument.FITR37.HasValue ? ParentDocument.FITR37.Value.ToString() : "null" },
                        new ProcessMessageParameter { Name = "FITR38", Value = ParentDocument.FITR38.HasValue ? ParentDocument.FITR38.Value.ToString() : "null" },
                        new ProcessMessageParameter { Name = "FITR39", Value = ParentDocument.FITR39.HasValue ? ParentDocument.FITR39.Value.ToString() : "null" },
                        new ProcessMessageParameter { Name = "FITR40", Value = ParentDocument.FITR40.HasValue ? ParentDocument.FITR40.Value.ToString() : "null" }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434614
    }
}