using System.Linq;
using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using VaTS;
using Ato.EN.IntegrationServices.CodeGenerationFITR;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;
using DataContracts;

namespace Ato.CD.Inbound.FITR202402
{
    internal class CrossFormValidatorDISTBENTRT : ICrossFormValidator
    {
        private FITR2024 ParentDocument { get; }

        private DISTBENTRT2024 ChildDocument { get; }

        public CrossFormValidatorDISTBENTRT(FITR2024 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (DISTBENTRT2024)childDocument.ConsumedReport;
        }

        public IEnumerable<ProcessMessageDocument> ValidateCrossFormRules()
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();

            VRATOFITR434611(response);
            VRATOFITR434612(response);
            VRATOFITR434613(response);
            VRATOFITRW00002(response);

            return response;
        }

        #region VR.ATO.FITR.434611

        /*  VR.ATO.FITR.434611
            Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form

            Technical Business Rule:
            CountDocument('DISTBENTRT') = 1 AND ^FITR300 <> ^DISTBENTRT1
        */
        private void VRATOFITR434611(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR300 != ChildDocument.DISTBENTRT1;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.434611",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form",
                    Location = "/tns:FITR/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434611" },
                        new ProcessMessageParameter { Name = "FITR300", Value = ParentDocument.FITR300.Value.ToString("yyyy-MM-dd") },
                        new ProcessMessageParameter { Name = "DISTBENTRT1", Value = ChildDocument.DISTBENTRT1.Value.ToString("yyyy-MM-dd") }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434611

        #region VR.ATO.FITR.434612

        /*  VR.ATO.FITR.434612
            Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form

            Technical Business Rule:
            CountDocument('DISTBENTRT') = 1 AND ^FITR301 <> ^DISTBENTRT2
        */
        private void VRATOFITR434612(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR301 != ChildDocument.DISTBENTRT2;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.434612",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form",
                    Location = "/tns:FITR/tns:RP/tns:PeriodEndD",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434612" },
                        new ProcessMessageParameter { Name = "FITR301", Value = ParentDocument.FITR301.Value.ToString("yyyy-MM-dd") },
                        new ProcessMessageParameter { Name = "DISTBENTRT2", Value = ChildDocument.DISTBENTRT2.Value.ToString("yyyy-MM-dd") }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434612

        #region VR.ATO.FITR.434613

        /*  VR.ATO.FITR.434613
            The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form

            Technical Business Rule:
            CountDocument('DISTBENTRT') = 1 AND ^FITR5 <> ^DISTBENTRT3
        */
        private void VRATOFITR434613(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR5 != ChildDocument.DISTBENTRT3;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.434613",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form",
                    Location = "/tns:FITR/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434613" },
                        new ProcessMessageParameter { Name = "FITR5", Value = ParentDocument.FITR5 },
                        new ProcessMessageParameter { Name = "DISTBENTRT3", Value = ChildDocument.DISTBENTRT3 }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434613

        #region VR.ATO.FITR.W00002

        /*  VR.ATO.FITR.W00002
            Distributions to Beneficiaries of Trust schedule is provided but Trust distributions - Income amounts or credits are not declared

            Technical Business Rule:
            CountDocument('DISTBENTRT') = 1 AND ^FITR14 = '022' AND ^FITR37 = NULL AND ^FITR38 = NULL AND ^FITR39 = NULL AND ^FITR40 = NULL
        */
        private void VRATOFITRW00002(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR14.Equals("022") && !ParentDocument.FITR37.HasValue && !ParentDocument.FITR38.HasValue && !ParentDocument.FITR39.HasValue && !ParentDocument.FITR40.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.W00002",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure Trust distributions - Income amounts or credits are declared",
                    LongDescription = @"Distributions to Beneficiaries of Trust schedule must be present when the type of fund or trust is a Small APRA fund and Trust distributions - Income amounts or credits are provided",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:PartyTypeC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434615" },
                        new ProcessMessageParameter { Name = "FITR14", Value = ParentDocument.FITR14 },
                        new ProcessMessageParameter { Name = "FITR37", Value = ParentDocument.FITR37.HasValue ? ParentDocument.FITR37.Value.ToString() : "null" },
                        new ProcessMessageParameter { Name = "FITR38", Value = ParentDocument.FITR38.HasValue ? ParentDocument.FITR38.Value.ToString() : "null" },
                        new ProcessMessageParameter { Name = "FITR39", Value = ParentDocument.FITR39.HasValue ? ParentDocument.FITR39.Value.ToString() : "null" },
                        new ProcessMessageParameter { Name = "FITR40", Value = ParentDocument.FITR40.HasValue ? ParentDocument.FITR40.Value.ToString() : "null" }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434615
    }
}