using System.Linq;
using System.Collections.Generic;
using Ato.EN.IntegrationServices.Document.DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationFTER
{
    public partial class FTER2021Validator
    {
        #region VR.ATO.FTER.408077
        /*  VR.ATO.FTER.408077
        If the answer to the question 'At any time during the relevant period was the trustee a non-resident for tax purposes?' is 'No' (false), then the Full period indicator must not be present and a period of non residency must NOT be present

        Legacy Rule Format:
            ^FTER1010 <> NULL AND ^FTER36 = FALSE AND (^FTER37 <> NULL OR Count(^FTER1013) > 0)

        Technical Business Rule Format:
            ^FTER1010 <> NULL AND ^FTER36 = FALSE AND (^FTER37 <> NULL OR Count(^FTER1013) > 0)

        Data Elements:
    
            ^FTER36 = FTER:RP:Trustee:Residency.TaxPurposesPersonStatus.Indicator
    
            ^FTER37 = FTER:RP:Trustee:Residency.NonResidentFullPeriod.Indicator
    
            ^FTER1010 = FTER:RP:Trustee
    
            ^FTER1013 = FTER:RP:Trustee:NonResidentPeriodDetails
    */
        protected void VRATOFTER408077(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.RP_TrusteeCollection != null)
                response.AddRange(report.RP_TrusteeCollection
                    .Where(f => !f.FTER36.GetValueOrDefault() && (f.FTER37 != null || Count(f.RP_Trustee_NonResidentPeriodDetailsCollection) > 0))
                    .Select(f =>
                        new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408077",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Non residency question must be 'yes' (true) or periods must not be present",
                            LongDescription = "If the answer to the question 'was trustee a non resident for tax purposes' is 'No' (false), then the Full period indicator must not be present and a period of non residency must NOT be present",
                            Location = $"/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:ResidencyTaxPurposesPersonStatusI",
                            Parameters = new ProcessMessageParameters
                            {
                                new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408077"},
                                new ProcessMessageParameter {Name = "FTER36", Value = GetValueOrEmpty(f.FTER36)},
                                new ProcessMessageParameter {Name = "FTER37", Value = GetValueOrEmpty(f.FTER37)}
                            }
                        }
                    ));
        }
        #endregion // VR.ATO.FTER.408077

        #region VR.ATO.FTER.408078
        /*  VR.ATO.FTER.408078
            Trustee full period indicator or period of non-residency dates must be present

        Legacy Rule Format:
            ^FTER36 = TRUE AND ((^FTER37 = NULL OR ^FTER37 = FALSE) AND Count(^FTER1013) = 0)

        Technical Business Rule Format:
            ^FTER36 = TRUE AND ((^FTER37 = NULL OR ^FTER37 = FALSE) AND Count(^FTER1013) = 0)

        Data Elements:
    
            ^FTER37 = FTER:RP:Trustee:Residency.NonResidentFullPeriod.Indicator
    
            ^FTER36 = FTER:RP:Trustee:Residency.TaxPurposesPersonStatus.Indicator
    
            ^FTER1013 = FTER:RP:Trustee:NonResidentPeriodDetails
        */
        protected void VRATOFTER408078(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.RP_TrusteeCollection != null)
                response.AddRange(report.RP_TrusteeCollection
                    .Where(f => f.FTER36.GetValueOrDefault() && !f.FTER37.GetValueOrDefault() && Count(f.RP_Trustee_NonResidentPeriodDetailsCollection) == 0)
                    .Select(f =>
                        new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408078",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Trustee full period indicator or period of non-residency dates must be present",
                            LongDescription = "If the answer to the question 'was trustee a non resident for tax purposes is 'Yes' (true), then the Full period indicator or Dates of non residency must be present",
                            Location = $"/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:ResidencyNonResidentFullPeriodI",
                            Parameters = new ProcessMessageParameters
                            {
                                new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408078"},
                                new ProcessMessageParameter {Name = "FTER36", Value = GetValueOrEmpty(f.FTER36)},
                                new ProcessMessageParameter {Name = "FTER37", Value = GetValueOrEmpty(f.FTER37)}
                            }
                        }
                    ));
        }
        #endregion // VR.ATO.FTER.408078

        #region VR.ATO.FTER.408079
        /*  VR.ATO.FTER.408079
            Full period indicator and dates must not both be present

        Legacy Rule Format:
            ^FTER37 = TRUE AND Count(^FTER1013) > 0

        Technical Business Rule Format:
            ^FTER37 = TRUE AND Count(^FTER1013) > 0

        Data Elements:
    
            ^FTER37 = FTER:RP:Trustee:Residency.NonResidentFullPeriod.Indicator
    
            ^FTER1013 = FTER:RP:Trustee:NonResidentPeriodDetails
        */
        protected void VRATOFTER408079(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.RP_TrusteeCollection != null)
                response.AddRange(report.RP_TrusteeCollection
                    .Where(f => f.FTER37.GetValueOrDefault() && f.RP_Trustee_NonResidentPeriodDetailsCollectionExists && f.RP_Trustee_NonResidentPeriodDetailsCollectionCount > 0)
                    .Select(f =>
                        new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408079",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Full period indicator and dates must not both be present",
                            LongDescription = "If trustee was a non-resident for the full period indicator is 'yes' (true), then specific dates specifying the time(s) the trustee was a non-resident must not be present",
                            Location = $"/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:ResidencyNonResidentFullPeriodI",
                            Parameters = new ProcessMessageParameters
                            {
                                new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408079"},
                                new ProcessMessageParameter {Name = "FTER37", Value = GetValueOrEmpty(f.FTER37)}
                            }
                        }
                    ));
        }
        #endregion // VR.ATO.FTER.408079

        #region VR.ATO.FTER.408084
        /*  VR.ATO.FTER.408084
            Trustee non-resident period must not be before the substituted accounting period

        Legacy Rule Format:
            ^FTER41 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER41)

        Technical Business Rule Format:
            ^FTER41 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER41)

        Data Elements:
    
            ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
    
            ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
        */
        protected void VRATOFTER408084(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.FTER41 != null && report.RP_TrusteeCollection != null)
                response.AddRange(report.RP_TrusteeCollection
                    .Where(f => f.RP_Trustee_NonResidentPeriodDetailsCollection?.Find(r => r.FTER38 != null && r.FTER38.GetValueOrDefault() < report.FTER41.GetValueOrDefault()) != null)
                    .Select(f =>
                        new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408084",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Trustee non-resident period must not be before the substituted accounting period",
                            LongDescription = "The start time during the relevant period at which the trustee was a non-resident for tax purposes must not be earlier than the substituted accounting period",
                            Location = $"/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:NonResidentPeriodDetailsCollection/tns:NonResidentPeriodDetails{OccurrenceIndex(1 + f.RP_Trustee_NonResidentPeriodDetailsCollection.FindIndex(r => r.FTER38 != null && r.FTER38.GetValueOrDefault() < report.FTER41.GetValueOrDefault()))}/tns:StartD",
                            Parameters = new ProcessMessageParameters
                            {
                                new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408084"},
                                new ProcessMessageParameter {Name = "FTER41", Value = GetValueOrEmptyDateTime(report.FTER41)}
                            }
                        }
                    ));
        }
        #endregion // VR.ATO.FTER.408084

        #region VR.ATO.FTER.408085
        /*  VR.ATO.FTER.408085
            Trustee non-resident period must not be before the election commencement time

        Legacy Rule Format:
            ^FTER43 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER43)

        Technical Business Rule Format:
            ^FTER43 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER43)

        Data Elements:
    
            ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
        */
        protected void VRATOFTER408085(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.FTER43 != null && report.RP_TrusteeCollection != null)
                response.AddRange(report.RP_TrusteeCollection
                    .Where(f => f.RP_Trustee_NonResidentPeriodDetailsCollection?.Find(r => r.FTER38 != null && r.FTER38.GetValueOrDefault() < report.FTER43.GetValueOrDefault()) != null)
                    .Select(f =>
                        new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408085",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Trustee non-resident period must not be before the election commencement time",
                            LongDescription = "The start time during the relevant period at which the trustee was a non-resident for tax purposes must not be earlier than start of the election commencement time",
                            Location = $"/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:NonResidentPeriodDetailsCollection/tns:NonResidentPeriodDetails{OccurrenceIndex(1 + f.RP_Trustee_NonResidentPeriodDetailsCollection.FindIndex(r => r.FTER38 != null && r.FTER38.GetValueOrDefault() < report.FTER43.GetValueOrDefault()))}/tns:StartD",
                            Parameters = new ProcessMessageParameters
                            {
                                new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408085"},
                                new ProcessMessageParameter {Name = "FTER43", Value = GetValueOrEmptyDateTime(report.FTER43)}
                            }
                        }
                    ));
        }
        #endregion // VR.ATO.FTER.408085

        #region VR.ATO.FTER.408086
        /*  VR.ATO.FTER.408086
            Trustee non-resident period must not be before the start of income year specified

        Legacy Rule Format:
            ^FTER40 <> NULL AND ^FTER41 = NULL AND ^FTER43 = NULL AND (AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < AsDate(Concat(^FTER40 - 1, '-07-01'))))

        Technical Business Rule Format:
            ^FTER40 <> NULL AND ^FTER41 = NULL AND ^FTER43 = NULL AND (AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < AsDate(Concat(^FTER40 - 1, '-07-01'))))

        Data Elements:
    
            ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
    
            ^FTER40 = FTER:RP:ElectionOrVariationIncomeYear:Elections.FamilyTrustElectionStatus.Year
    
            ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
        */
        protected void VRATOFTER408086(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.FTER40 != null && report.FTER41 == null && report.FTER43 == null && report.RP_TrusteeCollection != null)
                response.AddRange(report.RP_TrusteeCollection
                    .Where(f => f.RP_Trustee_NonResidentPeriodDetailsCollection?.Find(r => r.FTER38 != null && r.FTER38.GetValueOrDefault() < AsDate(string.Concat(report.FTER40.GetValueOrDefault() - 1, "-07-01"))) != null)
                    .Select(f =>
                        new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408086",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Trustee non-resident period must not be before the start of income year specified",
                            LongDescription = "The start time during the relevant period at which the trustee was a non-resident for tax purposes must not be earlier than 1 July of the income year specified",
                            Location = $"/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:NonResidentPeriodDetailsCollection/tns:NonResidentPeriodDetails{OccurrenceIndex(1 + f.RP_Trustee_NonResidentPeriodDetailsCollection.FindIndex(r => r.FTER38 != null && r.FTER38.GetValueOrDefault() < AsDate(string.Concat(report.FTER40.GetValueOrDefault() - 1, "-07-01"))))}/tns:StartD",
                            Parameters = new ProcessMessageParameters
                            {
                                new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408086"},
                                new ProcessMessageParameter { Name = "FTER40", Value = report.FTER40.GetValueOrDefault().ToString() },
                                new ProcessMessageParameter{ Name = "FTER41", Value = GetValueOrEmptyDateTime(report.FTER41) },
                                new ProcessMessageParameter {Name = "FTER43", Value = GetValueOrEmptyDateTime(report.FTER43)}
                            }
                        }
                    ));
        }

        #endregion // VR.ATO.FTER.408086

        #region VR.ATO.FTER.408164
        /*  VR.ATO.FTER.408164
             One Declaration Statement Type Code must be 'TrueAndCorrect'

         Legacy Rule Format:
             WHERE PARENT RETURN DOES NOT EXIST
             ^FTER1000 = NULL AND Count(^FTER99 = 'TrueAndCorrect') <> 1

        Technical Business Rule Format:
             ^FTER1000 = NULL AND Count(^FTER99 = 'TrueAndCorrect') <> 1

        Data Elements:

            ^FTER99 = FTER:RP:DeclarationOfTrustee:Declaration.StatementType.Code
    
            ^FTER1000 = FTER:Intermediary
        */
        protected void VRATOFTER408164(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (isStandalone)
            {
                if (!report.IntCollectionExists && report.RP_DeclarationOfTrusteeCollection != null)
                {
                    var collectionsRpTrueAndCorrect = report.RP_DeclarationOfTrusteeCollection.Where(f => f.FTER99 == "TrueAndCorrect");
                    if (collectionsRpTrueAndCorrect.Count() > 1)
                    {
                        response.AddRange(collectionsRpTrueAndCorrect
                            .Skip(1)
                            .Select(f =>
                                new ProcessMessageDocument
                                {
                                    Code = "CMN.ATO.FTER.408164",
                                    Severity = ProcessMessageSeverity.Error,
                                    Description = "Trustee declaration statement type code must have one 'TrueAndCorrect'",
                                    LongDescription = "The FTER form must have one declaration that has type 'TrueAndCorrect'",
                                    Location = $"/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee{OccurrenceIndex(f.OccurrenceIndex)}/tns:StatementTypeC",
                                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FTER.408164" } }
                                }
                            ));
                    }
                    else if (collectionsRpTrueAndCorrect.Count() < 1)
                    {
                        response.Add(new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408164",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Trustee declaration statement type code must have one 'TrueAndCorrect'",
                            LongDescription = "The FTER form must have one declaration that has type 'TrueAndCorrect'",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee[1]/tns:StatementTypeC",
                            Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FTER.408164" } }
                        });
                    }
                }
            }
        }
        #endregion // VR.ATO.FTER.408164

        #region VR.ATO.FTER.408031
        /*  VR.ATO.FTER.408031
            Variation must not include central management or trustee details

            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER95 = 'V' AND (^FTER1008 <> NULL OR Count(^FTER1010) > 0)

            Technical Business Rule Format:
            ^FTER95 = 'V' AND (^FTER1008 <> NULL OR Count(^FTER1010) > 0)

            Data Elements:

            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code

            ^FTER1008 = FTER:RP:CentralManagementOutsideAustralia

            ^FTER1010 = FTER:RP:Trustee
        */
        protected void VRATOFTER408031(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (!isStandalone)
            {
                if (report.FTER95 == "V")
                {
                    bool assertion = report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionExists || report.RP_TrusteeCollectionCount > 0;

                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument
                        {
                            Code = "CMN.ATO.FTER.408394",
                            Severity = ProcessMessageSeverity.Error,
                            Description = "Variation must not include central management or trustee details",
                            LongDescription = "If the form is used to advise of a variation, details regarding central management and control and of the trustees must not be completed",
                            Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                            Parameters = new ProcessMessageParameters
                        {
                            new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FTER.408031" },
                            new ProcessMessageParameter { Name = "FTER95", Value = report.FTER95 }
                        }
                        };

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion // VR.ATO.FTER.408031

        #region VR.ATO.FTER.408051
        /*  VR.ATO.FTER.408051
            Dates central management and control of trust was outside Australia must be present

        Legacy Rule Format:
            ^FTER16 = TRUE AND ^FTER17 <> TRUE AND Count(^FTER1009) = 0

        Technical Business Rule Format:
            ^FTER16 = TRUE AND ^FTER17 <> TRUE AND Count(^FTER1009) = 0

        Data Elements:
    
            ^FTER16 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustralia.Indicator
    
            ^FTER17 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
            ^FTER1009 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails
        */
        protected void VRATOFTER408051(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.FTER16.GetValueOrDefault())
            {
                bool assertion = !report.FTER17.GetValueOrDefault() && report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionCount == 0;

                if (assertion)
                {
                    var processMessage = new ProcessMessageDocument
                    {
                        Code = "CMN.ATO.FTER.408374",
                        Severity = ProcessMessageSeverity.Error,
                        Description = "Details of period during which the central management and control of trust was outside Australia must be present",
                        Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                        Parameters = new ProcessMessageParameters
                        {
                            new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FTER.408051" },
                            new ProcessMessageParameter{ Name = "FTER16", Value = GetValueOrEmpty(report.FTER16) },
                            new ProcessMessageParameter { Name = "FTER17", Value = GetValueOrEmpty(report.FTER17) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.FTER.408051

        #region VR.ATO.FTER.408052
        /*  VR.ATO.FTER.408052
            Time central management and control of trust was outside Australia must NOT be present

        Legacy Rule Format:
            ^FTER16 = FALSE AND (^FTER17 <> NULL OR Count(^FTER1009) > 0)

        Technical Business Rule Format:
            ^FTER16 = FALSE AND (^FTER17 <> NULL OR Count(^FTER1009) > 0)

        Data Elements:
    
            ^FTER16 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustralia.Indicator
    
            ^FTER17 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
            ^FTER1009 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails
        */
        protected void VRATOFTER408052(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.FTER16.HasValue && !report.FTER16.GetValueOrDefault())
            {
                bool assertion = report.FTER17 != null || report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionCount > 0;

                if (assertion)
                {
                    var processMessage = new ProcessMessageDocument
                    {
                        Code = "CMN.ATO.FTER.408375",
                        Severity = ProcessMessageSeverity.Error,
                        Description = "Details of period during which the central management and control of trust was outside Australia must not be present",
                        Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                        Parameters = new ProcessMessageParameters
                        {
                            new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FTER.408052" },
                            new ProcessMessageParameter { Name = "FTER16", Value = GetValueOrEmpty(report.FTER16) },
                            new ProcessMessageParameter { Name = "FTER17", Value = GetValueOrEmpty(report.FTER17) }
                        },
                    };

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.FTER.408052

        #region VR.ATO.FTER.408055
        /*  VR.ATO.FTER.408055
            Central management and control - full period indicator and dates must not both be present

        Legacy Rule Format:
            ^FTER17 = TRUE AND Count(^FTER1009) > 0

        Technical Business Rule Format:
            ^FTER17 = TRUE AND Count(^FTER1009) > 0

        Data Elements:
    
            ^FTER17 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
            ^FTER1009 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails
        */
        protected void VRATOFTER408055(List<ProcessMessageDocument> response, FTER2021 report, bool isStandalone)
        {
            if (report.FTER17.GetValueOrDefault())
            {
                bool assertion = report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionCount > 0;

                if (assertion)
                {
                    var processMessage = new ProcessMessageDocument
                    {
                        Code = "CMN.ATO.FTER.408055",
                        Severity = ProcessMessageSeverity.Error,
                        Description = "Central management and control - full period indicator and dates must not both be present",
                        LongDescription = "If Full period indicator for Central management and control outside Australia is 'yes' (true), then specific dates must not be present",
                        Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI",
                        Parameters = new ProcessMessageParameters
                        {
                            new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FTER.408055" },
                            new ProcessMessageParameter { Name = "FTER17", Value = GetValueOrEmpty(report.FTER17) }
                        },
                    };
                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.FTER.408055
    }
}