using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationFTER
{

    public partial class FTER2021Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private FTER2021 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response {get; private set;}

        /// <summary>
        /// Initializes a new instance of the <see cref="FTER2021Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FTER2021Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="FTER2021Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FTER2021Validator(FTER2021 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY"):this(maxParameterNameLength,maxParameterValueLength,emptyParameterValue)
        {                   
            this.ConsumedReport = reportIn;                    
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {                   
            return datetime.GetValueOrDefault().Date;
        }

				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }
        
        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public FTER2021 ConsumedReport { get {return report;} private set {report = value;}}

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();				

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(FTER2021 reportIn, bool isStandalone)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------
            #region Manual Rules - Rules marked for manual coding
            
                #region VR.ATO.FTER.408031
        
                /*  VR.ATO.FTER.408031
                Variation must not include central management or trustee details

                Legacy Rule Format:
                WHERE PARENT RETURN EXISTS
                ^FTER95 = 'V' AND (^FTER1008 <> NULL OR Count(^FTER1010) > 0)
    
                Technical Business Rule Format:
                ^FTER95 = 'V' AND (^FTER1008 <> NULL OR Count(^FTER1010) > 0)
        
                Data Elements:
        
                ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
        
                ^FTER1008 = FTER:RP:CentralManagementOutsideAustralia
        
                ^FTER1010 = FTER:RP:Trustee
                */
    VRATOFTER408031(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408031

    #region VR.ATO.FTER.408051

/*  VR.ATO.FTER.408051
            Details of period during which the central management and control of trust was outside Australia must be present

        Legacy Rule Format:
            ^FTER16 = TRUE AND ^FTER17 <> TRUE AND Count(^FTER1009) = 0

    Technical Business Rule Format:
            ^FTER16 = TRUE AND ^FTER17 <> TRUE AND Count(^FTER1009) = 0

Data Elements:
    
    ^FTER16 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustralia.Indicator
    
    ^FTER17 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
    ^FTER1009 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails
    */
    VRATOFTER408051(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408051

    #region VR.ATO.FTER.408052

/*  VR.ATO.FTER.408052
            Details of period during which the central management and control of trust was outside Australia must not be present

        Legacy Rule Format:
            ^FTER16 = FALSE AND (^FTER17 <> NULL OR Count(^FTER1009) > 0)

    Technical Business Rule Format:
            ^FTER16 = FALSE AND (^FTER17 <> NULL OR Count(^FTER1009) > 0)

Data Elements:
    
    ^FTER16 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustralia.Indicator
    
    ^FTER17 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
    ^FTER1009 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails
    */
    VRATOFTER408052(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408052

    #region VR.ATO.FTER.408055

/*  VR.ATO.FTER.408055
            Central management and control - full period indicator and dates must not both be present

        Legacy Rule Format:
            ^FTER17 = TRUE AND Count(^FTER1009) > 0

    Technical Business Rule Format:
            ^FTER17 = TRUE AND Count(^FTER1009) > 0

Data Elements:
    
    ^FTER17 = FTER:RP:CentralManagementOutsideAustralia:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
    ^FTER1009 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails
    */
    VRATOFTER408055(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408055

    #region VR.ATO.FTER.408077

/*  VR.ATO.FTER.408077
            If the answer to the question 'At any time during the relevant period was the trustee a non-resident for tax purposes?' is 'No' (false), then the Full period indicator must not be present and a period of non residency must NOT be present

        Legacy Rule Format:
            ^FTER1010 <> NULL AND ^FTER36 = FALSE AND (^FTER37 <> NULL OR Count(^FTER1013) > 0)

    Technical Business Rule Format:
            ^FTER1010 <> NULL AND ^FTER36 = FALSE AND (^FTER37 <> NULL OR Count(^FTER1013) > 0)

Data Elements:
    
    ^FTER36 = FTER:RP:Trustee:Residency.TaxPurposesPersonStatus.Indicator
    
    ^FTER37 = FTER:RP:Trustee:Residency.NonResidentFullPeriod.Indicator
    
    ^FTER1010 = FTER:RP:Trustee
    
    ^FTER1013 = FTER:RP:Trustee:NonResidentPeriodDetails
    */
    VRATOFTER408077(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408077

    #region VR.ATO.FTER.408078

/*  VR.ATO.FTER.408078
            Trustee full period indicator or period of non-residency dates must be present

        Legacy Rule Format:
            ^FTER1010 <> NULL AND ^FTER36 = TRUE AND ((^FTER37 = NULL OR ^FTER37 = FALSE) AND Count(^FTER1013) = 0)

    Technical Business Rule Format:
            ^FTER1010 <> NULL AND ^FTER36 = TRUE AND ((^FTER37 = NULL OR ^FTER37 = FALSE) AND Count(^FTER1013) = 0)

Data Elements:
    
    ^FTER37 = FTER:RP:Trustee:Residency.NonResidentFullPeriod.Indicator
    
    ^FTER36 = FTER:RP:Trustee:Residency.TaxPurposesPersonStatus.Indicator
    
    ^FTER1010 = FTER:RP:Trustee
    
    ^FTER1013 = FTER:RP:Trustee:NonResidentPeriodDetails
    */
    VRATOFTER408078(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408078

    #region VR.ATO.FTER.408079

/*  VR.ATO.FTER.408079
            Full period indicator and dates must not both be present

        Legacy Rule Format:
            ^FTER1010 <> NULL AND ^FTER37 = TRUE AND Count(^FTER1013) > 0

    Technical Business Rule Format:
            ^FTER1010 <> NULL AND ^FTER37 = TRUE AND Count(^FTER1013) > 0

Data Elements:
    
    ^FTER37 = FTER:RP:Trustee:Residency.NonResidentFullPeriod.Indicator
    
    ^FTER1010 = FTER:RP:Trustee
    
    ^FTER1013 = FTER:RP:Trustee:NonResidentPeriodDetails
    */
    VRATOFTER408079(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408079

    #region VR.ATO.FTER.408084

/*  VR.ATO.FTER.408084
            Trustee non-resident period must not be before the substituted accounting period

        Legacy Rule Format:
            ^FTER41 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER41)

    Technical Business Rule Format:
            ^FTER41 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER41)

Data Elements:
    
    ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
    
    ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
    */
    VRATOFTER408084(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408084

    #region VR.ATO.FTER.408085

/*  VR.ATO.FTER.408085
            Trustee non-resident period must not be before the election commencement time

        Legacy Rule Format:
            ^FTER43 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER43)

    Technical Business Rule Format:
            ^FTER43 <> NULL AND AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < ^FTER43)

Data Elements:
    
    ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
    
    ^FTER43 = FTER:RP:Elections.Commencement.Date
    */
    VRATOFTER408085(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408085

    #region VR.ATO.FTER.408086

/*  VR.ATO.FTER.408086
            Trustee non-resident period must not be before the start of income year specified

        Legacy Rule Format:
            ^FTER40 <> NULL AND ^FTER41 = NULL AND ^FTER43 = NULL AND (AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < AsDate(Concat(^FTER40 - 1, '-07-01'))))

    Technical Business Rule Format:
            ^FTER40 <> NULL AND ^FTER41 = NULL AND ^FTER43 = NULL AND (AnyOccurrence(^FTER38, ^FTER38 <> NULL AND ^FTER38 < AsDate(Concat(^FTER40 - 1, '-07-01'))))

Data Elements:
    
    ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
    
    ^FTER40 = FTER:RP:ElectionOrVariationIncomeYear:Elections.FamilyTrustElectionStatus.Year
    
    ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
    ^FTER43 = FTER:RP:Elections.Commencement.Date
    */
    VRATOFTER408086(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408086

    #region VR.ATO.FTER.408164

/*  VR.ATO.FTER.408164
            One Declaration Statement Type Code must be 'TrueAndCorrect'

        Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^FTER1000 = NULL AND Count(^FTER99 = 'TrueAndCorrect') <> 1

    Technical Business Rule Format:
            Count(^Context) = 0 AND Count(^FTER99 = 'TrueAndCorrect') <> 1

Data Elements:
    
    ^FTER99 = FTER:RP:DeclarationOfTrustee:Declaration.StatementType.Code
    
    ^Context = Context
    */
    VRATOFTER408164(response, report, isStandalone);
    #endregion // VR.ATO.FTER.408164
#endregion Manual Rules - Rules marked for manual coding

            #region Repeating report.RP_TrusteeCollection  
            if (report.RP_TrusteeCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (FTER2021.RP_Trustee trustee in report.RP_TrusteeCollection)
                {
                    itemIndex2++;
                        VRATOFTER408003( isStandalone, trustee, itemIndex2);
                        VRATOFTER408065( isStandalone, trustee, itemIndex2);
                        VRATOFTER408066( isStandalone, trustee, itemIndex2);
                        VRATOFTER408071( isStandalone, trustee, itemIndex2);
                        VRATOFTER408121( isStandalone, trustee, itemIndex2);
                        VRATOFTER408244( isStandalone, trustee, itemIndex2);
                        VRATOFTER408245( isStandalone, trustee, itemIndex2);
                        VRATOFTER408246( isStandalone, trustee, itemIndex2);
                        VRATOFTER408247( isStandalone, trustee, itemIndex2);
                        VRATOFTER408248( isStandalone, trustee, itemIndex2);
                        VRATOFTER408249( isStandalone, trustee, itemIndex2);
                        VRATOFTER408250( isStandalone, trustee, itemIndex2);
                        VRATOFTER408251( isStandalone, trustee, itemIndex2);
                        VRATOFTER408252( isStandalone, trustee, itemIndex2);
                        VRATOFTER408253( isStandalone, trustee, itemIndex2);
                        VRATOFTER408254( isStandalone, trustee, itemIndex2);
                        VRATOFTER408255( isStandalone, trustee, itemIndex2);
                        VRATOFTER408256( isStandalone, trustee, itemIndex2);
                        VRATOFTER408257( isStandalone, trustee, itemIndex2);
                        VRATOFTER408258( isStandalone, trustee, itemIndex2);
                        VRATOFTER408259( isStandalone, trustee, itemIndex2);
                        VRATOFTER408260( isStandalone, trustee, itemIndex2);
                        VRATOFTER408261( isStandalone, trustee, itemIndex2);
                        VRATOFTER408262( isStandalone, trustee, itemIndex2);
                        VRATOFTER408263( isStandalone, trustee, itemIndex2);
                        VRATOFTER408264( isStandalone, trustee, itemIndex2);
                        VRATOFTER408265( isStandalone, trustee, itemIndex2);
                        VRATOFTER408266( isStandalone, trustee, itemIndex2);
                        VRATOFTER408267( isStandalone, trustee, itemIndex2);
                        VRATOFTER408268( isStandalone, trustee, itemIndex2);
                        VRATOFTER408269( isStandalone, trustee, itemIndex2);
                        VRATOFTER408270( isStandalone, trustee, itemIndex2);
                        VRATOFTER408271( isStandalone, trustee, itemIndex2);
                        VRATOFTER408272( isStandalone, trustee, itemIndex2);
                        VRATOFTER408273( isStandalone, trustee, itemIndex2);
                        VRATOFTER408274( isStandalone, trustee, itemIndex2);
                        VRATOFTER408276( isStandalone, trustee, itemIndex2);
                        VRATOFTER408277( isStandalone, trustee, itemIndex2);
                        VRATOFTER408278( isStandalone, trustee, itemIndex2);
                        VRATOFTER408279( isStandalone, trustee, itemIndex2);
                        VRATOFTER408280( isStandalone, trustee, itemIndex2);
                        VRATOFTER408282( isStandalone, trustee, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop
                VRATOFTER408027( isStandalone);
                VRATOFTER408030( isStandalone);
                VRATOFTER408032( isStandalone);
                VRATOFTER408053( isStandalone);
                VRATOFTER408054( isStandalone);
                VRATOFTER408087( isStandalone);
                VRATOFTER408088( isStandalone);
                VRATOFTER408096( isStandalone);
                VRATOFTER408098( isStandalone);
                VRATOFTER408100( isStandalone);
                VRATOFTER408101( isStandalone);
                VRATOFTER408102( isStandalone);
                VRATOFTER408107( isStandalone);
                VRATOFTER408114( isStandalone);
                VRATOFTER408116( isStandalone);
                VRATOFTER408145( isStandalone);
                VRATOFTER408155( isStandalone);
                VRATOFTER408201( isStandalone);
                VRATOFTER408203( isStandalone);
                VRATOFTER408204( isStandalone);
                VRATOFTER408207( isStandalone);
                VRATOFTER408208( isStandalone);
                VRATOFTER408209( isStandalone);
                VRATOFTER408210( isStandalone);
                VRATOFTER408211( isStandalone);
                VRATOFTER408212( isStandalone);
                VRATOFTER408213( isStandalone);
                VRATOFTER408214( isStandalone);
                VRATOFTER408215( isStandalone);
                VRATOFTER408216( isStandalone);
                VRATOFTER408218( isStandalone);
                VRATOFTER408219( isStandalone);
                VRATOFTER408220( isStandalone);
                VRATOFTER408221( isStandalone);
                VRATOFTER408222( isStandalone);
                VRATOFTER408224( isStandalone);
                VRATOFTER408225( isStandalone);
                VRATOFTER408226( isStandalone);
                VRATOFTER408227( isStandalone);
                VRATOFTER408228( isStandalone);
                VRATOFTER408229( isStandalone);
                VRATOFTER408230( isStandalone);
                VRATOFTER408231( isStandalone);
                VRATOFTER408232( isStandalone);
                VRATOFTER408233( isStandalone);
                VRATOFTER408234( isStandalone);
                VRATOFTER408236( isStandalone);
                VRATOFTER408237( isStandalone);
                VRATOFTER408238( isStandalone);
                VRATOFTER408239( isStandalone);
                VRATOFTER408240( isStandalone);
                VRATOFTER408242( isStandalone);
                VRATOFTER408284( isStandalone);
                VRATOFTER408285( isStandalone);
                VRATOFTER408286( isStandalone);
                VRATOFTER408287( isStandalone);
                VRATOFTER408288( isStandalone);
                VRATOFTER408289( isStandalone);
                VRATOFTER408290( isStandalone);
                VRATOFTER408291( isStandalone);
                VRATOFTER408292( isStandalone);
                VRATOFTER408293( isStandalone);
                VRATOFTER408294( isStandalone);
                VRATOFTER408295( isStandalone);
                VRATOFTER408296( isStandalone);
                VRATOFTER408297( isStandalone);
                VRATOFTER408298( isStandalone);
                VRATOFTER408299( isStandalone);
                VRATOFTER408300( isStandalone);
                VRATOFTER408301( isStandalone);
                VRATOFTER408302( isStandalone);
                VRATOFTER408303( isStandalone);
                VRATOFTER408304( isStandalone);
                VRATOFTER408305( isStandalone);
                VRATOFTER408306( isStandalone);
                VRATOFTER408307( isStandalone);
                VRATOFTER408308( isStandalone);
                VRATOFTER408310( isStandalone);
                VRATOFTER408311( isStandalone);
                VRATOFTER408312( isStandalone);
                VRATOFTER408313( isStandalone);
                VRATOFTER408314( isStandalone);
                VRATOFTER408316( isStandalone);
                VRATOFTER408317( isStandalone);
                VRATOFTER408366( isStandalone);
                VRATOFTER408370( isStandalone);
                VRATOFTER408376( isStandalone);
                VRATOFTER408395( isStandalone);

            #region Repeating report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection  
            if (report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection != null)
            {    
                int itemIndex3 = 0;
                foreach (FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails periodDetails in report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection)
                {
                    itemIndex3++;
                        VRATOFTER408061( isStandalone, periodDetails, itemIndex3);
                        VRATOFTER408062( isStandalone, periodDetails, itemIndex3);
                        VRATOFTER408063( isStandalone, periodDetails, itemIndex3);
                        VRATOFTER408243( isStandalone, periodDetails, itemIndex3);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.RP_FamilyTrustRevocation_InterposedEntityCollection  
            if (report.RP_FamilyTrustRevocation_InterposedEntityCollection != null)
            {    
                int itemIndex3 = 0;
                foreach (FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity in report.RP_FamilyTrustRevocation_InterposedEntityCollection)
                {
                    itemIndex3++;
                        VRATOFTER408118( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408119( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408122( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408318( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408319( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408320( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408321( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408322( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408323( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408324( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408325( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408326( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408327( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408328( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408329( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408330( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408332( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408333( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408334( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408335( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408336( isStandalone, interposedEntity, itemIndex3);
                        VRATOFTER408338( isStandalone, interposedEntity, itemIndex3);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.RP_DeclarationOfTrusteeCollection  
            if (report.RP_DeclarationOfTrusteeCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (FTER2021.RP_DeclarationOfTrustee declarationOfTrustee in report.RP_DeclarationOfTrusteeCollection)
                {
                    itemIndex2++;
                        VRATOFTER408130( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408135( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408339( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408340( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408341( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408342( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408343( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408344( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408345( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408346( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408347( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408348( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408349( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408350( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408351( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408352( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408353( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408354( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408355( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408356( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408357( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408358( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408359( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408360( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408361( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408362( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408363( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408364( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408380( isStandalone, declarationOfTrustee, itemIndex2);
                        VRATOFTER408381( isStandalone, declarationOfTrustee, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.RP_TrusteeCollection  
            if (report.RP_TrusteeCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (FTER2021.RP_Trustee trustee in report.RP_TrusteeCollection)
                {
                    itemIndex2++;
        
                    #region Repeating trustee.RP_Trustee_NonResidentPeriodDetailsCollection  
                    if (trustee.RP_Trustee_NonResidentPeriodDetailsCollection != null)
                    {    
                        int itemIndex3 = 0;
                        foreach (FTER2021.RP_Trustee.RP_Trustee_NonResidentPeriodDetails nonResidentPeriodDetails in trustee.RP_Trustee_NonResidentPeriodDetailsCollection)
                        {
                            itemIndex3++;
                                VRATOFTER408283( isStandalone, trustee, nonResidentPeriodDetails, itemIndex3);
                            }
                        }
                
                        #endregion // Foreach loop
                        }
                    }
            
                    #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
				
			
    
      #region VR.ATO.FTER.408003
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408003( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408003
                    Trustee individual name or non-individual name must be present
    
                    Legacy Rule Format:
                    ^FTER1010 <> NULL AND ^FTER1011 = NULL AND ^FTER27 = BLANK
        
                    Technical Business Rule Format:
                    ^FTER1010 <> NULL AND ^FTER1011 = NULL AND ^FTER27 = BLANK
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
            
                    ^FTER1010 = FTER:RP:Trustee
            
                    ^FTER1011 = FTER:RP:Trustee:IndividualNameDetails
                    */
                    assertion = (report.RP_TrusteeCollectionExists != false && trustee.RP_Trustee_IndividualNameDetailsCollectionExists == false && string.IsNullOrWhiteSpace(trustee.FTER27) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408003", Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee individual name or non-individual name must be present",
                            LongDescription = @"For an election, details of each trustee's family name or non-individual name must be present",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408003"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408027
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408027( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408027
            Family trust revocation details must be left blank
    
            Legacy Rule Format:
            InSet(^FTER95, '"E", "V"') AND ^FTER1019 <> NULL

            Technical Business Rule Format:
            InSet(^FTER95, '"E", "V"') AND ^FTER1019 <> NULL
    
            Data Elements:
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER1019 = FTER:RP:FamilyTrustRevocation
            */
            assertion = (IsMatch(report.FTER95, @"^(E|V)$") && report.RP_FamilyTrustRevocationCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408027", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family trust revocation details must be left blank",
                    LongDescription = @"If the election-revocation code is  'E' - election or 'V' - variation, then revocation details must not be completed",
                    Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:ElectionsNotificationOfElectionBeingRevokedY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408027"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER95", Value = report.FTER95 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408030
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408030( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408030
            Details required for a variation must be completed
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER95 = 'V' AND ((^FTER40 = NULL AND ^FTER43 = NULL) OR ^FTER1016 = NULL)

            Technical Business Rule Format:
            ^FTER95 = 'V' AND ((^FTER40 = NULL AND ^FTER43 = NULL) OR ^FTER1016 = NULL)
    
            Data Elements:
    
            ^FTER40 = FTER:RP:ElectionOrVariationIncomeYear:Elections.FamilyTrustElectionStatus.Year
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
    
            ^FTER1016 = FTER:RP:SpecifiedIndividual
            */
            if (!isStandalone)
            {
                assertion = (report.FTER95 == @"V" && (report.FTER40 == null && report.FTER43 == null || report.RP_SpecifiedIndividualCollectionExists == false));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408030", Severity = ProcessMessageSeverity.Error,
                        Description = @"Details required for a variation must be completed",
                        LongDescription = @"When used for a variation the 'Income Year Specified' or 'Commencement Year' must be completed; and the TFN, name, address and date of birth of the Specified Individual must be completed",
                        Location = "/tns:FTER/tns:RP/tns:ElectionOrVariationIncomeYear/tns:ElectionsFamilyTrustElectionStatusY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408030"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER40", Value = GetValueOrEmpty(report.FTER40) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER43", Value = GetValueOrEmptyDate(report.FTER43) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408032
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408032( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408032
            Revocation indicated and the date family trust election ceases to be in force or income year of revocation is missing
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER95 = 'R' AND ^FTER57 = NULL AND ^FTER96 = NULL

            Technical Business Rule Format:
            ^FTER95 = 'R' AND ^FTER57 = NULL AND ^FTER96 = NULL
    
            Data Elements:
    
            ^FTER57 = FTER:RP:FamilyTrustRevocation:Elections.Revocation.Date
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER96 = FTER:RP:FamilyTrustRevocation:Elections.FamilyTrustRevocationStatusEffective.Year
            */
            if (!isStandalone)
            {
                assertion = (report.FTER95 == @"R" && report.FTER57 == null && report.FTER96 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408032", Severity = ProcessMessageSeverity.Error,
                        Description = @"Either the date the election ceases to be in force, or the income year of revocation must be present",
                        LongDescription = @"For a revocation, either the date the election ceases to be in force must be present or the income year from which the revocation is to be effective must be present",
                        Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:ElectionsD",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408032"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER57", Value = GetValueOrEmptyDate(report.FTER57) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER96", Value = GetValueOrEmpty(report.FTER96) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408053( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408053
            Election details must be completed
    
            Legacy Rule Format:
            ^FTER95 = 'E' AND (^FTER1008 = NULL OR ^FTER1014 = NULL OR ^FTER1016 = NULL OR Count(^FTER1010) = 0)

            Technical Business Rule Format:
            ^FTER95 = 'E' AND (^FTER1008 = NULL OR ^FTER1014 = NULL OR ^FTER1016 = NULL OR Count(^FTER1010) = 0)
    
            Data Elements:
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER1008 = FTER:RP:CentralManagementOutsideAustralia
    
            ^FTER1010 = FTER:RP:Trustee
    
            ^FTER1014 = FTER:RP:ElectionOrVariationIncomeYear
    
            ^FTER1016 = FTER:RP:SpecifiedIndividual
            */
            assertion = (report.FTER95 == @"E" && (report.RP_CentralManagementOutsideAustraliaCollectionExists == false || report.RP_ElectionOrVariationIncomeYearCollectionExists == false || report.RP_SpecifiedIndividualCollectionExists == false || Count(report.RP_TrusteeCollectionCount) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Election details must be completed",
                    LongDescription = @"For an election, the central management question, details of at least one trustee, income year and details of the specified individual must all be present",
                    Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408053"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER95", Value = report.FTER95 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408054( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408054
            Revocation details must not be provided with election or variation details
    
            Legacy Rule Format:
            (^FTER1008 <> NULL OR Count(^FTER1010) > 0 OR ^FTER1014 <> NULL OR ^FTER43 <> NULL OR ^FTER1016 <> NULL) AND ^FTER1019 <> NULL

            Technical Business Rule Format:
            (^FTER1008 <> NULL OR Count(^FTER1010) > 0 OR ^FTER1014 <> NULL OR ^FTER43 <> NULL OR ^FTER1016 <> NULL) AND ^FTER1019 <> NULL
    
            Data Elements:
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
    
            ^FTER1008 = FTER:RP:CentralManagementOutsideAustralia
    
            ^FTER1010 = FTER:RP:Trustee
    
            ^FTER1014 = FTER:RP:ElectionOrVariationIncomeYear
    
            ^FTER1016 = FTER:RP:SpecifiedIndividual
    
            ^FTER1019 = FTER:RP:FamilyTrustRevocation
            */
            assertion = ((report.RP_CentralManagementOutsideAustraliaCollectionExists != false || Count(report.RP_TrusteeCollectionCount) > 0 || report.RP_ElectionOrVariationIncomeYearCollectionExists != false || report.FTER43 != null || report.RP_SpecifiedIndividualCollectionExists != false) && report.RP_FamilyTrustRevocationCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Revocation details must not be provided with election or variation details",
                    LongDescription = @"A single FTER schedule must not contain details of a revocation in addition to details related to an election or a variation",
                    Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408054"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER43", Value = GetValueOrEmptyDate(report.FTER43) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408061
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408061( bool isStandalone, FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails periodDetails, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408061
                    Central management and control of the trust outside of Australia period must not be before start of the substituted accounting period
    
                    Legacy Rule Format:
                    ^FTER18 <> NULL AND ^FTER41 <> NULL AND (AnyOccurrence(^FTER18, ^FTER18 < ^FTER41))
        
                    Technical Business Rule Format:
                    ^FTER18 <> NULL AND ^FTER41 <> NULL AND (AnyOccurrence(^FTER18, ^FTER18 < ^FTER41))
            
                    Data Elements:
            
                    ^FTER18 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails:Period.Start.Date
            
                    ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
                    */
                    assertion = (periodDetails.FTER18 != null && report.FTER41 != null && (report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection == null ? false : report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection.Any(FTER18Repeat => FTER18Repeat.FTER18.GetValueOrDefault() < report.FTER41.GetValueOrDefault())));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408383", Severity = ProcessMessageSeverity.Error,
                            Description = @"Central management and control of the trust outside of Australia period must not be before start of the substituted accounting period",
                            Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:PeriodDetailsCollection/tns:PeriodDetails" + OccurrenceIndex(periodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408061"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER18", Value = GetValueOrEmptyDate(periodDetails.FTER18) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER41", Value = GetValueOrEmptyDate(report.FTER41) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408062
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408062( bool isStandalone, FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails periodDetails, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408062
                    Central management and control of the trust outside of Australia period must not be before the election commencement date
    
                    Legacy Rule Format:
                    ^FTER18 <> NULL AND ^FTER43 <> NULL AND (AnyOccurrence(^FTER18, ^FTER18 < ^FTER43))
        
                    Technical Business Rule Format:
                    ^FTER18 <> NULL AND ^FTER43 <> NULL AND (AnyOccurrence(^FTER18, ^FTER18 < ^FTER43))
            
                    Data Elements:
            
                    ^FTER18 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails:Period.Start.Date
            
                    ^FTER43 = FTER:RP:Elections.Commencement.Date
                    */
                    assertion = (periodDetails.FTER18 != null && report.FTER43 != null && (report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection == null ? false : report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection.Any(FTER18Repeat => FTER18Repeat.FTER18.GetValueOrDefault() < report.FTER43.GetValueOrDefault())));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408382", Severity = ProcessMessageSeverity.Error,
                            Description = @"Central management and control of the trust outside of Australia period must not be before the election commencement date",
                            Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:PeriodDetailsCollection/tns:PeriodDetails" + OccurrenceIndex(periodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408062"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER18", Value = GetValueOrEmptyDate(periodDetails.FTER18) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER43", Value = GetValueOrEmptyDate(report.FTER43) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408063
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408063( bool isStandalone, FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails periodDetails, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408063
                    Central management and control of the trust outside of Australia period must not be before the specified income year
    
                    Legacy Rule Format:
                    InSet(^FTER95, '"E", "V"') AND ^FTER40 <> NULL AND ^FTER41 = NULL AND ^FTER18 <> NULL AND (AnyOccurrence(^FTER18, ^FTER18 < AsDate(Concat(^FTER40 - 1, '-07-01'))))
        
                    Technical Business Rule Format:
                    InSet(^FTER95, '"E", "V"') AND ^FTER40 <> NULL AND ^FTER41 = NULL AND ^FTER18 <> NULL AND (AnyOccurrence(^FTER18, ^FTER18 < AsDate(Concat(^FTER40 - 1, '-07-01'))))
            
                    Data Elements:
            
                    ^FTER18 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails:Period.Start.Date
            
                    ^FTER40 = FTER:RP:ElectionOrVariationIncomeYear:Elections.FamilyTrustElectionStatus.Year
            
                    ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
            
                    ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
                    */
                    assertion = (IsMatch(report.FTER95, @"^(E|V)$") && report.FTER40 != null && report.FTER41 == null && periodDetails.FTER18 != null && (report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection == null ? false : report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection.Any(FTER18Repeat => FTER18Repeat.FTER18.GetValueOrDefault() < AsDate(string.Concat(report.FTER40.GetValueOrDefault() - 1, @"-07-01")))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408384", Severity = ProcessMessageSeverity.Error,
                            Description = @"Central management and control of the trust outside of Australia period must not be before the specified income year",
                            Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:PeriodDetailsCollection/tns:PeriodDetails" + OccurrenceIndex(periodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408063"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER95", Value = report.FTER95 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER40", Value = GetValueOrEmpty(report.FTER40) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER41", Value = GetValueOrEmptyDate(report.FTER41) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER18", Value = GetValueOrEmptyDate(periodDetails.FTER18) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408065
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408065( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408065
                    Trustee does not have a TFN indicator must not be 'yes' (true)
    
                    Legacy Rule Format:
                    ^FTER1010 <> NULL AND ^FTER21 = TRUE AND ^FTER20 <> BLANK
        
                    Technical Business Rule Format:
                    ^FTER1010 <> NULL AND ^FTER21 = TRUE AND ^FTER20 <> BLANK
            
                    Data Elements:
            
                    ^FTER21 = FTER:RP:Trustee:Identifiers.TaxFileNumberInexistent.Indicator
            
                    ^FTER20 = FTER:RP:Trustee:Identifiers.TaxFileNumber.Identifier
            
                    ^FTER1010 = FTER:RP:Trustee
                    */
                    assertion = (report.RP_TrusteeCollectionExists != false && trustee.FTER21 == true && string.IsNullOrWhiteSpace(trustee.FTER20) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408065", Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee does not have a TFN indicator must not be 'yes' (true)",
                            LongDescription = @"If the TFN of trustee is present, then the 'trustee does not have a TFN must not be 'yes' (true)",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:TaxFileNumberInexistentI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408065"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER21", Value = GetValueOrEmpty(trustee.FTER21) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER20", Value = GetValueOrEmpty(trustee.FTER20) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408066
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408066( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408066
                    Trustee Tax file number must be present or Trustee does not have a TFN indicator must be 'yes' (true)
    
                    Legacy Rule Format:
                    ^FTER1010 <> NULL AND ^FTER20 = BLANK AND ^FTER21 <> TRUE
        
                    Technical Business Rule Format:
                    ^FTER1010 <> NULL AND ^FTER20 = BLANK AND ^FTER21 <> TRUE
            
                    Data Elements:
            
                    ^FTER20 = FTER:RP:Trustee:Identifiers.TaxFileNumber.Identifier
            
                    ^FTER21 = FTER:RP:Trustee:Identifiers.TaxFileNumberInexistent.Indicator
            
                    ^FTER1010 = FTER:RP:Trustee
                    */
                    assertion = (report.RP_TrusteeCollectionExists != false && string.IsNullOrWhiteSpace(trustee.FTER20) == true && trustee.FTER21 != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408393", Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee Tax file number must be present or Trustee does not have a TFN indicator must be 'yes' (true)",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408066"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER20", Value = GetValueOrEmpty(trustee.FTER20) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER21", Value = GetValueOrEmpty(trustee.FTER21) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408071
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408071( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408071
                    A trustee must not include both an individual and a non-individual name
    
                    Legacy Rule Format:
                    ^FTER1010 <> NULL AND ^FTER1011 <> NULL AND ^FTER27 <> NULL
        
                    Technical Business Rule Format:
                    ^FTER1010 <> NULL AND ^FTER1011 <> NULL AND ^FTER27 <> NULL
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
            
                    ^FTER1010 = FTER:RP:Trustee
            
                    ^FTER1011 = FTER:RP:Trustee:IndividualNameDetails
                    */
                    assertion = (report.RP_TrusteeCollectionExists != false && trustee.RP_Trustee_IndividualNameDetailsCollectionExists != false && trustee.FTER27 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408071", Severity = ProcessMessageSeverity.Error,
                            Description = @"A trustee must not include both an individual and a non-individual name",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408071"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408087
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408087( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408087
            Income year specified must not be less than 2005
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^FTER95 = 'E' AND ^FTER40 < 2005

            Technical Business Rule Format:
            ^FTER95 = 'E' AND ^FTER40 < 2005
    
            Data Elements:
    
            ^FTER40 = FTER:RP:ElectionOrVariationIncomeYear:Elections.FamilyTrustElectionStatus.Year
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
            */
            if (isStandalone)
            {
                assertion = (report.FTER95 == @"E" && report.FTER40.GetValueOrDefault() < 2005);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408087", Severity = ProcessMessageSeverity.Error,
                        Description = @"Income year specified must not be less than 2005",
                        Location = "/tns:FTER/tns:RP/tns:ElectionOrVariationIncomeYear/tns:ElectionsFamilyTrustElectionStatusY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408087"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER40", Value = GetValueOrEmpty(report.FTER40) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408088
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408088( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408088
            Election commencement time must not be before the start of specified income year
    
            Legacy Rule Format:
            ^FTER40 <> NULL AND ^FTER43 <> NULL AND ^FTER41 = NULL AND ^FTER43 < AsDate(Concat(^FTER40 - 1, '-07-01'))

            Technical Business Rule Format:
            ^FTER40 <> NULL AND ^FTER43 <> NULL AND ^FTER41 = NULL AND ^FTER43 < AsDate(Concat(^FTER40 - 1, '-07-01'))
    
            Data Elements:
    
            ^FTER40 = FTER:RP:ElectionOrVariationIncomeYear:Elections.FamilyTrustElectionStatus.Year
    
            ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
            */
            assertion = (report.FTER40 != null && report.FTER43 != null && report.FTER41 == null && report.FTER43.GetValueOrDefault() < AsDate(string.Concat(report.FTER40.GetValueOrDefault() - 1, @"-07-01")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408088", Severity = ProcessMessageSeverity.Error,
                    Description = @"Election commencement time must not be before the start of specified income year",
                    Location = "/tns:FTER/tns:RP/tns:ElectionOrVariationIncomeYear/tns:ElectionsFamilyTrustElectionStatusY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408088"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER40", Value = GetValueOrEmpty(report.FTER40) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER43", Value = GetValueOrEmptyDate(report.FTER43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER41", Value = GetValueOrEmptyDate(report.FTER41) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408096
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408096( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408096
            Election commencement time must not be before the start of the substituted accounting period
    
            Legacy Rule Format:
            ^FTER43 <> NULL AND ^FTER41 <> NULL AND ^FTER43 < ^FTER41

            Technical Business Rule Format:
            ^FTER43 <> NULL AND ^FTER41 <> NULL AND ^FTER43 < ^FTER41
    
            Data Elements:
    
            ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
            */
            assertion = (report.FTER43 != null && report.FTER41 != null && report.FTER43.GetValueOrDefault() < report.FTER41.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408096", Severity = ProcessMessageSeverity.Error,
                    Description = @"Election commencement time must not be before the start of the substituted accounting period",
                    Location = "/tns:FTER/tns:RP/tns:ElectionOrVariationIncomeYear/tns:SubstitutedAccountingPeriodDetails/tns:StartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408096"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER43", Value = GetValueOrEmptyDate(report.FTER43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER41", Value = GetValueOrEmptyDate(report.FTER41) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408098
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408098( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408098
            Specified individual date of birth cannot be later than the election commencement date
    
            Legacy Rule Format:
            ^FTER43 <> NULL AND ^FTER46 <> NULL AND ^FTER46 > ^FTER43

            Technical Business Rule Format:
            ^FTER43 <> NULL AND ^FTER46 <> NULL AND ^FTER46 > ^FTER43
    
            Data Elements:
    
            ^FTER46 = FTER:RP:SpecifiedIndividual:PersonDemographicDetails.Birth.Date
    
            ^FTER43 = FTER:RP:Elections.Commencement.Date
            */
            assertion = (report.FTER43 != null && report.FTER46 != null && report.FTER46.GetValueOrDefault() > report.FTER43.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408378", Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified individual date of birth cannot be later than the election commencement date",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonDemographicDetailsBirthD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408098"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER43", Value = GetValueOrEmptyDate(report.FTER43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER46", Value = GetValueOrEmptyDate(report.FTER46) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408100
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408100( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408100
            Specified individual does not have a TFN indicator must not be 'yes' (true)
    
            Legacy Rule Format:
            ^FTER45 = TRUE AND ^FTER44 <> BLANK

            Technical Business Rule Format:
            ^FTER45 = TRUE AND ^FTER44 <> BLANK
    
            Data Elements:
    
            ^FTER45 = FTER:RP:SpecifiedIndividual:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^FTER44 = FTER:RP:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.FTER45 == true && string.IsNullOrWhiteSpace(report.FTER44) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408387", Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified individual does not have a TFN indicator must not be 'yes' (true)",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:TaxFileNumberInexistentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER45", Value = GetValueOrEmpty(report.FTER45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER44", Value = GetValueOrEmpty(report.FTER44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408101
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408101( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408101
            Specified individual for trust election being revoked does not have a TFN indicator must not be 'yes' (true)
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER61 = TRUE AND ^FTER60 <> BLANK

            Technical Business Rule Format:
            ^FTER61 = TRUE AND ^FTER60 <> BLANK
    
            Data Elements:
    
            ^FTER61 = FTER:RP:FamilyTrustRevocation:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^FTER60 = FTER:RP:FamilyTrustRevocation:Identifiers.TaxFileNumber.Identifier
            */
            if (!isStandalone)
            {
                assertion = (report.FTER61 == true && string.IsNullOrWhiteSpace(report.FTER60) != true);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408388", Severity = ProcessMessageSeverity.Error,
                        Description = @"Specified individual for trust election being revoked does not have a TFN indicator must not be 'yes' (true)",
                        Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:TaxFileNumberInexistentI",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408101"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER61", Value = GetValueOrEmpty(report.FTER61) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER60", Value = GetValueOrEmpty(report.FTER60) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408102
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408102( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408102
            Family trust revocation details must be present
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER95 = 'R' AND ^FTER1019 = NULL

            Technical Business Rule Format:
            ^FTER95 = 'R' AND ^FTER1019 = NULL
    
            Data Elements:
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER1019 = FTER:RP:FamilyTrustRevocation
            */
            if (!isStandalone)
            {
                assertion = (report.FTER95 == @"R" && report.RP_FamilyTrustRevocationCollectionExists == false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408369", Severity = ProcessMessageSeverity.Error,
                        Description = @"Family trust revocation details must be present",
                        Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:ElectionsNotificationOfElectionBeingRevokedY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408102"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408107
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408107( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408107
            Specified individual Tax file number must be present or Specified individual does not have a TFN indicator must be 'yes' (true)
    
            Legacy Rule Format:
            ^FTER1016 <> NULL AND ^FTER44 = NULL AND ^FTER45 <> TRUE

            Technical Business Rule Format:
            ^FTER1016 <> NULL AND ^FTER44 = NULL AND ^FTER45 <> TRUE
    
            Data Elements:
    
            ^FTER44 = FTER:RP:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
    
            ^FTER45 = FTER:RP:SpecifiedIndividual:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^FTER1016 = FTER:RP:SpecifiedIndividual
            */
            assertion = (report.RP_SpecifiedIndividualCollectionExists != false && report.FTER44 == null && report.FTER45 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408392", Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified individual Tax file number must be present or Specified individual does not have a TFN indicator must be 'yes' (true)",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408107"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER44", Value = GetValueOrEmpty(report.FTER44) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER45", Value = GetValueOrEmpty(report.FTER45) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408114
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408114( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408114
            Tax file number of the specified individual for the trust election being revoked must be present or Specified individual for trust election being revoked does not have a TFN indicator must be 'yes' (true)
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER95 = 'R' AND ^FTER60 = BLANK AND ^FTER61 <> TRUE

            Technical Business Rule Format:
            ^FTER95 = 'R' AND ^FTER60 = BLANK AND ^FTER61 <> TRUE
    
            Data Elements:
    
            ^FTER60 = FTER:RP:FamilyTrustRevocation:Identifiers.TaxFileNumber.Identifier
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER61 = FTER:RP:FamilyTrustRevocation:Identifiers.TaxFileNumberInexistent.Indicator
            */
            if (!isStandalone)
            {
                assertion = (report.FTER95 == @"R" && string.IsNullOrWhiteSpace(report.FTER60) == true && report.FTER61 != true);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408391", Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax file number of the specified individual for the trust election being revoked must be present or Specified individual for trust election being revoked does not have a TFN indicator must be 'yes' (true)",
                        Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:TaxFileNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408114"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER60", Value = GetValueOrEmpty(report.FTER60) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER61", Value = GetValueOrEmpty(report.FTER61) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408116
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408116( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408116
            Year of election must not be later than return year
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER95 = 'R' AND ^FTER58 <> NULL AND ^FTER172 <> NULL AND ^FTER172 > ^FTER58

            Technical Business Rule Format:
            ^FTER95 = 'R' AND ^FTER58 <> NULL AND ^FTER172 <> NULL AND ^FTER172 > ^FTER58
    
            Data Elements:
    
            ^FTER58 = FTER:RP:FamilyTrustRevocation:Elections.NotificationOfElectionBeingRevoked.Year
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER172 = FTER:RP:FamilyTrustRevocation:Elections.FamilyTrustElectionStatus.Year
            */
            if (!isStandalone)
            {
                assertion = (report.FTER95 == @"R" && report.FTER58 != null && report.FTER172 != null && report.FTER172.GetValueOrDefault() > report.FTER58.GetValueOrDefault());
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408116", Severity = ProcessMessageSeverity.Error,
                        Description = @"Year of election must not be later than return year",
                        LongDescription = @"For a revocation, the income year as it appeared in the family trust election included in the return for the trust, must not be later than the income year of the tax return in which the election was made",
                        Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:ElectionsNotificationOfElectionBeingRevokedY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408116"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER58", Value = GetValueOrEmpty(report.FTER58) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER172", Value = GetValueOrEmpty(report.FTER172) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408118
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408118( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408118
                    Entity does not have a TFN indicator must not be 'yes' (true)
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    ^FTER1020 <> NULL AND ^FTER63 = TRUE AND ^FTER64 <> BLANK
        
                    Technical Business Rule Format:
                    ^FTER1020 <> NULL AND ^FTER63 = TRUE AND ^FTER64 <> BLANK
            
                    Data Elements:
            
                    ^FTER63 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.TaxFileNumberInexistent.Indicator
            
                    ^FTER1020 = FTER:RP:FamilyTrustRevocation:InterposedEntity
            
                    ^FTER64 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.TaxFileNumber.Identifier
                    */
                    if (!isStandalone)
                    {
                        assertion = (report.RP_FamilyTrustRevocation_InterposedEntityCollectionExists != false && interposedEntity.FTER63 == true && string.IsNullOrWhiteSpace(interposedEntity.FTER64) != true);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408386", Severity = ProcessMessageSeverity.Error,
                                Description = @"Entity does not have a TFN indicator must not be 'yes' (true)",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:TaxFileNumberInexistentI",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408118"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER63", Value = GetValueOrEmpty(interposedEntity.FTER63) });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER64", Value = GetValueOrEmpty(interposedEntity.FTER64) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408119
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408119( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408119
                    Entity Tax file number must be present or Entity does not have a TFN indicator must be 'yes' (true)
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    ^FTER1020 <> NULL AND ^FTER64 = BLANK AND ^FTER63 <> TRUE
        
                    Technical Business Rule Format:
                    ^FTER1020 <> NULL AND ^FTER64 = BLANK AND ^FTER63 <> TRUE
            
                    Data Elements:
            
                    ^FTER64 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.TaxFileNumber.Identifier
            
                    ^FTER1020 = FTER:RP:FamilyTrustRevocation:InterposedEntity
            
                    ^FTER63 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.TaxFileNumberInexistent.Indicator
                    */
                    if (!isStandalone)
                    {
                        assertion = (report.RP_FamilyTrustRevocation_InterposedEntityCollectionExists != false && string.IsNullOrWhiteSpace(interposedEntity.FTER64) == true && interposedEntity.FTER63 != true);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408385", Severity = ProcessMessageSeverity.Error,
                                Description = @"Entity Tax file number must be present or Entity does not have a TFN indicator must be 'yes' (true)",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:TaxFileNumberId",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408119"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER64", Value = GetValueOrEmpty(interposedEntity.FTER64) });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER63", Value = GetValueOrEmpty(interposedEntity.FTER63) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408121
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408121( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408121
                    ACN and ARBN must not both be present
    
                    Legacy Rule Format:
                    ^FTER86 <> BLANK AND ^FTER29 <> BLANK
        
                    Technical Business Rule Format:
                    ^FTER86 <> BLANK AND ^FTER29 <> BLANK
            
                    Data Elements:
            
                    ^FTER86 = FTER:RP:Trustee:Identifiers.AustralianCompanyNumber.Identifier
            
                    ^FTER29 = FTER:RP:Trustee:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER86) != true && string.IsNullOrWhiteSpace(trustee.FTER29) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408122", Severity = ProcessMessageSeverity.Error,
                            Description = @"ACN and ARBN must not both be present",
                            LongDescription = @"Australian Company Number and Australian Registered Body Number for the company which has an interposed entity election in force must not both be supplied",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408121"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER86", Value = GetValueOrEmpty(trustee.FTER86) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER29", Value = GetValueOrEmpty(trustee.FTER29) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408122
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408122( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408122
                    ACN and ARBN must not both be present
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    ^FTER87 <> BLANK AND ^FTER67 <> BLANK
        
                    Technical Business Rule Format:
                    ^FTER87 <> BLANK AND ^FTER67 <> BLANK
            
                    Data Elements:
            
                    ^FTER87 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.AustralianCompanyNumber.Identifier
            
                    ^FTER67 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER87) != true && string.IsNullOrWhiteSpace(interposedEntity.FTER67) != true);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408122", Severity = ProcessMessageSeverity.Error,
                                Description = @"ACN and ARBN must not both be present",
                                LongDescription = @"Australian Company Number and Australian Registered Body Number for the company which has an interposed entity election in force must not both be supplied",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408122"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER87", Value = GetValueOrEmpty(interposedEntity.FTER87) });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER67", Value = GetValueOrEmpty(interposedEntity.FTER67) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408130
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408130( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408130
                    Trustee declaration must contain either an individual name or a non-individual name
    
                    Legacy Rule Format:
                    ^FTER1022 <> NULL AND ^FTER1023 = NULL AND ^FTER79 = BLANK
        
                    Technical Business Rule Format:
                    ^FTER1022 <> NULL AND ^FTER1023 = NULL AND ^FTER79 = BLANK
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
            
                    ^FTER1022 = FTER:RP:DeclarationOfTrustee
            
                    ^FTER1023 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails
                    */
                    assertion = (report.RP_DeclarationOfTrusteeCollectionExists != false && declarationOfTrustee.RP_DeclarationOfTrustee_IndividualNameDetailsCollectionExists == false && string.IsNullOrWhiteSpace(declarationOfTrustee.FTER79) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408130", Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee declaration must contain either an individual name or a non-individual name",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408130"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408135
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408135( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408135
                    Individual name and non-individual name of a trustee declaration must not both be present
    
                    Legacy Rule Format:
                    ^FTER1022 <> NULL AND ^FTER1023 <> NULL AND ^FTER79 <> NULL
        
                    Technical Business Rule Format:
                    ^FTER1022 <> NULL AND ^FTER1023 <> NULL AND ^FTER79 <> NULL
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
            
                    ^FTER1022 = FTER:RP:DeclarationOfTrustee
            
                    ^FTER1023 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails
                    */
                    assertion = (report.RP_DeclarationOfTrusteeCollectionExists != false && declarationOfTrustee.RP_DeclarationOfTrustee_IndividualNameDetailsCollectionExists != false && declarationOfTrustee.FTER79 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408135", Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name and non-individual name of a trustee declaration must not both be present",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408135"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408145
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408145( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408145
            Current trust details must be present
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^FTER1004 = NULL

            Technical Business Rule Format:
            ^FTER1004 = NULL
    
            Data Elements:
    
            ^FTER1004 = FTER:RP:CurrentTrustDetails
            */
            if (isStandalone)
            {
                assertion = (report.RP_CurrentTrustDetailsCollectionExists == false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408373", Severity = ProcessMessageSeverity.Error,
                        Description = @"Current trust details must be present",
                        Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408145"} },
                    };
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408155
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408155( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408155
            Number of Trustee Declarations supplied must not be greater than the number of trustees.
    
            Legacy Rule Format:
            ^FTER95 = 'E' AND Count(^FTER1022) > Count(^FTER1010)

            Technical Business Rule Format:
            ^FTER95 = 'E' AND Count(^FTER1022) > Count(^FTER1010)
    
            Data Elements:
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
    
            ^FTER1010 = FTER:RP:Trustee
    
            ^FTER1022 = FTER:RP:DeclarationOfTrustee
            */
            assertion = (report.FTER95 == @"E" && Count(report.RP_DeclarationOfTrusteeCollectionCount) > Count(report.RP_TrusteeCollectionCount));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408365", Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of Trustee Declarations supplied must not be greater than the number of trustees.",
                    Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(report.RP_DeclarationOfTrusteeCollection == null ? 0 : (report.RP_DeclarationOfTrusteeCollection.Count() == 0 ? 0 : report.RP_DeclarationOfTrusteeCollection.Last().OccurrenceIndex)) + "/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER95", Value = report.FTER95 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408201( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408201
            A schedule must not have its own intermediary as the intermediary on the main form will submit and declare any schedules attached to the main form
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^FTER1000 <> NULL

            Technical Business Rule Format:
            ^FTER1000 <> NULL
    
            Data Elements:
    
            ^FTER1000 = FTER:INT
            */
            if (!isStandalone)
            {
                assertion = (report.IntCollectionExists != false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408201", Severity = ProcessMessageSeverity.Error,
                        Description = @"Schedule must not have an intermediary",
                        Location = "/tns:FTER/tns:INT/tns:AustralianBusinessNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408201"} },
                    };
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408203
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408203( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408203
            Signature date must not be later than today
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^FTER162 > Today()

            Technical Business Rule Format:
            ^FTER162 > Today()
    
            Data Elements:
    
            ^FTER162 = FTER:INT:Declaration:Declaration.Signature.Date
            */
            if (isStandalone)
            {
                assertion = (report.FTER162.GetValueOrDefault() > DateTime.Now.Date);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                        Description = @"Signature date must not be later than today",
                        Location = "/tns:FTER/tns:INT/tns:Declaration/tns:SignatureD",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408203"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER162", Value = GetValueOrEmptyDate(report.FTER162) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408204
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408204( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408204
            Trust tax file number is invalid
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            (^FTER3 <> BLANK) AND (FailsTFNAlgorithm(^FTER3))

            Technical Business Rule Format:
            (^FTER3 <> BLANK) AND (FailsTFNAlgorithm(^FTER3))
    
            Data Elements:
    
            ^FTER3 = FTER:RP:Identifiers.TaxFileNumber.Identifier
            */
            if (isStandalone)
            {
                assertion = (string.IsNullOrWhiteSpace(report.FTER3) != true && FailsTFNAlgorithm(report.FTER3));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408204", Severity = ProcessMessageSeverity.Error,
                        Description = @"Trust tax file number is invalid",
                        Location = "/tns:FTER/tns:RP/tns:TaxFileNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408204"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER3", Value = report.FTER3 });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408207
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408207( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408207
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FTER4, 'T/A ')) AND (EndsWithSet(^FTER4, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FTER4, 'T/A ')) AND (EndsWithSet(^FTER4, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FTER4 = FTER:RP:CurrentTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FTER4, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.FTER4, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408207", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER4", Value = report.FTER4 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408208
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408208( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408208
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FTER4 <> BLANK) AND NotContainsSet(^FTER4, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            (^FTER4 <> BLANK) AND NotContainsSet(^FTER4, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^FTER4 = FTER:RP:CurrentTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER4) != true && !(IsMatch(report.FTER4, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER4", Value = report.FTER4 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408209
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408209( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408209
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FTER4, ' - ')

            Technical Business Rule Format:
            Contains(^FTER4, ' - ')
    
            Data Elements:
    
            ^FTER4 = FTER:RP:CurrentTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FTER4 == null ? false : report.FTER4.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER4", Value = report.FTER4 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408210
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408210( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408210
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FTER4, 'P/L')

            Technical Business Rule Format:
            Contains(^FTER4, 'P/L')
    
            Data Elements:
    
            ^FTER4 = FTER:RP:CurrentTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FTER4 == null ? false : report.FTER4.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408210"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER4", Value = report.FTER4 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408211
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408211( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408211
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FTER4, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FTER4, '"--","''","  "'))
    
            Data Elements:
    
            ^FTER4 = FTER:RP:CurrentTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FTER4, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER4", Value = report.FTER4 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408212
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408212( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408212
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^FTER4, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^FTER4, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^FTER4 = FTER:RP:CurrentTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FTER4, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408212"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER4", Value = report.FTER4 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408213
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408213( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408213
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FTER5 <> BLANK) AND (StartsWithSet(^FTER5,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER5, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FTER5 <> BLANK) AND (StartsWithSet(^FTER5,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER5, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FTER5 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER5) != true && IsMatch(report.FTER5, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FTER5, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408213", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408213"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER5", Value = report.FTER5 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408214
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408214( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408214
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            ((Contains(^FTER5, 'C/-')) AND (^FTER110 = BLANK))

            Technical Business Rule Format:
            ((Contains(^FTER5, 'C/-')) AND (^FTER110 = BLANK))
    
            Data Elements:
    
            ^FTER5 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
    
            ^FTER110 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.FTER5 == null ? false : report.FTER5.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.FTER110) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408214"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER5", Value = report.FTER5 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER110", Value = GetValueOrEmpty(report.FTER110) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408215
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408215( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408215
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            (FoundSet(^FTER5, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^FTER5, '"AS ABOVE"'))
    
            Data Elements:
    
            ^FTER5 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FTER5, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408215"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER5", Value = report.FTER5 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408216
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408216( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408216
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER5, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER5, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER5 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.FTER5 == null ? false : report.FTER5.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408216"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER5", Value = report.FTER5 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408218
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408218( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408218
            Address line 2 must not contain 'care of' reference
    
            Legacy Rule Format:
            (^FTER110 <> BLANK) AND (StartsWithSet(^FTER110,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER110, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FTER110 <> BLANK) AND (StartsWithSet(^FTER110,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER110, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FTER110 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER110) != true && (IsMatch(report.FTER110, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FTER110, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408218", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 must not contain 'care of' reference",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408218"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER110", Value = GetValueOrEmpty(report.FTER110) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408219
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408219( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408219
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER110, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER110, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER110 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.FTER110 == null ? false : report.FTER110.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408219"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER110", Value = GetValueOrEmpty(report.FTER110) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408220
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408220( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408220
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            (^FTER113 = 'au') AND (FoundSet(^FTER6, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^FTER113 = 'au') AND (FoundSet(^FTER6, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^FTER6 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^FTER113 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FTER113 == @"au" && IsMatch(report.FTER6, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408220"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER113", Value = report.FTER113 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER6", Value = report.FTER6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408221
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408221( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408221
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER6, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER6, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER6 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.FTER6 == null ? false : report.FTER6.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408221"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER6", Value = report.FTER6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408222
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408222( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408222
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            (^FTER113 = 'au') AND (ContainsSet(^FTER6, '0-9'))

            Technical Business Rule Format:
            (^FTER113 = 'au') AND (ContainsSet(^FTER6, '0-9'))
    
            Data Elements:
    
            ^FTER6 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^FTER113 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FTER113 == @"au" && IsMatch(report.FTER6, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408222"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER113", Value = report.FTER113 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER6", Value = report.FTER6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408224
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408224( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408224
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            (^FTER113 = 'au') AND ((^FTER8 = NULL) OR (^FTER7 = NULL))

            Technical Business Rule Format:
            (^FTER113 = 'au') AND ((^FTER8 = NULL) OR (^FTER7 = NULL))
    
            Data Elements:
    
            ^FTER113 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Country.Code
    
            ^FTER7 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FTER8 = FTER:RP:CurrentTrustDetails:PostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.FTER113 == @"au" && (report.FTER8 == null || report.FTER7 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:FTER/tns:RP/tns:CurrentTrustDetails/tns:PostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408224"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER113", Value = report.FTER113 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER8", Value = GetValueOrEmpty(report.FTER8) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER7", Value = GetValueOrEmpty(report.FTER7) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408225
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408225( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408225
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FTER10, 'T/A ')) AND (EndsWithSet(^FTER10, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FTER10, 'T/A ')) AND (EndsWithSet(^FTER10, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FTER10 = FTER:RP:PreviousTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FTER10, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.FTER10, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408225", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408225"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER10", Value = report.FTER10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408226
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408226( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408226
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FTER10 <> BLANK) AND NotContainsSet(^FTER10, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            (^FTER10 <> BLANK) AND NotContainsSet(^FTER10, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^FTER10 = FTER:RP:PreviousTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER10) != true && !(IsMatch(report.FTER10, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408226"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER10", Value = report.FTER10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408227
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408227( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408227
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FTER10, ' - ')

            Technical Business Rule Format:
            Contains(^FTER10, ' - ')
    
            Data Elements:
    
            ^FTER10 = FTER:RP:PreviousTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FTER10 == null ? false : report.FTER10.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408227"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER10", Value = report.FTER10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408228
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408228( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408228
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FTER10, 'P/L')

            Technical Business Rule Format:
            Contains(^FTER10, 'P/L')
    
            Data Elements:
    
            ^FTER10 = FTER:RP:PreviousTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FTER10 == null ? false : report.FTER10.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408228"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER10", Value = report.FTER10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408229
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408229( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408229
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FTER10, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FTER10, '"--","''","  "'))
    
            Data Elements:
    
            ^FTER10 = FTER:RP:PreviousTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FTER10, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408229"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER10", Value = report.FTER10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408230( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408230
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^FTER10, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^FTER10, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^FTER10 = FTER:RP:PreviousTrustDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FTER10, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER10", Value = report.FTER10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408231
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408231( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408231
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FTER11 <> BLANK) AND (StartsWithSet(^FTER11,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER11, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FTER11 <> BLANK) AND (StartsWithSet(^FTER11,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER11, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FTER11 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER11) != true && IsMatch(report.FTER11, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FTER11, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408231", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408231"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER11", Value = report.FTER11 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408232
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408232( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408232
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            ((Contains(^FTER11, 'C/-')) AND (^FTER115 = BLANK))

            Technical Business Rule Format:
            ((Contains(^FTER11, 'C/-')) AND (^FTER115 = BLANK))
    
            Data Elements:
    
            ^FTER11 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
    
            ^FTER115 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.FTER11 == null ? false : report.FTER11.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.FTER115) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408232"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER11", Value = report.FTER11 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER115", Value = GetValueOrEmpty(report.FTER115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408233
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408233( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408233
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            (FoundSet(^FTER11, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^FTER11, '"AS ABOVE"'))
    
            Data Elements:
    
            ^FTER11 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FTER11, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408233"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER11", Value = report.FTER11 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408234
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408234( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408234
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER11, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER11, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER11 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.FTER11 == null ? false : report.FTER11.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408234"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER11", Value = report.FTER11 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408236
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408236( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408236
            Address line 2 must not contain 'care of' reference
    
            Legacy Rule Format:
            (^FTER115 <> BLANK) AND (StartsWithSet(^FTER115,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER115, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FTER115 <> BLANK) AND (StartsWithSet(^FTER115,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER115, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FTER115 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER115) != true && (IsMatch(report.FTER115, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FTER115, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408236", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 must not contain 'care of' reference",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408236"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER115", Value = GetValueOrEmpty(report.FTER115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408237
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408237( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408237
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER115, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER115, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER115 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.FTER115 == null ? false : report.FTER115.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408237"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER115", Value = GetValueOrEmpty(report.FTER115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408238
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408238( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408238
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            (^FTER118 = 'au') AND (FoundSet(^FTER12, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^FTER118 = 'au') AND (FoundSet(^FTER12, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^FTER12 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^FTER118 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FTER118 == @"au" && IsMatch(report.FTER12, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408238"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER118", Value = report.FTER118 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER12", Value = report.FTER12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408239
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408239( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408239
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER12, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER12, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER12 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.FTER12 == null ? false : report.FTER12.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408239"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER12", Value = report.FTER12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408240
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408240( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408240
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            (^FTER118 = 'au') AND (ContainsSet(^FTER12, '0-9'))

            Technical Business Rule Format:
            (^FTER118 = 'au') AND (ContainsSet(^FTER12, '0-9'))
    
            Data Elements:
    
            ^FTER12 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^FTER118 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FTER118 == @"au" && IsMatch(report.FTER12, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408240"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER118", Value = report.FTER118 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER12", Value = report.FTER12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408242
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408242( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408242
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            (^FTER118 = 'au') AND ((^FTER14 = NULL) OR (^FTER13 = NULL))

            Technical Business Rule Format:
            (^FTER118 = 'au') AND ((^FTER14 = NULL) OR (^FTER13 = NULL))
    
            Data Elements:
    
            ^FTER118 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Country.Code
    
            ^FTER13 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FTER14 = FTER:RP:PreviousTrustDetails:PostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.FTER118 == @"au" && (report.FTER14 == null || report.FTER13 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:FTER/tns:RP/tns:PreviousTrustDetails/tns:PostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408242"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER118", Value = report.FTER118 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER14", Value = GetValueOrEmpty(report.FTER14) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER13", Value = GetValueOrEmpty(report.FTER13) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408243
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408243( bool isStandalone, FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails periodDetails, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408243
                    Start date must be before end date
    
                    Legacy Rule Format:
                    ^FTER19 <= ^FTER18
        
                    Technical Business Rule Format:
                    ^FTER19 <= ^FTER18
            
                    Data Elements:
            
                    ^FTER19 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails:Period.End.Date
            
                    ^FTER18 = FTER:RP:CentralManagementOutsideAustralia:PeriodDetails:Period.Start.Date
                    */
                    assertion = (periodDetails.FTER19.GetValueOrDefault() <= periodDetails.FTER18.GetValueOrDefault());
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.408060", Severity = ProcessMessageSeverity.Error,
                            Description = @"Start date must be before end date",
                            LongDescription = @"The start date must be earlier than the end date in a Period.",
                            Location = "/tns:FTER/tns:RP/tns:CentralManagementOutsideAustralia/tns:PeriodDetailsCollection/tns:PeriodDetails" + OccurrenceIndex(periodDetails.OccurrenceIndex) + "/tns:EndD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408243"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER19", Value = GetValueOrEmptyDate(periodDetails.FTER19) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER18", Value = GetValueOrEmptyDate(periodDetails.FTER18) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408244
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408244( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408244
                    Trustee tax file number is invalid
    
                    Legacy Rule Format:
                    (^FTER20 <> BLANK) AND (FailsTFNAlgorithm(^FTER20))
        
                    Technical Business Rule Format:
                    (^FTER20 <> BLANK) AND (FailsTFNAlgorithm(^FTER20))
            
                    Data Elements:
            
                    ^FTER20 = FTER:RP:Trustee:Identifiers.TaxFileNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER20) != true && FailsTFNAlgorithm(trustee.FTER20));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410031", Severity = ProcessMessageSeverity.Error,
                            Description = @"TFN must be a valid TFN",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408244"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER20", Value = GetValueOrEmpty(trustee.FTER20) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408245
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408245( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408245
                    Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER23, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER23, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^FTER23 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trustee.FTER23, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408245"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER23", Value = trustee.FTER23 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408246
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408246( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408246
                    Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER23, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER23, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^FTER23 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trustee.FTER23, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408246"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER23", Value = trustee.FTER23 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408247
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408247( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408247
                    Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^FTER23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^FTER23 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trustee.FTER23, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408247", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408247"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER23", Value = trustee.FTER23 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408248
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408248( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408248
                    Family Name cannot contain "space hyphen space"
    
                    Legacy Rule Format:
                    Contains(^FTER23, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER23, ' - ')
            
                    Data Elements:
            
                    ^FTER23 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (trustee.FTER23 == null ? false : trustee.FTER23.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408248"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER23", Value = trustee.FTER23 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408249
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408249( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408249
                    Family name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER23, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER23, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER23 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trustee.FTER23, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408249"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER23", Value = trustee.FTER23 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408250
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408250( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408250
                    Family name must contain at least one alphabetical character.
    
                    Legacy Rule Format:
                    (^FTER23 <> BLANK) AND (NotContainsSet(^FTER23, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^FTER23 <> BLANK) AND (NotContainsSet(^FTER23, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^FTER23 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER23) != true && !(IsMatch(trustee.FTER23, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name must contain at least one alphabetical character.",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408250"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER23", Value = trustee.FTER23 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408251
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408251( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408251
                    Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^FTER25 <> BLANK) AND (NotContainsSet(^FTER25, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^FTER25 <> BLANK) AND (NotContainsSet(^FTER25, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^FTER25 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER25) != true && !(IsMatch(trustee.FTER25, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408251"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER25", Value = trustee.FTER25 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408252
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408252( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408252
                    First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^FTER25, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER25, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^FTER25 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER25, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408252", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408252"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER25", Value = trustee.FTER25 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408253
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408253( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408253
                    First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER25, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER25, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^FTER25 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER25, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408253"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER25", Value = trustee.FTER25 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408254
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408254( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408254
                    First name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER25, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER25, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^FTER25 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER25, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408254"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER25", Value = trustee.FTER25 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408255( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408255
                    First name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^FTER25, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER25, ' - ')
            
                    Data Elements:
            
                    ^FTER25 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (trustee.FTER25 == null ? false : trustee.FTER25.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408255"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER25", Value = trustee.FTER25 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408256
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408256( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408256
                    First name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER25, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER25, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER25 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER25, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408256"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER25", Value = trustee.FTER25 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408257
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408257( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408257
                    Other Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^FTER26 <> BLANK) AND (NotContainsSet(^FTER26, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^FTER26 <> BLANK) AND (NotContainsSet(^FTER26, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^FTER26 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER26) != true && !(IsMatch(trustee.FTER26, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Given Name must contain at least one alphabetical character",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408257"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER26", Value = GetValueOrEmpty(trustee.FTER26) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408258( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408258
                    Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^FTER26, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER26, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^FTER26 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER26, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408258", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408258"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER26", Value = GetValueOrEmpty(trustee.FTER26) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408259( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408259
                    Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER26, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER26, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^FTER26 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER26, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408259"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER26", Value = GetValueOrEmpty(trustee.FTER26) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408260
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408260( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408260
                    Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER26, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER26, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^FTER26 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER26, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408260"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER26", Value = GetValueOrEmpty(trustee.FTER26) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408261( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408261
                    Other given name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^FTER26, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER26, ' - ')
            
                    Data Elements:
            
                    ^FTER26 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (trustee.FTER26 == null ? false : trustee.FTER26.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408261"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER26", Value = GetValueOrEmpty(trustee.FTER26) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408262
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408262( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408262
                    Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER26, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER26, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER26 = FTER:RP:Trustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trustee.FTER26, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408262"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER26", Value = GetValueOrEmpty(trustee.FTER26) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408263
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408263( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408263
                    Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    (StartsWith(^FTER27, 'T/A ')) AND (EndsWithSet(^FTER27, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^FTER27, 'T/A ')) AND (EndsWithSet(^FTER27, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (IsMatch(trustee.FTER27, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(trustee.FTER27, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408263", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408263"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408264
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408264( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408264
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    (^FTER27 <> BLANK) AND (NotContainsSet(^FTER27, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^FTER27 <> BLANK) AND (NotContainsSet(^FTER27, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER27) != true && !(IsMatch(trustee.FTER27, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408264"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408265
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408265( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408265
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    Contains(^FTER27, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER27, ' - ')
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (trustee.FTER27 == null ? false : trustee.FTER27.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408265"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408266
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408266( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408266
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    Contains(^FTER27, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^FTER27, 'P/L')
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (trustee.FTER27 == null ? false : trustee.FTER27.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408266"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408267
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408267( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408267
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER27, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER27, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(trustee.FTER27, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408267"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408268
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408268( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408268
                    Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    (EndsWithSet(^FTER27, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
        
                    Technical Business Rule Format:
                    (EndsWithSet(^FTER27, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
            
                    Data Elements:
            
                    ^FTER27 = FTER:RP:Trustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(trustee.FTER27, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408268"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER27", Value = GetValueOrEmpty(trustee.FTER27) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408269
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408269( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408269
                    Trustee ACN is invalid
    
                    Legacy Rule Format:
                    (^FTER86 <> BLANK) AND (FailsACNAlgorithm(^FTER86))
        
                    Technical Business Rule Format:
                    (^FTER86 <> BLANK) AND (FailsACNAlgorithm(^FTER86))
            
                    Data Elements:
            
                    ^FTER86 = FTER:RP:Trustee:Identifiers.AustralianCompanyNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER86) != true && FailsACNAlgorithm(trustee.FTER86));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428021", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ACN",
                            LongDescription = @"The Australian Company Number (ACN) has failed the algorithm check",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408269"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER86", Value = GetValueOrEmpty(trustee.FTER86) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408270
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408270( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408270
                    ARBN must be valid
    
                    Legacy Rule Format:
                    (^FTER29 <> BLANK) AND (FailsARBNAlgorithm(^FTER29))
        
                    Technical Business Rule Format:
                    (^FTER29 <> BLANK) AND (FailsARBNAlgorithm(^FTER29))
            
                    Data Elements:
            
                    ^FTER29 = FTER:RP:Trustee:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER29) != true && FailsACNAlgorithm(trustee.FTER29));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410105", Severity = ProcessMessageSeverity.Error,
                            Description = @"ARBN must be a valid number that passes the algorithm",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:AustralianRegisteredBodyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408270"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER29", Value = GetValueOrEmpty(trustee.FTER29) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408271
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408271( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408271
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    (^FTER30 <> BLANK) AND (StartsWithSet(^FTER30,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER30, '" C/O "," C/ "," Care Of "'))
        
                    Technical Business Rule Format:
                    (^FTER30 <> BLANK) AND (StartsWithSet(^FTER30,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER30, '" C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^FTER30 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line1.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER30) != true && IsMatch(trustee.FTER30, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(trustee.FTER30, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408271", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408271"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER30", Value = trustee.FTER30 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408272
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408272( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408272
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    ((Contains(^FTER30, 'C/-')) AND (^FTER31 = BLANK))
        
                    Technical Business Rule Format:
                    ((Contains(^FTER30, 'C/-')) AND (^FTER31 = BLANK))
            
                    Data Elements:
            
                    ^FTER30 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line1.Text
            
                    ^FTER31 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line2.Text
                    */
                    assertion = ((trustee.FTER30 == null ? false : trustee.FTER30.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(trustee.FTER31) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408272"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER30", Value = trustee.FTER30 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER31", Value = GetValueOrEmpty(trustee.FTER31) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408273
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408273( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408273
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    (FoundSet(^FTER30, '"AS ABOVE"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER30, '"AS ABOVE"'))
            
                    Data Elements:
            
                    ^FTER30 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line1.Text
                    */
                    assertion = IsMatch(trustee.FTER30, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not contain 'AS ABOVE'",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408273"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER30", Value = trustee.FTER30 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408274
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408274( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408274
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^FTER30, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^FTER30, 'UNKNOWN')
            
                    Data Elements:
            
                    ^FTER30 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line1.Text
                    */
                    assertion = (trustee.FTER30 == null ? false : trustee.FTER30.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 cannot contain ""UNKNOWN""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408274"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER30", Value = trustee.FTER30 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408276
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408276( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408276
                    Address line 2 must not contain 'care of' reference
    
                    Legacy Rule Format:
                    (^FTER31 <> BLANK) AND (StartsWithSet(^FTER31,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER31, '" C/- "," C/O "," C/ "," Care Of "'))
        
                    Technical Business Rule Format:
                    (^FTER31 <> BLANK) AND (StartsWithSet(^FTER31,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER31, '" C/- "," C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^FTER31 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line2.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trustee.FTER31) != true && (IsMatch(trustee.FTER31, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(trustee.FTER31, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408276", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 must not contain 'care of' reference",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line2T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408276"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER31", Value = GetValueOrEmpty(trustee.FTER31) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408277
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408277( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408277
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^FTER31, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^FTER31, 'UNKNOWN')
            
                    Data Elements:
            
                    ^FTER31 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Line2.Text
                    */
                    assertion = (trustee.FTER31 == null ? false : trustee.FTER31.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 cannot contain ""UNKNOWN""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line2T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408277"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER31", Value = GetValueOrEmpty(trustee.FTER31) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408278
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408278( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408278
                    Suburb/Town/City cannot contain state code
    
                    Legacy Rule Format:
                    (^FTER125 = 'au') AND (FoundSet(^FTER32, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
        
                    Technical Business Rule Format:
                    (^FTER125 = 'au') AND (FoundSet(^FTER32, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^FTER32 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.LocalityName.Text
            
                    ^FTER125 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Country.Code
                    */
                    assertion = (trustee.FTER125 == @"au" && IsMatch(trustee.FTER32, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain state code",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:LocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408278"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER125", Value = trustee.FTER125 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER32", Value = trustee.FTER32 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408279
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408279( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408279
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^FTER32, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^FTER32, 'UNKNOWN')
            
                    Data Elements:
            
                    ^FTER32 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.LocalityName.Text
                    */
                    assertion = (trustee.FTER32 == null ? false : trustee.FTER32.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:LocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408279"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER32", Value = trustee.FTER32 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408280
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408280( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408280
                    Suburb/Town/City cannot contain numeric characters
    
                    Legacy Rule Format:
                    (^FTER125 = 'au') AND (ContainsSet(^FTER32, '0-9'))
        
                    Technical Business Rule Format:
                    (^FTER125 = 'au') AND (ContainsSet(^FTER32, '0-9'))
            
                    Data Elements:
            
                    ^FTER32 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.LocalityName.Text
            
                    ^FTER125 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Country.Code
                    */
                    assertion = (trustee.FTER125 == @"au" && IsMatch(trustee.FTER32, @"[0-9]",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain numeric characters",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:LocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408280"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER125", Value = trustee.FTER125 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER32", Value = trustee.FTER32 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408282
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408282( bool isStandalone, FTER2021.RP_Trustee trustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408282
                    State and post code must be present for an Australian address
    
                    Legacy Rule Format:
                    (^FTER125 = 'au') AND ((^FTER34 = NULL) OR (^FTER33 = NULL))
        
                    Technical Business Rule Format:
                    (^FTER125 = 'au') AND ((^FTER34 = NULL) OR (^FTER33 = NULL))
            
                    Data Elements:
            
                    ^FTER125 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Country.Code
            
                    ^FTER33 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.StateOrTerritory.Code
            
                    ^FTER34 = FTER:RP:Trustee:PostalAddressDetails:AddressDetails.Postcode.Text
                    */
                    assertion = (trustee.FTER125 == @"au" && (trustee.FTER34 == null || trustee.FTER33 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                            Description = @"State and post code must be present for an Australian address",
                            LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                            Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:CountryC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408282"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER125", Value = trustee.FTER125 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER34", Value = GetValueOrEmpty(trustee.FTER34) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER33", Value = GetValueOrEmpty(trustee.FTER33) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408283
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408283( bool isStandalone, FTER2021.RP_Trustee trustee, FTER2021.RP_Trustee.RP_Trustee_NonResidentPeriodDetails nonResidentPeriodDetails, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
                    
                            /*  VR.ATO.FTER.408283
                            Start date must be before end date
            
                            Legacy Rule Format:
                            ^FTER39 <= ^FTER38
                
                            Technical Business Rule Format:
                            ^FTER39 <= ^FTER38
                    
                            Data Elements:
                    
                            ^FTER39 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.End.Date
                    
                            ^FTER38 = FTER:RP:Trustee:NonResidentPeriodDetails:Period.Start.Date
                            */
                            assertion = (nonResidentPeriodDetails.FTER39.GetValueOrDefault() <= nonResidentPeriodDetails.FTER38.GetValueOrDefault());
                            if (assertion)
                            {
                                processMessage = new ProcessMessageDocument()
                                {
                                    Code = "CMN.ATO.GEN.408060", Severity = ProcessMessageSeverity.Error,
                                    Description = @"Start date must be before end date",
                                    LongDescription = @"The start date must be earlier than the end date in a Period.",
                                    Location = "/tns:FTER/tns:RP/tns:TrusteeCollection/tns:Trustee" + OccurrenceIndex(trustee.OccurrenceIndex) + "/tns:NonResidentPeriodDetailsCollection/tns:NonResidentPeriodDetails" + OccurrenceIndex(nonResidentPeriodDetails.OccurrenceIndex) + "/tns:EndD",
                                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408283"} },
                                };
                                processMessage.Parameters.Add(new ProcessMessageParameter
                                    { Name = "FTER39", Value = GetValueOrEmptyDate(nonResidentPeriodDetails.FTER39) });
                    
                                processMessage.Parameters.Add(new ProcessMessageParameter
                                    { Name = "FTER38", Value = GetValueOrEmptyDate(nonResidentPeriodDetails.FTER38) });
                    
                                response.Add(processMessage);
                            }
                      }
                      #endregion 
    
      #region VR.ATO.FTER.408284
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408284( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408284
            Start date must be before end date
    
            Legacy Rule Format:
            ^FTER1015 <> NULL AND ^FTER42 <= ^FTER41

            Technical Business Rule Format:
            ^FTER1015 <> NULL AND ^FTER42 <= ^FTER41
    
            Data Elements:
    
            ^FTER42 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.End.Date
    
            ^FTER41 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^FTER1015 = FTER:RP:ElectionOrVariationIncomeYear:SubstitutedAccountingPeriodDetails
            */
            assertion = (report.RP_ElectionOrVariationIncomeYear_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.FTER42.GetValueOrDefault() <= report.FTER41.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408060", Severity = ProcessMessageSeverity.Error,
                    Description = @"Start date must be before end date",
                    LongDescription = @"The start date must be earlier than the end date in a Period.",
                    Location = "/tns:FTER/tns:RP/tns:ElectionOrVariationIncomeYear/tns:SubstitutedAccountingPeriodDetails/tns:EndD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408284"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER42", Value = GetValueOrEmptyDate(report.FTER42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER41", Value = GetValueOrEmptyDate(report.FTER41) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408285
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408285( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408285
            Specified individual tax file number is invalid
    
            Legacy Rule Format:
            (^FTER44 <> BLANK) AND (FailsTFNAlgorithm(^FTER44))

            Technical Business Rule Format:
            (^FTER44 <> BLANK) AND (FailsTFNAlgorithm(^FTER44))
    
            Data Elements:
    
            ^FTER44 = FTER:RP:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER44) != true && FailsTFNAlgorithm(report.FTER44));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410031", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN must be a valid TFN",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408285"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER44", Value = GetValueOrEmpty(report.FTER44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408286
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408286( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408286
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FTER50, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FTER50, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FTER50 = FTER:RP:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FTER50, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408286"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER50", Value = report.FTER50 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408287
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408287( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408287
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FTER50, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FTER50, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FTER50 = FTER:RP:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FTER50, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408287"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER50", Value = report.FTER50 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408288
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408288( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408288
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FTER50, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FTER50, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FTER50 = FTER:RP:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FTER50, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408288", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408288"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER50", Value = report.FTER50 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408289
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408289( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408289
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^FTER50, ' - ')

            Technical Business Rule Format:
            Contains(^FTER50, ' - ')
    
            Data Elements:
    
            ^FTER50 = FTER:RP:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FTER50 == null ? false : report.FTER50.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408289"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER50", Value = report.FTER50 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408290
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408290( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408290
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FTER50, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FTER50, '"--","''","  "'))
    
            Data Elements:
    
            ^FTER50 = FTER:RP:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FTER50, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408290"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER50", Value = report.FTER50 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408291
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408291( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408291
            Family name must contain at least one alphabetical character.
    
            Legacy Rule Format:
            (^FTER50 <> BLANK) AND (NotContainsSet(^FTER50, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FTER50 <> BLANK) AND (NotContainsSet(^FTER50, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FTER50 = FTER:RP:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER50) != true && !(IsMatch(report.FTER50, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408291"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER50", Value = report.FTER50 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408292
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408292( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408292
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FTER48 <> BLANK) AND (NotContainsSet(^FTER48, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FTER48 <> BLANK) AND (NotContainsSet(^FTER48, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER48) != true && !(IsMatch(report.FTER48, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408292"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408293
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408293( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408293
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FTER48, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FTER48, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FTER48, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408293", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408293"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408294
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408294( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408294
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FTER48, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FTER48, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FTER48, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408294"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408295
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408295( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408295
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FTER48, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FTER48, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FTER48, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408295"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408296
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408296( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408296
            First name must be provided if Other given names are provided
    
            Legacy Rule Format:
            ^FTER1016 <> NULL AND ^FTER49 <> BLANK AND ^FTER48 = BLANK

            Technical Business Rule Format:
            ^FTER1016 <> NULL AND ^FTER49 <> BLANK AND ^FTER48 = BLANK
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
    
            ^FTER1016 = FTER:RP:SpecifiedIndividual
            */
            assertion = (report.RP_SpecifiedIndividualCollectionExists != false && string.IsNullOrWhiteSpace(report.FTER49) != true && string.IsNullOrWhiteSpace(report.FTER48) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408296", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name must be provided if Other given names are provided",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408296"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408297
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408297( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408297
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FTER48, ' - ')

            Technical Business Rule Format:
            Contains(^FTER48, ' - ')
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FTER48 == null ? false : report.FTER48.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408297"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408298
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408298( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408298
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FTER48, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FTER48, '"--","''","  "'))
    
            Data Elements:
    
            ^FTER48 = FTER:RP:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FTER48, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408298"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER48", Value = GetValueOrEmpty(report.FTER48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408299
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408299( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408299
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FTER49 <> BLANK) AND (NotContainsSet(^FTER49, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FTER49 <> BLANK) AND (NotContainsSet(^FTER49, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER49) != true && !(IsMatch(report.FTER49, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408299"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408300
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408300( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408300
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FTER49, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FTER49, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FTER49, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408300", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408300"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408301
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408301( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408301
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FTER49, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FTER49, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FTER49, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408301"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408302
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408302( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408302
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FTER49, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FTER49, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FTER49, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408302"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408303
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408303( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408303
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FTER49, ' - ')

            Technical Business Rule Format:
            Contains(^FTER49, ' - ')
    
            Data Elements:
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FTER49 == null ? false : report.FTER49.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408303"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408304
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408304( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408304
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FTER49, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FTER49, '"--","''","  "'))
    
            Data Elements:
    
            ^FTER49 = FTER:RP:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FTER49, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408304"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER49", Value = GetValueOrEmpty(report.FTER49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408305
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408305( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408305
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^FTER52 <> BLANK) AND (StartsWithSet(^FTER52,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER52, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FTER52 <> BLANK) AND (StartsWithSet(^FTER52,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER52, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FTER52 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER52) != true && IsMatch(report.FTER52, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FTER52, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408305", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408305"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER52", Value = report.FTER52 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408306
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408306( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408306
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            ((Contains(^FTER52, 'C/-')) AND (^FTER134 = BLANK))

            Technical Business Rule Format:
            ((Contains(^FTER52, 'C/-')) AND (^FTER134 = BLANK))
    
            Data Elements:
    
            ^FTER52 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line1.Text
    
            ^FTER134 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.FTER52 == null ? false : report.FTER52.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.FTER134) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408306"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER52", Value = report.FTER52 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER134", Value = GetValueOrEmpty(report.FTER134) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408307
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408307( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408307
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            (FoundSet(^FTER52, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^FTER52, '"AS ABOVE"'))
    
            Data Elements:
    
            ^FTER52 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FTER52, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408307"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER52", Value = report.FTER52 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408308
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408308( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408308
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER52, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER52, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER52 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.FTER52 == null ? false : report.FTER52.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408308"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER52", Value = report.FTER52 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408310
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408310( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408310
            Address line 2 must not contain 'care of' reference
    
            Legacy Rule Format:
            (^FTER134 <> BLANK) AND (StartsWithSet(^FTER134,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER134, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FTER134 <> BLANK) AND (StartsWithSet(^FTER134,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER134, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FTER134 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.FTER134) != true && (IsMatch(report.FTER134, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FTER134, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408310", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 must not contain 'care of' reference",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408310"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER134", Value = GetValueOrEmpty(report.FTER134) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408311
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408311( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408311
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER134, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER134, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER134 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.FTER134 == null ? false : report.FTER134.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408311"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER134", Value = GetValueOrEmpty(report.FTER134) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408312
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408312( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408312
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            (^FTER137 = 'au') AND (FoundSet(^FTER53, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^FTER137 = 'au') AND (FoundSet(^FTER53, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^FTER53 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.LocalityName.Text
    
            ^FTER137 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FTER137 == @"au" && IsMatch(report.FTER53, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408312"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER137", Value = report.FTER137 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER53", Value = report.FTER53 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408313
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408313( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408313
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FTER53, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FTER53, 'UNKNOWN')
    
            Data Elements:
    
            ^FTER53 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.FTER53 == null ? false : report.FTER53.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408313"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER53", Value = report.FTER53 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408314
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408314( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408314
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            (^FTER137 = 'au') AND (ContainsSet(^FTER53, '0-9'))

            Technical Business Rule Format:
            (^FTER137 = 'au') AND (ContainsSet(^FTER53, '0-9'))
    
            Data Elements:
    
            ^FTER53 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.LocalityName.Text
    
            ^FTER137 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.FTER137 == @"au" && IsMatch(report.FTER53, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408314"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER137", Value = report.FTER137 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER53", Value = report.FTER53 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408316
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408316( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408316
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            (^FTER137 = 'au') AND ((^FTER55 = NULL) OR (^FTER54 = NULL))

            Technical Business Rule Format:
            (^FTER137 = 'au') AND ((^FTER55 = NULL) OR (^FTER54 = NULL))
    
            Data Elements:
    
            ^FTER137 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Country.Code
    
            ^FTER54 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^FTER55 = FTER:RP:SpecifiedIndividual:ResidentialAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.FTER137 == @"au" && (report.FTER55 == null || report.FTER54 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:FTER/tns:RP/tns:SpecifiedIndividual/tns:ResidentialAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408316"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER137", Value = report.FTER137 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER55", Value = GetValueOrEmpty(report.FTER55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER54", Value = GetValueOrEmpty(report.FTER54) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408317
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408317( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408317
            Tax file number of the specified individual for the trust election being revoked is invalid
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            (^FTER60 <> BLANK) AND (FailsTFNAlgorithm(^FTER60))

            Technical Business Rule Format:
            (^FTER60 <> BLANK) AND (FailsTFNAlgorithm(^FTER60))
    
            Data Elements:
    
            ^FTER60 = FTER:RP:FamilyTrustRevocation:Identifiers.TaxFileNumber.Identifier
            */
            if (!isStandalone)
            {
                assertion = (string.IsNullOrWhiteSpace(report.FTER60) != true && FailsTFNAlgorithm(report.FTER60));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408317", Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax file number of the specified individual is invalid",
                        Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:TaxFileNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408317"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER60", Value = GetValueOrEmpty(report.FTER60) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408318
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408318( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408318
                    Entity tax file number is invalid
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER64 <> BLANK) AND (FailsTFNAlgorithm(^FTER64))
        
                    Technical Business Rule Format:
                    (^FTER64 <> BLANK) AND (FailsTFNAlgorithm(^FTER64))
            
                    Data Elements:
            
                    ^FTER64 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.TaxFileNumber.Identifier
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER64) != true && FailsTFNAlgorithm(interposedEntity.FTER64));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.410031", Severity = ProcessMessageSeverity.Error,
                                Description = @"TFN must be a valid TFN",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:TaxFileNumberId",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408318"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER64", Value = GetValueOrEmpty(interposedEntity.FTER64) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408319
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408319( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408319
                    Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (StartsWith(^FTER65, 'T/A ')) AND (EndsWithSet(^FTER65, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^FTER65, 'T/A ')) AND (EndsWithSet(^FTER65, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^FTER65 = FTER:RP:FamilyTrustRevocation:InterposedEntity:OrganisationNameDetails.OrganisationalName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (IsMatch(interposedEntity.FTER65, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(interposedEntity.FTER65, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408319", Severity = ProcessMessageSeverity.Error,
                                Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408319"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER65", Value = GetValueOrEmpty(interposedEntity.FTER65) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408320
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408320( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408320
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER65 <> BLANK) AND (NotContainsSet(^FTER65, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^FTER65 <> BLANK) AND (NotContainsSet(^FTER65, '"A-Z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^FTER65 = FTER:RP:FamilyTrustRevocation:InterposedEntity:OrganisationNameDetails.OrganisationalName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER65) != true && !(IsMatch(interposedEntity.FTER65, @"[A-Z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                                Description = @"Organisation name must contain at least one alpha or numeric character",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408320"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER65", Value = GetValueOrEmpty(interposedEntity.FTER65) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408321
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408321( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408321
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    Contains(^FTER65, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER65, ' - ')
            
                    Data Elements:
            
                    ^FTER65 = FTER:RP:FamilyTrustRevocation:InterposedEntity:OrganisationNameDetails.OrganisationalName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER65 == null ? false : interposedEntity.FTER65.ToUpperInvariant().Contains(@" - "));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                                Description = @"Non Individual Name cannot contain space hyphen space combinations",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408321"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER65", Value = GetValueOrEmpty(interposedEntity.FTER65) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408322
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408322( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408322
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    Contains(^FTER65, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^FTER65, 'P/L')
            
                    Data Elements:
            
                    ^FTER65 = FTER:RP:FamilyTrustRevocation:InterposedEntity:OrganisationNameDetails.OrganisationalName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER65 == null ? false : interposedEntity.FTER65.ToUpperInvariant().Contains(@"P/L"));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                                Description = @"Non Individual Name cannot contain ""P/L""",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408322"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER65", Value = GetValueOrEmpty(interposedEntity.FTER65) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408323
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408323( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408323
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (ContainsSet(^FTER65, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER65, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER65 = FTER:RP:FamilyTrustRevocation:InterposedEntity:OrganisationNameDetails.OrganisationalName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = IsMatch(interposedEntity.FTER65, @"--|''|  ",RegexOptions.IgnoreCase);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                                Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408323"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER65", Value = GetValueOrEmpty(interposedEntity.FTER65) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408324
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408324( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408324
                    Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (EndsWithSet(^FTER65, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
        
                    Technical Business Rule Format:
                    (EndsWithSet(^FTER65, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
            
                    Data Elements:
            
                    ^FTER65 = FTER:RP:FamilyTrustRevocation:InterposedEntity:OrganisationNameDetails.OrganisationalName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = IsMatch(interposedEntity.FTER65, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                                Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408324"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER65", Value = GetValueOrEmpty(interposedEntity.FTER65) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408325
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408325( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408325
                    Company ACN is invalid
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER87 <> BLANK) AND (FailsACNAlgorithm(^FTER87))
        
                    Technical Business Rule Format:
                    (^FTER87 <> BLANK) AND (FailsACNAlgorithm(^FTER87))
            
                    Data Elements:
            
                    ^FTER87 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.AustralianCompanyNumber.Identifier
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER87) != true && FailsACNAlgorithm(interposedEntity.FTER87));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408325", Severity = ProcessMessageSeverity.Error,
                                Description = @"Company ACN is invalid",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408325"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER87", Value = GetValueOrEmpty(interposedEntity.FTER87) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408326
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408326( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408326
                    Company ARBN is invalid
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER67 <> BLANK) AND (FailsARBNAlgorithm(^FTER67))
        
                    Technical Business Rule Format:
                    (^FTER67 <> BLANK) AND (FailsARBNAlgorithm(^FTER67))
            
                    Data Elements:
            
                    ^FTER67 = FTER:RP:FamilyTrustRevocation:InterposedEntity:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER67) != true && FailsACNAlgorithm(interposedEntity.FTER67));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408326", Severity = ProcessMessageSeverity.Error,
                                Description = @"Company ARBN is invalid",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:AustralianRegisteredBodyNumberId",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408326"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER67", Value = GetValueOrEmpty(interposedEntity.FTER67) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408327
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408327( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408327
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER68 <> BLANK) AND (StartsWithSet(^FTER68,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FTER68, '" C/O "," C/ "," Care Of "'))
        
                    Technical Business Rule Format:
                    (^FTER68 <> BLANK) AND (StartsWithSet(^FTER68,'"C/O ","C/ ","CARE OF ","CO "')) OR (ContainsSet(^FTER68, '" C/O "," C/ "," CARE OF "'))
            
                    Data Elements:
            
                    ^FTER68 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line1.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER68) != true && IsMatch(interposedEntity.FTER68, @"^(C/O |C/ |CARE OF |CO )",RegexOptions.IgnoreCase) || IsMatch(interposedEntity.FTER68, @" C/O | C/ | CARE OF ",RegexOptions.IgnoreCase));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408327", Severity = ProcessMessageSeverity.Error,
                                Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408327"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER68", Value = interposedEntity.FTER68 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408328
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408328( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408328
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    ((Contains(^FTER68, 'C/-')) AND (^FTER69 = BLANK))
        
                    Technical Business Rule Format:
                    ((Contains(^FTER68, 'C/-')) AND (^FTER69 = BLANK))
            
                    Data Elements:
            
                    ^FTER68 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line1.Text
            
                    ^FTER69 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line2.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = ((interposedEntity.FTER68 == null ? false : interposedEntity.FTER68.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(interposedEntity.FTER69) == true);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                                Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408328"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER68", Value = interposedEntity.FTER68 });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER69", Value = GetValueOrEmpty(interposedEntity.FTER69) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408329
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408329( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408329
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (FoundSet(^FTER68, '"AS ABOVE"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER68, '"AS ABOVE"'))
            
                    Data Elements:
            
                    ^FTER68 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line1.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = IsMatch(interposedEntity.FTER68, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                                Description = @"Address Line 1 must not contain 'AS ABOVE'",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408329"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER68", Value = interposedEntity.FTER68 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408330
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408330( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408330
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    Contains(^FTER68, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^FTER68, 'UNKNOWN')
            
                    Data Elements:
            
                    ^FTER68 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line1.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER68 == null ? false : interposedEntity.FTER68.ToUpperInvariant().Contains(@"UNKNOWN"));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                                Description = @"Address line 1 cannot contain ""UNKNOWN""",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line1T",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408330"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER68", Value = interposedEntity.FTER68 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408332
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408332( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408332
                    Address line 2 must not contain 'care of' reference
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER69 <> BLANK) AND (StartsWithSet(^FTER69,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FTER69, '" C/- "," C/O "," C/ "," Care Of "'))
        
                    Technical Business Rule Format:
                    (^FTER69 <> BLANK) AND (StartsWithSet(^FTER69,'"C/- ","C/O ","C/ ","CARE OF ","CO "') OR ContainsSet(^FTER69, '" C/- "," C/O "," C/ "," CARE OF "'))
            
                    Data Elements:
            
                    ^FTER69 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line2.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (string.IsNullOrWhiteSpace(interposedEntity.FTER69) != true && (IsMatch(interposedEntity.FTER69, @"^(C/- |C/O |C/ |CARE OF |CO )",RegexOptions.IgnoreCase) || IsMatch(interposedEntity.FTER69, @" C/- | C/O | C/ | CARE OF ",RegexOptions.IgnoreCase)));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408332", Severity = ProcessMessageSeverity.Error,
                                Description = @"Address line 2 must not contain 'care of' reference",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line2T",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408332"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER69", Value = GetValueOrEmpty(interposedEntity.FTER69) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408333
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408333( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408333
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    Contains(^FTER69, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^FTER69, 'UNKNOWN')
            
                    Data Elements:
            
                    ^FTER69 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Line2.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER69 == null ? false : interposedEntity.FTER69.ToUpperInvariant().Contains(@"UNKNOWN"));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                                Description = @"Address line 2 cannot contain ""UNKNOWN""",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:Line2T",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408333"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER69", Value = GetValueOrEmpty(interposedEntity.FTER69) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408334
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408334( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408334
                    Suburb/Town/City cannot contain state code
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER145 = 'au') AND (FoundSet(^FTER70, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
        
                    Technical Business Rule Format:
                    (^FTER145 = 'au') AND (FoundSet(^FTER70, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^FTER70 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.LocalityName.Text
            
                    ^FTER145 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Country.Code
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER145 == @"au" && IsMatch(interposedEntity.FTER70, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                                Description = @"Suburb/Town/City cannot contain state code",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:LocalityNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408334"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER145", Value = interposedEntity.FTER145 });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER70", Value = interposedEntity.FTER70 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408335
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408335( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408335
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    Contains(^FTER70, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^FTER70, 'UNKNOWN')
            
                    Data Elements:
            
                    ^FTER70 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.LocalityName.Text
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER70 == null ? false : interposedEntity.FTER70.ToUpperInvariant().Contains(@"UNKNOWN"));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                                Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:LocalityNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408335"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER70", Value = interposedEntity.FTER70 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408336
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408336( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408336
                    Suburb/Town/City cannot contain numeric characters
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER145 = 'au') AND (ContainsSet(^FTER70, '0-9'))
        
                    Technical Business Rule Format:
                    (^FTER145 = 'AU') AND (ContainsSet(^FTER70, '0-9'))
            
                    Data Elements:
            
                    ^FTER70 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.LocalityName.Text
            
                    ^FTER145 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Country.Code
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER145 == @"au" && IsMatch(interposedEntity.FTER70, @"[0-9]",RegexOptions.IgnoreCase));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                                Description = @"Suburb/Town/City cannot contain numeric characters",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:LocalityNameT",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408336"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER145", Value = interposedEntity.FTER145 });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER70", Value = interposedEntity.FTER70 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408338
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408338( bool isStandalone, FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408338
                    State and post code must be present for an Australian address
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    (^FTER145 = 'au') AND ((^FTER72 = NULL) OR (^FTER71 = NULL))
        
                    Technical Business Rule Format:
                    (^FTER145 = 'au') AND ((^FTER72 = NULL) OR (^FTER71 = NULL))
            
                    Data Elements:
            
                    ^FTER145 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Country.Code
            
                    ^FTER72 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.Postcode.Text
            
                    ^FTER71 = FTER:RP:FamilyTrustRevocation:InterposedEntity:PostalAddressDetails:AddressDetails.StateOrTerritory.Code
                    */
                    if (!isStandalone)
                    {
                        assertion = (interposedEntity.FTER145 == @"au" && (interposedEntity.FTER72 == null || interposedEntity.FTER71 == null));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                                Description = @"State and post code must be present for an Australian address",
                                LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                                Location = "/tns:FTER/tns:RP/tns:FamilyTrustRevocation/tns:InterposedEntityCollection/tns:InterposedEntity" + OccurrenceIndex(interposedEntity.OccurrenceIndex) + "/tns:PostalAddressDetails/tns:CountryC",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408338"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER145", Value = interposedEntity.FTER145 });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER72", Value = GetValueOrEmpty(interposedEntity.FTER72) });
                
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER71", Value = GetValueOrEmpty(interposedEntity.FTER71) });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408339
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408339( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408339
                    Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER75, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER75, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^FTER75 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER75, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408339"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER75", Value = declarationOfTrustee.FTER75 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408340
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408340( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408340
                    Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER75, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER75, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^FTER75 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER75, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408340"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER75", Value = declarationOfTrustee.FTER75 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408341
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408341( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408341
                    Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^FTER75, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER75, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^FTER75 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER75, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408341", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408341"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER75", Value = declarationOfTrustee.FTER75 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408342
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408342( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408342
                    Family Name cannot contain "space hyphen space"
    
                    Legacy Rule Format:
                    Contains(^FTER75, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER75, ' - ')
            
                    Data Elements:
            
                    ^FTER75 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (declarationOfTrustee.FTER75 == null ? false : declarationOfTrustee.FTER75.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408342"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER75", Value = declarationOfTrustee.FTER75 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408343
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408343( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408343
                    Family name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER75, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER75, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER75 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER75, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408343"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER75", Value = declarationOfTrustee.FTER75 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408344
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408344( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408344
                    Family name must contain at least one alphabetical character.
    
                    Legacy Rule Format:
                    (^FTER75 <> BLANK) AND (NotContainsSet(^FTER75, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^FTER75 <> BLANK) AND (NotContainsSet(^FTER75, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^FTER75 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrustee.FTER75) != true && !(IsMatch(declarationOfTrustee.FTER75, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name must contain at least one alphabetical character.",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408344"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER75", Value = declarationOfTrustee.FTER75 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408345
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408345( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408345
                    Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^FTER77 <> BLANK) AND (NotContainsSet(^FTER77, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^FTER77 <> BLANK) AND (NotContainsSet(^FTER77, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^FTER77 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrustee.FTER77) != true && !(IsMatch(declarationOfTrustee.FTER77, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408345"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER77", Value = declarationOfTrustee.FTER77 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408346
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408346( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408346
                    First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^FTER77, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER77, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^FTER77 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER77, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408346", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408346"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER77", Value = declarationOfTrustee.FTER77 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408347
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408347( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408347
                    First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER77, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER77, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^FTER77 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER77, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408347"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER77", Value = declarationOfTrustee.FTER77 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408348
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408348( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408348
                    First name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER77, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER77, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^FTER77 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER77, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408348"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER77", Value = declarationOfTrustee.FTER77 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408349
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408349( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408349
                    First name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^FTER77, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER77, ' - ')
            
                    Data Elements:
            
                    ^FTER77 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (declarationOfTrustee.FTER77 == null ? false : declarationOfTrustee.FTER77.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408349"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER77", Value = declarationOfTrustee.FTER77 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408350
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408350( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408350
                    First name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER77, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER77, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER77 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER77, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408350"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER77", Value = declarationOfTrustee.FTER77 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408351
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408351( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408351
                    Other Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^FTER78 <> BLANK) AND (NotContainsSet(^FTER78, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^FTER78 <> BLANK) AND (NotContainsSet(^FTER78, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^FTER78 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrustee.FTER78) != true && !(IsMatch(declarationOfTrustee.FTER78, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Given Name must contain at least one alphabetical character",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408351"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER78", Value = GetValueOrEmpty(declarationOfTrustee.FTER78) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408352
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408352( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408352
                    Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^FTER78, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER78, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^FTER78 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER78, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408352", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408352"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER78", Value = GetValueOrEmpty(declarationOfTrustee.FTER78) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408353
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408353( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408353
                    Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER78, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER78, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^FTER78 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER78, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408353"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER78", Value = GetValueOrEmpty(declarationOfTrustee.FTER78) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408354
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408354( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408354
                    Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^FTER78, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^FTER78, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^FTER78 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER78, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408354"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER78", Value = GetValueOrEmpty(declarationOfTrustee.FTER78) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408355
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408355( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408355
                    Other given name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^FTER78, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER78, ' - ')
            
                    Data Elements:
            
                    ^FTER78 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (declarationOfTrustee.FTER78 == null ? false : declarationOfTrustee.FTER78.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408355"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER78", Value = GetValueOrEmpty(declarationOfTrustee.FTER78) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408356
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408356( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408356
                    Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER78, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER78, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER78 = FTER:RP:DeclarationOfTrustee:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER78, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408356"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER78", Value = GetValueOrEmpty(declarationOfTrustee.FTER78) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408357
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408357( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408357
                    Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    (StartsWith(^FTER79, 'T/A ')) AND (EndsWithSet(^FTER79, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^FTER79, 'T/A ')) AND (EndsWithSet(^FTER79, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (IsMatch(declarationOfTrustee.FTER79, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(declarationOfTrustee.FTER79, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408357", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408357"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408358
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408358( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408358
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    (^FTER79 <> BLANK) AND (NotContainsSet(^FTER79, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^FTER79 <> BLANK) AND (NotContainsSet(^FTER79, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrustee.FTER79) != true && !(IsMatch(declarationOfTrustee.FTER79, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408358"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408359
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408359( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408359
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    Contains(^FTER79, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FTER79, ' - ')
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (declarationOfTrustee.FTER79 == null ? false : declarationOfTrustee.FTER79.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408359"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408360
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408360( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408360
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    Contains(^FTER79, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^FTER79, 'P/L')
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (declarationOfTrustee.FTER79 == null ? false : declarationOfTrustee.FTER79.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408360"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408361
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408361( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408361
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FTER79, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FTER79, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER79, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408361"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408362
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408362( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408362
                    Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    (EndsWithSet(^FTER79, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
        
                    Technical Business Rule Format:
                    (EndsWithSet(^FTER79, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
            
                    Data Elements:
            
                    ^FTER79 = FTER:RP:DeclarationOfTrustee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(declarationOfTrustee.FTER79, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408362"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER79", Value = GetValueOrEmpty(declarationOfTrustee.FTER79) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408363
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408363( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408363
                    Signature date must not be later than today
    
                    Legacy Rule Format:
                    ^FTER168 > Today()
        
                    Technical Business Rule Format:
                    ^FTER168 > Today()
            
                    Data Elements:
            
                    ^FTER168 = FTER:RP:DeclarationOfTrustee:Declaration.Signature.Date
                    */
                    assertion = (declarationOfTrustee.FTER168.GetValueOrDefault() > DateTime.Now.Date);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                            Description = @"Signature date must not be later than today",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:SignatureD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408363"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER168", Value = GetValueOrEmptyDate(declarationOfTrustee.FTER168) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408364
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408364( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408364
                    Trustee declaration statement accepted indicator must be True
    
                    Legacy Rule Format:
                    ^FTER169 = FALSE
        
                    Technical Business Rule Format:
                    ^FTER169 = FALSE
            
                    Data Elements:
            
                    ^FTER169 = FTER:RP:DeclarationOfTrustee:Declaration.StatementAccepted.Indicator
                    */
                    assertion = (declarationOfTrustee.FTER169 == false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FTER.408364", Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee declaration statement accepted indicator must be True",
                            Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:StatementAcceptedI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408364"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FTER169", Value = GetValueOrEmpty(declarationOfTrustee.FTER169) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FTER.408366
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408366( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408366
            Tax Agent declaration statement accepted indicator must be True
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^FTER160 = FALSE

            Technical Business Rule Format:
            ^FTER160 = FALSE
    
            Data Elements:
    
            ^FTER160 = FTER:INT:Declaration:Declaration.StatementAccepted.Indicator
            */
            if (isStandalone)
            {
                assertion = (report.FTER160 == false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408366", Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax Agent declaration statement accepted indicator must be True",
                        Location = "/tns:FTER/tns:INT/tns:Declaration/tns:StatementAcceptedI",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408366"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER160", Value = GetValueOrEmpty(report.FTER160) });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408370
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408370( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408370
            Period start date should be before Period end date
    
            Legacy Rule Format:
            ^FTER173 >= ^FTER174

            Technical Business Rule Format:
            ^FTER173 >= ^FTER174
    
            Data Elements:
    
            ^FTER173 = FTER:RP:Period.Start.Date
    
            ^FTER174 = FTER:RP:Period.End.Date
            */
            assertion = (report.FTER173.GetValueOrDefault() >= report.FTER174.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408370", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date should be before Period end date",
                    Location = "/tns:FTER/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408370"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER173", Value = GetValueOrEmptyDate(report.FTER173) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FTER174", Value = GetValueOrEmptyDate(report.FTER174) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FTER.408376
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408376( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408376
            Family trust revocation or variation must be lodged as a schedule attached to the main form
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^FTER95 <> 'E'

            Technical Business Rule Format:
            ^FTER95 <> 'E'
    
            Data Elements:
    
            ^FTER95 = FTER:RP:Elections.FamilyTrustElectionRevocation.Code
            */
            if (isStandalone)
            {
                assertion = (report.FTER95 != @"E");
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.FTER.408376", Severity = ProcessMessageSeverity.Error,
                        Description = @"Family trust revocation or variation must be lodged as a schedule attached to the main form",
                        Location = "/tns:FTER/tns:RP/tns:ElectionsFamilyTrustElectionRevocationC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408376"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "FTER95", Value = report.FTER95 });
        
                    response.Add(processMessage);
                }
                }
      }
      #endregion 
    
      #region VR.ATO.FTER.408380
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408380( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408380
                    Declaration Statement Type Code must be 'HardCopy'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN DOES NOT EXIST
                    ^FTER1000 <> NULL AND ^FTER99 <> NULL AND (AnyOccurrence(^FTER99, ^FTER99 <> 'HardCopy'))
        
                    Technical Business Rule Format:
                    ^FTER1000 <> NULL AND ^FTER99 <> NULL AND (AnyOccurrence(^FTER99, ^FTER99 <> 'HardCopy'))
            
                    Data Elements:
            
                    ^FTER99 = FTER:RP:DeclarationOfTrustee:Declaration.StatementType.Code
            
                    ^FTER1000 = FTER:INT
                    */
                    if (isStandalone)
                    {
                        assertion = (report.IntCollectionExists != false && declarationOfTrustee.FTER99 != null && (report.RP_DeclarationOfTrusteeCollection == null ? false : report.RP_DeclarationOfTrusteeCollection.Any(FTER99Repeat => FTER99Repeat.FTER99 != @"HardCopy")));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408380", Severity = ProcessMessageSeverity.Error,
                                Description = @"Declaration Statement Type Code must be 'HardCopy'",
                                Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:StatementTypeC",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408380"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER99", Value = declarationOfTrustee.FTER99 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408381
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408381( bool isStandalone, FTER2021.RP_DeclarationOfTrustee declarationOfTrustee, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FTER.408381
                    Declaration Statement Type Code must be 'HardCopy'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    ^FTER99 <> NULL AND (AnyOccurrence(^FTER99, ^FTER99 <> 'HardCopy'))
        
                    Technical Business Rule Format:
                    ^FTER99 <> NULL AND (AnyOccurrence(^FTER99, ^FTER99 <> 'HardCopy'))
            
                    Data Elements:
            
                    ^FTER99 = FTER:RP:DeclarationOfTrustee:Declaration.StatementType.Code
                    */
                    if (!isStandalone)
                    {
                        assertion = (declarationOfTrustee.FTER99 != null && (report.RP_DeclarationOfTrusteeCollection == null ? false : report.RP_DeclarationOfTrusteeCollection.Any(FTER99Repeat => FTER99Repeat.FTER99 != @"HardCopy")));
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.FTER.408381", Severity = ProcessMessageSeverity.Error,
                                Description = @"Declaration Statement Type Code must be 'HardCopy'",
                                Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(declarationOfTrustee.OccurrenceIndex) + "/tns:StatementTypeC",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408381"} },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "FTER99", Value = declarationOfTrustee.FTER99 });
                
                            response.Add(processMessage);
                        }
                        }
              }
              #endregion 
    
      #region VR.ATO.FTER.408395
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFTER408395( bool isStandalone)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FTER.408395
            At least one declaration of trustee must be supplied.
    
            Legacy Rule Format:
            Count(^FTER1022) < 1

            Technical Business Rule Format:
            Count(^FTER1022) < 1
    
            Data Elements:
    
            ^FTER1022 = FTER:RP:DeclarationOfTrustee
            */
            assertion = (Count(report.RP_DeclarationOfTrusteeCollectionCount) < 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FTER.408395", Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one declaration of trustee must be supplied.",
                    Location = "/tns:FTER/tns:RP/tns:DeclarationOfTrusteeCollection/tns:DeclarationOfTrustee" + OccurrenceIndex(report.RP_DeclarationOfTrusteeCollection == null ? 0 : (report.RP_DeclarationOfTrusteeCollection.Count() == 0 ? 0 : report.RP_DeclarationOfTrusteeCollection.Last().OccurrenceIndex)) + "/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FTER.408395"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 

    }
} 
