using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using Ato.EN.IntegrationServices.Document.DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationIEE
{

    public partial class IEE2021Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="IEE2021Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public IEE2021Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());

        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public IEE2021 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(IEE2021 report, bool isStandalone)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------

            #region Manual Rules - Rules marked for manual coding

            #region VR.ATO.IEE.410497

            /*  VR.ATO.IEE.410497
                Trustee/partner Declaration Family name and Given name or non individual name must match at least one Trustee or partner Family name and Given name or non individual name.

            Legacy Rule Format:
                (^IEE116 = 'E') AND ^IEE1030 <> NULL AND ((AnyOccurrence(^IEE28, ^IEE28 <> ^IEE303)) OR (AnyOccurrence(^IEE30, ^IEE30 <> ^IEE304) WHERE ^IEE28 = ^IEE303) OR (^IEE300 <> BLANK AND AnyOccurrence(^IEE32, ^IEE32 <> ^IEE300)))

            Technical Business Rule Format:
                (^IEE116 = 'E') AND ^IEE1030 <> NULL AND ((AnyOccurrence(^IEE28, ^IEE28 <> ^IEE303)) OR (AnyOccurrence(^IEE30, ^IEE30 <> ^IEE304) WHERE ^IEE28 = ^IEE303) OR (^IEE300 <> BLANK AND AnyOccurrence(^IEE32, ^IEE32 <> ^IEE300)))

            Data Elements:

            ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text

            ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text

            ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text

            ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text

            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code

            ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text

            ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text

            ^IEE1030 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails
            */
            VRATOIEE410497(response, report, isStandalone);
            #endregion // VR.ATO.IEE.410497
            #endregion Manual Rules - Rules marked for manual coding

            #region VR.ATO.IEE.410004

            /*  VR.ATO.IEE.410004
            Where the interposed entity is a Self Managed Superannuation Fund or Trust, the question regarding the central management and control of the trust must be answered.
    
            Legacy Rule Format:
            (^IEE116 = 'E') AND (InSet(^IEE16, '"217","212"')) AND (^IEE1011 = NULL)

            Technical Business Rule Format:
            (^IEE116 = 'E') AND (InSet(^IEE16, '"217","212"')) AND (^IEE1011 = NULL)
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE1011 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation
            */
            assertion = (report.IEE116 == @"E" && IsMatch(report.IEE16, @"^(217|212)$") && report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformationCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Central management and control indicator must be present if interposed entity is a SMSF or Trust",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410004" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410004

            #region VR.ATO.IEE.410006

            /*  VR.ATO.IEE.410006
            Interposed entity election  is incomplete.
    
            Legacy Rule Format:
            (^IEE116 = 'E') AND ((^IEE1020 = NULL) OR (^IEE58 = NULL) OR (^IEE59 = NULL) OR (^IEE1025 = NULL))

            Technical Business Rule Format:
            (^IEE116 = 'E') AND ((^IEE1020 = NULL) OR (^IEE58 = NULL) OR (^IEE59 = NULL) OR (^IEE1025 = NULL))
    
            Data Elements:
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE58 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.FamilyTrustElectionStatus.Year
    
            ^IEE59 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.Commencement.Date
    
            ^IEE1020 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            assertion = (report.IEE116 == @"E" && (report.RP_InterposedEntityElectionRevocationDetails_ElectionDetailsCollectionExists == false || report.IEE58 == null || report.IEE59 == null || report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410006",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interposed entity election schedule must be complete",
                    LongDescription = @"The full name and address of the family trust in respect of which the election is made, the income year, day specified, election commencement time, and/or date of birth, name and address of the specified individual must all be completed to make a valid Interposed entity election",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsInterposedEntityElectionOrRevocationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410006" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE58", Value = GetValueOrEmpty(report.IEE58) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE59", Value = GetValueOrEmptyDate(report.IEE59) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410006

            #region VR.ATO.IEE.410034

            /*  VR.ATO.IEE.410034
            Where the interposed entity is a Company, the question regarding non-residency for tax purposes must be answered
    
            Legacy Rule Format:
            (^IEE16 = '051') AND (^IEE116 = 'E') AND (^IEE1013 = NULL)

            Technical Business Rule Format:
            (^IEE16 = '051') AND (^IEE116 = 'E') AND (^IEE1013 = NULL)
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE1013 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation
            */
            assertion = (report.IEE16 == @"051" && report.IEE116 == @"E" && report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformationCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-residency for tax purposes question must be answered if interposed entity is a Company",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:ResidencyTaxPurposesOrganisationStatusC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410034" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410034

            #region VR.ATO.IEE.410035

            /*  VR.ATO.IEE.410035
            If the Interposed entity is a Self Managed Superannuation Fund, Partnership or Trust, the question regarding non-residency for tax purposes must not be answered.
    
            Legacy Rule Format:
            (InSet(^IEE16, '"161","217","212"')) AND (^IEE116 = 'E') AND (^IEE1013 <> NULL)

            Technical Business Rule Format:
            (InSet(^IEE16, '"161","217","212"')) AND (^IEE116 = 'E') AND (^IEE1013 <> NULL)
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE1013 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation
            */
            assertion = (IsMatch(report.IEE16, @"^(161|217|212)$") && report.IEE116 == @"E" && report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformationCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-res for tax purposes question must not be answered if entity is SMSF, Partnership or Trust",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:ResidencyTaxPurposesOrganisationStatusC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410035" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410035

            #region VR.ATO.IEE.410041

            /*  VR.ATO.IEE.410041
            Family trust TFN must not equal TFN
    
            Legacy Rule Format:
            (^IEE50 <> NULL) AND (^IEE50 = ^IEE3)

            Technical Business Rule Format:
            (^IEE50 <> NULL) AND (^IEE50 = ^IEE3)
    
            Data Elements:
    
            ^IEE50 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Identifiers.TaxFileNumber.Identifier
    
            ^IEE3 = IEE:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.IEE50 != null && report.IEE50 == report.IEE3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family trust TFN must not equal TFN of trust, company or partnership",
                    LongDescription = @"An Entity cannot interpose itself; therefore the family trust tax file number quoted in Interposed Entity Election cannot be the same as the entity making the election",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410041" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE50", Value = GetValueOrEmpty(report.IEE50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE3", Value = report.IEE3 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410041

            #region VR.ATO.IEE.410059

            /*  VR.ATO.IEE.410059
            Election commencement time for the interposed entity cannot be earlier than the specified day for the interposed entity election.
    
            Legacy Rule Format:
            (^IEE49 <> NULL AND  ^IEE1020 <> NULL AND (^IEE49 < ^IEE46))

            Technical Business Rule Format:
            (^IEE49 <> NULL AND  ^IEE1020 <> NULL AND (^IEE49 < ^IEE46))
    
            Data Elements:
    
            ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
    
            ^IEE49 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Commencement.Date
    
            ^IEE1020 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails
            */
            assertion = (report.IEE49 != null && report.RP_InterposedEntityElectionRevocationDetails_ElectionDetailsCollectionExists != false && report.IEE49.GetValueOrDefault() < report.IEE46.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Election commencement time must not be before the specified day for the interposed entity election",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionDetails/tns:ElectionsDateSpecifiedD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410059" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE49", Value = GetValueOrEmptyDate(report.IEE49) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410059

            #region VR.ATO.IEE.410066

            /*  VR.ATO.IEE.410066
            For an election either a tax file number or 'No TFN' indicator must be present.
    
            Legacy Rule Format:
            (^IEE116 = 'E') AND (^IEE50 = NULL) AND (^IEE51 = NULL)

            Technical Business Rule Format:
            (^IEE116 = 'E') AND (^IEE50 = NULL) AND (^IEE51 = NULL)
    
            Data Elements:
    
            ^IEE50 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Identifiers.TaxFileNumber.Identifier
    
            ^IEE51 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            */
            assertion = (report.IEE116 == @"E" && report.IEE50 == null && report.IEE51 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410066",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either a TFN or the No TFN indictor must be present for the Family Trust",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410066" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE50", Value = GetValueOrEmpty(report.IEE50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE51", Value = GetValueOrEmpty(report.IEE51) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410066

            #region VR.ATO.IEE.410067

            /*  VR.ATO.IEE.410067
            Tax file number and 'No TFN' indicator are both absent and the surname is present. Either tax file number 'No TFN' indicator is required.
    
            Legacy Rule Format:
            (^IEE1025 <> NULL) AND (^IEE62 = NULL) AND (^IEE63 = NULL)

            Technical Business Rule Format:
            (^IEE1025 <> NULL) AND (^IEE62 = NULL) AND (^IEE63 = NULL)
    
            Data Elements:
    
            ^IEE62 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
    
            ^IEE63 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists != false && report.IEE62 == null && report.IEE63 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410067",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either a TFN or the No TFN indictor must be present if Specified Individual name is present",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410067" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE62", Value = GetValueOrEmpty(report.IEE62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE63", Value = GetValueOrEmpty(report.IEE63) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410067

            #region VR.ATO.IEE.410068

            /*  VR.ATO.IEE.410068
            End Date must be after a Start date
    
            Legacy Rule Format:
            ^IEE201 >= ^IEE202

            Technical Business Rule Format:
            ^IEE201 >= ^IEE202
    
            Data Elements:
    
            ^IEE201 = IEE:RP:Period.Start.Date
    
            ^IEE202 = IEE:RP:Period.End.Date
            */
            assertion = (report.IEE201.GetValueOrDefault() >= report.IEE202.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"End Date must be after a Start date",
                    Location = "/tns:IEE/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410068" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE201", Value = GetValueOrEmptyDate(report.IEE201) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE202", Value = GetValueOrEmptyDate(report.IEE202) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410068

            #region VR.ATO.IEE.410069

            /*  VR.ATO.IEE.410069
            Revocation details should only be completed when the client is revoking an IEE.
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
             (^IEE116 = 'E') AND (^IEE82 <> NULL)

            Technical Business Rule Format:
            (^IEE116 = 'E') AND (^IEE82 <> NULL)
    
            Data Elements:
    
            ^IEE82 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Revocation.Year
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            */
            if (!isStandalone)
            {
                assertion = (report.IEE116 == @"E" && report.IEE82 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410069",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Questions about revocation must not be answered if the form is used for an election",
                        Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410069" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE116", Value = report.IEE116 });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE82", Value = GetValueOrEmpty(report.IEE82) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410069

            #region VR.ATO.IEE.410072

            /*  VR.ATO.IEE.410072
            The date of birth of the specified individual cannot be later than the election commencement time or the substituted accounting period for the family trust.
    
            Legacy Rule Format:
            (^IEE1025 <> NULL) AND (((^IEE59 <> NULL) AND (^IEE64 > ^IEE59)) OR ((^IEE1023 <> NULL) AND (^IEE64 > ^IEE60)))

            Technical Business Rule Format:
            (^IEE1025 <> NULL) AND (((^IEE59 <> NULL) AND (^IEE64 > ^IEE59)) OR ((^IEE1023 <> NULL) AND (^IEE64 > ^IEE60)))
    
            Data Elements:
    
            ^IEE64 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonDemographicDetails.Birth.Date
    
            ^IEE59 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.Commencement.Date
    
            ^IEE60 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^IEE1023 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists != false && (report.IEE59 != null && report.IEE64.GetValueOrDefault() > report.IEE59.GetValueOrDefault() || report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE64.GetValueOrDefault() > report.IEE60.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410072",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The date of birth of the specified individual cannot be later than the election commencement time",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonDemographicDetailsBirthD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410072" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE59", Value = GetValueOrEmptyDate(report.IEE59) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE64", Value = GetValueOrEmptyDate(report.IEE64) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE60", Value = GetValueOrEmptyDate(report.IEE60) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410072

            #region VR.ATO.IEE.410073

            /*  VR.ATO.IEE.410073
            Day specified cannot be earlier than the substituted accounting period for the family trust election.
    
            Legacy Rule Format:
            (^IEE1020 <> NULL) AND (^IEE1023 <> NULL) AND (^IEE46 < ^IEE60)

            Technical Business Rule Format:
            (^IEE1020 <> NULL) AND (^IEE1023 <> NULL) AND (^IEE46 < ^IEE60)
    
            Data Elements:
    
            ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
    
            ^IEE60 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^IEE1020 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails
    
            ^IEE1023 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_ElectionDetailsCollectionExists != false && report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE46.GetValueOrDefault() < report.IEE60.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Day specified must not be before the Substituted accounting period for the family trust election",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionDetails/tns:ElectionsDateSpecifiedD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410073" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE60", Value = GetValueOrEmptyDate(report.IEE60) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410073

            #region VR.ATO.IEE.410074

            /*  VR.ATO.IEE.410074
            The day specified for the purpose of the interposed entity election cannot be earlier than the first day of the income year specified or later than the last day of the income year specified or earlier or later than the dates supplied for the substituted accounting period for the interposed entity.
    
            Legacy Rule Format:
            (^IEE1020 <> NULL) AND (((^IEE1021 = NULL) AND (^IEE46 < AsDate(Concat(^IEE45 - 1, '-07-01')))) OR ((^IEE1021 = NULL) AND (^IEE46 >  AsDate(Concat(^IEE45, '-06-30')))) OR ((^IEE1021 <> NULL) AND (^IEE46 < ^IEE47)) OR ((^IEE1021 <> NULL) AND (^IEE46 > ^IEE48)))

            Technical Business Rule Format:
            (^IEE1020 <> NULL) AND (((^IEE1021 = NULL) AND (^IEE46 < AsDate(Concat(^IEE45 - 1, '-07-01')))) OR ((^IEE1021 = NULL) AND (^IEE46 >  AsDate(Concat(^IEE45, '-06-30')))) OR ((^IEE1021 <> NULL) AND (^IEE46 < ^IEE47)) OR ((^IEE1021 <> NULL) AND (^IEE46 > ^IEE48)))
    
            Data Elements:
    
            ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
    
            ^IEE45 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.InterposedEntityElectionStatus.Year
    
            ^IEE47 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^IEE48 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:SubstitutedAccountingPeriodDetails:Period.End.Date
    
            ^IEE1020 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails
    
            ^IEE1021 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:SubstitutedAccountingPeriodDetails
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_ElectionDetailsCollectionExists != false && (report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionExists == false && report.IEE46.GetValueOrDefault() < AsDate(string.Concat(report.IEE45.GetValueOrDefault() - 1, @"-07-01")) || report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionExists == false && report.IEE46.GetValueOrDefault() > AsDate(string.Concat(report.IEE45.GetValueOrDefault(), @"-06-30")) || report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE46.GetValueOrDefault() < report.IEE47.GetValueOrDefault() || report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE46.GetValueOrDefault() > report.IEE48.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Day specified must be within the income year and the substituted accounting period",
                    LongDescription = @"The day specified for the purpose of the interposed entity election cannot be earlier than the first day of the income year specified or later than the last day of the income year specified or earlier or later than the dates supplied for the substituted accounting period for the interposed entity",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionDetails/tns:ElectionsDateSpecifiedD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410074" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE45", Value = GetValueOrEmpty(report.IEE45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE47", Value = GetValueOrEmptyDate(report.IEE47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE48", Value = GetValueOrEmptyDate(report.IEE48) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410074

            #region VR.ATO.IEE.410076

            /*  VR.ATO.IEE.410076
            An IEE must not contain both election and revocation details
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ((^IEE58 <> NULL) OR (^IEE59 <> NULL) OR (^IEE1023 <> NULL) OR (^IEE1025 <> NULL)) AND (^IEE82 <> NULL)

            Technical Business Rule Format:
            ((^IEE58 <> NULL) OR (^IEE59 <> NULL) OR (^IEE1023 <> NULL) OR (^IEE1025 <> NULL)) AND (^IEE82 <> NULL)
    
            Data Elements:
    
            ^IEE82 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Revocation.Year
    
            ^IEE58 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.FamilyTrustElectionStatus.Year
    
            ^IEE59 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.Commencement.Date
    
            ^IEE1023 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            if (!isStandalone)
            {
                assertion = ((report.IEE58 != null || report.IEE59 != null || report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionExists != false || report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists != false) && report.IEE82 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410076",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"An IEE election must not contain revocation Income Year",
                        LongDescription = @"This schedule contains election details, and also the 'income year from which the revocation is to be effective'. If both an election and a revocation are needed, a second schedule is to be submitted with the main return form",
                        Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410076" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE58", Value = GetValueOrEmpty(report.IEE58) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE59", Value = GetValueOrEmptyDate(report.IEE59) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE82", Value = GetValueOrEmpty(report.IEE82) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410076

            #region VR.ATO.IEE.410079

            /*  VR.ATO.IEE.410079
            Year must be 2005 or later
    
            Legacy Rule Format:
            (^IEE45 <> NULL) AND (^IEE45 < 2005)

            Technical Business Rule Format:
            (^IEE45 <> NULL) AND (^IEE45 < 2005)
    
            Data Elements:
    
            ^IEE45 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.InterposedEntityElectionStatus.Year
            */
            assertion = (report.IEE45 != null && report.IEE45.GetValueOrDefault() < 2005);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410233",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Election status year specified for IEE must be 2005 or later",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionDetails/tns:ElectionsInterposedEntityElectionStatusY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410079" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE45", Value = GetValueOrEmpty(report.IEE45) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410079

            #region VR.ATO.IEE.410080

            /*  VR.ATO.IEE.410080
            Year must be 1995 or later
    
            Legacy Rule Format:
            (^IEE58 <> NULL) AND  (^IEE58 < 1995)

            Technical Business Rule Format:
            (^IEE58 <> NULL) AND  (^IEE58 < 1995)
    
            Data Elements:
    
            ^IEE58 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.FamilyTrustElectionStatus.Year
            */
            assertion = (report.IEE58 != null && report.IEE58.GetValueOrDefault() < 1995);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410236",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family trust election specified income year must be 1995 or later",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:ElectionsElectionStatusY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410080" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE58", Value = GetValueOrEmpty(report.IEE58) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410080

            #region VR.ATO.IEE.410084

            /*  VR.ATO.IEE.410084
            Year must be 2008 or later
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            (^IEE82 <> NULL) AND (^IEE82 < 2008)

            Technical Business Rule Format:
            (^IEE82 <> NULL) AND (^IEE82 < 2008)
    
            Data Elements:
    
            ^IEE82 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Revocation.Year
            */
            if (!isStandalone)
            {
                assertion = (report.IEE82 != null && report.IEE82.GetValueOrDefault() < 2008);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410238",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Year specified for IEE revocation must be 2008 or later",
                        Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsY",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410084" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE82", Value = GetValueOrEmpty(report.IEE82) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410084

            #region VR.ATO.IEE.410088

            /*  VR.ATO.IEE.410088
            Full period indicator is 'True' and specific dates have been entered for central management and control of trust was outside Australia.
    
            Legacy Rule Format:
            (^IEE18 = 'TRUE') AND AnyOccurrence(^IEE1012, ^IEE1012 <> NULL)

            Technical Business Rule Format:
            (^IEE18 = 'TRUE') AND AnyOccurrence(^IEE1012, ^IEE1012 <> NULL)
    
            Data Elements:
    
            ^IEE18 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
            ^IEE1012 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails
            */
            assertion = (report.IEE18 == true && (report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection.Any(IEE1012Repeat => report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionExists != false)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410088",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If Full period indicator is 'True', specific dates must not be present",
                    LongDescription = @"Full period indicator is 'true' and and specific dates have been entered for central management and control of trust was outside Australia. Either delete dates or the full period indicator",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410088" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE18", Value = GetValueOrEmpty(report.IEE18) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410088

            #region VR.ATO.IEE.410089

            /*  VR.ATO.IEE.410089
            The answer to the company being a non resident for tax purposes is 'Yes' and the full period indicator and the dates the company was a non resident have not been supplied.
    
            Legacy Rule Format:
            (InSet(^IEE21, '"NonResidentNoPermEstab","NonResidentPermEstab"')) AND (^IEE22 = NULL) AND Count(^IEE1014) = 0

            Technical Business Rule Format:
            (InSet(^IEE21, '"NonResidentNoPermEstab","NonResidentPermEstab"')) AND (^IEE22 = NULL) AND Count(^IEE1014) = 0
    
            Data Elements:
    
            ^IEE21 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:Residency.TaxPurposesOrganisationStatus.Code
    
            ^IEE22 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:Residency.NonResidentFullPeriod.Indicator
    
            ^IEE1014 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails
            */
            assertion = (IsMatch(report.IEE21, @"^(NonResidentNoPermEstab|NonResidentPermEstab)$") && report.IEE22 == null && Count(report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410089",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If Company is non resident for tax purposes, full period indicator or dates must be present",
                    LongDescription = @"The answer to the company being a non resident for tax purposes is 'Yes' and the full period indicator and the dates the company was a non resident have not been supplied",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:ResidencyTaxPurposesOrganisationStatusC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410089" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE21", Value = report.IEE21 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE22", Value = GetValueOrEmpty(report.IEE22) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410089

            #region VR.ATO.IEE.410090

            /*  VR.ATO.IEE.410090
            The answer to the company a non resident for tax purposes question is 'No', therefore full period indicator or dates company was not resident are not required.
    
            Legacy Rule Format:
            (^IEE21 = 'Resident') AND ((^IEE22 <> NULL) OR AnyOccurrence(^IEE1014, ^IEE1014 <> NULL))

            Technical Business Rule Format:
            (^IEE21 = 'Resident') AND ((^IEE22 <> NULL) OR AnyOccurrence(^IEE1014, ^IEE1014 <> NULL))
    
            Data Elements:
    
            ^IEE21 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:Residency.TaxPurposesOrganisationStatus.Code
    
            ^IEE22 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:Residency.NonResidentFullPeriod.Indicator
    
            ^IEE1014 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails
            */
            assertion = (report.IEE21 == @"Resident" && (report.IEE22 != null || (report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection.Any(IEE1014Repeat => report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionExists != false))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410090",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If company is not a non resident, Full period indicator and dates must not be present",
                    LongDescription = @"The answer to the company a non resident for tax purposes question is 'No', therefore full period indicator or dates company was not resident are not required",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:ResidencyTaxPurposesOrganisationStatusC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410090" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE21", Value = report.IEE21 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE22", Value = GetValueOrEmpty(report.IEE22) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410090

            #region VR.ATO.IEE.410091

            /*  VR.ATO.IEE.410091
            Both full period indicator and specific dates company was a non resident for tax purposes are present when only one is allowed.
    
            Legacy Rule Format:
            (^IEE22 = TRUE) AND AnyOccurrence(^IEE1014, ^IEE1014 <> NULL)

            Technical Business Rule Format:
            (^IEE22 = TRUE) AND AnyOccurrence(^IEE1014, ^IEE1014 <> NULL)
    
            Data Elements:
    
            ^IEE22 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:Residency.NonResidentFullPeriod.Indicator
    
            ^IEE1014 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails
            */
            assertion = (report.IEE22 == true && (report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection.Any(IEE1014Repeat => report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionExists != false)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410091",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Full period indicator and specific dates company was a non resident must not both be present",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:ResidencyNonResidentFullPeriodI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410091" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE22", Value = GetValueOrEmpty(report.IEE22) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410091

            #region VR.ATO.IEE.410104

            /*  VR.ATO.IEE.410104
            The 'specified individual does not have a tax file number indicator' is present and a tax file number has been supplied
    
            Legacy Rule Format:
            (^IEE1025 <> NULL) AND (^IEE63 <> NULL) AND (^IEE62 <> NULL)

            Technical Business Rule Format:
            (^IEE1025 <> NULL) AND (^IEE63 <> NULL) AND (^IEE62 <> NULL)
    
            Data Elements:
    
            ^IEE62 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
    
            ^IEE63 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists != false && report.IEE63 != null && report.IEE62 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410104",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The 'no TFN' indicator must not be completed if the TFN of the specified individual is present",
                    LongDescription = @"The 'specified individual does not have a tax file number' indicator' is present and a tax file number has been supplied",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410104" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE63", Value = GetValueOrEmpty(report.IEE63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE62", Value = GetValueOrEmpty(report.IEE62) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410104

            #region VR.ATO.IEE.410193

            /*  VR.ATO.IEE.410193
            For a revocation, Family Trust tax file number must be present
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            (^IEE116 = 'R') AND (^IEE50 = NULL)

            Technical Business Rule Format:
            (^IEE116 = 'R') AND (^IEE50 = NULL)
    
            Data Elements:
    
            ^IEE50 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Identifiers.TaxFileNumber.Identifier
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            */
            if (!isStandalone)
            {
                assertion = (report.IEE116 == @"R" && report.IEE50 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410193",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"For a revocation, Family Trust tax file number must be present",
                        Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:TaxFileNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410193" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE116", Value = report.IEE116 });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE50", Value = GetValueOrEmpty(report.IEE50) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410193

            #region VR.ATO.IEE.410198

            /*  VR.ATO.IEE.410198
            TFN of specified individual must be entered if TFN non existent indicator is blank or 'FALSE'
    
            Legacy Rule Format:
            (^IEE116 = 'E') AND (^IEE1025 <> NULL) AND (^IEE62 = NULL) AND (^IEE63 <> TRUE)

            Technical Business Rule Format:
            (^IEE116 = 'E') AND (^IEE1025 <> NULL) AND (^IEE62 = NULL) AND (^IEE63 <> TRUE)
    
            Data Elements:
    
            ^IEE62 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
    
            ^IEE63 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumberInexistent.Indicator
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            assertion = (report.IEE116 == @"E" && report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists != false && report.IEE62 == null && report.IEE63 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410086",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN of specified individual must be present if TFN non existent indicator is not 'TRUE'",
                    LongDescription = @"TFN of specified individual must be present if TFN non existent indicator is blank or 'FALSE'",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410198" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE62", Value = GetValueOrEmpty(report.IEE62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE63", Value = GetValueOrEmpty(report.IEE63) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410198

            #region VR.ATO.IEE.410199

            /*  VR.ATO.IEE.410199
            Election Commencement Time must not be present if Income Year Specified in the election is not equal to Income Year of lodgment.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            (^IEE45 <> ^IEE1) AND (^IEE49 <> NULL)

            Technical Business Rule Format:
            (^IEE45 <> ^IEE1) AND (^IEE49 <> NULL)
    
            Data Elements:
    
            ^IEE49 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Commencement.Date
    
            ^IEE45 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.InterposedEntityElectionStatus.Year
    
            ^IEE1 = IEE:RP:Lodgment.PeriodYear.Year
            */
            if (isStandalone)
            {
                assertion = (report.IEE45.GetValueOrDefault() != report.IEE1.GetValueOrDefault() && report.IEE49 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410051",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Election Commencement Time must not be present if Income Year Specified in the election is not equal to Income Year of lodgment",
                        Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsCommencementD",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410199" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE45", Value = GetValueOrEmpty(report.IEE45) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE1", Value = GetValueOrEmpty(report.IEE1) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE49", Value = GetValueOrEmptyDate(report.IEE49) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410199

            #region VR.ATO.IEE.410227

            /*  VR.ATO.IEE.410227
            An ARBN and an ACN must not both be provided
    
            Legacy Rule Format:
            (^IEE100 <> NULL) AND (^IEE101 <> NULL)

            Technical Business Rule Format:
            (^IEE100 <> NULL) AND (^IEE101 <> NULL)
    
            Data Elements:
    
            ^IEE100 = IEE:RP:Identifiers.AustralianCompanyNumber.Identifier
    
            ^IEE101 = IEE:RP:Identifiers.AustralianRegisteredBodyNumber.Identifier
            */
            assertion = (report.IEE100 != null && report.IEE101 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410222",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An ARBN and an ACN must not both be provided",
                    Location = "/tns:IEE/tns:RP/tns:AustralianCompanyNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410227" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE100", Value = GetValueOrEmpty(report.IEE100) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE101", Value = GetValueOrEmpty(report.IEE101) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410227

            #region VR.ATO.IEE.410231

            /*  VR.ATO.IEE.410231
            Income year must not be less than 2005
    
            Legacy Rule Format:
            ^IEE1 < 2005

            Technical Business Rule Format:
            ^IEE1 < 2005
    
            Data Elements:
    
            ^IEE1 = IEE:RP:Lodgment.PeriodYear.Year
            */
            assertion = (report.IEE1.GetValueOrDefault() < 2005);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410231",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year must not be less than 2005",
                    Location = "/tns:IEE/tns:RP/tns:LodgmentPeriodYearY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410231" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE1", Value = GetValueOrEmpty(report.IEE1) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410231

            #region VR.ATO.IEE.410259

            /*  VR.ATO.IEE.410259
            If Entity type is Trust, Fund or Partnership, Public officer declaration must not be present.
    
            Legacy Rule Format:
            (InSet(^IEE16, '"217","212","161"') AND ^IEE1028 <> NULL)

            Technical Business Rule Format:
            (InSet(^IEE16, '"217","212","161"') AND ^IEE1028 <> NULL)
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE1028 = IEE:RP:DeclarationOfPublicOfficer
            */
            assertion = (IsMatch(report.IEE16, @"^(217|212|161)$") && report.RP_DeclarationOfPublicOfficerCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410498",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If Entity type is Trust, Fund or Partnership, Public officer declaration must not be present.",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:StatementTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410259" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410259

            #region VR.ATO.IEE.410260

            /*  VR.ATO.IEE.410260
            If Entity type is Company, trustee or partner declaration must not be present.
    
            Legacy Rule Format:
            ^IEE16 = "051" AND AnyOccurrence(^IEE1019, ^IEE1019 <> NULL)

            Technical Business Rule Format:
            ^IEE16 = '051' AND AnyOccurrence(^IEE1019, ^IEE1019 <> NULL)
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
            */
            assertion = (report.IEE16 == @"051" && (report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410260",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If Entity type is Company, trustee or partner declaration must not be present",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner/tns:StatementTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410260" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410260

            #region VR.ATO.IEE.410264

            /*  VR.ATO.IEE.410264
            Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^IEE192 = FALSE

            Technical Business Rule Format:
            ^IEE192 = FALSE
    
            Data Elements:
    
            ^IEE192 = IEE:RP:DeclarationOfPublicOfficer:Declaration.StatementAccepted.Indicator
            */
            assertion = (report.IEE192 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410264",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410264" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE192", Value = GetValueOrEmpty(report.IEE192) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410264

            #region VR.ATO.IEE.410265

            /*  VR.ATO.IEE.410265
            Declaration Statement Type Code must be 'TrueAndCorrect'.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1001 = NULL AND ^IEE1028 <> NULL AND ^IEE147 <> 'TrueAndCorrect'

            Technical Business Rule Format:
            ^IEE1001 = NULL AND ^IEE1028 <> NULL AND ^IEE147 <> 'TrueAndCorrect'
    
            Data Elements:
    
            ^IEE147 = IEE:RP:DeclarationOfPublicOfficer:Declaration.StatementType.Code
    
            ^IEE1001 = IEE:INT
    
            ^IEE1028 = IEE:RP:DeclarationOfPublicOfficer
            */
            if (isStandalone)
            {
                assertion = (report.IntCollectionExists == false && report.RP_DeclarationOfPublicOfficerCollectionExists != false && report.IEE147 != @"TrueAndCorrect");
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410265",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Declaration Statement Type Code must be 'TrueAndCorrect'",
                        Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:StatementTypeC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410265" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE147", Value = report.IEE147 });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410265

            #region VR.ATO.IEE.410267

            /*  VR.ATO.IEE.410267
            Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1001 <> NULL AND ^IEE179 = FALSE

            Technical Business Rule Format:
            ^IEE1001 <> NULL AND ^IEE179 = FALSE
    
            Data Elements:
    
            ^IEE179 = IEE:INT:Declaration:Declaration.StatementAccepted.Indicator
    
            ^IEE1001 = IEE:INT
            */
            if (isStandalone)
            {
                assertion = (report.IntCollectionExists != false && report.IEE179 == false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410264",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Declaration Statement Accepted Indicator must be 'TRUE'",
                        Location = "/tns:IEE/tns:INT/tns:Declaration/tns:StatementAcceptedI",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410267" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE179", Value = GetValueOrEmpty(report.IEE179) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410267

            #region VR.ATO.IEE.410300

            /*  VR.ATO.IEE.410300
            A schedule must not have its own intermediary as the intermediary on the main form will submit and declare any schedules attached to the main form.
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^IEE1001 <> NULL

            Technical Business Rule Format:
            ^IEE1001 <> NULL
    
            Data Elements:
    
            ^IEE1001 = IEE:INT
            */
            if (!isStandalone)
            {
                assertion = (report.IntCollectionExists != false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410300",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Schedule must not have an intermediary",
                        Location = "/tns:IEE/tns:INT/tns:AustralianBusinessNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410300" } },
                    };
                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410300

            #region VR.ATO.IEE.410302

            /*  VR.ATO.IEE.410302
            Current address must be provided.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1007 = NULL

            Technical Business Rule Format:
            ^IEE1007 = NULL
    
            Data Elements:
    
            ^IEE1007 = IEE:RP:CurrentPostalAddress
            */
            if (isStandalone)
            {
                assertion = (report.RP_CurrentPostalAddressCollectionExists == false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410302",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Current address must be provided",
                        Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410302" } },
                    };
                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410302

            #region VR.ATO.IEE.410303

            /*  VR.ATO.IEE.410303
            Postcode and State must be present for an Australian address
    
            Legacy Rule Format:
            (^IEE111 = 'au') AND ((^IEE7 = NULL) OR (^IEE8 = NULL))

            Technical Business Rule Format:
            (^IEE111 = 'au') AND ((^IEE7 = NULL) OR (^IEE8 = NULL))
    
            Data Elements:
    
            ^IEE111 = IEE:RP:CurrentPostalAddress:AddressDetails.Country.Code
    
            ^IEE7 = IEE:RP:CurrentPostalAddress:AddressDetails.StateOrTerritory.Code
    
            ^IEE8 = IEE:RP:CurrentPostalAddress:AddressDetails.Postcode.Text
            */
            assertion = (report.IEE111 == @"au" && (report.IEE7 == null || report.IEE8 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410303",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode and State must be present for an Australian address",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410303" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE111", Value = report.IEE111 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE7", Value = GetValueOrEmpty(report.IEE7) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE8", Value = GetValueOrEmpty(report.IEE8) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410303

            #region VR.ATO.IEE.410304

            /*  VR.ATO.IEE.410304
            Postcode and State must be present for an Australian address
    
            Legacy Rule Format:
            (^IEE112 = 'au') AND ((^IEE13 = NULL) OR (^IEE14 = NULL))

            Technical Business Rule Format:
            (^IEE112 = 'au') AND ((^IEE13 = NULL) OR (^IEE14 = NULL))
    
            Data Elements:
    
            ^IEE112 = IEE:RP:PreviousPostalAddress:AddressDetails.Country.Code
    
            ^IEE13 = IEE:RP:PreviousPostalAddress:AddressDetails.StateOrTerritory.Code
    
            ^IEE14 = IEE:RP:PreviousPostalAddress:AddressDetails.Postcode.Text
            */
            assertion = (report.IEE112 == @"au" && (report.IEE13 == null || report.IEE14 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410304",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode and State must be present for an Australian address",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410304" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE112", Value = report.IEE112 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE13", Value = GetValueOrEmpty(report.IEE13) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE14", Value = GetValueOrEmpty(report.IEE14) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410304

            #region VR.ATO.IEE.410306

            /*  VR.ATO.IEE.410306
            Postcode and State must be present for an Australian address
    
            Legacy Rule Format:
            (^IEE155 = 'au') AND ((^IEE55 = NULL) OR (^IEE56 = NULL))

            Technical Business Rule Format:
            (^IEE155 = 'au') AND ((^IEE55 = NULL) OR (^IEE56 = NULL))
    
            Data Elements:
    
            ^IEE55 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.StateOrTerritory.Code
    
            ^IEE56 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Postcode.Text
    
            ^IEE155 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE155 == @"au" && (report.IEE55 == null || report.IEE56 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410306",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode and State must be present for an Australian address",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsStateOrTerritoryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410306" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE155", Value = report.IEE155 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE55", Value = GetValueOrEmpty(report.IEE55) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE56", Value = GetValueOrEmpty(report.IEE56) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410306

            #region VR.ATO.IEE.410307

            /*  VR.ATO.IEE.410307
            Postcode and State must be present for an Australian address
    
            Legacy Rule Format:
            (^IEE114 = 'au') AND ((^IEE72 = NULL) OR (^IEE73 = NULL))

            Technical Business Rule Format:
            (^IEE114 = 'au') AND ((^IEE72 = NULL) OR (^IEE73 = NULL))
    
            Data Elements:
    
            ^IEE114 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Country.Code
    
            ^IEE72 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^IEE73 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.IEE114 == @"au" && (report.IEE72 == null || report.IEE73 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410307",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode and State must be present for an Australian address",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410307" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE114", Value = report.IEE114 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE72", Value = GetValueOrEmpty(report.IEE72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE73", Value = GetValueOrEmpty(report.IEE73) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410307

            #region VR.ATO.IEE.410311

            /*  VR.ATO.IEE.410311
            If Entity type is Company, Public Officer declaration must be present.
    
            Legacy Rule Format:
            ^IEE16 = '051' AND ^IEE1028 = NULL

            Technical Business Rule Format:
            ^IEE16 = '051' AND ^IEE1028 = NULL
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE1028 = IEE:RP:DeclarationOfPublicOfficer
            */
            assertion = (report.IEE16 == @"051" && report.RP_DeclarationOfPublicOfficerCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410311",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If Entity type is Company, Public Officer declaration must be present",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:StatementTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410311" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410311

            #region VR.ATO.IEE.410312

            /*  VR.ATO.IEE.410312
            Organisation Name must be present.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE4 = NULL

            Technical Business Rule Format:
            ^IEE4 = NULL
    
            Data Elements:
    
            ^IEE4 = IEE:RP:CurrentName:OrganisationNameDetails.OrganisationalName.Text
            */
            if (isStandalone)
            {
                assertion = (report.IEE4 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410312",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Organisation Name must be present.",
                        Location = "/tns:IEE/tns:RP/tns:CurrentNameOrganisationNameDetailsOrganisationalNameT",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410312" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE4", Value = GetValueOrEmpty(report.IEE4) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410312

            #region VR.ATO.IEE.410316

            /*  VR.ATO.IEE.410316
            Family name must not contain a suffix
    
            Legacy Rule Format:
            (FoundSet(^IEE66, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IEE66, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IEE66 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE66, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410313",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410316" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE66", Value = report.IEE66 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410316

            #region VR.ATO.IEE.410317

            /*  VR.ATO.IEE.410317
            Given Name must not contain a suffix
    
            Legacy Rule Format:
            (FoundSet(^IEE68, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IEE68, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE68, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410317" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410317

            #region VR.ATO.IEE.410318

            /*  VR.ATO.IEE.410318
            Other Given Name must not contain a suffix
    
            Legacy Rule Format:
            (FoundSet(^IEE69, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IEE69, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE69, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410315",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410318" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410318

            #region VR.ATO.IEE.410319

            /*  VR.ATO.IEE.410319
            Family name must not contain a suffix
    
            Legacy Rule Format:
            (FoundSet(^IEE84, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IEE84, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IEE84 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE84, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410313",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410319" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE84", Value = report.IEE84 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410319

            #region VR.ATO.IEE.410320

            /*  VR.ATO.IEE.410320
            Given Name must not contain a suffix
    
            Legacy Rule Format:
            (FoundSet(^IEE86, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IEE86, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE86, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410320" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410320

            #region VR.ATO.IEE.410321

            /*  VR.ATO.IEE.410321
            Other Given Name must not contain a suffix
    
            Legacy Rule Format:
            (FoundSet(^IEE87, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^IEE87, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE87, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410315",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410321" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410321

            #region VR.ATO.IEE.410323

            /*  VR.ATO.IEE.410323
            Revocation must be submitted only with main return being lodged
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            (^IEE116 = 'R') OR (^IEE82 <> NULL)

            Technical Business Rule Format:
            (^IEE116 = 'R') OR (^IEE82 <> NULL)
    
            Data Elements:
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE82 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Revocation.Year
            */
            if (isStandalone)
            {
                assertion = (report.IEE116 == @"R" || report.IEE82 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410323",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Revocation must be submitted only with main return being lodged",
                        Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsInterposedEntityElectionOrRevocationC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410323" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE116", Value = report.IEE116 });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE82", Value = GetValueOrEmpty(report.IEE82) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410323

            #region VR.ATO.IEE.410326

            /*  VR.ATO.IEE.410326
            Declaration Statement  type code must be 'TrueAndCorrect'
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1001 <> NULL AND ^IEE159 <> "TrueAndCorrect"

            Technical Business Rule Format:
            ^IEE1001 <> NULL AND ^IEE159 <> 'TrueAndCorrect'
    
            Data Elements:
    
            ^IEE159 = IEE:INT:Declaration:Declaration.StatementType.Code
    
            ^IEE1001 = IEE:INT
            */
            if (isStandalone)
            {
                assertion = (report.IntCollectionExists != false && report.IEE159 != @"TrueAndCorrect");
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410326",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Declaration Statement  type code must be 'TrueAndCorrect'",
                        Location = "/tns:IEE/tns:INT/tns:Declaration/tns:StatementTypeC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410326" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE159", Value = report.IEE159 });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410326

            #region VR.ATO.IEE.410327

            /*  VR.ATO.IEE.410327
            Specific dates company was a non resident must be present when  company was a non resident for full period for tax purpose.
    
            Legacy Rule Format:
            (^IEE22 = FALSE) AND Count(^IEE1014) = 0

            Technical Business Rule Format:
            (^IEE22 = FALSE) AND Count(^IEE1014) = 0
    
            Data Elements:
    
            ^IEE22 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:Residency.NonResidentFullPeriod.Indicator
    
            ^IEE1014 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails
            */
            assertion = (report.IEE22 == false && Count(report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410327",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Specific dates company was a non resident must be present when  company was a non resident for full period for tax purpose",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:ResidencyNonResidentFullPeriodI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410327" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE22", Value = GetValueOrEmpty(report.IEE22) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410327

            #region VR.ATO.IEE.410329

            /*  VR.ATO.IEE.410329
            Declaration Statement Type Code must be 'HardCopy'
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            ^IEE1028 <> NULL AND ^IEE147 <> 'HardCopy'

            Technical Business Rule Format:
            ^IEE1028 <> NULL AND ^IEE147 <> 'HardCopy'
    
            Data Elements:
    
            ^IEE147 = IEE:RP:DeclarationOfPublicOfficer:Declaration.StatementType.Code
    
            ^IEE1028 = IEE:RP:DeclarationOfPublicOfficer
            */
            if (!isStandalone)
            {
                assertion = (report.RP_DeclarationOfPublicOfficerCollectionExists != false && report.IEE147 != @"HardCopy");
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410329",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Declaration Statement Type Code must be 'HardCopy'",
                        Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:StatementTypeC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410329" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE147", Value = report.IEE147 });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410329

            #region VR.ATO.IEE.410330

            /*  VR.ATO.IEE.410330
            Declaration Statement Type Code must be 'HardCopy'
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1001 <> NULL AND ^IEE1028 <> NULL AND ^IEE147 <> 'HardCopy'

            Technical Business Rule Format:
            ^IEE1001 <> NULL AND ^IEE1028 <> NULL AND ^IEE147 <> 'HardCopy'
    
            Data Elements:
    
            ^IEE147 = IEE:RP:DeclarationOfPublicOfficer:Declaration.StatementType.Code
    
            ^IEE1001 = IEE:INT
    
            ^IEE1028 = IEE:RP:DeclarationOfPublicOfficer
            */
            if (isStandalone)
            {
                assertion = (report.IntCollectionExists != false && report.RP_DeclarationOfPublicOfficerCollectionExists != false && report.IEE147 != @"HardCopy");
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410330",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Declaration Statement Type Code must be 'HardCopy'",
                        Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:StatementTypeC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410330" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE147", Value = report.IEE147 });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410330

            #region VR.ATO.IEE.410331

            /*  VR.ATO.IEE.410331
            Name must not contain 'space hyphen space'
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            Contains(^IEE186, ' - ')

            Technical Business Rule Format:
            Contains(^IEE186, ' - ')
    
            Data Elements:
    
            ^IEE186 = IEE:INT:ContactName:PersonUnstructuredName.FullName.Text
            */
            if (isStandalone)
            {
                assertion = (report.IEE186 == null ? false : report.IEE186.ToUpperInvariant().Contains(@" - "));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410331",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax agent contact name  cannot contain ""space hyphen space""",
                        Location = "/tns:IEE/tns:INT/tns:ContactNamePersonUnstructuredNameFullNameT",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410331" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE186", Value = GetValueOrEmpty(report.IEE186) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410331

            #region VR.ATO.IEE.410332

            /*  VR.ATO.IEE.410332
            Full period indicator is 'False' and specific dates have not been entered for central management and control of trust was outside Australia.
    
            Legacy Rule Format:
            (^IEE18 = 'FALSE') AND Count(^IEE1012) = 0

            Technical Business Rule Format:
            (^IEE18 = 'FALSE') AND Count(^IEE1012) = 0
    
            Data Elements:
    
            ^IEE18 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
            ^IEE1012 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails
            */
            assertion = (report.IEE18 == false && Count(report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410332",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If Full period indicator is 'False', specific dates must be present",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410332" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE18", Value = GetValueOrEmpty(report.IEE18) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410332

            #region VR.ATO.IEE.410335

            /*  VR.ATO.IEE.410335
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE5, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE5, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE5 = IEE:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.IEE5 == null ? false : report.IEE5.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410335" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE5", Value = report.IEE5 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410335

            #region VR.ATO.IEE.410336

            /*  VR.ATO.IEE.410336
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE11, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE11, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE11 = IEE:RP:PreviousPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.IEE11 == null ? false : report.IEE11.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410336" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE11", Value = report.IEE11 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410336

            #region VR.ATO.IEE.410338

            /*  VR.ATO.IEE.410338
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE53, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE53, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE53 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.IEE53 == null ? false : report.IEE53.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410338" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE53", Value = report.IEE53 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410338

            #region VR.ATO.IEE.410339

            /*  VR.ATO.IEE.410339
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE70, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE70, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE70 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.IEE70 == null ? false : report.IEE70.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410339" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE70", Value = report.IEE70 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410339

            #region VR.ATO.IEE.410340

            /*  VR.ATO.IEE.410340
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^IEE5 <> NULL) AND (StartsWithSet(^IEE5,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^IEE5, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE5 <> NULL) AND (StartsWithSet(^IEE5,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^IEE5, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE5 = IEE:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.IEE5 != null && IsMatch(report.IEE5, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE5, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410340" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE5", Value = report.IEE5 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410340

            #region VR.ATO.IEE.410341

            /*  VR.ATO.IEE.410341
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^IEE11 <> NULL) AND (StartsWithSet(^IEE11,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE11, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE11 <> NULL) AND (StartsWithSet(^IEE11,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE11, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE11 = IEE:RP:PreviousPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.IEE11 != null && (IsMatch(report.IEE11, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE11, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410341" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE11", Value = report.IEE11 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410341

            #region VR.ATO.IEE.410343

            /*  VR.ATO.IEE.410343
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^IEE53 <> NULL) AND (StartsWithSet(^IEE53,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE53, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE53 <> NULL) AND (StartsWithSet(^IEE53,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE53, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE53 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.IEE53 != null && (IsMatch(report.IEE53, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE53, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410343" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE53", Value = report.IEE53 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410343

            #region VR.ATO.IEE.410344

            /*  VR.ATO.IEE.410344
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (^IEE70 <> NULL) AND (StartsWithSet(^IEE70,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE70, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE70 <> NULL) AND (StartsWithSet(^IEE70,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE70, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE70 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.IEE70 != null && (IsMatch(report.IEE70, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE70, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410344" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE70", Value = report.IEE70 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410344

            #region VR.ATO.IEE.410345

            /*  VR.ATO.IEE.410345
            Address Line 2 cannot contain 'UNKNOWN'
    
            Legacy Rule Format:
            Contains(^IEE122, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE122, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE122 = IEE:RP:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (report.IEE122 == null ? false : report.IEE122.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410345" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE122", Value = GetValueOrEmpty(report.IEE122) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410345

            #region VR.ATO.IEE.410346

            /*  VR.ATO.IEE.410346
            Address Line 2 cannot contain 'UNKNOWN'
    
            Legacy Rule Format:
            Contains(^IEE126, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE126, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE126 = IEE:RP:PreviousPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (report.IEE126 == null ? false : report.IEE126.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410346" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE126", Value = GetValueOrEmpty(report.IEE126) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410346

            #region VR.ATO.IEE.410348

            /*  VR.ATO.IEE.410348
            Address Line 2 cannot contain 'UNKNOWN'
    
            Legacy Rule Format:
            Contains(^IEE134, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE134, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE134 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (report.IEE134 == null ? false : report.IEE134.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410348" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE134", Value = GetValueOrEmpty(report.IEE134) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410348

            #region VR.ATO.IEE.410349

            /*  VR.ATO.IEE.410349
            Address Line 2 cannot contain 'UNKNOWN'
    
            Legacy Rule Format:
            Contains(^IEE138, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE138, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE138 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.IEE138 == null ? false : report.IEE138.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410349" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE138", Value = GetValueOrEmpty(report.IEE138) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410349

            #region VR.ATO.IEE.410350

            /*  VR.ATO.IEE.410350
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            (^IEE122 <> BLANK) AND (StartsWithSet(^IEE122,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE122, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE122 <> BLANK) AND (StartsWithSet(^IEE122,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE122, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE122 = IEE:RP:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE122) != true && (IsMatch(report.IEE122, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE122, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410350" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE122", Value = GetValueOrEmpty(report.IEE122) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410350

            #region VR.ATO.IEE.410351

            /*  VR.ATO.IEE.410351
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            (^IEE126 <> BLANK) AND (StartsWithSet(^IEE126,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE126, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE126 <> BLANK) AND (StartsWithSet(^IEE126,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE126, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE126 = IEE:RP:PreviousPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE126) != true && (IsMatch(report.IEE126, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE126, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410351" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE126", Value = GetValueOrEmpty(report.IEE126) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410351

            #region VR.ATO.IEE.410353

            /*  VR.ATO.IEE.410353
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            (^IEE134 <> BLANK) AND (StartsWithSet(^IEE134,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE134, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE134 <> BLANK) AND (StartsWithSet(^IEE134,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE134, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE134 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE134) != true && (IsMatch(report.IEE134, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE134, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410353" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE134", Value = GetValueOrEmpty(report.IEE134) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410353

            #region VR.ATO.IEE.410354

            /*  VR.ATO.IEE.410354
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            (^IEE138 <> BLANK) AND (StartsWithSet(^IEE138,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE138, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^IEE138 <> BLANK) AND (StartsWithSet(^IEE138,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE138, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^IEE138 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE138) != true && (IsMatch(report.IEE138, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.IEE138, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410354" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE138", Value = GetValueOrEmpty(report.IEE138) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410354

            #region VR.ATO.IEE.410355

            /*  VR.ATO.IEE.410355
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE6, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE6, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE6 = IEE:RP:CurrentPostalAddress:AddressDetails.LocalityName.Text
            */
            assertion = (report.IEE6 == null ? false : report.IEE6.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410355" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE6", Value = report.IEE6 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410355

            #region VR.ATO.IEE.410356

            /*  VR.ATO.IEE.410356
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE12, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE12, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE12 = IEE:RP:PreviousPostalAddress:AddressDetails.LocalityName.Text
            */
            assertion = (report.IEE12 == null ? false : report.IEE12.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410356" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE12", Value = report.IEE12 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410356

            #region VR.ATO.IEE.410358

            /*  VR.ATO.IEE.410358
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE54, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE54, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE54 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.LocalityName.Text
            */
            assertion = (report.IEE54 == null ? false : report.IEE54.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410358" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE54", Value = report.IEE54 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410358

            #region VR.ATO.IEE.410359

            /*  VR.ATO.IEE.410359
            Address must not contain "UNKNOWN".
    
            Legacy Rule Format:
            Contains(^IEE71, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^IEE71, 'UNKNOWN')
    
            Data Elements:
    
            ^IEE71 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.IEE71 == null ? false : report.IEE71.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410359" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE71", Value = report.IEE71 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410359

            #region VR.ATO.IEE.410360

            /*  VR.ATO.IEE.410360
            Address - Locality must not contain numerics for an Australian address
    
            Legacy Rule Format:
            (^IEE111 = 'au') AND (ContainsSet(^IEE6, '0-9'))

            Technical Business Rule Format:
            (^IEE111 = 'au') AND (ContainsSet(^IEE6, '0-9'))
    
            Data Elements:
    
            ^IEE6 = IEE:RP:CurrentPostalAddress:AddressDetails.LocalityName.Text
    
            ^IEE111 = IEE:RP:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE111 == @"au" && IsMatch(report.IEE6, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410360" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE111", Value = report.IEE111 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE6", Value = report.IEE6 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410360

            #region VR.ATO.IEE.410361

            /*  VR.ATO.IEE.410361
            Address - Locality must not contain numerics for an Australian address
    
            Legacy Rule Format:
            (^IEE112 = 'au') AND (ContainsSet(^IEE12, '0-9'))

            Technical Business Rule Format:
            (^IEE112 = 'au') AND (ContainsSet(^IEE12, '0-9'))
    
            Data Elements:
    
            ^IEE12 = IEE:RP:PreviousPostalAddress:AddressDetails.LocalityName.Text
    
            ^IEE112 = IEE:RP:PreviousPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE112 == @"au" && IsMatch(report.IEE12, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410361" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE112", Value = report.IEE112 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE12", Value = report.IEE12 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410361

            #region VR.ATO.IEE.410363

            /*  VR.ATO.IEE.410363
            Address - Locality must not contain numerics for an Australian address
    
            Legacy Rule Format:
            (^IEE155 = 'au') AND ContainsSet(^IEE54, '0-9')

            Technical Business Rule Format:
            (^IEE155 = 'au') AND ContainsSet(^IEE54, '0-9')
    
            Data Elements:
    
            ^IEE54 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.LocalityName.Text
    
            ^IEE155 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE155 == @"au" && IsMatch(report.IEE54, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410363" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE155", Value = report.IEE155 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE54", Value = report.IEE54 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410363

            #region VR.ATO.IEE.410364

            /*  VR.ATO.IEE.410364
            Address - Locality must not contain numericsfor an Australian address
    
            Legacy Rule Format:
            (^IEE114 = 'au') AND ContainsSet(^IEE71, '0-9')

            Technical Business Rule Format:
            (^IEE114 = 'au') AND ContainsSet(^IEE71, '0-9')
    
            Data Elements:
    
            ^IEE71 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.LocalityName.Text
    
            ^IEE114 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.IEE114 == @"au" && IsMatch(report.IEE71, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410364" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE114", Value = report.IEE114 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE71", Value = report.IEE71 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410364

            #region VR.ATO.IEE.410366

            /*  VR.ATO.IEE.410366
            Orqanisation Name cannot contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE52, ' - ')

            Technical Business Rule Format:
            Contains(^IEE52, ' - ')
    
            Data Elements:
    
            ^IEE52 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.IEE52 == null ? false : report.IEE52.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410366" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE52", Value = report.IEE52 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410366

            #region VR.ATO.IEE.410368

            /*  VR.ATO.IEE.410368
            Organisation Name must not contain 'P/L'
    
            Legacy Rule Format:
            Contains(^IEE52, 'P/L')

            Technical Business Rule Format:
            Contains(^IEE52, 'P/L')
    
            Data Elements:
    
            ^IEE52 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.IEE52 == null ? false : report.IEE52.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410368" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE52", Value = report.IEE52 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410368

            #region VR.ATO.IEE.410370

            /*  VR.ATO.IEE.410370
            Organisation Name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE52, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE52, '"--","''","  "')
    
            Data Elements:
    
            ^IEE52 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.IEE52, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410370" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE52", Value = report.IEE52 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410370

            #region VR.ATO.IEE.410372

            /*  VR.ATO.IEE.410372
            Non-individualname must not end with a leading space followed by 'T/A', 'T/A P'ship', 'T/A Pship', 'T/A P/Ship' or 'T/A Partnership'.
    
            Legacy Rule Format:
            EndsWithSet(^IEE52, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^IEE52, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^IEE52 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.IEE52, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410372" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE52", Value = report.IEE52 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410372

            #region VR.ATO.IEE.410374

            /*  VR.ATO.IEE.410374
            Family Name must not contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE66, ' - ')

            Technical Business Rule Format:
            Contains(^IEE66, ' - ')
    
            Data Elements:
    
            ^IEE66 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.IEE66 == null ? false : report.IEE66.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410374" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE66", Value = report.IEE66 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410374

            #region VR.ATO.IEE.410375

            /*  VR.ATO.IEE.410375
            Family Name must not contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE84, ' - ')

            Technical Business Rule Format:
            Contains(^IEE84, ' - ')
    
            Data Elements:
    
            ^IEE84 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.IEE84 == null ? false : report.IEE84.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410375" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE84", Value = report.IEE84 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410375

            #region VR.ATO.IEE.410377

            /*  VR.ATO.IEE.410377
            Family name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE66, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE66, '"--","''","  "')
    
            Data Elements:
    
            ^IEE66 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE66, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410377" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE66", Value = report.IEE66 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410377

            #region VR.ATO.IEE.410378

            /*  VR.ATO.IEE.410378
            Family name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE84, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE84, '"--","''","  "')
    
            Data Elements:
    
            ^IEE84 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE84, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410378" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE84", Value = report.IEE84 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410378

            #region VR.ATO.IEE.410380

            /*  VR.ATO.IEE.410380
            Given Name must not contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE68, ' - ')

            Technical Business Rule Format:
            Contains(^IEE68, ' - ')
    
            Data Elements:
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = (report.IEE68 == null ? false : report.IEE68.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410380" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410380

            #region VR.ATO.IEE.410381

            /*  VR.ATO.IEE.410381
            Given Name must not contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE86, ' - ')

            Technical Business Rule Format:
            Contains(^IEE86, ' - ')
    
            Data Elements:
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.IEE86 == null ? false : report.IEE86.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410381" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410381

            #region VR.ATO.IEE.410383

            /*  VR.ATO.IEE.410383
            Given name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE68, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE68, '"--","''","  "')
    
            Data Elements:
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE68, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410383" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410383

            #region VR.ATO.IEE.410384

            /*  VR.ATO.IEE.410384
            Given name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE86, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE86, '"--","''","  "')
    
            Data Elements:
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE86, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410384" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410384

            #region VR.ATO.IEE.410386

            /*  VR.ATO.IEE.410386
            Other Given Name must not contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE69, ' - ')

            Technical Business Rule Format:
            Contains(^IEE69, ' - ')
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.IEE69 == null ? false : report.IEE69.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410386" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410386

            #region VR.ATO.IEE.410387

            /*  VR.ATO.IEE.410387
            Other Given Name must not contain 'space hyphen space' .
    
            Legacy Rule Format:
            Contains(^IEE87, ' - ')

            Technical Business Rule Format:
            Contains(^IEE87, ' - ')
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.IEE87 == null ? false : report.IEE87.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410387" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410387

            #region VR.ATO.IEE.410389

            /*  VR.ATO.IEE.410389
            Other given name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE69, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE69, '"--","''","  "')
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE69, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410389" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410389

            #region VR.ATO.IEE.410390

            /*  VR.ATO.IEE.410390
            Other given name must not contain repeated hyphen, apostrophe, or space characters.
    
            Legacy Rule Format:
            ContainsSet(^IEE87, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^IEE87, '"--","''","  "')
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE87, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410390" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410390

            #region VR.ATO.IEE.410391

            /*  VR.ATO.IEE.410391
            Postcode must be a valid code for an Australian address
    
            Legacy Rule Format:
            ^IEE111 = 'au' AND ^IEE8 <> NULL AND (AsNumeric(^IEE8) < 200 OR AsNumeric(^IEE8) > 9999)

            Technical Business Rule Format:
            ^IEE111 = 'au' AND ^IEE8 <> NULL AND (AsNumeric(^IEE8) < 200 OR AsNumeric(^IEE8) > 9999)
    
            Data Elements:
    
            ^IEE8 = IEE:RP:CurrentPostalAddress:AddressDetails.Postcode.Text
    
            ^IEE111 = IEE:RP:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE111 == @"au" && report.IEE8 != null && (AsNumeric(report.IEE8) < 200 || AsNumeric(report.IEE8) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410391",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsPostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410391" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE111", Value = report.IEE111 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE8", Value = GetValueOrEmpty(report.IEE8) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410391

            #region VR.ATO.IEE.410392

            /*  VR.ATO.IEE.410392
            Postcode must be a valid code for an Australian address
    
            Legacy Rule Format:
            ^IEE112 = 'au' AND ^IEE14 <> NULL AND (AsNumeric(^IEE14) < 200 OR AsNumeric(^IEE14) > 9999)

            Technical Business Rule Format:
            ^IEE112 = 'au' AND ^IEE14 <> NULL AND (AsNumeric(^IEE14) < 200 OR AsNumeric(^IEE14) > 9999)
    
            Data Elements:
    
            ^IEE14 = IEE:RP:PreviousPostalAddress:AddressDetails.Postcode.Text
    
            ^IEE112 = IEE:RP:PreviousPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE112 == @"au" && report.IEE14 != null && (AsNumeric(report.IEE14) < 200 || AsNumeric(report.IEE14) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410391",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsPostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410392" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE112", Value = report.IEE112 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE14", Value = GetValueOrEmpty(report.IEE14) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410392

            #region VR.ATO.IEE.410394

            /*  VR.ATO.IEE.410394
            Postcode must be a valid code for an Australian address
    
            Legacy Rule Format:
            ^IEE155 = 'au' AND ^IEE56 <> NULL AND (AsNumeric(^IEE56) < 200 OR AsNumeric(^IEE56) > 9999)

            Technical Business Rule Format:
            ^IEE155 = 'au' AND ^IEE56 <> NULL AND (AsNumeric(^IEE56) < 200 OR AsNumeric(^IEE56) > 9999)
    
            Data Elements:
    
            ^IEE56 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Postcode.Text
    
            ^IEE155 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE155 == @"au" && report.IEE56 != null && (AsNumeric(report.IEE56) < 200 || AsNumeric(report.IEE56) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410391",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsPostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410394" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE155", Value = report.IEE155 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE56", Value = GetValueOrEmpty(report.IEE56) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410394

            #region VR.ATO.IEE.410395

            /*  VR.ATO.IEE.410395
            Postcode must be a valid code for an Australian address
    
            Legacy Rule Format:
            ^IEE114 = 'au' AND ^IEE73 <> NULL AND (AsNumeric(^IEE73) < 200 OR AsNumeric(^IEE73) > 9999)

            Technical Business Rule Format:
            ^IEE114 = 'au' AND ^IEE73 <> NULL AND (AsNumeric(^IEE73) < 200 OR AsNumeric(^IEE73) > 9999)
    
            Data Elements:
    
            ^IEE73 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Postcode.Text
    
            ^IEE114 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.IEE114 == @"au" && report.IEE73 != null && (AsNumeric(report.IEE73) < 200 || AsNumeric(report.IEE73) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410391",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:PostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410395" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE114", Value = report.IEE114 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE73", Value = GetValueOrEmpty(report.IEE73) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410395

            #region VR.ATO.IEE.410397

            /*  VR.ATO.IEE.410397
            If Address Line 1 contains C/-,  Address line 2 must be present
    
            Legacy Rule Format:
            ((Contains(^IEE5, 'C/-')) AND (^IEE122 = BLANK))

            Technical Business Rule Format:
            ((Contains(^IEE5, 'C/-')) AND (^IEE122 = BLANK))
    
            Data Elements:
    
            ^IEE5 = IEE:RP:CurrentPostalAddress:AddressDetails.Line1.Text
    
            ^IEE122 = IEE:RP:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = ((report.IEE5 == null ? false : report.IEE5.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.IEE122) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410397" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE5", Value = report.IEE5 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE122", Value = GetValueOrEmpty(report.IEE122) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410397

            #region VR.ATO.IEE.410398

            /*  VR.ATO.IEE.410398
            If Address Line 1 contains C/-,  Address line 2 must be present
    
            Legacy Rule Format:
            ((Contains(^IEE11, 'C/-')) AND (^IEE126 = BLANK))

            Technical Business Rule Format:
            ((Contains(^IEE11, 'C/-')) AND (^IEE126 = BLANK))
    
            Data Elements:
    
            ^IEE11 = IEE:RP:PreviousPostalAddress:AddressDetails.Line1.Text
    
            ^IEE126 = IEE:RP:PreviousPostalAddress:AddressDetails.Line2.Text
            */
            assertion = ((report.IEE11 == null ? false : report.IEE11.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.IEE126) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410398" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE11", Value = report.IEE11 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE126", Value = GetValueOrEmpty(report.IEE126) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410398

            #region VR.ATO.IEE.410400

            /*  VR.ATO.IEE.410400
            If Address Line 1 contains C/-,  Address line 2 must be present
    
            Legacy Rule Format:
            ((Contains(^IEE53, 'C/-')) AND (^IEE134 = BLANK))

            Technical Business Rule Format:
            ((Contains(^IEE53, 'C/-')) AND (^IEE134 = BLANK))
    
            Data Elements:
    
            ^IEE53 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line1.Text
    
            ^IEE134 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = ((report.IEE53 == null ? false : report.IEE53.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.IEE134) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410400" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE53", Value = report.IEE53 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE134", Value = GetValueOrEmpty(report.IEE134) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410400

            #region VR.ATO.IEE.410401

            /*  VR.ATO.IEE.410401
            If Address Line 1 contains C/-,  Address line 2 must be present
    
            Legacy Rule Format:
            ((Contains(^IEE70, 'C/-')) AND (^IEE138 = BLANK))

            Technical Business Rule Format:
            ((Contains(^IEE70, 'C/-')) AND (^IEE138 = BLANK))
    
            Data Elements:
    
            ^IEE70 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line1.Text
    
            ^IEE138 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.IEE70 == null ? false : report.IEE70.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.IEE138) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410401" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE70", Value = report.IEE70 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE138", Value = GetValueOrEmpty(report.IEE138) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410401

            #region VR.ATO.IEE.410403

            /*  VR.ATO.IEE.410403
            TFN must pass the TFN algorithm check
    
            Legacy Rule Format:
            (^IEE50 <> NULL) AND (FailsTFNAlgorithm(^IEE50))

            Technical Business Rule Format:
            (^IEE50 <> NULL) AND (FailsTFNAlgorithm(^IEE50))
    
            Data Elements:
    
            ^IEE50 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.IEE50 != null && FailsTFNAlgorithm(report.IEE50));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN must be a valid TFN",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410403" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE50", Value = GetValueOrEmpty(report.IEE50) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410403

            #region VR.ATO.IEE.410404

            /*  VR.ATO.IEE.410404
            TFN must pass the TFN algorithm check
    
            Legacy Rule Format:
            (^IEE62 <> NULL) AND (FailsTFNAlgorithm(^IEE62))

            Technical Business Rule Format:
            (^IEE62 <> NULL) AND (FailsTFNAlgorithm(^IEE62))
    
            Data Elements:
    
            ^IEE62 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.IEE62 != null && FailsTFNAlgorithm(report.IEE62));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN must be a valid TFN",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410404" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE62", Value = GetValueOrEmpty(report.IEE62) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410404

            #region VR.ATO.IEE.410406

            /*  VR.ATO.IEE.410406
            Non-individual Name cannot start with 'T/A' followed by a space and end with a space followed by 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^IEE52, 'T/A ')) AND (EndsWithSet(^IEE52, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^IEE52, 'T/A ')) AND (EndsWithSet(^IEE52, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^IEE52 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.IEE52, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.IEE52, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410406" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE52", Value = report.IEE52 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410406

            #region VR.ATO.IEE.410408

            /*  VR.ATO.IEE.410408
            ARBN must pass the ARBN algorithm check
    
            Legacy Rule Format:
            (^IEE101 <> NULL) AND (FailsARBNAlgorithm(^IEE101))

            Technical Business Rule Format:
            (^IEE101 <> NULL) AND (FailsARBNAlgorithm(^IEE101))
    
            Data Elements:
    
            ^IEE101 = IEE:RP:Identifiers.AustralianRegisteredBodyNumber.Identifier
            */
            assertion = (report.IEE101 != null && FailsACNAlgorithm(report.IEE101));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ARBN must be a valid number that passes the algorithm",
                    Location = "/tns:IEE/tns:RP/tns:AustralianRegisteredBodyNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410408" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE101", Value = GetValueOrEmpty(report.IEE101) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410408

            #region VR.ATO.IEE.410412

            /*  VR.ATO.IEE.410412
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^IEE69 <> BLANK) AND (NotContainsSet(^IEE69, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IEE69 <> BLANK) AND (NotContainsSet(^IEE69, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE69) != true && !(IsMatch(report.IEE69, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410412" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410412

            #region VR.ATO.IEE.410413

            /*  VR.ATO.IEE.410413
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^IEE87 <> BLANK) AND (NotContainsSet(^IEE87, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IEE87 <> BLANK) AND (NotContainsSet(^IEE87, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE87) != true && !(IsMatch(report.IEE87, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410413" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410413

            #region VR.ATO.IEE.410415

            /*  VR.ATO.IEE.410415
            Family name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
            Legacy Rule Format:
            (FoundSet(^IEE66, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IEE66, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IEE66 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE66, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410415" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE66", Value = report.IEE66 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410415

            #region VR.ATO.IEE.410416

            /*  VR.ATO.IEE.410416
            Family name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
            Legacy Rule Format:
            (FoundSet(^IEE84, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IEE84, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IEE84 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE84, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410416" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE84", Value = report.IEE84 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410416

            #region VR.ATO.IEE.410418

            /*  VR.ATO.IEE.410418
            Family name must not contain MR, MRS, MISS, MS with space before and after.
    
            Legacy Rule Format:
            (FoundSet(^IEE66, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IEE66, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IEE66 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE66, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410418" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE66", Value = report.IEE66 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410418

            #region VR.ATO.IEE.410419

            /*  VR.ATO.IEE.410419
            Family name must not contain MR, MRS, MISS, MS with space before and after.
    
            Legacy Rule Format:
            (FoundSet(^IEE84, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IEE84, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IEE84 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.IEE84, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410419" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE84", Value = report.IEE84 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410419

            #region VR.ATO.IEE.410421

            /*  VR.ATO.IEE.410421
            Given Name must not contain trustee reference
    
            Legacy Rule Format:
            (FoundSet(^IEE68, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IEE68, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE68, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410421" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410421

            #region VR.ATO.IEE.410422

            /*  VR.ATO.IEE.410422
            Given Name must not contain trustee reference
    
            Legacy Rule Format:
            (FoundSet(^IEE86, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IEE86, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE86, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410422" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410422

            #region VR.ATO.IEE.410424

            /*  VR.ATO.IEE.410424
            Given Name must not contain a title
    
            Legacy Rule Format:
            (FoundSet(^IEE68, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IEE68, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE68, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410424" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410424

            #region VR.ATO.IEE.410425

            /*  VR.ATO.IEE.410425
            Given Name must not contain a title
    
            Legacy Rule Format:
            (FoundSet(^IEE86, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IEE86, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.IEE86, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410425" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410425

            #region VR.ATO.IEE.410426

            /*  VR.ATO.IEE.410426
            Address Line 1 cannot contain 'as above'
    
            Legacy Rule Format:
            (FoundSet(^IEE5, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^IEE5, '"AS ABOVE"'))
    
            Data Elements:
    
            ^IEE5 = IEE:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.IEE5, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410426" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE5", Value = report.IEE5 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410426

            #region VR.ATO.IEE.410427

            /*  VR.ATO.IEE.410427
            Address Line 1 cannot contain 'as above'
    
            Legacy Rule Format:
            (FoundSet(^IEE11, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^IEE11, '"AS ABOVE"'))
    
            Data Elements:
    
            ^IEE11 = IEE:RP:PreviousPostalAddress:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.IEE11, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410427" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE11", Value = report.IEE11 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410427

            #region VR.ATO.IEE.410429

            /*  VR.ATO.IEE.410429
            Address Line 1 cannot contain 'as above'
    
            Legacy Rule Format:
            (FoundSet(^IEE53, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^IEE53, '"AS ABOVE"'))
    
            Data Elements:
    
            ^IEE53 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.IEE53, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410429" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE53", Value = report.IEE53 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410429

            #region VR.ATO.IEE.410430

            /*  VR.ATO.IEE.410430
            Address Line 1 cannot contain 'as above'
    
            Legacy Rule Format:
            (FoundSet(^IEE70, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^IEE70, '"AS ABOVE"'))
    
            Data Elements:
    
            ^IEE70 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.IEE70, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410430" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE70", Value = report.IEE70 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410430

            #region VR.ATO.IEE.410432

            /*  VR.ATO.IEE.410432
            Other given name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
            Legacy Rule Format:
            (FoundSet(^IEE69, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IEE69, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE69, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410432" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410432

            #region VR.ATO.IEE.410433

            /*  VR.ATO.IEE.410433
            Other given name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
            Legacy Rule Format:
            (FoundSet(^IEE87, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^IEE87, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE87, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410433" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410433

            #region VR.ATO.IEE.410435

            /*  VR.ATO.IEE.410435
            Other Given Name must not contain a title
    
            Legacy Rule Format:
            (FoundSet(^IEE69, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IEE69, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE69, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410435" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410435

            #region VR.ATO.IEE.410436

            /*  VR.ATO.IEE.410436
            Other Given Name must not contain a title
    
            Legacy Rule Format:
            (FoundSet(^IEE87, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^IEE87, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.IEE87, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410436" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410436

            #region VR.ATO.IEE.410438

            /*  VR.ATO.IEE.410438
            Firstname must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            (^IEE1025 <> NULL) AND ((^IEE69 <> BLANK) AND (^IEE68 = BLANK))

            Technical Business Rule Format:
            (^IEE1025 <> NULL) AND ((^IEE69 <> BLANK) AND (^IEE68 = BLANK))
    
            Data Elements:
    
            ^IEE69 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.OtherGivenName.Text
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
    
            ^IEE1025 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists != false && (string.IsNullOrWhiteSpace(report.IEE69) != true && string.IsNullOrWhiteSpace(report.IEE68) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410438" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE69", Value = GetValueOrEmpty(report.IEE69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410438

            #region VR.ATO.IEE.410439

            /*  VR.ATO.IEE.410439
            Firstname must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            (^IEE1029 <> NULL) AND ((^IEE87 <> BLANK) AND (^IEE86 = BLANK))

            Technical Business Rule Format:
            (^IEE1029 <> NULL) AND ((^IEE87 <> BLANK) AND (^IEE86 = BLANK))
    
            Data Elements:
    
            ^IEE87 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.OtherGivenName.Text
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
    
            ^IEE1029 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails
            */
            assertion = (report.RP_DeclarationOfPublicOfficer_PublicOfficerNameDetailsCollectionExists != false && (string.IsNullOrWhiteSpace(report.IEE87) != true && string.IsNullOrWhiteSpace(report.IEE86) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410439" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE87", Value = GetValueOrEmpty(report.IEE87) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410439

            #region VR.ATO.IEE.410440

            /*  VR.ATO.IEE.410440
            TFN must pass the TFN algorithm check
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            FailsTFNAlgorithm(^IEE3)

            Technical Business Rule Format:
            FailsTFNAlgorithm(^IEE3)
    
            Data Elements:
    
            ^IEE3 = IEE:RP:Identifiers.TaxFileNumber.Identifier
            */
            if (isStandalone)
            {
                assertion = FailsTFNAlgorithm(report.IEE3);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.GEN.428016",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax File Number has failed the algorithm check",
                        Location = "/tns:IEE/tns:RP/tns:TaxFileNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410440" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE3", Value = report.IEE3 });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410440

            #region VR.ATO.IEE.410441

            /*  VR.ATO.IEE.410441
            ACN must pass the ACN algorithm check
    
            Legacy Rule Format:
            (^IEE100 <> NULL) AND (FailsACNAlgorithm(^IEE100))

            Technical Business Rule Format:
            (^IEE100 <> NULL) AND (FailsACNAlgorithm(^IEE100))
    
            Data Elements:
    
            ^IEE100 = IEE:RP:Identifiers.AustralianCompanyNumber.Identifier
            */
            assertion = (report.IEE100 != null && FailsACNAlgorithm(report.IEE100));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ACN",
                    LongDescription = @"The Australian Company Number (ACN) has failed the algorithm check",
                    Location = "/tns:IEE/tns:RP/tns:AustralianCompanyNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410441" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE100", Value = GetValueOrEmpty(report.IEE100) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410441

            #region VR.ATO.IEE.410443

            /*  VR.ATO.IEE.410443
            ABN must pass the ABN algorithm check
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            (^IEE162 <> NULL) and (FailsABNAlgorithm(^IEE162))

            Technical Business Rule Format:
            (^IEE162 <> NULL) and (FailsABNAlgorithm(^IEE162))
    
            Data Elements:
    
            ^IEE162 = IEE:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            if (isStandalone)
            {
                assertion = (report.IEE162 != null && FailsABNAlgorithm(report.IEE162));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.GEN.000477",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"ABN is invalid",
                        Location = "/tns:IEE/tns:RP/tns:AustralianBusinessNumberId",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410443" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE162", Value = GetValueOrEmpty(report.IEE162) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410443

            #region VR.ATO.IEE.410444

            /*  VR.ATO.IEE.410444
            Address - Locality must not contain a state code for an Australian address
    
            Legacy Rule Format:
            (^IEE111 = 'au') AND (FoundSet(^IEE6, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^IEE111 = 'au') AND (FoundSet(^IEE6, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IEE6 = IEE:RP:CurrentPostalAddress:AddressDetails.LocalityName.Text
    
            ^IEE111 = IEE:RP:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE111 == @"au" && IsMatch(report.IEE6, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:IEE/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410444" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE111", Value = report.IEE111 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE6", Value = report.IEE6 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410444

            #region VR.ATO.IEE.410445

            /*  VR.ATO.IEE.410445
            Address - Locality must not contain a state code for an Australian address
    
            Legacy Rule Format:
            (^IEE112 = 'au') AND (FoundSet(^IEE12, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^IEE112 = 'au') AND (FoundSet(^IEE12, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IEE12 = IEE:RP:PreviousPostalAddress:AddressDetails.LocalityName.Text
    
            ^IEE112 = IEE:RP:PreviousPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE112 == @"au" && IsMatch(report.IEE12, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:IEE/tns:RP/tns:PreviousPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410445" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE112", Value = report.IEE112 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE12", Value = report.IEE12 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410445

            #region VR.ATO.IEE.410447

            /*  VR.ATO.IEE.410447
            Address - Locality must not contain a state code for an Australian address
    
            Legacy Rule Format:
            (^IEE155 = 'au') AND (FoundSet(^IEE54, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^IEE155 = 'au') AND (FoundSet(^IEE54, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IEE54 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.LocalityName.Text
    
            ^IEE155 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.IEE155 == @"au" && IsMatch(report.IEE54, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410447" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE155", Value = report.IEE155 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE54", Value = report.IEE54 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410447

            #region VR.ATO.IEE.410448

            /*  VR.ATO.IEE.410448
            Address - Locality must not contain a state code for an Australian address
    
            Legacy Rule Format:
            (^IEE114 = 'au') AND (FoundSet(^IEE71, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^IEE114 = 'au') AND (FoundSet(^IEE71, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^IEE71 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.LocalityName.Text
    
            ^IEE114 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:CurrentResidentialAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.IEE114 == @"au" && IsMatch(report.IEE71, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:CurrentResidentialAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410448" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE114", Value = report.IEE114 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE71", Value = report.IEE71 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410448

            #region VR.ATO.IEE.410449

            /*  VR.ATO.IEE.410449
            Hours taken to prepare and complete form must be 1 to 999
    
            Legacy Rule Format:
            (^IEE94 <> NULL) AND ((^IEE94 < 1 OR ^IEE94 > 999))

            Technical Business Rule Format:
            (^IEE94 <> NULL) AND ((^IEE94 < 1 OR ^IEE94 > 999))
    
            Data Elements:
    
            ^IEE94 = IEE:RP:Report.CompletionHours.Number
            */
            assertion = (report.IEE94 != null && (report.IEE94.GetValueOrDefault() < 1 || report.IEE94.GetValueOrDefault() > 999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430206",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hours taken to prepare and complete form must be 1 to 999",
                    Location = "/tns:IEE/tns:RP/tns:CompletionHoursN",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410449" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE94", Value = report.IEE94.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410449

            #region VR.ATO.IEE.410450

            /*  VR.ATO.IEE.410450
            Signature date must not be a future date
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1001 <> NULL AND ^IEE181 > Today()

            Technical Business Rule Format:
            ^IEE1001 <> NULL AND ^IEE181 > Today()
    
            Data Elements:
    
            ^IEE181 = IEE:INT:Declaration:Declaration.Signature.Date
    
            ^IEE1001 = IEE:INT
            */
            if (isStandalone)
            {
                assertion = (report.IntCollectionExists != false && report.IEE181.GetValueOrDefault() > DateTime.Now.Date);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.GEN.430255",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Signature date must not be later than today",
                        Location = "/tns:IEE/tns:INT/tns:Declaration/tns:SignatureD",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410450" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE181", Value = GetValueOrEmptyDate(report.IEE181) });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410450

            #region VR.ATO.IEE.410452

            /*  VR.ATO.IEE.410452
            Signature date must not be a future date
    
            Legacy Rule Format:
            ^IEE191 >Today()

            Technical Business Rule Format:
            ^IEE191 >Today()
    
            Data Elements:
    
            ^IEE191 = IEE:RP:DeclarationOfPublicOfficer:Declaration.Signature.Date
            */
            assertion = (report.IEE191.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410452" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE191", Value = GetValueOrEmptyDate(report.IEE191) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410452

            #region VR.ATO.IEE.410455

            /*  VR.ATO.IEE.410455
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^IEE68 <> BLANK) AND (NotContainsSet(^IEE68, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IEE68 <> BLANK) AND (NotContainsSet(^IEE68, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IEE68 = IEE:RP:InterposedEntityElectionRevocationDetails:SpecifiedIndividual:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE68) != true && !(IsMatch(report.IEE68, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500143",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character.",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:SpecifiedIndividual/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410455" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE68", Value = GetValueOrEmpty(report.IEE68) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410455

            #region VR.ATO.IEE.410456

            /*  VR.ATO.IEE.410456
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^IEE86<> BLANK) AND (NotContainsSet(^IEE86, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^IEE86<> BLANK) AND (NotContainsSet(^IEE86, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^IEE86 = IEE:RP:DeclarationOfPublicOfficer:PublicOfficerNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.IEE86) != true && !(IsMatch(report.IEE86, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500143",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character.",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfPublicOfficer/tns:PublicOfficerNameDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410456" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE86", Value = GetValueOrEmpty(report.IEE86) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410456

            #region VR.ATO.IEE.410457

            /*  VR.ATO.IEE.410457
            One Declaration Statement Type Code must be 'TrueAndCorrect'
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            ^IEE1001 = NULL AND ^IEE16 <> '051' AND CountOccurrence(^IEE1019, ^IEE193 = 'TrueAndCorrect') <>1

            Technical Business Rule Format:
            ^IEE1001 = NULL AND ^IEE16 <> '051' AND CountOccurrence(^IEE1019, ^IEE193 = 'TrueAndCorrect') <>1
    
            Data Elements:
    
            ^IEE193 = IEE:RP:DeclarationOfTrusteeOrPartner:Declaration.StatementType.Code
    
            ^IEE1001 = IEE:INT
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
            */
            if (isStandalone)
            {
                assertion = (report.IntCollectionExists == false && report.IEE16 != @"051" && (report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? 0 : report.RP_DeclarationOfTrusteeOrPartnerCollection.Count(IEE1019Repeat => IEE1019Repeat.IEE193 == @"TrueAndCorrect")) != 1);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.IEE.410457",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"One Declaration Statement Type Code must be 'TrueAndCorrect'",
                        Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner/tns:StatementTypeC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410457" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "IEE16", Value = report.IEE16 });

                    response.Add(processMessage);
                }
            }
            #endregion // VR.ATO.IEE.410457

            #region VR.ATO.IEE.410461

            /*  VR.ATO.IEE.410461
            End Date must be after a Start date
    
            Legacy Rule Format:
            (^IEE1021 <> NULL) AND (^IEE47>= ^IEE48)

            Technical Business Rule Format:
            (^IEE1021 <> NULL) AND (^IEE47>= ^IEE48)
    
            Data Elements:
    
            ^IEE47 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^IEE48 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:SubstitutedAccountingPeriodDetails:Period.End.Date
    
            ^IEE1021 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:SubstitutedAccountingPeriodDetails
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE47.GetValueOrDefault() >= report.IEE48.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"End Date must be after a Start date",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionDetails/tns:SubstitutedAccountingPeriodDetails/tns:StartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410461" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE47", Value = GetValueOrEmptyDate(report.IEE47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE48", Value = GetValueOrEmptyDate(report.IEE48) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410461

            #region VR.ATO.IEE.410462

            /*  VR.ATO.IEE.410462
            End Date must be after a Start date
    
            Legacy Rule Format:
            (^IEE1023 <> NULL) AND (^IEE60 >= ^IEE61)

            Technical Business Rule Format:
            (^IEE1023 <> NULL) AND (^IEE60 >= ^IEE61)
    
            Data Elements:
    
            ^IEE60 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails:Period.Start.Date
    
            ^IEE61 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails:Period.End.Date
    
            ^IEE1023 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails
            */
            assertion = (report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE60.GetValueOrDefault() >= report.IEE61.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"End Date must be after a Start date",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:SubstitutedAccountingPeriodDetails/tns:StartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410462" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE60", Value = GetValueOrEmptyDate(report.IEE60) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE61", Value = GetValueOrEmptyDate(report.IEE61) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410462

            #region VR.ATO.IEE.410463

            /*  VR.ATO.IEE.410463
            Where the interposed entity is a Company or Partnership, the question regarding the central management and control of the trust must not be answered.
    
            Legacy Rule Format:
            (^IEE116 = 'E') AND (InSet(^IEE16, '"051","161"')) AND (^IEE1011 <> NULL)

            Technical Business Rule Format:
            (^IEE116 = 'E') AND (InSet(^IEE16, '"051","161"')) AND (^IEE1011 <> NULL)
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE1011 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation
            */
            assertion = (report.IEE116 == @"E" && IsMatch(report.IEE16, @"^(051|161)$") && report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformationCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410463",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Central management and control indicator must not be present if interposed entity is a Company or Partnership",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410463" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410463

            #region VR.ATO.IEE.410465

            /*  VR.ATO.IEE.410465
            The day specified for the IEE cannot be earlier than the first day of the Family Trust Election specified  income year
    
            Legacy Rule Format:
            (^IEE58 <> NULL) AND (^IEE1023 <> NULL) AND (^IEE46 < AsDate(Concat(^IEE58 - 1, '-07-01')))

            Technical Business Rule Format:
            (^IEE58 <> NULL) AND (^IEE1023 <> NULL) AND (^IEE46 < AsDate(Concat(^IEE58 - 1, '-07-01')))
    
            Data Elements:
    
            ^IEE58 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:Elections.FamilyTrustElectionStatus.Year
    
            ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
    
            ^IEE1023 = IEE:RP:InterposedEntityElectionRevocationDetails:FamilyTrust:SubstitutedAccountingPeriodDetails
            */
            assertion = (report.IEE58 != null && report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionExists != false && report.IEE46.GetValueOrDefault() < AsDate(string.Concat(report.IEE58.GetValueOrDefault() - 1, @"-07-01")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410465",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Day specified must not be before the Family Trust Election specified income year of substituted accounting period",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:FamilyTrust/tns:ElectionsElectionStatusY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410465" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE58", Value = GetValueOrEmpty(report.IEE58) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410465

            #region VR.ATO.IEE.410466

            /*  VR.ATO.IEE.410466
            Full period indicator or dates for central management and control of trust was outside Australia have been entered when the answer to the question was central management and control of the trust outside Australia is 'No'.
    
            Legacy Rule Format:
            (^IEE17 = FALSE) AND ((^IEE18 <> NULL) OR AnyOccurrence(^IEE1012, ^IEE1012 <> NULL))

            Technical Business Rule Format:
            (^IEE17 = FALSE) AND ((^IEE18 <> NULL) OR AnyOccurrence(^IEE1012, ^IEE1012 <> NULL))
    
            Data Elements:
    
            ^IEE17 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:Residency.CentralManagementAndControlOutsideAustralia.Indicator
    
            ^IEE18 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:Residency.CentralManagementAndControlOutsideAustraliaFullPeriod.Indicator
    
            ^IEE1012 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails
            */
            assertion = (report.IEE17 == false && (report.IEE18 != null || (report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection.Any(IEE1012Repeat => report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionExists != false))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410466",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If central management and control indicator is 'false', full period or dates must not be present",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ResidencyCentralManagementAndControlOutsideAustraliaI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410466" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE17", Value = GetValueOrEmpty(report.IEE17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE18", Value = GetValueOrEmpty(report.IEE18) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410466

            #region VR.ATO.IEE.410496

            /*  VR.ATO.IEE.410496
            Number of Trustee/partner Declarations supplied must not be more than the number of trustees/partners in the lodgment.
    
            Legacy Rule Format:
            (^IEE116 = 'E') AND Count(^IEE1019) > Count(^IEE1015)

            Technical Business Rule Format:
            (^IEE116 = 'E') AND Count(^IEE1019) > Count(^IEE1015)
    
            Data Elements:
    
            ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
    
            ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
    
            ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
            */
            assertion = (report.IEE116 == @"E" && Count(report.RP_DeclarationOfTrusteeOrPartnerCollectionCount) > Count(report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionCount));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410496",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of trustee or partner declarations supplied must not be more than the number of trustees or partners in the lodgement.",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? 0 : (report.RP_DeclarationOfTrusteeOrPartnerCollection.Count() == 0 ? 0 : report.RP_DeclarationOfTrusteeOrPartnerCollection.Last().OccurrenceIndex)) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410496" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE116", Value = report.IEE116 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410496

            #region VR.ATO.IEE.410499

            /*  VR.ATO.IEE.410499
            If entity type is Trust, Partnership or Fund, at least one declaration of trustee or partner must be supplied.
    
            Legacy Rule Format:
            (InSet(^IEE16, '"161","217","212"')) AND Count(^IEE1019) < 1

            Technical Business Rule Format:
            (InSet(^IEE16, '"161","217","212"')) AND Count(^IEE1019) < 1
    
            Data Elements:
    
            ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
    
            ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
            */
            assertion = (IsMatch(report.IEE16, @"^(161|217|212)$") && Count(report.RP_DeclarationOfTrusteeOrPartnerCollectionCount) < 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IEE.410499",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If entity type is Trust, Partnership or Fund, at least one declaration of trustee or partner must be supplied.",
                    Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? 0 : (report.RP_DeclarationOfTrusteeOrPartnerCollection.Count() == 0 ? 0 : report.RP_DeclarationOfTrusteeOrPartnerCollection.Last().OccurrenceIndex)) + "/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410499" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IEE16", Value = report.IEE16 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.IEE.410499

            #region Repeating report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection
            if (report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection != null)
            {
                int itemIndex3 = 0;
                foreach (IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner trusteeOrPartner in report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection)
                {
                    itemIndex3++;

                    #region VR.ATO.IEE.410025

                    /*  VR.ATO.IEE.410025
                    If Entity type is Company, Address Details of trustee or partner must be blank.
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = "051") AND (^IEE1018 <> NULL)
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (^IEE1018 <> NULL)
            
                    Data Elements:
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            
                    ^IEE1018 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress
                    */
                    assertion = (report.IEE116 == @"E" && report.IEE16 == @"051" && trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_CurrentPostalAddressCollectionExists != false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410025",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If Entity type is Company, Address Details of trustee or partner must be blank",
                            LongDescription = @"If the Entity Type is a Company, you are not required to supply details of trustee or partner",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410025" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410025

                    #region VR.ATO.IEE.410026

                    /*  VR.ATO.IEE.410026
                    If Entity type is Company, ACN of trustee or partner must be blank.
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (^IEE34 <> NULL)
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (^IEE34 <> NULL)
            
                    Data Elements:
            
                    ^IEE34 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.AustralianCompanyNumber.Identifier
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
                    */
                    assertion = (report.IEE116 == @"E" && report.IEE16 == @"051" && trusteeOrPartner.IEE34 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410026",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If Entity type is Company, company ACN of trustee or partner must be blank",
                            LongDescription = @"If the Entity Type is a Company, you are not required to supply details of trustee or partner",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410026" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE34", Value = GetValueOrEmpty(trusteeOrPartner.IEE34) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410026

                    #region VR.ATO.IEE.410027

                    /*  VR.ATO.IEE.410027
                    If Entity type is Company, Organisation name of trustee or partner must be blank.
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (^IEE32 <> BLANK)
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (^IEE32 <> BLANK)
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
                    */
                    assertion = (report.IEE116 == @"E" && report.IEE16 == @"051" && string.IsNullOrWhiteSpace(trusteeOrPartner.IEE32) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410027",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If Entity type is Company, Organisation Name of trustee or partner must be blank",
                            LongDescription = @"If the Entity Type is a Company, you are not required to enter a Name under the 'If the trustee or partner is a company' question",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410027" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410027

                    #region VR.ATO.IEE.410028

                    /*  VR.ATO.IEE.410028
                    If Entity type is Company, Residency indicator  of trustee or partner must be blank.
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE41 <> NULL))
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE41 <> NULL))
            
                    Data Elements:
            
                    ^IEE41 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.TaxPurposesPersonStatus.Indicator
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
                    */
                    assertion = (report.IEE116 == @"E" && report.IEE16 == @"051" && ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE41 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410028",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If Entity type is Company, Residency Indicator of trustee or partner must be blank",
                            LongDescription = @"If the Entity Type is a Company, you are not required to supply details of trustee or partner",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:ResidencyTaxPurposesPersonStatusI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410028" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE41", Value = GetValueOrEmpty(trusteeOrPartner.IEE41) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410028

                    #region VR.ATO.IEE.410029

                    /*  VR.ATO.IEE.410029
                    If Entity type is Company, details of residency of trustee or partner must be blank.
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE42 <> NULL))
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE42 <> NULL))
            
                    Data Elements:
            
                    ^IEE42 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.NonResidentFullPeriod.Indicator
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
                    */
                    assertion = (report.IEE116 == @"E" && report.IEE16 == @"051" && ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE42 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410029",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If Entity type is Company, details of residency of trustee or partner must be blank",
                            LongDescription = @"If the Entity Type is a Company, you are not required to supply details of trustee or partner",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:ResidencyNonResidentFullPeriodI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410029" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE42", Value = GetValueOrEmpty(trusteeOrPartner.IEE42) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410029

                    #region VR.ATO.IEE.410032

                    /*  VR.ATO.IEE.410032
                    If Entity type is Company, TFN non-existent indicator of trustee or partner must be blank.
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE26 <> NULL))
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (^IEE16 = '051') AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE26 <> NULL))
            
                    Data Elements:
            
                    ^IEE26 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.TaxFileNumberInexistent.Indicator
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
                    */
                    assertion = (report.IEE116 == @"E" && report.IEE16 == @"051" && ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE26 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410032",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If Entity type is Company, TFN non-existent indicator of trustee or partner must not be provided",
                            LongDescription = @"If the Entity Type is a Company, you are not required to supply details of trustee or partner",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:TaxFileNumberInexistentI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410032" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE26", Value = GetValueOrEmpty(trusteeOrPartner.IEE26) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410032

                    #region VR.ATO.IEE.410042

                    /*  VR.ATO.IEE.410042
                    Trustee FamilyNameand Non-individualname must not both be entered.  Only one name is allowed.
    
                    Legacy Rule Format:
                    (^IEE28 <> NULL) AND (^IEE32 <> BLANK)
        
                    Technical Business Rule Format:
                    (^IEE28 <> NULL) AND (^IEE32 <> BLANK)
            
                    Data Elements:
            
                    ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (trusteeOrPartner.IEE28 != null && string.IsNullOrWhiteSpace(trusteeOrPartner.IEE32) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410042",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee Family Name and Non-individual Name must not both be present. Only one name is allowed",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410042" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE28", Value = trusteeOrPartner.IEE28 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410042

                    #region VR.ATO.IEE.410085

                    /*  VR.ATO.IEE.410085
                    TFN of Trustee must not be entered if TFN non existent indicator is entered
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE26 = TRUE) AND ^IEE25 <> NULL
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE26 = TRUE) AND ^IEE25 <> NULL
            
                    Data Elements:
            
                    ^IEE26 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.TaxFileNumberInexistent.Indicator
            
                    ^IEE25 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.TaxFileNumber.Identifier
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE26 == true && trusteeOrPartner.IEE25 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410085",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If TFN non existent indicator is 'true', TFN of Trustee must not be present",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:TaxFileNumberInexistentI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410085" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE26", Value = GetValueOrEmpty(trusteeOrPartner.IEE26) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE25", Value = GetValueOrEmpty(trusteeOrPartner.IEE25) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410085

                    #region VR.ATO.IEE.410092

                    /*  VR.ATO.IEE.410092
                    The answer to the question was trustee or partner a non-resident for tax purposes is 'Yes' and full period indicator or dates company was non-resident have not been supplied
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE41 = TRUE) AND (^IEE42 <> TRUE) AND Count(^IEE1016) = 0
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE41 = TRUE) AND (^IEE42 <> TRUE) AND Count(^IEE1016) = 0
            
                    Data Elements:
            
                    ^IEE41 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.TaxPurposesPersonStatus.Indicator
            
                    ^IEE42 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.NonResidentFullPeriod.Indicator
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
            
                    ^IEE1016 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE41 == true && trusteeOrPartner.IEE42 != true && Count(trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionCount) == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410092",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Full period indicator or dates must be present if trustee or partner is a non-resident",
                            LongDescription = @"The answer to the question was trustee or partner a non-resident for tax purposes is 'Yes' and full period indicator or dates trustee or partner was non-resident have not been supplied",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:ResidencyTaxPurposesPersonStatusI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410092" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE41", Value = GetValueOrEmpty(trusteeOrPartner.IEE41) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE42", Value = GetValueOrEmpty(trusteeOrPartner.IEE42) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410092

                    #region VR.ATO.IEE.410093

                    /*  VR.ATO.IEE.410093
                    The answer to the question was trustee or partner a non-resident for tax purposes is 'No' and full period indicator or dates company was non-resident have been supplied
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE41 = FALSE) AND ((^IEE42 = TRUE) OR AnyOccurrence(^IEE1016, ^IEE1016 <> NULL))
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE41 = FALSE) AND ((^IEE42 = TRUE) OR AnyOccurrence(^IEE1016, ^IEE1016 <> NULL))
            
                    Data Elements:
            
                    ^IEE41 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.TaxPurposesPersonStatus.Indicator
            
                    ^IEE42 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.NonResidentFullPeriod.Indicator
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
            
                    ^IEE1016 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE41 == false && (trusteeOrPartner.IEE42 == true || (trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection == null ? false : trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection.Any(IEE1016Repeat => trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionExists != false))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410093",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Full period indicator and dates must not be present if trustee or partner was not a non-resident",
                            LongDescription = @"The answer to the question was trustee or partner a non-resident for tax purposes is 'No' and full period indicator or dates company was non-resident have been supplied",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:ResidencyTaxPurposesPersonStatusI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410093" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE41", Value = GetValueOrEmpty(trusteeOrPartner.IEE41) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE42", Value = GetValueOrEmpty(trusteeOrPartner.IEE42) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410093

                    #region VR.ATO.IEE.410094

                    /*  VR.ATO.IEE.410094
                    Both full period indicator and specific dates have been supplied for non-resident trustee or partner
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE42 = TRUE) AND AnyOccurrence(^IEE1016, ^IEE1016 <> NULL)
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE42 = TRUE) AND AnyOccurrence(^IEE1016, ^IEE1016 <> NULL)
            
                    Data Elements:
            
                    ^IEE42 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Residency.NonResidentFullPeriod.Indicator
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
            
                    ^IEE1016 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE42 == true && (trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection == null ? false : trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection.Any(IEE1016Repeat => trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionExists != false)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410094",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Full period indicator and dates trustee or partner was a non-resident must not both be present",
                            LongDescription = @"Both full period indicator and specific dates have been supplied for non-resident trustee or partner",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:ResidencyNonResidentFullPeriodI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410094" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE42", Value = GetValueOrEmpty(trusteeOrPartner.IEE42) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410094

                    #region VR.ATO.IEE.410100

                    /*  VR.ATO.IEE.410100
                    The interposed entity code is '161', '212' or '217' and the field indicating that the trustee or partner does not have a tax file number is blank and no tax file number has been supplied
    
                    Legacy Rule Format:
                    (^IEE116 = 'E') AND (InSet(^IEE16, '"217","212","161"')) AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE26 <> TRUE) AND (^IEE25 = NULL))
        
                    Technical Business Rule Format:
                    (^IEE116 = 'E') AND (InSet(^IEE16, '"217","212","161"')) AND (AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE26 <> TRUE) AND (^IEE25 = NULL))
            
                    Data Elements:
            
                    ^IEE25 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.TaxFileNumber.Identifier
            
                    ^IEE16 = IEE:RP:InterposedEntityElectionRevocationDetails:Party.Type.Code
            
                    ^IEE26 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.TaxFileNumberInexistent.Indicator
            
                    ^IEE116 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.InterposedEntityElectionOrRevocation.Code
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
                    */
                    assertion = (report.IEE116 == @"E" && IsMatch(report.IEE16, @"^(217|212|161)$") && ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.IEE26 != true && trusteeOrPartner.IEE25 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410100",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"The TFN of the trustee or partner or the no TFN indicator must be present",
                            LongDescription = @"The interposed entity code is '161', '217' or '212' and the field indicating that the 'trustee or partner does not have a tax file number' is blank, and no tax file number has been supplied",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410100" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE116", Value = report.IEE116 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE16", Value = report.IEE16 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE26", Value = GetValueOrEmpty(trusteeOrPartner.IEE26) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE25", Value = GetValueOrEmpty(trusteeOrPartner.IEE25) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410100

                    #region VR.ATO.IEE.410222

                    /*  VR.ATO.IEE.410222
                    An ARBN and an ACN must not both be provided
    
                    Legacy Rule Format:
                    (^IEE34 <> NULL) AND (^IEE95 <> NULL)
        
                    Technical Business Rule Format:
                    (^IEE34 <> NULL) AND (^IEE95 <> NULL)
            
                    Data Elements:
            
                    ^IEE34 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.AustralianCompanyNumber.Identifier
            
                    ^IEE95 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (trusteeOrPartner.IEE34 != null && trusteeOrPartner.IEE95 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410222",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"An ARBN and an ACN must not both be provided",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410222" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE34", Value = GetValueOrEmpty(trusteeOrPartner.IEE34) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE95", Value = GetValueOrEmpty(trusteeOrPartner.IEE95) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410222

                    #region VR.ATO.IEE.410305

                    /*  VR.ATO.IEE.410305
                    Postcode and State must be present for an Australian address
    
                    Legacy Rule Format:
                    (^IEE113 = 'au') AND ((^IEE38 = NULL) OR (^IEE39 = NULL))
        
                    Technical Business Rule Format:
                    (^IEE113 = 'au') AND ((^IEE38 = NULL) OR (^IEE39 = NULL))
            
                    Data Elements:
            
                    ^IEE113 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Country.Code
            
                    ^IEE38 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.StateOrTerritory.Code
            
                    ^IEE39 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Postcode.Text
                    */
                    assertion = (trusteeOrPartner.IEE113 == @"au" && (trusteeOrPartner.IEE38 == null || trusteeOrPartner.IEE39 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410305",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Postcode and State must be present for an Australian address",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsCountryC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410305" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE113", Value = trusteeOrPartner.IEE113 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE38", Value = GetValueOrEmpty(trusteeOrPartner.IEE38) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE39", Value = GetValueOrEmpty(trusteeOrPartner.IEE39) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410305

                    #region VR.ATO.IEE.410313

                    /*  VR.ATO.IEE.410313
                    Family name must not contain a suffix
    
                    Legacy Rule Format:
                    (FoundSet(^IEE28, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE28, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE28, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410313",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410313" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE28", Value = trusteeOrPartner.IEE28 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410313

                    #region VR.ATO.IEE.410314

                    /*  VR.ATO.IEE.410314
                    Given Name must not contain a suffix
    
                    Legacy Rule Format:
                    (FoundSet(^IEE30, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE30, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE30, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410314",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410314" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410314

                    #region VR.ATO.IEE.410315

                    /*  VR.ATO.IEE.410315
                    Other Given Name must not contain a suffix
    
                    Legacy Rule Format:
                    (FoundSet(^IEE31, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE31, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE31, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410315",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410315" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410315

                    #region VR.ATO.IEE.410328

                    /*  VR.ATO.IEE.410328
                    Either Trustee FamilyName or Non-individualname must be present.
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE1017 = NULL) AND (^IEE32 = BLANK)
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1015, ^IEE1015 <> NULL) AND (^IEE1017 = NULL) AND (^IEE32 = BLANK)
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
            
                    ^IEE1015 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner
            
                    ^IEE1017 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Any(IEE1015Repeat => report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists != false)) && trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_IndividualNameDetailsCollectionExists == false && string.IsNullOrWhiteSpace(trusteeOrPartner.IEE32) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410328",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Either Trustee name or non-individual name must be present.",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410328" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410328

                    #region VR.ATO.IEE.410337

                    /*  VR.ATO.IEE.410337
                    Address must not contain "UNKNOWN".
    
                    Legacy Rule Format:
                    Contains(^IEE35, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^IEE35, 'UNKNOWN')
            
                    Data Elements:
            
                    ^IEE35 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line1.Text
                    */
                    assertion = (trusteeOrPartner.IEE35 == null ? false : trusteeOrPartner.IEE35.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000405",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 cannot contain ""UNKNOWN""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410337" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE35", Value = trusteeOrPartner.IEE35 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410337

                    #region VR.ATO.IEE.410342

                    /*  VR.ATO.IEE.410342
                    Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    (^IEE35 <> NULL) AND (StartsWithSet(^IEE35,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE35, '" C/O "," C/ "," Care Of "'))
        
                    Technical Business Rule Format:
                    (^IEE35 <> NULL) AND (StartsWithSet(^IEE35,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE35, '" C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^IEE35 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line1.Text
                    */
                    assertion = (trusteeOrPartner.IEE35 != null && (IsMatch(trusteeOrPartner.IEE35, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(trusteeOrPartner.IEE35, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000406",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410342" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE35", Value = trusteeOrPartner.IEE35 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410342

                    #region VR.ATO.IEE.410347

                    /*  VR.ATO.IEE.410347
                    Address Line 2 cannot contain 'UNKNOWN'
    
                    Legacy Rule Format:
                    Contains(^IEE36, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^IEE36, 'UNKNOWN')
            
                    Data Elements:
            
                    ^IEE36 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line2.Text
                    */
                    assertion = (trusteeOrPartner.IEE36 == null ? false : trusteeOrPartner.IEE36.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000407",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 cannot contain ""UNKNOWN""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410347" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE36", Value = GetValueOrEmpty(trusteeOrPartner.IEE36) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410347

                    #region VR.ATO.IEE.410352

                    /*  VR.ATO.IEE.410352
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    (^IEE36<> BLANK) AND (StartsWithSet(^IEE36,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE36, '" C/- "," C/O "," C/ "," Care Of "'))
        
                    Technical Business Rule Format:
                    (^IEE36<> BLANK) AND (StartsWithSet(^IEE36,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^IEE36, '" C/- "," C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^IEE36 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line2.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trusteeOrPartner.IEE36) != true && (IsMatch(trusteeOrPartner.IEE36, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(trusteeOrPartner.IEE36, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000408",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Care of' is not valid for address line 2",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410352" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE36", Value = GetValueOrEmpty(trusteeOrPartner.IEE36) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410352

                    #region VR.ATO.IEE.410357

                    /*  VR.ATO.IEE.410357
                    Address must not contain "UNKNOWN".
    
                    Legacy Rule Format:
                    Contains(^IEE37, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^IEE37, 'UNKNOWN')
            
                    Data Elements:
            
                    ^IEE37 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.LocalityName.Text
                    */
                    assertion = (trusteeOrPartner.IEE37 == null ? false : trusteeOrPartner.IEE37.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000409",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410357" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE37", Value = trusteeOrPartner.IEE37 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410357

                    #region VR.ATO.IEE.410362

                    /*  VR.ATO.IEE.410362
                    Address - Locality must not contain numerics for an Australian address
    
                    Legacy Rule Format:
                    (^IEE113 = 'au') AND ContainsSet(^IEE37, '0-9')
        
                    Technical Business Rule Format:
                    (^IEE113 = 'au') AND ContainsSet(^IEE37, '0-9')
            
                    Data Elements:
            
                    ^IEE37 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.LocalityName.Text
            
                    ^IEE113 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Country.Code
                    */
                    assertion = (trusteeOrPartner.IEE113 == @"au" && IsMatch(trusteeOrPartner.IEE37, @"[0-9]", RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000410",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain numeric characters",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410362" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE113", Value = trusteeOrPartner.IEE113 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE37", Value = trusteeOrPartner.IEE37 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410362

                    #region VR.ATO.IEE.410365

                    /*  VR.ATO.IEE.410365
                    Orqanisation Name cannot contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE32, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE32, ' - ')
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (trusteeOrPartner.IEE32 == null ? false : trusteeOrPartner.IEE32.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410365" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410365

                    #region VR.ATO.IEE.410367

                    /*  VR.ATO.IEE.410367
                    Organisation Name must not contain 'P/L'
    
                    Legacy Rule Format:
                    Contains(^IEE32, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^IEE32, 'P/L')
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (trusteeOrPartner.IEE32 == null ? false : trusteeOrPartner.IEE32.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410367" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410367

                    #region VR.ATO.IEE.410369

                    /*  VR.ATO.IEE.410369
                    Organisation Name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE32, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE32, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE32, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410369" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410369

                    #region VR.ATO.IEE.410371

                    /*  VR.ATO.IEE.410371
                    Non-individualname must not end with a leading space followed by 'T/A', 'T/A P'ship', 'T/A Pship', 'T/A P/Ship' or 'T/A Partnership'.
    
                    Legacy Rule Format:
                    EndsWithSet(^IEE32, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
        
                    Technical Business Rule Format:
                    EndsWithSet(^IEE32, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE32, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410371" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410371

                    #region VR.ATO.IEE.410373

                    /*  VR.ATO.IEE.410373
                    Family Name must not contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE28, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE28, ' - ')
            
                    Data Elements:
            
                    ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (trusteeOrPartner.IEE28 == null ? false : trusteeOrPartner.IEE28.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410373" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE28", Value = trusteeOrPartner.IEE28 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410373

                    #region VR.ATO.IEE.410376

                    /*  VR.ATO.IEE.410376
                    Family name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE28, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE28, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE28, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410376" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE28", Value = trusteeOrPartner.IEE28 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410376

                    #region VR.ATO.IEE.410379

                    /*  VR.ATO.IEE.410379
                    Given Name must not contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE30, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE30, ' - ')
            
                    Data Elements:
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (trusteeOrPartner.IEE30 == null ? false : trusteeOrPartner.IEE30.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410379" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410379

                    #region VR.ATO.IEE.410382

                    /*  VR.ATO.IEE.410382
                    Given name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE30, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE30, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE30, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410382" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410382

                    #region VR.ATO.IEE.410385

                    /*  VR.ATO.IEE.410385
                    Other Given Name must not contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE31, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE31, ' - ')
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (trusteeOrPartner.IEE31 == null ? false : trusteeOrPartner.IEE31.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410385" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410385

                    #region VR.ATO.IEE.410388

                    /*  VR.ATO.IEE.410388
                    Other given name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE31, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE31, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE31, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410388" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410388

                    #region VR.ATO.IEE.410393

                    /*  VR.ATO.IEE.410393
                    Postcode must be a valid code for an Australian address
    
                    Legacy Rule Format:
                    ^IEE113 = 'au' AND ^IEE39 <> NULL AND (AsNumeric(^IEE39) < 200 OR AsNumeric(^IEE39) > 9999)
        
                    Technical Business Rule Format:
                    ^IEE113 = 'au' AND ^IEE39 <> NULL AND (AsNumeric(^IEE39) < 200 OR AsNumeric(^IEE39) > 9999)
            
                    Data Elements:
            
                    ^IEE39 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Postcode.Text
            
                    ^IEE113 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Country.Code
                    */
                    assertion = (trusteeOrPartner.IEE113 == @"au" && trusteeOrPartner.IEE39 != null && (AsNumeric(trusteeOrPartner.IEE39) < 200 || AsNumeric(trusteeOrPartner.IEE39) > 9999));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410391",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsPostcodeT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410393" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE113", Value = trusteeOrPartner.IEE113 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE39", Value = GetValueOrEmpty(trusteeOrPartner.IEE39) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410393

                    #region VR.ATO.IEE.410399

                    /*  VR.ATO.IEE.410399
                    If Address Line 1 contains C/-,  Address line 2 must be present
    
                    Legacy Rule Format:
                    ((Contains(^IEE35, 'C/-')) AND (^IEE36 = BLANK))
        
                    Technical Business Rule Format:
                    ((Contains(^IEE35, 'C/-')) AND (^IEE36 = BLANK))
            
                    Data Elements:
            
                    ^IEE35 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line1.Text
            
                    ^IEE36 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line2.Text
                    */
                    assertion = ((trusteeOrPartner.IEE35 == null ? false : trusteeOrPartner.IEE35.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(trusteeOrPartner.IEE36) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410002",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410399" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE35", Value = trusteeOrPartner.IEE35 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE36", Value = GetValueOrEmpty(trusteeOrPartner.IEE36) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410399

                    #region VR.ATO.IEE.410402

                    /*  VR.ATO.IEE.410402
                    TFN must pass the TFN algorithm check
    
                    Legacy Rule Format:
                    (^IEE25 <> NULL) AND (FailsTFNAlgorithm(^IEE25))
        
                    Technical Business Rule Format:
                    (^IEE25 <> NULL) AND (FailsTFNAlgorithm(^IEE25))
            
                    Data Elements:
            
                    ^IEE25 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.TaxFileNumber.Identifier
                    */
                    assertion = (trusteeOrPartner.IEE25 != null && FailsTFNAlgorithm(trusteeOrPartner.IEE25));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410031",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"TFN must be a valid TFN",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410402" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE25", Value = GetValueOrEmpty(trusteeOrPartner.IEE25) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410402

                    #region VR.ATO.IEE.410405

                    /*  VR.ATO.IEE.410405
                    Non-individual Name cannot start with 'T/A' followed by a space and end with a space followed by 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    (StartsWith(^IEE32, 'T/A ')) AND (EndsWithSet(^IEE32, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^IEE32, 'T/A ')) AND (EndsWithSet(^IEE32, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (IsMatch(trusteeOrPartner.IEE32, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(trusteeOrPartner.IEE32, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410038",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410405" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410405

                    #region VR.ATO.IEE.410407

                    /*  VR.ATO.IEE.410407
                    Organisation name must contain at least one alpha or numeric character.
    
                    Legacy Rule Format:
                    (^IEE32  <> BLANK) AND (NotContainsSet(^IEE32, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^IEE32  <> BLANK) AND (NotContainsSet(^IEE32, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^IEE32 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trusteeOrPartner.IEE32) != true && !(IsMatch(trusteeOrPartner.IEE32, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410407" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE32", Value = GetValueOrEmpty(trusteeOrPartner.IEE32) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410407

                    #region VR.ATO.IEE.410409

                    /*  VR.ATO.IEE.410409
                    ABN must pass the ABN algorithm check
    
                    Legacy Rule Format:
                    (^IEE95 <> NULL) AND (FailsARBNAlgorithm(^IEE95))
        
                    Technical Business Rule Format:
                    (^IEE95 <> NULL) AND (FailsARBNAlgorithm(^IEE95))
            
                    Data Elements:
            
                    ^IEE95 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.AustralianRegisteredBodyNumber.Identifier
                    */
                    assertion = (trusteeOrPartner.IEE95 != null && FailsACNAlgorithm(trusteeOrPartner.IEE95));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410105",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"ARBN must be a valid number that passes the algorithm",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:AustralianRegisteredBodyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410409" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE95", Value = GetValueOrEmpty(trusteeOrPartner.IEE95) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410409

                    #region VR.ATO.IEE.410411

                    /*  VR.ATO.IEE.410411
                    Other given name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^IEE31 <> BLANK) AND (NotContainsSet(^IEE31, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^IEE31 <> BLANK) AND (NotContainsSet(^IEE31, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(trusteeOrPartner.IEE31) != true && !(IsMatch(trusteeOrPartner.IEE31, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410131",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Given Name must contain at least one alphabetical character",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410411" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410411

                    #region VR.ATO.IEE.410414

                    /*  VR.ATO.IEE.410414
                    Family name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
                    Legacy Rule Format:
                    (FoundSet(^IEE28, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE28, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE28, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410414" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE28", Value = trusteeOrPartner.IEE28 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410414

                    #region VR.ATO.IEE.410417

                    /*  VR.ATO.IEE.410417
                    Family name must not contain MR, MRS, MISS, MS with space before and after.
    
                    Legacy Rule Format:
                    (FoundSet(^IEE28, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE28, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^IEE28 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE28, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410417" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE28", Value = trusteeOrPartner.IEE28 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410417

                    #region VR.ATO.IEE.410420

                    /*  VR.ATO.IEE.410420
                    Given Name must not contain trustee reference
    
                    Legacy Rule Format:
                    (FoundSet(^IEE30, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE30, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE30, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410420" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410420

                    #region VR.ATO.IEE.410423

                    /*  VR.ATO.IEE.410423
                    Given Name must not contain a title
    
                    Legacy Rule Format:
                    (FoundSet(^IEE30, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE30, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE30, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410423" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410423

                    #region VR.ATO.IEE.410428

                    /*  VR.ATO.IEE.410428
                    Address Line 1 cannot contain 'as above'
    
                    Legacy Rule Format:
                    (FoundSet(^IEE35, '"AS ABOVE"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE35, '"AS ABOVE"'))
            
                    Data Elements:
            
                    ^IEE35 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Line1.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE35, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410001",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not contain 'AS ABOVE'",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410428" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE35", Value = trusteeOrPartner.IEE35 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410428

                    #region VR.ATO.IEE.410431

                    /*  VR.ATO.IEE.410431
                    Other given name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
                    Legacy Rule Format:
                    (FoundSet(^IEE31, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE31, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE31, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410431" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410431

                    #region VR.ATO.IEE.410434

                    /*  VR.ATO.IEE.410434
                    Other Given Name must not contain a title
    
                    Legacy Rule Format:
                    (FoundSet(^IEE31, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE31, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(trusteeOrPartner.IEE31, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410434" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410434

                    #region VR.ATO.IEE.410437

                    /*  VR.ATO.IEE.410437
                    Firstname must be entered if OtherGivenNames is entered
    
                    Legacy Rule Format:
                    (^IEE1017 <> NULL) AND ((^IEE31 <> BLANK) AND (^IEE30 = BLANK))
        
                    Technical Business Rule Format:
                    (^IEE1017 <> NULL) AND ((^IEE31 <> BLANK) AND (^IEE30 = BLANK))
            
                    Data Elements:
            
                    ^IEE31 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
            
                    ^IEE1017 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails
                    */
                    assertion = (trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_IndividualNameDetailsCollectionExists != false && (string.IsNullOrWhiteSpace(trusteeOrPartner.IEE31) != true && string.IsNullOrWhiteSpace(trusteeOrPartner.IEE30) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410218",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"FirstName must be entered if OtherGivenNames is entered",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410437" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE31", Value = GetValueOrEmpty(trusteeOrPartner.IEE31) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410437

                    #region VR.ATO.IEE.410442

                    /*  VR.ATO.IEE.410442
                    ACN must pass the ACN algorithm check
    
                    Legacy Rule Format:
                    (^IEE34 <> NULL) AND (FailsACNAlgorithm(^IEE34))
        
                    Technical Business Rule Format:
                    (^IEE34 <> NULL) AND (FailsACNAlgorithm(^IEE34))
            
                    Data Elements:
            
                    ^IEE34 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:Identifiers.AustralianCompanyNumber.Identifier
                    */
                    assertion = (trusteeOrPartner.IEE34 != null && FailsACNAlgorithm(trusteeOrPartner.IEE34));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428021",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid ACN",
                            LongDescription = @"The Australian Company Number (ACN) has failed the algorithm check",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:AustralianCompanyNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410442" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE34", Value = GetValueOrEmpty(trusteeOrPartner.IEE34) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410442

                    #region VR.ATO.IEE.410446

                    /*  VR.ATO.IEE.410446
                    Address - Locality must not contain a state code for an Australian address
    
                    Legacy Rule Format:
                    (^IEE113 = 'au') AND (FoundSet(^IEE37, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
        
                    Technical Business Rule Format:
                    (^IEE113 = 'au') AND (FoundSet(^IEE37, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^IEE37 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.LocalityName.Text
            
                    ^IEE113 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:CurrentPostalAddress:AddressDetails.Country.Code
                    */
                    assertion = (trusteeOrPartner.IEE113 == @"au" && IsMatch(trusteeOrPartner.IEE37, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000411",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain state code",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410446" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE113", Value = trusteeOrPartner.IEE113 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE37", Value = trusteeOrPartner.IEE37 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410446

                    #region VR.ATO.IEE.410454

                    /*  VR.ATO.IEE.410454
                    Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^IEE30<> NULL) AND (NotContainsSet(^IEE30, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^IEE30<> NULL) AND (NotContainsSet(^IEE30, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^IEE30 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (trusteeOrPartner.IEE30 != null && !(IsMatch(trusteeOrPartner.IEE30, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500143",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character.",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410454" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE30", Value = GetValueOrEmpty(trusteeOrPartner.IEE30) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410454
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection
            if (report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection != null)
            {
                int itemIndex4 = 0;
                foreach (IEE2021.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetails companyNonResidentPartPeriodDetails in report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection)
                {
                    itemIndex4++;

                    #region VR.ATO.IEE.410056

                    /*  VR.ATO.IEE.410056
                    Date company was non-resident must not be before the Day Specified or the Election Commencement Time
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1014, ^IEE1014 <> NULL) AND ((^IEE23 < ^IEE46) OR (^IEE23 < ^IEE49))
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1014, ^IEE1014 <> NULL) AND ((^IEE23 < ^IEE46) OR (^IEE23 < ^IEE49))
            
                    Data Elements:
            
                    ^IEE23 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails:Period.Start.Date
            
                    ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
            
                    ^IEE49 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Commencement.Date
            
                    ^IEE1014 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection.Any(IEE1014Repeat => report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionExists != false)) && (companyNonResidentPartPeriodDetails.IEE23.GetValueOrDefault() < report.IEE46.GetValueOrDefault() || companyNonResidentPartPeriodDetails.IEE23.GetValueOrDefault() < report.IEE49.GetValueOrDefault()));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410056",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Date company was non-resident must not be before the Day Specified or the Election Commencement Time",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:CompanyNonResidentPartPeriodDetailsCollection/tns:CompanyNonResidentPartPeriodDetails" + OccurrenceIndex(companyNonResidentPartPeriodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410056" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE23", Value = GetValueOrEmptyDate(companyNonResidentPartPeriodDetails.IEE23) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE49", Value = GetValueOrEmptyDate(report.IEE49) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410056

                    #region VR.ATO.IEE.410459

                    /*  VR.ATO.IEE.410459
                    End Date must be after a Start date
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1014, ^IEE1014 <> NULL) AND (^IEE23 >= ^IEE24)
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1014, ^IEE1014 <> NULL) AND (^IEE23 >= ^IEE24)
            
                    Data Elements:
            
                    ^IEE23 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails:Period.Start.Date
            
                    ^IEE24 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails:Period.End.Date
            
                    ^IEE1014 = IEE:RP:InterposedEntityElectionRevocationDetails:NonResidentInformation:CompanyNonResidentPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection.Any(IEE1014Repeat => report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionExists != false)) && companyNonResidentPartPeriodDetails.IEE23.GetValueOrDefault() >= companyNonResidentPartPeriodDetails.IEE24.GetValueOrDefault());
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410068",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"End Date must be after a Start date",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:NonResidentInformation/tns:CompanyNonResidentPartPeriodDetailsCollection/tns:CompanyNonResidentPartPeriodDetails" + OccurrenceIndex(companyNonResidentPartPeriodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410459" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE23", Value = GetValueOrEmptyDate(companyNonResidentPartPeriodDetails.IEE23) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE24", Value = GetValueOrEmptyDate(companyNonResidentPartPeriodDetails.IEE24) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410459
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection
            if (report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection != null)
            {
                int itemIndex3 = 0;
                foreach (IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner trusteeOrPartner in report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection)
                {
                    itemIndex3++;

                    #region Repeating trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection
                    if (trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection != null)
                    {
                        int itemIndex4 = 0;
                        foreach (IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetails nonResidentPartPeriodDetails in trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection)
                        {
                            itemIndex4++;

                            #region VR.ATO.IEE.410057

                            /*  VR.ATO.IEE.410057
                            The from date of the relevant period at which the trustee or partner was a non-resident for tax purposes (Section A Item 6) cannot be earlier than either the day specified for the interposed entity election (Section A Item 7) or the election commencement date (Section A Item 8).
            
                            Legacy Rule Format:
                            ^IEE1016 <> NULL AND ((^IEE43 < ^IEE46) OR  (^IEE43 < ^IEE49))
                
                            Technical Business Rule Format:
                            ^IEE1016 <> NULL AND ((^IEE43 < ^IEE46) OR  (^IEE43 < ^IEE49))
                    
                            Data Elements:
                    
                            ^IEE43 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails:Period.Start.Date
                    
                            ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
                    
                            ^IEE49 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Commencement.Date
                    
                            ^IEE1016 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails
                            */
                            assertion = (trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionExists != false && (nonResidentPartPeriodDetails.IEE43.GetValueOrDefault() < report.IEE46.GetValueOrDefault() || nonResidentPartPeriodDetails.IEE43.GetValueOrDefault() < report.IEE49.GetValueOrDefault()));
                            if (assertion)
                            {
                                processMessage = new ProcessMessageDocument()
                                {
                                    Code = "CMN.ATO.IEE.410057",
                                    Severity = ProcessMessageSeverity.Error,
                                    Description = @"Date trustee/partner was non-resident must not be before Day Specified or the Election Commence Date",
                                    LongDescription = @"The from date of the relevant period at which the trustee or partner was a non-resident for tax purposes (Section A Item 6) cannot be earlier than the either the day specified for the interposed entity election (Section A Item 7) or the election commencement date (Section A Item 8)",
                                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:NonResidentPartPeriodDetailsCollection/tns:NonResidentPartPeriodDetails" + OccurrenceIndex(nonResidentPartPeriodDetails.OccurrenceIndex) + "/tns:StartD",
                                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410057" } },
                                };
                                processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "IEE43", Value = GetValueOrEmptyDate(nonResidentPartPeriodDetails.IEE43) });

                                processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                                processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "IEE49", Value = GetValueOrEmptyDate(report.IEE49) });

                                response.Add(processMessage);
                            }
                            #endregion // VR.ATO.IEE.410057

                            #region VR.ATO.IEE.410460

                            /*  VR.ATO.IEE.410460
                            End Date must be after a Start date
            
                            Legacy Rule Format:
                            ^IEE1016 <> NULL AND (^IEE43 >= ^IEE44)
                
                            Technical Business Rule Format:
                            ^IEE1016 <> NULL AND (^IEE43 >= ^IEE44)
                    
                            Data Elements:
                    
                            ^IEE43 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails:Period.Start.Date
                    
                            ^IEE44 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails:Period.End.Date
                    
                            ^IEE1016 = IEE:RP:InterposedEntityElectionRevocationDetails:TrusteeOrPartner:NonResidentPartPeriodDetails
                            */
                            assertion = (trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionExists != false && nonResidentPartPeriodDetails.IEE43.GetValueOrDefault() >= nonResidentPartPeriodDetails.IEE44.GetValueOrDefault());
                            if (assertion)
                            {
                                processMessage = new ProcessMessageDocument()
                                {
                                    Code = "CMN.ATO.IEE.410068",
                                    Severity = ProcessMessageSeverity.Error,
                                    Description = @"End Date must be after a Start date",
                                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:TrusteeOrPartnerCollection/tns:TrusteeOrPartner" + OccurrenceIndex(trusteeOrPartner.OccurrenceIndex) + "/tns:NonResidentPartPeriodDetailsCollection/tns:NonResidentPartPeriodDetails" + OccurrenceIndex(nonResidentPartPeriodDetails.OccurrenceIndex) + "/tns:StartD",
                                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410460" } },
                                };
                                processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "IEE43", Value = GetValueOrEmptyDate(nonResidentPartPeriodDetails.IEE43) });

                                processMessage.Parameters.Add(new ProcessMessageParameter
                                { Name = "IEE44", Value = GetValueOrEmptyDate(nonResidentPartPeriodDetails.IEE44) });

                                response.Add(processMessage);
                            }
                            #endregion // VR.ATO.IEE.410460
                        }
                    }

                    #endregion // Foreach loop
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_DeclarationOfTrusteeOrPartnerCollection
            if (report.RP_DeclarationOfTrusteeOrPartnerCollection != null)
            {
                int itemIndex2 = 0;
                foreach (IEE2021.RP_DeclarationOfTrusteeOrPartner declarationOfTrusteeOrPartner in report.RP_DeclarationOfTrusteeOrPartnerCollection)
                {
                    itemIndex2++;

                    #region VR.ATO.IEE.410252

                    /*  VR.ATO.IEE.410252
                    Statement Accepted Indicator must be  'TRUE'
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE146 = FALSE
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE146 = FALSE
            
                    Data Elements:
            
                    ^IEE146 = IEE:RP:DeclarationOfTrusteeOrPartner:Declaration.StatementAccepted.Indicator
            
                    ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
                    */
                    assertion = ((report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)) && declarationOfTrusteeOrPartner.IEE146 == false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410252",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee or partner declaration - Statement Accepted Indicator must be 'TRUE'",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:StatementAcceptedI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410252" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE146", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE146) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410252

                    #region VR.ATO.IEE.410333

                    /*  VR.ATO.IEE.410333
                    Declaration Statement Type Code must be 'HardCopy'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE193 <> 'HardCopy'
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE193 <> 'HardCopy'
            
                    Data Elements:
            
                    ^IEE193 = IEE:RP:DeclarationOfTrusteeOrPartner:Declaration.StatementType.Code
            
                    ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
                    */
                    if (!isStandalone)
                    {
                        assertion = ((report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)) && declarationOfTrusteeOrPartner.IEE193 != @"HardCopy");
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.IEE.410333",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Declaration Statement Type Code must be 'HardCopy'",
                                Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:StatementTypeC",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410333" } },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "IEE193", Value = declarationOfTrusteeOrPartner.IEE193 });

                            response.Add(processMessage);
                        }
                    }
                    #endregion // VR.ATO.IEE.410333

                    #region VR.ATO.IEE.410334

                    /*  VR.ATO.IEE.410334
                    Declaration Statement Type Code must be 'HardCopy'
    
                    Legacy Rule Format:
                    WHERE PARENT RETURN DOES NOT EXIST
                    ^IEE1001 <> NULL AND AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE193 <> 'HardCopy'
        
                    Technical Business Rule Format:
                    ^IEE1001 <> NULL AND AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE193 <> 'HardCopy'
            
                    Data Elements:
            
                    ^IEE193 = IEE:RP:DeclarationOfTrusteeOrPartner:Declaration.StatementType.Code
            
                    ^IEE1001 = IEE:INT
            
                    ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
                    */
                    if (isStandalone)
                    {
                        assertion = (report.IntCollectionExists != false && (report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)) && declarationOfTrusteeOrPartner.IEE193 != @"HardCopy");
                        if (assertion)
                        {
                            processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.IEE.410334",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Declaration Statement Type Code must be 'HardCopy'",
                                Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:StatementTypeC",
                                Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410334" } },
                            };
                            processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "IEE193", Value = declarationOfTrusteeOrPartner.IEE193 });

                            response.Add(processMessage);
                        }
                    }
                    #endregion // VR.ATO.IEE.410334

                    #region VR.ATO.IEE.410451

                    /*  VR.ATO.IEE.410451
                    Signature date must not be a future date
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE90 > Today()
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ^IEE90 > Today()
            
                    Data Elements:
            
                    ^IEE90 = IEE:RP:DeclarationOfTrusteeOrPartner:Declaration.Signature.Date
            
                    ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
                    */
                    assertion = ((report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)) && declarationOfTrusteeOrPartner.IEE90.GetValueOrDefault() > DateTime.Now.Date);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430255",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Signature date must not be later than today",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:SignatureD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410451" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE90", Value = GetValueOrEmptyDate(declarationOfTrusteeOrPartner.IEE90) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410451

                    #region VR.ATO.IEE.410470

                    /*  VR.ATO.IEE.410470
                    Orqanisation Name cannot contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE300, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE300, ' - ')
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (declarationOfTrusteeOrPartner.IEE300 == null ? false : declarationOfTrusteeOrPartner.IEE300.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410470" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410470

                    #region VR.ATO.IEE.410471

                    /*  VR.ATO.IEE.410471
                    Organisation Name must not contain 'P/L'
    
                    Legacy Rule Format:
                    Contains(^IEE300, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^IEE300, 'P/L')
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (declarationOfTrusteeOrPartner.IEE300 == null ? false : declarationOfTrusteeOrPartner.IEE300.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410471" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410471

                    #region VR.ATO.IEE.410472

                    /*  VR.ATO.IEE.410472
                    Organisation Name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE300, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE300, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE300, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410472" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410472

                    #region VR.ATO.IEE.410473

                    /*  VR.ATO.IEE.410473
                    Non-individualname must not end with a leading space followed by 'T/A', 'T/A P'ship', 'T/A Pship', 'T/A P/Ship' or 'T/A Partnership'.
    
                    Legacy Rule Format:
                    EndsWithSet(^IEE300, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
        
                    Technical Business Rule Format:
                    EndsWithSet(^IEE300, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE300, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410473" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410473

                    #region VR.ATO.IEE.410474

                    /*  VR.ATO.IEE.410474
                    Non-individual Name cannot start with 'T/A' followed by a space and end with a space followed by 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    (StartsWith(^IEE300, 'T/A ')) AND (EndsWithSet(^IEE300, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^IEE300, 'T/A ')) AND (EndsWithSet(^IEE300, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (IsMatch(declarationOfTrusteeOrPartner.IEE300, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(declarationOfTrusteeOrPartner.IEE300, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410038",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410474" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410474

                    #region VR.ATO.IEE.410475

                    /*  VR.ATO.IEE.410475
                    Organisation name must contain at least one alpha or numeric character.
    
                    Legacy Rule Format:
                    (^IEE300  <> BLANK) AND (NotContainsSet(^IEE300, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^IEE300  <> BLANK) AND (NotContainsSet(^IEE300, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE300) != true && !(IsMatch(declarationOfTrusteeOrPartner.IEE300, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410475" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410475

                    #region VR.ATO.IEE.410476

                    /*  VR.ATO.IEE.410476
                    Family Name must not contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE303, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE303, ' - ')
            
                    Data Elements:
            
                    ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (declarationOfTrusteeOrPartner.IEE303 == null ? false : declarationOfTrusteeOrPartner.IEE303.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410476" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE303", Value = declarationOfTrusteeOrPartner.IEE303 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410476

                    #region VR.ATO.IEE.410477

                    /*  VR.ATO.IEE.410477
                    Family name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE303, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE303, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE303, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410477" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE303", Value = declarationOfTrusteeOrPartner.IEE303 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410477

                    #region VR.ATO.IEE.410478

                    /*  VR.ATO.IEE.410478
                    Family name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
                    Legacy Rule Format:
                    (FoundSet(^IEE303, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE303, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE303, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410478" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE303", Value = declarationOfTrusteeOrPartner.IEE303 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410478

                    #region VR.ATO.IEE.410479

                    /*  VR.ATO.IEE.410479
                    Family name must not contain MR, MRS, MISS, MS with space before and after.
    
                    Legacy Rule Format:
                    (FoundSet(^IEE303, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE303, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE303, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410479" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE303", Value = declarationOfTrusteeOrPartner.IEE303 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410479

                    #region VR.ATO.IEE.410480

                    /*  VR.ATO.IEE.410480
                    Family name must not contain a suffix
    
                    Legacy Rule Format:
                    (FoundSet(^IEE303, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE303, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE303, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410313",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410480" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE303", Value = declarationOfTrusteeOrPartner.IEE303 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410480

                    #region VR.ATO.IEE.410481

                    /*  VR.ATO.IEE.410481
                    Trustee FamilyNameand Non-individualname must not both be entered.  Only one name is allowed.
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ((^IEE303 <> NULL) AND (^IEE300 <> BLANK))
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ((^IEE303 <> NULL) AND (^IEE300 <> BLANK))
            
                    Data Elements:
            
                    ^IEE303 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
            
                    ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
                    */
                    assertion = ((report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)) && (declarationOfTrusteeOrPartner.IEE303 != null && string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE300) != true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410042",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Trustee Family Name and Non-individual Name must not both be present. Only one name is allowed",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410481" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE303", Value = declarationOfTrusteeOrPartner.IEE303 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410481

                    #region VR.ATO.IEE.410482

                    /*  VR.ATO.IEE.410482
                    Either Trustee FamilyName or Non-individualname must be present.
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ((^IEE1030 = NULL) AND (^IEE300 = BLANK))
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1019, ^IEE1019 <> NULL) AND ((^IEE1030 = NULL) AND (^IEE300 = BLANK))
            
                    Data Elements:
            
                    ^IEE300 = IEE:RP:DeclarationOfTrusteeOrPartner:OrganisationNameDetails.OrganisationalName.Text
            
                    ^IEE1019 = IEE:RP:DeclarationOfTrusteeOrPartner
            
                    ^IEE1030 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails
                    */
                    assertion = ((report.RP_DeclarationOfTrusteeOrPartnerCollection == null ? false : report.RP_DeclarationOfTrusteeOrPartnerCollection.Any(IEE1019Repeat => report.RP_DeclarationOfTrusteeOrPartnerCollectionExists != false)) && (declarationOfTrusteeOrPartner.RP_DeclarationOfTrusteeOrPartner_TrusteeOrPartnerIndividualNameDetailsCollectionExists == false && string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE300) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410328",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Either Trustee name or non-individual name must be present.",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410482" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE300", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE300) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410482

                    #region VR.ATO.IEE.410483

                    /*  VR.ATO.IEE.410483
                    Given Name must not contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE304, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE304, ' - ')
            
                    Data Elements:
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (declarationOfTrusteeOrPartner.IEE304 == null ? false : declarationOfTrusteeOrPartner.IEE304.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410483" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410483

                    #region VR.ATO.IEE.410484

                    /*  VR.ATO.IEE.410484
                    Given name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE304, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE304, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE304, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410484" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410484

                    #region VR.ATO.IEE.410485

                    /*  VR.ATO.IEE.410485
                    Given Name must not contain a suffix
    
                    Legacy Rule Format:
                    (FoundSet(^IEE304, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE304, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE304, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410314",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410485" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410485

                    #region VR.ATO.IEE.410486

                    /*  VR.ATO.IEE.410486
                    Given Name must not contain trustee reference
    
                    Legacy Rule Format:
                    (FoundSet(^IEE304, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE304, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE304, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410486" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410486

                    #region VR.ATO.IEE.410487

                    /*  VR.ATO.IEE.410487
                    Given Name must not contain a title
    
                    Legacy Rule Format:
                    (FoundSet(^IEE304, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE304, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE304, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410487" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410487

                    #region VR.ATO.IEE.410488

                    /*  VR.ATO.IEE.410488
                    Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^IEE304 <> BLANK) AND (NotContainsSet(^IEE304, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^IEE304 <> BLANK) AND (NotContainsSet(^IEE304, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE304) != true && !(IsMatch(declarationOfTrusteeOrPartner.IEE304, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500143",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character.",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410488" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410488

                    #region VR.ATO.IEE.410489

                    /*  VR.ATO.IEE.410489
                    Other Given Name must not contain 'space hyphen space' .
    
                    Legacy Rule Format:
                    Contains(^IEE305, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^IEE305, ' - ')
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (declarationOfTrusteeOrPartner.IEE305 == null ? false : declarationOfTrusteeOrPartner.IEE305.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410489" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410489

                    #region VR.ATO.IEE.410490

                    /*  VR.ATO.IEE.410490
                    Other given name must not contain repeated hyphen, apostrophe, or space characters.
    
                    Legacy Rule Format:
                    ContainsSet(^IEE305, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^IEE305, '"--","''","  "')
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE305, @"--|''|  ", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410490" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410490

                    #region VR.ATO.IEE.410491

                    /*  VR.ATO.IEE.410491
                    Other given name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    (^IEE305 <> BLANK) AND (NotContainsSet(^IEE305, '"A-Z","a-z"'))
        
                    Technical Business Rule Format:
                    (^IEE305 <> BLANK) AND (NotContainsSet(^IEE305, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE305) != true && !(IsMatch(declarationOfTrusteeOrPartner.IEE305, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410131",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Given Name must contain at least one alphabetical character",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410491" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410491

                    #region VR.ATO.IEE.410492

                    /*  VR.ATO.IEE.410492
                    Other Given Name must not contain a suffix
    
                    Legacy Rule Format:
                    (FoundSet(^IEE305, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE305, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE305, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410315",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410492" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410492

                    #region VR.ATO.IEE.410493

                    /*  VR.ATO.IEE.410493
                    Other given name must not contain 'Exec for', 'Rep for' or 'Trustee for'
    
                    Legacy Rule Format:
                    (FoundSet(^IEE305, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE305, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE305, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410493" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410493

                    #region VR.ATO.IEE.410494

                    /*  VR.ATO.IEE.410494
                    Other Given Name must not contain a title
    
                    Legacy Rule Format:
                    (FoundSet(^IEE305, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^IEE305, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(declarationOfTrusteeOrPartner.IEE305, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410494" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410494

                    #region VR.ATO.IEE.410495

                    /*  VR.ATO.IEE.410495
                    Firstname must be entered if OtherGivenNames is entered
    
                    Legacy Rule Format:
                    (^IEE1030 <> NULL) AND ((^IEE305 <> BLANK) AND (^IEE304 = BLANK))
        
                    Technical Business Rule Format:
                    (^IEE1030 <> NULL) AND ((^IEE305 <> BLANK) AND (^IEE304 = BLANK))
            
                    Data Elements:
            
                    ^IEE305 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.OtherGivenName.Text
            
                    ^IEE304 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails:PersonNameDetails.GivenName.Text
            
                    ^IEE1030 = IEE:RP:DeclarationOfTrusteeOrPartner:TrusteeOrPartnerIndividualNameDetails
                    */
                    assertion = (declarationOfTrusteeOrPartner.RP_DeclarationOfTrusteeOrPartner_TrusteeOrPartnerIndividualNameDetailsCollectionExists != false && (string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE305) != true && string.IsNullOrWhiteSpace(declarationOfTrusteeOrPartner.IEE304) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410218",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"FirstName must be entered if OtherGivenNames is entered",
                            Location = "/tns:IEE/tns:RP/tns:DeclarationOfTrusteeOrPartnerCollection/tns:DeclarationOfTrusteeOrPartner" + OccurrenceIndex(declarationOfTrusteeOrPartner.OccurrenceIndex) + "/tns:TrusteeOrPartnerIndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410495" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE305", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE305) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE304", Value = GetValueOrEmpty(declarationOfTrusteeOrPartner.IEE304) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410495
                }
            }

            #endregion // Foreach loop

            #region Repeating report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection
            if (report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection != null)
            {
                int itemIndex4 = 0;
                foreach (IEE2021.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetails controlOutsideAustraliaPartPeriodDetails in report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection)
                {
                    itemIndex4++;

                    #region VR.ATO.IEE.410458

                    /*  VR.ATO.IEE.410458
                    End Date must be after a Start date
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1012, ^IEE1012 <> NULL) AND (^IEE19 >= ^IEE20)
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1012, ^IEE1012 <> NULL) AND (^IEE19 >= ^IEE20)
            
                    Data Elements:
            
                    ^IEE19 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails:Period.Start.Date
            
                    ^IEE20 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails:Period.End.Date
            
                    ^IEE1012 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection.Any(IEE1012Repeat => report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionExists != false)) && controlOutsideAustraliaPartPeriodDetails.IEE19.GetValueOrDefault() >= controlOutsideAustraliaPartPeriodDetails.IEE20.GetValueOrDefault());
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410068",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"End Date must be after a Start date",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ControlOutsideAustraliaPartPeriodDetailsCollection/tns:ControlOutsideAustraliaPartPeriodDetails" + OccurrenceIndex(controlOutsideAustraliaPartPeriodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410458" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE19", Value = GetValueOrEmptyDate(controlOutsideAustraliaPartPeriodDetails.IEE19) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE20", Value = GetValueOrEmptyDate(controlOutsideAustraliaPartPeriodDetails.IEE20) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410458

                    #region VR.ATO.IEE.410464

                    /*  VR.ATO.IEE.410464
                    From Date for Central Mngmnt must not be before Election Cmcmnt or Day Specified for the election
    
                    Legacy Rule Format:
                    AnyOccurrence(^IEE1012, ^IEE1012 <> NULL) AND ((^IEE19 < ^IEE46) OR (^IEE19 < ^IEE49))
        
                    Technical Business Rule Format:
                    AnyOccurrence(^IEE1012, ^IEE1012 <> NULL) AND ((^IEE19 < ^IEE46) OR (^IEE19 < ^IEE49))
            
                    Data Elements:
            
                    ^IEE19 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails:Period.Start.Date
            
                    ^IEE46 = IEE:RP:InterposedEntityElectionRevocationDetails:ElectionDetails:Elections.DateSpecified.Date
            
                    ^IEE49 = IEE:RP:InterposedEntityElectionRevocationDetails:Elections.Commencement.Date
            
                    ^IEE1012 = IEE:RP:InterposedEntityElectionRevocationDetails:CentralManagementAndControlInformation:ControlOutsideAustraliaPartPeriodDetails
                    */
                    assertion = ((report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection == null ? false : report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection.Any(IEE1012Repeat => report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionExists != false)) && (controlOutsideAustraliaPartPeriodDetails.IEE19.GetValueOrDefault() < report.IEE46.GetValueOrDefault() || controlOutsideAustraliaPartPeriodDetails.IEE19.GetValueOrDefault() < report.IEE49.GetValueOrDefault()));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.IEE.410464",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"From Date for Central Management must not be before Election Commencement or Day Specified for the election",
                            Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:CentralManagementAndControlInformation/tns:ControlOutsideAustraliaPartPeriodDetailsCollection/tns:ControlOutsideAustraliaPartPeriodDetails" + OccurrenceIndex(controlOutsideAustraliaPartPeriodDetails.OccurrenceIndex) + "/tns:StartD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410464" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE19", Value = GetValueOrEmptyDate(controlOutsideAustraliaPartPeriodDetails.IEE19) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE46", Value = GetValueOrEmptyDate(report.IEE46) });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "IEE49", Value = GetValueOrEmptyDate(report.IEE49) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.IEE.410464
                }
            }

            #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
}
