using System;
using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using VaTS;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;
using Ato.EN.IntegrationServices.CodeGenerationPTR;
using DataContracts;

namespace Ato.CD.Inbound.PTR202402
{
    internal class CrossFormValidatorDISTBENTRT : ICrossFormValidator
    {
        private PTR2024 ParentDocument { get; }

        private DISTBENTRT2024 ChildDocument { get; }

        internal CrossFormValidatorDISTBENTRT(PTR2024 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (DISTBENTRT2024)childDocument.ConsumedReport;
        }

        public IEnumerable<ProcessMessageDocument> ValidateCrossFormRules()
        {
            var response = new List<ProcessMessageDocument>();


            VRATOPTR440035(response);
            VRATOPTR440036(response);
            VRATOPTR440037(response);
            VRATOPTRW00013(response);

            return response;
        }



        #region VR.ATO.PTR.440035

        /*  VR.ATO.PTR.440035
                    Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form

                Legacy Rule Format:
                    IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND RP:period.startDate <> [DISTBENTRT1]
                       RETURN VALIDATION MESSAGE
                    ENDIF

            Technical Business Rule Format:
                    CountDocument('DISTBENTRT') = 1 AND ^StartDate1 <> ^DISTBENTRT1

        Data Elements:
    
            ^StartDate1 = RP
    
            ^DISTBENTRT1 = DISTBENTRT1
            */

       
       private void VRATOPTR440035(List<ProcessMessageDocument> response)
        {
            bool assertion = (ParentDocument.RPExists == true) && (ChildDocument.DISTBENTRT1 != null) && (ChildDocument.DISTBENTRT1 != ParentDocument.RPStartDate);
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440035" } ,
                        new ProcessMessageParameter { Name = "RPStartDate", Value = ParentDocument.RPStartDate.HasValue ? ParentDocument.RPStartDate.Value.ToString("yyyy-MM-dd"):"Null" },
                        new ProcessMessageParameter { Name = "DISTBENTRT1", Value = ChildDocument.DISTBENTRT1.HasValue ?  ChildDocument.DISTBENTRT1.Value.ToString("yyyy-MM-dd"):"Null" },
                    }
                });           
            }
        }
        #endregion // VR.ATO.PTR.440035

        #region VR.ATO.PTR.440036

        /*  VR.ATO.PTR.440036
                    Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form

                Legacy Rule Format:
                    IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND RP:period.endDate <> [DISTBENTRT2]
                       RETURN VALIDATION MESSAGE
                    ENDIF

            Technical Business Rule Format:
                    CountDocument('DISTBENTRT') = 1 AND ^EndDate <> ^DISTBENTRT2

        Data Elements:
    
            ^EndDate = RP
    
            ^DISTBENTRT2 = DISTBENTRT2
            */
        private void VRATOPTR440036(List<ProcessMessageDocument> response)
        {
           bool assertion = (ParentDocument.RPExists == true) && (ChildDocument.DISTBENTRT2 != null) && (ChildDocument.DISTBENTRT2 != ParentDocument.RPEndDate);
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter{ Name = "RuleIdentifier", Value = "VR.ATO.PTR.440036" },
                        new ProcessMessageParameter { Name = "RPEndDate", Value = ParentDocument.RPEndDate.HasValue ? ParentDocument.RPEndDate.Value.ToString("yyyy-MM-dd"):"Null" },
                        new ProcessMessageParameter { Name = "DISTBENTRT1", Value =ChildDocument.DISTBENTRT1.HasValue ? ChildDocument.DISTBENTRT1.Value.ToString("yyyy-MM-dd") :"Null" }
                    }
                });            
            }
        }
        #endregion // VR.ATO.PTR.440036

        #region VR.ATO.PTR.440037

        /*  VR.ATO.PTR.440037
                    The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form

                Legacy Rule Format:
                    IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND RP:entity.identifier.TFN <> [DISTBENTRT3]
                       RETURN VALIDATION MESSAGE
                    ENDIF

            Technical Business Rule Format:
                    CountDocument('DISTBENTRT') = 1 AND ^TFN <> ^DISTBENTRT3

        Data Elements:
    
            ^DISTBENTRT3 = DISTBENTRT3
    
            ^TFN = RP
            */
        private void VRATOPTR440037(List<ProcessMessageDocument> response)
        {
           bool assertion = (ParentDocument.RPExists == true) && (ChildDocument.DISTBENTRT3 != null) && (ChildDocument.DISTBENTRT3 != ParentDocument.RPIdentifierTFN);      
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form",                  
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",

                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440037" },
                        new ProcessMessageParameter { Name = "RPIdentifierTFN", Value = ParentDocument.RPIdentifierTFN },
                        new ProcessMessageParameter { Name = "DISTBENTRT3", Value = ChildDocument.DISTBENTRT3 },
                    }
                });             
            }
        }


        #endregion // VR.ATO.PTR.440037

        #region VR.ATO.PTR.W00013
        private void VRATOPTRW00013(List<ProcessMessageDocument> response)
        {


            /*  VR.ATO.PTR.W00013
                        Distributions to Beneficiaries of Trust schedule is provided but Trust distributions - Income amounts or credits are not declared

                    Legacy Rule Format:
                        IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND [PTR66] = NULL AND [PTR70] = NULL AND [PTR322] = NULL
                           RETURN VALIDATION MESSAGE
                        ENDIF

                Technical Business Rule Format:
                        CountDocument('DISTBENTRT') = 1 AND ^PTR66 = NULL AND ^PTR70 = NULL AND ^PTR322 = NULL

            Data Elements:

                ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount

                ^PTR70 = RP.JAUS.NonPrim:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount

                ^PTR322 = RP.JAUS.NonPrim:Income.TrustDistributionFranked.Amount
            */
            bool assertion = (!ParentDocument.PTR66.HasValue) && (!ParentDocument.PTR70.HasValue) && (!ParentDocument.PTR322.HasValue);
            // = (@"DISTBENTRT") == 1 && report.PTR66 == null && report.PTR70 == null && report.PTR322 == null && report.PTR74 == null);
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.W00013",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure Trust distributions - Income amounts or credits from trusts income are declared",
                    LongDescription = @"Ensure Trust distributions - Income amounts or credits are declared when Distributions to Beneficiaries of Trust schedule is provided",
                    Location = "/xbrli:xbrl/tns:Income.TrustShareNet.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00013" },
                        new ProcessMessageParameter { Name = "PTR66", Value = ParentDocument.PTR66.HasValue ? ParentDocument.PTR66.GetValueOrDefault().ToString(): "null" },
                        new ProcessMessageParameter { Name = "PTR70", Value = ParentDocument.PTR70.HasValue ? ParentDocument.PTR70.GetValueOrDefault().ToString(): "null" },
                        new ProcessMessageParameter { Name = "PTR322", Value = ParentDocument.PTR322.HasValue ? ParentDocument.PTR322.GetValueOrDefault().ToString(): "null" }
                    }
                });
            }
        }
          
     #endregion
    }
}