using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationPTR
{

    public partial class PTR2024Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private PTR2024 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response {get; private set;}

        /// <summary>
        /// Initializes a new instance of the <see cref="PTR2024Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PTR2024Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PTR2024Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PTR2024Validator(PTR2024 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY"):this(maxParameterNameLength,maxParameterValueLength,emptyParameterValue)
        {                   
            this.ConsumedReport = reportIn;                    
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {                   
            return datetime.GetValueOrDefault().Date;
        }

				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }
        
        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public PTR2024 ConsumedReport { get {return report;} private set {report = value;}}

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();				

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(PTR2024 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------
                VRATOGEN000201();
                VRATOGEN000201_1();
                VRATOGEN000201_2();
                VRATOGEN000201_3();
                VRATOGEN000201_4();
                VRATOGEN000201_5();
                VRATOGEN000201_6();
                VRATOGEN000201_7();
                VRATOGEN000201_8();
                VRATOGEN000201_9();
                VRATOGEN000201_10();
                VRATOGEN000201_11();
                VRATOGEN000201_12();
                VRATOGEN000201_13();
                VRATOGEN000201_14();
                VRATOGEN000209();
                VRATOGEN000227();
                VRATOGEN000406();
                VRATOGEN000406_2();
                VRATOGEN000406_3();
                VRATOGEN000408();
                VRATOGEN000408_2();
                VRATOGEN000408_3();
                VRATOGEN000458_1();
                VRATOGEN000459_1();
                VRATOGEN001023();
                VRATOGEN001023_1();
                VRATOGEN001023_2();
                VRATOGEN001023_3();
                VRATOGEN001023_4();
                VRATOGEN001023_5();
                VRATOGEN001023_6();
                VRATOGEN001023_7();
                VRATOGEN001023_8();
                VRATOGEN001023_9();
                VRATOGEN001023_10();
                VRATOGEN001023_11();
                VRATOGEN001023_12();
                VRATOGEN001023_13();
                VRATOGEN001023_14();
                VRATOGEN010028();
                VRATOGEN010029();
                VRATOGEN010030();
                VRATOGEN408010();
                VRATOGEN408010_1();
                VRATOGEN408010_2();
                VRATOGEN410002();
                VRATOGEN410002_2();
                VRATOGEN410002_3();
                VRATOGEN410008();
                VRATOGEN410008_1();
                VRATOGEN410008_2();
                VRATOGEN410013();
                VRATOGEN410013_2();
                VRATOGEN410013_3();
                VRATOGEN410038();
                VRATOGEN410038_1();
                VRATOGEN410038_3();
                VRATOGEN410038_4();
                VRATOGEN410039();
                VRATOGEN410039_1();
                VRATOGEN410039_3();
                VRATOGEN410039_4();
                VRATOGEN410063_1();
                VRATOGEN410131_1();
                VRATOGEN410191();
                VRATOGEN410191_2();
                VRATOGEN410191_3();
                VRATOGEN410194();
                VRATOGEN410194_2();
                VRATOGEN410194_3();
                VRATOGEN410195();
                VRATOGEN410195_2();
                VRATOGEN410195_3();
                VRATOGEN410200_1();
                VRATOGEN410201_1();
                VRATOGEN410203_1();
                VRATOGEN410204_1();
                VRATOGEN410205();
                VRATOGEN410205_2();
                VRATOGEN410205_3();
                VRATOGEN410208_1();
                VRATOGEN410209_1();
                VRATOGEN410211();
                VRATOGEN410211_2();
                VRATOGEN410211_3();
                VRATOGEN410212();
                VRATOGEN410212_2();
                VRATOGEN410212_3();
                VRATOGEN410218_1();
                VRATOGEN428016();
                VRATOGEN428230();
                VRATOGEN428230_2();
                VRATOGEN428230_3();
                VRATOGEN428247();
                VRATOGEN428253();
                VRATOGEN428254();
                VRATOGEN428254_2();
                VRATOGEN428254_3();
                VRATOGEN428255();
                VRATOGEN428255_2();
                VRATOGEN428255_3();
                VRATOGEN428256();
                VRATOGEN428256_2();
                VRATOGEN428256_3();
                VRATOGEN428258();
                VRATOGEN428258_1();
                VRATOGEN428258_3();
                VRATOGEN428258_4();
                VRATOGEN428259();
                VRATOGEN428259_1();
                VRATOGEN428259_3();
                VRATOGEN428259_4();
                VRATOGEN428260();
                VRATOGEN428260_1();
                VRATOGEN428260_3();
                VRATOGEN428260_4();
                VRATOGEN428261();
                VRATOGEN428261_1();
                VRATOGEN428261_3();
                VRATOGEN428261_4();
                VRATOGEN428262_1();
                VRATOGEN428263_1();
                VRATOGEN428264_1();
                VRATOGEN428265_1();
                VRATOGEN428266_1();
                VRATOGEN428267_1();
                VRATOGEN430206();
                VRATOGEN430246();
                VRATOGEN430246_2();
                VRATOGEN430246_3();
                VRATOGEN430252();
                VRATOGEN430252_1();
                VRATOGEN430252_2();
                VRATOGEN430255();
                VRATOGEN430255_1();
                VRATOGEN430258();
                VRATOGEN430259();
                VRATOGEN430266();
                VRATOGEN430267();
                VRATOGEN430269();
                VRATOGEN430270();
                VRATOGEN430273();
                VRATOGEN430275();
                VRATOGEN430279();
                VRATOGEN430280();
                VRATOGEN430297();
                VRATOGEN430302();
                VRATOGEN430312();
                VRATOGEN430318();
                VRATOGEN432417();
                VRATOGEN434147();
                VRATOGEN434147_2();
                VRATOGEN434147_3();
                VRATOGEN434223();
                VRATOGEN434223_2();
                VRATOGEN438010();
                VRATOGEN438011();
                VRATOGEN438012();
                VRATOGEN438013();
                VRATOGEN438014();
                VRATOGEN438015();
                VRATOGEN438016();
                VRATOGEN438029();
                VRATOGEN438029_1();
                VRATOGEN438029_2();
                VRATOGEN438029_3();
                VRATOGEN438029_4();
                VRATOGEN438029_5();
                VRATOGEN438029_6();
                VRATOGEN438029_7();
                VRATOGEN438029_8();
                VRATOGEN438029_9();
                VRATOGEN438029_10();
                VRATOGEN438029_11();
                VRATOGEN438029_12();
                VRATOGEN438029_13();
                VRATOGEN438029_14();
                VRATOGEN438029_15();
                VRATOGEN438030();
                VRATOGEN438035();
                VRATOGEN438035_1();
                VRATOGEN438035_2();
                VRATOGEN438051();
                VRATOGEN438052();
                VRATOGEN438053();
                VRATOGEN438053_1();
                VRATOGEN438053_2();
                VRATOGEN438053_3();
                VRATOGEN438053_4();
                VRATOGEN438053_5();
                VRATOGEN438053_6();
                VRATOGEN438053_7();
                VRATOGEN438053_8();
                VRATOGEN438053_9();
                VRATOGEN438053_10();
                VRATOGEN438053_11();
                VRATOGEN438053_12();
                VRATOGEN438053_13();
                VRATOGEN438054();
                VRATOGEN438054_1();
                VRATOGEN438054_2();
                VRATOGEN438054_3();
                VRATOGEN438054_4();
                VRATOGEN438054_5();
                VRATOGEN438054_6();
                VRATOGEN438054_7();
                VRATOGEN438054_8();
                VRATOGEN438054_9();
                VRATOGEN438054_10();
                VRATOGEN438054_11();
                VRATOGEN438054_12();
                VRATOGEN438054_13();
                VRATOGEN438063();
                VRATOGEN438064();
                VRATOGEN438065();
                VRATOGEN438066();
                VRATOGEN438067();
                VRATOGEN438068();
                VRATOGEN500130();
                VRATOGEN500130_1();
                VRATOGEN500132();
                VRATOGEN500132_1();
                VRATOGEN500132_3();
                VRATOGEN500132_4();
                VRATOGEN500133();
                VRATOGEN500133_2();
                VRATOGEN500133_3();
                VRATOGEN500134();
                VRATOGEN500134_2();
                VRATOGEN500134_3();
                VRATOGEN500135_1();
                VRATOGEN500141();
                VRATOGEN500141_2();
                VRATOGEN500141_3();
                VRATOGEN500142_1();
                VRATOGEN500151_1();
                VRATOGEN500152_1();
                VRATOGEN500153_1();
                VRATOPTR430001();
                VRATOPTR430009();
                VRATOPTR430010();
                VRATOPTR430012();
                VRATOPTR430013();
                VRATOPTR430015();
                VRATOPTR430016();
                VRATOPTR430017();
                VRATOPTR430018();
                VRATOPTR430021();
                VRATOPTR430022();
                VRATOPTR430023();
                VRATOPTR430024();
                VRATOPTR430027();
                VRATOPTR430028();
                VRATOPTR430029();
                VRATOPTR430030();
                VRATOPTR430031();
                VRATOPTR430032();
                VRATOPTR430033();
                VRATOPTR430034();
                VRATOPTR430035();
                VRATOPTR430036();
                VRATOPTR430038();
                VRATOPTR430040();
                VRATOPTR430044();
                VRATOPTR430046();
                VRATOPTR430047();
                VRATOPTR430048();
                VRATOPTR430049();
                VRATOPTR430050();
                VRATOPTR430051();
                VRATOPTR430052();
                VRATOPTR430054();
                VRATOPTR430055();
                VRATOPTR430056();
                VRATOPTR430057();
                VRATOPTR430058();
                VRATOPTR430059();
                VRATOPTR430061();
                VRATOPTR430063();
                VRATOPTR430064();
                VRATOPTR430067();
                VRATOPTR430068();
                VRATOPTR430069();
                VRATOPTR430074();
                VRATOPTR430075();
                VRATOPTR430076();
                VRATOPTR430077();
                VRATOPTR430078();
                VRATOPTR430079();
                VRATOPTR430080();
                VRATOPTR430085();
                VRATOPTR430087();
                VRATOPTR430089();
                VRATOPTR430091();
                VRATOPTR430092();
                VRATOPTR430093();
                VRATOPTR430094();
                VRATOPTR430095();
                VRATOPTR430096();
                VRATOPTR430097();
                VRATOPTR430098();
                VRATOPTR430099();
                VRATOPTR430100();
                VRATOPTR430101();
                VRATOPTR430102();
                VRATOPTR430103();
                VRATOPTR430106();
                VRATOPTR430107();
                VRATOPTR430109();
                VRATOPTR430112();
                VRATOPTR430113();
                VRATOPTR430116();
                VRATOPTR430120();
                VRATOPTR430127();
                VRATOPTR430129();
                VRATOPTR430130();
                VRATOPTR430131();
                VRATOPTR430132();
                VRATOPTR430134();
                VRATOPTR430142();
                VRATOPTR430144();
                VRATOPTR430145();
                VRATOPTR430151();
                VRATOPTR430153();
                VRATOPTR430155();
                VRATOPTR430158();
                VRATOPTR430166();
                VRATOPTR430171();
                VRATOPTR430172();
                VRATOPTR430177();
                VRATOPTR430194();
                VRATOPTR430195();
                VRATOPTR430197();
                VRATOPTR430198();
                VRATOPTR430199();
                VRATOPTR430200();
                VRATOPTR430207();
                VRATOPTR430212();
                VRATOPTR430213();
                VRATOPTR430216();
                VRATOPTR430217();
                VRATOPTR430218();
                VRATOPTR430219();
                VRATOPTR430221();
                VRATOPTR430222();
                VRATOPTR430223();
                VRATOPTR430227();
                VRATOPTR430228();
                VRATOPTR430233();
                VRATOPTR430234();
                VRATOPTR430240();
                VRATOPTR430242();
                VRATOPTR430249();
                VRATOPTR430293();
                VRATOPTR430297();
                VRATOPTR430314();
                VRATOPTR430315();
                VRATOPTR430704();
                VRATOPTR430705();
                VRATOPTR430706();
                VRATOPTR430851();
                VRATOPTR430856();
                VRATOPTR430860();
                VRATOPTR430866();
                VRATOPTR430867();
                VRATOPTR430870();
                VRATOPTR430876();
                VRATOPTR430883();
                VRATOPTR430886();
                VRATOPTR430890();
                VRATOPTR430894();
                VRATOPTR430899();
                VRATOPTR430901();
                VRATOPTR430903();
                VRATOPTR430919();
                VRATOPTR430921();
                VRATOPTR430922();
                VRATOPTR430923();
                VRATOPTR430924();
                VRATOPTR430925();
                VRATOPTR430938();
                VRATOPTR430939();
                VRATOPTR430940();
                VRATOPTR430941();
                VRATOPTR430942();
                VRATOPTR430943();
                VRATOPTR430944();
                VRATOPTR430946();
                VRATOPTR430948();
                VRATOPTR430977();
                VRATOPTR430983();
                VRATOPTR430987();
                VRATOPTR430991();
                VRATOPTR431006();
                VRATOPTR431007();
                VRATOPTR431008();
                VRATOPTR431009();
                VRATOPTR431010();
                VRATOPTR431011();
                VRATOPTR431012();
                VRATOPTR431013();
                VRATOPTR431030();
                VRATOPTR431031();
                VRATOPTR431032();
                VRATOPTR431033();
                VRATOPTR431043();
                VRATOPTR431044();
                VRATOPTR432001();
                VRATOPTR438052();
                VRATOPTR438053();
                VRATOPTR438054();
                VRATOPTR438055();
                VRATOPTR438056();
                VRATOPTR438057();
                VRATOPTR438058();
                VRATOPTR438059();
                VRATOPTR438070();
                VRATOPTR438075();
                VRATOPTR438080();
                VRATOPTR438081();
                VRATOPTR438090();
                VRATOPTR438091();
                VRATOPTR438096();
                VRATOPTR438097();
                VRATOPTR438098();
                VRATOPTR438099();
                VRATOPTR438100();
                VRATOPTR438101();
                VRATOPTR438102();
                VRATOPTR438103();
                VRATOPTR438104();
                VRATOPTR438105();
                VRATOPTR438106();
                VRATOPTR438107();
                VRATOPTR439002();
                VRATOPTR439003();
                VRATOPTR439004();
                VRATOPTR439007();
                VRATOPTR439008();
                VRATOPTR439027();
                VRATOPTR439030();
                VRATOPTR440028();
                VRATOPTR440030();
                VRATOPTR440032();
                VRATOPTR440039();
                VRATOPTR440040();
                VRATOPTRW00001();
                VRATOPTRW00002();
                VRATOPTRW00003();
                VRATOPTRW00004();
                VRATOPTRW00005();
                VRATOPTRW00006();
                VRATOPTRW00007();
                VRATOPTRW00008();
                VRATOPTRW00009();
                VRATOPTRW00010();
                VRATOPTRW00011();
                VRATOPTRW00012();
                VRATOPTRW01014();

            #region Repeating report.StatementOfDistributionCollection  
            if (report.StatementOfDistributionCollection != null)
            {    
                int itemIndex1 = 0;
                foreach (PTR2024.StatementOfDistribution statementOfDistribution in report.StatementOfDistributionCollection)
                {
                    itemIndex1++;
                        VRATOGEN000406_1(statementOfDistribution, itemIndex1);
                        VRATOGEN000406_4(statementOfDistribution, itemIndex1);
                        VRATOGEN000408_1(statementOfDistribution, itemIndex1);
                        VRATOGEN000408_4(statementOfDistribution, itemIndex1);
                        VRATOGEN000458(statementOfDistribution, itemIndex1);
                        VRATOGEN000459(statementOfDistribution, itemIndex1);
                        VRATOGEN410002_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410002_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410008_3(statementOfDistribution, itemIndex1);
                        VRATOGEN410013_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410013_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410031(statementOfDistribution, itemIndex1);
                        VRATOGEN410038_2(statementOfDistribution, itemIndex1);
                        VRATOGEN410039_2(statementOfDistribution, itemIndex1);
                        VRATOGEN410063(statementOfDistribution, itemIndex1);
                        VRATOGEN410131(statementOfDistribution, itemIndex1);
                        VRATOGEN410191_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410191_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410194_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410194_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410195_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410195_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410200(statementOfDistribution, itemIndex1);
                        VRATOGEN410201(statementOfDistribution, itemIndex1);
                        VRATOGEN410203(statementOfDistribution, itemIndex1);
                        VRATOGEN410204(statementOfDistribution, itemIndex1);
                        VRATOGEN410205_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410205_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410208(statementOfDistribution, itemIndex1);
                        VRATOGEN410209(statementOfDistribution, itemIndex1);
                        VRATOGEN410211_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410211_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410212_1(statementOfDistribution, itemIndex1);
                        VRATOGEN410212_4(statementOfDistribution, itemIndex1);
                        VRATOGEN410218(statementOfDistribution, itemIndex1);
                        VRATOGEN428230_1(statementOfDistribution, itemIndex1);
                        VRATOGEN428230_4(statementOfDistribution, itemIndex1);
                        VRATOGEN428254_1(statementOfDistribution, itemIndex1);
                        VRATOGEN428254_4(statementOfDistribution, itemIndex1);
                        VRATOGEN428255_1(statementOfDistribution, itemIndex1);
                        VRATOGEN428255_4(statementOfDistribution, itemIndex1);
                        VRATOGEN428256_1(statementOfDistribution, itemIndex1);
                        VRATOGEN428256_4(statementOfDistribution, itemIndex1);
                        VRATOGEN428258_2(statementOfDistribution, itemIndex1);
                        VRATOGEN428259_2(statementOfDistribution, itemIndex1);
                        VRATOGEN428260_2(statementOfDistribution, itemIndex1);
                        VRATOGEN428261_2(statementOfDistribution, itemIndex1);
                        VRATOGEN428262(statementOfDistribution, itemIndex1);
                        VRATOGEN428263(statementOfDistribution, itemIndex1);
                        VRATOGEN428264(statementOfDistribution, itemIndex1);
                        VRATOGEN428265(statementOfDistribution, itemIndex1);
                        VRATOGEN428266(statementOfDistribution, itemIndex1);
                        VRATOGEN428267(statementOfDistribution, itemIndex1);
                        VRATOGEN430246_1(statementOfDistribution, itemIndex1);
                        VRATOGEN430246_4(statementOfDistribution, itemIndex1);
                        VRATOGEN434147_1(statementOfDistribution, itemIndex1);
                        VRATOGEN434147_4(statementOfDistribution, itemIndex1);
                        VRATOGEN434223_1(statementOfDistribution, itemIndex1);
                        VRATOGEN500132_2(statementOfDistribution, itemIndex1);
                        VRATOGEN500133_1(statementOfDistribution, itemIndex1);
                        VRATOGEN500133_4(statementOfDistribution, itemIndex1);
                        VRATOGEN500134_1(statementOfDistribution, itemIndex1);
                        VRATOGEN500134_4(statementOfDistribution, itemIndex1);
                        VRATOGEN500135(statementOfDistribution, itemIndex1);
                        VRATOGEN500141_1(statementOfDistribution, itemIndex1);
                        VRATOGEN500141_4(statementOfDistribution, itemIndex1);
                        VRATOGEN500142(statementOfDistribution, itemIndex1);
                        VRATOGEN500151(statementOfDistribution, itemIndex1);
                        VRATOGEN500152(statementOfDistribution, itemIndex1);
                        VRATOGEN500153(statementOfDistribution, itemIndex1);
                        VRATOPTR430136(statementOfDistribution, itemIndex1);
                        VRATOPTR430139(statementOfDistribution, itemIndex1);
                        VRATOPTR430141(statementOfDistribution, itemIndex1);
                        VRATOPTR430143(statementOfDistribution, itemIndex1);
                        VRATOPTR430146(statementOfDistribution, itemIndex1);
                        VRATOPTR430147(statementOfDistribution, itemIndex1);
                        VRATOPTR430148(statementOfDistribution, itemIndex1);
                        VRATOPTR430149(statementOfDistribution, itemIndex1);
                        VRATOPTR430150(statementOfDistribution, itemIndex1);
                        VRATOPTR430152(statementOfDistribution, itemIndex1);
                        VRATOPTR430154(statementOfDistribution, itemIndex1);
                        VRATOPTR430157(statementOfDistribution, itemIndex1);
                        VRATOPTR430161(statementOfDistribution, itemIndex1);
                        VRATOPTR430167(statementOfDistribution, itemIndex1);
                        VRATOPTR430235(statementOfDistribution, itemIndex1);
                        VRATOPTR430236(statementOfDistribution, itemIndex1);
                        VRATOPTR430280(statementOfDistribution, itemIndex1);
                        VRATOPTR430313(statementOfDistribution, itemIndex1);
                        VRATOPTR430711(statementOfDistribution, itemIndex1);
                        VRATOPTR431049(statementOfDistribution, itemIndex1);
                        VRATOPTR431050(statementOfDistribution, itemIndex1);
                        VRATOPTR431051(statementOfDistribution, itemIndex1);
                        VRATOPTR438036(statementOfDistribution, itemIndex1);
                        VRATOPTR438050(statementOfDistribution, itemIndex1);
                        VRATOPTR438051(statementOfDistribution, itemIndex1);
                        VRATOPTR438071(statementOfDistribution, itemIndex1);
                        VRATOPTR438092(statementOfDistribution, itemIndex1);
                        VRATOPTR438093(statementOfDistribution, itemIndex1);
                        VRATOPTR438094(statementOfDistribution, itemIndex1);
                        VRATOPTR438095(statementOfDistribution, itemIndex1);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.OtherAustralianIncomeCollection  
            if (report.OtherAustralianIncomeCollection != null)
            {    
                int itemIndex1 = 0;
                foreach (PTR2024.OtherAustralianIncome otherAustralianIncome in report.OtherAustralianIncomeCollection)
                {
                    itemIndex1++;
                        VRATOPTR430104(otherAustralianIncome, itemIndex1);
                        VRATOPTR430105(otherAustralianIncome, itemIndex1);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.OtherDeductionsCollection  
            if (report.OtherDeductionsCollection != null)
            {    
                int itemIndex1 = 0;
                foreach (PTR2024.OtherDeductions otherDeductions in report.OtherDeductionsCollection)
                {
                    itemIndex1++;
                        VRATOPTR430108(otherDeductions, itemIndex1);
                        VRATOPTR430110(otherDeductions, itemIndex1);
                        VRATOPTR430196(otherDeductions, itemIndex1);
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
				
			
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = INT
    
            ^EndDate = INT
            */
            assertion = ((report.INTExists == true) && (report.INTStartDate >= report.INTEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^EndDate = RP
            */
            assertion = ((report.RPExists == true) && (report.RPStartDate >= report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.GeneralPool
    
            ^EndDate = RP:RP.GeneralPool
            */
            assertion = ((report.RPGeneralPoolExists == true) && (report.RPGeneralPoolStartDate >= report.RPGeneralPoolEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.JAUS
    
            ^EndDate = RP:RP.JAUS
            */
            assertion = ((report.RPJAUSExists == true) && (report.RPJAUSStartDate >= report.RPJAUSEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.NonPrim
    
            ^EndDate = RP.JAUS:RP.JAUS.NonPrim
            */
            assertion = ((report.RPJAUSNonPrimExists == true) && (report.RPJAUSNonPrimStartDate >= report.RPJAUSNonPrimEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_5()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.Prim
    
            ^EndDate = RP.JAUS:RP.JAUS.Prim
            */
            assertion = ((report.RPJAUSPrimExists == true) && (report.RPJAUSPrimStartDate >= report.RPJAUSPrimEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_6()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.JFOREIGN
    
            ^EndDate = RP:RP.JFOREIGN
            */
            assertion = ((report.RPJFOREIGNExists == true) && (report.RPJFOREIGNStartDate >= report.RPJFOREIGNEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_7()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^EndDate = RP.JFOREIGN:RP.JFOREIGN.CFCLC
            */
            assertion = ((report.RPJFOREIGNCFCLCExists == true) && (report.RPJFOREIGNCFCLCStartDate >= report.RPJFOREIGNCFCLCEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCLCOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_8()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^EndDate = RP.JFOREIGN:RP.JFOREIGN.CFCUC
            */
            assertion = ((report.RPJFOREIGNCFCUCExists == true) && (report.RPJFOREIGNCFCUCStartDate >= report.RPJFOREIGNCFCUCEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCUCOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_9()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.LowCost
    
            ^EndDate = RP:RP.LowCost
            */
            assertion = ((report.RPLowCostExists == true) && (report.RPLowCostStartDate >= report.RPLowCostEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_10()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.NonPrim
    
            ^EndDate = RP:RP.NonPrim
            */
            assertion = ((report.RPNonPrimExists == true) && (report.RPNonPrimStartDate >= report.RPNonPrimEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_11()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Prim
    
            ^EndDate = RP:RP.Prim
            */
            assertion = ((report.RPPrimExists == true) && (report.RPPrimStartDate >= report.RPPrimEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_12()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Share.NonPrim
    
            ^EndDate = RP:RP.Share.NonPrim
            */
            assertion = ((report.RPShareNonPrimExists == true) && (report.RPShareNonPrimStartDate >= report.RPShareNonPrimEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPShareNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_13()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Share.Prim
    
            ^EndDate = RP:RP.Share.Prim
            */
            assertion = ((report.RPSharePrimExists == true) && (report.RPSharePrimStartDate >= report.RPSharePrimEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSharePrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000201_14()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.TOFA
    
            ^EndDate = RP:RP.TOFA
            */
            assertion = ((report.RPTOFAExists == true) && (report.RPTOFAStartDate >= report.RPTOFAEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000209
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000209()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000227
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000227()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000227
            Invalid context. The number of Intermediary contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(INT) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = INT
            */
            assertion = (Count(report.INTCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.436280", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Intermediary contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000227"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000406
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000406()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR117 <> BLANK) AND (StartsWithSet(^PTR117,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR117, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR117) != true && IsMatch(report.PTR117, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.PTR117, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000406
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000406_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.000406
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR151 <> BLANK) AND (StartsWithSet(^PTR151,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR151, '" C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR151) != true && IsMatch(statementOfDistribution.PTR151, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(statementOfDistribution.PTR151, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.000406
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000406_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR9 <> BLANK) AND (StartsWithSet(^PTR9,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR9, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR9) != true && IsMatch(report.PTR9, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.PTR9, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000406
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000406_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR14 <> BLANK) AND (StartsWithSet(^PTR14,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR14, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR14) != true && IsMatch(report.PTR14, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.PTR14, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000406
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000406_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.000406
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR332 <> BLANK) AND (StartsWithSet(^PTR332,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR332, '" C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR332) != true && IsMatch(statementOfDistribution.PTR332, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(statementOfDistribution.PTR332, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.000408
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000408()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR213 <> BLANK) AND (StartsWithSet(^PTR213,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR213, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR213 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR213) != true && (IsMatch(report.PTR213, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.PTR213, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000408
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000408_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.000408
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR152 <> BLANK) AND (StartsWithSet(^PTR152,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR152, '" C/- "," C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^PTR152 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR152) != true && (IsMatch(statementOfDistribution.PTR152, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(statementOfDistribution.PTR152, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                            Description = @"Care of' is not valid for address line 2",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.000408
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000408_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR211 <> BLANK) AND (StartsWithSet(^PTR211,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR211, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR211 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR211) != true && (IsMatch(report.PTR211, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.PTR211, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000408
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000408_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR212 <> BLANK) AND (StartsWithSet(^PTR212,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR212, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR212 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR212) != true && (IsMatch(report.PTR212, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.PTR212, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000408
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000408_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.000408
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR333 <> BLANK) AND (StartsWithSet(^PTR333,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR333, '" C/- "," C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^PTR333 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR333) != true && (IsMatch(statementOfDistribution.PTR333, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(statementOfDistribution.PTR333, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                            Description = @"Care of' is not valid for address line 2",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR333", Value = GetValueOrEmpty(statementOfDistribution.PTR333) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.000458
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000458(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.000458
                    Suffix is invalid
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR312 <> BLANK) AND (NotInDomain(^PTR312, 'SUFFIX CODES'))
            
                    Data Elements:
            
                    ^PTR312 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR312) != true && !(IsMatch(statementOfDistribution.PTR312, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000458", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suffix is invalid",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR312", Value = GetValueOrEmpty(statementOfDistribution.PTR312) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.000458
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000458_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000458
            Suffix is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR21 <> BLANK) AND (NotInDomain(^PTR21, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^PTR21 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR21) != true && !(IsMatch(report.PTR21, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR21", Value = GetValueOrEmpty(report.PTR21) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.000459
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000459(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.000459
                    Title is invalid
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR311 <> BLANK) AND (NotInDomain(^PTR311, 'TITLE CODES'))
            
                    Data Elements:
            
                    ^PTR311 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR311) != true && !(IsMatch(statementOfDistribution.PTR311, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000459", Severity = ProcessMessageSeverity.Error,
                            Description = @"Title is invalid",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR311", Value = GetValueOrEmpty(statementOfDistribution.PTR311) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.000459
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN000459_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.000459
            Title is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR19 <> BLANK) AND (NotInDomain(^PTR19, 'TITLE CODES'))
    
            Data Elements:
    
            ^PTR19 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR19) != true && !(IsMatch(report.PTR19, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459", Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR19", Value = GetValueOrEmpty(report.PTR19) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Closing(Instant)
    
            ^OtherTFN = RP:RP.Closing(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPClosingInstantExists != false && report.RPClosingInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.GeneralPool
    
            ^OtherTFN = RP:RP.GeneralPool
    
            ^TFN = RP
            */
            assertion = (report.RPGeneralPoolExists != false && report.RPGeneralPoolIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
    
            ^OtherTFN = RP:RP.JAUS
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.NonPrim
    
            ^OtherTFN = RP.JAUS:RP.JAUS.NonPrim
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSNonPrimExists != false && report.RPJAUSNonPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.Prim
    
            ^OtherTFN = RP.JAUS:RP.JAUS.Prim
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSPrimExists != false && report.RPJAUSPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_5()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
    
            ^OtherTFN = RP:RP.JFOREIGN
    
            ^TFN = RP
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_6()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^OtherTFN = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^TFN = RP
            */
            assertion = (report.RPJFOREIGNCFCLCExists != false && report.RPJFOREIGNCFCLCIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCLCOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_7()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^OtherTFN = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^TFN = RP
            */
            assertion = (report.RPJFOREIGNCFCUCExists != false && report.RPJFOREIGNCFCUCIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCUCOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_8()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.LowCost
    
            ^OtherTFN = RP:RP.LowCost
    
            ^TFN = RP
            */
            assertion = (report.RPLowCostExists != false && report.RPLowCostIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_9()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.NonPrim
    
            ^OtherTFN = RP:RP.NonPrim
    
            ^TFN = RP
            */
            assertion = (report.RPNonPrimExists != false && report.RPNonPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_10()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Opening(Instant)
    
            ^OtherTFN = RP:RP.Opening(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPOpeningInstantExists != false && report.RPOpeningInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_11()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Prim
    
            ^OtherTFN = RP:RP.Prim
    
            ^TFN = RP
            */
            assertion = (report.RPPrimExists != false && report.RPPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_12()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Share.NonPrim
    
            ^OtherTFN = RP:RP.Share.NonPrim
    
            ^TFN = RP
            */
            assertion = (report.RPShareNonPrimExists != false && report.RPShareNonPrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPShareNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_13()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Share.Prim
    
            ^OtherTFN = RP:RP.Share.Prim
    
            ^TFN = RP
            */
            assertion = (report.RPSharePrimExists != false && report.RPSharePrimIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSharePrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.001023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN001023_14()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
    
            ^OtherTFN = RP:RP.TOFA
    
            ^TFN = RP
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.010028
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN010028()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.010028
            If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.Amendment.Indicator = TRUE AND (pyin.xx.xx:Report.AmendmentSequence.Number = NULL OR pyin.xx.xx:Report.AmendmentType.Code = NULLORBLANK OR pyin.xx.xx:Report.AmendmentReason.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR316 = TRUE AND (^PTR319 = NULL OR ^PTR317 = BLANK OR ^PTR318 = BLANK)
    
            Data Elements:
    
            ^PTR316 = RP:Report.Amendment.Indicator
    
            ^PTR319 = RP:Report.AmendmentSequence.Number
    
            ^PTR317 = RP:Report.AmendmentType.Code
    
            ^PTR318 = RP:Report.AmendmentReason.Text
            */
            assertion = (report.PTR316 == true && (report.PTR319 == null || string.IsNullOrWhiteSpace(report.PTR317) == true || string.IsNullOrWhiteSpace(report.PTR318) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.010028"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR316", Value = GetValueOrEmpty(report.PTR316) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR319", Value = report.PTR319.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR317", Value = GetValueOrEmpty(report.PTR317) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR318", Value = GetValueOrEmpty(report.PTR318) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.010029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN010029()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.010029
            If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.Amendment.Indicator <> TRUE AND (pyin.xx.xx:Report.AmendmentSequence.Number <> NULL OR pyin.xx.xx:Report.AmendmentType.Code <> NULLORBLANK OR pyin.xx.xx:Report.AmendmentReason.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR316 <> TRUE AND (^PTR319 <> NULL OR ^PTR317 <> BLANK OR ^PTR318 <> BLANK)
    
            Data Elements:
    
            ^PTR316 = RP:Report.Amendment.Indicator
    
            ^PTR319 = RP:Report.AmendmentSequence.Number
    
            ^PTR317 = RP:Report.AmendmentType.Code
    
            ^PTR318 = RP:Report.AmendmentReason.Text
            */
            assertion = (report.PTR316 != true && (report.PTR319 != null || string.IsNullOrWhiteSpace(report.PTR317) != true || string.IsNullOrWhiteSpace(report.PTR318) != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.010029"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR316", Value = GetValueOrEmpty(report.PTR316) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR319", Value = report.PTR319.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR317", Value = GetValueOrEmpty(report.PTR317) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR318", Value = GetValueOrEmpty(report.PTR318) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.010030
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN010030()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.010030
            Amendment sequence number must be a number between 1 and 9
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.AmendmentSequence.Number <> NULL AND pyin.xx.xx:Report.AmendmentSequence.Number <> SET(1-9)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR319 <> NULL AND (^PTR319 < 1 OR ^PTR319 > 9)
    
            Data Elements:
    
            ^PTR319 = RP:Report.AmendmentSequence.Number
            */
            assertion = (report.PTR319 != null && (report.PTR319.GetValueOrDefault() < 1 || report.PTR319.GetValueOrDefault() > 9));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Amendment sequence number must be a number between 1 and 9",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentSequence.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.010030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR319", Value = report.PTR319.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.408010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN408010()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR242, ' - ')
    
            Data Elements:
    
            ^PTR242 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (report.PTR242 == null ? false : report.PTR242.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR242", Value = report.PTR242 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.408010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN408010_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR314, ' - ')
    
            Data Elements:
    
            ^PTR314 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.PTR314 == null ? false : report.PTR314.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR314", Value = report.PTR314 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.408010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN408010_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR264, ' - ')
    
            Data Elements:
    
            ^PTR264 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.PTR264 == null ? false : report.PTR264.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR264", Value = report.PTR264 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410002()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^PTR117, 'C/-')) AND (^PTR213 = BLANK))
    
            Data Elements:
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR213 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.PTR117 == null ? false : report.PTR117.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.PTR213) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410002_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410002
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ((Contains(^PTR151, 'C/-')) AND (^PTR152 = BLANK))
            
                    Data Elements:
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR152 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = ((statementOfDistribution.PTR151 == null ? false : statementOfDistribution.PTR151.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(statementOfDistribution.PTR152) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410002_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^PTR9, 'C/-')) AND (^PTR211 = BLANK))
    
            Data Elements:
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR211 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.PTR9 == null ? false : report.PTR9.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.PTR211) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410002_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^PTR14, 'C/-')) AND (^PTR212 = BLANK))
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR212 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = ((report.PTR14 == null ? false : report.PTR14.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.PTR212) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410002_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410002
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ((Contains(^PTR332, 'C/-')) AND (^PTR333 = BLANK))
            
                    Data Elements:
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR333 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = ((statementOfDistribution.PTR332 == null ? false : statementOfDistribution.PTR332.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(statementOfDistribution.PTR333) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR333", Value = GetValueOrEmpty(statementOfDistribution.PTR333) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410008()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR220 <> BLANK
    
            Data Elements:
    
            ^PTR220 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR220) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR220", Value = GetValueOrEmpty(report.PTR220) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410008_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR214 <> BLANK
    
            Data Elements:
    
            ^PTR214 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR214) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR214", Value = GetValueOrEmpty(report.PTR214) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410008_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR217 <> BLANK
    
            Data Elements:
    
            ^PTR217 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR217) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR217", Value = GetValueOrEmpty(report.PTR217) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410008_3(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410008
                    Address Line 3 must be left blank
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR339 <> BLANK
            
                    Data Elements:
            
                    ^PTR339 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR339) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 3 must be left blank",
                            LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR339", Value = GetValueOrEmpty(statementOfDistribution.PTR339) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410013()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR221 <> BLANK
    
            Data Elements:
    
            ^PTR221 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR221) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR221", Value = GetValueOrEmpty(report.PTR221) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410013_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410013
                    Address Line 4 must be left blank
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR239 <> BLANK
            
                    Data Elements:
            
                    ^PTR239 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR239) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 4 must be left blank",
                            LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR239", Value = GetValueOrEmpty(statementOfDistribution.PTR239) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410013_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR215 <> BLANK
    
            Data Elements:
    
            ^PTR215 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR215) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR215", Value = GetValueOrEmpty(report.PTR215) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410013_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR218 <> BLANK
    
            Data Elements:
    
            ^PTR218 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR218) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR218", Value = GetValueOrEmpty(report.PTR218) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410013_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410013
                    Address Line 4 must be left blank
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR340 <> BLANK
            
                    Data Elements:
            
                    ^PTR340 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR340) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 4 must be left blank",
                            LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR340", Value = GetValueOrEmpty(statementOfDistribution.PTR340) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410031
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410031(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410031
                    The Tax File Number has failed the Tax File Number algorithm check.
    
                    Legacy Rule Format:
                    IF (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier <> NULLORBLANK) AND (TFNALGORITHM (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier) = FALSE)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR157 <> BLANK) AND (FailsTFNAlgorithm(^PTR157))
            
                    Data Elements:
            
                    ^PTR157 = RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR157) != true && FailsTFNAlgorithm(statementOfDistribution.PTR157));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410031", Severity = ProcessMessageSeverity.Error,
                            Description = @"TFN must be a valid TFN",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Identifiers.TaxFileNumber.Identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410031"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410038()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^PTR116, 'T/A ')) AND (EndsWithSet(^PTR116, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.PTR116, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.PTR116, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410038_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^PTR24, 'T/A ')) AND (EndsWithSet(^PTR24, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.PTR24, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.PTR24, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410038_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410038
                    Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (StartsWith(^PTR149, 'T/A ')) AND (EndsWithSet(^PTR149, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = (IsMatch(statementOfDistribution.PTR149, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(statementOfDistribution.PTR149, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410038_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^PTR6, 'T/A ')) AND (EndsWithSet(^PTR6, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.PTR6, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.PTR6, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410038_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^PTR8, 'T/A ')) AND (EndsWithSet(^PTR8, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (IsMatch(report.PTR8, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.PTR8, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410039()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR116  <> BLANK) AND (NotContainsSet(^PTR116, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR116) != true && !(IsMatch(report.PTR116, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410039_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR24  <> BLANK) AND (NotContainsSet(^PTR24, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR24) != true && !(IsMatch(report.PTR24, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410039_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410039
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR149  <> BLANK) AND (NotContainsSet(^PTR149, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR149) != true && !(IsMatch(statementOfDistribution.PTR149, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410039_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR6  <> BLANK) AND (NotContainsSet(^PTR6, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR6) != true && !(IsMatch(report.PTR6, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410039_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR8  <> BLANK) AND (NotContainsSet(^PTR8, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR8) != true && !(IsMatch(report.PTR8, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410063
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410063(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410063
                    Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR147 <> BLANK) AND (NotContainsSet(^PTR147, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR147) != true && !(IsMatch(statementOfDistribution.PTR147, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410063
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410063_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410063
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR22 <> BLANK) AND (NotContainsSet(^PTR22, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR22) != true && !(IsMatch(report.PTR22, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410131
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410131(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410131
                    Other Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR148 <> BLANK) AND (NotContainsSet(^PTR148, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR148) != true && !(IsMatch(statementOfDistribution.PTR148, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Given Name must contain at least one alphabetical character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410131
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410131_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410131
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR23 <> BLANK) AND (NotContainsSet(^PTR23, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR23) != true && !(IsMatch(report.PTR23, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410191
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410191()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR222 = TRUE) AND (^PTR304 = NULL)
    
            Data Elements:
    
            ^PTR304 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR222 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR222 == true && report.PTR304 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR304", Value = GetValueOrEmpty(report.PTR304) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410191
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410191_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410191
                    Country code must be present for overseas address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR240 = TRUE) AND (^PTR308 = NULL)
            
                    Data Elements:
            
                    ^PTR308 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR240 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR240 == true && statementOfDistribution.PTR308 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                            Description = @"Country code must be present for overseas address",
                            LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR308", Value = GetValueOrEmpty(statementOfDistribution.PTR308) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410191
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410191_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR216 = TRUE) AND (^PTR302 = NULL)
    
            Data Elements:
    
            ^PTR302 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR216 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR216 == true && report.PTR302 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR302", Value = GetValueOrEmpty(report.PTR302) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410191
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410191_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR219 = TRUE) AND (^PTR303 = NULL)
    
            Data Elements:
    
            ^PTR303 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR219 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.PTR219 == true && report.PTR303 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR303", Value = GetValueOrEmpty(report.PTR303) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410191
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410191_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410191
                    Country code must be present for overseas address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR329 = TRUE) AND (^PTR337 = NULL)
            
                    Data Elements:
            
                    ^PTR337 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR329 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR329 == true && statementOfDistribution.PTR337 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                            Description = @"Country code must be present for overseas address",
                            LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR329", Value = GetValueOrEmpty(statementOfDistribution.PTR329) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR337", Value = GetValueOrEmpty(statementOfDistribution.PTR337) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410194
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410194()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR117) > 38
    
            Data Elements:
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.PTR117) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410194
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410194_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410194
                    Address Line 1 must not exceed 38 characters
    
                    Legacy Rule Format:
                    IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                         RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Length(^PTR151) > 38
            
                    Data Elements:
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (Length(statementOfDistribution.PTR151) > 38);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not exceed 38 characters",
                            LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410194
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410194_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR9) > 38
    
            Data Elements:
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.PTR9) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410194
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410194_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR14) > 38
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (Length(report.PTR14) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410194
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410194_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410194
                    Address Line 1 must not exceed 38 characters
    
                    Legacy Rule Format:
                    IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                         RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Length(^PTR332) > 38
            
                    Data Elements:
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (Length(statementOfDistribution.PTR332) > 38);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not exceed 38 characters",
                            LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410195
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410195()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR213) > 38
    
            Data Elements:
    
            ^PTR213 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.PTR213) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410195
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410195_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410195
                    Address Line 2 must not exceed 38 characters
    
                    Legacy Rule Format:
                    IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                         RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Length(^PTR152) > 38
            
                    Data Elements:
            
                    ^PTR152 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (Length(statementOfDistribution.PTR152) > 38);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must not exceed 38 characters",
                            LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410195
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410195_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR211) > 38
    
            Data Elements:
    
            ^PTR211 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.PTR211) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410195
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410195_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR212) > 38
    
            Data Elements:
    
            ^PTR212 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (Length(report.PTR212) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410195
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410195_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410195
                    Address Line 2 must not exceed 38 characters
    
                    Legacy Rule Format:
                    IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                         RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Length(^PTR333) > 38
            
                    Data Elements:
            
                    ^PTR333 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (Length(statementOfDistribution.PTR333) > 38);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must not exceed 38 characters",
                            LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR333", Value = GetValueOrEmpty(statementOfDistribution.PTR333) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410200
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410200(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410200
                    Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR146, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR146, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410200
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410200_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410200
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR20, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR20, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410201(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410201
                    Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR146, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR146, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410201
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410201_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410201
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR20, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR20, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410203
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410203(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410203
                    First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR147, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR147, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410203
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410203_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410203
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR22, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR22, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410204
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410204(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410204
                    First name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR147, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR147, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410204
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410204_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410204
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR22, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR22, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410205
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410205()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR117, '"AS ABOVE"'))
    
            Data Elements:
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR117, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410205
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410205_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410205
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR151, '"AS ABOVE"'))
            
                    Data Elements:
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR151, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not contain 'AS ABOVE'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410205
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410205_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR9, '"AS ABOVE"'))
    
            Data Elements:
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR9, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410205
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410205_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR14, '"AS ABOVE"'))
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = IsMatch(report.PTR14, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410205
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410205_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410205
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR332, '"AS ABOVE"'))
            
                    Data Elements:
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR332, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not contain 'AS ABOVE'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410208
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410208(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410208
                    Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR148, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR148, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410208
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410208_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410208
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR23, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR23, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410209
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410209(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410209
                    Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR148, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR148, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410209
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410209_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410209
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR23, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR23, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410211
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410211()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR304 = 'au') AND  (^PTR222 = TRUE)
    
            Data Elements:
    
            ^PTR304 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR222 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR304 == @"au" && report.PTR222 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR304", Value = GetValueOrEmpty(report.PTR304) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410211
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410211_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410211
                    Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR308 = 'au') AND  (^PTR240 = TRUE)
            
                    Data Elements:
            
                    ^PTR308 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR240 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR308 == @"au" && statementOfDistribution.PTR240 == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                            Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                            LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR308", Value = GetValueOrEmpty(statementOfDistribution.PTR308) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410211
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410211_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR302 = 'au') AND  (^PTR216 = TRUE)
    
            Data Elements:
    
            ^PTR302 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR216 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR302 == @"au" && report.PTR216 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR302", Value = GetValueOrEmpty(report.PTR302) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410211
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410211_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR303 = 'au') AND  (^PTR219 = TRUE)
    
            Data Elements:
    
            ^PTR303 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR219 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.PTR303 == @"au" && report.PTR219 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR303", Value = GetValueOrEmpty(report.PTR303) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410211
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410211_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410211
                    Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR337 = 'au') AND  (^PTR329 = TRUE)
            
                    Data Elements:
            
                    ^PTR337 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR329 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR337 == @"au" && statementOfDistribution.PTR329 == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                            Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                            LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR337", Value = GetValueOrEmpty(statementOfDistribution.PTR337) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR329", Value = GetValueOrEmpty(statementOfDistribution.PTR329) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410212
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410212()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR304 <> BLANK) AND (^PTR304 <> 'au') AND  (^PTR222 = FALSE)
    
            Data Elements:
    
            ^PTR304 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR222 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR304) != true && report.PTR304 != @"au" && report.PTR222 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR304", Value = GetValueOrEmpty(report.PTR304) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410212
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410212_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410212
                    If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR308 <> BLANK) AND (^PTR308 <> 'au') AND  (^PTR240 = FALSE)
            
                    Data Elements:
            
                    ^PTR308 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR240 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR308) != true && statementOfDistribution.PTR308 != @"au" && statementOfDistribution.PTR240 == false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                            Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                            LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR308", Value = GetValueOrEmpty(statementOfDistribution.PTR308) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410212
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410212_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR302 <> BLANK) AND (^PTR302 <> 'au') AND  (^PTR216 = FALSE)
    
            Data Elements:
    
            ^PTR302 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR216 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR302) != true && report.PTR302 != @"au" && report.PTR216 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR302", Value = GetValueOrEmpty(report.PTR302) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410212
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410212_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR303 <> BLANK) AND (^PTR303 <> 'au') AND  (^PTR219 = FALSE)
    
            Data Elements:
    
            ^PTR303 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR219 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR303) != true && report.PTR303 != @"au" && report.PTR219 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR303", Value = GetValueOrEmpty(report.PTR303) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.410212
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410212_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410212
                    If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR337 <> BLANK) AND (^PTR337 <> 'au') AND  (^PTR329 = FALSE)
            
                    Data Elements:
            
                    ^PTR337 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR329 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR337) != true && statementOfDistribution.PTR337 != @"au" && statementOfDistribution.PTR329 == false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                            Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                            LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR337", Value = GetValueOrEmpty(statementOfDistribution.PTR337) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR329", Value = GetValueOrEmpty(statementOfDistribution.PTR329) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410218
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410218(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.410218
                    FirstName must be entered if OtherGivenNames is entered
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails) 
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK) 
                        RETURN VALIDATION MESSAGE 
                     ENDIF
        
                    Technical Business Rule Format:
                    (^PersonNameDetails <> NULL) AND ((^PTR148 <> BLANK) AND (^PTR147 = BLANK))
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
            
                    ^PersonNameDetails = RP:StatementOfDistribution:PersonNameDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPExists != false && (string.IsNullOrWhiteSpace(statementOfDistribution.PTR148) != true && string.IsNullOrWhiteSpace(statementOfDistribution.PTR147) == true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410218", Severity = ProcessMessageSeverity.Error,
                            Description = @"FirstName must be entered if OtherGivenNames is entered",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410218"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.410218
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN410218_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.410218
            FirstName must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails) 
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK) 
                RETURN VALIDATION MESSAGE 
             ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND ((^PTR23 <> BLANK) AND (^PTR22 = BLANK))
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
    
            ^PersonNameDetails = RP:MailRecipient:PersonNameDetails
            */
            assertion = (report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPExists != false && (string.IsNullOrWhiteSpace(report.PTR23) != true && string.IsNullOrWhiteSpace(report.PTR22) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218", Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410218"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428016
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428016()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428016
            Tax File Number has failed the algorithm check.
    
            Legacy Rule Format:
            IF (RP:entity.identifier.TFN <> NULLORBLANK) AND (TFNALGORITHM (RP:entity.identifier.TFN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (FailsTFNAlgorithm(^TFN))
    
            Data Elements:
    
            ^TFN = RP
            */
            assertion = (string.IsNullOrWhiteSpace(report.RPIdentifierTFN) != true && FailsTFNAlgorithm(report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428016", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428016"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428230()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR118, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^PTR118 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR118, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR118", Value = report.PTR118 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428230_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428230
                    Suburb/Town/City cannot contain state code
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR153, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^PTR153 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR153, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain state code",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR153", Value = statementOfDistribution.PTR153 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428230_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR10, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^PTR10 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR10, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR10", Value = report.PTR10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428230_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR15, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^PTR15 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = IsMatch(report.PTR15, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR15", Value = report.PTR15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428230_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428230
                    Suburb/Town/City cannot contain state code
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR334, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^PTR334 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR334, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain state code",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR334", Value = statementOfDistribution.PTR334 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428247
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428247()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428247
            The Tax Agent number has failed the Tax Agent number algorithm check
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR2 <> BLANK) AND (FailsTANAlgorithm(^PTR2))
    
            Data Elements:
    
            ^PTR2 = INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR2) != true && FailsTANAlgorithm(report.PTR2));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR2", Value = GetValueOrEmpty(report.PTR2) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428253
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428253()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428253
            Invalid ABN
    
            Legacy Rule Format:
            IF (INT:entity.identifier.ABN <> NULLORBLANK) AND (ABNALGORITHM (INT:entity.identifier.ABN) = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^ABN <> BLANK) AND (FailsABNAlgorithm(^ABN))
    
            Data Elements:
    
            ^ABN = INT
            */
            assertion = (string.IsNullOrWhiteSpace(report.INTIdentifierABN) != true && FailsABNAlgorithm(report.INTIdentifierABN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428253"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428254
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428254()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR117, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR117 == null ? false : report.PTR117.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428254
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428254_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428254
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
                    RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR151, 'UNKNOWN')
            
                    Data Elements:
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR151 == null ? false : statementOfDistribution.PTR151.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428254
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428254_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR9, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR9 == null ? false : report.PTR9.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428254
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428254_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR14, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.PTR14 == null ? false : report.PTR14.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428254
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428254_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428254
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
                    RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR332, 'UNKNOWN')
            
                    Data Elements:
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR332 == null ? false : statementOfDistribution.PTR332.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428255()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR213, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR213 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR213 == null ? false : report.PTR213.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428255_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428255
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR152, 'UNKNOWN')
            
                    Data Elements:
            
                    ^PTR152 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR152 == null ? false : statementOfDistribution.PTR152.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428255_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR211, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR211 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR211 == null ? false : report.PTR211.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428255_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR212, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR212 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.PTR212 == null ? false : report.PTR212.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428255_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428255
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR333, 'UNKNOWN')
            
                    Data Elements:
            
                    ^PTR333 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR333 == null ? false : statementOfDistribution.PTR333.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR333", Value = GetValueOrEmpty(statementOfDistribution.PTR333) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428256
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428256()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR118, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR118 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR118 == null ? false : report.PTR118.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR118", Value = report.PTR118 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428256
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428256_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428256
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR153, 'UNKNOWN')
            
                    Data Elements:
            
                    ^PTR153 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR153 == null ? false : statementOfDistribution.PTR153.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR153", Value = statementOfDistribution.PTR153 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428256
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428256_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR10, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR10 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR10 == null ? false : report.PTR10.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR10", Value = report.PTR10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428256
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428256_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR15, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR15 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.PTR15 == null ? false : report.PTR15.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR15", Value = report.PTR15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428256
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428256_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428256
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR334, 'UNKNOWN')
            
                    Data Elements:
            
                    ^PTR334 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR334 == null ? false : statementOfDistribution.PTR334.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR334", Value = statementOfDistribution.PTR334 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428258()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR116, ' - ')
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.PTR116 == null ? false : report.PTR116.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428258_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR24, ' - ')
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.PTR24 == null ? false : report.PTR24.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428258_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428258
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR149, ' - ')
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR149 == null ? false : statementOfDistribution.PTR149.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428258_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR6, ' - ')
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.PTR6 == null ? false : report.PTR6.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428258_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR8, ' - ')
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (report.PTR8 == null ? false : report.PTR8.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428259()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR116, 'P/L')
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.PTR116 == null ? false : report.PTR116.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428259_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR24, 'P/L')
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.PTR24 == null ? false : report.PTR24.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428259_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428259
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR149, 'P/L')
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR149 == null ? false : statementOfDistribution.PTR149.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428259_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR6, 'P/L')
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.PTR6 == null ? false : report.PTR6.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428259_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR8, 'P/L')
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (report.PTR8 == null ? false : report.PTR8.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428260
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428260()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR116, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR116, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428260
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428260_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR24, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR24, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428260
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428260_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428260
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^PTR149, '"--","''","  "'))
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR149, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428260
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428260_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR6, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR6, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428260
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428260_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR8, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = IsMatch(report.PTR8, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428261()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^PTR116, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR116, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428261_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^PTR24, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR24, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428261_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428261
                    Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (EndsWithSet(^PTR149, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR149, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428261_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^PTR6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.PTR6, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428261_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^PTR8, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = IsMatch(report.PTR8, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428262
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428262(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428262
                    Family Name cannot contain "space hyphen space"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR146, ' - ')
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (statementOfDistribution.PTR146 == null ? false : statementOfDistribution.PTR146.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428262
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428262_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428262
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR20, ' - ')
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (report.PTR20 == null ? false : report.PTR20.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428263
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428263(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428263
                    Family name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^PTR146, '"--","''","  "'))
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR146, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428263
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428263_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428263
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR20, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR20, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428264
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428264(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428264
                    First name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR147, ' - ')
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (statementOfDistribution.PTR147 == null ? false : statementOfDistribution.PTR147.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428264
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428264_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428264
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR22, ' - ')
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (report.PTR22 == null ? false : report.PTR22.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428265
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428265(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428265
                    First name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^PTR147, '"--","''","  "'))
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR147, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428265
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428265_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428265
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR22, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR22, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428266
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428266(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428266
                    Other given name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^PTR148, ' - ')
            
                    Data Elements:
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (statementOfDistribution.PTR148 == null ? false : statementOfDistribution.PTR148.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428266
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428266_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428266
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^PTR23, ' - ')
    
            Data Elements:
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (report.PTR23 == null ? false : report.PTR23.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.428267
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428267(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.428267
                    Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^PTR148, '"--","''","  "'))
            
                    Data Elements:
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR148, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.428267
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN428267_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.428267
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^PTR23, '"--","''","  "'))
    
            Data Elements:
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR23, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430206
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430206()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430206
            Hours taken to prepare and complete form must be 1 to 999
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Report.CompletionHours.Number <> NULL) AND (pyin.xx.xx:Report.CompletionHours.Number <> SET (1-999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR169 <> NULL) AND ((^PTR169 < 1 OR ^PTR169 > 999))
    
            Data Elements:
    
            ^PTR169 = RP:Report.CompletionHours.Number
            */
            assertion = (report.PTR169 != null && (report.PTR169.GetValueOrDefault() < 1 || report.PTR169.GetValueOrDefault() > 999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430206", Severity = ProcessMessageSeverity.Error,
                    Description = @"Hours taken to prepare and complete form must be 1 to 999",
                    Location = "/xbrli:xbrl/tns:Report.CompletionHours.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430206"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR169", Value = report.PTR169.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430246
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430246()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR222 = FALSE) AND (ContainsSet(^PTR118, '0-9'))
    
            Data Elements:
    
            ^PTR118 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR222 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR222 == false && IsMatch(report.PTR118, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR118", Value = report.PTR118 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430246
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430246_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.430246
                    Suburb/Town/City cannot contain numeric characters
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR240 = FALSE) AND (ContainsSet(^PTR153, '0-9'))
            
                    Data Elements:
            
                    ^PTR153 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR240 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR240 == false && IsMatch(statementOfDistribution.PTR153, @"[0-9]",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain numeric characters",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR153", Value = statementOfDistribution.PTR153 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.430246
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430246_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR216 = FALSE) AND (ContainsSet(^PTR10, '0-9'))
    
            Data Elements:
    
            ^PTR10 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR216 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR216 == false && IsMatch(report.PTR10, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR10", Value = report.PTR10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430246
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430246_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR219 = FALSE) AND (ContainsSet(^PTR15, '0-9'))
    
            Data Elements:
    
            ^PTR15 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR219 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.PTR219 == false && IsMatch(report.PTR15, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR15", Value = report.PTR15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430246
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430246_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.430246
                    Suburb/Town/City cannot contain numeric characters
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR329 = FALSE) AND (ContainsSet(^PTR334, '0-9'))
            
                    Data Elements:
            
                    ^PTR334 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR329 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (statementOfDistribution.PTR329 == false && IsMatch(statementOfDistribution.PTR334, @"[0-9]",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain numeric characters",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR329", Value = GetValueOrEmpty(statementOfDistribution.PTR329) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR334", Value = statementOfDistribution.PTR334 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.430252
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430252()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR242) > 200
    
            Data Elements:
    
            ^PTR242 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (Length(report.PTR242) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR242", Value = report.PTR242 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430252
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430252_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR314) > 200
    
            Data Elements:
    
            ^PTR314 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (Length(report.PTR314) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR314", Value = report.PTR314 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430252
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430252_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR264) > 200
    
            Data Elements:
    
            ^PTR264 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (Length(report.PTR264) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR264", Value = report.PTR264 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430255()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430255
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR294 > Today()
    
            Data Elements:
    
            ^PTR294 = INT:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
            */
            assertion = (report.PTR294.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR294", Value = GetValueOrEmptyDate(report.PTR294) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430255
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430255_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430255
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR289 > Today()
    
            Data Elements:
    
            ^PTR289 = RP:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
            */
            assertion = (report.PTR289.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR289", Value = GetValueOrEmptyDate(report.PTR289) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430258
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430258()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430258
            Invalid context. The number of Reporting party: Primary Production contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Prim)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Prim
            */
            assertion = (Count(report.RPPrimCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430258", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Primary Production contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430258"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430259()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430259
            Invalid context. The number of Reporting party: Non-Primary Production contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.NonPrim)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.NonPrim
            */
            assertion = (Count(report.RPNonPrimCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430259", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Non-Primary Production contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430259"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430266
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430266()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430266
            Invalid context. The number of Reporting party: Australian contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.JAus)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
            */
            assertion = (Count(report.RPJAUSCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430266", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Australian contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430266"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430267
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430267()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430267
            Invalid context. The number of Reporting party: Foreign contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.JForeign)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (Count(report.RPJFOREIGNCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430267", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Foreign contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430267"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430269
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430269()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430269
            Invalid context. The number of Reporting party: Primary: Australian contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Prim.JAus)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.Prim
            */
            assertion = (Count(report.RPJAUSPrimCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430269", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Primary: Australian contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSPrimLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430269"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430270
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430270()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430270
            Invalid context. The number of Reporting party: Non-Primary: Australian contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.NonPrim.JAus)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.NonPrim
            */
            assertion = (Count(report.RPJAUSNonPrimCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430270", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Non-Primary: Australian contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSNonPrimLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430270"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430273
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430273()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430273
            Invalid context. The number of Reporting party: ListedCountry: Foreign contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.CFCLC.JForeign)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCLC
            */
            assertion = (Count(report.RPJFOREIGNCFCLCCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430273", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: ListedCountry: Foreign contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCLCLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430273"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430275
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430275()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430275
            Invalid context. The number of Reporting party: UnlistedCountry: Foreign contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.CFCUC.JForeign)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCUC
            */
            assertion = (Count(report.RPJFOREIGNCFCUCCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430275", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: UnlistedCountry: Foreign contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCUCLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430275"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430279
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430279()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430279
            Invalid context. The number of Reporting Party Instant: Opening contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Opening(Instant))) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Opening(Instant)
            */
            assertion = (Count(report.RPOpeningInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430279", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Instant: Opening contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430279"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430280
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430280()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430280
            Invalid context. The number of Reporting Party Instant: Opening contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Closing(Instant))) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Closing(Instant)
            */
            assertion = (Count(report.RPClosingInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430280", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Instant: Closing contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430280"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430297
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430297()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430297
            Invalid context. The number of Intermediary contexts must not exceed 1
    
            Legacy Rule Format:
            IF LENGTH(bafot.xx.xx:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text) > 3000
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^PTR315) > 3000
    
            Data Elements:
    
            ^PTR315 = RP:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
            */
            assertion = (Length(report.PTR315) > 3000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430297", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Other attachments schedule must not exceed 3000 characters",
                    LongDescription = @"The 'Other attachments' schedule is represented by the 'Regulatory Disclosures. General Information About Financial Statements.Text' element",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430297"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR315", Value = GetValueOrEmpty(report.PTR315) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430302
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430302()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430302
            Invalid context. The number of Reporting party: TOFA contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TOFA)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
            */
            assertion = (Count(report.RPTOFACount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430302", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: TOFA contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFALastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430302"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430312
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430312()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430312
            If an Intermediary context exists, then a Tax Agent (Intermediary) contact telephone must be present
    
            Legacy Rule Format:
            IF CONTEXT(INT) <> NULL AND INT:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^PTR173 = NULL
    
            Data Elements:
    
            ^PTR173 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Minimal.Number WHERE [ElectronicContact.Telephone.Usage.Code='03']
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && report.PTR173 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430312", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Agent (Intermediary) contact telephone must be present",
                    Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Minimal.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430312"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR173", Value = report.PTR173 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.430318
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN430318()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.430318
            ABN in the context definition must pass the ABN algorithm check
    
            Legacy Rule Format:
            IF (entity.identifier.ABN <> NULLORBLANK) AND (ABNALGORITHM(entity.identifier.ABN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^ABN <> BLANK) AND (FailsABNAlgorithm(^ABN))
    
            Data Elements:
    
            ^ABN = INT
            */
            assertion = (string.IsNullOrWhiteSpace(report.INTIdentifierABN) != true && FailsABNAlgorithm(report.INTIdentifierABN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430318", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN in context must be a valid ABN",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430318"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.432417
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN432417()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.432417
            ABN for reporting party must be provided when there is no intermediary
    
            Legacy Rule Format:
            IF CONTEXT(INT) = NULL AND RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier = NULLORBLANK
                    RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context = NULL AND ^PTR7 = BLANK
    
            Data Elements:
    
            ^PTR7 = RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^Context = INT
            */
            assertion = (report.INTExists == false && string.IsNullOrWhiteSpace(report.PTR7) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432417", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN for reporting party must be provided when there is no intermediary",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.432417"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR7", Value = GetValueOrEmpty(report.PTR7) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.434147
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434147()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.434147
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR222 = FALSE AND (^PTR119 = NULL OR ^PTR120 = NULL))
    
            Data Elements:
    
            ^PTR222 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR119 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR120 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && (report.PTR222 == false && (report.PTR119 == null || report.PTR120 == null)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR119", Value = GetValueOrEmpty(report.PTR119) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR120", Value = GetValueOrEmpty(report.PTR120) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.434147
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434147_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.434147
                    State and post code must be present for an Australian address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text = NULL)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^PTR240 = FALSE AND (^PTR154 = NULL OR ^PTR155 = NULL))
            
                    Data Elements:
            
                    ^PTR240 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR154 = RP:StatementOfDistribution:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR155 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:StatementOfDistribution:AddressDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPExists != false && (statementOfDistribution.PTR240 == false && (statementOfDistribution.PTR154 == null || statementOfDistribution.PTR155 == null)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                            Description = @"State and post code must be present for an Australian address",
                            LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR154", Value = GetValueOrEmpty(statementOfDistribution.PTR154) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR155", Value = GetValueOrEmpty(statementOfDistribution.PTR155) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.434147
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434147_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.434147
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR216 = FALSE AND (^PTR11 = NULL OR ^PTR12 = NULL))
    
            Data Elements:
    
            ^PTR216 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR11 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR12 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && (report.PTR216 == false && (report.PTR11 == null || report.PTR12 == null)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR11", Value = GetValueOrEmpty(report.PTR11) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR12", Value = GetValueOrEmpty(report.PTR12) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.434147
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434147_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.434147
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR219 = FALSE AND (^PTR16 = NULL OR ^PTR17 = NULL))
    
            Data Elements:
    
            ^PTR219 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR16 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR17 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_P_RPExists != false && (report.PTR219 == false && (report.PTR16 == null || report.PTR17 == null)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR16", Value = GetValueOrEmpty(report.PTR16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR17", Value = GetValueOrEmpty(report.PTR17) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.434147
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434147_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.434147
                    State and post code must be present for an Australian address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text = NULL)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^PTR329 = FALSE AND (^PTR336 = NULL OR ^PTR335 = NULL))
            
                    Data Elements:
            
                    ^PTR329 = RP:StatementOfDistribution:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR336 = RP:StatementOfDistribution:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR335 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:StatementOfDistribution:AddressDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPExists != false && (statementOfDistribution.PTR329 == false && (statementOfDistribution.PTR336 == null || statementOfDistribution.PTR335 == null)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                            Description = @"State and post code must be present for an Australian address",
                            LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR329", Value = GetValueOrEmpty(statementOfDistribution.PTR329) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR336", Value = GetValueOrEmpty(statementOfDistribution.PTR336) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR335", Value = GetValueOrEmpty(statementOfDistribution.PTR335) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.434223
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434223()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.434223
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR25 <> BLANK) and (FailsABNAlgorithm(^PTR25))
    
            Data Elements:
    
            ^PTR25 = RP:MailRecipient:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR25) != true && FailsABNAlgorithm(report.PTR25));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR25", Value = GetValueOrEmpty(report.PTR25) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.434223
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434223_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.434223
                    ABN must pass the ABN algorithm
    
                    Legacy Rule Format:
                    IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier) = FALSE)
                    RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR285 <> BLANK) and (FailsABNAlgorithm(^PTR285))
            
                    Data Elements:
            
                    ^PTR285 = RP:StatementOfDistribution:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR285) != true && FailsABNAlgorithm(statementOfDistribution.PTR285));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                            Description = @"ABN is invalid",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Identifiers.AustralianBusinessNumber.Identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.434223
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN434223_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.434223
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR7 <> BLANK) and (FailsABNAlgorithm(^PTR7))
    
            Data Elements:
    
            ^PTR7 = RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR7) != true && FailsABNAlgorithm(report.PTR7));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR7", Value = GetValueOrEmpty(report.PTR7) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438010()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438010
            If the income includes personal services income (PSI) then total amount of income and deductions, results test satisfied and business determination indicators must all be completed
    
            Legacy Rule Format:
            IF (RP:bafpr1.xx.xx:Income.PersonalServicesIncomeIncluded.Indicator = TRUE) AND ((RP:bafpr3.xx.xx:Income.PersonalServicesIncome.Total.Amount = NULL) OR (RP:bafpr3.xx.xx:Expense.PersonalServicesIncome.Deduction.Amount = NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator = NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator = NULL))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR110 = TRUE) AND ((^PTR265 = NULL) OR (^PTR266 = NULL) OR (^PTR268 = NULL) OR (^PTR271 = NULL))
    
            Data Elements:
    
            ^PTR110 = RP:Income.PersonalServicesIncomeIncluded.Indicator
    
            ^PTR265 = RP:Income.PersonalServicesIncome.Total.Amount
    
            ^PTR266 = RP:Expense.PersonalServicesIncome.Deduction.Amount
    
            ^PTR268 = RP:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
    
            ^PTR271 = RP:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator
            */
            assertion = (report.PTR110 == true && (report.PTR265 == null || report.PTR266 == null || report.PTR268 == null || report.PTR271 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal services income section completed incorrectly",
                    LongDescription = @"If the income includes personal services income (PSI) then total amount of income and deductions, results test satisfied and business determination indicators must all be completed",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncomeIncluded.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR110", Value = GetValueOrEmpty(report.PTR110) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR265", Value = GetValueOrEmpty(report.PTR265) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR266", Value = GetValueOrEmpty(report.PTR266) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR268", Value = GetValueOrEmpty(report.PTR268) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR271", Value = GetValueOrEmpty(report.PTR271) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438011
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438011()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438011
            If the income does not include personal services income (PSI) then total amount of income and deductions, results test satisfied , business determination,  unrelated clients test, employment test and business premises test indicators must not be completed
    
            Legacy Rule Format:
            IF (RP:bafpr1.xx.xx:Income.PersonalServicesIncomeIncluded.Indicator = FALSE) AND ((RP:bafpr3.xx.xx:Income.PersonalServicesIncome.Total.Amount <> NULL) OR (RP:bafpr3.xx.xx:Expense.PersonalServicesIncome.Deduction.Amount <> NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator <> NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator <> NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator <> NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator <> NULL) OR (RP:rvctc2.xx.xx:IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator <> NULL))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR110 = FALSE) AND ((^PTR265 <> NULL) OR (^PTR266 <> NULL) OR (^PTR268 <> NULL) OR (^PTR271 <> NULL) OR (^PTR276 <> NULL) OR (^PTR272 <> NULL) OR (^PTR275 <> NULL))
    
            Data Elements:
    
            ^PTR110 = RP:Income.PersonalServicesIncomeIncluded.Indicator
    
            ^PTR265 = RP:Income.PersonalServicesIncome.Total.Amount
    
            ^PTR266 = RP:Expense.PersonalServicesIncome.Deduction.Amount
    
            ^PTR268 = RP:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
    
            ^PTR271 = RP:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator
    
            ^PTR276 = RP:IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator
    
            ^PTR272 = RP:IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator
    
            ^PTR275 = RP:IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator
            */
            assertion = (report.PTR110 == false && (report.PTR265 != null || report.PTR266 != null || report.PTR268 != null || report.PTR271 != null || report.PTR276 != null || report.PTR272 != null || report.PTR275 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal services income section completed incorrectly",
                    LongDescription = @"If the income does not include personal services income (PSI) then total amount of income and deductions,results test satisfied,business determination,unrelated clients test,employment test and business premises test indicators must not be completed",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncomeIncluded.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438011"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR110", Value = GetValueOrEmpty(report.PTR110) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR265", Value = GetValueOrEmpty(report.PTR265) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR266", Value = GetValueOrEmpty(report.PTR266) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR268", Value = GetValueOrEmpty(report.PTR268) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR271", Value = GetValueOrEmpty(report.PTR271) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR276", Value = GetValueOrEmpty(report.PTR276) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR272", Value = GetValueOrEmpty(report.PTR272) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR275", Value = GetValueOrEmpty(report.PTR275) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438012
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438012()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438012
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF (bafpr3.xx.xx:Income.PersonalServicesIncome.Total.Amount <> NULL) AND (bafpr3.xx.xx:Income.PersonalServicesIncome.Total.Amount <> MONETARY(U,11,0))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR265, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR265 = RP:Income.PersonalServicesIncome.Total.Amount
            */
            assertion = NotMonetary(report.PTR265, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncome.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438012"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR265", Value = GetValueOrEmpty(report.PTR265) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438013()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438013
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF (bafpr3.xx.xx:Expense.PersonalServicesIncome.Deduction.Amount <> NULL) AND (bafpr3.xx.xx:Expense.PersonalServicesIncome.Deduction.Amount <> MONETARY(U,11,0))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR266, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR266 = RP:Expense.PersonalServicesIncome.Deduction.Amount
            */
            assertion = NotMonetary(report.PTR266, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.PersonalServicesIncome.Deduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR266", Value = GetValueOrEmpty(report.PTR266) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438014
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438014()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438014
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF (bafpo7.xx.xx:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount <> NULL) AND (bafpo7.xx.xx:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount <> MONETARY(U,11,0))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR279, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR279 = RP:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount
            */
            assertion = NotMonetary(report.PTR279, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438014"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR279", Value = GetValueOrEmpty(report.PTR279) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438015
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438015()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438015
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF (bafpr3.xx.xx:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount <> NULL) AND (bafpr3.xx.xx:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount <> MONETARY(U,11,0))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR282, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR282 = RP:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount
            */
            assertion = NotMonetary(report.PTR282, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438015"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR282", Value = GetValueOrEmpty(report.PTR282) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438016
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438016()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438016
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF (bafpr3.xx.xx:Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount <> NULL) AND (bafpr3.xx.xx:Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount <> MONETARY(U,11,0))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR281, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR281 = RP:Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount
            */
            assertion = NotMonetary(report.PTR281, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438016"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR281", Value = GetValueOrEmpty(report.PTR281) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Closing(Instant)
    
            ^Context = RP:RP.Closing(Instant)
            */
            assertion = (report.RPClosingInstantExists != false && report.RPClosingInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.GeneralPool
    
            ^Context = RP:RP.GeneralPool
            */
            assertion = (report.RPGeneralPoolExists != false && report.RPGeneralPoolIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.JAUS
    
            ^Context = RP:RP.JAUS
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.NonPrim
    
            ^Context = RP.JAUS:RP.JAUS.NonPrim
            */
            assertion = (report.RPJAUSNonPrimExists != false && report.RPJAUSNonPrimIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_5()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.Prim
    
            ^Context = RP.JAUS:RP.JAUS.Prim
            */
            assertion = (report.RPJAUSPrimExists != false && report.RPJAUSPrimIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_6()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.JFOREIGN
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_7()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCLC
            */
            assertion = (report.RPJFOREIGNCFCLCExists != false && report.RPJFOREIGNCFCLCIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCLCOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_8()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCUC
            */
            assertion = (report.RPJFOREIGNCFCUCExists != false && report.RPJFOREIGNCFCUCIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCUCOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_9()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.LowCost
    
            ^Context = RP:RP.LowCost
            */
            assertion = (report.RPLowCostExists != false && report.RPLowCostIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_10()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.NonPrim
    
            ^Context = RP:RP.NonPrim
            */
            assertion = (report.RPNonPrimExists != false && report.RPNonPrimIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_11()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Opening(Instant)
    
            ^Context = RP:RP.Opening(Instant)
            */
            assertion = (report.RPOpeningInstantExists != false && report.RPOpeningInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_12()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Prim
    
            ^Context = RP:RP.Prim
            */
            assertion = (report.RPPrimExists != false && report.RPPrimIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_13()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Share.NonPrim
    
            ^Context = RP:RP.Share.NonPrim
            */
            assertion = (report.RPShareNonPrimExists != false && report.RPShareNonPrimIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPShareNonPrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_14()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Share.Prim
    
            ^Context = RP:RP.Share.Prim
            */
            assertion = (report.RPSharePrimExists != false && report.RPSharePrimIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSharePrimOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438029_15()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TOFA
    
            ^Context = RP:RP.TOFA
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438030
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438030()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438030
            Entity identifier scheme for Intermediary must be "http://www.ato.gov.au/abn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = “RprtPyType.xx.xx:Intermediary”) AND (entity.identifier.scheme <> “http://www.ato.gov.au/abn”)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^IdentifierScheme <> 'http://www.ato.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = INT
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && report.INTIdentifierScheme != @"http://www.ato.gov.au/abn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Intermediary must be ""http://www.ato.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438030"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438035
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438035()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^PTR242, '",","*"')
    
            Data Elements:
    
            ^PTR242 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = IsMatch(report.PTR242, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR242", Value = report.PTR242 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438035
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438035_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^PTR314, '",","*"')
    
            Data Elements:
    
            ^PTR314 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.PTR314, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR314", Value = report.PTR314 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438035
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438035_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^PTR264, '",","*"')
    
            Data Elements:
    
            ^PTR264 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.PTR264, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR264", Value = report.PTR264 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438051
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438051()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438051
            Date of opening balance must match with the start date of Reporting period
    
            Legacy Rule Format:
            IF RP.Opening(Instant):period.instantDate <> RP:period.startDate
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^InstantDate1 <> ^StartDate1
    
            Data Elements:
    
            ^StartDate1 = RP
    
            ^InstantDate1 = RP:RP.Opening(Instant)
            */
            assertion = ((report.RPOpeningInstantExists == true) && (report.RPOpeningInstantInstantDate != report.RPStartDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438051", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of opening balance must match with the start date of Reporting period",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438051"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438052
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438052()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438052
            Date of closing balance must match with the end date of Reporting period
    
            Legacy Rule Format:
            IF RP.Closing(Instant):period.instantDate <> RP:period.endDate
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^InstantDate1 <> ^EndDate
    
            Data Elements:
    
            ^EndDate = RP
    
            ^InstantDate1 = RP:RP.Closing(Instant)
            */
            assertion = ((report.RPClosingInstantExists == true) && (report.RPClosingInstantInstantDate != report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438052", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of closing balance must match with the end date of Reporting period",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438052"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = INT
    
            ^StartDate1 = RP
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && report.INTStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.GeneralPool
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.GeneralPool
            */
            assertion = (report.RPGeneralPoolExists != false && report.RPGeneralPoolStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.JAUS
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.JAUS
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.NonPrim
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.NonPrim
            */
            assertion = (report.RPJAUSNonPrimExists != false && report.RPJAUSNonPrimStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.Prim
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.Prim
            */
            assertion = (report.RPJAUSPrimExists != false && report.RPJAUSPrimStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_5()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.JFOREIGN
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_6()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^StartDate1 = RP
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCLC
            */
            assertion = (report.RPJFOREIGNCFCLCExists != false && report.RPJFOREIGNCFCLCStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCLCOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_7()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^StartDate1 = RP
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCUC
            */
            assertion = (report.RPJFOREIGNCFCUCExists != false && report.RPJFOREIGNCFCUCStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCUCOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_8()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.LowCost
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.LowCost
            */
            assertion = (report.RPLowCostExists != false && report.RPLowCostStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_9()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.NonPrim
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.NonPrim
            */
            assertion = (report.RPNonPrimExists != false && report.RPNonPrimStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_10()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Prim
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Prim
            */
            assertion = (report.RPPrimExists != false && report.RPPrimStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_11()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Share.NonPrim
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Share.NonPrim
            */
            assertion = (report.RPShareNonPrimExists != false && report.RPShareNonPrimStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPShareNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_12()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Share.Prim
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Share.Prim
            */
            assertion = (report.RPSharePrimExists != false && report.RPSharePrimStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSharePrimOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438053_13()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.TOFA
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.TOFA
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = INT
    
            ^EndDate1 = INT
    
            ^EndDate = RP
            */
            assertion = (report.INTExists != false && report.INTEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.GeneralPool
    
            ^EndDate1 = RP:RP.GeneralPool
    
            ^EndDate = RP
            */
            assertion = (report.RPGeneralPoolExists != false && report.RPGeneralPoolEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
    
            ^EndDate1 = RP:RP.JAUS
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.NonPrim
    
            ^EndDate1 = RP.JAUS:RP.JAUS.NonPrim
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSNonPrimExists != false && report.RPJAUSNonPrimEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.Prim
    
            ^EndDate1 = RP.JAUS:RP.JAUS.Prim
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSPrimExists != false && report.RPJAUSPrimEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_5()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
    
            ^EndDate1 = RP:RP.JFOREIGN
    
            ^EndDate = RP
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_6()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^EndDate1 = RP.JFOREIGN:RP.JFOREIGN.CFCLC
    
            ^EndDate = RP
            */
            assertion = (report.RPJFOREIGNCFCLCExists != false && report.RPJFOREIGNCFCLCEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCLCOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_7()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^EndDate1 = RP.JFOREIGN:RP.JFOREIGN.CFCUC
    
            ^EndDate = RP
            */
            assertion = (report.RPJFOREIGNCFCUCExists != false && report.RPJFOREIGNCFCUCEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNCFCUCOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_8()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.LowCost
    
            ^EndDate1 = RP:RP.LowCost
    
            ^EndDate = RP
            */
            assertion = (report.RPLowCostExists != false && report.RPLowCostEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_9()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.NonPrim
    
            ^EndDate1 = RP:RP.NonPrim
    
            ^EndDate = RP
            */
            assertion = (report.RPNonPrimExists != false && report.RPNonPrimEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_10()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Prim
    
            ^EndDate1 = RP:RP.Prim
    
            ^EndDate = RP
            */
            assertion = (report.RPPrimExists != false && report.RPPrimEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_11()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Share.NonPrim
    
            ^EndDate1 = RP:RP.Share.NonPrim
    
            ^EndDate = RP
            */
            assertion = (report.RPShareNonPrimExists != false && report.RPShareNonPrimEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPShareNonPrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_12()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Share.Prim
    
            ^EndDate1 = RP:RP.Share.Prim
    
            ^EndDate = RP
            */
            assertion = (report.RPSharePrimExists != false && report.RPSharePrimEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSharePrimOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438054_13()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
    
            ^EndDate1 = RP:RP.TOFA
    
            ^EndDate = RP
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438063
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438063()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438063
            If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            IF INT<>NULL AND INT:pyin.xx.xx:Declaration.StatementAccepted.Indicator <> TRUE
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT <> NULL AND ^PTR295 <> TRUE
    
            Data Elements:
    
            ^PTR295 = INT:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists != false && report.PTR295 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.StatementAccepted.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR295", Value = GetValueOrEmpty(report.PTR295) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438064
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438064()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438064
            If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            IF INT=NULL AND RP:pyin.xx.xx:Declaration.StatementAccepted.Indicator <> TRUE
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT = NULL AND ^PTR290 <> TRUE
    
            Data Elements:
    
            ^PTR290 = RP:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists == false && report.PTR290 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438064", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.StatementAccepted.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438064"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR290", Value = GetValueOrEmpty(report.PTR290) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438065
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438065()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438065
            If an Intermediary is present, Intermediary declaration signatory identifier must be supplied.
    
            Legacy Rule Format:
            IF INT<>NULL AND INT:pyin.xx.xx:Declaration.SignatoryIdentifier.Text = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT <> NULL AND ^PTR297 = BLANK
    
            Data Elements:
    
            ^PTR297 = INT:Declaration:Declaration.SignatoryIdentifier.Text WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists != false && string.IsNullOrWhiteSpace(report.PTR297) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438065", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signatory identifier is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration signatory identifier must be supplied.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.SignatoryIdentifier.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438065"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR297", Value = report.PTR297 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438066
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438066()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438066
            If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.
    
            Legacy Rule Format:
            IF INT=NULL AND RP:pyin.xx.xx:Declaration.SignatoryIdentifier.Text = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT = NULL AND ^PTR292 = BLANK
    
            Data Elements:
    
            ^PTR292 = RP:Declaration:Declaration.SignatoryIdentifier.Text WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists == false && string.IsNullOrWhiteSpace(report.PTR292) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438066", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signatory identifier is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.SignatoryIdentifier.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR292", Value = report.PTR292 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438067
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438067()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438067
            If an Intermediary is present, Intermediary declaration signature date must be supplied.
    
            Legacy Rule Format:
            IF INT<>NULL AND INT:pyin.xx.xx:Declaration.Signature.Date = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT <> NULL AND ^PTR294 = NULL
    
            Data Elements:
    
            ^PTR294 = INT:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists != false && report.PTR294 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438067", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signature date is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration signature date must be supplied.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438067"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR294", Value = GetValueOrEmptyDate(report.PTR294) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.438068
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN438068()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.438068
            If an Intermediary is not present, Reporting Party declaration signature date must be supplied.
    
            Legacy Rule Format:
            IF INT=NULL AND RP:pyin.xx.xx:Declaration.Signature.Date = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT = NULL AND ^PTR289 = NULL
    
            Data Elements:
    
            ^PTR289 = RP:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists == false && report.PTR289 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438068", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signature date is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signature date must be supplied.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438068"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR289", Value = GetValueOrEmptyDate(report.PTR289) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500130
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500130()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500130
            Person unstructured name must be provided.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.FullName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^PTR314 = BLANK)
    
            Data Elements:
    
            ^PTR314 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
    
            ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
            */
            assertion = (report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && string.IsNullOrWhiteSpace(report.PTR314) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500130", Severity = ProcessMessageSeverity.Error,
                    Description = @"Person unstructured name must be provided.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500130"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR314", Value = report.PTR314 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500130
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500130_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500130
            Person unstructured name must be provided.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.FullName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^PTR264 = BLANK)
    
            Data Elements:
    
            ^PTR264 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
    
            ^PersonUnstructuredName = RP:Declaration:PersonUnstructuredName
            */
            assertion = (report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists != false && string.IsNullOrWhiteSpace(report.PTR264) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500130", Severity = ProcessMessageSeverity.Error,
                    Description = @"Person unstructured name must be provided.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500130"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR264", Value = report.PTR264 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500132
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500132()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^PTR116 = BLANK)
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
    
            ^OrganisationNameDetails = RP:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MTR_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR116) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500132
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500132_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^PTR24 = BLANK)
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^OrganisationNameDetails = RP:MailRecipient:OrganisationNameDetails
            */
            assertion = (report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR24) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500132
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500132_2(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500132
                    Organisation name must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
                    IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^OrganisationNameDetails <> NULL) AND (^PTR149 = BLANK)
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            
                    ^OrganisationNameDetails = RP:StatementOfDistribution:OrganisationNameDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPExists != false && string.IsNullOrWhiteSpace(statementOfDistribution.PTR149) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500132", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must be present.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500132
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500132_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^PTR6 = BLANK)
    
            Data Elements:
    
            ^PTR6 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^OrganisationNameDetails = RP:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MN_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR6) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR6", Value = report.PTR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500132
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500132_4()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^PTR8 = BLANK)
    
            Data Elements:
    
            ^PTR8 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.Currency.Code='P'].[OrganisationNameDetails.OrganisationalNameType.Code='MN']
    
            ^OrganisationNameDetails = RP:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_P_MN_RPExists != false && string.IsNullOrWhiteSpace(report.PTR8) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR8", Value = report.PTR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500133
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500133()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500133
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR117 = BLANK)
    
            Data Elements:
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR117) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500133", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500133
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500133_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500133
                    Address Line 1 must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^PTR151 = BLANK)
            
                    Data Elements:
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:StatementOfDistribution:AddressDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(statementOfDistribution.PTR151) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500133", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must be present.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500133
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500133_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500133
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR9 = BLANK)
    
            Data Elements:
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR9) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500133", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500133
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500133_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500133
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR14 = BLANK)
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_P_RPExists != false && string.IsNullOrWhiteSpace(report.PTR14) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500133", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500133
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500133_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500133
                    Address Line 1 must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^PTR332 = BLANK)
            
                    Data Elements:
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:StatementOfDistribution:AddressDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPExists != false && string.IsNullOrWhiteSpace(statementOfDistribution.PTR332) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500133", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must be present.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500134()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500134
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR118 = BLANK)
    
            Data Elements:
    
            ^PTR118 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR118) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500134", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR118", Value = report.PTR118 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500134_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500134
                    Suburb/Town/City must be present in an address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^PTR153 = BLANK)
            
                    Data Elements:
            
                    ^PTR153 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:StatementOfDistribution:AddressDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(statementOfDistribution.PTR153) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500134", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City must be present in an address.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR153", Value = statementOfDistribution.PTR153 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500134_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500134
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR10 = BLANK)
    
            Data Elements:
    
            ^PTR10 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.PTR10) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500134", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR10", Value = report.PTR10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500134_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500134
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^PTR15 = BLANK)
    
            Data Elements:
    
            ^PTR15 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_P_RPExists != false && string.IsNullOrWhiteSpace(report.PTR15) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500134", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR15", Value = report.PTR15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500134_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500134
                    Suburb/Town/City must be present in an address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^PTR334 = BLANK)
            
                    Data Elements:
            
                    ^PTR334 = RP:StatementOfDistribution:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:StatementOfDistribution:AddressDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPExists != false && string.IsNullOrWhiteSpace(statementOfDistribution.PTR334) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500134", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City must be present in an address.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR334", Value = statementOfDistribution.PTR334 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500135
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500135(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500135
                    Family Name must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
                    IF pyde.xx.xx:PersonNameDetails.FamilyName.Text = BLANK
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PersonNameDetails <> NULL) AND (^PTR146 = BLANK)
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
            
                    ^PersonNameDetails = RP:StatementOfDistribution:PersonNameDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPExists != false && string.IsNullOrWhiteSpace(statementOfDistribution.PTR146) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500135", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name must be present.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500135"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500135
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500135_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500135
            Family Name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text = BLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (^PTR20 = BLANK)
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
    
            ^PersonNameDetails = RP:MailRecipient:PersonNameDetails
            */
            assertion = (report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPExists != false && string.IsNullOrWhiteSpace(report.PTR20) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500135", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name must be present.",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500135"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500141
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500141()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
            AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR119 <> BLANK) AND (NotInSet(^PTR119, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^PTR119 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR119) != true && !(IsMatch(report.PTR119, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR119", Value = GetValueOrEmpty(report.PTR119) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500141
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500141_1(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500141
                    For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
                    AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR154 <> BLANK) AND (NotInSet(^PTR154, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
            
                    Data Elements:
            
                    ^PTR154 = RP:StatementOfDistribution:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR154) != true && !(IsMatch(statementOfDistribution.PTR154, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                            Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR154", Value = GetValueOrEmpty(statementOfDistribution.PTR154) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500141
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500141_2()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
            AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR11 <> BLANK) AND (NotInSet(^PTR11, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^PTR11 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR11) != true && !(IsMatch(report.PTR11, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR11", Value = GetValueOrEmpty(report.PTR11) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500141
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500141_3()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
            AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR16 <> BLANK) AND (NotInSet(^PTR16, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^PTR16 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR16) != true && !(IsMatch(report.PTR16, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR16", Value = GetValueOrEmpty(report.PTR16) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500141
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500141_4(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500141
                    For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
                    AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR336 <> BLANK) AND (NotInSet(^PTR336, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
            
                    Data Elements:
            
                    ^PTR336 = RP:StatementOfDistribution:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR336) != true && !(IsMatch(statementOfDistribution.PTR336, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                            Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR336", Value = GetValueOrEmpty(statementOfDistribution.PTR336) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500142
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500142(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500142
                    Family name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
                    IF pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PersonNameDetails <> NULL) AND (NotContainsSet(^PTR146, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
            
                    ^PersonNameDetails = RP:StatementOfDistribution:PersonNameDetails
                    */
                    assertion = (statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPExists != false && !(IsMatch(statementOfDistribution.PTR146, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name must contain at least one alphabetical character.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500142"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500142
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500142_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500142
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (NotContainsSet(^PTR20, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
    
            ^PersonNameDetails = RP:MailRecipient:PersonNameDetails
            */
            assertion = (report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPExists != false && !(IsMatch(report.PTR20, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500142"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500151
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500151(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500151
                    First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR147, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^PTR147 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR147, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.010436", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500151"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500151
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500151_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500151
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^PTR22 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR22, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500151"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500152
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500152(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500152
                    Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR148, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^PTR148 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR148, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.010448", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500152"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500152
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500152_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500152
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^PTR23 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR23, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500152"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.GEN.500153
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500153(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.GEN.500153
                    Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^PTR146, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = IsMatch(statementOfDistribution.PTR146, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500153", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500153"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.GEN.500153
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOGEN500153_1()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.GEN.500153
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^PTR20, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = IsMatch(report.PTR20, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500153"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430001
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430001()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430001
            Depreciation expenses must not exceed total Small business entity depreciating assets
    
            Legacy Rule Format:
            IF ([PTR54] < ([PTR141] + [PTR142]))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR54 < (^PTR141 + ^PTR142))
    
            Data Elements:
    
            ^PTR54 = RP.JAUS:Expense.DepreciationAndAmortisation.Amount
    
            ^PTR141 = RP.LowCost:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^PTR142 = RP.GeneralPool:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = (report.PTR54.GetValueOrDefault() < report.PTR141.GetValueOrDefault() + report.PTR142.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business entity depreciating assets must not exceed Depreciation expenses",
                    LongDescription = @"The sum of deductions for 'Small business entity depreciating assets' must not exceed the 'Depreciation expenses' amount",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430001"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR54", Value = GetValueOrEmpty(report.PTR54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR141", Value = GetValueOrEmpty(report.PTR141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR142", Value = GetValueOrEmpty(report.PTR142) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430009
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430009()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430009
            Previous postal address must not be the same as current postal address
    
            Legacy Rule Format:
            IF ([PTR14] = [PTR9] AND [PTR212] = [PTR211] AND [PTR15] = [PTR10] AND [PTR16] = [PTR11] AND [PTR17] = [PTR12]) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR14 = ^PTR9 AND ^PTR212 = ^PTR211 AND ^PTR15 = ^PTR10 AND ^PTR16 = ^PTR11 AND ^PTR17 = ^PTR12
    
            Data Elements:
    
            ^PTR14 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR9 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR212 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR211 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR15 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR10 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR16 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR11 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^PTR17 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^PTR12 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.PTR14 == report.PTR9 && report.PTR212 == report.PTR211 && report.PTR15 == report.PTR10 && report.PTR16 == report.PTR11 && report.PTR17 == report.PTR12);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous postal address must not be the same as current postal address",
                    LongDescription = @"Previous postal address is only required when an address change has occurred since the last return.  Delete previous postal address or input the old address if a change of address has occurred.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430009"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR14", Value = report.PTR14 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR9", Value = report.PTR9 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR15", Value = report.PTR15 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR10", Value = report.PTR10 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR16", Value = GetValueOrEmpty(report.PTR16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR11", Value = GetValueOrEmpty(report.PTR11) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR17", Value = GetValueOrEmpty(report.PTR17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR12", Value = GetValueOrEmpty(report.PTR12) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430010()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430010
            Name of company or trust and name of individual must not both be present
    
            Legacy Rule Format:
            IF [PTR24] <> NULLORBLANK AND [PTR20] <> NULLORBLANK 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR24 <> BLANK AND ^PTR20 <> BLANK
    
            Data Elements:
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR24) != true && string.IsNullOrWhiteSpace(report.PTR20) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of company or trust and name of individual must not both be present",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:PersonNameDetails" + OccurrenceIndex(report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430012
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430012()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430012
            Name of the partner to whom notices should be sent must be present
    
            Legacy Rule Format:
            IF [PTR20] = NULLORBLANK AND [PTR24] = NULLORBLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR20 = BLANK AND ^PTR24 = BLANK
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^PTR20 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR20) == true && string.IsNullOrWhiteSpace(report.PTR24) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430012", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the partner to whom notices should be sent must be present",
                    LongDescription = @"If the partner is an individual, provide the name of the individual. If the partner is a company or trust, provide the name of the company or trust",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430012"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR20", Value = report.PTR20 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430013()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430013
            Name of the company or trust must be present if the ABN is present
    
            Legacy Rule Format:
            IF [PTR25] <> NULLORBLANK AND [PTR24] = NULLORBLANK 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR25 <> BLANK AND ^PTR24 = BLANK
    
            Data Elements:
    
            ^PTR24 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^PTR25 = RP:MailRecipient:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR25) != true && string.IsNullOrWhiteSpace(report.PTR24) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the company or trust must be present if the ABN is present",
                    Location = "/xbrli:xbrl/tns:MailRecipient/tns:OrganisationNameDetails" + OccurrenceIndex(report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR25", Value = GetValueOrEmpty(report.PTR25) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR24", Value = report.PTR24 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430015
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430015()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430015
            Description of main business activity and total business income must be present
    
            Legacy Rule Format:
            IF [PTR31] <> NULL AND (([PTR30] = NULLORBLANK) OR ([PTR35] = NULL AND [PTR36] = NULL AND [PTR38] = NULL AND [PTR39] = NULL AND [PTR41] = NULL AND [PTR43] = NULL AND [PTR44] = NULL)) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR31 <> NULL AND ((^PTR30 = BLANK) OR (^PTR35 = NULL AND ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR39 = NULL AND ^PTR41 = NULL AND ^PTR43 = NULL AND ^PTR44 = NULL))
    
            Data Elements:
    
            ^PTR30 = RP:OrganisationDetails.MainIncomeActivity.Description
    
            ^PTR31 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
            */
            assertion = (report.PTR31 != null && (string.IsNullOrWhiteSpace(report.PTR30) == true || report.PTR35 == null && report.PTR36 == null && report.PTR38 == null && report.PTR39 == null && report.PTR41 == null && report.PTR43 == null && report.PTR44 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430015", Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of main business activity and total business income must be present",
                    LongDescription = @"If an Industry code is present, then the 'Description of main business activity' and business income (Primary Production and/or Non-Primary Production) must be present",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.MainIncomeActivity.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430015"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR30", Value = GetValueOrEmpty(report.PTR30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430016
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430016()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430016
            Description of main business and industry code must be present
    
            Legacy Rule Format:
            IF [PTR32] <> NULL AND ([PTR31] = NULL OR [PTR30] = NULLORBLANK) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR32 <> NULL AND (^PTR31 = NULL OR ^PTR30 = BLANK)
    
            Data Elements:
    
            ^PTR30 = RP:OrganisationDetails.MainIncomeActivity.Description
    
            ^PTR32 = RP:OrganisationDetails.ActivityEvent.Code
    
            ^PTR31 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (report.PTR32 != null && (report.PTR31 == null || string.IsNullOrWhiteSpace(report.PTR30) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430016", Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of main business and industry code must be present",
                    LongDescription = @"If Status of Business code is present, the Description of main business activity and Industry code are also required",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.MainIncomeActivity.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430016"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR32", Value = GetValueOrEmpty(report.PTR32) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR30", Value = GetValueOrEmpty(report.PTR30) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430017
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430017()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430017
            PP business income must be present
    
            Legacy Rule Format:
            IF [PTR61] <> NULL AND ([PTR35] = NULL  AND [PTR39] = NULL AND [PTR43] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR61 <> NULL AND (^PTR35 = NULL  AND ^PTR39 = NULL AND ^PTR43 = NULL)
    
            Data Elements:
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
            */
            assertion = (report.PTR61 != null && (report.PTR35 == null && report.PTR39 == null && report.PTR43 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430017", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP business income must be present",
                    LongDescription = @"Net primary production income/loss amount is present without total business income from primary production (Gross payments subject to Foreign Resident Withholding, Gross payments where ABN not quoted, Assessable government industry payments, Other business income).",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430017"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430018
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430018()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430018
            Total business income amount required
    
            Legacy Rule Format:
            IF ([PTR116] <> NULLORBLANK OR [PTR117] <> NULLORBLANK) AND ([PTR35] = NULL AND [PTR36] = NULL AND [PTR38] = NULL AND [PTR39] = NULL AND [PTR41] = NULL AND [PTR43] = NULL AND [PTR44] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR116 <> BLANK OR ^PTR117 <> BLANK) AND (^PTR35 = NULL AND ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR39 = NULL AND ^PTR41 = NULL AND ^PTR43 = NULL AND ^PTR44 = NULL)
    
            Data Elements:
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
            */
            assertion = ((string.IsNullOrWhiteSpace(report.PTR116) != true || string.IsNullOrWhiteSpace(report.PTR117) != true) && (report.PTR35 == null && report.PTR36 == null && report.PTR38 == null && report.PTR39 == null && report.PTR41 == null && report.PTR43 == null && report.PTR44 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430018", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total business income amount required",
                    LongDescription = @"Business name of main business or street address of main business is present and total business income (primary production or non-primary production) is missing at business income and expenses.",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430018"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430021
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430021()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430021
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR35] <> NULL AND [PTR35] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR35, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = NotMonetary(report.PTR35, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430021"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430022
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430022()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430022
            Non-PP business income must be present
    
            Legacy Rule Format:
            IF [PTR62] <> NULL AND ([PTR36] = NULL AND [PTR38] = NULL AND [PTR41] = NULL AND [PTR44] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR62 <> NULL AND (^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR41 = NULL AND ^PTR44 = NULL)
    
            Data Elements:
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
            */
            assertion = (report.PTR62 != null && (report.PTR36 == null && report.PTR38 == null && report.PTR41 == null && report.PTR44 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430022", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP business income must be present",
                    LongDescription = @"Net non-primary production income/loss amount is present without total business income non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding, Assessable government industry payments, Other business income).",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430022"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430023()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430023
            Net non-PP income must be present
    
            Legacy Rule Format:
            IF [PTR62] = NULL AND ([PTR36] <> NULL OR [PTR38] <> NULL OR [PTR41] <> NULL OR [PTR44] <> NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR62 = NULL AND (^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR41 <> NULL OR ^PTR44 <> NULL)
    
            Data Elements:
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
            */
            assertion = (report.PTR62 == null && (report.PTR36 != null || report.PTR38 != null || report.PTR41 != null || report.PTR44 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-PP income must be present",
                    LongDescription = @"Total business income from non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding, Assessable government industry payments, Other business income) is present without net non-primary production income/loss amount.",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430023"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430024
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430024()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430024
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR36] <> NULL AND [PTR36] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR36, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = NotMonetary(report.PTR36, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430024"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430027
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430027()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430027
            Credit for tax withheld - foreign resident withholding (exlcluding capital gains) must not exceed 50% of gross payments
    
            Legacy Rule Format:
            IF [PTR350] > 0 AND [PTR350] > ([PTR38] *0.50) + 1
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR350 > 0 AND ^PTR350 > (^PTR38 *0.50) + 1
    
            Data Elements:
    
            ^PTR350 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.PTR350.GetValueOrDefault() > 0 && report.PTR350.GetValueOrDefault() > report.PTR38.GetValueOrDefault() * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430027", Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must not exceed 50% of gross payments",
                    LongDescription = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must not exceed 50% of total of Primary production and Non-primary production Gross payments subject to foreign resident withholding",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430027"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430028
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430028()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430028
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR38] <> NULL AND [PTR38] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR38, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = NotMonetary(report.PTR38, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430028"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430029()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430029
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR39] <> NULL AND [PTR39] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR39, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
            */
            assertion = NotMonetary(report.PTR39, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430029"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430030
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430030()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430030
            PP Assessable government industry payments amount must be present
    
            Legacy Rule Format:
            IF [PTR258] = TRUE AND ([PTR39] = 0 OR [PTR39] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR258 = TRUE AND (^PTR39 = 0 OR ^PTR39 = NULL)
    
            Data Elements:
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR258 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator
            */
            assertion = (report.PTR258 == true && (report.PTR39 == 0 || report.PTR39 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430030", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Assessable government industry payments amount must be present",
                    LongDescription = @"If Primary production - Assessable government industry payments action code is present, Primary production - Assessable government industry payments amount must be present and not zero",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR258", Value = GetValueOrEmpty(report.PTR258) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430031
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430031()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430031
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR41] <> NULL AND [PTR41] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR41, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
            */
            assertion = NotMonetary(report.PTR41, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430031"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430032
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430032()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430032
            Non-PP Assessable government industry payments amount must be present
    
            Legacy Rule Format:
            IF [PTR259] = TRUE AND ([PTR41] = 0 OR [PTR41] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR259 = TRUE AND (^PTR41 = 0 OR ^PTR41 = NULL)
    
            Data Elements:
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR259 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator
            */
            assertion = (report.PTR259 == true && (report.PTR41 == 0 || report.PTR41 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430032", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP Assessable government industry payments amount must be present",
                    LongDescription = @"If Non-primary production - Assessable government industry payments action code is present, Non-primary production - Assessable government industry payments must be present and not zero",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430032"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR259", Value = GetValueOrEmpty(report.PTR259) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430033
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430033()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430033
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR43] <> NULL AND [PTR43] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR43, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
            */
            assertion = NotMonetary(report.PTR43, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430033"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430034
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430034()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430034
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR44] <> NULL AND [PTR44] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR44, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
            */
            assertion = NotMonetary(report.PTR44, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430034"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430035
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430035()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430035
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR45] <> NULL AND [PTR45] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR45, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR45 = RP.JAUS:Expense.ForeignResidentWithholding.Amount
            */
            assertion = NotMonetary(report.PTR45, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ForeignResidentWithholding.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR45", Value = GetValueOrEmpty(report.PTR45) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430036
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430036()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430036
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR47] <> NULL AND [PTR47] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR47, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR47 = RP.JAUS:SuperannuationContribution.EmployerContributions.Amount
            */
            assertion = NotMonetary(report.PTR47, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerContributions.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430036"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR47", Value = GetValueOrEmpty(report.PTR47) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430038()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430038
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR50] <> NULL AND [PTR50] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR50, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR50 = RP.JAUS:Expense.Operating.LeaseTotal.Amount
            */
            assertion = NotMonetary(report.PTR50, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.LeaseTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR50", Value = GetValueOrEmpty(report.PTR50) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430040
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430040()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430040
            Overseas royalty expenses must not exceed Total royalty expenses
    
            Legacy Rule Format:
            IF [PTR138] > 0 AND [PTR138] > [PTR347] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR138 > 0 AND ^PTR138 > ^PTR347
    
            Data Elements:
    
            ^PTR138 = RP.JFOREIGN:Expense.Royalties.Amount
    
            ^PTR347 = RP.JAUS:Expense.Royalties.Amount
            */
            assertion = (report.PTR138.GetValueOrDefault() > 0 && report.PTR138.GetValueOrDefault() > report.PTR347.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430040", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas royalty expenses must not exceed Total royalty expenses",
                    LongDescription = @"Overseas royalty expenses - under Business and professional items must not be greater than Total royalty expenses - under Business income and expenses",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430040"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR138", Value = GetValueOrEmpty(report.PTR138) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR347", Value = GetValueOrEmpty(report.PTR347) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430044
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430044()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430044
            Motor vehicle expenses amount must be present if action code is present
    
            Legacy Rule Format:
            IF [PTR56] <> NULL AND ([PTR348] = 0 OR [PTR348] = NULL)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR56 <> NULL AND (^PTR348 = 0 OR ^PTR348 = NULL)
    
            Data Elements:
    
            ^PTR348 = RP.JAUS:Expense.Operating.MotorVehicle.Amount
    
            ^PTR56 = RP.JAUS:Expense.Operating.MotorVehicleTypeOrClaimMethod.Code
            */
            assertion = (report.PTR56 != null && (report.PTR348 == 0 || report.PTR348 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430044", Severity = ProcessMessageSeverity.Error,
                    Description = @"Motor vehicle expenses amount must be present if action code is present",
                    LongDescription = @"If Motor vehicle expenses action code is present, there must be a motor vehicle expense amount",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.MotorVehicle.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430044"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR56", Value = GetValueOrEmpty(report.PTR56) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR348", Value = GetValueOrEmpty(report.PTR348) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430046
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430046()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430046
            Motor vehicle expenses action code must be present
    
            Legacy Rule Format:
            IF [PTR348] > 0 AND [PTR56] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR348 > 0 AND ^PTR56 = NULL
    
            Data Elements:
    
            ^PTR56 = RP.JAUS:Expense.Operating.MotorVehicleTypeOrClaimMethod.Code
    
            ^PTR348 = RP.JAUS:Expense.Operating.MotorVehicle.Amount
            */
            assertion = (report.PTR348.GetValueOrDefault() > 0 && report.PTR56 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430046", Severity = ProcessMessageSeverity.Error,
                    Description = @"Motor vehicle expenses action code must be present",
                    LongDescription = @"If a motor vehicle expense amount is present, a motor vehicle expense action code must be present. Codes are: S - Cents per Kilometre, T - 12% of cost of car, O - One-third actual expenses, B - Logbook, or N - Expenses of other vehicles",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.MotorVehicleTypeOrClaimMethod.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430046"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR348", Value = GetValueOrEmpty(report.PTR348) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR56", Value = GetValueOrEmpty(report.PTR56) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430047
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430047()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430047
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR59] <> NULL AND [PTR59] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR59, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR59 = RP.JAUS:Income.ReconciliationAdjustmentTotal.Amount
            */
            assertion = NotMonetary(report.PTR59, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.ReconciliationAdjustmentTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430047"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR59", Value = GetValueOrEmpty(report.PTR59) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430048
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430048()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430048
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR328] <> NULL AND [PTR328] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR328, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR328 = RP.JAUS:Expense.ReconciliationAdjustmentTotal.Amount
            */
            assertion = NotMonetary(report.PTR328, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ReconciliationAdjustmentTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430048"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR328", Value = GetValueOrEmpty(report.PTR328) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430049
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430049()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430049
            PP Net income or loss from business must be present
    
            Legacy Rule Format:
            IF ([PTR35] <> NULL OR [PTR39] <> NULL OR [PTR43] <> NULL) AND [PTR61] = NULL 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR35 <> NULL OR ^PTR39 <> NULL OR ^PTR43 <> NULL) AND ^PTR61 = NULL
    
            Data Elements:
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
            */
            assertion = ((report.PTR35 != null || report.PTR39 != null || report.PTR43 != null) && report.PTR61 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430049", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Net income or loss from business must be present",
                    LongDescription = @"If Total business income from primary production (Gross payments, Assessable government industry payments, other business income) is present, Primary production Net primary production income or loss from business amount must be present",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430049"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430050
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430050()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430050
            Total Net income or loss from business must equal total Income minus total Expenses
    
            Legacy Rule Format:
            IF ([PTR35] + [PTR36] + [PTR38] + [PTR39] + [PTR41] + [PTR43] + [PTR44]) - ([PTR45] + [PTR46] + [PTR47] + [PTR343] + [PTR344] + [PTR50] + [PTR345] + [PTR346] + [PTR347] + [PTR54] + [PTR348] + [PTR349] + [PTR58]) + [PTR59] + [PTR328] <> ([PTR61] + [PTR62]) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR35 + ^PTR36 + ^PTR38 + ^PTR39 + ^PTR41 + ^PTR43 + ^PTR44) - (^PTR45 + ^PTR46 + ^PTR47 + ^PTR343 + ^PTR344 + ^PTR50 + ^PTR345 + ^PTR346 + ^PTR347 + ^PTR54 + ^PTR348 + ^PTR349 + ^PTR58) + ^PTR59 + ^PTR328 <> (^PTR61 + ^PTR62)
    
            Data Elements:
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
    
            ^PTR45 = RP.JAUS:Expense.ForeignResidentWithholding.Amount
    
            ^PTR46 = RP.JAUS:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount
    
            ^PTR47 = RP.JAUS:SuperannuationContribution.EmployerContributions.Amount
    
            ^PTR343 = RP.JAUS:Expense.Operating.CostOfSales.Amount
    
            ^PTR344 = RP.JAUS:Expense.Operating.BadDebts.Amount
    
            ^PTR50 = RP.JAUS:Expense.Operating.LeaseTotal.Amount
    
            ^PTR345 = RP.JAUS:Expense.Operating.Rent.Amount
    
            ^PTR346 = RP.JAUS:Expense.Operating.Interest.Amount
    
            ^PTR347 = RP.JAUS:Expense.Royalties.Amount
    
            ^PTR54 = RP.JAUS:Expense.DepreciationAndAmortisation.Amount
    
            ^PTR348 = RP.JAUS:Expense.Operating.MotorVehicle.Amount
    
            ^PTR349 = RP.JAUS:Expense.Operating.RepairsAndMaintenance.Amount
    
            ^PTR58 = RP.JAUS:Expense.DeductibleOtherTotal.Amount
    
            ^PTR59 = RP.JAUS:Income.ReconciliationAdjustmentTotal.Amount
    
            ^PTR328 = RP.JAUS:Expense.ReconciliationAdjustmentTotal.Amount
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
            */
            assertion = (report.PTR35.GetValueOrDefault() + report.PTR36.GetValueOrDefault() + report.PTR38.GetValueOrDefault() + report.PTR39.GetValueOrDefault() + report.PTR41.GetValueOrDefault() + report.PTR43.GetValueOrDefault() + report.PTR44.GetValueOrDefault() - (report.PTR45.GetValueOrDefault() + report.PTR46.GetValueOrDefault() + report.PTR47.GetValueOrDefault() + report.PTR343.GetValueOrDefault() + report.PTR344.GetValueOrDefault() + report.PTR50.GetValueOrDefault() + report.PTR345.GetValueOrDefault() + report.PTR346.GetValueOrDefault() + report.PTR347.GetValueOrDefault() + report.PTR54.GetValueOrDefault() + report.PTR348.GetValueOrDefault() + report.PTR349.GetValueOrDefault() + report.PTR58.GetValueOrDefault()) + report.PTR59.GetValueOrDefault() + report.PTR328.GetValueOrDefault() != report.PTR61.GetValueOrDefault() + report.PTR62.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430050", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Net income or loss must equal total Income, plus Reconciliation items, minus total Expenses",
                    LongDescription = @"The total of Primary production and Non-primary production Net business income must equal the total of all income amounts, plus Reconciliation items, minus the sum of all expenses amounts",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430050"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR45", Value = GetValueOrEmpty(report.PTR45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR46", Value = GetValueOrEmpty(report.PTR46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR47", Value = GetValueOrEmpty(report.PTR47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR343", Value = GetValueOrEmpty(report.PTR343) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR344", Value = GetValueOrEmpty(report.PTR344) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR50", Value = GetValueOrEmpty(report.PTR50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR345", Value = GetValueOrEmpty(report.PTR345) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR346", Value = GetValueOrEmpty(report.PTR346) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR347", Value = GetValueOrEmpty(report.PTR347) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR54", Value = GetValueOrEmpty(report.PTR54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR348", Value = GetValueOrEmpty(report.PTR348) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR349", Value = GetValueOrEmpty(report.PTR349) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR58", Value = GetValueOrEmpty(report.PTR58) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR59", Value = GetValueOrEmpty(report.PTR59) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR328", Value = GetValueOrEmpty(report.PTR328) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430051
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430051()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430051
            Net income or loss from business must be present
    
            Legacy Rule Format:
            IF ([PTR63] <> NULL OR [PTR350] <> NULL) AND ([PTR61] = NULL AND [PTR62] = NULL)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR63 <> NULL OR ^PTR350 <> NULL) AND (^PTR61 = NULL AND ^PTR62 = NULL)
    
            Data Elements:
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR63 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR350 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
            */
            assertion = ((report.PTR63 != null || report.PTR350 != null) && (report.PTR61 == null && report.PTR62 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430051", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income or loss from business must be present",
                    LongDescription = @"If Tax withheld where ABN not quoted is present or Credit for tax withheld - Foreign resident withholding is present, then Net income or loss from business must be present",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430051"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430052
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430052()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430052
            PP Net income/loss from business or distribution from partnerships or trusts must be present
    
            Legacy Rule Format:
            IF (ANY OCCURRENCE OF([PTR158]) > 0) AND ([PTR61] = NULL AND [PTR65] = NULL AND [PTR66] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (AnyOccurrence(^PTR158, ^PTR158 > 0)) AND (^PTR61 = NULL AND ^PTR65 = NULL AND ^PTR66 = NULL)
    
            Data Elements:
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
    
            ^PTR65 = RP.JAUS.Prim:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount
            */
            assertion = ((report.StatementOfDistributionCollection == null ? false : report.StatementOfDistributionCollection.Any(PTR158Repeat => PTR158Repeat.PTR158.GetValueOrDefault() > 0)) && (report.PTR61 == null && report.PTR65 == null && report.PTR66 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430052", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Net income or loss from business or a distribution from partnerships or trusts must be present",
                    LongDescription = @"If an amount, greater than zero, is present for a Partner in the distribution statement at Share of income  - Primary production, there must also be an amount greater than zero at Primary production Net income or loss from business, or Primary production - Distribution from partnerships, or Primary production - Distribution from trusts",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430052"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR158", Value = "PTR158" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430054()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430054
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR61] <> NULL AND [PTR61] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR61, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
            */
            assertion = NotMonetary(report.PTR61, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430054"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430055
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430055()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430055
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR62] <> NULL AND [PTR62] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR62, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
            */
            assertion = NotMonetary(report.PTR62, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430056
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430056()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430056
            Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (PP and Non-PP) where ABN not quoted.
    
            Legacy Rule Format:
            IF [PTR63] > 0 AND [PTR63] > (([PTR35] + [PTR36]) *0.50) + 1 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR63 > 0 AND ^PTR63 > ((^PTR35 + ^PTR36) *0.50) + 1
    
            Data Elements:
    
            ^PTR63 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.PTR63.GetValueOrDefault() > 0 && report.PTR63.GetValueOrDefault() > (report.PTR35.GetValueOrDefault() + report.PTR36.GetValueOrDefault()) * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430056", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (PP and Non-PP) where ABN not quoted.",
                    LongDescription = @"Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (Primary production and Non-primary production) where ABN not quoted.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430056"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430057
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430057()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430057
            PP or Non-PP Gross payments where ABN not quoted must be present
    
            Legacy Rule Format:
            IF [PTR63] <> NULL AND [PTR35] = NULL AND [PTR36] = NULL 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR63 <> NULL AND ^PTR35 = NULL AND ^PTR36 = NULL
    
            Data Elements:
    
            ^PTR63 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.PTR63 != null && report.PTR35 == null && report.PTR36 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430057", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP or Non-PP Gross payments where ABN not quoted must be present",
                    LongDescription = @"If Tax withheld where ABN not quoted is present there must be an amount at Gross payments where ABN not quoted for either Primary production or Non-primary production",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430058
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430058()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430058
            Tax withheld where ABN not quoted must be present
    
            Legacy Rule Format:
            IF ([PTR35] > 0 OR [PTR36] > 0) AND [PTR63] = NULL 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR35 > 0 OR ^PTR36 > 0) AND ^PTR63 = NULL
    
            Data Elements:
    
            ^PTR63 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = ((report.PTR35.GetValueOrDefault() > 0 || report.PTR36.GetValueOrDefault() > 0) && report.PTR63 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430058", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted must be present",
                    LongDescription = @"If either Primary production or Non-primary production Gross payments where ABN not quoted are greater than zero, then an amount at Tax withheld where ABN not quoted must be present",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430059
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430059()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430059
            Credit for tax withheld - foreign resident withholding (excluding capital gains) must be present
    
            Legacy Rule Format:
            IF ([PTR38] > 0) AND [PTR350] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR38 > 0) AND ^PTR350 = NULL
    
            Data Elements:
    
            ^PTR350 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.PTR38.GetValueOrDefault() > 0 && report.PTR350 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430059", Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must be present",
                    LongDescription = @"If Primary production or Non-primary production Gross payments subject to foreign resident withholding is greater than zero, then there must be an amount at Credit for tax withheld - foreign resident withholding (excluding capital gains)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430061
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430061()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430061
            PP - Income distribution from partnerships and/or trusts must be present
    
            Legacy Rule Format:
            IF [PTR68] > 0 AND ([PTR65] = NULL AND [PTR66] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR68 > 0 AND (^PTR65 = NULL AND ^PTR66 = NULL)
    
            Data Elements:
    
            ^PTR65 = RP.JAUS.Prim:Income.PartnershipDistributionGross.Amount
    
            ^PTR68 = RP.JAUS.Prim:Expense.DistributionDeductionsOther.Amount
    
            ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount
            */
            assertion = (report.PTR68.GetValueOrDefault() > 0 && (report.PTR65 == null && report.PTR66 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430061", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP - Income distribution from partnerships and/or trusts must be present",
                    LongDescription = @"If there is an amount present at 'Primary production - Deductions relating to distribution', then there must also be an amount at 'Primary production - Distribution from partnerships' or 'Primary production - Distribution from trusts'",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430061"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430063
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430063()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430063
            Distribution from partnerships or trusts must be present
    
            Legacy Rule Format:
            IF ([PTR72] > 0 OR [PTR323] > 0 OR [PTR74] > 0 OR [PTR75] > 0 OR [PTR76] > 0) AND ([PTR65] = NULL AND [PTR66] = NULL AND [PTR69] = NULL AND [PTR70] = NULL AND [PTR322] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR72 > 0 OR ^PTR323 > 0 OR ^PTR74 > 0 OR ^PTR75 > 0 OR ^PTR76 > 0) AND (^PTR65 = NULL AND ^PTR66 = NULL AND ^PTR69 = NULL AND ^PTR70 = NULL AND ^PTR322 = NULL)
    
            Data Elements:
    
            ^PTR65 = RP.JAUS.Prim:Income.PartnershipDistributionGross.Amount
    
            ^PTR72 = RP.JAUS.NonPrim:Expense.DistributionDeductionsOther.Amount
    
            ^PTR323 = RP.JAUS.NonPrim:IncomeTax.Deduction.TrustDistributionFranked.Amount
    
            ^PTR74 = RP.JAUS:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^PTR75 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR76 = RP.JAUS:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
    
            ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount
    
            ^PTR69 = RP.JAUS.NonPrim:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR70 = RP.JAUS.NonPrim:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR322 = RP.JAUS.NonPrim:Income.TrustDistributionFranked.Amount
            */
            assertion = ((report.PTR72.GetValueOrDefault() > 0 || report.PTR323.GetValueOrDefault() > 0 || report.PTR74.GetValueOrDefault() > 0 || report.PTR75.GetValueOrDefault() > 0 || report.PTR76.GetValueOrDefault() > 0) && (report.PTR65 == null && report.PTR66 == null && report.PTR69 == null && report.PTR70 == null && report.PTR322 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from partnerships or trusts must be present",
                    LongDescription = @"If an amount is present at 'Deductions relating to distribution', 'Share of credit for tax withheld where ABN not quoted', 'Share of franking credit from franked dividends', 'Share of credit for TFN amounts withheld from interest dividends and unit trust distributions', or 'Share of credit for tax withheld from foreign resident withholding'; then there must be an amount present at 'Primary production - Distribution from partnerships', 'Primary production - Distribution from trusts', 'Non-primary production - Distribution from partnerships less foreign income', or 'Distribution from trusts less net capital gain and foreign income'",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR76", Value = GetValueOrEmpty(report.PTR76) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430064
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430064()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430064
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR65] <> NULL AND [PTR65] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR65, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR65 = RP.JAUS.Prim:Income.PartnershipDistributionGross.Amount
            */
            assertion = NotMonetary(report.PTR65, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430064"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430067
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430067()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430067
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR68] <> NULL AND [PTR68] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR68, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR68 = RP.JAUS.Prim:Expense.DistributionDeductionsOther.Amount
            */
            assertion = NotMonetary(report.PTR68, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DistributionDeductionsOther.Amount[@contextRef='" + report.RPJAUSPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430067"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430068
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430068()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430068
            Non-PP distribution amount must be present
    
            Legacy Rule Format:
            IF ([PTR74] <> NULL AND [PTR75] <> NULL) AND ([PTR69] = NULL AND [PTR70] = NULL AND [PTR72] = NULL AND [PTR322] = NULL AND [PTR323] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR74 <> NULL AND ^PTR75 <> NULL) AND (^PTR69 = NULL AND ^PTR70 = NULL AND ^PTR72 = NULL AND ^PTR322 = NULL AND ^PTR323 = NULL)
    
            Data Elements:
    
            ^PTR69 = RP.JAUS.NonPrim:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR74 = RP.JAUS:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^PTR75 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR70 = RP.JAUS.NonPrim:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR72 = RP.JAUS.NonPrim:Expense.DistributionDeductionsOther.Amount
    
            ^PTR322 = RP.JAUS.NonPrim:Income.TrustDistributionFranked.Amount
    
            ^PTR323 = RP.JAUS.NonPrim:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = (report.PTR74 != null && report.PTR75 != null && (report.PTR69 == null && report.PTR70 == null && report.PTR72 == null && report.PTR322 == null && report.PTR323 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430068", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP distribution amount must be present",
                    LongDescription = @"If there is an amount at Share of franking credits from franked dividends, and Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions; then there must be an amount at Non-Primary Production Distribution from partnerships less foreign income; Distribution from trusts less net capital gain and foreign income; or Deductions relating to distribution",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionLessForeignIncome.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430068"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430069
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430069()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430069
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR69] <> NULL AND [PTR69] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR69, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR69 = RP.JAUS.NonPrim:Income.PartnershipDistributionLessForeignIncome.Amount
            */
            assertion = NotMonetary(report.PTR69, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionLessForeignIncome.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430074
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430074()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430074
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR72] <> NULL AND [PTR72] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR72, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR72 = RP.JAUS.NonPrim:Expense.DistributionDeductionsOther.Amount
            */
            assertion = NotMonetary(report.PTR72, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DistributionDeductionsOther.Amount[@contextRef='" + report.RPJAUSNonPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430075
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430075()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430075
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR73] <> NULL AND [PTR73] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR73, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR73 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount
            */
            assertion = NotMonetary(report.PTR73, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR73", Value = GetValueOrEmpty(report.PTR73) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430076
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430076()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430076
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [PTR74] <> NULL AND [PTR74] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR74, 'U', 13, 2)
    
            Data Elements:
    
            ^PTR74 = RP.JAUS:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
            */
            assertion = NotMonetary(report.PTR74, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430077
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430077()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430077
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [PTR75] <> NULL AND [PTR75] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR75, 'U', 13, 2)
    
            Data Elements:
    
            ^PTR75 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
            */
            assertion = NotMonetary(report.PTR75, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430077"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430078
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430078()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430078
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR76] <> NULL AND [PTR76] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR76, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR76 = RP.JAUS:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
            */
            assertion = NotMonetary(report.PTR76, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR76", Value = GetValueOrEmpty(report.PTR76) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430079
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430079()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430079
            Gross rent must be present
    
            Legacy Rule Format:
            IF ([PTR78] > 0 OR [PTR79] > 0 OR [PTR80] > 0) AND [PTR77] = NULL 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR78 > 0 OR ^PTR79 > 0 OR ^PTR80 > 0) AND ^PTR77 = NULL
    
            Data Elements:
    
            ^PTR77 = RP.JAUS:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR78 = RP.JAUS:Expense.Interest.Amount
    
            ^PTR79 = RP.JAUS:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = RP.JAUS:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = ((report.PTR78.GetValueOrDefault() > 0 || report.PTR79.GetValueOrDefault() > 0 || report.PTR80.GetValueOrDefault() > 0) && report.PTR77 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430079", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent must be present",
                    LongDescription = @"If Interest deductions, Capital works deductions, or Other rental deductions are greater than zero, Gross rent must be present",
                    Location = "/xbrli:xbrl/tns:Income.Operating.RentalIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430079"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430080
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430080()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430080
            Rent deductions must be present
    
            Legacy Rule Format:
            IF [PTR77] > 0 AND ([PTR78] = NULL AND [PTR79] = NULL AND [PTR80] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR77 > 0 AND (^PTR78 = NULL AND ^PTR79 = NULL AND ^PTR80 = NULL)
    
            Data Elements:
    
            ^PTR78 = RP.JAUS:Expense.Interest.Amount
    
            ^PTR77 = RP.JAUS:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR79 = RP.JAUS:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = RP.JAUS:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = (report.PTR77.GetValueOrDefault() > 0 && (report.PTR78 == null && report.PTR79 == null && report.PTR80 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430080", Severity = ProcessMessageSeverity.Error,
                    Description = @"Rent deductions must be present",
                    LongDescription = @"If Gross rent is greater than zero, then there must be an amount at Interest deductions, Capital works deductions or Other rental deductions. If there are no rental deductions, then zero must be present in at least one of these fields",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430080"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430085
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430085()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430085
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR77] <> NULL AND [PTR77] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR77, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR77 = RP.JAUS:Income.Operating.RentalIncomeGross.Amount
            */
            assertion = NotMonetary(report.PTR77, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Operating.RentalIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430085"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430087
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430087()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430087
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR78] <> NULL AND [PTR78] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR78, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR78 = RP.JAUS:Expense.Interest.Amount
            */
            assertion = NotMonetary(report.PTR78, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430087"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430089
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430089()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430089
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR79] <> NULL AND [PTR79] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR79, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR79 = RP.JAUS:Expense.CapitalWorksDeduction.Amount
            */
            assertion = NotMonetary(report.PTR79, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalWorksDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430089"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430091
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430091()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430091
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR80] <> NULL AND [PTR80] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR80, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR80 = RP.JAUS:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = NotMonetary(report.PTR80, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430091"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430092
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430092()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430092
            Gross interest amount greater than zero must be present
    
            Legacy Rule Format:
            IF [PTR83] > 0 AND ([PTR82] = 0 OR [PTR82] = NULL)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR83 > 0 AND (^PTR82 = 0 OR ^PTR82 = NULL)
    
            Data Elements:
    
            ^PTR82 = RP.JAUS:Income.Interest.Gross.Amount
    
            ^PTR83 = RP.JAUS:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            */
            assertion = (report.PTR83.GetValueOrDefault() > 0 && (report.PTR82 == 0 || report.PTR82 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430092", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest amount greater than zero must be present",
                    LongDescription = @"If an amount greater than zero is present at TFN amounts withheld from gross interest, then an amount greater than zero must also be present at Gross interest",
                    Location = "/xbrli:xbrl/tns:Income.Interest.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430092"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430093
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430093()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430093
            TFN amounts withheld from dividends must not exceed 50% of Gross interest
    
            Legacy Rule Format:
            IF [PTR83] > 0 AND ([PTR82] * 0.5) <= [PTR83] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR83 > 0 AND (^PTR82 * 0.5) <= ^PTR83
    
            Data Elements:
    
            ^PTR83 = RP.JAUS:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^PTR82 = RP.JAUS:Income.Interest.Gross.Amount
            */
            assertion = (report.PTR83.GetValueOrDefault() > 0 && report.PTR82.GetValueOrDefault() * 0.5M <= report.PTR83.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430093", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must not exceed 50% of Gross interest",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430093"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430094
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430094()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430094
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [PTR83] <> NULL AND [PTR83] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR83, 'U', 13, 2)
    
            Data Elements:
    
            ^PTR83 = RP.JAUS:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            */
            assertion = NotMonetary(report.PTR83, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430094"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430095
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430095()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430095
            Dividends amounts must be present
    
            Legacy Rule Format:
            IF [PTR87] > 0 AND ([PTR84] = 0 OR [PTR84] = NULL) AND ([PTR85] = 0 OR [PTR85] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR87 > 0 AND (^PTR84 = 0 OR ^PTR84 = NULL) AND (^PTR85 = 0 OR ^PTR85 = NULL)
    
            Data Elements:
    
            ^PTR84 = RP.JAUS:Income.DividendsUnfranked.Amount
    
            ^PTR87 = RP.JAUS:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR87.GetValueOrDefault() > 0 && (report.PTR84 == 0 || report.PTR84 == null) && (report.PTR85 == 0 || report.PTR85 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430095", Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends amounts must be present",
                    LongDescription = @"If an amount greater than zero is present at TFN amounts withheld from dividends, then there must also be an amount greater than zero at Unfranked amount or Franked amount",
                    Location = "/xbrli:xbrl/tns:Income.DividendsUnfranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430095"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430096
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430096()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430096
            TFN amount withheld must not exceed unfranked dividends plus 50% of franked dividends
    
            Legacy Rule Format:
            IF ([PTR87] > 0 AND [PTR84] > 0 AND [PTR85] > 0) AND ([PTR84] + (([PTR85]*0.5) + 1)) < [PTR87] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR87 > 0 AND ^PTR84 > 0 AND ^PTR85 > 0) AND (^PTR84 + ((^PTR85*0.5) + 1)) < ^PTR87
    
            Data Elements:
    
            ^PTR84 = RP.JAUS:Income.DividendsUnfranked.Amount
    
            ^PTR87 = RP.JAUS:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR87.GetValueOrDefault() > 0 && report.PTR84.GetValueOrDefault() > 0 && report.PTR85.GetValueOrDefault() > 0 && report.PTR84.GetValueOrDefault() + (report.PTR85.GetValueOrDefault() * 0.5M + 1) < report.PTR87.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430096", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amount withheld must not exceed unfranked dividends plus 50% of franked dividends",
                    LongDescription = @"The amount at TFN amounts withheld from dividends must not be greater than Unfranked amount plus 50% of Franked amount",
                    Location = "/xbrli:xbrl/tns:Income.DividendsUnfranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430096"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430097
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430097()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430097
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR84] <> NULL AND [PTR84] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR84, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR84 = RP.JAUS:Income.DividendsUnfranked.Amount
            */
            assertion = NotMonetary(report.PTR84, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.DividendsUnfranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430097"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430098
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430098()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430098
            Franking credit cannot be greater than 48% of dividends-franked amount.
    
            Legacy Rule Format:
            IF [PTR86] > 0 AND (([PTR85] * 0.48) < ([PTR86] - 1)) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR86 > 0 AND ((^PTR85 * 0.48) < (^PTR86 - 1))
    
            Data Elements:
    
            ^PTR86 = RP.JAUS:IncomeTax.FrankingCredits.Amount
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR86.GetValueOrDefault() > 0 && report.PTR85.GetValueOrDefault() * 0.48M < report.PTR86.GetValueOrDefault() - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430098", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit cannot be greater than 48% of dividends-franked amount.",
                    LongDescription = @"Franking credit must not exceed 48% of Dividends - Franked amount",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430098"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430099
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430099()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430099
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR85] <> NULL AND [PTR85] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR85, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
            */
            assertion = NotMonetary(report.PTR85, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.DividendsFranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430099"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430100
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430100()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430100
            Franking credit amount must be present
    
            Legacy Rule Format:
            IF [PTR85] > 0 AND [PTR86] = NULL 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR85 > 0 AND ^PTR86 = NULL
    
            Data Elements:
    
            ^PTR86 = RP.JAUS:IncomeTax.FrankingCredits.Amount
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR85.GetValueOrDefault() > 0 && report.PTR86 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430100", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit amount must be present",
                    LongDescription = @"If Dividends - Franked amount is present, then Franking credit must be present",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430101
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430101()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430101
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR86] <> NULL AND [PTR86] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR86, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR86 = RP.JAUS:IncomeTax.FrankingCredits.Amount
            */
            assertion = NotMonetary(report.PTR86, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430101"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430102
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430102()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430102
            TFN amounts withheld from dividends must not exceed 50% of the Unfranked amount
    
            Legacy Rule Format:
            IF [PTR87] > 0 AND (([PTR84]*0.5) + 1) < [PTR87] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR87 > 0 AND ((^PTR84*0.5) + 1) < ^PTR87
    
            Data Elements:
    
            ^PTR87 = RP.JAUS:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^PTR84 = RP.JAUS:Income.DividendsUnfranked.Amount
            */
            assertion = (report.PTR87.GetValueOrDefault() > 0 && report.PTR84.GetValueOrDefault() * 0.5M + 1 < report.PTR87.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430102", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must not exceed 50% of the Unfranked amount",
                    Location = "/xbrli:xbrl/tns:IncomeTax.DividendsTFNAmountsWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430102"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430103
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430103()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430103
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [PTR87] <> NULL AND [PTR87] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR87, 'U', 13, 2)
    
            Data Elements:
    
            ^PTR87 = RP.JAUS:IncomeTax.DividendsTFNAmountsWithheld.Amount
            */
            assertion = NotMonetary(report.PTR87, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.DividendsTFNAmountsWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430103"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430104
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430104(PTR2024.OtherAustralianIncome otherAustralianIncome, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430104
                    Other Australian income - type and amount must both be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:OtherAustralianIncome)
                    IF (bafpr1.xx.xx:Income.Other.Description <> NULLORBLANK OR bafpr1.xx.xx:Income.Other.Amount <> NULL) AND (bafpr1.xx.xx:Income.Other.Description = BLANK OR bafpr1.xx.xx:Income.Other.Amount  = NULL)
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR88 <> BLANK OR ^PTR89 <> NULL) AND (^PTR88 = BLANK OR ^PTR89  = NULL)
            
                    Data Elements:
            
                    ^PTR89 = RP.JAUS:OtherAustralianIncome:Income.Other.Amount
            
                    ^PTR88 = RP.JAUS:OtherAustralianIncome:Income.Other.Description
                    */
                    assertion = ((string.IsNullOrWhiteSpace(otherAustralianIncome.PTR88) != true || otherAustralianIncome.PTR89 != null) && (string.IsNullOrWhiteSpace(otherAustralianIncome.PTR88) == true || otherAustralianIncome.PTR89 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.430104", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Australian income - type and amount must both be present",
                            LongDescription = @"If Other Australian income - Type of income is present, then the income amount must be present. If Other Australian income - amount is present, then the Type of income must be present.",
                            Location = "/xbrli:xbrl/tns:OtherAustralianIncome" + OccurrenceIndex(otherAustralianIncome.OccurrenceIndex) + "/tns:Income.Other.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430104"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR88", Value = otherAustralianIncome.PTR88 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR89", Value = GetValueOrEmpty(otherAustralianIncome.PTR89) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430105
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430105(PTR2024.OtherAustralianIncome otherAustralianIncome, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430105
                    Monetary field may be signed, maximum of 11 chars, no decimals
    
                    Legacy Rule Format:
                    IF [PTR89] <> NULL AND [PTR89] <> MONETARY(S,11,0)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR89, 'S', 11, 0)
            
                    Data Elements:
            
                    ^PTR89 = RP.JAUS:OtherAustralianIncome:Income.Other.Amount
                    */
                    assertion = NotMonetary(otherAustralianIncome.PTR89, @"S", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:OtherAustralianIncome" + OccurrenceIndex(otherAustralianIncome.OccurrenceIndex) + "/tns:Income.Other.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430105"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR89", Value = GetValueOrEmpty(otherAustralianIncome.PTR89) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430106
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430106()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430106
            Gross interest or dividend income must be present
    
            Legacy Rule Format:
            IF [PTR90] > 0 AND ([PTR82] = NULL AND [PTR84] = NULL AND [PTR85] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR90 > 0 AND (^PTR82 = NULL AND ^PTR84 = NULL AND ^PTR85 = NULL)
    
            Data Elements:
    
            ^PTR82 = RP.JAUS:Income.Interest.Gross.Amount
    
            ^PTR90 = RP.JAUS:Expense.AustralianInvestmentAllowableDeduction.Amount
    
            ^PTR84 = RP.JAUS:Income.DividendsUnfranked.Amount
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR90.GetValueOrDefault() > 0 && (report.PTR82 == null && report.PTR84 == null && report.PTR85 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430106", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest or dividend income must be present",
                    LongDescription = @"If Deductions relating to Australian investment income is greater than zero, then Gross interest, Unfranked amount, or Franked amount must be present",
                    Location = "/xbrli:xbrl/tns:Income.Interest.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430106"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR90", Value = GetValueOrEmpty(report.PTR90) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430107
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430107()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430107
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR90] <> NULL AND [PTR90] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR90, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR90 = RP.JAUS:Expense.AustralianInvestmentAllowableDeduction.Amount
            */
            assertion = NotMonetary(report.PTR90, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.AustralianInvestmentAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430107"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR90", Value = GetValueOrEmpty(report.PTR90) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430108
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430108(PTR2024.OtherDeductions otherDeductions, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430108
                    Other deductions - Name of each item of deduction must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:OtherDeductions)
                    IF (bafpr1.xx.xx:Expense.DeductibleOther.Amount > 0) AND (bafpr1.xx.xx:Expense.DeductibleOther.Text = BLANK)
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR96 > 0) AND (^PTR95 = BLANK)
            
                    Data Elements:
            
                    ^PTR95 = RP.JAUS:OtherDeductions:Expense.DeductibleOther.Text
            
                    ^PTR96 = RP.JAUS:OtherDeductions:Expense.DeductibleOther.Amount
                    */
                    assertion = (otherDeductions.PTR96.GetValueOrDefault() > 0 && string.IsNullOrWhiteSpace(otherDeductions.PTR95) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430108", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other deductions - Name of each item of deduction must be present",
                            LongDescription = @"If Other deductions - Amount is present, the Name of each item of deduction must be present",
                            Location = "/xbrli:xbrl/tns:OtherDeductions" + OccurrenceIndex(otherDeductions.OccurrenceIndex) + "/tns:Expense.DeductibleOther.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430108"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR96", Value = GetValueOrEmpty(otherDeductions.PTR96) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR95", Value = otherDeductions.PTR95 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430109
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430109()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430109
            Other deductions must not exceed 99 instances
    
            Legacy Rule Format:
            IF COUNT(ptr.0012.lodge.req.xx.xx:OtherDeductions) > 99
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^OtherDeductions) > 99
    
            Data Elements:
    
            ^OtherDeductions = RP.JAUS:OtherDeductions
            */
            assertion = (Count(report.OtherDeductionsCollectionCount) > 99);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430109", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other deductions must not exceed 99 instances",
                    Location = "/xbrli:xbrl/tns:OtherDeductions" + OccurrenceIndex(report.OtherDeductionsCollection == null ? 0 : (report.OtherDeductionsCollection.Count() == 0 ? 0 : report.OtherDeductionsCollection.Last().OccurrenceIndex)) + "/tns:Expense.DeductibleOther.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430109"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430110
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430110(PTR2024.OtherDeductions otherDeductions, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430110
                    Other deductions amount must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:OtherDeductions)
                    IF [PTR95] <> NULLORBLANK AND ([PTR96] = 0) 
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR95 <> BLANK AND (^PTR96 = 0)
            
                    Data Elements:
            
                    ^PTR96 = RP.JAUS:OtherDeductions:Expense.DeductibleOther.Amount
            
                    ^PTR95 = RP.JAUS:OtherDeductions:Expense.DeductibleOther.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(otherDeductions.PTR95) != true && otherDeductions.PTR96 == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430110", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other deductions amount must be present",
                            LongDescription = @"If Name of each item of deduction is present, then the Other deductions - Amount must be present and greater than zero",
                            Location = "/xbrli:xbrl/tns:OtherDeductions" + OccurrenceIndex(otherDeductions.OccurrenceIndex) + "/tns:Expense.DeductibleOther.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430110"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR95", Value = otherDeductions.PTR95 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR96", Value = GetValueOrEmpty(otherDeductions.PTR96) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430112
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430112()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430112
            Net Australian income must equal income less deductions
    
            Legacy Rule Format:
            IF [PTR97] <> NULL AND [PTR97] <> ([PTR61] + [PTR62] + [PTR65] + [PTR66] - [PTR68] + [PTR69] + [PTR70] - [PTR72] + [PTR322] - [PTR323] + [PTR77] - [PTR78] - [PTR79] - [PTR80] + [PTR81] + [PTR82] + [PTR84] + [PTR85] + [PTR86] + SUM([PTR89])) - ([PTR90] + [PTR91] + SUM([PTR96])) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR97 <> NULL AND ^PTR97 <> (^PTR61 + ^PTR62 + ^PTR65 + ^PTR66 - ^PTR68 + ^PTR69 + ^PTR70 - ^PTR72 + ^PTR322 - ^PTR323 + ^PTR77 - ^PTR78 - ^PTR79 - ^PTR80 + ^PTR81 + ^PTR82 + ^PTR84 + ^PTR85 + ^PTR86 + Sum(^PTR89)) - (^PTR90 + ^PTR91 + Sum(^PTR96))
    
            Data Elements:
    
            ^PTR97 = RP.JAUS:Income.Net.Amount
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
    
            ^PTR65 = RP.JAUS.Prim:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount
    
            ^PTR68 = RP.JAUS.Prim:Expense.DistributionDeductionsOther.Amount
    
            ^PTR69 = RP.JAUS.NonPrim:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR70 = RP.JAUS.NonPrim:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR72 = RP.JAUS.NonPrim:Expense.DistributionDeductionsOther.Amount
    
            ^PTR322 = RP.JAUS.NonPrim:Income.TrustDistributionFranked.Amount
    
            ^PTR323 = RP.JAUS.NonPrim:IncomeTax.Deduction.TrustDistributionFranked.Amount
    
            ^PTR77 = RP.JAUS:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR78 = RP.JAUS:Expense.Interest.Amount
    
            ^PTR79 = RP.JAUS:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = RP.JAUS:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
            ^PTR81 = RP.JAUS:Income.ForestryManagedInvestmentScheme.Amount
    
            ^PTR82 = RP.JAUS:Income.Interest.Gross.Amount
    
            ^PTR84 = RP.JAUS:Income.DividendsUnfranked.Amount
    
            ^PTR85 = RP.JAUS:Income.DividendsFranked.Amount
    
            ^PTR86 = RP.JAUS:IncomeTax.FrankingCredits.Amount
    
            ^PTR89 = RP.JAUS:OtherAustralianIncome:Income.Other.Amount
    
            ^PTR90 = RP.JAUS:Expense.AustralianInvestmentAllowableDeduction.Amount
    
            ^PTR91 = RP.JAUS:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^PTR96 = RP.JAUS:OtherDeductions:Expense.DeductibleOther.Amount
            */
            assertion = (report.PTR97 != null && report.PTR97.GetValueOrDefault() != report.PTR61.GetValueOrDefault() + report.PTR62.GetValueOrDefault() + report.PTR65.GetValueOrDefault() + report.PTR66.GetValueOrDefault() - report.PTR68.GetValueOrDefault() + report.PTR69.GetValueOrDefault() + report.PTR70.GetValueOrDefault() - report.PTR72.GetValueOrDefault() + report.PTR322.GetValueOrDefault() - report.PTR323.GetValueOrDefault() + report.PTR77.GetValueOrDefault() - report.PTR78.GetValueOrDefault() - report.PTR79.GetValueOrDefault() - report.PTR80.GetValueOrDefault() + report.PTR81.GetValueOrDefault() + report.PTR82.GetValueOrDefault() + report.PTR84.GetValueOrDefault() + report.PTR85.GetValueOrDefault() + report.PTR86.GetValueOrDefault() + (report.OtherAustralianIncomeCollection == null ? 0 : report.OtherAustralianIncomeCollection.Sum(f => f.PTR89.GetValueOrDefault())) - (report.PTR90.GetValueOrDefault() + report.PTR91.GetValueOrDefault() + (report.OtherDeductionsCollection == null ? 0 : report.OtherDeductionsCollection.Sum(f => f.PTR96.GetValueOrDefault()))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430112", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Australian income must equal income less deductions",
                    LongDescription = @"The Net Australian income or loss amount must equal the sum of all income less the sum of all deductions",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430112"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR97", Value = GetValueOrEmpty(report.PTR97) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR81", Value = GetValueOrEmpty(report.PTR81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR89", Value = "PTR89" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR90", Value = GetValueOrEmpty(report.PTR90) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR91", Value = GetValueOrEmpty(report.PTR91) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR96", Value = "PTR96" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430113
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430113()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430113
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [PTR97] <> NULL AND [PTR97] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR97, 'S', 11, 0)
    
            Data Elements:
    
            ^PTR97 = RP.JAUS:Income.Net.Amount
            */
            assertion = NotMonetary(report.PTR97, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430113"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR97", Value = GetValueOrEmpty(report.PTR97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430116
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430116()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430116
            Answer to Attributed foreign income question must be 'yes'
    
            Legacy Rule Format:
            IF ([PTR98] > 0 OR [PTR100] > 0) AND [PTR102] <> TRUE 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR98 > 0 OR ^PTR100 > 0) AND ^PTR102 <> TRUE
    
            Data Elements:
    
            ^PTR102 = RP.JFOREIGN:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
    
            ^PTR98 = RP.JFOREIGN.CFCLC:Income.InternationalDealings.Attributed.Amount
    
            ^PTR100 = RP.JFOREIGN.CFCUC:Income.InternationalDealings.Attributed.Amount
            */
            assertion = ((report.PTR98.GetValueOrDefault() > 0 || report.PTR100.GetValueOrDefault() > 0) && report.PTR102 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430116", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Attributed foreign income question must be 'yes'",
                    LongDescription = @"If Attributed foreign income - Listed country, Section 404 country, or Unlisted country is greater than zero, then a yes (true) response to the question ""Did you have either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust?"" is required",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430116"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR98", Value = GetValueOrEmpty(report.PTR98) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR100", Value = GetValueOrEmpty(report.PTR100) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR102", Value = GetValueOrEmpty(report.PTR102) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430120
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430120()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430120
            Other assessable foreign source income 'Gross' amount must not be less than 'Net' amount
    
            Legacy Rule Format:
            IF [PTR104] < [PTR105] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR104 < ^PTR105
    
            Data Elements:
    
            ^PTR104 = RP.JFOREIGN:Income.Other.Amount
    
            ^PTR105 = RP.JFOREIGN:Income.Net.Amount
            */
            assertion = (report.PTR104.GetValueOrDefault() < report.PTR105.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430120", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other assessable foreign source income 'Gross' amount must not be less than 'Net' amount",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430120"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR104", Value = GetValueOrEmpty(report.PTR104) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR105", Value = GetValueOrEmpty(report.PTR105) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430127
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430127()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430127
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR114] <> NULL AND [PTR114] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR114, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR114 = RP.Closing(Instant):Liabilities.Total.Amount
            */
            assertion = NotMonetary(report.PTR114, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430127"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR114", Value = GetValueOrEmpty(report.PTR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430129
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430129()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430129
            Business name and Business address of main business must be present
    
            Legacy Rule Format:
            IF ([PTR35] <> NULL OR [PTR36] <> NULL OR [PTR38] <> NULL OR [PTR39] <> NULL OR [PTR41] <> NULL OR [PTR43] <> NULL OR [PTR44] <> NULL) AND ([PTR116] = NULLORBLANK OR [PTR117] = NULLORBLANK) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR35 <> NULL OR ^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR39 <> NULL OR ^PTR41 <> NULL OR ^PTR43 <> NULL OR ^PTR44 <> NULL) AND (^PTR116 = BLANK OR ^PTR117 = BLANK)
    
            Data Elements:
    
            ^PTR116 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MTR'].[OrganisationNameDetails.Currency.Code='C']
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
    
            ^PTR117 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.PTR35 != null || report.PTR36 != null || report.PTR38 != null || report.PTR39 != null || report.PTR41 != null || report.PTR43 != null || report.PTR44 != null) && (string.IsNullOrWhiteSpace(report.PTR116) == true || string.IsNullOrWhiteSpace(report.PTR117) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430129", Severity = ProcessMessageSeverity.Error,
                    Description = @"Business name and business address of main business must be present",
                    LongDescription = @"If Total business income (Primary production or Non-primary production) is present at business income and expenses, then both the Business name and Business address of main business must be present",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430129"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR116", Value = report.PTR116 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR117", Value = report.PTR117 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430130
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430130()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430130
            Business and professional amounts must be present
    
            Legacy Rule Format:
            IF ([PTR61] <> NULL OR [PTR62] <> NULL) 
            AND (([PTR111] = NULL) 
            AND ([PTR112] = NULL) 
            AND ([PTR113] = NULL)
            AND ([PTR114] = NULL)
            AND ([PTR121] = NULL)
            AND ([PTR122] = NULL)
            AND ([PTR123] = NULL)
            AND ([PTR125] = NULL)
            AND ([PTR126] = NULL)
            AND ([PTR127] = NULL)
            AND ([PTR129] = NULL)
            AND ([PTR136] = NULL)) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR61 <> NULL OR ^PTR62 <> NULL) AND ((^PTR111 = NULL) AND (^PTR112 = NULL) AND (^PTR113 = NULL) AND (^PTR114 = NULL) AND (^PTR121 = NULL) AND (^PTR122 = NULL) AND (^PTR123 = NULL) AND (^PTR125 = NULL) AND (^PTR126 = NULL) AND (^PTR127 = NULL) AND (^PTR129 = NULL) AND (^PTR136 = NULL))
    
            Data Elements:
    
            ^PTR121 = RP.Opening(Instant):Assets.Inventories.Total.Amount
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR62 = RP.JAUS.NonPrim:Income.Net.Amount
    
            ^PTR111 = RP.Closing(Instant):Assets.CurrentTotal.Amount
    
            ^PTR112 = RP.Closing(Instant):Assets.Total.Amount
    
            ^PTR113 = RP.Closing(Instant):Liabilities.CurrentTotal.Amount
    
            ^PTR114 = RP.Closing(Instant):Liabilities.Total.Amount
    
            ^PTR122 = RP:Expense.Purchases.AndOtherCosts.Amount
    
            ^PTR123 = RP.Closing(Instant):Assets.Inventories.Total.Amount
    
            ^PTR125 = RP.Closing(Instant):Assets.TradeandReceivablesOther.Amount
    
            ^PTR126 = RP.Closing(Instant):Liabilities.TradeAndOtherPayablesTotal.Amount
    
            ^PTR127 = RP:Remuneration.WagesAndSalaries.Total.Amount
    
            ^PTR129 = RP:Remuneration.PaymentToRelatedPartiesGross.Amount
    
            ^PTR136 = RP:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = ((report.PTR61 != null || report.PTR62 != null) && (report.PTR111 == null && report.PTR112 == null && report.PTR113 == null && report.PTR114 == null && report.PTR121 == null && report.PTR122 == null && report.PTR123 == null && report.PTR125 == null && report.PTR126 == null && report.PTR127 == null && report.PTR129 == null && report.PTR136 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.439000", Severity = ProcessMessageSeverity.Error,
                    Description = @"Key financial information amounts and/or Business and professional amounts must be present",
                    LongDescription = @"If Net income or loss from business (either Primary production or Non-primary production) is present, then there must be an amount included under Assets, Liabilities or under Business and professional items",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.Total.Amount[@contextRef='" + report.RPOpeningInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430130"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR111", Value = GetValueOrEmpty(report.PTR111) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR112", Value = GetValueOrEmpty(report.PTR112) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR113", Value = GetValueOrEmpty(report.PTR113) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR114", Value = GetValueOrEmpty(report.PTR114) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR121", Value = GetValueOrEmpty(report.PTR121) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR122", Value = GetValueOrEmpty(report.PTR122) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR123", Value = GetValueOrEmpty(report.PTR123) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR125", Value = GetValueOrEmpty(report.PTR125) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR126", Value = GetValueOrEmpty(report.PTR126) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR127", Value = GetValueOrEmpty(report.PTR127) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR129", Value = GetValueOrEmpty(report.PTR129) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR136", Value = GetValueOrEmpty(report.PTR136) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430131
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430131()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430131
            Total salary and wage expenses amount must be present if the action code is present
    
            Legacy Rule Format:
            IF [PTR128] <> NULL AND ([PTR127] = 0 OR [PTR127] = NULL) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR128 <> NULL AND (^PTR127 = 0 OR ^PTR127 = NULL)
    
            Data Elements:
    
            ^PTR127 = RP:Remuneration.WagesAndSalaries.Total.Amount
    
            ^PTR128 = RP:Remuneration.WagesAndSalariesAction.Code
            */
            assertion = (report.PTR128 != null && (report.PTR127 == 0 || report.PTR127 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses amount must be present if the action code is present",
                    LongDescription = @"If a Total salary and wage expenses action code is present an amount must be present. Either enter an amount or delete the action code",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalaries.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430131"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR128", Value = GetValueOrEmpty(report.PTR128) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR127", Value = GetValueOrEmpty(report.PTR127) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430132
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430132()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430132
            Total salary and wage expenses action code must be present
    
            Legacy Rule Format:
            IF [PTR127] > 0 AND [PTR128] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR127 > 0 AND ^PTR128 = NULL
    
            Data Elements:
    
            ^PTR128 = RP:Remuneration.WagesAndSalariesAction.Code
    
            ^PTR127 = RP:Remuneration.WagesAndSalaries.Total.Amount
            */
            assertion = (report.PTR127.GetValueOrDefault() > 0 && report.PTR128 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses action code must be present",
                    LongDescription = @"If a Total salary and wage expenses amount is present, then an action code must be present",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalariesAction.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430132"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR127", Value = GetValueOrEmpty(report.PTR127) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR128", Value = GetValueOrEmpty(report.PTR128) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430134()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430134
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [PTR138] <> NULL AND [PTR138] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^PTR138, 'U', 11, 0)
    
            Data Elements:
    
            ^PTR138 = RP.JFOREIGN:Expense.Royalties.Amount
            */
            assertion = NotMonetary(report.PTR138, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430134"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR138", Value = GetValueOrEmpty(report.PTR138) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430136
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430136(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430136
                    Statement of distribution - name must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF [PTR146] = NULLORBLANK AND [PTR149] = NULLORBLANK 
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR146 = BLANK AND ^PTR149 = BLANK
            
                    Data Elements:
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR146) == true && string.IsNullOrWhiteSpace(statementOfDistribution.PTR149) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430136", Severity = ProcessMessageSeverity.Error,
                            Description = @"Statement of distribution - name must be present",
                            LongDescription = @"If the partner/beneficiary tax file number or current postal address is present in the Statement of distribution, then an individual  name or non-individual name must be present",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430136"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430139
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430139(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430139
                    Statement of distribution must not contain both an individual name and a non-individual name
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF [PTR146] <> NULLORBLANK AND [PTR149] <> NULLORBLANK
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR146 <> BLANK AND ^PTR149 <> BLANK
            
                    Data Elements:
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR146) != true && string.IsNullOrWhiteSpace(statementOfDistribution.PTR149) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430139", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both family name and company name present",
                            LongDescription = @"Family name and non-individual name line 1 are both present in the distribution statement (Item 65). Please delete the incorrect name as only one individual or non-individual name may be present.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430139"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430141
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430141(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430141
                    Statement of distribution -  Individual's  residential address  or Non-Individual's business address or Tax File Number or Australian Business Number must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF [PTR332] = NULLORBLANK AND [PTR157] = NULL AND [PTR151] = NULLORBLANK AND [PTR285] = NULL
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR332 = BLANK AND ^PTR157 = NULL AND ^PTR151 = BLANK AND ^PTR285 = NULL
            
                    Data Elements:
            
                    ^PTR157 = RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR285 = RP:StatementOfDistribution:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR332) == true && statementOfDistribution.PTR157 == null && string.IsNullOrWhiteSpace(statementOfDistribution.PTR151) == true && statementOfDistribution.PTR285 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.431053", Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual's  residential address  or Non-Individual's business address or Tax File Number or Australian Business Number  must be present in each Statement of Distribution",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Identifiers.TaxFileNumber.Identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430141"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430142
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430142()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430142
            Tax file number must not be the TFN of the Trust or Partnership
    
            Legacy Rule Format:
            IF (ANY OCCURRENCE OF[PTR157]) = RP:entity.identfier.TFN
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (AnyOccurrence(^PTR157, ^PTR157 = ^TFN))
    
            Data Elements:
    
            ^PTR157 = RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier
    
            ^TFN = RP
            */
            assertion = (report.StatementOfDistributionCollection == null ? false : report.StatementOfDistributionCollection.Any(PTR157Repeat => PTR157Repeat.PTR157 == report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax file number must not be the TFN of the Trust or Partnership",
                    LongDescription = @"The TFN provided for a Trustee or Partner in a Statement of distribution must not be the same as the TFN of the Trust or Partnership",
                    Location = "/xbrli:xbrl/tns:StatementOfDistribution/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430142"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR157", Value = "PTR157" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430143
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430143(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430143
                    PP, Non-PP or Franked distribution required.
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF [PTR160] > 0 AND [PTR158] = NULL AND [PTR159] = NULL AND [PTR324] = NULL 
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR160 > 0 AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL
            
                    Data Elements:
            
                    ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR160 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^PTR159 = RP.NonPrim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR324 = RP:StatementOfDistribution:Income.DistributionFranked.Amount
                    */
                    assertion = (statementOfDistribution.PTR160.GetValueOrDefault() > 0 && statementOfDistribution.PTR158 == null && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.430707", Severity = ProcessMessageSeverity.Error,
                            Description = @"PP, Non-PP or Franked distributions required",
                            LongDescription = @"Credit for tax withheld where ABN not quoted is greater than zero and there is no Primary production or Non-primary production income distribution, or Franked Distributions",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPPrimId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430143"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR160", Value = GetValueOrEmpty(statementOfDistribution.PTR160) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430144
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430144()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430144
            Distribution of PP income cannot exceed net PP income
    
            Legacy Rule Format:
            IF SUM([PTR158]) > ([PTR61] + [PTR65] + [PTR66] - [PTR68])
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Sum(^PTR158) > (^PTR61 + ^PTR65 + ^PTR66 - ^PTR68)
    
            Data Elements:
    
            ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
    
            ^PTR61 = RP.JAUS.Prim:Income.Net.Amount
    
            ^PTR65 = RP.JAUS.Prim:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount
    
            ^PTR68 = RP.JAUS.Prim:Expense.DistributionDeductionsOther.Amount
            */
            assertion = ((report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR158.GetValueOrDefault())) > report.PTR61.GetValueOrDefault() + report.PTR65.GetValueOrDefault() + report.PTR66.GetValueOrDefault() - report.PTR68.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430144", Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution of PP income cannot exceed net PP income",
                    LongDescription = @"The amount of primary production income in the distribution statement (Item 65 Label A) is greater than the net primary production income received - net primary production income (Item 5 Label Q) plus primary production distribution from partnerships (Item 8 Label A) plus primary production distribution from trusts (Item 8 Label Z) less primary production deductions relating to distribution (Item 8 Label S).",
                    Location = "/xbrli:xbrl/tns:StatementOfDistribution/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPPrimId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430144"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR158", Value = "PTR158" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430145
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430145()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430145
            Net Australian income/loss must equal total Distributed PP and non-PP income
    
            Legacy Rule Format:
            IF [PTR97] <> NULL AND [PTR97] <> SUM([PTR158]) + SUM([PTR159]) + SUM([PTR324])
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR97 <> NULL AND ^PTR97 <> Sum(^PTR158) + Sum(^PTR159) + Sum(^PTR324)
    
            Data Elements:
    
            ^PTR97 = RP.JAUS:Income.Net.Amount
    
            ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
    
            ^PTR159 = RP.NonPrim:StatementOfDistribution:Income.BeneficiaryShare.Amount
    
            ^PTR324 = RP:StatementOfDistribution:Income.DistributionFranked.Amount
            */
            assertion = (report.PTR97 != null && report.PTR97.GetValueOrDefault() != (report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR158.GetValueOrDefault())) + (report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR159.GetValueOrDefault())) + (report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR324.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430706", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Australian income/loss must equal total Distributed PP, non-PP income and Franked distributions",
                    LongDescription = @"The Net Australian income/loss amount must equal the sum of Share of income - Primary production plus Share of income - Non-primary production in all Statements of distribution plus Franked distributions",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430145"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR97", Value = GetValueOrEmpty(report.PTR97) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR158", Value = "PTR158" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR159", Value = "PTR159" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR324", Value = "PTR324" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430146
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430146(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430146
                    Partner distribution amount(s) missing
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF(
                    ([PTR146] <> NULLORBLANK) OR 
                    ([PTR149] <> NULLORBLANK) OR 
                    ([PTR151] <> NULLORBLANK) OR 
                    ([PTR157] <> NULLORBLANK) OR 
                    ([PTR332] <> NULLORBLANK)
                    ) AND
                    ([PTR158] = NULL) AND 
                    ([PTR159] = NULL) AND 
                    ([PTR324] = NULL) AND 
                    ([PTR161] = NULL) AND 
                    ([PTR162] = NULL) AND 
                    ([PTR164] = NULL) 
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ((^PTR146 <> BLANK) OR (^PTR149 <> BLANK) OR (^PTR151 <> BLANK) OR (^PTR157 <> BLANK) OR (^PTR332 <> BLANK)) AND (^PTR158 = NULL) AND (^PTR159 = NULL) AND (^PTR324 = NULL) AND (^PTR161 = NULL) AND (^PTR162 = NULL) AND (^PTR164 = NULL)
            
                    Data Elements:
            
                    ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR146 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']
            
                    ^PTR149 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            
                    ^PTR151 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            
                    ^PTR157 = RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier
            
                    ^PTR332 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']
            
                    ^PTR159 = RP.NonPrim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR324 = RP:StatementOfDistribution:Income.DistributionFranked.Amount
            
                    ^PTR161 = RP:StatementOfDistribution:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^PTR162 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^PTR164 = RP:StatementOfDistribution:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
                    */
                    assertion = ((string.IsNullOrWhiteSpace(statementOfDistribution.PTR146) != true || string.IsNullOrWhiteSpace(statementOfDistribution.PTR149) != true || string.IsNullOrWhiteSpace(statementOfDistribution.PTR151) != true || string.IsNullOrWhiteSpace(statementOfDistribution.PTR157) != true || string.IsNullOrWhiteSpace(statementOfDistribution.PTR332) != true) && statementOfDistribution.PTR158 == null && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null && statementOfDistribution.PTR161 == null && statementOfDistribution.PTR162 == null && statementOfDistribution.PTR164 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.431137", Severity = ProcessMessageSeverity.Error,
                            Description = @"Partner distribution amount(s) must be provided",
                            LongDescription = @"There must be an amount provided for the Partner distribution amount(s) - Share of income Primary Production, Share of income Non-primary production, Franked Distributions, Franking credit, TFN amounts withheld or Australian franking credits from a New Zealand company, for each partner listed in the Statement of distribution.",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPPrimId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430146"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR146", Value = statementOfDistribution.PTR146 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR149", Value = statementOfDistribution.PTR149 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR151", Value = statementOfDistribution.PTR151 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR332", Value = statementOfDistribution.PTR332 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR161", Value = GetValueOrEmpty(statementOfDistribution.PTR161) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR162", Value = GetValueOrEmpty(statementOfDistribution.PTR162) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR164", Value = GetValueOrEmpty(statementOfDistribution.PTR164) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430147
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430147(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430147
                    Monetary field may be signed, maximum of 11 chars, no decimals
    
                    Legacy Rule Format:
                    IF [PTR158] <> NULL AND [PTR158] <> MONETARY(S,11,0)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR158, 'S', 11, 0)
            
                    Data Elements:
            
                    ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR158, @"S", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPPrimId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430147"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430148
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430148(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430148
                    Share of income - Non-PP or Franked distributions must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF ([PTR160] <> NULL OR [PTR161] <> NULL OR [PTR162] <> NULL OR [PTR351] <> NULL) AND [PTR159] = NULL AND [PTR324] = NULL
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PTR160 <> NULL OR ^PTR161 <> NULL OR ^PTR162 <> NULL OR ^PTR351 <> NULL) AND ^PTR159 = NULL AND ^PTR324 = NULL
            
                    Data Elements:
            
                    ^PTR159 = RP.NonPrim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR160 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^PTR161 = RP:StatementOfDistribution:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^PTR162 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^PTR351 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            
                    ^PTR324 = RP:StatementOfDistribution:Income.DistributionFranked.Amount
                    */
                    assertion = ((statementOfDistribution.PTR160 != null || statementOfDistribution.PTR161 != null || statementOfDistribution.PTR162 != null || statementOfDistribution.PTR351 != null) && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.430710", Severity = ProcessMessageSeverity.Error,
                            Description = @"Share of income - Non-PP income distribution or Franked distributions must be present",
                            LongDescription = @"In the statement of distribution, if there is an amount at either Credit for tax withheld where ABN not quoted, Franking Credit, TFN amounts withheld, or Credit for tax withheld - foreign resident withholding, then there must be an amount at either Share of income - Non-primary production or Franked distributions",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPNonPrimId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430148"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR160", Value = GetValueOrEmpty(statementOfDistribution.PTR160) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR161", Value = GetValueOrEmpty(statementOfDistribution.PTR161) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR162", Value = GetValueOrEmpty(statementOfDistribution.PTR162) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR351", Value = GetValueOrEmpty(statementOfDistribution.PTR351) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430149
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430149(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430149
                    Share of income must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
                    IF [PTR351] > 0 AND [PTR158] = NULL AND [PTR159] = NULL AND [PTR324] = NULL
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^PTR351 > 0 AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL
            
                    Data Elements:
            
                    ^PTR158 = RP.Prim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR351 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            
                    ^PTR159 = RP.NonPrim:StatementOfDistribution:Income.BeneficiaryShare.Amount
            
                    ^PTR324 = RP:StatementOfDistribution:Income.DistributionFranked.Amount
                    */
                    assertion = (statementOfDistribution.PTR351.GetValueOrDefault() > 0 && statementOfDistribution.PTR158 == null && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.430709", Severity = ProcessMessageSeverity.Error,
                            Description = @"Share of income must be present",
                            LongDescription = @"In a Statement of distribution, if Credit for tax withheld - foreign resident withholding is greater than zero, then there must be an amount at Share of income - primary production or Share of income - non-primary production or Franked distibutions",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPPrimId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430149"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR351", Value = GetValueOrEmpty(statementOfDistribution.PTR351) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430150
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430150(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430150
                    Monetary field may be signed, maximum of 11 chars, no decimals
    
                    Legacy Rule Format:
                    IF [PTR159] <> NULL AND [PTR159] <> MONETARY(S,11,0)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR159, 'S', 11, 0)
            
                    Data Elements:
            
                    ^PTR159 = RP.NonPrim:StatementOfDistribution:Income.BeneficiaryShare.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR159, @"S", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:Income.BeneficiaryShare.Amount[@contextRef='" + report.RPNonPrimId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430150"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430151
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430151()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430151
            Tax withheld where ABN not quoted must equal distributed amounts
    
            Legacy Rule Format:
            IF SUM([PTR160]) <> [PTR63] + [PTR73] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Sum(^PTR160) <> ^PTR63 + ^PTR73
    
            Data Elements:
    
            ^PTR63 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR160 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR73 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount
            */
            assertion = ((report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR160.GetValueOrDefault())) != report.PTR63.GetValueOrDefault() + report.PTR73.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430151", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted must equal distributed amounts",
                    LongDescription = @"The total amount of Credit for tax withheld where ABN not quoted in the Statements of distribution, must equal the sum of Tax withheld where ABN not quoted, and Share of credit for tax withheld where ABN not quoted",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430151"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR160", Value = "PTR160" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR73", Value = GetValueOrEmpty(report.PTR73) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430152
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430152(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430152
                    Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
                    Legacy Rule Format:
                    IF [PTR160] <> NULL AND [PTR160] <> MONETARY(U,11,0)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR160, 'U', 11, 0)
            
                    Data Elements:
            
                    ^PTR160 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR160, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430152"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR160", Value = GetValueOrEmpty(statementOfDistribution.PTR160) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430153
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430153()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430153
            Franking credit amounts must equal distributed amounts
    
            Legacy Rule Format:
            IF SUM([PTR161]) <> [PTR74] + [PTR86] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Sum(^PTR161) <> ^PTR74 + ^PTR86
    
            Data Elements:
    
            ^PTR74 = RP.JAUS:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^PTR161 = RP:StatementOfDistribution:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^PTR86 = RP.JAUS:IncomeTax.FrankingCredits.Amount
            */
            assertion = ((report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR161.GetValueOrDefault())) != report.PTR74.GetValueOrDefault() + report.PTR86.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431007", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP - Deductions relating to distribution amounts can not be more than zero",
                    LongDescription = @"If there is an amount present at Non-primary production- Deductions relating to amounts shown at Distribution from partnerships less foreign income or Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions there must be an amount at either Non-primary production - Distribution from partnerships less foreign income or Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430153"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR161", Value = "PTR161" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430154
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430154(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430154
                    Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
                    Legacy Rule Format:
                    IF [PTR161] <> NULL AND [PTR161] <> MONETARY(U,13,2)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR161, 'U', 13, 2)
            
                    Data Elements:
            
                    ^PTR161 = RP:StatementOfDistribution:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR161, @"U", 13, 2);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430154"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR161", Value = GetValueOrEmpty(statementOfDistribution.PTR161) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430155
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430155()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430155
            TFN amounts withheld must equal distributed amounts
    
            Legacy Rule Format:
            IF SUM([PTR162]) <> [PTR75] + [PTR83] + [PTR87] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Sum(^PTR162) <> ^PTR75 + ^PTR83 + ^PTR87
    
            Data Elements:
    
            ^PTR75 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR162 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
            ^PTR83 = RP.JAUS:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^PTR87 = RP.JAUS:IncomeTax.DividendsTFNAmountsWithheld.Amount
            */
            assertion = ((report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR162.GetValueOrDefault())) != report.PTR75.GetValueOrDefault() + report.PTR83.GetValueOrDefault() + report.PTR87.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430155", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld must equal distributed amounts",
                    LongDescription = @"The sum of TFN amounts withheld in the Statements of distribution must equal the sum of Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions, TFN amounts withheld from gross interest, and TFN amounts withheld from dividends",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR162", Value = "PTR162" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430157
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430157(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430157
                    Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
                    Legacy Rule Format:
                    IF [PTR162] <> NULL AND [PTR162] <> MONETARY(U,13,2)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR162, 'U', 13, 2)
            
                    Data Elements:
            
                    ^PTR162 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR162, @"U", 13, 2);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430157"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR162", Value = GetValueOrEmpty(statementOfDistribution.PTR162) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430158
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430158()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430158
            Credit for tax withheld - foreign resident withholding (excluding capital gains) amounts must equal distributed amounts
    
            Legacy Rule Format:
            IF SUM([PTR351]) <> [PTR350] + [PTR76] 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Sum(^PTR351) <> ^PTR350 + ^PTR76
    
            Data Elements:
    
            ^PTR350 = RP.JAUS:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR351 = RP:StatementOfDistribution:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR76 = RP.JAUS:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
            */
            assertion = ((report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR351.GetValueOrDefault())) != report.PTR350.GetValueOrDefault() + report.PTR76.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430158", Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) amounts must equal distributed amounts",
                    LongDescription = @"The total of Credit for tax withheld - foreign resident withholding (excluding capital gains) in the Statements of distribution, must equal the sum of Credit for tax withheld - foreign resident withholding (excluding capital gains), plus Share of credit for tax withheld from foreign resident withholding (excluding capital gains)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430158"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR351", Value = "PTR351" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR76", Value = GetValueOrEmpty(report.PTR76) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430161
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430161(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430161
                    Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
                    Legacy Rule Format:
                    IF [PTR164] <> NULL AND [PTR164] <> MONETARY(U,11,0)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR164, 'U', 11, 0)
            
                    Data Elements:
            
                    ^PTR164 = RP:StatementOfDistribution:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR164, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430161"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR164", Value = GetValueOrEmpty(statementOfDistribution.PTR164) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430166
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430166()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430166
            National rental affordability scheme tax offset amounts
    
            Legacy Rule Format:
            IF [PTR248] <> SUM([PTR249]) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR248 <> Sum(^PTR249)
    
            Data Elements:
    
            ^PTR248 = RP:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
            ^PTR249 = RP:StatementOfDistribution:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
            */
            assertion = (report.PTR248.GetValueOrDefault() != (report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR249.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430166", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total National rental affordability scheme tax offset amount must equal sum of distributed amounts",
                    LongDescription = @"The National rental affordability scheme tax offset entitlement amount must equal the sum of the Share of National rental affordability scheme tax offset amounts in the Statements of distribution",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430166"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR248", Value = GetValueOrEmpty(report.PTR248) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR249", Value = "PTR249" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430167
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430167(PTR2024.StatementOfDistribution statementOfDistribution, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430167
                    Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
                    Legacy Rule Format:
                    IF [PTR249] <> NULL AND [PTR249] <> MONETARY(U,13,2)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR249, 'U', 13, 2)
            
                    Data Elements:
            
                    ^PTR249 = RP:StatementOfDistribution:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
                    */
                    assertion = NotMonetary(statementOfDistribution.PTR249, @"U", 13, 2);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430167"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR249", Value = GetValueOrEmpty(statementOfDistribution.PTR249) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430171
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430171()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430171
            Net rental income must equal the sum of all partners' Share of net rental income
    
            Legacy Rule Format:
            IF [PTR255] <> SUM([PTR257])
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR255 <> Sum(^PTR257)
    
            Data Elements:
    
            ^PTR255 = RP:Income.RentalIncomeOrLossNet.Amount
    
            ^PTR257 = RP:StatementOfDistribution:Income.RentalIncomeOrLossNet.Amount
            */
            assertion = (report.PTR255.GetValueOrDefault() != (report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR257.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430171", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net rental income must equal the sum of all partners' Share of net rental income",
                    LongDescription = @"The amount at Net rental property income or loss must equal the sum of all amounts recorded in the Statement of distribution - Share of net rental property income or loss",
                    Location = "/xbrli:xbrl/tns:Income.RentalIncomeOrLossNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430171"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR255", Value = GetValueOrEmpty(report.PTR255) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR257", Value = "PTR257" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430172
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430172()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430172
            Net financial investment income must equal the sum of all partners' Share of net financial income
    
            Legacy Rule Format:
            IF ([PTR254] <> SUM([PTR256]))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR254 <> Sum(^PTR256))
    
            Data Elements:
    
            ^PTR254 = RP:Income.Investment.FinancialIncomeOrLossNet.Amount
    
            ^PTR256 = RP:StatementOfDistribution:Income.Investment.FinancialIncomeOrLossNet.Amount
            */
            assertion = (report.PTR254.GetValueOrDefault() != (report.StatementOfDistributionCollection == null ? 0 : report.StatementOfDistributionCollection.Sum(f => f.PTR256.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430172", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net financial investment income must equal the sum of all partners' Share of net financial income",
                    LongDescription = @"The amount at Net financial investment income or loss must equal the sum of all amounts recorded in the Statement of distribution - Share of net financial investment income or loss",
                    Location = "/xbrli:xbrl/tns:Income.Investment.FinancialIncomeOrLossNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430172"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR254", Value = GetValueOrEmpty(report.PTR254) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR256", Value = "PTR256" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430177
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430177()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430177
            Interposed Entity Election Status - Specified income year must be in range 1995 to the current income year
    
            Legacy Rule Format:
            IF [PTR26] <> NULL AND ([PTR26] < 1995 OR [PTR26] > [PTR1])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR26 <> NULL AND (^PTR26 < 1995 OR ^PTR26 > ^PTR1)
    
            Data Elements:
    
            ^PTR26 = RP:Elections.InterposedEntityElectionStatus.Year
    
            ^PTR1 = RP:Report.TargetFinancial.Year
            */
            assertion = (report.PTR26 != null && (report.PTR26.GetValueOrDefault() < 1995 || report.PTR26.GetValueOrDefault() > report.PTR1.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430702", Severity = ProcessMessageSeverity.Error,
                    Description = @"Interposed Entity Election Status - Specified income year must be in range 1995 to current year",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionStatus.Year",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430177"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR26", Value = GetValueOrEmpty(report.PTR26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR1", Value = GetValueOrEmpty(report.PTR1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430194
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430194()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430194
            All current assets must not exceed Total assets
    
            Legacy Rule Format:
            IF  [PTR112] < [PTR111]
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR112 < ^PTR111
    
            Data Elements:
    
            ^PTR111 = RP.Closing(Instant):Assets.CurrentTotal.Amount
    
            ^PTR112 = RP.Closing(Instant):Assets.Total.Amount
            */
            assertion = (report.PTR112.GetValueOrDefault() < report.PTR111.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430194", Severity = ProcessMessageSeverity.Error,
                    Description = @"All current assets must not exceed Total assets",
                    Location = "/xbrli:xbrl/tns:Assets.CurrentTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR112", Value = GetValueOrEmpty(report.PTR112) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR111", Value = GetValueOrEmpty(report.PTR111) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430195
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430195()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430195
            All current liabilities must not exceed Total liabilities
    
            Legacy Rule Format:
            IF [PTR114] < [PTR113]
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR114 < ^PTR113
    
            Data Elements:
    
            ^PTR113 = RP.Closing(Instant):Liabilities.CurrentTotal.Amount
    
            ^PTR114 = RP.Closing(Instant):Liabilities.Total.Amount
            */
            assertion = (report.PTR114.GetValueOrDefault() < report.PTR113.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430195", Severity = ProcessMessageSeverity.Error,
                    Description = @"All current liabilities must not exceed Total liabilities",
                    Location = "/xbrli:xbrl/tns:Liabilities.CurrentTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR114", Value = GetValueOrEmpty(report.PTR114) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR113", Value = GetValueOrEmpty(report.PTR113) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430196
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430196(PTR2024.OtherDeductions otherDeductions, int itemIndex1)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.PTR.430196
                    Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
                    Legacy Rule Format:
                    IF [PTR96] <> NULL AND [PTR96] <> MONETARY(U,11,0)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^PTR96, 'U', 11, 0)
            
                    Data Elements:
            
                    ^PTR96 = RP.JAUS:OtherDeductions:Expense.DeductibleOther.Amount
                    */
                    assertion = NotMonetary(otherDeductions.PTR96, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:OtherDeductions" + OccurrenceIndex(otherDeductions.OccurrenceIndex) + "/tns:Expense.DeductibleOther.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430196"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "PTR96", Value = GetValueOrEmpty(otherDeductions.PTR96) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.PTR.430197
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430197()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430197
            Closing stock amount must be present if Code is present
    
            Legacy Rule Format:
            IF ([PTR124] <> NULL) AND ([PTR123] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR124 <> NULL) AND (^PTR123 = NULL)
    
            Data Elements:
    
            ^PTR123 = RP.Closing(Instant):Assets.Inventories.Total.Amount
    
            ^PTR124 = RP.Closing(Instant):Assets.Inventories.ValuationMethod.Code
            */
            assertion = (report.PTR124 != null && report.PTR123 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430197", Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock amount must be present if Code is present",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.Total.Amount[@contextRef='" + report.RPClosingInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430197"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR124", Value = GetValueOrEmpty(report.PTR124) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR123", Value = GetValueOrEmpty(report.PTR123) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430198
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430198()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430198
            Closing stock action code must be present if Closing stock amount is present
    
            Legacy Rule Format:
            IF ([PTR124] = NULL) AND ([PTR123] >0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR124 = NULL) AND (^PTR123 >0)
    
            Data Elements:
    
            ^PTR124 = RP.Closing(Instant):Assets.Inventories.ValuationMethod.Code
    
            ^PTR123 = RP.Closing(Instant):Assets.Inventories.Total.Amount
            */
            assertion = (report.PTR124 == null && report.PTR123.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430198", Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock action code must be present if Closing stock amount is present",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.ValuationMethod.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430198"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR124", Value = GetValueOrEmpty(report.PTR124) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR123", Value = GetValueOrEmpty(report.PTR123) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430199
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430199()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430199
            Trade debtors amount must not exceed All current assets amount
    
            Legacy Rule Format:
            IF ([PTR111] < [PTR125])
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR111 < ^PTR125
    
            Data Elements:
    
            ^PTR125 = RP.Closing(Instant):Assets.TradeandReceivablesOther.Amount
    
            ^PTR111 = RP.Closing(Instant):Assets.CurrentTotal.Amount
            */
            assertion = (report.PTR111.GetValueOrDefault() < report.PTR125.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430199", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade debtors amount must not exceed All current assets amount",
                    Location = "/xbrli:xbrl/tns:Assets.TradeandReceivablesOther.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430199"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR111", Value = GetValueOrEmpty(report.PTR111) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR125", Value = GetValueOrEmpty(report.PTR125) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430200
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430200()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430200
            Trade creditors cannot exceed all current liabilities in Financial and other information
    
            Legacy Rule Format:
            IF [PTR113] < [PTR126]
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^PTR113 < ^PTR126
    
            Data Elements:
    
            ^PTR126 = RP.Closing(Instant):Liabilities.TradeAndOtherPayablesTotal.Amount
    
            ^PTR113 = RP.Closing(Instant):Liabilities.CurrentTotal.Amount
            */
            assertion = (report.PTR113.GetValueOrDefault() < report.PTR126.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430200", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade creditors cannot exceed all current liabilities in Financial and other information",
                    Location = "/xbrli:xbrl/tns:Liabilities.TradeAndOtherPayablesTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430200"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR113", Value = GetValueOrEmpty(report.PTR113) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR126", Value = GetValueOrEmpty(report.PTR126) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.PTR.430207
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOPTR430207()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.PTR.430207
            Industry code must be present
    
            Legacy Rule Format:
            IF ([PTR35] <> NULL OR [PTR36] <> NULL OR [PTR38] <> NULL OR [PTR39] <> NULL OR [PTR41] <> NULL OR [PTR43] <> NULL OR [PTR44] <> NULL) AND [PTR31] = NULLORBLANK 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PTR35 <> NULL OR ^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR39 <> NULL OR ^PTR41 <> NULL OR ^PTR43 <> NULL OR ^PTR44 <> NULL) AND ^PTR31 = BLANK
    
            Data Elements:
    
            ^PTR31 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^PTR35 = RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = RP.JAUS.NonPrim:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = RP.JAUS.Prim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = RP.JAUS.NonPrim:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = RP.JAUS.Prim:Income.Other.Amount
    
            ^PTR44 = RP.JAUS.NonPrim:Income.Other.Amount
            */
            assertion = ((report.PTR35 != null || report.PTR36 != null || report.PTR38 != null || report.PTR39 != null || report.PTR41 != null || report.PTR43 != null || report.PTR44 != null) && string.IsNullOrWhiteSpace(report.PTR31) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430207", Severity = ProcessMessageSeverity.Error,
                    Description = @"If  business income (PP or non-PP) is present, an Industry code (ANZSIC) must be present",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationIndustry2006Extended.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParamet