using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationFTER
{
    /// <summary>
    /// XML Consumer for FTER
    /// </summary>
    public class FTER2021XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static FTER2021XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.fter.0003.2021.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;
  
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();        
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count-1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements) 
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
           return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory) 
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader) 
        {
            bool retval = false;

            if ( (reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if(_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if(_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
             }

             //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf , string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if(numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if(numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader) 
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }
      
                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public FTER2021 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            FTER2021 report = new FTER2021();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:FTER");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "FTER" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/fter")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

    
            #region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader,false) && reader.LocalName == "INT")
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true; 
        
                #region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxAgentNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FTER2 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxAgentNumberId
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FTER171 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region PersonUnstructuredNameFullNameT
                _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PersonUnstructuredNameFullNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FTER82 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PersonUnstructuredNameFullNameT
        
                #region TelephoneDetails
                _currentXPath.Push("/tns:TelephoneDetails");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TelephoneDetails")
                {
                    report.InT_TelephoneDetailsCollectionExists = true;
                    report.InT_TelephoneDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region ElectronicContactAreaC
                    _currentXPath.Push("/tns:ElectronicContactAreaC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactAreaC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER83 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactAreaC
            
                    #region ElectronicContactMinimalN
                    _currentXPath.Push("/tns:ElectronicContactMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectronicContactMinimalN")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER84 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactMinimalN
                } // End of if TelephoneDetails node exists
        
                _currentXPath.Pop();
                #endregion TelephoneDetails
        
                #region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "Declaration")
                {
                    report.InT_DeclarationCollectionExists = true;
                    report.InT_DeclarationCollectionCount += 1;
                    _found = true; 
            
                    #region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER159 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementTypeC
            
                    #region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FTER160 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementAcceptedI
            
                    #region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FTER162 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatureD
            
                    #region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER163 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatoryIdentifierT
            
                    #region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonUnstructuredNameFullNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER165 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonUnstructuredNameFullNameT
                } // End of if Declaration node exists
        
                _currentXPath.Pop();
                #endregion Declaration
            } // End of if INT node exists
    
            _currentXPath.Pop();
            #endregion INT
    
            #region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader,true) && reader.LocalName == "RP")
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true; 
        
                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "TaxFileNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FTER3 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxFileNumberId
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FTER170 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region LodgmentPeriodYearY
                _currentXPath.Push("/tns:LodgmentPeriodYearY");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "LodgmentPeriodYearY")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.FTER1 = currentIntValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion LodgmentPeriodYearY
        
                #region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodStartD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.FTER173 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodStartD
        
                #region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodEndD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.FTER174 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodEndD
        
                #region ElectionsFamilyTrustElectionRevocationC
                _currentXPath.Push("/tns:ElectionsFamilyTrustElectionRevocationC");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsFamilyTrustElectionRevocationC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FTER95 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ElectionsFamilyTrustElectionRevocationC
        
                #region CompletionHoursN
                _currentXPath.Push("/tns:CompletionHoursN");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CompletionHoursN")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.FTER85 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion CompletionHoursN
        
                #region CurrentTrustDetails
                _currentXPath.Push("/tns:CurrentTrustDetails");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CurrentTrustDetails")
                {
                    report.RP_CurrentTrustDetailsCollectionExists = true;
                    report.RP_CurrentTrustDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER4 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion OrganisationNameDetailsOrganisationalNameT
            
                    #region PostalAddressDetails
                    _currentXPath.Push("/tns:PostalAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PostalAddressDetails")
                    {
                        report.RP_CurrentTrustDetails_PostalAddressDetailsCollectionExists = true;
                        report.RP_CurrentTrustDetails_PostalAddressDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER5 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion Line1T
                
                        #region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER110 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion Line2T
                
                        #region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER6 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion LocalityNameT
                
                        #region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER8 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PostcodeT
                
                        #region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER7 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StateOrTerritoryC
                
                        #region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "CountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER113 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion CountryC
                    } // End of if PostalAddressDetails node exists
            
                    _currentXPath.Pop();
                    #endregion PostalAddressDetails
                } // End of if CurrentTrustDetails node exists
        
                _currentXPath.Pop();
                #endregion CurrentTrustDetails
        
                #region PreviousTrustDetails
                _currentXPath.Push("/tns:PreviousTrustDetails");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PreviousTrustDetails")
                {
                    report.RP_PreviousTrustDetailsCollectionExists = true;
                    report.RP_PreviousTrustDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER10 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion OrganisationNameDetailsOrganisationalNameT
            
                    #region PostalAddressDetails
                    _currentXPath.Push("/tns:PostalAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PostalAddressDetails")
                    {
                        report.RP_PreviousTrustDetails_PostalAddressDetailsCollectionExists = true;
                        report.RP_PreviousTrustDetails_PostalAddressDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER11 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion Line1T
                
                        #region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER115 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion Line2T
                
                        #region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER12 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion LocalityNameT
                
                        #region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER14 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PostcodeT
                
                        #region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER13 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StateOrTerritoryC
                
                        #region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "CountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER118 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion CountryC
                    } // End of if PostalAddressDetails node exists
            
                    _currentXPath.Pop();
                    #endregion PostalAddressDetails
                } // End of if PreviousTrustDetails node exists
        
                _currentXPath.Pop();
                #endregion PreviousTrustDetails
        
                #region CentralManagementOutsideAustralia
                _currentXPath.Push("/tns:CentralManagementOutsideAustralia");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CentralManagementOutsideAustralia")
                {
                    report.RP_CentralManagementOutsideAustraliaCollectionExists = true;
                    report.RP_CentralManagementOutsideAustraliaCollectionCount += 1;
                    _found = true; 
            
                    #region ResidencyCentralManagementAndControlOutsideAustraliaI
                    _currentXPath.Push("/tns:ResidencyCentralManagementAndControlOutsideAustraliaI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ResidencyCentralManagementAndControlOutsideAustraliaI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FTER16 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ResidencyCentralManagementAndControlOutsideAustraliaI
            
                    #region ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI
                    _currentXPath.Push("/tns:ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FTER17 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI
            
                    #region PeriodDetails
            
                    _currentXPath.Push("/tns:PeriodDetailsCollection");
                    // 4. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PeriodDetailsCollection")
                    {
                        report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection = new List<FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails>();
                        report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionExists = true;
            
                        _found = true;
            
                        while ( ReadToNextElement(reader,false) && reader.LocalName == "PeriodDetails" ) 
                        {
                            _found = true;
            
                            FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails periodDetails = new FTER2021.RP_CentralManagementOutsideAustralia_PeriodDetails();
                            report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollection.Add(periodDetails);
                            report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionCount += 1;
                            periodDetails.OccurrenceIndex = report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionCount;
            
                            _currentXPath.Push("/tns:PeriodDetails[" + report.RP_CentralManagementOutsideAustralia_PeriodDetailsCollectionCount + "]");
                    
                            #region StartD
                            _currentXPath.Push("/tns:StartD");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            periodDetails.FTER18 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion StartD
                    
                            #region EndD
                            _currentXPath.Push("/tns:EndD");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            periodDetails.FTER19 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion EndD
                        _currentXPath.Pop();
                        } // End of while loop for PeriodDetails
                    } // End of if PeriodDetailsCollection node exists
            
                    _currentXPath.Pop();
                    #endregion PeriodDetails
                } // End of if CentralManagementOutsideAustralia node exists
        
                _currentXPath.Pop();
                #endregion CentralManagementOutsideAustralia
        
                #region Trustee
        
                _currentXPath.Push("/tns:TrusteeCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TrusteeCollection")
                {
                    report.RP_TrusteeCollection = new List<FTER2021.RP_Trustee>();
                    report.RP_TrusteeCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "Trustee" ) 
                    {
                        _found = true;
        
                        FTER2021.RP_Trustee trustee = new FTER2021.RP_Trustee();
                        report.RP_TrusteeCollection.Add(trustee);
                        report.RP_TrusteeCollectionCount += 1;
                        trustee.OccurrenceIndex = report.RP_TrusteeCollectionCount;
        
                        _currentXPath.Push("/tns:Trustee[" + report.RP_TrusteeCollectionCount + "]");
                
                        #region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    trustee.FTER20 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberId
                
                        #region TaxFileNumberInexistentI
                        _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        trustee.FTER21 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberInexistentI
                
                        #region IndividualNameDetails
                        _currentXPath.Push("/tns:IndividualNameDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IndividualNameDetails")
                        {
                            trustee.RP_Trustee_IndividualNameDetailsCollectionExists = true;
                            trustee.RP_Trustee_IndividualNameDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region PersonNameDetailsTitleT
                            _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER22 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsTitleT
                    
                            #region PersonNameDetailsNameSuffixT
                            _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER24 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsNameSuffixT
                    
                            #region PersonNameDetailsFamilyNameT
                            _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER23 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsFamilyNameT
                    
                            #region PersonNameDetailsGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER25 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsGivenNameT
                    
                            #region PersonNameDetailsOtherGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER26 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsOtherGivenNameT
                        } // End of if IndividualNameDetails node exists
                
                        _currentXPath.Pop();
                        #endregion IndividualNameDetails
                
                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    trustee.FTER27 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT
                
                        #region AustralianCompanyNumberId
                        _currentXPath.Push("/tns:AustralianCompanyNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianCompanyNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    trustee.FTER86 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AustralianCompanyNumberId
                
                        #region AustralianRegisteredBodyNumberId
                        _currentXPath.Push("/tns:AustralianRegisteredBodyNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianRegisteredBodyNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    trustee.FTER29 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AustralianRegisteredBodyNumberId
                
                        #region PostalAddressDetails
                        _currentXPath.Push("/tns:PostalAddressDetails");
                        //3. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "PostalAddressDetails")
                        {
                            trustee.RP_Trustee_PostalAddressDetailsCollectionExists = true;
                            trustee.RP_Trustee_PostalAddressDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region Line1T
                            _currentXPath.Push("/tns:Line1T");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER30 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion Line1T
                    
                            #region Line2T
                            _currentXPath.Push("/tns:Line2T");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER31 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion Line2T
                    
                            #region LocalityNameT
                            _currentXPath.Push("/tns:LocalityNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER32 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion LocalityNameT
                    
                            #region PostcodeT
                            _currentXPath.Push("/tns:PostcodeT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER34 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PostcodeT
                    
                            #region StateOrTerritoryC
                            _currentXPath.Push("/tns:StateOrTerritoryC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER33 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion StateOrTerritoryC
                    
                            #region CountryC
                            _currentXPath.Push("/tns:CountryC");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "CountryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustee.FTER125 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion CountryC
                        } // End of if PostalAddressDetails node exists
                
                        _currentXPath.Pop();
                        #endregion PostalAddressDetails
                
                        #region ResidencyTaxPurposesPersonStatusI
                        _currentXPath.Push("/tns:ResidencyTaxPurposesPersonStatusI");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ResidencyTaxPurposesPersonStatusI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        trustee.FTER36 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ResidencyTaxPurposesPersonStatusI
                
                        #region ResidencyNonResidentFullPeriodI
                        _currentXPath.Push("/tns:ResidencyNonResidentFullPeriodI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ResidencyNonResidentFullPeriodI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        trustee.FTER37 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ResidencyNonResidentFullPeriodI
                
                        #region NonResidentPeriodDetails
                
                        _currentXPath.Push("/tns:NonResidentPeriodDetailsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NonResidentPeriodDetailsCollection")
                        {
                            trustee.RP_Trustee_NonResidentPeriodDetailsCollection = new List<FTER2021.RP_Trustee.RP_Trustee_NonResidentPeriodDetails>();
                            trustee.RP_Trustee_NonResidentPeriodDetailsCollectionExists = true;
                
                            _found = true;
                
                            while ( ReadToNextElement(reader,false) && reader.LocalName == "NonResidentPeriodDetails" ) 
                            {
                                _found = true;
                
                                FTER2021.RP_Trustee.RP_Trustee_NonResidentPeriodDetails nonResidentPeriodDetails = new FTER2021.RP_Trustee.RP_Trustee_NonResidentPeriodDetails();
                                trustee.RP_Trustee_NonResidentPeriodDetailsCollection.Add(nonResidentPeriodDetails);
                                trustee.RP_Trustee_NonResidentPeriodDetailsCollectionCount += 1;
                                nonResidentPeriodDetails.OccurrenceIndex = trustee.RP_Trustee_NonResidentPeriodDetailsCollectionCount;
                
                                _currentXPath.Push("/tns:NonResidentPeriodDetails[" + trustee.RP_Trustee_NonResidentPeriodDetailsCollectionCount + "]");
                        
                                #region StartD
                                _currentXPath.Push("/tns:StartD");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                nonResidentPeriodDetails.FTER38 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion StartD
                        
                                #region EndD
                                _currentXPath.Push("/tns:EndD");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                nonResidentPeriodDetails.FTER39 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion EndD
                            _currentXPath.Pop();
                            } // End of while loop for NonResidentPeriodDetails
                        } // End of if NonResidentPeriodDetailsCollection node exists
                
                        _currentXPath.Pop();
                        #endregion NonResidentPeriodDetails
                    _currentXPath.Pop();
                    } // End of while loop for Trustee
                } // End of if TrusteeCollection node exists
        
                _currentXPath.Pop();
                #endregion Trustee
        
                #region ElectionOrVariationIncomeYear
                _currentXPath.Push("/tns:ElectionOrVariationIncomeYear");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionOrVariationIncomeYear")
                {
                    report.RP_ElectionOrVariationIncomeYearCollectionExists = true;
                    report.RP_ElectionOrVariationIncomeYearCollectionCount += 1;
                    _found = true; 
            
                    #region ElectionsFamilyTrustElectionStatusY
                    _currentXPath.Push("/tns:ElectionsFamilyTrustElectionStatusY");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsFamilyTrustElectionStatusY")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.FTER40 = currentIntValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsFamilyTrustElectionStatusY
            
                    #region SubstitutedAccountingPeriodDetails
                    _currentXPath.Push("/tns:SubstitutedAccountingPeriodDetails");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "SubstitutedAccountingPeriodDetails")
                    {
                        report.RP_ElectionOrVariationIncomeYear_SubstitutedAccountingPeriodDetailsCollectionExists = true;
                        report.RP_ElectionOrVariationIncomeYear_SubstitutedAccountingPeriodDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region StartD
                        _currentXPath.Push("/tns:StartD");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        report.FTER41 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StartD
                
                        #region EndD
                        _currentXPath.Push("/tns:EndD");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        report.FTER42 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion EndD
                    } // End of if SubstitutedAccountingPeriodDetails node exists
            
                    _currentXPath.Pop();
                    #endregion SubstitutedAccountingPeriodDetails
                } // End of if ElectionOrVariationIncomeYear node exists
        
                _currentXPath.Pop();
                #endregion ElectionOrVariationIncomeYear
        
                #region ElectionsCommencementD
                _currentXPath.Push("/tns:ElectionsCommencementD");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsCommencementD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.FTER43 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ElectionsCommencementD
        
                #region SpecifiedIndividual
                _currentXPath.Push("/tns:SpecifiedIndividual");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "SpecifiedIndividual")
                {
                    report.RP_SpecifiedIndividualCollectionExists = true;
                    report.RP_SpecifiedIndividualCollectionCount += 1;
                    _found = true; 
            
                    #region TaxFileNumberId
                    _currentXPath.Push("/tns:TaxFileNumberId");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER44 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxFileNumberId
            
                    #region TaxFileNumberInexistentI
                    _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FTER45 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxFileNumberInexistentI
            
                    #region PersonDemographicDetailsBirthD
                    _currentXPath.Push("/tns:PersonDemographicDetailsBirthD");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonDemographicDetailsBirthD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FTER46 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonDemographicDetailsBirthD
            
                    #region PersonNameDetailsTitleT
                    _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER47 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonNameDetailsTitleT
            
                    #region PersonNameDetailsNameSuffixT
                    _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER51 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonNameDetailsNameSuffixT
            
                    #region PersonNameDetailsFamilyNameT
                    _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER50 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonNameDetailsFamilyNameT
            
                    #region PersonNameDetailsGivenNameT
                    _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsGivenNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER48 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonNameDetailsGivenNameT
            
                    #region PersonNameDetailsOtherGivenNameT
                    _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER49 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonNameDetailsOtherGivenNameT
            
                    #region ResidentialAddressDetails
                    _currentXPath.Push("/tns:ResidentialAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ResidentialAddressDetails")
                    {
                        report.RP_SpecifiedIndividual_ResidentialAddressDetailsCollectionExists = true;
                        report.RP_SpecifiedIndividual_ResidentialAddressDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER52 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion Line1T
                
                        #region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER134 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion Line2T
                
                        #region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER53 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion LocalityNameT
                
                        #region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER55 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PostcodeT
                
                        #region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER54 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StateOrTerritoryC
                
                        #region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "CountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.FTER137 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion CountryC
                    } // End of if ResidentialAddressDetails node exists
            
                    _currentXPath.Pop();
                    #endregion ResidentialAddressDetails
                } // End of if SpecifiedIndividual node exists
        
                _currentXPath.Pop();
                #endregion SpecifiedIndividual
        
                #region FamilyTrustRevocation
                _currentXPath.Push("/tns:FamilyTrustRevocation");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "FamilyTrustRevocation")
                {
                    report.RP_FamilyTrustRevocationCollectionExists = true;
                    report.RP_FamilyTrustRevocationCollectionCount += 1;
                    _found = true; 
            
                    #region ElectionsD
                    _currentXPath.Push("/tns:ElectionsD");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FTER57 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsD
            
                    #region ElectionsStatusEffectiveY
                    _currentXPath.Push("/tns:ElectionsStatusEffectiveY");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsStatusEffectiveY")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.FTER96 = currentIntValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsStatusEffectiveY
            
                    #region ElectionsNotificationOfElectionBeingRevokedY
                    _currentXPath.Push("/tns:ElectionsNotificationOfElectionBeingRevokedY");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsNotificationOfElectionBeingRevokedY")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.FTER58 = currentIntValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsNotificationOfElectionBeingRevokedY
            
                    #region ElectionsFamilyTrustElectionStatusY
                    _currentXPath.Push("/tns:ElectionsFamilyTrustElectionStatusY");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsFamilyTrustElectionStatusY")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.FTER172 = currentIntValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsFamilyTrustElectionStatusY
            
                    #region TaxFileNumberId
                    _currentXPath.Push("/tns:TaxFileNumberId");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FTER60 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxFileNumberId
            
                    #region TaxFileNumberInexistentI
                    _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FTER61 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxFileNumberInexistentI
            
                    #region InterposedEntity
            
                    _currentXPath.Push("/tns:InterposedEntityCollection");
                    // 4. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "InterposedEntityCollection")
                    {
                        report.RP_FamilyTrustRevocation_InterposedEntityCollection = new List<FTER2021.RP_FamilyTrustRevocation_InterposedEntity>();
                        report.RP_FamilyTrustRevocation_InterposedEntityCollectionExists = true;
            
                        _found = true;
            
                        while ( ReadToNextElement(reader,false) && reader.LocalName == "InterposedEntity" ) 
                        {
                            _found = true;
            
                            FTER2021.RP_FamilyTrustRevocation_InterposedEntity interposedEntity = new FTER2021.RP_FamilyTrustRevocation_InterposedEntity();
                            report.RP_FamilyTrustRevocation_InterposedEntityCollection.Add(interposedEntity);
                            report.RP_FamilyTrustRevocation_InterposedEntityCollectionCount += 1;
                            interposedEntity.OccurrenceIndex = report.RP_FamilyTrustRevocation_InterposedEntityCollectionCount;
            
                            _currentXPath.Push("/tns:InterposedEntity[" + report.RP_FamilyTrustRevocation_InterposedEntityCollectionCount + "]");
                    
                            #region PartyTypeC
                            _currentXPath.Push("/tns:PartyTypeC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PartyTypeC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        interposedEntity.FTER62 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PartyTypeC
                    
                            #region TaxFileNumberId
                            _currentXPath.Push("/tns:TaxFileNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        interposedEntity.FTER64 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxFileNumberId
                    
                            #region TaxFileNumberInexistentI
                            _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            interposedEntity.FTER63 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxFileNumberInexistentI
                    
                            #region OrganisationNameDetailsOrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        interposedEntity.FTER65 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OrganisationNameDetailsOrganisationalNameT
                    
                            #region AustralianCompanyNumberId
                            _currentXPath.Push("/tns:AustralianCompanyNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianCompanyNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        interposedEntity.FTER87 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AustralianCompanyNumberId
                    
                            #region AustralianRegisteredBodyNumberId
                            _currentXPath.Push("/tns:AustralianRegisteredBodyNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianRegisteredBodyNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        interposedEntity.FTER67 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AustralianRegisteredBodyNumberId
                    
                            #region PostalAddressDetails
                            _currentXPath.Push("/tns:PostalAddressDetails");
                            //3. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PostalAddressDetails")
                            {
                                interposedEntity.RP_FamilyTrustRevocation_InterposedEntity_PostalAddressDetailsCollectionExists = true;
                                interposedEntity.RP_FamilyTrustRevocation_InterposedEntity_PostalAddressDetailsCollectionCount += 1;
                                _found = true; 
                        
                                #region Line1T
                                _currentXPath.Push("/tns:Line1T");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            interposedEntity.FTER68 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion Line1T
                        
                                #region Line2T
                                _currentXPath.Push("/tns:Line2T");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            interposedEntity.FTER69 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion Line2T
                        
                                #region LocalityNameT
                                _currentXPath.Push("/tns:LocalityNameT");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            interposedEntity.FTER70 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion LocalityNameT
                        
                                #region PostcodeT
                                _currentXPath.Push("/tns:PostcodeT");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            interposedEntity.FTER72 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion PostcodeT
                        
                                #region StateOrTerritoryC
                                _currentXPath.Push("/tns:StateOrTerritoryC");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            interposedEntity.FTER71 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion StateOrTerritoryC
                        
                                #region CountryC
                                _currentXPath.Push("/tns:CountryC");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "CountryC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            interposedEntity.FTER145 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion CountryC
                            } // End of if PostalAddressDetails node exists
                    
                            _currentXPath.Pop();
                            #endregion PostalAddressDetails
                        _currentXPath.Pop();
                        } // End of while loop for InterposedEntity
                    } // End of if InterposedEntityCollection node exists
            
                    _currentXPath.Pop();
                    #endregion InterposedEntity
                } // End of if FamilyTrustRevocation node exists
        
                _currentXPath.Pop();
                #endregion FamilyTrustRevocation
        
                #region DeclarationOfTrustee
        
                _currentXPath.Push("/tns:DeclarationOfTrusteeCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "DeclarationOfTrusteeCollection")
                {
                    report.RP_DeclarationOfTrusteeCollection = new List<FTER2021.RP_DeclarationOfTrustee>();
                    report.RP_DeclarationOfTrusteeCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "DeclarationOfTrustee" ) 
                    {
                        _found = true;
        
                        FTER2021.RP_DeclarationOfTrustee declarationOfTrustee = new FTER2021.RP_DeclarationOfTrustee();
                        report.RP_DeclarationOfTrusteeCollection.Add(declarationOfTrustee);
                        report.RP_DeclarationOfTrusteeCollectionCount += 1;
                        declarationOfTrustee.OccurrenceIndex = report.RP_DeclarationOfTrusteeCollectionCount;
        
                        _currentXPath.Push("/tns:DeclarationOfTrustee[" + report.RP_DeclarationOfTrusteeCollectionCount + "]");
                
                        #region IndividualNameDetails
                        _currentXPath.Push("/tns:IndividualNameDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IndividualNameDetails")
                        {
                            declarationOfTrustee.RP_DeclarationOfTrustee_IndividualNameDetailsCollectionExists = true;
                            declarationOfTrustee.RP_DeclarationOfTrustee_IndividualNameDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region PersonNameDetailsTitleT
                            _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrustee.FTER74 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsTitleT
                    
                            #region PersonNameDetailsNameSuffixT
                            _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrustee.FTER76 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsNameSuffixT
                    
                            #region PersonNameDetailsFamilyNameT
                            _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrustee.FTER75 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsFamilyNameT
                    
                            #region PersonNameDetailsGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrustee.FTER77 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsGivenNameT
                    
                            #region PersonNameDetailsOtherGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrustee.FTER78 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsOtherGivenNameT
                        } // End of if IndividualNameDetails node exists
                
                        _currentXPath.Pop();
                        #endregion IndividualNameDetails
                
                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    declarationOfTrustee.FTER79 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT
                
                        #region StatementTypeC
                        _currentXPath.Push("/tns:StatementTypeC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    declarationOfTrustee.FTER99 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StatementTypeC
                
                        #region StatementAcceptedI
                        _currentXPath.Push("/tns:StatementAcceptedI");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        declarationOfTrustee.FTER169 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StatementAcceptedI
                
                        #region SignatureD
                        _currentXPath.Push("/tns:SignatureD");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        declarationOfTrustee.FTER168 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SignatureD
                
                        #region SignatoryIdentifierT
                        _currentXPath.Push("/tns:SignatoryIdentifierT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    declarationOfTrustee.FTER100 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SignatoryIdentifierT
                    _currentXPath.Pop();
                    } // End of while loop for DeclarationOfTrustee
                } // End of if DeclarationOfTrusteeCollection node exists
        
                _currentXPath.Pop();
                #endregion DeclarationOfTrustee
            } // End of if RP node exists
    
            _currentXPath.Pop();
            #endregion RP

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "INT#TaxAgentNumberId,INT#AustralianBusinessNumberId,INT#PersonUnstructuredNameFullNameT,TelephoneDetails#ElectronicContactAreaC,TelephoneDetails#ElectronicContactMinimalN,TelephoneDetails,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatureD,Declaration#SignatoryIdentifierT,Declaration#PersonUnstructuredNameFullNameT,Declaration,INT,RP#TaxFileNumberId,RP#AustralianBusinessNumberId,RP#LodgmentPeriodYearY,RP#PeriodStartD,RP#PeriodEndD,RP#ElectionsFamilyTrustElectionRevocationC,RP#CompletionHoursN,CurrentTrustDetails#OrganisationNameDetailsOrganisationalNameT,PostalAddressDetails#Line1T,PostalAddressDetails#Line2T,PostalAddressDetails#LocalityNameT,PostalAddressDetails#PostcodeT,PostalAddressDetails#StateOrTerritoryC,PostalAddressDetails#CountryC,PostalAddressDetails,CurrentTrustDetails,PreviousTrustDetails#OrganisationNameDetailsOrganisationalNameT,PostalAddressDetails#Line1T,PostalAddressDetails#Line2T,PostalAddressDetails#LocalityNameT,PostalAddressDetails#PostcodeT,PostalAddressDetails#StateOrTerritoryC,PostalAddressDetails#CountryC,PostalAddressDetails,PreviousTrustDetails,CentralManagementOutsideAustralia#ResidencyCentralManagementAndControlOutsideAustraliaI,CentralManagementOutsideAustralia#ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI,PeriodDetails#StartD,PeriodDetails#EndD,PeriodDetails,PeriodDetailsCollection,CentralManagementOutsideAustralia,Trustee#TaxFileNumberId,Trustee#TaxFileNumberInexistentI,Trustee#PersonNameDetailsTitleT,Trustee#PersonNameDetailsNameSuffixT,Trustee#PersonNameDetailsFamilyNameT,Trustee#PersonNameDetailsGivenNameT,Trustee#PersonNameDetailsOtherGivenNameT,IndividualNameDetails,Trustee#OrganisationNameDetailsOrganisationalNameT,Trustee#AustralianCompanyNumberId,Trustee#AustralianRegisteredBodyNumberId,Trustee#Line1T,Trustee#Line2T,Trustee#LocalityNameT,Trustee#PostcodeT,Trustee#StateOrTerritoryC,Trustee#CountryC,PostalAddressDetails,Trustee#ResidencyTaxPurposesPersonStatusI,Trustee#ResidencyNonResidentFullPeriodI,NonResidentPeriodDetails#StartD,NonResidentPeriodDetails#EndD,NonResidentPeriodDetails,NonResidentPeriodDetailsCollection,Trustee,TrusteeCollection,ElectionOrVariationIncomeYear#ElectionsFamilyTrustElectionStatusY,SubstitutedAccountingPeriodDetails#StartD,SubstitutedAccountingPeriodDetails#EndD,SubstitutedAccountingPeriodDetails,ElectionOrVariationIncomeYear,RP#ElectionsCommencementD,SpecifiedIndividual#TaxFileNumberId,SpecifiedIndividual#TaxFileNumberInexistentI,SpecifiedIndividual#PersonDemographicDetailsBirthD,SpecifiedIndividual#PersonNameDetailsTitleT,SpecifiedIndividual#PersonNameDetailsNameSuffixT,SpecifiedIndividual#PersonNameDetailsFamilyNameT,SpecifiedIndividual#PersonNameDetailsGivenNameT,SpecifiedIndividual#PersonNameDetailsOtherGivenNameT,ResidentialAddressDetails#Line1T,ResidentialAddressDetails#Line2T,ResidentialAddressDetails#LocalityNameT,ResidentialAddressDetails#PostcodeT,ResidentialAddressDetails#StateOrTerritoryC,ResidentialAddressDetails#CountryC,ResidentialAddressDetails,SpecifiedIndividual,FamilyTrustRevocation#ElectionsD,FamilyTrustRevocation#ElectionsStatusEffectiveY,FamilyTrustRevocation#ElectionsNotificationOfElectionBeingRevokedY,FamilyTrustRevocation#ElectionsFamilyTrustElectionStatusY,FamilyTrustRevocation#TaxFileNumberId,FamilyTrustRevocation#TaxFileNumberInexistentI,InterposedEntity#PartyTypeC,InterposedEntity#TaxFileNumberId,InterposedEntity#TaxFileNumberInexistentI,InterposedEntity#OrganisationNameDetailsOrganisationalNameT,InterposedEntity#AustralianCompanyNumberId,InterposedEntity#AustralianRegisteredBodyNumberId,InterposedEntity#Line1T,InterposedEntity#Line2T,InterposedEntity#LocalityNameT,InterposedEntity#PostcodeT,InterposedEntity#StateOrTerritoryC,InterposedEntity#CountryC,PostalAddressDetails,InterposedEntity,InterposedEntityCollection,FamilyTrustRevocation,DeclarationOfTrustee#PersonNameDetailsTitleT,DeclarationOfTrustee#PersonNameDetailsNameSuffixT,DeclarationOfTrustee#PersonNameDetailsFamilyNameT,DeclarationOfTrustee#PersonNameDetailsGivenNameT,DeclarationOfTrustee#PersonNameDetailsOtherGivenNameT,IndividualNameDetails,DeclarationOfTrustee#OrganisationNameDetailsOrganisationalNameT,DeclarationOfTrustee#StatementTypeC,DeclarationOfTrustee#StatementAcceptedI,DeclarationOfTrustee#SignatureD,DeclarationOfTrustee#SignatoryIdentifierT,DeclarationOfTrustee,DeclarationOfTrusteeCollection,RP";

    }
}

