using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;
using Ato.EN.IntegrationServices.CodeGenerationTRT;
using DataContracts;
using System.Collections.Generic;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.TRT202402
{
    public class CrossFormValidatorDISTBENTRT : ICrossFormValidator
    {
        private TRT2024 parentDocument { get; set; }
        public List<ProcessMessageDocument> response { get; private set; }
        private DISTBENTRT2024 childDocument { get; }

        public CrossFormValidatorDISTBENTRT(TRT2024 report, object child)
        {
            parentDocument = report;
            childDocument = (DISTBENTRT2024)child;
            response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //Parent form cross form rules
            VRATOTRT442894();
            VRATOTRT442895();
            VRATOTRT442896();
            VRATOTRTW00021();

            return response;
        }

        #region "Validation rules"

        #region VR.ATO.TRT.442894

        /*  VR.ATO.TRT.442894
        Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form

        Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^TRT573 <> ^DISTBENTRT1

        Data Elements:
        ^TRT573 = TRT:RP:Period.Start.Date    
        ^DISTBENTRT1 = DISTBENTRT1
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public void VRATOTRT442894()
        {
            bool assertion = (childDocument.DISTBENTRT1 != null) && (childDocument.DISTBENTRT1.GetValueOrDefault() != parentDocument.TRT573.GetValueOrDefault());

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442894",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form",
                    Location = "/tns:TRT/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRT.442894" } ,
                        new ProcessMessageParameter { Name = "TRT573", Value = TRT2024Validator.GetValueOrEmptyDate(parentDocument.TRT573) },
                        new ProcessMessageParameter { Name = "DISTBENTRT1", Value = TRT2024Validator.GetValueOrEmptyDate(childDocument.DISTBENTRT1) }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }

        #endregion //VR.ATO.TRT.442894

        #region VR.ATO.TRT.442895

        /*  VR.ATO.TRT.442895
        Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form

        Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^TRT574 <> ^DISTBENTRT2

        Data Elements:    
        ^TRT574 = TRT:RP:Period.End.Date
        ^DISTBENTRT2 = DISTBENTRT2
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public void VRATOTRT442895()
        {
            bool assertion = (childDocument.DISTBENTRT2 != null) && (childDocument.DISTBENTRT2.GetValueOrDefault() != parentDocument.TRT574.GetValueOrDefault());

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442895",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form",
                    Location = "/tns:TRT/tns:RP/tns:PeriodEndD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRT.442895" } ,
                        new ProcessMessageParameter { Name = "TRT574", Value = TRT2024Validator.GetValueOrEmptyDate(parentDocument.TRT574) },
                        new ProcessMessageParameter { Name = "DISTBENTRT1", Value = TRT2024Validator.GetValueOrEmptyDate(childDocument.DISTBENTRT2) }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }

        #endregion //VR.ATO.TRT.442895

        #region VR.ATO.TRT.442896

        /*  VR.ATO.TRT.442896
        The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form

        Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^TRT5 <> ^DISTBENTRT3

        Data Elements:    
        ^TRT5 = TRT:RP:Identifiers.TaxFileNumber.Identifier    
        ^DISTBENTRT3 = DISTBENTRT3
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public void VRATOTRT442896()
        {
            bool assertion = (childDocument.DISTBENTRT3 != null) && (childDocument.DISTBENTRT3 != parentDocument.TRT5);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442896",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form",
                    Location = "/tns:TRT/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRT.442896" } ,
                        new ProcessMessageParameter { Name = "TRT5", Value =  TRT2024Validator.GetValueOrEmpty(parentDocument.TRT5) },
                        new ProcessMessageParameter { Name = "DISTBENTRT3", Value = TRT2024Validator.GetValueOrEmpty(childDocument.DISTBENTRT3) },
                    },
                    DocumentSequence = "parent_doc"
                });
            }

        }

        #endregion //VR.ATO.TRT.442896

        #region VR.ATO.TRT.W00021

        /*  VR.ATO.TRT.W00021
        Distributions to Beneficiaries of Trust schedule is provided but Trust distributions - Income amounts or credits are not declared

        Technical Business Rule Format:
            CountDocument('DISTBENTRT') = 1 AND ^TRT77 = NULL AND ^TRT81 = NULL AND ^TRT437 = NULL AND ^TRT85 = NULL

        Data Elements:    
        ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount    
        ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount    
        ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount    
        ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public void VRATOTRTW00021()
        {
            bool assertion = (!parentDocument.TRT77.HasValue) && (!parentDocument.TRT81.HasValue) && (!parentDocument.TRT85.HasValue) && (!parentDocument.TRT437.HasValue);
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.W00021",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure Trust distributions - Income amounts or credits from trusts income are declared",
                    LongDescription = @"Ensure Trust distributions - Income amounts or credits are declared when Distributions to Beneficiaries of Trust schedule is provided",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00021" },
                        new ProcessMessageParameter { Name = "TRT77", Value = TRT2024Validator.GetValueOrEmpty(parentDocument.TRT77) },
                        new ProcessMessageParameter { Name = "TRT81", Value = TRT2024Validator.GetValueOrEmpty(parentDocument.TRT81) },
                        new ProcessMessageParameter { Name = "TRT85", Value = TRT2024Validator.GetValueOrEmpty(parentDocument.TRT85) },
                        new ProcessMessageParameter { Name = "TRT437", Value =TRT2024Validator.GetValueOrEmpty(parentDocument.TRT437) }
                      },
                    DocumentSequence = "parent_doc"
                });
            }
        }

        #endregion //VR.ATO.TRT.W00021

        #endregion Validation Rules

    }
}