﻿using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationIEE;
using Ato.EN.IntegrationServices.CodeGenerationTRT;
using DataContracts;
using System.Collections.Generic;

namespace Ato.CD.Inbound.TRT202402
{
    public class CrossFormValidatorIEE : ICrossFormValidator
    {
        private IEE2021 childDocument { get; set; }
        
        public List<ProcessMessageDocument> response { get; private set; }

        private TRT2024 parentDocument { get; set; }

        public CrossFormValidatorIEE(TRT2024 parent, object child)
        {
            childDocument = (IEE2021)child;
            parentDocument = parent;
            response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOIEE410051();
            VRATOIEE410324();
            VRATOIEE410396();
            VRATOIEE410410();
            
            return response;
        }

        #region Validation Rules

        #region VR.ATO.IEE.410051

        /*  VR.ATO.IEE.410051
            Election Commencement Time must not be entered if Income Year Specified is different to the parent return.

            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF ([IEE49] <> NULL) AND ([IEE45] <> PARENT RETURN:Report.TargetFinancial.Year) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^IEE49 <> NULL) AND (^IEE45 <> ^ParentTargetFinancialYear)

            Data Elements:
            
            RP:^IEE49 = tns:ElectionsCommencementD
            
            RP:^ParentTargetFinancialYear = ParentTargetFinancialYear
            
            RP:^IEE45 = tns:ElectionsInterposedEntityElectionStatusY
        */
        public void VRATOIEE410051()
        {
            bool assertion = childDocument.IEE49.HasValue && !parentDocument.TRT1.GetValueOrDefault().Equals(childDocument.IEE45.GetValueOrDefault());

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410235",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Election Commencement Time must not be present if Income Year Specified is different to the parent return target financial year",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsCommencementD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410051" },
                        new ProcessMessageParameter { Name = "IEE45", Value = childDocument.IEE45.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "IEE49", Value = childDocument.IEE49.GetValueOrDefault().ToString() }
                    }
                });
            }
        }

        #endregion // VR.ATO.IEE.410051

        #region VR.ATO.IEE.410324

        /* VR.ATO.IEE.410324
            Reporting period on Schedule must match reporting period on main form

            WHERE PARENT RETURN EXISTS
            IF [IEE201] <> PARENT RETURN:RP:period.startDate OR [IEE202] <> PARENT RETURN:RP:period.endDate 
                RETURN VALIDATION MESSAGE
            ENDIF
        
            IEE201 = tns:PeriodStartD

            IEE202 = tns:PeriodEndD
        */
        public void VRATOIEE410324()
        {
            bool assertion = childDocument.IEE201 != parentDocument.TRT573 || childDocument.IEE202 != parentDocument.TRT574;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410324",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Reporting period does not match the reporting period supplied on the form it was submitted with",
                    Location = "/tns:IEE/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410324" },
                        new ProcessMessageParameter { Name = "IEE201", Value = childDocument.IEE201.ToString() },
                        new ProcessMessageParameter { Name = "IEE202", Value = childDocument.IEE202.ToString() },
                        new ProcessMessageParameter { Name = "RP:period.startDate", Value = parentDocument.TRT573.ToString() },
                        new ProcessMessageParameter { Name = "RP:period.endDate", Value = parentDocument.TRT574.ToString() }
                    }
                });
            }
        }

        #endregion VR.ATO.IEE.410324

        #region VR.ATO.IEE.410396

        /*  VR.ATO.IEE.410396
            ABN on Schedule must match ABN on main form

            WHERE PARENT RETURN EXISTS
            IF (^IEE162 <> BLANK) AND ([TRT7] <> BLANK) AND (^IEE162 <> [TRT7])
                RETURN VALIDATION MESSAGE
            ENDIF

            Data Elements:
            
            IEE162 = tns:AustralianBusinessNumberId
            
            RP:^ParentABN = ParentABN
        */
        public void VRATOIEE410396()
        {
            bool assertion = !string.IsNullOrWhiteSpace(childDocument.IEE162) && !string.IsNullOrWhiteSpace(parentDocument.TRT7) && !parentDocument.TRT7.Equals(childDocument.IEE162);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied ABN does not match the ABN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied ABN does not match the ABN supplied on the parent form it was submitted with",
                    Location = "/tns:IEE/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410396" },
                        new ProcessMessageParameter { Name = "IEE162", Value = childDocument.IEE162 },
                        new ProcessMessageParameter { Name = "TRT7", Value = parentDocument.TRT7 }
                    }
                });
            }
        }

        #endregion // VR.ATO.IEE.410396

        #region VR.ATO.IEE.410410

        /*  VR.ATO.IEE.410410
            TFN on Schedule must match TFN on main form

            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS 
            IF ([IEE3] <> PARENT RETURN:RP:entity.identifier.TFN)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (^TFN <> ^ParentTFN)

            Data Elements:
            
            ^ParentTFN = ParentTFN
           
            ^TFN = RP
        */
        public void VRATOIEE410410()
        {
            bool assertion = !string.IsNullOrWhiteSpace(childDocument.IEE3) && !string.IsNullOrWhiteSpace(parentDocument.TRT5) && !parentDocument.TRT5.Equals(childDocument.IEE3);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.410115",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN on Schedule must match TFN on main form",
                    Location = "/tns:IEE/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410410" },
                        new ProcessMessageParameter { Name = "IEE3", Value = childDocument.IEE3 },
                        new ProcessMessageParameter { Name = "TRT5", Value = parentDocument.TRT5 }
                    }
                });
            }
        }

        #endregion // VR.ATO.IEE.410410

        #endregion Validation Rules
    }
}