using System;
using System.Collections.Generic;
using Au.Gov.Sbr.Xbrl.Document;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationTRTAMIS
{
    public class TRTAMIS2022Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }
				
        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public TRTAMIS2022 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public TRTAMIS2022 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            TRTAMIS2022 report;
            report = new TRTAMIS2022();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrences being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.TRTAMIS2 = report.RPStartDate;
                    report.TRTAMIS3 = report.RPEndDate;
                    report.TRTAMIS1 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS4 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS5 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}IncomeTax.AttributionManagedInvestmentTrustClass.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS313 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Lodgment.FinalReturn.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS317 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}IncomeTax.AttributionManagedInvestmentTrustClassMembers.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS323 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Liabilities.ThinCapitalisation.DeductionDebt.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS335 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS309 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.Uplift.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS318 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.ForgivenDebtNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS319 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.Incurred.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS320 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.ExemptIncomeNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS321 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.Forgone.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS322 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.Deducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS310 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS311 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPInstantContext
            IEnumerable<SBRContext> sbrRPInstantContexts = null;
            SBRContext sbrRPInstantContext = null;
    
            sbrRPInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInstantContexts != null && sbrRPInstantContexts.Count() > 0)
            {
                sbrRPInstantContext = sbrRPInstantContexts.First();
        
                report.RPInstantId = sbrRPInstantContext.Id;
                report.RPInstantIdentifier = sbrRPInstantContext.EntityIdentifier;
                report.RPInstantIdentifierScheme = sbrRPInstantContext.EntityScheme;
        
                report.RPInstantCount = sbrRPInstantContexts.Count();
                report.RPInstantExists = true;
                report.RPInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContext) + 1;
                report.RPInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContexts.Last()) + 1;
        

                if (report.RPInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInstantIdentifierTFN = report.RPInstantIdentifier;
    
                report.RPInstantInstantDate = ((SBRInstantPeriod)sbrRPInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInstantContexts.Count(); i++)
                {
                    sbrRPInstantContext = sbrRPInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Assets.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS333 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Liabilities.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS334 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInstantContext
    
            #region sbrRPNetExemptionIncomeContext
            IEnumerable<SBRContext> sbrRPNetExemptionIncomeContexts = null;
            SBRContext sbrRPNetExemptionIncomeContext = null;
    
            sbrRPNetExemptionIncomeContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NetExemptionIncome")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNetExemptionIncomeElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNetExemptionIncomeContexts != null && sbrRPNetExemptionIncomeContexts.Count() > 0)
            {
                sbrRPNetExemptionIncomeContext = sbrRPNetExemptionIncomeContexts.First();
        
                report.RPNetExemptionIncomeId = sbrRPNetExemptionIncomeContext.Id;
                report.RPNetExemptionIncomeIdentifier = sbrRPNetExemptionIncomeContext.EntityIdentifier;
                report.RPNetExemptionIncomeIdentifierScheme = sbrRPNetExemptionIncomeContext.EntityScheme;
        
                report.RPNetExemptionIncomeCount = sbrRPNetExemptionIncomeContexts.Count();
                report.RPNetExemptionIncomeExists = true;
                report.RPNetExemptionIncomeOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNetExemptionIncomeContext) + 1;
                report.RPNetExemptionIncomeLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNetExemptionIncomeContexts.Last()) + 1;
        

                if (report.RPNetExemptionIncomeIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNetExemptionIncomeIdentifierTFN = report.RPNetExemptionIncomeIdentifier;
    
                report.RPNetExemptionIncomeStartDate = ((SBRDurationPeriod)sbrRPNetExemptionIncomeContext.Period).DurationStartAsDateObject;
                report.RPNetExemptionIncomeEndDate = ((SBRDurationPeriod)sbrRPNetExemptionIncomeContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPNetExemptionIncomeElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNetExemptionIncomeContexts.Count(); i++)
                {
                    sbrRPNetExemptionIncomeContext = sbrRPNetExemptionIncomeContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNetExemptionIncomeContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNetExemptionIncomeElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNetExemptionIncomeElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNetExemptionIncomeElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPNetExemptionIncomeElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS285 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPNetExemptionIncomeContext
    
            #region sbrRPNonAssessableNonExemptAmountsContext
            IEnumerable<SBRContext> sbrRPNonAssessableNonExemptAmountsContexts = null;
            SBRContext sbrRPNonAssessableNonExemptAmountsContext = null;
    
            sbrRPNonAssessableNonExemptAmountsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonAssessableNonExemptAmounts")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonAssessableNonExemptAmountsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonAssessableNonExemptAmountsContexts != null && sbrRPNonAssessableNonExemptAmountsContexts.Count() > 0)
            {
                sbrRPNonAssessableNonExemptAmountsContext = sbrRPNonAssessableNonExemptAmountsContexts.First();
        
                report.RPNonAssessableNonExemptAmountsId = sbrRPNonAssessableNonExemptAmountsContext.Id;
                report.RPNonAssessableNonExemptAmountsIdentifier = sbrRPNonAssessableNonExemptAmountsContext.EntityIdentifier;
                report.RPNonAssessableNonExemptAmountsIdentifierScheme = sbrRPNonAssessableNonExemptAmountsContext.EntityScheme;
        
                report.RPNonAssessableNonExemptAmountsCount = sbrRPNonAssessableNonExemptAmountsContexts.Count();
                report.RPNonAssessableNonExemptAmountsExists = true;
                report.RPNonAssessableNonExemptAmountsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonAssessableNonExemptAmountsContext) + 1;
                report.RPNonAssessableNonExemptAmountsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonAssessableNonExemptAmountsContexts.Last()) + 1;
        

                if (report.RPNonAssessableNonExemptAmountsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonAssessableNonExemptAmountsIdentifierTFN = report.RPNonAssessableNonExemptAmountsIdentifier;
    
                report.RPNonAssessableNonExemptAmountsStartDate = ((SBRDurationPeriod)sbrRPNonAssessableNonExemptAmountsContext.Period).DurationStartAsDateObject;
                report.RPNonAssessableNonExemptAmountsEndDate = ((SBRDurationPeriod)sbrRPNonAssessableNonExemptAmountsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPNonAssessableNonExemptAmountsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonAssessableNonExemptAmountsContexts.Count(); i++)
                {
                    sbrRPNonAssessableNonExemptAmountsContext = sbrRPNonAssessableNonExemptAmountsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonAssessableNonExemptAmountsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonAssessableNonExemptAmountsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonAssessableNonExemptAmountsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonAssessableNonExemptAmountsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPNonAssessableNonExemptAmountsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS332 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPNonAssessableNonExemptAmountsContext
    
            #region sbrRPOffsetContext
            IEnumerable<SBRContext> sbrRPOffsetContexts = null;
            SBRContext sbrRPOffsetContext = null;
    
            sbrRPOffsetContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Offset")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOffsetElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOffsetContexts != null && sbrRPOffsetContexts.Count() > 0)
            {
                sbrRPOffsetContext = sbrRPOffsetContexts.First();
        
                report.RPOffsetId = sbrRPOffsetContext.Id;
                report.RPOffsetIdentifier = sbrRPOffsetContext.EntityIdentifier;
                report.RPOffsetIdentifierScheme = sbrRPOffsetContext.EntityScheme;
        
                report.RPOffsetCount = sbrRPOffsetContexts.Count();
                report.RPOffsetExists = true;
                report.RPOffsetOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOffsetContext) + 1;
                report.RPOffsetLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOffsetContexts.Last()) + 1;
        

                if (report.RPOffsetIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOffsetIdentifierTFN = report.RPOffsetIdentifier;
    
                report.RPOffsetStartDate = ((SBRDurationPeriod)sbrRPOffsetContext.Period).DurationStartAsDateObject;
                report.RPOffsetEndDate = ((SBRDurationPeriod)sbrRPOffsetContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPOffsetElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOffsetContexts.Count(); i++)
                {
                    sbrRPOffsetContext = sbrRPOffsetContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOffsetContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOffsetElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOffsetElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOffsetElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOffsetElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.TrustComponent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS295 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOffsetElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.UnderstatedEntitlement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS296 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOffsetElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.Overstated.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS297 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOffsetElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS299 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPOffsetElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.TrustDeficitComponent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS300 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOffsetContext
    
            #region sbrRPAssessableIncomeOtherThanCapitalGainsContext
            IEnumerable<SBRContext> sbrRPAssessableIncomeOtherThanCapitalGainsContexts = null;
            SBRContext sbrRPAssessableIncomeOtherThanCapitalGainsContext = null;
    
            sbrRPAssessableIncomeOtherThanCapitalGainsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AssessableIncomeOtherThanCapitalGains")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAssessableIncomeOtherThanCapitalGainsContexts != null && sbrRPAssessableIncomeOtherThanCapitalGainsContexts.Count() > 0)
            {
                sbrRPAssessableIncomeOtherThanCapitalGainsContext = sbrRPAssessableIncomeOtherThanCapitalGainsContexts.First();
        
                report.RPAssessableIncomeOtherThanCapitalGainsId = sbrRPAssessableIncomeOtherThanCapitalGainsContext.Id;
                report.RPAssessableIncomeOtherThanCapitalGainsIdentifier = sbrRPAssessableIncomeOtherThanCapitalGainsContext.EntityIdentifier;
                report.RPAssessableIncomeOtherThanCapitalGainsIdentifierScheme = sbrRPAssessableIncomeOtherThanCapitalGainsContext.EntityScheme;
        
                report.RPAssessableIncomeOtherThanCapitalGainsCount = sbrRPAssessableIncomeOtherThanCapitalGainsContexts.Count();
                report.RPAssessableIncomeOtherThanCapitalGainsExists = true;
                report.RPAssessableIncomeOtherThanCapitalGainsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeOtherThanCapitalGainsContext) + 1;
                report.RPAssessableIncomeOtherThanCapitalGainsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeOtherThanCapitalGainsContexts.Last()) + 1;
        

                if (report.RPAssessableIncomeOtherThanCapitalGainsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAssessableIncomeOtherThanCapitalGainsIdentifierTFN = report.RPAssessableIncomeOtherThanCapitalGainsIdentifier;
    
                report.RPAssessableIncomeOtherThanCapitalGainsStartDate = ((SBRDurationPeriod)sbrRPAssessableIncomeOtherThanCapitalGainsContext.Period).DurationStartAsDateObject;
                report.RPAssessableIncomeOtherThanCapitalGainsEndDate = ((SBRDurationPeriod)sbrRPAssessableIncomeOtherThanCapitalGainsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAssessableIncomeOtherThanCapitalGainsContexts.Count(); i++)
                {
                    sbrRPAssessableIncomeOtherThanCapitalGainsContext = sbrRPAssessableIncomeOtherThanCapitalGainsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAssessableIncomeOtherThanCapitalGainsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS260 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS336 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS337 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.TrustComponent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS314 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.UnderstatedEntitlement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS315 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.Overstated.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS316 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.CarryForwardTrustComponentDeficit.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS265 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS266 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAssessableIncomeOtherThanCapitalGainsContext
    
            #region sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext
            IEnumerable<SBRContext> sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts = null;
            SBRContext sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext = null;
    
            sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AssessableIncomeOtherThanCapitalGains"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExpenseTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Direct")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsDirectElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts != null && sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts.Count() > 0)
            {
                sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext = sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts.First();
        
                report.RPAssessableIncomeOtherThanCapitalGainsDirectId = sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext.Id;
                report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifier = sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext.EntityIdentifier;
                report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifierScheme = sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext.EntityScheme;
        
                report.RPAssessableIncomeOtherThanCapitalGainsDirectCount = sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts.Count();
                report.RPAssessableIncomeOtherThanCapitalGainsDirectExists = true;
                report.RPAssessableIncomeOtherThanCapitalGainsDirectOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext) + 1;
                report.RPAssessableIncomeOtherThanCapitalGainsDirectLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts.Last()) + 1;
        

                if (report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifierTFN = report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifier;
    
                report.RPAssessableIncomeOtherThanCapitalGainsDirectStartDate = ((SBRDurationPeriod)sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext.Period).DurationStartAsDateObject;
                report.RPAssessableIncomeOtherThanCapitalGainsDirectEndDate = ((SBRDurationPeriod)sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsDirectElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts.Count(); i++)
                {
                    sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext = sbrRPAssessableIncomeOtherThanCapitalGainsDirectContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsDirectElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsDirectElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsDirectElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsDirectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.DeductionsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS261 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAssessableIncomeOtherThanCapitalGainsDirectContext
    
            #region sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext
            IEnumerable<SBRContext> sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts = null;
            SBRContext sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext = null;
    
            sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AssessableIncomeOtherThanCapitalGains"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExpenseTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Indirect")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsIndirectElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts != null && sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts.Count() > 0)
            {
                sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext = sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts.First();
        
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectId = sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext.Id;
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifier = sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext.EntityIdentifier;
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifierScheme = sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext.EntityScheme;
        
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectCount = sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts.Count();
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectExists = true;
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext) + 1;
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts.Last()) + 1;
        

                if (report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifierTFN = report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifier;
    
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectStartDate = ((SBRDurationPeriod)sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext.Period).DurationStartAsDateObject;
                report.RPAssessableIncomeOtherThanCapitalGainsIndirectEndDate = ((SBRDurationPeriod)sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsIndirectElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts.Count(); i++)
                {
                    sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext = sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsIndirectElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsIndirectElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsIndirectElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAssessableIncomeOtherThanCapitalGainsIndirectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.DeductionsOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS262 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAssessableIncomeOtherThanCapitalGainsIndirectContext
    
            #region sbrRPAssessableIncomeCapitalGainsContext
            IEnumerable<SBRContext> sbrRPAssessableIncomeCapitalGainsContexts = null;
            SBRContext sbrRPAssessableIncomeCapitalGainsContext = null;
    
            sbrRPAssessableIncomeCapitalGainsContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AssessableIncomeCapitalGains")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAssessableIncomeCapitalGainsContexts != null && sbrRPAssessableIncomeCapitalGainsContexts.Count() > 0)
            {
                sbrRPAssessableIncomeCapitalGainsContext = sbrRPAssessableIncomeCapitalGainsContexts.First();
        
                report.RPAssessableIncomeCapitalGainsId = sbrRPAssessableIncomeCapitalGainsContext.Id;
                report.RPAssessableIncomeCapitalGainsIdentifier = sbrRPAssessableIncomeCapitalGainsContext.EntityIdentifier;
                report.RPAssessableIncomeCapitalGainsIdentifierScheme = sbrRPAssessableIncomeCapitalGainsContext.EntityScheme;
        
                report.RPAssessableIncomeCapitalGainsCount = sbrRPAssessableIncomeCapitalGainsContexts.Count();
                report.RPAssessableIncomeCapitalGainsExists = true;
                report.RPAssessableIncomeCapitalGainsOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeCapitalGainsContext) + 1;
                report.RPAssessableIncomeCapitalGainsLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeCapitalGainsContexts.Last()) + 1;
        

                if (report.RPAssessableIncomeCapitalGainsIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAssessableIncomeCapitalGainsIdentifierTFN = report.RPAssessableIncomeCapitalGainsIdentifier;
    
                report.RPAssessableIncomeCapitalGainsStartDate = ((SBRDurationPeriod)sbrRPAssessableIncomeCapitalGainsContext.Period).DurationStartAsDateObject;
                report.RPAssessableIncomeCapitalGainsEndDate = ((SBRDurationPeriod)sbrRPAssessableIncomeCapitalGainsContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAssessableIncomeCapitalGainsContexts.Count(); i++)
                {
                    sbrRPAssessableIncomeCapitalGainsContext = sbrRPAssessableIncomeCapitalGainsContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAssessableIncomeCapitalGainsContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.CapitalGainsNetCredit.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS324 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS338 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS339 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.TrustComponent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS327 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.UnderstatedEntitlement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS328 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.Overstated.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS329 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS330 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.CarryForwardTrustComponentDeficit.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS331 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAssessableIncomeCapitalGainsContext
    
            #region sbrRPAssessableIncomeCapitalGainsDirectContext
            IEnumerable<SBRContext> sbrRPAssessableIncomeCapitalGainsDirectContexts = null;
            SBRContext sbrRPAssessableIncomeCapitalGainsDirectContext = null;
    
            sbrRPAssessableIncomeCapitalGainsDirectContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AssessableIncomeCapitalGains"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExpenseTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Direct")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAssessableIncomeCapitalGainsDirectElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAssessableIncomeCapitalGainsDirectContexts != null && sbrRPAssessableIncomeCapitalGainsDirectContexts.Count() > 0)
            {
                sbrRPAssessableIncomeCapitalGainsDirectContext = sbrRPAssessableIncomeCapitalGainsDirectContexts.First();
        
                report.RPAssessableIncomeCapitalGainsDirectId = sbrRPAssessableIncomeCapitalGainsDirectContext.Id;
                report.RPAssessableIncomeCapitalGainsDirectIdentifier = sbrRPAssessableIncomeCapitalGainsDirectContext.EntityIdentifier;
                report.RPAssessableIncomeCapitalGainsDirectIdentifierScheme = sbrRPAssessableIncomeCapitalGainsDirectContext.EntityScheme;
        
                report.RPAssessableIncomeCapitalGainsDirectCount = sbrRPAssessableIncomeCapitalGainsDirectContexts.Count();
                report.RPAssessableIncomeCapitalGainsDirectExists = true;
                report.RPAssessableIncomeCapitalGainsDirectOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeCapitalGainsDirectContext) + 1;
                report.RPAssessableIncomeCapitalGainsDirectLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeCapitalGainsDirectContexts.Last()) + 1;
        

                if (report.RPAssessableIncomeCapitalGainsDirectIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAssessableIncomeCapitalGainsDirectIdentifierTFN = report.RPAssessableIncomeCapitalGainsDirectIdentifier;
    
                report.RPAssessableIncomeCapitalGainsDirectStartDate = ((SBRDurationPeriod)sbrRPAssessableIncomeCapitalGainsDirectContext.Period).DurationStartAsDateObject;
                report.RPAssessableIncomeCapitalGainsDirectEndDate = ((SBRDurationPeriod)sbrRPAssessableIncomeCapitalGainsDirectContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAssessableIncomeCapitalGainsDirectElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAssessableIncomeCapitalGainsDirectContexts.Count(); i++)
                {
                    sbrRPAssessableIncomeCapitalGainsDirectContext = sbrRPAssessableIncomeCapitalGainsDirectContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAssessableIncomeCapitalGainsDirectContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAssessableIncomeCapitalGainsDirectElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAssessableIncomeCapitalGainsDirectElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAssessableIncomeCapitalGainsDirectElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsDirectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.DeductionsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS325 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAssessableIncomeCapitalGainsDirectContext
    
            #region sbrRPAssessableIncomeCapitalGainsIndirectContext
            IEnumerable<SBRContext> sbrRPAssessableIncomeCapitalGainsIndirectContexts = null;
            SBRContext sbrRPAssessableIncomeCapitalGainsIndirectContext = null;
    
            sbrRPAssessableIncomeCapitalGainsIndirectContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AMITCharacterTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AssessableIncomeCapitalGains"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExpenseTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Indirect")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAssessableIncomeCapitalGainsIndirectElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAssessableIncomeCapitalGainsIndirectContexts != null && sbrRPAssessableIncomeCapitalGainsIndirectContexts.Count() > 0)
            {
                sbrRPAssessableIncomeCapitalGainsIndirectContext = sbrRPAssessableIncomeCapitalGainsIndirectContexts.First();
        
                report.RPAssessableIncomeCapitalGainsIndirectId = sbrRPAssessableIncomeCapitalGainsIndirectContext.Id;
                report.RPAssessableIncomeCapitalGainsIndirectIdentifier = sbrRPAssessableIncomeCapitalGainsIndirectContext.EntityIdentifier;
                report.RPAssessableIncomeCapitalGainsIndirectIdentifierScheme = sbrRPAssessableIncomeCapitalGainsIndirectContext.EntityScheme;
        
                report.RPAssessableIncomeCapitalGainsIndirectCount = sbrRPAssessableIncomeCapitalGainsIndirectContexts.Count();
                report.RPAssessableIncomeCapitalGainsIndirectExists = true;
                report.RPAssessableIncomeCapitalGainsIndirectOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeCapitalGainsIndirectContext) + 1;
                report.RPAssessableIncomeCapitalGainsIndirectLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAssessableIncomeCapitalGainsIndirectContexts.Last()) + 1;
        

                if (report.RPAssessableIncomeCapitalGainsIndirectIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAssessableIncomeCapitalGainsIndirectIdentifierTFN = report.RPAssessableIncomeCapitalGainsIndirectIdentifier;
    
                report.RPAssessableIncomeCapitalGainsIndirectStartDate = ((SBRDurationPeriod)sbrRPAssessableIncomeCapitalGainsIndirectContext.Period).DurationStartAsDateObject;
                report.RPAssessableIncomeCapitalGainsIndirectEndDate = ((SBRDurationPeriod)sbrRPAssessableIncomeCapitalGainsIndirectContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPAssessableIncomeCapitalGainsIndirectElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAssessableIncomeCapitalGainsIndirectContexts.Count(); i++)
                {
                    sbrRPAssessableIncomeCapitalGainsIndirectContext = sbrRPAssessableIncomeCapitalGainsIndirectContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAssessableIncomeCapitalGainsIndirectContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAssessableIncomeCapitalGainsIndirectElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAssessableIncomeCapitalGainsIndirectElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAssessableIncomeCapitalGainsIndirectElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPAssessableIncomeCapitalGainsIndirectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/trtamis}Expense.DeductionsOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.TRTAMIS326 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPAssessableIncomeCapitalGainsIndirectContext
            return report;

        } // Of Consume Method
    } // Of Class
} // Of Namespace

