
using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationIEE
{
    /// <summary>
    /// XML Consumer for IEE
    /// </summary>
    public class IEE2021XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static IEE2021XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.iee.0004.2021.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;
  
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();        
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count-1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements) 
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
           return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory) 
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader) 
        {
            bool retval = false;

            if ( (reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if(_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if(_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
             }

             //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf , string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if(numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if(numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader) 
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }
      
                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public IEE2021 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            IEE2021 report = new IEE2021();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:IEE");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "IEE" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/iee")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

    
            #region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader,false) && reader.LocalName == "INT")
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true; 
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE200 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxAgentNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE2 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxAgentNumberId
        
                #region ContactNamePersonUnstructuredNameFullNameT
                _currentXPath.Push("/tns:ContactNamePersonUnstructuredNameFullNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ContactNamePersonUnstructuredNameFullNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE186 = currentValue;
                            report.InT_ContactNameCollectionExists = true;
                            report.InT_ContactNameCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ContactNamePersonUnstructuredNameFullNameT
        
                #region ContactTelephone
                _currentXPath.Push("/tns:ContactTelephone");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ContactTelephone")
                {
                    report.InT_ContactTelephoneCollectionExists = true;
                    report.InT_ContactTelephoneCollectionCount += 1;
                    _found = true; 
            
                    #region ElectronicContactAreaC
                    _currentXPath.Push("/tns:ElectronicContactAreaC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactAreaC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE189 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactAreaC
            
                    #region ElectronicContactMinimalN
                    _currentXPath.Push("/tns:ElectronicContactMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectronicContactMinimalN")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE190 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactMinimalN
                } // End of if ContactTelephone node exists
        
                _currentXPath.Pop();
                #endregion ContactTelephone
        
                #region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "Declaration")
                {
                    report.InT_DeclarationCollectionExists = true;
                    report.InT_DeclarationCollectionCount += 1;
                    _found = true; 
            
                    #region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE159 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementTypeC
            
                    #region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IEE179 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementAcceptedI
            
                    #region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.IEE181 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatureD
            
                    #region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE182 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatoryIdentifierT
            
                    #region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonUnstructuredNameFullNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE184 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonUnstructuredNameFullNameT
                } // End of if Declaration node exists
        
                _currentXPath.Pop();
                #endregion Declaration
            } // End of if INT node exists
    
            _currentXPath.Pop();
            #endregion INT
    
            #region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader,true) && reader.LocalName == "RP")
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true; 
        
                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "TaxFileNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE3 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxFileNumberId
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE162 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region OrganisationDetailsOrganisationBranchC
                _currentXPath.Push("/tns:OrganisationDetailsOrganisationBranchC");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetailsOrganisationBranchC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE163 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationDetailsOrganisationBranchC
        
                #region LodgmentPeriodYearY
                _currentXPath.Push("/tns:LodgmentPeriodYearY");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "LodgmentPeriodYearY")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.IEE1 = currentIntValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion LodgmentPeriodYearY
        
                #region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodStartD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.IEE201 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodStartD
        
                #region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodEndD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.IEE202 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodEndD
        
                #region CompletionHoursN
                _currentXPath.Push("/tns:CompletionHoursN");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CompletionHoursN")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IEE94 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion CompletionHoursN
        
                #region CurrentNameOrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:CurrentNameOrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CurrentNameOrganisationNameDetailsOrganisationalNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE4 = currentValue;
                            report.RP_CurrentNameCollectionExists = true;
                            report.RP_CurrentNameCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion CurrentNameOrganisationNameDetailsOrganisationalNameT
        
                #region AustralianCompanyNumberId
                _currentXPath.Push("/tns:AustralianCompanyNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianCompanyNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE100 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianCompanyNumberId
        
                #region AustralianRegisteredBodyNumberId
                _currentXPath.Push("/tns:AustralianRegisteredBodyNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianRegisteredBodyNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE101 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianRegisteredBodyNumberId
        
                #region CurrentPostalAddress
                _currentXPath.Push("/tns:CurrentPostalAddress");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CurrentPostalAddress")
                {
                    report.RP_CurrentPostalAddressCollectionExists = true;
                    report.RP_CurrentPostalAddressCollectionCount += 1;
                    _found = true; 
            
                    #region AddressDetailsLine1T
                    _currentXPath.Push("/tns:AddressDetailsLine1T");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLine1T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE5 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsLine1T
            
                    #region AddressDetailsLine2T
                    _currentXPath.Push("/tns:AddressDetailsLine2T");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsLine2T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE122 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsLine2T
            
                    #region AddressDetailsLocalityNameT
                    _currentXPath.Push("/tns:AddressDetailsLocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLocalityNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE6 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsLocalityNameT
            
                    #region AddressDetailsPostcodeT
                    _currentXPath.Push("/tns:AddressDetailsPostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsPostcodeT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE8 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsPostcodeT
            
                    #region AddressDetailsStateOrTerritoryC
                    _currentXPath.Push("/tns:AddressDetailsStateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsStateOrTerritoryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE7 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsStateOrTerritoryC
            
                    #region AddressDetailsCountryC
                    _currentXPath.Push("/tns:AddressDetailsCountryC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsCountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE111 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsCountryC
                } // End of if CurrentPostalAddress node exists
        
                _currentXPath.Pop();
                #endregion CurrentPostalAddress
        
                #region PreviousNameOrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:PreviousNameOrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PreviousNameOrganisationNameDetailsOrganisationalNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IEE10 = currentValue;
                            report.RP_PreviousNameCollectionExists = true;
                            report.RP_PreviousNameCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PreviousNameOrganisationNameDetailsOrganisationalNameT
        
                #region PreviousPostalAddress
                _currentXPath.Push("/tns:PreviousPostalAddress");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PreviousPostalAddress")
                {
                    report.RP_PreviousPostalAddressCollectionExists = true;
                    report.RP_PreviousPostalAddressCollectionCount += 1;
                    _found = true; 
            
                    #region AddressDetailsLine1T
                    _currentXPath.Push("/tns:AddressDetailsLine1T");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLine1T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE11 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsLine1T
            
                    #region AddressDetailsLine2T
                    _currentXPath.Push("/tns:AddressDetailsLine2T");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsLine2T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE126 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsLine2T
            
                    #region AddressDetailsLocalityNameT
                    _currentXPath.Push("/tns:AddressDetailsLocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLocalityNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE12 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsLocalityNameT
            
                    #region AddressDetailsPostcodeT
                    _currentXPath.Push("/tns:AddressDetailsPostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsPostcodeT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE14 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsPostcodeT
            
                    #region AddressDetailsStateOrTerritoryC
                    _currentXPath.Push("/tns:AddressDetailsStateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsStateOrTerritoryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE13 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsStateOrTerritoryC
            
                    #region AddressDetailsCountryC
                    _currentXPath.Push("/tns:AddressDetailsCountryC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsCountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE112 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AddressDetailsCountryC
                } // End of if PreviousPostalAddress node exists
        
                _currentXPath.Pop();
                #endregion PreviousPostalAddress
        
                #region InterposedEntityElectionRevocationDetails
                _currentXPath.Push("/tns:InterposedEntityElectionRevocationDetails");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "InterposedEntityElectionRevocationDetails")
                {
                    report.RP_InterposedEntityElectionRevocationDetailsCollectionExists = true;
                    report.RP_InterposedEntityElectionRevocationDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region PartyTypeC
                    _currentXPath.Push("/tns:PartyTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PartyTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE16 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PartyTypeC
            
                    #region ElectionsInterposedEntityElectionOrRevocationC
                    _currentXPath.Push("/tns:ElectionsInterposedEntityElectionOrRevocationC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsInterposedEntityElectionOrRevocationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE116 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsInterposedEntityElectionOrRevocationC
            
                    #region ElectionsCommencementD
                    _currentXPath.Push("/tns:ElectionsCommencementD");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsCommencementD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.IEE49 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsCommencementD
            
                    #region ElectionsY
                    _currentXPath.Push("/tns:ElectionsY");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsY")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.IEE82 = currentIntValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsY
            
                    #region CentralManagementAndControlInformation
                    _currentXPath.Push("/tns:CentralManagementAndControlInformation");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "CentralManagementAndControlInformation")
                    {
                        report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformationCollectionExists = true;
                        report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformationCollectionCount += 1;
                        _found = true; 
                
                        #region ResidencyCentralManagementAndControlOutsideAustraliaI
                        _currentXPath.Push("/tns:ResidencyCentralManagementAndControlOutsideAustraliaI");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ResidencyCentralManagementAndControlOutsideAustraliaI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IEE17 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ResidencyCentralManagementAndControlOutsideAustraliaI
                
                        #region ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI
                        _currentXPath.Push("/tns:ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IEE18 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI
                
                        #region ControlOutsideAustraliaPartPeriodDetails
                
                        _currentXPath.Push("/tns:ControlOutsideAustraliaPartPeriodDetailsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ControlOutsideAustraliaPartPeriodDetailsCollection")
                        {
                            report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection = new List<IEE2021.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetails>();
                            report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionExists = true;
                
                            _found = true;
                
                            while ( ReadToNextElement(reader,false) && reader.LocalName == "ControlOutsideAustraliaPartPeriodDetails" ) 
                            {
                                _found = true;
                
                                IEE2021.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetails controlOutsideAustraliaPartPeriodDetails = new IEE2021.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetails();
                                report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollection.Add(controlOutsideAustraliaPartPeriodDetails);
                                report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionCount += 1;
                                controlOutsideAustraliaPartPeriodDetails.OccurrenceIndex = report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionCount;
                
                                _currentXPath.Push("/tns:ControlOutsideAustraliaPartPeriodDetails[" + report.RP_InterposedEntityElectionRevocationDetails_CentralManagementAndControlInformation_ControlOutsideAustraliaPartPeriodDetailsCollectionCount + "]");
                        
                                #region StartD
                                _currentXPath.Push("/tns:StartD");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                controlOutsideAustraliaPartPeriodDetails.IEE19 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion StartD
                        
                                #region EndD
                                _currentXPath.Push("/tns:EndD");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                controlOutsideAustraliaPartPeriodDetails.IEE20 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion EndD
                            _currentXPath.Pop();
                            } // End of while loop for ControlOutsideAustraliaPartPeriodDetails
                        } // End of if ControlOutsideAustraliaPartPeriodDetailsCollection node exists
                
                        _currentXPath.Pop();
                        #endregion ControlOutsideAustraliaPartPeriodDetails
                    } // End of if CentralManagementAndControlInformation node exists
            
                    _currentXPath.Pop();
                    #endregion CentralManagementAndControlInformation
            
                    #region NonResidentInformation
                    _currentXPath.Push("/tns:NonResidentInformation");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NonResidentInformation")
                    {
                        report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformationCollectionExists = true;
                        report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformationCollectionCount += 1;
                        _found = true; 
                
                        #region ResidencyTaxPurposesOrganisationStatusC
                        _currentXPath.Push("/tns:ResidencyTaxPurposesOrganisationStatusC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ResidencyTaxPurposesOrganisationStatusC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE21 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ResidencyTaxPurposesOrganisationStatusC
                
                        #region ResidencyNonResidentFullPeriodI
                        _currentXPath.Push("/tns:ResidencyNonResidentFullPeriodI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ResidencyNonResidentFullPeriodI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IEE22 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ResidencyNonResidentFullPeriodI
                
                        #region CompanyNonResidentPartPeriodDetails
                
                        _currentXPath.Push("/tns:CompanyNonResidentPartPeriodDetailsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "CompanyNonResidentPartPeriodDetailsCollection")
                        {
                            report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection = new List<IEE2021.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetails>();
                            report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionExists = true;
                
                            _found = true;
                
                            while ( ReadToNextElement(reader,false) && reader.LocalName == "CompanyNonResidentPartPeriodDetails" ) 
                            {
                                _found = true;
                
                                IEE2021.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetails companyNonResidentPartPeriodDetails = new IEE2021.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetails();
                                report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollection.Add(companyNonResidentPartPeriodDetails);
                                report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionCount += 1;
                                companyNonResidentPartPeriodDetails.OccurrenceIndex = report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionCount;
                
                                _currentXPath.Push("/tns:CompanyNonResidentPartPeriodDetails[" + report.RP_InterposedEntityElectionRevocationDetails_NonResidentInformation_CompanyNonResidentPartPeriodDetailsCollectionCount + "]");
                        
                                #region StartD
                                _currentXPath.Push("/tns:StartD");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                companyNonResidentPartPeriodDetails.IEE23 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion StartD
                        
                                #region EndD
                                _currentXPath.Push("/tns:EndD");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                companyNonResidentPartPeriodDetails.IEE24 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion EndD
                            _currentXPath.Pop();
                            } // End of while loop for CompanyNonResidentPartPeriodDetails
                        } // End of if CompanyNonResidentPartPeriodDetailsCollection node exists
                
                        _currentXPath.Pop();
                        #endregion CompanyNonResidentPartPeriodDetails
                    } // End of if NonResidentInformation node exists
            
                    _currentXPath.Pop();
                    #endregion NonResidentInformation
            
                    #region TrusteeOrPartner
            
                    _currentXPath.Push("/tns:TrusteeOrPartnerCollection");
                    // 4. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TrusteeOrPartnerCollection")
                    {
                        report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection = new List<IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner>();
                        report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionExists = true;
            
                        _found = true;
            
                        while ( ReadToNextElement(reader,false) && reader.LocalName == "TrusteeOrPartner" ) 
                        {
                            _found = true;
            
                            IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner trusteeOrPartner = new IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner();
                            report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollection.Add(trusteeOrPartner);
                            report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionCount += 1;
                            trusteeOrPartner.OccurrenceIndex = report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionCount;
            
                            _currentXPath.Push("/tns:TrusteeOrPartner[" + report.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartnerCollectionCount + "]");
                    
                            #region TaxFileNumberId
                            _currentXPath.Push("/tns:TaxFileNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trusteeOrPartner.IEE25 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxFileNumberId
                    
                            #region TaxFileNumberInexistentI
                            _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            trusteeOrPartner.IEE26 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxFileNumberInexistentI
                    
                            #region ResidencyNonResidentFullPeriodI
                            _currentXPath.Push("/tns:ResidencyNonResidentFullPeriodI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ResidencyNonResidentFullPeriodI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            trusteeOrPartner.IEE42 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ResidencyNonResidentFullPeriodI
                    
                            #region ResidencyTaxPurposesPersonStatusI
                            _currentXPath.Push("/tns:ResidencyTaxPurposesPersonStatusI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ResidencyTaxPurposesPersonStatusI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            trusteeOrPartner.IEE41 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ResidencyTaxPurposesPersonStatusI
                    
                            #region NonResidentPartPeriodDetails
                    
                            _currentXPath.Push("/tns:NonResidentPartPeriodDetailsCollection");
                            // 4. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "NonResidentPartPeriodDetailsCollection")
                            {
                                trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection = new List<IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetails>();
                                trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionExists = true;
                    
                                _found = true;
                    
                                while ( ReadToNextElement(reader,false) && reader.LocalName == "NonResidentPartPeriodDetails" ) 
                                {
                                    _found = true;
                    
                                    IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetails nonResidentPartPeriodDetails = new IEE2021.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetails();
                                    trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollection.Add(nonResidentPartPeriodDetails);
                                    trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionCount += 1;
                                    nonResidentPartPeriodDetails.OccurrenceIndex = trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionCount;
                    
                                    _currentXPath.Push("/tns:NonResidentPartPeriodDetails[" + trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_NonResidentPartPeriodDetailsCollectionCount + "]");
                            
                                    #region StartD
                                    _currentXPath.Push("/tns:StartD");
                                    //6. use case
                                    if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                                {
                                                    nonResidentPartPeriodDetails.IEE43 = currentDateTimeValue.Date;
                                                }
                                            }
                                        }
                                        _found = true;
                                    }
                            
                                    _currentXPath.Pop();
                                    #endregion StartD
                            
                                    #region EndD
                                    _currentXPath.Push("/tns:EndD");
                                    //6. use case
                                    if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                                {
                                                    nonResidentPartPeriodDetails.IEE44 = currentDateTimeValue.Date;
                                                }
                                            }
                                        }
                                        _found = true;
                                    }
                            
                                    _currentXPath.Pop();
                                    #endregion EndD
                                _currentXPath.Pop();
                                } // End of while loop for NonResidentPartPeriodDetails
                            } // End of if NonResidentPartPeriodDetailsCollection node exists
                    
                            _currentXPath.Pop();
                            #endregion NonResidentPartPeriodDetails
                    
                            #region OrganisationNameDetailsOrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trusteeOrPartner.IEE32 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OrganisationNameDetailsOrganisationalNameT
                    
                            #region IndividualNameDetails
                            _currentXPath.Push("/tns:IndividualNameDetails");
                            //3. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IndividualNameDetails")
                            {
                                trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_IndividualNameDetailsCollectionExists = true;
                                trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_IndividualNameDetailsCollectionCount += 1;
                                _found = true; 
                        
                                #region PersonNameDetailsTitleT
                                _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE27 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion PersonNameDetailsTitleT
                        
                                #region PersonNameDetailsNameSuffixT
                                _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE29 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion PersonNameDetailsNameSuffixT
                        
                                #region PersonNameDetailsFamilyNameT
                                _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE28 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion PersonNameDetailsFamilyNameT
                        
                                #region PersonNameDetailsGivenNameT
                                _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsGivenNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE30 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion PersonNameDetailsGivenNameT
                        
                                #region PersonNameDetailsOtherGivenNameT
                                _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE31 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion PersonNameDetailsOtherGivenNameT
                            } // End of if IndividualNameDetails node exists
                    
                            _currentXPath.Pop();
                            #endregion IndividualNameDetails
                    
                            #region AustralianCompanyNumberId
                            _currentXPath.Push("/tns:AustralianCompanyNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianCompanyNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trusteeOrPartner.IEE34 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AustralianCompanyNumberId
                    
                            #region AustralianRegisteredBodyNumberId
                            _currentXPath.Push("/tns:AustralianRegisteredBodyNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianRegisteredBodyNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trusteeOrPartner.IEE95 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AustralianRegisteredBodyNumberId
                    
                            #region CurrentPostalAddress
                            _currentXPath.Push("/tns:CurrentPostalAddress");
                            //3. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "CurrentPostalAddress")
                            {
                                trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_CurrentPostalAddressCollectionExists = true;
                                trusteeOrPartner.RP_InterposedEntityElectionRevocationDetails_TrusteeOrPartner_CurrentPostalAddressCollectionCount += 1;
                                _found = true; 
                        
                                #region AddressDetailsLine1T
                                _currentXPath.Push("/tns:AddressDetailsLine1T");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLine1T")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE35 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion AddressDetailsLine1T
                        
                                #region AddressDetailsLine2T
                                _currentXPath.Push("/tns:AddressDetailsLine2T");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsLine2T")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE36 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion AddressDetailsLine2T
                        
                                #region AddressDetailsLocalityNameT
                                _currentXPath.Push("/tns:AddressDetailsLocalityNameT");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLocalityNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE37 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion AddressDetailsLocalityNameT
                        
                                #region AddressDetailsPostcodeT
                                _currentXPath.Push("/tns:AddressDetailsPostcodeT");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsPostcodeT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE39 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion AddressDetailsPostcodeT
                        
                                #region AddressDetailsStateOrTerritoryC
                                _currentXPath.Push("/tns:AddressDetailsStateOrTerritoryC");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsStateOrTerritoryC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE38 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion AddressDetailsStateOrTerritoryC
                        
                                #region AddressDetailsCountryC
                                _currentXPath.Push("/tns:AddressDetailsCountryC");
                                //6. use case
                                if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsCountryC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            trusteeOrPartner.IEE113 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion AddressDetailsCountryC
                            } // End of if CurrentPostalAddress node exists
                    
                            _currentXPath.Pop();
                            #endregion CurrentPostalAddress
                        _currentXPath.Pop();
                        } // End of while loop for TrusteeOrPartner
                    } // End of if TrusteeOrPartnerCollection node exists
            
                    _currentXPath.Pop();
                    #endregion TrusteeOrPartner
            
                    #region ElectionDetails
                    _currentXPath.Push("/tns:ElectionDetails");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionDetails")
                    {
                        report.RP_InterposedEntityElectionRevocationDetails_ElectionDetailsCollectionExists = true;
                        report.RP_InterposedEntityElectionRevocationDetails_ElectionDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region ElectionsInterposedEntityElectionStatusY
                        _currentXPath.Push("/tns:ElectionsInterposedEntityElectionStatusY");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsInterposedEntityElectionStatusY")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Int32.TryParse(currentValue, out currentIntValue))
                                    {
                                        report.IEE45 = currentIntValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ElectionsInterposedEntityElectionStatusY
                
                        #region ElectionsDateSpecifiedD
                        _currentXPath.Push("/tns:ElectionsDateSpecifiedD");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ElectionsDateSpecifiedD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        report.IEE46 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ElectionsDateSpecifiedD
                
                        #region SubstitutedAccountingPeriodDetails
                        _currentXPath.Push("/tns:SubstitutedAccountingPeriodDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SubstitutedAccountingPeriodDetails")
                        {
                            report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionExists = true;
                            report.RP_InterposedEntityElectionRevocationDetails_ElectionDetails_SubstitutedAccountingPeriodDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region StartD
                            _currentXPath.Push("/tns:StartD");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            report.IEE47 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion StartD
                    
                            #region EndD
                            _currentXPath.Push("/tns:EndD");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            report.IEE48 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion EndD
                        } // End of if SubstitutedAccountingPeriodDetails node exists
                
                        _currentXPath.Pop();
                        #endregion SubstitutedAccountingPeriodDetails
                    } // End of if ElectionDetails node exists
            
                    _currentXPath.Pop();
                    #endregion ElectionDetails
            
                    #region FamilyTrust
                    _currentXPath.Push("/tns:FamilyTrust");
                    //3. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "FamilyTrust")
                    {
                        report.RP_InterposedEntityElectionRevocationDetails_FamilyTrustCollectionExists = true;
                        report.RP_InterposedEntityElectionRevocationDetails_FamilyTrustCollectionCount += 1;
                        _found = true; 
                
                        #region ElectionsCommencementD
                        _currentXPath.Push("/tns:ElectionsCommencementD");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsCommencementD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        report.IEE59 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ElectionsCommencementD
                
                        #region ElectionsElectionStatusY
                        _currentXPath.Push("/tns:ElectionsElectionStatusY");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsElectionStatusY")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Int32.TryParse(currentValue, out currentIntValue))
                                    {
                                        report.IEE58 = currentIntValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ElectionsElectionStatusY
                
                        #region SubstitutedAccountingPeriodDetails
                        _currentXPath.Push("/tns:SubstitutedAccountingPeriodDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SubstitutedAccountingPeriodDetails")
                        {
                            report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionExists = true;
                            report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_SubstitutedAccountingPeriodDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region StartD
                            _currentXPath.Push("/tns:StartD");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "StartD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            report.IEE60 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion StartD
                    
                            #region EndD
                            _currentXPath.Push("/tns:EndD");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "EndD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            report.IEE61 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion EndD
                        } // End of if SubstitutedAccountingPeriodDetails node exists
                
                        _currentXPath.Pop();
                        #endregion SubstitutedAccountingPeriodDetails
                
                        #region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE50 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberId
                
                        #region TaxFileNumberInexistentI
                        _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IEE51 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberInexistentI
                
                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE52 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT
                
                        #region CurrentPostalAddress
                        _currentXPath.Push("/tns:CurrentPostalAddress");
                        //3. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "CurrentPostalAddress")
                        {
                            report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_CurrentPostalAddressCollectionExists = true;
                            report.RP_InterposedEntityElectionRevocationDetails_FamilyTrust_CurrentPostalAddressCollectionCount += 1;
                            _found = true; 
                    
                            #region AddressDetailsLine1T
                            _currentXPath.Push("/tns:AddressDetailsLine1T");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLine1T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE53 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AddressDetailsLine1T
                    
                            #region AddressDetailsLine2T
                            _currentXPath.Push("/tns:AddressDetailsLine2T");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsLine2T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE134 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AddressDetailsLine2T
                    
                            #region AddressDetailsLocalityNameT
                            _currentXPath.Push("/tns:AddressDetailsLocalityNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsLocalityNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE54 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AddressDetailsLocalityNameT
                    
                            #region AddressDetailsPostcodeT
                            _currentXPath.Push("/tns:AddressDetailsPostcodeT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsPostcodeT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE56 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AddressDetailsPostcodeT
                    
                            #region AddressDetailsStateOrTerritoryC
                            _currentXPath.Push("/tns:AddressDetailsStateOrTerritoryC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AddressDetailsStateOrTerritoryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE55 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AddressDetailsStateOrTerritoryC
                    
                            #region AddressDetailsCountryC
                            _currentXPath.Push("/tns:AddressDetailsCountryC");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "AddressDetailsCountryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE155 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AddressDetailsCountryC
                        } // End of if CurrentPostalAddress node exists
                
                        _currentXPath.Pop();
                        #endregion CurrentPostalAddress
                    } // End of if FamilyTrust node exists
            
                    _currentXPath.Pop();
                    #endregion FamilyTrust
            
                    #region SpecifiedIndividual
                    _currentXPath.Push("/tns:SpecifiedIndividual");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "SpecifiedIndividual")
                    {
                        report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionExists = true;
                        report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividualCollectionCount += 1;
                        _found = true; 
                
                        #region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE62 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberId
                
                        #region TaxFileNumberInexistentI
                        _currentXPath.Push("/tns:TaxFileNumberInexistentI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberInexistentI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IEE63 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberInexistentI
                
                        #region PersonDemographicDetailsBirthD
                        _currentXPath.Push("/tns:PersonDemographicDetailsBirthD");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "PersonDemographicDetailsBirthD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        report.IEE64 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonDemographicDetailsBirthD
                
                        #region PersonNameDetailsTitleT
                        _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE65 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsTitleT
                
                        #region PersonNameDetailsNameSuffixT
                        _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE67 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsNameSuffixT
                
                        #region PersonNameDetailsFamilyNameT
                        _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE66 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsFamilyNameT
                
                        #region PersonNameDetailsGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsGivenNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE68 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsGivenNameT
                
                        #region PersonNameDetailsOtherGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE69 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsOtherGivenNameT
                
                        #region CurrentResidentialAddressDetails
                        _currentXPath.Push("/tns:CurrentResidentialAddressDetails");
                        //3. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "CurrentResidentialAddressDetails")
                        {
                            report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividual_CurrentResidentialAddressDetailsCollectionExists = true;
                            report.RP_InterposedEntityElectionRevocationDetails_SpecifiedIndividual_CurrentResidentialAddressDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region Line1T
                            _currentXPath.Push("/tns:Line1T");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE70 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion Line1T
                    
                            #region Line2T
                            _currentXPath.Push("/tns:Line2T");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE138 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion Line2T
                    
                            #region LocalityNameT
                            _currentXPath.Push("/tns:LocalityNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE71 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion LocalityNameT
                    
                            #region PostcodeT
                            _currentXPath.Push("/tns:PostcodeT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE73 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PostcodeT
                    
                            #region StateOrTerritoryC
                            _currentXPath.Push("/tns:StateOrTerritoryC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE72 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion StateOrTerritoryC
                    
                            #region CountryC
                            _currentXPath.Push("/tns:CountryC");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "CountryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.IEE114 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion CountryC
                        } // End of if CurrentResidentialAddressDetails node exists
                
                        _currentXPath.Pop();
                        #endregion CurrentResidentialAddressDetails
                    } // End of if SpecifiedIndividual node exists
            
                    _currentXPath.Pop();
                    #endregion SpecifiedIndividual
                } // End of if InterposedEntityElectionRevocationDetails node exists
        
                _currentXPath.Pop();
                #endregion InterposedEntityElectionRevocationDetails
        
                #region DeclarationOfTrusteeOrPartner
        
                _currentXPath.Push("/tns:DeclarationOfTrusteeOrPartnerCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "DeclarationOfTrusteeOrPartnerCollection")
                {
                    report.RP_DeclarationOfTrusteeOrPartnerCollection = new List<IEE2021.RP_DeclarationOfTrusteeOrPartner>();
                    report.RP_DeclarationOfTrusteeOrPartnerCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "DeclarationOfTrusteeOrPartner" ) 
                    {
                        _found = true;
        
                        IEE2021.RP_DeclarationOfTrusteeOrPartner declarationOfTrusteeOrPartner = new IEE2021.RP_DeclarationOfTrusteeOrPartner();
                        report.RP_DeclarationOfTrusteeOrPartnerCollection.Add(declarationOfTrusteeOrPartner);
                        report.RP_DeclarationOfTrusteeOrPartnerCollectionCount += 1;
                        declarationOfTrusteeOrPartner.OccurrenceIndex = report.RP_DeclarationOfTrusteeOrPartnerCollectionCount;
        
                        _currentXPath.Push("/tns:DeclarationOfTrusteeOrPartner[" + report.RP_DeclarationOfTrusteeOrPartnerCollectionCount + "]");
                
                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    declarationOfTrusteeOrPartner.IEE300 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT
                
                        #region TrusteeOrPartnerIndividualNameDetails
                        _currentXPath.Push("/tns:TrusteeOrPartnerIndividualNameDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TrusteeOrPartnerIndividualNameDetails")
                        {
                            declarationOfTrusteeOrPartner.RP_DeclarationOfTrusteeOrPartner_TrusteeOrPartnerIndividualNameDetailsCollectionExists = true;
                            declarationOfTrusteeOrPartner.RP_DeclarationOfTrusteeOrPartner_TrusteeOrPartnerIndividualNameDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region PersonNameDetailsTitleT
                            _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrusteeOrPartner.IEE301 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsTitleT
                    
                            #region PersonNameDetailsNameSuffixT
                            _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrusteeOrPartner.IEE302 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsNameSuffixT
                    
                            #region PersonNameDetailsFamilyNameT
                            _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrusteeOrPartner.IEE303 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsFamilyNameT
                    
                            #region PersonNameDetailsGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrusteeOrPartner.IEE304 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsGivenNameT
                    
                            #region PersonNameDetailsOtherGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        declarationOfTrusteeOrPartner.IEE305 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsOtherGivenNameT
                        } // End of if TrusteeOrPartnerIndividualNameDetails node exists
                
                        _currentXPath.Pop();
                        #endregion TrusteeOrPartnerIndividualNameDetails
                
                        #region StatementTypeC
                        _currentXPath.Push("/tns:StatementTypeC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    declarationOfTrusteeOrPartner.IEE193 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StatementTypeC
                
                        #region StatementAcceptedI
                        _currentXPath.Push("/tns:StatementAcceptedI");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        declarationOfTrusteeOrPartner.IEE146 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion StatementAcceptedI
                
                        #region SignatureD
                        _currentXPath.Push("/tns:SignatureD");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        declarationOfTrusteeOrPartner.IEE90 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SignatureD
                
                        #region SignatoryIdentifierT
                        _currentXPath.Push("/tns:SignatoryIdentifierT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    declarationOfTrusteeOrPartner.IEE195 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SignatoryIdentifierT
                    _currentXPath.Pop();
                    } // End of while loop for DeclarationOfTrusteeOrPartner
                } // End of if DeclarationOfTrusteeOrPartnerCollection node exists
        
                _currentXPath.Pop();
                #endregion DeclarationOfTrusteeOrPartner
        
                #region DeclarationOfPublicOfficer
                _currentXPath.Push("/tns:DeclarationOfPublicOfficer");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "DeclarationOfPublicOfficer")
                {
                    report.RP_DeclarationOfPublicOfficerCollectionExists = true;
                    report.RP_DeclarationOfPublicOfficerCollectionCount += 1;
                    _found = true; 
            
                    #region PublicOfficerNameDetails
                    _currentXPath.Push("/tns:PublicOfficerNameDetails");
                    //3. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PublicOfficerNameDetails")
                    {
                        report.RP_DeclarationOfPublicOfficer_PublicOfficerNameDetailsCollectionExists = true;
                        report.RP_DeclarationOfPublicOfficer_PublicOfficerNameDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region PersonNameDetailsTitleT
                        _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE83 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsTitleT
                
                        #region PersonNameDetailsNameSuffixT
                        _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE85 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsNameSuffixT
                
                        #region PersonNameDetailsFamilyNameT
                        _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE84 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsFamilyNameT
                
                        #region PersonNameDetailsGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsGivenNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE86 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsGivenNameT
                
                        #region PersonNameDetailsOtherGivenNameT
                        _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IEE87 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PersonNameDetailsOtherGivenNameT
                    } // End of if PublicOfficerNameDetails node exists
            
                    _currentXPath.Pop();
                    #endregion PublicOfficerNameDetails
            
                    #region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE147 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementTypeC
            
                    #region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IEE192 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementAcceptedI
            
                    #region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.IEE191 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatureD
            
                    #region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IEE148 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatoryIdentifierT
                } // End of if DeclarationOfPublicOfficer node exists
        
                _currentXPath.Pop();
                #endregion DeclarationOfPublicOfficer
            } // End of if RP node exists
    
            _currentXPath.Pop();
            #endregion RP

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "INT#AustralianBusinessNumberId,INT#TaxAgentNumberId,ContactNamePersonUnstructuredNameFullNameT,ContactTelephone#ElectronicContactAreaC,ContactTelephone#ElectronicContactMinimalN,ContactTelephone,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatureD,Declaration#SignatoryIdentifierT,Declaration#PersonUnstructuredNameFullNameT,Declaration,INT,RP#TaxFileNumberId,RP#AustralianBusinessNumberId,RP#OrganisationDetailsOrganisationBranchC,RP#LodgmentPeriodYearY,RP#PeriodStartD,RP#PeriodEndD,RP#CompletionHoursN,CurrentNameOrganisationNameDetailsOrganisationalNameT,RP#AustralianCompanyNumberId,RP#AustralianRegisteredBodyNumberId,CurrentPostalAddress#AddressDetailsLine1T,CurrentPostalAddress#AddressDetailsLine2T,CurrentPostalAddress#AddressDetailsLocalityNameT,CurrentPostalAddress#AddressDetailsPostcodeT,CurrentPostalAddress#AddressDetailsStateOrTerritoryC,CurrentPostalAddress#AddressDetailsCountryC,CurrentPostalAddress,PreviousNameOrganisationNameDetailsOrganisationalNameT,PreviousPostalAddress#AddressDetailsLine1T,PreviousPostalAddress#AddressDetailsLine2T,PreviousPostalAddress#AddressDetailsLocalityNameT,PreviousPostalAddress#AddressDetailsPostcodeT,PreviousPostalAddress#AddressDetailsStateOrTerritoryC,PreviousPostalAddress#AddressDetailsCountryC,PreviousPostalAddress,InterposedEntityElectionRevocationDetails#PartyTypeC,InterposedEntityElectionRevocationDetails#ElectionsInterposedEntityElectionOrRevocationC,InterposedEntityElectionRevocationDetails#ElectionsCommencementD,InterposedEntityElectionRevocationDetails#ElectionsY,CentralManagementAndControlInformation#ResidencyCentralManagementAndControlOutsideAustraliaI,CentralManagementAndControlInformation#ResidencyCentralManagementAndControlOutsideAustraliaFullPeriodI,ControlOutsideAustraliaPartPeriodDetails#StartD,ControlOutsideAustraliaPartPeriodDetails#EndD,ControlOutsideAustraliaPartPeriodDetails,ControlOutsideAustraliaPartPeriodDetailsCollection,CentralManagementAndControlInformation,NonResidentInformation#ResidencyTaxPurposesOrganisationStatusC,NonResidentInformation#ResidencyNonResidentFullPeriodI,CompanyNonResidentPartPeriodDetails#StartD,CompanyNonResidentPartPeriodDetails#EndD,CompanyNonResidentPartPeriodDetails,CompanyNonResidentPartPeriodDetailsCollection,NonResidentInformation,TrusteeOrPartner#TaxFileNumberId,TrusteeOrPartner#TaxFileNumberInexistentI,TrusteeOrPartner#ResidencyNonResidentFullPeriodI,TrusteeOrPartner#ResidencyTaxPurposesPersonStatusI,NonResidentPartPeriodDetails#StartD,NonResidentPartPeriodDetails#EndD,NonResidentPartPeriodDetails,NonResidentPartPeriodDetailsCollection,TrusteeOrPartner#OrganisationNameDetailsOrganisationalNameT,TrusteeOrPartner#PersonNameDetailsTitleT,TrusteeOrPartner#PersonNameDetailsNameSuffixT,TrusteeOrPartner#PersonNameDetailsFamilyNameT,TrusteeOrPartner#PersonNameDetailsGivenNameT,TrusteeOrPartner#PersonNameDetailsOtherGivenNameT,IndividualNameDetails,TrusteeOrPartner#AustralianCompanyNumberId,TrusteeOrPartner#AustralianRegisteredBodyNumberId,TrusteeOrPartner#AddressDetailsLine1T,TrusteeOrPartner#AddressDetailsLine2T,TrusteeOrPartner#AddressDetailsLocalityNameT,TrusteeOrPartner#AddressDetailsPostcodeT,TrusteeOrPartner#AddressDetailsStateOrTerritoryC,TrusteeOrPartner#AddressDetailsCountryC,CurrentPostalAddress,TrusteeOrPartner,TrusteeOrPartnerCollection,ElectionDetails#ElectionsInterposedEntityElectionStatusY,ElectionDetails#ElectionsDateSpecifiedD,SubstitutedAccountingPeriodDetails#StartD,SubstitutedAccountingPeriodDetails#EndD,SubstitutedAccountingPeriodDetails,ElectionDetails,FamilyTrust#ElectionsCommencementD,FamilyTrust#ElectionsElectionStatusY,SubstitutedAccountingPeriodDetails#StartD,SubstitutedAccountingPeriodDetails#EndD,SubstitutedAccountingPeriodDetails,FamilyTrust#TaxFileNumberId,FamilyTrust#TaxFileNumberInexistentI,FamilyTrust#OrganisationNameDetailsOrganisationalNameT,CurrentPostalAddress#AddressDetailsLine1T,CurrentPostalAddress#AddressDetailsLine2T,CurrentPostalAddress#AddressDetailsLocalityNameT,CurrentPostalAddress#AddressDetailsPostcodeT,CurrentPostalAddress#AddressDetailsStateOrTerritoryC,CurrentPostalAddress#AddressDetailsCountryC,CurrentPostalAddress,FamilyTrust,SpecifiedIndividual#TaxFileNumberId,SpecifiedIndividual#TaxFileNumberInexistentI,SpecifiedIndividual#PersonDemographicDetailsBirthD,SpecifiedIndividual#PersonNameDetailsTitleT,SpecifiedIndividual#PersonNameDetailsNameSuffixT,SpecifiedIndividual#PersonNameDetailsFamilyNameT,SpecifiedIndividual#PersonNameDetailsGivenNameT,SpecifiedIndividual#PersonNameDetailsOtherGivenNameT,CurrentResidentialAddressDetails#Line1T,CurrentResidentialAddressDetails#Line2T,CurrentResidentialAddressDetails#LocalityNameT,CurrentResidentialAddressDetails#PostcodeT,CurrentResidentialAddressDetails#StateOrTerritoryC,CurrentResidentialAddressDetails#CountryC,CurrentResidentialAddressDetails,SpecifiedIndividual,InterposedEntityElectionRevocationDetails,DeclarationOfTrusteeOrPartner#OrganisationNameDetailsOrganisationalNameT,DeclarationOfTrusteeOrPartner#PersonNameDetailsTitleT,DeclarationOfTrusteeOrPartner#PersonNameDetailsNameSuffixT,DeclarationOfTrusteeOrPartner#PersonNameDetailsFamilyNameT,DeclarationOfTrusteeOrPartner#PersonNameDetailsGivenNameT,DeclarationOfTrusteeOrPartner#PersonNameDetailsOtherGivenNameT,TrusteeOrPartnerIndividualNameDetails,DeclarationOfTrusteeOrPartner#StatementTypeC,DeclarationOfTrusteeOrPartner#StatementAcceptedI,DeclarationOfTrusteeOrPartner#SignatureD,DeclarationOfTrusteeOrPartner#SignatoryIdentifierT,DeclarationOfTrusteeOrPartner,DeclarationOfTrusteeOrPartnerCollection,PublicOfficerNameDetails#PersonNameDetailsTitleT,PublicOfficerNameDetails#PersonNameDetailsNameSuffixT,PublicOfficerNameDetails#PersonNameDetailsFamilyNameT,PublicOfficerNameDetails#PersonNameDetailsGivenNameT,PublicOfficerNameDetails#PersonNameDetailsOtherGivenNameT,PublicOfficerNameDetails,DeclarationOfPublicOfficer#StatementTypeC,DeclarationOfPublicOfficer#StatementAcceptedI,DeclarationOfPublicOfficer#SignatureD,DeclarationOfPublicOfficer#SignatoryIdentifierT,DeclarationOfPublicOfficer,RP";

    }
}