
using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationIDS
{
    /// <summary>
    /// XML Consumer for IDS
    /// </summary>
    public class IDS2025XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static IDS2025XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.ids.0012.2025.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;

            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];

                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count - 1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements)
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
            return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));

            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory)
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader)
        {
            bool retval = false;

            if ((reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if (_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if (_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
            }

            //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf, string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if (numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if (numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader)
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }

                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public IDS2025 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            IDS2025 report = new IDS2025();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:IDS");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "IDS" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/ids")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }


            #region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader, true) && reader.LocalName == "RP")
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true;

                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxFileNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IDS500 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion TaxFileNumberId

                #region InternationalDealingsCountryByCountryReportApplicabilityI
                _currentXPath.Push("/tns:InternationalDealingsCountryByCountryReportApplicabilityI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsCountryByCountryReportApplicabilityI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS330 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsCountryByCountryReportApplicabilityI

                #region InternationalDealingsLocalFilePartAVoluntaryLodgmentI
                _currentXPath.Push("/tns:InternationalDealingsLocalFilePartAVoluntaryLodgmentI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsLocalFilePartAVoluntaryLodgmentI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS331 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsLocalFilePartAVoluntaryLodgmentI

                #region InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
                _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS1 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI

                #region InternationalDealingsSmallBusinessConcessionsEligibilityI
                _currentXPath.Push("/tns:InternationalDealingsSmallBusinessConcessionsEligibilityI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsSmallBusinessConcessionsEligibilityI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS425 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsSmallBusinessConcessionsEligibilityI

                #region InternationalDealingsRelatedPartiesTransactionsI
                _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS2 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsRelatedPartiesTransactionsI

                #region ForeignCountry

                _currentXPath.Push("/tns:ForeignCountryCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignCountryCollection")
                {
                    report.RP_ForeignCountryCollection = new List<IDS2025.RP_ForeignCountry>();
                    report.RP_ForeignCountryCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "ForeignCountry")
                    {
                        _found = true;

                        IDS2025.RP_ForeignCountry foreignCountry = new IDS2025.RP_ForeignCountry();
                        report.RP_ForeignCountryCollection.Add(foreignCountry);
                        report.RP_ForeignCountryCollectionCount += 1;
                        foreignCountry.OccurrenceIndex = report.RP_ForeignCountryCollectionCount;

                        _currentXPath.Push("/tns:ForeignCountry[" + report.RP_ForeignCountryCollectionCount + "]");

                        #region OrganisationDetailsInternationalRelatedPartyCountryC
                        _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    foreignCountry.IDS418 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                        #region ForeignDealings

                        _currentXPath.Push("/tns:ForeignDealingsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignDealingsCollection")
                        {
                            foreignCountry.RP_ForeignCountry_ForeignDealingsCollection = new List<IDS2025.RP_ForeignCountry.RP_ForeignCountry_ForeignDealings>();
                            foreignCountry.RP_ForeignCountry_ForeignDealingsCollectionExists = true;

                            _found = true;

                            while (ReadToNextElement(reader, false) && reader.LocalName == "ForeignDealings")
                            {
                                _found = true;

                                IDS2025.RP_ForeignCountry.RP_ForeignCountry_ForeignDealings foreignDealings = new IDS2025.RP_ForeignCountry.RP_ForeignCountry_ForeignDealings();
                                foreignCountry.RP_ForeignCountry_ForeignDealingsCollection.Add(foreignDealings);
                                foreignCountry.RP_ForeignCountry_ForeignDealingsCollectionCount += 1;
                                foreignDealings.OccurrenceIndex = foreignCountry.RP_ForeignCountry_ForeignDealingsCollectionCount;

                                _currentXPath.Push("/tns:ForeignDealings[" + foreignCountry.RP_ForeignCountry_ForeignDealingsCollectionCount + "]");

                                #region OrganisationDetailsInternationalRelatedPartyCountryActivityC
                                _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryActivityC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryActivityC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            foreignDealings.IDS419 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion OrganisationDetailsInternationalRelatedPartyCountryActivityC

                                #region ExpenseTotalA
                                _currentXPath.Push("/tns:ExpenseTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                foreignDealings.IDS5 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion ExpenseTotalA

                                #region RevenueTotalA
                                _currentXPath.Push("/tns:RevenueTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                foreignDealings.IDS6 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion RevenueTotalA
                                _currentXPath.Pop();
                            } // End of while loop for ForeignDealings
                        } // End of if ForeignDealingsCollection node exists

                        _currentXPath.Pop();
                        #endregion ForeignDealings

                        #region ExpenseOtherA
                        _currentXPath.Push("/tns:ExpenseOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignCountry.IDS7 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseOtherA

                        #region RevenueOtherA
                        _currentXPath.Push("/tns:RevenueOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignCountry.IDS8 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueOtherA
                        _currentXPath.Pop();
                    } // End of while loop for ForeignCountry
                } // End of if ForeignCountryCollection node exists

                _currentXPath.Pop();
                #endregion ForeignCountry

                #region InternationalDealingsSpecifiedCountriesTransactionsI
                _currentXPath.Push("/tns:InternationalDealingsSpecifiedCountriesTransactionsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsSpecifiedCountriesTransactionsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS9 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsSpecifiedCountriesTransactionsI

                #region SpecifiedCountry

                _currentXPath.Push("/tns:SpecifiedCountryCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "SpecifiedCountryCollection")
                {
                    report.RP_SpecifiedCountryCollection = new List<IDS2025.RP_SpecifiedCountry>();
                    report.RP_SpecifiedCountryCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "SpecifiedCountry")
                    {
                        _found = true;

                        IDS2025.RP_SpecifiedCountry specifiedCountry = new IDS2025.RP_SpecifiedCountry();
                        report.RP_SpecifiedCountryCollection.Add(specifiedCountry);
                        report.RP_SpecifiedCountryCollectionCount += 1;
                        specifiedCountry.OccurrenceIndex = report.RP_SpecifiedCountryCollectionCount;

                        _currentXPath.Push("/tns:SpecifiedCountry[" + report.RP_SpecifiedCountryCollectionCount + "]");

                        #region OrganisationDetailsInternationalRelatedPartyCountryC
                        _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    specifiedCountry.IDS420 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                        #region SpecifiedDealings

                        _currentXPath.Push("/tns:SpecifiedDealingsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "SpecifiedDealingsCollection")
                        {
                            specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollection = new List<IDS2025.RP_SpecifiedCountry.RP_SpecifiedCountry_SpecifiedDealings>();
                            specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollectionExists = true;

                            _found = true;

                            while (ReadToNextElement(reader, false) && reader.LocalName == "SpecifiedDealings")
                            {
                                _found = true;

                                IDS2025.RP_SpecifiedCountry.RP_SpecifiedCountry_SpecifiedDealings specifiedDealings = new IDS2025.RP_SpecifiedCountry.RP_SpecifiedCountry_SpecifiedDealings();
                                specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollection.Add(specifiedDealings);
                                specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollectionCount += 1;
                                specifiedDealings.OccurrenceIndex = specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollectionCount;

                                _currentXPath.Push("/tns:SpecifiedDealings[" + specifiedCountry.RP_SpecifiedCountry_SpecifiedDealingsCollectionCount + "]");

                                #region OrganisationDetailsInternationalRelatedPartyCountryActivityC
                                _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryActivityC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryActivityC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            specifiedDealings.IDS421 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion OrganisationDetailsInternationalRelatedPartyCountryActivityC

                                #region ExpenseTotalA
                                _currentXPath.Push("/tns:ExpenseTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                specifiedDealings.IDS12 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion ExpenseTotalA

                                #region RevenueTotalA
                                _currentXPath.Push("/tns:RevenueTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                specifiedDealings.IDS13 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion RevenueTotalA
                                _currentXPath.Pop();
                            } // End of while loop for SpecifiedDealings
                        } // End of if SpecifiedDealingsCollection node exists

                        _currentXPath.Pop();
                        #endregion SpecifiedDealings

                        #region ExpenseOtherA
                        _currentXPath.Push("/tns:ExpenseOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        specifiedCountry.IDS14 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseOtherA

                        #region RevenueOtherA
                        _currentXPath.Push("/tns:RevenueOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        specifiedCountry.IDS15 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueOtherA
                        _currentXPath.Pop();
                    } // End of while loop for SpecifiedCountry
                } // End of if SpecifiedCountryCollection node exists

                _currentXPath.Pop();
                #endregion SpecifiedCountry

                #region InternationalDealingsTransactionsTangiblePropertyRevenueI
                _currentXPath.Push("/tns:InternationalDealingsTransactionsTangiblePropertyRevenueI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransactionsTangiblePropertyRevenueI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS16 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsTransactionsTangiblePropertyRevenueI

                #region TangiblePropertyRevenue
                _currentXPath.Push("/tns:TangiblePropertyRevenue");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TangiblePropertyRevenue")
                {
                    report.RP_TangiblePropertyRevenueCollectionExists = true;
                    report.RP_TangiblePropertyRevenueCollectionCount += 1;
                    _found = true;

                    #region ExpenseTotalA
                    _currentXPath.Push("/tns:ExpenseTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS17 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseTotalA

                    #region TotalA
                    _currentXPath.Push("/tns:TotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS18 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion TotalA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS19 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS20 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                } // End of if TangiblePropertyRevenue node exists

                _currentXPath.Pop();
                #endregion TangiblePropertyRevenue

                #region InternationalDealingsTransactionsRoyaltiesLicenceFeesI
                _currentXPath.Push("/tns:InternationalDealingsTransactionsRoyaltiesLicenceFeesI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransactionsRoyaltiesLicenceFeesI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS21 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsTransactionsRoyaltiesLicenceFeesI

                #region Royalties
                _currentXPath.Push("/tns:Royalties");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Royalties")
                {
                    report.RP_RoyaltiesCollectionExists = true;
                    report.RP_RoyaltiesCollectionCount += 1;
                    _found = true;

                    #region ExpenseDeductionsTotalA
                    _currentXPath.Push("/tns:ExpenseDeductionsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductionsTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS22 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseDeductionsTotalA

                    #region IncomeIncludedInAssessableTotalA
                    _currentXPath.Push("/tns:IncomeIncludedInAssessableTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeIncludedInAssessableTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS23 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeIncludedInAssessableTotalA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS24 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS25 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                } // End of if Royalties node exists

                _currentXPath.Pop();
                #endregion Royalties

                #region LicenceFees
                _currentXPath.Push("/tns:LicenceFees");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LicenceFees")
                {
                    report.RP_LicenceFeesCollectionExists = true;
                    report.RP_LicenceFeesCollectionCount += 1;
                    _found = true;

                    #region ExpenseDeductionsTotalA
                    _currentXPath.Push("/tns:ExpenseDeductionsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductionsTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS26 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseDeductionsTotalA

                    #region IncomeIncludedInAssessableTotalA
                    _currentXPath.Push("/tns:IncomeIncludedInAssessableTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeIncludedInAssessableTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS27 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeIncludedInAssessableTotalA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS28 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS29 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                } // End of if LicenceFees node exists

                _currentXPath.Pop();
                #endregion LicenceFees

                #region InternationalDealingsTransactionsRentLeasingI
                _currentXPath.Push("/tns:InternationalDealingsTransactionsRentLeasingI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransactionsRentLeasingI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS30 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsTransactionsRentLeasingI

                #region RentOrLeasing
                _currentXPath.Push("/tns:RentOrLeasing");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RentOrLeasing")
                {
                    report.RP_RentOrLeasingCollectionExists = true;
                    report.RP_RentOrLeasingCollectionCount += 1;
                    _found = true;

                    #region ExpenseTotalA
                    _currentXPath.Push("/tns:ExpenseTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS31 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseTotalA

                    #region RevenueTotalA
                    _currentXPath.Push("/tns:RevenueTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS32 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion RevenueTotalA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS33 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS34 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                } // End of if RentOrLeasing node exists

                _currentXPath.Pop();
                #endregion RentOrLeasing

                #region InternationalDealingsTransactionsServiceArrangementsI
                _currentXPath.Push("/tns:InternationalDealingsTransactionsServiceArrangementsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransactionsServiceArrangementsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS35 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsTransactionsServiceArrangementsI

                #region ServiceArrangements
                _currentXPath.Push("/tns:ServiceArrangements");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ServiceArrangements")
                {
                    report.RP_ServiceArrangementsCollectionExists = true;
                    report.RP_ServiceArrangementsCollectionCount += 1;
                    _found = true;

                    #region TreasuryServices
                    _currentXPath.Push("/tns:TreasuryServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TreasuryServices")
                    {
                        report.RP_ServiceArrangements_TreasuryServicesCollectionExists = true;
                        report.RP_ServiceArrangements_TreasuryServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS36 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS37 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS38 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS39 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if TreasuryServices node exists

                    _currentXPath.Pop();
                    #endregion TreasuryServices

                    #region ManagementAndAdministrationServices
                    _currentXPath.Push("/tns:ManagementAndAdministrationServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ManagementAndAdministrationServices")
                    {
                        report.RP_ServiceArrangements_ManagementAndAdministrationServicesCollectionExists = true;
                        report.RP_ServiceArrangements_ManagementAndAdministrationServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS40 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS41 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS42 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS43 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if ManagementAndAdministrationServices node exists

                    _currentXPath.Pop();
                    #endregion ManagementAndAdministrationServices

                    #region InsuranceServices
                    _currentXPath.Push("/tns:InsuranceServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InsuranceServices")
                    {
                        report.RP_ServiceArrangements_InsuranceServicesCollectionExists = true;
                        report.RP_ServiceArrangements_InsuranceServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS44 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS45 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS46 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS47 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if InsuranceServices node exists

                    _currentXPath.Pop();
                    #endregion InsuranceServices

                    #region ReinsuranceServices
                    _currentXPath.Push("/tns:ReinsuranceServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ReinsuranceServices")
                    {
                        report.RP_ServiceArrangements_ReinsuranceServicesCollectionExists = true;
                        report.RP_ServiceArrangements_ReinsuranceServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS48 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS49 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS50 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS51 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if ReinsuranceServices node exists

                    _currentXPath.Pop();
                    #endregion ReinsuranceServices

                    #region RAndDServices
                    _currentXPath.Push("/tns:RAndDServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RAndDServices")
                    {
                        report.RP_ServiceArrangements_RAndDServicesCollectionExists = true;
                        report.RP_ServiceArrangements_RAndDServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS52 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS53 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS54 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS55 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if RAndDServices node exists

                    _currentXPath.Pop();
                    #endregion RAndDServices

                    #region SalesAndMarketingServices
                    _currentXPath.Push("/tns:SalesAndMarketingServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SalesAndMarketingServices")
                    {
                        report.RP_ServiceArrangements_SalesAndMarketingServicesCollectionExists = true;
                        report.RP_ServiceArrangements_SalesAndMarketingServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS56 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS57 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS58 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS59 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if SalesAndMarketingServices node exists

                    _currentXPath.Pop();
                    #endregion SalesAndMarketingServices

                    #region SoftwareAndITServices
                    _currentXPath.Push("/tns:SoftwareAndITServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SoftwareAndITServices")
                    {
                        report.RP_ServiceArrangements_SoftwareAndITServicesCollectionExists = true;
                        report.RP_ServiceArrangements_SoftwareAndITServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS60 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS61 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS62 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS63 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if SoftwareAndITServices node exists

                    _currentXPath.Pop();
                    #endregion SoftwareAndITServices

                    #region TechnicalServices
                    _currentXPath.Push("/tns:TechnicalServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TechnicalServices")
                    {
                        report.RP_ServiceArrangements_TechnicalServicesCollectionExists = true;
                        report.RP_ServiceArrangements_TechnicalServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS64 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS65 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS66 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS67 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if TechnicalServices node exists

                    _currentXPath.Pop();
                    #endregion TechnicalServices

                    #region LogisticsServices
                    _currentXPath.Push("/tns:LogisticsServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LogisticsServices")
                    {
                        report.RP_ServiceArrangements_LogisticsServicesCollectionExists = true;
                        report.RP_ServiceArrangements_LogisticsServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS68 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS69 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS70 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS71 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if LogisticsServices node exists

                    _currentXPath.Pop();
                    #endregion LogisticsServices

                    #region AssetManagementServices
                    _currentXPath.Push("/tns:AssetManagementServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetManagementServices")
                    {
                        report.RP_ServiceArrangements_AssetManagementServicesCollectionExists = true;
                        report.RP_ServiceArrangements_AssetManagementServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS72 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS73 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS74 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS75 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if AssetManagementServices node exists

                    _currentXPath.Pop();
                    #endregion AssetManagementServices

                    #region OtherServices
                    _currentXPath.Push("/tns:OtherServices");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OtherServices")
                    {
                        report.RP_ServiceArrangements_OtherServicesCollectionExists = true;
                        report.RP_ServiceArrangements_OtherServicesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS76 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS77 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS78 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS79 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS80 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT
                    } // End of if OtherServices node exists

                    _currentXPath.Pop();
                    #endregion OtherServices
                } // End of if ServiceArrangements node exists

                _currentXPath.Pop();
                #endregion ServiceArrangements

                #region InternationalDealingsDerivativeTransactionsI
                _currentXPath.Push("/tns:InternationalDealingsDerivativeTransactionsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDerivativeTransactionsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS81 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsDerivativeTransactionsI

                #region Derivatives
                _currentXPath.Push("/tns:Derivatives");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Derivatives")
                {
                    report.RP_DerivativesCollectionExists = true;
                    report.RP_DerivativesCollectionCount += 1;
                    _found = true;

                    #region ExpenseTotalA
                    _currentXPath.Push("/tns:ExpenseTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS82 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseTotalA

                    #region RevenueTotalA
                    _currentXPath.Push("/tns:RevenueTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS83 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion RevenueTotalA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS84 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS85 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC

                    #region InternationalDealingsRelatedPartyTransactionTypeCCollection
                    _currentXPath.Push("/tns:InternationalDealingsRelatedPartyTransactionTypeCCollection");
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsRelatedPartyTransactionTypeCCollection")
                    {
                        report.IDS332Collection = new List<string>();
                        ReadNext(reader);
                        _currentXPath.Push("/tns:InternationalDealingsRelatedPartyTransactionTypeC");
                        // 5. use case
                        while (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsRelatedPartyTransactionTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before the currentXPath is updated.
                                    report.IDS332Collection.Add(currentValue);
                                }
                            }
                            _found = true;
                        }
                        _currentXPath.Pop();
                    }
                    _currentXPath.Pop();
                    #endregion InternationalDealingsRelatedPartyTransactionTypeCCollection

                    #region InternationalDealingsTradingGlobalI
                    _currentXPath.Push("/tns:InternationalDealingsTradingGlobalI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTradingGlobalI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS87 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsTradingGlobalI
                } // End of if Derivatives node exists

                _currentXPath.Pop();
                #endregion Derivatives

                #region InternationalDealingsDebtFactoringSecuritisationI
                _currentXPath.Push("/tns:InternationalDealingsDebtFactoringSecuritisationI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDebtFactoringSecuritisationI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS88 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsDebtFactoringSecuritisationI

                #region DebtFactoring
                _currentXPath.Push("/tns:DebtFactoring");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "DebtFactoring")
                {
                    report.RP_DebtFactoringCollectionExists = true;
                    report.RP_DebtFactoringCollectionCount += 1;
                    _found = true;

                    #region ExpenseBookValueA
                    _currentXPath.Push("/tns:ExpenseBookValueA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseBookValueA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS89 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseBookValueA

                    #region ExpenseConsiderationA
                    _currentXPath.Push("/tns:ExpenseConsiderationA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseConsiderationA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS90 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseConsiderationA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS91 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS92 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                } // End of if DebtFactoring node exists

                _currentXPath.Pop();
                #endregion DebtFactoring

                #region Securitisation
                _currentXPath.Push("/tns:Securitisation");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Securitisation")
                {
                    report.RP_SecuritisationCollectionExists = true;
                    report.RP_SecuritisationCollectionCount += 1;
                    _found = true;

                    #region ExpenseBookValueA
                    _currentXPath.Push("/tns:ExpenseBookValueA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseBookValueA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS93 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseBookValueA

                    #region ExpenseConsiderationA
                    _currentXPath.Push("/tns:ExpenseConsiderationA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseConsiderationA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS94 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseConsiderationA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS95 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS96 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                } // End of if Securitisation node exists

                _currentXPath.Pop();
                #endregion Securitisation

                #region InternationalDealingsOtherFinancialDealingsI
                _currentXPath.Push("/tns:InternationalDealingsOtherFinancialDealingsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsOtherFinancialDealingsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS97 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsOtherFinancialDealingsI

                #region FinancialDealings
                _currentXPath.Push("/tns:FinancialDealings");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialDealings")
                {
                    report.RP_FinancialDealingsCollectionExists = true;
                    report.RP_FinancialDealingsCollectionCount += 1;
                    _found = true;

                    #region InterestBearingLoans
                    _currentXPath.Push("/tns:InterestBearingLoans");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InterestBearingLoans")
                    {
                        report.RP_FinancialDealings_InterestBearingLoansCollectionExists = true;
                        report.RP_FinancialDealings_InterestBearingLoansCollectionCount += 1;
                        _found = true;

                        #region LiabilitiesBorrowedAverageBalanceA
                        _currentXPath.Push("/tns:LiabilitiesBorrowedAverageBalanceA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesBorrowedAverageBalanceA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS98 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesBorrowedAverageBalanceA

                        #region AssetsLoanedAverageBalanceA
                        _currentXPath.Push("/tns:AssetsLoanedAverageBalanceA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsLoanedAverageBalanceA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS100 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsLoanedAverageBalanceA
                    } // End of if InterestBearingLoans node exists

                    _currentXPath.Pop();
                    #endregion InterestBearingLoans

                    #region InterestFreeLoans
                    _currentXPath.Push("/tns:InterestFreeLoans");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InterestFreeLoans")
                    {
                        report.RP_FinancialDealings_InterestFreeLoansCollectionExists = true;
                        report.RP_FinancialDealings_InterestFreeLoansCollectionCount += 1;
                        _found = true;

                        #region LiabilitiesBorrowedAverageBalanceA
                        _currentXPath.Push("/tns:LiabilitiesBorrowedAverageBalanceA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesBorrowedAverageBalanceA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS99 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesBorrowedAverageBalanceA

                        #region AssetsLoanedAverageBalanceA
                        _currentXPath.Push("/tns:AssetsLoanedAverageBalanceA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsLoanedAverageBalanceA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS101 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsLoanedAverageBalanceA
                    } // End of if InterestFreeLoans node exists

                    _currentXPath.Pop();
                    #endregion InterestFreeLoans

                    #region Interest
                    _currentXPath.Push("/tns:Interest");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "Interest")
                    {
                        report.RP_FinancialDealings_InterestCollectionExists = true;
                        report.RP_FinancialDealings_InterestCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS102 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS103 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS104 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS105 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if Interest node exists

                    _currentXPath.Pop();
                    #endregion Interest

                    #region Guarantees
                    _currentXPath.Push("/tns:Guarantees");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "Guarantees")
                    {
                        report.RP_FinancialDealings_GuaranteesCollectionExists = true;
                        report.RP_FinancialDealings_GuaranteesCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS106 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS107 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS108 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS109 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if Guarantees node exists

                    _currentXPath.Pop();
                    #endregion Guarantees

                    #region Insurance
                    _currentXPath.Push("/tns:Insurance");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "Insurance")
                    {
                        report.RP_FinancialDealings_InsuranceCollectionExists = true;
                        report.RP_FinancialDealings_InsuranceCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS110 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS111 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS112 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS113 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if Insurance node exists

                    _currentXPath.Pop();
                    #endregion Insurance

                    #region Reinsurance
                    _currentXPath.Push("/tns:Reinsurance");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "Reinsurance")
                    {
                        report.RP_FinancialDealings_ReinsuranceCollectionExists = true;
                        report.RP_FinancialDealings_ReinsuranceCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS114 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS115 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS116 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS117 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if Reinsurance node exists

                    _currentXPath.Pop();
                    #endregion Reinsurance

                    #region OtherFinancial
                    _currentXPath.Push("/tns:OtherFinancial");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OtherFinancial")
                    {
                        report.RP_FinancialDealings_OtherFinancialCollectionExists = true;
                        report.RP_FinancialDealings_OtherFinancialCollectionCount += 1;
                        _found = true;

                        #region ExpenseTotalA
                        _currentXPath.Push("/tns:ExpenseTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS118 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseTotalA

                        #region RevenueTotalA
                        _currentXPath.Push("/tns:RevenueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS119 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueTotalA

                        #region MiscellaneousArmsLengthPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS120 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousArmsLengthPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS121 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS122 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT
                    } // End of if OtherFinancial node exists

                    _currentXPath.Pop();
                    #endregion OtherFinancial
                } // End of if FinancialDealings node exists

                _currentXPath.Pop();
                #endregion FinancialDealings

                #region InternationalDealingsForeignExchangeGainLossI
                _currentXPath.Push("/tns:InternationalDealingsForeignExchangeGainLossI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignExchangeGainLossI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS333 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsForeignExchangeGainLossI

                #region ForeignExchange

                _currentXPath.Push("/tns:ForeignExchangeCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignExchangeCollection")
                {
                    report.RP_ForeignExchangeCollection = new List<IDS2025.RP_ForeignExchange>();
                    report.RP_ForeignExchangeCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "ForeignExchange")
                    {
                        _found = true;

                        IDS2025.RP_ForeignExchange foreignExchange = new IDS2025.RP_ForeignExchange();
                        report.RP_ForeignExchangeCollection.Add(foreignExchange);
                        report.RP_ForeignExchangeCollectionCount += 1;
                        foreignExchange.OccurrenceIndex = report.RP_ForeignExchangeCollectionCount;

                        _currentXPath.Push("/tns:ForeignExchange[" + report.RP_ForeignExchangeCollectionCount + "]");

                        #region InternationalDealingsTransactionTypeC
                        _currentXPath.Push("/tns:InternationalDealingsTransactionTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTransactionTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    foreignExchange.IDS422 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsTransactionTypeC

                        #region GainOrLoss

                        _currentXPath.Push("/tns:GainOrLossCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GainOrLossCollection")
                        {
                            foreignExchange.RP_ForeignExchange_GainOrLossCollection = new List<IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss>();
                            foreignExchange.RP_ForeignExchange_GainOrLossCollectionExists = true;

                            _found = true;

                            while (ReadToNextElement(reader, false) && reader.LocalName == "GainOrLoss")
                            {
                                _found = true;

                                IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss gainOrLoss = new IDS2025.RP_ForeignExchange.RP_ForeignExchange_GainOrLoss();
                                foreignExchange.RP_ForeignExchange_GainOrLossCollection.Add(gainOrLoss);
                                foreignExchange.RP_ForeignExchange_GainOrLossCollectionCount += 1;
                                gainOrLoss.OccurrenceIndex = foreignExchange.RP_ForeignExchange_GainOrLossCollectionCount;

                                _currentXPath.Push("/tns:GainOrLoss[" + foreignExchange.RP_ForeignExchange_GainOrLossCollectionCount + "]");

                                #region MiscellaneousFunctionalCurrencyC
                                _currentXPath.Push("/tns:MiscellaneousFunctionalCurrencyC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousFunctionalCurrencyC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            gainOrLoss.IDS423 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion MiscellaneousFunctionalCurrencyC

                                #region InternationalDealingsLossesHighestNominalA
                                _currentXPath.Push("/tns:InternationalDealingsLossesHighestNominalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsLossesHighestNominalA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                gainOrLoss.IDS336 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion InternationalDealingsLossesHighestNominalA

                                #region InternationalDealingsGainsHighestNominalA
                                _currentXPath.Push("/tns:InternationalDealingsGainsHighestNominalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsGainsHighestNominalA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                gainOrLoss.IDS337 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion InternationalDealingsGainsHighestNominalA
                                _currentXPath.Pop();
                            } // End of while loop for GainOrLoss
                        } // End of if GainOrLossCollection node exists

                        _currentXPath.Pop();
                        #endregion GainOrLoss

                        #region InternationalDealingsLossesOtherA
                        _currentXPath.Push("/tns:InternationalDealingsLossesOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsLossesOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignExchange.IDS338 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsLossesOtherA

                        #region InternationalDealingsGainsOtherA
                        _currentXPath.Push("/tns:InternationalDealingsGainsOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsGainsOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignExchange.IDS339 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsGainsOtherA
                        _currentXPath.Pop();
                    } // End of while loop for ForeignExchange
                } // End of if ForeignExchangeCollection node exists

                _currentXPath.Pop();
                #endregion ForeignExchange

                #region InternationalDealingsOtherRevenueI
                _currentXPath.Push("/tns:InternationalDealingsOtherRevenueI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsOtherRevenueI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS123 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsOtherRevenueI

                #region RevenueDealings
                _currentXPath.Push("/tns:RevenueDealings");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueDealings")
                {
                    report.RP_RevenueDealingsCollectionExists = true;
                    report.RP_RevenueDealingsCollectionCount += 1;
                    _found = true;

                    #region ExpenseTotalA
                    _currentXPath.Push("/tns:ExpenseTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS124 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseTotalA

                    #region TotalA
                    _currentXPath.Push("/tns:TotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS125 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion TotalA

                    #region MiscellaneousArmsLengthPricingMethodC
                    _currentXPath.Push("/tns:MiscellaneousArmsLengthPricingMethodC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousArmsLengthPricingMethodC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS126 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousArmsLengthPricingMethodC

                    #region MiscellaneousPercentageOfDealingsWithDocumentationC
                    _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS127 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousPercentageOfDealingsWithDocumentationC

                    #region InternationalDealingsDescriptionT
                    _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS128 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsDescriptionT
                } // End of if RevenueDealings node exists

                _currentXPath.Pop();
                #endregion RevenueDealings

                #region MiscellaneousCapitalNatureI
                _currentXPath.Push("/tns:MiscellaneousCapitalNatureI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousCapitalNatureI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS129 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousCapitalNatureI

                #region DisposalOrAcquisition
                _currentXPath.Push("/tns:DisposalOrAcquisition");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "DisposalOrAcquisition")
                {
                    report.RP_DisposalOrAcquisitionCollectionExists = true;
                    report.RP_DisposalOrAcquisitionCollectionCount += 1;
                    _found = true;

                    #region TangiblePropertyNonRevenue
                    _currentXPath.Push("/tns:TangiblePropertyNonRevenue");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TangiblePropertyNonRevenue")
                    {
                        report.RP_DisposalOrAcquisition_TangiblePropertyNonRevenueCollectionExists = true;
                        report.RP_DisposalOrAcquisition_TangiblePropertyNonRevenueCollectionCount += 1;
                        _found = true;

                        #region AssetsConsiderationPaidA
                        _currentXPath.Push("/tns:AssetsConsiderationPaidA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationPaidA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS130 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationPaidA

                        #region AssetsConsiderationReceivedA
                        _currentXPath.Push("/tns:AssetsConsiderationReceivedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationReceivedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS131 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationReceivedA

                        #region MiscellaneousAcquisitionsAndDisposalsPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousAcquisitionsAndDisposalsPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousAcquisitionsAndDisposalsPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS132 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousAcquisitionsAndDisposalsPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS133 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if TangiblePropertyNonRevenue node exists

                    _currentXPath.Pop();
                    #endregion TangiblePropertyNonRevenue

                    #region IntellectualProperty
                    _currentXPath.Push("/tns:IntellectualProperty");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IntellectualProperty")
                    {
                        report.RP_DisposalOrAcquisition_IntellectualPropertyCollectionExists = true;
                        report.RP_DisposalOrAcquisition_IntellectualPropertyCollectionCount += 1;
                        _found = true;

                        #region AssetsConsiderationPaidA
                        _currentXPath.Push("/tns:AssetsConsiderationPaidA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationPaidA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS340 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationPaidA

                        #region AssetsConsiderationReceivedA
                        _currentXPath.Push("/tns:AssetsConsiderationReceivedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationReceivedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS341 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationReceivedA

                        #region MiscellaneousAcquisitionsAndDisposalsPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousAcquisitionsAndDisposalsPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousAcquisitionsAndDisposalsPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS342 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousAcquisitionsAndDisposalsPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS343 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if IntellectualProperty node exists

                    _currentXPath.Pop();
                    #endregion IntellectualProperty

                    #region SharesOrOtherEquityInterests
                    _currentXPath.Push("/tns:SharesOrOtherEquityInterests");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SharesOrOtherEquityInterests")
                    {
                        report.RP_DisposalOrAcquisition_SharesOrOtherEquityInterestsCollectionExists = true;
                        report.RP_DisposalOrAcquisition_SharesOrOtherEquityInterestsCollectionCount += 1;
                        _found = true;

                        #region AssetsConsiderationPaidA
                        _currentXPath.Push("/tns:AssetsConsiderationPaidA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationPaidA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS344 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationPaidA

                        #region AssetsConsiderationReceivedA
                        _currentXPath.Push("/tns:AssetsConsiderationReceivedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationReceivedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS345 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationReceivedA

                        #region MiscellaneousAcquisitionsAndDisposalsPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousAcquisitionsAndDisposalsPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousAcquisitionsAndDisposalsPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS346 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousAcquisitionsAndDisposalsPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS347 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if SharesOrOtherEquityInterests node exists

                    _currentXPath.Pop();
                    #endregion SharesOrOtherEquityInterests

                    #region LoansOrDebts
                    _currentXPath.Push("/tns:LoansOrDebts");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LoansOrDebts")
                    {
                        report.RP_DisposalOrAcquisition_LoansOrDebtsCollectionExists = true;
                        report.RP_DisposalOrAcquisition_LoansOrDebtsCollectionCount += 1;
                        _found = true;

                        #region AssetsConsiderationPaidA
                        _currentXPath.Push("/tns:AssetsConsiderationPaidA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationPaidA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS348 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationPaidA

                        #region AssetsConsiderationReceivedA
                        _currentXPath.Push("/tns:AssetsConsiderationReceivedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationReceivedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS349 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationReceivedA

                        #region MiscellaneousAcquisitionsAndDisposalsPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousAcquisitionsAndDisposalsPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousAcquisitionsAndDisposalsPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS350 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousAcquisitionsAndDisposalsPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS351 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if LoansOrDebts node exists

                    _currentXPath.Pop();
                    #endregion LoansOrDebts

                    #region IntangibleProperty
                    _currentXPath.Push("/tns:IntangibleProperty");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IntangibleProperty")
                    {
                        report.RP_DisposalOrAcquisition_IntangiblePropertyCollectionExists = true;
                        report.RP_DisposalOrAcquisition_IntangiblePropertyCollectionCount += 1;
                        _found = true;

                        #region AssetsConsiderationPaidA
                        _currentXPath.Push("/tns:AssetsConsiderationPaidA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationPaidA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS134 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationPaidA

                        #region AssetsConsiderationReceivedA
                        _currentXPath.Push("/tns:AssetsConsiderationReceivedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsConsiderationReceivedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS135 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsConsiderationReceivedA

                        #region MiscellaneousAcquisitionsAndDisposalsPricingMethodC
                        _currentXPath.Push("/tns:MiscellaneousAcquisitionsAndDisposalsPricingMethodC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousAcquisitionsAndDisposalsPricingMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS136 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousAcquisitionsAndDisposalsPricingMethodC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS137 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC
                    } // End of if IntangibleProperty node exists

                    _currentXPath.Pop();
                    #endregion IntangibleProperty
                } // End of if DisposalOrAcquisition node exists

                _currentXPath.Pop();
                #endregion DisposalOrAcquisition

                #region MiscellaneousResearchAndDevelopmentCostPlusRemunerationI
                _currentXPath.Push("/tns:MiscellaneousResearchAndDevelopmentCostPlusRemunerationI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousResearchAndDevelopmentCostPlusRemunerationI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS426 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousResearchAndDevelopmentCostPlusRemunerationI

                #region ResearchAndDevelopment
                _currentXPath.Push("/tns:ResearchAndDevelopment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ResearchAndDevelopment")
                {
                    report.RP_ResearchAndDevelopmentCollectionExists = true;
                    report.RP_ResearchAndDevelopmentCollectionCount += 1;
                    _found = true;

                    #region ExpenseCostPlusRemunerationA
                    _currentXPath.Push("/tns:ExpenseCostPlusRemunerationA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ExpenseCostPlusRemunerationA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS427 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseCostPlusRemunerationA

                    #region IncomeCostPlusRemunerationA
                    _currentXPath.Push("/tns:IncomeCostPlusRemunerationA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeCostPlusRemunerationA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS428 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeCostPlusRemunerationA
                } // End of if ResearchAndDevelopment node exists

                _currentXPath.Pop();
                #endregion ResearchAndDevelopment

                #region MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI
                _currentXPath.Push("/tns:MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS138 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI

                #region CapitalNoPayment
                _currentXPath.Push("/tns:CapitalNoPayment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalNoPayment")
                {
                    report.RP_CapitalNoPaymentCollectionExists = true;
                    report.RP_CapitalNoPaymentCollectionCount += 1;
                    _found = true;

                    #region OrganisationDetailsInternationalRelatedPartyCountryC
                    _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS139 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                    #region MiscellaneousNatureOfItemC
                    _currentXPath.Push("/tns:MiscellaneousNatureOfItemC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousNatureOfItemC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS140 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousNatureOfItemC
                } // End of if CapitalNoPayment node exists

                _currentXPath.Pop();
                #endregion CapitalNoPayment

                #region RevenueNoPayment
                _currentXPath.Push("/tns:RevenueNoPayment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueNoPayment")
                {
                    report.RP_RevenueNoPaymentCollectionExists = true;
                    report.RP_RevenueNoPaymentCollectionCount += 1;
                    _found = true;

                    #region OrganisationDetailsInternationalRelatedPartyCountryC
                    _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS141 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                    #region MiscellaneousNatureOfItemC
                    _currentXPath.Push("/tns:MiscellaneousNatureOfItemC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousNatureOfItemC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS142 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousNatureOfItemC
                } // End of if RevenueNoPayment node exists

                _currentXPath.Pop();
                #endregion RevenueNoPayment

                #region CapitalNonMonetaryPayment
                _currentXPath.Push("/tns:CapitalNonMonetaryPayment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalNonMonetaryPayment")
                {
                    report.RP_CapitalNonMonetaryPaymentCollectionExists = true;
                    report.RP_CapitalNonMonetaryPaymentCollectionCount += 1;
                    _found = true;

                    #region OrganisationDetailsInternationalRelatedPartyCountryC
                    _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS143 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                    #region MiscellaneousNatureOfItemC
                    _currentXPath.Push("/tns:MiscellaneousNatureOfItemC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousNatureOfItemC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS144 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousNatureOfItemC
                } // End of if CapitalNonMonetaryPayment node exists

                _currentXPath.Pop();
                #endregion CapitalNonMonetaryPayment

                #region RevenueNonMonetaryPayment
                _currentXPath.Push("/tns:RevenueNonMonetaryPayment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RevenueNonMonetaryPayment")
                {
                    report.RP_RevenueNonMonetaryPaymentCollectionExists = true;
                    report.RP_RevenueNonMonetaryPaymentCollectionCount += 1;
                    _found = true;

                    #region OrganisationDetailsInternationalRelatedPartyCountryC
                    _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS145 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                    #region MiscellaneousNatureOfItemC
                    _currentXPath.Push("/tns:MiscellaneousNatureOfItemC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousNatureOfItemC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS146 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousNatureOfItemC
                } // End of if RevenueNonMonetaryPayment node exists

                _currentXPath.Pop();
                #endregion RevenueNonMonetaryPayment

                #region InternationalDealingsShareBasedEmployeeRemunerationI
                _currentXPath.Push("/tns:InternationalDealingsShareBasedEmployeeRemunerationI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsShareBasedEmployeeRemunerationI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS147 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsShareBasedEmployeeRemunerationI

                #region ShareBasedRemuneration
                _currentXPath.Push("/tns:ShareBasedRemuneration");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ShareBasedRemuneration")
                {
                    report.RP_ShareBasedRemunerationCollectionExists = true;
                    report.RP_ShareBasedRemunerationCollectionCount += 1;
                    _found = true;

                    #region ExpenseDeductionsTotalA
                    _currentXPath.Push("/tns:ExpenseDeductionsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductionsTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS148 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseDeductionsTotalA

                    #region IncomeIncludedInAssessableTotalA
                    _currentXPath.Push("/tns:IncomeIncludedInAssessableTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeIncludedInAssessableTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS149 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeIncludedInAssessableTotalA
                } // End of if ShareBasedRemuneration node exists

                _currentXPath.Pop();
                #endregion ShareBasedRemuneration

                #region InternationalDealingsCostContributionArrangementsI
                _currentXPath.Push("/tns:InternationalDealingsCostContributionArrangementsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsCostContributionArrangementsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS150 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsCostContributionArrangementsI

                #region InternationalDealingsRestructuringEventsI
                _currentXPath.Push("/tns:InternationalDealingsRestructuringEventsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsRestructuringEventsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS151 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsRestructuringEventsI

                #region RestructuringEvent

                _currentXPath.Push("/tns:RestructuringEventCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RestructuringEventCollection")
                {
                    report.RP_RestructuringEventCollection = new List<IDS2025.RP_RestructuringEvent>();
                    report.RP_RestructuringEventCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "RestructuringEvent")
                    {
                        _found = true;

                        IDS2025.RP_RestructuringEvent restructuringEvent = new IDS2025.RP_RestructuringEvent();
                        report.RP_RestructuringEventCollection.Add(restructuringEvent);
                        report.RP_RestructuringEventCollectionCount += 1;
                        restructuringEvent.OccurrenceIndex = report.RP_RestructuringEventCollectionCount;

                        _currentXPath.Push("/tns:RestructuringEvent[" + report.RP_RestructuringEventCollectionCount + "]");

                        #region OrganisationDetailsEntityC
                        _currentXPath.Push("/tns:OrganisationDetailsEntityC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsEntityC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEvent.IDS152 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsEntityC

                        #region MiscellaneousCapitalValueC
                        _currentXPath.Push("/tns:MiscellaneousCapitalValueC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousCapitalValueC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEvent.IDS153 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousCapitalValueC

                        #region MiscellaneousNatureOfItemC
                        _currentXPath.Push("/tns:MiscellaneousNatureOfItemC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousNatureOfItemC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEvent.IDS154 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousNatureOfItemC

                        #region OrganisationDetailsInternationalRelatedPartyCountryC
                        _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEvent.IDS155 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                        #region MiscellaneousPercentageOfDealingsWithDocumentationC
                        _currentXPath.Push("/tns:MiscellaneousPercentageOfDealingsWithDocumentationC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPercentageOfDealingsWithDocumentationC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEvent.IDS156 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPercentageOfDealingsWithDocumentationC

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEvent.IDS157 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT

                        #region InternationalDealingsEventAnalysisI
                        _currentXPath.Push("/tns:InternationalDealingsEventAnalysisI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsEventAnalysisI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        restructuringEvent.IDS158 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsEventAnalysisI
                        _currentXPath.Pop();
                    } // End of while loop for RestructuringEvent
                } // End of if RestructuringEventCollection node exists

                _currentXPath.Pop();
                #endregion RestructuringEvent

                #region InternationalDealingsTransactionsRestructuringEventsAssetRevaluationI
                _currentXPath.Push("/tns:InternationalDealingsTransactionsRestructuringEventsAssetRevaluationI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransactionsRestructuringEventsAssetRevaluationI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS159 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsTransactionsRestructuringEventsAssetRevaluationI

                #region InternationalDealingsBranchOperationsI
                _currentXPath.Push("/tns:InternationalDealingsBranchOperationsI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsBranchOperationsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS160 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsBranchOperationsI

                #region BranchOperationsInterestBearingLoans
                _currentXPath.Push("/tns:BranchOperationsInterestBearingLoans");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BranchOperationsInterestBearingLoans")
                {
                    report.RP_BranchOperationsInterestBearingLoansCollectionExists = true;
                    report.RP_BranchOperationsInterestBearingLoansCollectionCount += 1;
                    _found = true;

                    #region LiabilitiesBorrowedAverageBalanceA
                    _currentXPath.Push("/tns:LiabilitiesBorrowedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesBorrowedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS161 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesBorrowedAverageBalanceA

                    #region AssetsLoanedAverageBalanceA
                    _currentXPath.Push("/tns:AssetsLoanedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsLoanedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS162 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AssetsLoanedAverageBalanceA

                    #region ExpenseInterestBorrowedTotalA
                    _currentXPath.Push("/tns:ExpenseInterestBorrowedTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInterestBorrowedTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS163 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseInterestBorrowedTotalA

                    #region IncomeInterestLoanedTotalA
                    _currentXPath.Push("/tns:IncomeInterestLoanedTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInterestLoanedTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS164 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeInterestLoanedTotalA
                } // End of if BranchOperationsInterestBearingLoans node exists

                _currentXPath.Pop();
                #endregion BranchOperationsInterestBearingLoans

                #region BranchOperationsInterestFreeLoans
                _currentXPath.Push("/tns:BranchOperationsInterestFreeLoans");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BranchOperationsInterestFreeLoans")
                {
                    report.RP_BranchOperationsInterestFreeLoansCollectionExists = true;
                    report.RP_BranchOperationsInterestFreeLoansCollectionCount += 1;
                    _found = true;

                    #region LiabilitiesBorrowedAverageBalanceA
                    _currentXPath.Push("/tns:LiabilitiesBorrowedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesBorrowedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS165 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesBorrowedAverageBalanceA

                    #region AssetsLoanedAverageBalanceA
                    _currentXPath.Push("/tns:AssetsLoanedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsLoanedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS166 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AssetsLoanedAverageBalanceA
                } // End of if BranchOperationsInterestFreeLoans node exists

                _currentXPath.Pop();
                #endregion BranchOperationsInterestFreeLoans

                #region InternalTrading
                _currentXPath.Push("/tns:InternalTrading");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternalTrading")
                {
                    report.RP_InternalTradingCollectionExists = true;
                    report.RP_InternalTradingCollectionCount += 1;
                    _found = true;

                    #region ExpenseTradingStockPurchaseCostsA
                    _currentXPath.Push("/tns:ExpenseTradingStockPurchaseCostsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseTradingStockPurchaseCostsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS167 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseTradingStockPurchaseCostsA

                    #region IncomeTradingStockSalesTotalA
                    _currentXPath.Push("/tns:IncomeTradingStockSalesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTradingStockSalesTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS168 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTradingStockSalesTotalA
                } // End of if InternalTrading node exists

                _currentXPath.Pop();
                #endregion InternalTrading

                #region BranchOperationsOther
                _currentXPath.Push("/tns:BranchOperationsOther");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BranchOperationsOther")
                {
                    report.RP_BranchOperationsOtherCollectionExists = true;
                    report.RP_BranchOperationsOtherCollectionCount += 1;
                    _found = true;

                    #region MiscellaneousInternallyRecordedDealingsAmountsClaimedTotalA
                    _currentXPath.Push("/tns:MiscellaneousInternallyRecordedDealingsAmountsClaimedTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousInternallyRecordedDealingsAmountsClaimedTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS169 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousInternallyRecordedDealingsAmountsClaimedTotalA

                    #region MiscellaneousInternallyRecordedDealingsAmountsReturnedTotalA
                    _currentXPath.Push("/tns:MiscellaneousInternallyRecordedDealingsAmountsReturnedTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousInternallyRecordedDealingsAmountsReturnedTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS170 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousInternallyRecordedDealingsAmountsReturnedTotalA

                    #region IncomeDe
                    _currentXPath.Push("/tns:IncomeDe");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeDe")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS171 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeDe
                } // End of if BranchOperationsOther node exists

                _currentXPath.Pop();
                #endregion BranchOperationsOther

                #region InternationalDealingsFinancialArrangementsI
                _currentXPath.Push("/tns:InternationalDealingsFinancialArrangementsI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsFinancialArrangementsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS172 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsFinancialArrangementsI

                #region FinancialArrangements
                _currentXPath.Push("/tns:FinancialArrangements");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialArrangements")
                {
                    report.RP_FinancialArrangementsCollectionExists = true;
                    report.RP_FinancialArrangementsCollectionCount += 1;
                    _found = true;

                    #region ExpenseInterestDebtReceivedAverageBalanceA
                    _currentXPath.Push("/tns:ExpenseInterestDebtReceivedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInterestDebtReceivedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS173 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseInterestDebtReceivedAverageBalanceA

                    #region IncomeInterestDebtProvidedAverageBalanceA
                    _currentXPath.Push("/tns:IncomeInterestDebtProvidedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInterestDebtProvidedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS174 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeInterestDebtProvidedAverageBalanceA

                    #region ExpenseInterestEquityReceivedAverageBalanceA
                    _currentXPath.Push("/tns:ExpenseInterestEquityReceivedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInterestEquityReceivedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS175 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseInterestEquityReceivedAverageBalanceA

                    #region IncomeInterestEquityProvidedAverageBalanceA
                    _currentXPath.Push("/tns:IncomeInterestEquityProvidedAverageBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInterestEquityProvidedAverageBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS176 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeInterestEquityProvidedAverageBalanceA
                } // End of if FinancialArrangements node exists

                _currentXPath.Pop();
                #endregion FinancialArrangements

                #region InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI
                _currentXPath.Push("/tns:InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS177 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI

                #region InternationalDealingsTaxTimingMethodElectionsCCollection
                _currentXPath.Push("/tns:InternationalDealingsTaxTimingMethodElectionsCCollection");
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxTimingMethodElectionsCCollection")
                {
                    report.IDS178Collection = new List<string>();
                    ReadNext(reader);
                    _currentXPath.Push("/tns:InternationalDealingsTaxTimingMethodElectionsC");
                    // 5. use case
                    while (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxTimingMethodElectionsC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before the currentXPath is updated.
                                report.IDS178Collection.Add(currentValue);
                            }
                        }
                        _found = true;
                    }
                    _currentXPath.Pop();
                }
                _currentXPath.Pop();
                #endregion InternationalDealingsTaxTimingMethodElectionsCCollection

                #region MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI
                _currentXPath.Push("/tns:MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS184 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI

                #region MiscellaneousControlledForeignCompanyTrustInterestAcquiredI
                _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestAcquiredI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestAcquiredI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS367 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousControlledForeignCompanyTrustInterestAcquiredI

                #region MiscellaneousControlledForeignCompanyTrustInterestDisposedI
                _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestDisposedI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestDisposedI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS371 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousControlledForeignCompanyTrustInterestDisposedI

                #region Listed
                _currentXPath.Push("/tns:Listed");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Listed")
                {
                    report.RP_ListedCollectionExists = true;
                    report.RP_ListedCollectionCount += 1;
                    _found = true;

                    #region MiscellaneousControlledForeignCompaniesTrustsInterestCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesTrustsInterestCt");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousControlledForeignCompaniesTrustsInterestCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS185 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompaniesTrustsInterestCt

                    #region MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS368 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt

                    #region MiscellaneousControlledForeignCompanyTrustInterestDisposedCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestDisposedCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestDisposedCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS372 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyTrustInterestDisposedCt

                    #region MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS375 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC
                } // End of if Listed node exists

                _currentXPath.Pop();
                #endregion Listed

                #region Specified
                _currentXPath.Push("/tns:Specified");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Specified")
                {
                    report.RP_SpecifiedCollectionExists = true;
                    report.RP_SpecifiedCollectionCount += 1;
                    _found = true;

                    #region MiscellaneousControlledForeignCompaniesTrustsInterestCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesTrustsInterestCt");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousControlledForeignCompaniesTrustsInterestCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS186 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompaniesTrustsInterestCt

                    #region MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS369 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt

                    #region MiscellaneousControlledForeignCompanyTrustInterestDisposedCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestDisposedCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestDisposedCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS373 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyTrustInterestDisposedCt

                    #region MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS376 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC
                } // End of if Specified node exists

                _currentXPath.Pop();
                #endregion Specified

                #region Unlisted
                _currentXPath.Push("/tns:Unlisted");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Unlisted")
                {
                    report.RP_UnlistedCollectionExists = true;
                    report.RP_UnlistedCollectionCount += 1;
                    _found = true;

                    #region MiscellaneousControlledForeignCompaniesTrustsInterestCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesTrustsInterestCt");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousControlledForeignCompaniesTrustsInterestCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS187 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompaniesTrustsInterestCt

                    #region MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS370 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt

                    #region MiscellaneousControlledForeignCompanyTrustInterestDisposedCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyTrustInterestDisposedCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyTrustInterestDisposedCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS374 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyTrustInterestDisposedCt

                    #region MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS377 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC
                } // End of if Unlisted node exists

                _currentXPath.Pop();
                #endregion Unlisted

                #region MiscellaneousControlledForeignCompaniesExcludedTaintedInterestIncomeI
                _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesExcludedTaintedInterestIncomeI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompaniesExcludedTaintedInterestIncomeI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS433 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousControlledForeignCompaniesExcludedTaintedInterestIncomeI

                #region MiscellaneousControlledForeignCompaniesBankingLicencesI
                _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesBankingLicencesI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompaniesBankingLicencesI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS434 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousControlledForeignCompaniesBankingLicencesI

                #region MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI
                _currentXPath.Push("/tns:MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS435 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI

                #region ListedCFCAmounts
                _currentXPath.Push("/tns:ListedCFCAmounts");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ListedCFCAmounts")
                {
                    report.RP_ListedCFCAmountsCollectionExists = true;
                    report.RP_ListedCFCAmountsCollectionCount += 1;
                    _found = true;

                    #region IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS436 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA

                    #region MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS440 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt

                    #region IncomeControlledForeignCompaniesAttributableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS188 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAttributableIncomeA

                    #region IncomeGrossRevenueActiveIncomeTestSatisfiedA
                    _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestSatisfiedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS443 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeGrossRevenueActiveIncomeTestSatisfiedA

                    #region IncomeGrossRevenueActiveIncomeTestNotSatisfiedA
                    _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestNotSatisfiedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS447 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeGrossRevenueActiveIncomeTestNotSatisfiedA

                    #region IncomeControlledForeignCompaniesAdjustedTaintedConcessionIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAdjustedTaintedConcessionIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAdjustedTaintedConcessionIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS378 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAdjustedTaintedConcessionIncomeA

                    #region IncomeControlledForeignCompaniesAdjustedTaintedIncomeNotDerivedA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAdjustedTaintedIncomeNotDerivedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAdjustedTaintedIncomeNotDerivedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS379 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAdjustedTaintedIncomeNotDerivedA

                    #region IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS380 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA

                    #region IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS381 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA

                    #region IncomeControlledForeignCompaniesNotionalAllowableDeductionsA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAllowableDeductionsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAllowableDeductionsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS451 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesNotionalAllowableDeductionsA
                } // End of if ListedCFCAmounts node exists

                _currentXPath.Pop();
                #endregion ListedCFCAmounts

                #region SpecifiedCFCAmounts
                _currentXPath.Push("/tns:SpecifiedCFCAmounts");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "SpecifiedCFCAmounts")
                {
                    report.RP_SpecifiedCFCAmountsCollectionExists = true;
                    report.RP_SpecifiedCFCAmountsCollectionCount += 1;
                    _found = true;

                    #region IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS437 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA

                    #region MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS441 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt

                    #region IncomeControlledForeignCompaniesAttributableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS189 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAttributableIncomeA

                    #region IncomeGrossRevenueActiveIncomeTestSatisfiedA
                    _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestSatisfiedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS444 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeGrossRevenueActiveIncomeTestSatisfiedA

                    #region IncomeGrossRevenueActiveIncomeTestNotSatisfiedA
                    _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestNotSatisfiedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS448 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeGrossRevenueActiveIncomeTestNotSatisfiedA

                    #region IncomeControlledForeignCompaniesAdjustedTaintedIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAdjustedTaintedIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAdjustedTaintedIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS382 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAdjustedTaintedIncomeA

                    #region IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS383 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA

                    #region IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS384 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA

                    #region IncomeControlledForeignCompaniesNotionalAllowableDeductionsA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAllowableDeductionsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAllowableDeductionsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS452 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesNotionalAllowableDeductionsA
                } // End of if SpecifiedCFCAmounts node exists

                _currentXPath.Pop();
                #endregion SpecifiedCFCAmounts

                #region UnlistedCFCAmounts
                _currentXPath.Push("/tns:UnlistedCFCAmounts");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "UnlistedCFCAmounts")
                {
                    report.RP_UnlistedCFCAmountsCollectionExists = true;
                    report.RP_UnlistedCFCAmountsCollectionCount += 1;
                    _found = true;

                    #region IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS438 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA

                    #region MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt
                    _currentXPath.Push("/tns:MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS442 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt

                    #region IncomeControlledForeignCompaniesAttributableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS190 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAttributableIncomeA

                    #region IncomeGrossRevenueActiveIncomeTestSatisfiedA
                    _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestSatisfiedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS445 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeGrossRevenueActiveIncomeTestSatisfiedA

                    #region IncomeGrossRevenueActiveIncomeTestNotSatisfiedA
                    _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestNotSatisfiedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS449 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeGrossRevenueActiveIncomeTestNotSatisfiedA

                    #region IncomeControlledForeignCompaniesAdjustedTaintedIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAdjustedTaintedIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAdjustedTaintedIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS385 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesAdjustedTaintedIncomeA

                    #region IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS386 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA

                    #region IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS387 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA

                    #region IncomeControlledForeignCompaniesNotionalAllowableDeductionsA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAllowableDeductionsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAllowableDeductionsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS453 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeControlledForeignCompaniesNotionalAllowableDeductionsA
                } // End of if UnlistedCFCAmounts node exists

                _currentXPath.Pop();
                #endregion UnlistedCFCAmounts

                #region IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeTotalA
                _currentXPath.Push("/tns:IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS439 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeTotalA

                #region IncomeControlledForeignCompaniesAttributableIncomeTotalA
                _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS191 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeControlledForeignCompaniesAttributableIncomeTotalA

                #region IncomeControlledForeignCompaniesChangeOfResidenceIncomeA
                _currentXPath.Push("/tns:IncomeControlledForeignCompaniesChangeOfResidenceIncomeA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesChangeOfResidenceIncomeA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS192 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeControlledForeignCompaniesChangeOfResidenceIncomeA

                #region IncomeControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotalA
                _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS193 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotalA

                #region IncomeGrossRevenueActiveIncomeTestSatisfiedTotalA
                _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestSatisfiedTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestSatisfiedTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS446 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeGrossRevenueActiveIncomeTestSatisfiedTotalA

                #region IncomeGrossRevenueActiveIncomeTestNotSatisfiedTotalA
                _currentXPath.Push("/tns:IncomeGrossRevenueActiveIncomeTestNotSatisfiedTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGrossRevenueActiveIncomeTestNotSatisfiedTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS450 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeGrossRevenueActiveIncomeTestNotSatisfiedTotalA

                #region IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA
                _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS388 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA

                #region IncomeControlledForeignCompaniesNotionalAllowableDeductionsTotalA
                _currentXPath.Push("/tns:IncomeControlledForeignCompaniesNotionalAllowableDeductionsTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesNotionalAllowableDeductionsTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS454 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeControlledForeignCompaniesNotionalAllowableDeductionsTotalA

                #region IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI
                _currentXPath.Push("/tns:IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS195 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI

                #region ListedForeignBranchOperations
                _currentXPath.Push("/tns:ListedForeignBranchOperations");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ListedForeignBranchOperations")
                {
                    report.RP_ListedForeignBranchOperationsCollectionExists = true;
                    report.RP_ListedForeignBranchOperationsCollectionCount += 1;
                    _found = true;

                    #region IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS196 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA

                    #region IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS200 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA

                    #region IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS320 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA
                } // End of if ListedForeignBranchOperations node exists

                _currentXPath.Pop();
                #endregion ListedForeignBranchOperations

                #region SpecifiedForeignBranchOperations
                _currentXPath.Push("/tns:SpecifiedForeignBranchOperations");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "SpecifiedForeignBranchOperations")
                {
                    report.RP_SpecifiedForeignBranchOperationsCollectionExists = true;
                    report.RP_SpecifiedForeignBranchOperationsCollectionCount += 1;
                    _found = true;

                    #region IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS197 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA

                    #region IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS201 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA

                    #region IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS321 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA
                } // End of if SpecifiedForeignBranchOperations node exists

                _currentXPath.Pop();
                #endregion SpecifiedForeignBranchOperations

                #region UnlistedForeignBranchOperations
                _currentXPath.Push("/tns:UnlistedForeignBranchOperations");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "UnlistedForeignBranchOperations")
                {
                    report.RP_UnlistedForeignBranchOperationsCollectionExists = true;
                    report.RP_UnlistedForeignBranchOperationsCollectionCount += 1;
                    _found = true;

                    #region IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS198 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA

                    #region IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS202 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA

                    #region IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA
                    _currentXPath.Push("/tns:IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS322 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA
                } // End of if UnlistedForeignBranchOperations node exists

                _currentXPath.Pop();
                #endregion UnlistedForeignBranchOperations

                #region ExpenseNonDeductibleExemptIncomeA
                _currentXPath.Push("/tns:ExpenseNonDeductibleExemptIncomeA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseNonDeductibleExemptIncomeA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS199 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion ExpenseNonDeductibleExemptIncomeA

                #region IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI
                _currentXPath.Push("/tns:IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS206 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI

                #region IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeTotalA
                _currentXPath.Push("/tns:IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS207 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeTotalA

                #region CapitalGainsTaxEventI
                _currentXPath.Push("/tns:CapitalGainsTaxEventI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxEventI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS208 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion CapitalGainsTaxEventI

                #region CGTEvents
                _currentXPath.Push("/tns:CGTEvents");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CGTEvents")
                {
                    report.RP_CGTEventsCollectionExists = true;
                    report.RP_CGTEventsCollectionCount += 1;
                    _found = true;

                    #region CapitalGainsTaxActiveForeignBusinessCapitalGainsA
                    _currentXPath.Push("/tns:CapitalGainsTaxActiveForeignBusinessCapitalGainsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxActiveForeignBusinessCapitalGainsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS209 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion CapitalGainsTaxActiveForeignBusinessCapitalGainsA

                    #region IncomeTaxActiveForeignBusinessCapitalGainsReductionA
                    _currentXPath.Push("/tns:IncomeTaxActiveForeignBusinessCapitalGainsReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxActiveForeignBusinessCapitalGainsReductionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS210 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxActiveForeignBusinessCapitalGainsReductionA

                    #region CapitalGainsTaxActiveForeignBusinessCapitalLossesA
                    _currentXPath.Push("/tns:CapitalGainsTaxActiveForeignBusinessCapitalLossesA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxActiveForeignBusinessCapitalLossesA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS211 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion CapitalGainsTaxActiveForeignBusinessCapitalLossesA

                    #region IncomeTaxActiveForeignBusinessCapitalLossesReductionA
                    _currentXPath.Push("/tns:IncomeTaxActiveForeignBusinessCapitalLossesReductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxActiveForeignBusinessCapitalLossesReductionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS212 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxActiveForeignBusinessCapitalLossesReductionA
                } // End of if CGTEvents node exists

                _currentXPath.Pop();
                #endregion CGTEvents

                #region MiscellaneousTransferPropertyMoneyOrServicesI
                _currentXPath.Push("/tns:MiscellaneousTransferPropertyMoneyOrServicesI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousTransferPropertyMoneyOrServicesI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS213 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousTransferPropertyMoneyOrServicesI

                #region Transfer

                _currentXPath.Push("/tns:TransferCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TransferCollection")
                {
                    report.RP_TransferCollection = new List<IDS2025.RP_Transfer>();
                    report.RP_TransferCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "Transfer")
                    {
                        _found = true;

                        IDS2025.RP_Transfer transfer = new IDS2025.RP_Transfer();
                        report.RP_TransferCollection.Add(transfer);
                        report.RP_TransferCollectionCount += 1;
                        transfer.OccurrenceIndex = report.RP_TransferCollectionCount;

                        _currentXPath.Push("/tns:Transfer[" + report.RP_TransferCollectionCount + "]");

                        #region MiscellaneousPropertyMoneyOrServicesA
                        _currentXPath.Push("/tns:MiscellaneousPropertyMoneyOrServicesA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousPropertyMoneyOrServicesA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        transfer.IDS214 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousPropertyMoneyOrServicesA

                        #region ItemOrderN
                        _currentXPath.Push("/tns:ItemOrderN");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "ItemOrderN")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    transfer.IDS215 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ItemOrderN
                        _currentXPath.Pop();
                    } // End of while loop for Transfer
                } // End of if TransferCollection node exists

                _currentXPath.Pop();
                #endregion Transfer

                #region MiscellaneousNonResidentTrustInterestEntitlementOrBeneficiaryI
                _currentXPath.Push("/tns:MiscellaneousNonResidentTrustInterestEntitlementOrBeneficiaryI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousNonResidentTrustInterestEntitlementOrBeneficiaryI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS216 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousNonResidentTrustInterestEntitlementOrBeneficiaryI

                #region MiscellaneousOffshoreDealingsComplianceI
                _currentXPath.Push("/tns:MiscellaneousOffshoreDealingsComplianceI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousOffshoreDealingsComplianceI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS393 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousOffshoreDealingsComplianceI

                #region HubArrangement

                _currentXPath.Push("/tns:HubArrangementCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "HubArrangementCollection")
                {
                    report.RP_HubArrangementCollection = new List<IDS2025.RP_HubArrangement>();
                    report.RP_HubArrangementCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "HubArrangement")
                    {
                        _found = true;

                        IDS2025.RP_HubArrangement hubArrangement = new IDS2025.RP_HubArrangement();
                        report.RP_HubArrangementCollection.Add(hubArrangement);
                        report.RP_HubArrangementCollectionCount += 1;
                        hubArrangement.OccurrenceIndex = report.RP_HubArrangementCollectionCount;

                        _currentXPath.Push("/tns:HubArrangement[" + report.RP_HubArrangementCollectionCount + "]");

                        #region MiscellaneousC
                        _currentXPath.Push("/tns:MiscellaneousC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    hubArrangement.IDS394 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousC

                        #region ExpenseImportsA
                        _currentXPath.Push("/tns:ExpenseImportsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "ExpenseImportsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        hubArrangement.IDS395 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExpenseImportsA

                        #region RevenueExportsA
                        _currentXPath.Push("/tns:RevenueExportsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "RevenueExportsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        hubArrangement.IDS396 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RevenueExportsA
                        _currentXPath.Pop();
                    } // End of while loop for HubArrangement
                } // End of if HubArrangementCollection node exists

                _currentXPath.Pop();
                #endregion HubArrangement

                #region MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI
                _currentXPath.Push("/tns:MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS217 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI

                #region MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestInCt
                _currentXPath.Push("/tns:MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestInCt");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestInCt")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS218 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestInCt

                #region AssetsInvestmentForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotalA
                _currentXPath.Push("/tns:AssetsInvestmentForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotalA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsInvestmentForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotalA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS219 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion AssetsInvestmentForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotalA

                #region LiabilitiesThinCapitalisationProvisionsAppliedI
                _currentXPath.Push("/tns:LiabilitiesThinCapitalisationProvisionsAppliedI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesThinCapitalisationProvisionsAppliedI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS220 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesThinCapitalisationProvisionsAppliedI

                #region LiabilitiesThinCapitalisation2MillionThresholdTestI
                _currentXPath.Push("/tns:LiabilitiesThinCapitalisation2MillionThresholdTestI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThinCapitalisation2MillionThresholdTestI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS400 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesThinCapitalisation2MillionThresholdTestI

                #region LiabilitiesThinCapitalisation90PercentAssetThresholdTestI
                _currentXPath.Push("/tns:LiabilitiesThinCapitalisation90PercentAssetThresholdTestI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThinCapitalisation90PercentAssetThresholdTestI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS401 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesThinCapitalisation90PercentAssetThresholdTestI

                #region LiabilitiesThinCapitalisationExemptionSpecialPurposeEntitiesI
                _currentXPath.Push("/tns:LiabilitiesThinCapitalisationExemptionSpecialPurposeEntitiesI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThinCapitalisationExemptionSpecialPurposeEntitiesI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS402 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesThinCapitalisationExemptionSpecialPurposeEntitiesI

                #region LiabilitiesThinCapitalisationAustralianResidentEntityNonInOutInvestingEntityI
                _currentXPath.Push("/tns:LiabilitiesThinCapitalisationAustralianResidentEntityNonInOutInvestingEntityI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThinCapitalisationAustralianResidentEntityNonInOutInvestingEntityI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS630 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesThinCapitalisationAustralianResidentEntityNonInOutInvestingEntityI

                #region LiabilitiesThinCapitalisationAustralianResidentCompanyElectedI
                _currentXPath.Push("/tns:LiabilitiesThinCapitalisationAustralianResidentCompanyElectedI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThinCapitalisationAustralianResidentCompanyElectedI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS221 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesThinCapitalisationAustralianResidentCompanyElectedI

                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IDS222 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId

                #region ThinCapitalisation
                _currentXPath.Push("/tns:ThinCapitalisation");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ThinCapitalisation")
                {
                    report.RP_ThinCapitalisationCollectionExists = true;
                    report.RP_ThinCapitalisationCollectionCount += 1;
                    _found = true;

                    #region LiabilitiesEntityTypeC
                    _currentXPath.Push("/tns:LiabilitiesEntityTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesEntityTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS223 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesEntityTypeC

                    #region LiabilitiesFinancialEntityTypeC
                    _currentXPath.Push("/tns:LiabilitiesFinancialEntityTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFinancialEntityTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS560 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesFinancialEntityTypeC

                    #region LiabilitiesAveragingMethodAdoptedC
                    _currentXPath.Push("/tns:LiabilitiesAveragingMethodAdoptedC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesAveragingMethodAdoptedC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS225 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesAveragingMethodAdoptedC

                    #region LiabilitiesDeductionDebtA
                    _currentXPath.Push("/tns:LiabilitiesDeductionDebtA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDeductionDebtA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS226 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDeductionDebtA

                    #region LiabilitiesRelatedNonResidentsDebtDeductionA
                    _currentXPath.Push("/tns:LiabilitiesRelatedNonResidentsDebtDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesRelatedNonResidentsDebtDeductionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS227 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesRelatedNonResidentsDebtDeductionA

                    #region LiabilitiesDeductionDebtDisallowedA
                    _currentXPath.Push("/tns:LiabilitiesDeductionDebtDisallowedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDeductionDebtDisallowedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS228 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDeductionDebtDisallowedA

                    #region LiabilitiesDebtAdjustedAverageA
                    _currentXPath.Push("/tns:LiabilitiesDebtAdjustedAverageA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtAdjustedAverageA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS229 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDebtAdjustedAverageA

                    #region InterestIncomeAndOtherAmountsA
                    _currentXPath.Push("/tns:InterestIncomeAndOtherAmountsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InterestIncomeAndOtherAmountsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS563 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InterestIncomeAndOtherAmountsA

                    #region DerivedNonResidentInterestIncomeAndOtherAmountsA
                    _currentXPath.Push("/tns:DerivedNonResidentInterestIncomeAndOtherAmountsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "DerivedNonResidentInterestIncomeAndOtherAmountsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS564 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion DerivedNonResidentInterestIncomeAndOtherAmountsA

                    #region EBITDA
                    _currentXPath.Push("/tns:EBITDA");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "EBITDA")
                    {
                        report.RP_ThinCapitalisation_EBITDACollectionExists = true;
                        report.RP_ThinCapitalisation_EBITDACollectionCount += 1;
                        _found = true;

                        #region IncomeTaxModifiedTaxableIncomeOrLossA
                        _currentXPath.Push("/tns:IncomeTaxModifiedTaxableIncomeOrLossA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxModifiedTaxableIncomeOrLossA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS565 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion IncomeTaxModifiedTaxableIncomeOrLossA

                        #region LiabilitiesNetDeductionDebtA
                        _currentXPath.Push("/tns:LiabilitiesNetDeductionDebtA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesNetDeductionDebtA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS566 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNetDeductionDebtA

                        #region LiabilitiesModifiedDepreciationAndForestryCostsA
                        _currentXPath.Push("/tns:LiabilitiesModifiedDepreciationAndForestryCostsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesModifiedDepreciationAndForestryCostsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS567 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesModifiedDepreciationAndForestryCostsA

                        #region LiabilitiesExcessTaxEBITDAA
                        _currentXPath.Push("/tns:LiabilitiesExcessTaxEBITDAA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesExcessTaxEBITDAA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS568 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesExcessTaxEBITDAA

                        #region LiabilitiesEntityTaxEBITDAA
                        _currentXPath.Push("/tns:LiabilitiesEntityTaxEBITDAA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesEntityTaxEBITDAA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS569 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesEntityTaxEBITDAA

                        #region EarningsLimitA
                        _currentXPath.Push("/tns:EarningsLimitA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "EarningsLimitA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS570 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion EarningsLimitA

                        #region TotalDisregardedAmountsA
                        _currentXPath.Push("/tns:TotalDisregardedAmountsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "TotalDisregardedAmountsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS571 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion TotalDisregardedAmountsA

                        #region HighestExcessEBITDA

                        _currentXPath.Push("/tns:HighestExcessEBITDACollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "HighestExcessEBITDACollection")
                        {
                            report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollection = new List<IDS2025.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDA>();
                            report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollectionExists = true;

                            _found = true;

                            while (ReadToNextElement(reader, false) && reader.LocalName == "HighestExcessEBITDA")
                            {
                                _found = true;

                                IDS2025.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDA highestExcessEBITDA = new IDS2025.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDA();
                                report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollection.Add(highestExcessEBITDA);
                                report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollectionCount += 1;
                                highestExcessEBITDA.OccurrenceIndex = report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollectionCount;

                                _currentXPath.Push("/tns:HighestExcessEBITDA[" + report.RP_ThinCapitalisation_EBITDA_HighestExcessEBITDACollectionCount + "]");

                                #region OrganisationNameDetailsOrganisationalNameT
                                _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            highestExcessEBITDA.IDS572 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion OrganisationNameDetailsOrganisationalNameT

                                #region ControlledEntityA
                                _currentXPath.Push("/tns:ControlledEntityA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "ControlledEntityA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                highestExcessEBITDA.IDS573 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion ControlledEntityA

                                #region ControlledEntityDividedByA
                                _currentXPath.Push("/tns:ControlledEntityDividedByA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "ControlledEntityDividedByA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                highestExcessEBITDA.IDS574 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion ControlledEntityDividedByA
                                _currentXPath.Pop();
                            } // End of while loop for HighestExcessEBITDA
                        } // End of if HighestExcessEBITDACollection node exists

                        _currentXPath.Pop();
                        #endregion HighestExcessEBITDA
                    } // End of if EBITDA node exists

                    _currentXPath.Pop();
                    #endregion EBITDA

                    #region LiabilitiesGroupRatioTestI
                    _currentXPath.Push("/tns:LiabilitiesGroupRatioTestI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesGroupRatioTestI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS575 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesGroupRatioTestI

                    #region GroupRatioTest
                    _currentXPath.Push("/tns:GroupRatioTest");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "GroupRatioTest")
                    {
                        report.RP_ThinCapitalisation_GroupRatioTestCollectionExists = true;
                        report.RP_ThinCapitalisation_GroupRatioTestCollectionCount += 1;
                        _found = true;

                        #region LiabilitiesGroupEntitiesCt
                        _currentXPath.Push("/tns:LiabilitiesGroupEntitiesCt");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesGroupEntitiesCt")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS576 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesGroupEntitiesCt

                        #region LiabilitiesGroupMembersCt
                        _currentXPath.Push("/tns:LiabilitiesGroupMembersCt");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesGroupMembersCt")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS577 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesGroupMembersCt

                        #region LiabilitiesNetThirdPartyInterestExpenseA
                        _currentXPath.Push("/tns:LiabilitiesNetThirdPartyInterestExpenseA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesNetThirdPartyInterestExpenseA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS578 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNetThirdPartyInterestExpenseA

                        #region LiabilitiesNetProfitA
                        _currentXPath.Push("/tns:LiabilitiesNetProfitA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesNetProfitA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS579 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNetProfitA

                        #region LiabilitiesNetThirdPartyInterestExpenseAdjustedA
                        _currentXPath.Push("/tns:LiabilitiesNetThirdPartyInterestExpenseAdjustedA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesNetThirdPartyInterestExpenseAdjustedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS580 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNetThirdPartyInterestExpenseAdjustedA

                        #region LiabilitiesDepreciationAndAmortisationExpensesA
                        _currentXPath.Push("/tns:LiabilitiesDepreciationAndAmortisationExpensesA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDepreciationAndAmortisationExpensesA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS581 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDepreciationAndAmortisationExpensesA

                        #region LiabilitiesGroupTaxEBITDAA
                        _currentXPath.Push("/tns:LiabilitiesGroupTaxEBITDAA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesGroupTaxEBITDAA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS582 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesGroupTaxEBITDAA

                        #region LiabilitiesA
                        _currentXPath.Push("/tns:LiabilitiesA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS583 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesA

                        #region GroupRatioEarningsLimitA
                        _currentXPath.Push("/tns:GroupRatioEarningsLimitA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "GroupRatioEarningsLimitA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS584 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion GroupRatioEarningsLimitA

                        #region GRGroupMembers

                        _currentXPath.Push("/tns:GRGroupMembersCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GRGroupMembersCollection")
                        {
                            report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection = new List<IDS2025.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembers>();
                            report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionExists = true;

                            _found = true;

                            while (ReadToNextElement(reader, false) && reader.LocalName == "GRGroupMembers")
                            {
                                _found = true;

                                IDS2025.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembers grGroupMembers = new IDS2025.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembers();
                                report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollection.Add(grGroupMembers);
                                report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionCount += 1;
                                grGroupMembers.OccurrenceIndex = report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionCount;

                                _currentXPath.Push("/tns:GRGroupMembers[" + report.RP_ThinCapitalisation_GroupRatioTest_GRGroupMembersCollectionCount + "]");

                                #region OrganisationNameDetailsOrganisationalNameT
                                _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            grGroupMembers.IDS585 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion OrganisationNameDetailsOrganisationalNameT

                                #region LiabilitiesGroupMemberEBITDAA
                                _currentXPath.Push("/tns:LiabilitiesGroupMemberEBITDAA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesGroupMemberEBITDAA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                grGroupMembers.IDS586 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion LiabilitiesGroupMemberEBITDAA

                                #region LiabilitiesNetThirdPartyInterestExpenseAdjustedA
                                _currentXPath.Push("/tns:LiabilitiesNetThirdPartyInterestExpenseAdjustedA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesNetThirdPartyInterestExpenseAdjustedA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                grGroupMembers.IDS587 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion LiabilitiesNetThirdPartyInterestExpenseAdjustedA

                                #region OrganisationDetailsInternationalRelatedPartyCountryC
                                _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            grGroupMembers.IDS588 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion OrganisationDetailsInternationalRelatedPartyCountryC
                                _currentXPath.Pop();
                            } // End of while loop for GRGroupMembers
                        } // End of if GRGroupMembersCollection node exists

                        _currentXPath.Pop();
                        #endregion GRGroupMembers
                    } // End of if GroupRatioTest node exists

                    _currentXPath.Pop();
                    #endregion GroupRatioTest

                    #region LiabilitiesThirdPartyDebtTestI
                    _currentXPath.Push("/tns:LiabilitiesThirdPartyDebtTestI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThirdPartyDebtTestI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS589 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesThirdPartyDebtTestI

                    #region ThirdPartyDebtTest
                    _currentXPath.Push("/tns:ThirdPartyDebtTest");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ThirdPartyDebtTest")
                    {
                        report.RP_ThinCapitalisation_ThirdPartyDebtTestCollectionExists = true;
                        report.RP_ThinCapitalisation_ThirdPartyDebtTestCollectionCount += 1;
                        _found = true;

                        #region LiabilitiesChoiceC
                        _currentXPath.Push("/tns:LiabilitiesChoiceC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesChoiceC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS590 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesChoiceC

                        #region EntityThirdPartyEarningsLimitA
                        _currentXPath.Push("/tns:EntityThirdPartyEarningsLimitA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "EntityThirdPartyEarningsLimitA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS591 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion EntityThirdPartyEarningsLimitA

                        #region LiabilitiesSubsections820Dash427AI
                        _currentXPath.Push("/tns:LiabilitiesSubsections820Dash427AI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesSubsections820Dash427AI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS592 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesSubsections820Dash427AI

                        #region LiabilitiesAssetsHeldByAnotherMemberOfTheObligorGroupI
                        _currentXPath.Push("/tns:LiabilitiesAssetsHeldByAnotherMemberOfTheObligorGroupI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesAssetsHeldByAnotherMemberOfTheObligorGroupI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS593 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesAssetsHeldByAnotherMemberOfTheObligorGroupI

                        #region LiabilitiesMembershipInterests820Dash427AI
                        _currentXPath.Push("/tns:LiabilitiesMembershipInterests820Dash427AI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesMembershipInterests820Dash427AI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS594 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesMembershipInterests820Dash427AI

                        #region LiabilitiesMinorOrInsignificantAssetsDisregardedPursuant820Dash427AI
                        _currentXPath.Push("/tns:LiabilitiesMinorOrInsignificantAssetsDisregardedPursuant820Dash427AI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesMinorOrInsignificantAssetsDisregardedPursuant820Dash427AI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS595 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesMinorOrInsignificantAssetsDisregardedPursuant820Dash427AI

                        #region LiabilitiesThirdPartyDebtInterestA
                        _currentXPath.Push("/tns:LiabilitiesThirdPartyDebtInterestA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesThirdPartyDebtInterestA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS596 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesThirdPartyDebtInterestA

                        #region LiabilitiesConduitFinancingArrangementI
                        _currentXPath.Push("/tns:LiabilitiesConduitFinancingArrangementI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesConduitFinancingArrangementI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS597 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesConduitFinancingArrangementI

                        #region LiabilitiesConduitFinancerI
                        _currentXPath.Push("/tns:LiabilitiesConduitFinancerI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesConduitFinancerI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS598 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesConduitFinancerI

                        #region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS599 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AustralianBusinessNumberId

                        #region LiabilitiesConduitFinancerCostsIncurredAndDifferFromTermsI
                        _currentXPath.Push("/tns:LiabilitiesConduitFinancerCostsIncurredAndDifferFromTermsI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesConduitFinancerCostsIncurredAndDifferFromTermsI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS600 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesConduitFinancerCostsIncurredAndDifferFromTermsI

                        #region LiabilitiesConduitFinancerFinancingArrangementTypeC
                        _currentXPath.Push("/tns:LiabilitiesConduitFinancerFinancingArrangementTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesConduitFinancerFinancingArrangementTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS601 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesConduitFinancerFinancingArrangementTypeC

                        #region LiabilitiesNotConduitFinancerCostsIncurredAndDifferFromTermsI
                        _currentXPath.Push("/tns:LiabilitiesNotConduitFinancerCostsIncurredAndDifferFromTermsI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesNotConduitFinancerCostsIncurredAndDifferFromTermsI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS602 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNotConduitFinancerCostsIncurredAndDifferFromTermsI

                        #region LiabilitiesNonConduitFinancerFinancingArrangementTypeC
                        _currentXPath.Push("/tns:LiabilitiesNonConduitFinancerFinancingArrangementTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesNonConduitFinancerFinancingArrangementTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS603 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNonConduitFinancerFinancingArrangementTypeC

                        #region LiabilitiesCrossStapleArrangementsI
                        _currentXPath.Push("/tns:LiabilitiesCrossStapleArrangementsI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesCrossStapleArrangementsI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS604 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesCrossStapleArrangementsI

                        #region LiabilitiesMemberOfAnObligorGroupNotIssuedByTheEntityI
                        _currentXPath.Push("/tns:LiabilitiesMemberOfAnObligorGroupNotIssuedByTheEntityI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesMemberOfAnObligorGroupNotIssuedByTheEntityI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS605 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesMemberOfAnObligorGroupNotIssuedByTheEntityI
                    } // End of if ThirdPartyDebtTest node exists

                    _currentXPath.Pop();
                    #endregion ThirdPartyDebtTest

                    #region LiabilitiesSpecialPurposeEntityI
                    _currentXPath.Push("/tns:LiabilitiesSpecialPurposeEntityI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesSpecialPurposeEntityI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS606 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesSpecialPurposeEntityI

                    #region SpecialPurposeEntity
                    _currentXPath.Push("/tns:SpecialPurposeEntity");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SpecialPurposeEntity")
                    {
                        report.RP_ThinCapitalisation_SpecialPurposeEntityCollectionExists = true;
                        report.RP_ThinCapitalisation_SpecialPurposeEntityCollectionCount += 1;
                        _found = true;

                        #region LiabilitiesDeductionDebtA
                        _currentXPath.Push("/tns:LiabilitiesDeductionDebtA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDeductionDebtA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS607 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDeductionDebtA

                        #region LiabilitiesRelatedNonResidentsDebtDeductionA
                        _currentXPath.Push("/tns:LiabilitiesRelatedNonResidentsDebtDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesRelatedNonResidentsDebtDeductionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS608 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesRelatedNonResidentsDebtDeductionA

                        #region LiabilitiesDebtAverageA
                        _currentXPath.Push("/tns:LiabilitiesDebtAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS609 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtAverageA

                        #region InterestIncomeAndOtherAmountsA
                        _currentXPath.Push("/tns:InterestIncomeAndOtherAmountsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InterestIncomeAndOtherAmountsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS610 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InterestIncomeAndOtherAmountsA

                        #region DerivedNonResidentInterestIncomeAndOtherAmountsA
                        _currentXPath.Push("/tns:DerivedNonResidentInterestIncomeAndOtherAmountsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "DerivedNonResidentInterestIncomeAndOtherAmountsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS611 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion DerivedNonResidentInterestIncomeAndOtherAmountsA
                    } // End of if SpecialPurposeEntity node exists

                    _currentXPath.Pop();
                    #endregion SpecialPurposeEntity

                    #region FRTDisallowedamount
                    _currentXPath.Push("/tns:FRTDisallowedamount");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FRTDisallowedamount")
                    {
                        report.RP_ThinCapitalisation_FRTDisallowedamountCollectionExists = true;
                        report.RP_ThinCapitalisation_FRTDisallowedamountCollectionCount += 1;
                        _found = true;

                        #region LiabilitiesFixedRatioTestDisallowedAmountI
                        _currentXPath.Push("/tns:LiabilitiesFixedRatioTestDisallowedAmountI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFixedRatioTestDisallowedAmountI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS631 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesFixedRatioTestDisallowedAmountI

                        #region LiabilitiesFixedRatioTestDisallowedC
                        _currentXPath.Push("/tns:LiabilitiesFixedRatioTestDisallowedC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFixedRatioTestDisallowedC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.IDS632 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesFixedRatioTestDisallowedC

                        #region LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA
                        _currentXPath.Push("/tns:LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS633 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA

                        #region LiabilitiesFixedRatioTestDisallowedCarriedForwardA
                        _currentXPath.Push("/tns:LiabilitiesFixedRatioTestDisallowedCarriedForwardA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFixedRatioTestDisallowedCarriedForwardA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS634 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesFixedRatioTestDisallowedCarriedForwardA

                        #region LiabilitiesTaxConsolidatedGroupMultipleEntryConsolidatedI
                        _currentXPath.Push("/tns:LiabilitiesTaxConsolidatedGroupMultipleEntryConsolidatedI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesTaxConsolidatedGroupMultipleEntryConsolidatedI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS635 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesTaxConsolidatedGroupMultipleEntryConsolidatedI

                        #region LiabilitiesFixedRatioTestDisallowedAmountsTransferredI
                        _currentXPath.Push("/tns:LiabilitiesFixedRatioTestDisallowedAmountsTransferredI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFixedRatioTestDisallowedAmountsTransferredI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.IDS636 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesFixedRatioTestDisallowedAmountsTransferredI

                        #region LiabilitiesFixedRatioTestDisallowedCalculatedA
                        _currentXPath.Push("/tns:LiabilitiesFixedRatioTestDisallowedCalculatedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesFixedRatioTestDisallowedCalculatedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS637 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesFixedRatioTestDisallowedCalculatedA
                    } // End of if FRTDisallowedamount node exists

                    _currentXPath.Pop();
                    #endregion FRTDisallowedamount

                    #region AllADI
                    _currentXPath.Push("/tns:AllADI");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AllADI")
                    {
                        report.RP_ThinCapitalisation_AllADICollectionExists = true;
                        report.RP_ThinCapitalisation_AllADICollectionCount += 1;
                        _found = true;

                        #region LiabilitiesEquityCapitalAdjustedAverageA
                        _currentXPath.Push("/tns:LiabilitiesEquityCapitalAdjustedAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesEquityCapitalAdjustedAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS231 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesEquityCapitalAdjustedAverageA

                        #region SafeHarbourA
                        _currentXPath.Push("/tns:SafeHarbourA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "SafeHarbourA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS232 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion SafeHarbourA

                        #region EquityShortfallA
                        _currentXPath.Push("/tns:EquityShortfallA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "EquityShortfallA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS233 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion EquityShortfallA
                    } // End of if AllADI node exists

                    _currentXPath.Pop();
                    #endregion AllADI

                    #region OutwardInvestingADI
                    _currentXPath.Push("/tns:OutwardInvestingADI");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OutwardInvestingADI")
                    {
                        report.RP_ThinCapitalisation_OutwardInvestingADICollectionExists = true;
                        report.RP_ThinCapitalisation_OutwardInvestingADICollectionCount += 1;
                        _found = true;

                        #region LiabilitiesRiskWeightedAssetsAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesRiskWeightedAssetsAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesRiskWeightedAssetsAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS234 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesRiskWeightedAssetsAverageValueA

                        #region LiabilitiesEquityCapitalAttributableToOverseasPermanentEstablishmentsA
                        _currentXPath.Push("/tns:LiabilitiesEquityCapitalAttributableToOverseasPermanentEstablishmentsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesEquityCapitalAttributableToOverseasPermanentEstablishmentsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS235 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesEquityCapitalAttributableToOverseasPermanentEstablishmentsA

                        #region LiabilitiesRiskWeightedAssetsOverseasPermanentEstablishmentsAverageValueTotalA
                        _currentXPath.Push("/tns:LiabilitiesRiskWeightedAssetsOverseasPermanentEstablishmentsAverageValueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesRiskWeightedAssetsOverseasPermanentEstablishmentsAverageValueTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS236 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesRiskWeightedAssetsOverseasPermanentEstablishmentsAverageValueTotalA

                        #region LiabilitiesControlledForeignADIEntityEquityAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesControlledForeignADIEntityEquityAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesControlledForeignADIEntityEquityAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS237 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesControlledForeignADIEntityEquityAverageValueA

                        #region LiabilitiesTier1PrudentialCapitalDeductionA
                        _currentXPath.Push("/tns:LiabilitiesTier1PrudentialCapitalDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesTier1PrudentialCapitalDeductionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS238 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesTier1PrudentialCapitalDeductionA
                    } // End of if OutwardInvestingADI node exists

                    _currentXPath.Pop();
                    #endregion OutwardInvestingADI

                    #region InwardInvestingADI
                    _currentXPath.Push("/tns:InwardInvestingADI");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InwardInvestingADI")
                    {
                        report.RP_ThinCapitalisation_InwardInvestingADICollectionExists = true;
                        report.RP_ThinCapitalisation_InwardInvestingADICollectionCount += 1;
                        _found = true;

                        #region RiskWeightedAssetsAttributableToAustralianPermanentEstablishmentsAverageA
                        _currentXPath.Push("/tns:RiskWeightedAssetsAttributableToAustralianPermanentEstablishmentsAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RiskWeightedAssetsAttributableToAustralianPermanentEstablishmentsAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS239 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion RiskWeightedAssetsAttributableToAustralianPermanentEstablishmentsAverageA

                        #region DotationCapitalAverageValueA
                        _currentXPath.Push("/tns:DotationCapitalAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "DotationCapitalAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS240 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion DotationCapitalAverageValueA

                        #region ArmsLengthCapitalA
                        _currentXPath.Push("/tns:ArmsLengthCapitalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ArmsLengthCapitalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS638 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ArmsLengthCapitalA
                    } // End of if InwardInvestingADI node exists

                    _currentXPath.Pop();
                    #endregion InwardInvestingADI

                    #region AustralianPlantationForestryEntity
                    _currentXPath.Push("/tns:AustralianPlantationForestryEntity");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianPlantationForestryEntity")
                    {
                        report.RP_ThinCapitalisation_AustralianPlantationForestryEntityCollectionExists = true;
                        report.RP_ThinCapitalisation_AustralianPlantationForestryEntityCollectionCount += 1;
                        _found = true;

                        #region AssetsValueAverageA
                        _currentXPath.Push("/tns:AssetsValueAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsValueAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS242 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsValueAverageA

                        #region AssetRevaluationA
                        _currentXPath.Push("/tns:AssetRevaluationA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetRevaluationA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS243 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetRevaluationA

                        #region LiabilitiesDebtMaximumAllowableA
                        _currentXPath.Push("/tns:LiabilitiesDebtMaximumAllowableA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtMaximumAllowableA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS244 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtMaximumAllowableA

                        #region LiabilitiesExcessDebtA
                        _currentXPath.Push("/tns:LiabilitiesExcessDebtA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesExcessDebtA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS245 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesExcessDebtA

                        #region LiabilitiesNonDebtLiabilitiesAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesNonDebtLiabilitiesAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesNonDebtLiabilitiesAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS246 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNonDebtLiabilitiesAverageValueA

                        #region LiabilitiesAssociateEntityDebtAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesAssociateEntityDebtAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesAssociateEntityDebtAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS247 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesAssociateEntityDebtAverageValueA

                        #region AssociateEntityEquityAverageValueA
                        _currentXPath.Push("/tns:AssociateEntityEquityAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssociateEntityEquityAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS248 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssociateEntityEquityAverageValueA

                        #region LiabilitiesAssociateEntityExcessAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesAssociateEntityExcessAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesAssociateEntityExcessAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS249 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesAssociateEntityExcessAverageValueA

                        #region ExcludedEquityInterestsAverageA
                        _currentXPath.Push("/tns:ExcludedEquityInterestsAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedEquityInterestsAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS250 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExcludedEquityInterestsAverageA
                    } // End of if AustralianPlantationForestryEntity node exists

                    _currentXPath.Pop();
                    #endregion AustralianPlantationForestryEntity

                    #region FinancialEntity
                    _currentXPath.Push("/tns:FinancialEntity");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialEntity")
                    {
                        report.RP_ThinCapitalisation_FinancialEntityCollectionExists = true;
                        report.RP_ThinCapitalisation_FinancialEntityCollectionCount += 1;
                        _found = true;

                        #region ZeroCapitalAverageA
                        _currentXPath.Push("/tns:ZeroCapitalAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ZeroCapitalAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS251 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ZeroCapitalAverageA

                        #region LiabilitiesOnLentAverageA
                        _currentXPath.Push("/tns:LiabilitiesOnLentAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesOnLentAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS252 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesOnLentAverageA
                    } // End of if FinancialEntity node exists

                    _currentXPath.Pop();
                    #endregion FinancialEntity

                    #region OutwardInvestingEntity
                    _currentXPath.Push("/tns:OutwardInvestingEntity");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OutwardInvestingEntity")
                    {
                        report.RP_ThinCapitalisation_OutwardInvestingEntityCollectionExists = true;
                        report.RP_ThinCapitalisation_OutwardInvestingEntityCollectionCount += 1;
                        _found = true;

                        #region ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA
                        _currentXPath.Push("/tns:ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS253 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA

                        #region ControlledForeignEntityDebtAverageValueA
                        _currentXPath.Push("/tns:ControlledForeignEntityDebtAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ControlledForeignEntityDebtAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS254 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ControlledForeignEntityDebtAverageValueA

                        #region ArmsLengthDebtA
                        _currentXPath.Push("/tns:ArmsLengthDebtA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ArmsLengthDebtA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS639 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ArmsLengthDebtA
                    } // End of if OutwardInvestingEntity node exists

                    _currentXPath.Pop();
                    #endregion OutwardInvestingEntity

                    #region InvestingFinancialEntityNonADI
                    _currentXPath.Push("/tns:InvestingFinancialEntityNonADI");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InvestingFinancialEntityNonADI")
                    {
                        report.RP_ThinCapitalisation_InvestingFinancialEntityNonADICollectionExists = true;
                        report.RP_ThinCapitalisation_InvestingFinancialEntityNonADICollectionCount += 1;
                        _found = true;

                        #region AssetsValueAverageA
                        _currentXPath.Push("/tns:AssetsValueAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsValueAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS613 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetsValueAverageA

                        #region AssetRevaluationA
                        _currentXPath.Push("/tns:AssetRevaluationA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetRevaluationA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS614 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssetRevaluationA

                        #region LiabilitiesDebtMaximumAllowableA
                        _currentXPath.Push("/tns:LiabilitiesDebtMaximumAllowableA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtMaximumAllowableA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS615 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtMaximumAllowableA

                        #region LiabilitiesExcessDebtA
                        _currentXPath.Push("/tns:LiabilitiesExcessDebtA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesExcessDebtA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS616 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesExcessDebtA

                        #region LiabilitiesNonDebtLiabilitiesAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesNonDebtLiabilitiesAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesNonDebtLiabilitiesAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS617 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesNonDebtLiabilitiesAverageValueA

                        #region LiabilitiesAssociateEntityDebtAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesAssociateEntityDebtAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesAssociateEntityDebtAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS618 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesAssociateEntityDebtAverageValueA

                        #region AssociateEntityEquityAverageValueA
                        _currentXPath.Push("/tns:AssociateEntityEquityAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssociateEntityEquityAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS619 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AssociateEntityEquityAverageValueA

                        #region LiabilitiesAssociateEntityExcessAverageValueA
                        _currentXPath.Push("/tns:LiabilitiesAssociateEntityExcessAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesAssociateEntityExcessAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS620 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesAssociateEntityExcessAverageValueA

                        #region ExcludedEquityInterestsAverageA
                        _currentXPath.Push("/tns:ExcludedEquityInterestsAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedEquityInterestsAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS621 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ExcludedEquityInterestsAverageA

                        #region ZeroCapitalAverageA
                        _currentXPath.Push("/tns:ZeroCapitalAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ZeroCapitalAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS622 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ZeroCapitalAverageA

                        #region LiabilitiesOnLentAverageA
                        _currentXPath.Push("/tns:LiabilitiesOnLentAverageA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesOnLentAverageA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS623 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesOnLentAverageA
                    } // End of if InvestingFinancialEntityNonADI node exists

                    _currentXPath.Pop();
                    #endregion InvestingFinancialEntityNonADI

                    #region OutwardInvestingFinancialEntity
                    _currentXPath.Push("/tns:OutwardInvestingFinancialEntity");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OutwardInvestingFinancialEntity")
                    {
                        report.RP_ThinCapitalisation_OutwardInvestingFinancialEntityCollectionExists = true;
                        report.RP_ThinCapitalisation_OutwardInvestingFinancialEntityCollectionCount += 1;
                        _found = true;

                        #region ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA
                        _currentXPath.Push("/tns:ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS624 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA

                        #region ControlledForeignEntityDebtAverageValueA
                        _currentXPath.Push("/tns:ControlledForeignEntityDebtAverageValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ControlledForeignEntityDebtAverageValueA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS625 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion ControlledForeignEntityDebtAverageValueA
                    } // End of if OutwardInvestingFinancialEntity node exists

                    _currentXPath.Pop();
                    #endregion OutwardInvestingFinancialEntity

                    #region LiabilitiesGearingDebtCapitalTestI
                    _currentXPath.Push("/tns:LiabilitiesGearingDebtCapitalTestI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesGearingDebtCapitalTestI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS257 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesGearingDebtCapitalTestI

                    #region ADI
                    _currentXPath.Push("/tns:ADI");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ADI")
                    {
                        report.RP_ThinCapitalisation_ADICollectionExists = true;
                        report.RP_ThinCapitalisation_ADICollectionCount += 1;
                        _found = true;

                        #region WorldwideGroupCapitalRatioF
                        _currentXPath.Push("/tns:WorldwideGroupCapitalRatioF");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "WorldwideGroupCapitalRatioF")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS258 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion WorldwideGroupCapitalRatioF

                        #region WorldwideCapitalA
                        _currentXPath.Push("/tns:WorldwideCapitalA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "WorldwideCapitalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS259 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion WorldwideCapitalA
                    } // End of if ADI node exists

                    _currentXPath.Pop();
                    #endregion ADI

                    #region NonADI
                    _currentXPath.Push("/tns:NonADI");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "NonADI")
                    {
                        report.RP_ThinCapitalisation_NonADICollectionExists = true;
                        report.RP_ThinCapitalisation_NonADICollectionCount += 1;
                        _found = true;

                        #region WorldwideDebtTotalA
                        _currentXPath.Push("/tns:WorldwideDebtTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "WorldwideDebtTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS260 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion WorldwideDebtTotalA

                        #region LiabilitiesWorldwideEquityA
                        _currentXPath.Push("/tns:LiabilitiesWorldwideEquityA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesWorldwideEquityA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS261 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesWorldwideEquityA

                        #region WorldwideGearingDebtA
                        _currentXPath.Push("/tns:WorldwideGearingDebtA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "WorldwideGearingDebtA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS262 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion WorldwideGearingDebtA

                        #region StatementWorldwideAssetA
                        _currentXPath.Push("/tns:StatementWorldwideAssetA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "StatementWorldwideAssetA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS323 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion StatementWorldwideAssetA

                        #region AverageAustralianAssetA
                        _currentXPath.Push("/tns:AverageAustralianAssetA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "AverageAustralianAssetA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.IDS324 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AverageAustralianAssetA
                    } // End of if NonADI node exists

                    _currentXPath.Pop();
                    #endregion NonADI
                } // End of if ThinCapitalisation node exists

                _currentXPath.Pop();
                #endregion ThinCapitalisation

                #region FinancialServicesForeignBankOrQualifyingFinancialEntityI
                _currentXPath.Push("/tns:FinancialServicesForeignBankOrQualifyingFinancialEntityI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesForeignBankOrQualifyingFinancialEntityI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS263 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion FinancialServicesForeignBankOrQualifyingFinancialEntityI

                #region NotionalAmount
                _currentXPath.Push("/tns:NotionalAmount");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "NotionalAmount")
                {
                    report.RP_NotionalAmountCollectionExists = true;
                    report.RP_NotionalAmountCollectionCount += 1;
                    _found = true;

                    #region FinancialServicesNotionalAverageBorrowingsA
                    _currentXPath.Push("/tns:FinancialServicesNotionalAverageBorrowingsA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalAverageBorrowingsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS264 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalAverageBorrowingsA

                    #region FinancialServicesNotionalAverageBorrowingsC
                    _currentXPath.Push("/tns:FinancialServicesNotionalAverageBorrowingsC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalAverageBorrowingsC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS407 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalAverageBorrowingsC

                    #region FinancialServicesNonOffshoreBankingUnitNotionalInterestA
                    _currentXPath.Push("/tns:FinancialServicesNonOffshoreBankingUnitNotionalInterestA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNonOffshoreBankingUnitNotionalInterestA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS265 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNonOffshoreBankingUnitNotionalInterestA

                    #region FinancialServicesNotionalInterestTaxWithholdingA
                    _currentXPath.Push("/tns:FinancialServicesNotionalInterestTaxWithholdingA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalInterestTaxWithholdingA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS267 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalInterestTaxWithholdingA
                } // End of if NotionalAmount node exists

                _currentXPath.Pop();
                #endregion NotionalAmount

                #region NotionalDerivativeAndForeignExchange
                _currentXPath.Push("/tns:NotionalDerivativeAndForeignExchange");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "NotionalDerivativeAndForeignExchange")
                {
                    report.RP_NotionalDerivativeAndForeignExchangeCollectionExists = true;
                    report.RP_NotionalDerivativeAndForeignExchangeCollectionCount += 1;
                    _found = true;

                    #region FinancialServicesNotionalDerivativePaidA
                    _currentXPath.Push("/tns:FinancialServicesNotionalDerivativePaidA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalDerivativePaidA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS408 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalDerivativePaidA

                    #region FinancialServicesNotionalDerivativeReceivedA
                    _currentXPath.Push("/tns:FinancialServicesNotionalDerivativeReceivedA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalDerivativeReceivedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS409 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalDerivativeReceivedA

                    #region FinancialServicesNotionalForeignExchangeTransactionPaidA
                    _currentXPath.Push("/tns:FinancialServicesNotionalForeignExchangeTransactionPaidA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalForeignExchangeTransactionPaidA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS410 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalForeignExchangeTransactionPaidA

                    #region FinancialServicesNotionalForeignExchangeTransactionReceivedA
                    _currentXPath.Push("/tns:FinancialServicesNotionalForeignExchangeTransactionReceivedA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalForeignExchangeTransactionReceivedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS411 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesNotionalForeignExchangeTransactionReceivedA
                } // End of if NotionalDerivativeAndForeignExchange node exists

                _currentXPath.Pop();
                #endregion NotionalDerivativeAndForeignExchange

                #region FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI
                _currentXPath.Push("/tns:FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS412 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI

                #region NotionalAmountElectedOut
                _currentXPath.Push("/tns:NotionalAmountElectedOut");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "NotionalAmountElectedOut")
                {
                    report.RP_NotionalAmountElectedOutCollectionExists = true;
                    report.RP_NotionalAmountElectedOutCollectionCount += 1;
                    _found = true;

                    #region FinancialServicesElectedOutNotionalAverageBorrowingsA
                    _currentXPath.Push("/tns:FinancialServicesElectedOutNotionalAverageBorrowingsA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesElectedOutNotionalAverageBorrowingsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS413 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesElectedOutNotionalAverageBorrowingsA

                    #region FinancialServicesElectedOutNotionalAverageBorrowingsC
                    _currentXPath.Push("/tns:FinancialServicesElectedOutNotionalAverageBorrowingsC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesElectedOutNotionalAverageBorrowingsC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.IDS414 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesElectedOutNotionalAverageBorrowingsC

                    #region FinancialServicesElectedOutNonOffshoreBankingUnitNotionalInterestA
                    _currentXPath.Push("/tns:FinancialServicesElectedOutNonOffshoreBankingUnitNotionalInterestA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesElectedOutNonOffshoreBankingUnitNotionalInterestA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS415 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesElectedOutNonOffshoreBankingUnitNotionalInterestA

                    #region FinancialServicesElectedOutNotionalInterestTaxWithholdingA
                    _currentXPath.Push("/tns:FinancialServicesElectedOutNotionalInterestTaxWithholdingA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesElectedOutNotionalInterestTaxWithholdingA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS417 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion FinancialServicesElectedOutNotionalInterestTaxWithholdingA
                } // End of if NotionalAmountElectedOut node exists

                _currentXPath.Pop();
                #endregion NotionalAmountElectedOut

                #region FinancialServicesNotionalPaymentDeniedI
                _currentXPath.Push("/tns:FinancialServicesNotionalPaymentDeniedI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialServicesNotionalPaymentDeniedI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS455 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion FinancialServicesNotionalPaymentDeniedI

                #region FinancialServicesNotionalPaymentDeniedA
                _currentXPath.Push("/tns:FinancialServicesNotionalPaymentDeniedA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialServicesNotionalPaymentDeniedA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS456 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion FinancialServicesNotionalPaymentDeniedA

                #region InternationalDealingsConduitForeignIncomeI
                _currentXPath.Push("/tns:InternationalDealingsConduitForeignIncomeI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsConduitForeignIncomeI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS309 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsConduitForeignIncomeI

                #region ConduitForeignIncome
                _currentXPath.Push("/tns:ConduitForeignIncome");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ConduitForeignIncome")
                {
                    report.RP_ConduitForeignIncomeCollectionExists = true;
                    report.RP_ConduitForeignIncomeCollectionCount += 1;
                    _found = true;

                    #region InternationalDealingsBalanceA
                    _currentXPath.Push("/tns:InternationalDealingsBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsBalanceA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS310 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsBalanceA

                    #region InternationalDealingsDistributedA
                    _currentXPath.Push("/tns:InternationalDealingsDistributedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDistributedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS311 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsDistributedA
                } // End of if ConduitForeignIncome node exists

                _currentXPath.Pop();
                #endregion ConduitForeignIncome

                #region InternationalDealingsUnfrankedNonPortfolioDividendAccountI
                _currentXPath.Push("/tns:InternationalDealingsUnfrankedNonPortfolioDividendAccountI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsUnfrankedNonPortfolioDividendAccountI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS312 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsUnfrankedNonPortfolioDividendAccountI

                #region Unfranked
                _currentXPath.Push("/tns:Unfranked");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Unfranked")
                {
                    report.RP_UnfrankedCollectionExists = true;
                    report.RP_UnfrankedCollectionCount += 1;
                    _found = true;

                    #region InternationalDealingsNonPortfolioDividendAccountA
                    _currentXPath.Push("/tns:InternationalDealingsNonPortfolioDividendAccountA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsNonPortfolioDividendAccountA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS313 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsNonPortfolioDividendAccountA

                    #region InternationalDealingsNonPortfolioAccountConduitForeignIncomeIncludedI
                    _currentXPath.Push("/tns:InternationalDealingsNonPortfolioAccountConduitForeignIncomeIncludedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsNonPortfolioAccountConduitForeignIncomeIncludedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS314 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsNonPortfolioAccountConduitForeignIncomeIncludedI
                } // End of if Unfranked node exists

                _currentXPath.Pop();
                #endregion Unfranked

                #region InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidI
                _currentXPath.Push("/tns:InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS315 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidI

                #region InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA
                _currentXPath.Push("/tns:InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS316 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA

                #region InternationalDealingsDoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountryC
                _currentXPath.Push("/tns:InternationalDealingsDoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountryC");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountryC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IDS317 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InternationalDealingsDoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountryC

                #region FInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA
                _currentXPath.Push("/tns:FInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS318 = currentDecimalValue;
                            }
                            report.RP_128FCollectionExists = true;
                            report.RP_128FCollectionCount += 1;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion FInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA

                #region FAInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA
                _currentXPath.Push("/tns:FAInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FAInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS319 = currentDecimalValue;
                            }
                            report.RP_128FACollectionExists = true;
                            report.RP_128FACollectionCount += 1;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion FAInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA

                #region MiscellaneousHybridMismatchI
                _currentXPath.Push("/tns:MiscellaneousHybridMismatchI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS481 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousHybridMismatchI

                #region MiscellaneousHybridMismatchDeductionNonInclusionI
                _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionNonInclusionI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousHybridMismatchDeductionNonInclusionI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS458 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousHybridMismatchDeductionNonInclusionI

                #region MiscellaneousHybridMismatchPaymentsA
                _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentsA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousHybridMismatchPaymentsA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS482 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousHybridMismatchPaymentsA

                #region MaterialArrangements

                _currentXPath.Push("/tns:MaterialArrangementsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MaterialArrangementsCollection")
                {
                    report.RP_MaterialArrangementsCollection = new List<IDS2025.RP_MaterialArrangements>();
                    report.RP_MaterialArrangementsCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "MaterialArrangements")
                    {
                        _found = true;

                        IDS2025.RP_MaterialArrangements materialArrangements = new IDS2025.RP_MaterialArrangements();
                        report.RP_MaterialArrangementsCollection.Add(materialArrangements);
                        report.RP_MaterialArrangementsCollectionCount += 1;
                        materialArrangements.OccurrenceIndex = report.RP_MaterialArrangementsCollectionCount;

                        _currentXPath.Push("/tns:MaterialArrangements[" + report.RP_MaterialArrangementsCollectionCount + "]");

                        #region MiscellaneousHybridMismatchArrangementC
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchArrangementC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchArrangementC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    materialArrangements.IDS460 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchArrangementC

                        #region OrganisationDetailsInternationalRelatedPartyCountryC
                        _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    materialArrangements.IDS461 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                        #region MiscellaneousHybridMismatchPaymentsA
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchPaymentsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        materialArrangements.IDS462 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchPaymentsA

                        #region MiscellaneousHybridMismatchDeductionsDeniedOrIncomeInclusionA
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionsDeniedOrIncomeInclusionA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchDeductionsDeniedOrIncomeInclusionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        materialArrangements.IDS463 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchDeductionsDeniedOrIncomeInclusionA

                        #region MiscellaneousHybridMismatchDifferenceReasonC
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchDifferenceReasonC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousHybridMismatchDifferenceReasonC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    materialArrangements.IDS464 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchDifferenceReasonC

                        #region MiscellaneousHybridMismatchPaymentRecipientI
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentRecipientI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchPaymentRecipientI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        materialArrangements.IDS465 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchPaymentRecipientI
                        _currentXPath.Pop();
                    } // End of while loop for MaterialArrangements
                } // End of if MaterialArrangementsCollection node exists

                _currentXPath.Pop();
                #endregion MaterialArrangements

                #region MiscellaneousHybridMismatchPaymentI
                _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousHybridMismatchPaymentI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS501 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousHybridMismatchPaymentI

                #region StructuredArrangement
                _currentXPath.Push("/tns:StructuredArrangement");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "StructuredArrangement")
                {
                    report.RP_StructuredArrangementCollectionExists = true;
                    report.RP_StructuredArrangementCollectionCount += 1;
                    _found = true;

                    #region MiscellaneousHybridMismatchPaymentsA
                    _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentsA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchPaymentsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS502 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousHybridMismatchPaymentsA

                    #region MiscellaneousHybridMismatchA
                    _currentXPath.Push("/tns:MiscellaneousHybridMismatchA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS503 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousHybridMismatchA

                    #region MiscellaneousHybridMismatchDeductionDisallowedA
                    _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionDisallowedA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchDeductionDisallowedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS504 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousHybridMismatchDeductionDisallowedA
                } // End of if StructuredArrangement node exists

                _currentXPath.Pop();
                #endregion StructuredArrangement

                #region MiscellaneousOffshoreHybridMismatchI
                _currentXPath.Push("/tns:MiscellaneousOffshoreHybridMismatchI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousOffshoreHybridMismatchI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS505 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousOffshoreHybridMismatchI

                #region NonStructuredArrangement
                _currentXPath.Push("/tns:NonStructuredArrangement");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "NonStructuredArrangement")
                {
                    report.RP_NonStructuredArrangementCollectionExists = true;
                    report.RP_NonStructuredArrangementCollectionCount += 1;
                    _found = true;

                    #region MiscellaneousHybridMismatchA
                    _currentXPath.Push("/tns:MiscellaneousHybridMismatchA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS506 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousHybridMismatchA

                    #region MiscellaneousHybridMismatchDeductionDisallowedA
                    _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionDisallowedA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchDeductionDisallowedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS507 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion MiscellaneousHybridMismatchDeductionDisallowedA
                } // End of if NonStructuredArrangement node exists

                _currentXPath.Pop();
                #endregion NonStructuredArrangement

                #region MaterialOffshoreHybridMismatch

                _currentXPath.Push("/tns:MaterialOffshoreHybridMismatchCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MaterialOffshoreHybridMismatchCollection")
                {
                    report.RP_MaterialOffshoreHybridMismatchCollection = new List<IDS2025.RP_MaterialOffshoreHybridMismatch>();
                    report.RP_MaterialOffshoreHybridMismatchCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "MaterialOffshoreHybridMismatch")
                    {
                        _found = true;

                        IDS2025.RP_MaterialOffshoreHybridMismatch materialOffshoreHybridMismatch = new IDS2025.RP_MaterialOffshoreHybridMismatch();
                        report.RP_MaterialOffshoreHybridMismatchCollection.Add(materialOffshoreHybridMismatch);
                        report.RP_MaterialOffshoreHybridMismatchCollectionCount += 1;
                        materialOffshoreHybridMismatch.OccurrenceIndex = report.RP_MaterialOffshoreHybridMismatchCollectionCount;

                        _currentXPath.Push("/tns:MaterialOffshoreHybridMismatch[" + report.RP_MaterialOffshoreHybridMismatchCollectionCount + "]");

                        #region MiscellaneousHybridMismatchArrangementC
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchArrangementC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchArrangementC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    materialOffshoreHybridMismatch.IDS508 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchArrangementC

                        #region MiscellaneousA
                        _currentXPath.Push("/tns:MiscellaneousA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        materialOffshoreHybridMismatch.IDS509 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousA

                        #region MiscellaneousHybridMismatchPaymentsC
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentsC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchPaymentsC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    materialOffshoreHybridMismatch.IDS510 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchPaymentsC

                        #region MiscellaneousHybridMismatchPaymentsA
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchPaymentsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        materialOffshoreHybridMismatch.IDS511 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchPaymentsA

                        #region MiscellaneousHybridMismatchDeductionDisallowedA
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionDisallowedA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchDeductionDisallowedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        materialOffshoreHybridMismatch.IDS512 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchDeductionDisallowedA

                        #region MiscellaneousHybridMismatchDeductionDisallowedDifferenceReasonC
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionDisallowedDifferenceReasonC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchDeductionDisallowedDifferenceReasonC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    materialOffshoreHybridMismatch.IDS513 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchDeductionDisallowedDifferenceReasonC
                        _currentXPath.Pop();
                    } // End of while loop for MaterialOffshoreHybridMismatch
                } // End of if MaterialOffshoreHybridMismatchCollection node exists

                _currentXPath.Pop();
                #endregion MaterialOffshoreHybridMismatch

                #region MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI
                _currentXPath.Push("/tns:MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS468 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI

                #region MiscellaneousHybridMismatchDeductionsA
                _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionsA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousHybridMismatchDeductionsA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS483 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousHybridMismatchDeductionsA

                #region HybridPayerMismatch

                _currentXPath.Push("/tns:HybridPayerMismatchCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "HybridPayerMismatchCollection")
                {
                    report.RP_HybridPayerMismatchCollection = new List<IDS2025.RP_HybridPayerMismatch>();
                    report.RP_HybridPayerMismatchCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "HybridPayerMismatch")
                    {
                        _found = true;

                        IDS2025.RP_HybridPayerMismatch hybridPayerMismatch = new IDS2025.RP_HybridPayerMismatch();
                        report.RP_HybridPayerMismatchCollection.Add(hybridPayerMismatch);
                        report.RP_HybridPayerMismatchCollectionCount += 1;
                        hybridPayerMismatch.OccurrenceIndex = report.RP_HybridPayerMismatchCollectionCount;

                        _currentXPath.Push("/tns:HybridPayerMismatch[" + report.RP_HybridPayerMismatchCollectionCount + "]");

                        #region OrganisationDetailsInternationalRelatedPartyCountryC
                        _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    hybridPayerMismatch.IDS470 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                        #region MiscellaneousHybridMismatchPaymentsA
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchPaymentsA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchPaymentsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        hybridPayerMismatch.IDS471 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchPaymentsA

                        #region MiscellaneousHybridMismatchDeductionsDeniedA
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchDeductionsDeniedA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridMismatchDeductionsDeniedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        hybridPayerMismatch.IDS472 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchDeductionsDeniedA

                        #region MiscellaneousHybridMismatchExceptionReasonC
                        _currentXPath.Push("/tns:MiscellaneousHybridMismatchExceptionReasonC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousHybridMismatchExceptionReasonC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    hybridPayerMismatch.IDS473 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion MiscellaneousHybridMismatchExceptionReasonC
                        _currentXPath.Pop();
                    } // End of while loop for HybridPayerMismatch
                } // End of if HybridPayerMismatchCollection node exists

                _currentXPath.Pop();
                #endregion HybridPayerMismatch

                #region MiscellaneousHybridArrangementRestructureReplacementI
                _currentXPath.Push("/tns:MiscellaneousHybridArrangementRestructureReplacementI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "MiscellaneousHybridArrangementRestructureReplacementI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS474 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion MiscellaneousHybridArrangementRestructureReplacementI

                #region HybridArrangementRestructure

                _currentXPath.Push("/tns:HybridArrangementRestructureCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "HybridArrangementRestructureCollection")
                {
                    report.RP_HybridArrangementRestructureCollection = new List<IDS2025.RP_HybridArrangementRestructure>();
                    report.RP_HybridArrangementRestructureCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "HybridArrangementRestructure")
                    {
                        _found = true;

                        IDS2025.RP_HybridArrangementRestructure hybridArrangementRestructure = new IDS2025.RP_HybridArrangementRestructure();
                        report.RP_HybridArrangementRestructureCollection.Add(hybridArrangementRestructure);
                        report.RP_HybridArrangementRestructureCollectionCount += 1;
                        hybridArrangementRestructure.OccurrenceIndex = report.RP_HybridArrangementRestructureCollectionCount;

                        _currentXPath.Push("/tns:HybridArrangementRestructure[" + report.RP_HybridArrangementRestructureCollectionCount + "]");

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    hybridArrangementRestructure.IDS475 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT

                        #region InternationalDealingsHybridEventAnalysisI
                        _currentXPath.Push("/tns:InternationalDealingsHybridEventAnalysisI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsHybridEventAnalysisI")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        hybridArrangementRestructure.IDS476 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsHybridEventAnalysisI
                        _currentXPath.Pop();
                    } // End of while loop for HybridArrangementRestructure
                } // End of if HybridArrangementRestructureCollection node exists

                _currentXPath.Pop();
                #endregion HybridArrangementRestructure

                #region IncomeForeignEquityDistributionForeignTaxDeductionI
                _currentXPath.Push("/tns:IncomeForeignEquityDistributionForeignTaxDeductionI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignEquityDistributionForeignTaxDeductionI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS477 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeForeignEquityDistributionForeignTaxDeductionI

                #region IncomeForeignDeductionNotNonAssessableNonExemptA
                _currentXPath.Push("/tns:IncomeForeignDeductionNotNonAssessableNonExemptA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignDeductionNotNonAssessableNonExemptA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS478 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeForeignDeductionNotNonAssessableNonExemptA

                #region IncomeDerivedBranchHybridMismatchIncomeI
                _currentXPath.Push("/tns:IncomeDerivedBranchHybridMismatchIncomeI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeDerivedBranchHybridMismatchIncomeI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS479 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeDerivedBranchHybridMismatchIncomeI

                #region IncomeForeignHybridIncomeNotNonAssessableNonExemptA
                _currentXPath.Push("/tns:IncomeForeignHybridIncomeNotNonAssessableNonExemptA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignHybridIncomeNotNonAssessableNonExemptA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS480 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion IncomeForeignHybridIncomeNotNonAssessableNonExemptA

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS640 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IDS641 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC

                #region LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS628 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI

                #region RestructuringEventCurrentOrPriorIncomeYear

                _currentXPath.Push("/tns:RestructuringEventCurrentOrPriorIncomeYearCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RestructuringEventCurrentOrPriorIncomeYearCollection")
                {
                    report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection = new List<IDS2025.RP_RestructuringEventCurrentOrPriorIncomeYear>();
                    report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "RestructuringEventCurrentOrPriorIncomeYear")
                    {
                        _found = true;

                        IDS2025.RP_RestructuringEventCurrentOrPriorIncomeYear restructuringEventCurrentOrPriorIncomeYear = new IDS2025.RP_RestructuringEventCurrentOrPriorIncomeYear();
                        report.RP_RestructuringEventCurrentOrPriorIncomeYearCollection.Add(restructuringEventCurrentOrPriorIncomeYear);
                        report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionCount += 1;
                        restructuringEventCurrentOrPriorIncomeYear.OccurrenceIndex = report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionCount;

                        _currentXPath.Push("/tns:RestructuringEventCurrentOrPriorIncomeYear[" + report.RP_RestructuringEventCurrentOrPriorIncomeYearCollectionCount + "]");

                        #region LiabilitiesThinCapitalisationRestructuringEventsDescriptionT
                        _currentXPath.Push("/tns:LiabilitiesThinCapitalisationRestructuringEventsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesThinCapitalisationRestructuringEventsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEventCurrentOrPriorIncomeYear.IDS629 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesThinCapitalisationRestructuringEventsDescriptionT

                        #region LiabilitiesDebtDeductionCreationRulesTaxableIncomeChangesAfterRestructureC
                        _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesTaxableIncomeChangesAfterRestructureC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesTaxableIncomeChangesAfterRestructureC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    restructuringEventCurrentOrPriorIncomeYear.IDS642 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtDeductionCreationRulesTaxableIncomeChangesAfterRestructureC
                        _currentXPath.Pop();
                    } // End of while loop for RestructuringEventCurrentOrPriorIncomeYear
                } // End of if RestructuringEventCurrentOrPriorIncomeYearCollection node exists

                _currentXPath.Pop();
                #endregion RestructuringEventCurrentOrPriorIncomeYear

                #region LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairPriorIncomeYearI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairPriorIncomeYearI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairPriorIncomeYearI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS643 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairPriorIncomeYearI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS644 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS645 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS646 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA

                #region PriorIncomeYearTopMaterialArrangements

                _currentXPath.Push("/tns:PriorIncomeYearTopMaterialArrangementsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PriorIncomeYearTopMaterialArrangementsCollection")
                {
                    report.RP_PriorIncomeYearTopMaterialArrangementsCollection = new List<IDS2025.RP_PriorIncomeYearTopMaterialArrangements>();
                    report.RP_PriorIncomeYearTopMaterialArrangementsCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "PriorIncomeYearTopMaterialArrangements")
                    {
                        _found = true;

                        IDS2025.RP_PriorIncomeYearTopMaterialArrangements priorIncomeYearTopMaterialArrangements = new IDS2025.RP_PriorIncomeYearTopMaterialArrangements();
                        report.RP_PriorIncomeYearTopMaterialArrangementsCollection.Add(priorIncomeYearTopMaterialArrangements);
                        report.RP_PriorIncomeYearTopMaterialArrangementsCollectionCount += 1;
                        priorIncomeYearTopMaterialArrangements.OccurrenceIndex = report.RP_PriorIncomeYearTopMaterialArrangementsCollectionCount;

                        _currentXPath.Push("/tns:PriorIncomeYearTopMaterialArrangements[" + report.RP_PriorIncomeYearTopMaterialArrangementsCollectionCount + "]");

                        #region InteractionTransactionA
                        _currentXPath.Push("/tns:InteractionTransactionA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InteractionTransactionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        priorIncomeYearTopMaterialArrangements.IDS647 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InteractionTransactionA

                        #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA
                        _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        priorIncomeYearTopMaterialArrangements.IDS648 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    priorIncomeYearTopMaterialArrangements.IDS649 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT
                        _currentXPath.Pop();
                    } // End of while loop for PriorIncomeYearTopMaterialArrangements
                } // End of if PriorIncomeYearTopMaterialArrangementsCollection node exists

                _currentXPath.Pop();
                #endregion PriorIncomeYearTopMaterialArrangements

                #region LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS650 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI

                #region LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS651 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS652 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS653 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS654 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA

                #region PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement

                _currentXPath.Push("/tns:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection")
                {
                    report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection = new List<IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement>();
                    report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement")
                    {
                        _found = true;

                        IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement = new IDS2025.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement();
                        report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection.Add(priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement);
                        report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount += 1;
                        priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.OccurrenceIndex = report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount;

                        _currentXPath.Push("/tns:PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement[" + report.RP_PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount + "]");

                        #region InteractionTransactionA
                        _currentXPath.Push("/tns:InteractionTransactionA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InteractionTransactionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS655 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InteractionTransactionA

                        #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA
                        _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS656 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    priorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS657 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT
                        _currentXPath.Pop();
                    } // End of while loop for PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
                } // End of if PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection node exists

                _currentXPath.Pop();
                #endregion PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement

                #region LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS658 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI

                #region InteractionTransactionA
                _currentXPath.Push("/tns:InteractionTransactionA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InteractionTransactionA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS659 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion InteractionTransactionA

                #region LiabilitiesDebtDeductionCreationRulesAssetExceptionI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesAssetExceptionI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesAssetExceptionI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS660 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesAssetExceptionI

                #region LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.IDS661 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS662 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS663 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS664 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA

                #region Arrangements

                _currentXPath.Push("/tns:ArrangementsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ArrangementsCollection")
                {
                    report.RP_ArrangementsCollection = new List<IDS2025.RP_Arrangements>();
                    report.RP_ArrangementsCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "Arrangements")
                    {
                        _found = true;

                        IDS2025.RP_Arrangements arrangements = new IDS2025.RP_Arrangements();
                        report.RP_ArrangementsCollection.Add(arrangements);
                        report.RP_ArrangementsCollectionCount += 1;
                        arrangements.OccurrenceIndex = report.RP_ArrangementsCollectionCount;

                        _currentXPath.Push("/tns:Arrangements[" + report.RP_ArrangementsCollectionCount + "]");

                        #region InteractionTransactionA
                        _currentXPath.Push("/tns:InteractionTransactionA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InteractionTransactionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        arrangements.IDS665 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InteractionTransactionA

                        #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA
                        _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        arrangements.IDS666 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA

                        #region LiabilitiesDebtDeductionCreationRulesAcquisitionTypeC
                        _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesAcquisitionTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesAcquisitionTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    arrangements.IDS667 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtDeductionCreationRulesAcquisitionTypeC

                        #region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    arrangements.IDS668 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InternationalDealingsDescriptionT
                        _currentXPath.Pop();
                    } // End of while loop for Arrangements
                } // End of if ArrangementsCollection node exists

                _currentXPath.Pop();
                #endregion Arrangements

                #region CurrentIncomeYearPaymentsorDistributionsSelection
                _currentXPath.Push("/tns:CurrentIncomeYearPaymentsorDistributionsSelection");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CurrentIncomeYearPaymentsorDistributionsSelection")
                {
                    report.RP_CurrentIncomeYearPaymentsorDistributionsSelectionCollectionExists = true;
                    report.RP_CurrentIncomeYearPaymentsorDistributionsSelectionCollectionCount += 1;
                    _found = true;

                    #region LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI
                    _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS669 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI

                    #region PaymentRecordA
                    _currentXPath.Push("/tns:PaymentRecordA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PaymentRecordA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS670 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion PaymentRecordA

                    #region LiabilitiesDebtDeductionCreationRulesFinancialArrangementFundingI
                    _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesFinancialArrangementFundingI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesFinancialArrangementFundingI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS671 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDebtDeductionCreationRulesFinancialArrangementFundingI

                    #region LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI
                    _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS672 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI

                    #region LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI
                    _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.IDS673 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI

                    #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA
                    _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.IDS674 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA

                    #region CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement

                    _currentXPath.Push("/tns:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection");
                    // 4. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection")
                    {
                        report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection = new List<IDS2025.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement>();
                        report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionExists = true;

                        _found = true;

                        while (ReadToNextElement(reader, false) && reader.LocalName == "CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement")
                        {
                            _found = true;

                            IDS2025.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement = new IDS2025.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement();
                            report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection.Add(currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement);
                            report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount += 1;
                            currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.OccurrenceIndex = report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount;

                            _currentXPath.Push("/tns:CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement[" + report.RP_CurrentIncomeYearPaymentsorDistributionsSelection_CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollectionCount + "]");

                            #region PaymentRecordA
                            _currentXPath.Push("/tns:PaymentRecordA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PaymentRecordA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS675 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion PaymentRecordA

                            #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA
                            _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS676 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA

                            #region LiabilitiesDebtDeductionCreationRulesPaymentOrDistributionC
                            _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesPaymentOrDistributionC");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesPaymentOrDistributionC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS677 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion LiabilitiesDebtDeductionCreationRulesPaymentOrDistributionC

                            #region InternationalDealingsDescriptionT
                            _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDescriptionT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement.IDS678 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion InternationalDealingsDescriptionT
                            _currentXPath.Pop();
                        } // End of while loop for CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
                    } // End of if CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection node exists

                    _currentXPath.Pop();
                    #endregion CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement
                } // End of if CurrentIncomeYearPaymentsorDistributionsSelection node exists

                _currentXPath.Pop();
                #endregion CurrentIncomeYearPaymentsorDistributionsSelection

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.IDS679 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI

                #region LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA
                _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.IDS680 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA

                #region AssociatePairs

                _currentXPath.Push("/tns:AssociatePairsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AssociatePairsCollection")
                {
                    report.RP_AssociatePairsCollection = new List<IDS2025.RP_AssociatePairs>();
                    report.RP_AssociatePairsCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "AssociatePairs")
                    {
                        _found = true;

                        IDS2025.RP_AssociatePairs associatePairs = new IDS2025.RP_AssociatePairs();
                        report.RP_AssociatePairsCollection.Add(associatePairs);
                        report.RP_AssociatePairsCollectionCount += 1;
                        associatePairs.OccurrenceIndex = report.RP_AssociatePairsCollectionCount;

                        _currentXPath.Push("/tns:AssociatePairs[" + report.RP_AssociatePairsCollectionCount + "]");

                        #region LiabilitiesDebtDeductionCreationRulesDeductionDebtRevelantEntityAssociatePairT
                        _currentXPath.Push("/tns:LiabilitiesDebtDeductionCreationRulesDeductionDebtRevelantEntityAssociatePairT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesDebtDeductionCreationRulesDeductionDebtRevelantEntityAssociatePairT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    associatePairs.IDS681 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LiabilitiesDebtDeductionCreationRulesDeductionDebtRevelantEntityAssociatePairT

                        #region OrganisationDetailsInternationalRelatedPartyCountryC
                        _currentXPath.Push("/tns:OrganisationDetailsInternationalRelatedPartyCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsInternationalRelatedPartyCountryC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    associatePairs.IDS682 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationDetailsInternationalRelatedPartyCountryC

                        #region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    associatePairs.IDS683 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AustralianBusinessNumberId

                        #region InteractionTransactionA
                        _currentXPath.Push("/tns:InteractionTransactionA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InteractionTransactionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        associatePairs.IDS684 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion InteractionTransactionA
                        _currentXPath.Pop();
                    } // End of while loop for AssociatePairs
                } // End of if AssociatePairsCollection node exists

                _currentXPath.Pop();
                #endregion AssociatePairs
            } // End of if RP node exists

            _currentXPath.Pop();
            #endregion RP

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "RP#TaxFileNumberId,RP#InternationalDealingsCountryByCountryReportApplicabilityI,RP#InternationalDealingsLocalFilePartAVoluntaryLodgmentI,RP#InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI,RP#InternationalDealingsSmallBusinessConcessionsEligibilityI,RP#InternationalDealingsRelatedPartiesTransactionsI,ForeignCountry#OrganisationDetailsInternationalRelatedPartyCountryC,ForeignDealings#OrganisationDetailsInternationalRelatedPartyCountryActivityC,ForeignDealings#ExpenseTotalA,ForeignDealings#RevenueTotalA,ForeignDealings,ForeignDealingsCollection,ForeignCountry#ExpenseOtherA,ForeignCountry#RevenueOtherA,ForeignCountry,ForeignCountryCollection,RP#InternationalDealingsSpecifiedCountriesTransactionsI,SpecifiedCountry#OrganisationDetailsInternationalRelatedPartyCountryC,SpecifiedDealings#OrganisationDetailsInternationalRelatedPartyCountryActivityC,SpecifiedDealings#ExpenseTotalA,SpecifiedDealings#RevenueTotalA,SpecifiedDealings,SpecifiedDealingsCollection,SpecifiedCountry#ExpenseOtherA,SpecifiedCountry#RevenueOtherA,SpecifiedCountry,SpecifiedCountryCollection,RP#InternationalDealingsTransactionsTangiblePropertyRevenueI,TangiblePropertyRevenue#ExpenseTotalA,TangiblePropertyRevenue#TotalA,TangiblePropertyRevenue#MiscellaneousArmsLengthPricingMethodC,TangiblePropertyRevenue#MiscellaneousPercentageOfDealingsWithDocumentationC,TangiblePropertyRevenue,RP#InternationalDealingsTransactionsRoyaltiesLicenceFeesI,Royalties#ExpenseDeductionsTotalA,Royalties#IncomeIncludedInAssessableTotalA,Royalties#MiscellaneousArmsLengthPricingMethodC,Royalties#MiscellaneousPercentageOfDealingsWithDocumentationC,Royalties,LicenceFees#ExpenseDeductionsTotalA,LicenceFees#IncomeIncludedInAssessableTotalA,LicenceFees#MiscellaneousArmsLengthPricingMethodC,LicenceFees#MiscellaneousPercentageOfDealingsWithDocumentationC,LicenceFees,RP#InternationalDealingsTransactionsRentLeasingI,RentOrLeasing#ExpenseTotalA,RentOrLeasing#RevenueTotalA,RentOrLeasing#MiscellaneousArmsLengthPricingMethodC,RentOrLeasing#MiscellaneousPercentageOfDealingsWithDocumentationC,RentOrLeasing,RP#InternationalDealingsTransactionsServiceArrangementsI,TreasuryServices#ExpenseTotalA,TreasuryServices#RevenueTotalA,TreasuryServices#MiscellaneousArmsLengthPricingMethodC,TreasuryServices#MiscellaneousPercentageOfDealingsWithDocumentationC,TreasuryServices,ManagementAndAdministrationServices#ExpenseTotalA,ManagementAndAdministrationServices#RevenueTotalA,ManagementAndAdministrationServices#MiscellaneousArmsLengthPricingMethodC,ManagementAndAdministrationServices#MiscellaneousPercentageOfDealingsWithDocumentationC,ManagementAndAdministrationServices,InsuranceServices#ExpenseTotalA,InsuranceServices#RevenueTotalA,InsuranceServices#MiscellaneousArmsLengthPricingMethodC,InsuranceServices#MiscellaneousPercentageOfDealingsWithDocumentationC,InsuranceServices,ReinsuranceServices#ExpenseTotalA,ReinsuranceServices#RevenueTotalA,ReinsuranceServices#MiscellaneousArmsLengthPricingMethodC,ReinsuranceServices#MiscellaneousPercentageOfDealingsWithDocumentationC,ReinsuranceServices,RAndDServices#ExpenseTotalA,RAndDServices#RevenueTotalA,RAndDServices#MiscellaneousArmsLengthPricingMethodC,RAndDServices#MiscellaneousPercentageOfDealingsWithDocumentationC,RAndDServices,SalesAndMarketingServices#ExpenseTotalA,SalesAndMarketingServices#RevenueTotalA,SalesAndMarketingServices#MiscellaneousArmsLengthPricingMethodC,SalesAndMarketingServices#MiscellaneousPercentageOfDealingsWithDocumentationC,SalesAndMarketingServices,SoftwareAndITServices#ExpenseTotalA,SoftwareAndITServices#RevenueTotalA,SoftwareAndITServices#MiscellaneousArmsLengthPricingMethodC,SoftwareAndITServices#MiscellaneousPercentageOfDealingsWithDocumentationC,SoftwareAndITServices,TechnicalServices#ExpenseTotalA,TechnicalServices#RevenueTotalA,TechnicalServices#MiscellaneousArmsLengthPricingMethodC,TechnicalServices#MiscellaneousPercentageOfDealingsWithDocumentationC,TechnicalServices,LogisticsServices#ExpenseTotalA,LogisticsServices#RevenueTotalA,LogisticsServices#MiscellaneousArmsLengthPricingMethodC,LogisticsServices#MiscellaneousPercentageOfDealingsWithDocumentationC,LogisticsServices,AssetManagementServices#ExpenseTotalA,AssetManagementServices#RevenueTotalA,AssetManagementServices#MiscellaneousArmsLengthPricingMethodC,AssetManagementServices#MiscellaneousPercentageOfDealingsWithDocumentationC,AssetManagementServices,OtherServices#ExpenseTotalA,OtherServices#RevenueTotalA,OtherServices#MiscellaneousArmsLengthPricingMethodC,OtherServices#MiscellaneousPercentageOfDealingsWithDocumentationC,OtherServices#InternationalDealingsDescriptionT,OtherServices,ServiceArrangements,RP#InternationalDealingsDerivativeTransactionsI,Derivatives#ExpenseTotalA,Derivatives#RevenueTotalA,Derivatives#MiscellaneousArmsLengthPricingMethodC,Derivatives#MiscellaneousPercentageOfDealingsWithDocumentationC,InternationalDealingsRelatedPartyTransactionTypeC,InternationalDealingsRelatedPartyTransactionTypeCCollection,Derivatives#InternationalDealingsTradingGlobalI,Derivatives,RP#InternationalDealingsDebtFactoringSecuritisationI,DebtFactoring#ExpenseBookValueA,DebtFactoring#ExpenseConsiderationA,DebtFactoring#MiscellaneousArmsLengthPricingMethodC,DebtFactoring#MiscellaneousPercentageOfDealingsWithDocumentationC,DebtFactoring,Securitisation#ExpenseBookValueA,Securitisation#ExpenseConsiderationA,Securitisation#MiscellaneousArmsLengthPricingMethodC,Securitisation#MiscellaneousPercentageOfDealingsWithDocumentationC,Securitisation,RP#InternationalDealingsOtherFinancialDealingsI,InterestBearingLoans#LiabilitiesBorrowedAverageBalanceA,InterestBearingLoans#AssetsLoanedAverageBalanceA,InterestBearingLoans,InterestFreeLoans#LiabilitiesBorrowedAverageBalanceA,InterestFreeLoans#AssetsLoanedAverageBalanceA,InterestFreeLoans,Interest#ExpenseTotalA,Interest#RevenueTotalA,Interest#MiscellaneousArmsLengthPricingMethodC,Interest#MiscellaneousPercentageOfDealingsWithDocumentationC,Interest,Guarantees#ExpenseTotalA,Guarantees#RevenueTotalA,Guarantees#MiscellaneousArmsLengthPricingMethodC,Guarantees#MiscellaneousPercentageOfDealingsWithDocumentationC,Guarantees,Insurance#ExpenseTotalA,Insurance#RevenueTotalA,Insurance#MiscellaneousArmsLengthPricingMethodC,Insurance#MiscellaneousPercentageOfDealingsWithDocumentationC,Insurance,Reinsurance#ExpenseTotalA,Reinsurance#RevenueTotalA,Reinsurance#MiscellaneousArmsLengthPricingMethodC,Reinsurance#MiscellaneousPercentageOfDealingsWithDocumentationC,Reinsurance,OtherFinancial#ExpenseTotalA,OtherFinancial#RevenueTotalA,OtherFinancial#MiscellaneousArmsLengthPricingMethodC,OtherFinancial#MiscellaneousPercentageOfDealingsWithDocumentationC,OtherFinancial#InternationalDealingsDescriptionT,OtherFinancial,FinancialDealings,RP#InternationalDealingsForeignExchangeGainLossI,ForeignExchange#InternationalDealingsTransactionTypeC,GainOrLoss#MiscellaneousFunctionalCurrencyC,GainOrLoss#InternationalDealingsLossesHighestNominalA,GainOrLoss#InternationalDealingsGainsHighestNominalA,GainOrLoss,GainOrLossCollection,ForeignExchange#InternationalDealingsLossesOtherA,ForeignExchange#InternationalDealingsGainsOtherA,ForeignExchange,ForeignExchangeCollection,RP#InternationalDealingsOtherRevenueI,RevenueDealings#ExpenseTotalA,RevenueDealings#TotalA,RevenueDealings#MiscellaneousArmsLengthPricingMethodC,RevenueDealings#MiscellaneousPercentageOfDealingsWithDocumentationC,RevenueDealings#InternationalDealingsDescriptionT,RevenueDealings,RP#MiscellaneousCapitalNatureI,TangiblePropertyNonRevenue#AssetsConsiderationPaidA,TangiblePropertyNonRevenue#AssetsConsiderationReceivedA,TangiblePropertyNonRevenue#MiscellaneousAcquisitionsAndDisposalsPricingMethodC,TangiblePropertyNonRevenue#MiscellaneousPercentageOfDealingsWithDocumentationC,TangiblePropertyNonRevenue,IntellectualProperty#AssetsConsiderationPaidA,IntellectualProperty#AssetsConsiderationReceivedA,IntellectualProperty#MiscellaneousAcquisitionsAndDisposalsPricingMethodC,IntellectualProperty#MiscellaneousPercentageOfDealingsWithDocumentationC,IntellectualProperty,SharesOrOtherEquityInterests#AssetsConsiderationPaidA,SharesOrOtherEquityInterests#AssetsConsiderationReceivedA,SharesOrOtherEquityInterests#MiscellaneousAcquisitionsAndDisposalsPricingMethodC,SharesOrOtherEquityInterests#MiscellaneousPercentageOfDealingsWithDocumentationC,SharesOrOtherEquityInterests,LoansOrDebts#AssetsConsiderationPaidA,LoansOrDebts#AssetsConsiderationReceivedA,LoansOrDebts#MiscellaneousAcquisitionsAndDisposalsPricingMethodC,LoansOrDebts#MiscellaneousPercentageOfDealingsWithDocumentationC,LoansOrDebts,IntangibleProperty#AssetsConsiderationPaidA,IntangibleProperty#AssetsConsiderationReceivedA,IntangibleProperty#MiscellaneousAcquisitionsAndDisposalsPricingMethodC,IntangibleProperty#MiscellaneousPercentageOfDealingsWithDocumentationC,IntangibleProperty,DisposalOrAcquisition,RP#MiscellaneousResearchAndDevelopmentCostPlusRemunerationI,ResearchAndDevelopment#ExpenseCostPlusRemunerationA,ResearchAndDevelopment#IncomeCostPlusRemunerationA,ResearchAndDevelopment,RP#MiscellaneousReceivedOrProvidedServicesPropertyProcessesRightsOrObligationsForNonMonetaryConsiderationI,CapitalNoPayment#OrganisationDetailsInternationalRelatedPartyCountryC,CapitalNoPayment#MiscellaneousNatureOfItemC,CapitalNoPayment,RevenueNoPayment#OrganisationDetailsInternationalRelatedPartyCountryC,RevenueNoPayment#MiscellaneousNatureOfItemC,RevenueNoPayment,CapitalNonMonetaryPayment#OrganisationDetailsInternationalRelatedPartyCountryC,CapitalNonMonetaryPayment#MiscellaneousNatureOfItemC,CapitalNonMonetaryPayment,RevenueNonMonetaryPayment#OrganisationDetailsInternationalRelatedPartyCountryC,RevenueNonMonetaryPayment#MiscellaneousNatureOfItemC,RevenueNonMonetaryPayment,RP#InternationalDealingsShareBasedEmployeeRemunerationI,ShareBasedRemuneration#ExpenseDeductionsTotalA,ShareBasedRemuneration#IncomeIncludedInAssessableTotalA,ShareBasedRemuneration,RP#InternationalDealingsCostContributionArrangementsI,RP#InternationalDealingsRestructuringEventsI,RestructuringEvent#OrganisationDetailsEntityC,RestructuringEvent#MiscellaneousCapitalValueC,RestructuringEvent#MiscellaneousNatureOfItemC,RestructuringEvent#OrganisationDetailsInternationalRelatedPartyCountryC,RestructuringEvent#MiscellaneousPercentageOfDealingsWithDocumentationC,RestructuringEvent#InternationalDealingsDescriptionT,RestructuringEvent#InternationalDealingsEventAnalysisI,RestructuringEvent,RestructuringEventCollection,RP#InternationalDealingsTransactionsRestructuringEventsAssetRevaluationI,RP#InternationalDealingsBranchOperationsI,BranchOperationsInterestBearingLoans#LiabilitiesBorrowedAverageBalanceA,BranchOperationsInterestBearingLoans#AssetsLoanedAverageBalanceA,BranchOperationsInterestBearingLoans#ExpenseInterestBorrowedTotalA,BranchOperationsInterestBearingLoans#IncomeInterestLoanedTotalA,BranchOperationsInterestBearingLoans,BranchOperationsInterestFreeLoans#LiabilitiesBorrowedAverageBalanceA,BranchOperationsInterestFreeLoans#AssetsLoanedAverageBalanceA,BranchOperationsInterestFreeLoans,InternalTrading#ExpenseTradingStockPurchaseCostsA,InternalTrading#IncomeTradingStockSalesTotalA,InternalTrading,BranchOperationsOther#MiscellaneousInternallyRecordedDealingsAmountsClaimedTotalA,BranchOperationsOther#MiscellaneousInternallyRecordedDealingsAmountsReturnedTotalA,BranchOperationsOther#IncomeDe,BranchOperationsOther,RP#InternationalDealingsFinancialArrangementsI,FinancialArrangements#ExpenseInterestDebtReceivedAverageBalanceA,FinancialArrangements#IncomeInterestDebtProvidedAverageBalanceA,FinancialArrangements#ExpenseInterestEquityReceivedAverageBalanceA,FinancialArrangements#IncomeInterestEquityProvidedAverageBalanceA,FinancialArrangements,RP#InternationalDealingsTaxationOfFinancialArrangementsRulesAppliedI,InternationalDealingsTaxTimingMethodElectionsC,InternationalDealingsTaxTimingMethodElectionsCCollection,RP#MiscellaneousInterestInControlledForeignCompaniesAndControlledForeignTrustsI,RP#MiscellaneousControlledForeignCompanyTrustInterestAcquiredI,RP#MiscellaneousControlledForeignCompanyTrustInterestDisposedI,Listed#MiscellaneousControlledForeignCompaniesTrustsInterestCt,Listed#MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt,Listed#MiscellaneousControlledForeignCompanyTrustInterestDisposedCt,Listed#MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC,Listed,Specified#MiscellaneousControlledForeignCompaniesTrustsInterestCt,Specified#MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt,Specified#MiscellaneousControlledForeignCompanyTrustInterestDisposedCt,Specified#MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC,Specified,Unlisted#MiscellaneousControlledForeignCompaniesTrustsInterestCt,Unlisted#MiscellaneousControlledForeignCompanyTrustInterestAcquiredCt,Unlisted#MiscellaneousControlledForeignCompanyTrustInterestDisposedCt,Unlisted#MiscellaneousControlledForeignCompaniesActiveIncomeTestSatisfiedC,Unlisted,RP#MiscellaneousControlledForeignCompaniesExcludedTaintedInterestIncomeI,RP#MiscellaneousControlledForeignCompaniesBankingLicencesI,RP#MiscellaneousControlledForeignCompaniesIncomeMoneyLendingI,ListedCFCAmounts#IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA,ListedCFCAmounts#MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt,ListedCFCAmounts#IncomeControlledForeignCompaniesAttributableIncomeA,ListedCFCAmounts#IncomeGrossRevenueActiveIncomeTestSatisfiedA,ListedCFCAmounts#IncomeGrossRevenueActiveIncomeTestNotSatisfiedA,ListedCFCAmounts#IncomeControlledForeignCompaniesAdjustedTaintedConcessionIncomeA,ListedCFCAmounts#IncomeControlledForeignCompaniesAdjustedTaintedIncomeNotDerivedA,ListedCFCAmounts#IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA,ListedCFCAmounts#IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA,ListedCFCAmounts#IncomeControlledForeignCompaniesNotionalAllowableDeductionsA,ListedCFCAmounts,SpecifiedCFCAmounts#IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA,SpecifiedCFCAmounts#MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt,SpecifiedCFCAmounts#IncomeControlledForeignCompaniesAttributableIncomeA,SpecifiedCFCAmounts#IncomeGrossRevenueActiveIncomeTestSatisfiedA,SpecifiedCFCAmounts#IncomeGrossRevenueActiveIncomeTestNotSatisfiedA,SpecifiedCFCAmounts#IncomeControlledForeignCompaniesAdjustedTaintedIncomeA,SpecifiedCFCAmounts#IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA,SpecifiedCFCAmounts#IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA,SpecifiedCFCAmounts#IncomeControlledForeignCompaniesNotionalAllowableDeductionsA,SpecifiedCFCAmounts,UnlistedCFCAmounts#IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeA,UnlistedCFCAmounts#MiscellaneousControlledForeignCompanyExcludedTaintedInterestIncomeCt,UnlistedCFCAmounts#IncomeControlledForeignCompaniesAttributableIncomeA,UnlistedCFCAmounts#IncomeGrossRevenueActiveIncomeTestSatisfiedA,UnlistedCFCAmounts#IncomeGrossRevenueActiveIncomeTestNotSatisfiedA,UnlistedCFCAmounts#IncomeControlledForeignCompaniesAdjustedTaintedIncomeA,UnlistedCFCAmounts#IncomeControlledForeignCompaniesOtherNotionalAssessableIncomeA,UnlistedCFCAmounts#IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA,UnlistedCFCAmounts#IncomeControlledForeignCompaniesNotionalAllowableDeductionsA,UnlistedCFCAmounts,RP#IncomeControlledForeignCompaniesExcludedTaintedInterestIncomeTotalA,RP#IncomeControlledForeignCompaniesAttributableIncomeTotalA,RP#IncomeControlledForeignCompaniesChangeOfResidenceIncomeA,RP#IncomeControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotalA,RP#IncomeGrossRevenueActiveIncomeTestSatisfiedTotalA,RP#IncomeGrossRevenueActiveIncomeTestNotSatisfiedTotalA,RP#IncomeControlledForeignCompaniesNotionalAssessableIncomeTotalA,RP#IncomeControlledForeignCompaniesNotionalAllowableDeductionsTotalA,RP#IncomeForeignBranchOperationsInterestsInForeignCompaniesOrForeignTrustsI,ListedForeignBranchOperations#IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA,ListedForeignBranchOperations#IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA,ListedForeignBranchOperations#IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA,ListedForeignBranchOperations,SpecifiedForeignBranchOperations#IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA,SpecifiedForeignBranchOperations#IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA,SpecifiedForeignBranchOperations#IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA,SpecifiedForeignBranchOperations,UnlistedForeignBranchOperations#IncomeForeignBranchProfitsOfAustralianCompaniesNonAssessableNonExemptIncomeA,UnlistedForeignBranchOperations#IncomePaidOutOfAttributedControlledForeignCompanyIncomeNonAssessableNonExemptIncomeA,UnlistedForeignBranchOperations#IncomeNonPortfolioDividendFromForeignCompanyNonAssessableNonExemptIncomeA,UnlistedForeignBranchOperations,RP#ExpenseNonDeductibleExemptIncomeA,RP#IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeI,RP#IncomeTaxDeductionDebtEarningNonAssessableNonExemptForeignIncomeTotalA,RP#CapitalGainsTaxEventI,CGTEvents#CapitalGainsTaxActiveForeignBusinessCapitalGainsA,CGTEvents#IncomeTaxActiveForeignBusinessCapitalGainsReductionA,CGTEvents#CapitalGainsTaxActiveForeignBusinessCapitalLossesA,CGTEvents#IncomeTaxActiveForeignBusinessCapitalLossesReductionA,CGTEvents,RP#MiscellaneousTransferPropertyMoneyOrServicesI,Transfer#MiscellaneousPropertyMoneyOrServicesA,Transfer#ItemOrderN,Transfer,TransferCollection,RP#MiscellaneousNonResidentTrustInterestEntitlementOrBeneficiaryI,RP#MiscellaneousOffshoreDealingsComplianceI,HubArrangement#MiscellaneousC,HubArrangement#ExpenseImportsA,HubArrangement#RevenueExportsA,HubArrangement,HubArrangementCollection,RP#MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyShareholderI,RP#MiscellaneousForeignHybridLimitedPartnershipOrForeignHybridCompanyInterestInCt,RP#AssetsInvestmentForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotalA,RP#LiabilitiesThinCapitalisationProvisionsAppliedI,RP#LiabilitiesThinCapitalisation2MillionThresholdTestI,RP#LiabilitiesThinCapitalisation90PercentAssetThresholdTestI,RP#LiabilitiesThinCapitalisationExemptionSpecialPurposeEntitiesI,RP#LiabilitiesThinCapitalisationAustralianResidentEntityNonInOutInvestingEntityI,RP#LiabilitiesThinCapitalisationAustralianResidentCompanyElectedI,RP#AustralianBusinessNumberId,ThinCapitalisation#LiabilitiesEntityTypeC,ThinCapitalisation#LiabilitiesFinancialEntityTypeC,ThinCapitalisation#LiabilitiesAveragingMethodAdoptedC,ThinCapitalisation#LiabilitiesDeductionDebtA,ThinCapitalisation#LiabilitiesRelatedNonResidentsDebtDeductionA,ThinCapitalisation#LiabilitiesDeductionDebtDisallowedA,ThinCapitalisation#LiabilitiesDebtAdjustedAverageA,ThinCapitalisation#InterestIncomeAndOtherAmountsA,ThinCapitalisation#DerivedNonResidentInterestIncomeAndOtherAmountsA,EBITDA#IncomeTaxModifiedTaxableIncomeOrLossA,EBITDA#LiabilitiesNetDeductionDebtA,EBITDA#LiabilitiesModifiedDepreciationAndForestryCostsA,EBITDA#LiabilitiesExcessTaxEBITDAA,EBITDA#LiabilitiesEntityTaxEBITDAA,EBITDA#EarningsLimitA,EBITDA#TotalDisregardedAmountsA,HighestExcessEBITDA#OrganisationNameDetailsOrganisationalNameT,HighestExcessEBITDA#ControlledEntityA,HighestExcessEBITDA#ControlledEntityDividedByA,HighestExcessEBITDA,HighestExcessEBITDACollection,EBITDA,ThinCapitalisation#LiabilitiesGroupRatioTestI,GroupRatioTest#LiabilitiesGroupEntitiesCt,GroupRatioTest#LiabilitiesGroupMembersCt,GroupRatioTest#LiabilitiesNetThirdPartyInterestExpenseA,GroupRatioTest#LiabilitiesNetProfitA,GroupRatioTest#LiabilitiesNetThirdPartyInterestExpenseAdjustedA,GroupRatioTest#LiabilitiesDepreciationAndAmortisationExpensesA,GroupRatioTest#LiabilitiesGroupTaxEBITDAA,GroupRatioTest#LiabilitiesA,GroupRatioTest#GroupRatioEarningsLimitA,GRGroupMembers#OrganisationNameDetailsOrganisationalNameT,GRGroupMembers#LiabilitiesGroupMemberEBITDAA,GRGroupMembers#LiabilitiesNetThirdPartyInterestExpenseAdjustedA,GRGroupMembers#OrganisationDetailsInternationalRelatedPartyCountryC,GRGroupMembers,GRGroupMembersCollection,GroupRatioTest,ThinCapitalisation#LiabilitiesThirdPartyDebtTestI,ThirdPartyDebtTest#LiabilitiesChoiceC,ThirdPartyDebtTest#EntityThirdPartyEarningsLimitA,ThirdPartyDebtTest#LiabilitiesSubsections820Dash427AI,ThirdPartyDebtTest#LiabilitiesAssetsHeldByAnotherMemberOfTheObligorGroupI,ThirdPartyDebtTest#LiabilitiesMembershipInterests820Dash427AI,ThirdPartyDebtTest#LiabilitiesMinorOrInsignificantAssetsDisregardedPursuant820Dash427AI,ThirdPartyDebtTest#LiabilitiesThirdPartyDebtInterestA,ThirdPartyDebtTest#LiabilitiesConduitFinancingArrangementI,ThirdPartyDebtTest#LiabilitiesConduitFinancerI,ThirdPartyDebtTest#AustralianBusinessNumberId,ThirdPartyDebtTest#LiabilitiesConduitFinancerCostsIncurredAndDifferFromTermsI,ThirdPartyDebtTest#LiabilitiesConduitFinancerFinancingArrangementTypeC,ThirdPartyDebtTest#LiabilitiesNotConduitFinancerCostsIncurredAndDifferFromTermsI,ThirdPartyDebtTest#LiabilitiesNonConduitFinancerFinancingArrangementTypeC,ThirdPartyDebtTest#LiabilitiesCrossStapleArrangementsI,ThirdPartyDebtTest#LiabilitiesMemberOfAnObligorGroupNotIssuedByTheEntityI,ThirdPartyDebtTest,ThinCapitalisation#LiabilitiesSpecialPurposeEntityI,SpecialPurposeEntity#LiabilitiesDeductionDebtA,SpecialPurposeEntity#LiabilitiesRelatedNonResidentsDebtDeductionA,SpecialPurposeEntity#LiabilitiesDebtAverageA,SpecialPurposeEntity#InterestIncomeAndOtherAmountsA,SpecialPurposeEntity#DerivedNonResidentInterestIncomeAndOtherAmountsA,SpecialPurposeEntity,FRTDisallowedamount#LiabilitiesFixedRatioTestDisallowedAmountI,FRTDisallowedamount#LiabilitiesFixedRatioTestDisallowedC,FRTDisallowedamount#LiabilitiesFixedRatioTestDisallowedAppliedIncomeYearA,FRTDisallowedamount#LiabilitiesFixedRatioTestDisallowedCarriedForwardA,FRTDisallowedamount#LiabilitiesTaxConsolidatedGroupMultipleEntryConsolidatedI,FRTDisallowedamount#LiabilitiesFixedRatioTestDisallowedAmountsTransferredI,FRTDisallowedamount#LiabilitiesFixedRatioTestDisallowedCalculatedA,FRTDisallowedamount,AllADI#LiabilitiesEquityCapitalAdjustedAverageA,AllADI#SafeHarbourA,AllADI#EquityShortfallA,AllADI,OutwardInvestingADI#LiabilitiesRiskWeightedAssetsAverageValueA,OutwardInvestingADI#LiabilitiesEquityCapitalAttributableToOverseasPermanentEstablishmentsA,OutwardInvestingADI#LiabilitiesRiskWeightedAssetsOverseasPermanentEstablishmentsAverageValueTotalA,OutwardInvestingADI#LiabilitiesControlledForeignADIEntityEquityAverageValueA,OutwardInvestingADI#LiabilitiesTier1PrudentialCapitalDeductionA,OutwardInvestingADI,InwardInvestingADI#RiskWeightedAssetsAttributableToAustralianPermanentEstablishmentsAverageA,InwardInvestingADI#DotationCapitalAverageValueA,InwardInvestingADI#ArmsLengthCapitalA,InwardInvestingADI,AustralianPlantationForestryEntity#AssetsValueAverageA,AustralianPlantationForestryEntity#AssetRevaluationA,AustralianPlantationForestryEntity#LiabilitiesDebtMaximumAllowableA,AustralianPlantationForestryEntity#LiabilitiesExcessDebtA,AustralianPlantationForestryEntity#LiabilitiesNonDebtLiabilitiesAverageValueA,AustralianPlantationForestryEntity#LiabilitiesAssociateEntityDebtAverageValueA,AustralianPlantationForestryEntity#AssociateEntityEquityAverageValueA,AustralianPlantationForestryEntity#LiabilitiesAssociateEntityExcessAverageValueA,AustralianPlantationForestryEntity#ExcludedEquityInterestsAverageA,AustralianPlantationForestryEntity,FinancialEntity#ZeroCapitalAverageA,FinancialEntity#LiabilitiesOnLentAverageA,FinancialEntity,OutwardInvestingEntity#ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA,OutwardInvestingEntity#ControlledForeignEntityDebtAverageValueA,OutwardInvestingEntity#ArmsLengthDebtA,OutwardInvestingEntity,InvestingFinancialEntityNonADI#AssetsValueAverageA,InvestingFinancialEntityNonADI#AssetRevaluationA,InvestingFinancialEntityNonADI#LiabilitiesDebtMaximumAllowableA,InvestingFinancialEntityNonADI#LiabilitiesExcessDebtA,InvestingFinancialEntityNonADI#LiabilitiesNonDebtLiabilitiesAverageValueA,InvestingFinancialEntityNonADI#LiabilitiesAssociateEntityDebtAverageValueA,InvestingFinancialEntityNonADI#AssociateEntityEquityAverageValueA,InvestingFinancialEntityNonADI#LiabilitiesAssociateEntityExcessAverageValueA,InvestingFinancialEntityNonADI#ExcludedEquityInterestsAverageA,InvestingFinancialEntityNonADI#ZeroCapitalAverageA,InvestingFinancialEntityNonADI#LiabilitiesOnLentAverageA,InvestingFinancialEntityNonADI,OutwardInvestingFinancialEntity#ControlledForeignNonAuthorisedDepositTakingInstitutionEntityEquityAverageValueA,OutwardInvestingFinancialEntity#ControlledForeignEntityDebtAverageValueA,OutwardInvestingFinancialEntity,ThinCapitalisation#LiabilitiesGearingDebtCapitalTestI,ADI#WorldwideGroupCapitalRatioF,ADI#WorldwideCapitalA,ADI,NonADI#WorldwideDebtTotalA,NonADI#LiabilitiesWorldwideEquityA,NonADI#WorldwideGearingDebtA,NonADI#StatementWorldwideAssetA,NonADI#AverageAustralianAssetA,NonADI,ThinCapitalisation,RP#FinancialServicesForeignBankOrQualifyingFinancialEntityI,NotionalAmount#FinancialServicesNotionalAverageBorrowingsA,NotionalAmount#FinancialServicesNotionalAverageBorrowingsC,NotionalAmount#FinancialServicesNonOffshoreBankingUnitNotionalInterestA,NotionalAmount#FinancialServicesNotionalInterestTaxWithholdingA,NotionalAmount,NotionalDerivativeAndForeignExchange#FinancialServicesNotionalDerivativePaidA,NotionalDerivativeAndForeignExchange#FinancialServicesNotionalDerivativeReceivedA,NotionalDerivativeAndForeignExchange#FinancialServicesNotionalForeignExchangeTransactionPaidA,NotionalDerivativeAndForeignExchange#FinancialServicesNotionalForeignExchangeTransactionReceivedA,NotionalDerivativeAndForeignExchange,RP#FinancialServicesForeignBankOrQualifyingFinancialEntityElectedOutI,NotionalAmountElectedOut#FinancialServicesElectedOutNotionalAverageBorrowingsA,NotionalAmountElectedOut#FinancialServicesElectedOutNotionalAverageBorrowingsC,NotionalAmountElectedOut#FinancialServicesElectedOutNonOffshoreBankingUnitNotionalInterestA,NotionalAmountElectedOut#FinancialServicesElectedOutNotionalInterestTaxWithholdingA,NotionalAmountElectedOut,RP#FinancialServicesNotionalPaymentDeniedI,RP#FinancialServicesNotionalPaymentDeniedA,RP#InternationalDealingsConduitForeignIncomeI,ConduitForeignIncome#InternationalDealingsBalanceA,ConduitForeignIncome#InternationalDealingsDistributedA,ConduitForeignIncome,RP#InternationalDealingsUnfrankedNonPortfolioDividendAccountI,Unfranked#InternationalDealingsNonPortfolioDividendAccountA,Unfranked#InternationalDealingsNonPortfolioAccountConduitForeignIncomeIncludedI,Unfranked,RP#InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidI,RP#InternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA,RP#InternationalDealingsDoubleTaxAgreementFinancialInstitutionsWithholdingExemptionCountryC,FInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA,FAInternationalDealingsWithholdingTaxExemptNonResidentInterestPaidA,RP#MiscellaneousHybridMismatchI,RP#MiscellaneousHybridMismatchDeductionNonInclusionI,RP#MiscellaneousHybridMismatchPaymentsA,MaterialArrangements#MiscellaneousHybridMismatchArrangementC,MaterialArrangements#OrganisationDetailsInternationalRelatedPartyCountryC,MaterialArrangements#MiscellaneousHybridMismatchPaymentsA,MaterialArrangements#MiscellaneousHybridMismatchDeductionsDeniedOrIncomeInclusionA,MaterialArrangements#MiscellaneousHybridMismatchDifferenceReasonC,MaterialArrangements#MiscellaneousHybridMismatchPaymentRecipientI,MaterialArrangements,MaterialArrangementsCollection,RP#MiscellaneousHybridMismatchPaymentI,StructuredArrangement#MiscellaneousHybridMismatchPaymentsA,StructuredArrangement#MiscellaneousHybridMismatchA,StructuredArrangement#MiscellaneousHybridMismatchDeductionDisallowedA,StructuredArrangement,RP#MiscellaneousOffshoreHybridMismatchI,NonStructuredArrangement#MiscellaneousHybridMismatchA,NonStructuredArrangement#MiscellaneousHybridMismatchDeductionDisallowedA,NonStructuredArrangement,MaterialOffshoreHybridMismatch#MiscellaneousHybridMismatchArrangementC,MaterialOffshoreHybridMismatch#MiscellaneousA,MaterialOffshoreHybridMismatch#MiscellaneousHybridMismatchPaymentsC,MaterialOffshoreHybridMismatch#MiscellaneousHybridMismatchPaymentsA,MaterialOffshoreHybridMismatch#MiscellaneousHybridMismatchDeductionDisallowedA,MaterialOffshoreHybridMismatch#MiscellaneousHybridMismatchDeductionDisallowedDifferenceReasonC,MaterialOffshoreHybridMismatch,MaterialOffshoreHybridMismatchCollection,RP#MiscellaneousPaymentNotTaxedOrTaxed10PercentOrLessI,RP#MiscellaneousHybridMismatchDeductionsA,HybridPayerMismatch#OrganisationDetailsInternationalRelatedPartyCountryC,HybridPayerMismatch#MiscellaneousHybridMismatchPaymentsA,HybridPayerMismatch#MiscellaneousHybridMismatchDeductionsDeniedA,HybridPayerMismatch#MiscellaneousHybridMismatchExceptionReasonC,HybridPayerMismatch,HybridPayerMismatchCollection,RP#MiscellaneousHybridArrangementRestructureReplacementI,HybridArrangementRestructure#InternationalDealingsDescriptionT,HybridArrangementRestructure#InternationalDealingsHybridEventAnalysisI,HybridArrangementRestructure,HybridArrangementRestructureCollection,RP#IncomeForeignEquityDistributionForeignTaxDeductionI,RP#IncomeForeignDeductionNotNonAssessableNonExemptA,RP#IncomeDerivedBranchHybridMismatchIncomeI,RP#IncomeForeignHybridIncomeNotNonAssessableNonExemptA,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtApplicableI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtChoiceC,RP#LiabilitiesDebtDeductionCreationRulesRestructuringEventsDescriptionI,RestructuringEventCurrentOrPriorIncomeYear#LiabilitiesThinCapitalisationRestructuringEventsDescriptionT,RestructuringEventCurrentOrPriorIncomeYear#LiabilitiesDebtDeductionCreationRulesTaxableIncomeChangesAfterRestructureC,RestructuringEventCurrentOrPriorIncomeYear,RestructuringEventCurrentOrPriorIncomeYearCollection,RP#LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairPriorIncomeYearI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetAcquisitionI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearCGTI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearCGTA,PriorIncomeYearTopMaterialArrangements#InteractionTransactionA,PriorIncomeYearTopMaterialArrangements#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA,PriorIncomeYearTopMaterialArrangements#InternationalDealingsDescriptionT,PriorIncomeYearTopMaterialArrangements,PriorIncomeYearTopMaterialArrangementsCollection,RP#LiabilitiesDebtDeductionCreationRulesPriorYearDistributionsAssociatePairI,RP#LiabilitiesDebtDeductionCreationRulesFinancialArrangementsFundingI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtDistributionsPriorIncomeYearRelatedI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairPriorIncomeYearDistributionsI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedPriorIncomeYearDistributionsA,PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#InteractionTransactionA,PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA,PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#InternationalDealingsDescriptionT,PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement,PriorIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection,RP#LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionAssociatePairCurrentIncomeYearI,RP#InteractionTransactionA,RP#LiabilitiesDebtDeductionCreationRulesAssetExceptionI,RP#LiabilitiesDebtDeductionCreationRulesCGTAssetAquisitionExceptionC,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtCGTAssetsI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearCGTI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearCGTA,Arrangements#InteractionTransactionA,Arrangements#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA,Arrangements#LiabilitiesDebtDeductionCreationRulesAcquisitionTypeC,Arrangements#InternationalDealingsDescriptionT,Arrangements,ArrangementsCollection,CurrentIncomeYearPaymentsorDistributionsSelection#LiabilitiesDebtDeductionCreationRulesCurrentYearDistributionsAssociatePairI,CurrentIncomeYearPaymentsorDistributionsSelection#PaymentRecordA,CurrentIncomeYearPaymentsorDistributionsSelection#LiabilitiesDebtDeductionCreationRulesFinancialArrangementFundingI,CurrentIncomeYearPaymentsorDistributionsSelection#LiabilitiesDebtDeductionCreationRulesDeductionDebtCurrentIncomeYearDistributionsI,CurrentIncomeYearPaymentsorDistributionsSelection#LiabilitiesDebtDeductionCreationRulesDeductionDebtAssociatePairCurrentIncomeYearDistributionsI,CurrentIncomeYearPaymentsorDistributionsSelection#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentIncomeYearDistributionsA,CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#PaymentRecordA,CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedA,CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#LiabilitiesDebtDeductionCreationRulesPaymentOrDistributionC,CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement#InternationalDealingsDescriptionT,CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangement,CurrentIncomeYearTopPaymentsorDistributionstoAssociatedPairArrangementCollection,CurrentIncomeYearPaymentsorDistributionsSelection,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedAssociatePairTransactionI,RP#LiabilitiesDebtDeductionCreationRulesDeductionDebtDisallowedCurrentYearPaymentsA,AssociatePairs#LiabilitiesDebtDeductionCreationRulesDeductionDebtRevelantEntityAssociatePairT,AssociatePairs#OrganisationDetailsInternationalRelatedPartyCountryC,AssociatePairs#AustralianBusinessNumberId,AssociatePairs#InteractionTransactionA,AssociatePairs,AssociatePairsCollection,RP";

    }
}