using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationCTR;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;
using DataContracts;
using System.Collections.Generic;
using System.Runtime.CompilerServices;
using VaTS;

namespace Ato.CD.Inbound.CTR202602
{
    public class CrossFormValidatorDISTBENTRT : ICrossFormValidator
    {
        private CTR2026 ParentDocument { get; }
        private DISTBENTRT2024 ChildDocument { get; }
        private CTR2026Validator validator = new CTR2026Validator();
        public List<ProcessMessageDocument> Response { get; private set; }

        public CrossFormValidatorDISTBENTRT(CTR2026 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (DISTBENTRT2024)childDocument.ConsumedReport;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //Parent Cross From Validation Rules
            VRATOCTR500390();
            VRATOCTR500392();
            VRATOCTR500393();
            VRATOCTRW00027();

            return Response;
        }

        #region VR.ATO.CTR.500390 

        /*  VR.ATO.CTR.500390
        The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form

        Legacy Rule Format:
        IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND RP:entity.identifier.TFN <> [DISTBENTRT3]
        RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^CTR TFN = CTR:RP:entity.identifier.TFN
        ^DISTBENTRT3 = DISTBENTRT:RP:Identifiers.TaxFileNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500390()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.RPIdentifierTFN.Equals(ChildDocument.DISTBENTRT3);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500390",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"The Tax File Number (TFN) provided for the Distributions to Beneficiaries of Trust schedule must be the same as the TFN of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500390" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR TFN", Value = ParentDocument.RPIdentifierTFN });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "DISTBENTRT3", Value = ChildDocument.DISTBENTRT3 });
                Response.Add(processMessage);
            }
        }

        #endregion VR.ATO.CTR.500390

        #region VR.ATO.CTR.500392

        /*  VR.ATO.CTR.500392
        Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form

        Legacy Rule Format:
        IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND RP:period.startDate <> [DISTBENTRT1]
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^CTR Start Date = CTR Start Date
        ^DISTBENTRT1 = DISTBENTRT:RP:Period.Start.Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500392()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.RPStartDate.Equals(ChildDocument.DISTBENTRT1);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CTR.500392",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period start date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period start date of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context" + validator.OccurrenceIndex(ParentDocument.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500392" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR Start Date", Value = CTR2026Validator.GetValueOrEmptyDate(ParentDocument.RPStartDate) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "DISTBENTRT1", Value = CTR2026Validator.GetValueOrEmptyDate(ChildDocument.DISTBENTRT1) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500392

        #region  VR.ATO.CTR.500393

        /*  VR.ATO.CTR.500393
        Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form

        Legacy Rule Format:
        IF COUNT(SCHEDULE = "DISTBENTRT") = 1 AND RP:period.endDate <> [DISTBENTRT2]
        RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^CTR End Date = CTR End Date
        ^DISTBENTRT2 = DISTBENTRT:RP:Period.End.Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500393()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.RPEndDate.Equals(ChildDocument.DISTBENTRT2);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500393",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule is invalid",
                    LongDescription = @"Reporting period end date for the Distributions to Beneficiaries of Trust schedule must be the same as the Reporting period end date of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context" + validator.OccurrenceIndex(ParentDocument.RPOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500393" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR End Date", Value = CTR2026Validator.GetValueOrEmptyDate(ParentDocument.RPEndDate) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "DISTBENTRT2", Value = CTR2026Validator.GetValueOrEmptyDate(ChildDocument.DISTBENTRT2) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500393

        #region VR.ATO.CTR.W00027

        /*  VR.ATO.CTR.W00027
        Distributions to Beneficiaries of Trust schedule is provided but Gross distribution from trusts amount is not declared

        Legacy Rule Format:
        IF (COUNT(SCHEDULE= "DISTBENTRT") = 1) AND [CTR61] = NULL
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00027()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR61 == null;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00027",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure Gross distribution from trusts amount is declared",
                    LongDescription = @"Ensure Gross distribution from trusts amount is declared when Distributions to Beneficiaries of Trust schedule is provided",
                    Location = "/xbrli:xbrl/tns:TrustDistributionIncome/tns:Income.TrustDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.W00027" } },
                    DocumentSequence = "parent_doc"
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR61) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.W00027
    }
}
