using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationFITR
{

    public partial class FITR2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private FITR2026 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response {get; private set;}

        /// <summary>
        /// Initializes a new instance of the <see cref="FITR2026Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FITR2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="FITR2026Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public FITR2026Validator(FITR2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY"):this(maxParameterNameLength,maxParameterValueLength,emptyParameterValue)
        {                   
            this.ConsumedReport = reportIn;                    
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        public static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        public static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        public static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {                   
            return datetime.GetValueOrDefault().Date;
        }

				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }
        
        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public FITR2026 ConsumedReport { get {return report;} private set {report = value;}}

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();				

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(FITR2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------
                VRATOFITR434022();
                VRATOFITR434024();
                VRATOFITR434026();
                VRATOFITR434035();
                VRATOFITR434041();
                VRATOFITR434042();
                VRATOFITR434043();
                VRATOFITR434050();
                VRATOFITR434051();
                VRATOFITR434053();
                VRATOFITR434054();
                VRATOFITR434055();
                VRATOFITR434058();
                VRATOFITR434059();
                VRATOFITR434060();
                VRATOFITR434068();
                VRATOFITR434069();
                VRATOFITR434072();
                VRATOFITR434073();
                VRATOFITR434074();
                VRATOFITR434075();
                VRATOFITR434081();
                VRATOFITR434082();
                VRATOFITR434098();
                VRATOFITR434100();
                VRATOFITR434102();
                VRATOFITR434103();
                VRATOFITR434110();
                VRATOFITR434119();
                VRATOFITR434123();
                VRATOFITR434124();
                VRATOFITR434145();
                VRATOFITR434244();
                VRATOFITR434261();
                VRATOFITR434262();
                VRATOFITR434263();
                VRATOFITR434264();
                VRATOFITR434265();
                VRATOFITR434266();
                VRATOFITR434267();
                VRATOFITR434268();
                VRATOFITR434269();
                VRATOFITR434271();
                VRATOFITR434286();
                VRATOFITR434292();
                VRATOFITR434310();
                VRATOFITR434311();
                VRATOFITR434314();
                VRATOFITR434315();
                VRATOFITR434316();
                VRATOFITR434330();
                VRATOFITR434333();
                VRATOFITR434335();
                VRATOFITR434337();
                VRATOFITR434338();
                VRATOFITR434350();
                VRATOFITR434500();
                VRATOFITR434501();
                VRATOFITR434502();
                VRATOFITR434503();
                VRATOFITR434504();
                VRATOFITR434505();
                VRATOFITR434506();
                VRATOFITR434507();
                VRATOFITR434508();
                VRATOFITR434509();
                VRATOFITR434510();
                VRATOFITR434511();
                VRATOFITR434512();
                VRATOFITR434513();
                VRATOFITR434514();
                VRATOFITR434515();
                VRATOFITR434516();
                VRATOFITR434517();
                VRATOFITR434518();
                VRATOFITR434519();
                VRATOFITR434520();
                VRATOFITR434521();
                VRATOFITR434522();
                VRATOFITR434523();
                VRATOFITR434524();
                VRATOFITR434525();
                VRATOFITR434526();
                VRATOFITR434527();
                VRATOFITR434528();
                VRATOFITR434529();
                VRATOFITR434530();
                VRATOFITR434531();
                VRATOFITR434532();
                VRATOFITR434537();
                VRATOFITR434538();
                VRATOFITR434547();
                VRATOFITR434548();
                VRATOFITR434549();
                VRATOFITR434550();
                VRATOFITR434551();
                VRATOFITR434552();
                VRATOFITR434553();
                VRATOFITR434554();
                VRATOFITR434555();
                VRATOFITR434556();
                VRATOFITR434557();
                VRATOFITR434558();
                VRATOFITR434559();
                VRATOFITR434560();
                VRATOFITR434561();
                VRATOFITR434562();
                VRATOFITR434563();
                VRATOFITR434564();
                VRATOFITR434565();
                VRATOFITR434566();
                VRATOFITR434567();
                VRATOFITR434568();
                VRATOFITR434569();
                VRATOFITR434570();
                VRATOFITR434571();
                VRATOFITR434572();
                VRATOFITR434573();
                VRATOFITR434574();
                VRATOFITR434575();
                VRATOFITR434576();
                VRATOFITR434577();
                VRATOFITR434578();
                VRATOFITR434579();
                VRATOFITR434580();
                VRATOFITR434581();
                VRATOFITR434582();
                VRATOFITR434583();
                VRATOFITR434584();
                VRATOFITR434585();
                VRATOFITR434586();
                VRATOFITR434587();
                VRATOFITR434588();
                VRATOFITR434589();
                VRATOFITR434590();
                VRATOFITR434591();
                VRATOFITR434592();
                VRATOFITR434593();
                VRATOFITR434594();
                VRATOFITR434595();
                VRATOFITR434596();
                VRATOFITR434597();
                VRATOFITR434598();
                VRATOFITR434599();
                VRATOFITR434600();
                VRATOFITR434601();
                VRATOFITR434602();
                VRATOFITR434603();
                VRATOFITR434604();
                VRATOFITR434605();
                VRATOFITR434606();
                VRATOFITR434607();
                VRATOFITR434608();
                VRATOFITR434609();
                VRATOFITR434610();
                VRATOFITRW00001();
                VRATOFITRW00003();

            #region Repeating report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection  
            if (report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection != null)
            {    
                int itemIndex4 = 0;
                foreach (FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee in report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection)
                {
                    itemIndex4++;
                        VRATOFITR434539(transferee, itemIndex4);
                        VRATOFITR434540(transferee, itemIndex4);
                        VRATOFITR434541(transferee, itemIndex4);
                        VRATOFITR434542(transferee, itemIndex4);
                        VRATOFITR434543(transferee, itemIndex4);
                        VRATOFITR434544(transferee, itemIndex4);
                        VRATOFITR434545(transferee, itemIndex4);
                        VRATOFITR434546(transferee, itemIndex4);
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
				
			
    
      #region VR.ATO.FITR.434022
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434022()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434022
            If the fund or trust wound up during the income year, the Number of members of the superannuation fund/trust must not be provided
    
            Legacy Rule Format:
            ^FITR19 = TRUE AND ^FITR208 <> NULL

            Technical Business Rule Format:
            ^FITR19 = TRUE AND ^FITR208 <> NULL
    
            Data Elements:
    
            ^FITR208 = FITR:RP:StatusOfFundOrTrust:OrganisationControl.RegisteredMember.Count
    
            ^FITR19 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.FITR19 == true && report.FITR208 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434297", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the fund or trust wound up during the income year, the number of members must not be provided",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:OrganisationControlRegisteredMemberCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434022"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR19", Value = GetValueOrEmpty(report.FITR19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR208", Value = report.FITR208.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434024
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434024()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434024
            The date the superannuation fund was established must not be after the financial year
    
            Legacy Rule Format:
            ^FITR18 > AsDate(Concat(^FITR1, '-06-30'))

            Technical Business Rule Format:
            ^FITR18 > AsDate(Concat(^FITR1, '-06-30'))
    
            Data Elements:
    
            ^FITR18 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.Establishment.Date
    
            ^FITR1 = FITR:RP:Report.TargetFinancial.Year
            */
            assertion = (report.FITR18.GetValueOrDefault() > AsDate(string.Concat(report.FITR1.GetValueOrDefault(), @"-06-30")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434024", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of establishment must not be after the financial year",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:OrganisationDetailsEstablishmentD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434024"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR18", Value = GetValueOrEmptyDate(report.FITR18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR1", Value = GetValueOrEmpty(report.FITR1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434026
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434026()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434026
            If the superannuation fund was wound up during the year, the date the fund ceased operations must be present
    
            Legacy Rule Format:
            ^FITR19 = TRUE AND ^FITR20 = NULL

            Technical Business Rule Format:
            ^FITR19 = TRUE AND ^FITR20 = NULL
    
            Data Elements:
    
            ^FITR20 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.OrganisationActivityStatusEnd.Date
    
            ^FITR19 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.FITR19 == true && report.FITR20 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434026", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date wound up must be present if the fund wound up during the income year",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:OrganisationDetailsOrganisationActivityStatusEndD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434026"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR19", Value = GetValueOrEmpty(report.FITR19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR20", Value = GetValueOrEmptyDate(report.FITR20) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434035
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434035()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434035
            Answer to CGT event question must be TRUE if Net capital gain is present
    
            Legacy Rule Format:
            ^FITR23 > 0 AND ^FITR21 = FALSE

            Technical Business Rule Format:
            ^FITR23 > 0 AND ^FITR21 = FALSE
    
            Data Elements:
    
            ^FITR21 = FITR:RP:Income:CapitalGainsTax.Event.Indicator
    
            ^FITR23 = FITR:RP:Income:Income.CapitalGainsNet.Amount
            */
            assertion = (report.FITR23.GetValueOrDefault() > 0 && report.FITR21 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to CGT event question must be 'yes' (true)",
                    LongDescription = @"If a Net capital gain amount is entered, then the answer to the question 'Did you have a CGT event during the year' must be 'yes' (true)",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxEventI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR23", Value = GetValueOrEmpty(report.FITR23) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR21", Value = GetValueOrEmpty(report.FITR21) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434041
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434041()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434041
            Gross foreign income must not be less than Net foreign income
    
            Legacy Rule Format:
            ^FITR27 <> NULL AND ^FITR27 < ^FITR28

            Technical Business Rule Format:
            ^FITR27 <> NULL AND ^FITR27 < ^FITR28
    
            Data Elements:
    
            ^FITR27 = FITR:RP:Income:Income.Gross.Amount
    
            ^FITR28 = FITR:RP:Income:Income.Net.Amount
            */
            assertion = (report.FITR27 != null && report.FITR27.GetValueOrDefault() < report.FITR28.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434041", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income must not be less than Net foreign income.",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:GrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR27", Value = GetValueOrEmpty(report.FITR27) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR28", Value = GetValueOrEmpty(report.FITR28) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434042
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434042()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434042
            Gross foreign income must be present if Net foreign income is present
    
            Legacy Rule Format:
            ^FITR28 > 0 AND ^FITR27 = NULL

            Technical Business Rule Format:
            ^FITR28 > 0 AND ^FITR27 = NULL
    
            Data Elements:
    
            ^FITR27 = FITR:RP:Income:Income.Gross.Amount
    
            ^FITR28 = FITR:RP:Income:Income.Net.Amount
            */
            assertion = (report.FITR28.GetValueOrDefault() > 0 && report.FITR27 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434042", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income must be present if Net foreign income is present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:GrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434042"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR28", Value = GetValueOrEmpty(report.FITR28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR27", Value = GetValueOrEmpty(report.FITR27) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434043
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434043()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434043
            Number of transfers received from foreign funds for the current income year must be present if a transfer amount is present
    
            Legacy Rule Format:
            ^FITR30 > 0 AND ^FITR31 = NULL

            Technical Business Rule Format:
            ^FITR30 > 0 AND ^FITR31 = NULL
    
            Data Elements:
    
            ^FITR31 = FITR:RP:Income:InternationalDealings.ForeignFundTransfers.Count
    
            ^FITR30 = FITR:RP:Income:Equity.InternationalDealings.ForeignFundTransfers.Amount
            */
            assertion = (report.FITR30.GetValueOrDefault() > 0 && report.FITR31 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434043", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Number of transfers from foreign funds must be present if an amount is present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:InternationalDealingsForeignFundTransfersCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434043"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR30", Value = GetValueOrEmpty(report.FITR30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR31", Value = GetValueOrEmpty(report.FITR31) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434050
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434050()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434050
            Dividends - Franking credit cannot be greater than 48% of Dividends - Franked amount
    
            Legacy Rule Format:
            ^FITR36 > 0 AND ((^FITR35 * 0.48) < (^FITR36 - 1))

            Technical Business Rule Format:
            ^FITR36 > 0 AND ((^FITR35 * 0.48) < (^FITR36 - 1))
    
            Data Elements:
    
            ^FITR36 = FITR:RP:Income:IncomeTax.FrankingCredits.Amount
    
            ^FITR35 = FITR:RP:Income:Income.DividendsFranked.Amount
            */
            assertion = (report.FITR36.GetValueOrDefault() > 0 && report.FITR35.GetValueOrDefault() * 0.48M < report.FITR36.GetValueOrDefault() - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430098", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit cannot be greater than 48% of dividends-franked amount.",
                    LongDescription = @"Franking credit must not exceed 48% of Dividends - Franked amount",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:TaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434050"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR36", Value = GetValueOrEmpty(report.FITR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR35", Value = GetValueOrEmpty(report.FITR35) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434051
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434051()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434051
            Dividends - Franking credit must be present if Dividends - Franked amount is present
    
            Legacy Rule Format:
            ^FITR35 > 0 AND ^FITR36 = NULL

            Technical Business Rule Format:
            ^FITR35 > 0 AND ^FITR36 = NULL
    
            Data Elements:
    
            ^FITR36 = FITR:RP:Income:IncomeTax.FrankingCredits.Amount
    
            ^FITR35 = FITR:RP:Income:Income.DividendsFranked.Amount
            */
            assertion = (report.FITR35.GetValueOrDefault() > 0 && report.FITR36 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430100", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit amount must be present",
                    LongDescription = @"If Dividends - Franked amount is present, then Franking credit must be present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:TaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434051"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR35", Value = GetValueOrEmpty(report.FITR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR36", Value = GetValueOrEmpty(report.FITR36) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434053()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434053
            Transfer of liability to LIC or PST must not exceed the sum of Assessable employer contributions and Assessable personal contributions
    
            Legacy Rule Format:
            ^FITR46 > (^FITR41 + ^FITR42)

            Technical Business Rule Format:
            ^FITR46 > (^FITR41 + ^FITR42)
    
            Data Elements:
    
            ^FITR46 = FITR:RP:Income:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^FITR41 = FITR:RP:Income:SuperannuationContribution.EmployerAssessable.Amount
    
            ^FITR42 = FITR:RP:Income:SuperannuationContribution.PersonalAssessable.Amount
            */
            assertion = (report.FITR46.GetValueOrDefault() > report.FITR41.GetValueOrDefault() + report.FITR42.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434053", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transfer of liability to LIC or PST must not exceed the sum of Assessable contributions",
                    LongDescription = @"The 'Transfer of liability to life insurance company or PST' amount must not be greater than the sum of 'Assessable employer contributions' and 'Assessable personal contributions'",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434053"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR46", Value = GetValueOrEmpty(report.FITR46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR41", Value = GetValueOrEmpty(report.FITR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR42", Value = GetValueOrEmpty(report.FITR42) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434054()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434054
            Transfer of liability to LIC or PST amount must be present
    
            Legacy Rule Format:
            ^FITR84 = TRUE AND (^FITR46 = 0 OR ^FITR46 = NULL)

            Technical Business Rule Format:
            ^FITR84 = TRUE AND (^FITR46 = 0 OR ^FITR46 = NULL)
    
            Data Elements:
    
            ^FITR46 = FITR:RP:Income:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^FITR84 = FITR:RP:OtherInformation:TransferOfLiabilities:IncomeTax.Superannuation.TransferredAssessable.Indicator
            */
            assertion = (report.FITR84 == true && (report.FITR46 == 0 || report.FITR46 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434054", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transfer of liability to LIC or PST amount must be present",
                    LongDescription = @"If the answer to the question 'Has the fund or trust, with the consent of the transferee, transferred assessable contributions...' is 'yes' (true), then a 'Transfer of liability to a LIC or PST' amount, other than zero, must be present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434054"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR84", Value = GetValueOrEmpty(report.FITR84) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR46", Value = GetValueOrEmpty(report.FITR46) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434055
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434055()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434055
            Other assessable income received must be present if Other assessable income received - Code is present
    
            Legacy Rule Format:
            ^FITR48 <> NULL AND (^FITR47 = NULL OR ^FITR47 = 0)

            Technical Business Rule Format:
            ^FITR48 <> NULL AND (^FITR47 = NULL OR ^FITR47 = 0)
    
            Data Elements:
    
            ^FITR47 = FITR:RP:Income:Income.Other.Amount
    
            ^FITR48 = FITR:RP:Income:Income.Other.Code
            */
            assertion = (report.FITR48 != null && (report.FITR47 == null || report.FITR47 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434055", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income amount must be present",
                    LongDescription = @"The 'Other income' amount must be greater than zero if the 'Other income - code' is present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:OtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR48", Value = GetValueOrEmpty(report.FITR48) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR47", Value = GetValueOrEmpty(report.FITR47) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434058
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434058()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434058
            Other assessable income received - Code must be present if Other assessable income received amount is present
    
            Legacy Rule Format:
            ^FITR47 > 0 AND ^FITR48 = NULL

            Technical Business Rule Format:
            ^FITR47 > 0 AND ^FITR48 = NULL
    
            Data Elements:
    
            ^FITR48 = FITR:RP:Income:Income.Other.Code
    
            ^FITR47 = FITR:RP:Income:Income.Other.Amount
            */
            assertion = (report.FITR47.GetValueOrDefault() > 0 && report.FITR48 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434282", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income code must be present if Other income amount is present.",
                    LongDescription = @"Other income codes are:    'B' - Assessable balancing adjustment amount,    'C' - LIC capital gain amount,  'O' - Other income received not listed,    'R' - Rebate or refund of death or disability premiums,    'W' - Gross payments subject to foreign resident withholding, 'T' - TOFA gains or assessable TOFA balancing adjustment",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:OtherC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR47", Value = GetValueOrEmpty(report.FITR47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR48", Value = GetValueOrEmpty(report.FITR48) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434059
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434059()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434059
            Exempt current pension income must not be greater than Gross income
    
            Legacy Rule Format:
            ^FITR200 > (^FITR23 + ^FITR24 + ^FITR25 + ^FITR26 + ^FITR28 + ^FITR29 + ^FITR30 + ^FITR32 + ^FITR33 + ^FITR34 + ^FITR35 + ^FITR36 + ^FITR37 + ^FITR38 + ^FITR39 + ^FITR40 + ^FITR41 + ^FITR42 + ^FITR43 - ^FITR44 - ^FITR45 - ^FITR46 + ^FITR186 + ^FITR47 + ^FITR49 + ^FITR50)

            Technical Business Rule Format:
            ^FITR200 > (^FITR23 + ^FITR24 + ^FITR25 + ^FITR26 + ^FITR28 + ^FITR29 + ^FITR30 + ^FITR32 + ^FITR33 + ^FITR34 + ^FITR35 + ^FITR36 + ^FITR37 + ^FITR38 + ^FITR39 + ^FITR40 + ^FITR41 + ^FITR42 + ^FITR43 - ^FITR44 - ^FITR45 - ^FITR46 + ^FITR186 + ^FITR47 + ^FITR49 + ^FITR50)
    
            Data Elements:
    
            ^FITR200 = FITR:RP:Income:TaxConcession.PensionIncomeExemptDeduction.Amount
    
            ^FITR23 = FITR:RP:Income:Income.CapitalGainsNet.Amount
    
            ^FITR24 = FITR:RP:Income:Income.RentLeasingHiringGross.Amount
    
            ^FITR25 = FITR:RP:Income:Income.Interest.Gross.Amount
    
            ^FITR26 = FITR:RP:Income:Income.ForestryManagedInvestmentScheme.Amount
    
            ^FITR28 = FITR:RP:Income:Income.Net.Amount
    
            ^FITR29 = FITR:RP:Income:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^FITR30 = FITR:RP:Income:Equity.InternationalDealings.ForeignFundTransfers.Amount
    
            ^FITR32 = FITR:RP:Income:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^FITR33 = FITR:RP:Income:Income.PartnershipDistributionGross.Amount
    
            ^FITR34 = FITR:RP:Income:Income.DividendsUnfranked.Amount
    
            ^FITR35 = FITR:RP:Income:Income.DividendsFranked.Amount
    
            ^FITR36 = FITR:RP:Income:IncomeTax.FrankingCredits.Amount
    
            ^FITR37 = FITR:RP:Income:Income.TrustDistributionUnfranked.Amount
    
            ^FITR38 = FITR:RP:Income:Income.TrustDistributionFranked.Amount
    
            ^FITR39 = FITR:RP:Income:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^FITR40 = FITR:RP:Income:Income.TrustDistributionOther.Amount
    
            ^FITR41 = FITR:RP:Income:SuperannuationContribution.EmployerAssessable.Amount
    
            ^FITR42 = FITR:RP:Income:SuperannuationContribution.PersonalAssessable.Amount
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^FITR44 = FITR:RP:Income:SuperannuationContribution.ContributionsExcludedByTrustee.Amount
    
            ^FITR45 = FITR:RP:Income:SuperannuationContribution.ReducedContributions.Amount
    
            ^FITR46 = FITR:RP:Income:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^FITR47 = FITR:RP:Income:Income.Other.Amount
    
            ^FITR49 = FITR:RP:Income:Income.AssessableIncomeDueToFundTaxStatusChange.Amount
    
            ^FITR50 = FITR:RP:Income:Income.NonArmsLengthOtherNet.Amount
    
            ^FITR186 = FITR:RP:Income:Income.ForeignExchangeGains.Amount
            */
            assertion = (report.FITR200.GetValueOrDefault() > report.FITR23.GetValueOrDefault() + report.FITR24.GetValueOrDefault() + report.FITR25.GetValueOrDefault() + report.FITR26.GetValueOrDefault() + report.FITR28.GetValueOrDefault() + report.FITR29.GetValueOrDefault() + report.FITR30.GetValueOrDefault() + report.FITR32.GetValueOrDefault() + report.FITR33.GetValueOrDefault() + report.FITR34.GetValueOrDefault() + report.FITR35.GetValueOrDefault() + report.FITR36.GetValueOrDefault() + report.FITR37.GetValueOrDefault() + report.FITR38.GetValueOrDefault() + report.FITR39.GetValueOrDefault() + report.FITR40.GetValueOrDefault() + report.FITR41.GetValueOrDefault() + report.FITR42.GetValueOrDefault() + report.FITR43.GetValueOrDefault() - report.FITR44.GetValueOrDefault() - report.FITR45.GetValueOrDefault() - report.FITR46.GetValueOrDefault() + report.FITR186.GetValueOrDefault() + report.FITR47.GetValueOrDefault() + report.FITR49.GetValueOrDefault() + report.FITR50.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt current pension income must not be greater than Gross income.",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:TaxConcessionPensionIncomeExemptDeductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR200", Value = GetValueOrEmpty(report.FITR200) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR23", Value = GetValueOrEmpty(report.FITR23) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR24", Value = GetValueOrEmpty(report.FITR24) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR25", Value = GetValueOrEmpty(report.FITR25) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR26", Value = GetValueOrEmpty(report.FITR26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR28", Value = GetValueOrEmpty(report.FITR28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR29", Value = GetValueOrEmpty(report.FITR29) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR30", Value = GetValueOrEmpty(report.FITR30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR32", Value = GetValueOrEmpty(report.FITR32) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR33", Value = GetValueOrEmpty(report.FITR33) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR34", Value = GetValueOrEmpty(report.FITR34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR35", Value = GetValueOrEmpty(report.FITR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR36", Value = GetValueOrEmpty(report.FITR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR37", Value = GetValueOrEmpty(report.FITR37) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR38", Value = GetValueOrEmpty(report.FITR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR39", Value = GetValueOrEmpty(report.FITR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR40", Value = GetValueOrEmpty(report.FITR40) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR41", Value = GetValueOrEmpty(report.FITR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR42", Value = GetValueOrEmpty(report.FITR42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR44", Value = GetValueOrEmpty(report.FITR44) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR45", Value = GetValueOrEmpty(report.FITR45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR46", Value = GetValueOrEmpty(report.FITR46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR186", Value = GetValueOrEmpty(report.FITR186) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR47", Value = GetValueOrEmpty(report.FITR47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR49", Value = GetValueOrEmpty(report.FITR49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR50", Value = GetValueOrEmpty(report.FITR50) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434060
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434060()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434060
            Exempt current pension income must not equal Gross income when Deductions are present
    
            Legacy Rule Format:
            ^FITR200 = (^FITR23 + ^FITR24 + ^FITR25 + ^FITR26 + ^FITR28 + ^FITR29 + ^FITR30 + ^FITR32 + ^FITR33 + ^FITR34 + ^FITR35 + ^FITR36 + ^FITR37 + ^FITR38 + ^FITR39 + ^FITR40 + ^FITR41 + ^FITR42 + ^FITR43 - ^FITR44 - ^FITR45 - ^FITR46 + ^FITR186 + ^FITR47 + ^FITR49 + ^FITR50) AND (^FITR52 > 0 OR ^FITR53 > 0 OR ^FITR54 > 0 OR ^FITR55 > 0 OR ^FITR56 > 0 OR ^FITR57 > 0 OR ^FITR59 > 0 OR ^FITR190 > 0 OR ^FITR188 > 0 OR ^FITR61 > 0  OR ^FITR187 > 0 OR ^FITR62 > 0 OR ^FITR64 > 0)

            Technical Business Rule Format:
            ^FITR200 = (^FITR23 + ^FITR24 + ^FITR25 + ^FITR26 + ^FITR28 + ^FITR29 + ^FITR30 + ^FITR32 + ^FITR33 + ^FITR34 + ^FITR35 + ^FITR36 + ^FITR37 + ^FITR38 + ^FITR39 + ^FITR40 + ^FITR41 + ^FITR42 + ^FITR43 - ^FITR44 - ^FITR45 - ^FITR46 + ^FITR186 + ^FITR47 + ^FITR49 + ^FITR50) AND (^FITR52 > 0 OR ^FITR53 > 0 OR ^FITR54 > 0 OR ^FITR55 > 0 OR ^FITR56 > 0 OR ^FITR57 > 0 OR ^FITR59 > 0 OR ^FITR190 > 0 OR ^FITR188 > 0 OR ^FITR61 > 0  OR ^FITR187 > 0 OR ^FITR62 > 0 OR ^FITR64 > 0)
    
            Data Elements:
    
            ^FITR200 = FITR:RP:Income:TaxConcession.PensionIncomeExemptDeduction.Amount
    
            ^FITR23 = FITR:RP:Income:Income.CapitalGainsNet.Amount
    
            ^FITR24 = FITR:RP:Income:Income.RentLeasingHiringGross.Amount
    
            ^FITR25 = FITR:RP:Income:Income.Interest.Gross.Amount
    
            ^FITR26 = FITR:RP:Income:Income.ForestryManagedInvestmentScheme.Amount
    
            ^FITR28 = FITR:RP:Income:Income.Net.Amount
    
            ^FITR29 = FITR:RP:Income:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^FITR30 = FITR:RP:Income:Equity.InternationalDealings.ForeignFundTransfers.Amount
    
            ^FITR32 = FITR:RP:Income:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^FITR33 = FITR:RP:Income:Income.PartnershipDistributionGross.Amount
    
            ^FITR34 = FITR:RP:Income:Income.DividendsUnfranked.Amount
    
            ^FITR35 = FITR:RP:Income:Income.DividendsFranked.Amount
    
            ^FITR36 = FITR:RP:Income:IncomeTax.FrankingCredits.Amount
    
            ^FITR37 = FITR:RP:Income:Income.TrustDistributionUnfranked.Amount
    
            ^FITR38 = FITR:RP:Income:Income.TrustDistributionFranked.Amount
    
            ^FITR39 = FITR:RP:Income:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^FITR40 = FITR:RP:Income:Income.TrustDistributionOther.Amount
    
            ^FITR41 = FITR:RP:Income:SuperannuationContribution.EmployerAssessable.Amount
    
            ^FITR42 = FITR:RP:Income:SuperannuationContribution.PersonalAssessable.Amount
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^FITR44 = FITR:RP:Income:SuperannuationContribution.ContributionsExcludedByTrustee.Amount
    
            ^FITR45 = FITR:RP:Income:SuperannuationContribution.ReducedContributions.Amount
    
            ^FITR46 = FITR:RP:Income:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^FITR47 = FITR:RP:Income:Income.Other.Amount
    
            ^FITR49 = FITR:RP:Income:Income.AssessableIncomeDueToFundTaxStatusChange.Amount
    
            ^FITR50 = FITR:RP:Income:Income.NonArmsLengthOtherNet.Amount
    
            ^FITR52 = FITR:RP:Deductions:InterestExpensesAustralia:Expense.Interest.Amount
    
            ^FITR53 = FITR:RP:Deductions:InterestExpensesOverseas:Expense.Interest.Amount
    
            ^FITR54 = FITR:RP:Deductions:Remuneration.WagesandSalaries.Payments.Amount
    
            ^FITR55 = FITR:RP:Deductions:Expense.CapitalWorksDeduction.Amount
    
            ^FITR56 = FITR:RP:Deductions:Expense.DepreciationAllowableDeduction.Amount
    
            ^FITR57 = FITR:RP:Deductions:Expense.PremiumsDeathOrDisability.Amount
    
            ^FITR59 = FITR:RP:Deductions:Expense.Investment.Amount
    
            ^FITR61 = FITR:RP:Deductions:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^FITR62 = FITR:RP:Deductions:Expense.DeductibleOther.Amount
    
            ^FITR64 = FITR:RP:Deductions:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^FITR186 = FITR:RP:Income:Income.ForeignExchangeGains.Amount
    
            ^FITR187 = FITR:RP:Deductions:Income.ForeignExchangeLosses.Amount
    
            ^FITR188 = FITR:RP:Deductions:Expense.Operating.AdministrativeExpense.Amount
    
            ^FITR190 = FITR:RP:Deductions:Expense.Operating.ManagementExpense.Amount
            */
            assertion = (report.FITR200.GetValueOrDefault() == report.FITR23.GetValueOrDefault() + report.FITR24.GetValueOrDefault() + report.FITR25.GetValueOrDefault() + report.FITR26.GetValueOrDefault() + report.FITR28.GetValueOrDefault() + report.FITR29.GetValueOrDefault() + report.FITR30.GetValueOrDefault() + report.FITR32.GetValueOrDefault() + report.FITR33.GetValueOrDefault() + report.FITR34.GetValueOrDefault() + report.FITR35.GetValueOrDefault() + report.FITR36.GetValueOrDefault() + report.FITR37.GetValueOrDefault() + report.FITR38.GetValueOrDefault() + report.FITR39.GetValueOrDefault() + report.FITR40.GetValueOrDefault() + report.FITR41.GetValueOrDefault() + report.FITR42.GetValueOrDefault() + report.FITR43.GetValueOrDefault() - report.FITR44.GetValueOrDefault() - report.FITR45.GetValueOrDefault() - report.FITR46.GetValueOrDefault() + report.FITR186.GetValueOrDefault() + report.FITR47.GetValueOrDefault() + report.FITR49.GetValueOrDefault() + report.FITR50.GetValueOrDefault() && (report.FITR52.GetValueOrDefault() > 0 || report.FITR53.GetValueOrDefault() > 0 || report.FITR54.GetValueOrDefault() > 0 || report.FITR55.GetValueOrDefault() > 0 || report.FITR56.GetValueOrDefault() > 0 || report.FITR57.GetValueOrDefault() > 0 || report.FITR59.GetValueOrDefault() > 0 || report.FITR190.GetValueOrDefault() > 0 || report.FITR188.GetValueOrDefault() > 0 || report.FITR61.GetValueOrDefault() > 0 || report.FITR187.GetValueOrDefault() > 0 || report.FITR62.GetValueOrDefault() > 0 || report.FITR64.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434279", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt current pension income must not equal Gross income when Deductions are present",
                    LongDescription = @"The 'Exempt current pension income' amount must not equal the Gross income when one or more Deduction amounts are greater than zero",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:TaxConcessionPensionIncomeExemptDeductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434060"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR200", Value = GetValueOrEmpty(report.FITR200) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR23", Value = GetValueOrEmpty(report.FITR23) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR24", Value = GetValueOrEmpty(report.FITR24) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR25", Value = GetValueOrEmpty(report.FITR25) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR26", Value = GetValueOrEmpty(report.FITR26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR28", Value = GetValueOrEmpty(report.FITR28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR29", Value = GetValueOrEmpty(report.FITR29) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR30", Value = GetValueOrEmpty(report.FITR30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR32", Value = GetValueOrEmpty(report.FITR32) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR33", Value = GetValueOrEmpty(report.FITR33) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR34", Value = GetValueOrEmpty(report.FITR34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR35", Value = GetValueOrEmpty(report.FITR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR36", Value = GetValueOrEmpty(report.FITR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR37", Value = GetValueOrEmpty(report.FITR37) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR38", Value = GetValueOrEmpty(report.FITR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR39", Value = GetValueOrEmpty(report.FITR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR40", Value = GetValueOrEmpty(report.FITR40) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR41", Value = GetValueOrEmpty(report.FITR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR42", Value = GetValueOrEmpty(report.FITR42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR44", Value = GetValueOrEmpty(report.FITR44) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR45", Value = GetValueOrEmpty(report.FITR45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR46", Value = GetValueOrEmpty(report.FITR46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR186", Value = GetValueOrEmpty(report.FITR186) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR47", Value = GetValueOrEmpty(report.FITR47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR49", Value = GetValueOrEmpty(report.FITR49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR50", Value = GetValueOrEmpty(report.FITR50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR52", Value = GetValueOrEmpty(report.FITR52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR53", Value = GetValueOrEmpty(report.FITR53) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR54", Value = GetValueOrEmpty(report.FITR54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR55", Value = GetValueOrEmpty(report.FITR55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR56", Value = GetValueOrEmpty(report.FITR56) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR57", Value = GetValueOrEmpty(report.FITR57) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR59", Value = GetValueOrEmpty(report.FITR59) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR190", Value = GetValueOrEmpty(report.FITR190) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR188", Value = GetValueOrEmpty(report.FITR188) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR61", Value = GetValueOrEmpty(report.FITR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR187", Value = GetValueOrEmpty(report.FITR187) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR62", Value = GetValueOrEmpty(report.FITR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR64", Value = GetValueOrEmpty(report.FITR64) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434068
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434068()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434068
            If Other deductions - Code is present, then an Other deductions amount greater than zero must be present
    
            Legacy Rule Format:
            (^FITR62 = 0 OR ^FITR62 = NULL) AND ^FITR63 <> NULL

            Technical Business Rule Format:
            (^FITR62 = 0 OR ^FITR62 = NULL) AND ^FITR63 <> NULL
    
            Data Elements:
    
            ^FITR62 = FITR:RP:Deductions:Expense.DeductibleOther.Amount
    
            ^FITR63 = FITR:RP:Deductions:Expense.DeductibleOther.Code
            */
            assertion = ((report.FITR62 == 0 || report.FITR62 == null) && report.FITR63 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434068", Severity = ProcessMessageSeverity.Error,
                    Description = @"If 'Other deductions - code' is present, 'Other deductions' amount greater than zero must be present",
                    Location = "/tns:FITR/tns:RP/tns:Deductions/tns:ExpenseDeductibleOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434068"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR62", Value = GetValueOrEmpty(report.FITR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR63", Value = GetValueOrEmpty(report.FITR63) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434069
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434069()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434069
            If Other deductions amount is present, then Other deductions - Code must be present
    
            Legacy Rule Format:
            ^FITR62 > 0 AND ^FITR63 = NULL

            Technical Business Rule Format:
            ^FITR62 > 0 AND ^FITR63 = NULL
    
            Data Elements:
    
            ^FITR62 = FITR:RP:Deductions:Expense.DeductibleOther.Amount
    
            ^FITR63 = FITR:RP:Deductions:Expense.DeductibleOther.Code
            */
            assertion = (report.FITR62.GetValueOrDefault() > 0 && report.FITR63 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434069", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an 'Other deductions' amount is present, an 'Other deductions' code must be present",
                    Location = "/tns:FITR/tns:RP/tns:Deductions/tns:ExpenseDeductibleOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR62", Value = GetValueOrEmpty(report.FITR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR63", Value = GetValueOrEmpty(report.FITR63) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434072
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434072()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434072
            Taxable income or loss amount must be equal to the sum of all income amounts minus sum of all deduction amounts
    
            Legacy Rule Format:
            (^FITR65 <> (^FITR23 + ^FITR24 + ^FITR25 + ^FITR26 + ^FITR28 + ^FITR29 + ^FITR30 + ^FITR32 + ^FITR33 + ^FITR34 + ^FITR35 + ^FITR36 + ^FITR37 + ^FITR38 + ^FITR39 + ^FITR40 + ^FITR41 + ^FITR42 + ^FITR43 - ^FITR44 - ^FITR45 - ^FITR46 + ^FITR186 + ^FITR47 + ^FITR49 + ^FITR50) - ^FITR200 - (^FITR52 + ^FITR53 + ^FITR54 + ^FITR55 + ^FITR56 + ^FITR57 + ^FITR59 + ^FITR190 + ^FITR188 + ^FITR61 + ^FITR187 + ^FITR62 + ^FITR64))

            Technical Business Rule Format:
            (^FITR65 <> (^FITR23 + ^FITR24 + ^FITR25 + ^FITR26 + ^FITR28 + ^FITR29 + ^FITR30 + ^FITR32 + ^FITR33 + ^FITR34 + ^FITR35 + ^FITR36 + ^FITR37 + ^FITR38 + ^FITR39 + ^FITR40 + ^FITR41 + ^FITR42 + ^FITR43 - ^FITR44 - ^FITR45 - ^FITR46 + ^FITR186 + ^FITR47 + ^FITR49 + ^FITR50) - ^FITR200 - (^FITR52 + ^FITR53 + ^FITR54 + ^FITR55 + ^FITR56 + ^FITR57 + ^FITR59 + ^FITR190 + ^FITR188 + ^FITR61 + ^FITR187 + ^FITR62 + ^FITR64))
    
            Data Elements:
    
            ^FITR65 = FITR:RP:Deductions:Income.Taxable.Amount
    
            ^FITR23 = FITR:RP:Income:Income.CapitalGainsNet.Amount
    
            ^FITR24 = FITR:RP:Income:Income.RentLeasingHiringGross.Amount
    
            ^FITR25 = FITR:RP:Income:Income.Interest.Gross.Amount
    
            ^FITR26 = FITR:RP:Income:Income.ForestryManagedInvestmentScheme.Amount
    
            ^FITR28 = FITR:RP:Income:Income.Net.Amount
    
            ^FITR29 = FITR:RP:Income:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^FITR30 = FITR:RP:Income:Equity.InternationalDealings.ForeignFundTransfers.Amount
    
            ^FITR32 = FITR:RP:Income:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^FITR33 = FITR:RP:Income:Income.PartnershipDistributionGross.Amount
    
            ^FITR34 = FITR:RP:Income:Income.DividendsUnfranked.Amount
    
            ^FITR35 = FITR:RP:Income:Income.DividendsFranked.Amount
    
            ^FITR36 = FITR:RP:Income:IncomeTax.FrankingCredits.Amount
    
            ^FITR37 = FITR:RP:Income:Income.TrustDistributionUnfranked.Amount
    
            ^FITR38 = FITR:RP:Income:Income.TrustDistributionFranked.Amount
    
            ^FITR39 = FITR:RP:Income:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^FITR40 = FITR:RP:Income:Income.TrustDistributionOther.Amount
    
            ^FITR41 = FITR:RP:Income:SuperannuationContribution.EmployerAssessable.Amount
    
            ^FITR42 = FITR:RP:Income:SuperannuationContribution.PersonalAssessable.Amount
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^FITR44 = FITR:RP:Income:SuperannuationContribution.ContributionsExcludedByTrustee.Amount
    
            ^FITR45 = FITR:RP:Income:SuperannuationContribution.ReducedContributions.Amount
    
            ^FITR46 = FITR:RP:Income:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^FITR47 = FITR:RP:Income:Income.Other.Amount
    
            ^FITR49 = FITR:RP:Income:Income.AssessableIncomeDueToFundTaxStatusChange.Amount
    
            ^FITR50 = FITR:RP:Income:Income.NonArmsLengthOtherNet.Amount
    
            ^FITR52 = FITR:RP:Deductions:InterestExpensesAustralia:Expense.Interest.Amount
    
            ^FITR53 = FITR:RP:Deductions:InterestExpensesOverseas:Expense.Interest.Amount
    
            ^FITR54 = FITR:RP:Deductions:Remuneration.WagesandSalaries.Payments.Amount
    
            ^FITR55 = FITR:RP:Deductions:Expense.CapitalWorksDeduction.Amount
    
            ^FITR56 = FITR:RP:Deductions:Expense.DepreciationAllowableDeduction.Amount
    
            ^FITR57 = FITR:RP:Deductions:Expense.PremiumsDeathOrDisability.Amount
    
            ^FITR59 = FITR:RP:Deductions:Expense.Investment.Amount
    
            ^FITR61 = FITR:RP:Deductions:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^FITR62 = FITR:RP:Deductions:Expense.DeductibleOther.Amount
    
            ^FITR64 = FITR:RP:Deductions:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^FITR186 = FITR:RP:Income:Income.ForeignExchangeGains.Amount
    
            ^FITR187 = FITR:RP:Deductions:Income.ForeignExchangeLosses.Amount
    
            ^FITR188 = FITR:RP:Deductions:Expense.Operating.AdministrativeExpense.Amount
    
            ^FITR190 = FITR:RP:Deductions:Expense.Operating.ManagementExpense.Amount
    
            ^FITR200 = FITR:RP:Income:TaxConcession.PensionIncomeExemptDeduction.Amount
            */
            assertion = (report.FITR65.GetValueOrDefault() != report.FITR23.GetValueOrDefault() + report.FITR24.GetValueOrDefault() + report.FITR25.GetValueOrDefault() + report.FITR26.GetValueOrDefault() + report.FITR28.GetValueOrDefault() + report.FITR29.GetValueOrDefault() + report.FITR30.GetValueOrDefault() + report.FITR32.GetValueOrDefault() + report.FITR33.GetValueOrDefault() + report.FITR34.GetValueOrDefault() + report.FITR35.GetValueOrDefault() + report.FITR36.GetValueOrDefault() + report.FITR37.GetValueOrDefault() + report.FITR38.GetValueOrDefault() + report.FITR39.GetValueOrDefault() + report.FITR40.GetValueOrDefault() + report.FITR41.GetValueOrDefault() + report.FITR42.GetValueOrDefault() + report.FITR43.GetValueOrDefault() - report.FITR44.GetValueOrDefault() - report.FITR45.GetValueOrDefault() - report.FITR46.GetValueOrDefault() + report.FITR186.GetValueOrDefault() + report.FITR47.GetValueOrDefault() + report.FITR49.GetValueOrDefault() + report.FITR50.GetValueOrDefault() - report.FITR200.GetValueOrDefault() - (report.FITR52.GetValueOrDefault() + report.FITR53.GetValueOrDefault() + report.FITR54.GetValueOrDefault() + report.FITR55.GetValueOrDefault() + report.FITR56.GetValueOrDefault() + report.FITR57.GetValueOrDefault() + report.FITR59.GetValueOrDefault() + report.FITR190.GetValueOrDefault() + report.FITR188.GetValueOrDefault() + report.FITR61.GetValueOrDefault() + report.FITR187.GetValueOrDefault() + report.FITR62.GetValueOrDefault() + report.FITR64.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434072", Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable income/loss amount must equal income minus deductions",
                    LongDescription = @"The Taxable income or loss amount must be equal to the sum of all income amounts minus the sum of all deductions amounts",
                    Location = "/tns:FITR/tns:RP/tns:Deductions/tns:IncomeTaxableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR65", Value = GetValueOrEmpty(report.FITR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR23", Value = GetValueOrEmpty(report.FITR23) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR24", Value = GetValueOrEmpty(report.FITR24) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR25", Value = GetValueOrEmpty(report.FITR25) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR26", Value = GetValueOrEmpty(report.FITR26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR28", Value = GetValueOrEmpty(report.FITR28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR29", Value = GetValueOrEmpty(report.FITR29) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR30", Value = GetValueOrEmpty(report.FITR30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR32", Value = GetValueOrEmpty(report.FITR32) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR33", Value = GetValueOrEmpty(report.FITR33) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR34", Value = GetValueOrEmpty(report.FITR34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR35", Value = GetValueOrEmpty(report.FITR35) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR36", Value = GetValueOrEmpty(report.FITR36) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR37", Value = GetValueOrEmpty(report.FITR37) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR38", Value = GetValueOrEmpty(report.FITR38) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR39", Value = GetValueOrEmpty(report.FITR39) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR40", Value = GetValueOrEmpty(report.FITR40) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR41", Value = GetValueOrEmpty(report.FITR41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR42", Value = GetValueOrEmpty(report.FITR42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR44", Value = GetValueOrEmpty(report.FITR44) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR45", Value = GetValueOrEmpty(report.FITR45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR46", Value = GetValueOrEmpty(report.FITR46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR186", Value = GetValueOrEmpty(report.FITR186) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR47", Value = GetValueOrEmpty(report.FITR47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR49", Value = GetValueOrEmpty(report.FITR49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR50", Value = GetValueOrEmpty(report.FITR50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR200", Value = GetValueOrEmpty(report.FITR200) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR52", Value = GetValueOrEmpty(report.FITR52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR53", Value = GetValueOrEmpty(report.FITR53) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR54", Value = GetValueOrEmpty(report.FITR54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR55", Value = GetValueOrEmpty(report.FITR55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR56", Value = GetValueOrEmpty(report.FITR56) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR57", Value = GetValueOrEmpty(report.FITR57) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR59", Value = GetValueOrEmpty(report.FITR59) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR190", Value = GetValueOrEmpty(report.FITR190) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR188", Value = GetValueOrEmpty(report.FITR188) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR61", Value = GetValueOrEmpty(report.FITR61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR187", Value = GetValueOrEmpty(report.FITR187) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR62", Value = GetValueOrEmpty(report.FITR62) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR64", Value = GetValueOrEmpty(report.FITR64) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434073
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434073()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434073
            If there is a loss reported at Taxable income or loss, and there is no Exempt current pension income reported, then Tax losses carried forward to later income years must be present
    
            Legacy Rule Format:
            ^FITR65 < 0 AND (^FITR200 = 0 OR ^FITR200 = NULL) AND (^FITR76 = 0 OR ^FITR76 = NULL)

            Technical Business Rule Format:
            ^FITR65 < 0 AND (^FITR200 = 0 OR ^FITR200 = NULL) AND (^FITR76 = 0 OR ^FITR76 = NULL)
    
            Data Elements:
    
            ^FITR76 = FITR:RP:Losses:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            ^FITR65 = FITR:RP:Deductions:Income.Taxable.Amount
    
            ^FITR200 = FITR:RP:Income:TaxConcession.PensionIncomeExemptDeduction.Amount
            */
            assertion = (report.FITR65.GetValueOrDefault() < 0 && (report.FITR200 == 0 || report.FITR200 == null) && (report.FITR76 == 0 || report.FITR76 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434073", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses carried forward amount must be present",
                    LongDescription = @"If the Taxable income or loss amount is less than zero, there must be an 'Exempt current pension income' amount or a 'Tax losses carried forward to later income years' amount",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:TaxCarriedForwardLaterIncomeYearsTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434073"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR65", Value = GetValueOrEmpty(report.FITR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR200", Value = GetValueOrEmpty(report.FITR200) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR76", Value = GetValueOrEmpty(report.FITR76) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434074
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434074()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434074
            If Taxable income or loss is greater than zero and Gross tax is present, then Gross tax must not be less than 15% of Taxable income or loss
    
            Legacy Rule Format:
            ^FITR65 > 0 AND ^FITR66 <> NULL AND (^FITR66 < (^FITR65 * 0.15) - 1)

            Technical Business Rule Format:
            ^FITR65 > 0 AND ^FITR66 <> NULL AND (^FITR66 < (^FITR65 * 0.15) - 1)
    
            Data Elements:
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
    
            ^FITR65 = FITR:RP:Deductions:Income.Taxable.Amount
            */
            assertion = (report.FITR65.GetValueOrDefault() > 0 && report.FITR66 != null && report.FITR66.GetValueOrDefault() < report.FITR65.GetValueOrDefault() * 0.15M - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434074", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross tax must not be less than 15% of Taxable income",
                    LongDescription = @"If the Taxable income or loss amount is greater than zero, then the Gross tax amount must not be less than 15% of the Taxable income or loss amount",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxableGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR65", Value = GetValueOrEmpty(report.FITR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434075
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434075()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434075
            If No-TFN-quoted contributions amount is equal to zero and Taxable income or loss is less than or equal zero, then Gross tax must not be greater than zero
    
            Legacy Rule Format:
            (^FITR43 = 0 AND ^FITR65 <= 0) AND ^FITR66 > 0

            Technical Business Rule Format:
            (^FITR43 = 0 AND ^FITR65 <= 0) AND ^FITR66 > 0
    
            Data Elements:
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^FITR65 = FITR:RP:Deductions:Income.Taxable.Amount
            */
            assertion = (report.FITR43 == 0 && report.FITR65.GetValueOrDefault() <= 0 && report.FITR66.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434075", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross tax must not be greater than zero",
                    LongDescription = @"If a 'No-TFN quoted contributions' amount is not present, and 'Taxable income or loss' amount is less than or equal to zero, then the Gross Tax amount cannot be greater than zero",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxableGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR65", Value = GetValueOrEmpty(report.FITR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434081
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434081()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434081
            Gross distribution from partnerships or Trust distributions - Income - Other amounts or Other assessable income received must be present
    
            Legacy Rule Format:
            ^FITR71 > 0 AND (^FITR33 = NULL AND ^FITR40 = NULL AND ^FITR47 = NULL)

            Technical Business Rule Format:
            ^FITR71 > 0 AND (^FITR33 = NULL AND ^FITR40 = NULL AND ^FITR47 = NULL)
    
            Data Elements:
    
            ^FITR33 = FITR:RP:Income:Income.PartnershipDistributionGross.Amount
    
            ^FITR40 = FITR:RP:Income:Income.TrustDistributionOther.Amount
    
            ^FITR47 = FITR:RP:Income:Income.Other.Amount
    
            ^FITR71 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = (report.FITR71.GetValueOrDefault() > 0 && (report.FITR33 == null && report.FITR40 == null && report.FITR47 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434081", Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution or Other income amounts must be present",
                    LongDescription = @"If 'Credit: foreign resident withholding' is greater than zero, then 'Gross distributions from partnerships', 'Trust distribution other amounts' and/or 'Other income' must be present",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434081"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR71", Value = GetValueOrEmpty(report.FITR71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR33", Value = GetValueOrEmpty(report.FITR33) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR40", Value = GetValueOrEmpty(report.FITR40) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR47", Value = GetValueOrEmpty(report.FITR47) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434082
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434082()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434082
            If Gross payments made to the fund that were subject to withholding where an Australian Business Number (ABN) was not quoted is greater than zero, then Credit for tax withheld - where ABN or TFN not quoted (non-individual) must be present
    
            Legacy Rule Format:
            ^FITR32 > 0 AND ^FITR72 = NULL

            Technical Business Rule Format:
            ^FITR32 > 0 AND ^FITR72 = NULL
    
            Data Elements:
    
            ^FITR72 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount
    
            ^FITR32 = FITR:RP:Income:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.FITR32.GetValueOrDefault() > 0 && report.FITR72 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434082", Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit: ABN/TFN not quoted amount must be present",
                    LongDescription = @"If 'Gross payments where ABN not quoted' is greater than zero, then 'Credit: TFN/ABN not quoted' amount must be present",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434082"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR32", Value = GetValueOrEmpty(report.FITR32) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR72", Value = GetValueOrEmpty(report.FITR72) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434098
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434098()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434098
            If the fund has wound up during the year, Net assets of the fund available to pay benefits must not be present
    
            Legacy Rule Format:
            ^FITR19 = TRUE AND ^FITR83 <> NULL

            Technical Business Rule Format:
            ^FITR19 = TRUE AND ^FITR83 <> NULL
    
            Data Elements:
    
            ^FITR83 = FITR:RP:OtherInformation:Equity.NetAssetsCalculated.Amount
    
            ^FITR19 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.FITR19 == true && report.FITR83 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434098", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the fund has been wound-up during the year then 'Net assets available to pay benefits' must not be present.",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:EquityNetAssetsCalculatedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434098"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR19", Value = GetValueOrEmpty(report.FITR19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR83", Value = GetValueOrEmpty(report.FITR83) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434100
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434100()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434100
            If the fund or trust has transferred assessable contributions to an LIC or PST, then transferee details must be present
    
            Legacy Rule Format:
            ^FITR84 = TRUE AND (Count(^FITR1021) = 0)

            Technical Business Rule Format:
            ^FITR84 = TRUE AND (Count(^FITR1021) = 0)
    
            Data Elements:
    
            ^FITR84 = FITR:RP:OtherInformation:TransferOfLiabilities:IncomeTax.Superannuation.TransferredAssessable.Indicator
    
            ^FITR1021 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee
            */
            assertion = (report.FITR84 == true && Count(report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434588", Severity = ProcessMessageSeverity.Error,
                    Description = @"Transfer of liabilities to LIC or PST transferee details must be present",
                    LongDescription = @"If the answer to ""Has the fund or trust.....transferred assessable contributions... to a life insurance company (LIC) or pooled superannuation trust (PST)?"" is TRUE, then the name, ABN and amounts for at least one transferee must be present",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:IncomeTaxSuperannuationTransferredAssessableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR84", Value = GetValueOrEmpty(report.FITR84) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434102
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434102()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434102
            If the fund or trust has not transferred assessable contributions to any LIC or PST, then transferee details must not be present
    
            Legacy Rule Format:
            ^FITR84 = FALSE AND (Count(^FITR1021) > 0)

            Technical Business Rule Format:
            ^FITR84 = FALSE AND (Count(^FITR1021) > 0)
    
            Data Elements:
    
            ^FITR84 = FITR:RP:OtherInformation:TransferOfLiabilities:IncomeTax.Superannuation.TransferredAssessable.Indicator
    
            ^FITR1021 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee
            */
            assertion = (report.FITR84 == false && Count(report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionCount) > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434102", Severity = ProcessMessageSeverity.Error,
                    Description = @"If fund has not transferred contributions, Transferee details must not be present",
                    LongDescription = @"If the answer to 'Has the fund or trust,... transferred assessable income contributions ...' is 'no' (false), then the Name, ABN and Amounts for a trusferee must not be present",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection == null ? 0 : (report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection.Count() == 0 ? 0 : report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection.Last().OccurrenceIndex)) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434102"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR84", Value = GetValueOrEmpty(report.FITR84) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434103
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434103()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434103
            Answer to Transfer of liabilities question must be TRUE
    
            Legacy Rule Format:
            ^FITR46 > 0 AND ^FITR84 = FALSE

            Technical Business Rule Format:
            ^FITR46 > 0 AND ^FITR84 = FALSE
    
            Data Elements:
    
            ^FITR84 = FITR:RP:OtherInformation:TransferOfLiabilities:IncomeTax.Superannuation.TransferredAssessable.Indicator
    
            ^FITR46 = FITR:RP:Income:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
            */
            assertion = (report.FITR46.GetValueOrDefault() > 0 && report.FITR84 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434103", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Transfer of liabilities question must be 'yes'",
                    LongDescription = @"If an amount is present at 'Transfer of liability to LIC or PST', then the answer to the question 'Has the fund or trust, ... transferred assessable income contributions ...' must be 'yes' (true).",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:IncomeTaxSuperannuationTransferredAssessableI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434103"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR46", Value = GetValueOrEmpty(report.FITR46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR84", Value = GetValueOrEmpty(report.FITR84) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434110
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434110()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434110
            Answer to international related party dealings question must be TRUE
    
            Legacy Rule Format:
            ^FITR91 = TRUE AND ^FITR90 = FALSE

            Technical Business Rule Format:
            ^FITR91 = TRUE AND ^FITR90 = FALSE
    
            Data Elements:
    
            ^FITR91 = FITR:RP:OtherInformation:OverseasTransactions:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
            ^FITR90 = FITR:RP:OtherInformation:OverseasTransactions:InternationalDealings.RelatedPartiesTransactions.Indicator
            */
            assertion = (report.FITR91 == true && report.FITR90 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434249", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to international related party dealings question must be true.",
                    LongDescription = @"If the answer to the question 'Was the aggregate amount of transactions...greater than $2 million', is 'yes' (true), then the answer to the question 'Did the fund have any transactions or dealing with international parties...', must be yes' (true)",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:OverseasTransactions/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434110"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR91", Value = GetValueOrEmpty(report.FITR91) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR90", Value = GetValueOrEmpty(report.FITR90) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434119
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434119()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434119
            Answer to Overseas interests question must be TRUE if any Attributed foreign income is present
    
            Legacy Rule Format:
            (^FITR78 > 0 OR ^FITR80 > 0) AND ^FITR92 = FALSE

            Technical Business Rule Format:
            (^FITR78 > 0 OR ^FITR80 > 0) AND ^FITR92 = FALSE
    
            Data Elements:
    
            ^FITR92 = FITR:RP:OtherInformation:OverseasTransactions:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
    
            ^FITR78 = FITR:RP:OtherInformation:ListedCountries:Income.InternationalDealings.Attributed.Amount
    
            ^FITR80 = FITR:RP:OtherInformation:UnlistedCountries:Income.InternationalDealings.Attributed.Amount
            */
            assertion = ((report.FITR78.GetValueOrDefault() > 0 || report.FITR80.GetValueOrDefault() > 0) && report.FITR92 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434589", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Overseas interests question must be TRUE if any Attributed foreign income is present",
                    LongDescription = @"If there is any amount reported at Attributed foreign income in listed countries or Attributed foreign income in unlisted countries, then answer to Overseas interests question must be TRUE",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:OverseasTransactions/tns:InternationalDealingsDirectOrIndirectOverseasInterestI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434119"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR78", Value = GetValueOrEmpty(report.FITR78) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR80", Value = GetValueOrEmpty(report.FITR80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR92", Value = GetValueOrEmpty(report.FITR92) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434123
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434123()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434123
            Answer to Death or disability actuary's certificate question must be present
    
            Legacy Rule Format:
            ^FITR96 = TRUE AND ^FITR97 = NULL

            Technical Business Rule Format:
            ^FITR96 = TRUE AND ^FITR97 = NULL
    
            Data Elements:
    
            ^FITR97 = FITR:RP:OtherInformation:OtherTransactions:IncomeTax.Deduction.ClaimedDeathOrDisabilityDeductionCertificateObtained.Indicator
    
            ^FITR96 = FITR:RP:OtherInformation:OtherTransactions:IncomeTax.Deduction.ClaimedDeathOrDisabilityDeductionWhichRequiresCertificate.Indicator
            */
            assertion = (report.FITR96 == true && report.FITR97 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434123", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Death or diability -  'relevant certificate' question must be present",
                    LongDescription = @"If the answer to the question 'Is the fund or trust is claiming a deduction for premiums for death or disability....' is 'yes' (true), then an answer to the question 'If yes, has the fund or trust obtained the relevant certificate', must be present",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:OtherTransactions/tns:IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434123"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR96", Value = GetValueOrEmpty(report.FITR96) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR97", Value = GetValueOrEmpty(report.FITR97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434124
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434124()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434124
            Answer to Death or disability actuary's certificate question must not be present
    
            Legacy Rule Format:
            ^FITR96 = FALSE AND ^FITR97 <> NULL

            Technical Business Rule Format:
            ^FITR96 = FALSE AND ^FITR97 <> NULL
    
            Data Elements:
    
            ^FITR97 = FITR:RP:OtherInformation:OtherTransactions:IncomeTax.Deduction.ClaimedDeathOrDisabilityDeductionCertificateObtained.Indicator
    
            ^FITR96 = FITR:RP:OtherInformation:OtherTransactions:IncomeTax.Deduction.ClaimedDeathOrDisabilityDeductionWhichRequiresCertificate.Indicator
            */
            assertion = (report.FITR96 == false && report.FITR97 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434124", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Death or diability -  'relevant certificate' question must NOT be present",
                    LongDescription = @"If the answer to the question 'Is the fund or trust is claiming a deduction for premiums for death or disability....' is 'no' (false), then an answer to the question 'If yes, has the fund or trust obtained the relevant certificate', must NOT be present",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:OtherTransactions/tns:IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434124"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR96", Value = GetValueOrEmpty(report.FITR96) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR97", Value = GetValueOrEmpty(report.FITR97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434145
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434145()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434145
            The date the superannuation fund ceased operations must be within the year of assessment
    
            Legacy Rule Format:
            ^FITR20 <> NULL AND (^FITR20 < ConvertToDate(1, 7, ^FITR1-1) OR ^FITR20 > ConvertToDate(30, 6, ^FITR1))

            Technical Business Rule Format:
            ^FITR20 <> NULL AND (^FITR20 < ConvertToDate(1, 7, ^FITR1-1) OR ^FITR20 > ConvertToDate(30, 6, ^FITR1))
    
            Data Elements:
    
            ^FITR20 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.OrganisationActivityStatusEnd.Date
    
            ^FITR1 = FITR:RP:Report.TargetFinancial.Year
            */
            assertion = (report.FITR20 != null && (report.FITR20.GetValueOrDefault() < ConvertToDate(1, 7, report.FITR1.GetValueOrDefault() - 1) || report.FITR20.GetValueOrDefault() > ConvertToDate(30, 6, report.FITR1.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434145", Severity = ProcessMessageSeverity.Error,
                    Description = @"Fund wound up date must be within year of assessment.",
                    LongDescription = @"The date the fund wound up must fall within the year of assessment.  For example, if the year of the return being lodged is 2010, then the date must not be before 1 July 2009 and must not be after 30 June 2010.",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:OrganisationDetailsOrganisationActivityStatusEndD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434145"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR20", Value = GetValueOrEmptyDate(report.FITR20) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR1", Value = GetValueOrEmpty(report.FITR1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434244
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434244()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434244
            If the fund is not an Australian superannuation fund, then Tax on no-TFN-quoted contributions must be equal to No-TFN-quoted contributions amount multiplied by 0.02
    
            Legacy Rule Format:
            ^FITR15 = FALSE AND OutsideRange(^FITR181, ^FITR43 * 0.02, 1)

            Technical Business Rule Format:
            ^FITR15 = FALSE AND OutsideRange(^FITR181, ^FITR43 * 0.02, 1)
    
            Data Elements:
    
            ^FITR181 = FITR:RP:IncomeTaxCalculation:IncomeTax.TFNNotQuotedContributionsTax.Amount
    
            ^FITR15 = FITR:RP:StatusOfFundOrTrust:SuperannuationFundDetails.AustralianSuperannuationFundStatus.Indicator
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
            */
            assertion = (report.FITR15 == false && OutsideRange(report.FITR181.GetValueOrDefault(), report.FITR43.GetValueOrDefault() * 0.02M, 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434307", Severity = ProcessMessageSeverity.Error,
                    Description = @"The amount at Tax on no-TFN-quoted contributions is incorrect",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TFNNotQuotedContributionsTaxA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434244"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR15", Value = GetValueOrEmpty(report.FITR15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR181", Value = GetValueOrEmpty(report.FITR181) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434261
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434261()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434261
            Gross tax must be equal to sum of Tax on taxable income and Tax on no-TFN-quoted contributions
    
            Legacy Rule Format:
            ^FITR66 <> (^FITR182 + ^FITR181)

            Technical Business Rule Format:
            ^FITR66 <> (^FITR182 + ^FITR181)
    
            Data Elements:
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
    
            ^FITR181 = FITR:RP:IncomeTaxCalculation:IncomeTax.TFNNotQuotedContributionsTax.Amount
    
            ^FITR182 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalTaxOnTaxableIncome.Amount
            */
            assertion = (report.FITR66.GetValueOrDefault() != report.FITR182.GetValueOrDefault() + report.FITR181.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434300", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross Tax amount incorrect.",
                    LongDescription = @"Gross Tax must be equal to sum of Tax on taxable income and Tax on no-TFN-quoted contributions",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxableGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434261"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR182", Value = GetValueOrEmpty(report.FITR182) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR181", Value = GetValueOrEmpty(report.FITR181) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434262
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434262()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434262
            Non-refundable non-carry forward tax offsets must be equal to sum of Foreign income tax offset and Rebates and tax offsets
    
            Legacy Rule Format:
            (^FITR67 <> NULL OR ^FITR68 <> NULL) AND (^FITR189 <> (^FITR67 + ^FITR68))

            Technical Business Rule Format:
            (^FITR67 <> NULL OR ^FITR68 <> NULL) AND (^FITR189 <> (^FITR67 + ^FITR68))
    
            Data Elements:
    
            ^FITR189 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount
    
            ^FITR67 = FITR:RP:IncomeTaxCalculation:Income.InternationalDealings.TaxOffset.Amount
    
            ^FITR68 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetsAndRebatesTotal.Amount
            */
            assertion = ((report.FITR67 != null || report.FITR68 != null) && report.FITR189.GetValueOrDefault() != report.FITR67.GetValueOrDefault() + report.FITR68.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434262", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-refundable non-carry forward tax offsets amount incorrect.",
                    LongDescription = @"Non-refundable non-carry forward tax offsets amount must be equal to sum of Foreign income tax offset and Rebates and tax offsets.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxOffsetNonRefundableNonCarryForwardA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434262"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR67", Value = GetValueOrEmpty(report.FITR67) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR68", Value = GetValueOrEmpty(report.FITR68) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR189", Value = GetValueOrEmpty(report.FITR189) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434263
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434263()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434263
            Refundable tax offsets amount must be equal to sum of Complying fund's franking credits tax offset, No-TFN tax offset, National rental affordability scheme tax offset and Exploration credit tax offset
    
            Legacy Rule Format:
            ^FITR184 <> (^FITR73 + ^FITR131 + ^FITR133 + ^FITR206)

            Technical Business Rule Format:
            ^FITR184 <> (^FITR73 + ^FITR131 + ^FITR133 + ^FITR206)
    
            Data Elements:
    
            ^FITR184 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^FITR73 = FITR:RP:IncomeTaxCalculation:IncomeTax.FrankingCredits.Refundable.Amount
    
            ^FITR131 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetTFNNotQuoted.Amount
    
            ^FITR133 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetEntitlementForNationalRentalAffordabilityScheme.Amount
    
            ^FITR206 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetExplorationCredits.Amount
            */
            assertion = (report.FITR184.GetValueOrDefault() != report.FITR73.GetValueOrDefault() + report.FITR131.GetValueOrDefault() + report.FITR133.GetValueOrDefault() + report.FITR206.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434263", Severity = ProcessMessageSeverity.Error,
                    Description = @"Refundable tax offsets amount incorrect.",
                    LongDescription = @"Refundable tax offsets amount must be equal to sum of Complying fund's franking credits tax offset, No-TFN tax offset, National rental affordability scheme tax offset and Exploration credit tax offset.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TotalRefundableTaxOffsetsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434263"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR184", Value = GetValueOrEmpty(report.FITR184) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR73", Value = GetValueOrEmpty(report.FITR73) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR131", Value = GetValueOrEmpty(report.FITR131) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR133", Value = GetValueOrEmpty(report.FITR133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR206", Value = GetValueOrEmpty(report.FITR206) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434264
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434264()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434264
            If Non-refundable non-carry forward tax offsets amount is less than Gross tax, then Subtotal 1 amount must be equal to Gross tax minus Non-refundable non-carry forward tax offsets amount
    
            Legacy Rule Format:
            (^FITR189 < ^FITR66) AND (^FITR183 <> (^FITR66 - ^FITR189))

            Technical Business Rule Format:
            (^FITR189 < ^FITR66) AND (^FITR183 <> (^FITR66 - ^FITR189))
    
            Data Elements:
    
            ^FITR183 = FITR:RP:IncomeTaxCalculation:IncomeTax.SubtotalTaxPayable.Amount
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
    
            ^FITR189 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount
            */
            assertion = (report.FITR189.GetValueOrDefault() < report.FITR66.GetValueOrDefault() && report.FITR183.GetValueOrDefault() != report.FITR66.GetValueOrDefault() - report.FITR189.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434579", Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 1 amount incorrect",
                    LongDescription = @"If Non-refundable non-carry forward tax offsets amount is less than Gross tax, then Subtotal 1 amount must be equal to Gross tax minus Non-refundable non-carry forward tax offsets amount",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:SubtotalTaxPayableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434264"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR189", Value = GetValueOrEmpty(report.FITR189) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR183", Value = GetValueOrEmpty(report.FITR183) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434265
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434265()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434265
            If Non-refundable non-carry forward tax offsets amount is greater than or equal to Gross tax, then Subtotal 1 amount must be equal to zero
    
            Legacy Rule Format:
            (^FITR189 >= ^FITR66) AND (^FITR183 <> 0)

            Technical Business Rule Format:
            (^FITR189 >= ^FITR66) AND (^FITR183 <> 0)
    
            Data Elements:
    
            ^FITR183 = FITR:RP:IncomeTaxCalculation:IncomeTax.SubtotalTaxPayable.Amount
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
    
            ^FITR189 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount
            */
            assertion = (report.FITR189.GetValueOrDefault() >= report.FITR66.GetValueOrDefault() && report.FITR183 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434580", Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 1 amount must equal zero",
                    LongDescription = @"If Non-refundable non-carry forward tax offsets amount is greater than or equal to Gross tax, then Subtotal 1 amount must be equal to zero",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:SubtotalTaxPayableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434265"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR189", Value = GetValueOrEmpty(report.FITR189) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR183", Value = GetValueOrEmpty(report.FITR183) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434266
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434266()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434266
            Tax payable amount must be equal to zero if Subtotal 2 amount is zero
    
            Legacy Rule Format:
            ^FITR216 = 0 AND ^FITR199 <> 0

            Technical Business Rule Format:
            ^FITR216 = 0 AND ^FITR199 <> 0
    
            Data Elements:
    
            ^FITR199 = FITR:RP:IncomeTaxCalculation:IncomeTax.Payable.Amount
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.FITR216 == 0 && report.FITR199 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434583", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax payable amount must equal zero",
                    LongDescription = @"Tax payable amount must be equal to zero if Subtotal 2 amount is zero",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434266"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR199", Value = GetValueOrEmpty(report.FITR199) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434267
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434267()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434267
            If Refundable tax offsets amount is less than Subtotal 2 amount, then Tax payable amount must be equal to Subtotal 2 amount minus Refundable tax offsets amount
    
            Legacy Rule Format:
            (^FITR184 < ^FITR216) AND (^FITR199 <> (^FITR216 - ^FITR184))

            Technical Business Rule Format:
            (^FITR184 < ^FITR216) AND (^FITR199 <> (^FITR216 - ^FITR184))
    
            Data Elements:
    
            ^FITR199 = FITR:RP:IncomeTaxCalculation:IncomeTax.Payable.Amount
    
            ^FITR184 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.FITR184.GetValueOrDefault() < report.FITR216.GetValueOrDefault() && report.FITR199.GetValueOrDefault() != report.FITR216.GetValueOrDefault() - report.FITR184.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434584", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax payable amount incorrect",
                    LongDescription = @"If Refundable tax offsets amount is less than Subtotal 2 amount, then Tax payable amount must be equal to Subtotal 2 amount minus Refundable tax offsets amount",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434267"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR184", Value = GetValueOrEmpty(report.FITR184) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR199", Value = GetValueOrEmpty(report.FITR199) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434268
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434268()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434268
            If Refundable tax offsets amount is greater than or equal to Subtotal 2, then Tax payable amount must be equal to zero
    
            Legacy Rule Format:
            (^FITR184 >= ^FITR216) AND (^FITR199 <> 0)

            Technical Business Rule Format:
            (^FITR184 >= ^FITR216) AND (^FITR199 <> 0)
    
            Data Elements:
    
            ^FITR199 = FITR:RP:IncomeTaxCalculation:IncomeTax.Payable.Amount
    
            ^FITR184 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.FITR184.GetValueOrDefault() >= report.FITR216.GetValueOrDefault() && report.FITR199 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434585", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax payable amount must equal zero",
                    LongDescription = @"If Refundable tax offsets amount is greater than or equal to Subtotal 2, then Tax payable amount must be equal to zero",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434268"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR184", Value = GetValueOrEmpty(report.FITR184) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR199", Value = GetValueOrEmpty(report.FITR199) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434269
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434269()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434269
            If Refundable tax offsets amount is greater than or equal to Subtotal 2 amount, then Tax offset refunds (Remainder of refundable tax offsets) amount must be equal to Refundable tax offsets amount minus Subtotal 2 amount
    
            Legacy Rule Format:
            (^FITR184 >= ^FITR216) AND (^FITR185 <> (^FITR184 - ^FITR216))

            Technical Business Rule Format:
            (^FITR184 >= ^FITR216) AND (^FITR185 <> (^FITR184 - ^FITR216))
    
            Data Elements:
    
            ^FITR185 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
    
            ^FITR184 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.FITR184.GetValueOrDefault() >= report.FITR216.GetValueOrDefault() && report.FITR185.GetValueOrDefault() != report.FITR184.GetValueOrDefault() - report.FITR216.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434586", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax offset refunds (Remainder of refundable tax offsets) amount incorrect",
                    LongDescription = @"If Refundable tax offsets amount is greater than or equal to Subtotal 2 amount, then Tax offset refunds (Remainder of refundable tax offsets) amount must be equal to Refundable tax offsets amount minus Subtotal 2 amount",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TotalRemainderOfRefundableTaxOffsetsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434269"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR184", Value = GetValueOrEmpty(report.FITR184) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR185", Value = GetValueOrEmpty(report.FITR185) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434271
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434271()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434271
            Amount due or refundable must be equal to Tax payable plus Section 102AAM interest charge less Eligible credits, Tax offset refunds (Remainder of refundable tax offsets) and PAYG instalments raised
    
            Legacy Rule Format:
            ^FITR75 <> (^FITR199 + ^FITR69 - (^FITR201 + ^FITR185 + ^FITR74))

            Technical Business Rule Format:
            ^FITR75 <> (^FITR199 + ^FITR69 - (^FITR201 + ^FITR185 + ^FITR74))
    
            Data Elements:
    
            ^FITR75 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayableOrRefundableTotal.Amount
    
            ^FITR69 = FITR:RP:IncomeTaxCalculation:InternationalDealings.AAMInterestSection102.Amount
    
            ^FITR74 = FITR:RP:IncomeTaxCalculation:IncomeTax.LiabilityInstalmentsTotal.Amount
    
            ^FITR185 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
    
            ^FITR199 = FITR:RP:IncomeTaxCalculation:IncomeTax.Payable.Amount
    
            ^FITR201 = FITR:RP:IncomeTaxCalculation:IncomeTax.Deduction.CreditsEligibleTotal.Amount
            */
            assertion = (report.FITR75.GetValueOrDefault() != report.FITR199.GetValueOrDefault() + report.FITR69.GetValueOrDefault() - (report.FITR201.GetValueOrDefault() + report.FITR185.GetValueOrDefault() + report.FITR74.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434271", Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount due or refundable incorrect.",
                    LongDescription = @"Amount due or refundable must equal Tax Payable plus Section 102AAM interest charge less Eligible credits, Remainder of refundable tax offsets and PAYG instalments raised.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayableOrRefundableTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434271"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR75", Value = GetValueOrEmpty(report.FITR75) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR199", Value = GetValueOrEmpty(report.FITR199) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR69", Value = GetValueOrEmpty(report.FITR69) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR201", Value = GetValueOrEmpty(report.FITR201) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR185", Value = GetValueOrEmpty(report.FITR185) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR74", Value = GetValueOrEmpty(report.FITR74) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434286
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434286()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434286
            Exemption or rollover applied - Code must be unique
    
            Legacy Rule Format:
            HasDuplicateValues(^FITR204)

            Technical Business Rule Format:
            HasDuplicateValues(^FITR204)
    
            Data Elements:
    
            ^FITR204 = FITR:RP:Income:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = HasDuplicateValues(report.FITR204Collection);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434286", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code must be unique",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC" + OccurrenceIndex(DuplicateValueIndex(report.FITR204Collection) + 1) + "",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434286"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434292
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434292()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434292
            If Refundable tax offsets amount is less than Subtotal 2 amount, then Tax offset refunds (Remainder of refundable tax offsets) must be equal to zero
    
            Legacy Rule Format:
            ^FITR184 < ^FITR216 AND ^FITR185 > 0

            Technical Business Rule Format:
            ^FITR184 < ^FITR216 AND ^FITR185 > 0
    
            Data Elements:
    
            ^FITR185 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
    
            ^FITR184 = FITR:RP:IncomeTaxCalculation:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.FITR184.GetValueOrDefault() < report.FITR216.GetValueOrDefault() && report.FITR185.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434587", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax offset refunds (Remainder of refundable tax offsets) must be zero",
                    LongDescription = @"If Refundable tax offsets amount is less than Subtotal 2 amount, then Tax offset refunds (Remainder of refundable tax offsets) must be equal to zero",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TotalRemainderOfRefundableTaxOffsetsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434292"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR184", Value = GetValueOrEmpty(report.FITR184) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR185", Value = GetValueOrEmpty(report.FITR185) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434310
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434310()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434310
            Total Taxation of financial arrangements (TOFA) gains must be present
    
            Legacy Rule Format:
            (^FITR48 = 'T' AND (^FITR134 = NULL OR ^FITR134 = 0))

            Technical Business Rule Format:
            (^FITR48 = 'T' AND (^FITR134 = NULL OR ^FITR134 = 0))
    
            Data Elements:
    
            ^FITR134 = FITR:RP:OtherInformation:TaxationOfFinancialArrangements:Income.GainsTotal.Amount
    
            ^FITR48 = FITR:RP:Income:Income.Other.Code
            */
            assertion = (report.FITR48 == @"T" && (report.FITR134 == null || report.FITR134 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434310", Severity = ProcessMessageSeverity.Error,
                    Description = @"A TOFA gain must be present",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TaxationOfFinancialArrangements/tns:IncomeGainsTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434310"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR48", Value = GetValueOrEmpty(report.FITR48) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR134", Value = GetValueOrEmpty(report.FITR134) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434311
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434311()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434311
            Total Taxation of financial arrangements (TOFA) losses must be present
    
            Legacy Rule Format:
            (^FITR63 = 'T - TOFA losses' AND (^FITR135 = NULL OR ^FITR135 = 0))

            Technical Business Rule Format:
            (^FITR63 = 'T - TOFA losses' AND (^FITR135 = NULL OR ^FITR135 = 0))
    
            Data Elements:
    
            ^FITR135 = FITR:RP:OtherInformation:TaxationOfFinancialArrangements:Expense.LossesTotal.Amount
    
            ^FITR63 = FITR:RP:Deductions:Expense.DeductibleOther.Code
            */
            assertion = (report.FITR63 == @"T - TOFA losses" && (report.FITR135 == null || report.FITR135 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434311", Severity = ProcessMessageSeverity.Error,
                    Description = @"A TOFA loss must be present",
                    Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TaxationOfFinancialArrangements/tns:ExpenseLossesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434311"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR63", Value = GetValueOrEmpty(report.FITR63) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR135", Value = GetValueOrEmpty(report.FITR135) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434314
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434314()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434314
            If the fund is an Australian superannuation fund and the No-TFN-quoted contributions amount is greater than zero, then Tax on no-TFN-quoted contributions amount must be equal to No-TFN-quoted contributions multiplied by either 0.32 or 0.02
    
            Legacy Rule Format:
            ^FITR15 = TRUE AND  ^FITR43 > 0 AND (OutsideRange(^FITR181, ^FITR43 * 0.32, 1) AND OutsideRange(^FITR181, ^FITR43 * 0.02, 1))

            Technical Business Rule Format:
            ^FITR15 = TRUE AND  ^FITR43 > 0 AND (OutsideRange(^FITR181, ^FITR43 * 0.32, 1) AND OutsideRange(^FITR181, ^FITR43 * 0.02, 1))
    
            Data Elements:
    
            ^FITR181 = FITR:RP:IncomeTaxCalculation:IncomeTax.TFNNotQuotedContributionsTax.Amount
    
            ^FITR15 = FITR:RP:StatusOfFundOrTrust:SuperannuationFundDetails.AustralianSuperannuationFundStatus.Indicator
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
            */
            assertion = (report.FITR15 == true && report.FITR43.GetValueOrDefault() > 0 && (OutsideRange(report.FITR181.GetValueOrDefault(), report.FITR43.GetValueOrDefault() * 0.32M, 1) && OutsideRange(report.FITR181.GetValueOrDefault(), report.FITR43.GetValueOrDefault() * 0.02M, 1)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434314", Severity = ProcessMessageSeverity.Error,
                    Description = @"The amount at Tax on no-TFN-quoted contributions is incorrect",
                    LongDescription = @"A 'yes' response has been given to the Australian superannuation fund question however the No-TFN quoted contributions amount is greater than zero and the Tax on no-TFN contributions amount is not equal to No-TFN quoted contributions amount multiplied by 0.02 or 0.32",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TFNNotQuotedContributionsTaxA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434314"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR15", Value = GetValueOrEmpty(report.FITR15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR181", Value = GetValueOrEmpty(report.FITR181) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434315
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434315()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434315
            If the fund is not an Australian superannuation fund or if it is a Non-Regulated Fund, then Gross tax must not be less than 45% of Taxable income or loss
    
            Legacy Rule Format:
            ((^FITR15 = FALSE OR ^FITR14 = '138') AND ^FITR65 > 0 AND ^FITR66 < (^FITR65 * 0.45))

            Technical Business Rule Format:
            ((^FITR15 = FALSE OR ^FITR14 = '138') AND ^FITR65 > 0 AND ^FITR66 < (^FITR65 * 0.45))
    
            Data Elements:
    
            ^FITR65 = FITR:RP:Deductions:Income.Taxable.Amount
    
            ^FITR14 = FITR:RP:StatusOfFundOrTrust:Party.Type.Code
    
            ^FITR15 = FITR:RP:StatusOfFundOrTrust:SuperannuationFundDetails.AustralianSuperannuationFundStatus.Indicator
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
            */
            assertion = ((report.FITR15 == false || report.FITR14 == @"138") && report.FITR65.GetValueOrDefault() > 0 && report.FITR66.GetValueOrDefault() < report.FITR65.GetValueOrDefault() * 0.45M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434315", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross tax must not be less than 45% of Taxable income",
                    LongDescription = @"Gross tax amount must not be less than 45% of the Taxable income or loss amount if the Status of the fund is 'Non-regulated' (code 138) or if the fund is not an Australian superannuation fund",
                    Location = "/tns:FITR/tns:RP/tns:Deductions/tns:IncomeTaxableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434315"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR15", Value = GetValueOrEmpty(report.FITR15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR14", Value = report.FITR14 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR65", Value = GetValueOrEmpty(report.FITR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434316
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434316()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434316
            Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year
    
            Legacy Rule Format:
            (NotInSet(^FITR1, '"2025","2026"')) OR (^FITR1 = 2026 AND Today() > AsDate('2026-06-15') AND ^FITR171 <> TRUE)

            Technical Business Rule Format:
            (NotInSet(^FITR1, '"2025","2026"')) OR (^FITR1 = 2026 AND Today() > AsDate('2026-06-15') AND ^FITR171 <> TRUE)
    
            Data Elements:
    
            ^FITR1 = FITR:RP:Report.TargetFinancial.Year
    
            ^FITR171 = FITR:RP:AmendmentDetails:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.FITR1.GetValueOrDefault().ToString(), @"^(2026|2027)$",RegexOptions.IgnoreCase)) || report.FITR1.GetValueOrDefault() == 2027 && DateTime.Now.Date > AsDate(@"2026-06-15") && report.FITR171 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500127", Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.",
                    Location = "/tns:FITR/tns:RP/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434316"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR1", Value = GetValueOrEmpty(report.FITR1) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR171", Value = GetValueOrEmpty(report.FITR171) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434330
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434330()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434330
            Gross tax must be greater than zero if No-TFN-quoted contributions amount is greater than zero
    
            Legacy Rule Format:
            ^FITR43 > 0 AND ^FITR66 = 0

            Technical Business Rule Format:
            ^FITR43 > 0 AND ^FITR66 = 0
    
            Data Elements:
    
            ^FITR66 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxableGross.Amount
    
            ^FITR43 = FITR:RP:Income:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
            */
            assertion = (report.FITR43.GetValueOrDefault() > 0 && report.FITR66 == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434330", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross Tax must be greater than zero if 'No-TFN-quoted contributions' amount is greater than zero.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxableGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434330"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR43", Value = GetValueOrEmpty(report.FITR43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR66", Value = GetValueOrEmpty(report.FITR66) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434333
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434333()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434333
            The sum of Early stage investor tax offset and Early stage investor tax offset carried forward from previous year cannot exceed $200,000
    
            Legacy Rule Format:
            (^FITR214 <> NULL OR ^FITR218 <> NULL) AND (^FITR214 + ^FITR218 > 200000)

            Technical Business Rule Format:
            (^FITR214 <> NULL OR ^FITR218 <> NULL) AND (^FITR214 + ^FITR218 > 200000)
    
            Data Elements:
    
            ^FITR214 = FITR:RP:IncomeTaxCalculation:EarlyStageInvestor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^FITR218 = FITR:RP:IncomeTaxCalculation:PreviousYearEarlyStageInvestor:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = ((report.FITR214 != null || report.FITR218 != null) && report.FITR214.GetValueOrDefault() + report.FITR218.GetValueOrDefault() > 200000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434333", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Early stage investor tax offset is capped at $200,000 per investor and their affiliates combined each income year.",
                    LongDescription = @"The Early stage investor tax offset is capped at $200,000 per investor and their affiliates combined each income year. This $200,000 limit includes the sum of both the current year tax offset and the amount of early stage investor tax offset carried forward from the previous year.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:EarlyStageInvestorTaxOffsetClaimNonRefundableOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434333"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR214", Value = GetValueOrEmpty(report.FITR214) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR218", Value = GetValueOrEmpty(report.FITR218) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434335
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434335()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434335
            The Non-refundable carry forward tax offsets amount must be equal to the sum of Early stage venture capital limited partnership (ESVCLP) tax offset, Early stage investor tax offset, Early stage venture capital limited partnership (ESVCLP) tax offset carried forward from previous year and Early stage investor tax offset carried forward from previous year
    
            Legacy Rule Format:
            (^FITR213 <> NULL OR ^FITR214 <> NULL OR ^FITR217 <> NULL OR ^FITR218 <> NULL) AND (^FITR215 <> (^FITR213 + ^FITR214 + ^FITR217 + ^FITR218))

            Technical Business Rule Format:
            (^FITR213 <> NULL OR ^FITR214 <> NULL OR ^FITR217 <> NULL OR ^FITR218 <> NULL) AND (^FITR215 <> (^FITR213 + ^FITR214 + ^FITR217 + ^FITR218))
    
            Data Elements:
    
            ^FITR215 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
    
            ^FITR213 = FITR:RP:IncomeTaxCalculation:ESVCLP:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^FITR214 = FITR:RP:IncomeTaxCalculation:EarlyStageInvestor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^FITR217 = FITR:RP:IncomeTaxCalculation:PreviousYearESVCLP:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^FITR218 = FITR:RP:IncomeTaxCalculation:PreviousYearEarlyStageInvestor:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = ((report.FITR213 != null || report.FITR214 != null || report.FITR217 != null || report.FITR218 != null) && report.FITR215.GetValueOrDefault() != report.FITR213.GetValueOrDefault() + report.FITR214.GetValueOrDefault() + report.FITR217.GetValueOrDefault() + report.FITR218.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434335", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-refundable carry forward tax offsets amount incorrect.",
                    LongDescription = @"Non-refundable carry forward tax offsets amount must be equal to sum of Early stage Venture Capital Limited Partnership (ESVCLP) tax offset, Early stage venture capital limited partnership tax offset carried forward from previous year, Early stage investor tax offset, and Early stage investor tax offset carried forward from previous year.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxOffsetNonRefundableCarryForwardA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434335"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR213", Value = GetValueOrEmpty(report.FITR213) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR214", Value = GetValueOrEmpty(report.FITR214) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR217", Value = GetValueOrEmpty(report.FITR217) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR218", Value = GetValueOrEmpty(report.FITR218) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR215", Value = GetValueOrEmpty(report.FITR215) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434337
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434337()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434337
            If Non-refundable carry forward tax offset amount is less than Subtotal 1 amount, then Subtotal 2 amount must be equal to Subtotal 1 amount minus Non-refundable carry forward tax offsets amount
    
            Legacy Rule Format:
            (^FITR215 < ^FITR183) AND (^FITR216 <> (^FITR183 - ^FITR215))

            Technical Business Rule Format:
            (^FITR215 < ^FITR183) AND (^FITR216 <> (^FITR183 - ^FITR215))
    
            Data Elements:
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^FITR183 = FITR:RP:IncomeTaxCalculation:IncomeTax.SubtotalTaxPayable.Amount
    
            ^FITR215 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
            */
            assertion = (report.FITR215.GetValueOrDefault() < report.FITR183.GetValueOrDefault() && report.FITR216.GetValueOrDefault() != report.FITR183.GetValueOrDefault() - report.FITR215.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434581", Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 2 amount incorrect",
                    LongDescription = @"If Non-refundable carry forward tax offset amount is less than Subtotal 1 amount, then Subtotal 2 amount must be equal to Subtotal 1 amount minus Non-refundable carry forward tax offsets amount",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxOffsetCarriedForwardSubTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434337"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR215", Value = GetValueOrEmpty(report.FITR215) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR183", Value = GetValueOrEmpty(report.FITR183) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434338
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434338()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434338
            If Non-refundable carry forward tax offset amount is greater than or equal to Subtotal 1 amount, then Subtotal 2 amount must be equal to zero
    
            Legacy Rule Format:
            (^FITR215 >= ^FITR183) AND (^FITR216 <> 0)

            Technical Business Rule Format:
            (^FITR215 >= ^FITR183) AND (^FITR216 <> 0)
    
            Data Elements:
    
            ^FITR216 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^FITR183 = FITR:RP:IncomeTaxCalculation:IncomeTax.SubtotalTaxPayable.Amount
    
            ^FITR215 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
            */
            assertion = (report.FITR215.GetValueOrDefault() >= report.FITR183.GetValueOrDefault() && report.FITR216 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434582", Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 2 amount must equal zero",
                    LongDescription = @"If Non-refundable carry forward tax offset amount is greater than or equal to Subtotal 1 amount, then Subtotal 2 amount must be equal to zero",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:TaxOffsetCarriedForwardSubTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434338"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR215", Value = GetValueOrEmpty(report.FITR215) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR183", Value = GetValueOrEmpty(report.FITR183) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR216", Value = GetValueOrEmpty(report.FITR216) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434350
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434350()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434350
            If the answer to the Country-by-country reporting entity indicator is TRUE, then the Significant global entity indicator must be TRUE
    
            Legacy Rule Format:
            ^FITR212 <> TRUE AND ^FITR219 = TRUE

            Technical Business Rule Format:
            ^FITR212 <> TRUE AND ^FITR219 = TRUE
    
            Data Elements:
    
            ^FITR212 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
    
            ^FITR219 = FITR:RP:StatusOfFundOrTrust:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
            */
            assertion = (report.FITR212 != true && report.FITR219 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434350", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Significant Global Entity indicator must be 'yes' (true)",
                    LongDescription = @"If the answer to the Country by Country Reporting Entity indicator is 'yes' (true), then the Significant Global Entity indicator must be 'yes' (true)",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434350"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR212", Value = GetValueOrEmpty(report.FITR212) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR219", Value = GetValueOrEmpty(report.FITR219) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434500
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434500()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434500
            Reporting period start date must be prior to Reporting period end date
    
            Legacy Rule Format:
            ^FITR300 >= ^FITR301

            Technical Business Rule Format:
            ^FITR300 >= ^FITR301
    
            Data Elements:
    
            ^FITR300 = FITR:RP:Period.Start.Date
    
            ^FITR301 = FITR:RP:Period.End.Date
            */
            assertion = (report.FITR300.GetValueOrDefault() >= report.FITR301.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/tns:FITR/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434500"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR300", Value = GetValueOrEmptyDate(report.FITR300) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR301", Value = GetValueOrEmptyDate(report.FITR301) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434501
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434501()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434501
            Tax file number must pass the algorithm check
    
            Legacy Rule Format:
            (FailsTFNAlgorithm(^FITR5))

            Technical Business Rule Format:
            (FailsTFNAlgorithm(^FITR5))
    
            Data Elements:
    
            ^FITR5 = FITR:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = FailsTFNAlgorithm(report.FITR5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428016", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number has failed the algorithm check",
                    Location = "/tns:FITR/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434501"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR5", Value = report.FITR5 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434502
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434502()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434502
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FITR6, 'T/A ')) AND (EndsWithSet(^FITR6, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FITR6, 'T/A ')) AND (EndsWithSet(^FITR6, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FITR6 = FITR:RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FITR6, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.FITR6, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FITR/tns:RP/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434502"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR6", Value = report.FITR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434503
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434503()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434503
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (NotContainsSet(^FITR6, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (NotContainsSet(^FITR6, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FITR6 = FITR:RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = !(IsMatch(report.FITR6, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FITR/tns:RP/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434503"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR6", Value = report.FITR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434504
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434504()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434504
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FITR6, ' - ')

            Technical Business Rule Format:
            Contains(^FITR6, ' - ')
    
            Data Elements:
    
            ^FITR6 = FITR:RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR6 == null ? false : report.FITR6.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FITR/tns:RP/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434504"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR6", Value = report.FITR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434505
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434505()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434505
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FITR6, 'P/L')

            Technical Business Rule Format:
            Contains(^FITR6, 'P/L')
    
            Data Elements:
    
            ^FITR6 = FITR:RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR6 == null ? false : report.FITR6.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FITR/tns:RP/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434505"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR6", Value = report.FITR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434506
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434506()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434506
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR6, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR6, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR6 = FITR:RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FITR6, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:RP/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434506"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR6", Value = report.FITR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434507
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434507()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434507
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^FITR6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^FITR6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^FITR6 = FITR:RP:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FITR6, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FITR/tns:RP/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434507"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR6", Value = report.FITR6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434508
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434508()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434508
            ABN of reporting party must be provided when there is no intermediary
    
            Legacy Rule Format:
            ^FITR1027 = NULL AND ^FITR7 = NULL

            Technical Business Rule Format:
            ^FITR1027 = NULL AND ^FITR7 = NULL
    
            Data Elements:
    
            ^FITR7 = FITR:RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^FITR1027 = FITR:INT
            */
            assertion = (report.IntCollectionExists == false && report.FITR7 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432417", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN for reporting party must be provided when there is no intermediary",
                    Location = "/tns:FITR/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434508"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR7", Value = GetValueOrEmpty(report.FITR7) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434509
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434509()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434509
            Reporting party ABN must pass the ABN algorithm check
    
            Legacy Rule Format:
            (^FITR7 <> NULL) and (FailsABNAlgorithm(^FITR7))

            Technical Business Rule Format:
            (^FITR7 <> NULL) and (FailsABNAlgorithm(^FITR7))
    
            Data Elements:
    
            ^FITR7 = FITR:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.FITR7 != null && FailsABNAlgorithm(report.FITR7));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:FITR/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434509"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR7", Value = GetValueOrEmpty(report.FITR7) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434510
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434510()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434510
            If Amendment indicator is TRUE, then Amendment sequence number, Amendment reason and Amendment type code must be present
    
            Legacy Rule Format:
            ^FITR171 = TRUE AND (^FITR174 = NULL OR ^FITR172 = NULL OR ^FITR173 = NULL)

            Technical Business Rule Format:
            ^FITR171 = TRUE AND (^FITR174 = NULL OR ^FITR172 = NULL OR ^FITR173 = NULL)
    
            Data Elements:
    
            ^FITR171 = FITR:RP:AmendmentDetails:Report.Amendment.Indicator
    
            ^FITR172 = FITR:RP:AmendmentDetails:Report.AmendmentType.Code
    
            ^FITR173 = FITR:RP:AmendmentDetails:Report.AmendmentReason.Text
    
            ^FITR174 = FITR:RP:AmendmentDetails:Report.AmendmentSequence.Number
            */
            assertion = (report.FITR171 == true && (report.FITR174 == null || report.FITR172 == null || report.FITR173 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/tns:FITR/tns:RP/tns:AmendmentDetails/tns:I",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434510"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR171", Value = GetValueOrEmpty(report.FITR171) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR174", Value = GetValueOrEmpty(report.FITR174) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR172", Value = GetValueOrEmpty(report.FITR172) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR173", Value = GetValueOrEmpty(report.FITR173) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434511
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434511()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434511
            If Amendment sequence number or Amendment reason or Amendment type code is present, Amendment indicator must be TRUE
    
            Legacy Rule Format:
            ^FITR171 <> TRUE AND (^FITR174 <> NULL OR ^FITR172 <> NULL OR ^FITR173 <> NULL)

            Technical Business Rule Format:
            ^FITR171 <> TRUE AND (^FITR174 <> NULL OR ^FITR172 <> NULL OR ^FITR173 <> NULL)
    
            Data Elements:
    
            ^FITR171 = FITR:RP:AmendmentDetails:Report.Amendment.Indicator
    
            ^FITR172 = FITR:RP:AmendmentDetails:Report.AmendmentType.Code
    
            ^FITR173 = FITR:RP:AmendmentDetails:Report.AmendmentReason.Text
    
            ^FITR174 = FITR:RP:AmendmentDetails:Report.AmendmentSequence.Number
            */
            assertion = (report.FITR171 != true && (report.FITR174 != null || report.FITR172 != null || report.FITR173 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/tns:FITR/tns:RP/tns:AmendmentDetails/tns:I",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434511"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR171", Value = GetValueOrEmpty(report.FITR171) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR174", Value = GetValueOrEmpty(report.FITR174) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR172", Value = GetValueOrEmpty(report.FITR172) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR173", Value = GetValueOrEmpty(report.FITR173) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434512
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434512()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434512
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            (StartsWithSet(^FITR8,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FITR8, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (StartsWithSet(^FITR8,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^FITR8, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FITR8 = FITR:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (IsMatch(report.FITR8, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FITR8, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434512"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR8", Value = report.FITR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434513
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434513()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434513
            Address line 1 must not contain "AS ABOVE"
    
            Legacy Rule Format:
            (FoundSet(^FITR8, '"AS ABOVE"'))

            Technical Business Rule Format:
            (FoundSet(^FITR8, '"AS ABOVE"'))
    
            Data Elements:
    
            ^FITR8 = FITR:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.FITR8, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434513"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR8", Value = report.FITR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434514
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434514()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434514
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FITR8, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FITR8, 'UNKNOWN')
    
            Data Elements:
    
            ^FITR8 = FITR:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = (report.FITR8 == null ? false : report.FITR8.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434514"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR8", Value = report.FITR8 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434515
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434515()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434515
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            (^FITR146 <> NULL) AND (StartsWithSet(^FITR146,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FITR146, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            (^FITR146 <> NULL) AND (StartsWithSet(^FITR146,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^FITR146, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^FITR146 = FITR:RP:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (report.FITR146 != null && (IsMatch(report.FITR146, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.FITR146, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434515"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR146", Value = GetValueOrEmpty(report.FITR146) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434516
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434516()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434516
            Address line 2 must be present if Address line 1 contains 'C/-'
    
            Legacy Rule Format:
            ((Contains(^FITR8, 'C/-')) AND (^FITR146 = NULL))

            Technical Business Rule Format:
            ((Contains(^FITR8, 'C/-')) AND (^FITR146 = NULL))
    
            Data Elements:
    
            ^FITR146 = FITR:RP:CurrentPostalAddress:AddressDetails.Line2.Text
    
            ^FITR8 = FITR:RP:CurrentPostalAddress:AddressDetails.Line1.Text
            */
            assertion = ((report.FITR8 == null ? false : report.FITR8.ToUpperInvariant().Contains(@"C/-")) && report.FITR146 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434516"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR8", Value = report.FITR8 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR146", Value = GetValueOrEmpty(report.FITR146) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434517
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434517()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434517
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FITR146, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FITR146, 'UNKNOWN')
    
            Data Elements:
    
            ^FITR146 = FITR:RP:CurrentPostalAddress:AddressDetails.Line2.Text
            */
            assertion = (report.FITR146 == null ? false : report.FITR146.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLine2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434517"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR146", Value = GetValueOrEmpty(report.FITR146) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434518
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434518()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434518
            Suburb/Town cannot contain state code
    
            Legacy Rule Format:
            (FoundSet(^FITR9, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR9, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^FITR9 = FITR:RP:CurrentPostalAddress:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.FITR9, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434518"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR9", Value = report.FITR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434519
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434519()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434519
            Suburb/Town cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^FITR9, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^FITR9, 'UNKNOWN')
    
            Data Elements:
    
            ^FITR9 = FITR:RP:CurrentPostalAddress:AddressDetails.LocalityName.Text
            */
            assertion = (report.FITR9 == null ? false : report.FITR9.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434519"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR9", Value = report.FITR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434520
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434520()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434520
            Suburb/Town cannot contain numeric characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR9, '0-9'))

            Technical Business Rule Format:
            (ContainsSet(^FITR9, '0-9'))
    
            Data Elements:
    
            ^FITR9 = FITR:RP:CurrentPostalAddress:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.FITR9, @"[0-9]",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsLocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434520"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR9", Value = report.FITR9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434521
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434521()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434521
            Current postal address of superannuation fund or trust - Country code must be "au"
    
            Legacy Rule Format:
            ^FITR150 <>  "au"

            Technical Business Rule Format:
            ^FITR150 <> 'au'
    
            Data Elements:
    
            ^FITR150 = FITR:RP:CurrentPostalAddress:AddressDetails.Country.Code
            */
            assertion = (report.FITR150 != @"au");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434521", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be set to Australia (au)",
                    Location = "/tns:FITR/tns:RP/tns:CurrentPostalAddress/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434521"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR150", Value = report.FITR150 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434522
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434522()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434522
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FITR12, 'T/A ')) AND (EndsWithSet(^FITR12, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FITR12, 'T/A ')) AND (EndsWithSet(^FITR12, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FITR12 = FITR:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FITR12, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.FITR12, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434522"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR12", Value = report.FITR12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434523
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434523()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434523
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FITR12 <> NULL) AND (NotContainsSet(^FITR12, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FITR12 <> NULL) AND (NotContainsSet(^FITR12, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FITR12 = FITR:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR12 != null && !(IsMatch(report.FITR12, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434523"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR12", Value = report.FITR12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434524
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434524()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434524
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FITR12, ' - ')

            Technical Business Rule Format:
            Contains(^FITR12, ' - ')
    
            Data Elements:
    
            ^FITR12 = FITR:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR12 == null ? false : report.FITR12.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434524"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR12", Value = report.FITR12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434525
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434525()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434525
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FITR12, 'P/L')

            Technical Business Rule Format:
            Contains(^FITR12, 'P/L')
    
            Data Elements:
    
            ^FITR12 = FITR:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR12 == null ? false : report.FITR12.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434525"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR12", Value = report.FITR12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434526
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434526()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434526
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR12, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR12, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR12 = FITR:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FITR12, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434526"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR12", Value = report.FITR12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434527
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434527()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434527
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^FITR12, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^FITR12, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^FITR12 = FITR:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FITR12, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434527"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR12", Value = report.FITR12 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434528
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434528()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434528
            Non-individual trustee ABN must pass the ABN algorithm check
    
            Legacy Rule Format:
            (^FITR13 <> NULL) AND (FailsABNAlgorithm(^FITR13))

            Technical Business Rule Format:
            (^FITR13 <> NULL) AND (FailsABNAlgorithm(^FITR13))
    
            Data Elements:
    
            ^FITR13 = FITR:RP:TrusteeDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.FITR13 != null && FailsABNAlgorithm(report.FITR13));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:FITR/tns:RP/tns:TrusteeDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434528"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR13", Value = GetValueOrEmpty(report.FITR13) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434529
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434529()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434529
            BSB Number must be 6 digits and greater than 012000
    
            Legacy Rule Format:
            (^FITR176 <> NULL) AND ((^FITR176 < 012001) OR (^FITR176 > 999999))

            Technical Business Rule Format:
            (^FITR176 <> NULL) AND ((^FITR176 < 012001) OR (^FITR176 > 999999))
    
            Data Elements:
    
            ^FITR176 = FITR:RP:ElectronicFundTransfer:FinancialInstitutionAccount.BankStateBranch.Number
            */
            assertion = (report.FITR176 != null && (AsNumeric(report.FITR176) < 12001 || AsNumeric(report.FITR176) > 999999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500136", Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be greater than 012000.",
                    Location = "/tns:FITR/tns:RP/tns:ElectronicFundTransfer/tns:FinancialInstitutionAccountBankStateBranchN",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434529"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR176", Value = report.FITR176 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434530
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434530()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434530
            Account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; " ,
    
            Legacy Rule Format:
            ContainsSet(^FITR178, '"_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ","')

            Technical Business Rule Format:
            ContainsSet(^FITR178, '"_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ","')
    
            Data Elements:
    
            ^FITR178 = FITR:RP:ElectronicFundTransfer:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = IsMatch(report.FITR178, @"_|!|@|\$|%|\]|\[|\?|=|:|;|""|,",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432037", Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; "" ,",
                    Location = "/tns:FITR/tns:RP/tns:ElectronicFundTransfer/tns:FinancialInstitutionAccountFinancialInstitutionAccountNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434530"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR178", Value = report.FITR178 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434531
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434531()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434531
            If an exemption or rollover has been applied, then Exemption or rollover applied - Code(s) must be supplied
    
            Legacy Rule Format:
            (^FITR203 = TRUE) AND (Count(^FITR204) = 0)

            Technical Business Rule Format:
            (^FITR203 = TRUE) AND (Count(^FITR204) = 0)
    
            Data Elements:
    
            ^FITR203 = FITR:RP:Income:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^FITR204 = FITR:RP:Income:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (report.FITR203 == true && Count(report.FITR204Collection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438019", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code not supplied",
                    LongDescription = @"If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434531"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR203", Value = GetValueOrEmpty(report.FITR203) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434532
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434532()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434532
            If one or more Exemption or rollover applied - Code(s) is provided, then the exemption or rollover applied indicator must be TRUE
    
            Legacy Rule Format:
            (Count(^FITR204) > 0) AND (^FITR203 <> TRUE)

            Technical Business Rule Format:
            (Count(^FITR204) > 0) AND (^FITR203 <> TRUE)
    
            Data Elements:
    
            ^FITR203 = FITR:RP:Income:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^FITR204 = FITR:RP:Income:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (Count(report.FITR204Collection) > 0 && report.FITR203 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover applied indicator must be true",
                    LongDescription = @"If an exemption or rollover code(s) is provided then the exemption or rollover applied indicator must be true",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434532"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR203", Value = GetValueOrEmpty(report.FITR203) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434537
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434537()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434537
            Based on the return calculation details, a credit or refund is due. In order for the ATO to pay this credit or refund, bank account details must be provided
    
            Legacy Rule Format:
            (^FITR75 < 0) AND ^FITR1005 = NULL

            Technical Business Rule Format:
            (^FITR75 < 0) AND ^FITR1005 = NULL
    
            Data Elements:
    
            ^FITR75 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayableOrRefundableTotal.Amount
    
            ^FITR1005 = FITR:RP:ElectronicFundTransfer
            */
            assertion = (report.FITR75.GetValueOrDefault() < 0 && report.RP_ElectronicFundTransferCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432268", Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial institution details must be entered when a credit (refund) is due.",
                    LongDescription = @"Based on the return calculation details a credit or refund is due. In order for the ATO to pay this credit or refund, Australian financial institution account details must be provided. If account details are not available please phone the Tax Office on: For self lodgers 13 28 61 options 2 2 and for tax agents 13 72 86 options 1 3 1 2.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayableOrRefundableTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434537"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR75", Value = GetValueOrEmpty(report.FITR75) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434538
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434538()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434538
            Both, Taxable income or loss amount and Tax losses carried forward to later income years cannot be greater than zero
    
            Legacy Rule Format:
            ^FITR65 > 0 AND ^FITR76 > 0

            Technical Business Rule Format:
            ^FITR65 > 0 AND ^FITR76 > 0
    
            Data Elements:
    
            ^FITR76 = FITR:RP:Losses:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            ^FITR65 = FITR:RP:Deductions:Income.Taxable.Amount
            */
            assertion = (report.FITR65.GetValueOrDefault() > 0 && report.FITR76.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438034", Severity = ProcessMessageSeverity.Error,
                    Description = @"Both taxable income and tax losses carried forward to later income years are present.",
                    LongDescription = @"Tax losses carried forward to later income years cannot be present with a positive taxable income.",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:TaxCarriedForwardLaterIncomeYearsTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434538"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR65", Value = GetValueOrEmpty(report.FITR65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR76", Value = GetValueOrEmpty(report.FITR76) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434539
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434539(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434539
                    Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    (StartsWith(^FITR85, 'T/A ')) AND (EndsWithSet(^FITR85, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^FITR85, 'T/A ')) AND (EndsWithSet(^FITR85, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^FITR85 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (IsMatch(transferee.FITR85, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(transferee.FITR85, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434539"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR85", Value = transferee.FITR85 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434540
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434540(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434540
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    (^FITR85 <> NULL) AND (NotContainsSet(^FITR85, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^FITR85 <> NULL) AND (NotContainsSet(^FITR85, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^FITR85 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (transferee.FITR85 != null && !(IsMatch(transferee.FITR85, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434540"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR85", Value = transferee.FITR85 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434541
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434541(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434541
                    Non-individual name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    Contains(^FITR85, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^FITR85, ' - ')
            
                    Data Elements:
            
                    ^FITR85 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (transferee.FITR85 == null ? false : transferee.FITR85.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434541"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR85", Value = transferee.FITR85 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434542
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434542(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434542
                    Non-individual name cannot contain "P/L"
    
                    Legacy Rule Format:
                    Contains(^FITR85, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^FITR85, 'P/L')
            
                    Data Elements:
            
                    ^FITR85 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (transferee.FITR85 == null ? false : transferee.FITR85.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434542"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR85", Value = transferee.FITR85 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434543
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434543(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434543
                    Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^FITR85, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^FITR85, '"--","''","  "'))
            
                    Data Elements:
            
                    ^FITR85 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(transferee.FITR85, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434543"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR85", Value = transferee.FITR85 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434544
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434544(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434544
                    Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    (EndsWithSet(^FITR85, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
        
                    Technical Business Rule Format:
                    (EndsWithSet(^FITR85, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
            
                    Data Elements:
            
                    ^FITR85 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(transferee.FITR85, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:OrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434544"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR85", Value = transferee.FITR85 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434545
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434545(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434545
                    Transferee ABN must pass the ABN algorithm check
    
                    Legacy Rule Format:
                    (^FITR86 <> NULL) AND (FailsABNAlgorithm(^FITR86))
        
                    Technical Business Rule Format:
                    (^FITR86 <> NULL) AND (FailsABNAlgorithm(^FITR86))
            
                    Data Elements:
            
                    ^FITR86 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (transferee.FITR86 != null && FailsABNAlgorithm(transferee.FITR86));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                            Description = @"ABN is invalid",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:AustralianBusinessNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434545"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR86", Value = transferee.FITR86 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434546
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434546(FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.FITR.434546
                    If Transferee is present, Transferee amount must be greater than zero
    
                    Legacy Rule Format:
                    ^FITR1021 <> NULL AND ^FITR87 = 0
        
                    Technical Business Rule Format:
                    ^FITR1021 <> NULL AND ^FITR87 = 0
            
                    Data Elements:
            
                    ^FITR87 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee:IncomeTax.Superannuation.TransferredAssessable.Amount
            
                    ^FITR1021 = FITR:RP:OtherInformation:TransferOfLiabilities:Transferee
                    */
                    assertion = (report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionExists != false && transferee.FITR87 == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.FITR.434546", Severity = ProcessMessageSeverity.Error,
                            Description = @"Transferee amount must be greater than zero",
                            Location = "/tns:FITR/tns:RP/tns:OtherInformation/tns:TransferOfLiabilities/tns:TransfereeCollection/tns:Transferee" + OccurrenceIndex(transferee.OccurrenceIndex) + "/tns:IncomeTaxSuperannuationTransferredAssessableA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434546"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "FITR87", Value = GetValueOrEmpty(transferee.FITR87) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.FITR.434547
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434547()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434547
            If an Intermediary is not present, Reporting party declaration statement accepted indicator must be TRUE
    
            Legacy Rule Format:
            ^FITR1027 = NULL AND ^FITR162 <> TRUE

            Technical Business Rule Format:
            ^FITR1027 = NULL AND ^FITR162 <> TRUE
    
            Data Elements:
    
            ^FITR162 = FITR:RP:Declaration:Declaration.StatementAccepted.Indicator
    
            ^FITR1027 = FITR:INT
            */
            assertion = (report.IntCollectionExists == false && report.FITR162 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438064", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:FITR/tns:RP/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434547"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR162", Value = GetValueOrEmpty(report.FITR162) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434548
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434548()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434548
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^FITR123 > Today()

            Technical Business Rule Format:
            ^FITR123 > Today()
    
            Data Elements:
    
            ^FITR123 = FITR:RP:Declaration:Declaration.Signature.Date
            */
            assertion = (report.FITR123.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:FITR/tns:RP/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434548"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR123", Value = GetValueOrEmptyDate(report.FITR123) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434549
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434549()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434549
            Name field cannot contain 'space hyphen space'
    
            Legacy Rule Format:
            Contains(^FITR167, ' - ')

            Technical Business Rule Format:
            Contains(^FITR167, ' - ')
    
            Data Elements:
    
            ^FITR167 = FITR:RP:Declaration:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.FITR167 == null ? false : report.FITR167.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:FITR/tns:RP/tns:Declaration/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434549"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR167", Value = report.FITR167 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434550
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434550()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434550
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FITR103, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FITR103, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FITR103 = FITR:RP:ContactPersonDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR103, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434550"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR103", Value = report.FITR103 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434551
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434551()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434551
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FITR103, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR103, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FITR103 = FITR:RP:ContactPersonDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR103, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434551"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR103", Value = report.FITR103 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434552
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434552()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434552
            Family name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^FITR103, ' - ')

            Technical Business Rule Format:
            Contains(^FITR103, ' - ')
    
            Data Elements:
    
            ^FITR103 = FITR:RP:ContactPersonDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FITR103 == null ? false : report.FITR103.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434552"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR103", Value = report.FITR103 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434553
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434553()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434553
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FITR103, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FITR103, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FITR103 = FITR:RP:ContactPersonDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR103, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434553"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR103", Value = report.FITR103 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434554
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434554()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434554
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR103, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR103, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR103 = FITR:RP:ContactPersonDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR103, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434554"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR103", Value = report.FITR103 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434555
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434555()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434555
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (NotContainsSet(^FITR103, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (NotContainsSet(^FITR103, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR103 = FITR:RP:ContactPersonDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = !(IsMatch(report.FITR103, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434555"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR103", Value = report.FITR103 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434556
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434556()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434556
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FITR105, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FITR105, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR105, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434556"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434557
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434557()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434557
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR105, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR105, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR105, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434557"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434558
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434558()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434558
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FITR105, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR105, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR105, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434558"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434559
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434559()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434559
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FITR105, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FITR105, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR105, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434559"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434560
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434560()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434560
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FITR105, ' - ')

            Technical Business Rule Format:
            Contains(^FITR105, ' - ')
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FITR105 == null ? false : report.FITR105.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434560"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434561
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434561()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434561
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FITR106, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FITR106, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FITR106 = FITR:RP:ContactPersonDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR106, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434561"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR106", Value = GetValueOrEmpty(report.FITR106) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434562
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434562()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434562
            Given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (NotContainsSet(^FITR105, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (NotContainsSet(^FITR105, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = !(IsMatch(report.FITR105, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500143", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character.",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434562"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434563
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434563()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434563
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FITR106, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FITR106, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FITR106 = FITR:RP:ContactPersonDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR106, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434563"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR106", Value = GetValueOrEmpty(report.FITR106) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434564
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434564()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434564
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR106, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR106, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR106 = FITR:RP:ContactPersonDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR106, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434564"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR106", Value = GetValueOrEmpty(report.FITR106) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434565
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434565()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434565
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FITR106, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR106, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FITR106 = FITR:RP:ContactPersonDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR106, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434565"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR106", Value = GetValueOrEmpty(report.FITR106) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434566
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434566()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434566
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FITR106, ' - ')

            Technical Business Rule Format:
            Contains(^FITR106, ' - ')
    
            Data Elements:
    
            ^FITR106 = FITR:RP:ContactPersonDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FITR106 == null ? false : report.FITR106.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434566"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR106", Value = GetValueOrEmpty(report.FITR106) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434567
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434567()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434567
            Email address must not start or end with a dot (.)
    
            Legacy Rule Format:
            (StartsWith(^FITR109, '.')) OR (EndsWith(^FITR109, '.'))

            Technical Business Rule Format:
            (StartsWith(^FITR109, '.')) OR (EndsWith(^FITR109, '.'))
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (IsMatch(report.FITR109, @"^(\.)",RegexOptions.IgnoreCase) || IsMatch(report.FITR109, @"(\.)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430224", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with a dot (.)",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434567"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434568
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434568()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434568
            Email address must not contain consecutive dots (..)
    
            Legacy Rule Format:
            Contains(^FITR109, '..')

            Technical Business Rule Format:
            Contains(^FITR109, '..')
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FITR109 == null ? false : report.FITR109.ToUpperInvariant().Contains(@".."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430226", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one consecutive dots (..)",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434568"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434569
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434569()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434569
            Email address must not contain more than one @ symbol
    
            Legacy Rule Format:
            ContainsMultiple(^FITR109, '@')

            Technical Business Rule Format:
            ContainsMultiple(^FITR109, '@')
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = IsMatch(report.FITR109, @"@(.*)@",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430223", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one @ symbol",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434569"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434570
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434570()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434570
            Email address must not contain an invalid character
    
            Legacy Rule Format:
            ContainsSet(^FITR109, '"(", ")", "<", ">", ",", ";", ":", "\",  """, "]", "["')

            Technical Business Rule Format:
            ContainsSet(^FITR109, '"(", ")", "<", ">", ",", ";", ":", "\",  """, "]", "["')
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = IsMatch(report.FITR109, @"\(|\)|<|>|,|;|:|\\|""|\]|\[",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430229", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain an invalid character",
                    LongDescription = @"An email address must not contain any of the following invalid characters: ( ) < > , ; : \ "" [ ]",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434570"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434571
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434571()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434571
            Email address must contain an @ symbol
    
            Legacy Rule Format:
            ((^FITR109 <> NULL) AND (NotContains(^FITR109, '@')))

            Technical Business Rule Format:
            ((^FITR109 <> NULL) AND (NotContains(^FITR109, '@')))
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FITR109 != null && !(report.FITR109 == null ? true : report.FITR109.ToUpperInvariant().Contains(@"@")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430230", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain an @ symbol",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434571"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434572
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434572()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434572
            Email address must not contain a dot followed by an @ symbol (.@)
    
            Legacy Rule Format:
            Contains(^FITR109, '.@')

            Technical Business Rule Format:
            Contains(^FITR109, '.@')
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FITR109 == null ? false : report.FITR109.ToUpperInvariant().Contains(@".@"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430227", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a dot followed by a @ symbol (.@)",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434572"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434573
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434573()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434573
            Email address must not contain an @ symbol followed by a dot (@.)
    
            Legacy Rule Format:
            Contains(^FITR109, '@.')

            Technical Business Rule Format:
            Contains(^FITR109, '@.')
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FITR109 == null ? false : report.FITR109.ToUpperInvariant().Contains(@"@."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430228", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a @ symbol followed by a dot (@.)",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434573"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434574
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434574()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434574
            Email address must contain a dot (.)
    
            Legacy Rule Format:
            ((^FITR109 <> NULL) AND (NotContains(^FITR109, '.')))

            Technical Business Rule Format:
            ((^FITR109 <> NULL) AND (NotContains(^FITR109, '.')))
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FITR109 != null && !(report.FITR109 == null ? true : report.FITR109.ToUpperInvariant().Contains(@".")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430232", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain a dot (.)",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434574"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434575
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434575()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434575
            Email address must not start or end with an @ symbol
    
            Legacy Rule Format:
            (StartsWith(^FITR109, '@')) OR (EndsWith(^FITR109, '@'))

            Technical Business Rule Format:
            (StartsWith(^FITR109, '@')) OR (EndsWith(^FITR109, '@'))
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (IsMatch(report.FITR109, @"^(@)",RegexOptions.IgnoreCase) || IsMatch(report.FITR109, @"(@)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430222", Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with an @ symbol",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434575"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434576
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434576()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434576
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FITR106 <> NULL) AND (NotContainsSet(^FITR106, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FITR106 <> NULL) AND (NotContainsSet(^FITR106, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR106 = FITR:RP:ContactPersonDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FITR106 != null && !(IsMatch(report.FITR106, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434576"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR106", Value = GetValueOrEmpty(report.FITR106) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434577
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434577()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434577
            Email address must not start with a space or include a space within it
    
            Legacy Rule Format:
            Contains(^FITR109, ' ')

            Technical Business Rule Format:
            Contains(^FITR109, ' ')
    
            Data Elements:
    
            ^FITR109 = FITR:RP:ContactPersonDetails:ElectronicContact.ElectronicMail.Address.Text
            */
            assertion = (report.FITR109 == null ? false : report.FITR109.ToUpperInvariant().Contains(@" "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430231", Severity = ProcessMessageSeverity.Error,
                    Description = @"An email address must not start with a space or include a space within it",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:ElectronicContactElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434577"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR109", Value = GetValueOrEmpty(report.FITR109) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434578
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434578()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434578
            Reporting party declaration must be present in the absence of an Intermediary
    
            Legacy Rule Format:
            ^FITR1027 = NULL AND ^FITR1025 = NULL

            Technical Business Rule Format:
            ^FITR1027 = NULL AND ^FITR1025 = NULL
    
            Data Elements:
    
            ^FITR1027 = FITR:INT
    
            ^FITR1025 = FITR:RP:Declaration
            */
            assertion = (report.IntCollectionExists == false && report.RP_DeclarationCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434578", Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting party declaration must be present in the absence of an Intermediary",
                    Location = "/tns:FITR/tns:INT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434578"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434579
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434579()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434579
            Tax Agent number must pass the Tax Agent Number algorithm check
    
            Legacy Rule Format:
            (^FITR2 <> NULL) AND (FailsTANAlgorithm(^FITR2))

            Technical Business Rule Format:
            (^FITR2 <> NULL) AND (FailsTANAlgorithm(^FITR2))
    
            Data Elements:
    
            ^FITR2 = FITR:INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (report.FITR2 != null && FailsTANAlgorithm(report.FITR2));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/tns:FITR/tns:INT/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434579"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR2", Value = GetValueOrEmpty(report.FITR2) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434580
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434580()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434580
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FITR112, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FITR112, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FITR112 = FITR:INT:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR112, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434580"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR112", Value = GetValueOrEmpty(report.FITR112) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434581
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434581()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434581
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FITR112, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR112, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FITR112 = FITR:INT:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR112, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434581"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR112", Value = GetValueOrEmpty(report.FITR112) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434582
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434582()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434582
            Family name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^FITR112, ' - ')

            Technical Business Rule Format:
            Contains(^FITR112, ' - ')
    
            Data Elements:
    
            ^FITR112 = FITR:INT:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FITR112 == null ? false : report.FITR112.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434582"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR112", Value = GetValueOrEmpty(report.FITR112) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434583
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434583()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434583
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FITR112, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FITR112, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FITR112 = FITR:INT:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR112, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434583"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR112", Value = GetValueOrEmpty(report.FITR112) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434584
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434584()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434584
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR112, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR112, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR112 = FITR:INT:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.FITR112, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434584"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR112", Value = GetValueOrEmpty(report.FITR112) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434585
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434585()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434585
            An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'
    
            Legacy Rule Format:
            (^FITR114 <> NULL) AND (NOT StartsWithSet(^FITR114,'"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FITR114 <> NULL) AND (NOT StartsWithSet(^FITR114,'"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FITR114 != null &&  !(IsMatch(report.FITR114, @"^([A-Z]|[a-z])",RegexOptions.IgnoreCase) ));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500138", Severity = ProcessMessageSeverity.Error,
                    Description = @"An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'.",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434585"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434586
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434586()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434586
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FITR114, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FITR114, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR114, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434586"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434587
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434587()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434587
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR114, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR114, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR114, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434587"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434588
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434588()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434588
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FITR114, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR114, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR114, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434588"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434589
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434589()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434589
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FITR112 <> NULL) AND (NotContainsSet(^FITR112, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FITR112 <> NULL) AND (NotContainsSet(^FITR112, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR112 = FITR:INT:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.FITR112 != null && !(IsMatch(report.FITR112, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434589"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR112", Value = GetValueOrEmpty(report.FITR112) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434590
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434590()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434590
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FITR114, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FITR114, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.FITR114, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434590"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434591
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434591()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434591
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FITR114, ' - ')

            Technical Business Rule Format:
            Contains(^FITR114, ' - ')
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FITR114 == null ? false : report.FITR114.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434591"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434592
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434592()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434592
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^FITR115, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^FITR115, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^FITR115 = FITR:INT:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR115, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434592"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR115", Value = GetValueOrEmpty(report.FITR115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434593
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434593()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434593
            Given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FITR114 <> NULL) AND (NotContainsSet(^FITR114, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FITR114 <> NULL) AND (NotContainsSet(^FITR114, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR114 = FITR:INT:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FITR114 != null && !(IsMatch(report.FITR114, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500143", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character.",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434593"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR114", Value = GetValueOrEmpty(report.FITR114) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434594
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434594()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434594
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^FITR115, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^FITR115, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^FITR115 = FITR:INT:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR115, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434594"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR115", Value = GetValueOrEmpty(report.FITR115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434595
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434595()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434595
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR115, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR115, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR115 = FITR:INT:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR115, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434595"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR115", Value = GetValueOrEmpty(report.FITR115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434596
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434596()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434596
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^FITR115, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^FITR115, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^FITR115 = FITR:INT:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.FITR115, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434596"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR115", Value = GetValueOrEmpty(report.FITR115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434597
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434597()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434597
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^FITR115, ' - ')

            Technical Business Rule Format:
            Contains(^FITR115, ' - ')
    
            Data Elements:
    
            ^FITR115 = FITR:INT:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FITR115 == null ? false : report.FITR115.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434597"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR115", Value = GetValueOrEmpty(report.FITR115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434598
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434598()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434598
            Non-individual name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^FITR116, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^FITR116, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^FITR116 = FITR:INT:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FITR116, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:FITR/tns:INT/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434598"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR116", Value = GetValueOrEmpty(report.FITR116) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434599
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434599()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434599
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            (StartsWith(^FITR116, 'T/A ')) AND (EndsWithSet(^FITR116, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^FITR116, 'T/A ')) AND (EndsWithSet(^FITR116, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^FITR116 = FITR:INT:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.FITR116, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.FITR116, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:FITR/tns:INT/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434599"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR116", Value = GetValueOrEmpty(report.FITR116) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434600
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434600()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434600
            Non-individual name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^FITR116, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^FITR116, '"--","''","  "'))
    
            Data Elements:
    
            ^FITR116 = FITR:INT:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.FITR116, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:FITR/tns:INT/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434600"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR116", Value = GetValueOrEmpty(report.FITR116) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434601
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434601()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434601
            Other given name must contain at least one alphabetical character
    
            Legacy Rule Format:
            (^FITR115 <> NULL) AND (NotContainsSet(^FITR115, '"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FITR115 <> NULL) AND (NotContainsSet(^FITR115, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR115 = FITR:INT:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.FITR115 != null && !(IsMatch(report.FITR115, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:FITR/tns:INT/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434601"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR115", Value = GetValueOrEmpty(report.FITR115) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434602
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434602()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434602
            Non-individual name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^FITR116, 'P/L')

            Technical Business Rule Format:
            Contains(^FITR116, 'P/L')
    
            Data Elements:
    
            ^FITR116 = FITR:INT:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR116 == null ? false : report.FITR116.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:FITR/tns:INT/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434602"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR116", Value = GetValueOrEmpty(report.FITR116) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434603
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434603()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434603
            Non-individual name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^FITR116, ' - ')

            Technical Business Rule Format:
            Contains(^FITR116, ' - ')
    
            Data Elements:
    
            ^FITR116 = FITR:INT:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR116 == null ? false : report.FITR116.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:FITR/tns:INT/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434603"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR116", Value = GetValueOrEmpty(report.FITR116) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434604
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434604()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434604
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            (^FITR116 <> NULL) AND (NotContainsSet(^FITR116, '"a-z", "A-Z", "0-9"'))

            Technical Business Rule Format:
            (^FITR116 <> NULL) AND (NotContainsSet(^FITR116, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^FITR116 = FITR:INT:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.FITR116 != null && !(IsMatch(report.FITR116, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:FITR/tns:INT/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434604"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR116", Value = GetValueOrEmpty(report.FITR116) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434605
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434605()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434605
            If an Intermediary is present, Intermediary declaration statement accepted indicator must be TRUE
    
            Legacy Rule Format:
            ^FITR1027 <> NULL AND ^FITR165 <> TRUE

            Technical Business Rule Format:
            ^FITR1027 <> NULL AND ^FITR165 <> TRUE
    
            Data Elements:
    
            ^FITR165 = FITR:INT:Declaration:Declaration.StatementAccepted.Indicator
    
            ^FITR1027 = FITR:INT
            */
            assertion = (report.IntCollectionExists != false && report.FITR165 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:FITR/tns:INT/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434605"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR165", Value = GetValueOrEmpty(report.FITR165) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434606
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434606()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434606
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^FITR125 > Today()

            Technical Business Rule Format:
            ^FITR125 > Today()
    
            Data Elements:
    
            ^FITR125 = FITR:INT:Declaration:Declaration.Signature.Date
            */
            assertion = (report.FITR125.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:FITR/tns:INT/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434606"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR125", Value = GetValueOrEmptyDate(report.FITR125) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434607
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434607()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434607
            Name field cannot contain 'space hyphen space'
    
            Legacy Rule Format:
            Contains(^FITR169, ' - ')

            Technical Business Rule Format:
            Contains(^FITR169, ' - ')
    
            Data Elements:
    
            ^FITR169 = FITR:INT:Declaration:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.FITR169 == null ? false : report.FITR169.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:FITR/tns:INT/tns:Declaration/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434607"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR169", Value = report.FITR169 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434608
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434608()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434608
            An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'
    
            Legacy Rule Format:
            (^FITR105 <> NULL) AND (NOT StartsWithSet(^FITR105,'"A-Z","a-z"'))

            Technical Business Rule Format:
            (^FITR105 <> NULL) AND (NOT StartsWithSet(^FITR105,'"A-Z","a-z"'))
    
            Data Elements:
    
            ^FITR105 = FITR:RP:ContactPersonDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.FITR105 != null &&  !(IsMatch(report.FITR105, @"^([A-Z]|[a-z])",RegexOptions.IgnoreCase) ));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500138", Severity = ProcessMessageSeverity.Error,
                    Description = @"An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'.",
                    Location = "/tns:FITR/tns:RP/tns:ContactPersonDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434608"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR105", Value = report.FITR105 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434609
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434609()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434609
            Intermediary Australian Business Number must pass the ABN algorithm check
    
            Legacy Rule Format:
            (^FITR302 <> NULL) AND (FailsABNAlgorithm(^FITR302))

            Technical Business Rule Format:
            (^FITR302 <> NULL) AND (FailsABNAlgorithm(^FITR302))
    
            Data Elements:
    
            ^FITR302 = FITR:INT:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.FITR302 != null && FailsABNAlgorithm(report.FITR302));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:FITR/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434609"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR302", Value = report.FITR302 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.434610
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITR434610()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.434610
            Eligible credits must be equal to the sum of Credit for tax withheld - foreign resident withholding (excluding capital gains), Credit for tax withheld - where ABN or TFN not quoted (non-individual), Credit for TFN amounts withheld from payments from closely held trusts, Credit for interest on no-TFN tax offset and Credit for foreign resident capital gains withholding amounts
    
            Legacy Rule Format:
            ^FITR201 <> (^FITR71 + ^FITR72 + ^FITR175 + ^FITR132 + ^FITR209)

            Technical Business Rule Format:
            ^FITR201 <> (^FITR71 + ^FITR72 + ^FITR175 + ^FITR132 + ^FITR209)
    
            Data Elements:
    
            ^FITR201 = FITR:RP:IncomeTaxCalculation:IncomeTax.Deduction.CreditsEligibleTotal.Amount
    
            ^FITR71 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^FITR72 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount
    
            ^FITR132 = FITR:RP:IncomeTaxCalculation:IncomeTax.TaxOffsetCalculatedInterestTFNNotQuoted.Amount
    
            ^FITR175 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectly.Amount
    
            ^FITR209 = FITR:RP:IncomeTaxCalculation:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            */
            assertion = (report.FITR201.GetValueOrDefault() != report.FITR71.GetValueOrDefault() + report.FITR72.GetValueOrDefault() + report.FITR175.GetValueOrDefault() + report.FITR132.GetValueOrDefault() + report.FITR209.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434610", Severity = ProcessMessageSeverity.Error,
                    Description = @"Eligible credits amount incorrect.",
                    LongDescription = @"Eligible credits must equal the sum of Credit for tax withheld - foreign resident withholding (excluding capital gains), Credit for tax withheld - where ABN or TFN not quoted (non-individual), Credit for TFN amounts withheld from payments from closely held trusts, Credit for interest on no-TFN tax offset and Credit for amounts withheld from foreign resident capital gains withholding.",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:DeductionCreditsEligibleTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434610"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR201", Value = GetValueOrEmpty(report.FITR201) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR71", Value = GetValueOrEmpty(report.FITR71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR72", Value = GetValueOrEmpty(report.FITR72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR175", Value = GetValueOrEmpty(report.FITR175) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR132", Value = GetValueOrEmpty(report.FITR132) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR209", Value = GetValueOrEmpty(report.FITR209) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.W00001
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITRW00001()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.W00001
            The answer to the Significant global entity indicator is TRUE and Country-by-country reporting entity indicator has not been provided. Check and correct if necessary
    
            Legacy Rule Format:
            ^FITR219 = NULL AND ^FITR212 = TRUE

            Technical Business Rule Format:
            ^FITR219 = NULL AND ^FITR212 = TRUE
    
            Data Elements:
    
            ^FITR219 = FITR:RP:StatusOfFundOrTrust:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^FITR212 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.FITR219 == null && report.FITR212 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.W00001", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check Country by Country Reporting Entity indicator",
                    LongDescription = @"The answer to the Significant Global Entity indicator is 'yes' (true) and Country by Country Reporting Entity indicator has not been provided. Check and correct if necessary.",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:InternationalDealingsCountryByCountryReportingEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.W00001"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR219", Value = GetValueOrEmpty(report.FITR219) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR212", Value = GetValueOrEmpty(report.FITR212) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.FITR.W00003
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOFITRW00003()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.FITR.W00003
            Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
            - a global parent entity (GPE) with an annual global income of A$1 billion or more
            - a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
            - a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.
            
            If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete label N2, and may have additional reporting obligations. 
            
            For more information on SGEs and CBC reporting, visit our website at ato.gov.au.
    
            Legacy Rule Format:
            ^FITR212 = TRUE

            Technical Business Rule Format:
            ^FITR212 = TRUE
    
            Data Elements:
    
            ^FITR212 = FITR:RP:StatusOfFundOrTrust:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.FITR212 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.W00003", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Confirm whether taxpayer is a Significant Global Entity (SGE)",
                    LongDescription = @"Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
- a global parent entity (GPE) with an annual global income of A$1 billion or more
- a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
- a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.

If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete label N2, and may have additional reporting obligations. 

For more information on SGEs and CBC reporting, visit our website at ato.gov.au.",
                    Location = "/tns:FITR/tns:RP/tns:StatusOfFundOrTrust/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.FITR.W00003"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "FITR212", Value = GetValueOrEmpty(report.FITR212) });
    
                response.Add(processMessage);
            }
      }
      #endregion 

    }
} 
