using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationCGTS
{

    public partial class CGTS2018Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="CGTS2018Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CGTS2018Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }
				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public CGTS2018 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CGTS2018 report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            #region VR.ATO.CGTS.402132
    
            /*  VR.ATO.CGTS.402132
            Invalid context. The number of reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT RP <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context)<> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402132"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402132
    
            #region VR.ATO.CGTS.402137
    
            /*  VR.ATO.CGTS.402137
            The earnout arrangement group number must be unique
    
            Legacy Rule Format:
            WHERE IN SET {RP.EarnoutArrangementGroupNumber}
            IF
            entity.segment.RP.EarnoutArrgmtGrpItmDim.EarnoutArrangementGroupItemContainer = 
            ANY OTHER OCCURRENCE OF
            entity:segment:EarnoutArrgmtGrpItmDim.EarnoutArrangementGroupItemContainer
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^CGTS160)
    
            Data Elements:
    
            RP.{EarnoutArrangementGroupNumber}:^CGTS160 = tns:Report.ItemOrder.Number
            */
            assertion = HasDuplicateValues(report.RPEarnoutArrangementGroupNumberContextCollection == null ? null : report.RPEarnoutArrangementGroupNumberContextCollection.Select(f => f.CGTS160).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402137", Severity = ProcessMessageSeverity.Error,
                    Description = @"The earnout arrangement report item order number must be unique",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.RPEarnoutArrangementGroupNumberContextCollection[DuplicateValueIndex(report.RPEarnoutArrangementGroupNumberContextCollection.Select(f => f.CGTS160))].Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402137"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402137
    
            #region VR.ATO.CGTS.402138
    
            /*  VR.ATO.CGTS.402138
            The number of Company Shares contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.ListedCompanyShares)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.ListedCompanyShares
            */
            assertion = (Count(report.RPListedCompanySharesCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPListedCompanySharesLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402138"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402138
    
            #region VR.ATO.CGTS.402139
    
            /*  VR.ATO.CGTS.402139
            The number of Other Shares contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.OtherShares)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.OtherShares
            */
            assertion = (Count(report.RPOtherSharesCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherSharesLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402139"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402139
    
            #region VR.ATO.CGTS.402140
    
            /*  VR.ATO.CGTS.402140
            The number of Listed Unit Trusts contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.ListedUnitTrusts)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.ListedUnitTrusts
            */
            assertion = (Count(report.RPListedUnitTrustsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPListedUnitTrustsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402140"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402140
    
            #region VR.ATO.CGTS.402141
    
            /*  VR.ATO.CGTS.402141
            The number of Other Units contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.OtherUnits)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.OtherUnits
            */
            assertion = (Count(report.RPOtherUnitsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherUnitsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402141"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402141
    
            #region VR.ATO.CGTS.402142
    
            /*  VR.ATO.CGTS.402142
            The number of Australian Real Property contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.AustralianRealProperty)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.AustralianRealProperty
            */
            assertion = (Count(report.RPAustralianRealPropertyCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAustralianRealPropertyLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402142"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402142
    
            #region VR.ATO.CGTS.402143
    
            /*  VR.ATO.CGTS.402143
            The number of Overseas Real Property contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.OverseasRealProperty)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.OverseasRealProperty
            */
            assertion = (Count(report.RPOverseasRealPropertyCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOverseasRealPropertyLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402143"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402143
    
            #region VR.ATO.CGTS.402144
    
            /*  VR.ATO.CGTS.402144
            The number of Capital Gains Trust Distributions contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.CapitalGainsTrustDistributions)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.CapitalGainsTrustDistributions
            */
            assertion = (Count(report.RPCapitalGainsTrustDistributionsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCapitalGainsTrustDistributionsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402144"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402144
    
            #region VR.ATO.CGTS.402145
    
            /*  VR.ATO.CGTS.402145
            The number of Collectables contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.Collectables)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Collectables
            */
            assertion = (Count(report.RPCollectablesCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCollectablesLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402145"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402145
    
            #region VR.ATO.CGTS.402147
    
            /*  VR.ATO.CGTS.402147
            The number of applied contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.Applied)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Applied
            */
            assertion = (Count(report.RPAppliedCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402147"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402147
    
            #region VR.ATO.CGTS.402148
    
            /*  VR.ATO.CGTS.402148
            The number of brought forward from previous years (Y0-) contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.Applied.Y0-)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Applied:RP.Applied.Y0-
            */
            assertion = (Count(report.RPAppliedY0MinusCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedY0MinusLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402148"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402148
    
            #region VR.ATO.CGTS.402149
    
            /*  VR.ATO.CGTS.402149
            The number of collectables unapplied contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.Collectables.Unapplied(Instant))) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Collectables:RP.Collectables.Unapplied(Instant)
            */
            assertion = (Count(report.RPCollectablesUnappliedInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCollectablesUnappliedInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402149"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402149
    
            #region VR.ATO.CGTS.402150
    
            /*  VR.ATO.CGTS.402150
            The number of CGT Other unapplied contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.CGTOther.Unapplied(Instant))) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.CGTOther.Unapplied(Instant)
            */
            assertion = (Count(report.RPCGTOtherUnappliedInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCGTOtherUnappliedInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402150"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402150
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS113, 'U', 11, 0)
    
            Data Elements:
    
            RP.CapitalGainsTrustDistributions:^CGTS113 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS113, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPCapitalGainsTrustDistributionsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS113", Value = report.CGTS113.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS114, 'U', 11, 0)
    
            Data Elements:
    
            RP.Collectables:^CGTS114 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS114, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPCollectablesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS114", Value = report.CGTS114.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS116, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherCapitalAssets:^CGTS116 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS116, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPOtherCapitalAssetsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS116", Value = report.CGTS116.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS111, 'U', 11, 0)
    
            Data Elements:
    
            RP.OverseasRealProperty:^CGTS111 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS111, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPOverseasRealPropertyId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS111", Value = report.CGTS111.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS97, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherShares:^CGTS97 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS97, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPOtherSharesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS97", Value = report.CGTS97.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS101, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherUnits:^CGTS101 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS101, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPOtherUnitsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS101", Value = report.CGTS101.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS103, 'U', 11, 0)
    
            Data Elements:
    
            RP.AustralianRealProperty:^CGTS103 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS103, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPAustralianRealPropertyId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS103", Value = report.CGTS103.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS95, 'U', 11, 0)
    
            Data Elements:
    
            RP.ListedCompanyShares:^CGTS95 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS95, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPListedCompanySharesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS95", Value = report.CGTS95.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS118, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS118 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS118, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS118", Value = report.CGTS118.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402155
    
            /*  VR.ATO.CGTS.402155
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Gains.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Gains.Total.Amount <> MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS99, 'U', 11, 0)
    
            Data Elements:
    
            RP.ListedUnitTrusts:^CGTS99 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS99, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPListedUnitTrustsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402155"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS99", Value = report.CGTS99.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402155
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS115, 'U', 11, 0)
    
            Data Elements:
    
            RP.Collectables:^CGTS115 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS115, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPCollectablesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS115", Value = report.CGTS115.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS117, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherCapitalAssets:^CGTS117 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS117, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPOtherCapitalAssetsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS117", Value = report.CGTS117.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS112, 'U', 11, 0)
    
            Data Elements:
    
            RP.OverseasRealProperty:^CGTS112 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS112, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPOverseasRealPropertyId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS112", Value = report.CGTS112.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS98, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherShares:^CGTS98 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS98, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPOtherSharesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS98", Value = report.CGTS98.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS102, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherUnits:^CGTS102 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS102, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPOtherUnitsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS102", Value = report.CGTS102.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS110, 'U', 11, 0)
    
            Data Elements:
    
            RP.AustralianRealProperty:^CGTS110 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS110, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPAustralianRealPropertyId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS110", Value = report.CGTS110.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS96, 'U', 11, 0)
    
            Data Elements:
    
            RP.ListedCompanyShares:^CGTS96 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS96, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPListedCompanySharesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS96", Value = report.CGTS96.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402156
    
            /*  VR.ATO.CGTS.402156
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount
            <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS100, 'U', 11, 0)
    
            Data Elements:
    
            RP.ListedUnitTrusts:^CGTS100 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS100, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPListedUnitTrustsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402156"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS100", Value = report.CGTS100.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402156
    
            #region VR.ATO.CGTS.402157
    
            /*  VR.ATO.CGTS.402157
            Total current year capital gains must equal the sum of each unique capital gain  reported
    
            Legacy Rule Format:
            IF ([CGTS118]) <> ([CGTS95]+[CGTS97]+[CGTS99]+[CGTS101]+[CGTS103]+[CGTS111]+[CGTS113]+[CGTS114]+[CGTS116]+[CGTS163])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS118 <> (^CGTS95+^CGTS97+^CGTS99+^CGTS101+^CGTS103+^CGTS111+^CGTS113+^CGTS114+^CGTS116+^CGTS163)
    
            Data Elements:
    
            RP:^CGTS118 = tns:Capital.Gains.Total.Amount
    
            RP.ListedCompanyShares:^CGTS95 = tns:Capital.Gains.Total.Amount
    
            RP.OtherShares:^CGTS97 = tns:Capital.Gains.Total.Amount
    
            RP.ListedUnitTrusts:^CGTS99 = tns:Capital.Gains.Total.Amount
    
            RP.OtherUnits:^CGTS101 = tns:Capital.Gains.Total.Amount
    
            RP.AustralianRealProperty:^CGTS103 = tns:Capital.Gains.Total.Amount
    
            RP.OverseasRealProperty:^CGTS111 = tns:Capital.Gains.Total.Amount
    
            RP.CapitalGainsTrustDistributions:^CGTS113 = tns:Capital.Gains.Total.Amount
    
            RP.Collectables:^CGTS114 = tns:Capital.Gains.Total.Amount
    
            RP.OtherCapitalAssets:^CGTS116 = tns:Capital.Gains.Total.Amount
    
            RP.OtherCapitalAssets:^CGTS163 = tns:CapitalGainsTax.DeferredRelief.Amount
            */
            assertion = (report.CGTS118.GetValueOrDefault() != report.CGTS95.GetValueOrDefault() + report.CGTS97.GetValueOrDefault() + report.CGTS99.GetValueOrDefault() + report.CGTS101.GetValueOrDefault() + report.CGTS103.GetValueOrDefault() + report.CGTS111.GetValueOrDefault() + report.CGTS113.GetValueOrDefault() + report.CGTS114.GetValueOrDefault() + report.CGTS116.GetValueOrDefault() + report.CGTS163.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402157", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total current year capital gains must equal the sum of each unique capital gain  reported",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402157"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS118", Value = report.CGTS118.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS95", Value = report.CGTS95.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS97", Value = report.CGTS97.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS99", Value = report.CGTS99.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS101", Value = report.CGTS101.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS103", Value = report.CGTS103.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS111", Value = report.CGTS111.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS113", Value = report.CGTS113.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS114", Value = report.CGTS114.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS116", Value = report.CGTS116.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS163", Value = report.CGTS163.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402157
    
            #region VR.ATO.CGTS.402158
    
            /*  VR.ATO.CGTS.402158
            Where a capital loss, from collectables, is greater than the capital gain, from collectables, there can only be a capital loss amount, for collectables, applied in the current year that equates to the amount of capital gain reported for collectables.
    
            Legacy Rule Format:
            IF ([CGTS115] > [CGTS114]) AND [CGTS120] >([CGTS96]+[CGTS98]+[CGTS100]+[CGTS102]+[CGTS110]+[CGTS112]+[CGTS117]+[CGTS114])
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS115 > ^CGTS114) AND ^CGTS120 >(^CGTS96+^CGTS98+^CGTS100+^CGTS102+^CGTS110+^CGTS112+^CGTS117+^CGTS114)
    
            Data Elements:
    
            RP.Collectables:^CGTS114 = tns:Capital.Gains.Total.Amount
    
            RP.Collectables:^CGTS115 = tns:Capital.Losses.Total.Amount
    
            RP.Applied.Y0:^CGTS120 = tns:Capital.Losses.Total.Amount
    
            RP.ListedCompanyShares:^CGTS96 = tns:Capital.Losses.Total.Amount
    
            RP.OtherShares:^CGTS98 = tns:Capital.Losses.Total.Amount
    
            RP.ListedUnitTrusts:^CGTS100 = tns:Capital.Losses.Total.Amount
    
            RP.OtherUnits:^CGTS102 = tns:Capital.Losses.Total.Amount
    
            RP.AustralianRealProperty:^CGTS110 = tns:Capital.Losses.Total.Amount
    
            RP.OverseasRealProperty:^CGTS112 = tns:Capital.Losses.Total.Amount
    
            RP.OtherCapitalAssets:^CGTS117 = tns:Capital.Losses.Total.Amount
            */
            assertion = (report.CGTS115.GetValueOrDefault() > report.CGTS114.GetValueOrDefault() && report.CGTS120.GetValueOrDefault() > report.CGTS96.GetValueOrDefault() + report.CGTS98.GetValueOrDefault() + report.CGTS100.GetValueOrDefault() + report.CGTS102.GetValueOrDefault() + report.CGTS110.GetValueOrDefault() + report.CGTS112.GetValueOrDefault() + report.CGTS117.GetValueOrDefault() + report.CGTS114.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402158", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid capital loss amount applied",
                    LongDescription = @"Current year capital losses from collectables are greatear than capital gains from collectables. The capital loss amount applied in the current year, for collectables, must be less than or equal to the amount of capital gain from collectables.  The reamining capital loss amount must be carried froward.",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPCollectablesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402158"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS115", Value = report.CGTS115.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS114", Value = report.CGTS114.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS120", Value = report.CGTS120.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS96", Value = report.CGTS96.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS98", Value = report.CGTS98.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS100", Value = report.CGTS100.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS102", Value = report.CGTS102.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS110", Value = report.CGTS110.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS112", Value = report.CGTS112.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS117", Value = report.CGTS117.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402158
    
            #region VR.ATO.CGTS.402159
    
            /*  VR.ATO.CGTS.402159
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS119, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS119 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS119, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402159"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS119", Value = report.CGTS119.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402159
    
            #region VR.ATO.CGTS.402160
    
            /*  VR.ATO.CGTS.402160
            Total current year capital losses must equal the sum of each unique capital loss reported
    
            Legacy Rule Format:
            IF ([CGTS119]) <>
            ([CGTS96]+[CGTS98]+[CGTS100]+[CGTS102]+[CGTS110]+
            [CGTS112]+[CGTS115]+[CGTS117])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS119 <>(^CGTS96+^CGTS98+^CGTS100+^CGTS102+^CGTS110+^CGTS112+^CGTS115+^CGTS117)
    
            Data Elements:
    
            RP:^CGTS119 = tns:Capital.Losses.Total.Amount
    
            RP.ListedCompanyShares:^CGTS96 = tns:Capital.Losses.Total.Amount
    
            RP.OtherShares:^CGTS98 = tns:Capital.Losses.Total.Amount
    
            RP.ListedUnitTrusts:^CGTS100 = tns:Capital.Losses.Total.Amount
    
            RP.OtherUnits:^CGTS102 = tns:Capital.Losses.Total.Amount
    
            RP.AustralianRealProperty:^CGTS110 = tns:Capital.Losses.Total.Amount
    
            RP.OverseasRealProperty:^CGTS112 = tns:Capital.Losses.Total.Amount
    
            RP.Collectables:^CGTS115 = tns:Capital.Losses.Total.Amount
    
            RP.OtherCapitalAssets:^CGTS117 = tns:Capital.Losses.Total.Amount
            */
            assertion = (report.CGTS119.GetValueOrDefault() != report.CGTS96.GetValueOrDefault() + report.CGTS98.GetValueOrDefault() + report.CGTS100.GetValueOrDefault() + report.CGTS102.GetValueOrDefault() + report.CGTS110.GetValueOrDefault() + report.CGTS112.GetValueOrDefault() + report.CGTS115.GetValueOrDefault() + report.CGTS117.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402160", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total current year capital losses does not equal the sum of each unique capital loss reported",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402160"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS119", Value = report.CGTS119.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS96", Value = report.CGTS96.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS98", Value = report.CGTS98.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS100", Value = report.CGTS100.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS102", Value = report.CGTS102.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS110", Value = report.CGTS110.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS112", Value = report.CGTS112.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS115", Value = report.CGTS115.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS117", Value = report.CGTS117.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402160
    
            #region VR.ATO.CGTS.402161
    
            /*  VR.ATO.CGTS.402161
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (rvctc3.xx.xx:Capital.Losses.Total.Amount) <> NULL AND (rvctc3.xx.xx:Capital.Losses.Total.Amount <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS120, 'U', 11, 0)
    
            Data Elements:
    
            RP.Applied.Y0:^CGTS120 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS120, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPAppliedY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402161"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS120", Value = report.CGTS120.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402161
    
            #region VR.ATO.CGTS.402162
    
            /*  VR.ATO.CGTS.402162
            Total current year capital losses applied can not be greater than total current year capital losses
    
            Legacy Rule Format:
            IF ([CGTS120] > [CGTS119])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS120 > ^CGTS119
    
            Data Elements:
    
            RP.Applied.Y0:^CGTS120 = tns:Capital.Losses.Total.Amount
    
            RP:^CGTS119 = tns:Capital.Losses.Total.Amount
            */
            assertion = (report.CGTS120.GetValueOrDefault() > report.CGTS119.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402162", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total current year capital losses applied can not be greater than total current year capital losses",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPAppliedY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402162"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS120", Value = report.CGTS120.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS119", Value = report.CGTS119.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402162
    
            #region VR.ATO.CGTS.402163
    
            /*  VR.ATO.CGTS.402163
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS43] <> NULL) AND ([CGTS43] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS43, 'U', 11, 0)
    
            Data Elements:
    
            RP.Applied.Y0-:^CGTS43 = tns:Capital.Losses.NetTotal.Amount
            */
            assertion = NotMonetary(report.CGTS43, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.NetTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402163"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS43", Value = report.CGTS43.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402163
    
            #region VR.ATO.CGTS.402164
    
            /*  VR.ATO.CGTS.402164
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS122] <> NULL) AND ([CGTS122] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS122, 'U', 11, 0)
    
            Data Elements:
    
            RP.Applied:^CGTS122 = tns:Capital.Losses.TransferredTotal.Amount
            */
            assertion = NotMonetary(report.CGTS122, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402164"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS122", Value = report.CGTS122.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402164
    
            #region VR.ATO.CGTS.402166
    
            /*  VR.ATO.CGTS.402166
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS123] <> NULL) AND ([CGTS123] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS123, 'U', 11, 0)
    
            Data Elements:
    
            RP.Applied:^CGTS123 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS123, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPAppliedId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402166"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS123", Value = report.CGTS123.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402166
    
            #region VR.ATO.CGTS.402167
    
            /*  VR.ATO.CGTS.402167
            Total capital losses applied must be equal to the sum of current year capital losses, prior year net capital losses and total capital losses transferred in applied
    
            Legacy Rule Format:
            IF ([CGTS123] <> NULL) AND ([CGTS123]  <> ([CGTS120]+[CGTS43]+[CGTS122]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS123 <> NULL) AND (^CGTS123  <> (^CGTS120+^CGTS43+^CGTS122))
    
            Data Elements:
    
            RP.Applied:^CGTS123 = tns:Capital.Losses.Total.Amount
    
            RP.Applied.Y0:^CGTS120 = tns:Capital.Losses.Total.Amount
    
            RP.Applied.Y0-:^CGTS43 = tns:Capital.Losses.NetTotal.Amount
    
            RP.Applied:^CGTS122 = tns:Capital.Losses.TransferredTotal.Amount
            */
            assertion = (report.CGTS123 != null && report.CGTS123.GetValueOrDefault() != report.CGTS120.GetValueOrDefault() + report.CGTS43.GetValueOrDefault() + report.CGTS122.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402167", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid amount. Total current year capital losses applied amount is incorrect",
                    LongDescription = @"Total capital losses applied must equal the sum of current year capital losses, prior year net capital losses and total capital losses transferred in applied",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPAppliedId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402167"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS123", Value = report.CGTS123.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS120", Value = report.CGTS120.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS43", Value = report.CGTS43.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS122", Value = report.CGTS122.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402167
    
            #region VR.ATO.CGTS.402168
    
            /*  VR.ATO.CGTS.402168
            Total capital losses applied must not be greater than total current year capital gains
    
            Legacy Rule Format:
            IF ([CGTS123] > [CGTS118])
               RETURN VALIDATION MESSSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS123 > ^CGTS118
    
            Data Elements:
    
            RP.Applied:^CGTS123 = tns:Capital.Losses.Total.Amount
    
            RP:^CGTS118 = tns:Capital.Gains.Total.Amount
            */
            assertion = (report.CGTS123.GetValueOrDefault() > report.CGTS118.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402168", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid amount. Total capital losses applied must not be greater than current year capital gains",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPAppliedId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402168"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS123", Value = report.CGTS123.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS118", Value = report.CGTS118.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402168
    
            #region VR.ATO.CGTS.402169
    
            /*  VR.ATO.CGTS.402169
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS124] <> NULL AND [CGTS124] <>
            MONETARY (U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS124, 'U', 11, 0)
    
            Data Elements:
    
            RP.Collectables.Unapplied(Instant):^CGTS124 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.CGTS124, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPCollectablesUnappliedInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402169"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS124", Value = report.CGTS124.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402169
    
            #region VR.ATO.CGTS.402170
    
            /*  VR.ATO.CGTS.402170
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS125] <> NULL AND [CGTS125] <>
            MONETARY (U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS125, 'U', 11, 0)
    
            Data Elements:
    
            RP.CGTOther.Unapplied(Instant):^CGTS125 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.CGTS125, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPCGTOtherUnappliedInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402170"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS125", Value = report.CGTS125.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402170
    
            #region VR.ATO.CGTS.402172
    
            /*  VR.ATO.CGTS.402172
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS126] <> NULL) AND ([CGTS126] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS126, 'U', 11, 0)
    
            Data Elements:
    
            RP.Applied:^CGTS126 = tns:TaxConcession.CapitalGains.DiscountTotal.Amount
            */
            assertion = NotMonetary(report.CGTS126, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.DiscountTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402172"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS126", Value = report.CGTS126.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402172
    
            #region VR.ATO.CGTS.402175
    
            /*  VR.ATO.CGTS.402175
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS127] <> NULL) AND ([CGTS127] <> 
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS127, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS127 = tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount
            */
            assertion = NotMonetary(report.CGTS127, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402175"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS127", Value = report.CGTS127.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402175
    
            #region VR.ATO.CGTS.402176
    
            /*  VR.ATO.CGTS.402176
            The small business active asset reduction must be less than or equal to 50% of current year capital gains after applying capital losses
    
            Legacy Rule Format:
            IF ([CGTS127] <> NULL) AND [CGTS127] > ((([CGTS118] - [CGTS123] - [CGTS126]) *0.5) +1)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS127 <> NULL) AND ^CGTS127 > (((^CGTS118 - ^CGTS123 - ^CGTS126) *0.5) +1)
    
            Data Elements:
    
            RP:^CGTS127 = tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount
    
            RP:^CGTS118 = tns:Capital.Gains.Total.Amount
    
            RP.Applied:^CGTS123 = tns:Capital.Losses.Total.Amount
    
            RP.Applied:^CGTS126 = tns:TaxConcession.CapitalGains.DiscountTotal.Amount
            */
            assertion = (report.CGTS127 != null && report.CGTS127.GetValueOrDefault() > (report.CGTS118.GetValueOrDefault() - report.CGTS123.GetValueOrDefault() - report.CGTS126.GetValueOrDefault()) * 0.5M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402176", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid small business active asset reduction amount reported",
                    LongDescription = @"The small business active asset reduction must be less than or equal to 50% of current year capital gains after applying capital losses",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402176"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS127", Value = report.CGTS127.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS118", Value = report.CGTS118.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS123", Value = report.CGTS123.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS126", Value = report.CGTS126.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402176
    
            #region VR.ATO.CGTS.402177
    
            /*  VR.ATO.CGTS.402177
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS128] <> NULL AND [CGTS128] <> 
            MONETARY (U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS128, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS128 = tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount
            */
            assertion = NotMonetary(report.CGTS128, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402177"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS128", Value = report.CGTS128.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402177
    
            #region VR.ATO.CGTS.402179
    
            /*  VR.ATO.CGTS.402179
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS129] <> NULL) AND ([CGTS129] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS129, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS129 = tns:TaxConcession.CapitalGains.SmallBusinessRollover.Amount
            */
            assertion = NotMonetary(report.CGTS129, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessRollover.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402179"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS129", Value = report.CGTS129.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402179
    
            #region VR.ATO.CGTS.402180
    
            /*  VR.ATO.CGTS.402180
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS130] <> NULL) AND ([CGTS130] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            END IF

            Technical Business Rule Format:
            NotMonetary(^CGTS130, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS130 = tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount
            */
            assertion = NotMonetary(report.CGTS130, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402180"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS130", Value = report.CGTS130.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402180
    
            #region VR.ATO.CGTS.402181
    
            /*  VR.ATO.CGTS.402181
            Total small business concessions can not be greater than current year capital gains remaining after applying capital losses and the Capital Gains Tax (CGT) discount
    
            Legacy Rule Format:
            IF ([CGTS130] <> NULL) AND ([CGTS130] > ([CGTS118] - [CGTS123] - [CGTS126]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS130 <> NULL) AND (^CGTS130 > (^CGTS118 - ^CGTS123 - ^CGTS126))
    
            Data Elements:
    
            RP:^CGTS130 = tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount
    
            RP:^CGTS118 = tns:Capital.Gains.Total.Amount
    
            RP.Applied:^CGTS123 = tns:Capital.Losses.Total.Amount
    
            RP.Applied:^CGTS126 = tns:TaxConcession.CapitalGains.DiscountTotal.Amount
            */
            assertion = (report.CGTS130 != null && report.CGTS130.GetValueOrDefault() > report.CGTS118.GetValueOrDefault() - report.CGTS123.GetValueOrDefault() - report.CGTS126.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402181", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid small business concessions total amount",
                    LongDescription = @"Total small business concessions can not be greater than current year capital gains remaining after applying capital losses and the Capital Gains Tax (CGT) discount",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402181"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS130", Value = report.CGTS130.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS118", Value = report.CGTS118.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS123", Value = report.CGTS123.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS126", Value = report.CGTS126.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402181
    
            #region VR.ATO.CGTS.402182
    
            /*  VR.ATO.CGTS.402182
            Total small business concessions applied must be the sum of the small business active asset reduction, the small business retirement exemption and the small business rollover
    
            Legacy Rule Format:
            IF [CGTS130] <> NULL AND [CGTS130] <> ([CGTS127]+[CGTS128]+[CGTS129]) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS130 <> NULL AND ^CGTS130 <> (^CGTS127+^CGTS128+^CGTS129)
    
            Data Elements:
    
            RP:^CGTS130 = tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount
    
            RP:^CGTS127 = tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount
    
            RP:^CGTS128 = tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount
    
            RP:^CGTS129 = tns:TaxConcession.CapitalGains.SmallBusinessRollover.Amount
            */
            assertion = (report.CGTS130 != null && report.CGTS130.GetValueOrDefault() != report.CGTS127.GetValueOrDefault() + report.CGTS128.GetValueOrDefault() + report.CGTS129.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402182", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid small business concessions total amount",
                    LongDescription = @"Total small business concessions can not be greater than current year capital gains remaining after applying capital losses and the Capital Gains Tax (CGT) discount",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402182"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS130", Value = report.CGTS130.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS127", Value = report.CGTS127.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS128", Value = report.CGTS128.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS129", Value = report.CGTS129.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402182
    
            #region VR.ATO.CGTS.402183
    
            /*  VR.ATO.CGTS.402183
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS131] <> NULL AND [CGTS131] <>
            MONETARY (U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS131, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS131 = tns:Income.CapitalGainsNet.Amount
            */
            assertion = NotMonetary(report.CGTS131, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402183"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS131", Value = report.CGTS131.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402183
    
            #region VR.ATO.CGTS.402184
    
            /*  VR.ATO.CGTS.402184
            Net capital gain must equal the sum of total current year capital gains less total capital losses applied  less total capital gains discount applied less total small business concessions applied
    
            Legacy Rule Format:
            IF [CGTS131] <> NULL AND [CGTS131] <> ([CGTS118]-[CGTS123]-[CGTS126]-[CGTS130])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS131 <> NULL AND ^CGTS131 <> (^CGTS118-^CGTS123-^CGTS126-^CGTS130)
    
            Data Elements:
    
            RP:^CGTS131 = tns:Income.CapitalGainsNet.Amount
    
            RP:^CGTS118 = tns:Capital.Gains.Total.Amount
    
            RP.Applied:^CGTS123 = tns:Capital.Losses.Total.Amount
    
            RP.Applied:^CGTS126 = tns:TaxConcession.CapitalGains.DiscountTotal.Amount
    
            RP:^CGTS130 = tns:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount
            */
            assertion = (report.CGTS131 != null && report.CGTS131.GetValueOrDefault() != report.CGTS118.GetValueOrDefault() - report.CGTS123.GetValueOrDefault() - report.CGTS126.GetValueOrDefault() - report.CGTS130.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402184", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid amount. Total net capital gain amount incorrect",
                    LongDescription = @"Net capital gain must equal the sum of total current year capital gains less total capital losses applied  less total capital gains discount applied less total small business concessions applied",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402184"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS131", Value = report.CGTS131.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS118", Value = report.CGTS118.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS123", Value = report.CGTS123.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS126", Value = report.CGTS126.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS130", Value = report.CGTS130.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402184
    
            #region VR.ATO.CGTS.402194
    
            /*  VR.ATO.CGTS.402194
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS138] <> NULL) AND ([CGTS138] <> 
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS138, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS138 = tns:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount
            */
            assertion = NotMonetary(report.CGTS138, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS138", Value = report.CGTS138.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402194
    
            #region VR.ATO.CGTS.402195
    
            /*  VR.ATO.CGTS.402195
            An asset type code must be present where a reporting party has exempt capital gains from a small business 15 year exemption concession
    
            Legacy Rule Format:
            IF ([CGTS138] <> NULL) AND ([CGTS139] = NULL)
               RETURN VALIDATION MESSAGE 
            ENDIF

            Technical Business Rule Format:
            (^CGTS138 <> NULL) AND (^CGTS139 = NULL)
    
            Data Elements:
    
            RP:^CGTS139 = tns:TaxConcession.CapitalGains.SmallBusinessExemptionAssetOrEvent.Code
    
            RP:^CGTS138 = tns:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount
            */
            assertion = (report.CGTS138 != null && report.CGTS139 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402195", Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business 15 year exemption code is not present",
                    LongDescription = @"Small business 15 year exemption amount reported. Asset type code must be present",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessExemptionAssetOrEvent.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS138", Value = report.CGTS138.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS139", Value = report.CGTS139 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402195
    
            #region VR.ATO.CGTS.402196
    
            /*  VR.ATO.CGTS.402196
            An asset type code must not be be present where a reporting party has no exempt capital gains from a small business 15 year exemption concession
    
            Legacy Rule Format:
            IF ([CGTS139] <> NULL) AND ([CGTS138] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS139 <> NULL) AND (^CGTS138 = NULL)
    
            Data Elements:
    
            RP:^CGTS139 = tns:TaxConcession.CapitalGains.SmallBusinessExemptionAssetOrEvent.Code
    
            RP:^CGTS138 = tns:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount
            */
            assertion = (report.CGTS139 != null && report.CGTS138 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402196", Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business 15 year exemption code should not be present",
                    LongDescription = @"No small business 15 year exemption amount reported. Asset type code must not be present",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessExemptionAssetOrEvent.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402196"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS139", Value = report.CGTS139 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS138", Value = report.CGTS138.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402196
    
            #region VR.ATO.CGTS.402197
    
            /*  VR.ATO.CGTS.402197
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF([CGTS140] <> NULL) AND ([CGTS140] <> 
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS140, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS140 = tns:Capital.Gains.ForeignResidentExemption.Amount
            */
            assertion = NotMonetary(report.CGTS140, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.ForeignResidentExemption.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402197"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS140", Value = report.CGTS140.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402197
    
            #region VR.ATO.CGTS.402198
    
            /*  VR.ATO.CGTS.402198
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF([CGTS45] <> NULL) AND ([CGTS45] <> 
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS45, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS45 = tns:Capital.Gains.ScriptForScriptRolloverDeferral.Amount
            */
            assertion = NotMonetary(report.CGTS45, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.ScriptForScriptRolloverDeferral.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402198"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS45", Value = report.CGTS45.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402198
    
            #region VR.ATO.CGTS.402199
    
            /*  VR.ATO.CGTS.402199
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS46] <> NULL) AND ([CGTS46] <> 
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS46, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS46 = tns:Capital.Gains.InterCompanyAssetRolloverDeferral.Amount
            */
            assertion = NotMonetary(report.CGTS46, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.InterCompanyAssetRolloverDeferral.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402199"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS46", Value = report.CGTS46.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402199
    
            #region VR.ATO.CGTS.402200
    
            /*  VR.ATO.CGTS.402200
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF ([CGTS47] <> NULL) AND ([CGTS47] <>
            MONETARY (U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS47, 'U', 11, 0)
    
            Data Elements:
    
            RP:^CGTS47 = tns:Capital.Gains.DemergingEntityExemption.Amount
            */
            assertion = NotMonetary(report.CGTS47, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.DemergingEntityExemption.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402200"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS47", Value = report.CGTS47.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402200
    
            #region VR.ATO.CGTS.402201
    
            /*  VR.ATO.CGTS.402201
            The number of current year applied contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.Applied.Y0)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Applied:RP.Applied.Y0
            */
            assertion = (Count(report.RPAppliedY0Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedY0LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402201"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402201
    
            #region VR.ATO.CGTS.402202
    
            /*  VR.ATO.CGTS.402202
            The number of Other Capital Assets contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT (CONTEXT(RP.OtherCapitalAssets)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.OtherCapitalAssets
            */
            assertion = (Count(report.RPOtherCapitalAssetsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278", Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherCapitalAssetsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402202"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402202
    
            #region VR.ATO.CGTS.402222
    
            /*  VR.ATO.CGTS.402222
            Income year earnout right created cannot be NULL when Amount to be amended is present.
    
            Legacy Rule Format:
            IF ([CGTS151] <> NULL) AND ([CGTS150] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS151 <> NULL) AND (^CGTS150 = NULL)
    
            Data Elements:
    
            RP:^CGTS150 = tns:Report.TargetFinancial.Year
    
            RP:^CGTS151 = tns:Miscellaneous.RequestedAmendment.Amount
            */
            assertion = (report.CGTS151 != null && report.CGTS150 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402222", Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created must be provided when Amount to be amended is present",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402222"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS151", Value = report.CGTS151.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS150", Value = report.CGTS150.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402222
    
            #region VR.ATO.CGTS.402223
    
            /*  VR.ATO.CGTS.402223
            Amount to be amended cannot be NULL when Income year earnout right created is present.
    
            Legacy Rule Format:
            IF ([CGTS150] <> NULL) AND ([CGTS151] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CGTS150 <> NULL) AND (^CGTS151 = NULL)
    
            Data Elements:
    
            RP:^CGTS151 = tns:Miscellaneous.RequestedAmendment.Amount
    
            RP:^CGTS150 = tns:Report.TargetFinancial.Year
            */
            assertion = (report.CGTS150 != null && report.CGTS151 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402223", Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount to be amended must be provided when Income year earnout right is present",
                    Location = "/xbrli:xbrl/tns:Miscellaneous.RequestedAmendment.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402223"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS150", Value = report.CGTS150.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS151", Value = report.CGTS151.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402223
    
            #region VR.ATO.CGTS.402225
    
            /*  VR.ATO.CGTS.402225
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character.
    
            Legacy Rule Format:
            IF ([CGTS151] <> NULL) AND ([CGTS151] <> MONETARY (S,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS151, 'S', 11, 0)
    
            Data Elements:
    
            RP:^CGTS151 = tns:Miscellaneous.RequestedAmendment.Amount
            */
            assertion = NotMonetary(report.CGTS151, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Miscellaneous.RequestedAmendment.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402225"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS151", Value = report.CGTS151.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402225
    
            #region VR.ATO.CGTS.402231
    
            /*  VR.ATO.CGTS.402231
            Multi-class treatment fields are mandatory where 'Name of AMIT class' is present
    
            Legacy Rule Format:
            IF [CGTS153] <> NULL AND ([CGTS154] = NULL OR [CGTS155] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CGTS153 <> NULL AND (^CGTS154 = NULL OR ^CGTS155 = NULL)
    
            Data Elements:
    
            RP:^CGTS153 = tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description
    
            RP.TransfersToOtherClasses:^CGTS154 = tns:Capital.Gains.Total.Amount
    
            RP.TransfersToOtherClasses:^CGTS155 = tns:Capital.Losses.Total.Amount
            */
            assertion = (report.CGTS153 != null && (report.CGTS154 == null || report.CGTS155 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGTS.402231", Severity = ProcessMessageSeverity.Error,
                    Description = @"Multi-class treatment fields 'Total capital gains from transfers to other classes' and 'Total capital losses from transfers to other classes' must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402231"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS153", Value = report.CGTS153 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS154", Value = report.CGTS154.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS155", Value = report.CGTS155.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402231
    
            #region VR.ATO.CGTS.402232
    
            /*  VR.ATO.CGTS.402232
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS154] <> NULL AND [CGTS154] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS154, 'U', 11, 0)
    
            Data Elements:
    
            RP.TransfersToOtherClasses:^CGTS154 = tns:Capital.Gains.Total.Amount
            */
            assertion = NotMonetary(report.CGTS154, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Gains.Total.Amount[@contextRef='" + report.RPTransfersToOtherClassesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402232"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS154", Value = report.CGTS154.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402232
    
            #region VR.ATO.CGTS.402233
    
            /*  VR.ATO.CGTS.402233
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS155] <> NULL AND [CGTS155] <> 
            MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS155, 'U', 11, 0)
    
            Data Elements:
    
            RP.TransfersToOtherClasses:^CGTS155 = tns:Capital.Losses.Total.Amount
            */
            assertion = NotMonetary(report.CGTS155, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.Total.Amount[@contextRef='" + report.RPTransfersToOtherClassesId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402233"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS155", Value = report.CGTS155.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402233
    
            #region VR.ATO.CGTS.402245
    
            /*  VR.ATO.CGTS.402245
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CGTS163] <> NULL AND [CGTS163] <> 
            MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CGTS163, 'U', 11, 0)
    
            Data Elements:
    
            RP.OtherCapitalAssets:^CGTS163 = tns:CapitalGainsTax.DeferredRelief.Amount
            */
            assertion = NotMonetary(report.CGTS163, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.DeferredRelief.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402245"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "CGTS163", Value = report.CGTS163.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.CGTS.402245
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Applied
    
            ^Context = RP:RP.Applied
            */
            assertion = (report.RPAppliedExists != false && report.RPAppliedIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Applied:RP.Applied.Y0
    
            ^Context = RP.Applied:RP.Applied.Y0
            */
            assertion = (report.RPAppliedY0Exists != false && report.RPAppliedY0IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedY0OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Applied:RP.Applied.Y0-
    
            ^Context = RP.Applied:RP.Applied.Y0-
            */
            assertion = (report.RPAppliedY0MinusExists != false && report.RPAppliedY0MinusIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedY0MinusOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.AustralianRealProperty
    
            ^Context = RP:RP.AustralianRealProperty
            */
            assertion = (report.RPAustralianRealPropertyExists != false && report.RPAustralianRealPropertyIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAustralianRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.CapitalGainsTrustDistributions
    
            ^Context = RP:RP.CapitalGainsTrustDistributions
            */
            assertion = (report.RPCapitalGainsTrustDistributionsExists != false && report.RPCapitalGainsTrustDistributionsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCapitalGainsTrustDistributionsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.CGTOther.Unapplied(Instant)
    
            ^Context = RP:RP.CGTOther.Unapplied(Instant)
            */
            assertion = (report.RPCGTOtherUnappliedInstantExists != false && report.RPCGTOtherUnappliedInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCGTOtherUnappliedInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Collectables
    
            ^Context = RP:RP.Collectables
            */
            assertion = (report.RPCollectablesExists != false && report.RPCollectablesIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCollectablesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Collectables:RP.Collectables.Unapplied(Instant)
    
            ^Context = RP.Collectables:RP.Collectables.Unapplied(Instant)
            */
            assertion = (report.RPCollectablesUnappliedInstantExists != false && report.RPCollectablesUnappliedInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCollectablesUnappliedInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.ListedCompanyShares
    
            ^Context = RP:RP.ListedCompanyShares
            */
            assertion = (report.RPListedCompanySharesExists != false && report.RPListedCompanySharesIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPListedCompanySharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.ListedUnitTrusts
    
            ^Context = RP:RP.ListedUnitTrusts
            */
            assertion = (report.RPListedUnitTrustsExists != false && report.RPListedUnitTrustsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPListedUnitTrustsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.OtherCapitalAssets
    
            ^Context = RP:RP.OtherCapitalAssets
            */
            assertion = (report.RPOtherCapitalAssetsExists != false && report.RPOtherCapitalAssetsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherCapitalAssetsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.OtherShares
    
            ^Context = RP:RP.OtherShares
            */
            assertion = (report.RPOtherSharesExists != false && report.RPOtherSharesIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherSharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.OtherUnits
    
            ^Context = RP:RP.OtherUnits
            */
            assertion = (report.RPOtherUnitsExists != false && report.RPOtherUnitsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherUnitsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.OverseasRealProperty
    
            ^Context = RP:RP.OverseasRealProperty
            */
            assertion = (report.RPOverseasRealPropertyExists != false && report.RPOverseasRealPropertyIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOverseasRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TransfersToOtherClasses
    
            ^Context = RP:RP.TransfersToOtherClasses
            */
            assertion = (report.RPTransfersToOtherClassesExists != false && report.RPTransfersToOtherClassesIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTransfersToOtherClassesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Applied
    
            ^OtherTFN = RP:RP.Applied
    
            ^TFN = RP
            */
            assertion = (report.RPAppliedExists != false && report.RPAppliedIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Applied:RP.Applied.Y0
    
            ^OtherTFN = RP.Applied:RP.Applied.Y0
    
            ^TFN = RP
            */
            assertion = (report.RPAppliedY0Exists != false && report.RPAppliedY0IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedY0OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Applied:RP.Applied.Y0-
    
            ^OtherTFN = RP.Applied:RP.Applied.Y0-
    
            ^TFN = RP
            */
            assertion = (report.RPAppliedY0MinusExists != false && report.RPAppliedY0MinusIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAppliedY0MinusOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.AustralianRealProperty
    
            ^OtherTFN = RP:RP.AustralianRealProperty
    
            ^TFN = RP
            */
            assertion = (report.RPAustralianRealPropertyExists != false && report.RPAustralianRealPropertyIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAustralianRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.CapitalGainsTrustDistributions
    
            ^OtherTFN = RP:RP.CapitalGainsTrustDistributions
    
            ^TFN = RP
            */
            assertion = (report.RPCapitalGainsTrustDistributionsExists != false && report.RPCapitalGainsTrustDistributionsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCapitalGainsTrustDistributionsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.CGTOther.Unapplied(Instant)
    
            ^OtherTFN = RP:RP.CGTOther.Unapplied(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPCGTOtherUnappliedInstantExists != false && report.RPCGTOtherUnappliedInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCGTOtherUnappliedInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Collectables
    
            ^OtherTFN = RP:RP.Collectables
    
            ^TFN = RP
            */
            assertion = (report.RPCollectablesExists != false && report.RPCollectablesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCollectablesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Collectables:RP.Collectables.Unapplied(Instant)
    
            ^OtherTFN = RP.Collectables:RP.Collectables.Unapplied(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPCollectablesUnappliedInstantExists != false && report.RPCollectablesUnappliedInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCollectablesUnappliedInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.ListedCompanyShares
    
            ^OtherTFN = RP:RP.ListedCompanyShares
    
            ^TFN = RP
            */
            assertion = (report.RPListedCompanySharesExists != false && report.RPListedCompanySharesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPListedCompanySharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.ListedUnitTrusts
    
            ^OtherTFN = RP:RP.ListedUnitTrusts
    
            ^TFN = RP
            */
            assertion = (report.RPListedUnitTrustsExists != false && report.RPListedUnitTrustsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPListedUnitTrustsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.OtherCapitalAssets
    
            ^OtherTFN = RP:RP.OtherCapitalAssets
    
            ^TFN = RP
            */
            assertion = (report.RPOtherCapitalAssetsExists != false && report.RPOtherCapitalAssetsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherCapitalAssetsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.OtherShares
    
            ^OtherTFN = RP:RP.OtherShares
    
            ^TFN = RP
            */
            assertion = (report.RPOtherSharesExists != false && report.RPOtherSharesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherSharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.OtherUnits
    
            ^OtherTFN = RP:RP.OtherUnits
    
            ^TFN = RP
            */
            assertion = (report.RPOtherUnitsExists != false && report.RPOtherUnitsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOtherUnitsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.OverseasRealProperty
    
            ^OtherTFN = RP:RP.OverseasRealProperty
    
            ^TFN = RP
            */
            assertion = (report.RPOverseasRealPropertyExists != false && report.RPOverseasRealPropertyIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOverseasRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.TransfersToOtherClasses
    
            ^OtherTFN = RP:RP.TransfersToOtherClasses
    
            ^TFN = RP
            */
            assertion = (report.RPTransfersToOtherClassesExists != false && report.RPTransfersToOtherClassesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTransfersToOtherClassesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region Repeating RPEarnoutArrangementGroupNumberContext
            if (report.RPEarnoutArrangementGroupNumberContextCollection != null)
            {
                int itemIndexContext2 = 0;
                foreach (CGTS2018.RPEarnoutArrangementGroupNumberContext rpEarnoutArrangementGroupNumberContext in report.RPEarnoutArrangementGroupNumberContextCollection)
                {
                    itemIndexContext2++;
            
                    #region VR.ATO.CGTS.402136
            
                    /*  VR.ATO.CGTS.402136
                    If the earnout arrangement group number exists, then the number of earnout arrangement group item dimensions must be between 1 - 20
    
                    Legacy Rule Format:
                    IF ([CGTS160] <> NULL) AND ([CGTS160] < 1 OR [CGTS160] > 20)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS160 <> NULL) AND (^CGTS160 < 1 OR ^CGTS160 > 20)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS160 = tns:Report.ItemOrder.Number
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS160 != null && (AsNumeric(rpEarnoutArrangementGroupNumberContext.CGTS160) < 1 || AsNumeric(rpEarnoutArrangementGroupNumberContext.CGTS160) > 20));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402136", Severity = ProcessMessageSeverity.Error,
                            Description = @"The number of earnout arrangement group item dimensions must be between 1 - 20",
                            Location = "/xbrli:xbrl/xbrli:context[@id='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402136"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS160", Value = rpEarnoutArrangementGroupNumberContext.CGTS160 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402136
            
                    #region VR.ATO.CGTS.402185
            
                    /*  VR.ATO.CGTS.402185
                    Are you party to an earnout arrangement is a mandatory field
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT
                    (RP.{EarnoutArrangementGroupNumber})
                    IF ([CGTS133] = NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS133 = BLANK)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS133 = tns:Miscellaneous.EarnoutArrangementEligibility.Code
                    */
                    assertion = (string.IsNullOrWhiteSpace(rpEarnoutArrangementGroupNumberContext.CGTS133) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402185", Severity = ProcessMessageSeverity.Error,
                            Description = @"Mandatory field not completed, must be set to either 'Buyer', 'Seller' or 'No'",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementEligibility.Code[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402185"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS133", Value = rpEarnoutArrangementGroupNumberContext.CGTS133 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402185
            
                    #region VR.ATO.CGTS.402186
            
                    /*  VR.ATO.CGTS.402186
                    Where a reporting party is party to an earnout arrangement as a buyer, the number of years the earnout arrangement will run for must be completed
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT
                    (RP.{EarnoutArrangementGroupNumber}) 
                    IF ([CGTS133] = "Buyer") AND ([CGTS134] = NULL)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS133 = 'Buyer') AND (^CGTS134 = NULL)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS133 = tns:Miscellaneous.EarnoutArrangementEligibility.Code
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS134 = tns:Miscellaneous.EarnoutArrangementDuration.Count
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS133 == @"Buyer" && rpEarnoutArrangementGroupNumberContext.CGTS134 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402186", Severity = ProcessMessageSeverity.Error,
                            Description = @"The number of years the earnout arrangement will run for must be completed",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementEligibility.Code[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402186"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS133", Value = rpEarnoutArrangementGroupNumberContext.CGTS133 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS134", Value = rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402186
            
                    #region VR.ATO.CGTS.402187
            
                    /*  VR.ATO.CGTS.402187
                    Where a reporting party is party to an earnout arrangement as a buyer, the earnout arrangement relative year must be completed
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT
                    (RP.{EarnoutArrangementGroupNumber})
                    IF ([CGTS133] = "Buyer") AND ([CGTS135] = NULL)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS133 = 'Buyer') AND (^CGTS135 = NULL)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS133 = tns:Miscellaneous.EarnoutArrangementEligibility.Code
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS135 = tns:Miscellaneous.EarnoutArrangementApplicableYear.Number
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS133 == @"Buyer" && rpEarnoutArrangementGroupNumberContext.CGTS135 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402187", Severity = ProcessMessageSeverity.Error,
                            Description = @"The earnout arrangement relative year must be completed",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementEligibility.Code[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402187"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS133", Value = rpEarnoutArrangementGroupNumberContext.CGTS133 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS135", Value = rpEarnoutArrangementGroupNumberContext.CGTS135.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402187
            
                    #region VR.ATO.CGTS.402188
            
                    /*  VR.ATO.CGTS.402188
                    The fields 'How many years does the earnout arrangement run for?' and 'What year of the earnout arrangement are you in?' are mandatory where the party code type is "Seller".
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT (RP.{EarnoutArrangementGroupNumber}) 
                    IF ([CGTS133] = "Seller") AND ([CGTS134] = NULL OR [CGTS135] = NULL)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS133 = 'Seller') AND (^CGTS134 = NULL OR ^CGTS135 = NULL)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS133 = tns:Miscellaneous.EarnoutArrangementEligibility.Code
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS134 = tns:Miscellaneous.EarnoutArrangementDuration.Count
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS135 = tns:Miscellaneous.EarnoutArrangementApplicableYear.Number
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS133 == @"Seller" && (rpEarnoutArrangementGroupNumberContext.CGTS134 == null || rpEarnoutArrangementGroupNumberContext.CGTS135 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402249", Severity = ProcessMessageSeverity.Error,
                            Description = @"Earnout Arrangement information missing.",
                            LongDescription = @"When the reporting party is a ""Seller"", then earnout arrangement fields 'How many years does the earnout arrangement run for?' and 'What year of the earnout arrangement are you in?' must be supplied.",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementEligibility.Code[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402188"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS133", Value = rpEarnoutArrangementGroupNumberContext.CGTS133 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS134", Value = rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS135", Value = rpEarnoutArrangementGroupNumberContext.CGTS135.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402188
            
                    #region VR.ATO.CGTS.402189
            
                    /*  VR.ATO.CGTS.402189
                    Where a reporting party is not party to an earnout arrangement, no fields within the earnout arrangement category should be present
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT
                    (RP.{EarnoutArrangementGroupNumber}) 
                    IF ([CGTS133] = "No") AND ([CGTS134] <> NULL OR [CGTS135] <> NULL OR [CGTS136] <> NULL OR [CGTS152] <> NULL OR [CGTS150] <> NULL OR [CGTS151] <> NULL)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS133 = 'No') AND (^CGTS134 <> NULL OR ^CGTS135 <> NULL OR ^CGTS136 <> NULL OR ^CGTS152 <> NULL OR ^CGTS150 <> NULL OR ^CGTS151 <> NULL)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS133 = tns:Miscellaneous.EarnoutArrangementEligibility.Code
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS134 = tns:Miscellaneous.EarnoutArrangementDuration.Count
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS135 = tns:Miscellaneous.EarnoutArrangementApplicableYear.Number
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS136 = tns:CapitalGainsTax.EarnoutArrangementEstimatedCapitalProceedsTotal.Amount
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS152 = tns:CapitalGainsTax.CapitalGainsOrLossesOnNonQualifyingEarnoutArrangementNet.Amount
            
                    RP:^CGTS150 = tns:Report.TargetFinancial.Year
            
                    RP:^CGTS151 = tns:Miscellaneous.RequestedAmendment.Amount
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS133 == @"No" && (rpEarnoutArrangementGroupNumberContext.CGTS134 != null || rpEarnoutArrangementGroupNumberContext.CGTS135 != null || rpEarnoutArrangementGroupNumberContext.CGTS136 != null || rpEarnoutArrangementGroupNumberContext.CGTS152 != null || report.CGTS150 != null || report.CGTS151 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402189", Severity = ProcessMessageSeverity.Error,
                            Description = @"No earnout arrangement reported. No other values should be present for an earnout arrangement",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementEligibility.Code[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402189"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS133", Value = rpEarnoutArrangementGroupNumberContext.CGTS133 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS134", Value = rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS135", Value = rpEarnoutArrangementGroupNumberContext.CGTS135.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS136", Value = rpEarnoutArrangementGroupNumberContext.CGTS136.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS152", Value = rpEarnoutArrangementGroupNumberContext.CGTS152.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS150", Value = report.CGTS150.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS151", Value = report.CGTS151.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402189
            
                    #region VR.ATO.CGTS.402190
            
                    /*  VR.ATO.CGTS.402190
                    The Earnout Arrangement duration must be set between 1 -99
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT 
                    (RP.{EarnoutArrangementGroupNumber}) 
                    IF ([CGTS133] <> NULL AND [CGTS133] <> "No") AND ([CGTS134] <> SET (1-99))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS133 <> NULL AND ^CGTS133 <> 'No') AND (^CGTS134 < 1 or ^CGTS134 > 99)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS134 = tns:Miscellaneous.EarnoutArrangementDuration.Count
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS133 = tns:Miscellaneous.EarnoutArrangementEligibility.Code
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS133 != null && rpEarnoutArrangementGroupNumberContext.CGTS133 != @"No" && (rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault() < 1 || rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault() > 99));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402190", Severity = ProcessMessageSeverity.Error,
                            Description = @"Total number of years an earnout arrangement can run is not set between 1-99",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementDuration.Count[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402190"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS133", Value = rpEarnoutArrangementGroupNumberContext.CGTS133 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS134", Value = rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402190
            
                    #region VR.ATO.CGTS.402191
            
                    /*  VR.ATO.CGTS.402191
                    The total number of years an Earnout Arrangement can run for must be less than the current earnout arrangement year
    
                    Legacy Rule Format:
                    WHERE IN CONTEXT
                    (RP.{EarnoutArrangementGroupNumber}) 
                    IF ([CGTS135] <> NULL) AND ([CGTS134] <> NULL) AND ([CGTS135] > [CGTS134])
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS135 <> NULL) AND (^CGTS134 <> NULL) AND (^CGTS135 > ^CGTS134)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS135 = tns:Miscellaneous.EarnoutArrangementApplicableYear.Number
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS134 = tns:Miscellaneous.EarnoutArrangementDuration.Count
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS135 != null && rpEarnoutArrangementGroupNumberContext.CGTS134 != null && rpEarnoutArrangementGroupNumberContext.CGTS135.GetValueOrDefault() > rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault());
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402191", Severity = ProcessMessageSeverity.Error,
                            Description = @"Current year of earnout arrangement is greater than total earnout duration",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementApplicableYear.Number[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402191"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS135", Value = rpEarnoutArrangementGroupNumberContext.CGTS135.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS134", Value = rpEarnoutArrangementGroupNumberContext.CGTS134.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402191
            
                    #region VR.ATO.CGTS.402192
            
                    /*  VR.ATO.CGTS.402192
                    Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
                    Legacy Rule Format:
                    IF ([CGTS136] <> NULL) AND ([CGTS136] <> 
                    MONETARY (U,11,0))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^CGTS136, 'U', 11, 0)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS136 = tns:CapitalGainsTax.EarnoutArrangementEstimatedCapitalProceedsTotal.Amount
                    */
                    assertion = NotMonetary(rpEarnoutArrangementGroupNumberContext.CGTS136, @"U", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:CapitalGainsTax.EarnoutArrangementEstimatedCapitalProceedsTotal.Amount[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402192"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS136", Value = rpEarnoutArrangementGroupNumberContext.CGTS136.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402192
            
                    #region VR.ATO.CGTS.402204
            
                    /*  VR.ATO.CGTS.402204
                    Field is in incorrect format for a numeric field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
                    Legacy Rule Format:
                    IF ([CGTS135] <> NULL) AND ([CGTS135] <> NUMBER(U,2,0))
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotNumeric(^CGTS135, 'U', 2, 0)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS135 = tns:Miscellaneous.EarnoutArrangementApplicableYear.Number
                    */
                    assertion = NotNumeric(rpEarnoutArrangementGroupNumberContext.CGTS135, @"U", 2, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400012", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a numeric field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Miscellaneous.EarnoutArrangementApplicableYear.Number[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402204"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS135", Value = rpEarnoutArrangementGroupNumberContext.CGTS135.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402204
            
                    #region VR.ATO.CGTS.402220
            
                    /*  VR.ATO.CGTS.402220
                    Amount of any capital gain or loss you made under your non-qualifying earnout arrangement in the income year must be NULL if Income year earnout right created or Amount to be amended is present.
    
                    Legacy Rule Format:
                    IF ([CGTS152] <> NULL) AND ([CGTS150] <> NULL AND [CGTS151] <> NULL)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^CGTS152 <> NULL) AND (^CGTS150 <> NULL AND ^CGTS151 <> NULL)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS152 = tns:CapitalGainsTax.CapitalGainsOrLossesOnNonQualifyingEarnoutArrangementNet.Amount
            
                    RP:^CGTS150 = tns:Report.TargetFinancial.Year
            
                    RP:^CGTS151 = tns:Miscellaneous.RequestedAmendment.Amount
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.CGTS152 != null && (report.CGTS150 != null && report.CGTS151 != null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CGTS.402220", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-qualifying value cannot be reported when request for amendment details are present.",
                            Location = "/xbrli:xbrl/tns:CapitalGainsTax.CapitalGainsOrLossesOnNonQualifyingEarnoutArrangementNet.Amount[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402220"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS152", Value = rpEarnoutArrangementGroupNumberContext.CGTS152.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS150", Value = report.CGTS150.GetValueOrDefault().ToString() });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS151", Value = report.CGTS151.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402220
            
                    #region VR.ATO.CGTS.402224
            
                    /*  VR.ATO.CGTS.402224
                    Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character.
    
                    Legacy Rule Format:
                    IF ([CGTS152] <> NULL) AND ([CGTS152] <> 
                    MONETARY (S,11,0))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    NotMonetary(^CGTS152, 'S', 11, 0)
            
                    Data Elements:
            
                    RP.{EarnoutArrangementGroupNumber}:^CGTS152 = tns:CapitalGainsTax.CapitalGainsOrLossesOnNonQualifyingEarnoutArrangementNet.Amount
                    */
                    assertion = NotMonetary(rpEarnoutArrangementGroupNumberContext.CGTS152, @"S", 11, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:CapitalGainsTax.CapitalGainsOrLossesOnNonQualifyingEarnoutArrangementNet.Amount[@contextRef='" + rpEarnoutArrangementGroupNumberContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402224"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "CGTS152", Value = rpEarnoutArrangementGroupNumberContext.CGTS152.GetValueOrDefault().ToString() });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.CGTS.402224
            
                    #region VR.ATO.GEN.001021
            
                    /*  VR.ATO.GEN.001021
                    Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
                    Legacy Rule Format:
                    IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
            
                    Data Elements:
            
                    ^IdentifierScheme = RP:RP.{EarnoutArrangementGroupNumber}
            
                    ^Context = RP:RP.{EarnoutArrangementGroupNumber}
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.Exists != false && rpEarnoutArrangementGroupNumberContext.IdentifierScheme != @"http://www.ato.gov.au/tfn");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid identifier scheme.",
                            LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rpEarnoutArrangementGroupNumberContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.001021
            
                    #region VR.ATO.GEN.001023
            
                    /*  VR.ATO.GEN.001023
                    TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
                    Legacy Rule Format:
                    IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^Context <> NULL AND ^OtherTFN <> ^TFN
            
                    Data Elements:
            
                    ^Context = RP:RP.{EarnoutArrangementGroupNumber}
            
                    ^OtherTFN = RP:RP.{EarnoutArrangementGroupNumber}
            
                    ^TFN = RP
                    */
                    assertion = (rpEarnoutArrangementGroupNumberContext.Exists != false && rpEarnoutArrangementGroupNumberContext.IdentifierTFN != report.RPIdentifierTFN);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                            Description = @"Unexpected TFN mismatch with Reporting Party",
                            LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                            Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(rpEarnoutArrangementGroupNumberContext.OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                        };
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.001023
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 