﻿using System;
using System.Runtime.CompilerServices;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using Ato.EN.IntegrationServices.CodeGenerationINCDTLS;

namespace Ato.EN.IntegrationServices.CodeGenerationIITR
{
    public partial class IITR2026CrossformValidator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="IITR2026CrossformValidator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public IITR2026CrossformValidator(int maxParameterNameLength = 80, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(IITR2026Crossform report)
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();

            AddProcessMessage(VR_ATO_IITR_310090(report), response); 

            ValidateCGTS(report, response);
            ValidateDDCTNS(report, response);
            ValidateINCDTLS(report, response);
            ValidateNRFI(report, response);
            ValidateRNTLPRPTY(report, response); 

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

        #region VR.ATO.IITR.310090

        /* VR.ATO.IITR.310090
        When the 'Income Tax Return Type' is provided, implying that the return is to be treated as a short Individual return, no attached schedules are allowed except INCDTLS, and no entries for Employment termination payment (ETP) summary or Australian superannuation lump sum (SLS) payments are permitted

        Legacy Rule Format:
            ^IITR13 <> NULL AND (CountDocument('CGTS') = 1 OR CountDocument('NRFI') = 1 OR CountDocument('DDCTNS') = 1 OR CountDocument('RNTLPRPTY') = 1 OR (CountDocument('INCDTLS') = 1 AND (Count(^INCDTLS135) > 0 OR Count(^INCDTLS327) > 0)))

        Technical Business Rule Format:
            ^IITR13 <> NULL AND (CountDocument('CGTS') = 1 OR CountDocument('NRFI') = 1 OR CountDocument('DDCTNS') = 1 OR CountDocument('RNTLPRPTY') = 1 OR (CountDocument('INCDTLS') = 1 AND (Count(^INCDTLS135) > 0 OR Count(^INCDTLS327) > 0)))

        Data Elements:
    
        ^IITR13 = RP:IncomeTax.ReturnType.Code
    
        ^INCDTLS135 = INCDTLS135
    
        ^INCDTLS327 = INCDTLS327
        */
        public static ProcessMessageDocument VR_ATO_IITR_310090(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;
            List<INCDTLS2026.Rp_Income> incomeTuples = new List<INCDTLS2026.Rp_Income>();

            if (Count(report.incdtlsChildDocuments) != 0 && report.incdtlsChildDocuments[0].Rp_IncomeCollection != null)
            {
                incomeTuples = report.incdtlsChildDocuments[0].Rp_IncomeCollection;
            }

            bool etpTuplesGreaterThanZero = (incomeTuples.FindAll(tuple => tuple.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists).Count > 0);
            bool slspTuplesGreaterThanZero = (incomeTuples.FindAll(tuple => tuple.Rp_Income_OrganisationNameDetails_SuperannuationLumpSumPaymentCollectionExists).Count > 0);

            bool assertion = (report.iitrParentDocument.IITR13 != null && ( Count(report.cgtsChildDocuments) == 1 || Count(report.nrfiChildDocuments) == 1 || Count(report.ddctnsChildDocuments) == 1 || Count(report.rntlprptyChildDocuments) == 1 || (Count(report.incdtlsChildDocuments) == 1) && (etpTuplesGreaterThanZero || slspTuplesGreaterThanZero)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.311090",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attached schedules or ETP/SLSP details are invalid",
                    LongDescription = @"When the 'Income Tax Return Type' is provided, implying that the return is to be treated as a short Individual return, no attached schedules are allowed, and no entries for Employment termination payment (ETP) summary or Australian superannuation lump sum (SLS) payments are permitted",
                    Location = "/xbrli:xbrl/tns:IncomeTax.ReturnType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.310090" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR13", Value = report.iitrParentDocument.IITR13 });
            }

            return processMessage;
        }
        #endregion // VR.ATO.IITR.310090

    }
}