﻿using Ato.EN.IntegrationServices.CodeGenerationNRFI;
using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Ato.EN.IntegrationServices.CodeGenerationIITR
{
    public partial class IITR2026CrossformValidator
    {
        /// <summary>
        /// Takes a report to validate against all NRFI rules and store the results in the response paramaeter object.
        /// </summary>
        /// <param name="report">Document to be validated with NRFI rules.</param>
        /// <param name="response">Response to add error codes to.</param>
        public static void ValidateNRFI(IITR2026Crossform report, List<ProcessMessageDocument> response)
        {
            AddProcessMessage(VR_ATO_IITR_000639(report), response);
            AddProcessMessage(VR_ATO_IITR_310093(report), response);

            #region NRFI Child Present

            if (report.nrfiChildDocuments != null && Count(report.nrfiChildDocuments) == 1)
            {
                NRFI2024 nrfiChildDocument = report.nrfiChildDocuments[0];

                AddProcessMessage(VR_ATO_IITR_000641(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_000642(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_000643(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_000644(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_000645(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_000651(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_310094(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_310095(report, nrfiChildDocument), response);
                AddProcessMessage(VR_ATO_IITR_600046(report, nrfiChildDocument), response);
            }

            #endregion // NRFI Child Present
        }

        #region VR.ATO.IITR.000639

        /* VR.ATO.IITR.000639
        An amount has been entered at Non-resident foreign income amount and there is no Non-Resident Foreign Income (NRFI) schedule attached. Please complete and attach a NRFI schedule.

        Legacy Rule Format:
            ^IITR494 <> NULL AND CountDocument('NRFI') = 0

        Technical Business Rule Format:
            ^IITR494 <> NULL AND CountDocument('NRFI') = 0

        Data Elements:
    
        ^IITR494 = RP.JFOREIGN:Income.Net.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_000639(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR494 != null && (report.nrfiChildDocuments == null || Count(report.nrfiChildDocuments) == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000639",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-Resident Foreign Income (NRFI) schedule is required for Non-resident foreign income amount",
                    LongDescription = @"An amount has been entered at Non-resident foreign income amount and there is no Non-Resident Foreign Income (NRFI) schedule attached. Please complete and attach a NRFI schedule.",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.iitrParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000639" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.310093

        /* VR.ATO.IITR.310093
        The number of Non-Resident Foreign Income schedules must not exceed 1

        Legacy Rule Format:
            CountDocument('NRFI') > 1

        Technical Business Rule Format:
            CountDocument('NRFI') > 1

       Data Elements:
       */
        public static ProcessMessageDocument VR_ATO_IITR_310093(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.nrfiChildDocuments != null && Count(report.nrfiChildDocuments) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.310093",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Non-Resident Foreign Income schedules must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.310093" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000641

        /* VR.ATO.IITR.000641
            When the simple self-assessment method is selected on the Non-Resident Foreign Income (NRFI) schedule, the Gross foreign income amount, less standard occupation deductions must equal the Non-resident foreign income amount on the parent form.

        Legacy Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 <> NULL AND ^NRFI3 = 'Simple' AND NotInDomain(Substring(^NRFI2,1,4), 'OCCUPATION PREFIX CODES') = FALSE AND ^IITR494 <> (^NRFI4 - RoundDown(^NRFI4 * ConditionalLookup(Substring(^NRFI2,1,4), 'Occupation Prefix Codes', NULL, 'Deduction Median Rate', 0),0))

        Technical Business Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 <> NULL AND ^NRFI3 = 'Simple' AND NotInDomain(Substring(^NRFI2,1,4), 'OCCUPATION PREFIX CODES') = FALSE AND ^IITR494 <> (^NRFI4 - RoundDown(^NRFI4 * ConditionalLookup(Substring(^NRFI2,1,4), 'Occupation Prefix Codes', NULL, 'Deduction Median Rate', 0),0))

        Data Elements:
    
        ^IITR494 = RP.JFOREIGN:Income.Net.Amount
    
        ^NRFI2 = NRFI2
    
        ^NRFI3 = NRFI3
    
        ^NRFI4 = NRFI4
        */

        public static ProcessMessageDocument VR_ATO_IITR_000641(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool condition1_NRFI3Simple = false;
            bool condition2_NRFI2Exists = false;
            bool condition3_IITR494NotEqualToNRFI4 = false;

            condition1_NRFI3Simple = nrfiChildDocument.NRFI3 != null && nrfiChildDocument.NRFI3 == "Simple";
            condition2_NRFI2Exists = DeductionMedianRates.deductionMedianRates != null && nrfiChildDocument.NRFI2 != null && DeductionMedianRates.deductionMedianRates.Keys.Contains(nrfiChildDocument.NRFI2.Substring(0, 4));

            if (condition2_NRFI2Exists)
                condition3_IITR494NotEqualToNRFI4 = nrfiChildDocument.NRFI4 != null && report.iitrParentDocument.IITR494 != null && report.iitrParentDocument.IITR494.GetValueOrDefault() != (nrfiChildDocument.NRFI4.GetValueOrDefault() - Math.Floor(nrfiChildDocument.NRFI4.GetValueOrDefault() * DeductionMedianRates.deductionMedianRates[nrfiChildDocument.NRFI2.Substring(0, 4)]));

            bool assertion = condition1_NRFI3Simple && condition2_NRFI2Exists && condition3_IITR494NotEqualToNRFI4;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000641",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident foreign income amount using the simple self-assessment method is incorrect",
                    LongDescription = @"When the simple self-assessment method is selected on the Non-Resident Foreign Income (NRFI) schedule, the Gross foreign income amount, less standard occupation deductions must equal the Non-resident foreign income amount on the parent form",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.iitrParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000641" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000642

        /* VR.ATO.IITR.000642
        When the overseas assessed method is selected on the Non-Resident Foreign Income (NRFI) schedule, the Foreign tax return taxable income must equal the Non-resident foreign income amount on the parent form, with the exception that when Foreign tax return taxable income on the NRFI schedule is negative, the Non-resident foreign income amount on the parent form must be zero (0).

        Legacy Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 <> NULL AND ^NRFI3 = 'Overseas' AND ((^NRFI10 >= 0 AND ^IITR494 <> ^NRFI10) OR (^NRFI10 < 0 AND ^IITR494 <> 0))

        Technical Business Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 <> NULL AND ^NRFI3 = 'Overseas' AND ((^NRFI10 >= 0 AND ^IITR494 <> ^NRFI10) OR (^NRFI10 < 0 AND ^IITR494 <> 0))

        Data Elements:
    
        ^IITR494 = RP.JFOREIGN:Income.Net.Amount
    
        ^NRFI3 = NRFI3
    
        ^NRFI10 = NRFI10
        */
        public static ProcessMessageDocument VR_ATO_IITR_000642(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR494 != null && nrfiChildDocument.NRFI3 == "Overseas" && ((nrfiChildDocument.NRFI10.GetValueOrDefault() >= 0 && report.iitrParentDocument.IITR494.GetValueOrDefault() != nrfiChildDocument.NRFI10.GetValueOrDefault()) || (nrfiChildDocument.NRFI10.GetValueOrDefault() < 0 && report.iitrParentDocument.IITR494.GetValueOrDefault() != 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000642",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident foreign income amount using the overseas assessed method is incorrect",
                    LongDescription = @"When the overseas assessed method is selected on the Non-Resident Foreign Income (NRFI) schedule, the Foreign tax return taxable income must equal the Non-resident foreign income amount on the parent form, with the exception that when Foreign tax return taxable income on the NRFI schedule is negative, the Non-resident foreign income amount on the parent form must be zero (0)",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.iitrParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000642" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000643

        /* VR.ATO.IITR.000643
        When the comprehensive assessment method is selected on the Non-Resident Foreign Income (NRFI) schedule, the sum of Total salary or wages amount, Total allowance amount, Total government allowances amount, Total pension amount, Net business income, Total personal services income, Total interest amount, Total dividend amount, Net partnership and trust income amount, Net capital gains amount, Net rent amount and Total other income amount, less the sum of Total employee work related deduction amount, Total depreciation amount, Total interest and dividend amount, Total undeducted purchase price of a foreign pension amount, Total personal super contributions for business, Total other deduction amount and Prior year tax losses amount, must equal the Non-resident foreign income amount on the parent form, with the exception that when the sum of amounts on the NRFI schedule is negative, the Non-resident foreign income amount on the parent form must be zero (0).

        Legacy Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 <> NULL AND ^NRFI3 = 'Comprehensive' AND (((^NRFI11 + ^NRFI12 + ^NRFI13 + ^NRFI14 + ^NRFI15 + ^NRFI16 + ^NRFI17 + ^NRFI18 + ^NRFI19 + ^NRFI20 + ^NRFI21 + ^NRFI22) >= (^NRFI23 + ^NRFI24 + ^NRFI25 + ^NRFI26 + ^NRFI27 + ^NRFI28 + ^NRFI32) AND ^IITR494 <> ((^NRFI11 + ^NRFI12 + ^NRFI13 + ^NRFI14 + ^NRFI15 + ^NRFI16 + ^NRFI17 + ^NRFI18 + ^NRFI19 + ^NRFI20 + ^NRFI21 + ^NRFI22) - (^NRFI23 + ^NRFI24 + ^NRFI25 + ^NRFI26 + ^NRFI27 + ^NRFI28 + ^NRFI32))) OR ((^NRFI11 + ^NRFI12 + ^NRFI13 + ^NRFI14 + ^NRFI15 + ^NRFI16 + ^NRFI17 + ^NRFI18 + ^NRFI19 + ^NRFI20 + ^NRFI21 + ^NRFI22) < (^NRFI23 + ^NRFI24 + ^NRFI25 + ^NRFI26 + ^NRFI27 + ^NRFI28 + ^NRFI32) AND ^IITR494 <> 0))

        Technical Business Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 <> NULL AND ^NRFI3 = 'Comprehensive' AND (((^NRFI11 + ^NRFI12 + ^NRFI13 + ^NRFI14 + ^NRFI15 + ^NRFI16 + ^NRFI17 + ^NRFI18 + ^NRFI19 + ^NRFI20 + ^NRFI21 + ^NRFI22) >= (^NRFI23 + ^NRFI24 + ^NRFI25 + ^NRFI26 + ^NRFI27 + ^NRFI28 + ^NRFI32) AND ^IITR494 <> ((^NRFI11 + ^NRFI12 + ^NRFI13 + ^NRFI14 + ^NRFI15 + ^NRFI16 + ^NRFI17 + ^NRFI18 + ^NRFI19 + ^NRFI20 + ^NRFI21 + ^NRFI22) - (^NRFI23 + ^NRFI24 + ^NRFI25 + ^NRFI26 + ^NRFI27 + ^NRFI28 + ^NRFI32))) OR ((^NRFI11 + ^NRFI12 + ^NRFI13 + ^NRFI14 + ^NRFI15 + ^NRFI16 + ^NRFI17 + ^NRFI18 + ^NRFI19 + ^NRFI20 + ^NRFI21 + ^NRFI22) < (^NRFI23 + ^NRFI24 + ^NRFI25 + ^NRFI26 + ^NRFI27 + ^NRFI28 + ^NRFI32) AND ^IITR494 <> 0))

        Data Elements:
    
        ^IITR494 = RP.JFOREIGN:Income.Net.Amount
    
        ^NRFI3 = NRFI3
    
        ^NRFI11 = NRFI11
    
        ^NRFI12 = NRFI12
    
        ^NRFI13 = NRFI13
    
        ^NRFI14 = NRFI14
    
        ^NRFI15 = NRFI15
    
        ^NRFI16 = NRFI16
    
        ^NRFI17 = NRFI17
    
        ^NRFI18 = NRFI18
    
        ^NRFI19 = NRFI19
    
        ^NRFI20 = NRFI20
    
        ^NRFI21 = NRFI21
    
        ^NRFI22 = NRFI22
    
        ^NRFI23 = NRFI23
    
        ^NRFI24 = NRFI24
    
        ^NRFI25 = NRFI25
    
        ^NRFI26 = NRFI26
    
        ^NRFI27 = NRFI27
    
        ^NRFI28 = NRFI28
    
        ^NRFI32 = NRFI32
        */
        public static ProcessMessageDocument VR_ATO_IITR_000643(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR494 != null && nrfiChildDocument.NRFI3 == "Comprehensive" && (((nrfiChildDocument.NRFI11.GetValueOrDefault() + nrfiChildDocument.NRFI12.GetValueOrDefault() + nrfiChildDocument.NRFI13.GetValueOrDefault() + nrfiChildDocument.NRFI14.GetValueOrDefault() + nrfiChildDocument.NRFI15.GetValueOrDefault() + nrfiChildDocument.NRFI16.GetValueOrDefault() + nrfiChildDocument.NRFI17.GetValueOrDefault() + nrfiChildDocument.NRFI18.GetValueOrDefault() + nrfiChildDocument.NRFI19.GetValueOrDefault() + nrfiChildDocument.NRFI20.GetValueOrDefault() + nrfiChildDocument.NRFI21.GetValueOrDefault() + nrfiChildDocument.NRFI22.GetValueOrDefault()) >= (nrfiChildDocument.NRFI23.GetValueOrDefault() + nrfiChildDocument.NRFI24.GetValueOrDefault() + nrfiChildDocument.NRFI25.GetValueOrDefault() + nrfiChildDocument.NRFI26.GetValueOrDefault() + nrfiChildDocument.NRFI27.GetValueOrDefault() + nrfiChildDocument.NRFI28.GetValueOrDefault() + nrfiChildDocument.NRFI32.GetValueOrDefault()) && report.iitrParentDocument.IITR494.GetValueOrDefault() != (nrfiChildDocument.NRFI11.GetValueOrDefault() + nrfiChildDocument.NRFI12.GetValueOrDefault() + nrfiChildDocument.NRFI13.GetValueOrDefault() + nrfiChildDocument.NRFI14.GetValueOrDefault() + nrfiChildDocument.NRFI15.GetValueOrDefault() + nrfiChildDocument.NRFI16.GetValueOrDefault() + nrfiChildDocument.NRFI17.GetValueOrDefault() + nrfiChildDocument.NRFI18.GetValueOrDefault() + nrfiChildDocument.NRFI19.GetValueOrDefault() + nrfiChildDocument.NRFI20.GetValueOrDefault() + nrfiChildDocument.NRFI21.GetValueOrDefault() + nrfiChildDocument.NRFI22.GetValueOrDefault()) - (nrfiChildDocument.NRFI23.GetValueOrDefault() + nrfiChildDocument.NRFI24.GetValueOrDefault() + nrfiChildDocument.NRFI25.GetValueOrDefault() + nrfiChildDocument.NRFI26.GetValueOrDefault() + nrfiChildDocument.NRFI27.GetValueOrDefault() + nrfiChildDocument.NRFI28.GetValueOrDefault() + nrfiChildDocument.NRFI32.GetValueOrDefault())) || ((nrfiChildDocument.NRFI11.GetValueOrDefault() + nrfiChildDocument.NRFI12.GetValueOrDefault() + nrfiChildDocument.NRFI13.GetValueOrDefault() + nrfiChildDocument.NRFI14.GetValueOrDefault() + nrfiChildDocument.NRFI15.GetValueOrDefault() + nrfiChildDocument.NRFI16.GetValueOrDefault() + nrfiChildDocument.NRFI17.GetValueOrDefault() + nrfiChildDocument.NRFI18.GetValueOrDefault() + nrfiChildDocument.NRFI19.GetValueOrDefault() + nrfiChildDocument.NRFI20.GetValueOrDefault() + nrfiChildDocument.NRFI21.GetValueOrDefault() + nrfiChildDocument.NRFI22.GetValueOrDefault()) < (nrfiChildDocument.NRFI23.GetValueOrDefault() + nrfiChildDocument.NRFI24.GetValueOrDefault() + nrfiChildDocument.NRFI25.GetValueOrDefault() + nrfiChildDocument.NRFI26.GetValueOrDefault() + nrfiChildDocument.NRFI27.GetValueOrDefault() + nrfiChildDocument.NRFI28.GetValueOrDefault() + nrfiChildDocument.NRFI32.GetValueOrDefault()) && report.iitrParentDocument.IITR494.GetValueOrDefault() != 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000643",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident foreign income amount using the comprehensive assessment method is incorrect",
                    LongDescription = @"When the comprehensive assessment method is selected on the Non-Resident Foreign Income (NRFI) schedule, the sum of Total salary or wages amount, Total allowance amount, Total government allowances amount, Total pension amount, Net business income, Total personal services income, Total interest amount, Total dividend amount, Net partnership and trust income amount, Net capital gains amount, Net rent amount and Total other income amount, less the sum of Total employee work related deduction amount, Total depreciation amount, Total interest and dividend amount, Total undeducted purchase price of a foreign pension amount, Total personal super contributions for business, Total other deduction amount and Prior year tax losses amount, must equal the Non-resident foreign income amount on the parent form, with the exception that when the sum of amounts on the NRFI schedule is negative, the Non-resident foreign income amount on the parent form must be zero (0)",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.iitrParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000643" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000644

        /* VR.ATO.IITR.000644
        Foreign tax return financial year provided on Non-Resident Foreign Income (NRFI) schedule must be within one year of Year of return on parent form

        Legacy Rule Format:
            CountDocument('NRFI') = 1 AND ^NRFI7 <> NULL AND ^NRFI7 <> ^IITR10 +/- 1

        Technical Business Rule Format:
            CountDocument('NRFI') = 1 AND ^NRFI7 <> NULL AND ^NRFI7 <> ^IITR10 +/- 1

        Data Elements:
    
        ^IITR10 = RP:Report.TargetFinancial.Year
    
        ^NRFI7 = NRFI7
        */
        public static ProcessMessageDocument VR_ATO_IITR_000644(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (nrfiChildDocument.NRFI7 != null && OutsideRange((Decimal)nrfiChildDocument.NRFI7, (Decimal)report.iitrParentDocument.IITR10, 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000644",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign tax return financial year is invalid",
                    LongDescription = @"Foreign tax return financial year provided on Non-Resident Foreign Income (NRFI) schedule must be within one year of Year of return on parent form",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000644" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000645

        /* VR.ATO.IITR.000645
        Unable to confirm whether the correct Non-resident foreign income amount has been provided for the simple self-assessment method, as the Foreign occupation code provided on the Non-Resident Foreign Income (NRFI) schedule is not recognised. Record the details of the message and contact the agency service desk.
    
        Legacy Rule Format:
        CountDocument('NRFI') = 1 AND ^NRFI3 = 'Simple' AND ^NRFI2 <> NULL AND NotInDomain(Substring(^NRFI2,1,4), 'OCCUPATION PREFIX CODES')

        Technical Business Rule Format:
        CountDocument('NRFI') = 1 AND ^NRFI3 = 'Simple' AND ^NRFI2 <> NULL AND NotInDomain(Substring(^NRFI2,1,4), 'OCCUPATION PREFIX CODES')
    
        Data Elements:
    
        ^NRFI2 = NRFI2
    
        ^NRFI3 = NRFI3
        */
        public static ProcessMessageDocument VR_ATO_IITR_000645(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool condition1_NRFI3Simple = (nrfiChildDocument.NRFI3 != null && nrfiChildDocument.NRFI3 == "Simple");
            bool condition3_NRFI2DeductionMedianRates = (nrfiChildDocument.NRFI2 != null && DeductionMedianRates.deductionMedianRates != null && !DeductionMedianRates.deductionMedianRates.Keys.Contains(nrfiChildDocument.NRFI2.Substring(0, 4)));
            bool assertion = (condition1_NRFI3Simple && condition3_NRFI2DeductionMedianRates);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000645",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Foreign occupation code is unknown",
                    LongDescription = @"Unable to calculate the correct Non-resident foreign income amount for the simple self-assessment method, as the Foreign occupation code provided on the Non-Resident Foreign Income (NRFI) schedule is not configured. Record the details of the message and contact the agency service desk.",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.iitrParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000645" },
                        new ProcessMessageParameter() { Name = "NRFI3", Value = nrfiChildDocument.NRFI3.ToString() },
                        new ProcessMessageParameter() { Name = "NRFI2", Value = nrfiChildDocument.NRFI2.ToString() }
                    }
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.000651

        /* VR.ATO.IITR.000651
        When a Non-Resident Foreign Income (NRFI) schedule is attached, the Non-resident foreign income amount must be provided

        Legacy Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 = NULL

        Technical Business Rule Format:
            CountDocument('NRFI') = 1 AND ^IITR494 = NULL

        Data Elements:
    
        ^IITR494 = RP.JFOREIGN:Income.Net.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_000651(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.iitrParentDocument.IITR494 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.000651",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident foreign income amount must be provided",
                    LongDescription = @"When a Non-Resident Foreign Income (NRFI) schedule is attached, the Non-resident foreign income amount must be provided",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.iitrParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.000651" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.310094

        /* VR.ATO.IITR.310094
        Invalid context. Period start date for the Non-Resident Foreign Income Schedule must be the same as the Period start date of the parent form.

        Legacy Rule Format:
            (CountDocument('NRFI') = 1) AND (NRFI:^RP:StartDate <> ^RP:StartDate)

        Technical Business Rule Format:
            (CountDocument('NRFI') = 1) AND (NRFI:^RP:StartDate <> ^RP:StartDate)

        Data Elements:
    
        ^StartDate = StartDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_310094(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (nrfiChildDocument.RPStartDate != report.iitrParentDocument.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.310094",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date for the Non-Resident Foreign Income Schedule is invalid",
                    LongDescription = @"Invalid context. Period start date for the Non-Resident Foreign Income Schedule must be the same as the Period start date of the parent form.",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + nrfiChildDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.310094" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.310095

        /* VR.ATO.IITR.310095
        Invalid context. Period end date for the Non-Resident Foreign Income Schedule must be the same as the Period end date of the parent form.

        Legacy Rule Format:
            (CountDocument('NRFI') = 1) AND (NRFI:^RP:EndDate <> ^RP:EndDate)

        Technical Business Rule Format:
            (CountDocument('NRFI') = 1) AND (NRFI:^RP:EndDate <> ^RP:EndDate)

        Data Elements:
    
        ^EndDate = EndDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_310095(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (nrfiChildDocument.RPEndDate != report.iitrParentDocument.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.310095",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date for the Non-Resident Foreign Income Schedule is invalid",
                    LongDescription = @"Invalid context. Period end date for the Non-Resident Foreign Income Schedule must be the same as the Period end date of the parent form.",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + nrfiChildDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.310095" } },
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.600046

        /* VR.ATO.IITR.600046
        The TFN provided for the Non-Resident Foreign Income schedule must be the same as the TFN of the parent form

        Legacy Rule Format:
            (CountDocument('NRFI') = 1) AND (NRFI:^RP:TFN <> ^RP:TFN)

        Technical Business Rule Format:
            (CountDocument('NRFI') = 1) AND (NRFI:^RP:TFN <> ^RP:TFN)

        Data Elements:
    
        ^TFN = TFN
        */
        public static ProcessMessageDocument VR_ATO_IITR_600046(IITR2026Crossform report, NRFI2024 nrfiChildDocument)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (nrfiChildDocument.RPIdentifierTFN != report.iitrParentDocument.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.600046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax file number invalid",
                    LongDescription = @"The TFN provided for the Non-Resident Foreign Income schedule must be the same as the TFN of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + nrfiChildDocument.RPId + "']/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.600046" } },
                };
            }

            return processMessage;
        }

        #endregion
    }
}