﻿using Ato.EN.IntegrationServices.CodeGenerationRNTLPRPTY;
using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Ato.EN.IntegrationServices.CodeGenerationIITR
{
    public partial class IITR2026CrossformValidator
    {
        /// <summary>
        /// Takes a report to validate against all RNTLPRPTY rules and store the results in the response paramaeter object.
        /// </summary>
        /// <param name="report">Document to be validated with RNTLPRPTY rules.</param>
        /// <param name="response">Response to add error codes to.</param>
        public static void ValidateRNTLPRPTY(IITR2026Crossform report, List<ProcessMessageDocument> response)
        {
            AddProcessMessage(VR_ATO_IITR_730313(report), response);
            AddProcessMessage(VR_ATO_IITR_310083(report), response);

            #region RNTLPRPTY Child Present
            if (report.rntlprptyChildDocuments != null && Count(report.rntlprptyChildDocuments) == 1)
            {
                RNTLPRPTY2019 rntlprptyChildDocument = report.rntlprptyChildDocuments[0];

                AddProcessMessage(VR_ATO_IITR_730214(report), response);
                AddProcessMessage(VR_ATO_IITR_730215(report), response);
                AddProcessMessage(VR_ATO_IITR_730216(report), response);
                AddProcessMessage(VR_ATO_IITR_730217(report), response);
                AddProcessMessage(VR_ATO_IITR_730218(report), response);
                AddProcessMessage(VR_ATO_IITR_730219(report), response);
                AddProcessMessage(VR_ATO_IITR_730220(report), response);
                AddProcessMessage(VR_ATO_IITR_730221(report), response);

                
                AddProcessMessage(VR_ATO_IITR_310084(report), response);

                #region Repeating RPPrptySeqNumContext
                if (rntlprptyChildDocument.RPPrptySeqNumContextCollection != null)
                {
                    foreach (RNTLPRPTY2019.RPPrptySeqNumContext rntlprptyContext in rntlprptyChildDocument.RPPrptySeqNumContextCollection)
                    {
                        AddProcessMessage(VR_ATO_IITR_730310(report, rntlprptyContext), response);
                        AddProcessMessage(VR_ATO_IITR_730311(report, rntlprptyContext), response);
                        AddProcessMessage(VR_ATO_IITR_730312(report, rntlprptyContext), response);
                    }
                }
                #endregion // Repeating RPPrptySeqNumContext
            }

            #endregion // RNTLPRPTY Child Present
        }

        public static string[] BaseSequencesFullShare(IITR2026Crossform report)
        {
            return report.rntlprptyChildDocuments[0].RPPrptySeqNumContextCollection.FindAll(context => context.RNTLPRPTY104 == 100).Select(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue).ToArray();
        }

        public static string[] BaseSequencesPartShare(IITR2026Crossform report)
        {
            // Calculate the sum of the other deductions amounts
            return report.rntlprptyChildDocuments[0].RPPrptySeqNumContextCollection.FindAll(context => context.RNTLPRPTY104 < 100).Select(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue).ToArray();
        }

        #region VR.ATO.IITR.730313

        /* VR.ATO.IITR.730313
        The number of Multi-Property Rental schedules must not exceed 1

        Legacy Rule Format:
            CountDocument('RNTLPRPTY') > 1

        Technical Business Rule Format:
            CountDocument('RNTLPRPTY') > 1

        Data Elements:
        */
        public static ProcessMessageDocument VR_ATO_IITR_730313(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = Count(report.rntlprptyChildDocuments) > 1;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730313",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Multi-Property Rental schedules must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730313" } }
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730214

        /* VR.ATO.IITR.730214
        When any Rental income, Other rental related income, Your share of rental income, or Your share of other rental related income amount is present on the attached Multi-Property Rental schedule, then Gross rent must be provided

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR334 = NULL AND (AnyOccurrence(^RNTLPRPTY114, ^RNTLPRPTY114 <> NULL) OR AnyOccurrence(^RNTLPRPTY115, ^RNTLPRPTY115 <> NULL) OR AnyOccurrence(^RNTLPRPTY136, ^RNTLPRPTY136 <> NULL) OR AnyOccurrence(^RNTLPRPTY137, ^RNTLPRPTY137 <> NULL))

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR334 = NULL AND (AnyOccurrence(^RNTLPRPTY114, ^RNTLPRPTY114 <> NULL) OR AnyOccurrence(^RNTLPRPTY115, ^RNTLPRPTY115 <> NULL) OR AnyOccurrence(^RNTLPRPTY136, ^RNTLPRPTY136 <> NULL) OR AnyOccurrence(^RNTLPRPTY137, ^RNTLPRPTY137 <> NULL))

        Data Elements:
    
        ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount
    
        ^RNTLPRPTY114 = RNTLPRPTY114
    
        ^RNTLPRPTY115 = RNTLPRPTY115
    
        ^RNTLPRPTY136 = RNTLPRPTY136
    
        ^RNTLPRPTY137 = RNTLPRPTY137
        */
        public static ProcessMessageDocument VR_ATO_IITR_730214(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = report.iitrParentDocument.IITR334 == null && ((report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Any(context => context.RNTLPRPTY114 != null || context.RNTLPRPTY115 != null) : false) || (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Any(context => context.RNTLPRPTY136 != null || context.RNTLPRPTY137 != null) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730214",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent must be provided",
                    LongDescription = @"When any Rental income, Other rental related income, Your share of rental income, or Your share of other rental related income amount is present on the attached Multi-Property Rental schedule, then Gross rent must be provided",
                    Location = "/xbrli:xbrl/tns:Income.RealEstateProperty.Rental.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730214" } },
                };

                RNTLPRPTY2019.RPPrptySeqNumTotalContext totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RNTLPRPTY114 != null || context.RNTLPRPTY115 != null) : null);
                RNTLPRPTY2019.RPPrptySeqNumPortionContext portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RNTLPRPTY136 != null || context.RNTLPRPTY137 != null) : null);

                if (totalContext != null && portionContext != null)
                {
                    if (Convert.ToInt16(totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) < Convert.ToInt16(portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue))
                        portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                    else
                        totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                }

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR334", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY114", Value = (totalContext != null && totalContext.RNTLPRPTY114.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY114) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY115", Value = (totalContext != null && totalContext.RNTLPRPTY115.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY115) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY136", Value = (portionContext != null && portionContext.RNTLPRPTY136.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY136) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY137", Value = (portionContext != null && portionContext.RNTLPRPTY137.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY137) : "null") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730215

        /* VR.ATO.IITR.730215
        Gross rent must equal the sum of all Rental income and Other rental income, where the Ownership percentage is 100%, plus the sum of all Your share of rental income and Your share of other rental related income, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR334 <> NULL AND ^IITR334 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY114, ^RNTLPRPTY136)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY115, ^RNTLPRPTY137)),0)

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR334 <> NULL AND ^IITR334 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY114, ^RNTLPRPTY136)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY115, ^RNTLPRPTY137)),0)

        Data Elements:
    
        ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount
    
        ^RNTLPRPTY104 = RNTLPRPTY104
    
        ^RNTLPRPTY114 = RNTLPRPTY114
    
        ^RNTLPRPTY115 = RNTLPRPTY115
    
        ^RNTLPRPTY136 = RNTLPRPTY136
    
        ^RNTLPRPTY137 = RNTLPRPTY137
        */
        public static ProcessMessageDocument VR_ATO_IITR_730215(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            string[] baseSequencesFullShare = BaseSequencesFullShare(report);
            string[] baseSequencesPartShare = BaseSequencesPartShare(report);

            decimal rntlprpty114Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.FindAll(context => baseSequencesFullShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY114.GetValueOrDefault()) : 0);
            decimal rntlprpty115Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.FindAll(context => baseSequencesFullShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY115.GetValueOrDefault()) : 0);
            decimal rntlprpty136Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.FindAll(context => baseSequencesPartShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY136.GetValueOrDefault()) : 0);
            decimal rntlprpty137Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.FindAll(context => baseSequencesPartShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY137.GetValueOrDefault()) : 0);
            decimal vr730215Total = Math.Truncate(rntlprpty114Total + rntlprpty115Total + rntlprpty136Total + rntlprpty137Total);

            bool assertion = report.iitrParentDocument.IITR334 != null && report.iitrParentDocument.IITR334.GetValueOrDefault() != vr730215Total;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730215",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent amount is incorrect",
                    LongDescription = @"Gross rent must equal the sum of all Rental income and Other rental income, where the Ownership percentage is 100%, plus the sum of all Your share of rental income and Your share of other rental related income, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule",
                    Location = "/xbrli:xbrl/tns:Income.RealEstateProperty.Rental.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730215" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR334", Value = report.iitrParentDocument.IITR334.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY114) WHERE RNTLPRPTY104 = 100", Value = rntlprpty114Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY115) WHERE RNTLPRPTY104 = 100", Value = rntlprpty115Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY136) WHERE RNTLPRPTY104 < 100", Value = rntlprpty136Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY137) WHERE RNTLPRPTY104 < 100", Value = rntlprpty137Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Gross Rent Total", Value = vr730215Total.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730216

        /* VR.ATO.IITR.730216
        When any Interest on loans, or Your share of interest on loans amount is present on the attached Multi-Property Rental schedule, then Interest deductions must be provided

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR335 = NULL AND (AnyOccurrence(^RNTLPRPTY124, ^RNTLPRPTY124 <> NULL) OR AnyOccurrence(^RNTLPRPTY146, ^RNTLPRPTY146 <> NULL))

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR335 = NULL AND (AnyOccurrence(^RNTLPRPTY124, ^RNTLPRPTY124 <> NULL) OR AnyOccurrence(^RNTLPRPTY146, ^RNTLPRPTY146 <> NULL))

        Data Elements:
    
        ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount
    
        ^RNTLPRPTY124 = RNTLPRPTY124
    
        ^RNTLPRPTY146 = RNTLPRPTY146
        */
        public static ProcessMessageDocument VR_ATO_IITR_730216(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = report.iitrParentDocument.IITR335 == null && ((report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Any(context => context.RNTLPRPTY124 != null) : false) || (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Any(context => context.RNTLPRPTY146 != null) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730216",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest deductions must be provided",
                    LongDescription = @"When any Interest on loans, or Your share of interest on loans amount is present on the attached Multi-Property Rental schedule, then Interest deductions must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730216" } },
                };

                RNTLPRPTY2019.RPPrptySeqNumTotalContext totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RNTLPRPTY124 != null) : null);
                RNTLPRPTY2019.RPPrptySeqNumPortionContext portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RNTLPRPTY146 != null) : null);

                if (totalContext != null && portionContext != null)
                {
                    if (Convert.ToInt16(totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) < Convert.ToInt16(portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue))
                        portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                    else
                        totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                }

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR335", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY124", Value = (totalContext != null && totalContext.RNTLPRPTY124.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY124) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY146", Value = (portionContext != null && portionContext.RNTLPRPTY146.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY146) : "null") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730217

        /* VR.ATO.IITR.730217
            Interest deductions must equal the sum of all Interest on loans, where the Ownership percentage is 100%, plus the sum of all Your share of interest on loans, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR335 <> NULL AND ^IITR335 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY124, ^RNTLPRPTY146)),0)

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR335 <> NULL AND ^IITR335 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY124, ^RNTLPRPTY146)),0)

        Data Elements:
    
            ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount
    
            ^RNTLPRPTY104 = RNTLPRPTY104
    
            ^RNTLPRPTY124 = RNTLPRPTY124
    
            ^RNTLPRPTY146 = RNTLPRPTY146
        */
        public static ProcessMessageDocument VR_ATO_IITR_730217(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            string[] baseSequencesFullShare = BaseSequencesFullShare(report);
            string[] baseSequencesPartShare = BaseSequencesPartShare(report);

            decimal rntlprpty124Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.FindAll(context => baseSequencesFullShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY124.GetValueOrDefault()) : 0);
            decimal rntlprpty146Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.FindAll(context => baseSequencesPartShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY146.GetValueOrDefault()) : 0);
            decimal vr730217Total = Math.Truncate(rntlprpty124Total + rntlprpty146Total);

            bool assertion = report.iitrParentDocument.IITR335 != null && report.iitrParentDocument.IITR335.GetValueOrDefault() != vr730217Total;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730217",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest deductions amount is incorrect",
                    LongDescription = @"Interest deductions must equal the sum of all Interest on loans, where the Ownership percentage is 100%, plus the sum of all Your share of interest on loans, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730217" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR335", Value = report.iitrParentDocument.IITR335.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY124) WHERE RNTLPRPTY104 = 100", Value = rntlprpty124Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY146) WHERE RNTLPRPTY104 < 100", Value = rntlprpty146Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Interest on Loans Total", Value = vr730217Total.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730218

        /* VR.ATO.IITR.730218
        When any Capital works deductions, or Your share of capital works deductions amount is present on the attached Multi-Property Rental schedule, then Capital works deductions must be provided

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR336 = NULL AND (AnyOccurrence(^RNTLPRPTY130, ^RNTLPRPTY130 <> NULL) OR AnyOccurrence(^RNTLPRPTY152, ^RNTLPRPTY152 <> NULL))

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR336 = NULL AND (AnyOccurrence(^RNTLPRPTY130, ^RNTLPRPTY130 <> NULL) OR AnyOccurrence(^RNTLPRPTY152, ^RNTLPRPTY152 <> NULL))

        Data Elements:
    
        ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount
    
        ^RNTLPRPTY130 = RNTLPRPTY130
    
        ^RNTLPRPTY152 = RNTLPRPTY152
        */
        public static ProcessMessageDocument VR_ATO_IITR_730218(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = report.iitrParentDocument.IITR336 == null && ((report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Any(context => context.RNTLPRPTY130 != null) : false) || (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Any(context => context.RNTLPRPTY152 != null) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital works deductions must be provided",
                    LongDescription = @"When any Capital works deductions, or Your share of capital works deductions amount is present on the attached Multi-Property Rental schedule, then Capital works deductions must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.CapitalWorksDeductions.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730218" } },
                };

                RNTLPRPTY2019.RPPrptySeqNumTotalContext totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RNTLPRPTY130 != null) : null);
                RNTLPRPTY2019.RPPrptySeqNumPortionContext portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RNTLPRPTY152 != null) : null);

                if (totalContext != null && portionContext != null)
                {
                    if (Convert.ToInt16(totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) < Convert.ToInt16(portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue))
                        portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                    else
                        totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                }

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR336", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY130", Value = (totalContext != null && totalContext.RNTLPRPTY130.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY130) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY152", Value = (portionContext != null && portionContext.RNTLPRPTY152.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY152) : "null") });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730219

        /* VR.ATO.IITR.730219
        Capital works deductions must equal the sum of all Capital works deductions, where the Ownership percentage is 100%, plus the sum of all Your share capital works deductions, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR336 <> NULL AND ^IITR336 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY130, ^RNTLPRPTY152)),0)

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR336 <> NULL AND ^IITR336 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY130, ^RNTLPRPTY152)),0)

        Data Elements:
    
        ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount
    
        ^RNTLPRPTY104 = RNTLPRPTY104
    
        ^RNTLPRPTY130 = RNTLPRPTY130
    
        ^RNTLPRPTY152 = RNTLPRPTY152
        */
        public static ProcessMessageDocument VR_ATO_IITR_730219(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            string[] baseSequencesFullShare = BaseSequencesFullShare(report);
            string[] baseSequencesPartShare = BaseSequencesPartShare(report);

            decimal rntlprpty130Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.FindAll(context => baseSequencesFullShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY130.GetValueOrDefault()) : 0);
            decimal rntlprpty152Total = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.FindAll(context => baseSequencesPartShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)).Sum(context => context.RNTLPRPTY152.GetValueOrDefault()) : 0);
            decimal vr730219Total = Math.Truncate(rntlprpty130Total + rntlprpty152Total);

            bool assertion = report.iitrParentDocument.IITR336 != null && report.iitrParentDocument.IITR336.GetValueOrDefault() != vr730219Total;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730219",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital works deductions amount is incorrect",
                    LongDescription = @"Capital works deductions must equal the sum of all Capital works deductions, where the Ownership percentage is 100%, plus the sum of all Your share capital works deductions, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.CapitalWorksDeductions.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730219" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR336", Value = report.iitrParentDocument.IITR336.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY130) WHERE RNTLPRPTY104 = 100", Value = rntlprpty130Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY152) WHERE RNTLPRPTY104 < 100", Value = rntlprpty152Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Capital Works Deductions Total", Value = vr730219Total.ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730220

        /* VR.ATO.IITR.730220
        When any other rental deductions, or your share of other rental deductions amount is present on the attached Multi-Property Rental schedule, then Other rental deductions must be provided

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR337 = NULL AND (AnyOccurrence(^RNTLPRPTY116, ^RNTLPRPTY116 <> NULL) OR AnyOccurrence(^RNTLPRPTY117, ^RNTLPRPTY117 <> NULL) OR AnyOccurrence(^RNTLPRPTY118, ^RNTLPRPTY118 <> NULL) OR AnyOccurrence(^RNTLPRPTY119, ^RNTLPRPTY119 <> NULL) OR AnyOccurrence(^RNTLPRPTY120, ^RNTLPRPTY120 <> NULL) OR AnyOccurrence(^RNTLPRPTY121, ^RNTLPRPTY121 <> NULL) OR AnyOccurrence(^RNTLPRPTY122, ^RNTLPRPTY122 <> NULL) OR AnyOccurrence(^RNTLPRPTY123, ^RNTLPRPTY123 <> NULL) OR AnyOccurrence(^RNTLPRPTY125, ^RNTLPRPTY125 <> NULL) OR AnyOccurrence(^RNTLPRPTY126, ^RNTLPRPTY126 <> NULL) OR AnyOccurrence(^RNTLPRPTY127, ^RNTLPRPTY127 <> NULL) OR AnyOccurrence(^RNTLPRPTY128, ^RNTLPRPTY128 <> NULL) OR AnyOccurrence(^RNTLPRPTY129, ^RNTLPRPTY129 <> NULL) OR AnyOccurrence(^RNTLPRPTY131, ^RNTLPRPTY131 <> NULL) OR AnyOccurrence(^RNTLPRPTY132, ^RNTLPRPTY132 <> NULL) OR AnyOccurrence(^RNTLPRPTY133, ^RNTLPRPTY133 <> NULL) OR AnyOccurrence(^RNTLPRPTY134, ^RNTLPRPTY134 <> NULL) OR AnyOccurrence(^RNTLPRPTY138, ^RNTLPRPTY138 <> NULL) OR AnyOccurrence(^RNTLPRPTY139, ^RNTLPRPTY139 <> NULL) OR AnyOccurrence(^RNTLPRPTY140, ^RNTLPRPTY140 <> NULL) OR AnyOccurrence(^RNTLPRPTY141, ^RNTLPRPTY141 <> NULL) OR AnyOccurrence(^RNTLPRPTY142, ^RNTLPRPTY142 <> NULL) OR AnyOccurrence(^RNTLPRPTY143, ^RNTLPRPTY143 <> NULL) OR AnyOccurrence(^RNTLPRPTY144, ^RNTLPRPTY144 <> NULL) OR AnyOccurrence(^RNTLPRPTY145, ^RNTLPRPTY145 <> NULL) OR AnyOccurrence(^RNTLPRPTY147, ^RNTLPRPTY147 <> NULL) OR AnyOccurrence(^RNTLPRPTY148, ^RNTLPRPTY148 <> NULL) OR AnyOccurrence(^RNTLPRPTY149, ^RNTLPRPTY149 <> NULL) OR AnyOccurrence(^RNTLPRPTY150, ^RNTLPRPTY150 <> NULL) OR AnyOccurrence(^RNTLPRPTY151, ^RNTLPRPTY151 <> NULL) OR AnyOccurrence(^RNTLPRPTY153, ^RNTLPRPTY153 <> NULL) OR AnyOccurrence(^RNTLPRPTY154, ^RNTLPRPTY154 <> NULL) OR AnyOccurrence(^RNTLPRPTY155, ^RNTLPRPTY155 <> NULL) OR AnyOccurrence(^RNTLPRPTY156, ^RNTLPRPTY156 <> NULL))

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR337 = NULL AND (AnyOccurrence(^RNTLPRPTY116, ^RNTLPRPTY116 <> NULL) OR AnyOccurrence(^RNTLPRPTY117, ^RNTLPRPTY117 <> NULL) OR AnyOccurrence(^RNTLPRPTY118, ^RNTLPRPTY118 <> NULL) OR AnyOccurrence(^RNTLPRPTY119, ^RNTLPRPTY119 <> NULL) OR AnyOccurrence(^RNTLPRPTY120, ^RNTLPRPTY120 <> NULL) OR AnyOccurrence(^RNTLPRPTY121, ^RNTLPRPTY121 <> NULL) OR AnyOccurrence(^RNTLPRPTY122, ^RNTLPRPTY122 <> NULL) OR AnyOccurrence(^RNTLPRPTY123, ^RNTLPRPTY123 <> NULL) OR AnyOccurrence(^RNTLPRPTY125, ^RNTLPRPTY125 <> NULL) OR AnyOccurrence(^RNTLPRPTY126, ^RNTLPRPTY126 <> NULL) OR AnyOccurrence(^RNTLPRPTY127, ^RNTLPRPTY127 <> NULL) OR AnyOccurrence(^RNTLPRPTY128, ^RNTLPRPTY128 <> NULL) OR AnyOccurrence(^RNTLPRPTY129, ^RNTLPRPTY129 <> NULL) OR AnyOccurrence(^RNTLPRPTY131, ^RNTLPRPTY131 <> NULL) OR AnyOccurrence(^RNTLPRPTY132, ^RNTLPRPTY132 <> NULL) OR AnyOccurrence(^RNTLPRPTY133, ^RNTLPRPTY133 <> NULL) OR AnyOccurrence(^RNTLPRPTY134, ^RNTLPRPTY134 <> NULL) OR AnyOccurrence(^RNTLPRPTY138, ^RNTLPRPTY138 <> NULL) OR AnyOccurrence(^RNTLPRPTY139, ^RNTLPRPTY139 <> NULL) OR AnyOccurrence(^RNTLPRPTY140, ^RNTLPRPTY140 <> NULL) OR AnyOccurrence(^RNTLPRPTY141, ^RNTLPRPTY141 <> NULL) OR AnyOccurrence(^RNTLPRPTY142, ^RNTLPRPTY142 <> NULL) OR AnyOccurrence(^RNTLPRPTY143, ^RNTLPRPTY143 <> NULL) OR AnyOccurrence(^RNTLPRPTY144, ^RNTLPRPTY144 <> NULL) OR AnyOccurrence(^RNTLPRPTY145, ^RNTLPRPTY145 <> NULL) OR AnyOccurrence(^RNTLPRPTY147, ^RNTLPRPTY147 <> NULL) OR AnyOccurrence(^RNTLPRPTY148, ^RNTLPRPTY148 <> NULL) OR AnyOccurrence(^RNTLPRPTY149, ^RNTLPRPTY149 <> NULL) OR AnyOccurrence(^RNTLPRPTY150, ^RNTLPRPTY150 <> NULL) OR AnyOccurrence(^RNTLPRPTY151, ^RNTLPRPTY151 <> NULL) OR AnyOccurrence(^RNTLPRPTY153, ^RNTLPRPTY153 <> NULL) OR AnyOccurrence(^RNTLPRPTY154, ^RNTLPRPTY154 <> NULL) OR AnyOccurrence(^RNTLPRPTY155, ^RNTLPRPTY155 <> NULL) OR AnyOccurrence(^RNTLPRPTY156, ^RNTLPRPTY156 <> NULL))

        Data Elements:
    
        ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
        ^RNTLPRPTY116 = RNTLPRPTY116
    
        ^RNTLPRPTY117 = RNTLPRPTY117
    
        ^RNTLPRPTY118 = RNTLPRPTY118
    
        ^RNTLPRPTY119 = RNTLPRPTY119
    
        ^RNTLPRPTY120 = RNTLPRPTY120
    
        ^RNTLPRPTY121 = RNTLPRPTY121
    
        ^RNTLPRPTY122 = RNTLPRPTY122
    
        ^RNTLPRPTY123 = RNTLPRPTY123
    
        ^RNTLPRPTY125 = RNTLPRPTY125
    
        ^RNTLPRPTY126 = RNTLPRPTY126
    
        ^RNTLPRPTY127 = RNTLPRPTY127
    
        ^RNTLPRPTY128 = RNTLPRPTY128
    
        ^RNTLPRPTY129 = RNTLPRPTY129
    
        ^RNTLPRPTY131 = RNTLPRPTY131
    
        ^RNTLPRPTY132 = RNTLPRPTY132
    
        ^RNTLPRPTY133 = RNTLPRPTY133
    
        ^RNTLPRPTY134 = RNTLPRPTY134
    
        ^RNTLPRPTY138 = RNTLPRPTY138
    
        ^RNTLPRPTY139 = RNTLPRPTY139
    
        ^RNTLPRPTY140 = RNTLPRPTY140
    
        ^RNTLPRPTY141 = RNTLPRPTY141
    
        ^RNTLPRPTY142 = RNTLPRPTY142
    
        ^RNTLPRPTY143 = RNTLPRPTY143
    
        ^RNTLPRPTY144 = RNTLPRPTY144
    
        ^RNTLPRPTY145 = RNTLPRPTY145
    
        ^RNTLPRPTY147 = RNTLPRPTY147
    
        ^RNTLPRPTY148 = RNTLPRPTY148
    
        ^RNTLPRPTY149 = RNTLPRPTY149
    
        ^RNTLPRPTY150 = RNTLPRPTY150
    
        ^RNTLPRPTY151 = RNTLPRPTY151
    
        ^RNTLPRPTY153 = RNTLPRPTY153
    
        ^RNTLPRPTY154 = RNTLPRPTY154
    
        ^RNTLPRPTY155 = RNTLPRPTY155
    
        ^RNTLPRPTY156 = RNTLPRPTY156
        */
        public static ProcessMessageDocument VR_ATO_IITR_730220(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = report.iitrParentDocument.IITR337 == null && ((report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Any(context => context.RNTLPRPTY116 != null || context.RNTLPRPTY117 != null || context.RNTLPRPTY118 != null || context.RNTLPRPTY119 != null || context.RNTLPRPTY120 != null || context.RNTLPRPTY121 != null || context.RNTLPRPTY122 != null || context.RNTLPRPTY123 != null || context.RNTLPRPTY125 != null || context.RNTLPRPTY126 != null || context.RNTLPRPTY127 != null || context.RNTLPRPTY128 != null || context.RNTLPRPTY129 != null || context.RNTLPRPTY131 != null || context.RNTLPRPTY132 != null || context.RNTLPRPTY133 != null || context.RNTLPRPTY134 != null) : false) || (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Any(context => context.RNTLPRPTY138 != null || context.RNTLPRPTY139 != null || context.RNTLPRPTY140 != null || context.RNTLPRPTY141 != null || context.RNTLPRPTY142 != null || context.RNTLPRPTY143 != null || context.RNTLPRPTY144 != null || context.RNTLPRPTY145 != null || context.RNTLPRPTY147 != null || context.RNTLPRPTY148 != null || context.RNTLPRPTY149 != null || context.RNTLPRPTY150 != null || context.RNTLPRPTY151 != null || context.RNTLPRPTY153 != null || context.RNTLPRPTY154 != null || context.RNTLPRPTY155 != null || context.RNTLPRPTY156 != null) : false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730220",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other rental deductions must be provided",
                    LongDescription = @"When any other rental deductions, or your share of other rental deductions amount is present on the attached Multi-Property Rental schedule, then Other rental deductions must be provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730220" } },
                };

                #region ProcessMessage Parameters

                // Get the first RP.{PrptySeqNum}.Total and RP.{PrptySeqNum}.Portion context that contains an Other rental deductions field that is not null
                RNTLPRPTY2019.RPPrptySeqNumTotalContext totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RNTLPRPTY116 != null || context.RNTLPRPTY117 != null || context.RNTLPRPTY118 != null || context.RNTLPRPTY119 != null || context.RNTLPRPTY120 != null || context.RNTLPRPTY121 != null || context.RNTLPRPTY122 != null || context.RNTLPRPTY123 != null || context.RNTLPRPTY125 != null || context.RNTLPRPTY126 != null || context.RNTLPRPTY127 != null || context.RNTLPRPTY128 != null || context.RNTLPRPTY129 != null || context.RNTLPRPTY131 != null || context.RNTLPRPTY132 != null || context.RNTLPRPTY133 != null || context.RNTLPRPTY134 != null) : null);
                RNTLPRPTY2019.RPPrptySeqNumPortionContext portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RNTLPRPTY138 != null || context.RNTLPRPTY139 != null || context.RNTLPRPTY140 != null || context.RNTLPRPTY141 != null || context.RNTLPRPTY142 != null || context.RNTLPRPTY143 != null || context.RNTLPRPTY144 != null || context.RNTLPRPTY145 != null || context.RNTLPRPTY147 != null || context.RNTLPRPTY148 != null || context.RNTLPRPTY149 != null || context.RNTLPRPTY150 != null || context.RNTLPRPTY151 != null || context.RNTLPRPTY153 != null || context.RNTLPRPTY154 != null || context.RNTLPRPTY155 != null || context.RNTLPRPTY156 != null) : null);

                // Check which of the above contexts has the lowest sequence number and retrieve the other context that has a matching sequence number (if applicable in the case of a RP.{PrptySeqNum}.Portion context)
                if (totalContext != null && portionContext != null)
                {
                    if (Convert.ToInt16(totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) < Convert.ToInt16(portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue))
                        portionContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == totalContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                    else
                        totalContext = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.Find(context => context.RentalPropertyFinancialDetailsSequenceDimensionValue == portionContext.RentalPropertyFinancialDetailsSequenceDimensionValue) : null);
                }
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR337", Value = "null" });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY116", Value = (totalContext != null && totalContext.RNTLPRPTY116.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY116) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY117", Value = (totalContext != null && totalContext.RNTLPRPTY117.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY117) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY118", Value = (totalContext != null && totalContext.RNTLPRPTY118.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY118) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY119", Value = (totalContext != null && totalContext.RNTLPRPTY119.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY119) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY120", Value = (totalContext != null && totalContext.RNTLPRPTY120.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY120) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY121", Value = (totalContext != null && totalContext.RNTLPRPTY121.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY121) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY122", Value = (totalContext != null && totalContext.RNTLPRPTY122.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY122) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY123", Value = (totalContext != null && totalContext.RNTLPRPTY123.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY123) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY125", Value = (totalContext != null && totalContext.RNTLPRPTY125.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY125) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY126", Value = (totalContext != null && totalContext.RNTLPRPTY126.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY126) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY127", Value = (totalContext != null && totalContext.RNTLPRPTY127.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY127) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY128", Value = (totalContext != null && totalContext.RNTLPRPTY128.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY128) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY129", Value = (totalContext != null && totalContext.RNTLPRPTY129.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY129) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY131", Value = (totalContext != null && totalContext.RNTLPRPTY131.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY131) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY132", Value = (totalContext != null && totalContext.RNTLPRPTY132.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY132) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY133", Value = (totalContext != null && totalContext.RNTLPRPTY133.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY133) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY134", Value = (totalContext != null && totalContext.RNTLPRPTY134.HasValue ? GetValueOrEmpty(totalContext.RNTLPRPTY134) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY138", Value = (portionContext != null && portionContext.RNTLPRPTY138.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY138) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY139", Value = (portionContext != null && portionContext.RNTLPRPTY139.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY139) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY140", Value = (portionContext != null && portionContext.RNTLPRPTY140.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY140) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY141", Value = (portionContext != null && portionContext.RNTLPRPTY141.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY141) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY142", Value = (portionContext != null && portionContext.RNTLPRPTY142.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY142) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY143", Value = (portionContext != null && portionContext.RNTLPRPTY143.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY143) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY144", Value = (portionContext != null && portionContext.RNTLPRPTY144.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY144) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY145", Value = (portionContext != null && portionContext.RNTLPRPTY145.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY145) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY147", Value = (portionContext != null && portionContext.RNTLPRPTY147.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY147) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY148", Value = (portionContext != null && portionContext.RNTLPRPTY148.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY148) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY149", Value = (portionContext != null && portionContext.RNTLPRPTY149.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY149) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY150", Value = (portionContext != null && portionContext.RNTLPRPTY150.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY150) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY151", Value = (portionContext != null && portionContext.RNTLPRPTY151.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY151) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY153", Value = (portionContext != null && portionContext.RNTLPRPTY153.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY153) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY154", Value = (portionContext != null && portionContext.RNTLPRPTY154.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY154) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY155", Value = (portionContext != null && portionContext.RNTLPRPTY155.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY155) : "null") });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "RNTLPRPTY156", Value = (portionContext != null && portionContext.RNTLPRPTY156.HasValue ? GetValueOrEmpty(portionContext.RNTLPRPTY156) : "null") });

                #endregion
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730221

        /* VR.ATO.IITR.730221
        Other rental deductions must equal the sum of all other rental deductions, where the Ownership percentage is 100%, plus the sum of all your share of other rental deductions, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule

        Legacy Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR337 <> NULL AND ^IITR337 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY116, ^RNTLPRPTY138)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY117, ^RNTLPRPTY139)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY118, ^RNTLPRPTY140)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY119, ^RNTLPRPTY141)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY120, ^RNTLPRPTY142)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY121, ^RNTLPRPTY143)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY122, ^RNTLPRPTY144)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY123, ^RNTLPRPTY145)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY125, ^RNTLPRPTY147)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY126, ^RNTLPRPTY148)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY127, ^RNTLPRPTY149)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY128, ^RNTLPRPTY150)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY129, ^RNTLPRPTY151)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY131, ^RNTLPRPTY153)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY132, ^RNTLPRPTY154)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY133, ^RNTLPRPTY155)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY134, ^RNTLPRPTY156)),0)

        Technical Business Rule Format:
            (CountDocument('RNTLPRPTY') = 1) AND ^IITR337 <> NULL AND ^IITR337 <> RoundDown(Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY116, ^RNTLPRPTY138)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY117, ^RNTLPRPTY139)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY118, ^RNTLPRPTY140)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY119, ^RNTLPRPTY141)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY120, ^RNTLPRPTY142)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY121, ^RNTLPRPTY143)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY122, ^RNTLPRPTY144)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY123, ^RNTLPRPTY145)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY125, ^RNTLPRPTY147)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY126, ^RNTLPRPTY148)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY127, ^RNTLPRPTY149)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY128, ^RNTLPRPTY150)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY129, ^RNTLPRPTY151)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY131, ^RNTLPRPTY153)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY132, ^RNTLPRPTY154)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY133, ^RNTLPRPTY155)) + Sum(ConditionalValue(^RNTLPRPTY104 = 100, ^RNTLPRPTY134, ^RNTLPRPTY156)),0)

        Data Elements:
    
        ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
        ^RNTLPRPTY104 = RNTLPRPTY104
    
        ^RNTLPRPTY116 = RNTLPRPTY116
    
        ^RNTLPRPTY117 = RNTLPRPTY117
    
        ^RNTLPRPTY118 = RNTLPRPTY118
    
        ^RNTLPRPTY119 = RNTLPRPTY119
    
        ^RNTLPRPTY120 = RNTLPRPTY120
    
        ^RNTLPRPTY121 = RNTLPRPTY121
    
        ^RNTLPRPTY122 = RNTLPRPTY122
    
        ^RNTLPRPTY123 = RNTLPRPTY123
    
        ^RNTLPRPTY125 = RNTLPRPTY125
    
        ^RNTLPRPTY126 = RNTLPRPTY126
    
        ^RNTLPRPTY127 = RNTLPRPTY127
    
        ^RNTLPRPTY128 = RNTLPRPTY128
    
        ^RNTLPRPTY129 = RNTLPRPTY129
    
        ^RNTLPRPTY131 = RNTLPRPTY131
    
        ^RNTLPRPTY132 = RNTLPRPTY132
    
        ^RNTLPRPTY133 = RNTLPRPTY133
    
        ^RNTLPRPTY134 = RNTLPRPTY134
    
        ^RNTLPRPTY138 = RNTLPRPTY138
    
        ^RNTLPRPTY139 = RNTLPRPTY139
    
        ^RNTLPRPTY140 = RNTLPRPTY140
    
        ^RNTLPRPTY141 = RNTLPRPTY141
    
        ^RNTLPRPTY142 = RNTLPRPTY142
    
        ^RNTLPRPTY143 = RNTLPRPTY143
    
        ^RNTLPRPTY144 = RNTLPRPTY144
    
        ^RNTLPRPTY145 = RNTLPRPTY145
    
        ^RNTLPRPTY147 = RNTLPRPTY147
    
        ^RNTLPRPTY148 = RNTLPRPTY148
    
        ^RNTLPRPTY149 = RNTLPRPTY149
    
        ^RNTLPRPTY150 = RNTLPRPTY150
    
        ^RNTLPRPTY151 = RNTLPRPTY151
    
        ^RNTLPRPTY153 = RNTLPRPTY153
    
        ^RNTLPRPTY154 = RNTLPRPTY154
    
        ^RNTLPRPTY155 = RNTLPRPTY155
    
        ^RNTLPRPTY156 = RNTLPRPTY156
        */
        public static ProcessMessageDocument VR_ATO_IITR_730221(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            string[] baseSequencesFullShare = BaseSequencesFullShare(report);
            string[] baseSequencesPartShare = BaseSequencesPartShare(report);

            // Get every RP.{PrptySeqNum}.Total context where the Ownership percentage in the base context = 100%
            List<RNTLPRPTY2019.RPPrptySeqNumTotalContext> totalContexts = (report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumTotalContextCollection.FindAll(context => baseSequencesFullShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)) : null);
            // Get every RP.{PrptySeqNum}.Portion context where the Ownership percentage in the base context < 100%
            List<RNTLPRPTY2019.RPPrptySeqNumPortionContext> portionContexts = (report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection != null ? report.rntlprptyChildDocuments[0].RPPrptySeqNumPortionContextCollection.FindAll(context => baseSequencesPartShare.Contains(context.RentalPropertyFinancialDetailsSequenceDimensionValue)) : null);

            #region Sum Totals

            decimal rntlprpty116Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY116.GetValueOrDefault()) : 0);
            decimal rntlprpty117Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY117.GetValueOrDefault()) : 0);
            decimal rntlprpty118Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY118.GetValueOrDefault()) : 0);
            decimal rntlprpty119Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY119.GetValueOrDefault()) : 0);
            decimal rntlprpty120Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY120.GetValueOrDefault()) : 0);
            decimal rntlprpty121Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY121.GetValueOrDefault()) : 0);
            decimal rntlprpty122Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY122.GetValueOrDefault()) : 0);
            decimal rntlprpty123Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY123.GetValueOrDefault()) : 0);
            decimal rntlprpty125Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY125.GetValueOrDefault()) : 0);
            decimal rntlprpty126Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY126.GetValueOrDefault()) : 0);
            decimal rntlprpty127Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY127.GetValueOrDefault()) : 0);
            decimal rntlprpty128Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY128.GetValueOrDefault()) : 0);
            decimal rntlprpty129Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY129.GetValueOrDefault()) : 0);
            decimal rntlprpty131Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY131.GetValueOrDefault()) : 0);
            decimal rntlprpty132Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY132.GetValueOrDefault()) : 0);
            decimal rntlprpty133Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY133.GetValueOrDefault()) : 0);
            decimal rntlprpty134Total = (totalContexts != null ? totalContexts.Sum(context => context.RNTLPRPTY134.GetValueOrDefault()) : 0);
            decimal rntlprpty138Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY138.GetValueOrDefault()) : 0);
            decimal rntlprpty139Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY139.GetValueOrDefault()) : 0);
            decimal rntlprpty140Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY140.GetValueOrDefault()) : 0);
            decimal rntlprpty141Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY141.GetValueOrDefault()) : 0);
            decimal rntlprpty142Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY142.GetValueOrDefault()) : 0);
            decimal rntlprpty143Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY143.GetValueOrDefault()) : 0);
            decimal rntlprpty144Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY144.GetValueOrDefault()) : 0);
            decimal rntlprpty145Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY145.GetValueOrDefault()) : 0);
            decimal rntlprpty147Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY147.GetValueOrDefault()) : 0);
            decimal rntlprpty148Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY148.GetValueOrDefault()) : 0);
            decimal rntlprpty149Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY149.GetValueOrDefault()) : 0);
            decimal rntlprpty150Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY150.GetValueOrDefault()) : 0);
            decimal rntlprpty151Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY151.GetValueOrDefault()) : 0);
            decimal rntlprpty153Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY153.GetValueOrDefault()) : 0);
            decimal rntlprpty154Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY154.GetValueOrDefault()) : 0);
            decimal rntlprpty155Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY155.GetValueOrDefault()) : 0);
            decimal rntlprpty156Total = (portionContexts != null ? portionContexts.Sum(context => context.RNTLPRPTY156.GetValueOrDefault()) : 0);
            decimal vr730221Total = Math.Truncate(rntlprpty116Total + rntlprpty117Total + rntlprpty118Total + rntlprpty119Total + rntlprpty120Total + rntlprpty121Total + rntlprpty122Total + rntlprpty123Total + rntlprpty125Total + rntlprpty126Total + rntlprpty127Total + rntlprpty128Total + rntlprpty129Total + rntlprpty131Total + rntlprpty132Total + rntlprpty133Total + rntlprpty134Total + rntlprpty138Total + rntlprpty139Total + rntlprpty140Total + rntlprpty141Total + rntlprpty142Total + rntlprpty143Total + rntlprpty144Total + rntlprpty145Total + rntlprpty147Total + rntlprpty148Total + rntlprpty149Total + rntlprpty150Total + rntlprpty151Total + rntlprpty153Total + rntlprpty154Total + rntlprpty155Total + rntlprpty156Total);

            #endregion

            bool assertion = report.iitrParentDocument.IITR337 != null && report.iitrParentDocument.IITR337.GetValueOrDefault() != vr730221Total;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730221",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other rental deductions amount is incorrect",
                    LongDescription = @"Other rental deductions must equal the sum of all other rental deductions, where the Ownership percentage is 100%, plus the sum of all your share of other rental deductions, where the Ownership percentage is less than 100%, in the attached Multi-Property Rental schedule",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730221" } },
                };

                #region ProcessMessage Parameters

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR337", Value = report.iitrParentDocument.IITR337.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY116) WHERE RNTLPRPTY104 = 100", Value = rntlprpty116Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY117) WHERE RNTLPRPTY104 = 100", Value = rntlprpty117Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY118) WHERE RNTLPRPTY104 = 100", Value = rntlprpty118Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY119) WHERE RNTLPRPTY104 = 100", Value = rntlprpty119Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY120) WHERE RNTLPRPTY104 = 100", Value = rntlprpty120Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY121) WHERE RNTLPRPTY104 = 100", Value = rntlprpty121Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY122) WHERE RNTLPRPTY104 = 100", Value = rntlprpty122Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY123) WHERE RNTLPRPTY104 = 100", Value = rntlprpty123Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY125) WHERE RNTLPRPTY104 = 100", Value = rntlprpty125Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY126) WHERE RNTLPRPTY104 = 100", Value = rntlprpty126Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY127) WHERE RNTLPRPTY104 = 100", Value = rntlprpty127Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY128) WHERE RNTLPRPTY104 = 100", Value = rntlprpty128Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY129) WHERE RNTLPRPTY104 = 100", Value = rntlprpty129Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY131) WHERE RNTLPRPTY104 = 100", Value = rntlprpty131Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY132) WHERE RNTLPRPTY104 = 100", Value = rntlprpty132Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY133) WHERE RNTLPRPTY104 = 100", Value = rntlprpty133Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY134) WHERE RNTLPRPTY104 = 100", Value = rntlprpty134Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY138) WHERE RNTLPRPTY104 < 100", Value = rntlprpty138Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY139) WHERE RNTLPRPTY104 < 100", Value = rntlprpty139Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY140) WHERE RNTLPRPTY104 < 100", Value = rntlprpty140Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY141) WHERE RNTLPRPTY104 < 100", Value = rntlprpty141Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY142) WHERE RNTLPRPTY104 < 100", Value = rntlprpty142Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY143) WHERE RNTLPRPTY104 < 100", Value = rntlprpty143Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY144) WHERE RNTLPRPTY104 < 100", Value = rntlprpty144Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY145) WHERE RNTLPRPTY104 < 100", Value = rntlprpty145Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY147) WHERE RNTLPRPTY104 < 100", Value = rntlprpty147Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY148) WHERE RNTLPRPTY104 < 100", Value = rntlprpty148Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY149) WHERE RNTLPRPTY104 < 100", Value = rntlprpty149Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY150) WHERE RNTLPRPTY104 < 100", Value = rntlprpty150Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY151) WHERE RNTLPRPTY104 < 100", Value = rntlprpty151Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY153) WHERE RNTLPRPTY104 < 100", Value = rntlprpty153Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY154) WHERE RNTLPRPTY104 < 100", Value = rntlprpty154Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY155) WHERE RNTLPRPTY104 < 100", Value = rntlprpty155Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SUM(RNTLPRPTY156) WHERE RNTLPRPTY104 < 100", Value = rntlprpty156Total.ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Calculated Other Rental Deductions Total", Value = vr730221Total.ToString() });

                #endregion
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730310

        /* VR.ATO.IITR.730310
        The TFN provided for the Multi-Property Rental schedule must be the same as the TFN of the parent form

        Legacy Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND AnyOccurrence(^RNTLPRPTY:RP.{PrptySeqNum}:TFN, ^RNTLPRPTY:RP.{PrptySeqNum}:TFN <> ^RP:TFN)

        Technical Business Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND AnyOccurrence(^RNTLPRPTY:RP.{PrptySeqNum}:TFN, ^RNTLPRPTY:RP.{PrptySeqNum}:TFN <> ^RP:TFN)

        Data Elements:
    
        ^TFN = TFN
        */
        public static ProcessMessageDocument VR_ATO_IITR_730310(IITR2026Crossform report, RNTLPRPTY2019.RPPrptySeqNumContext rntlprptyContext)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (rntlprptyContext.IdentifierTFN != report.iitrParentDocument.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730310",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax file number invalid",
                    LongDescription = @"The TFN provided for the Multi-Property Rental schedule must be the same as the TFN of the parent form",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + rntlprptyContext.Id + "']/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730310" } }
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730311

        /* VR.ATO.IITR.730311
        Invalid context. Period start date for the Multi-Property Rental schedule must be the same as the Period start date of the parent form.

        Legacy Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND AnyOccurrence(^RNTLPRPTY:RP.{PrptySeqNum}:StartDate, ^RNTLPRPTY:RP.{PrptySeqNum}:StartDate <> ^RP:StartDate)

        Technical Business Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND AnyOccurrence(^RNTLPRPTY:RP.{PrptySeqNum}:StartDate, ^RNTLPRPTY:RP.{PrptySeqNum}:StartDate <> ^RP:StartDate)

        Data Elements:
    
        ^StartDate = StartDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_730311(IITR2026Crossform report, RNTLPRPTY2019.RPPrptySeqNumContext rntlprptyContext)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (rntlprptyContext.StartDate != report.iitrParentDocument.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730311",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date for the Multi-Property Rental schedule is invalid",
                    LongDescription = @"Invalid context. Period start date for the Multi-Property Rental schedule must be the same as the Period start date of the parent form.",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + rntlprptyContext.Id + "']/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730311" } }
                };
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.730312

        /* VR.ATO.IITR.730312
        Invalid context. Period end date for the Multi-Property Rental schedule must be the same as the Period end date of the parent form.

        Legacy Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND AnyOccurrence(^RNTLPRPTY:RP.{PrptySeqNum}:EndDate, ^RNTLPRPTY:RP.{PrptySeqNum}:EndDate <> ^RP:EndDate)

        Technical Business Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND AnyOccurrence(^RNTLPRPTY:RP.{PrptySeqNum}:EndDate, ^RNTLPRPTY:RP.{PrptySeqNum}:EndDate <> ^RP:EndDate)

        Data Elements:
    
        ^EndDate = EndDate
        */
        public static ProcessMessageDocument VR_ATO_IITR_730312(IITR2026Crossform report, RNTLPRPTY2019.RPPrptySeqNumContext rntlprptyContext)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (rntlprptyContext.EndDate != report.iitrParentDocument.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.730312",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date for the Multi-Property Rental schedule is invalid",
                    LongDescription = @"Invalid context. Period end date for the Multi-Property Rental schedule must be the same as the Period end date of the parent form.",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + rntlprptyContext.Id + "']/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.730312" } }
                };
            }

            return processMessage;
        }

        #endregion


        #region VR.ATO.IITR.310083

        /* VR.ATO.IITR.310083
        If there is a positive amount at Gross rent or at any rental deductions item, then a Multi-Property Rental schedule must be provided

        Legacy Rule Format:
            (^IITR334 > 0 OR ^IITR335 > 0 OR ^IITR336 > 0 OR ^IITR337 > 0) AND CountDocument('RNTLPRPTY') = 0

        Technical Business Rule Format:
            (^IITR334 > 0 OR ^IITR335 > 0 OR ^IITR336 > 0 OR ^IITR337 > 0) AND CountDocument('RNTLPRPTY') = 0

        Data Elements:
    
        ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount
    
        ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount
    
        ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount
    
        ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_310083(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = ((report.iitrParentDocument.IITR334 > 0 || report.iitrParentDocument.IITR335 > 0 || report.iitrParentDocument.IITR336 > 0 || report.iitrParentDocument.IITR337 > 0)  && Count(report.rntlprptyChildDocuments) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.311083",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A rental schedule must be provided",
                    LongDescription = @"If there is a positive amount at Gross rent or at any rental deductions item, then a Multi-Property Rental schedule must be provided",
                    Location = "/xbrli:xbrl/tns:Income.RealEstateProperty.Rental.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.310083" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR334", Value = report.iitrParentDocument.IITR334.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR335", Value = report.iitrParentDocument.IITR335.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR336", Value = report.iitrParentDocument.IITR336.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR337", Value = report.iitrParentDocument.IITR337.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion

        #region VR.ATO.IITR.310084

        /* VR.ATO.IITR.310084
        If a Multi-Property Rental schedule is present, Gross rent and rental deduction amounts must be present

        Legacy Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND (^IITR334 = NULL AND (^IITR335 = NULL OR ^IITR335 = 0) AND (^IITR336 = NULL OR ^IITR336 = 0) AND (^IITR337 = NULL OR ^IITR337 = 0))

        Technical Business Rule Format:
            CountDocument('RNTLPRPTY') = 1 AND (^IITR334 = NULL AND (^IITR335 = NULL OR ^IITR335 = 0) AND (^IITR336 = NULL OR ^IITR336 = 0) AND (^IITR337 = NULL OR ^IITR337 = 0))

        Data Elements:
    
        ^IITR334 = RP:Income.RealEstateProperty.Rental.Amount
    
        ^IITR335 = RP:IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount
    
        ^IITR336 = RP:IncomeTax.Deduction.CapitalWorksDeductions.Amount
    
        ^IITR337 = RP:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
        */
        public static ProcessMessageDocument VR_ATO_IITR_310084(IITR2026Crossform report)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (( Count(report.rntlprptyChildDocuments) == 1) && (report.iitrParentDocument.IITR334 == null && (report.iitrParentDocument.IITR335 == null || report.iitrParentDocument.IITR335 == 0) && (report.iitrParentDocument.IITR336 == null || report.iitrParentDocument.IITR336 == 0) && (report.iitrParentDocument.IITR337 == null || report.iitrParentDocument.IITR337 == 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.IITR.311084",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent and rental deduction amounts must be present",
                    LongDescription = @"If a Multi-Property Rental schedule is present, Gross rent and rental deduction amounts must be present",
                    Location = "/xbrli:xbrl/tns:Income.RealEstateProperty.Rental.Amount[@contextRef='" + report.iitrParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IITR.310084" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR334", Value = report.iitrParentDocument.IITR334.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR335", Value = report.iitrParentDocument.IITR335.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR336", Value = report.iitrParentDocument.IITR336.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "IITR337", Value = report.iitrParentDocument.IITR337.GetValueOrDefault().ToString() });
            }

            return processMessage;
        }

        #endregion
    }
}