using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationINCDTLS
{
    public partial class INCDTLS2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;
        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;
        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;
        /// <summary>
        /// The report parameter
        /// </summary>
        private INCDTLS2026 report;
        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref = "INCDTLS2026Validator"/> class.
        /// </summary>
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public INCDTLS2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref = "INCDTLS2026Validator"/> class.
        /// </summary>
        /// <param name = "report">Report variable </param>                
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public INCDTLS2026Validator(INCDTLS2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;
            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }

        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }

            if (posB == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }

            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }

            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            return (
                from x in nodes
                select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            nodes = nodes.Where(x => x != null).ToArray();
            return !((
                from x in nodes
                select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values)
            where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;
            if (year == null || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;
            if (year == 0 || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }

        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;
        }

        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }

        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }

            return date.Value.Month;
        }

        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }

        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }

        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;
                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }

            return response;
        }

        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;
            response = (field < (expression - range)) || (field > (expression + range));
            return response;
        }

        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;
                }
                else if (usi.Length == 9)
                {
                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }

            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;
            if (tan == null)
                return false;
            tan = tan.Trim();
            if (!decimal.TryParse(tan, out decimalTan))
                return true;
            if (tan.Length != 8)
                return true;
            decimal tanSum = 7 * int.Parse(tan.Substring(0, 1)) + 9 * int.Parse(tan.Substring(1, 1)) + 8 * int.Parse(tan.Substring(2, 1)) + 4 * int.Parse(tan.Substring(3, 1)) + 6 * int.Parse(tan.Substring(4, 1)) + 3 * int.Parse(tan.Substring(5, 1)) + 5 * int.Parse(tan.Substring(6, 1)) + 1 * int.Parse(tan.Substring(7, 1));
            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;
            if (abn == null)
                return false;
            abn = abn.Trim();
            if (!decimal.TryParse(abn, out decimalAbn))
                return true;
            if (abn.Length != 11)
                return true;
            decimal abnSum = 10 * (int.Parse(abn.Substring(0, 1)) - 1) + 1 * int.Parse(abn.Substring(1, 1)) + 3 * int.Parse(abn.Substring(2, 1)) + 5 * int.Parse(abn.Substring(3, 1)) + 7 * int.Parse(abn.Substring(4, 1)) + 9 * int.Parse(abn.Substring(5, 1)) + 11 * int.Parse(abn.Substring(6, 1)) + 13 * int.Parse(abn.Substring(7, 1)) + 15 * int.Parse(abn.Substring(8, 1)) + 17 * int.Parse(abn.Substring(9, 1)) + 19 * int.Parse(abn.Substring(10, 1));
            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;
            acn = acn.Trim();
            if (!decimal.TryParse(acn, out decimalAbn))
                return true;
            if (acn.Length != 9)
                return true;
            decimal abnSum = 8 * int.Parse(acn.Substring(0, 1)) + 7 * int.Parse(acn.Substring(1, 1)) + 6 * int.Parse(acn.Substring(2, 1)) + 5 * int.Parse(acn.Substring(3, 1)) + 4 * int.Parse(acn.Substring(4, 1)) + 3 * int.Parse(acn.Substring(5, 1)) + 2 * int.Parse(acn.Substring(6, 1)) + 1 * int.Parse(acn.Substring(7, 1));
            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;
            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;
            if (tfn == null)
                return false;
            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");
            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;
            if (tfn.Length < 8)
                return true;
            decimal tfn1To7Sum = 1 * int.Parse(tfn.Substring(0, 1)) + 4 * int.Parse(tfn.Substring(1, 1)) + 3 * int.Parse(tfn.Substring(2, 1)) + 7 * int.Parse(tfn.Substring(3, 1)) + 5 * int.Parse(tfn.Substring(4, 1)) + 8 * int.Parse(tfn.Substring(5, 1)) + 6 * int.Parse(tfn.Substring(6, 1));
            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));
            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;
                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;
                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }

        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;
            char[] characters = flags.ToCharArray();
            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name = "occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public INCDTLS2026 ConsumedReport
        {
            get
            {
                return report;
            }

            private set
            {
                report = value;
            }
        }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;
            this.ConsumedReport = reportIn;
            #region Manual Rules - Rules marked for manual coding
            #region VR.ATO.INCDTLS.000019
            /*  VR.ATO.INCDTLS.000019
                The number of Australian superannuation income stream payment summary instances must not exceed 20

                Legacy Rule Format:
                (Count(^INCDTLS108) > 20)
    
                Technical Business Rule Format:
                (Count(^INCDTLS108) > 20)
        
                Data Elements:
        
                ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
                */
            VRATOINCDTLS000019(response, report);
            #endregion // VR.ATO.INCDTLS.000019

            #region VR.ATO.INCDTLS.000023
            /*  VR.ATO.INCDTLS.000023
            The number of Australian government benefit types must not exceed 10 per benefit type

        Legacy Rule Format:
            (CountOccurrence(^INCDTLS126, ^INCDTLS126 = "Allowance") > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = "Pension") > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = "Special") > 10)

    Technical Business Rule Format:
            (CountOccurrence(^INCDTLS126, ^INCDTLS126 = 'Allowance') > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = 'Pension') > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = 'Special') > 10)

Data Elements:
    
    ^INCDTLS126 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.Type.Code
    */
            VRATOINCDTLS000023(response, report);
            #endregion // VR.ATO.INCDTLS.000023

            #region VR.ATO.INCDTLS.000100
            /*  VR.ATO.INCDTLS.000100
            The total number of Salary or wages payment summary instances plus Australian annuities payment summary instances must not exceed 57

        Legacy Rule Format:
            ((Count(^INCDTLS309) + Count(^INCDTLS145)) > 57)

    Technical Business Rule Format:
            ((Count(^INCDTLS309) + Count(^INCDTLS145)) > 57)

Data Elements:
    
    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
    
    ^INCDTLS145 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities
    */
            VRATOINCDTLS000100(response, report);
            #endregion // VR.ATO.INCDTLS.000100

            #region VR.ATO.INCDTLS.000123
            /*  VR.ATO.INCDTLS.000123
            The number of Australian Superannuation lump sum (SLS) payments instances must not exceed 25

        Legacy Rule Format:
            (Count(^INCDTLS327) > 25)

    Technical Business Rule Format:
            (Count(^INCDTLS327) > 25)

Data Elements:
    
    ^INCDTLS327 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment
    */
            VRATOINCDTLS000123(response, report);
            #endregion // VR.ATO.INCDTLS.000123

            #region VR.ATO.INCDTLS.000129
            /*  VR.ATO.INCDTLS.000129
            The number of Attributed personal services (Attributed PSI) payments instances must not exceed 10

        Legacy Rule Format:
            (Count(^INCDTLS340) > 10)

    Technical Business Rule Format:
            (Count(^INCDTLS340) > 10)

Data Elements:
    
    ^INCDTLS340 = INCDTLS:Rp:Income:OrganisationNameDetails:AttributedPersonalServicesIncome
    */
            VRATOINCDTLS000129(response, report);
            #endregion // VR.ATO.INCDTLS.000129

            #region VR.ATO.INCDTLS.000274
            /*  VR.ATO.INCDTLS.000274
            The Foreign employment income payment summary lump sum in arrears year within the Foreign employment income payment summary instance must not be duplicated

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS160, HasDuplicateValues(^INCDTLS523)))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS160, HasDuplicateValues(^INCDTLS523)))

Data Elements:
    
    ^INCDTLS523 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment:Report.TargetFinancial.Year
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    */
            VRATOINCDTLS000274(response, report);
            #endregion // VR.ATO.INCDTLS.000274

            #region VR.ATO.INCDTLS.000275
            /*  VR.ATO.INCDTLS.000275
            The Superannuation income stream lump sum in arrears year within the Australian superannuation income stream payment summary instance must not be duplicated

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS108, HasDuplicateValues(^INCDTLS526)))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS108, HasDuplicateValues(^INCDTLS526)))

Data Elements:
    
    ^INCDTLS526 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment:Report.TargetFinancial.Year
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    */
            VRATOINCDTLS000275(response, report);
            #endregion // VR.ATO.INCDTLS.000275

            #region VR.ATO.INCDTLS.000276
            /*  VR.ATO.INCDTLS.000276
            The Foreign employment income non-payment summary lump sum in arrears year within the Foreign employment income non-payment summary instance must not be duplicated

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS185, HasDuplicateValues(^INCDTLS529)))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS185, HasDuplicateValues(^INCDTLS529)))

Data Elements:
    
    ^INCDTLS529 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment:Report.TargetFinancial.Year
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    */
            VRATOINCDTLS000276(response, report);
            #endregion // VR.ATO.INCDTLS.000276

            #region VR.ATO.INCDTLS.000277
            /*  VR.ATO.INCDTLS.000277
            The Foreign pensions or annuities lump sum in arrears year within the Foreign pensions or annuities instance must not be duplicated

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS207, HasDuplicateValues(^INCDTLS532)))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS207, HasDuplicateValues(^INCDTLS532)))

Data Elements:
    
    ^INCDTLS532 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment:Report.TargetFinancial.Year
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    */
            VRATOINCDTLS000277(response, report);
            #endregion // VR.ATO.INCDTLS.000277

            #region VR.ATO.INCDTLS.000278
            /*  VR.ATO.INCDTLS.000278
            The Lump sum in arrears payment type code within the Salary or wages payment summary instance must not be duplicated

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS309, HasDuplicateValues(^INCDTLS151)))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS309, HasDuplicateValues(^INCDTLS151)))

Data Elements:
    
    ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code
    
    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
    */
            VRATOINCDTLS000278(response, report);
            #endregion // VR.ATO.INCDTLS.000278

            #region VR.ATO.INCDTLS.000308
            /*  VR.ATO.INCDTLS.000308
            The number of Employment termination payment (ETP) summary instances must not exceed 25 for Australian ETP payments

        Legacy Rule Format:
            (CountOccurrence(^INCDTLS135, (InSet(^INCDTLS136, '"au","cc","cx","nf","hm"') OR ^INCDTLS136 = NULL)) > 25)

    Technical Business Rule Format:
            (CountOccurrence(^INCDTLS135, (InSet(^INCDTLS136, '"au","cc","cx","nf","hm"') OR ^INCDTLS136 = NULL)) > 25)

Data Elements:
    
    ^INCDTLS136 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:AddressDetails.Country.Code
    
    ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
    */
            VRATOINCDTLS000308(response, report);
            #endregion // VR.ATO.INCDTLS.000308

            #region VR.ATO.INCDTLS.000309
            /*  VR.ATO.INCDTLS.000309
            The number of Employment termination payment (ETP) summary instances must not exceed 25 for Foreign ETP payments

        Legacy Rule Format:
            (CountOccurrence(^INCDTLS135, (NotInSet(^INCDTLS136, '"au","cc","cx","nf","hm"') AND ^INCDTLS136 <> NULL)) > 25)

    Technical Business Rule Format:
            (CountOccurrence(^INCDTLS135, (NotInSet(^INCDTLS136, '"au","cc","cx","nf","hm"') AND ^INCDTLS136 <> NULL)) > 25)

Data Elements:
    
    ^INCDTLS136 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:AddressDetails.Country.Code
    
    ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
    */
            VRATOINCDTLS000309(response, report);
            #endregion // VR.ATO.INCDTLS.000309

            #region VR.ATO.INCDTLS.000316
            /*  VR.ATO.INCDTLS.000316
            The total number of sets of lump sum in arrear details across all payment types must not exceed 15 - this includes the following payment summaries/types; Salary or wages payment summary, Foreign employment income payment summary, Australian superannuation income stream payment summary, Foreign employment income non-payment summary, and Foreign pensions or annuities

        Legacy Rule Format:
            ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) > 15)

    Technical Business Rule Format:
            ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) > 15)

Data Elements:
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    */
            VRATOINCDTLS000316(response, report);
            #endregion // VR.ATO.INCDTLS.000316

            #region VR.ATO.INCDTLS.000317
            /*  VR.ATO.INCDTLS.000317
            The Salary or wages lump sum in arrears year within the Salary or wages lump sum in arrears instance must not be duplicated

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS150, HasDuplicateValues(^INCDTLS153)))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS150, HasDuplicateValues(^INCDTLS153)))

Data Elements:
    
    ^INCDTLS153 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:LumpSumArrearsPayment:Report.TargetFinancial.Year
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    */
            VRATOINCDTLS000317(response, report);
            #endregion // VR.ATO.INCDTLS.000317

            #region VR.ATO.INCDTLS.000318
            /*  VR.ATO.INCDTLS.000318
            Where a positive amount is present for any of the following Superannuation income stream lump sum in arrears amounts for an Australian superannuation income stream payment summary, at least one Superannuation income stream lump sum in arrears payment instance must be provided; Superannuation income stream lump sum in arrears taxable component taxed element, Superannuation income stream lump sum in arrears taxable component untaxed element, or Superannuation income stream lump sum in arrears tax free component where the answer to "Are you under 60 years of age and a death benefits dependant, where the deceased died at 60 years or over?" is 'yes' (true)

        Legacy Rule Format:
            (^INCDTLS108 <> NULL AND (^INCDTLS117 > 0 OR ^INCDTLS118 > 0 OR (^INCDTLS109 = "true" AND ^INCDTLS119 > 0)) AND Count(^INCDTLS525) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

    Technical Business Rule Format:
            (^INCDTLS108 <> NULL AND (^INCDTLS117 > 0 OR ^INCDTLS118 > 0 OR (^INCDTLS109 = 'true' AND ^INCDTLS119 > 0)) AND Count(^INCDTLS525) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

Data Elements:
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS109 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationReversionaryIncomeStream.Indicator
    
    ^INCDTLS117 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
    
    ^INCDTLS118 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
    
    ^INCDTLS119 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.LumpSumArrearsTaxFree.Amount
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    */
            VRATOINCDTLS000318(response, report);
            #endregion // VR.ATO.INCDTLS.000318

            #region VR.ATO.INCDTLS.000321
            /*  VR.ATO.INCDTLS.000321
            Where a positive Salary or wages lump sum E amount is present for a Salary or wages payment summary, a Salary or wages lump sum in arrears instance with a Lump sum in arrears payment type code of "Salary or wage lump sum E" must be provided

        Legacy Rule Format:
            (^INCDTLS309 <> NULL AND ^INCDTLS326 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Salary or wage lump sum E") = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

    Technical Business Rule Format:
            (^INCDTLS309 <> NULL AND ^INCDTLS326 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = 'Salary or wage lump sum E') = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

Data Elements:
    
    ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
    
    ^INCDTLS326 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.BackPaymentLumpSumE.Amount
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    */
            VRATOINCDTLS000321(response, report);
            #endregion // VR.ATO.INCDTLS.000321

            #region VR.ATO.INCDTLS.000323
            /*  VR.ATO.INCDTLS.000323
            Where a positive Exempt foreign employment income lump sum in arrears amount is present for a Salary or wages payment summary, a Salary or wages lump sum in arrears instance with a Lump sum in arrears payment type code of "Exempt FEI PS lump sum in arrears" must be provided

        Legacy Rule Format:
            (^INCDTLS309 <> NULL AND ^INCDTLS338 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Exempt FEI PS lump sum in arrears") = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

    Technical Business Rule Format:
            (^INCDTLS309 <> NULL AND ^INCDTLS338 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = 'Exempt FEI PS lump sum in arrears') = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

Data Elements:
    
    ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
    
    ^INCDTLS338 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:ExemptForeignEmployment:Income.LumpSumArrearsPayment.Amount
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    */
            VRATOINCDTLS000323(response, report);
            #endregion // VR.ATO.INCDTLS.000323

            #region VR.ATO.INCDTLS.000333
            /*  VR.ATO.INCDTLS.000333
            The number of Foreign employment payment summary instances must not exceed 3

        Legacy Rule Format:
            (Count(^INCDTLS160) > 3)

    Technical Business Rule Format:
            (Count(^INCDTLS160) > 3)

Data Elements:
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    */
            VRATOINCDTLS000333(response, report);
            #endregion // VR.ATO.INCDTLS.000333

            #region VR.ATO.INCDTLS.000362
            /*  VR.ATO.INCDTLS.000362
            Where a positive Foreign employment income lump sum E amount is present for a Foreign employment income payment summary, at least one Foreign employment income payment summary lump sum in arrears payment instance must be provided

        Legacy Rule Format:
            (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

    Technical Business Rule Format:
            (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

Data Elements:
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS177 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.BackPaymentLumpSumE.Amount
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    */
            VRATOINCDTLS000362(response, report);
            #endregion // VR.ATO.INCDTLS.000362

            #region VR.ATO.INCDTLS.000363
            /*  VR.ATO.INCDTLS.000363
            Where a positive Exempt foreign income lump sum in arrears amount is present for a Foreign employment income non-payment summary, at least one Foreign employment income non-payment summary lump sum in arrears payment instance must be provided

        Legacy Rule Format:
            (^INCDTLS185 <> NULL AND ^INCDTLS195 > 0 AND Count(^INCDTLS528) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

    Technical Business Rule Format:
            (^INCDTLS185 <> NULL AND ^INCDTLS195 > 0 AND Count(^INCDTLS528) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

Data Elements:
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS195 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome:Income.LumpSumArrearsPayment.Amount
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    */
            VRATOINCDTLS000363(response, report);
            #endregion // VR.ATO.INCDTLS.000363

            #region VR.ATO.INCDTLS.000364
            /*  VR.ATO.INCDTLS.000364
            Where a positive Foreign pension or annuity lump sum in arrears amount is present for a Foreign pensions or annuities, at least one Foreign pensions or annuities lump sum in arrears payment instance must be provided

        Legacy Rule Format:
            (^INCDTLS207 <> NULL AND ^INCDTLS213 > 0 AND Count(^INCDTLS531) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

    Technical Business Rule Format:
            (^INCDTLS207 <> NULL AND ^INCDTLS213 > 0 AND Count(^INCDTLS531) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

Data Elements:
    
    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
    
    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
    
    ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
    
    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
    
    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
    
    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
    
    ^INCDTLS213 = INCDTLS:Rp:ForeignPensionsOrAnnuities:Income.LumpSumArrearsPayment.Amount
    
    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
    
    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
    
    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
    */
            VRATOINCDTLS000364(response, report);
            #endregion // VR.ATO.INCDTLS.000364

            #region VR.ATO.INCDTLS.000412
            /*  VR.ATO.INCDTLS.000412
            A positive ETP taxable component amount must be provided. When the Employment Termination Payment (ETP) only includes an ETP tax free component, it must not to be included in the lodgment.

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS140, ^INCDTLS140 = 0))

    Technical Business Rule Format:
            (AnyOccurrence(^INCDTLS140, ^INCDTLS140 = 0))

Data Elements:
    
    ^INCDTLS140 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:Income.EmploymentTerminationPaymentTaxable.Amount
    */
            VRATOINCDTLS000412(response, report);
            #endregion // VR.ATO.INCDTLS.000412

            #region VR.ATO.INCDTLS.W00311
            /*  VR.ATO.INCDTLS.W00311
            Where the total ETP payments for Death benefit ETP paid to a non-dependant of the deceased, from the same payer ABN, is greater than $245,000 (low CAP rate), and total tax withheld from ETP payments for Death benefit ETP paid to a non-dependant of the deceased from the same payer ABN is less than ETP taxable component * 32% (30% is ETP rate 2 + 2% Medicare), check ETP Tax withheld amount

        Legacy Rule Format:
            (^INCDTLS138 = 'N') AND (((^INCDTLS308 <> NULL) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) > 245000) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) * 0.32))) OR ((^INCDTLS307 <> NULL) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) > 245000) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) * 0.32))))
            
            [INCDTLS308:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS308,NULL) 
            [INCDTLS307:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS307,NULL)

    Technical Business Rule Format:
            (^INCDTLS138 = 'N') AND (((^INCDTLS308 <> NULL) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) > 245000) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) * 0.32))) OR ((^INCDTLS307 <> NULL) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) > 245000) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) * 0.32))))
            
            [INCDTLS308:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS308,NULL) 
            [INCDTLS307:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS307,NULL)

Data Elements:
    
    ^INCDTLS139 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmploymentTerminationPayment.Amount
    
    ^INCDTLS138 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:IncomeTax.PayAsYouGoWithholding.EmploymentTerminationPaymentType.Code
    
    ^INCDTLS140 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:Income.EmploymentTerminationPaymentTaxable.Amount
    
    ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
    
    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
    */
            VRATOINCDTLSW00311(response, report);
            #endregion // VR.ATO.INCDTLS.W00311

            #endregion Manual Rules - Rules marked for manual coding
            #region Repeating report.Rp_IncomeCollection  
            if (report.Rp_IncomeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_Income income in report.Rp_IncomeCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000002(income, itemIndex2);
                    VRATOINCDTLS000004(income, itemIndex2);
                    VRATOINCDTLS000005(income, itemIndex2);
                    VRATOINCDTLS000006(income, itemIndex2);
                    VRATOINCDTLS000007(income, itemIndex2);
                    VRATOINCDTLS000008(income, itemIndex2);
                    VRATOINCDTLS000010(income, itemIndex2);
                    VRATOINCDTLS000011(income, itemIndex2);
                    VRATOINCDTLS000012(income, itemIndex2);
                    VRATOINCDTLS000013(income, itemIndex2);
                    VRATOINCDTLS000014(income, itemIndex2);
                    VRATOINCDTLS000015(income, itemIndex2);
                    VRATOINCDTLS000016(income, itemIndex2);
                    VRATOINCDTLS000116(income, itemIndex2);
                    VRATOINCDTLS000117(income, itemIndex2);
                    VRATOINCDTLS000120(income, itemIndex2);
                    VRATOINCDTLS000126(income, itemIndex2);
                    VRATOINCDTLS000127(income, itemIndex2);
                    VRATOINCDTLS000130(income, itemIndex2);
                    VRATOINCDTLS000159(income, itemIndex2);
                    VRATOINCDTLS000301(income, itemIndex2);
                    VRATOINCDTLS000310(income, itemIndex2);
                    VRATOINCDTLS000313(income, itemIndex2);
                    VRATOINCDTLS000314(income, itemIndex2);
                    VRATOINCDTLS000315(income, itemIndex2);
                    VRATOINCDTLS000319(income, itemIndex2);
                    VRATOINCDTLS000322(income, itemIndex2);
                    VRATOINCDTLS000324(income, itemIndex2);
                    VRATOINCDTLS000332(income, itemIndex2);
                    VRATOINCDTLS000343(income, itemIndex2);
                    VRATOINCDTLS000344(income, itemIndex2);
                    VRATOINCDTLS000345(income, itemIndex2);
                    VRATOINCDTLS000346(income, itemIndex2);
                    VRATOINCDTLS000347(income, itemIndex2);
                    VRATOINCDTLS000348(income, itemIndex2);
                    VRATOINCDTLS000359(income, itemIndex2);
                    VRATOINCDTLS000397(income, itemIndex2);
                    VRATOINCDTLSW00012(income, itemIndex2);
                    VRATOINCDTLSW00121(income, itemIndex2);
                    VRATOINCDTLSW00128(income, itemIndex2);
                    VRATOINCDTLSW00304(income, itemIndex2);
                    VRATOINCDTLSW00312(income, itemIndex2);
                    VRATOINCDTLSW00342(income, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_IncomeCollection  
            if (report.Rp_IncomeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_Income income in report.Rp_IncomeCollection)
                {
                    itemIndex2++;
                    #region Repeating income.Rp_Income_GovernmentBenefitCollection  
                    if (income.Rp_Income_GovernmentBenefitCollection != null)
                    {
                        int itemIndex3 = 0;
                        foreach (INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit in income.Rp_Income_GovernmentBenefitCollection)
                        {
                            itemIndex3++;
                            VRATOINCDTLS000020(income, governmentBenefit, itemIndex3);
                            VRATOINCDTLS000024(income, governmentBenefit, itemIndex3);
                            VRATOINCDTLS000025(income, governmentBenefit, itemIndex3);
                            VRATOINCDTLS000425(income, governmentBenefit, itemIndex3);
                            VRATOINCDTLS000426(income, governmentBenefit, itemIndex3);
                            VRATOINCDTLS000427(income, governmentBenefit, itemIndex3);
                        }
                    }
                    #endregion // Foreach loop

                }
            }

            #endregion // Foreach loop

            VRATOINCDTLS000114();
            VRATOINCDTLS000115();
            VRATOINCDTLS000118();
            VRATOINCDTLS000119();
            VRATOINCDTLS000260();
            VRATOINCDTLS000353();
            VRATOINCDTLS000354();
            VRATOINCDTLS000393();
            VRATOINCDTLS000419();
            VRATOINCDTLS000423();
            #region Repeating report.Rp_DividendsCollection  
            if (report.Rp_DividendsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_Dividends dividends in report.Rp_DividendsCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000131(dividends, itemIndex2);
                    VRATOINCDTLS000132(dividends, itemIndex2);
                    VRATOINCDTLS000133(dividends, itemIndex2);
                    VRATOINCDTLS000134(dividends, itemIndex2);
                    VRATOINCDTLS000135(dividends, itemIndex2);
                    VRATOINCDTLS000136(dividends, itemIndex2);
                    VRATOINCDTLS000137(dividends, itemIndex2);
                    VRATOINCDTLS000138(dividends, itemIndex2);
                    VRATOINCDTLS000139(dividends, itemIndex2);
                    VRATOINCDTLS000140(dividends, itemIndex2);
                    VRATOINCDTLS000141(dividends, itemIndex2);
                    VRATOINCDTLS000142(dividends, itemIndex2);
                    VRATOINCDTLS000143(dividends, itemIndex2);
                    VRATOINCDTLS000144(dividends, itemIndex2);
                    VRATOINCDTLS000145(dividends, itemIndex2);
                    VRATOINCDTLS000146(dividends, itemIndex2);
                    VRATOINCDTLS000147(dividends, itemIndex2);
                    VRATOINCDTLS000148(dividends, itemIndex2);
                    VRATOINCDTLS000149(dividends, itemIndex2);
                    VRATOINCDTLS000150(dividends, itemIndex2);
                    VRATOINCDTLS000151(dividends, itemIndex2);
                    VRATOINCDTLS000152(dividends, itemIndex2);
                    VRATOINCDTLS000153(dividends, itemIndex2);
                    VRATOINCDTLS000156(dividends, itemIndex2);
                    VRATOINCDTLS000157(dividends, itemIndex2);
                    VRATOINCDTLS000158(dividends, itemIndex2);
                    VRATOINCDTLS000268(dividends, itemIndex2);
                    VRATOINCDTLS000269(dividends, itemIndex2);
                    VRATOINCDTLS000270(dividends, itemIndex2);
                    VRATOINCDTLS000271(dividends, itemIndex2);
                    VRATOINCDTLS000272(dividends, itemIndex2);
                    VRATOINCDTLS000273(dividends, itemIndex2);
                    VRATOINCDTLS000398(dividends, itemIndex2);
                    VRATOINCDTLS000399(dividends, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_ManagedFundsCollection  
            if (report.Rp_ManagedFundsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_ManagedFunds managedFunds in report.Rp_ManagedFundsCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000160(managedFunds, itemIndex2);
                    VRATOINCDTLS000161(managedFunds, itemIndex2);
                    VRATOINCDTLS000162(managedFunds, itemIndex2);
                    VRATOINCDTLS000163(managedFunds, itemIndex2);
                    VRATOINCDTLS000164(managedFunds, itemIndex2);
                    VRATOINCDTLS000165(managedFunds, itemIndex2);
                    VRATOINCDTLS000166(managedFunds, itemIndex2);
                    VRATOINCDTLS000167(managedFunds, itemIndex2);
                    VRATOINCDTLS000168(managedFunds, itemIndex2);
                    VRATOINCDTLS000169(managedFunds, itemIndex2);
                    VRATOINCDTLS000170(managedFunds, itemIndex2);
                    VRATOINCDTLS000171(managedFunds, itemIndex2);
                    VRATOINCDTLS000172(managedFunds, itemIndex2);
                    VRATOINCDTLS000173(managedFunds, itemIndex2);
                    VRATOINCDTLS000174(managedFunds, itemIndex2);
                    VRATOINCDTLS000175(managedFunds, itemIndex2);
                    VRATOINCDTLS000176(managedFunds, itemIndex2);
                    VRATOINCDTLS000178(managedFunds, itemIndex2);
                    VRATOINCDTLS000179(managedFunds, itemIndex2);
                    VRATOINCDTLS000180(managedFunds, itemIndex2);
                    VRATOINCDTLS000181(managedFunds, itemIndex2);
                    VRATOINCDTLS000182(managedFunds, itemIndex2);
                    VRATOINCDTLS000183(managedFunds, itemIndex2);
                    VRATOINCDTLS000184(managedFunds, itemIndex2);
                    VRATOINCDTLS000185(managedFunds, itemIndex2);
                    VRATOINCDTLS000186(managedFunds, itemIndex2);
                    VRATOINCDTLS000187(managedFunds, itemIndex2);
                    VRATOINCDTLS000188(managedFunds, itemIndex2);
                    VRATOINCDTLS000189(managedFunds, itemIndex2);
                    VRATOINCDTLS000190(managedFunds, itemIndex2);
                    VRATOINCDTLS000191(managedFunds, itemIndex2);
                    VRATOINCDTLS000192(managedFunds, itemIndex2);
                    VRATOINCDTLS000193(managedFunds, itemIndex2);
                    VRATOINCDTLS000194(managedFunds, itemIndex2);
                    VRATOINCDTLS000195(managedFunds, itemIndex2);
                    VRATOINCDTLS000196(managedFunds, itemIndex2);
                    VRATOINCDTLS000197(managedFunds, itemIndex2);
                    VRATOINCDTLS000198(managedFunds, itemIndex2);
                    VRATOINCDTLS000199(managedFunds, itemIndex2);
                    VRATOINCDTLS000200(managedFunds, itemIndex2);
                    VRATOINCDTLS000201(managedFunds, itemIndex2);
                    VRATOINCDTLS000202(managedFunds, itemIndex2);
                    VRATOINCDTLS000203(managedFunds, itemIndex2);
                    VRATOINCDTLS000204(managedFunds, itemIndex2);
                    VRATOINCDTLS000205(managedFunds, itemIndex2);
                    VRATOINCDTLS000206(managedFunds, itemIndex2);
                    VRATOINCDTLS000207(managedFunds, itemIndex2);
                    VRATOINCDTLS000208(managedFunds, itemIndex2);
                    VRATOINCDTLS000209(managedFunds, itemIndex2);
                    VRATOINCDTLS000210(managedFunds, itemIndex2);
                    VRATOINCDTLS000211(managedFunds, itemIndex2);
                    VRATOINCDTLS000212(managedFunds, itemIndex2);
                    VRATOINCDTLS000213(managedFunds, itemIndex2);
                    VRATOINCDTLS000214(managedFunds, itemIndex2);
                    VRATOINCDTLS000215(managedFunds, itemIndex2);
                    VRATOINCDTLS000216(managedFunds, itemIndex2);
                    VRATOINCDTLS000217(managedFunds, itemIndex2);
                    VRATOINCDTLS000218(managedFunds, itemIndex2);
                    VRATOINCDTLS000219(managedFunds, itemIndex2);
                    VRATOINCDTLS000220(managedFunds, itemIndex2);
                    VRATOINCDTLS000221(managedFunds, itemIndex2);
                    VRATOINCDTLS000222(managedFunds, itemIndex2);
                    VRATOINCDTLS000223(managedFunds, itemIndex2);
                    VRATOINCDTLS000224(managedFunds, itemIndex2);
                    VRATOINCDTLS000225(managedFunds, itemIndex2);
                    VRATOINCDTLS000226(managedFunds, itemIndex2);
                    VRATOINCDTLS000245(managedFunds, itemIndex2);
                    VRATOINCDTLS000246(managedFunds, itemIndex2);
                    VRATOINCDTLS000247(managedFunds, itemIndex2);
                    VRATOINCDTLS000248(managedFunds, itemIndex2);
                    VRATOINCDTLS000249(managedFunds, itemIndex2);
                    VRATOINCDTLS000250(managedFunds, itemIndex2);
                    VRATOINCDTLS000251(managedFunds, itemIndex2);
                    VRATOINCDTLS000252(managedFunds, itemIndex2);
                    VRATOINCDTLS000253(managedFunds, itemIndex2);
                    VRATOINCDTLS000254(managedFunds, itemIndex2);
                    VRATOINCDTLS000255(managedFunds, itemIndex2);
                    VRATOINCDTLS000256(managedFunds, itemIndex2);
                    VRATOINCDTLS000267(managedFunds, itemIndex2);
                    VRATOINCDTLS000394(managedFunds, itemIndex2);
                    VRATOINCDTLS000395(managedFunds, itemIndex2);
                    VRATOINCDTLS000396(managedFunds, itemIndex2);
                    VRATOINCDTLS000428(managedFunds, itemIndex2);
                    VRATOINCDTLS000429(managedFunds, itemIndex2);
                    VRATOINCDTLS000430(managedFunds, itemIndex2);
                    VRATOINCDTLS000431(managedFunds, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_IncomeOtherCollection  
            if (report.Rp_IncomeOtherCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_IncomeOther incomeOther in report.Rp_IncomeOtherCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000257(incomeOther, itemIndex2);
                    VRATOINCDTLS000258(incomeOther, itemIndex2);
                    VRATOINCDTLS000259(incomeOther, itemIndex2);
                    VRATOINCDTLS000261(incomeOther, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_PartnershipsCollection  
            if (report.Rp_PartnershipsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_Partnerships partnerships in report.Rp_PartnershipsCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000262(partnerships, itemIndex2);
                    VRATOINCDTLS000263(partnerships, itemIndex2);
                    VRATOINCDTLS000264(partnerships, itemIndex2);
                    VRATOINCDTLS000265(partnerships, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_EmployeeShareSchemeCollection  
            if (report.Rp_EmployeeShareSchemeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_EmployeeShareScheme employeeShareScheme in report.Rp_EmployeeShareSchemeCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000266(employeeShareScheme, itemIndex2);
                    VRATOINCDTLS000367(employeeShareScheme, itemIndex2);
                    VRATOINCDTLS000368(employeeShareScheme, itemIndex2);
                    VRATOINCDTLS000369(employeeShareScheme, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_InterestCollection  
            if (report.Rp_InterestCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_Interest interest in report.Rp_InterestCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000320(interest, itemIndex2);
                    VRATOINCDTLS000325(interest, itemIndex2);
                    VRATOINCDTLS000326(interest, itemIndex2);
                    VRATOINCDTLS000327(interest, itemIndex2);
                    VRATOINCDTLS000328(interest, itemIndex2);
                    VRATOINCDTLS000329(interest, itemIndex2);
                    VRATOINCDTLS000330(interest, itemIndex2);
                    VRATOINCDTLS000331(interest, itemIndex2);
                    VRATOINCDTLS000415(interest, itemIndex2);
                    VRATOINCDTLS000416(interest, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_ForeignIncomeCollection  
            if (report.Rp_ForeignIncomeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_ForeignIncome foreignIncome in report.Rp_ForeignIncomeCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000356(foreignIncome, itemIndex2);
                    VRATOINCDTLS000357(foreignIncome, itemIndex2);
                    VRATOINCDTLS000360(foreignIncome, itemIndex2);
                    VRATOINCDTLS000413(foreignIncome, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_ForeignPensionsOrAnnuitiesCollection  
            if (report.Rp_ForeignPensionsOrAnnuitiesCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_ForeignPensionsOrAnnuities foreignPensionsOrAnnuities in report.Rp_ForeignPensionsOrAnnuitiesCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000361(foreignPensionsOrAnnuities, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_OtherForeignIncomeCollection  
            if (report.Rp_OtherForeignIncomeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_OtherForeignIncome otherForeignIncome in report.Rp_OtherForeignIncomeCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000365(otherForeignIncome, itemIndex2);
                    VRATOINCDTLS000366(otherForeignIncome, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_BusinessIncomePaymentSummariesCollection  
            if (report.Rp_BusinessIncomePaymentSummariesCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries in report.Rp_BusinessIncomePaymentSummariesCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000371(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000372(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000373(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000374(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000375(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000376(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000377(businessIncomePaymentSummaries, itemIndex2);
                    VRATOINCDTLS000378(businessIncomePaymentSummaries, itemIndex2);
                }
            }

            #endregion // Foreach loop

            #region Repeating report.Rp_TrustIncomeCollection  
            if (report.Rp_TrustIncomeCollection != null)
            {
                int itemIndex2 = 0;
                foreach (INCDTLS2026.Rp_TrustIncome trustIncome in report.Rp_TrustIncomeCollection)
                {
                    itemIndex2++;
                    VRATOINCDTLS000379(trustIncome, itemIndex2);
                    VRATOINCDTLS000380(trustIncome, itemIndex2);
                    VRATOINCDTLS000381(trustIncome, itemIndex2);
                    VRATOINCDTLS000389(trustIncome, itemIndex2);
                    VRATOINCDTLS000390(trustIncome, itemIndex2);
                    VRATOINCDTLS000391(trustIncome, itemIndex2);
                    VRATOINCDTLS000392(trustIncome, itemIndex2);
                    VRATOINCDTLS000418(trustIncome, itemIndex2);
                    VRATOINCDTLS000420(trustIncome, itemIndex2);
                    VRATOINCDTLS000421(trustIncome, itemIndex2);
                    VRATOINCDTLS000422(trustIncome, itemIndex2);
                    VRATOINCDTLS000424(trustIncome, itemIndex2);
                }
            }

            #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

        #region VR.ATO.INCDTLS.000002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000002(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000002
                    The Superannuation income stream payment period must be within the current financial year or approved substitute period
    
                    Legacy Rule Format:
                    (^INCDTLS110 <> NULL AND (^INCDTLS110 < ^INCDTLS100 OR ^INCDTLS111 > ^INCDTLS101))
        
                    Technical Business Rule Format:
                    (^INCDTLS110 <> NULL AND (^INCDTLS110 < ^INCDTLS100 OR ^INCDTLS111 > ^INCDTLS101))
            
                    Data Elements:
            
                    ^INCDTLS110 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationPaymentPeriodStart.Date
            
                    ^INCDTLS100 = INCDTLS:Rp:Lodgment.PeriodStart.Date
            
                    ^INCDTLS101 = INCDTLS:Rp:Lodgment.PeriodEnd.Date
            
                    ^INCDTLS111 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationPaymentPeriodEnd.Date
                    */
            assertion = (income.INCDTLS110 != null && (income.INCDTLS110.GetValueOrDefault() < report.INCDTLS100.GetValueOrDefault() || income.INCDTLS111.GetValueOrDefault() > report.INCDTLS101.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation income stream payment period must be within income year",
                    LongDescription = @"The Superannuation income stream payment period must be within the current financial year or approved substitute period",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:PaymentPeriodStartD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000002"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS110", Value = GetValueOrEmptyDate(income.INCDTLS110) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS100", Value = GetValueOrEmptyDate(report.INCDTLS100) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS111", Value = GetValueOrEmptyDate(income.INCDTLS111) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS101", Value = GetValueOrEmptyDate(report.INCDTLS101) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000004(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000004
                    The Superannuation income stream payment period end date must not be prior to the Superannuation income stream payment period start date
    
                    Legacy Rule Format:
                    (^INCDTLS111 <> NULL AND ^INCDTLS111 < ^INCDTLS110)
        
                    Technical Business Rule Format:
                    ^INCDTLS111 <> NULL AND ^INCDTLS111 < ^INCDTLS110
            
                    Data Elements:
            
                    ^INCDTLS111 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationPaymentPeriodEnd.Date
            
                    ^INCDTLS110 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationPaymentPeriodStart.Date
                    */
            assertion = (income.INCDTLS111 != null && income.INCDTLS111.GetValueOrDefault() < income.INCDTLS110.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation income stream payment period dates are incorrect",
                    LongDescription = @"The Superannuation income stream payment period end date must not be prior to the Superannuation income stream payment period start date",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:PaymentPeriodEndD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000004"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS111", Value = GetValueOrEmptyDate(income.INCDTLS111) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS110", Value = GetValueOrEmptyDate(income.INCDTLS110) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000005
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000005(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000005
                    The Payer's Australian business number has failed the ABN algorithm check
    
                    Legacy Rule Format:
                    (FailsABNAlgorithm(^INCDTLS307))
        
                    Technical Business Rule Format:
                    (FailsABNAlgorithm(^INCDTLS307))
            
                    Data Elements:
            
                    ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = FailsABNAlgorithm(income.INCDTLS307);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000005",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer's Australian business number is invalid",
                    LongDescription = @"The Payer's Australian business number has failed the ABN algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000005"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS307", Value = GetValueOrEmpty(income.INCDTLS307) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000006(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000006
                    The Payer's withholding payer number has failed the WPN algorithm check
    
                    Legacy Rule Format:
                    (FailsWPNAlgorithm(^INCDTLS308))
        
                    Technical Business Rule Format:
                    (FailsWPNAlgorithm(^INCDTLS308))
            
                    Data Elements:
            
                    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
                    */
            assertion = FailsTFNAlgorithm(income.INCDTLS308);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000006",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer's withholding payer number is invalid",
                    LongDescription = @"The Payer's withholding payer number has failed the WPN algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:WithholdingPayerNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000006"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS308", Value = GetValueOrEmpty(income.INCDTLS308) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000007(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000007
                    Either Payer's Australian business number or Payer's withholding payer number may be populated, but not both
    
                    Legacy Rule Format:
                    (^INCDTLS307 <> NULL) AND (^INCDTLS308 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS307 <> NULL) AND (^INCDTLS308 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
            
                    ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = (income.INCDTLS307 != null && income.INCDTLS308 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either Payer's Australian business number or Payer's withholding payer number may be populated, but not both",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:WithholdingPayerNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000007"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS307", Value = GetValueOrEmpty(income.INCDTLS307) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS308", Value = GetValueOrEmpty(income.INCDTLS308) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000008(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000008
                    When a positive Salary or wages gross amount has been provided for a Salary or wages payment summary instance, either the Payer's Australian business number or Payer's withholding payer number must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS312 > 0) AND (^INCDTLS307 = NULL) AND (^INCDTLS308 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS312 > 0) AND (^INCDTLS307 = NULL) AND (^INCDTLS308 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS312 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.SalaryOrWages.Amount
            
                    ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
            
                    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
                    */
            assertion = (income.INCDTLS312.GetValueOrDefault() > 0 && income.INCDTLS307 == null && income.INCDTLS308 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer's Australian business number or Payer's withholding payer number must be provided",
                    LongDescription = @"When a positive Salary or wages gross amount has been provided for a Salary or wages payment summary instance, either the Payer's Australian business number or Payer's withholding payer number must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:A",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000008"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS312", Value = GetValueOrEmpty(income.INCDTLS312) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS307", Value = GetValueOrEmpty(income.INCDTLS307) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS308", Value = GetValueOrEmpty(income.INCDTLS308) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000010(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000010
                    When the answer to the question "Did you earn this income while you held a Working Holiday visa (subclass 417) or Work and Holiday visa (subclass 462)?" is 'yes' (true), the Salary or wages gross amount, or Allowances including commissions, director's fees, bonuses, tips or consultation fees etc, or Salary or wages lump sum B must be provided.
    
                    Legacy Rule Format:
                    (^INCDTLS310 = TRUE) AND (^INCDTLS312 = NULL AND ^INCDTLS313 = NULL AND ^INCDTLS324 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS310 = TRUE) AND (^INCDTLS312 = NULL AND ^INCDTLS313 = NULL AND ^INCDTLS324 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS310 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Remuneration.WorkingHolidayMakerIncome.Indicator
            
                    ^INCDTLS312 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.SalaryOrWages.Amount
            
                    ^INCDTLS313 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.AllowancesEarningsTipsDirectorsFees.Amount
            
                    ^INCDTLS324 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
                    */
            assertion = (income.INCDTLS310 == true && (income.INCDTLS312 == null && income.INCDTLS313 == null && income.INCDTLS324 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified fields must be provided",
                    LongDescription = @"When the answer to the question ""Did you earn this income while you held a Working Holiday visa (subclass 417) or Work and Holiday visa (subclass 462)?"" is 'yes' (true), the Salary or wages gross amount, or Allowances including commissions, director's fees, bonuses, tips or consultation fees etc, or Salary or wages lump sum B must be provided.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:RemunerationWorkingHolidayMakerIncomeI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000010"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS310", Value = GetValueOrEmpty(income.INCDTLS310) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS312", Value = GetValueOrEmpty(income.INCDTLS312) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS313", Value = GetValueOrEmpty(income.INCDTLS313) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS324", Value = GetValueOrEmpty(income.INCDTLS324) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000011
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000011(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000011
                    When a positive Salary or wages gross amount is present, the Salary or wages tax withheld amount must be provided. Where no tax was withheld, enter zero.
    
                    Legacy Rule Format:
                    (^INCDTLS312 > 0) AND (^INCDTLS311 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS312 > 0) AND (^INCDTLS311 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS312 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.SalaryOrWages.Amount
            
                    ^INCDTLS311 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSalaryOrWages.Amount
                    */
            assertion = (income.INCDTLS312.GetValueOrDefault() > 0 && income.INCDTLS311 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Salary or wages tax withheld amount must be provided",
                    LongDescription = @"When a positive Salary or wages gross amount is present, the Salary or wages tax withheld amount must be provided. Where no tax was withheld, enter zero.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:A",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000011"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS312", Value = GetValueOrEmpty(income.INCDTLS312) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS311", Value = GetValueOrEmpty(income.INCDTLS311) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000012(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000012
                    Where a positive Reportable fringe benefits amount is present, an answer to "Is the employer exempt from FBT under section 57A of the FBTAA 1986?" must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS314 > 0) AND (^INCDTLS315 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS314 > 0) AND (^INCDTLS315 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS315 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:FringeBenefitsTax.Exemption.Indicator
            
                    ^INCDTLS314 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.FringeBenefitsReportable.Amount
                    */
            assertion = (income.INCDTLS314.GetValueOrDefault() > 0 && income.INCDTLS315 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Is the employer exempt from FBT under section 57A of the FBTAA 1986?"" must be provided",
                    LongDescription = @"Where a positive Reportable fringe benefits amount is present, an answer to ""Is the employer exempt from FBT under section 57A of the FBTAA 1986?"" must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:FringeBenefitsTaxExemptionI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000012"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS314", Value = GetValueOrEmpty(income.INCDTLS314) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS315", Value = GetValueOrEmpty(income.INCDTLS315) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000013(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000013
                    When any positive amount is present for Superannuation income stream taxable component taxed element, Superannuation income stream taxable component untaxed element, Superannuation income stream lump sum in arrears taxable component taxed element or Superannuation income stream lump sum in arrears taxable component untaxed element, Superannuation income stream tax withheld must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS112 = NULL AND (^INCDTLS113 > 0 OR ^INCDTLS114 > 0 OR ^INCDTLS117 > 0 OR ^INCDTLS118 > 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS112 = NULL AND (^INCDTLS113 > 0 OR ^INCDTLS114 > 0 OR ^INCDTLS117 > 0 OR ^INCDTLS118 > 0))
            
                    Data Elements:
            
                    ^INCDTLS112 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount
            
                    ^INCDTLS113 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS114 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount
            
                    ^INCDTLS117 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS118 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
                    */
            assertion = (income.INCDTLS112 == null && (income.INCDTLS113.GetValueOrDefault() > 0 || income.INCDTLS114.GetValueOrDefault() > 0 || income.INCDTLS117.GetValueOrDefault() > 0 || income.INCDTLS118.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation income stream tax withheld must be provided",
                    LongDescription = @"When any positive amount is present for Superannuation income stream taxable component taxed element, Superannuation income stream taxable component untaxed element, Superannuation income stream lump sum in arrears taxable component taxed element or Superannuation income stream lump sum in arrears taxable component untaxed element, Superannuation income stream tax withheld must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000013"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS112", Value = GetValueOrEmpty(income.INCDTLS112) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS113", Value = GetValueOrEmpty(income.INCDTLS113) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS114", Value = GetValueOrEmpty(income.INCDTLS114) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS117", Value = GetValueOrEmpty(income.INCDTLS117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS118", Value = GetValueOrEmpty(income.INCDTLS118) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000014(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000014
                    When a positive Superannuation income stream tax withheld amount is present and the answer to the question "Are you under 60 years of age and a death benefits dependant, where the deceased died at 60 years or over?" is 'no' (false), a positive amount for Superannuation income stream taxable component taxed element, Superannuation income stream taxable component untaxed element, Superannuation income stream tax free component, Superannuation income stream lump sum in arrears taxable component taxed element, Superannuation income stream lump sum in arrears taxable component untaxed element or Superannuation income stream lump sum in arrears tax free component must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS112 > 0 AND ^INCDTLS109 = FALSE AND (^INCDTLS113 = NULL OR ^INCDTLS113 = 0) AND (^INCDTLS114 = NULL OR ^INCDTLS114 = 0) AND (^INCDTLS115 = NULL OR ^INCDTLS115 = 0) AND (^INCDTLS117 = NULL OR ^INCDTLS117 = 0) AND (^INCDTLS118 = NULL OR ^INCDTLS118 = 0) AND (^INCDTLS119 = NULL OR ^INCDTLS119 = 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS112 > 0 AND ^INCDTLS109 = FALSE AND (^INCDTLS113 = NULL OR ^INCDTLS113 = 0) AND (^INCDTLS114 = NULL OR ^INCDTLS114 = 0) AND (^INCDTLS115 = NULL OR ^INCDTLS115 = 0) AND (^INCDTLS117 = NULL OR ^INCDTLS117 = 0) AND (^INCDTLS118 = NULL OR ^INCDTLS118 = 0) AND (^INCDTLS119 = NULL OR ^INCDTLS119 = 0))
            
                    Data Elements:
            
                    ^INCDTLS112 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount
            
                    ^INCDTLS109 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationReversionaryIncomeStream.Indicator
            
                    ^INCDTLS113 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS114 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount
            
                    ^INCDTLS115 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:IncomeTax.Superannuation.TaxFreeComponent.Amount
            
                    ^INCDTLS117 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS118 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
            
                    ^INCDTLS119 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.LumpSumArrearsTaxFree.Amount
                    */
            assertion = (income.INCDTLS112.GetValueOrDefault() > 0 && income.INCDTLS109 == false && (income.INCDTLS113 == null || income.INCDTLS113 == 0) && (income.INCDTLS114 == null || income.INCDTLS114 == 0) && (income.INCDTLS115 == null || income.INCDTLS115 == 0) && (income.INCDTLS117 == null || income.INCDTLS117 == 0) && (income.INCDTLS118 == null || income.INCDTLS118 == 0) && (income.INCDTLS119 == null || income.INCDTLS119 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation income stream component amounts are incorrect",
                    LongDescription = @"When a positive Superannuation income stream tax withheld amount is present and the answer to the question ""Are you under 60 years of age and a death benefits dependant, where the deceased died at 60 years or over?"" is 'no' (false), a positive amount for Superannuation income stream taxable component taxed element, Superannuation income stream taxable component untaxed element, Superannuation income stream tax free component, Superannuation income stream lump sum in arrears taxable component taxed element, Superannuation income stream lump sum in arrears taxable component untaxed element or Superannuation income stream lump sum in arrears tax free component must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000014"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS112", Value = GetValueOrEmpty(income.INCDTLS112) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS109", Value = GetValueOrEmpty(income.INCDTLS109) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS113", Value = GetValueOrEmpty(income.INCDTLS113) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS114", Value = GetValueOrEmpty(income.INCDTLS114) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS115", Value = GetValueOrEmpty(income.INCDTLS115) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS117", Value = GetValueOrEmpty(income.INCDTLS117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS118", Value = GetValueOrEmpty(income.INCDTLS118) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS119", Value = GetValueOrEmpty(income.INCDTLS119) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000015
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000015(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000015
                    When a positive Superannuation income stream tax offset amount is present, a positive amount for Superannuation income stream taxable component taxed element or Superannuation income stream taxable component untaxed element must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS116 > 0 AND (^INCDTLS113 = NULL OR ^INCDTLS113 = 0) AND (^INCDTLS114 = NULL OR ^INCDTLS114 = 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS116 > 0 AND (^INCDTLS113 = NULL OR ^INCDTLS113 = 0) AND (^INCDTLS114 = NULL OR ^INCDTLS114 = 0))
            
                    Data Elements:
            
                    ^INCDTLS116 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:TaxOffsetClaim.SuperannuationIncomeStream.Amount
            
                    ^INCDTLS113 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS114 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount
                    */
            assertion = (income.INCDTLS116.GetValueOrDefault() > 0 && (income.INCDTLS113 == null || income.INCDTLS113 == 0) && (income.INCDTLS114 == null || income.INCDTLS114 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation income stream taxable component amounts are incorrect",
                    LongDescription = @"When a positive Superannuation income stream tax offset amount is present, a positive amount for Superannuation income stream taxable component taxed element or Superannuation income stream taxable component untaxed element must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:TaxOffsetClaimSuperannuationIncomeStreamA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000015"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS116", Value = GetValueOrEmpty(income.INCDTLS116) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS113", Value = GetValueOrEmpty(income.INCDTLS113) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS114", Value = GetValueOrEmpty(income.INCDTLS114) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000016(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000016
                    When an answer to the question "Is the employer exempt from FBT under section 57A of the FBTAA 1986?" is provided, the Reportable fringe benefits amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS315 <> NULL) AND (^INCDTLS314 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS315 <> NULL) AND (^INCDTLS314 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS314 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.FringeBenefitsReportable.Amount
            
                    ^INCDTLS315 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:FringeBenefitsTax.Exemption.Indicator
                    */
            assertion = (income.INCDTLS315 != null && income.INCDTLS314 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reportable fringe benefits amount must be provided",
                    LongDescription = @"When an answer to the question ""Is the employer exempt from FBT under section 57A of the FBTAA 1986?"" is provided, the Reportable fringe benefits amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:FringeBenefitsReportableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000016"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS315", Value = GetValueOrEmpty(income.INCDTLS315) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS314", Value = GetValueOrEmpty(income.INCDTLS314) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000020(INCDTLS2026.Rp_Income income, INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000020
                            When a positive Australian government benefit tax withheld amount is provided, it must be less than the Australian government benefit taxable amount
            
                            Legacy Rule Format:
                            (^INCDTLS129 > 0 AND (^INCDTLS128 = NULL OR ^INCDTLS129 >= ^INCDTLS128))
                
                            Technical Business Rule Format:
                            (^INCDTLS129 > 0 AND (^INCDTLS128 = NULL OR ^INCDTLS129 >= ^INCDTLS128))
                    
                            Data Elements:
                    
                            ^INCDTLS129 = INCDTLS:Rp:Income:GovernmentBenefit:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount
                    
                            ^INCDTLS128 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.Taxable.Amount
                            */
            assertion = (governmentBenefit.INCDTLS129.GetValueOrDefault() > 0 && (governmentBenefit.INCDTLS128 == null || governmentBenefit.INCDTLS129.GetValueOrDefault() >= governmentBenefit.INCDTLS128.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian government benefit tax withheld amount is incorrect",
                    LongDescription = @"When a positive Australian government benefit tax withheld amount is provided, it must be less than the Australian government benefit taxable amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:GovernmentBenefitCollection/tns:GovernmentBenefit" + OccurrenceIndex(governmentBenefit.OccurrenceIndex) + "/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000020"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS129", Value = GetValueOrEmpty(governmentBenefit.INCDTLS129) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS128", Value = GetValueOrEmpty(governmentBenefit.INCDTLS128) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000024(INCDTLS2026.Rp_Income income, INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000024
                            Where an Australian government benefit payment summary instance is present, at least one of the following amounts must be provided: Australian government benefit taxable amount; or Australian government benefit tax withheld
            
                            Legacy Rule Format:
                            (^INCDTLS125 <> NULL AND ^INCDTLS128 = NULL AND ^INCDTLS129 = NULL)
                
                            Technical Business Rule Format:
                            ^INCDTLS125 <> NULL AND ^INCDTLS128 = NULL AND ^INCDTLS129 = NULL
                    
                            Data Elements:
                    
                            ^INCDTLS128 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.Taxable.Amount
                    
                            ^INCDTLS125 = INCDTLS:Rp:Income:GovernmentBenefit
                    
                            ^INCDTLS129 = INCDTLS:Rp:Income:GovernmentBenefit:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount
                            */
            assertion = (income.Rp_Income_GovernmentBenefitCollectionExists != false && governmentBenefit.INCDTLS128 == null && governmentBenefit.INCDTLS129 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian government benefit amount details are incomplete",
                    LongDescription = @"Where an Australian government benefit payment summary instance is present, at least one of the following amounts must be provided: Australian government benefit taxable amount; or Australian government benefit tax withheld",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:GovernmentBenefitCollection/tns:GovernmentBenefit" + OccurrenceIndex(governmentBenefit.OccurrenceIndex) + "/tns:PaymentTaxableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000024"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS128", Value = GetValueOrEmpty(governmentBenefit.INCDTLS128) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS129", Value = GetValueOrEmpty(governmentBenefit.INCDTLS129) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000025(INCDTLS2026.Rp_Income income, INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000025
                            When the Australian government benefit type is "Special" (Australian government special payment), Australian government benefit tax withheld amount must not be provided
            
                            Legacy Rule Format:
                            (^INCDTLS126 = "Special" AND ^INCDTLS129 <> NULL)
                
                            Technical Business Rule Format:
                            ^INCDTLS126 = 'Special' AND ^INCDTLS129 <> NULL
                    
                            Data Elements:
                    
                            ^INCDTLS129 = INCDTLS:Rp:Income:GovernmentBenefit:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount
                    
                            ^INCDTLS126 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.Type.Code
                            */
            assertion = (governmentBenefit.INCDTLS126 == @"Special" && governmentBenefit.INCDTLS129 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian government benefit tax withheld amount must not be provided",
                    LongDescription = @"When the Australian government benefit type is ""Special"" (Australian government special payment), Australian government benefit tax withheld amount must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:GovernmentBenefitCollection/tns:GovernmentBenefit" + OccurrenceIndex(governmentBenefit.OccurrenceIndex) + "/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000025"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS126", Value = governmentBenefit.INCDTLS126 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS129", Value = GetValueOrEmpty(governmentBenefit.INCDTLS129) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000114
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000114()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000114
            The number of Salary or wages lump sum A tax withheld amounts must not exceed 50
    
            Legacy Rule Format:
            (Count(^INCDTLS320) > 50)

            Technical Business Rule Format:
            (Count(^INCDTLS320) > 50)
    
            Data Elements:
    
            ^INCDTLS320 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            */
            assertion = (Count(report.Rp_IncomeCollection == null ? new List<decimal?>() : report.Rp_IncomeCollection.Select(f => f.INCDTLS320)) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000114",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Salary or wages lump sum A tax withheld amounts must not exceed 50",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000114"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000115
            The number of Salary or wages lump sum A amounts must not exceed 50
    
            Legacy Rule Format:
            (Count(^INCDTLS321) > 50)

            Technical Business Rule Format:
            (Count(^INCDTLS321) > 50)
    
            Data Elements:
    
            ^INCDTLS321 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            */
            assertion = (Count(report.Rp_IncomeCollection == null ? new List<decimal?>() : report.Rp_IncomeCollection.Select(f => f.INCDTLS321)) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000115",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Salary or wages lump sum A amounts must not exceed 50",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumAA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000115"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000116
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000116(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000116
                    Salary or wages lump sum A amount is required where Salary or wages lump sum A payment type is present
    
                    Legacy Rule Format:
                    (^INCDTLS321 = NULL OR ^INCDTLS321 = 0) AND (^INCDTLS322 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS321 = NULL OR ^INCDTLS321 = 0) AND (^INCDTLS322 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS321 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS322 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Remuneration.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Code
                    */
            assertion = ((income.INCDTLS321 == null || income.INCDTLS321 == 0) && income.INCDTLS322 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000116",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Salary or wages lump sum A amount must be provided",
                    LongDescription = @"Salary or wages lump sum A amount is required where Salary or wages lump sum A payment type is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumAA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000116"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS321", Value = GetValueOrEmpty(income.INCDTLS321) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS322", Value = GetValueOrEmpty(income.INCDTLS322) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000117
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000117(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000117
                    Salary or wages lump sum A payment type is required where Salary or wages lump sum A amount is present
    
                    Legacy Rule Format:
                    (^INCDTLS322 = NULL) AND (^INCDTLS321 > 0)
        
                    Technical Business Rule Format:
                    (^INCDTLS322 = NULL) AND (^INCDTLS321 > 0)
            
                    Data Elements:
            
                    ^INCDTLS322 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Remuneration.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Code
            
                    ^INCDTLS321 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
                    */
            assertion = (income.INCDTLS322 == null && income.INCDTLS321.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000117",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Salary or wages lump sum A payment type must be provided",
                    LongDescription = @"Salary or wages lump sum A payment type is required where Salary or wages lump sum A amount is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000117"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS322", Value = GetValueOrEmpty(income.INCDTLS322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS321", Value = GetValueOrEmpty(income.INCDTLS321) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000118
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000118()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000118
            The number of Salary or wages lump sum B tax withheld amounts must not exceed 50
    
            Legacy Rule Format:
            (Count(^INCDTLS323) > 50)

            Technical Business Rule Format:
            (Count(^INCDTLS323) > 50)
    
            Data Elements:
    
            ^INCDTLS323 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
            */
            assertion = (Count(report.Rp_IncomeCollection == null ? new List<decimal?>() : report.Rp_IncomeCollection.Select(f => f.INCDTLS323)) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000118",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Salary or wages lump sum B tax withheld amounts must not exceed 50",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000118"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000119
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000119()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000119
            The number of Salary or wages lump sum B amounts must not exceed 50
    
            Legacy Rule Format:
            (Count(^INCDTLS324) > 50)

            Technical Business Rule Format:
            (Count(^INCDTLS324) > 50)
    
            Data Elements:
    
            ^INCDTLS324 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
            */
            assertion = (Count(report.Rp_IncomeCollection == null ? new List<decimal?>() : report.Rp_IncomeCollection.Select(f => f.INCDTLS324)) > 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000119",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Salary or wages lump sum B amounts must not exceed 50",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumBA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000119"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000120
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000120(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000120
                    Where a Salary or wages payment summary instance is provided, at least one Salary or wages payment summary amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS309 <> NULL) AND (^INCDTLS311 = NULL) AND (^INCDTLS312 = NULL) AND (^INCDTLS313 = NULL) AND (^INCDTLS314 = NULL) AND (^INCDTLS316 = NULL) AND (^INCDTLS319 = NULL) AND (^INCDTLS320 = NULL) AND (^INCDTLS321 = NULL) AND (^INCDTLS322 = NULL) AND (^INCDTLS323 = NULL) AND (^INCDTLS324 = NULL) AND (^INCDTLS325 = NULL) AND (^INCDTLS326 = NULL) AND (^INCDTLS334 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS309 <> NULL) AND (^INCDTLS311 = NULL) AND (^INCDTLS312 = NULL) AND (^INCDTLS313 = NULL) AND (^INCDTLS314 = NULL) AND (^INCDTLS316 = NULL) AND (^INCDTLS319 = NULL) AND (^INCDTLS320 = NULL) AND (^INCDTLS321 = NULL) AND (^INCDTLS322 = NULL) AND (^INCDTLS323 = NULL) AND (^INCDTLS324 = NULL) AND (^INCDTLS325 = NULL) AND (^INCDTLS326 = NULL) AND (^INCDTLS334 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
            
                    ^INCDTLS311 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSalaryOrWages.Amount
            
                    ^INCDTLS312 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.SalaryOrWages.Amount
            
                    ^INCDTLS313 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.AllowancesEarningsTipsDirectorsFees.Amount
            
                    ^INCDTLS314 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.FringeBenefitsReportable.Amount
            
                    ^INCDTLS316 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.CommunityDevelopmentEmploymentProject.Amount
            
                    ^INCDTLS319 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:SuperannuationContribution.EmployerReportable.Amount
            
                    ^INCDTLS320 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS321 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS322 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Remuneration.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Code
            
                    ^INCDTLS323 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
            
                    ^INCDTLS324 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
            
                    ^INCDTLS325 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.GenuineRedundancyPaymentLumpSumD.Amount
            
                    ^INCDTLS326 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.BackPaymentLumpSumE.Amount
            
                    ^INCDTLS334 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:ExemptForeignEmployment
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists != false && income.INCDTLS311 == null && income.INCDTLS312 == null && income.INCDTLS313 == null && income.INCDTLS314 == null && income.INCDTLS316 == null && income.INCDTLS319 == null && income.INCDTLS320 == null && income.INCDTLS321 == null && income.INCDTLS322 == null && income.INCDTLS323 == null && income.INCDTLS324 == null && income.INCDTLS325 == null && income.INCDTLS326 == null && income.Rp_Income_OrganisationNameDetails_SalaryOrWages_ExemptForeignEmploymentCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000120",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one Salary or wages payment summary amount must be provided",
                    LongDescription = @"Where a Salary or wages payment summary instance is provided, at least one Salary or wages payment summary amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000120"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS311", Value = GetValueOrEmpty(income.INCDTLS311) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS312", Value = GetValueOrEmpty(income.INCDTLS312) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS313", Value = GetValueOrEmpty(income.INCDTLS313) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS314", Value = GetValueOrEmpty(income.INCDTLS314) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS316", Value = GetValueOrEmpty(income.INCDTLS316) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS319", Value = GetValueOrEmpty(income.INCDTLS319) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS320", Value = GetValueOrEmpty(income.INCDTLS320) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS321", Value = GetValueOrEmpty(income.INCDTLS321) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS322", Value = GetValueOrEmpty(income.INCDTLS322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS323", Value = GetValueOrEmpty(income.INCDTLS323) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS324", Value = GetValueOrEmpty(income.INCDTLS324) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS325", Value = GetValueOrEmpty(income.INCDTLS325) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS326", Value = GetValueOrEmpty(income.INCDTLS326) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000126
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000126(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000126
                    Where an Australian Superannuation lump sum (SLS) payments instance is provided, either SLS taxed element or SLS untaxed element must be provided with an amount greater than 0
    
                    Legacy Rule Format:
                    (^INCDTLS330 = 0) AND (^INCDTLS331 = 0)
        
                    Technical Business Rule Format:
                    (^INCDTLS330 = 0) AND (^INCDTLS331 = 0)
            
                    Data Elements:
            
                    ^INCDTLS330 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment:Income.Superannuation.LumpSumPaymentTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS331 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment:Income.Superannuation.LumpSumPaymentTaxableComponentUntaxedElement.Amount
                    */
            assertion = (income.INCDTLS330 == 0 && income.INCDTLS331 == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000126",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An amount above zero must be provided for either SLS taxed element or SLS untaxed element",
                    LongDescription = @"Where an Australian Superannuation lump sum (SLS) payments instance is provided, either SLS taxed element or SLS untaxed element must be provided with an amount greater than 0",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SuperannuationLumpSumPayment/tns:LumpSumPaymentTaxableComponentTaxedElementA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000126"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS330", Value = GetValueOrEmpty(income.INCDTLS330) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS331", Value = GetValueOrEmpty(income.INCDTLS331) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000127
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000127(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000127
                    The Superannuation lump sum (SLS) payment SLS date of payment must be within the current financial year or approved substitute period.
    
                    Legacy Rule Format:
                    (^INCDTLS327 <> NULL AND ((^INCDTLS328 < ^INCDTLS100) OR (^INCDTLS328 > ^INCDTLS101)))
        
                    Technical Business Rule Format:
                    (^INCDTLS327 <> NULL AND ((^INCDTLS328 < ^INCDTLS100) OR (^INCDTLS328 > ^INCDTLS101)))
            
                    Data Elements:
            
                    ^INCDTLS328 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment:Income.Superannuation.LumpSumPayment.Date
            
                    ^INCDTLS100 = INCDTLS:Rp:Lodgment.PeriodStart.Date
            
                    ^INCDTLS101 = INCDTLS:Rp:Lodgment.PeriodEnd.Date
            
                    ^INCDTLS327 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_SuperannuationLumpSumPaymentCollectionExists != false && (income.INCDTLS328.GetValueOrDefault() < report.INCDTLS100.GetValueOrDefault() || income.INCDTLS328.GetValueOrDefault() > report.INCDTLS101.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000127",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"SLS date of payment must be within income year",
                    LongDescription = @"The Superannuation lump sum (SLS) payment SLS date of payment must be within the current financial year or approved substitute period.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SuperannuationLumpSumPayment/tns:D",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000127"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS328", Value = GetValueOrEmptyDate(income.INCDTLS328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS100", Value = GetValueOrEmptyDate(report.INCDTLS100) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS101", Value = GetValueOrEmptyDate(report.INCDTLS101) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000130
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000130(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000130
                    Where Attributed PSI tax withheld amount is provided, it must not be greater than the Attributed PSI gross amount
    
                    Legacy Rule Format:
                    (^INCDTLS341 > 0 AND  (^INCDTLS342 = NULL OR ^INCDTLS341 >= ^INCDTLS342))
        
                    Technical Business Rule Format:
                    (^INCDTLS341 > 0 AND  (^INCDTLS342 = NULL OR ^INCDTLS341 >= ^INCDTLS342))
            
                    Data Elements:
            
                    ^INCDTLS341 = INCDTLS:Rp:Income:OrganisationNameDetails:AttributedPersonalServicesIncome:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount
            
                    ^INCDTLS342 = INCDTLS:Rp:Income:OrganisationNameDetails:AttributedPersonalServicesIncome:Income.PersonalServicesIncome.Total.Amount
                    */
            assertion = (income.INCDTLS341.GetValueOrDefault() > 0 && (income.INCDTLS342 == null || income.INCDTLS341.GetValueOrDefault() >= income.INCDTLS342.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000130",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed PSI tax withheld amount is incorrect",
                    LongDescription = @"Where Attributed PSI tax withheld amount is provided, it must not be greater than the Attributed PSI gross amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AttributedPersonalServicesIncome/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldPersonalServicesIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000130"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS341", Value = GetValueOrEmpty(income.INCDTLS341) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS342", Value = GetValueOrEmpty(income.INCDTLS342) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000131(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000131
                    Where Country of residence when the dividend was paid or credited is not "Australia", "Cocos (Keeling) Islands", "Christmas Island", "Heard Island and McDonald Islands" or "Norfolk Island", Total franked amount must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS372 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS372 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS348 = INCDTLS:Rp:Dividends:AddressDetails.Country.Code
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS348 != null && !(IsMatch(dividends.INCDTLS348, @"^(au|cc|cx|nf|hm)$", RegexOptions.IgnoreCase)) && dividends.INCDTLS372 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franked amount must not be provided",
                    LongDescription = @"Where Country of residence when the dividend was paid or credited is not ""Australia"", ""Cocos (Keeling) Islands"", ""Christmas Island"", ""Heard Island and McDonald Islands"" or ""Norfolk Island"", Total franked amount must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000131"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS348", Value = GetValueOrEmpty(dividends.INCDTLS348) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000132(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000132
                    Where Country of residence when the dividend was paid or credited is not "Australia", "Cocos (Keeling) Islands", "Christmas Island", "Heard Island and McDonald Islands" or "Norfolk Island", Total franking credit amount must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS373 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS373 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS348 = INCDTLS:Rp:Dividends:AddressDetails.Country.Code
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS348 != null && !(IsMatch(dividends.INCDTLS348, @"^(au|cc|cx|nf|hm)$", RegexOptions.IgnoreCase)) && dividends.INCDTLS373 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franking credit amount must not be provided",
                    LongDescription = @"Where Country of residence when the dividend was paid or credited is not ""Australia"", ""Cocos (Keeling) Islands"", ""Christmas Island"", ""Heard Island and McDonald Islands"" or ""Norfolk Island"", Total franking credit amount must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000132"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS348", Value = GetValueOrEmpty(dividends.INCDTLS348) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000133(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000133
                    Where Total franking credit from dividends amount is provided, Total franked amount from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS372 = NULL AND (^INCDTLS373 <> NULL OR  ^INCDTLS373 = 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS372 = NULL AND (^INCDTLS373 <> NULL OR  ^INCDTLS373 = 0))
            
                    Data Elements:
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS372 == null && (dividends.INCDTLS373 != null || dividends.INCDTLS373 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franked amount from dividends must be provided",
                    LongDescription = @"Where Total franking credit from dividends amount is provided, Total franked amount from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeFrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000133"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000134(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000134
                    Where Total franked amount from dividends is provided, Total franking credit from dividends amount must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS373 = NULL AND (^INCDTLS372 <> NULL OR  ^INCDTLS372 = 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS373 = NULL AND (^INCDTLS372 <> NULL OR  ^INCDTLS372 = 0))
            
                    Data Elements:
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS373 == null && (dividends.INCDTLS372 != null || dividends.INCDTLS372 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000134",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franking credit from dividends amount must be provided",
                    LongDescription = @"Where Total franked amount from dividends is provided, Total franking credit from dividends amount must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000134"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000135(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000135
                    Total franking credit from dividends must not exceed 48% of the Total franked amount from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS373 > (^INCDTLS372 * 0.48))
        
                    Technical Business Rule Format:
                    (^INCDTLS373 > (^INCDTLS372 * 0.48))
            
                    Data Elements:
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS373.GetValueOrDefault() > dividends.INCDTLS372.GetValueOrDefault() * 0.48M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franking credit from dividends must not exceed 48% of the Total franked amount from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000135"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000136(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000136
                    Total TFN amounts withheld from dividends must not exceed 50% of the Total unfranked amount from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS374 > (^INCDTLS371 * 0.5))
        
                    Technical Business Rule Format:
                    (^INCDTLS374 > (^INCDTLS371 * 0.5))
            
                    Data Elements:
            
                    ^INCDTLS374 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS371 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsUnfranked.Amount
                    */
            assertion = (dividends.INCDTLS374.GetValueOrDefault() > dividends.INCDTLS371.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000136",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total TFN amounts withheld from dividends must not exceed 50% of the Total unfranked amount from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000136"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS374", Value = GetValueOrEmpty(dividends.INCDTLS374) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS371", Value = GetValueOrEmpty(dividends.INCDTLS371) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000137(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000137
                    Total TFN amounts withheld from dividends must not exceed the sum of Total unfranked amount from dividends, Total franked amount from dividends and Total franking credit from dividends amount
    
                    Legacy Rule Format:
                    (^INCDTLS374 > (^INCDTLS371 + ^INCDTLS372 + ^INCDTLS373))
        
                    Technical Business Rule Format:
                    (^INCDTLS374 > (^INCDTLS371 + ^INCDTLS372 + ^INCDTLS373))
            
                    Data Elements:
            
                    ^INCDTLS374 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS371 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS374.GetValueOrDefault() > dividends.INCDTLS371.GetValueOrDefault() + dividends.INCDTLS372.GetValueOrDefault() + dividends.INCDTLS373.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total TFN amounts withheld from dividends is incorrect",
                    LongDescription = @"Total TFN amounts withheld from dividends must not exceed the sum of Total unfranked amount from dividends, Total franked amount from dividends and Total franking credit from dividends amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000137"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS374", Value = GetValueOrEmpty(dividends.INCDTLS374) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS371", Value = GetValueOrEmpty(dividends.INCDTLS371) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000138
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000138(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000138
                    Your share of unfranked amount from dividends must not exceed Total unfranked amount from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS378 > ^INCDTLS371)
        
                    Technical Business Rule Format:
                    ^INCDTLS378 > ^INCDTLS371
            
                    Data Elements:
            
                    ^INCDTLS378 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS371 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsUnfranked.Amount
                    */
            assertion = (dividends.INCDTLS378.GetValueOrDefault() > dividends.INCDTLS371.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000138",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of unfranked amount from dividends must not exceed Total unfranked amount from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeUnfrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000138"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS378", Value = GetValueOrEmpty(dividends.INCDTLS378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS371", Value = GetValueOrEmpty(dividends.INCDTLS371) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000139
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000139(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000139
                    Your share of franked amount from dividends must not exceed Total franked amount from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS379 > ^INCDTLS372)
        
                    Technical Business Rule Format:
                    ^INCDTLS379 > ^INCDTLS372
            
                    Data Elements:
            
                    ^INCDTLS379 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsFranked.Amount
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS379.GetValueOrDefault() > dividends.INCDTLS372.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000139",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franked amount from dividends must not exceed Total franked amount from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeFrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000139"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS379", Value = GetValueOrEmpty(dividends.INCDTLS379) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000140
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000140(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000140
                    Your share of franking credit from dividends amount must not exceed Total franking credit from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS380 > ^INCDTLS373)
        
                    Technical Business Rule Format:
                    ^INCDTLS380 > ^INCDTLS373
            
                    Data Elements:
            
                    ^INCDTLS380 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS380.GetValueOrDefault() > dividends.INCDTLS373.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000140",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franking credit from dividends amount must not exceed Total franking credit from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000140"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS380", Value = GetValueOrEmpty(dividends.INCDTLS380) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000141(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000141
                    Your share of franking credit from dividends must not exceed 48% of the Your share of franked amount
    
                    Legacy Rule Format:
                    (^INCDTLS380 > (^INCDTLS379 * 0.48))
        
                    Technical Business Rule Format:
                    (^INCDTLS380 > (^INCDTLS379 * 0.48))
            
                    Data Elements:
            
                    ^INCDTLS380 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS379 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS380.GetValueOrDefault() > dividends.INCDTLS379.GetValueOrDefault() * 0.48M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000141",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franking credit from dividends must not exceed 48% of the Your share of franked amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000141"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS380", Value = GetValueOrEmpty(dividends.INCDTLS380) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS379", Value = GetValueOrEmpty(dividends.INCDTLS379) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000142(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000142
                    Your share of TFN amounts withheld from dividends must not exceed Total TFN amounts withheld from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS381 > ^INCDTLS374)
        
                    Technical Business Rule Format:
                    ^INCDTLS381 > ^INCDTLS374
            
                    Data Elements:
            
                    ^INCDTLS381 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS374 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
                    */
            assertion = (dividends.INCDTLS381.GetValueOrDefault() > dividends.INCDTLS374.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from dividends must not exceed Total TFN amounts withheld from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000142"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS381", Value = GetValueOrEmpty(dividends.INCDTLS381) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS374", Value = GetValueOrEmpty(dividends.INCDTLS374) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000143
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000143(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000143
                    Your share of exploration credits from dividends must not exceed Total exploration credits from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS382 > ^INCDTLS375)
        
                    Technical Business Rule Format:
                    ^INCDTLS382 > ^INCDTLS375
            
                    Data Elements:
            
                    ^INCDTLS382 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS375 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (dividends.INCDTLS382.GetValueOrDefault() > dividends.INCDTLS375.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000143",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of exploration credits from dividends must not exceed Total exploration credits from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxExplorationCreditsDistributedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000143"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS382", Value = GetValueOrEmpty(dividends.INCDTLS382) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS375", Value = GetValueOrEmpty(dividends.INCDTLS375) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000144
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000144(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000144
                    Your share of listed investment company capital gain deduction from dividends must not exceed Total listed investment company capital gain deduction from dividends
    
                    Legacy Rule Format:
                    (^INCDTLS383 > ^INCDTLS376)
        
                    Technical Business Rule Format:
                    ^INCDTLS383 > ^INCDTLS376
            
                    Data Elements:
            
                    ^INCDTLS383 = INCDTLS:Rp:Dividends:DividendYourShare:Capital.Gains.Total.Amount
            
                    ^INCDTLS376 = INCDTLS:Rp:Dividends:DividendTotal:Capital.Gains.Total.Amount
                    */
            assertion = (dividends.INCDTLS383.GetValueOrDefault() > dividends.INCDTLS376.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000144",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of listed investment company capital gain deduction from dividends is incorrect",
                    LongDescription = @"Your share of listed investment company capital gain deduction from dividends must not exceed Total listed investment company capital gain deduction from dividends",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:CapitalGainsTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000144"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS383", Value = GetValueOrEmpty(dividends.INCDTLS383) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS376", Value = GetValueOrEmpty(dividends.INCDTLS376) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000145(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000145
                    Where Your share of unfranked amount from dividends is provided, Total unfranked amount from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS371 = NULL AND ^INCDTLS378 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS371 = NULL AND ^INCDTLS378 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS371 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS378 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsUnfranked.Amount
                    */
            assertion = (dividends.INCDTLS371 == null && dividends.INCDTLS378 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total unfranked amount from dividends must be provided",
                    LongDescription = @"Where Your share of unfranked amount from dividends is provided, Total unfranked amount from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeUnfrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000145"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS371", Value = GetValueOrEmpty(dividends.INCDTLS371) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS378", Value = GetValueOrEmpty(dividends.INCDTLS378) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000146
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000146(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000146
                    Where Your share of franked amount from dividends is provided, Total franked amount from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS372 = NULL AND ^INCDTLS379 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS372 = NULL AND ^INCDTLS379 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
            
                    ^INCDTLS379 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS372 == null && dividends.INCDTLS379 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000146",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franked amount from dividends must be provided",
                    LongDescription = @"Where Your share of franked amount from dividends is provided, Total franked amount from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeFrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000146"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS379", Value = GetValueOrEmpty(dividends.INCDTLS379) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000147
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000147(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000147
                    Where Your share of franking credit from dividends is provided, Total franking credit from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS373 = NULL AND ^INCDTLS380 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS373 = NULL AND ^INCDTLS380 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS380 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS373 == null && dividends.INCDTLS380 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total franking credit from dividends must be provided",
                    LongDescription = @"Where Your share of franking credit from dividends is provided, Total franking credit from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000147"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS380", Value = GetValueOrEmpty(dividends.INCDTLS380) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000148
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000148(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000148
                    Where Your share of TFN amounts withheld from dividends is provided, Total TFN amounts withheld from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS374 = NULL AND ^INCDTLS381 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS374 = NULL AND ^INCDTLS381 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS374 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS381 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
                    */
            assertion = (dividends.INCDTLS374 == null && dividends.INCDTLS381 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000148",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total TFN amounts withheld from dividends must be provided",
                    LongDescription = @"Where Your share of TFN amounts withheld from dividends is provided, Total TFN amounts withheld from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000148"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS374", Value = GetValueOrEmpty(dividends.INCDTLS374) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS381", Value = GetValueOrEmpty(dividends.INCDTLS381) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000149(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000149
                    Where Your share exploration credits from dividends is provided, Total exploration credits from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS375 = NULL AND ^INCDTLS382 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS375 = NULL AND ^INCDTLS382 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS375 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS382 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (dividends.INCDTLS375 == null && dividends.INCDTLS382 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000149",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total exploration credits from dividends must be provided",
                    LongDescription = @"Where Your share exploration credits from dividends is provided, Total exploration credits from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:IncomeTaxExplorationCreditsDistributedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000149"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS375", Value = GetValueOrEmpty(dividends.INCDTLS375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS382", Value = GetValueOrEmpty(dividends.INCDTLS382) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000150
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000150(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000150
                    Where Your share of listed investment company capital gain deduction from dividends is provided, Total listed investment company capital gain deduction from dividends must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS376 = NULL AND ^INCDTLS383 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS376 = NULL AND ^INCDTLS383 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS376 = INCDTLS:Rp:Dividends:DividendTotal:Capital.Gains.Total.Amount
            
                    ^INCDTLS383 = INCDTLS:Rp:Dividends:DividendYourShare:Capital.Gains.Total.Amount
                    */
            assertion = (dividends.INCDTLS376 == null && dividends.INCDTLS383 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000150",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total listed investment company capital gain deduction from dividends must be provided",
                    LongDescription = @"Where Your share of listed investment company capital gain deduction from dividends is provided, Total listed investment company capital gain deduction from dividends must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal/tns:CapitalGainsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000150"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS376", Value = GetValueOrEmpty(dividends.INCDTLS376) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS383", Value = GetValueOrEmpty(dividends.INCDTLS383) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000151(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000151
                    Where a Dividend instance is present, at least 1 Dividend Total amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS370 <> NULL AND ^INCDTLS371  = NULL AND ^INCDTLS372  = NULL AND ^INCDTLS373 = NULL AND ^INCDTLS374  = NULL AND ^INCDTLS375  = NULL AND ^INCDTLS376  = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS370 <> NULL AND ^INCDTLS371  = NULL AND ^INCDTLS372  = NULL AND ^INCDTLS373 = NULL AND ^INCDTLS374  = NULL AND ^INCDTLS375  = NULL AND ^INCDTLS376  = NULL
            
                    Data Elements:
            
                    ^INCDTLS370 = INCDTLS:Rp:Dividends:DividendTotal
            
                    ^INCDTLS371 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS374 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS375 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS376 = INCDTLS:Rp:Dividends:DividendTotal:Capital.Gains.Total.Amount
                    */
            assertion = (dividends.Rp_Dividends_DividendTotalCollectionExists != false && dividends.INCDTLS371 == null && dividends.INCDTLS372 == null && dividends.INCDTLS373 == null && dividends.INCDTLS374 == null && dividends.INCDTLS375 == null && dividends.INCDTLS376 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000151",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where a Dividend instance is present, at least 1 Dividend Total amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendTotal",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000151"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS371", Value = GetValueOrEmpty(dividends.INCDTLS371) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS374", Value = GetValueOrEmpty(dividends.INCDTLS374) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS375", Value = GetValueOrEmpty(dividends.INCDTLS375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS376", Value = GetValueOrEmpty(dividends.INCDTLS376) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000152(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000152
                    Where the Dividends number of account holders exceeds 1, the Dividend Your Share instance must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS377 = NULL AND ^INCDTLS347 > 1)
        
                    Technical Business Rule Format:
                    ^INCDTLS377 = NULL AND ^INCDTLS347 > 1
            
                    Data Elements:
            
                    ^INCDTLS377 = INCDTLS:Rp:Dividends:DividendYourShare
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (dividends.Rp_Dividends_DividendYourShareCollectionExists == false && dividends.INCDTLS347.GetValueOrDefault() > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.010152",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where the Dividends number of account holders exceeds 1, the Dividend Your Share instance must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000152"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000153(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000153
                    Where a Dividend Your Share instance is present, at least 1 Dividend Your Share amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS377 <> NULL AND ^INCDTLS378  = NULL AND ^INCDTLS379 = NULL AND ^INCDTLS380  = NULL AND ^INCDTLS381  = NULL AND ^INCDTLS382  = NULL AND ^INCDTLS383  = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS377 <> NULL AND ^INCDTLS378  = NULL AND ^INCDTLS379 = NULL AND ^INCDTLS380  = NULL AND ^INCDTLS381  = NULL AND ^INCDTLS382  = NULL AND ^INCDTLS383  = NULL
            
                    Data Elements:
            
                    ^INCDTLS377 = INCDTLS:Rp:Dividends:DividendYourShare
            
                    ^INCDTLS378 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS379 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsFranked.Amount
            
                    ^INCDTLS380 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS381 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS382 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS383 = INCDTLS:Rp:Dividends:DividendYourShare:Capital.Gains.Total.Amount
                    */
            assertion = (dividends.Rp_Dividends_DividendYourShareCollectionExists != false && dividends.INCDTLS378 == null && dividends.INCDTLS379 == null && dividends.INCDTLS380 == null && dividends.INCDTLS381 == null && dividends.INCDTLS382 == null && dividends.INCDTLS383 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where a Dividend Your Share instance is present, at least 1 Dividend Your Share amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000153"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS378", Value = GetValueOrEmpty(dividends.INCDTLS378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS379", Value = GetValueOrEmpty(dividends.INCDTLS379) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS380", Value = GetValueOrEmpty(dividends.INCDTLS380) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS381", Value = GetValueOrEmpty(dividends.INCDTLS381) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS382", Value = GetValueOrEmpty(dividends.INCDTLS382) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS383", Value = GetValueOrEmpty(dividends.INCDTLS383) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000156
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000156(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000156
                    Where a Dividend Your Share instance is present, Dividends number of account holders must be greater than 1
    
                    Legacy Rule Format:
                    (^INCDTLS377 <> NULL AND ^INCDTLS347 = 1)
        
                    Technical Business Rule Format:
                    ^INCDTLS377 <> NULL AND ^INCDTLS347 = 1
            
                    Data Elements:
            
                    ^INCDTLS377 = INCDTLS:Rp:Dividends:DividendYourShare
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (dividends.Rp_Dividends_DividendYourShareCollectionExists != false && dividends.INCDTLS347.GetValueOrDefault() == 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.010156",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where a Dividend Your Share instance is present, Dividends number of account holders must be greater than 1",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000156"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000157
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000157(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000157
                    Your share of TFN amounts withheld from dividends must not exceed 50% of the Your share of unfranked amount
    
                    Legacy Rule Format:
                    (^INCDTLS381 > (^INCDTLS378 * 0.5))
        
                    Technical Business Rule Format:
                    (^INCDTLS381 > (^INCDTLS378 * 0.5))
            
                    Data Elements:
            
                    ^INCDTLS381 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS378 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsUnfranked.Amount
                    */
            assertion = (dividends.INCDTLS381.GetValueOrDefault() > dividends.INCDTLS378.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000157",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from dividends must not exceed 50% of the Your share of unfranked amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000157"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS381", Value = GetValueOrEmpty(dividends.INCDTLS381) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS378", Value = GetValueOrEmpty(dividends.INCDTLS378) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000158
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000158(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000158
                    Your share of TFN amounts withheld from dividends must not exceed the sum of Your share of unfranked amount, Your share of franked amount and Your share of franking credit from dividends amount
    
                    Legacy Rule Format:
                    (^INCDTLS381 > (^INCDTLS378 + ^INCDTLS379 + ^INCDTLS380))
        
                    Technical Business Rule Format:
                    (^INCDTLS381 > (^INCDTLS378 + ^INCDTLS379 + ^INCDTLS380))
            
                    Data Elements:
            
                    ^INCDTLS381 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS378 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS379 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsFranked.Amount
            
                    ^INCDTLS380 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS381.GetValueOrDefault() > dividends.INCDTLS378.GetValueOrDefault() + dividends.INCDTLS379.GetValueOrDefault() + dividends.INCDTLS380.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000158",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts is incorrect",
                    LongDescription = @"Your share of TFN amounts withheld from dividends must not exceed the sum of Your share of unfranked amount, Your share of franked amount and Your share of franking credit from dividends amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000158"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS381", Value = GetValueOrEmpty(dividends.INCDTLS381) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS378", Value = GetValueOrEmpty(dividends.INCDTLS378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS379", Value = GetValueOrEmpty(dividends.INCDTLS379) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS380", Value = GetValueOrEmpty(dividends.INCDTLS380) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000159
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000159(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000159
                    Exempt foreign employment income amount is required where Exempt foreign employment income lump sum in arrears amount is present
    
                    Legacy Rule Format:
                    (^INCDTLS335 = NULL OR ^INCDTLS335 = 0) AND (^INCDTLS338 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS335 = NULL OR ^INCDTLS335 = 0) AND (^INCDTLS338 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS335 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:ExemptForeignEmployment:InternationalDealings.ExemptForeignEmploymentIncome.Amount
            
                    ^INCDTLS338 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:ExemptForeignEmployment:Income.LumpSumArrearsPayment.Amount
                    */
            assertion = ((income.INCDTLS335 == null || income.INCDTLS335 == 0) && income.INCDTLS338 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000159",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt foreign employment income amount must be provided",
                    LongDescription = @"Exempt foreign employment income amount is required where Exempt foreign employment income lump sum in arrears amount is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:ExemptForeignEmployment/tns:InternationalDealingsIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000159"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS335", Value = GetValueOrEmpty(income.INCDTLS335) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS338", Value = GetValueOrEmpty(income.INCDTLS338) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000160
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000160(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000160
                    Total net capital gain from managed fund must be provided when a positive amount for Total current year capital gains from managed fund is present
    
                    Legacy Rule Format:
                    (^INCDTLS394 = NULL AND ^INCDTLS395 > 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS394 = NULL AND ^INCDTLS395 > 0
            
                    Data Elements:
            
                    ^INCDTLS394 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
                    */
            assertion = (managedFunds.INCDTLS394 == null && managedFunds.INCDTLS395.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000160",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total net capital gain from managed fund must be provided",
                    LongDescription = @"Total net capital gain from managed fund must be provided when a positive amount for Total current year capital gains from managed fund is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:CapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000160"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS394", Value = GetValueOrEmpty(managedFunds.INCDTLS394) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000161
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000161(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000161
                    Where the number of account holders is equal to 1, the Total current year capital gains from managed fund must not be less than Total net capital gain from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS387 = 1 AND ^INCDTLS395 < ^INCDTLS394)
        
                    Technical Business Rule Format:
                    ^INCDTLS387 = 1 AND ^INCDTLS395 < ^INCDTLS394
            
                    Data Elements:
            
                    ^INCDTLS394 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() == 1 && managedFunds.INCDTLS395.GetValueOrDefault() < managedFunds.INCDTLS394.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000161",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total net capital gain from managed fund is incorrect",
                    LongDescription = @"Where the number of account holders is equal to 1, the Total current year capital gains from managed fund must not be less than Total net capital gain from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:CapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000161"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS394", Value = GetValueOrEmpty(managedFunds.INCDTLS394) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000162
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000162(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000162
                    When the number of Managed fund number of account holders is one, Total primary production income from managed fund, Total non-primary production income from managed fund, Total franked distribution from managed fund, Total current year capital gains from managed fund, Total net foreign rent from managed fund, Total Australian credits from a NZ franking company from managed fund, and Total other foreign income from managed fund must sum to zero or a positive value
    
                    Legacy Rule Format:
                    (^INCDTLS387 = 1 AND (^INCDTLS402 + ^INCDTLS389 + ^INCDTLS390 + ^INCDTLS395 + ^INCDTLS413 + ^INCDTLS414 + ^INCDTLS399) < 0)
        
                    Technical Business Rule Format:
                    (^INCDTLS387 = 1 AND (^INCDTLS402 + ^INCDTLS389 + ^INCDTLS390 + ^INCDTLS395 + ^INCDTLS413 + ^INCDTLS414 + ^INCDTLS399) < 0)
            
                    Data Elements:
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS414 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsNewZealand.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() == 1 && managedFunds.INCDTLS402.GetValueOrDefault() + managedFunds.INCDTLS389.GetValueOrDefault() + managedFunds.INCDTLS390.GetValueOrDefault() + managedFunds.INCDTLS395.GetValueOrDefault() + managedFunds.INCDTLS413.GetValueOrDefault() + managedFunds.INCDTLS414.GetValueOrDefault() + managedFunds.INCDTLS399.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000162",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The specified amounts must sum to a positive value or zero",
                    LongDescription = @"When the number of Managed fund number of account holders is one, Total primary production income from managed fund, Total non-primary production income from managed fund, Total franked distribution from managed fund, Total current year capital gains from managed fund, Total net foreign rent from managed fund, Total Australian credits from a NZ franking company from managed fund, and Total other foreign income from managed fund must sum to zero or a positive value",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000162"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS414", Value = GetValueOrEmpty(managedFunds.INCDTLS414) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000163
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000163(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000163
                    When the number of Managed fund number of account holders is one, and Total franking credits from managed fund, Total TFN amounts withheld from managed fund, Total tax withheld when ABN not quoted from managed fund, or Total credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Total primary production income from managed fund, Total non-primary production income from managed fund, or Total franked distribution from managed fund must be provided
    
                    Legacy Rule Format:
                    ((^INCDTLS387 = 1) AND (^INCDTLS402 = NULL AND ^INCDTLS389 = NULL AND ^INCDTLS390 = NULL) AND (^INCDTLS404 > 0 OR ^INCDTLS392 > 0 OR ^INCDTLS393 > 0 OR ^INCDTLS408 > 0))
        
                    Technical Business Rule Format:
                    ((^INCDTLS387 = 1) AND (^INCDTLS402 = NULL AND ^INCDTLS389 = NULL AND ^INCDTLS390 = NULL) AND (^INCDTLS404 > 0 OR ^INCDTLS392 > 0 OR ^INCDTLS393 > 0 OR ^INCDTLS408 > 0))
            
                    Data Elements:
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS392 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS393 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS404 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS408 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() == 1 && (managedFunds.INCDTLS402 == null && managedFunds.INCDTLS389 == null && managedFunds.INCDTLS390 == null) && (managedFunds.INCDTLS404.GetValueOrDefault() > 0 || managedFunds.INCDTLS392.GetValueOrDefault() > 0 || managedFunds.INCDTLS393.GetValueOrDefault() > 0 || managedFunds.INCDTLS408.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000163",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Totals of primary production income from managed fund, non-primary production income from managed fund, or franked distribution from managed fund must be provided",
                    LongDescription = @"When the number of Managed fund number of account holders is one, and Total franking credits from managed fund, Total TFN amounts withheld from managed fund, Total tax withheld when ABN not quoted from managed fund, or Total credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Total primary production income from managed fund, Total non-primary production income from managed fund, or Total franked distribution from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000163"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS404", Value = GetValueOrEmpty(managedFunds.INCDTLS404) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS392", Value = GetValueOrEmpty(managedFunds.INCDTLS392) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS393", Value = GetValueOrEmpty(managedFunds.INCDTLS393) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS408", Value = GetValueOrEmpty(managedFunds.INCDTLS408) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000164
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000164(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000164
                    When the number of Managed fund number of account holders is one, Total primary production income from managed fund, Total non-primary production income from managed fund, Total franked distribution from managed fund, Total current year capital gains from managed fund, Total transferor trust income from managed fund, Total CFC income from managed fund, Total net foreign rent from managed fund, and Total other foreign income from managed fund must sum to zero or a positive value
    
                    Legacy Rule Format:
                    (^INCDTLS387 = 1 AND (^INCDTLS402 + ^INCDTLS389 + ^INCDTLS390 + ^INCDTLS395 + ^INCDTLS411 + ^INCDTLS412 + ^INCDTLS413 + ^INCDTLS399 < 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS387 = 1 AND (^INCDTLS402 + ^INCDTLS389 + ^INCDTLS390 + ^INCDTLS395 + ^INCDTLS411 + ^INCDTLS412 + ^INCDTLS413 + ^INCDTLS399 < 0))
            
                    Data Elements:
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS411 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS412 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() == 1 && managedFunds.INCDTLS402.GetValueOrDefault() + managedFunds.INCDTLS389.GetValueOrDefault() + managedFunds.INCDTLS390.GetValueOrDefault() + managedFunds.INCDTLS395.GetValueOrDefault() + managedFunds.INCDTLS411.GetValueOrDefault() + managedFunds.INCDTLS412.GetValueOrDefault() + managedFunds.INCDTLS413.GetValueOrDefault() + managedFunds.INCDTLS399.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000164",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The specified amounts must sum to a positive value or zero",
                    LongDescription = @"When the number of Managed fund number of account holders is one, Total primary production income from managed fund, Total non-primary production income from managed fund, Total franked distribution from managed fund, Total current year capital gains from managed fund, Total transferor trust income from managed fund, Total CFC income from managed fund, Total net foreign rent from managed fund, and Total other foreign income from managed fund must sum to zero or a positive value",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000164"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS411", Value = GetValueOrEmpty(managedFunds.INCDTLS411) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS412", Value = GetValueOrEmpty(managedFunds.INCDTLS412) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000165
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000165(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000165
                    When the number of Managed fund number of account holders is one, and total credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Total primary production income from managed fund, Total non-primary production income from managed fund, Total franked distribution from managed fund, and Total current year capital gains from managed fund must sum to a positive value
    
                    Legacy Rule Format:
                    (^INCDTLS387 = 1 AND ^INCDTLS408 > 0 AND ((^INCDTLS402 + ^INCDTLS389 + ^INCDTLS390 + ^INCDTLS395) <= 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS387 = 1 AND ^INCDTLS408 > 0 AND ((^INCDTLS402 + ^INCDTLS389 + ^INCDTLS390 + ^INCDTLS395) <= 0))
            
                    Data Elements:
            
                    ^INCDTLS408 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() == 1 && managedFunds.INCDTLS408.GetValueOrDefault() > 0 && managedFunds.INCDTLS402.GetValueOrDefault() + managedFunds.INCDTLS389.GetValueOrDefault() + managedFunds.INCDTLS390.GetValueOrDefault() + managedFunds.INCDTLS395.GetValueOrDefault() <= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000165",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The specified amounts must sum to a positive value",
                    LongDescription = @"When the number of Managed fund number of account holders is one, and total credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Total primary production income from managed fund, Total non-primary production income from managed fund, Total franked distribution from managed fund, and Total current year capital gains from managed fund must sum to a positive value",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000165"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS408", Value = GetValueOrEmpty(managedFunds.INCDTLS408) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000166
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000166(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000166
                    Where the number of account holders is equal to 1, the Total current year early stage investor tax offset from managed fund must not be greater than $200,000
    
                    Legacy Rule Format:
                    (^INCDTLS387 = 1 AND ^INCDTLS416 > 200000)
        
                    Technical Business Rule Format:
                    ^INCDTLS387 = 1 AND ^INCDTLS416 > 200000
            
                    Data Elements:
            
                    ^INCDTLS416 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageInvestorTaxOffset.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() == 1 && managedFunds.INCDTLS416.GetValueOrDefault() > 200000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000166",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total current year early stage investor tax offset from managed fund is incorrect",
                    LongDescription = @"Where the number of account holders is equal to 1, the Total current year early stage investor tax offset from managed fund must not be greater than $200,000",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTaxEarlyStageInvestorTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000166"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS416", Value = GetValueOrEmpty(managedFunds.INCDTLS416) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000167
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000167(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000167
                    Your share of net capital gain from managed fund must be provided when a positive amount for Your share of current year capital gain from managed fund is present
    
                    Legacy Rule Format:
                    (^INCDTLS424 = NULL AND ^INCDTLS425 > 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS424 = NULL AND ^INCDTLS425 > 0
            
                    Data Elements:
            
                    ^INCDTLS424 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
                    */
            assertion = (managedFunds.INCDTLS424 == null && managedFunds.INCDTLS425.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000167",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net capital gain from managed fund must be provided",
                    LongDescription = @"Your share of net capital gain from managed fund must be provided when a positive amount for Your share of current year capital gain from managed fund is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000167"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS424", Value = GetValueOrEmpty(managedFunds.INCDTLS424) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000168
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000168(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000168
                    Where the number of account holders is greater than 1, the Your share of current year capital gain from managed fund must not be less than Your share of net capital gain from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS387 > 1 AND ^INCDTLS425 < ^INCDTLS424)
        
                    Technical Business Rule Format:
                    ^INCDTLS387 > 1 AND ^INCDTLS425 < ^INCDTLS424
            
                    Data Elements:
            
                    ^INCDTLS424 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS425.GetValueOrDefault() < managedFunds.INCDTLS424.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000168",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net capital gain from managed fund is incorrect",
                    LongDescription = @"Where the number of account holders is greater than 1, the Your share of current year capital gain from managed fund must not be less than Your share of net capital gain from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000168"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS424", Value = GetValueOrEmpty(managedFunds.INCDTLS424) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000169
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000169(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000169
                    When the Total other foreign income from managed fund amount provided is negative, then Your share of other foreign income from managed fund must not be positive or less than Total other foreign income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS399 < 0 AND (^INCDTLS429 > 0 OR ^INCDTLS429 < ^INCDTLS399))
        
                    Technical Business Rule Format:
                    (^INCDTLS399 < 0 AND (^INCDTLS429 > 0 OR ^INCDTLS429 < ^INCDTLS399))
            
                    Data Elements:
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
                    */
            assertion = (managedFunds.INCDTLS399.GetValueOrDefault() < 0 && (managedFunds.INCDTLS429.GetValueOrDefault() > 0 || managedFunds.INCDTLS429.GetValueOrDefault() < managedFunds.INCDTLS399.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000169",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other foreign income from managed fund is incorrect",
                    LongDescription = @"When the Total other foreign income from managed fund amount provided is negative, then Your share of other foreign income from managed fund must not be positive or less than Total other foreign income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000169"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000170
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000170(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000170
                    Where Total other foreign income from managed fund is positive, Your share of other foreign income from managed fund must not be greater than Total other foreign income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS429 > ^INCDTLS399 AND ^INCDTLS399 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS429 > ^INCDTLS399 AND ^INCDTLS399 >= 0
            
                    Data Elements:
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
                    */
            assertion = (managedFunds.INCDTLS429.GetValueOrDefault() > managedFunds.INCDTLS399.GetValueOrDefault() && managedFunds.INCDTLS399.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000170",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other foreign income from managed fund is incorrect",
                    LongDescription = @"Where Total other foreign income from managed fund is positive, Your share of other foreign income from managed fund must not be greater than Total other foreign income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000170"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000171
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000171(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000171
                    Your share of other foreign income from managed fund must not be negative where Total other foreign income from managed fund is greater than or equal to 0
    
                    Legacy Rule Format:
                    (^INCDTLS429 < 0 AND ^INCDTLS399 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS429 < 0 AND ^INCDTLS399 >= 0
            
                    Data Elements:
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
                    */
            assertion = (managedFunds.INCDTLS429.GetValueOrDefault() < 0 && managedFunds.INCDTLS399.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other foreign income from managed fund is incorrect",
                    LongDescription = @"Your share of other foreign income from managed fund must not be negative where Total other foreign income from managed fund is greater than or equal to 0",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000171"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000172
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000172(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000172
                    When the Total primary production income from managed fund amount provided is negative, then Your share of primary production income from managed fund must not be positive or less than Total primary production income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS402 < 0 AND (^INCDTLS432 > 0 OR ^INCDTLS432 < ^INCDTLS402))
        
                    Technical Business Rule Format:
                    (^INCDTLS402 < 0 AND (^INCDTLS432 > 0 OR ^INCDTLS432 < ^INCDTLS402))
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
                    */
            assertion = (managedFunds.INCDTLS402.GetValueOrDefault() < 0 && (managedFunds.INCDTLS432.GetValueOrDefault() > 0 || managedFunds.INCDTLS432.GetValueOrDefault() < managedFunds.INCDTLS402.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000172",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of primary production income from managed fund is incorrect",
                    LongDescription = @"When the Total primary production income from managed fund amount provided is negative, then Your share of primary production income from managed fund must not be positive or less than Total primary production income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000172"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000173
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000173(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000173
                    Where Total primary production income from managed fund is positive, Your share of primary production income from managed fund must not be greater than Total primary production income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS432 > ^INCDTLS402 AND ^INCDTLS402 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS432 > ^INCDTLS402 AND ^INCDTLS402 >= 0
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
                    */
            assertion = (managedFunds.INCDTLS432.GetValueOrDefault() > managedFunds.INCDTLS402.GetValueOrDefault() && managedFunds.INCDTLS402.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000173",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of primary production income from managed fund is incorrect",
                    LongDescription = @"Where Total primary production income from managed fund is positive, Your share of primary production income from managed fund must not be greater than Total primary production income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000173"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000174
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000174(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000174
                    Your share of primary production income from managed fund must not be negative where Total primary production income from managed fund is greater than or equal to 0
    
                    Legacy Rule Format:
                    (^INCDTLS432 < 0 AND ^INCDTLS402 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS432 < 0 AND ^INCDTLS402 >= 0
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
                    */
            assertion = (managedFunds.INCDTLS432.GetValueOrDefault() < 0 && managedFunds.INCDTLS402.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000174",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of primary production income from managed fund is incorrect",
                    LongDescription = @"Your share of primary production income from managed fund must not be negative where Total primary production income from managed fund is greater than or equal to 0",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000174"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000175
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000175(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000175
                    Your share of current year early stage investor tax offset from managed fund cannot be greater than $200,000
    
                    Legacy Rule Format:
                    (^INCDTLS444 > 200000)
        
                    Technical Business Rule Format:
                    ^INCDTLS444 > 200000
            
                    Data Elements:
            
                    ^INCDTLS444 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageInvestorTaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS444.GetValueOrDefault() > 200000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000175",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year early stage investor tax offset from managed fund cannot be greater than $200,000",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxEarlyStageInvestorTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000175"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS444", Value = GetValueOrEmpty(managedFunds.INCDTLS444) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000176
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000176(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000176
                    Where the Managed fund number of account holders is greater than 1 and Total non-primary production income from managed fund is provided, Your share of non-primary production income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS419 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS389 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS419 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS389 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
                    */
            assertion = (managedFunds.INCDTLS419 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS389 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000176",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of non-primary production income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total non-primary production income from managed fund is provided, Your share of non-primary production income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeNonPrimaryProductionNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000176"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000178
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000178(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000178
                    Where the Managed fund number of account holders is greater than 1 and Total franked distribution from managed fund is provided, Your share of franked distribution from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS420 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS390 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS420 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS390 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
                    */
            assertion = (managedFunds.INCDTLS420 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS390 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000178",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franked distribution from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total franked distribution from managed fund is provided, Your share of franked distribution from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustFrankedNonPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000178"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000179
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000179(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000179
                    Your share of franked distribution amount from managed fund must not exceed Total franked distribution from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS420 > ^INCDTLS390)
        
                    Technical Business Rule Format:
                    ^INCDTLS420 > ^INCDTLS390
            
                    Data Elements:
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
                    */
            assertion = (managedFunds.INCDTLS420.GetValueOrDefault() > managedFunds.INCDTLS390.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000179",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franked distribution from managed fund is incorrect",
                    LongDescription = @"Your share of franked distribution amount from managed fund must not exceed Total franked distribution from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustFrankedNonPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000179"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000180
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000180(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000180
                    Where the Managed fund number of account holders is greater than 1 and Total other deductions relating to non-primary production from managed fund is provided, Your share of other deductions relating to non-primary production from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS421 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS391 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS421 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS391 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS421 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS391 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
                    */
            assertion = (managedFunds.INCDTLS421 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS391 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000180",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other deductions relating to non-primary production from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total other deductions relating to non-primary production from managed fund is provided, Your share of other deductions relating to non-primary production from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000180"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS421", Value = GetValueOrEmpty(managedFunds.INCDTLS421) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS391", Value = GetValueOrEmpty(managedFunds.INCDTLS391) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000181
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000181(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000181
                    Your share of other deductions relating to non-primary production from managed fund must not exceed Total other deductions relating to non-primary production from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS421 > ^INCDTLS391)
        
                    Technical Business Rule Format:
                    ^INCDTLS421 > ^INCDTLS391
            
                    Data Elements:
            
                    ^INCDTLS421 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
            
                    ^INCDTLS391 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
                    */
            assertion = (managedFunds.INCDTLS421.GetValueOrDefault() > managedFunds.INCDTLS391.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000181",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other deductions relating to non-primary production from managed fund is incorrect",
                    LongDescription = @"Your share of other deductions relating to non-primary production from managed fund must not exceed Total other deductions relating to non-primary production from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000181"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS421", Value = GetValueOrEmpty(managedFunds.INCDTLS421) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS391", Value = GetValueOrEmpty(managedFunds.INCDTLS391) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000182
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000182(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000182
                    Where the Managed fund number of account holders is greater than 1 and Total franking credits from managed fund is provided, Your share of franking credits from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS422 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS392 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS422 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS392 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS422 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS392 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsShare.Amount
                    */
            assertion = (managedFunds.INCDTLS422 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS392 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000182",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franking credits from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total franking credits from managed fund is provided, Your share of franking credits from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxFrankingCreditsShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000182"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS422", Value = GetValueOrEmpty(managedFunds.INCDTLS422) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS392", Value = GetValueOrEmpty(managedFunds.INCDTLS392) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000183
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000183(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000183
                    Your share of franking credits from managed fund must not exceed Total franking credits from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS422 > ^INCDTLS392)
        
                    Technical Business Rule Format:
                    ^INCDTLS422 > ^INCDTLS392
            
                    Data Elements:
            
                    ^INCDTLS422 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS392 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsShare.Amount
                    */
            assertion = (managedFunds.INCDTLS422.GetValueOrDefault() > managedFunds.INCDTLS392.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000183",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franking credits from managed fund is incorrect",
                    LongDescription = @"Your share of franking credits from managed fund must not exceed Total franking credits from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxFrankingCreditsShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000183"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS422", Value = GetValueOrEmpty(managedFunds.INCDTLS422) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS392", Value = GetValueOrEmpty(managedFunds.INCDTLS392) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000184
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000184(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000184
                    Where the Managed fund number of account holders is greater than 1 and Total TFN amounts withheld from managed fund is provided, Your share of TFN amounts withheld from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS423 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS393 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS423 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS393 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS423 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS393 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
                    */
            assertion = (managedFunds.INCDTLS423 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS393 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000184",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total TFN amounts withheld from managed fund is provided, Your share of TFN amounts withheld from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000184"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS423", Value = GetValueOrEmpty(managedFunds.INCDTLS423) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS393", Value = GetValueOrEmpty(managedFunds.INCDTLS393) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000185
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000185(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000185
                    Your share of TFN amounts withheld from managed fund must not exceed Total TFN amounts withheld from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS423 > ^INCDTLS393)
        
                    Technical Business Rule Format:
                    ^INCDTLS423 > ^INCDTLS393
            
                    Data Elements:
            
                    ^INCDTLS423 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS393 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
                    */
            assertion = (managedFunds.INCDTLS423.GetValueOrDefault() > managedFunds.INCDTLS393.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000185",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from managed fund is incorrect",
                    LongDescription = @"Your share of TFN amounts withheld from managed fund must not exceed Total TFN amounts withheld from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000185"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS423", Value = GetValueOrEmpty(managedFunds.INCDTLS423) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS393", Value = GetValueOrEmpty(managedFunds.INCDTLS393) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000186
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000186(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000186
                    Where the Managed fund number of account holders is greater than 1 and Total net capital gain from managed fund is provided, Your share of net capital gain from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS424 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS394 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS424 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS394 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS424 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS394 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Net.Amount
                    */
            assertion = (managedFunds.INCDTLS424 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS394 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000186",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net capital gain from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total net capital gain from managed fund is provided, Your share of net capital gain from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000186"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS424", Value = GetValueOrEmpty(managedFunds.INCDTLS424) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS394", Value = GetValueOrEmpty(managedFunds.INCDTLS394) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000187
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000187(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000187
                    Your share of net capital gain from managed fund must not exceed Total net capital gain from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS424 > ^INCDTLS394)
        
                    Technical Business Rule Format:
                    ^INCDTLS424 > ^INCDTLS394
            
                    Data Elements:
            
                    ^INCDTLS424 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS394 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Net.Amount
                    */
            assertion = (managedFunds.INCDTLS424.GetValueOrDefault() > managedFunds.INCDTLS394.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000187",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net capital gain withheld from managed fund is incorrect",
                    LongDescription = @"Your share of net capital gain from managed fund must not exceed Total net capital gain from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000187"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS424", Value = GetValueOrEmpty(managedFunds.INCDTLS424) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS394", Value = GetValueOrEmpty(managedFunds.INCDTLS394) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000188
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000188(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000188
                    Where the Managed fund number of account holders is greater than 1 and Total current year capital gains from managed fund is provided, Your share of current year capital gains from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS425 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS395 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS425 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS395 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
                    */
            assertion = (managedFunds.INCDTLS425 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS395 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000188",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year capital gains from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total current year capital gains from managed fund is provided, Your share of current year capital gains from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:TotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000188"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000189
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000189(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000189
                    Your share of current year capital gains from managed fund must not exceed Total current year capital gains from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS425 > ^INCDTLS395)
        
                    Technical Business Rule Format:
                    ^INCDTLS425 > ^INCDTLS395
            
                    Data Elements:
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
                    */
            assertion = (managedFunds.INCDTLS425.GetValueOrDefault() > managedFunds.INCDTLS395.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000189",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year capital gains from managed fund is incorrect",
                    LongDescription = @"Your share of current year capital gains from managed fund must not exceed Total current year capital gains from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:TotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000189"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000190
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000190(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000190
                    Where the Managed fund number of account holders is greater than 1 and Total capital gains - discount method from managed fund is provided, Your share of capital gains - discount method from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS426 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS396 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS426 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS396 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS426 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.DiscountedMethod.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS396 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.DiscountedMethod.Amount
                    */
            assertion = (managedFunds.INCDTLS426 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS396 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000190",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of capital gains - discount method from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total capital gains - discount method from managed fund is provided, Your share of capital gains - discount method from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:DiscountedMethodA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000190"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS426", Value = GetValueOrEmpty(managedFunds.INCDTLS426) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS396", Value = GetValueOrEmpty(managedFunds.INCDTLS396) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000191
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000191(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000191
                    Your share of capital gains - discount method from managed fund must not exceed Total capital gains - discount method from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS426 > ^INCDTLS396)
        
                    Technical Business Rule Format:
                    ^INCDTLS426 > ^INCDTLS396
            
                    Data Elements:
            
                    ^INCDTLS426 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.DiscountedMethod.Amount
            
                    ^INCDTLS396 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.DiscountedMethod.Amount
                    */
            assertion = (managedFunds.INCDTLS426.GetValueOrDefault() > managedFunds.INCDTLS396.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of capital gains - discount method from managed fund is incorrect",
                    LongDescription = @"Your share of capital gains - discount method from managed fund must not exceed Total capital gains - discount method from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:DiscountedMethodA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000191"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS426", Value = GetValueOrEmpty(managedFunds.INCDTLS426) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS396", Value = GetValueOrEmpty(managedFunds.INCDTLS396) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000192
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000192(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000192
                    Where the Managed fund number of account holders is greater than 1 and Total capital gains - other method from managed fund is provided, Your share of capital gains - other method from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS427 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS397 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS427 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS397 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS427 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.OtherMethod.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS397 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.OtherMethod.Amount
                    */
            assertion = (managedFunds.INCDTLS427 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS397 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000192",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of capital gains - other method from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total capital gains - other method from managed fund is provided, Your share of capital gains - other method from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:OtherMethodA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000192"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS427", Value = GetValueOrEmpty(managedFunds.INCDTLS427) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS397", Value = GetValueOrEmpty(managedFunds.INCDTLS397) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000193
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000193(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000193
                    Your share of capital gains - other method from managed fund must not exceed Total capital gains - other method from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS427 > ^INCDTLS397)
        
                    Technical Business Rule Format:
                    ^INCDTLS427 > ^INCDTLS397
            
                    Data Elements:
            
                    ^INCDTLS427 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.OtherMethod.Amount
            
                    ^INCDTLS397 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.OtherMethod.Amount
                    */
            assertion = (managedFunds.INCDTLS427.GetValueOrDefault() > managedFunds.INCDTLS397.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000193",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of capital gains - other method from managed fund is incorrect",
                    LongDescription = @"Your share of capital gains - other method from managed fund must not exceed Total capital gains - other method from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:YourShareCapitalGains/tns:OtherMethodA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000193"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS427", Value = GetValueOrEmpty(managedFunds.INCDTLS427) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS397", Value = GetValueOrEmpty(managedFunds.INCDTLS397) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000194(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000194
                    Where the Managed fund number of account holders is greater than 1 and Total assessable foreign income from managed fund is provided, Your share of assessable foreign income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS428 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS398 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS428 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS398 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS428 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS398 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignIncomeGross.Amount
                    */
            assertion = (managedFunds.INCDTLS428 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS398 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of assessable foreign income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total assessable foreign income from managed fund is provided, Your share of assessable foreign income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignIncomeGrossA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000194"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS428", Value = GetValueOrEmpty(managedFunds.INCDTLS428) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS398", Value = GetValueOrEmpty(managedFunds.INCDTLS398) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000195(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000195
                    Your share of assessable foreign income from managed fund must not exceed Total assessable foreign income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS428 > ^INCDTLS398)
        
                    Technical Business Rule Format:
                    ^INCDTLS428 > ^INCDTLS398
            
                    Data Elements:
            
                    ^INCDTLS428 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS398 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignIncomeGross.Amount
                    */
            assertion = (managedFunds.INCDTLS428.GetValueOrDefault() > managedFunds.INCDTLS398.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of assessable foreign income from managed fund is incorrect",
                    LongDescription = @"Your share of assessable foreign income from managed fund must not exceed Total assessable foreign income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignIncomeGrossA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000195"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS428", Value = GetValueOrEmpty(managedFunds.INCDTLS428) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS398", Value = GetValueOrEmpty(managedFunds.INCDTLS398) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000196
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000196(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000196
                    Where the Managed fund number of account holders is greater than 1 and Total other foreign income from managed fund is provided, Your share of other foreign income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS429 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS399 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS429 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS399 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
                    */
            assertion = (managedFunds.INCDTLS429 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS399 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000196",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other foreign income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total other foreign income from managed fund is provided, Your share of other foreign income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000196"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000197
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000197(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000197
                    Where the Managed fund number of account holders is greater than 1 and Total foreign income tax offset from managed fund is provided, Your share of foreign income tax offset from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS430 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS400 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS430 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS400 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS430 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.TaxOffset.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS400 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.TaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS430 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS400 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000197",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of foreign income tax offset from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total foreign income tax offset from managed fund is provided, Your share of foreign income tax offset from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000197"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS430", Value = GetValueOrEmpty(managedFunds.INCDTLS430) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS400", Value = GetValueOrEmpty(managedFunds.INCDTLS400) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000198
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000198(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000198
                    Your share of foreign income tax offset from managed fund must not exceed Total foreign income tax offset from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS430 > ^INCDTLS400)
        
                    Technical Business Rule Format:
                    ^INCDTLS430 > ^INCDTLS400
            
                    Data Elements:
            
                    ^INCDTLS430 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.TaxOffset.Amount
            
                    ^INCDTLS400 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.TaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS430.GetValueOrDefault() > managedFunds.INCDTLS400.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000198",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of foreign income tax offset from managed fund is incorrect",
                    LongDescription = @"Your share of foreign income tax offset from managed fund must not exceed Total foreign income tax offset from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000198"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS430", Value = GetValueOrEmpty(managedFunds.INCDTLS430) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS400", Value = GetValueOrEmpty(managedFunds.INCDTLS400) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000199
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000199(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000199
                    Where the Managed fund number of account holders is greater than 1 and Total other income from managed fund is provided, Your share of other income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS431 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS401 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS431 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS401 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS431 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS401 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
                    */
            assertion = (managedFunds.INCDTLS431 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS401 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000199",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total other income from managed fund is provided, Your share of other income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxIncomeCategoryMiscellaneousTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000199"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS431", Value = GetValueOrEmpty(managedFunds.INCDTLS431) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS401", Value = GetValueOrEmpty(managedFunds.INCDTLS401) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000200(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000200
                    Your share of other income from managed fund must not exceed Total other income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS431 > ^INCDTLS401)
        
                    Technical Business Rule Format:
                    ^INCDTLS431 > ^INCDTLS401
            
                    Data Elements:
            
                    ^INCDTLS431 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
            
                    ^INCDTLS401 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
                    */
            assertion = (managedFunds.INCDTLS431.GetValueOrDefault() > managedFunds.INCDTLS401.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000200",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other income from managed fund is incorrect",
                    LongDescription = @"Your share of other income from managed fund must not exceed Total other income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxIncomeCategoryMiscellaneousTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000200"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS431", Value = GetValueOrEmpty(managedFunds.INCDTLS431) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS401", Value = GetValueOrEmpty(managedFunds.INCDTLS401) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000201(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000201
                    Where the Managed fund number of account holders is greater than 1 and the Total primary production income from managed fund is provided, Your share of primary production income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS432 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS402 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS432 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS402 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
                    */
            assertion = (managedFunds.INCDTLS432 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS402 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000201",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of primary production income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total primary production income from managed fund is provided, Your share of primary production income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000201"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000202
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000202(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000202
                    Where the Managed fund number of account holders is greater than 1 and the Total other deductions for primary production distributions from managed fund is provided, Your share of other deductions for primary production distributions from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS433 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS403 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS433 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS403 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS433 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS403 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
                    */
            assertion = (managedFunds.INCDTLS433 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS403 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000202",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other deductions for primary production distributions from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total other deductions for primary production distributions from managed fund is provided, Your share of other deductions for primary production distributions from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000202"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS433", Value = GetValueOrEmpty(managedFunds.INCDTLS433) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS403", Value = GetValueOrEmpty(managedFunds.INCDTLS403) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000203(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000203
                    Your share of other deductions for primary production distributions from managed fund must not exceed Total other deductions for primary production distributions from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS433 > ^INCDTLS403)
        
                    Technical Business Rule Format:
                    ^INCDTLS433 > ^INCDTLS403
            
                    Data Elements:
            
                    ^INCDTLS433 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            
                    ^INCDTLS403 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
                    */
            assertion = (managedFunds.INCDTLS433.GetValueOrDefault() > managedFunds.INCDTLS403.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other deductions for primary production distributions from managed fund is incorrect",
                    LongDescription = @"Your share of other deductions for primary production distributions from managed fund must not exceed Total other deductions for primary production distributions from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000203"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS433", Value = GetValueOrEmpty(managedFunds.INCDTLS433) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS403", Value = GetValueOrEmpty(managedFunds.INCDTLS403) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000204(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000204
                    Where the Managed fund number of account holders is greater than 1 and the Total tax withheld when ABN not quoted from managed fund is provided, Your share of tax withheld when ABN not quoted from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS434 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS404 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS434 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS404 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS434 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS404 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
                    */
            assertion = (managedFunds.INCDTLS434 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS404 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000204",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of tax withheld when ABN not quoted from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total tax withheld when ABN not quoted from managed fund is provided, Your share of tax withheld when ABN not quoted from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000204"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS434", Value = GetValueOrEmpty(managedFunds.INCDTLS434) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS404", Value = GetValueOrEmpty(managedFunds.INCDTLS404) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000205(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000205
                    Your share of tax withheld when ABN not quoted from managed fund must not exceed Total tax withheld when ABN not quoted from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS434 > ^INCDTLS404)
        
                    Technical Business Rule Format:
                    ^INCDTLS434 > ^INCDTLS404
            
                    Data Elements:
            
                    ^INCDTLS434 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS404 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
                    */
            assertion = (managedFunds.INCDTLS434.GetValueOrDefault() > managedFunds.INCDTLS404.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000205",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of tax withheld when ABN not quoted from managed fund is incorrect",
                    LongDescription = @"Your share of tax withheld when ABN not quoted from managed fund must not exceed Total tax withheld when ABN not quoted from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000205"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS434", Value = GetValueOrEmpty(managedFunds.INCDTLS434) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS404", Value = GetValueOrEmpty(managedFunds.INCDTLS404) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000206
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000206(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000206
                    Where the Managed fund number of account holders is greater than 1 and the Total credit for tax paid by trustee from managed fund is provided, Your share of credit for tax paid by trustee from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS435 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS405 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS435 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS405 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS435 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS405 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
                    */
            assertion = (managedFunds.INCDTLS435 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS405 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000206",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of credit for tax paid by trustee from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total credit for tax paid by trustee from managed fund is provided, Your share of credit for tax paid by trustee from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxCreditTrusteeTaxPaidTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000206"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS435", Value = GetValueOrEmpty(managedFunds.INCDTLS435) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS405", Value = GetValueOrEmpty(managedFunds.INCDTLS405) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000207
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000207(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000207
                    Your share of credit for tax paid by trustee from managed fund must not exceed Total credit for tax paid by trustee from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS435 > ^INCDTLS405)
        
                    Technical Business Rule Format:
                    ^INCDTLS435 > ^INCDTLS405
            
                    Data Elements:
            
                    ^INCDTLS435 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS405 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
                    */
            assertion = (managedFunds.INCDTLS435.GetValueOrDefault() > managedFunds.INCDTLS405.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000207",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of credit for tax paid by trustee from managed fund is incorrect",
                    LongDescription = @"Your share of credit for tax paid by trustee from managed fund must not exceed Total credit for tax paid by trustee from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxCreditTrusteeTaxPaidTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000207"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS435", Value = GetValueOrEmpty(managedFunds.INCDTLS435) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS405", Value = GetValueOrEmpty(managedFunds.INCDTLS405) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000208(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000208
                    Where the Managed fund number of account holders is greater than 1 and the Total credit for foreign resident withholding amounts (excluding capital gains) from managed fund is provided, Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS436 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS408 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS436 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS408 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS436 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS408 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
                    */
            assertion = (managedFunds.INCDTLS436 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS408 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total credit for foreign resident withholding amounts (excluding capital gains) from managed fund is provided, Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000208"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS436", Value = GetValueOrEmpty(managedFunds.INCDTLS436) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS408", Value = GetValueOrEmpty(managedFunds.INCDTLS408) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000209(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000209
                    Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund must not exceed Total credit for foreign resident withholding amounts (excluding capital gains) from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS436 > ^INCDTLS408)
        
                    Technical Business Rule Format:
                    ^INCDTLS436 > ^INCDTLS408
            
                    Data Elements:
            
                    ^INCDTLS436 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS408 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
                    */
            assertion = (managedFunds.INCDTLS436.GetValueOrDefault() > managedFunds.INCDTLS408.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000209",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund is incorrect",
                    LongDescription = @"Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund must not exceed Total credit for foreign resident withholding amounts (excluding capital gains) from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000209"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS436", Value = GetValueOrEmpty(managedFunds.INCDTLS436) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS408", Value = GetValueOrEmpty(managedFunds.INCDTLS408) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000210
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000210(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000210
                    Where the Managed fund number of account holders is greater than 1 and the Total Australian credits from a NZ franking company from managed fund is provided, Your share of Australian credits from a NZ franking company from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS442 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS414 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS442 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS414 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS442 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsNewZealand.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS414 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsNewZealand.Amount
                    */
            assertion = (managedFunds.INCDTLS442 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS414 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000210",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of Australian credits from a NZ franking company from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total Australian credits from a NZ franking company from managed fund is provided, Your share of Australian credits from a NZ franking company from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxFrankingCreditsNewZealandA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000210"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS442", Value = GetValueOrEmpty(managedFunds.INCDTLS442) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS414", Value = GetValueOrEmpty(managedFunds.INCDTLS414) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000211(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000211
                    Your share of Australian credits from a NZ franking company from managed fund must not exceed the Total Australian credits from a NZ franking company from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS442 > ^INCDTLS414)
        
                    Technical Business Rule Format:
                    ^INCDTLS442 > ^INCDTLS414
            
                    Data Elements:
            
                    ^INCDTLS442 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsNewZealand.Amount
            
                    ^INCDTLS414 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsNewZealand.Amount
                    */
            assertion = (managedFunds.INCDTLS442.GetValueOrDefault() > managedFunds.INCDTLS414.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of Australian credits from a NZ franking company from managed fund is incorrect",
                    LongDescription = @"Your share of Australian credits from a NZ franking company from managed fund must not exceed the Total Australian credits from a NZ franking company from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxFrankingCreditsNewZealandA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000211"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS442", Value = GetValueOrEmpty(managedFunds.INCDTLS442) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS414", Value = GetValueOrEmpty(managedFunds.INCDTLS414) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000212
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000212(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000212
                    Where the Managed fund number of account holders is greater than 1 and the Total current year early stage venture capital limited partnership tax offset from managed fund is provided, Your share of current year early stage venture capital limited partnership tax offset from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS443 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS415 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS443 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS415 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS443 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS415 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS443 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS415 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year early stage venture capital limited partnership tax offset from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total current year early stage venture capital limited partnership tax offset from managed fund is provided, Your share of current year early stage venture capital limited partnership tax offset from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000212"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS443", Value = GetValueOrEmpty(managedFunds.INCDTLS443) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS415", Value = GetValueOrEmpty(managedFunds.INCDTLS415) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000213
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000213(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000213
                    Your share of current year early stage venture capital limited partnership tax offset from managed fund must not exceed the Total current year early stage venture capital limited partnership tax offset from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS443 > ^INCDTLS415)
        
                    Technical Business Rule Format:
                    ^INCDTLS443 > ^INCDTLS415
            
                    Data Elements:
            
                    ^INCDTLS443 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
            
                    ^INCDTLS415 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS443.GetValueOrDefault() > managedFunds.INCDTLS415.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000213",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year early stage venture capital limited partnership tax offset from managed fund is incorrect",
                    LongDescription = @"Your share of current year early stage venture capital limited partnership tax offset from managed fund must not exceed the Total current year early stage venture capital limited partnership tax offset from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000213"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS443", Value = GetValueOrEmpty(managedFunds.INCDTLS443) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS415", Value = GetValueOrEmpty(managedFunds.INCDTLS415) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000214
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000214(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000214
                    Where the Managed fund number of account holders is greater than 1 and the Total current year early stage investor tax offset from managed fund is provided, Your share of current year early stage investor tax offset from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS444 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS416 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS444 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS416 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS444 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageInvestorTaxOffset.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS416 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageInvestorTaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS444 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS416 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000214",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year early stage investor tax offset from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total current year early stage investor tax offset from managed fund is provided, Your share of current year early stage investor tax offset from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxEarlyStageInvestorTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000214"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS444", Value = GetValueOrEmpty(managedFunds.INCDTLS444) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS416", Value = GetValueOrEmpty(managedFunds.INCDTLS416) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000215
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000215(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000215
                    Your share of current year early stage investor tax offset from managed fund must not exceed the Total current year early stage investor tax offset from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS444 > ^INCDTLS416)
        
                    Technical Business Rule Format:
                    ^INCDTLS444 > ^INCDTLS416
            
                    Data Elements:
            
                    ^INCDTLS444 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageInvestorTaxOffset.Amount
            
                    ^INCDTLS416 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageInvestorTaxOffset.Amount
                    */
            assertion = (managedFunds.INCDTLS444.GetValueOrDefault() > managedFunds.INCDTLS416.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000215",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of current year early stage investor tax offset from managed fund is incorrect",
                    LongDescription = @"Your share of current year early stage investor tax offset from managed fund must not exceed the Total current year early stage investor tax offset from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxEarlyStageInvestorTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000215"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS444", Value = GetValueOrEmpty(managedFunds.INCDTLS444) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS416", Value = GetValueOrEmpty(managedFunds.INCDTLS416) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000216
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000216(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000216
                    Where the Managed fund number of account holders is greater than 1 and the Total other refundable tax offsets: exploration credits from managed fund is provided, Your share of other refundable tax offsets: exploration credits from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS445 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS417 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS445 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS417 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS445 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS417 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (managedFunds.INCDTLS445 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS417 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000216",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other refundable tax offsets: exploration credits from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and the Total other refundable tax offsets: exploration credits from managed fund is provided, Your share of other refundable tax offsets: exploration credits from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxExplorationCreditsDistributedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000216"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS445", Value = GetValueOrEmpty(managedFunds.INCDTLS445) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS417", Value = GetValueOrEmpty(managedFunds.INCDTLS417) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000217
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000217(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000217
                    Your share of other refundable tax offsets: exploration credits from managed fund must not exceed the Total other refundable tax offsets: exploration credits from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS445 > ^INCDTLS417)
        
                    Technical Business Rule Format:
                    ^INCDTLS445 > ^INCDTLS417
            
                    Data Elements:
            
                    ^INCDTLS445 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS417 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (managedFunds.INCDTLS445.GetValueOrDefault() > managedFunds.INCDTLS417.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000217",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other refundable tax offsets: exploration credits from managed fund is incorrect",
                    LongDescription = @"Your share of other refundable tax offsets: exploration credits from managed fund must not exceed the Total other refundable tax offsets: exploration credits from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxExplorationCreditsDistributedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000217"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS445", Value = GetValueOrEmpty(managedFunds.INCDTLS445) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS417", Value = GetValueOrEmpty(managedFunds.INCDTLS417) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000218
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000218(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000218
                    Where the Managed fund number of account holders is greater than 1 and Total national rental affordability scheme offset from managed fund is provided, Your share of national rental affordability scheme offset from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS437 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS409 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS437 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS409 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS437 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS409 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
                    */
            assertion = (managedFunds.INCDTLS437 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS409 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of national rental affordability scheme offset from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total national rental affordability scheme offset from managed fund is provided, Your share of national rental affordability scheme offset from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000218"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS437", Value = GetValueOrEmpty(managedFunds.INCDTLS437) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS409", Value = GetValueOrEmpty(managedFunds.INCDTLS409) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000219
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000219(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000219
                    Your share of national rental affordability scheme offset from managed fund must not exceed Total national rental affordability scheme offset from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS437 > ^INCDTLS409)
        
                    Technical Business Rule Format:
                    ^INCDTLS437 > ^INCDTLS409
            
                    Data Elements:
            
                    ^INCDTLS437 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            
                    ^INCDTLS409 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
                    */
            assertion = (managedFunds.INCDTLS437.GetValueOrDefault() > managedFunds.INCDTLS409.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000219",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of national rental affordability scheme offset from managed fund is incorrect",
                    LongDescription = @"Your share of national rental affordability scheme offset from managed fund must not exceed Total national rental affordability scheme offset from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000219"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS437", Value = GetValueOrEmpty(managedFunds.INCDTLS437) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS409", Value = GetValueOrEmpty(managedFunds.INCDTLS409) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000220
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000220(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000220
                    Where the Managed fund number of account holders is greater than 1 and Total credits for foreign resident capital gains withholding amounts from managed fund is provided, Your share of credits for foreign resident capital gains withholding amounts from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS438 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS410 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS438 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS410 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS438 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS410 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
                    */
            assertion = (managedFunds.INCDTLS438 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS410 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000220",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of credits for foreign resident capital gains withholding amounts from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total credits for foreign resident capital gains withholding amounts from managed fund is provided, Your share of credits for foreign resident capital gains withholding amounts from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000220"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS438", Value = GetValueOrEmpty(managedFunds.INCDTLS438) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS410", Value = GetValueOrEmpty(managedFunds.INCDTLS410) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000221
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000221(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000221
                    Your share of credits for foreign resident capital gains withholding amounts from managed fund must not exceed Total credits for foreign resident capital gains withholding amounts from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS438 > ^INCDTLS410)
        
                    Technical Business Rule Format:
                    ^INCDTLS438 > ^INCDTLS410
            
                    Data Elements:
            
                    ^INCDTLS438 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            
                    ^INCDTLS410 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
                    */
            assertion = (managedFunds.INCDTLS438.GetValueOrDefault() > managedFunds.INCDTLS410.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000221",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of credits for foreign resident capital gains withholding amounts from managed fund is incorrect",
                    LongDescription = @"Your share of credits for foreign resident capital gains withholding amounts from managed fund must not exceed Total credits for foreign resident capital gains withholding amounts from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000221"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS438", Value = GetValueOrEmpty(managedFunds.INCDTLS438) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS410", Value = GetValueOrEmpty(managedFunds.INCDTLS410) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000222
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000222(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000222
                    Where the Managed fund number of account holders is greater than 1 and Total transferor trust income from managed fund is provided, Your share of transferor trust income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS439 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS411 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS439 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS411 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS439 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS411 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
                    */
            assertion = (managedFunds.INCDTLS439 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS411 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000222",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of transferor trust income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total transferor trust income from managed fund is provided, Your share of transferor trust income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsTransferorTrustAttributedForeignIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000222"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS439", Value = GetValueOrEmpty(managedFunds.INCDTLS439) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS411", Value = GetValueOrEmpty(managedFunds.INCDTLS411) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000223
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000223(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000223
                    Your share of transferor trust income from managed fund must not exceed Total transferor trust income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS439 > ^INCDTLS411)
        
                    Technical Business Rule Format:
                    ^INCDTLS439 > ^INCDTLS411
            
                    Data Elements:
            
                    ^INCDTLS439 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS411 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
                    */
            assertion = (managedFunds.INCDTLS439.GetValueOrDefault() > managedFunds.INCDTLS411.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of transferor trust income from managed fund is incorrect",
                    LongDescription = @"Your share of transferor trust income from managed fund must not exceed Total transferor trust income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsTransferorTrustAttributedForeignIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000223"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS439", Value = GetValueOrEmpty(managedFunds.INCDTLS439) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS411", Value = GetValueOrEmpty(managedFunds.INCDTLS411) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000224
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000224(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000224
                    Where the Managed fund number of account holders is greater than 1 and Total CFC income from managed fund is provided, Your share of CFC income from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS440 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS412 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS440 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS412 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS440 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS412 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.ControlledForeignCompaniesAttributableIncome.Amount
                    */
            assertion = (managedFunds.INCDTLS440 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS412 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of CFC income from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total CFC income from managed fund is provided, Your share of CFC income from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeControlledForeignCompaniesAttributableIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000224"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS440", Value = GetValueOrEmpty(managedFunds.INCDTLS440) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS412", Value = GetValueOrEmpty(managedFunds.INCDTLS412) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000225
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000225(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000225
                    Your share of CFC income from managed fund must not exceed Total CFC income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS440 > ^INCDTLS412)
        
                    Technical Business Rule Format:
                    ^INCDTLS440 > ^INCDTLS412
            
                    Data Elements:
            
                    ^INCDTLS440 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS412 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.ControlledForeignCompaniesAttributableIncome.Amount
                    */
            assertion = (managedFunds.INCDTLS440.GetValueOrDefault() > managedFunds.INCDTLS412.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000225",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of CFC income from managed fund is incorrect",
                    LongDescription = @"Your share of CFC income from managed fund must not exceed Total CFC income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeControlledForeignCompaniesAttributableIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000225"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS440", Value = GetValueOrEmpty(managedFunds.INCDTLS440) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS412", Value = GetValueOrEmpty(managedFunds.INCDTLS412) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000226
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000226(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000226
                    Where the Managed fund number of account holders is greater than 1 and Total net foreign rent from managed fund is provided, Your share of net foreign rent from managed fund must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS441 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS413 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS441 = NULL AND ^INCDTLS387 > 1 AND ^INCDTLS413 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS441 == null && managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS413 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net foreign rent from managed fund must be provided",
                    LongDescription = @"Where the Managed fund number of account holders is greater than 1 and Total net foreign rent from managed fund is provided, Your share of net foreign rent from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignRentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000226"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000245
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000245(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000245
                    Where a Managed Fund Distributions instance is present, at least one Managed Fund Distributions Total or Managed Fund Distributions Your Share amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS388 <> NULL AND ^INCDTLS389 = NULL AND ^INCDTLS390 = NULL AND ^INCDTLS391 = NULL AND ^INCDTLS392 = NULL AND ^INCDTLS393 = NULL AND ^INCDTLS398 = NULL AND ^INCDTLS399 = NULL AND ^INCDTLS400 = NULL AND ^INCDTLS401 = NULL AND ^INCDTLS402 = NULL AND ^INCDTLS403 = NULL AND ^INCDTLS404 = NULL AND ^INCDTLS405 = NULL AND ^INCDTLS406 = NULL AND ^INCDTLS407 = NULL AND ^INCDTLS408 = NULL AND ^INCDTLS409 = NULL AND ^INCDTLS410 = NULL AND ^INCDTLS411 = NULL AND ^INCDTLS412 = NULL AND ^INCDTLS413 = NULL AND ^INCDTLS414 = NULL AND ^INCDTLS415 = NULL AND ^INCDTLS416 = NULL AND ^INCDTLS417 = NULL AND ^INCDTLS394 = NULL AND ^INCDTLS395 = NULL AND ^INCDTLS396 = NULL AND ^INCDTLS397 = NULL AND ^INCDTLS419 = NULL AND ^INCDTLS420 = NULL AND ^INCDTLS421 = NULL AND ^INCDTLS422 = NULL AND ^INCDTLS423 = NULL AND ^INCDTLS428 = NULL AND ^INCDTLS429 = NULL AND ^INCDTLS430 = NULL AND ^INCDTLS431 = NULL AND ^INCDTLS432 = NULL AND ^INCDTLS433 = NULL AND ^INCDTLS434 = NULL AND ^INCDTLS435 = NULL AND ^INCDTLS436 = NULL AND ^INCDTLS437 = NULL AND ^INCDTLS438 = NULL AND ^INCDTLS439 = NULL AND ^INCDTLS440 = NULL AND ^INCDTLS441 = NULL AND ^INCDTLS442 = NULL AND ^INCDTLS443 = NULL AND ^INCDTLS444 = NULL AND ^INCDTLS445 = NULL AND ^INCDTLS424 = NULL AND ^INCDTLS425 = NULL AND ^INCDTLS426 = NULL AND ^INCDTLS427 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS388 <> NULL AND ^INCDTLS389 = NULL AND ^INCDTLS390 = NULL AND ^INCDTLS391 = NULL AND ^INCDTLS392 = NULL AND ^INCDTLS393 = NULL AND ^INCDTLS398 = NULL AND ^INCDTLS399 = NULL AND ^INCDTLS400 = NULL AND ^INCDTLS401 = NULL AND ^INCDTLS402 = NULL AND ^INCDTLS403 = NULL AND ^INCDTLS404 = NULL AND ^INCDTLS405 = NULL AND ^INCDTLS406 = NULL AND ^INCDTLS407 = NULL AND ^INCDTLS408 = NULL AND ^INCDTLS409 = NULL AND ^INCDTLS410 = NULL AND ^INCDTLS411 = NULL AND ^INCDTLS412 = NULL AND ^INCDTLS413 = NULL AND ^INCDTLS414 = NULL AND ^INCDTLS415 = NULL AND ^INCDTLS416 = NULL AND ^INCDTLS417 = NULL AND ^INCDTLS394 = NULL AND ^INCDTLS395 = NULL AND ^INCDTLS396 = NULL AND ^INCDTLS397 = NULL AND ^INCDTLS419 = NULL AND ^INCDTLS420 = NULL AND ^INCDTLS421 = NULL AND ^INCDTLS422 = NULL AND ^INCDTLS423 = NULL AND ^INCDTLS428 = NULL AND ^INCDTLS429 = NULL AND ^INCDTLS430 = NULL AND ^INCDTLS431 = NULL AND ^INCDTLS432 = NULL AND ^INCDTLS433 = NULL AND ^INCDTLS434 = NULL AND ^INCDTLS435 = NULL AND ^INCDTLS436 = NULL AND ^INCDTLS437 = NULL AND ^INCDTLS438 = NULL AND ^INCDTLS439 = NULL AND ^INCDTLS440 = NULL AND ^INCDTLS441 = NULL AND ^INCDTLS442 = NULL AND ^INCDTLS443 = NULL AND ^INCDTLS444 = NULL AND ^INCDTLS445 = NULL AND ^INCDTLS424 = NULL AND ^INCDTLS425 = NULL AND ^INCDTLS426 = NULL AND ^INCDTLS427 = NULL
            
                    Data Elements:
            
                    ^INCDTLS388 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS390 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS391 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
            
                    ^INCDTLS392 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS393 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS394 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS395 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS396 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.DiscountedMethod.Amount
            
                    ^INCDTLS397 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:CapitalGains:Capital.Gains.OtherMethod.Amount
            
                    ^INCDTLS398 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS400 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.TaxOffset.Amount
            
                    ^INCDTLS401 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
            
                    ^INCDTLS402 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS403 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            
                    ^INCDTLS404 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS405 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS406 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidReason.Code
            
                    ^INCDTLS407 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustDistributions.Amount
            
                    ^INCDTLS408 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS409 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            
                    ^INCDTLS410 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            
                    ^INCDTLS411 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS412 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS414 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.FrankingCreditsNewZealand.Amount
            
                    ^INCDTLS415 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
            
                    ^INCDTLS416 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.EarlyStageInvestorTaxOffset.Amount
            
                    ^INCDTLS417 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS421 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
            
                    ^INCDTLS422 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS423 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS424 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS426 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.DiscountedMethod.Amount
            
                    ^INCDTLS427 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.OtherMethod.Amount
            
                    ^INCDTLS428 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS430 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.TaxOffset.Amount
            
                    ^INCDTLS431 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS433 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            
                    ^INCDTLS434 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS435 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS436 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS437 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            
                    ^INCDTLS438 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            
                    ^INCDTLS439 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS440 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS442 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsNewZealand.Amount
            
                    ^INCDTLS443 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
            
                    ^INCDTLS444 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageInvestorTaxOffset.Amount
            
                    ^INCDTLS445 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (managedFunds.Rp_ManagedFunds_ManagedFundsTotalCollectionExists != false && managedFunds.INCDTLS389 == null && managedFunds.INCDTLS390 == null && managedFunds.INCDTLS391 == null && managedFunds.INCDTLS392 == null && managedFunds.INCDTLS393 == null && managedFunds.INCDTLS398 == null && managedFunds.INCDTLS399 == null && managedFunds.INCDTLS400 == null && managedFunds.INCDTLS401 == null && managedFunds.INCDTLS402 == null && managedFunds.INCDTLS403 == null && managedFunds.INCDTLS404 == null && managedFunds.INCDTLS405 == null && managedFunds.INCDTLS406 == null && managedFunds.INCDTLS407 == null && managedFunds.INCDTLS408 == null && managedFunds.INCDTLS409 == null && managedFunds.INCDTLS410 == null && managedFunds.INCDTLS411 == null && managedFunds.INCDTLS412 == null && managedFunds.INCDTLS413 == null && managedFunds.INCDTLS414 == null && managedFunds.INCDTLS415 == null && managedFunds.INCDTLS416 == null && managedFunds.INCDTLS417 == null && managedFunds.INCDTLS394 == null && managedFunds.INCDTLS395 == null && managedFunds.INCDTLS396 == null && managedFunds.INCDTLS397 == null && managedFunds.INCDTLS419 == null && managedFunds.INCDTLS420 == null && managedFunds.INCDTLS421 == null && managedFunds.INCDTLS422 == null && managedFunds.INCDTLS423 == null && managedFunds.INCDTLS428 == null && managedFunds.INCDTLS429 == null && managedFunds.INCDTLS430 == null && managedFunds.INCDTLS431 == null && managedFunds.INCDTLS432 == null && managedFunds.INCDTLS433 == null && managedFunds.INCDTLS434 == null && managedFunds.INCDTLS435 == null && managedFunds.INCDTLS436 == null && managedFunds.INCDTLS437 == null && managedFunds.INCDTLS438 == null && managedFunds.INCDTLS439 == null && managedFunds.INCDTLS440 == null && managedFunds.INCDTLS441 == null && managedFunds.INCDTLS442 == null && managedFunds.INCDTLS443 == null && managedFunds.INCDTLS444 == null && managedFunds.INCDTLS445 == null && managedFunds.INCDTLS424 == null && managedFunds.INCDTLS425 == null && managedFunds.INCDTLS426 == null && managedFunds.INCDTLS427 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000245",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Managed Fund Distributions Total amounts and Managed Fund Distributions Your Share amounts must be provided",
                    LongDescription = @"Where a Managed Fund Distributions instance is present, at least one Managed Fund Distributions Total or Managed Fund Distributions Your Share amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000245"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS390", Value = GetValueOrEmpty(managedFunds.INCDTLS390) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS391", Value = GetValueOrEmpty(managedFunds.INCDTLS391) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS392", Value = GetValueOrEmpty(managedFunds.INCDTLS392) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS393", Value = GetValueOrEmpty(managedFunds.INCDTLS393) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS398", Value = GetValueOrEmpty(managedFunds.INCDTLS398) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS400", Value = GetValueOrEmpty(managedFunds.INCDTLS400) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS401", Value = GetValueOrEmpty(managedFunds.INCDTLS401) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS402", Value = GetValueOrEmpty(managedFunds.INCDTLS402) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS403", Value = GetValueOrEmpty(managedFunds.INCDTLS403) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS404", Value = GetValueOrEmpty(managedFunds.INCDTLS404) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS405", Value = GetValueOrEmpty(managedFunds.INCDTLS405) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS406", Value = GetValueOrEmpty(managedFunds.INCDTLS406) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS407", Value = GetValueOrEmpty(managedFunds.INCDTLS407) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS408", Value = GetValueOrEmpty(managedFunds.INCDTLS408) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS409", Value = GetValueOrEmpty(managedFunds.INCDTLS409) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS410", Value = GetValueOrEmpty(managedFunds.INCDTLS410) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS411", Value = GetValueOrEmpty(managedFunds.INCDTLS411) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS412", Value = GetValueOrEmpty(managedFunds.INCDTLS412) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS414", Value = GetValueOrEmpty(managedFunds.INCDTLS414) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS415", Value = GetValueOrEmpty(managedFunds.INCDTLS415) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS416", Value = GetValueOrEmpty(managedFunds.INCDTLS416) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS417", Value = GetValueOrEmpty(managedFunds.INCDTLS417) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS394", Value = GetValueOrEmpty(managedFunds.INCDTLS394) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS395", Value = GetValueOrEmpty(managedFunds.INCDTLS395) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS396", Value = GetValueOrEmpty(managedFunds.INCDTLS396) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS397", Value = GetValueOrEmpty(managedFunds.INCDTLS397) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS421", Value = GetValueOrEmpty(managedFunds.INCDTLS421) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS422", Value = GetValueOrEmpty(managedFunds.INCDTLS422) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS423", Value = GetValueOrEmpty(managedFunds.INCDTLS423) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS428", Value = GetValueOrEmpty(managedFunds.INCDTLS428) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS430", Value = GetValueOrEmpty(managedFunds.INCDTLS430) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS431", Value = GetValueOrEmpty(managedFunds.INCDTLS431) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS433", Value = GetValueOrEmpty(managedFunds.INCDTLS433) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS434", Value = GetValueOrEmpty(managedFunds.INCDTLS434) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS435", Value = GetValueOrEmpty(managedFunds.INCDTLS435) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS436", Value = GetValueOrEmpty(managedFunds.INCDTLS436) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS437", Value = GetValueOrEmpty(managedFunds.INCDTLS437) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS438", Value = GetValueOrEmpty(managedFunds.INCDTLS438) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS439", Value = GetValueOrEmpty(managedFunds.INCDTLS439) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS440", Value = GetValueOrEmpty(managedFunds.INCDTLS440) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS442", Value = GetValueOrEmpty(managedFunds.INCDTLS442) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS443", Value = GetValueOrEmpty(managedFunds.INCDTLS443) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS444", Value = GetValueOrEmpty(managedFunds.INCDTLS444) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS445", Value = GetValueOrEmpty(managedFunds.INCDTLS445) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS424", Value = GetValueOrEmpty(managedFunds.INCDTLS424) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS426", Value = GetValueOrEmpty(managedFunds.INCDTLS426) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS427", Value = GetValueOrEmpty(managedFunds.INCDTLS427) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000246
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000246(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000246
                    Where Managed fund number of account holders is greater than 1, Managed Fund Distributions Your Share instance must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS387 > 1 AND ^INCDTLS418 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS387 > 1 AND ^INCDTLS418 = NULL
            
                    Data Elements:
            
                    ^INCDTLS418 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.Rp_ManagedFunds_ManagedFundsYourShareCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000246",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Managed Fund Distributions Your Share must be provided",
                    LongDescription = @"Where Managed fund number of account holders is greater than 1, Managed Fund Distributions Your Share instance must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000246"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000247
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000247(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000247
                    Where a Managed Fund Distributions Your Share instance is present, at least one Managed Fund Distributions Your Share amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS418 <> NULL AND ^INCDTLS419 = NULL AND ^INCDTLS420 = NULL AND ^INCDTLS421 = NULL AND ^INCDTLS422 = NULL AND ^INCDTLS423 = NULL AND ^INCDTLS428 = NULL AND ^INCDTLS429 = NULL AND ^INCDTLS430 = NULL AND ^INCDTLS431 = NULL AND ^INCDTLS432 = NULL AND ^INCDTLS433 = NULL AND ^INCDTLS434 = NULL AND ^INCDTLS435 = NULL AND ^INCDTLS436 = NULL AND ^INCDTLS437 = NULL AND ^INCDTLS438 = NULL AND ^INCDTLS439 = NULL AND ^INCDTLS440 = NULL AND ^INCDTLS441 = NULL AND ^INCDTLS442 = NULL AND ^INCDTLS443 = NULL AND ^INCDTLS444 = NULL AND ^INCDTLS445 = NULL AND ^INCDTLS424 = NULL AND ^INCDTLS425 = NULL AND ^INCDTLS426 = NULL AND ^INCDTLS427 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS418 <> NULL AND ^INCDTLS419 = NULL AND ^INCDTLS420 = NULL AND ^INCDTLS421 = NULL AND ^INCDTLS422 = NULL AND ^INCDTLS423 = NULL AND ^INCDTLS428 = NULL AND ^INCDTLS429 = NULL AND ^INCDTLS430 = NULL AND ^INCDTLS431 = NULL AND ^INCDTLS432 = NULL AND ^INCDTLS433 = NULL AND ^INCDTLS434 = NULL AND ^INCDTLS435 = NULL AND ^INCDTLS436 = NULL AND ^INCDTLS437 = NULL AND ^INCDTLS438 = NULL AND ^INCDTLS439 = NULL AND ^INCDTLS440 = NULL AND ^INCDTLS441 = NULL AND ^INCDTLS442 = NULL AND ^INCDTLS443 = NULL AND ^INCDTLS444 = NULL AND ^INCDTLS445 = NULL AND ^INCDTLS424 = NULL AND ^INCDTLS425 = NULL AND ^INCDTLS426 = NULL AND ^INCDTLS427 = NULL
            
                    Data Elements:
            
                    ^INCDTLS418 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS421 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount
            
                    ^INCDTLS422 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS423 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS424 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Net.Amount
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS426 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.DiscountedMethod.Amount
            
                    ^INCDTLS427 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.OtherMethod.Amount
            
                    ^INCDTLS428 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS430 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.TaxOffset.Amount
            
                    ^INCDTLS431 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.IncomeCategoryMiscellaneousTotal.Amount
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS433 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount
            
                    ^INCDTLS434 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS435 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS436 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS437 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            
                    ^INCDTLS438 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            
                    ^INCDTLS439 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS440 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS442 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsNewZealand.Amount
            
                    ^INCDTLS443 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
            
                    ^INCDTLS444 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.EarlyStageInvestorTaxOffset.Amount
            
                    ^INCDTLS445 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (managedFunds.Rp_ManagedFunds_ManagedFundsYourShareCollectionExists != false && managedFunds.INCDTLS419 == null && managedFunds.INCDTLS420 == null && managedFunds.INCDTLS421 == null && managedFunds.INCDTLS422 == null && managedFunds.INCDTLS423 == null && managedFunds.INCDTLS428 == null && managedFunds.INCDTLS429 == null && managedFunds.INCDTLS430 == null && managedFunds.INCDTLS431 == null && managedFunds.INCDTLS432 == null && managedFunds.INCDTLS433 == null && managedFunds.INCDTLS434 == null && managedFunds.INCDTLS435 == null && managedFunds.INCDTLS436 == null && managedFunds.INCDTLS437 == null && managedFunds.INCDTLS438 == null && managedFunds.INCDTLS439 == null && managedFunds.INCDTLS440 == null && managedFunds.INCDTLS441 == null && managedFunds.INCDTLS442 == null && managedFunds.INCDTLS443 == null && managedFunds.INCDTLS444 == null && managedFunds.INCDTLS445 == null && managedFunds.INCDTLS424 == null && managedFunds.INCDTLS425 == null && managedFunds.INCDTLS426 == null && managedFunds.INCDTLS427 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000247",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one Managed Fund Distributions Your Share amount must be provided",
                    LongDescription = @"Where a Managed Fund Distributions Your Share instance is present, at least one Managed Fund Distributions Your Share amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000247"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS421", Value = GetValueOrEmpty(managedFunds.INCDTLS421) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS422", Value = GetValueOrEmpty(managedFunds.INCDTLS422) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS423", Value = GetValueOrEmpty(managedFunds.INCDTLS423) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS428", Value = GetValueOrEmpty(managedFunds.INCDTLS428) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS430", Value = GetValueOrEmpty(managedFunds.INCDTLS430) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS431", Value = GetValueOrEmpty(managedFunds.INCDTLS431) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS433", Value = GetValueOrEmpty(managedFunds.INCDTLS433) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS434", Value = GetValueOrEmpty(managedFunds.INCDTLS434) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS435", Value = GetValueOrEmpty(managedFunds.INCDTLS435) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS436", Value = GetValueOrEmpty(managedFunds.INCDTLS436) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS437", Value = GetValueOrEmpty(managedFunds.INCDTLS437) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS438", Value = GetValueOrEmpty(managedFunds.INCDTLS438) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS439", Value = GetValueOrEmpty(managedFunds.INCDTLS439) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS440", Value = GetValueOrEmpty(managedFunds.INCDTLS440) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS442", Value = GetValueOrEmpty(managedFunds.INCDTLS442) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS443", Value = GetValueOrEmpty(managedFunds.INCDTLS443) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS444", Value = GetValueOrEmpty(managedFunds.INCDTLS444) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS445", Value = GetValueOrEmpty(managedFunds.INCDTLS445) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS424", Value = GetValueOrEmpty(managedFunds.INCDTLS424) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS426", Value = GetValueOrEmpty(managedFunds.INCDTLS426) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS427", Value = GetValueOrEmpty(managedFunds.INCDTLS427) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000248
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000248(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000248
                    When the number of Managed fund number of account holders is greater than 1, and Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, Your share of franked distribution from managed fund, and Your share of current year capital gains from managed fund must sum to a positive value
    
                    Legacy Rule Format:
                    (^INCDTLS387 > 1 AND ^INCDTLS436 > 0 AND ((^INCDTLS432 + ^INCDTLS419 + ^INCDTLS420 + ^INCDTLS425) <= 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS387 > 1 AND ^INCDTLS436 > 0 AND ((^INCDTLS432 + ^INCDTLS419 + ^INCDTLS420 + ^INCDTLS425) <= 0))
            
                    Data Elements:
            
                    ^INCDTLS436 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS436.GetValueOrDefault() > 0 && managedFunds.INCDTLS432.GetValueOrDefault() + managedFunds.INCDTLS419.GetValueOrDefault() + managedFunds.INCDTLS420.GetValueOrDefault() + managedFunds.INCDTLS425.GetValueOrDefault() <= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000248",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The specified amounts must sum to a positive value",
                    LongDescription = @"When the number of Managed fund number of account holders is greater than 1, and Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, Your share of franked distribution from managed fund, and Your share of current year capital gains from managed fund must sum to a positive value",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000248"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS436", Value = GetValueOrEmpty(managedFunds.INCDTLS436) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000249
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000249(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000249
                    When the number of Managed fund number of account holders is greater than 1, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, Your share of franked distribution from managed fund, Your share of current year capital gains from managed fund, Your share of net foreign rent from managed fund, Your share of Australian credits from a NZ franking company from managed fund, and Your share of other foreign income from managed fund must sum to zero or a positive value
    
                    Legacy Rule Format:
                    (^INCDTLS387 > 1 AND (^INCDTLS432 + ^INCDTLS419 + ^INCDTLS420 + ^INCDTLS425 + ^INCDTLS441 + ^INCDTLS442 + ^INCDTLS429) < 0)
        
                    Technical Business Rule Format:
                    (^INCDTLS387 > 1 AND (^INCDTLS432 + ^INCDTLS419 + ^INCDTLS420 + ^INCDTLS425 + ^INCDTLS441 + ^INCDTLS442 + ^INCDTLS429) < 0)
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS442 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsNewZealand.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS432.GetValueOrDefault() + managedFunds.INCDTLS419.GetValueOrDefault() + managedFunds.INCDTLS420.GetValueOrDefault() + managedFunds.INCDTLS425.GetValueOrDefault() + managedFunds.INCDTLS441.GetValueOrDefault() + managedFunds.INCDTLS442.GetValueOrDefault() + managedFunds.INCDTLS429.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000249",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The specified amounts must sum to a positive value or zero",
                    LongDescription = @"When the number of Managed fund number of account holders is greater than 1, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, Your share of franked distribution from managed fund, Your share of current year capital gains from managed fund, Your share of net foreign rent from managed fund, Your share of Australian credits from a NZ franking company from managed fund, and Your share of other foreign income from managed fund must sum to zero or a positive value",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000249"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS442", Value = GetValueOrEmpty(managedFunds.INCDTLS442) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000250
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000250(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000250
                    When the number of Managed fund number of account holders is greater than 1, and Your share of franking credits from managed fund, Your share of TFN amounts withheld from managed fund, Your share of tax withheld when ABN not quoted from managed fund, or Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, or Your share of franked distribution from managed fund must be provided
    
                    Legacy Rule Format:
                    ((^INCDTLS387 > 1) AND (^INCDTLS432 = NULL AND ^INCDTLS419 = NULL AND ^INCDTLS420 = NULL) AND (^INCDTLS422 > 0 OR ^INCDTLS423 > 0 OR ^INCDTLS434 > 0 OR ^INCDTLS436 > 0))
        
                    Technical Business Rule Format:
                    ((^INCDTLS387 > 1) AND (^INCDTLS432 = NULL AND ^INCDTLS419 = NULL AND ^INCDTLS420 = NULL) AND (^INCDTLS422 > 0 OR ^INCDTLS423 > 0 OR ^INCDTLS434 > 0 OR ^INCDTLS436 > 0))
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS422 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS423 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^INCDTLS434 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS436 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() > 1 && (managedFunds.INCDTLS432 == null && managedFunds.INCDTLS419 == null && managedFunds.INCDTLS420 == null) && (managedFunds.INCDTLS422.GetValueOrDefault() > 0 || managedFunds.INCDTLS423.GetValueOrDefault() > 0 || managedFunds.INCDTLS434.GetValueOrDefault() > 0 || managedFunds.INCDTLS436.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000250",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your shares of primary production income from managed fund, non-primary production income from managed fund, or franked distribution from managed fund must be provided",
                    LongDescription = @"When the number of Managed fund number of account holders is greater than 1, and Your share of franking credits from managed fund, Your share of TFN amounts withheld from managed fund, Your share of tax withheld when ABN not quoted from managed fund, or Your share of credit for foreign resident withholding amounts (excluding capital gains) from managed fund is positive, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, or Your share of franked distribution from managed fund must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000250"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS422", Value = GetValueOrEmpty(managedFunds.INCDTLS422) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS423", Value = GetValueOrEmpty(managedFunds.INCDTLS423) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS434", Value = GetValueOrEmpty(managedFunds.INCDTLS434) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS436", Value = GetValueOrEmpty(managedFunds.INCDTLS436) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000251
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000251(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000251
                    When the number of Managed fund number of account holders is greater than 1, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, Your share of franked distribution from managed fund, Your share of current year capital gains from managed fund, Your share of transferor trust income from managed fund, Your share of CFC income from managed fund, Your share of net foreign rent from managed fund, and Your share of other foreign income from managed fund must sum to zero or a positive value
    
                    Legacy Rule Format:
                    (^INCDTLS387 > 1 AND (^INCDTLS432 + ^INCDTLS419 + ^INCDTLS420 + ^INCDTLS425 + ^INCDTLS439 + ^INCDTLS440 + ^INCDTLS441 + ^INCDTLS429 < 0))
        
                    Technical Business Rule Format:
                    (^INCDTLS387 > 1 AND (^INCDTLS432 + ^INCDTLS419 + ^INCDTLS420 + ^INCDTLS425 + ^INCDTLS439 + ^INCDTLS440 + ^INCDTLS441 + ^INCDTLS429 < 0))
            
                    Data Elements:
            
                    ^INCDTLS432 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS420 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.TrustFrankedNonPrimaryProductionShare.Amount
            
                    ^INCDTLS425 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:YourShareCapitalGains:Capital.Gains.Total.Amount
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS439 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            
                    ^INCDTLS440 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.ControlledForeignCompaniesAttributableIncome.Amount
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS387.GetValueOrDefault() > 1 && managedFunds.INCDTLS432.GetValueOrDefault() + managedFunds.INCDTLS419.GetValueOrDefault() + managedFunds.INCDTLS420.GetValueOrDefault() + managedFunds.INCDTLS425.GetValueOrDefault() + managedFunds.INCDTLS439.GetValueOrDefault() + managedFunds.INCDTLS440.GetValueOrDefault() + managedFunds.INCDTLS441.GetValueOrDefault() + managedFunds.INCDTLS429.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000251",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The specified amounts must sum to a positive value or zero",
                    LongDescription = @"When the number of Managed fund number of account holders is greater than 1, Your share of primary production income from managed fund, Your share of non-primary production income from managed fund, Your share of franked distribution from managed fund, Your share of current year capital gains from managed fund, Your share of transferor trust income from managed fund, Your share of CFC income from managed fund, Your share of net foreign rent from managed fund, and Your share of other foreign income from managed fund must sum to zero or a positive value",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeTrustPrimaryProductionNetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000251"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS432", Value = GetValueOrEmpty(managedFunds.INCDTLS432) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS420", Value = GetValueOrEmpty(managedFunds.INCDTLS420) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS425", Value = GetValueOrEmpty(managedFunds.INCDTLS425) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS439", Value = GetValueOrEmpty(managedFunds.INCDTLS439) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS440", Value = GetValueOrEmpty(managedFunds.INCDTLS440) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000252
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000252(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000252
                    When the Total non-primary production income from managed fund amount provided is negative, then Your share of non-primary production income from managed fund must not be positive or less than Total non-primary production income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS389 < 0 AND (^INCDTLS419 > 0 OR ^INCDTLS419 < ^INCDTLS389))
        
                    Technical Business Rule Format:
                    (^INCDTLS389 < 0 AND (^INCDTLS419 > 0 OR ^INCDTLS419 < ^INCDTLS389))
            
                    Data Elements:
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
                    */
            assertion = (managedFunds.INCDTLS389.GetValueOrDefault() < 0 && (managedFunds.INCDTLS419.GetValueOrDefault() > 0 || managedFunds.INCDTLS419.GetValueOrDefault() < managedFunds.INCDTLS389.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000252",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of non-primary production income from managed fund is incorrect",
                    LongDescription = @"When the Total non-primary production income from managed fund amount provided is negative, then Your share of non-primary production income from managed fund must not be positive or less than Total non-primary production income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeNonPrimaryProductionNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000252"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000253
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000253(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000253
                    Where Total non-primary production income from managed fund is positive, Your share of non-primary production income from managed fund must not be greater than Total non-primary production income from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS419 > ^INCDTLS389 AND ^INCDTLS389 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS419 > ^INCDTLS389 AND ^INCDTLS389 >= 0
            
                    Data Elements:
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
                    */
            assertion = (managedFunds.INCDTLS419.GetValueOrDefault() > managedFunds.INCDTLS389.GetValueOrDefault() && managedFunds.INCDTLS389.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000253",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of non-primary production income from managed fund is incorrect",
                    LongDescription = @"Where Total non-primary production income from managed fund is positive, Your share of non-primary production income from managed fund must not be greater than Total non-primary production income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeNonPrimaryProductionNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000253"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000254
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000254(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000254
                    Your share of non-primary production income from managed fund must not be negative where Total non-primary production income from managed fund is greater than or equal to 0
    
                    Legacy Rule Format:
                    (^INCDTLS419 < 0 AND ^INCDTLS389 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS419 < 0 AND ^INCDTLS389 >= 0
            
                    Data Elements:
            
                    ^INCDTLS419 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.NonPrimaryProductionNet.Amount
            
                    ^INCDTLS389 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.NonPrimaryProductionNet.Amount
                    */
            assertion = (managedFunds.INCDTLS419.GetValueOrDefault() < 0 && managedFunds.INCDTLS389.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000254",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of non-primary production income from managed fund is incorrect",
                    LongDescription = @"Your share of non-primary production income from managed fund must not be negative where Total non-primary production income from managed fund is greater than or equal to 0",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeNonPrimaryProductionNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000254"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS419", Value = GetValueOrEmpty(managedFunds.INCDTLS419) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS389", Value = GetValueOrEmpty(managedFunds.INCDTLS389) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000255(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000255
                    Where Reason the trustee paid tax from managed fund or Share of income from managed fund is provided, the Total credit for tax paid by trustee from managed fund must also be provided
    
                    Legacy Rule Format:
                    ((^INCDTLS405 = NULL OR ^INCDTLS405 = 0) AND (^INCDTLS406 <> NULL OR ^INCDTLS407 <> NULL))
        
                    Technical Business Rule Format:
                    ((^INCDTLS405 = NULL OR ^INCDTLS405 = 0) AND (^INCDTLS406 <> NULL OR ^INCDTLS407 <> NULL))
            
                    Data Elements:
            
                    ^INCDTLS405 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS406 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidReason.Code
            
                    ^INCDTLS407 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustDistributions.Amount
                    */
            assertion = ((managedFunds.INCDTLS405 == null || managedFunds.INCDTLS405 == 0) && (managedFunds.INCDTLS406 != null || managedFunds.INCDTLS407 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total credit for tax paid by trustee from managed fund must be provided",
                    LongDescription = @"Where Reason the trustee paid tax from managed fund or Share of income from managed fund is provided, the Total credit for tax paid by trustee from managed fund must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTaxCreditTrusteeTaxPaidTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000255"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS405", Value = GetValueOrEmpty(managedFunds.INCDTLS405) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS406", Value = GetValueOrEmpty(managedFunds.INCDTLS406) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS407", Value = GetValueOrEmpty(managedFunds.INCDTLS407) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000256
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000256(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000256
                    Where Total credit for tax paid by trustee from managed fund is provided, Reason the trustee paid tax from managed fund and Share of income from managed fund must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS405 > 0 AND (^INCDTLS406 = NULL OR ^INCDTLS407 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS405 > 0 AND (^INCDTLS406 = NULL OR ^INCDTLS407 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS407 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.TrustDistributions.Amount
            
                    ^INCDTLS405 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS406 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:IncomeTax.CreditTrusteeTaxPaidReason.Code
                    */
            assertion = (managedFunds.INCDTLS405.GetValueOrDefault() > 0 && (managedFunds.INCDTLS406 == null || managedFunds.INCDTLS407 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000256",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reason the trustee paid tax from managed fund and Share of income from managed fund must be provided",
                    LongDescription = @"Where Total credit for tax paid by trustee from managed fund is provided, Reason the trustee paid tax from managed fund and Share of income from managed fund must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeTrustDistributionsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000256"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS405", Value = GetValueOrEmpty(managedFunds.INCDTLS405) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS406", Value = GetValueOrEmpty(managedFunds.INCDTLS406) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS407", Value = GetValueOrEmpty(managedFunds.INCDTLS407) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000257
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000257(INCDTLS2026.Rp_IncomeOther incomeOther, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000257
                    Where Other income type is "Assessable balancing adjustment", the sum of Assessable balancing adjustment from low value pool relating to financial investments, Assessable balancing adjustment from low value pool relating to rental property, and Remaining assessable balancing adjustment must be equal to the Other income amount
    
                    Legacy Rule Format:
                    (^INCDTLS456 <> NULL) AND (^INCDTLS451 <> (^INCDTLS457 + ^INCDTLS458 + ^INCDTLS459))
        
                    Technical Business Rule Format:
                    (^INCDTLS456 <> NULL) AND (^INCDTLS451 <> (^INCDTLS457 + ^INCDTLS458 + ^INCDTLS459))
            
                    Data Elements:
            
                    ^INCDTLS451 = INCDTLS:Rp:IncomeOther:Income.Other.Amount
            
                    ^INCDTLS456 = INCDTLS:Rp:IncomeOther:IncomeTaxAssessableLowValuePoolDeduction
            
                    ^INCDTLS457 = INCDTLS:Rp:IncomeOther:IncomeTaxAssessableLowValuePoolDeduction:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount
            
                    ^INCDTLS458 = INCDTLS:Rp:IncomeOther:IncomeTaxAssessableLowValuePoolDeduction:Income.DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotal.Amount
            
                    ^INCDTLS459 = INCDTLS:Rp:IncomeOther:IncomeTaxAssessableLowValuePoolDeduction:Income.BalancingAdjustmentTotal.Amount
                    */
            assertion = (incomeOther.Rp_IncomeOther_IncomeTaxAssessableLowValuePoolDeductionCollectionExists != false && incomeOther.INCDTLS451.GetValueOrDefault() != incomeOther.INCDTLS457.GetValueOrDefault() + incomeOther.INCDTLS458.GetValueOrDefault() + incomeOther.INCDTLS459.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000257",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income amount is incorrect",
                    LongDescription = @"Where Other income type is ""Assessable balancing adjustment"", the sum of Assessable balancing adjustment from low value pool relating to financial investments, Assessable balancing adjustment from low value pool relating to rental property, and Remaining assessable balancing adjustment must be equal to the Other income amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther" + OccurrenceIndex(incomeOther.OccurrenceIndex) + "/tns:A",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000257"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS451", Value = GetValueOrEmpty(incomeOther.INCDTLS451) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS457", Value = GetValueOrEmpty(incomeOther.INCDTLS457) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS458", Value = GetValueOrEmpty(incomeOther.INCDTLS458) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS459", Value = GetValueOrEmpty(incomeOther.INCDTLS459) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000258
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000258(INCDTLS2026.Rp_IncomeOther incomeOther, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000258
                    Where Other income type is "Assessable balancing adjustment" and Other income amount field is greater than zero, then Assessable balancing adjustment instance must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS449 = "Assessable balancing adjustment") AND (^INCDTLS451 > 0) AND (^INCDTLS456 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS449 = 'Assessable balancing adjustment') AND (^INCDTLS451 > 0) AND (^INCDTLS456 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS451 = INCDTLS:Rp:IncomeOther:Income.Other.Amount
            
                    ^INCDTLS449 = INCDTLS:Rp:IncomeOther:Income.OtherIncomeType.Code
            
                    ^INCDTLS456 = INCDTLS:Rp:IncomeOther:IncomeTaxAssessableLowValuePoolDeduction
                    */
            assertion = (incomeOther.INCDTLS449 == @"Assessable balancing adjustment" && incomeOther.INCDTLS451.GetValueOrDefault() > 0 && incomeOther.Rp_IncomeOther_IncomeTaxAssessableLowValuePoolDeductionCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000258",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessable balancing adjustment instance must be provided",
                    LongDescription = @"Where Other income type is ""Assessable balancing adjustment"" and Other income amount field is greater than zero, then Assessable balancing adjustment instance must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther" + OccurrenceIndex(incomeOther.OccurrenceIndex) + "/tns:A",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000258"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS449", Value = incomeOther.INCDTLS449 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS451", Value = GetValueOrEmpty(incomeOther.INCDTLS451) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000259(INCDTLS2026.Rp_IncomeOther incomeOther, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000259
                    When Other income type is not "Assessable balancing adjustment" the Assessable balancing adjustment instance must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS449 <> "Assessable balancing adjustment") AND (^INCDTLS456 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS449 <> 'Assessable balancing adjustment') AND (^INCDTLS456 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS449 = INCDTLS:Rp:IncomeOther:Income.OtherIncomeType.Code
            
                    ^INCDTLS456 = INCDTLS:Rp:IncomeOther:IncomeTaxAssessableLowValuePoolDeduction
                    */
            assertion = (incomeOther.INCDTLS449 != @"Assessable balancing adjustment" && incomeOther.Rp_IncomeOther_IncomeTaxAssessableLowValuePoolDeductionCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000259",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessable balancing adjustment instance must not be provided",
                    LongDescription = @"When Other income type is not ""Assessable balancing adjustment"" the Assessable balancing adjustment instance must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther" + OccurrenceIndex(incomeOther.OccurrenceIndex) + "/tns:OtherIncomeTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000259"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS449", Value = incomeOther.INCDTLS449 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000260
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000260()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000260
            Only one instance of "Special professional income" type can be provided
    
            Legacy Rule Format:
            (CountOccurrence(^INCDTLS449, ^INCDTLS449 = "Special professional income") > 1)

            Technical Business Rule Format:
            (CountOccurrence(^INCDTLS449, ^INCDTLS449 = 'Special professional income') > 1)
    
            Data Elements:
    
            ^INCDTLS449 = INCDTLS:Rp:IncomeOther:Income.OtherIncomeType.Code
            */
            assertion = ((report.Rp_IncomeOtherCollection == null ? 0 : report.Rp_IncomeOtherCollection.Count(INCDTLS449Repeat => INCDTLS449Repeat.INCDTLS449 == @"Special professional income")) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000260",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Only one instance of ""Special professional income"" type can be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther/tns:OtherIncomeTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000260"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000261
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000261(INCDTLS2026.Rp_IncomeOther incomeOther, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000261
                    Where there is no instance of "Special professional income" type provided, Special professional additional information must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS449 <> "Special professional income" AND ^INCDTLS452 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS449 <> 'Special professional income' AND ^INCDTLS452 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS452 = INCDTLS:Rp:IncomeOther:SpecialProfessionalExpense
            
                    ^INCDTLS449 = INCDTLS:Rp:IncomeOther:Income.OtherIncomeType.Code
                    */
            assertion = (incomeOther.INCDTLS449 != @"Special professional income" && incomeOther.Rp_IncomeOther_SpecialProfessionalExpenseCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000261",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Special professional additional information must not be provided",
                    LongDescription = @"Where there is no instance of ""Special professional income"" type provided, Special professional additional information must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther" + OccurrenceIndex(incomeOther.OccurrenceIndex) + "/tns:SpecialProfessionalExpense",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000261"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS449", Value = incomeOther.INCDTLS449 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000262(INCDTLS2026.Rp_Partnerships partnerships, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000262
                    The Partnership's Australian business number has failed the ABN algorithm check
    
                    Legacy Rule Format:
                    (FailsABNAlgorithm(^INCDTLS462))
        
                    Technical Business Rule Format:
                    (FailsABNAlgorithm(^INCDTLS462))
            
                    Data Elements:
            
                    ^INCDTLS462 = INCDTLS:Rp:Partnerships:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = FailsABNAlgorithm(partnerships.INCDTLS462);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000262",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Partnership's Australian business number is invalid",
                    LongDescription = @"The Partnership's Australian business number has failed the ABN algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:PartnershipsCollection/tns:Partnerships" + OccurrenceIndex(partnerships.OccurrenceIndex) + "/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000262"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS462", Value = GetValueOrEmpty(partnerships.INCDTLS462) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000263
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000263(INCDTLS2026.Rp_Partnerships partnerships, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000263
                    Where a Partnership credits from income and tax offsets instance is provided, at least one credit or offset amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS479 <> NULL AND (^INCDTLS480 = NULL AND ^INCDTLS481 = NULL AND ^INCDTLS482 = NULL AND ^INCDTLS483 = NULL AND ^INCDTLS484 = NULL AND ^INCDTLS485 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS479 <> NULL AND (^INCDTLS480 = NULL AND ^INCDTLS481 = NULL AND ^INCDTLS482 = NULL AND ^INCDTLS483 = NULL AND ^INCDTLS484 = NULL AND ^INCDTLS485 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS479 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets
            
                    ^INCDTLS480 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS481 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS482 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
            
                    ^INCDTLS483 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount
            
                    ^INCDTLS484 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            
                    ^INCDTLS485 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
                    */
            assertion = (partnerships.Rp_Partnerships_PartnershipsCreditsIncomeTaxOffsetsCollectionExists != false && (partnerships.INCDTLS480 == null && partnerships.INCDTLS481 == null && partnerships.INCDTLS482 == null && partnerships.INCDTLS483 == null && partnerships.INCDTLS484 == null && partnerships.INCDTLS485 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000263",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where a Partnership credits from income and tax offsets instance is provided, at least one credit or offset amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:PartnershipsCollection/tns:Partnerships" + OccurrenceIndex(partnerships.OccurrenceIndex) + "/tns:PartnershipsCreditsIncomeTaxOffsets",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000263"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS480", Value = GetValueOrEmpty(partnerships.INCDTLS480) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS481", Value = GetValueOrEmpty(partnerships.INCDTLS481) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS482", Value = GetValueOrEmpty(partnerships.INCDTLS482) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS483", Value = GetValueOrEmpty(partnerships.INCDTLS483) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS484", Value = GetValueOrEmpty(partnerships.INCDTLS484) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS485", Value = GetValueOrEmpty(partnerships.INCDTLS485) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000264
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000264(INCDTLS2026.Rp_Partnerships partnerships, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000264
                    Where Tax withheld where Australian business number not quoted from partnership, TFN amounts withheld from payments from closely held trusts from partnership or Credit for foreign resident withholding amounts (excluding capital gains) from partnership are provided, the Partnership primary production instance or the Partnership non-primary production distribution instance must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS480 > 0 OR ^INCDTLS483 > 0 OR ^INCDTLS484 > 0) AND (^INCDTLS463 = NULL AND ^INCDTLS469 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS480 > 0 OR ^INCDTLS483 > 0 OR ^INCDTLS484 > 0) AND (^INCDTLS463 = NULL AND ^INCDTLS469 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS463 = INCDTLS:Rp:Partnerships:PartnershipsPrimary
            
                    ^INCDTLS469 = INCDTLS:Rp:Partnerships:PartnershipsNonPrimary:PartnershipsNonPrimaryDistribution
            
                    ^INCDTLS480 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS483 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount
            
                    ^INCDTLS484 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
                    */
            assertion = ((partnerships.INCDTLS480.GetValueOrDefault() > 0 || partnerships.INCDTLS483.GetValueOrDefault() > 0 || partnerships.INCDTLS484.GetValueOrDefault() > 0) && (partnerships.Rp_Partnerships_PartnershipsPrimaryCollectionExists == false && partnerships.Rp_Partnerships_PartnershipsNonPrimary_PartnershipsNonPrimaryDistributionCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000264",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Partnership primary or non-primary distribution amounts must be provided",
                    LongDescription = @"Where Tax withheld where Australian business number not quoted from partnership, TFN amounts withheld from payments from closely held trusts from partnership or Credit for foreign resident withholding amounts (excluding capital gains) from partnership are provided, the Partnership primary production instance or the Partnership non-primary production distribution instance must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:PartnershipsCollection/tns:Partnerships" + OccurrenceIndex(partnerships.OccurrenceIndex) + "/tns:PartnershipsPrimary",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000264"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS480", Value = GetValueOrEmpty(partnerships.INCDTLS480) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS483", Value = GetValueOrEmpty(partnerships.INCDTLS483) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS484", Value = GetValueOrEmpty(partnerships.INCDTLS484) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000265
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000265(INCDTLS2026.Rp_Partnerships partnerships, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000265
                    Where Franking credits from partnership or TFN amounts withheld from interest, dividends and unit trust distributions from partnership are provided, Partnership non-primary production distribution must also be provided
    
                    Legacy Rule Format:
                    ((^INCDTLS481 > 0 OR ^INCDTLS482 > 0) AND ^INCDTLS469 = NULL)
        
                    Technical Business Rule Format:
                    ((^INCDTLS481 > 0 OR ^INCDTLS482 > 0) AND ^INCDTLS469 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS469 = INCDTLS:Rp:Partnerships:PartnershipsNonPrimary:PartnershipsNonPrimaryDistribution
            
                    ^INCDTLS481 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS482 = INCDTLS:Rp:Partnerships:PartnershipsCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
                    */
            assertion = ((partnerships.INCDTLS481.GetValueOrDefault() > 0 || partnerships.INCDTLS482.GetValueOrDefault() > 0) && partnerships.Rp_Partnerships_PartnershipsNonPrimary_PartnershipsNonPrimaryDistributionCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000265",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Partnership non-primary production distribution must be provided",
                    LongDescription = @"Where Franking credits from partnership or TFN amounts withheld from interest, dividends and unit trust distributions from partnership are provided, Partnership non-primary production distribution must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:PartnershipsCollection/tns:Partnerships" + OccurrenceIndex(partnerships.OccurrenceIndex) + "/tns:PartnershipsNonPrimary/tns:PartnershipsNonPrimaryDistribution",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000265"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS481", Value = GetValueOrEmpty(partnerships.INCDTLS481) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS482", Value = GetValueOrEmpty(partnerships.INCDTLS482) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000266(INCDTLS2026.Rp_EmployeeShareScheme employeeShareScheme, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000266
                    The ESS payer's Australian business number has failed the ABN algorithm check
    
                    Legacy Rule Format:
                    (FailsABNAlgorithm(^INCDTLS496))
        
                    Technical Business Rule Format:
                    (FailsABNAlgorithm(^INCDTLS496))
            
                    Data Elements:
            
                    ^INCDTLS496 = INCDTLS:Rp:EmployeeShareScheme:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = FailsABNAlgorithm(employeeShareScheme.INCDTLS496);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000266",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer's Australian business number is invalid",
                    LongDescription = @"The Payer's Australian business number has failed the ABN algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:EmployeeShareSchemeCollection/tns:EmployeeShareScheme" + OccurrenceIndex(employeeShareScheme.OccurrenceIndex) + "/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000266"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS496", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS496) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000267(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000267
                    Where amounts are declared for a fund with 1 account holder, Managed Fund Distributions Your Share must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS418 <> NULL AND ^INCDTLS387 = 1)
        
                    Technical Business Rule Format:
                    ^INCDTLS418 <> NULL AND ^INCDTLS387 = 1
            
                    Data Elements:
            
                    ^INCDTLS418 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare
            
                    ^INCDTLS387 = INCDTLS:Rp:ManagedFunds:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (managedFunds.Rp_ManagedFunds_ManagedFundsYourShareCollectionExists != false && managedFunds.INCDTLS387.GetValueOrDefault() == 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000267",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where amounts are declared for a fund with 1 account holder, Managed Fund Distributions Your Share must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000267"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS387", Value = managedFunds.INCDTLS387.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000268
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000268(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000268
                    Where the Dividends number of account holders is greater than 1 and Total unfranked amount from dividends is provided, Your share of unfranked amount from dividends must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS378 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS371 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS378 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS371 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS378 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsUnfranked.Amount
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS371 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsUnfranked.Amount
                    */
            assertion = (dividends.INCDTLS378 == null && dividends.INCDTLS347.GetValueOrDefault() > 1 && dividends.INCDTLS371 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000268",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of unfranked amount from dividends must be provided",
                    LongDescription = @"Where the Dividends number of account holders is greater than 1 and Total unfranked amount from dividends is provided, Your share of unfranked amount from dividends must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeUnfrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000268"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS378", Value = GetValueOrEmpty(dividends.INCDTLS378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS371", Value = GetValueOrEmpty(dividends.INCDTLS371) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000269
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000269(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000269
                    Where the Dividends number of account holders is greater than 1 and Total franked amount from dividends is provided, Your share of franked amount from dividends must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS379 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS372 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS379 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS372 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS379 = INCDTLS:Rp:Dividends:DividendYourShare:Income.DividendsFranked.Amount
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS372 = INCDTLS:Rp:Dividends:DividendTotal:Income.DividendsFranked.Amount
                    */
            assertion = (dividends.INCDTLS379 == null && dividends.INCDTLS347.GetValueOrDefault() > 1 && dividends.INCDTLS372 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000269",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franked amount from dividends must be provided",
                    LongDescription = @"Where the Dividends number of account holders is greater than 1 and Total franked amount from dividends is provided, Your share of franked amount from dividends must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeFrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000269"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS379", Value = GetValueOrEmpty(dividends.INCDTLS379) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS372", Value = GetValueOrEmpty(dividends.INCDTLS372) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000270
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000270(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000270
                    Where the Dividends number of account holders is greater than 1 and Total franking credit from dividends is provided, Your share of franking credit from dividends must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS380 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS373 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS380 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS373 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS380 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.FrankingCredits.Amount
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS373 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.FrankingCredits.Amount
                    */
            assertion = (dividends.INCDTLS380 == null && dividends.INCDTLS347.GetValueOrDefault() > 1 && dividends.INCDTLS373 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000270",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of franking credit from dividends must be provided",
                    LongDescription = @"Where the Dividends number of account holders is greater than 1 and Total franking credit from dividends is provided, Your share of franking credit from dividends must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000270"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS380", Value = GetValueOrEmpty(dividends.INCDTLS380) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS373", Value = GetValueOrEmpty(dividends.INCDTLS373) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000271
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000271(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000271
                    Where the Dividends number of account holders is greater than 1 and Total TFN amounts withheld from dividends is provided, Your share of TFN amounts withheld from dividends must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS381 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS374 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS381 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS374 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS381 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS374 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount
                    */
            assertion = (dividends.INCDTLS381 == null && dividends.INCDTLS347.GetValueOrDefault() > 1 && dividends.INCDTLS374 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000271",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from dividends must be provided",
                    LongDescription = @"Where the Dividends number of account holders is greater than 1 and Total TFN amounts withheld from dividends is provided, Your share of TFN amounts withheld from dividends must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000271"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS381", Value = GetValueOrEmpty(dividends.INCDTLS381) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS374", Value = GetValueOrEmpty(dividends.INCDTLS374) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000272
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000272(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000272
                    Where the Dividends number of account holders is greater than 1 and Total exploration credits from dividends is provided, Your share of exploration credits from dividends must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS382 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS375 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS382 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS375 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS382 = INCDTLS:Rp:Dividends:DividendYourShare:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS375 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (dividends.INCDTLS382 == null && dividends.INCDTLS347.GetValueOrDefault() > 1 && dividends.INCDTLS375 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000272",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of exploration credits from dividends must be provided",
                    LongDescription = @"Where the Dividends number of account holders is greater than 1 and Total exploration credits from dividends is provided, Your share of exploration credits from dividends must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:IncomeTaxExplorationCreditsDistributedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000272"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS382", Value = GetValueOrEmpty(dividends.INCDTLS382) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS375", Value = GetValueOrEmpty(dividends.INCDTLS375) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000273(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000273
                    Where the Dividends number of account holders is greater than 1 and Total listed investment company capital gain deduction from dividends is provided, Your share of listed investment company capital gain deduction from dividends must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS383 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS376 <> NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS383 = NULL AND ^INCDTLS347 > 1 AND ^INCDTLS376 <> NULL
            
                    Data Elements:
            
                    ^INCDTLS383 = INCDTLS:Rp:Dividends:DividendYourShare:Capital.Gains.Total.Amount
            
                    ^INCDTLS347 = INCDTLS:Rp:Dividends:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS376 = INCDTLS:Rp:Dividends:DividendTotal:Capital.Gains.Total.Amount
                    */
            assertion = (dividends.INCDTLS383 == null && dividends.INCDTLS347.GetValueOrDefault() > 1 && dividends.INCDTLS376 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000273",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of listed investment company capital gain deduction from dividends must be provided",
                    LongDescription = @"Where the Dividends number of account holders is greater than 1 and Total listed investment company capital gain deduction from dividends is provided, Your share of listed investment company capital gain deduction from dividends must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:DividendYourShare/tns:CapitalGainsTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000273"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS383", Value = GetValueOrEmpty(dividends.INCDTLS383) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS347", Value = dividends.INCDTLS347.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS376", Value = GetValueOrEmpty(dividends.INCDTLS376) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000301
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000301(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000301
                    The ETP date of payment must be within the current financial year or approved substitute period
    
                    Legacy Rule Format:
                    (^INCDTLS137 <> NULL AND (^INCDTLS137 < ^INCDTLS100 OR ^INCDTLS137 > ^INCDTLS101))
        
                    Technical Business Rule Format:
                    (^INCDTLS137 <> NULL AND (^INCDTLS137 < ^INCDTLS100 OR ^INCDTLS137 > ^INCDTLS101))
            
                    Data Elements:
            
                    ^INCDTLS137 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:Income.EmploymentTerminationPayment.Date
            
                    ^INCDTLS100 = INCDTLS:Rp:Lodgment.PeriodStart.Date
            
                    ^INCDTLS101 = INCDTLS:Rp:Lodgment.PeriodEnd.Date
                    */
            assertion = (income.INCDTLS137 != null && (income.INCDTLS137.GetValueOrDefault() < report.INCDTLS100.GetValueOrDefault() || income.INCDTLS137.GetValueOrDefault() > report.INCDTLS101.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000301",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ETP date of payment must be within income year",
                    LongDescription = @"The ETP date of payment must be within the current financial year or approved substitute period",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:EmploymentTerminationPayment/tns:D",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000301"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS137", Value = GetValueOrEmptyDate(income.INCDTLS137) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS100", Value = GetValueOrEmptyDate(report.INCDTLS100) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS101", Value = GetValueOrEmptyDate(report.INCDTLS101) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000310
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000310(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000310
                    Where an Australian Employment termination payment summary instance is present, either the Payer's Australian business number or Payer's withholding payer number must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS135 <> NULL AND (InSet(^INCDTLS136, '"au","cc","cx","nf","hm"') OR ^INCDTLS136 = NULL) AND (^INCDTLS307 = NULL) AND (^INCDTLS308 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS135 <> NULL AND (InSet(^INCDTLS136, '"au","cc","cx","nf","hm"') OR ^INCDTLS136 = NULL) AND (^INCDTLS307 = NULL) AND (^INCDTLS308 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
            
                    ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
            
                    ^INCDTLS136 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:AddressDetails.Country.Code
            
                    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists != false && (IsMatch(income.INCDTLS136, @"^(au|cc|cx|nf|hm)$") || income.INCDTLS136 == null) && income.INCDTLS307 == null && income.INCDTLS308 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000310",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer's Australian business number or Payer's withholding payer number must be provided",
                    LongDescription = @"Where an Australian Employment termination payment summary instance is present, either the Payer's Australian business number or Payer's withholding payer number must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000310"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS136", Value = GetValueOrEmpty(income.INCDTLS136) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS307", Value = GetValueOrEmpty(income.INCDTLS307) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS308", Value = GetValueOrEmpty(income.INCDTLS308) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000313
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000313(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000313
                    Where the result of Annuity gross amount less Annuity deductible amount of undeducted purchase price is positive, the Annuity tax withheld amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS146 = NULL AND (^INCDTLS147 - ^INCDTLS148) > 0)
        
                    Technical Business Rule Format:
                    (^INCDTLS146 = NULL AND (^INCDTLS147 - ^INCDTLS148) > 0)
            
                    Data Elements:
            
                    ^INCDTLS146 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount
            
                    ^INCDTLS147 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:Income.Insurance.LifePremiumDirectGross.Amount
            
                    ^INCDTLS148 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.Deduction.PensionOrAnnuityPurchasePriceUndeducted.Amount
                    */
            assertion = (income.INCDTLS146 == null && income.INCDTLS147.GetValueOrDefault() - income.INCDTLS148.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000313",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Annuity tax withheld amount must be provided",
                    LongDescription = @"Where the result of Annuity gross amount less Annuity deductible amount of undeducted purchase price is positive, the Annuity tax withheld amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AustralianAnnuities/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000313"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS146", Value = GetValueOrEmpty(income.INCDTLS146) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS147", Value = GetValueOrEmpty(income.INCDTLS147) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS148", Value = GetValueOrEmpty(income.INCDTLS148) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000314
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000314(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000314
                    When Annuity deductible amount of undeducted purchase price amount is provided, it must not be greater than the Annuity gross amount
    
                    Legacy Rule Format:
                    (^INCDTLS148 > 0 AND (^INCDTLS147 = NULL OR ^INCDTLS148 > ^INCDTLS147))
        
                    Technical Business Rule Format:
                    (^INCDTLS148 > 0 AND (^INCDTLS147 = NULL OR ^INCDTLS148 > ^INCDTLS147))
            
                    Data Elements:
            
                    ^INCDTLS148 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.Deduction.PensionOrAnnuityPurchasePriceUndeducted.Amount
            
                    ^INCDTLS147 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:Income.Insurance.LifePremiumDirectGross.Amount
                    */
            assertion = (income.INCDTLS148.GetValueOrDefault() > 0 && (income.INCDTLS147 == null || income.INCDTLS148.GetValueOrDefault() > income.INCDTLS147.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Annuity deductible amount of undeducted purchase price is incorrect",
                    LongDescription = @"When Annuity deductible amount of undeducted purchase price amount is provided, it must not be greater than the Annuity gross amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AustralianAnnuities/tns:TaxDeductionPensionOrAnnuityPurchasePriceUndeductedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000314"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS148", Value = GetValueOrEmpty(income.INCDTLS148) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS147", Value = GetValueOrEmpty(income.INCDTLS147) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000315
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000315(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000315
                    Where an Australian annuities payment summary instance is present, at least one of the following amounts must be provided: Annuity tax withheld; Annuity gross amount; or Annuity deductible amount of undeducted purchase price
    
                    Legacy Rule Format:
                    (^INCDTLS145 <> NULL AND ^INCDTLS146 = NULL AND ^INCDTLS147 = NULL AND ^INCDTLS148 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS145 <> NULL AND ^INCDTLS146 = NULL AND ^INCDTLS147 = NULL AND ^INCDTLS148 = NULL
            
                    Data Elements:
            
                    ^INCDTLS145 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities
            
                    ^INCDTLS146 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount
            
                    ^INCDTLS147 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:Income.Insurance.LifePremiumDirectGross.Amount
            
                    ^INCDTLS148 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.Deduction.PensionOrAnnuityPurchasePriceUndeducted.Amount
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_AustralianAnnuitiesCollectionExists != false && income.INCDTLS146 == null && income.INCDTLS147 == null && income.INCDTLS148 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000315",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian annuities payment summary details are incomplete",
                    LongDescription = @"Where an Australian annuities payment summary instance is present, at least one of the following amounts must be provided: Annuity tax withheld; Annuity gross amount; or Annuity deductible amount of undeducted purchase price",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AustralianAnnuities",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000315"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS146", Value = GetValueOrEmpty(income.INCDTLS146) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS147", Value = GetValueOrEmpty(income.INCDTLS147) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS148", Value = GetValueOrEmpty(income.INCDTLS148) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000319
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000319(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000319
                    Where any Superannuation income stream lump sum in arrears payment instances are present for an Australian superannuation income stream payment summary, at least one of the following amounts must be provided: Superannuation income stream lump sum in arrears taxable component taxed element; Superannuation income stream lump sum in arrears taxable component untaxed element; or Superannuation income stream lump sum in arrears tax free component
    
                    Legacy Rule Format:
                    (^INCDTLS108 <> NULL AND Count(^INCDTLS525) > 0 AND ^INCDTLS117 = NULL AND ^INCDTLS118 = NULL AND ^INCDTLS119 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS108 <> NULL AND Count(^INCDTLS525) > 0 AND ^INCDTLS117 = NULL AND ^INCDTLS118 = NULL AND ^INCDTLS119 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS117 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
            
                    ^INCDTLS118 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
            
                    ^INCDTLS119 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.LumpSumArrearsTaxFree.Amount
            
                    ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuationCollectionExists != false && Count(income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionCount) > 0 && income.INCDTLS117 == null && income.INCDTLS118 == null && income.INCDTLS119 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000319",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Superannuation income stream lump sum in arrears amount details are incomplete",
                    LongDescription = @"Where any Superannuation income stream lump sum in arrears payment instances are present for an Australian superannuation income stream payment summary, at least one of the following amounts must be provided: Superannuation income stream lump sum in arrears taxable component taxed element; Superannuation income stream lump sum in arrears taxable component untaxed element; or Superannuation income stream lump sum in arrears tax free component",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:LumpSumArrearsTaxableComponentTaxedElementA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000319"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS117", Value = GetValueOrEmpty(income.INCDTLS117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS118", Value = GetValueOrEmpty(income.INCDTLS118) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS119", Value = GetValueOrEmpty(income.INCDTLS119) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000320
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000320(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000320
                    Total TFN amounts withheld from gross interest must not exceed 50% of the Total gross interest
    
                    Legacy Rule Format:
                    (^INCDTLS358 > (^INCDTLS359 * 0.5))
        
                    Technical Business Rule Format:
                    (^INCDTLS358 > (^INCDTLS359 * 0.5))
            
                    Data Elements:
            
                    ^INCDTLS358 = INCDTLS:Rp:Interest:InterestTotal:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            
                    ^INCDTLS359 = INCDTLS:Rp:Interest:InterestTotal:Income.Interest.Gross.Amount
                    */
            assertion = (interest.INCDTLS358.GetValueOrDefault() > interest.INCDTLS359.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000320",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total TFN amounts withheld from gross interest must not exceed 50% of the Total gross interest",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestTotal/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000320"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS358", Value = GetValueOrEmpty(interest.INCDTLS358) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS359", Value = GetValueOrEmpty(interest.INCDTLS359) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000322
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000322(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000322
                    Where a Salary or wages lump sum in arrears instance is present for a Salary or wages payment summary, with a Lump sum in arrears payment type code of "Salary or wage lump sum E", the Salary or wages lump sum E amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS309 <> NULL AND AnyOccurrence(^INCDTLS151, ^INCDTLS151 = "Salary or wage lump sum E") AND ^INCDTLS326 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS309 <> NULL AND AnyOccurrence(^INCDTLS151, ^INCDTLS151 = 'Salary or wage lump sum E') AND ^INCDTLS326 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS326 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.BackPaymentLumpSumE.Amount
            
                    ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code
            
                    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists != false && (income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection == null ? false : income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Any(INCDTLS151Repeat => INCDTLS151Repeat.INCDTLS151 == @"Salary or wage lump sum E")) && income.INCDTLS326 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000322",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Salary or wages lump sum E amount must be provided",
                    LongDescription = @"Where a Salary or wages lump sum in arrears instance is present for a Salary or wages payment summary, with a Lump sum in arrears payment type code of ""Salary or wage lump sum E"", the Salary or wages lump sum E amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:BackPaymentLumpSumEA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000322"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS151", Value = "INCDTLS151" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS326", Value = GetValueOrEmpty(income.INCDTLS326) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000324
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000324(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000324
                    Where a Salary or wages lump sum in arrears instance is present for a Salary or wages payment summary, with a Lump sum in arrears payment type code of "Exempt FEI PS lump sum in arrears", the Exempt foreign employment income lump sum in arrears amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS309 <> NULL AND AnyOccurrence(^INCDTLS151, ^INCDTLS151 = "Exempt FEI PS lump sum in arrears") AND ^INCDTLS338 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS309 <> NULL AND AnyOccurrence(^INCDTLS151, ^INCDTLS151 = 'Exempt FEI PS lump sum in arrears') AND ^INCDTLS338 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS338 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:ExemptForeignEmployment:Income.LumpSumArrearsPayment.Amount
            
                    ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code
            
                    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists != false && (income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection == null ? false : income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Any(INCDTLS151Repeat => INCDTLS151Repeat.INCDTLS151 == @"Exempt FEI PS lump sum in arrears")) && income.INCDTLS338 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000324",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt foreign employment income lump sum in arrears must be provided",
                    LongDescription = @"Where a Salary or wages lump sum in arrears instance is present for a Salary or wages payment summary, with a Lump sum in arrears payment type code of ""Exempt FEI PS lump sum in arrears"", the Exempt foreign employment income lump sum in arrears amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:ExemptForeignEmployment/tns:LumpSumArrearsPaymentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000324"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS151", Value = "INCDTLS151" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS338", Value = GetValueOrEmpty(income.INCDTLS338) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000325
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000325(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000325
                    Where interest is declared for a joint account, Interest your share must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS360 = NULL AND ^INCDTLS353 > 1)
        
                    Technical Business Rule Format:
                    ^INCDTLS360 = NULL AND ^INCDTLS353 > 1
            
                    Data Elements:
            
                    ^INCDTLS360 = INCDTLS:Rp:Interest:InterestYourShare
            
                    ^INCDTLS353 = INCDTLS:Rp:Interest:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (interest.Rp_Interest_InterestYourShareCollectionExists == false && interest.INCDTLS353.GetValueOrDefault() > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000325",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where interest is declared for a joint account, Interest your share must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000325"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS353", Value = interest.INCDTLS353.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000326
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000326(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000326
                    Where interest is declared for an individually owned account, Interest your share must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS360 <> NULL AND ^INCDTLS353 = 1)
        
                    Technical Business Rule Format:
                    ^INCDTLS360 <> NULL AND ^INCDTLS353 = 1
            
                    Data Elements:
            
                    ^INCDTLS360 = INCDTLS:Rp:Interest:InterestYourShare
            
                    ^INCDTLS353 = INCDTLS:Rp:Interest:RegulatoryDisclosures.AccountHolder.Count
                    */
            assertion = (interest.Rp_Interest_InterestYourShareCollectionExists != false && interest.INCDTLS353.GetValueOrDefault() == 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000326",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where interest is declared for an individually owned account, Interest your share must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestYourShare",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000326"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS353", Value = interest.INCDTLS353.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000327
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000327(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000327
                    Your share of TFN amounts withheld from gross interest must not exceed 50% of Your share of gross interest
    
                    Legacy Rule Format:
                    (^INCDTLS361 > (^INCDTLS362 * 0.5))
        
                    Technical Business Rule Format:
                    (^INCDTLS361 > (^INCDTLS362 * 0.5))
            
                    Data Elements:
            
                    ^INCDTLS361 = INCDTLS:Rp:Interest:InterestYourShare:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            
                    ^INCDTLS362 = INCDTLS:Rp:Interest:InterestYourShare:Income.Interest.Gross.Amount
                    */
            assertion = (interest.INCDTLS361.GetValueOrDefault() > interest.INCDTLS362.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000327",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from gross interest must not exceed 50% of Your share of gross interest",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestYourShare/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000327"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS361", Value = GetValueOrEmpty(interest.INCDTLS361) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS362", Value = GetValueOrEmpty(interest.INCDTLS362) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000328
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000328(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000328
                    Where a positive amount for Your share of TFN amounts withheld from gross interest is present, Total TFN amounts withheld from gross interest must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS361 > 0 AND ^INCDTLS358 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS361 > 0 AND ^INCDTLS358 = NULL
            
                    Data Elements:
            
                    ^INCDTLS358 = INCDTLS:Rp:Interest:InterestTotal:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            
                    ^INCDTLS361 = INCDTLS:Rp:Interest:InterestYourShare:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
                    */
            assertion = (interest.INCDTLS361.GetValueOrDefault() > 0 && interest.INCDTLS358 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000328",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total TFN amounts withheld from gross interest must be provided",
                    LongDescription = @"Where a positive amount for Your share of TFN amounts withheld from gross interest is present, Total TFN amounts withheld from gross interest must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestTotal/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000328"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS361", Value = GetValueOrEmpty(interest.INCDTLS361) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS358", Value = GetValueOrEmpty(interest.INCDTLS358) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000329
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000329(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000329
                    Where Total TFN amounts withheld from gross interest is present for a joint account, Your share of TFN amounts withheld from gross interest must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS358 > 0 AND ^INCDTLS353 > 1 AND ^INCDTLS361 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS358 > 0 AND ^INCDTLS353 > 1 AND ^INCDTLS361 = NULL
            
                    Data Elements:
            
                    ^INCDTLS361 = INCDTLS:Rp:Interest:InterestYourShare:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            
                    ^INCDTLS353 = INCDTLS:Rp:Interest:RegulatoryDisclosures.AccountHolder.Count
            
                    ^INCDTLS358 = INCDTLS:Rp:Interest:InterestTotal:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
                    */
            assertion = (interest.INCDTLS358.GetValueOrDefault() > 0 && interest.INCDTLS353.GetValueOrDefault() > 1 && interest.INCDTLS361 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000329",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from gross interest must be provided",
                    LongDescription = @"Where Total TFN amounts withheld from gross interest is present for a joint account, Your share of TFN amounts withheld from gross interest must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestYourShare/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000329"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS358", Value = GetValueOrEmpty(interest.INCDTLS358) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS353", Value = interest.INCDTLS353.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS361", Value = GetValueOrEmpty(interest.INCDTLS361) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000330
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000330(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000330
                    When Your share of TFN amounts withheld from gross interest is provided, the amount must not exceed Total TFN amounts withheld from gross interest
    
                    Legacy Rule Format:
                    (^INCDTLS361 > 0 AND (^INCDTLS361 > ^INCDTLS358))
        
                    Technical Business Rule Format:
                    (^INCDTLS361 > 0 AND (^INCDTLS361 > ^INCDTLS358))
            
                    Data Elements:
            
                    ^INCDTLS361 = INCDTLS:Rp:Interest:InterestYourShare:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            
                    ^INCDTLS358 = INCDTLS:Rp:Interest:InterestTotal:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
                    */
            assertion = (interest.INCDTLS361.GetValueOrDefault() > 0 && interest.INCDTLS361.GetValueOrDefault() > interest.INCDTLS358.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000330",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of TFN amounts withheld from gross interest is incorrect",
                    LongDescription = @"When Your share of TFN amounts withheld from gross interest is provided, the amount must not exceed Total TFN amounts withheld from gross interest",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestYourShare/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000330"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS361", Value = GetValueOrEmpty(interest.INCDTLS361) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS358", Value = GetValueOrEmpty(interest.INCDTLS358) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000331
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000331(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000331
                    When Your share of gross interest is provided, the amount must not exceed the Total gross interest amount
    
                    Legacy Rule Format:
                    (^INCDTLS362 > 0 AND (^INCDTLS362 > ^INCDTLS359))
        
                    Technical Business Rule Format:
                    (^INCDTLS362 > 0 AND (^INCDTLS362 > ^INCDTLS359))
            
                    Data Elements:
            
                    ^INCDTLS362 = INCDTLS:Rp:Interest:InterestYourShare:Income.Interest.Gross.Amount
            
                    ^INCDTLS359 = INCDTLS:Rp:Interest:InterestTotal:Income.Interest.Gross.Amount
                    */
            assertion = (interest.INCDTLS362.GetValueOrDefault() > 0 && interest.INCDTLS362.GetValueOrDefault() > interest.INCDTLS359.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000331",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of gross interest is incorrect",
                    LongDescription = @"When Your share of gross interest is provided, the amount must not exceed the Total gross interest amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:InterestYourShare/tns:IncomeGrossA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000331"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS362", Value = GetValueOrEmpty(interest.INCDTLS362) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS359", Value = GetValueOrEmpty(interest.INCDTLS359) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000332
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000332(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000332
                    Where a Payer's withholding payer number is present, either a Salary or wages payment summary or Employment termination payment (ETP) summary must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS308 <> NULL AND ^INCDTLS309 = NULL AND ^INCDTLS135 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS308 <> NULL AND ^INCDTLS309 = NULL AND ^INCDTLS135 = NULL
            
                    Data Elements:
            
                    ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
            
                    ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
            
                    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
                    */
            assertion = (income.INCDTLS308 != null && income.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists == false && income.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000332",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either a Salary or wages payment summary or Employment termination payment (ETP) summary must be provided",
                    LongDescription = @"Where a Payer's withholding payer number is present, either a Salary or wages payment summary or Employment termination payment (ETP) summary must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000332"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS308", Value = GetValueOrEmpty(income.INCDTLS308) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000343
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000343(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000343
                    When Foreign employment income tax foreign tax paid is provided, the amount must be less than the sum of Foreign employment income gross amount, Foreign employment income lump sum A, Foreign employment income lump sum D, Foreign employment income lump sum E
    
                    Legacy Rule Format:
                    (^INCDTLS169 > 0) AND (^INCDTLS169 >= (^INCDTLS168 + ^INCDTLS174 + ^INCDTLS176 + ^INCDTLS177))
        
                    Technical Business Rule Format:
                    (^INCDTLS169 > 0) AND (^INCDTLS169 >= (^INCDTLS168 + ^INCDTLS174 + ^INCDTLS176 + ^INCDTLS177))
            
                    Data Elements:
            
                    ^INCDTLS169 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.InternationalDealings.TaxOffset.Amount
            
                    ^INCDTLS168 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS174 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS176 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.GenuineRedundancyPaymentLumpSumD.Amount
            
                    ^INCDTLS177 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.BackPaymentLumpSumE.Amount
                    */
            assertion = (income.INCDTLS169.GetValueOrDefault() > 0 && income.INCDTLS169.GetValueOrDefault() >= income.INCDTLS168.GetValueOrDefault() + income.INCDTLS174.GetValueOrDefault() + income.INCDTLS176.GetValueOrDefault() + income.INCDTLS177.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.010343",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income foreign tax paid is incorrect",
                    LongDescription = @"When Foreign employment income tax foreign tax paid is provided, the amount must be less than the sum of Foreign employment income gross amount, Foreign employment income lump sum A, Foreign employment income lump sum D, Foreign employment income lump sum E",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:InternationalDealingsTaxOffsetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000343"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS169", Value = GetValueOrEmpty(income.INCDTLS169) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS168", Value = GetValueOrEmpty(income.INCDTLS168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS174", Value = GetValueOrEmpty(income.INCDTLS174) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS176", Value = GetValueOrEmpty(income.INCDTLS176) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS177", Value = GetValueOrEmpty(income.INCDTLS177) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000344
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000344(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000344
                    When an answer to the question "Is the employer exempt from FBT under section 57A of the FBTAA 1986?" is provided within a Foreign employment income payment summary instance, the Foreign employment income reportable fringe benefits amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS180 <> NULL) AND (^INCDTLS170 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS180 <> NULL) AND (^INCDTLS170 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS170 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.FringeBenefitsReportable.Amount
            
                    ^INCDTLS180 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FringeBenefitsTax.Exemption.Indicator
                    */
            assertion = (income.INCDTLS180 != null && income.INCDTLS170 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000344",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income reportable fringe benefits amount must be provided",
                    LongDescription = @"When an answer to the question ""Is the employer exempt from FBT under section 57A of the FBTAA 1986?"" is provided within a Foreign employment income payment summary instance, the Foreign employment income reportable fringe benefits amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:FringeBenefitsReportableA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000344"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS180", Value = GetValueOrEmpty(income.INCDTLS180) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS170", Value = GetValueOrEmpty(income.INCDTLS170) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000345
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000345(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000345
                    Where a positive Foreign employment income reportable fringe benefits amount is present, an answer to "Is the employer exempt from FBT under section 57A of the FBTAA 1986?" must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS170 > 0) AND (^INCDTLS180 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS170 > 0) AND (^INCDTLS180 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS180 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FringeBenefitsTax.Exemption.Indicator
            
                    ^INCDTLS170 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.FringeBenefitsReportable.Amount
                    */
            assertion = (income.INCDTLS170.GetValueOrDefault() > 0 && income.INCDTLS180 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000345",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"""Is the employer exempt from FBT under section 57A of the FBTAA 1986?"" must be provided",
                    LongDescription = @"Where a positive Reportable fringe benefits amount is present, an answer to ""Is the employer exempt from FBT under section 57A of the FBTAA 1986?"" must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:FringeBenefitsTaxExemptionI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000345"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS170", Value = GetValueOrEmpty(income.INCDTLS170) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS180", Value = GetValueOrEmpty(income.INCDTLS180) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000346
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000346(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000346
                    Foreign employment income lump sum A amount is required where Foreign employment income lump sum A payment type is present
    
                    Legacy Rule Format:
                    (^INCDTLS174 = NULL OR ^INCDTLS174 = 0) AND (^INCDTLS175 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS174 = NULL OR ^INCDTLS174 = 0) AND (^INCDTLS175 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS174 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS175 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Remuneration.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Code
                    */
            assertion = ((income.INCDTLS174 == null || income.INCDTLS174 == 0) && income.INCDTLS175 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000346",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income lump sum A amount must be provided",
                    LongDescription = @"Foreign employment income lump sum A amount is required where Foreign employment income lump sum A payment type is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumAA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000346"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS174", Value = GetValueOrEmpty(income.INCDTLS174) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS175", Value = GetValueOrEmpty(income.INCDTLS175) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000347
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000347(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000347
                    Foreign employment income lump sum A payment type is required where Foreign employment income lump sum A is present
    
                    Legacy Rule Format:
                    (^INCDTLS175 = NULL) AND (^INCDTLS174 > 0)
        
                    Technical Business Rule Format:
                    (^INCDTLS175 = NULL) AND (^INCDTLS174 > 0)
            
                    Data Elements:
            
                    ^INCDTLS175 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Remuneration.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Code
            
                    ^INCDTLS174 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
                    */
            assertion = (income.INCDTLS175 == null && income.INCDTLS174.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000347",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income lump sum A payment type must be provided",
                    LongDescription = @"Foreign employment income lump sum A payment type is required where Foreign employment income lump sum A is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000347"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS175", Value = GetValueOrEmpty(income.INCDTLS175) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS174", Value = GetValueOrEmpty(income.INCDTLS174) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000348
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000348(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000348
                    When the Foreign employment income net amount is provided, the amount must not exceed the sum of Foreign employment income gross amount, Foreign employment income lump sum A amount and Foreign employment income lump sum E amount
    
                    Legacy Rule Format:
                    (^INCDTLS178 > 0) AND (^INCDTLS178 > (^INCDTLS168 + ^INCDTLS174 + ^INCDTLS177))
        
                    Technical Business Rule Format:
                    (^INCDTLS178 > 0) AND (^INCDTLS178 > (^INCDTLS168 + ^INCDTLS174 + ^INCDTLS177))
            
                    Data Elements:
            
                    ^INCDTLS178 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount
            
                    ^INCDTLS168 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS174 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS177 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.BackPaymentLumpSumE.Amount
                    */
            assertion = (income.INCDTLS178.GetValueOrDefault() > 0 && income.INCDTLS178.GetValueOrDefault() > income.INCDTLS168.GetValueOrDefault() + income.INCDTLS174.GetValueOrDefault() + income.INCDTLS177.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000348",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income net amount is incorrect",
                    LongDescription = @"When the Foreign employment income net amount is provided, the amount must not exceed the sum of Foreign employment income gross amount, Foreign employment income lump sum A amount and Foreign employment income lump sum E amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:InternationalDealingsIncomePaymentSummaryA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000348"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS178", Value = GetValueOrEmpty(income.INCDTLS178) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS168", Value = GetValueOrEmpty(income.INCDTLS168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS174", Value = GetValueOrEmpty(income.INCDTLS174) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS177", Value = GetValueOrEmpty(income.INCDTLS177) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000353
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000353()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000353
            When a positive amount has been provided for the Foreign entity transferor trust income, both the Foreign entity non-resident trust name and Foreign entity trustee or trustees name must be provided
    
            Legacy Rule Format:
            (^INCDTLS204 > 0) AND (^INCDTLS201 = NULL OR ^INCDTLS202 = NULL)

            Technical Business Rule Format:
            (^INCDTLS204 > 0) AND (^INCDTLS201 = NULL OR ^INCDTLS202 = NULL)
    
            Data Elements:
    
            ^INCDTLS201 = INCDTLS:Rp:ForeignEntities:OrganisationNameDetails.OrganisationalName.Text
    
            ^INCDTLS202 = INCDTLS:Rp:ForeignEntities:PersonUnstructuredName.FullName.Text
    
            ^INCDTLS204 = INCDTLS:Rp:ForeignEntities:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            */
            assertion = (report.INCDTLS204.GetValueOrDefault() > 0 && (report.INCDTLS201 == null || report.INCDTLS202 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000353",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign entity non-resident trust name and Foreign entity trustee or trustees name must both be provided",
                    LongDescription = @"When a positive amount has been provided for the Foreign entity transferor trust income, both the Foreign entity non-resident trust name and Foreign entity trustee or trustees name must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignEntities/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000353"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS204", Value = GetValueOrEmpty(report.INCDTLS204) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS201", Value = GetValueOrEmpty(report.INCDTLS201) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS202", Value = GetValueOrEmpty(report.INCDTLS202) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000354
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000354()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000354
            Where a Foreign entities instance is present, either the Foreign entity CFC income or the Foreign entity transferor trust income must be provided
    
            Legacy Rule Format:
            (^INCDTLS200 <> NULL) AND (^INCDTLS203 = NULL AND ^INCDTLS204 = NULL)

            Technical Business Rule Format:
            (^INCDTLS200 <> NULL) AND (^INCDTLS203 = NULL AND ^INCDTLS204 = NULL)
    
            Data Elements:
    
            ^INCDTLS203 = INCDTLS:Rp:ForeignEntities:Income.ControlledForeignCompaniesAttributableIncome.Amount
    
            ^INCDTLS200 = INCDTLS:Rp:ForeignEntities
    
            ^INCDTLS204 = INCDTLS:Rp:ForeignEntities:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            */
            assertion = (report.Rp_ForeignEntitiesCollectionExists != false && (report.INCDTLS203 == null && report.INCDTLS204 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000354",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign entities amount details are incomplete",
                    LongDescription = @"Where a Foreign entities instance is present, either the Foreign entity CFC income or the Foreign entity transferor trust income must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignEntities/tns:IncomeControlledForeignCompaniesAttributableIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000354"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS203", Value = GetValueOrEmpty(report.INCDTLS203) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS204", Value = GetValueOrEmpty(report.INCDTLS204) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000356
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000356(INCDTLS2026.Rp_ForeignIncome foreignIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000356
                    Foreign income employment details and Exempt foreign income details must not both be provided within any Foreign employment income non-payment summary instance
    
                    Legacy Rule Format:
                    (^INCDTLS185 <> NULL) AND ((^INCDTLS188 <> NULL) AND (^INCDTLS191 <> NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS185 <> NULL) AND ((^INCDTLS188 <> NULL) AND (^INCDTLS191 <> NULL))
            
                    Data Elements:
            
                    ^INCDTLS188 = INCDTLS:Rp:ForeignIncome:ForeignIncomeEmploymentDetails
            
                    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
            
                    ^INCDTLS191 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome
                    */
            assertion = (report.Rp_ForeignIncomeCollectionExists != false && (foreignIncome.Rp_ForeignIncome_ForeignIncomeEmploymentDetailsCollectionExists != false && foreignIncome.Rp_ForeignIncome_ExemptForeignIncomeCollectionExists != false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000356",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income employment details and Exempt foreign income details must not both be provided",
                    LongDescription = @"Foreign income employment details and Exempt foreign income details must not both be provided within any Foreign employment income non-payment summary instance",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignIncomeCollection/tns:ForeignIncome" + OccurrenceIndex(foreignIncome.OccurrenceIndex) + "/tns:ForeignIncomeEmploymentDetails",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000356"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000357
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000357(INCDTLS2026.Rp_ForeignIncome foreignIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000357
                    Where a Foreign employment income non-payment summary instance is present, either the Foreign income employment details or Exempt foreign income details must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS185 <> NULL) AND ((^INCDTLS188 = NULL) AND (^INCDTLS191 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS185 <> NULL) AND ((^INCDTLS188 = NULL) AND (^INCDTLS191 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS188 = INCDTLS:Rp:ForeignIncome:ForeignIncomeEmploymentDetails
            
                    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
            
                    ^INCDTLS191 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome
                    */
            assertion = (report.Rp_ForeignIncomeCollectionExists != false && (foreignIncome.Rp_ForeignIncome_ForeignIncomeEmploymentDetailsCollectionExists == false && foreignIncome.Rp_ForeignIncome_ExemptForeignIncomeCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000357",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income non-payment summary details are incomplete",
                    LongDescription = @"Where a Foreign employment income non-payment summary instance is present, either the Foreign income employment details or Exempt foreign income details must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignIncomeCollection/tns:ForeignIncome" + OccurrenceIndex(foreignIncome.OccurrenceIndex) + "/tns:ForeignIncomeEmploymentDetails",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000357"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000359
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000359(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000359
                    Where any Foreign employment income payment summary lump sum in arrears payment instances are present for a Foreign employment income payment summary, the Foreign employment income lump sum E amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS160 <> NULL AND Count(^INCDTLS522) > 0 AND ^INCDTLS177 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS160 <> NULL AND Count(^INCDTLS522) > 0 AND ^INCDTLS177 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS177 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.BackPaymentLumpSumE.Amount
            
                    ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
            
                    ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_ForeignEmploymentCollectionExists != false && Count(income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionCount) > 0 && income.INCDTLS177 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000359",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign employment income lump sum E amount must be provided",
                    LongDescription = @"Where any Foreign employment income payment summary lump sum in arrears payment instances are present for a Foreign employment income payment summary, the Foreign employment income lump sum E amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:BackPaymentLumpSumEA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000359"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS177", Value = GetValueOrEmpty(income.INCDTLS177) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000360
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000360(INCDTLS2026.Rp_ForeignIncome foreignIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000360
                    Where any Foreign employment income non-payment summary lump sum in arrears payment instances are present for a Foreign employment income non-payment summary, the Exempt foreign income lump sum in arrears amount or Foreign income gross amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS185 <> NULL AND Count(^INCDTLS528) > 0 AND (^INCDTLS195 = NULL AND ^INCDTLS189 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS185 <> NULL AND Count(^INCDTLS528) > 0 AND (^INCDTLS195 = NULL AND ^INCDTLS189 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS195 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome:Income.LumpSumArrearsPayment.Amount
            
                    ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
            
                    ^INCDTLS189 = INCDTLS:Rp:ForeignIncome:ForeignIncomeEmploymentDetails:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
                    */
            assertion = (report.Rp_ForeignIncomeCollectionExists != false && Count(foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionCount) > 0 && (foreignIncome.INCDTLS195 == null && foreignIncome.INCDTLS189 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000360",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt foreign income lump sum in arrears or Foreign income gross amount must be provided",
                    LongDescription = @"Where any Foreign employment income non-payment summary lump sum in arrears payment instances are present for a Foreign employment income non-payment summary, the Exempt foreign income lump sum in arrears amount or Foreign income gross amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignIncomeCollection/tns:ForeignIncome" + OccurrenceIndex(foreignIncome.OccurrenceIndex) + "/tns:ExemptForeignIncome/tns:LumpSumArrearsPaymentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000360"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS195", Value = GetValueOrEmpty(foreignIncome.INCDTLS195) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS189", Value = GetValueOrEmpty(foreignIncome.INCDTLS189) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000361
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000361(INCDTLS2026.Rp_ForeignPensionsOrAnnuities foreignPensionsOrAnnuities, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000361
                    Where any Foreign pensions or annuities lump sum in arrears payment instances are present for a Foreign pensions or annuities, the Foreign pension or annuity lump sum in arrears amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS207 <> NULL AND Count(^INCDTLS531) > 0 AND ^INCDTLS213 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS207 <> NULL AND Count(^INCDTLS531) > 0 AND ^INCDTLS213 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS213 = INCDTLS:Rp:ForeignPensionsOrAnnuities:Income.LumpSumArrearsPayment.Amount
            
                    ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
            
                    ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
                    */
            assertion = (report.Rp_ForeignPensionsOrAnnuitiesCollectionExists != false && Count(foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionCount) > 0 && foreignPensionsOrAnnuities.INCDTLS213 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000361",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign pension or annuity lump sum in arrears amount must be provided",
                    LongDescription = @"Where any Foreign pensions or annuities lump sum in arrears payment instances are present for a Foreign pensions or annuities, the Foreign pension or annuity lump sum in arrears amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignPensionsOrAnnuitiesCollection/tns:ForeignPensionsOrAnnuities" + OccurrenceIndex(foreignPensionsOrAnnuities.OccurrenceIndex) + "/tns:IncomeLumpSumArrearsPaymentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000361"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS213", Value = GetValueOrEmpty(foreignPensionsOrAnnuities.INCDTLS213) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000365
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000365(INCDTLS2026.Rp_OtherForeignIncome otherForeignIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000365
                    Where an Other foreign income instance is present, at least one of the following must be provided: Foreign rental income details, Foreign financial investment details or Other foreign income details
    
                    Legacy Rule Format:
                    (^INCDTLS215 <> NULL) AND ((^INCDTLS218 = NULL AND ^INCDTLS221 = NULL AND ^INCDTLS225 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS215 <> NULL) AND ((^INCDTLS218 = NULL AND ^INCDTLS221 = NULL AND ^INCDTLS225 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS218 = INCDTLS:Rp:OtherForeignIncome:ForeignRentalIncomeDetails
            
                    ^INCDTLS215 = INCDTLS:Rp:OtherForeignIncome
            
                    ^INCDTLS221 = INCDTLS:Rp:OtherForeignIncome:ForeignFinancialInvestmentDetails
            
                    ^INCDTLS225 = INCDTLS:Rp:OtherForeignIncome:OtherForeignIncomeDetails
                    */
            assertion = (report.Rp_OtherForeignIncomeCollectionExists != false && (otherForeignIncome.Rp_OtherForeignIncome_ForeignRentalIncomeDetailsCollectionExists == false && otherForeignIncome.Rp_OtherForeignIncome_ForeignFinancialInvestmentDetailsCollectionExists == false && otherForeignIncome.Rp_OtherForeignIncome_OtherForeignIncomeDetailsCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000365",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other foreign income instance is incomplete",
                    LongDescription = @"Where an Other foreign income instance is present, at least one of the following must be provided: Foreign rental income details, Foreign financial investment details or Other foreign income details",
                    Location = "/tns:INCDTLS/tns:Rp/tns:OtherForeignIncomeCollection/tns:OtherForeignIncome" + OccurrenceIndex(otherForeignIncome.OccurrenceIndex) + "/tns:ForeignRentalIncomeDetails",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000365"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000366
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000366(INCDTLS2026.Rp_OtherForeignIncome otherForeignIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000366
                    Where an Other foreign income instance is present, only one of the following may be provided: Foreign rental income details, Foreign financial investment details or Other foreign income details
    
                    Legacy Rule Format:
                    (^INCDTLS215 <> NULL) AND (((Count(^INCDTLS218) + Count(^INCDTLS221) + Count(^INCDTLS225)) > 1))
        
                    Technical Business Rule Format:
                    (^INCDTLS215 <> NULL) AND (((Count(^INCDTLS218) + Count(^INCDTLS221) + Count(^INCDTLS225)) > 1))
            
                    Data Elements:
            
                    ^INCDTLS218 = INCDTLS:Rp:OtherForeignIncome:ForeignRentalIncomeDetails
            
                    ^INCDTLS215 = INCDTLS:Rp:OtherForeignIncome
            
                    ^INCDTLS221 = INCDTLS:Rp:OtherForeignIncome:ForeignFinancialInvestmentDetails
            
                    ^INCDTLS225 = INCDTLS:Rp:OtherForeignIncome:OtherForeignIncomeDetails
                    */
            assertion = (report.Rp_OtherForeignIncomeCollectionExists != false && Count(otherForeignIncome.Rp_OtherForeignIncome_ForeignRentalIncomeDetailsCollectionCount) + Count(otherForeignIncome.Rp_OtherForeignIncome_ForeignFinancialInvestmentDetailsCollectionCount) + Count(otherForeignIncome.Rp_OtherForeignIncome_OtherForeignIncomeDetailsCollectionCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000366",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other foreign income instance is incorrect",
                    LongDescription = @"Where an Other foreign income instance is present, only one of the following may be provided: Foreign rental income details, Foreign financial investment details or Other foreign income details",
                    Location = "/tns:INCDTLS/tns:Rp/tns:OtherForeignIncomeCollection/tns:OtherForeignIncome" + OccurrenceIndex(otherForeignIncome.OccurrenceIndex) + "/tns:ForeignRentalIncomeDetails",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000366"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000367
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000367(INCDTLS2026.Rp_EmployeeShareScheme employeeShareScheme, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000367
                    Where an Employee share scheme instance is present, at least one Employee share scheme amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS500 <> NULL AND ^INCDTLS501 = NULL AND ^INCDTLS502 = NULL AND ^INCDTLS503 = NULL AND ^INCDTLS505 = NULL AND ^INCDTLS506 = NULL AND ^INCDTLS507 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS500 <> NULL AND ^INCDTLS501 = NULL AND ^INCDTLS502 = NULL AND ^INCDTLS503 = NULL AND ^INCDTLS505 = NULL AND ^INCDTLS506 = NULL AND ^INCDTLS507 = NULL
            
                    Data Elements:
            
                    ^INCDTLS501 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
            
                    ^INCDTLS500 = INCDTLS:Rp:EmployeeShareScheme
            
                    ^INCDTLS502 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
            
                    ^INCDTLS503 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.DeferralDiscount.Amount
            
                    ^INCDTLS505 = INCDTLS:Rp:EmployeeShareScheme:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount
            
                    ^INCDTLS506 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.ForeignSourceDiscount.Amount
            
                    ^INCDTLS507 = INCDTLS:Rp:EmployeeShareScheme:Income.InternationalDealings.TaxOffset.Amount
                    */
            assertion = (report.Rp_EmployeeShareSchemeCollectionExists != false && employeeShareScheme.INCDTLS501 == null && employeeShareScheme.INCDTLS502 == null && employeeShareScheme.INCDTLS503 == null && employeeShareScheme.INCDTLS505 == null && employeeShareScheme.INCDTLS506 == null && employeeShareScheme.INCDTLS507 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000367",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Employee share scheme amount details are incomplete",
                    LongDescription = @"Where an Employee share scheme instance is present, at least one Employee share scheme amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:EmployeeShareSchemeCollection/tns:EmployeeShareScheme" + OccurrenceIndex(employeeShareScheme.OccurrenceIndex) + "/tns:IncomeTaxUpfrontReducibleDiscountA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000367"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS501", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS501) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS502", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS502) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS503", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS503) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS505", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS505) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS506", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS506) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS507", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS507) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000368
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000368(INCDTLS2026.Rp_EmployeeShareScheme employeeShareScheme, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000368
                    The TFN amounts withheld from discounts must not exceed 50% of the sum of Discount from taxed-upfront schemes eligible for reduction, Discount from taxed-upfront schemes not eligible for reduction and Discount from deferral schemes
    
                    Legacy Rule Format:
                    (^INCDTLS505 > ((^INCDTLS501 + ^INCDTLS502 + ^INCDTLS503) * 0.5))
        
                    Technical Business Rule Format:
                    (^INCDTLS505 > ((^INCDTLS501 + ^INCDTLS502 + ^INCDTLS503) * 0.5))
            
                    Data Elements:
            
                    ^INCDTLS505 = INCDTLS:Rp:EmployeeShareScheme:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount
            
                    ^INCDTLS501 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
            
                    ^INCDTLS502 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
            
                    ^INCDTLS503 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.DeferralDiscount.Amount
                    */
            assertion = (employeeShareScheme.INCDTLS505.GetValueOrDefault() > (employeeShareScheme.INCDTLS501.GetValueOrDefault() + employeeShareScheme.INCDTLS502.GetValueOrDefault() + employeeShareScheme.INCDTLS503.GetValueOrDefault()) * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000368",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from discounts is incorrect",
                    LongDescription = @"The TFN amounts withheld from discounts must not exceed 50% of the sum of Discount from taxed-upfront schemes eligible for reduction, Discount from taxed-upfront schemes not eligible for reduction and Discount from deferral schemes",
                    Location = "/tns:INCDTLS/tns:Rp/tns:EmployeeShareSchemeCollection/tns:EmployeeShareScheme" + OccurrenceIndex(employeeShareScheme.OccurrenceIndex) + "/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000368"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS505", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS505) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS501", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS501) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS502", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS502) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS503", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS503) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000369
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000369(INCDTLS2026.Rp_EmployeeShareScheme employeeShareScheme, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000369
                    The Foreign source discounts amount must not exceed the sum of Discount from taxed-upfront schemes eligible for reduction, Discount from taxed-upfront schemes not eligible for reduction, and Discount from deferral schemes
    
                    Legacy Rule Format:
                    (^INCDTLS506 > (^INCDTLS501 + ^INCDTLS502 + ^INCDTLS503))
        
                    Technical Business Rule Format:
                    (^INCDTLS506 > (^INCDTLS501 + ^INCDTLS502 + ^INCDTLS503))
            
                    Data Elements:
            
                    ^INCDTLS506 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.ForeignSourceDiscount.Amount
            
                    ^INCDTLS501 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount
            
                    ^INCDTLS502 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount
            
                    ^INCDTLS503 = INCDTLS:Rp:EmployeeShareScheme:Income.EmployeeShareScheme.DeferralDiscount.Amount
                    */
            assertion = (employeeShareScheme.INCDTLS506.GetValueOrDefault() > employeeShareScheme.INCDTLS501.GetValueOrDefault() + employeeShareScheme.INCDTLS502.GetValueOrDefault() + employeeShareScheme.INCDTLS503.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.010369",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign source discounts amount is incorrect",
                    LongDescription = @"The Foreign source discounts amount must not exceed the sum of Discount from taxed-upfront schemes eligible for reduction, Discount from taxed-upfront schemes not eligible for reduction, and Discount from deferral schemes",
                    Location = "/tns:INCDTLS/tns:Rp/tns:EmployeeShareSchemeCollection/tns:EmployeeShareScheme" + OccurrenceIndex(employeeShareScheme.OccurrenceIndex) + "/tns:IncomeForeignSourceDiscountA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000369"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS506", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS506) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS501", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS501) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS502", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS502) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS503", Value = GetValueOrEmpty(employeeShareScheme.INCDTLS503) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000371
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000371(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000371
                    The Business income statement payer's Australian business number has failed the ABN algorithm check
    
                    Legacy Rule Format:
                    (FailsABNAlgorithm(^INCDTLS488))
        
                    Technical Business Rule Format:
                    (FailsABNAlgorithm(^INCDTLS488))
            
                    Data Elements:
            
                    ^INCDTLS488 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = FailsABNAlgorithm(businessIncomePaymentSummaries.INCDTLS488);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000371",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement payer's Australian business number is invalid",
                    LongDescription = @"Business income statement payer's Australian business number has failed the ABN algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000371"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS488", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS488) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000372
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000372(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000372
                    Where the Business income statement payment type is "011" (foreign resident withholding), the Business income statement type must be set to "Business"
    
                    Legacy Rule Format:
                    (^INCDTLS489 = "011") AND (^INCDTLS490 <> "Business")
        
                    Technical Business Rule Format:
                    (^INCDTLS489 = '011') AND (^INCDTLS490 <> 'Business')
            
                    Data Elements:
            
                    ^INCDTLS489 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Remuneration.WithholdingPaymentIncomeType.Code
            
                    ^INCDTLS490 = INCDTLS:Rp:BusinessIncomePaymentSummaries:IncomeTax.IncomeType.Code
                    */
            assertion = (businessIncomePaymentSummaries.INCDTLS489 == @"011" && businessIncomePaymentSummaries.INCDTLS490 != @"Business");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000372",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement payment type is incorrect",
                    LongDescription = @"Where the Business income statement payment type is ""011"" (foreign resident withholding), the Business income statement type must be set to ""Business""",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:RemunerationWithholdingPaymentIncomeTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000372"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS489", Value = businessIncomePaymentSummaries.INCDTLS489 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS490", Value = businessIncomePaymentSummaries.INCDTLS490 });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000373
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000373(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000373
                    Where the Business income statement payment type is "011" (foreign resident withholding), the Business income statement industry production type must be set to "Non-Primary"
    
                    Legacy Rule Format:
                    (^INCDTLS489 = "011") AND (^INCDTLS491 <> "Non-Primary")
        
                    Technical Business Rule Format:
                    (^INCDTLS489 = '011') AND (^INCDTLS491 <> 'Non-Primary')
            
                    Data Elements:
            
                    ^INCDTLS489 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Remuneration.WithholdingPaymentIncomeType.Code
            
                    ^INCDTLS491 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Income.IndustryProductionType.Code
                    */
            assertion = (businessIncomePaymentSummaries.INCDTLS489 == @"011" && businessIncomePaymentSummaries.INCDTLS491 != @"Non-Primary");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000373",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement industry production type is incorrect",
                    LongDescription = @"Where the Business income statement payment type is ""011"" (foreign resident withholding), the Business income statement industry production type must be set to ""Non-Primary""",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:RemunerationWithholdingPaymentIncomeTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000373"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS489", Value = businessIncomePaymentSummaries.INCDTLS489 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS491", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS491) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000374
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000374(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000374
                    Where the Business income statement type is "Business", the Business income statement industry production type must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS490 = "Business") AND (^INCDTLS491 = NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS490 = 'Business') AND (^INCDTLS491 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS490 = INCDTLS:Rp:BusinessIncomePaymentSummaries:IncomeTax.IncomeType.Code
            
                    ^INCDTLS491 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Income.IndustryProductionType.Code
                    */
            assertion = (businessIncomePaymentSummaries.INCDTLS490 == @"Business" && businessIncomePaymentSummaries.INCDTLS491 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000374",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement industry production type must be provided",
                    LongDescription = @"Where the Business income statement type is ""Business"", the Business income statement industry production type must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:IncomeTaxIncomeTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000374"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS490", Value = businessIncomePaymentSummaries.INCDTLS490 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS491", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS491) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000375
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000375(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000375
                    Where the Business income statement type is "Personal Services Income", the Business income statement industry production type must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS490 = "Personal Services Income") AND (^INCDTLS491 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS490 = 'Personal Services Income') AND (^INCDTLS491 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS490 = INCDTLS:Rp:BusinessIncomePaymentSummaries:IncomeTax.IncomeType.Code
            
                    ^INCDTLS491 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Income.IndustryProductionType.Code
                    */
            assertion = (businessIncomePaymentSummaries.INCDTLS490 == @"Personal Services Income" && businessIncomePaymentSummaries.INCDTLS491 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000375",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement industry production type must not be provided",
                    LongDescription = @"Where the Business income statement type is ""Personal Services Income"", the Business income statement industry production type must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:IncomeTaxIncomeTypeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000375"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS490", Value = businessIncomePaymentSummaries.INCDTLS490 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS491", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS491) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000376
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000376(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000376
                    Business income statement tax withheld amount must not exceed 50% of the Business income statement gross payment amount
    
                    Legacy Rule Format:
                    (^INCDTLS493 > (^INCDTLS492 * 0.5))
        
                    Technical Business Rule Format:
                    (^INCDTLS493 > (^INCDTLS492 * 0.5))
            
                    Data Elements:
            
                    ^INCDTLS493 = INCDTLS:Rp:BusinessIncomePaymentSummaries:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheld.Amount
            
                    ^INCDTLS492 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Income.BusinessGrossTotal.Amount
                    */
            assertion = (businessIncomePaymentSummaries.INCDTLS493.GetValueOrDefault() > businessIncomePaymentSummaries.INCDTLS492.GetValueOrDefault() * 0.5M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000376",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement tax withheld amount is incorrect",
                    LongDescription = @"Business income statement tax withheld amount must not exceed 50% of the Business income statement gross payment amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000376"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS493", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS493) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS492", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS492) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000377
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000377(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000377
                    Where Business income statement payment type is "001" (labour hire), "002" (other specified payments) or "003" (voluntary agreement), Business income statement tax withheld amount must not include cents
    
                    Legacy Rule Format:
                    (InSet(^INCDTLS489, '"001", "002", "003"') AND NotMonetary(^INCDTLS493, 'U',11,0))
        
                    Technical Business Rule Format:
                    (InSet(^INCDTLS489, '"001", "002", "003"') AND NotMonetary(^INCDTLS493, 'U',11,0))
            
                    Data Elements:
            
                    ^INCDTLS493 = INCDTLS:Rp:BusinessIncomePaymentSummaries:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheld.Amount
            
                    ^INCDTLS489 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Remuneration.WithholdingPaymentIncomeType.Code
                    */
            assertion = (IsMatch(businessIncomePaymentSummaries.INCDTLS489, @"^(001|002|003)$") && NotMonetary(businessIncomePaymentSummaries.INCDTLS493, @"U", 11, 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000377",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement tax withheld amount must not include cents",
                    LongDescription = @"Where Business income statement payment type is ""001"" (labour hire), ""002"" (other specified payments) or ""003"" (voluntary agreement), Business income statement tax withheld amount must not include cents",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000377"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS489", Value = businessIncomePaymentSummaries.INCDTLS489 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS493", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS493) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000378
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000378(INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000378
                    Where Business income statement payment type is "011" (foreign resident withholding) or "012" (no ABN quoted), Business income statement reportable employer super contribution amount must not be provided
    
                    Legacy Rule Format:
                    (InSet(^INCDTLS489, '"011", "012"') AND ^INCDTLS494 <> NULL)
        
                    Technical Business Rule Format:
                    (InSet(^INCDTLS489, '"011", "012"') AND ^INCDTLS494 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS494 = INCDTLS:Rp:BusinessIncomePaymentSummaries:SuperannuationContribution.EmployerContributions.Amount
            
                    ^INCDTLS489 = INCDTLS:Rp:BusinessIncomePaymentSummaries:Remuneration.WithholdingPaymentIncomeType.Code
                    */
            assertion = (IsMatch(businessIncomePaymentSummaries.INCDTLS489, @"^(011|012)$") && businessIncomePaymentSummaries.INCDTLS494 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000378",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business income statement reportable employer super contribution amount must not be provided",
                    LongDescription = @"Where Business income statement payment type is ""011"" (foreign resident withholding) or ""012"" (no ABN quoted), Business income statement reportable employer super contribution amount must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:BusinessIncomePaymentSummariesCollection/tns:BusinessIncomePaymentSummaries" + OccurrenceIndex(businessIncomePaymentSummaries.OccurrenceIndex) + "/tns:SuperannuationContributionEmployerContributionsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000378"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS489", Value = businessIncomePaymentSummaries.INCDTLS489 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS494", Value = GetValueOrEmpty(businessIncomePaymentSummaries.INCDTLS494) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000379
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000379(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000379
                    The Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must equal the sum of the Non-primary production managed investment scheme amount and the Non-primary production remaining trust amount
    
                    Legacy Rule Format:
                    (^INCDTLS250 <> NULL AND ^INCDTLS250 <> (^INCDTLS251 + ^INCDTLS252))
        
                    Technical Business Rule Format:
                    (^INCDTLS250 <> NULL AND ^INCDTLS250 <> (^INCDTLS251 + ^INCDTLS252))
            
                    Data Elements:
            
                    ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS251 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS252 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustOtherNonPrimaryProductionShare.Amount
                    */
            assertion = (trustIncome.INCDTLS250 != null && trustIncome.INCDTLS250.GetValueOrDefault() != trustIncome.INCDTLS251.GetValueOrDefault() + trustIncome.INCDTLS252.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000379",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) is incorrect",
                    LongDescription = @"The Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must equal the sum of the Non-primary production managed investment scheme amount and the Non-primary production remaining trust amount",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustNonPrimaryProduction/tns:TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000379"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = GetValueOrEmpty(trustIncome.INCDTLS250) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS251", Value = GetValueOrEmpty(trustIncome.INCDTLS251) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS252", Value = GetValueOrEmpty(trustIncome.INCDTLS252) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000380
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000380(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000380
                    When either the Non-primary production managed investment scheme amount or the Non-primary production remaining trust amount is present, the Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS250 = NULL AND (^INCDTLS251 <> NULL OR ^INCDTLS252 <> NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS250 = NULL AND (^INCDTLS251 <> NULL OR ^INCDTLS252 <> NULL))
            
                    Data Elements:
            
                    ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS251 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS252 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustOtherNonPrimaryProductionShare.Amount
                    */
            assertion = (trustIncome.INCDTLS250 == null && (trustIncome.INCDTLS251 != null || trustIncome.INCDTLS252 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000380",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-primary production net income from trusts amount details are incomplete",
                    LongDescription = @"When either the Non-primary production managed investment scheme amount or the Non-primary production remaining trust amount is present, the Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustNonPrimaryProduction/tns:TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000380"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = GetValueOrEmpty(trustIncome.INCDTLS250) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS251", Value = GetValueOrEmpty(trustIncome.INCDTLS251) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS252", Value = GetValueOrEmpty(trustIncome.INCDTLS252) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000381
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000381(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000381
                    Where Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) is present, both the Non-primary production managed investment scheme amount and the Non-primary production remaining trust amount must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS250 <> NULL AND (^INCDTLS251 = NULL OR ^INCDTLS252 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS250 <> NULL AND (^INCDTLS251 = NULL OR ^INCDTLS252 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS251 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS252 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustOtherNonPrimaryProductionShare.Amount
                    */
            assertion = (trustIncome.INCDTLS250 != null && (trustIncome.INCDTLS251 == null || trustIncome.INCDTLS252 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000381",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-primary production net income from trusts amount details are incomplete",
                    LongDescription = @"Where Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) is present, both the Non-primary production managed investment scheme amount and the Non-primary production remaining trust amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustNonPrimaryProduction/tns:TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000381"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = GetValueOrEmpty(trustIncome.INCDTLS250) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS251", Value = GetValueOrEmpty(trustIncome.INCDTLS251) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS252", Value = GetValueOrEmpty(trustIncome.INCDTLS252) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000389
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000389(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000389
                    Where a Trust non-primary production instance is present, Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS249 <> NULL AND ^INCDTLS250 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS249 <> NULL AND ^INCDTLS250 = NULL
            
                    Data Elements:
            
                    ^INCDTLS249 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction
            
                    ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
                    */
            assertion = (trustIncome.Rp_TrustIncome_TrustNonPrimaryProductionCollectionExists != false && trustIncome.INCDTLS250 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.001389",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided",
                    LongDescription = @"Where a Trust non-primary production instance is present, Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must also be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustNonPrimaryProduction",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000389"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = GetValueOrEmpty(trustIncome.INCDTLS250) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000390
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000390(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000390
                    Where a Trust income instance is present, at least one of the following must be provided: Trust primary production details, Trust non-primary production details, Trust credits from income and tax offsets details, Trust Capital Gains, or Foreign and Non-Resident Income
    
                    Legacy Rule Format:
                    (^INCDTLS245 <> NULL AND (^INCDTLS246 = NULL AND ^INCDTLS249 = NULL AND ^INCDTLS259 = NULL AND ^INCDTLS551 = NULL AND ^INCDTLS559 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS245 <> NULL AND (^INCDTLS246 = NULL AND ^INCDTLS249 = NULL AND ^INCDTLS259 = NULL AND ^INCDTLS551 = NULL AND ^INCDTLS559 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS245 = INCDTLS:Rp:TrustIncome
            
                    ^INCDTLS246 = INCDTLS:Rp:TrustIncome:TrustPrimaryProduction
            
                    ^INCDTLS249 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction
            
                    ^INCDTLS259 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets
            
                    ^INCDTLS551 = INCDTLS:Rp:TrustIncome:TrustCapitalGains
            
                    ^INCDTLS559 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident
                    */
            assertion = (report.Rp_TrustIncomeCollectionExists != false && (trustIncome.Rp_TrustIncome_TrustPrimaryProductionCollectionExists == false && trustIncome.Rp_TrustIncome_TrustNonPrimaryProductionCollectionExists == false && trustIncome.Rp_TrustIncome_TrustCreditsIncomeTaxOffsetsCollectionExists == false && trustIncome.Rp_TrustIncome_TrustCapitalGainsCollectionExists == false && trustIncome.Rp_TrustIncome_ForeignAndNonResidentCollectionExists == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.010390",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trusts income instance is incomplete",
                    LongDescription = @"Where a Trust income instance is present, at least one of the following must be provided: Trust primary production details, Trust non-primary production details, Trust credits from income and tax offsets details, Trust Capital Gains, or Foreign and Non-Resident Income",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000390"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000391
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000391(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000391
                    Where a Trust credits from income and tax offsets instance is present, at least one of the following amounts must be provided: Tax withheld where Australian business number not quoted from trusts; Franking credit from franked dividends from trusts; TFN amounts withheld from interest, dividends and unit trust distributions from trusts; TFN amounts withheld from payments from closely held trusts; Credit for tax paid by trustee; National rental affordability scheme tax offset from trusts; Australian franking credits from a New Zealand company - Beneficiary share; Exploration credits distributed; Share of net small business income; Early stage venture capital limited partnership tax offset - Beneficiary; or Early stage investor tax offset - Beneficiary
    
                    Legacy Rule Format:
                    ^INCDTLS259 <> NULL AND (^INCDTLS260 = NULL AND ^INCDTLS261 = NULL AND ^INCDTLS262 = NULL AND ^INCDTLS263 = NULL AND ^INCDTLS264 = NULL AND ^INCDTLS269 = NULL AND ^INCDTLS546 = NULL AND ^INCDTLS547 = NULL AND ^INCDTLS548 = NULL AND ^INCDTLS549 = NULL AND ^INCDTLS550 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS259 <> NULL AND (^INCDTLS260 = NULL AND ^INCDTLS261 = NULL AND ^INCDTLS262 = NULL AND ^INCDTLS263 = NULL AND ^INCDTLS264 = NULL AND ^INCDTLS269 = NULL AND ^INCDTLS546 = NULL AND ^INCDTLS547 = NULL AND ^INCDTLS548 = NULL AND ^INCDTLS549 = NULL AND ^INCDTLS550 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS259 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets
            
                    ^INCDTLS260 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS261 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS262 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
            
                    ^INCDTLS263 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount
            
                    ^INCDTLS264 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS269 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount
            
                    ^INCDTLS546 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            
                    ^INCDTLS547 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.ExplorationCreditsDistributed.Amount
            
                    ^INCDTLS548 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:Income.SmallBusinessEntityNet.Amount
            
                    ^INCDTLS549 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.EarlyStageVentureCapitalLimitedPartnershipTaxOffset.Amount
            
                    ^INCDTLS550 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.EarlyStageInvestorTaxOffset.Amount
                    */
            assertion = (trustIncome.Rp_TrustIncome_TrustCreditsIncomeTaxOffsetsCollectionExists != false && (trustIncome.INCDTLS260 == null && trustIncome.INCDTLS261 == null && trustIncome.INCDTLS262 == null && trustIncome.INCDTLS263 == null && trustIncome.INCDTLS264 == null && trustIncome.INCDTLS269 == null && trustIncome.INCDTLS546 == null && trustIncome.INCDTLS547 == null && trustIncome.INCDTLS548 == null && trustIncome.INCDTLS549 == null && trustIncome.INCDTLS550 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.010391",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust credits from income and tax offsets amount details are incomplete",
                    LongDescription = @"Where a Trust credits from income and tax offsets instance is present, at least one of the following amounts must be provided: Tax withheld where Australian business number not quoted from trusts; Franking credit from franked dividends from trusts; TFN amounts withheld from interest, dividends and unit trust distributions from trusts; TFN amounts withheld from payments from closely held trusts; Credit for tax paid by trustee; National rental affordability scheme tax offset from trusts; Australian franking credits from a New Zealand company - Beneficiary share; Exploration credits distributed; Share of net small business income; Early stage venture capital limited partnership tax offset - Beneficiary; or Early stage investor tax offset - Beneficiary",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustCreditsIncomeTaxOffsets",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000391"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS260", Value = GetValueOrEmpty(trustIncome.INCDTLS260) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS261", Value = GetValueOrEmpty(trustIncome.INCDTLS261) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS262", Value = GetValueOrEmpty(trustIncome.INCDTLS262) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS263", Value = GetValueOrEmpty(trustIncome.INCDTLS263) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS264", Value = GetValueOrEmpty(trustIncome.INCDTLS264) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS269", Value = GetValueOrEmpty(trustIncome.INCDTLS269) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS546", Value = GetValueOrEmpty(trustIncome.INCDTLS546) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS547", Value = GetValueOrEmpty(trustIncome.INCDTLS547) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS548", Value = GetValueOrEmpty(trustIncome.INCDTLS548) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS549", Value = GetValueOrEmpty(trustIncome.INCDTLS549) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS550", Value = GetValueOrEmpty(trustIncome.INCDTLS550) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000392
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000392(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000392
                    Where Reason the trustee paid tax from trusts, Share of income from trusts or Tax credits for tax paid by trustee is present, the Credit for tax paid by trustee amount must be provided
    
                    Legacy Rule Format:
                    ((^INCDTLS265 <> NULL OR ^INCDTLS266 <> NULL OR ^INCDTLS267 <> NULL) AND ^INCDTLS264 = NULL)
        
                    Technical Business Rule Format:
                    ((^INCDTLS265 <> NULL OR ^INCDTLS266 <> NULL OR ^INCDTLS267 <> NULL) AND ^INCDTLS264 = NULL)
            
                    Data Elements:
            
                    ^INCDTLS264 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.CreditTrusteeTaxPaidTotal.Amount
            
                    ^INCDTLS265 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.CreditTrusteeTaxPaidReason.Code
            
                    ^INCDTLS266 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:Income.TrustDistributions.Amount
            
                    ^INCDTLS267 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.CreditTrusteeTaxPaidCredits.Amount
                    */
            assertion = ((trustIncome.INCDTLS265 != null || trustIncome.INCDTLS266 != null || trustIncome.INCDTLS267 != null) && trustIncome.INCDTLS264 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000392",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax paid by trustee must be provided",
                    LongDescription = @"Where Reason the trustee paid tax from trusts, Share of income from trusts or Tax credits for tax paid by trustee is present, the Credit for tax paid by trustee amount must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustCreditsIncomeTaxOffsets/tns:CreditTrusteeTaxPaidTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000392"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS265", Value = GetValueOrEmpty(trustIncome.INCDTLS265) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS266", Value = GetValueOrEmpty(trustIncome.INCDTLS266) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS267", Value = GetValueOrEmpty(trustIncome.INCDTLS267) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS264", Value = GetValueOrEmpty(trustIncome.INCDTLS264) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000393
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000393()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000393
            When a positive amount is present at any one of Tax withheld where Australian business number not quoted from trusts; Franking credit from franked dividends from trusts; TFN amounts withheld from interest, dividends and unit trust distributions from trusts; TFN amounts withheld from payments from closely held trusts; or Credit for foreign resident withholding amounts (excluding capital gains) from trusts; at least one of the following amounts must be provided: Primary production net income from trusts or Non-primary production net income from trusts (less capital gains, foreign income and franked distributions)
    
            Legacy Rule Format:
            AnyOccurrence(^INCDTLS245, (^INCDTLS260 > 0 OR ^INCDTLS261 > 0 OR ^INCDTLS262 > 0 OR ^INCDTLS263 > 0 OR ^INCDTLS268 > 0) AND ^INCDTLS247 = NULL AND ^INCDTLS250 = NULL)

            Technical Business Rule Format:
            AnyOccurrence(^INCDTLS245, (^INCDTLS260 > 0 OR ^INCDTLS261 > 0 OR ^INCDTLS262 > 0 OR ^INCDTLS263 > 0 OR ^INCDTLS268 > 0) AND ^INCDTLS247 = NULL AND ^INCDTLS250 = NULL)
    
            Data Elements:
    
            ^INCDTLS247 = INCDTLS:Rp:TrustIncome:TrustPrimaryProduction:Income.TrustPrimaryProductionNetShare.Amount
    
            ^INCDTLS245 = INCDTLS:Rp:TrustIncome
    
            ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^INCDTLS260 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^INCDTLS261 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.FrankingCreditsShare.Amount
    
            ^INCDTLS262 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
    
            ^INCDTLS263 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount
    
            ^INCDTLS268 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount
            */
            assertion = (report.Rp_TrustIncomeCollection == null ? false : report.Rp_TrustIncomeCollection.Any(INCDTLS245Repeat => (INCDTLS245Repeat.INCDTLS260.GetValueOrDefault() > 0 || INCDTLS245Repeat.INCDTLS261.GetValueOrDefault() > 0 || INCDTLS245Repeat.INCDTLS262.GetValueOrDefault() > 0 || INCDTLS245Repeat.INCDTLS263.GetValueOrDefault() > 0 || INCDTLS245Repeat.INCDTLS268.GetValueOrDefault() > 0) && INCDTLS245Repeat.INCDTLS247 == null && INCDTLS245Repeat.INCDTLS250 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.001393",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust income details are incomplete",
                    LongDescription = @"When a positive amount is present at any one of Tax withheld where Australian business number not quoted from trusts; Franking credit from franked dividends from trusts; TFN amounts withheld from interest, dividends and unit trust distributions from trusts; TFN amounts withheld from payments from closely held trusts; or Credit for foreign resident withholding amounts (excluding capital gains) from trusts; at least one of the following amounts must be provided: Primary production net income from trusts or Non-primary production net income from trusts (less capital gains, foreign income and franked distributions)",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome/tns:TrustPrimaryProduction/tns:NetShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000393"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS260", Value = "INCDTLS260" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS261", Value = "INCDTLS261" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS262", Value = "INCDTLS262" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS263", Value = "INCDTLS263" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS268", Value = "INCDTLS268" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS247", Value = "INCDTLS247" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = "INCDTLS250" });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000394
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000394(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000394
                    When the Total net foreign rent from managed fund amount provided is negative, then Your share of net foreign rent from managed fund must not be positive or less than Total net foreign rent from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS413 < 0 AND (^INCDTLS441 > 0 OR ^INCDTLS441 < ^INCDTLS413))
        
                    Technical Business Rule Format:
                    (^INCDTLS413 < 0 AND (^INCDTLS441 > 0 OR ^INCDTLS441 < ^INCDTLS413))
            
                    Data Elements:
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS413.GetValueOrDefault() < 0 && (managedFunds.INCDTLS441.GetValueOrDefault() > 0 || managedFunds.INCDTLS441.GetValueOrDefault() < managedFunds.INCDTLS413.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000394",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net foreign rent from managed fund is incorrect",
                    LongDescription = @"When the Total net foreign rent from managed fund amount provided is negative, then Your share of net foreign rent from managed fund must not be positive or less than Total net foreign rent from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignRentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000394"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000395
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000395(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000395
                    Where Total net foreign rent from managed fund is positive, Your share of net foreign rent from managed fund must not be greater than Total net foreign rent from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS441 > ^INCDTLS413 AND ^INCDTLS413 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS441 > ^INCDTLS413 AND ^INCDTLS413 >= 0
            
                    Data Elements:
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS441.GetValueOrDefault() > managedFunds.INCDTLS413.GetValueOrDefault() && managedFunds.INCDTLS413.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000395",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net foreign rent from managed fund is incorrect",
                    LongDescription = @"Where Total net foreign rent from managed fund is positive, Your share of net foreign rent from managed fund must not be greater than Total net foreign rent from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignRentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000395"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000396
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000396(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000396
                    Your share of net foreign rent from managed fund must not be negative where Total net foreign rent from managed fund is greater than or equal to 0
    
                    Legacy Rule Format:
                    (^INCDTLS441 < 0 AND ^INCDTLS413 >= 0)
        
                    Technical Business Rule Format:
                    ^INCDTLS441 < 0 AND ^INCDTLS413 >= 0
            
                    Data Elements:
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS441.GetValueOrDefault() < 0 && managedFunds.INCDTLS413.GetValueOrDefault() >= 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000396",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of net foreign rent from managed fund is incorrect",
                    LongDescription = @"Your share of net foreign rent from managed fund must not be negative where Total net foreign rent from managed fund is greater than or equal to 0",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignRentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000396"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000397
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000397(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000397
                    Where a foreign Employment termination summary instance is present, both the Payer's Australian business number and the Payer's withholding payer number must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS135 <> NULL AND (NotInSet(^INCDTLS136, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS136 <> NULL) AND (^INCDTLS307 <> NULL OR ^INCDTLS308 <> NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS135 <> NULL AND (NotInSet(^INCDTLS136, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS136 <> NULL) AND (^INCDTLS307 <> NULL OR ^INCDTLS308 <> NULL))
            
                    Data Elements:
            
                    ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
            
                    ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
            
                    ^INCDTLS136 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:AddressDetails.Country.Code
            
                    ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = (income.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists != false && (!(IsMatch(income.INCDTLS136, @"^(au|cc|cx|nf|hm)$", RegexOptions.IgnoreCase)) && income.INCDTLS136 != null) && (income.INCDTLS307 != null || income.INCDTLS308 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000397",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer's Australian business number and Payer's withholding payer number must not be provided",
                    LongDescription = @"Where a foreign Employment termination summary instance is present, both the Payer's Australian business number and the Payer's withholding payer number must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:WithholdingPayerNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000397"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS136", Value = GetValueOrEmpty(income.INCDTLS136) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS307", Value = GetValueOrEmpty(income.INCDTLS307) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS308", Value = GetValueOrEmpty(income.INCDTLS308) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000398
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000398(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000398
                    Where Country of residence when the dividend was paid or credited is not "Australia", "Cocos (Keeling) Islands", "Christmas Island", "Heard Island and McDonald Islands" or "Norfolk Island", Total exploration credits from dividends amount must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS375 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS375 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS348 = INCDTLS:Rp:Dividends:AddressDetails.Country.Code
            
                    ^INCDTLS375 = INCDTLS:Rp:Dividends:DividendTotal:IncomeTax.ExplorationCreditsDistributed.Amount
                    */
            assertion = (dividends.INCDTLS348 != null && !(IsMatch(dividends.INCDTLS348, @"^(au|cc|cx|nf|hm)$", RegexOptions.IgnoreCase)) && dividends.INCDTLS375 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000398",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total exploration credits from dividends amount must not be provided",
                    LongDescription = @"Where Country of residence when the dividend was paid or credited is not ""Australia"", ""Cocos (Keeling) Islands"", ""Christmas Island"", ""Heard Island and McDonald Islands"" or ""Norfolk Island"", Total exploration credits from dividends amount must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000398"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS348", Value = GetValueOrEmpty(dividends.INCDTLS348) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS375", Value = GetValueOrEmpty(dividends.INCDTLS375) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000399
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000399(INCDTLS2026.Rp_Dividends dividends, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000399
                    Where Country of residence when the dividend was paid or credited is not "Australia", "Cocos (Keeling) Islands", "Christmas Island", "Heard Island and McDonald Islands" or "Norfolk Island", Total listed investment company capital gain deduction from dividends amount must not be provided
    
                    Legacy Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS376 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS348 <> NULL AND NotInSet(^INCDTLS348, '"au", "cc", "cx", "nf", "hm"') AND ^INCDTLS376 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS348 = INCDTLS:Rp:Dividends:AddressDetails.Country.Code
            
                    ^INCDTLS376 = INCDTLS:Rp:Dividends:DividendTotal:Capital.Gains.Total.Amount
                    */
            assertion = (dividends.INCDTLS348 != null && !(IsMatch(dividends.INCDTLS348, @"^(au|cc|cx|nf|hm)$", RegexOptions.IgnoreCase)) && dividends.INCDTLS376 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000399",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total listed investment company capital gain deduction amount must not be provided",
                    LongDescription = @"Where Country of residence when the dividend was paid or credited is not ""Australia"", ""Cocos (Keeling) Islands"", ""Christmas Island"", ""Heard Island and McDonald Islands"" or ""Norfolk Island"", Total listed investment company capital gain deduction from dividends amount must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:DividendsCollection/tns:Dividends" + OccurrenceIndex(dividends.OccurrenceIndex) + "/tns:AddressDetailsCountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000399"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS348", Value = GetValueOrEmpty(dividends.INCDTLS348) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS376", Value = GetValueOrEmpty(dividends.INCDTLS376) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000413
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000413(INCDTLS2026.Rp_ForeignIncome foreignIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000413
                    Exempt foreign income amount is required where Exempt foreign income lump sum in arrears amount is present
    
                    Legacy Rule Format:
                    (^INCDTLS193 = NULL OR ^INCDTLS193 = 0) AND (^INCDTLS195 <> NULL)
        
                    Technical Business Rule Format:
                    (^INCDTLS193 = NULL OR ^INCDTLS193 = 0) AND (^INCDTLS195 <> NULL)
            
                    Data Elements:
            
                    ^INCDTLS193 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome:InternationalDealings.ExemptForeignEmploymentIncome.Amount
            
                    ^INCDTLS195 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome:Income.LumpSumArrearsPayment.Amount
                    */
            assertion = ((foreignIncome.INCDTLS193 == null || foreignIncome.INCDTLS193 == 0) && foreignIncome.INCDTLS195 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt foreign income amount must be provided",
                    LongDescription = @"Exempt foreign income amount is required where Exempt foreign income lump sum in arrears amount is present",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignIncomeCollection/tns:ForeignIncome" + OccurrenceIndex(foreignIncome.OccurrenceIndex) + "/tns:ExemptForeignIncome/tns:InternationalDealingsExemptForeignEmploymentIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000413"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS193", Value = GetValueOrEmpty(foreignIncome.INCDTLS193) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS195", Value = GetValueOrEmpty(foreignIncome.INCDTLS195) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000415
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000415(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000415
                    When Interest adjustment reason is "Other", Interest adjustment reason description must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS534 = "Other" AND ^INCDTLS535 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS534 = 'Other' AND ^INCDTLS535 = NULL
            
                    Data Elements:
            
                    ^INCDTLS535 = INCDTLS:Rp:Interest:Income.Interest.AdjustmentReason.Description
            
                    ^INCDTLS534 = INCDTLS:Rp:Interest:Income.Interest.AdjustmentReason.Code
                    */
            assertion = (interest.INCDTLS534 == @"Other" && interest.INCDTLS535 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000415",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest adjustment reason description must be provided",
                    LongDescription = @"When Interest adjustment reason is ""Other"", Interest adjustment reason description must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:IncomeAdjustmentReasonDe",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000415"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS534", Value = GetValueOrEmpty(interest.INCDTLS534) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS535", Value = GetValueOrEmpty(interest.INCDTLS535) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000416
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000416(INCDTLS2026.Rp_Interest interest, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000416
                    When the Interest adjustment reason description is present, Interest adjustment reason must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS535 <> NULLORBLANK AND ^INCDTLS534 = NULL)
        
                    Technical Business Rule Format:
                    ^INCDTLS535 <> BLANK AND ^INCDTLS534 = NULL
            
                    Data Elements:
            
                    ^INCDTLS534 = INCDTLS:Rp:Interest:Income.Interest.AdjustmentReason.Code
            
                    ^INCDTLS535 = INCDTLS:Rp:Interest:Income.Interest.AdjustmentReason.Description
                    */
            assertion = (string.IsNullOrWhiteSpace(interest.INCDTLS535) != true && interest.INCDTLS534 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest adjustment reason must be provided",
                    LongDescription = @"When the Interest adjustment reason description is present, Interest adjustment reason must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:InterestCollection/tns:Interest" + OccurrenceIndex(interest.OccurrenceIndex) + "/tns:IncomeAdjustmentReasonC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000416"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS535", Value = GetValueOrEmpty(interest.INCDTLS535) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS534", Value = GetValueOrEmpty(interest.INCDTLS534) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000418
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000418(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000418
                    Where an amount greater than 0 is present at Tax withheld where Australian business number not quoted from trusts; Primary production net income from trusts or Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided
    
                    Legacy Rule Format:
                    ^INCDTLS260 > 0 AND ^INCDTLS247 = NULL AND ^INCDTLS250 = NULL
        
                    Technical Business Rule Format:
                    ^INCDTLS260 > 0 AND ^INCDTLS247 = NULL AND ^INCDTLS250 = NULL
            
                    Data Elements:
            
                    ^INCDTLS260 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^INCDTLS247 = INCDTLS:Rp:TrustIncome:TrustPrimaryProduction:Income.TrustPrimaryProductionNetShare.Amount
            
                    ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
                    */
            assertion = (trustIncome.INCDTLS260.GetValueOrDefault() > 0 && trustIncome.INCDTLS247 == null && trustIncome.INCDTLS250 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.001418",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP or Non-PP income distribution net income from trusts amount must be provided",
                    LongDescription = @"Where an amount greater than 0 is present at Tax withheld where Australian business number not quoted from trusts; Primary production net income from trusts or Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustCreditsIncomeTaxOffsets/tns:PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000418"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS260", Value = GetValueOrEmpty(trustIncome.INCDTLS260) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS247", Value = GetValueOrEmpty(trustIncome.INCDTLS247) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = GetValueOrEmpty(trustIncome.INCDTLS250) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000419
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000419()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000419
            Where Name of distributing trust is present, then Primary production net income from trusts or Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) or Franking credit from franked dividends from trusts or TFN amounts withheld from interest, dividends and unit trust distributions from trusts or Australian franking credits from a New Zealand company or Capital gains or Share of credit for foreign resident capital gains withholding amounts or Attributed foreign income or Other assessable foreign source income or Foreign income tax offset must be provided
    
            Legacy Rule Format:
            AnyOccurrence(^INCDTLS245, ^INCDTLS539 <> NULL AND ^INCDTLS247 = NULL AND ^INCDTLS250 = NULL AND ^INCDTLS261 = NULL AND ^INCDTLS262 = NULL AND ^INCDTLS546 = NULL AND ^INCDTLS552 = NULL AND ^INCDTLS561 = NULL AND ^INCDTLS562 = NULL AND ^INCDTLS563 = NULL AND ^INCDTLS564 = NULL)

            Technical Business Rule Format:
            AnyOccurrence(^INCDTLS245, ^INCDTLS539 <> NULL AND ^INCDTLS247 = NULL AND ^INCDTLS250 = NULL AND ^INCDTLS261 = NULL AND ^INCDTLS262 = NULL AND ^INCDTLS546 = NULL AND ^INCDTLS552 = NULL AND ^INCDTLS561 = NULL AND ^INCDTLS562 = NULL AND ^INCDTLS563 = NULL AND ^INCDTLS564 = NULL)
    
            Data Elements:
    
            ^INCDTLS539 = INCDTLS:Rp:TrustIncome:DistributingTrust:OrganisationNameDetails.OrganisationalName.Text
    
            ^INCDTLS245 = INCDTLS:Rp:TrustIncome
    
            ^INCDTLS247 = INCDTLS:Rp:TrustIncome:TrustPrimaryProduction:Income.TrustPrimaryProductionNetShare.Amount
    
            ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
    
            ^INCDTLS261 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.FrankingCreditsShare.Amount
    
            ^INCDTLS262 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
    
            ^INCDTLS546 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^INCDTLS552 = INCDTLS:Rp:TrustIncome:TrustCapitalGains:Income.CapitalGainsNet.Amount
    
            ^INCDTLS561 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
            ^INCDTLS562 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:IncomeTax.AttributedForeignIncome.Amount
    
            ^INCDTLS563 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:Income.InternationalDealings.Net.Amount
    
            ^INCDTLS564 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:Income.InternationalDealings.TaxOffset.Amount
            */
            assertion = (report.Rp_TrustIncomeCollection == null ? false : report.Rp_TrustIncomeCollection.Any(INCDTLS245Repeat => INCDTLS245Repeat.INCDTLS539 != null && INCDTLS245Repeat.INCDTLS247 == null && INCDTLS245Repeat.INCDTLS250 == null && INCDTLS245Repeat.INCDTLS261 == null && INCDTLS245Repeat.INCDTLS262 == null && INCDTLS245Repeat.INCDTLS546 == null && INCDTLS245Repeat.INCDTLS552 == null && INCDTLS245Repeat.INCDTLS561 == null && INCDTLS245Repeat.INCDTLS562 == null && INCDTLS245Repeat.INCDTLS563 == null && INCDTLS245Repeat.INCDTLS564 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.001419",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust distribution amounts must be provided",
                    LongDescription = @"Where Name of distributing trust is present, then Primary production net income from trusts or Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) or Franking credit from franked dividends from trusts or TFN amounts withheld from interest, dividends and unit trust distributions from trusts or Australian franking credits from a New Zealand company or Capital gains or Share of credit for foreign resident capital gains withholding amounts or Attributed foreign income or Other assessable foreign source income or Foreign income tax offset must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome/tns:DistributingTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000419"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS539", Value = "INCDTLS539" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS247", Value = "INCDTLS247" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = "INCDTLS250" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS261", Value = "INCDTLS261" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS262", Value = "INCDTLS262" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS546", Value = "INCDTLS546" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS552", Value = "INCDTLS552" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS561", Value = "INCDTLS561" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS562", Value = "INCDTLS562" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS563", Value = "INCDTLS563" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS564", Value = "INCDTLS564" });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000420
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000420(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000420
                    When a positive amount is present at Franking credit from franked dividends from trusts or TFN amounts withheld from interest, dividends and unit trust distributions from trusts then Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided
    
                    Legacy Rule Format:
                    (^INCDTLS261 > 0 OR ^INCDTLS262 > 0) AND ^INCDTLS250 = NULL
        
                    Technical Business Rule Format:
                    (^INCDTLS261 > 0 OR ^INCDTLS262 > 0) AND ^INCDTLS250 = NULL
            
                    Data Elements:
            
                    ^INCDTLS250 = INCDTLS:Rp:TrustIncome:TrustNonPrimaryProduction:Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount
            
                    ^INCDTLS261 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.FrankingCreditsShare.Amount
            
                    ^INCDTLS262 = INCDTLS:Rp:TrustIncome:TrustCreditsIncomeTaxOffsets:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount
                    */
            assertion = ((trustIncome.INCDTLS261.GetValueOrDefault() > 0 || trustIncome.INCDTLS262.GetValueOrDefault() > 0) && trustIncome.INCDTLS250 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.001420",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-primary production net income from trusts must be provided",
                    LongDescription = @"When a positive amount is present at Franking credit from franked dividends from trusts or TFN amounts withheld from interest, dividends and unit trust distributions from trusts then Non-primary production net income from trusts (less capital gains, foreign income and franked distributions) must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:TrustNonPrimaryProduction/tns:TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000420"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS261", Value = GetValueOrEmpty(trustIncome.INCDTLS261) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS262", Value = GetValueOrEmpty(trustIncome.INCDTLS262) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS250", Value = GetValueOrEmpty(trustIncome.INCDTLS250) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000421
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000421(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000421
                    Australian Business Number must pass the ABN algorithm check.
    
                    Legacy Rule Format:
                    ^INCDTLS537 <> NULL AND FailsABNAlgorithm(^INCDTLS537)
        
                    Technical Business Rule Format:
                    ^INCDTLS537 <> NULL AND FailsABNAlgorithm(^INCDTLS537)
            
                    Data Elements:
            
                    ^INCDTLS537 = INCDTLS:Rp:TrustIncome:DistributingTrust:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = (trustIncome.INCDTLS537 != null && FailsABNAlgorithm(trustIncome.INCDTLS537));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000421",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian Business Number has failed the algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:DistributingTrust/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000421"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS537", Value = GetValueOrEmpty(trustIncome.INCDTLS537) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000422
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000422(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000422
                    Australian Company Number must pass the ACN algorithm check.
    
                    Legacy Rule Format:
                    ^INCDTLS538 <> NULL AND FailsACNAlgorithm(^INCDTLS538)
        
                    Technical Business Rule Format:
                    ^INCDTLS538 <> NULL AND FailsACNAlgorithm(^INCDTLS538)
            
                    Data Elements:
            
                    ^INCDTLS538 = INCDTLS:Rp:TrustIncome:DistributingTrust:Identifiers.AustralianCompanyNumber.Identifier
                    */
            assertion = (trustIncome.INCDTLS538 != null && FailsACNAlgorithm(trustIncome.INCDTLS538));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000422",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian Company Number has failed the algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:DistributingTrust/tns:AustralianCompanyNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000422"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS538", Value = GetValueOrEmpty(trustIncome.INCDTLS538) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000423
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000423()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000423
            Capital gains must be within +/- $5 of the total of gross capital gain minus the sum of capital losses applied, CGT discounts applied and CGT small business concessions applied
    
            Legacy Rule Format:
            AnyOccurrence(^INCDTLS245, OutsideRange(^INCDTLS552, ^INCDTLS553 - ^INCDTLS554 - ^INCDTLS555 - ^INCDTLS556, 5))

            Technical Business Rule Format:
            AnyOccurrence(^INCDTLS245, OutsideRange(^INCDTLS552, ^INCDTLS553 - ^INCDTLS554 - ^INCDTLS555 - ^INCDTLS556, 5))
    
            Data Elements:
    
            ^INCDTLS552 = INCDTLS:Rp:TrustIncome:TrustCapitalGains:Income.CapitalGainsNet.Amount
    
            ^INCDTLS245 = INCDTLS:Rp:TrustIncome
    
            ^INCDTLS553 = INCDTLS:Rp:TrustIncome:TrustCapitalGains:Capital.Gains.Total.Amount
    
            ^INCDTLS554 = INCDTLS:Rp:TrustIncome:TrustCapitalGains:Capital.Losses.Total.Amount
    
            ^INCDTLS555 = INCDTLS:Rp:TrustIncome:TrustCapitalGains:TaxConcession.CapitalGains.DiscountTotal.Amount
    
            ^INCDTLS556 = INCDTLS:Rp:TrustIncome:TrustCapitalGains:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount
            */
            assertion = (report.Rp_TrustIncomeCollection == null ? false : report.Rp_TrustIncomeCollection.Any(INCDTLS245Repeat => OutsideRange(INCDTLS245Repeat.INCDTLS552.GetValueOrDefault(), INCDTLS245Repeat.INCDTLS553.GetValueOrDefault() - INCDTLS245Repeat.INCDTLS554.GetValueOrDefault() - INCDTLS245Repeat.INCDTLS555.GetValueOrDefault() - INCDTLS245Repeat.INCDTLS556.GetValueOrDefault(), 5)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains amount is incorrect",
                    LongDescription = @"Capital gains must be within +/- $5 of the total of gross capital gain minus the sum of capital losses applied, CGT discounts applied and CGT small business concessions applied",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome/tns:TrustCapitalGains/tns:CapitalGainsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000423"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS552", Value = "Count(INCDTLS552)" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS553", Value = "Count(INCDTLS553)" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS554", Value = "Count(INCDTLS554)" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS555", Value = "Count(INCDTLS555)" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS556", Value = "Count(INCDTLS556)" });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000424
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000424(INCDTLS2026.Rp_TrustIncome trustIncome, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000424
                    When an amount greater than 0 is present at Foreign income tax offset, Attributed foreign income or Other assessable foreign source income must be provided
    
                    Legacy Rule Format:
                    ^INCDTLS564 > 0 AND ^INCDTLS562 = NULL AND ^INCDTLS563 = NULL
        
                    Technical Business Rule Format:
                    ^INCDTLS564 > 0 AND ^INCDTLS562 = NULL AND ^INCDTLS563 = NULL
            
                    Data Elements:
            
                    ^INCDTLS562 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:IncomeTax.AttributedForeignIncome.Amount
            
                    ^INCDTLS563 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS564 = INCDTLS:Rp:TrustIncome:ForeignAndNonResident:Income.InternationalDealings.TaxOffset.Amount
                    */
            assertion = (trustIncome.INCDTLS564.GetValueOrDefault() > 0 && trustIncome.INCDTLS562 == null && trustIncome.INCDTLS563 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income or Other assessable foreign source income must be provided",
                    LongDescription = @"When an amount greater than 0 is present at Foreign income tax offset, Attributed foreign income or Other assessable foreign source income must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TrustIncomeCollection/tns:TrustIncome" + OccurrenceIndex(trustIncome.OccurrenceIndex) + "/tns:ForeignAndNonResident/tns:IncomeTaxAttributedForeignIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000424"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS564", Value = GetValueOrEmpty(trustIncome.INCDTLS564) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS562", Value = GetValueOrEmpty(trustIncome.INCDTLS562) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS563", Value = GetValueOrEmpty(trustIncome.INCDTLS563) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000425
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000425(INCDTLS2026.Rp_Income income, INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000425
                            Where the Australian government benefit type is 'Special', then an Australian government benefit adjustment reason and Australian government benefit adjustment reason description must not be provided
            
                            Legacy Rule Format:
                            (^INCDTLS126 = "Special" AND (^INCDTLS131 <> NULL OR ^INCDTLS132 <> NULL))
                
                            Technical Business Rule Format:
                            (^INCDTLS126 = 'Special' AND (^INCDTLS131 <> NULL OR ^INCDTLS132 <> NULL))
                    
                            Data Elements:
                    
                            ^INCDTLS131 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.AdjustmentReason.Code
                    
                            ^INCDTLS126 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.Type.Code
                    
                            ^INCDTLS132 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.AdjustmentReason.Description
                            */
            assertion = (governmentBenefit.INCDTLS126 == @"Special" && (governmentBenefit.INCDTLS131 != null || governmentBenefit.INCDTLS132 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000425",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian government benefit adjustment reason or Australian government benefit adjustment reason description must not be provided",
                    LongDescription = @"Where the Australian government benefit type is 'Special', then an Australian government benefit adjustment reason and Australian government benefit adjustment reason description must not be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:GovernmentBenefitCollection/tns:GovernmentBenefit" + OccurrenceIndex(governmentBenefit.OccurrenceIndex) + "/tns:PaymentAdjustmentReasonC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000425"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS126", Value = governmentBenefit.INCDTLS126 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS131", Value = GetValueOrEmpty(governmentBenefit.INCDTLS131) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS132", Value = GetValueOrEmpty(governmentBenefit.INCDTLS132) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000426
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000426(INCDTLS2026.Rp_Income income, INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000426
                            When the Australian government benefit adjustment reason description is present, Australian government benefit adjustment reason must be provided
            
                            Legacy Rule Format:
                            (^INCDTLS131 = NULL AND ^INCDTLS132 <> NULL)
                
                            Technical Business Rule Format:
                            ^INCDTLS131 = NULL AND ^INCDTLS132 <> NULL
                    
                            Data Elements:
                    
                            ^INCDTLS131 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.AdjustmentReason.Code
                    
                            ^INCDTLS132 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.AdjustmentReason.Description
                            */
            assertion = (governmentBenefit.INCDTLS131 == null && governmentBenefit.INCDTLS132 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian government benefit adjustment reason must be provided",
                    LongDescription = @"When the Australian government benefit adjustment reason description is present, Australian government benefit adjustment reason must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:GovernmentBenefitCollection/tns:GovernmentBenefit" + OccurrenceIndex(governmentBenefit.OccurrenceIndex) + "/tns:PaymentAdjustmentReasonC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000426"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS131", Value = GetValueOrEmpty(governmentBenefit.INCDTLS131) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS132", Value = GetValueOrEmpty(governmentBenefit.INCDTLS132) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000427
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000427(INCDTLS2026.Rp_Income income, INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000427
                            Where Australian government benefit adjustment reason is 'Other', an Australian government benefit adjustment reason description must be provided
            
                            Legacy Rule Format:
                            (^INCDTLS131 = "Other" AND ^INCDTLS132 = NULL)
                
                            Technical Business Rule Format:
                            ^INCDTLS131 = 'Other' AND ^INCDTLS132 = NULL
                    
                            Data Elements:
                    
                            ^INCDTLS132 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.AdjustmentReason.Description
                    
                            ^INCDTLS131 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.AdjustmentReason.Code
                            */
            assertion = (governmentBenefit.INCDTLS131 == @"Other" && governmentBenefit.INCDTLS132 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian government benefit adjustment reason description must be provided",
                    LongDescription = @"Where Australian government benefit adjustment reason is 'Other', a Australian government benefit adjustment reason description must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:GovernmentBenefitCollection/tns:GovernmentBenefit" + OccurrenceIndex(governmentBenefit.OccurrenceIndex) + "/tns:PaymentAdjustmentReasonDe",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000427"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS131", Value = GetValueOrEmpty(governmentBenefit.INCDTLS131) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS132", Value = GetValueOrEmpty(governmentBenefit.INCDTLS132) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000428
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000428(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000428
                    Where Total assessable foreign income from managed fund is provided, the Total other foreign income from managed fund or Total net foreign rent from managed fund must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS398 > 0 AND (^INCDTLS399 = NULL AND ^INCDTLS413 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS398 > 0 AND (^INCDTLS399 = NULL AND ^INCDTLS413 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS398 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS398.GetValueOrDefault() > 0 && (managedFunds.INCDTLS399 == null && managedFunds.INCDTLS413 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000428",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total other foreign income from managed fund or Total net foreign rent from managed fund must be provided",
                    LongDescription = @"Total other foreign income from managed fund or Total net foreign rent from managed fund must be provided when Total assessable foreign income from managed fund is provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:InternationalDealingsForeignIncomeGrossA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000428"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS398", Value = GetValueOrEmpty(managedFunds.INCDTLS398) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000429
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000429(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000429
                    Total assessable foreign income from managed fund must be greater than or equal to the sum of Total other foreign income from managed fund and Total net foreign rent from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS398 < (^INCDTLS399 + ^INCDTLS413))
        
                    Technical Business Rule Format:
                    (^INCDTLS398 < (^INCDTLS399 + ^INCDTLS413))
            
                    Data Elements:
            
                    ^INCDTLS399 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS398 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS413 = INCDTLS:Rp:ManagedFunds:ManagedFundsTotal:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS398.GetValueOrDefault() < managedFunds.INCDTLS399.GetValueOrDefault() + managedFunds.INCDTLS413.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000429",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total other foreign income from managed fund or Total net foreign rent from managed fund is incorrect",
                    LongDescription = @"The sum of Total other foreign income from managed fund and Total net foreign rent from managed fund must not exceed Total assessable foreign income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsTotal/tns:IncomeInternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000429"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS398", Value = GetValueOrEmpty(managedFunds.INCDTLS398) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS399", Value = GetValueOrEmpty(managedFunds.INCDTLS399) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS413", Value = GetValueOrEmpty(managedFunds.INCDTLS413) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000430
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000430(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000430
                    Where Your share of assessable foreign income from managed fund is provided, the Your share of other foreign income from managed fund or Your share of net foreign rent from managed fund must also be provided
    
                    Legacy Rule Format:
                    (^INCDTLS428 > 0 AND (^INCDTLS429 = NULL AND ^INCDTLS441 = NULL))
        
                    Technical Business Rule Format:
                    (^INCDTLS428 > 0 AND (^INCDTLS429 = NULL AND ^INCDTLS441 = NULL))
            
                    Data Elements:
            
                    ^INCDTLS428 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS428.GetValueOrDefault() > 0 && (managedFunds.INCDTLS429 == null && managedFunds.INCDTLS441 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000430",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other foreign income from managed fund or your share of net foreign rent from managed fund must be provided",
                    LongDescription = @"Your share of other foreign income from managed fund or Your share of net foreign rent from managed fund must be provided when Your share of assessable foreign income from managed fund is provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:InternationalDealingsForeignIncomeGrossA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000430"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS428", Value = GetValueOrEmpty(managedFunds.INCDTLS428) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000431
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000431(INCDTLS2026.Rp_ManagedFunds managedFunds, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000431
                    Your share of assessable foreign income from managed fund must be greater than or equal to the sum of Your share of other foreign income from managed fund and Your share of net foreign rent from managed fund
    
                    Legacy Rule Format:
                    (^INCDTLS428 < (^INCDTLS429 + ^INCDTLS441))
        
                    Technical Business Rule Format:
                    (^INCDTLS428 < (^INCDTLS429 + ^INCDTLS441))
            
                    Data Elements:
            
                    ^INCDTLS429 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:Income.InternationalDealings.Net.Amount
            
                    ^INCDTLS428 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS441 = INCDTLS:Rp:ManagedFunds:ManagedFundsYourShare:InternationalDealings.ForeignRent.Amount
                    */
            assertion = (managedFunds.INCDTLS428.GetValueOrDefault() < managedFunds.INCDTLS429.GetValueOrDefault() + managedFunds.INCDTLS441.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000431",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of other foreign income from managed fund or Your share of net foreign rent from managed fund is incorrect",
                    LongDescription = @"The sum of Your share of other foreign income from managed fund and Your share of net foreign rent from managed fund must not exceed Your share of assessable foreign income from managed fund",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ManagedFundsCollection/tns:ManagedFunds" + OccurrenceIndex(managedFunds.OccurrenceIndex) + "/tns:ManagedFundsYourShare/tns:IncomeInternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.000431"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS428", Value = GetValueOrEmpty(managedFunds.INCDTLS428) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS429", Value = GetValueOrEmpty(managedFunds.INCDTLS429) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS441", Value = GetValueOrEmpty(managedFunds.INCDTLS441) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.000433
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000433(INCDTLS2026.Rp_TaxablePayment taxablePayment, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.000433
                    Taxable payments payer's australian business number is invalid

                    Legacy Rule Format:
                    (FailsABNAlgorithm(^INCDTLS573))

                    Technical Business Rule Format:
                    (FailsABNAlgorithm(^INCDTLS573))

                    Data Elements:

                    ^INCDTLS573 = INCDTLS:Rp:TaxablePaymentCollection:TaxablePayment:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = (taxablePayment.INCDTLS573 != null && FailsABNAlgorithm(taxablePayment.INCDTLS573));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000433",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable payments payer's australian business number is invalid",
                    LongDescription = @"The Taxable payments payer's australian business number has failed the ABN algorithm check",
                    Location = "/tns:INCDTLS/tns:Rp/tns:TaxablePaymentCollection/tns:Taxablepayment" + OccurrenceIndex(taxablePayment.OccurrenceIndex) + "/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
            {
                new ProcessMessageParameter()
                {
                    Name = "RuleIdentifier",
                    Value = "VR.ATO.INCDTLS.000433"
                }
            },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS573", Value = GetValueOrEmpty(taxablePayment.INCDTLS573) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.W00012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLSW00012(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.W00012
                    The Superannuation income stream tax withheld amount is greater than or equal to the Superannuation income stream taxable and tax free components. Check and correct if necessary.
    
                    Legacy Rule Format:
                    (^INCDTLS112 > 0 AND ^INCDTLS112 >= (^INCDTLS113 + ^INCDTLS114 + ^INCDTLS115 + ^INCDTLS117 + ^INCDTLS118 + ^INCDTLS119))
        
                    Technical Business Rule Format:
                    (^INCDTLS112 > 0 AND ^INCDTLS112 >= (^INCDTLS113 + ^INCDTLS114 + ^INCDTLS115 + ^INCDTLS117 + ^INCDTLS118 + ^INCDTLS119))
            
                    Data Elements:
            
                    ^INCDTLS112 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount
            
                    ^INCDTLS113 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS114 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount
            
                    ^INCDTLS115 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:IncomeTax.Superannuation.TaxFreeComponent.Amount
            
                    ^INCDTLS117 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS118 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount
            
                    ^INCDTLS119 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.LumpSumArrearsTaxFree.Amount
                    */
            assertion = (income.INCDTLS112.GetValueOrDefault() > 0 && income.INCDTLS112.GetValueOrDefault() >= income.INCDTLS113.GetValueOrDefault() + income.INCDTLS114.GetValueOrDefault() + income.INCDTLS115.GetValueOrDefault() + income.INCDTLS117.GetValueOrDefault() + income.INCDTLS118.GetValueOrDefault() + income.INCDTLS119.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.W00012",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Superannuation income stream tax withheld amount",
                    LongDescription = @"The Superannuation income stream tax withheld amount is greater than or equal to the Superannuation income stream taxable and tax free components. Check and correct if necessary.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.W00012"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS112", Value = GetValueOrEmpty(income.INCDTLS112) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS113", Value = GetValueOrEmpty(income.INCDTLS113) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS114", Value = GetValueOrEmpty(income.INCDTLS114) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS115", Value = GetValueOrEmpty(income.INCDTLS115) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS117", Value = GetValueOrEmpty(income.INCDTLS117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS118", Value = GetValueOrEmpty(income.INCDTLS118) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS119", Value = GetValueOrEmpty(income.INCDTLS119) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.W00121
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLSW00121(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.W00121
                    Salary or wages tax withheld amount is greater than or equal to Salary or wages gross amount. Check and correct if necessary.
    
                    Legacy Rule Format:
                    (^INCDTLS311 > 0) AND (^INCDTLS311 >= ^INCDTLS312)
        
                    Technical Business Rule Format:
                    (^INCDTLS311 > 0) AND (^INCDTLS311 >= ^INCDTLS312)
            
                    Data Elements:
            
                    ^INCDTLS311 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSalaryOrWages.Amount
            
                    ^INCDTLS312 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.SalaryOrWages.Amount
                    */
            assertion = (income.INCDTLS311.GetValueOrDefault() > 0 && income.INCDTLS311.GetValueOrDefault() >= income.INCDTLS312.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.W00121",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check Salary or wages tax withheld amount and Salary or wages gross amount",
                    LongDescription = @"Salary or wages tax withheld amount is greater than or equal to Salary or wages gross amount. Check and correct if necessary.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldSalaryOrWagesA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.W00121"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS311", Value = GetValueOrEmpty(income.INCDTLS311) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS312", Value = GetValueOrEmpty(income.INCDTLS312) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.W00128
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLSW00128(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.W00128
                    SLS tax withheld amount is greater than 50% of the combined SLS taxed element and SLS untaxed. Check and correct if necessary.
    
                    Legacy Rule Format:
                    (^INCDTLS329 > (0.5 * (^INCDTLS330 + ^INCDTLS331)))
        
                    Technical Business Rule Format:
                    (^INCDTLS329 > (0.5 * (^INCDTLS330 + ^INCDTLS331)))
            
                    Data Elements:
            
                    ^INCDTLS329 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSuperannuationLumpSumPayment.Amount
            
                    ^INCDTLS330 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment:Income.Superannuation.LumpSumPaymentTaxableComponentTaxedElement.Amount
            
                    ^INCDTLS331 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment:Income.Superannuation.LumpSumPaymentTaxableComponentUntaxedElement.Amount
                    */
            assertion = (income.INCDTLS329.GetValueOrDefault() > 0.5M * (income.INCDTLS330.GetValueOrDefault() + income.INCDTLS331.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.W00128",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check SLS tax withheld amount",
                    LongDescription = @"SLS tax withheld amount is greater than 50% of the combined SLS taxed element and SLS untaxed. Check and correct if necessary.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:SuperannuationLumpSumPayment/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldSuperannuationLumpSumPaymentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.W00128"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS329", Value = GetValueOrEmpty(income.INCDTLS329) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS330", Value = GetValueOrEmpty(income.INCDTLS330) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS331", Value = GetValueOrEmpty(income.INCDTLS331) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.W00304
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLSW00304(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.W00304
                    The ETP tax withheld amount exceeds 50% of the ETP Taxable component. Check and correct if necessary.
    
                    Legacy Rule Format:
                    (^INCDTLS139 > ^INCDTLS140 * 0.50)
        
                    Technical Business Rule Format:
                    ^INCDTLS139 > ^INCDTLS140 * 0.50
            
                    Data Elements:
            
                    ^INCDTLS139 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmploymentTerminationPayment.Amount
            
                    ^INCDTLS140 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:Income.EmploymentTerminationPaymentTaxable.Amount
                    */
            assertion = (income.INCDTLS139.GetValueOrDefault() > income.INCDTLS140.GetValueOrDefault() * 0.50M);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.W00304",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check ETP tax withheld amount",
                    LongDescription = @"The ETP tax withheld amount exceeds 50% of the ETP Taxable component. Check and correct if necessary.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:EmploymentTerminationPayment/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.W00304"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS139", Value = GetValueOrEmpty(income.INCDTLS139) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS140", Value = GetValueOrEmpty(income.INCDTLS140) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.W00312
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLSW00312(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.W00312
                    The Annuity tax withheld amount is greater than or equal to the result of the Annuity gross amount less Annuity deductible amount of undeducted purchase price. Check and correct if necessary.
    
                    Legacy Rule Format:
                    (^INCDTLS146 > 0 AND ^INCDTLS146 >= (^INCDTLS147 - ^INCDTLS148))
        
                    Technical Business Rule Format:
                    (^INCDTLS146 > 0 AND ^INCDTLS146 >= (^INCDTLS147 - ^INCDTLS148))
            
                    Data Elements:
            
                    ^INCDTLS146 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount
            
                    ^INCDTLS147 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:Income.Insurance.LifePremiumDirectGross.Amount
            
                    ^INCDTLS148 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities:IncomeTax.Deduction.PensionOrAnnuityPurchasePriceUndeducted.Amount
                    */
            assertion = (income.INCDTLS146.GetValueOrDefault() > 0 && income.INCDTLS146.GetValueOrDefault() >= income.INCDTLS147.GetValueOrDefault() - income.INCDTLS148.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.W00312",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check Annuity tax withheld amount",
                    LongDescription = @"The Annuity tax withheld amount is greater than or equal to the result of the Annuity gross amount less Annuity deductible amount of undeducted purchase price. Check and correct if necessary.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:AustralianAnnuities/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.W00312"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS146", Value = GetValueOrEmpty(income.INCDTLS146) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS147", Value = GetValueOrEmpty(income.INCDTLS147) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS148", Value = GetValueOrEmpty(income.INCDTLS148) });
                response.Add(processMessage);
            }
        }

        #endregion
        #region VR.ATO.INCDTLS.W00342
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLSW00342(INCDTLS2026.Rp_Income income, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.INCDTLS.W00342
                    The Foreign employment income tax withheld amount is greater than or equal to the sum of Foreign employment income gross amount, Foreign employment income lump sum A and Foreign employment income lump sum E. Check and correct if necessary.
    
                    Legacy Rule Format:
                    (^INCDTLS167 > 0) AND (^INCDTLS167 >= (^INCDTLS168 + ^INCDTLS174 + ^INCDTLS177))
        
                    Technical Business Rule Format:
                    (^INCDTLS167 > 0) AND (^INCDTLS167 >= (^INCDTLS168 + ^INCDTLS174 + ^INCDTLS177))
            
                    Data Elements:
            
                    ^INCDTLS167 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignEmploymentIncome.Amount
            
                    ^INCDTLS168 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:InternationalDealings.ForeignIncomeGross.Amount
            
                    ^INCDTLS174 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^INCDTLS177 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.BackPaymentLumpSumE.Amount
                    */
            assertion = (income.INCDTLS167.GetValueOrDefault() > 0 && income.INCDTLS167.GetValueOrDefault() >= income.INCDTLS168.GetValueOrDefault() + income.INCDTLS174.GetValueOrDefault() + income.INCDTLS177.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.W00342",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Check Foreign employment income tax withheld amount",
                    LongDescription = @"The Foreign employment income tax withheld amount is greater than or equal to the sum of Foreign employment income gross amount, Foreign employment income lump sum A and Foreign employment income lump sum E. Check and correct if necessary.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(income.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldForeignEmploymentIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.INCDTLS.W00342"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS167", Value = GetValueOrEmpty(income.INCDTLS167) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS168", Value = GetValueOrEmpty(income.INCDTLS168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS174", Value = GetValueOrEmpty(income.INCDTLS174) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "INCDTLS177", Value = GetValueOrEmpty(income.INCDTLS177) });
                response.Add(processMessage);
            }
        }
        #endregion
    }
}