using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationRNTLPRPTY
{

    public partial class RNTLPRPTY2019Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="RNTLPRPTY2019Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public RNTLPRPTY2019Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() != 1);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());

        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public RNTLPRPTY2019 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(RNTLPRPTY2019 report)
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            #region Manual Rules - Rules marked for manual coding

            #region VR.ATO.RNTLPRPTY.000001

            /*  VR.ATO.RNTLPRPTY.000001
                When the Ownership percentage is less than 100% and (total) Land tax is present, Your share of land tax must be provided

            Legacy Rule Format:
                (^RNTLPRPTY147 = NULL) AND (^RNTLPRPTY125 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
                (^RNTLPRPTY147 = NULL) AND (^RNTLPRPTY125 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

            ^RNTLPRPTY147 = RP.{PrptySeqNum}.Portion:Expense.LandTax.Amount

            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

            ^RNTLPRPTY125 = RP.{PrptySeqNum}.Total:Expense.LandTax.Amount
            */
            VRATORNTLPRPTY000001(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000001

            #region VR.ATO.RNTLPRPTY.000002

            /*  VR.ATO.RNTLPRPTY.000002
                        When Your share of land tax is provided, the amount must not exceed (total) Land tax

                    Legacy Rule Format:
                        (^RNTLPRPTY147 <> NULL) AND (^RNTLPRPTY125 = NULL OR ^RNTLPRPTY147 > ^RNTLPRPTY125)

                Technical Business Rule Format:
                        (^RNTLPRPTY147 <> NULL) AND (^RNTLPRPTY125 = NULL OR ^RNTLPRPTY147 > ^RNTLPRPTY125)

            Data Elements:

                ^RNTLPRPTY147 = RP.{PrptySeqNum}.Portion:Expense.LandTax.Amount

                ^RNTLPRPTY125 = RP.{PrptySeqNum}.Total:Expense.LandTax.Amount
                */
            VRATORNTLPRPTY000002(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000002
            
            #region VR.ATO.RNTLPRPTY.000003

            /*  VR.ATO.RNTLPRPTY.000003
            When the Ownership percentage is less than 100% and (total) Legal fees is present, Your share of legal fees must be provided

        Legacy Rule Format:
            (^RNTLPRPTY148 = NULL) AND (^RNTLPRPTY126 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

    Technical Business Rule Format:
            (^RNTLPRPTY148 = NULL) AND (^RNTLPRPTY126 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

Data Elements:
    
    ^RNTLPRPTY148 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LegalFees.Amount
    
    ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
    ^RNTLPRPTY126 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LegalFees.Amount
    */
            VRATORNTLPRPTY000003(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000003

            #region VR.ATO.RNTLPRPTY.000004

            /*  VR.ATO.RNTLPRPTY.000004
            When Your share of legal fees is provided, the amount must not exceed (total) Legal fees

        Legacy Rule Format:
            (^RNTLPRPTY148 <> NULL) AND (^RNTLPRPTY126 = NULL OR ^RNTLPRPTY148 > ^RNTLPRPTY126)

    Technical Business Rule Format:
            (^RNTLPRPTY148 <> NULL) AND (^RNTLPRPTY126 = NULL OR ^RNTLPRPTY148 > ^RNTLPRPTY126)

Data Elements:
    
    ^RNTLPRPTY148 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LegalFees.Amount
    
    ^RNTLPRPTY126 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LegalFees.Amount
    */
            VRATORNTLPRPTY000004(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000004

            #region VR.ATO.RNTLPRPTY.000005

            /*  VR.ATO.RNTLPRPTY.000005
            When the Ownership percentage is less than 100% and (total) Pest control is present, Your share of pest control must be provided

        Legacy Rule Format:
            (^RNTLPRPTY149 = NULL) AND (^RNTLPRPTY127 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

    Technical Business Rule Format:
            (^RNTLPRPTY149 = NULL) AND (^RNTLPRPTY127 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

Data Elements:
    
    ^RNTLPRPTY149 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.PestControl.Amount
    
    ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
    ^RNTLPRPTY127 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.PestControl.Amount
    */
            VRATORNTLPRPTY000005(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000005

            #region VR.ATO.RNTLPRPTY.000006

            /*  VR.ATO.RNTLPRPTY.000006
            When Your share of pest control is provided, the amount must not exceed (total) Pest control

        Legacy Rule Format:
            (^RNTLPRPTY149 <> NULL) AND (^RNTLPRPTY127 = NULL OR ^RNTLPRPTY149 > ^RNTLPRPTY127)

    Technical Business Rule Format:
            (^RNTLPRPTY149 <> NULL) AND (^RNTLPRPTY127 = NULL OR ^RNTLPRPTY149 > ^RNTLPRPTY127)

Data Elements:
    
    ^RNTLPRPTY149 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.PestControl.Amount
    
    ^RNTLPRPTY127 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.PestControl.Amount
    */
            VRATORNTLPRPTY000006(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000006

            #region VR.ATO.RNTLPRPTY.000007

            /*  VR.ATO.RNTLPRPTY.000007
                        When the Ownership percentage is less than 100% and (total) Property agent fees/commission is present, Your share of property agent fees/commission must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY150 = NULL) AND (^RNTLPRPTY128 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY150 = NULL) AND (^RNTLPRPTY128 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY150 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.AgentFeesCommission.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY128 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.AgentFeesCommission.Amount
                */
            VRATORNTLPRPTY000007(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000007

            #region VR.ATO.RNTLPRPTY.000008

            /*  VR.ATO.RNTLPRPTY.000008
                        When Your share of property agent fees/commission is provided, the amount must not exceed (total) Property agent fees/commission

                    Legacy Rule Format:
                        (^RNTLPRPTY150 <> NULL) AND (^RNTLPRPTY128 = NULL OR ^RNTLPRPTY150 > ^RNTLPRPTY128)

                Technical Business Rule Format:
                        (^RNTLPRPTY150 <> NULL) AND (^RNTLPRPTY128 = NULL OR ^RNTLPRPTY150 > ^RNTLPRPTY128)

            Data Elements:

                ^RNTLPRPTY150 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.AgentFeesCommission.Amount

                ^RNTLPRPTY128 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.AgentFeesCommission.Amount
                */
            VRATORNTLPRPTY000008(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000008

            #region VR.ATO.RNTLPRPTY.000009

            /*  VR.ATO.RNTLPRPTY.000009
                        When the Ownership percentage is less than 100% and (total) Repairs and maintenance is present, Your share of repairs and maintenance must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY151 = NULL) AND (^RNTLPRPTY129 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY151 = NULL) AND (^RNTLPRPTY129 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY151 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.RepairsAndMaintenance.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY129 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
                */
            VRATORNTLPRPTY000009(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000009

            #region VR.ATO.RNTLPRPTY.000010

            /*  VR.ATO.RNTLPRPTY.000010
                        When Your share of repairs and maintenance is provided, the amount must not exceed (total) Repairs and maintenance

                    Legacy Rule Format:
                        (^RNTLPRPTY151 <> NULL) AND (^RNTLPRPTY129 = NULL OR ^RNTLPRPTY151 > ^RNTLPRPTY129)

                Technical Business Rule Format:
                        (^RNTLPRPTY151 <> NULL) AND (^RNTLPRPTY129 = NULL OR ^RNTLPRPTY151 > ^RNTLPRPTY129)

            Data Elements:

                ^RNTLPRPTY151 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.RepairsAndMaintenance.Amount

                ^RNTLPRPTY129 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
                */
            VRATORNTLPRPTY000010(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000010
            
            #region VR.ATO.RNTLPRPTY.000011

            /*  VR.ATO.RNTLPRPTY.000011
                        When the Ownership percentage is less than 100% and (total) Stationery, telephone, and postage is present, Your share of stationery, telephone, and postage must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY153 = NULL) AND (^RNTLPRPTY131 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY153 = NULL) AND (^RNTLPRPTY131 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY153 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.OfficeSupplies.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY131 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.OfficeSupplies.Amount
                */
            VRATORNTLPRPTY000011(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000011

            #region VR.ATO.RNTLPRPTY.000012

            /*  VR.ATO.RNTLPRPTY.000012
                        When Your share of stationery, telephone, and postage is provided, the amount must not exceed (total) Stationery, telephone, and postage

                    Legacy Rule Format:
                        (^RNTLPRPTY153 <> NULL) AND (^RNTLPRPTY131 = NULL OR ^RNTLPRPTY153 > ^RNTLPRPTY131)

                Technical Business Rule Format:
                        (^RNTLPRPTY153 <> NULL) AND (^RNTLPRPTY131 = NULL OR ^RNTLPRPTY153 > ^RNTLPRPTY131)

            Data Elements:

                ^RNTLPRPTY153 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.OfficeSupplies.Amount

                ^RNTLPRPTY131 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.OfficeSupplies.Amount
                */
            VRATORNTLPRPTY000012(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000012
            
            #region VR.ATO.RNTLPRPTY.000015

            /*  VR.ATO.RNTLPRPTY.000015
                        When the Ownership percentage is less than 100% and (total) Water charges is present, Your share of water charges must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY155 = NULL) AND (^RNTLPRPTY133 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY155 = NULL) AND (^RNTLPRPTY133 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY155 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Water.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY133 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Water.Amount
                */
            VRATORNTLPRPTY000015(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000015

            #region VR.ATO.RNTLPRPTY.000016

            /*  VR.ATO.RNTLPRPTY.000016
                        When Your share of water charges is provided, the amount must not exceed (total) Water charges

                    Legacy Rule Format:
                        (^RNTLPRPTY155 <> NULL) AND (^RNTLPRPTY133 = NULL OR ^RNTLPRPTY155 > ^RNTLPRPTY133)

                Technical Business Rule Format:
                        (^RNTLPRPTY155 <> NULL) AND (^RNTLPRPTY133 = NULL OR ^RNTLPRPTY155 > ^RNTLPRPTY133)

            Data Elements:

                ^RNTLPRPTY155 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Water.Amount

                ^RNTLPRPTY133 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Water.Amount
                */
            VRATORNTLPRPTY000016(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000016

            #region VR.ATO.RNTLPRPTY.000013
            /*  VR.ATO.RNTLPRPTY.000013
            When the Ownership percentage is less than 100% and (total) Travel expenses is present, Your share of travel expenses must be provided

            Legacy Rule Format:
            (^RNTLPRPTY154 = NULL) AND (^RNTLPRPTY132 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Technical Business Rule Format:
            (^RNTLPRPTY154 = NULL) AND (^RNTLPRPTY132 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:
    
            ^RNTLPRPTY154 = RP.{PrptySeqNum}.Portion:Expense.Travel.Amount
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
    
            ^RNTLPRPTY132 = RP.{PrptySeqNum}.Total:Expense.Travel.Amount
        */
            VRATORNTLPRPTY000013(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000013

            #region VR.ATO.RNTLPRPTY.000014

            /*  VR.ATO.RNTLPRPTY.000014
                        When Your share of travel expenses is provided, the amount must not exceed (total) Travel expenses

                    Legacy Rule Format:
                        (^RNTLPRPTY154 <> NULL) AND (^RNTLPRPTY132 = NULL OR ^RNTLPRPTY154 > ^RNTLPRPTY132)

                Technical Business Rule Format:
                        (^RNTLPRPTY154 <> NULL) AND (^RNTLPRPTY132 = NULL OR ^RNTLPRPTY154 > ^RNTLPRPTY132)

            Data Elements:
    
                ^RNTLPRPTY154 = RP.{PrptySeqNum}.Portion:Expense.Travel.Amount
    
                ^RNTLPRPTY132 = RP.{PrptySeqNum}.Total:Expense.Travel.Amount
    */
            VRATORNTLPRPTY000014(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000014

            #region VR.ATO.RNTLPRPTY.000017

            /*  VR.ATO.RNTLPRPTY.000017
                        When the Ownership percentage is less than 100% and (total) Sundry rental expenses is present, Your share of sundry rental expenses must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY156 = NULL) AND (^RNTLPRPTY134 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY156 = NULL) AND (^RNTLPRPTY134 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY156 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Sundry.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY134 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Sundry.Amount
                */
            VRATORNTLPRPTY000017(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000017

            #region VR.ATO.RNTLPRPTY.000018

            /*  VR.ATO.RNTLPRPTY.000018
                        When Your share of sundry rental expenses is provided, the amount must not exceed (total) Sundry rental expenses

                    Legacy Rule Format:
                        (^RNTLPRPTY156 <> NULL) AND (^RNTLPRPTY134 = NULL OR ^RNTLPRPTY156 > ^RNTLPRPTY134)

                Technical Business Rule Format:
                        (^RNTLPRPTY156 <> NULL) AND (^RNTLPRPTY134 = NULL OR ^RNTLPRPTY156 > ^RNTLPRPTY134)

            Data Elements:

                ^RNTLPRPTY156 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Sundry.Amount

                ^RNTLPRPTY134 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Sundry.Amount
                */
            VRATORNTLPRPTY000018(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000018

            #region VR.ATO.RNTLPRPTY.000019

            /*  VR.ATO.RNTLPRPTY.000019
                        Entity identifier scheme must be "http://www.ato.gov.au/tfn" across all contexts

                    Legacy Rule Format:
                        (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')

                Technical Business Rule Format:
                        ^IdentifierScheme <> 'http://www.ato.gov.au/tfn'

            Data Elements:

                ^IdentifierScheme = IdentifierScheme
                */
            VRATORNTLPRPTY000019(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000019

            #region VR.ATO.RNTLPRPTY.000020

            /*  VR.ATO.RNTLPRPTY.000020
                        Entity identifier TFN must be the same across all contexts

                    Legacy Rule Format:
                        (NotSameValues(^TFN))

                Technical Business Rule Format:
                        (NotSameValues(^TFN))

            Data Elements:

                ^TFN = TFN
                */
            VRATORNTLPRPTY000020(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000020

            #region VR.ATO.RNTLPRPTY.000021

            /*  VR.ATO.RNTLPRPTY.000021
                        Period start date must be the same across all duration contexts

                    Legacy Rule Format:
                        (NotSameValues(^StartDate))

                Technical Business Rule Format:
                        (NotSameValues(^StartDate))

            Data Elements:

                ^StartDate = StartDate
                */
            VRATORNTLPRPTY000021(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000021

            #region VR.ATO.RNTLPRPTY.000022

            /*  VR.ATO.RNTLPRPTY.000022
                        Period end date must be the same across all duration contexts

                    Legacy Rule Format:
                        (NotSameValues(^EndDate))

                Technical Business Rule Format:
                        (NotSameValues(^EndDate))

            Data Elements:

                ^EndDate = EndDate
                */
            VRATORNTLPRPTY000022(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000022
            
            #region VR.ATO.RNTLPRPTY.000024

            /*  VR.ATO.RNTLPRPTY.000024
            The Sequence Container within the Rental Property Sequence Dimension context instance must not be duplicated
    
            Legacy Rule Format:
            (HasDuplicateValues(^RNTLPRPTY103))

            Technical Business Rule Format:
            (HasDuplicateValues(^RNTLPRPTY103))
    
            Data Elements:
    
            ^RNTLPRPTY103 = RP.{PrptySeqNum}:Report.ItemOrder.Number
            */
            assertion = HasDuplicateValues(report.RPPrptySeqNumContextCollection == null ? null : report.RPPrptySeqNumContextCollection.Select(f => f.RNTLPRPTY103).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sequence Container must not be duplicated",
                    LongDescription = @"The Sequence Container within the Rental Property Sequence Dimension context instance must not be duplicated",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.RPPrptySeqNumContextCollection[DuplicateValueIndex(report.RPPrptySeqNumContextCollection.Select(f => f.RNTLPRPTY103))].Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000024" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000024

            #region VR.ATO.RNTLPRPTY.000025

            /*  VR.ATO.RNTLPRPTY.000025
            The number of Reporting Party Rental Property Total Financial detail context instances provided for each Reporting Party Rental Property context must not exceed 1
    
            Legacy Rule Format:
            (HasDuplicateValues(^RNTLPRPTY113))

            Technical Business Rule Format:
            (HasDuplicateValues(^RNTLPRPTY113))
    
            Data Elements:
    
            ^RNTLPRPTY113 = RP.{PrptySeqNum}.Total:Report.ItemOrder.Number
            */
            assertion = HasDuplicateValues(report.RPPrptySeqNumTotalContextCollection == null ? null : report.RPPrptySeqNumTotalContextCollection.Select(f => f.RNTLPRPTY113).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sequence Container for Rental Property Total Financial detail contexts must not be duplicated",
                    LongDescription = @"The number of Reporting Party Rental Property Total Financial detail context instances provided for each Reporting Party Rental Property context must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.RPPrptySeqNumTotalContextCollection[DuplicateValueIndex(report.RPPrptySeqNumTotalContextCollection.Select(f => f.RNTLPRPTY113))].Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000025" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000025

            #region VR.ATO.RNTLPRPTY.000026

            /*  VR.ATO.RNTLPRPTY.000026
            Each Reporting Party Rental Property Total Financial detail context instance must have a matching (parent) Reporting Party Rental Property context
    
            Legacy Rule Format:
            (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY113) = 0)

            Technical Business Rule Format:
            (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY113) = 0)
    
            Data Elements:
    
            ^RNTLPRPTY113 = RP.{PrptySeqNum}.Total:Report.ItemOrder.Number
    
            ^RNTLPRPTY103 = RP.{PrptySeqNum}:Report.ItemOrder.Number
            */

            VRATORNTLPRPTY000026(response, report);

            #endregion // VR.ATO.RNTLPRPTY.000026

            #region VR.ATO.RNTLPRPTY.000027

            /*  VR.ATO.RNTLPRPTY.000027
            When a Reporting Party Rental Property Total Financial detail context instance has been provided, and the Ownership percentage is less than 100%, then a paired Reporting Party Rental Property Portion Financial detail context instance must be provided

            Legacy Rule Format:
            (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100 AND CountOccurrence(^RNTLPRPTY135, ^RNTLPRPTY135 = ^RNTLPRPTY113) = 0)

            Technical Business Rule Format:
            (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100 AND CountOccurrence(^RNTLPRPTY135, ^RNTLPRPTY135 = ^RNTLPRPTY113) = 0)

            Data Elements:

            ^RNTLPRPTY113 = RP.{PrptySeqNum}.Total:Report.ItemOrder.Number

            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

            ^RNTLPRPTY135 = RP.{PrptySeqNum}.Portion:Report.ItemOrder.Number
            */
            VRATORNTLPRPTY000027(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000027

            #region VR.ATO.RNTLPRPTY.000028

            /*  VR.ATO.RNTLPRPTY.000028
            The number of Reporting Party Rental Property Portion Financial detail context instances provided for each Reporting Party Rental Property context must not exceed 1
    
            Legacy Rule Format:
            (HasDuplicateValues(^RNTLPRPTY135))

            Technical Business Rule Format:
            (HasDuplicateValues(^RNTLPRPTY135))
    
            Data Elements:
    
            ^RNTLPRPTY135 = RP.{PrptySeqNum}.Portion:Report.ItemOrder.Number
            */
            assertion = HasDuplicateValues(report.RPPrptySeqNumPortionContextCollection == null ? null : report.RPPrptySeqNumPortionContextCollection.Select(f => f.RNTLPRPTY135).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sequence Container for Rental Property Portion Financial detail contexts must not be duplicated",
                    LongDescription = @"The number of Reporting Party Rental Property Portion Financial detail context instances provided for each Reporting Party Rental Property context must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context[@id='" + report.RPPrptySeqNumPortionContextCollection[DuplicateValueIndex(report.RPPrptySeqNumPortionContextCollection.Select(f => f.RNTLPRPTY135))].Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000028" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000028

            #region VR.ATO.RNTLPRPTY.000029

            /*  VR.ATO.RNTLPRPTY.000029
            Each Reporting Party Rental Property Portion Financial detail context instance must have a matching (parent) Reporting Party Rental Property context
    
            Legacy Rule Format:
            (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY135) = 0)

            Technical Business Rule Format:
            (CountOccurrence(^RNTLPRPTY103, ^RNTLPRPTY103 = ^RNTLPRPTY135) = 0)
    
            Data Elements:
    
            ^RNTLPRPTY135 = RP.{PrptySeqNum}.Portion:Report.ItemOrder.Number
    
            ^RNTLPRPTY103 = RP.{PrptySeqNum}:Report.ItemOrder.Number
            */

            VRATORNTLPRPTY000029(response, report);

            #endregion // VR.ATO.RNTLPRPTY.000029
            
            #region VR.ATO.RNTLPRPTY.000107

            /*  VR.ATO.RNTLPRPTY.000107
                        When Your share of rental income is provided, the amount must not exceed (total) Rental income

                    Legacy Rule Format:
                        (^RNTLPRPTY136 <> NULL) AND (^RNTLPRPTY136 > ^RNTLPRPTY114)

                Technical Business Rule Format:
                        (^RNTLPRPTY136 <> NULL) AND (^RNTLPRPTY136 > ^RNTLPRPTY114)

            Data Elements:

                ^RNTLPRPTY136 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.Rental.Amount

                ^RNTLPRPTY114 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.Rental.Amount
                */
            VRATORNTLPRPTY000107(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000107

            #region VR.ATO.RNTLPRPTY.000108

            /*  VR.ATO.RNTLPRPTY.000108
                        When the Ownership percentage is less than 100% and (total) Other rental related income is present, Your share of other rental related income must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY137 = NULL) AND (^RNTLPRPTY115 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY137 = NULL) AND (^RNTLPRPTY115 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY137 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.RentalRelatedOther.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY115 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.RentalRelatedOther.Amount
                */
            VRATORNTLPRPTY000108(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000108

            #region VR.ATO.RNTLPRPTY.000109

            /*  VR.ATO.RNTLPRPTY.000109
                        When Your share of other rental related income is provided, the amount must not exceed (total) Other rental related income

                    Legacy Rule Format:
                        (^RNTLPRPTY137 <> NULL) AND (^RNTLPRPTY115 = NULL OR ^RNTLPRPTY137 > ^RNTLPRPTY115)

                Technical Business Rule Format:
                        (^RNTLPRPTY137 <> NULL) AND (^RNTLPRPTY115 = NULL OR ^RNTLPRPTY137 > ^RNTLPRPTY115)

            Data Elements:

                ^RNTLPRPTY137 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.RentalRelatedOther.Amount

                ^RNTLPRPTY115 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.RentalRelatedOther.Amount
                */
            VRATORNTLPRPTY000109(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000109

            #region VR.ATO.RNTLPRPTY.000110

            /*  VR.ATO.RNTLPRPTY.000110
                        When the Ownership percentage is less than 100% and (total) Interest on loans is present, Your share of interest on loans must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY146 = NULL) AND (^RNTLPRPTY124 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY146 = NULL) AND (^RNTLPRPTY124 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY146 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LoanInterest.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY124 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LoanInterest.Amount
                */
            VRATORNTLPRPTY000110(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000110

            #region VR.ATO.RNTLPRPTY.000111

            /*  VR.ATO.RNTLPRPTY.000111
                        When Your share of interest on loans is provided, the amount must not exceed (total) Interest on loans

                    Legacy Rule Format:
                        (^RNTLPRPTY146 <> NULL) AND (^RNTLPRPTY124 = NULL OR ^RNTLPRPTY146 > ^RNTLPRPTY124)

                Technical Business Rule Format:
                        (^RNTLPRPTY146 <> NULL) AND (^RNTLPRPTY124 = NULL OR ^RNTLPRPTY146 > ^RNTLPRPTY124)

            Data Elements:

                ^RNTLPRPTY146 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.LoanInterest.Amount

                ^RNTLPRPTY124 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.LoanInterest.Amount
                */
            VRATORNTLPRPTY000111(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000111

            #region VR.ATO.RNTLPRPTY.000112

            /*  VR.ATO.RNTLPRPTY.000112
                        When the Ownership percentage is less than 100% and (total) Capital works deductions is present, Your share of capital works deductions must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY152 = NULL) AND (^RNTLPRPTY130 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY152 = NULL) AND (^RNTLPRPTY130 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY152 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CapitalWorksDeduction.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY130 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
                */
            VRATORNTLPRPTY000112(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000112

            #region VR.ATO.RNTLPRPTY.000113

            /*  VR.ATO.RNTLPRPTY.000113
                        When Your share of capital works deductions is provided, the amount must not exceed (total) Capital works deductions

                    Legacy Rule Format:
                        (^RNTLPRPTY152 <> NULL) AND (^RNTLPRPTY130 = NULL OR ^RNTLPRPTY152 > ^RNTLPRPTY130)

                Technical Business Rule Format:
                        (^RNTLPRPTY152 <> NULL) AND (^RNTLPRPTY130 = NULL OR ^RNTLPRPTY152 > ^RNTLPRPTY130)

            Data Elements:

                ^RNTLPRPTY152 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CapitalWorksDeduction.Amount

                ^RNTLPRPTY130 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
                */
            VRATORNTLPRPTY000113(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000113

            #region VR.ATO.RNTLPRPTY.000120

            /*  VR.ATO.RNTLPRPTY.000120
                        Date property first earned rental income must be prior to form submission end date

                    Legacy Rule Format:
                        (AnyOccurrence(^RNTLPRPTY105, ^RNTLPRPTY105 > ^EndDate))

                Technical Business Rule Format:
                        (AnyOccurrence(^RNTLPRPTY105, ^RNTLPRPTY105 > ^EndDate))

            Data Elements:

                ^RNTLPRPTY105 = RP.{PrptySeqNum}:RealEstateProperty.RentalIncomeFirstEarned.Date

                ^EndDate = EndDate
                */
            VRATORNTLPRPTY000120(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000120

            #region VR.ATO.RNTLPRPTY.000200

            /*  VR.ATO.RNTLPRPTY.000200
                        When the Ownership percentage is less than 100% and (total) Advertising for tenants is present, Your share of advertising for tenants must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY138 = NULL) AND (^RNTLPRPTY116 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY138 = NULL) AND (^RNTLPRPTY116 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY138 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Marketing.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY116 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Marketing.Amount
                */
            VRATORNTLPRPTY000200(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000200

            #region VR.ATO.RNTLPRPTY.000201

            /*  VR.ATO.RNTLPRPTY.000201
                        When Your share of advertising for tenants is provided, the amount must not exceed (total) Advertising for tenants

                    Legacy Rule Format:
                        (^RNTLPRPTY138 <> NULL) AND (^RNTLPRPTY116 = NULL OR ^RNTLPRPTY138 > ^RNTLPRPTY116)

                Technical Business Rule Format:
                        (^RNTLPRPTY138 <> NULL) AND (^RNTLPRPTY116 = NULL OR ^RNTLPRPTY138 > ^RNTLPRPTY116)

            Data Elements:

                ^RNTLPRPTY138 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Marketing.Amount

                ^RNTLPRPTY116 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Marketing.Amount
                */
            VRATORNTLPRPTY000201(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000201

            #region VR.ATO.RNTLPRPTY.000202

            /*  VR.ATO.RNTLPRPTY.000202
                        When the Ownership percentage is less than 100% and (total) Body corporate fees is present, Your share of body corporate fees must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY139 = NULL) AND (^RNTLPRPTY117 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY139 = NULL) AND (^RNTLPRPTY117 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY139 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.BodyCorporate.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY117 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.BodyCorporate.Amount
                */
            VRATORNTLPRPTY000202(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000202

            #region VR.ATO.RNTLPRPTY.000203

            /*  VR.ATO.RNTLPRPTY.000203
                        When Your share of body corporate fees is provided, the amount must not exceed (total) Body corporate fees

                    Legacy Rule Format:
                        (^RNTLPRPTY139 <> NULL) AND (^RNTLPRPTY117 = NULL OR ^RNTLPRPTY139 > ^RNTLPRPTY117)

                Technical Business Rule Format:
                        (^RNTLPRPTY139 <> NULL) AND (^RNTLPRPTY117 = NULL OR ^RNTLPRPTY139 > ^RNTLPRPTY117)

            Data Elements:

                ^RNTLPRPTY139 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.BodyCorporate.Amount

                ^RNTLPRPTY117 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.BodyCorporate.Amount
                */
            VRATORNTLPRPTY000203(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000203

            #region VR.ATO.RNTLPRPTY.000204

            /*  VR.ATO.RNTLPRPTY.000204
                        When the Ownership percentage is less than 100% and (total) Borrowing expenses is present, Your share of borrowing expenses must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY140 = NULL) AND (^RNTLPRPTY118 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY140 = NULL) AND (^RNTLPRPTY118 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY140 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Borrowing.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY118 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Borrowing.Amount
                */
            VRATORNTLPRPTY000204(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000204

            #region VR.ATO.RNTLPRPTY.000205

            /*  VR.ATO.RNTLPRPTY.000205
                        When Your share of borrowing expenses is provided, the amount must not exceed (total) Borrowing expenses

                    Legacy Rule Format:
                        (^RNTLPRPTY140 <> NULL) AND (^RNTLPRPTY118 = NULL OR ^RNTLPRPTY140 > ^RNTLPRPTY118)

                Technical Business Rule Format:
                        (^RNTLPRPTY140 <> NULL) AND (^RNTLPRPTY118 = NULL OR ^RNTLPRPTY140 > ^RNTLPRPTY118)

            Data Elements:

                ^RNTLPRPTY140 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Borrowing.Amount

                ^RNTLPRPTY118 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Borrowing.Amount
                */
            VRATORNTLPRPTY000205(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000205

            #region VR.ATO.RNTLPRPTY.000206

            /*  VR.ATO.RNTLPRPTY.000206
                        When the Ownership percentage is less than 100% and (total) Cleaning is present, Your share of cleaning must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY141 = NULL) AND (^RNTLPRPTY119 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY141 = NULL) AND (^RNTLPRPTY119 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY141 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Cleaning.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY119 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Cleaning.Amount
                */
            VRATORNTLPRPTY000206(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000206

            #region VR.ATO.RNTLPRPTY.000207

            /*  VR.ATO.RNTLPRPTY.000207
                        When Your share of cleaning is provided, the amount must not exceed (total) Cleaning

                    Legacy Rule Format:
                        (^RNTLPRPTY141 <> NULL) AND (^RNTLPRPTY119 = NULL OR ^RNTLPRPTY141 > ^RNTLPRPTY119)

                Technical Business Rule Format:
                        (^RNTLPRPTY141 <> NULL) AND (^RNTLPRPTY119 = NULL OR ^RNTLPRPTY141 > ^RNTLPRPTY119)

            Data Elements:

                ^RNTLPRPTY141 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Cleaning.Amount

                ^RNTLPRPTY119 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Cleaning.Amount
                */
            VRATORNTLPRPTY000207(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000207

            #region VR.ATO.RNTLPRPTY.000208

            /*  VR.ATO.RNTLPRPTY.000208
                        When the Ownership percentage is less than 100% and (total) Council rates is present, Your share of council rates must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY142 = NULL) AND (^RNTLPRPTY120 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY142 = NULL) AND (^RNTLPRPTY120 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY142 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CouncilRates.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY120 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CouncilRates.Amount
                */
            VRATORNTLPRPTY000208(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000208

            #region VR.ATO.RNTLPRPTY.000209

            /*  VR.ATO.RNTLPRPTY.000209
                        When Your share of council rates is provided, the amount must not exceed (total) Council rates

                    Legacy Rule Format:
                        (^RNTLPRPTY142 <> NULL) AND (^RNTLPRPTY120 = NULL OR ^RNTLPRPTY142 > ^RNTLPRPTY120)

                Technical Business Rule Format:
                        (^RNTLPRPTY142 <> NULL) AND (^RNTLPRPTY120 = NULL OR ^RNTLPRPTY142 > ^RNTLPRPTY120)

            Data Elements:

                ^RNTLPRPTY142 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.CouncilRates.Amount

                ^RNTLPRPTY120 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.CouncilRates.Amount
                */
            VRATORNTLPRPTY000209(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000209

            #region VR.ATO.RNTLPRPTY.000210

            /*  VR.ATO.RNTLPRPTY.000210
                        When the Ownership percentage is less than 100% and (total) Capital allowances (depreciation on plant) is present, Your share of capital allowances (depreciation on plant) must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY143 = NULL) AND (^RNTLPRPTY121 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY143 = NULL) AND (^RNTLPRPTY121 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY143 = RP.{PrptySeqNum}.Portion:Expense.DepreciationandAmortisation.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY121 = RP.{PrptySeqNum}.Total:Expense.DepreciationandAmortisation.Amount
                */
            VRATORNTLPRPTY000210(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000210

            #region VR.ATO.RNTLPRPTY.000211

            /*  VR.ATO.RNTLPRPTY.000211
                        When Your share of capital allowances (depreciation on plant) is provided, the amount must not exceed (total) Capital allowances (depreciation on plant)

                    Legacy Rule Format:
                        (^RNTLPRPTY143 <> NULL) AND (^RNTLPRPTY121 = NULL OR ^RNTLPRPTY143 > ^RNTLPRPTY121)

                Technical Business Rule Format:
                        (^RNTLPRPTY143 <> NULL) AND (^RNTLPRPTY121 = NULL OR ^RNTLPRPTY143 > ^RNTLPRPTY121)

            Data Elements:

                ^RNTLPRPTY143 = RP.{PrptySeqNum}.Portion:Expense.DepreciationandAmortisation.Amount

                ^RNTLPRPTY121 = RP.{PrptySeqNum}.Total:Expense.DepreciationandAmortisation.Amount
                */
            VRATORNTLPRPTY000211(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000211

            #region VR.ATO.RNTLPRPTY.000212

            /*  VR.ATO.RNTLPRPTY.000212
                        When the Ownership percentage is less than 100% and (total) Gardening/lawn mowing is present, Your share of gardening/lawn mowing must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY144 = NULL) AND (^RNTLPRPTY122 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY144 = NULL) AND (^RNTLPRPTY122 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY144 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Gardening.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY122 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Gardening.Amount
                */
            VRATORNTLPRPTY000212(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000212

            #region VR.ATO.RNTLPRPTY.000213

            /*  VR.ATO.RNTLPRPTY.000213
                        When Your share of gardening/lawn mowing is provided, the amount must not exceed (total) Gardening/lawn mowing

                    Legacy Rule Format:
                        (^RNTLPRPTY144 <> NULL) AND (^RNTLPRPTY122 = NULL OR ^RNTLPRPTY144 > ^RNTLPRPTY122)

                Technical Business Rule Format:
                        (^RNTLPRPTY144 <> NULL) AND (^RNTLPRPTY122 = NULL OR ^RNTLPRPTY144 > ^RNTLPRPTY122)

            Data Elements:

                ^RNTLPRPTY144 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Gardening.Amount

                ^RNTLPRPTY122 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Gardening.Amount
                */
            VRATORNTLPRPTY000213(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000213

            #region VR.ATO.RNTLPRPTY.000214

            /*  VR.ATO.RNTLPRPTY.000214
                        When the Ownership percentage is less than 100% and (total) Insurance is present, Your share of insurance must be provided

                    Legacy Rule Format:
                        (^RNTLPRPTY145 = NULL) AND (^RNTLPRPTY123 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

                Technical Business Rule Format:
                        (^RNTLPRPTY145 = NULL) AND (^RNTLPRPTY123 <> NULL) AND (^RNTLPRPTY104 <> NULL AND ^RNTLPRPTY104 < 100)

            Data Elements:

                ^RNTLPRPTY145 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Insurance.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY123 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Insurance.Amount
                */
            VRATORNTLPRPTY000214(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000214

            #region VR.ATO.RNTLPRPTY.000215

            /*  VR.ATO.RNTLPRPTY.000215
                        When Your share of insurance is provided, the amount must not exceed (total) Insurance

                    Legacy Rule Format:
                        (^RNTLPRPTY145 <> NULL) AND (^RNTLPRPTY123 = NULL OR ^RNTLPRPTY145 > ^RNTLPRPTY123)

                Technical Business Rule Format:
                        (^RNTLPRPTY145 <> NULL) AND (^RNTLPRPTY123 = NULL OR ^RNTLPRPTY145 > ^RNTLPRPTY123)

            Data Elements:

                ^RNTLPRPTY145 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.Insurance.Amount

                ^RNTLPRPTY123 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Insurance.Amount
                */
            VRATORNTLPRPTY000215(response, report);
            #endregion // VR.ATO.RNTLPRPTY.000215

            #region VR.ATO.RNTLPRPTY.W00216

            /*  VR.ATO.RNTLPRPTY.W00216
                        Only deductions for repairs and maintenance may be claimed that relate directly to wear and tear or other damage to rental properties and must not include capital works or depreciation

                    Legacy Rule Format:
                        ((^RNTLPRPTY104 = 100) AND (^RNTLPRPTY129 <> NULL AND ^RNTLPRPTY129 > 3500)) OR ((^RNTLPRPTY104 < 100) AND (^RNTLPRPTY151 <> NULL AND ^RNTLPRPTY151 > 3500))

                Technical Business Rule Format:
                        ((^RNTLPRPTY104 = 100) AND (^RNTLPRPTY129 <> NULL AND ^RNTLPRPTY129 > 3500)) OR ((^RNTLPRPTY104 < 100) AND (^RNTLPRPTY151 <> NULL AND ^RNTLPRPTY151 > 3500))

            Data Elements:

                ^RNTLPRPTY129 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.RepairsAndMaintenance.Amount

                ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent

                ^RNTLPRPTY151 = RP.{PrptySeqNum}.Portion:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
                */
            VRATORNTLPRPTYW00216(response, report);
            #endregion // VR.ATO.RNTLPRPTY.W00216

            #region VR.ATO.RNTLPRPTY.W00217

            /*  VR.ATO.RNTLPRPTY.W00217
                        Total borrowing expenses claimed is very high. Borrowing expenses must be claimed over five years from when the loan was established and must not include stamp duty, the loan balance or interest.

                    Legacy Rule Format:
                        (AnyOccurrence(^RNTLPRPTY118, ^RNTLPRPTY118 > 10000))

                Technical Business Rule Format:
                        (AnyOccurrence(^RNTLPRPTY118, ^RNTLPRPTY118 > 10000))

            Data Elements:

                ^RNTLPRPTY118 = RP.{PrptySeqNum}.Total:Expense.RealEstateProperty.Borrowing.Amount
                */
            VRATORNTLPRPTYW00217(response, report);
            #endregion // VR.ATO.RNTLPRPTY.W00217

            #endregion Manual Rules - Rules marked for manual coding

            #region VR.ATO.RNTLPRPTY.000023

            /*  VR.ATO.RNTLPRPTY.000023
            At least one Reporting Party Rental Property context must be provided
    
            Legacy Rule Format:
            (Count(RP.{PrptySeqNum}) = 0)

            Technical Business Rule Format:
            (Count(^RP.{PrptySeqNum}:Context) = 0)
    
            Data Elements:
            */
            assertion = (Count(report.RPPrptySeqNumContextCollection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. At least one Reporting Party Rental Property context must be provided.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrptySeqNumContextCollection == null ? 0 : (report.RPPrptySeqNumContextCollection.Count() == 0 ? 0 : report.RPPrptySeqNumContextCollection.Last().LastOccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000023

            #region VR.ATO.RNTLPRPTY.000100

            /*  VR.ATO.RNTLPRPTY.000100
            Address line 1 must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY108, ^RNTLPRPTY108 = BLANK))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY108, ^RNTLPRPTY108 = BLANK))
    
            Data Elements:
    
            ^RNTLPRPTY108 = RP.{PrptySeqNum}:AddressDetails.Line1.Text
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY108Repeat => string.IsNullOrWhiteSpace(RNTLPRPTY108Repeat.RNTLPRPTY108) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Address line 1 must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY108", Value = "RNTLPRPTY108" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000100

            #region VR.ATO.RNTLPRPTY.000101

            /*  VR.ATO.RNTLPRPTY.000101
            Suburb/Town must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY110, ^RNTLPRPTY110 = BLANK))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY110, ^RNTLPRPTY110 = BLANK))
    
            Data Elements:
    
            ^RNTLPRPTY110 = RP.{PrptySeqNum}:AddressDetails.LocalityName.Text
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY110Repeat => string.IsNullOrWhiteSpace(RNTLPRPTY110Repeat.RNTLPRPTY110) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000101",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Suburb/Town must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000101" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY110", Value = "RNTLPRPTY110" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000101

            #region VR.ATO.RNTLPRPTY.000102

            /*  VR.ATO.RNTLPRPTY.000102
            State must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY111, ^RNTLPRPTY111 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY111, ^RNTLPRPTY111 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY111 = RP.{PrptySeqNum}:AddressDetails.StateOrTerritory.Code
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY111Repeat => RNTLPRPTY111Repeat.RNTLPRPTY111 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000102",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State must be provided",
                    LongDescription = @"Where a Rental Property instance is present, State must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000102" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY111", Value = "RNTLPRPTY111" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000102

            #region VR.ATO.RNTLPRPTY.000103

            /*  VR.ATO.RNTLPRPTY.000103
            Postcode must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY112, ^RNTLPRPTY112 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY112, ^RNTLPRPTY112 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY112 = RP.{PrptySeqNum}:AddressDetails.Postcode.Text
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY112Repeat => RNTLPRPTY112Repeat.RNTLPRPTY112 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000103",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Postcode must be provided",
                    Location = "/xbrli:xbrl/tns:AddressDetails.Postcode.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000103" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY112", Value = "RNTLPRPTY112" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000103

            #region VR.ATO.RNTLPRPTY.000104

            /*  VR.ATO.RNTLPRPTY.000104
            Suburb/Town must not contain a State code
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY110, FoundSet(^RNTLPRPTY110, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY110, FoundSet(^RNTLPRPTY110, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')))
    
            Data Elements:
    
            ^RNTLPRPTY110 = RP.{PrptySeqNum}:AddressDetails.LocalityName.Text
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY110Repeat => IsMatch(RNTLPRPTY110Repeat.RNTLPRPTY110, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000104",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000104" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY110", Value = "Count(RNTLPRPTY110)" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000104

            #region VR.ATO.RNTLPRPTY.000105

            /*  VR.ATO.RNTLPRPTY.000105
            Rental income must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY114, ^RNTLPRPTY114 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY114, ^RNTLPRPTY114 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY114 = RP.{PrptySeqNum}.Total:Income.RealEstateProperty.Rental.Amount
            */
            assertion = (report.RPPrptySeqNumTotalContextCollection == null ? false : report.RPPrptySeqNumTotalContextCollection.Any(RNTLPRPTY114Repeat => RNTLPRPTY114Repeat.RNTLPRPTY114 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Rental income must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Rental income must be provided",
                    Location = "/xbrli:xbrl/tns:Income.RealEstateProperty.Rental.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000105" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY114", Value = "RNTLPRPTY114" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000105

            #region VR.ATO.RNTLPRPTY.000106

            /*  VR.ATO.RNTLPRPTY.000106
            Your share of rental income must be provided in every Rental Income - Your Share context
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY136, ^RNTLPRPTY136 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY136, ^RNTLPRPTY136 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY136 = RP.{PrptySeqNum}.Portion:Income.RealEstateProperty.Rental.Amount
            */
            assertion = (report.RPPrptySeqNumPortionContextCollection == null ? false : report.RPPrptySeqNumPortionContextCollection.Any(RNTLPRPTY136Repeat => RNTLPRPTY136Repeat.RNTLPRPTY136 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000106",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your share of rental income must be provided",
                    LongDescription = @"Where a Rental Income - Your Share instance is present, Your share of rental income must be provided",
                    Location = "/xbrli:xbrl/tns:Income.RealEstateProperty.Rental.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY136", Value = "RNTLPRPTY136" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000106

            #region VR.ATO.RNTLPRPTY.000117

            /*  VR.ATO.RNTLPRPTY.000117
            The number of Reporting Party Rental Property Financial details contexts must not exceed 45
    
            Legacy Rule Format:
            (COUNT(RP.{PrptySeqNum}) > 45)

            Technical Business Rule Format:
            (Count(^RP.{PrptySeqNum}:Context) > 45)
    
            Data Elements:
            */
            assertion = (Count(report.RPPrptySeqNumContextCollection) > 45);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000117",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Rental Property contexts must not exceed 45.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPPrptySeqNumContextCollection == null ? 0 : (report.RPPrptySeqNumContextCollection.Count() == 0 ? 0 : report.RPPrptySeqNumContextCollection.Last().LastOccurrenceIndex)),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000117" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000117

            #region VR.ATO.RNTLPRPTY.000118

            /*  VR.ATO.RNTLPRPTY.000118
            Ownership percentage must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY104, ^RNTLPRPTY104 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY104, ^RNTLPRPTY104 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY104 = RP.{PrptySeqNum}:RealEstateProperty.Ownership.Percent
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY104Repeat => RNTLPRPTY104Repeat.RNTLPRPTY104 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000118",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Ownership percentage must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Ownership percentage must be provided",
                    Location = "/xbrli:xbrl/tns:RealEstateProperty.Ownership.Percent",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000118" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY104", Value = "RNTLPRPTY104" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000118

            #region VR.ATO.RNTLPRPTY.000119

            /*  VR.ATO.RNTLPRPTY.000119
            Date property first earned rental income must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY105, ^RNTLPRPTY105 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY105, ^RNTLPRPTY105 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY105 = RP.{PrptySeqNum}:RealEstateProperty.RentalIncomeFirstEarned.Date
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY105Repeat => RNTLPRPTY105Repeat.RNTLPRPTY105 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000119",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date property first earned rental income must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Date property first earned rental income must be provided",
                    Location = "/xbrli:xbrl/tns:RealEstateProperty.RentalIncomeFirstEarned.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000119" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY105", Value = "RNTLPRPTY105" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000119

            #region VR.ATO.RNTLPRPTY.000121

            /*  VR.ATO.RNTLPRPTY.000121
            Number of weeks property was rented this year must be provided
    
            Legacy Rule Format:
            (AnyOccurrence(^RNTLPRPTY106, ^RNTLPRPTY106 = NULL))

            Technical Business Rule Format:
            (AnyOccurrence(^RNTLPRPTY106, ^RNTLPRPTY106 = NULL))
    
            Data Elements:
    
            ^RNTLPRPTY106 = RP.{PrptySeqNum}:RealEstateProperty.RentalWeeks.Count
            */
            assertion = (report.RPPrptySeqNumContextCollection == null ? false : report.RPPrptySeqNumContextCollection.Any(RNTLPRPTY106Repeat => RNTLPRPTY106Repeat.RNTLPRPTY106 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.RNTLPRPTY.000121",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of weeks property was rented this year must be provided",
                    LongDescription = @"Where a Rental Property instance is present, Number of weeks property was rented this year must be provided",
                    Location = "/xbrli:xbrl/tns:RealEstateProperty.RentalWeeks.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000121" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY106", Value = "RNTLPRPTY106" });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.RNTLPRPTY.000121

            #region Repeating RPPrptySeqNumContext
            if (report.RPPrptySeqNumContextCollection != null)
            {
                int itemIndexContext1 = 0;
                foreach (RNTLPRPTY2019.RPPrptySeqNumContext rpPrptySeqNumContext in report.RPPrptySeqNumContextCollection)
                {
                    itemIndexContext1++;

                    #region VR.ATO.RNTLPRPTY.000114

                    /*  VR.ATO.RNTLPRPTY.000114
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.
    
                    Legacy Rule Format:
                    (^RNTLPRPTY108 <> BLANK AND (StartsWith(^RNTLPRPTY108, 'CO ') OR FoundSet(^RNTLPRPTY108, '"C/O","C/","Care Of"')))
        
                    Technical Business Rule Format:
                    (^RNTLPRPTY108 <> BLANK AND (StartsWith(^RNTLPRPTY108, 'CO ') OR FoundSet(^RNTLPRPTY108, '"C/O","C/","Care Of"')))
            
                    Data Elements:
            
                    ^RNTLPRPTY108 = RP.{PrptySeqNum}:AddressDetails.Line1.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(rpPrptySeqNumContext.RNTLPRPTY108) != true && (IsMatch(rpPrptySeqNumContext.RNTLPRPTY108, @"^(CO )", RegexOptions.IgnoreCase) || IsMatch(rpPrptySeqNumContext.RNTLPRPTY108, @"(^(C/O|C/|Care Of)$)|(^(.*)( C/O | C/ | Care Of )(.*)$)|(^(C/O |C/ |Care Of )(.*)$)|(^(.*)( C/O| C/| Care Of)$)", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000114",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 is incorrect",
                            LongDescription = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.",
                            Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + rpPrptySeqNumContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000114" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY108", Value = rpPrptySeqNumContext.RNTLPRPTY108 });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.RNTLPRPTY.000114

                    #region VR.ATO.RNTLPRPTY.000115

                    /*  VR.ATO.RNTLPRPTY.000115
                    Address line 1 contains a 'care of' reference and there is no street address in Address line 2.  If a 'care of' reference is on Address line 1, it must be followed by a name and Address line 2 must contain the street address.
    
                    Legacy Rule Format:
                    (Contains(^RNTLPRPTY108, 'C/-') AND ^RNTLPRPTY109 = BLANK)
        
                    Technical Business Rule Format:
                    (Contains(^RNTLPRPTY108, 'C/-') AND ^RNTLPRPTY109 = BLANK)
            
                    Data Elements:
            
                    ^RNTLPRPTY108 = RP.{PrptySeqNum}:AddressDetails.Line1.Text
            
                    ^RNTLPRPTY109 = RP.{PrptySeqNum}:AddressDetails.Line2.Text
                    */
                    assertion = ((rpPrptySeqNumContext.RNTLPRPTY108 == null ? false : rpPrptySeqNumContext.RNTLPRPTY108.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(rpPrptySeqNumContext.RNTLPRPTY109) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000115",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"C/- must be followed by a Name and the Address on line 2",
                            Location = "/xbrli:xbrl/tns:AddressDetails.Line1.Text[@contextRef='" + rpPrptySeqNumContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000115" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY108", Value = rpPrptySeqNumContext.RNTLPRPTY108 });

                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY109", Value = GetValueOrEmpty(rpPrptySeqNumContext.RNTLPRPTY109) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.RNTLPRPTY.000115

                    #region VR.ATO.RNTLPRPTY.000116

                    /*  VR.ATO.RNTLPRPTY.000116
                    Address line 2 cannot contain a 'care of' reference.
    
                    Legacy Rule Format:
                    (^RNTLPRPTY109 <> BLANK AND (StartsWith(^RNTLPRPTY109, 'CO ') OR FoundSet(^RNTLPRPTY109, '"C/O","C/","Care Of","C/-"')))
        
                    Technical Business Rule Format:
                    (^RNTLPRPTY109 <> BLANK AND (StartsWith(^RNTLPRPTY109, 'CO ') OR FoundSet(^RNTLPRPTY109, '"C/O","C/","Care Of","C/-"')))
            
                    Data Elements:
            
                    ^RNTLPRPTY109 = RP.{PrptySeqNum}:AddressDetails.Line2.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(rpPrptySeqNumContext.RNTLPRPTY109) != true && (IsMatch(rpPrptySeqNumContext.RNTLPRPTY109, @"^(CO )", RegexOptions.IgnoreCase) || IsMatch(rpPrptySeqNumContext.RNTLPRPTY109, @"(^(C/O|C/|Care Of|C/-)$)|(^(.*)( C/O | C/ | Care Of | C/- )(.*)$)|(^(C/O |C/ |Care Of |C/- )(.*)$)|(^(.*)( C/O| C/| Care Of| C/-)$)", RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.RNTLPRPTY.000116",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 must not contain a 'care of' reference",
                            Location = "/xbrli:xbrl/tns:AddressDetails.Line2.Text[@contextRef='" + rpPrptySeqNumContext.Id + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.RNTLPRPTY.000116" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RNTLPRPTY109", Value = GetValueOrEmpty(rpPrptySeqNumContext.RNTLPRPTY109) });

                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.RNTLPRPTY.000116
                }
            }

            #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
    }
}