using System.Linq;
using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using Ato.CD.Inbound.Shared201802;
using VaTS;
using Ato.EN.IntegrationServices.CodeGenerationCGTS;
using Ato.EN.IntegrationServices.CodeGenerationFITR;
using DataContracts;

namespace Ato.CD.Inbound.FITR202402
{
    internal class CrossFormValidatorCGTS : ICrossFormValidator
    {
        private FITR2024 ParentDocument { get; }

        private CGTS2018 ChildDocument { get; }

        private List<Context> Contexts { get; }

        public List<ProcessMessageDocument> response { get; private set; }

        internal CrossFormValidatorCGTS(FITR2024 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (CGTS2018)childDocument.ConsumedReport;
            Contexts = ChildDocument.GetContexts();
        }

        public IEnumerable<ProcessMessageDocument> ValidateCrossFormRules()
        {
            response = new List<ProcessMessageDocument>();

            VRATOGEN402009(response);
            VRATOGEN438000(response);
            VRATOGEN438001(response);
            VRATOCGTS402165(response);
            VRATOCGTS402171(response);
            VRATOCGTS402178(response);
            VRATOCGTS402226(response);
            VRATOCGTS402230(response);
            VRATOCGTS402240(response);
            VRATOFITR434088(response);
            VRATOFITR434036(response);
            VRATOFITR434533(response);
            VRATOFITR434534(response);
            VRATOFITR434535(response);
            VRATOFITR434536(response);

            return response;
        }
        
        #region VR.ATO.GEN.402009

        /*  VR.ATO.GEN.402009
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
                RETURN VALIDATION MESSAGE
            ENDIF 
        */
        private void VRATOGEN402009(List<ProcessMessageDocument> response)
        {
            bool assertion = !ParentDocument.FITR5.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "FITR5", Value = ParentDocument.FITR5 }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.GEN.402009

        #region VR.ATO.GEN.438000

        /*  VR.ATO.GEN.438000
            IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate
               RETURN VALIDATION MESSAGE
            ENDIF
        */
        private void VRATOGEN438000(List<ProcessMessageDocument> response)
        {
            IEnumerable<Context> contexts = Contexts.Where(context => context.StartDate.GetValueOrDefault() != ParentDocument.FITR300.GetValueOrDefault());

            IEnumerable<ProcessMessageDocument> processMessages = contexts.Select(context => new ProcessMessageDocument
            {
                Code = "CMN.ATO.GEN.438000",
                Severity = ProcessMessageSeverity.Error,
                Description = @"The context period start date is incorrect.",
                LongDescription = @"Period start date within context on the schedule must match period start date within the parent return reporting party context",
                Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:startDate",
                Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438000" } }
            });

            response.AddRange(processMessages);
        }

        #endregion // VR.ATO.CGTS.438000

        #region VR.ATO.GEN.438001

        /*  VR.ATO.GEN.438001
            IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
               RETURN VALIDATION MESSAGE
            ENDIF
        */
        private void VRATOGEN438001(List<ProcessMessageDocument> response)
        {
            IEnumerable<Context> contexts = Contexts.Where(context => context.EndDate.GetValueOrDefault() != ParentDocument.FITR301.GetValueOrDefault());

            IEnumerable<ProcessMessageDocument> processMessages = contexts.Select(context => new ProcessMessageDocument
            {
                Code = "CMN.ATO.GEN.438001",
                Severity = ProcessMessageSeverity.Error,
                Description = @"The context period end date is incorrect.",
                LongDescription = @"Period end date within context on the schedule must match period end date within the parent return reporting party context",
                Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:endDate",
                Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438001" } }
            });

            response.AddRange(processMessages);
        }

        #endregion // VR.ATO.CGTS.438001

        #region VR.ATO.CGTS.402165

        /*  VR.ATO.CGTS.402165

            IF (PARENT RETURN <> "CTR") AND ([CGTS122] <> NULL)  
                RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS122] = CGTS:RP.Applied:tns:Capital.Losses.TransferredTotal.Amount
        */
        private void VRATOCGTS402165(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS122.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402165",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total losses transferred in applied is only valid for a company",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredTotal.Amount",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402165" },
                        new ProcessMessageParameter { Name = "CGTS122", Value = ChildDocument.CGTS122.Value.ToString() }
                    }
                };
                
                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.CGTS.402165

        #region VR.ATO.CGTS.402171

        /*  VR.ATO.CGTS.402171

            IF PARENT RETURN = "SMSFAR", "FITR", "TRT" AND ([CGTS131] > 0) AND ([CGTS125] > 0)
                RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS125] = CGTS:RP.CGTOther.Unapplied:rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [CGTS131] = CGTS:RP:bafpr1.02.00:Income.CapitalGainsNet.Amount
        */
        private void VRATOCGTS402171(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS131.GetValueOrDefault() > 0 && ChildDocument.CGTS125.GetValueOrDefault() > 0;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A capital loss must not be carried forward when a positive net capital gain amount has been provided",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402171" },
                        new ProcessMessageParameter { Name = "CGTS125", Value = ChildDocument.CGTS125.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "CGTS131", Value = ChildDocument.CGTS131.GetValueOrDefault().ToString() }
                    }
                };
                
                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402171

        #region VR.ATO.CGTS.402178
        
        /*  VR.ATO.CGTS.402178

            IF PARENT RETURN = "SMSFAR","FITR" AND ([CGTS128] <> NULL)
                RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS128] = CGTS:RP:gfagc.02.01:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount   
        */
        private void VRATOCGTS402178(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS128.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402178",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business retirement exemption is not valid for Self Managed Super Funds",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402178" },
                        new ProcessMessageParameter { Name = "CGTS128", Value = ChildDocument.CGTS128.Value.ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402178

        #region VR.ATO.CGTS.402226

        /*  VR.ATO.CGTS.402226
          
            IF (PARENT RETURN:RP:pyin.xx.xx:Report.Amendment.Indicator = TRUE) AND ([CGTS150] <> NULL AND [CGTS151] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS150] = CGTS:RP:pyin.02.37:Report.TargetFinancial.Year
            [CGTS151] = CGTS:RP:bafot.02.26:Miscellaneous.RequestedAmendment.Amount
        */
        private void VRATOCGTS402226(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR171.GetValueOrDefault() && ChildDocument.CGTS150.HasValue && ChildDocument.CGTS151.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created and Amount to be amended cannot be supplied when Parent return Amendment indicator is present.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402226" },
                        new ProcessMessageParameter { Name = "FITR171", Value = ParentDocument.FITR171.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "CGTS150", Value = ChildDocument.CGTS150.Value.ToString() },
                        new ProcessMessageParameter { Name = "CGTS151", Value = ChildDocument.CGTS151.Value.ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402226       

        #region VR.ATO.CGTS.402230

        /*  VR.ATO.CGTS.402230
          
            IF PARENT RETURN  <> "TRTAMI" AND ([CGTS153] <> NULL OR [CGTS154] <> NULL OR [CGTS155] <> NULL)
                RETURN VALIDATION MESSAGE
            ENDIF
        */
        private void VRATOCGTS402230(List<ProcessMessageDocument> response)
        {
            bool assertion = !string.IsNullOrEmpty(ChildDocument.CGTS153) || ChildDocument.CGTS154.HasValue || ChildDocument.CGTS155.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Multi-class treatment fields incorrectly supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402230" },
                        new ProcessMessageParameter { Name = "CGTS153", Value = !string.IsNullOrEmpty(ChildDocument.CGTS153) ? ChildDocument.CGTS153 : string.Empty },
                        new ProcessMessageParameter { Name = "CGTS154", Value = ChildDocument.CGTS154.HasValue ? ChildDocument.CGTS154.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "CGTS155", Value = ChildDocument.CGTS155.HasValue ? ChildDocument.CGTS155.Value.ToString() : string.Empty }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402230

        #region VR.ATO.CGTS.402240

        /*  VR.ATO.CGTS.402240

            IF([CGTS150] <> NULL) AND ([CGTS150] = (PARENT RETURN:RP:pyin.xx.xx:Report.TargetFinancial.Year))
               RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS150] = CGTS:RP:pyin.02.37:Report.TargetFinancial.Year         
        */
        private void VRATOCGTS402240(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS150.HasValue && ChildDocument.CGTS150 == ParentDocument.FITR1.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402240",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created incorrect",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402240" },
                        new ProcessMessageParameter { Name = "FITR1", Value = ParentDocument.FITR1.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "CGTS150", Value = ChildDocument.CGTS150.Value.ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402240

        #region VR.ATO.FITR.434036

        private void VRATOFITR434036(List<ProcessMessageDocument> response)
        {
            /*
                IF (COUNT(SCHEDULE = ""CGTS"") = 1) AND [FITR23] <> [CGTS131]
                    RETURN VALIDATION MESSAGE
                ENDIF

                [CGTS131] = CGTS:RP: bafpr1.02.00:Income.CapitalGainsNet.Amount
                [FITR23] = FITR:RP: bafpr1.02.00:Income.CapitalGainsNet.Amount
            */

            bool assertion = ParentDocument.FITR23.HasValue && ParentDocument.FITR23.Value != ChildDocument.CGTS131.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gains in return must equal total net capital gain in CGT schedule.",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsNetA",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434036" },
                        new ProcessMessageParameter { Name = "FITR23", Value = ParentDocument.FITR23.Value.ToString() },
                        new ProcessMessageParameter { Name = "CGTS131", Value = ChildDocument.CGTS131.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion

        #region VR.ATO.FITR.434088

        /*  VR.ATO.FITR.434088
            Net capital losses carried forward to later income years must equal amount in CGT schedule

            Technical Business Rule:
            IF (COUNT(SCHEDULE = "CGTS") = 1) AND  [FITR207] <> ([CGTS124]+[CGTS125])
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS124] = CGTS:RP.Collectables.Unapplied(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [CGTS125] = CGTS:RP.CGTOther.Unapplied(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [FITR207] = FITR:RP.Closing(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            (CountDocument('CGTS') = 1) AND ^FITR207 <> (^CGTS124+^CGTS125)

            Data Elements:
            
            RP.Closing(Instant):^FITR207 = tns:Capital.Losses.CarriedForward.Net.Amount
            
            RP.Closing(Instant):^CGTS124 = CGTS:RP.Collectables.Unapplied(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            
            RP.Closing(Instant):^CGTS125 = CGTS:RP.CGTOther.Unapplied(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
        */
        protected void VRATOFITR434088(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR207.GetValueOrDefault() != ChildDocument.CGTS124.GetValueOrDefault() + ChildDocument.CGTS125.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434289",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses carried forward to later income years must equal amount in CGT schedule",
                    LongDescription = @"Net capital losses carried forward to later income years' on the main form must equal the total of the 'Unapplied net capital losses carried forward'  in the Capital Gains Tax schedule",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:CapitalCarriedForwardNetA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434088" },
                        new ProcessMessageParameter() { Name = "FITR207", Value = ParentDocument.FITR207.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "CGTS124", Value = ChildDocument.CGTS124.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "CGTS125", Value = ChildDocument.CGTS125.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434088

        #region VR.ATO.FITR.434533

        /*  VR.ATO.FITR.434533
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business active asset reduction (Subdivision 152-C)", then Small business active asset reduction amount on attached CGT schedule must be completed

            Technical Business Rule:
            (^FITR204 = 'A - Small business active asset reduction') AND (CountDocument('CGTS') = 1) AND ^CGTS127 = NULL
        */
        private void VRATOFITR434533(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("A - Small business active asset reduction")) && !ChildDocument.CGTS127.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business active asset reduction amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business active asset reduction (Subdivision 152-C)"", then Small business active asset reduction amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434533" }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434533

        #region VR.ATO.FITR.434534

        /*  VR.ATO.FITR.434534
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business roll-over (Subdivision 152-E)", then Small business roll-over amount on attached CGT schedule must be completed

            Technical Business Rule:
            (^FITR204 = 'C - Small business roll-over') AND (CountDocument('CGTS') = 1) AND ^CGTS129 = NULL
        */
        private void VRATOFITR434534(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("C - Small business roll-over")) && !ChildDocument.CGTS129.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business roll-over (Subdivision 152-E)"", then Small business roll-over amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434534" }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434534

        #region VR.ATO.FITR.434535

        /*  VR.ATO.FITR.434535
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business 15 year exemption (Subdivision152-B)", then Small business 15 year exemption amount on attached CGT schedule must be completed

            Technical Business Rule:
            (^FITR204 = 'D - Small business 15 year exemption') AND (CountDocument('CGTS') = 1) AND ^CGTS138 = NULL
        */
        private void VRATOFITR434535(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("D - Small business 15 year exemption")) && !ChildDocument.CGTS138.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business 15 year exemption amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business 15 year exemption (Subdivision152-B)"", then Small business 15 year exemption amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434535" }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434535

        #region VR.ATO.FITR.434536

        /*  VR.ATO.FITR.434536
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Scrip for scrip roll-over (Subdivision 124-M)", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed

            Technical Business Rule:
            (^FITR204 = 'F - Scrip for scrip roll-over') AND (CountDocument('CGTS') = 1) AND ^CGTS45 = NULL
        */
        private void VRATOFITR434536(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("F - Scrip for scrip roll-over")) && !ChildDocument.CGTS45.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains disregarded as a result of a scrip for scrip roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Scrip for scrip roll-over (Subdivision 124-M)"", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434536" }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434536
    }
}