using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using VaTS;
using Ato.EN.IntegrationServices.CodeGenerationLS;
using Ato.EN.IntegrationServices.CodeGenerationFITR;
using DataContracts;

namespace Ato.CD.Inbound.FITR202402
{
    internal class CrossFormValidatorLS : ICrossFormValidator
    {
        private FITR2024 ParentDocument { get; }

        private LS2017 ChildDocument { get; }

        private List<Context> Contexts { get; }

        internal CrossFormValidatorLS(FITR2024 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (LS2017)childDocument.ConsumedReport;
            Contexts = GetContexts();
        }

        public IEnumerable<ProcessMessageDocument> ValidateCrossFormRules()
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();

            VRATOFITR434086(response);
            VRATOFITR434087(response);
            VRATOGEN402009(response);
            VRATOGEN402010(response);
            VRATOGEN428045(response);
            VRATOLS414017(response);
            VRATOLS414022(response);
            
            return response;
        }

        #region VR.ATO.FITR.434086

        /*  VR.ATO.FITR.434086
            Tax losses carried forward to later income years in schedule must equal amount in main form

            Legacy Rule Format:
            (CountDocument('LS') = 1) AND ^LS50 <> ^FITR76
            
            Technical Business Rule Format:
            (CountDocument('LS') = 1) AND ^LS50 <> ^FITR76

            Data Elements:
            
            RP:^FITR76 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            RP.Closing:^LS50 = LS:RP.Closing:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
        */
        private void VRATOFITR434086(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR76.GetValueOrDefault() != ChildDocument.LS50.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434086",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses carried forward to later income years in schedule must equal amount in main form",
                    LongDescription = @"The total of the 'Tax losses carried forward to later income years'  for unconsolidated entities in the Losses schedule must equal 'Tax losses carried forward to later income years' on the main form",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:TaxCarriedForwardLaterIncomeYearsTotalA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434086" },
                        new ProcessMessageParameter() { Name = "FITR76", Value = ParentDocument.FITR76.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "LS50", Value = ChildDocument.LS50.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434086

        #region VR.ATO.FITR.434087

        /*  VR.ATO.FITR.434087
            Net capital losses carried forward to later income years must equal amount in Losses schedule

            Legacy Rule Format:
            (CountDocument('LS') = 1) AND ^LS52 <> ^FITR207

            Technical Business Rule Format:
            (CountDocument('LS') = 1) AND ^LS52 <> ^FITR207

            Data Elements:
            
            RP:^FITR207 = tns:Capital.Losses.CarriedForward.Net.Amount
            RP.Closing(Instant):^LS52 = LS:RP.Closing(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
        */
        private void VRATOFITR434087(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.FITR207.GetValueOrDefault() != ChildDocument.LS52.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434087",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses carried forward to later income years must equal amount in Losses schedule",
                    LongDescription = @"The total of the 'Net capital losses carried forward to later income years'  for unconsolidated entities in the Losses schedule must equal 'Net capital losses carried forward to later income years' on the main form",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:CapitalCarriedForwardNetA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434087" },
                        new ProcessMessageParameter() { Name = "FITR207", Value = ParentDocument.FITR207.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "LS52", Value = ChildDocument.LS52.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434087

        #region VR.ATO.GEN.402009

        /*  VR.ATO.GEN.402009
            Net capital losses carried forward to later income years must equal amount in Losses schedule

            Legacy Rule Format:
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
                RETURN VALIDATION MESSAGE
            ENDIF
            
            Technical Business Rule Format:
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
                RETURN VALIDATION MESSAGE
            ENDIF
            
            Data Elements:
            
            RP:^FITR5 = tns:Identifiers.TaxFileNumber.Identifier
        */
        private void VRATOGEN402009(List<ProcessMessageDocument> response)
        {
            bool assertion = !ParentDocument.FITR5.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "FITR5", Value = ParentDocument.FITR5 }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.GEN.402009

        #region VR.ATO.GEN.402010

        /*  VR.ATO.GEN.402010
            Your supplied ABN does not match the ABN supplied on the form it was submitted with

            Legacy Rule Format:
            IF (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            IF (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier)
                RETURN VALIDATION MESSAGE
            ENDIF

            Data Elements:
            
            RP:^FITR7 = tns:Identifiers.AustralianBusinessNumber.Identifier
            RP:^LS3 = LS:RP:pyin.02.00:Identifiers.AustralianBusinessNumber.Identifier
        */
        private void VRATOGEN402010(List<ProcessMessageDocument> response)
        {
            bool assertion = (!string.IsNullOrWhiteSpace(ParentDocument.FITR7) && !string.IsNullOrWhiteSpace(ChildDocument.LS3)) && !(ParentDocument.FITR7.Equals(ChildDocument.LS3));

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.402010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied ABN does not match the ABN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied ABN does not match the ABN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402010" },
                        new ProcessMessageParameter() { Name = "FITR7", Value = ParentDocument.FITR7 },
                        new ProcessMessageParameter() { Name = "LS3", Value = ChildDocument.LS3 }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion

        #region VR.ATO.GEN.428045

        /*  VR.ATO.GEN.402010
            Organisation Name in schedule should match Organisation Name in parent return

            Legacy Rule Format:
            IF RP:pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text IN TUPLE (xbrli\organisationname1.xx.xx:OrganisationNameDetails) <> NULL
            AND RP:pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text IN TUPLE (xbrli\organisationname1.xx.xx:OrganisationNameDetails) <> (PARENT RETURN:RP:pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text IN TUPLE (xbrli\organisationname2.xx.xx:OrganisationNameDetails)
            WHERE (TUPLE EXPLICIT pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code = "MN")
            AND (TUPLE EXPLICIT pyde.xx.xx:OrganisationNameDetails.Currency.Code = "C")) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            IF RP:pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text IN TUPLE (xbrli\organisationname1.xx.xx:OrganisationNameDetails) <> NULL
            AND RP:pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text IN TUPLE (xbrli\organisationname1.xx.xx:OrganisationNameDetails) <> (PARENT RETURN:RP:pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text IN TUPLE (xbrli\organisationname2.xx.xx:OrganisationNameDetails)
            WHERE (TUPLE EXPLICIT pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code = "MN")
            AND (TUPLE EXPLICIT pyde.xx.xx:OrganisationNameDetails.Currency.Code = "C")) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Data Elements:
            
            RP:^FITR6 = tns:OrganisationNameDetails.OrganisationalName.Text
            RP:^LS2 = LS:RP:pyin.02.00:OrganisationNameDetails.OrganisationalName.Text
        */
        private void VRATOGEN428045(List<ProcessMessageDocument> response)
        {
            bool assertion = !string.IsNullOrWhiteSpace(ParentDocument.FITR6) && !string.IsNullOrWhiteSpace(ChildDocument.LS2) && !ParentDocument.FITR6.Equals(ChildDocument.LS2);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428045",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation Name in schedule should match Organisation Name in parent return",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428045" },
                        new ProcessMessageParameter() { Name = "FITR6", Value = ParentDocument.FITR6 },
                        new ProcessMessageParameter() { Name = "LS2", Value = ChildDocument.LS2 }
                    }
                };

                response.Add(processMessage);
            }
        }
        
        #endregion

        #region VR.ATO.LS.414017

        /*  VR.ATO.LS.414017

            Technical Business Rule:
            IF (PARENT RETURN <> SET("CTR", "TRT")) AND (([LS15] <>NULL) OR ([LS16] <> NULL) OR ([LS17] <> NULL) OR ([LS18] <> NULL) OR ([LS19] <> NULL) OR ([LS53] <> NULL))
                RETURN VALIDATION MESSAGE
            ENDIF

            [LS15] = LS:RP.Y0:rvctc3.xx.xx:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.xx.xx:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.xx.xx:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.xx.xx:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.xx.xx:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS53] = LS:RP.Y0-5+:rvctc3.xx.xx:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
        */
        private void VRATOLS414017(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.LS15.HasValue || ChildDocument.LS16.HasValue || ChildDocument.LS17.HasValue || ChildDocument.LS18.HasValue || ChildDocument.LS19.HasValue || ChildDocument.LS53.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.LS.414017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Continuity of majority ownership test only required for companies or widely held trusts",
                    Location = "/xbrli:xbrl/ls:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.LS.414017" },
                        new ProcessMessageParameter { Name = "LS15", Value = ChildDocument.LS15.HasValue ? ChildDocument.LS15.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS16", Value = ChildDocument.LS16.HasValue ? ChildDocument.LS16.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS17", Value = ChildDocument.LS17.HasValue ? ChildDocument.LS17.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS18", Value = ChildDocument.LS18.HasValue ? ChildDocument.LS18.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS19", Value = ChildDocument.LS19.HasValue ? ChildDocument.LS19.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS53", Value = ChildDocument.LS53.HasValue ? ChildDocument.LS53.Value.ToString() : string.Empty }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.LS.414017

        #region VR.ATO.LS.414022

        /*  VR.ATO.LS.414022
            
            Technical Business Rule:
            IF (PARENT RETURN <> "CTR") AND ([LS25] <> NULL OR [LS26] <> NULL OR [LS27] <> NULL OR [LS28] <> NULL)
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS25] = LS:RP:rvctc3.xx.xx:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator
            [LS26] = LS:RP:rvctc3.xx.xx:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
            [LS27] = LS:RP:rvctc3.xx.xx:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
            [LS28] = LS:RP:rvctc3.xx.xx:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount
        */
        private void VRATOLS414022(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.LS25.HasValue || ChildDocument.LS26.HasValue || ChildDocument.LS27.HasValue || ChildDocument.LS28.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.LS.414022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unrealised losses are not required",
                    Location = "/xbrli:xbrl/ls:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.LS.414022" },
                        new ProcessMessageParameter { Name = "LS25", Value = ChildDocument.LS25.HasValue ? ChildDocument.LS25.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS26", Value = ChildDocument.LS26.HasValue ? ChildDocument.LS26.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS27", Value = ChildDocument.LS27.HasValue ? ChildDocument.LS27.Value.ToString() : string.Empty },
                        new ProcessMessageParameter { Name = "LS28", Value = ChildDocument.LS28.HasValue ? ChildDocument.LS28.Value.ToString() : string.Empty }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.LS.414022

        private List<Context> GetContexts() => new List<Context>()
            .AddContext(ChildDocument.RPStartDate, ChildDocument.RPEndDate, ChildDocument.RPOccurrenceIndex, ChildDocument.RPId)
            .AddContext(ChildDocument.RPY0StartDate, ChildDocument.RPY0EndDate, ChildDocument.RPY0OccurrenceIndex, ChildDocument.RPY0Id)
            .AddContext(ChildDocument.RPY0Minus1StartDate, ChildDocument.RPY0Minus1EndDate, ChildDocument.RPY0Minus1OccurrenceIndex, ChildDocument.RPY0Minus1Id)
            .AddContext(ChildDocument.RPY0Minus2StartDate, ChildDocument.RPY0Minus2EndDate, ChildDocument.RPY0Minus2OccurrenceIndex, ChildDocument.RPY0Minus2Id)
            .AddContext(ChildDocument.RPY0Minus3StartDate, ChildDocument.RPY0Minus3EndDate, ChildDocument.RPY0Minus3OccurrenceIndex, ChildDocument.RPY0Minus3Id)
            .AddContext(ChildDocument.RPY0Minus4StartDate, ChildDocument.RPY0Minus4EndDate, ChildDocument.RPY0Minus4OccurrenceIndex, ChildDocument.RPY0Minus4Id)
            .AddContext(ChildDocument.RPY0Minus5PStartDate, ChildDocument.RPY0Minus5PEndDate, ChildDocument.RPY0Minus5POccurrenceIndex, ChildDocument.RPY0Minus5PId)
            .AddContext(ChildDocument.RPALLStartDate, ChildDocument.RPALLEndDate, ChildDocument.RPALLOccurrenceIndex, ChildDocument.RPALLId);
    }
}