using System.Linq;
using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using VaTS;
using Ato.EN.IntegrationServices.CodeGenerationFITR;
using Ato.EN.IntegrationServices.CodeGenerationPSS;
using DataContracts;

namespace Ato.CD.Inbound.FITR202402
{
    internal class CrossFormValidatorPSS : ICrossFormValidator
    {
        private FITR2024 ParentDocument { get; }

        private PSS2018 ChildDocument { get; }

        public CrossFormValidatorPSS(FITR2024 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (PSS2018)childDocument.ConsumedReport;
        }

        public IEnumerable<ProcessMessageDocument> ValidateCrossFormRules()
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            
            VRATOFITR434045(response);
            VRATOFITR434046(response);
            VRATOFITR434056(response);
            VRATOFITR434080(response);
            VRATOPSS000035(response);
            VRATOPSS000036(response);

            return response;
        }

        #region VR.ATO.FITR.434045

        /*  VR.ATO.FITR.434045
            Tax withheld amount must not be less than the sum of amounts present in the schedule

            Legacy Rule Format:
            (CountDocument('PSS') = 1) AND ^FITR32 > 0 AND ((^FITR72 + 1) < Sum(^PSS23))
            
            Technical Business Rule Format:
            (CountDocument('PSS') = 1) AND ^FITR32 > 0 AND ((^FITR72 + 1) < Sum(^PSS23))

            Data Elements:

            RP:^FITR32 = tns:Remuneration.ABNNotQuotedPaymentGross.Amount
            RP:^FITR72 = tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount
            RP:^PSS23 = PSS:RP.Payer.{PSSeqNum}:lrla.02.00:Remuneration.PaymentToForeignResidentGross.Amount
        */
        private void VRATOFITR434045(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.RPPayerPSSeqNumContextCollection != null && ParentDocument.FITR32 > 0 &&
                (ParentDocument.FITR72 + 1) < ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS23).GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434045",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Tax withheld amount must not be less than the sum of amounts present in the schedule",
                    LongDescription = "The 'Credit for tax withheld where ABN/TFN not quoted' amount must equal the sum of the Tax withheld amounts on the Payment Summary schedule",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434045" },
                        new ProcessMessageParameter() { Name = "FITR32", Value = ParentDocument.FITR72.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "FITR72", Value = ParentDocument.FITR72.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(PSS23)", Value = ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS23).GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion

        #region VR.ATO.FITR.434046

        /*  VR.ATO.FITR.434046
            Gross payment amount must equal the sum of amounts in the Payment Summary schedule

            Legacy Rule Format:
            IF (COUNT(SCHEDULE = "PSS") = 1) AND SUM([PSS19]) <> [FITR32]

            Technical Business Rule Format:
            (CountDocument('PSS') = 1) AND Sum(^PSS19) <> ^FITR32

            Data Elements:
            
            RP:^FITR32 = tns:Remuneration.ABNNotQuotedPaymentGross.Amount
            RP:^PSS19 = PSS:RP.Payer.{PSSeqNum}:lrla.02.00:Remuneration.ABNNotQuotedPaymentGross.Amount
        */
        private void VRATOFITR434046(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.RPPayerPSSeqNumContextCollection != null &&
                ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS19).GetValueOrDefault() != ParentDocument.FITR32.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross payment amount must equal the sum of amounts in the Payment Summary schedule",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434046" },
                        new ProcessMessageParameter { Name = "FITR32", Value = ParentDocument.FITR32.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "PSS19", Value = ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS19).GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion

        #region VR.ATO.FITR.434056

        /*  VR.ATO.FITR.434056
            Other income must not be less than Gross payment amounts on Payment Summary schedule

            Legacy Rule Format:
            (CountDocument('PSS') = 1) AND Sum(^PSS18) > ^FITR47

            Technical Business Rule Format:
            (CountDocument('PSS') = 1) AND Sum(^PSS18) > ^FITR47

            Data Elements:

            RP:^FITR47 = tns:Income.Other.Amount
            RP:^PSS18 = PSS:RP.Payer.{PSSeqNum}:lrla.02.00:Remuneration.PaymentToForeignResidentGross.Amount
        */
        private void VRATOFITR434056(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.RPPayerPSSeqNumContextCollection != null &&
                ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS18).GetValueOrDefault() > ParentDocument.FITR47.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434056",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income must not be less than Gross payment amounts on Payment Summary schedule",
                    LongDescription = @"The 'Other income' amount must not be less than the sum of all 'Gross payment' amounts on the Payment Summary schedule",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:OtherA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434056" },
                        new ProcessMessageParameter() { Name = "FITR47", Value = ParentDocument.FITR47.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "PSS18", Value = ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS18).GetValueOrDefault().ToString() }
                    }
                };
                
                response.Add(processMessage);
            }
        }

        #endregion  

        #region VR.ATO.FITR.434080

        /*  VR.ATO.FITR.434080
            Credit: FRW (excluding capital gains) must not be less than the total of all FRW amounts in Payment Summary schedule

            Legacy Rule Format:
            (CountDocument('PSS') = 1) AND Sum(^PSS22) > (^FITR71 + 1)

            Technical Business Rule Format:
            (CountDocument('PSS') = 1) AND Sum(^PSS22) > (^FITR71 + 1)

            Data Elements:

            RP:^FITR71 = tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            RP:^PSS22 = PSS:RP.Payer.{PSSeqNum}:rvctc2.02.14:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount
        */
        private void VRATOFITR434080(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.RPPayerPSSeqNumContextCollection != null &&
                ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS22).GetValueOrDefault() > (ParentDocument.FITR71 + 1);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Credit: FRW (excluding capital gains) must not be less than the total of all FRW amounts in Payment Summary schedule",
                    LongDescription = "The 'Credit: foreign resident withholding (excluding capital gains)' amount on the main form must not be less than the sum of all 'Foreign resident withholding' amounts on the Payment Summary schedule",
                    Location = "/tns:FITR/tns:RP/tns:IncomeTaxCalculation/tns:PayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434080" },
                        new ProcessMessageParameter { Name = "FITR71", Value = ParentDocument.FITR71.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "PSS22", Value = ChildDocument.RPPayerPSSeqNumContextCollection.Sum(x => x.PSS22).GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion      

        #region VR.ATO.PSS.000035

        /*  VR.ATO.PSS.000035
            Payment Type not applicable to Non-individual income tax return

            Legacy Rule Format:
            IF PARENT RETURN <> “IITR” AND ([PSS20] <> NULLORBLANK OR [PSS21] <> NULLORBLANK OR [PSS31] <> NULLORBLANK OR [PSS24] <> NULLORBLANK OR [PSS25] <> NULLORBLANK OR [PSS32] <> NULLORBLANK)
                RETURN VALIDATION MESSAGE
            END IF
            
            Data Elements:

            RP:^PSS20 = PSS:RP.Payer.{PSSeqNum}:tns:Remuneration.VoluntaryAgreementGross.Amount
            RP:^PSS21 = PSS:RP.Payer.{PSSeqNum}:tns:Remuneration.LabourHireArrangementPaymentGross.Amount
            RP:^PSS24 = PSS:RP.Payer.{PSSeqNum}:tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount
            RP:^PSS25 = PSS:RP.Payer.{PSSeqNum}:tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount
            RP:^PSS31 = PSS:RP.Payer.{PSSeqNum}:tns:Remuneration.PersonalServicesIncome.AttributedGross.Amount
            RP:^PSS32 = PSS:RP.Payer.{PSSeqNum}:tns:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount   
       */
        private void VRATOPSS000035(List<ProcessMessageDocument> response)
        {
            if (ChildDocument.RPPayerPSSeqNumContextCollection != null)
            {
                IEnumerable<PSS2018.RPPayerPSSeqNumContext> contexts = ChildDocument.RPPayerPSSeqNumContextCollection.Where(
                    p => p.PSS20.HasValue || p.PSS21.HasValue || p.PSS24.HasValue || p.PSS25.HasValue || p.PSS31.HasValue || p.PSS32.HasValue
                );

                IEnumerable<ProcessMessageDocument> processMessages = contexts.Select(p =>
                    new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PSS.000035",
                        Severity = ProcessMessageSeverity.Error,
                        Description = "Payment Type not applicable to Non-individual income tax return",
                        Location = $"/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount[contextRef=\"{p.Id}\"]",
                        Parameters = new ProcessMessageParameters {
                            new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PSS.000035" },
                            new ProcessMessageParameter { Name = "PSS20", Value = p.PSS20.Value.ToString() },
                            new ProcessMessageParameter { Name = "PSS21", Value = p.PSS21.Value.ToString() },
                            new ProcessMessageParameter { Name = "PSS24", Value = p.PSS24.Value.ToString() },
                            new ProcessMessageParameter { Name = "PSS25", Value = p.PSS25.Value.ToString() },
                            new ProcessMessageParameter { Name = "PSS31", Value = p.PSS31.Value.ToString() },
                            new ProcessMessageParameter { Name = "PSS32", Value = p.PSS32.Value.ToString() }
                        }
                    }
                );

                response.AddRange(processMessages);
            }
        }

        #endregion // VR.ATO.PSS.000035

        #region VR.ATO.PSS.000036

        /*  VR.ATO.PSS.000036
            Your supplied TFN does not match the TFN supplied on the form it was submitted with

            Legacy Rule Format:
            WHERE IN CONTEXT (RP.Payer.{PSSeqNum)
            IF (RP.Payer.{PSSeqNum}.entity.identifier.TFN <> PARENT RETURN:RP.entity.identifier.TFN)
               RETURN VALIDATION MESSAGE
            ENDIF
        */
        private void VRATOPSS000036(List<ProcessMessageDocument> response)
        {
            if (ChildDocument.RPPayerPSSeqNumContextCollection != null)
            {
                IEnumerable<PSS2018.RPPayerPSSeqNumContext> contexts = ChildDocument.RPPayerPSSeqNumContextCollection.Where(
                    p => p.IdentifierTFN != ParentDocument.FITR5
                );

                IEnumerable<ProcessMessageDocument> processMessages = contexts.Select((p, i) => new { RPPayer = p, Index = i }).Select(p =>
                    new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PSS.000036",
                        Severity = ProcessMessageSeverity.Error,
                        Description = "Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                        Location = p.Index == 0 ? "xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier" : $"xbrli:xbrl/xbrli:context[{p.Index + 1}]/xbrli:entity/xbrli:identifier",
                        Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PSS.000036" } }
                    }
                );

                response.AddRange(processMessages);
            }
        }

        #endregion // VR.ATO.PSS.000036
    }
}