using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using Au.Gov.Sbr.Xbrl.Document;
using DataContracts;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Runtime.CompilerServices;
using System.IO;
using System.Text;
using System.Xml.Schema;
using System.Reflection;
using System.Runtime.Serialization; 

namespace Ato.EN.IntegrationServices.CodeGenerationLS
{
    public class LS2017Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public LS2017 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public LS2017 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            LS2017 report;
            report = new LS2017();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrances being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) ));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.LS1 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS3 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                        report.OrganisationNameDetailsCollectionExists = true;
                        report.OrganisationNameDetailsCollectionCount = sbrOrganisationNameDetailsCollection.Count();
                        SBRElement sbrOrganisationNameDetails;
                        sbrOrganisationNameDetails = sbrOrganisationNameDetailsCollection.ElementAt(0);
                
                        IDictionary<string, SBRElement> sbrOrganisationNameDetailsTupleElementMap;
                        sbrOrganisationNameDetailsTupleElementMap = sbrOrganisationNameDetails.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                        if (sbrOrganisationNameDetailsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                        {
                            if (!currentValue.IsNil) report.LS75 = currentValue.Value;
                        }
                
                        if (sbrOrganisationNameDetailsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.LS2 = currentValue.Value;
                        }
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CurrentYearLossProvisionsApply.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS24 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS20 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS21 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS22 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS80 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS25 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS26 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS27 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS28 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS81 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS82 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.ControlledForeignCompany.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS64 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.ControlledForeignCompanyDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS44 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ControlledForeignCompany.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS45 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS65 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.Uplift.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS79 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.ForgivenDebtNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS67 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.Incurred.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS68 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.ConversionOfExcessFrankingOffsets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS69 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.ExemptIncomeNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS70 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.Forgone.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS71 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS72 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.TransferredOut.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS73 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS74 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPY0Context
            IEnumerable<SBRContext> sbrRPY0Contexts = null;
            SBRContext sbrRPY0Context = null;
    
            sbrRPY0Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Contexts != null && sbrRPY0Contexts.Count() > 0)
            {
                sbrRPY0Context = sbrRPY0Contexts.First();
        
                report.RPY0Id = sbrRPY0Context.Id;
                report.RPY0Identifier = sbrRPY0Context.EntityIdentifier;
                report.RPY0IdentifierScheme = sbrRPY0Context.EntityScheme;
        
                report.RPY0Count = sbrRPY0Contexts.Count();
                report.RPY0Exists = true;
                report.RPY0OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Context) + 1;
                report.RPY0LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Contexts.Last()) + 1;
        

                if (report.RPY0IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0IdentifierTFN = report.RPY0Identifier;
    
                report.RPY0StartDate = ((SBRDurationPeriod)sbrRPY0Context.Period).DurationStartAsDateObject;
                report.RPY0EndDate = ((SBRDurationPeriod)sbrRPY0Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Contexts.Count(); i++)
                {
                    sbrRPY0Context = sbrRPY0Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS5 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPY0ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS15 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Context
    
            #region sbrRPY0Minus1Context
            IEnumerable<SBRContext> sbrRPY0Minus1Contexts = null;
            SBRContext sbrRPY0Minus1Context = null;
    
            sbrRPY0Minus1Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus1")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus1ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus1Contexts != null && sbrRPY0Minus1Contexts.Count() > 0)
            {
                sbrRPY0Minus1Context = sbrRPY0Minus1Contexts.First();
        
                report.RPY0Minus1Id = sbrRPY0Minus1Context.Id;
                report.RPY0Minus1Identifier = sbrRPY0Minus1Context.EntityIdentifier;
                report.RPY0Minus1IdentifierScheme = sbrRPY0Minus1Context.EntityScheme;
        
                report.RPY0Minus1Count = sbrRPY0Minus1Contexts.Count();
                report.RPY0Minus1Exists = true;
                report.RPY0Minus1OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus1Context) + 1;
                report.RPY0Minus1LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus1Contexts.Last()) + 1;
        

                if (report.RPY0Minus1IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus1IdentifierTFN = report.RPY0Minus1Identifier;
    
                report.RPY0Minus1StartDate = ((SBRDurationPeriod)sbrRPY0Minus1Context.Period).DurationStartAsDateObject;
                report.RPY0Minus1EndDate = ((SBRDurationPeriod)sbrRPY0Minus1Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Minus1ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus1Contexts.Count(); i++)
                {
                    sbrRPY0Minus1Context = sbrRPY0Minus1Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus1Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus1ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus1ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus1ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus1ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS6 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPY0Minus1ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS16 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus1Context
    
            #region sbrRPY0Minus2Context
            IEnumerable<SBRContext> sbrRPY0Minus2Contexts = null;
            SBRContext sbrRPY0Minus2Context = null;
    
            sbrRPY0Minus2Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus2")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus2ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus2Contexts != null && sbrRPY0Minus2Contexts.Count() > 0)
            {
                sbrRPY0Minus2Context = sbrRPY0Minus2Contexts.First();
        
                report.RPY0Minus2Id = sbrRPY0Minus2Context.Id;
                report.RPY0Minus2Identifier = sbrRPY0Minus2Context.EntityIdentifier;
                report.RPY0Minus2IdentifierScheme = sbrRPY0Minus2Context.EntityScheme;
        
                report.RPY0Minus2Count = sbrRPY0Minus2Contexts.Count();
                report.RPY0Minus2Exists = true;
                report.RPY0Minus2OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus2Context) + 1;
                report.RPY0Minus2LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus2Contexts.Last()) + 1;
        

                if (report.RPY0Minus2IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus2IdentifierTFN = report.RPY0Minus2Identifier;
    
                report.RPY0Minus2StartDate = ((SBRDurationPeriod)sbrRPY0Minus2Context.Period).DurationStartAsDateObject;
                report.RPY0Minus2EndDate = ((SBRDurationPeriod)sbrRPY0Minus2Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Minus2ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus2Contexts.Count(); i++)
                {
                    sbrRPY0Minus2Context = sbrRPY0Minus2Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus2Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus2ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus2ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus2ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus2ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS7 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPY0Minus2ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS17 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus2Context
    
            #region sbrRPY0Minus3Context
            IEnumerable<SBRContext> sbrRPY0Minus3Contexts = null;
            SBRContext sbrRPY0Minus3Context = null;
    
            sbrRPY0Minus3Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus3")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus3ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus3Contexts != null && sbrRPY0Minus3Contexts.Count() > 0)
            {
                sbrRPY0Minus3Context = sbrRPY0Minus3Contexts.First();
        
                report.RPY0Minus3Id = sbrRPY0Minus3Context.Id;
                report.RPY0Minus3Identifier = sbrRPY0Minus3Context.EntityIdentifier;
                report.RPY0Minus3IdentifierScheme = sbrRPY0Minus3Context.EntityScheme;
        
                report.RPY0Minus3Count = sbrRPY0Minus3Contexts.Count();
                report.RPY0Minus3Exists = true;
                report.RPY0Minus3OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus3Context) + 1;
                report.RPY0Minus3LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus3Contexts.Last()) + 1;
        

                if (report.RPY0Minus3IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus3IdentifierTFN = report.RPY0Minus3Identifier;
    
                report.RPY0Minus3StartDate = ((SBRDurationPeriod)sbrRPY0Minus3Context.Period).DurationStartAsDateObject;
                report.RPY0Minus3EndDate = ((SBRDurationPeriod)sbrRPY0Minus3Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Minus3ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus3Contexts.Count(); i++)
                {
                    sbrRPY0Minus3Context = sbrRPY0Minus3Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus3Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus3ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus3ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus3ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus3ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS8 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPY0Minus3ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS18 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus3Context
    
            #region sbrRPY0Minus4Context
            IEnumerable<SBRContext> sbrRPY0Minus4Contexts = null;
            SBRContext sbrRPY0Minus4Context = null;
    
            sbrRPY0Minus4Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus4")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus4ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus4Contexts != null && sbrRPY0Minus4Contexts.Count() > 0)
            {
                sbrRPY0Minus4Context = sbrRPY0Minus4Contexts.First();
        
                report.RPY0Minus4Id = sbrRPY0Minus4Context.Id;
                report.RPY0Minus4Identifier = sbrRPY0Minus4Context.EntityIdentifier;
                report.RPY0Minus4IdentifierScheme = sbrRPY0Minus4Context.EntityScheme;
        
                report.RPY0Minus4Count = sbrRPY0Minus4Contexts.Count();
                report.RPY0Minus4Exists = true;
                report.RPY0Minus4OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus4Context) + 1;
                report.RPY0Minus4LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus4Contexts.Last()) + 1;
        

                if (report.RPY0Minus4IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus4IdentifierTFN = report.RPY0Minus4Identifier;
    
                report.RPY0Minus4StartDate = ((SBRDurationPeriod)sbrRPY0Minus4Context.Period).DurationStartAsDateObject;
                report.RPY0Minus4EndDate = ((SBRDurationPeriod)sbrRPY0Minus4Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Minus4ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus4Contexts.Count(); i++)
                {
                    sbrRPY0Minus4Context = sbrRPY0Minus4Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus4Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus4ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus4ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus4ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus4ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS9 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPY0Minus4ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS19 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus4Context
    
            #region sbrRPY0Minus5PContext
            IEnumerable<SBRContext> sbrRPY0Minus5PContexts = null;
            SBRContext sbrRPY0Minus5PContext = null;
    
            sbrRPY0Minus5PContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus5Plus")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus5PElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus5PContexts != null && sbrRPY0Minus5PContexts.Count() > 0)
            {
                sbrRPY0Minus5PContext = sbrRPY0Minus5PContexts.First();
        
                report.RPY0Minus5PId = sbrRPY0Minus5PContext.Id;
                report.RPY0Minus5PIdentifier = sbrRPY0Minus5PContext.EntityIdentifier;
                report.RPY0Minus5PIdentifierScheme = sbrRPY0Minus5PContext.EntityScheme;
        
                report.RPY0Minus5PCount = sbrRPY0Minus5PContexts.Count();
                report.RPY0Minus5PExists = true;
                report.RPY0Minus5POccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus5PContext) + 1;
                report.RPY0Minus5PLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus5PContexts.Last()) + 1;
        

                if (report.RPY0Minus5PIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus5PIdentifierTFN = report.RPY0Minus5PIdentifier;
    
                report.RPY0Minus5PStartDate = ((SBRDurationPeriod)sbrRPY0Minus5PContext.Period).DurationStartAsDateObject;
                report.RPY0Minus5PEndDate = ((SBRDurationPeriod)sbrRPY0Minus5PContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Minus5PElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus5PContexts.Count(); i++)
                {
                    sbrRPY0Minus5PContext = sbrRPY0Minus5PContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus5PContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus5PElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus5PElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus5PElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus5PElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS49 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPY0Minus5PElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS53 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus5PContext
    
            #region sbrRPALLContext
            IEnumerable<SBRContext> sbrRPALLContexts = null;
            SBRContext sbrRPALLContext = null;
    
            sbrRPALLContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}All")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPALLElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPALLContexts != null && sbrRPALLContexts.Count() > 0)
            {
                sbrRPALLContext = sbrRPALLContexts.First();
        
                report.RPALLId = sbrRPALLContext.Id;
                report.RPALLIdentifier = sbrRPALLContext.EntityIdentifier;
                report.RPALLIdentifierScheme = sbrRPALLContext.EntityScheme;
        
                report.RPALLCount = sbrRPALLContexts.Count();
                report.RPALLExists = true;
                report.RPALLOccurrenceIndex = Array.IndexOf(contextArray, sbrRPALLContext) + 1;
                report.RPALLLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPALLContexts.Last()) + 1;
        

                if (report.RPALLIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPALLIdentifierTFN = report.RPALLIdentifier;
    
                report.RPALLStartDate = ((SBRDurationPeriod)sbrRPALLContext.Period).DurationStartAsDateObject;
                report.RPALLEndDate = ((SBRDurationPeriod)sbrRPALLContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPALLElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPALLContexts.Count(); i++)
                {
                    sbrRPALLContext = sbrRPALLContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPALLContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPALLElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPALLElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPALLElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPALLElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS50 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPALLContext
    
            #region sbrRPY0InstantContext
            IEnumerable<SBRContext> sbrRPY0InstantContexts = null;
            SBRContext sbrRPY0InstantContext = null;
    
            sbrRPY0InstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0InstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0InstantContexts != null && sbrRPY0InstantContexts.Count() > 0)
            {
                sbrRPY0InstantContext = sbrRPY0InstantContexts.First();
        
                report.RPY0InstantId = sbrRPY0InstantContext.Id;
                report.RPY0InstantIdentifier = sbrRPY0InstantContext.EntityIdentifier;
                report.RPY0InstantIdentifierScheme = sbrRPY0InstantContext.EntityScheme;
        
                report.RPY0InstantCount = sbrRPY0InstantContexts.Count();
                report.RPY0InstantExists = true;
                report.RPY0InstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0InstantContext) + 1;
                report.RPY0InstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0InstantContexts.Last()) + 1;
        

                if (report.RPY0InstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0InstantIdentifierTFN = report.RPY0InstantIdentifier;
    
                report.RPY0InstantInstantDate = ((SBRInstantPeriod)sbrRPY0InstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPY0InstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0InstantContexts.Count(); i++)
                {
                    sbrRPY0InstantContext = sbrRPY0InstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0InstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0InstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0InstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0InstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0InstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS10 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0InstantContext
    
            #region sbrRPY0Minus1InstantContext
            IEnumerable<SBRContext> sbrRPY0Minus1InstantContexts = null;
            SBRContext sbrRPY0Minus1InstantContext = null;
    
            sbrRPY0Minus1InstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus1")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus1InstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus1InstantContexts != null && sbrRPY0Minus1InstantContexts.Count() > 0)
            {
                sbrRPY0Minus1InstantContext = sbrRPY0Minus1InstantContexts.First();
        
                report.RPY0Minus1InstantId = sbrRPY0Minus1InstantContext.Id;
                report.RPY0Minus1InstantIdentifier = sbrRPY0Minus1InstantContext.EntityIdentifier;
                report.RPY0Minus1InstantIdentifierScheme = sbrRPY0Minus1InstantContext.EntityScheme;
        
                report.RPY0Minus1InstantCount = sbrRPY0Minus1InstantContexts.Count();
                report.RPY0Minus1InstantExists = true;
                report.RPY0Minus1InstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus1InstantContext) + 1;
                report.RPY0Minus1InstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus1InstantContexts.Last()) + 1;
        

                if (report.RPY0Minus1InstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus1InstantIdentifierTFN = report.RPY0Minus1InstantIdentifier;
    
                report.RPY0Minus1InstantInstantDate = ((SBRInstantPeriod)sbrRPY0Minus1InstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPY0Minus1InstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus1InstantContexts.Count(); i++)
                {
                    sbrRPY0Minus1InstantContext = sbrRPY0Minus1InstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus1InstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus1InstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus1InstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus1InstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus1InstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS11 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus1InstantContext
    
            #region sbrRPY0Minus2InstantContext
            IEnumerable<SBRContext> sbrRPY0Minus2InstantContexts = null;
            SBRContext sbrRPY0Minus2InstantContext = null;
    
            sbrRPY0Minus2InstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus2")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus2InstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus2InstantContexts != null && sbrRPY0Minus2InstantContexts.Count() > 0)
            {
                sbrRPY0Minus2InstantContext = sbrRPY0Minus2InstantContexts.First();
        
                report.RPY0Minus2InstantId = sbrRPY0Minus2InstantContext.Id;
                report.RPY0Minus2InstantIdentifier = sbrRPY0Minus2InstantContext.EntityIdentifier;
                report.RPY0Minus2InstantIdentifierScheme = sbrRPY0Minus2InstantContext.EntityScheme;
        
                report.RPY0Minus2InstantCount = sbrRPY0Minus2InstantContexts.Count();
                report.RPY0Minus2InstantExists = true;
                report.RPY0Minus2InstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus2InstantContext) + 1;
                report.RPY0Minus2InstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus2InstantContexts.Last()) + 1;
        

                if (report.RPY0Minus2InstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus2InstantIdentifierTFN = report.RPY0Minus2InstantIdentifier;
    
                report.RPY0Minus2InstantInstantDate = ((SBRInstantPeriod)sbrRPY0Minus2InstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPY0Minus2InstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus2InstantContexts.Count(); i++)
                {
                    sbrRPY0Minus2InstantContext = sbrRPY0Minus2InstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus2InstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus2InstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus2InstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus2InstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus2InstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS12 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus2InstantContext
    
            #region sbrRPY0Minus3InstantContext
            IEnumerable<SBRContext> sbrRPY0Minus3InstantContexts = null;
            SBRContext sbrRPY0Minus3InstantContext = null;
    
            sbrRPY0Minus3InstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus3")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus3InstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus3InstantContexts != null && sbrRPY0Minus3InstantContexts.Count() > 0)
            {
                sbrRPY0Minus3InstantContext = sbrRPY0Minus3InstantContexts.First();
        
                report.RPY0Minus3InstantId = sbrRPY0Minus3InstantContext.Id;
                report.RPY0Minus3InstantIdentifier = sbrRPY0Minus3InstantContext.EntityIdentifier;
                report.RPY0Minus3InstantIdentifierScheme = sbrRPY0Minus3InstantContext.EntityScheme;
        
                report.RPY0Minus3InstantCount = sbrRPY0Minus3InstantContexts.Count();
                report.RPY0Minus3InstantExists = true;
                report.RPY0Minus3InstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus3InstantContext) + 1;
                report.RPY0Minus3InstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus3InstantContexts.Last()) + 1;
        

                if (report.RPY0Minus3InstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus3InstantIdentifierTFN = report.RPY0Minus3InstantIdentifier;
    
                report.RPY0Minus3InstantInstantDate = ((SBRInstantPeriod)sbrRPY0Minus3InstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPY0Minus3InstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus3InstantContexts.Count(); i++)
                {
                    sbrRPY0Minus3InstantContext = sbrRPY0Minus3InstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus3InstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus3InstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus3InstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus3InstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus3InstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS13 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus3InstantContext
    
            #region sbrRPY0Minus4InstantContext
            IEnumerable<SBRContext> sbrRPY0Minus4InstantContexts = null;
            SBRContext sbrRPY0Minus4InstantContext = null;
    
            sbrRPY0Minus4InstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus4")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus4InstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus4InstantContexts != null && sbrRPY0Minus4InstantContexts.Count() > 0)
            {
                sbrRPY0Minus4InstantContext = sbrRPY0Minus4InstantContexts.First();
        
                report.RPY0Minus4InstantId = sbrRPY0Minus4InstantContext.Id;
                report.RPY0Minus4InstantIdentifier = sbrRPY0Minus4InstantContext.EntityIdentifier;
                report.RPY0Minus4InstantIdentifierScheme = sbrRPY0Minus4InstantContext.EntityScheme;
        
                report.RPY0Minus4InstantCount = sbrRPY0Minus4InstantContexts.Count();
                report.RPY0Minus4InstantExists = true;
                report.RPY0Minus4InstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus4InstantContext) + 1;
                report.RPY0Minus4InstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus4InstantContexts.Last()) + 1;
        

                if (report.RPY0Minus4InstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus4InstantIdentifierTFN = report.RPY0Minus4InstantIdentifier;
    
                report.RPY0Minus4InstantInstantDate = ((SBRInstantPeriod)sbrRPY0Minus4InstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPY0Minus4InstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus4InstantContexts.Count(); i++)
                {
                    sbrRPY0Minus4InstantContext = sbrRPY0Minus4InstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus4InstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus4InstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus4InstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus4InstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus4InstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS14 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus4InstantContext
    
            #region sbrRPY0Minus5PInstantContext
            IEnumerable<SBRContext> sbrRPY0Minus5PInstantContexts = null;
            SBRContext sbrRPY0Minus5PInstantContext = null;
    
            sbrRPY0Minus5PInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus5Plus")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus5PInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus5PInstantContexts != null && sbrRPY0Minus5PInstantContexts.Count() > 0)
            {
                sbrRPY0Minus5PInstantContext = sbrRPY0Minus5PInstantContexts.First();
        
                report.RPY0Minus5PInstantId = sbrRPY0Minus5PInstantContext.Id;
                report.RPY0Minus5PInstantIdentifier = sbrRPY0Minus5PInstantContext.EntityIdentifier;
                report.RPY0Minus5PInstantIdentifierScheme = sbrRPY0Minus5PInstantContext.EntityScheme;
        
                report.RPY0Minus5PInstantCount = sbrRPY0Minus5PInstantContexts.Count();
                report.RPY0Minus5PInstantExists = true;
                report.RPY0Minus5PInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus5PInstantContext) + 1;
                report.RPY0Minus5PInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus5PInstantContexts.Last()) + 1;
        

                if (report.RPY0Minus5PInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus5PInstantIdentifierTFN = report.RPY0Minus5PInstantIdentifier;
    
                report.RPY0Minus5PInstantInstantDate = ((SBRInstantPeriod)sbrRPY0Minus5PInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPY0Minus5PInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus5PInstantContexts.Count(); i++)
                {
                    sbrRPY0Minus5PInstantContext = sbrRPY0Minus5PInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus5PInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus5PInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus5PInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus5PInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus5PInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS51 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus5PInstantContext
    
            #region sbrRPALLInstantContext
            IEnumerable<SBRContext> sbrRPALLInstantContexts = null;
            SBRContext sbrRPALLInstantContext = null;
    
            sbrRPALLInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}All")) )));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPALLInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPALLInstantContexts != null && sbrRPALLInstantContexts.Count() > 0)
            {
                sbrRPALLInstantContext = sbrRPALLInstantContexts.First();
        
                report.RPALLInstantId = sbrRPALLInstantContext.Id;
                report.RPALLInstantIdentifier = sbrRPALLInstantContext.EntityIdentifier;
                report.RPALLInstantIdentifierScheme = sbrRPALLInstantContext.EntityScheme;
        
                report.RPALLInstantCount = sbrRPALLInstantContexts.Count();
                report.RPALLInstantExists = true;
                report.RPALLInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPALLInstantContext) + 1;
                report.RPALLInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPALLInstantContexts.Last()) + 1;
        

                if (report.RPALLInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPALLInstantIdentifierTFN = report.RPALLInstantIdentifier;
    
                report.RPALLInstantInstantDate = ((SBRInstantPeriod)sbrRPALLInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPALLInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPALLInstantContexts.Count(); i++)
                {
                    sbrRPALLInstantContext = sbrRPALLInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPALLInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPALLInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPALLInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPALLInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPALLInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ls}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.LS52 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPALLInstantContext
            return report;

        } // Of Consume Method
    } // Of Class
} // Of Namespace

