using System.Linq;
using System.Collections.Generic;
using VaTS;
using DataContracts;
using Ato.EN.IntegrationServices.CodeGenerationPSS;
using Ato.EN.IntegrationServices.CodeGenerationRS;
using Ato.EN.IntegrationServices.CodeGenerationIEE;
using Ato.EN.IntegrationServices.CodeGenerationIDS;
using Ato.EN.IntegrationServices.CodeGenerationPTR;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;

namespace Ato.CD.Inbound.PTR202402
{
    internal class CrossFormValidator
    {
        private PTR2024 ParentDocument { get; }
        private BusinessDocumentCollection ChildDocuments { get; }

        private IEnumerable<PSS2018> PSSChildDocuments { get; }
        private IEnumerable<RS2018> RSChildDocuments { get; }
        private IEnumerable<IEE2021> IEEChildDocuments { get; }
        private IEnumerable<IDS2024> IDSChildDocuments { get; }
        private IEnumerable<DISTBENTRT2024> DISTBENTRTChildDocuments { get; }

        public CrossFormValidator(PTR2024 report, BusinessDocumentCollection childDocuments)
        {
            ParentDocument = report;
            ChildDocuments = childDocuments ?? new BusinessDocumentCollection();

            PSSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.PSS.ToString()))
                    .Select(p => (PSS2018)p.ConsumedReport);

            RSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.RS.ToString()))
                    .Select(p => (RS2018)p.ConsumedReport);

            IEEChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.IEE.ToString()))
                    .Select(p => (IEE2021)p.ConsumedReport);

            IDSChildDocuments = ChildDocuments
                        .Where(s => s.DocumentName.Equals(Schedule.IDS.ToString()))
                        .Select(p => (IDS2024)p.ConsumedReport);

            DISTBENTRTChildDocuments = ChildDocuments
                      .Where(s => s.DocumentName.Equals(Schedule.DISTBENTRT.ToString()))
                      .Select(p => (DISTBENTRT2024)p.ConsumedReport);
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            var response = new List<ProcessMessageDocument>();

            //PSS CrossForm Validation
            VRATOPTR430025(response);
            VRATOPTR430183(response);
            VRATOPTR430020(response);
            VRATOPTR430019(response);

            //RS CrossForm Validation
            VRATOPTR430090(response);
            VRATOPTR430081(response);
            VRATOPTR430088(response);
            VRATOPTR430083(response);
            VRATOPTR430084(response);
            VRATOPTR430086(response);

            //IEE CrossForm Validation
            VRATOPTR430180(response);

            //IDS CrossForm Validation
            VRATOGEN438002(response);
            VRATOGEN438003(response);
            VRATOPTR430998(response);
            VRATOPTR430999(response);
            VRATOPTR430996(response);
            VRATOPTR430995(response);
            VRATOPTR431003(response);
            VRATOPTR431001(response);
            VRATOPTR431000(response);
            VRATOPTR431002(response);
            VRATOPTR431014(response);
            VRATOPTR431015(response);
            VRATOPTR431045(response);
            VRATOPTR432000(response);
            VRATOGEN438017(response);
            VRATOPTR440038(response);

            return response;
        }

        #region IDS CrossForm Validation Rules


        #region VR.ATO.GEN.438002
        /*
        Rule ID: VR.ATO.GEN.438002
            
        IF (bafpr3.xx.xx:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator = TRUE)
        AND (COUNT(SCHEDULE = "IDS") = 0)
            RETURN VALIDATION MESSAGE
        ENDIF
           
        [PTR102] = PTR:RPJFOREIGN:tns:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
            */
        private void VRATOGEN438002(List<ProcessMessageDocument> response)
        {
            var PTR102 = ParentDocument.PTR102.GetValueOrDefault();

            bool assertion = (PTR102 == true) && (IDSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "International Dealings Schedule is not attached",
                    LongDescription = "The answer to the question 'Did the entity have either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust?' is 'Yes', and no International Dealings Schedule is attached.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = " VR.ATO.GEN.438002" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438003
        /*
        Rule ID: VR.ATO.GEN.438003
            
        IF (rvctc4.xx.xx:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator = TRUE) AND (COUNT(SCHEDULE = "IDS") = 0)
        RETURN VALIDATION MESSAGE
        ENDIF
           
        [PTR108] = PTR:RP:tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
            */
        private void VRATOGEN438003(List<ProcessMessageDocument> response)
        {
            var PTR108 = ParentDocument.PTR108.GetValueOrDefault();

            bool assertion = (PTR108 == true) && (IDSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "International Dealings Schedule is not attached",
                    LongDescription = "The answer to the question 'Was the aggregate amount of your transactions or dealings with international related parties (including the value of any property or service transferred or the balance of any loans) greater than $2 million?'(Overseas transactions) is 'Yes' , and no International Dealings Schedule is attached.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438003" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430998
        /*
        Rule ID: VR.ATO.PTR.430998
            
        IF ([PTR98] > 0 OR [PTR100] > 0) AND (([PTR98] +  [PTR100]) < ([IDS191] + [IDS192] + [IDS193]))
           RETURN VALIDATION MESSAGE
        ENDIF

        [IDS191] = IDS:RP:bafpr1.02.12:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
        [IDS192] = IDS:RP:bafpr1.02.12:Income.ControlledForeignCompaniesChangeOfResidenceIncome.Amount
        [IDS193] = IDS:RP:bafpr1.02.12:Income.ControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotal.Amount
        [PTR98] = PTR:RP.JFOREIGN.CFCLC:bafpr3.02.02:Income.InternationalDealings.Attributed.Amount
        [PTR100] = PTR:RP.JFOREIGN.CFCUC:bafpr3.02.02:Income.InternationalDealings.Attributed.Amount
        */
        private void VRATOPTR430998(List<ProcessMessageDocument> response)
        {
            var PTR98 = ParentDocument.PTR98.GetValueOrDefault();
            var PTR100 = ParentDocument.PTR100.GetValueOrDefault();
            var IDS191 = IDSChildDocuments.Select(ids => ids.IDS191.GetValueOrDefault()).Sum();
            var IDS192 = IDSChildDocuments.Select(ids => ids.IDS192.GetValueOrDefault()).Sum();
            var IDS193 = IDSChildDocuments.Select(ids => ids.IDS193.GetValueOrDefault()).Sum();

            bool assertion = (PTR98 > 0 || PTR100 > 0) && ((PTR98 + PTR100) < (IDS191 + IDS192 + IDS193));

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.430998",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Attributed foreign income incorrect",
                    LongDescription = "The sum of amounts at attributable income of controlled foreign companies and controlled foreign trusts on the IDS should be equal to or less than the sum of amounts at Listed country and Unlisted country on the partnership return form.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = " VR.ATO.PTR.430998" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430999
        /*
        Rule ID: VR.ATO.PTR.430999
            
        IF ([PTR98] > 0) AND ([IDS188] > [PTR98])
           RETURN VALIDATION MESSAGE
        ENDIF

        [IDS188] = IDS:RP.CFCLC:bafpr1.02.12:Income.ControlledForeignCompaniesAttributableIncome.Amount
        [PTR98] = PTR:RP.JFOREIGN.CFCLC:bafpr3.02.02:Income.InternationalDealings.Attributed.Amount
        */
        private void VRATOPTR430999(List<ProcessMessageDocument> response)
        {
            var PTR98 = ParentDocument.PTR98.GetValueOrDefault();
            var IDS188 = IDSChildDocuments.Select(ids => ids.IDS188.GetValueOrDefault()).Sum();

            bool assertion = (PTR98 > 0) && (IDS188 > PTR98);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.430999",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Attributed foreign income incorrect",
                    LongDescription = "Assessable income of controlled foreign companies and controlled foreign trusts under section 456 for Listed countries on the IDS should be equal to or less than Attributed foreign income - Listed country on the partnership return.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430999" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430996
        /*
        Rule ID: VR.ATO.PTR.430996
            
        IF ([PTR138] >  0) AND ([PTR138] < [IDS22])
           RETURN VALIDATION MESSAGE
        ENDIF

        [IDS22] = IDS:RP.Royalties:bafpr1.02.12:Expense.DeductionsTotal.Amount
        [PTR138] = PTR:RP.JFOREIGN:bafpr1.02.27:Expense.Royalties.Amount
        */
        private void VRATOPTR430996(List<ProcessMessageDocument> response)
        {
            var PTR138 = ParentDocument.PTR138.GetValueOrDefault();
            var IDS22 = IDSChildDocuments.Select(ids => ids.IDS22.GetValueOrDefault()).Sum();

            bool assertion = (PTR138 > 0) && (PTR138 < IDS22);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.430996",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Royalties deductions incorrect",
                    LongDescription = "If Royalties deductions has been completed on the IDS then it must be less than or equal to Royalty expenses overseas on the partnership return form.",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430996" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430995
        /*
        Rule ID: VR.ATO.PTR.430995
            
        IF ([PTR138] = 0 OR [PTR138] = NULL) AND ([IDS22] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        [IDS22] = IDS:RP.Royalties:bafpr1.02.12:Expense.DeductionsTotal.Amount
        [PTR138] = PTR:RP.JFOREIGN:bafpr1.02.27:Expense.Royalties.Amount
        */
        private void VRATOPTR430995(List<ProcessMessageDocument> response)
        {
            var PTR138 = ParentDocument.PTR138;
            var IDS22 = IDSChildDocuments.Select(ids => ids.IDS22.GetValueOrDefault()).Sum();

            bool assertion = (PTR138 == 0 || PTR138 == null) && (IDS22 > 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.430995",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Royalty expenses overseas required",
                    LongDescription = "If a Royalties deduction amount has been completed on the IDS then Royalty expenses overseas must be greater than zero on the partnership form.",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430995" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431003
        /*
        Rule ID: VR.ATO.PTR.431003
            
        IF ([PTR104] = NULL) AND ([IDS219] > 0)
           RETURN VALIDATION MESSAGE
        ENDIF

        [IDS219] = IDS:RP:bafpo7.02.11:Assets.Investment.ForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotal.Amount
        [PTR104] = PTR:RP.JFOREIGN:bafpr1.02.04:Income.Other.Amount
        */
        private void VRATOPTR431003(List<ProcessMessageDocument> response)
        {
            var PTR104 = ParentDocument.PTR104;
            var IDS219 = IDSChildDocuments.Select(ids => ids.IDS219.GetValueOrDefault()).Sum();

            bool assertion = (PTR104 == null) && (IDS219 > 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.431003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross other assessable foreign source income required",
                    LongDescription = "If Total amount of your share of net income/distribution of profit has been completed on the IDS then the Gross Other assessable foreign source income must be completed on the partnership return form.",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431003" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431001
        /*
        Rule ID: VR.ATO.PTR.431001
            
        IF [PTR109] = FALSE AND [IDS9] = TRUE
           RETURN VALIDATION MESSAGE
        ENDIF

        [IDS9] = IDS:RP:rvctc4.02.05:InternationalDealings.SpecifiedCountriesTransactions.Indicator
        [PTR109] = PTR:RP:rvctc4.02.11:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator
        */
        private void VRATOPTR431001(List<ProcessMessageDocument> response)
        {
            var PTR109 = ParentDocument.PTR109;
            var IDS9 = IDSChildDocuments.Select(ids => ids.IDS9);

            bool assertion = PTR109 == false && IDS9.Contains(true);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.431001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Transactions with specified countries question incorrect",
                    LongDescription = "If there is a 'True' response at 'Did you have dealings with international related parties located in specified countries?' on the IDS then the response at Overseas transactions - Transactions with specified countries on the partnership return must be 'True'.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431001" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431000
        /*  VR.ATO.PTR.431000
                    Attributed foreign income - Listed country or Attributed foreign income - Unlisted country has been completed and the number of controlled foreign companies or trusts on the IDS, has not been shown. Show zero at Number of CFCs and CFTs - Listed countries, Number of CFCs and CFTs - Specified countries, Number of CFC and CFTs - Other unlisted countries on the IDS if at the end of your income year you no longer held an interest in CFCs or CFTs.

                Legacy Rule Format:
                    IF ([PTR98] > 0 OR [PTR100] > 0) AND ([IDS185] = NULL AND [IDS186] = NULL AND [IDS187] = NULL)
                       RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [IDS185] = /tns:IDS/tns:RP/tns:Listed/tns:MiscellaneousControlledForeignCompaniesTrustsInterestCt
                    [IDS186] = /tns:IDS/tns:RP/tns:Specified/tns:MiscellaneousControlledForeignCompaniesTrustsInterestCt
                    [IDS187] = /tns:IDS/tns:RP/tns:Unlisted/tns:MiscellaneousControlledForeignCompaniesTrustsInterestCt
                    [PTR98] = PTR:RP.JFOREIGN.CFCLC:bafpr3.02.02:Income.InternationalDealings.Attributed.Amount
                    [PTR100] = PTR:RP.JFOREIGN.CFCUC:bafpr3.02.02:Income.InternationalDealings.Attributed.Amount

            Technical Business Rule Format:
                    (^PTR98 > 0 OR ^PTR100 > 0) AND (^IDS185 = NULL AND ^IDS186 = NULL AND ^IDS187 = NULL)

            Data Elements:
                ^PTR98 = RP.JFOREIGN.CFCLC:Income.InternationalDealings.Attributed.Amount
                ^IDS185 = IDS185
                ^IDS186 = IDS186
                ^IDS187 = IDS187
                ^PTR100 = RP.JFOREIGN.CFCUC:Income.InternationalDealings.Attributed.Amount
        */
        private void VRATOPTR431000(List<ProcessMessageDocument> response)
        {
            var PTR98 = ParentDocument.PTR98.GetValueOrDefault();
            var PTR100 = ParentDocument.PTR100.GetValueOrDefault();
            var IDS185 = IDSChildDocuments.Select(ids => ids.IDS185).Count(i => i != null) == 0;
            var IDS186 = IDSChildDocuments.Select(ids => ids.IDS186).Count(i => i != null) == 0;
            var IDS187 = IDSChildDocuments.Select(ids => ids.IDS187).Count(i => i != null) == 0;

            bool assertion = (PTR98 > 0 || PTR100 > 0) && (IDS185 && IDS186 && IDS187);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.434000",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Number of controlled foreign companies or trusts required",
                    LongDescription = @"Attributed foreign income - Listed country or Attributed foreign income - Unlisted country has been completed and the number of controlled foreign companies or trusts on the IDS has not been shown. Show zero on the IDS if at the end of your income year you no longer held an interest in CFCs or CFTs.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431000" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431002
        /*
        Rule ID: VR.ATO.PTR.431002
            
        IF ([PTR346] = NULL OR [PTR346] = 0 ) AND (ANY OCCURRENCE OF([PTR96] = NULL OR [PTR96] = 0)) AND ([PTR137] = NULL OR [PTR137] = 0 ) AND ([IDS207] > 0)
           RETURN VALIDATION MESSAGE
        ENDIF

        [IDS207] = IDS:RP:rvctc2.02.09:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncomeTotal.Amount
        [PTR96] = PTR:RP.JAUS:bafpr1.02.04:Expense.DeductibleOther.Amount IN TUPLE(ptr.0007.lodge.req.02.00:OtherDeductions)
        [PTR137] = PTR:RP.JFOREIGN:bafpr1.02.05:Expense.Interest.Amount
        [PTR346] = PTR:RP.JAUS:bafpr2.02.17:Expense.Operating.Interest.Amount
        */
        private void VRATOPTR431002(List<ProcessMessageDocument> response)
        {
            var PTR346 = ParentDocument.PTR346;
            var PTR96IsNullOccurs = ParentDocument.OtherDeductionsCollection != null ? ParentDocument.OtherDeductionsCollection.Any(o => o.PTR96 == null) : true;
            var PTR96ValueIsZeroOccurs = ParentDocument.OtherDeductionsCollection != null ? ParentDocument.OtherDeductionsCollection.Any(o => o.PTR96 == 0) : true;
            var PTR137 = ParentDocument.PTR137;
            var IDS207 = IDSChildDocuments.Select(ids => ids.IDS207).Sum();

            bool assertion = (PTR346 == null || PTR346 == 0) && ((PTR96IsNullOccurs || PTR96ValueIsZeroOccurs)) && (PTR137 == null || PTR137 == 0) && (IDS207 > 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.431002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Interest expenses required",
                    LongDescription = "If Debt deductions in earning non-assessable foreign income claimed under s25-90 has been completed on the IDS then Total interest expenses or Interest expenses overseas must be greater than zero on the partnership form.",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Interest.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431002" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431014
        /*
        Rule ID: VR.ATO.PTR.431014
            
        IF ([PTR137] > 0) AND (COUNT(SCHEDULE = "IDS") = 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        [PTR137] = PTR:RP.JFOREIGN:bafpr1.02.05:Expense.Interest.Amount
        */
        private void VRATOPTR431014(List<ProcessMessageDocument> response)
        {
            var PTR137 = ParentDocument.PTR137.GetValueOrDefault();

            bool assertion = (PTR137 > 0) && (IDSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.431014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "International Dealings Schedule not attached",
                    LongDescription = "An ‘International dealings schedule’ (IDS) is not attached when a value greater than zero is present at label Interest expenses overseas",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431014" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431015
        /*
        Rule ID: VR.ATO.PTR.431015
            
        IF ([PTR138] > 0) AND (COUNT(SCHEDULE = "IDS") = 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        [PTR138] = PTR:RP.JFOREIGN:bafpr1.02.27:Expense.Royalties.Amount
        */
        private void VRATOPTR431015(List<ProcessMessageDocument> response)
        {
            var PTR138 = ParentDocument.PTR138.GetValueOrDefault();

            bool assertion = (PTR138 > 0) && (IDSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.431015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "International Dealings Schedule not attached",
                    LongDescription = "An ‘International dealings schedule’ (IDS) is not attached when a value greater than zero is present at label Royalty expenses overseas",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431015" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.431045
        /*
        Rule ID: VR.ATO.PTR.431045
            
        IF
        ([PTR102] = FALSE AND [PTR108] = FALSE AND [PTR277] = FALSE) AND (([PTR137] = NULL OR [PTR137] = 0) AND ([PTR138] = NULL OR [PTR138] = 0)) AND (COUNT (Schedule = “IDS”) = 1)
            RETURN VALIDATION MESSAGE
        ENDIF

        [PTR102] = PTR:RP.JFOREIGN:bafpr3.02.03:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
        [PTR108] = PTR:RP:rvctc4.02.10:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
        [PTR137] = PTR:RP.JFOREIGN:bafpr1.02.05:Expense.Interest.Amount
        [PTR138] = PTR:RP.JFOREIGN:bafpr1.02.27:Expense.Royalties.Amount
        [PTR277] = PTR:RP:bafpo6.02.15:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        private void VRATOPTR431045(List<ProcessMessageDocument> response)
        {
            var PTR102 = ParentDocument.PTR102.GetValueOrDefault();
            var PTR108 = ParentDocument.PTR108.GetValueOrDefault();
            var PTR277 = ParentDocument.PTR277.GetValueOrDefault();
            var PTR137 = ParentDocument.PTR137;
            var PTR138 = ParentDocument.PTR138;

            bool assertion = (PTR102 == false && PTR108 == false && PTR277 == false) && ((PTR137 == null || PTR137 == 0) && (PTR138 == null || PTR138 == 0)) && (IDSChildDocuments.Count() == 1);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "International Dealings Schedule is not required",
                    LongDescription = "If there is a 'NO' response to the questions concerning whether the entity had either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust and whether the value of dealings with international related parties is greater than $2 million and whether the thin capitalisation provisions apply, and no amounts have been completed at Interest expenses overseas and Royalty expenses overseas, an International Dealings schedule must not be attached.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431045" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.432000
        /*
        Rule ID: VR.ATO.PTR.432000
            
        IF ([PTR251] > 0 OR [PTR252] > 0) 
        AND (COUNT(SCHEDULE = "IDS") = 1 AND [IDS177] <> TRUE)
            RETURN VALIDATION MESSAGE
        ENDIF

        [IDS177] = IDS:RP:rvctc4.02.05:InternationalDealings.TaxationOfFinancialArrangementsRulesApplied.Indicator
        [PTR251] = PTR:RP.TOFA:bafpr1.02.04:Income.GainsTotal.Amount
        [PTR252] = PTR:RP.TOFA:bafpr1.02.04:Expense.LossesTotal.Amount
        */
        private void VRATOPTR432000(List<ProcessMessageDocument> response)
        {
            var PTR251 = ParentDocument.PTR251.GetValueOrDefault();
            var PTR252 = ParentDocument.PTR252.GetValueOrDefault();
            var IDS177 = IDSChildDocuments.Select(ids => ids.IDS177).FirstOrDefault();

            bool assertion = (PTR251 > 0 || PTR252 > 0)
            && (IDSChildDocuments.Count() == 1 && IDS177 != true);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.433000",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "If there is an amount for Total TOFA losses or Total TOFA gains, then 'Are you subject to the Taxation of Financial Arrangements rules' must be true.",
                    LongDescription = "If there is an amount at either 'Total TOFA gains' or 'Total TOFA Losses' that indicates you made a gain or loss from a financial arrangement subject to the TOFA rules, then there must be a 'True' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' on the IDS.",
                    Location = "/xbrli:xbrl/tns:Income.GainsTotal.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.432000" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438017
        /*
        Rule ID: VR.ATO.GEN.438017
            
        IF (RP:bafpo6.xx.xx:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator = TRUE) AND (COUNT(SCHEDULE = "IDS") = 0)
          RETURN VALIDATION MESSAGE
        ENDIF

        [PTR277] = PTR:RP:tns:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        private void VRATOGEN438017(List<ProcessMessageDocument> response)
        {
            var PTR277 = ParentDocument.PTR277.GetValueOrDefault();
            var PTR252 = ParentDocument.PTR252.GetValueOrDefault();
            var IDS177 = IDSChildDocuments.Select(ids => ids.IDS177).FirstOrDefault();

            bool assertion = (PTR277 == true) && (IDSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "International Dealings Schedule not attached",
                    LongDescription = "If the thin capitalisation rules apply then International Dealings Schedule must be attached",
                    Location = "/xbrli:xbrl/tns:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438017" }
                    }
                });
            }
        }
        #endregion

        #endregion

        #region IEE CrossForm Validation Rules
        
        #region VR.ATO.PTR.430180
        /*
        Rule ID: VR.ATO.PTR.430180
        
        IF ([PTR27] = "R") AND (COUNT(SCHEDULE = "IEE") = 0)
            RETURN VALIDATION MESSAGE
        ENDIF
            
        [PTR27] = PTR:RP:Elections.InterposedEntityElectionRevocation.Code
            */
        private void VRATOPTR430180(List<ProcessMessageDocument> response)
        {
            bool assertion = !string.IsNullOrWhiteSpace(ParentDocument.PTR27)
                && ParentDocument.PTR27.ToUpper() == "R" && IEEChildDocuments.Count() == 0;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430180",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "If the Interposed Entity Election revocation code is present, an IEE schedule must be present",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionRevocation.Code",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430180" }
                    }
                });
            }
        }
        #endregion


        #endregion


        #region PSS CrossForm Validation Rules


        #region VR.ATO.PTR.430025
        private void VRATOPTR430025(List<ProcessMessageDocument> response)
        {
            /*
        IF (COUNT(SCHEDULE = "PSS") > 0) AND [PTR38] <> SUM([PSS18])
          RETURN VALIDATION MESSAGE
        ENDIF

        [PSS18] = PSS:RP.Payer.{PSSeqNum}:lrla.02.00:Remuneration.PaymentToForeignResidentGross.Amount
        [PTR38] = PTR:RP.JAUS.NonPrim:lrla.02.00:Remuneration.PaymentToForeignResidentGross.Amount
            */
            var PTR38 = ParentDocument.PTR38.GetValueOrDefault();
            var sumPSS18 = PSSChildDocuments
                    .Where(rp => rp.RPPayerPSSeqNumContextCollection != null)
                    .SelectMany(rp => rp.RPPayerPSSeqNumContextCollection
                    .Select(p => p.PSS18.GetValueOrDefault())).Sum();

            var assertion = (PSSChildDocuments.Count() > 0) && PTR38 != sumPSS18;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.430025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross payment - Foreign Resident Withholding (excluding capital gains) is incorrect ",
                    LongDescription = "The sum of gross payments with indicator 'F' in the Payment Summary schedule must be equal to the sum of total gross payment subject to foreign resident withholding (excluding capital gains) primary production and total gross payment subject to foreign resident withholding (excluding capital gains) non-primary production on the main return form",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430025" } }
                });
            }
        }

        #endregion

        #region VR.ATO.PTR.430183
        private void VRATOPTR430183(List<ProcessMessageDocument> response)
        {
            /*
                IF ([PTR63] > 0) AND ((COUNT(SCHEDULE = "PSS") = 0) OR (SUM([PSS23]) <> [PTR63]))
                RETURN VALIDATION ERROR
                ENDIF

                [PSS23] = #ELEMENT PSS23 NOT IN TIMS#
                [PTR63] = PTR:RP.JAUS:rvctc2.02.00:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            */
            var PTR63 = ParentDocument.PTR63.GetValueOrDefault();
            var sumPSS23 = PSSChildDocuments
                    .Where(rp => rp.RPPayerPSSeqNumContextCollection != null)
                    .SelectMany(rp => rp.RPPayerPSSeqNumContextCollection
                    .Select(p => p.PSS23.GetValueOrDefault())).Sum();

            bool assertion = (PTR63 > 0) && ((PSSChildDocuments.Count() == 0) || (sumPSS23 != PTR63));

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430183",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Payment Summary schedule must include Tax withheld where ABN not quoted amount",
                    LongDescription = "If an amount greater than zero has been entered at Tax withheld from payments where ABN no quoted, then a Payment Summary schedule must be present and it must include the same total amount of Tax withheld from payments where ABN no quoted",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430183" } }
                });
            }
        }

        #endregion

        #region VR.ATO.PTR.430020

        //        IF(COUNT(SCHEDULE = "PSS") > 0)  AND SUM([PSS19]) <> [PTR35] + [PTR36]
        //        RETURN VALIDATION MESSAGE
        //      ENDIF

        //      [PSS19] = PSS:RP.Payer.{PSSeqNum
        //    }:lrla.02.00:Remuneration.ABNNotQuotedPaymentGross.Amount
        //[PTR35] = PTR:RP.JAUS.Prim:lrla.02.00:Remuneration.ABNNotQuotedPaymentGross.Amount
        //[PTR36] = PTR:RP.JAUS.NonPrim:lrla.02.00:Remuneration.ABNNotQuotedPaymentGross.Amount
        private void VRATOPTR430020(List<ProcessMessageDocument> response)
        {
            var sumPSS19 = PSSChildDocuments
                    .Where(rp => rp.RPPayerPSSeqNumContextCollection != null)
                    .SelectMany(rp => rp.RPPayerPSSeqNumContextCollection
                    .Select(p => p.PSS19.GetValueOrDefault())).Sum();

            var PTR35 = ParentDocument.PTR35.GetValueOrDefault();
            var PTR36 = ParentDocument.PTR36.GetValueOrDefault();

            var assertion = (PSSChildDocuments.Count() > 0) && sumPSS19 != PTR35 + PTR36;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross payment where ABN not quoted must equal sum of payments on schedule",
                    LongDescription = "The sum of gross payment with indicator 'N' in the Payment Summary schedule must equal the sum of Primary production and Non-primary production Gross payments where ABN not quoted on main return form",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430020" } }
                });
            }
        }
        #endregion


        #region VR.ATO.PTR.430019
        //Rule ID: VR.ATO.PTR.430019

        //IF[PTR35] + [PTR36] > 0 AND(COUNT(SCHEDULE = "PSS") = 0)
        //    RETURN VALIDATION MESSAGE
        //ENDIF

        //[PTR35] = PTR:RP.JAUS.Prim:Remuneration.ABNNotQuotedPaymentGross.Amount
        //[PTR36] = PTR:RP.JAUS.NonPrim:Remuneration.ABNNotQuotedPaymentGross.Amount
        private void VRATOPTR430019(List<ProcessMessageDocument> response)
        {
            var PTR35 = ParentDocument.PTR35.GetValueOrDefault();
            var PTR36 = ParentDocument.PTR36.GetValueOrDefault();

            bool assertion = PTR35 + PTR36 > 0 && (PSSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Payment Summary schedule must be present if Gross payments where ABN not quoted is present",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430019" } }
                });
            }
        }
        #endregion


        #endregion


        #region RS CrossForm Validation Rules

        #region VR.ATO.PTR.430090
        /*   IF  (COUNT(SCHEDULE = "RS") > 0) AND 
        (SUM(ALL OCCURRENCES OF([RS40] + [RS41] + [RS42] + [RS43] + [RS44] + [RS45] + [RS46] + [RS47] + 
        [RS49] + [RS50] + [RS51] + [RS52] + [RS53] + [RS55] + [RS56] + [RS57] + [RS58])) <> [PTR80])
        RETURN VALIDATION MESSAGE
        ENDIF

        [PTR80] = PTR:RP.JAUS:rvctc2.02.00:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
        [RS40] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Marketing.Amount
        [RS41] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.BodyCorporate.Amount
        [RS42] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Borrowing.Amount
        [RS43] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Cleaning.Amount
        [RS44] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.CouncilRates.Amount
        [RS45] = RS:RP:bafpr1.02.04:Expense.DepreciationandAmortisation.Amount
        [RS46] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Gardening.Amount
        [RS47] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Insurance.Amount
        [RS49] = RS:RP:bafpr1.02.00:Expense.LandTax.Amount
        [RS50] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.LegalFees.Amount
        [RS51] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.PestControl.Amount
        [RS52] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.AgentFeesCommission.Amount
        [RS53] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.RepairsAndMaintenance.Amount
        [RS55] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.OfficeSupplies.Amount
        [RS56] = RS:RP:bafpr1.02.00:Expense.Travel.Amount
        [RS57] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Water.Amount
        [RS58] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.Sundry.Amount
        */
        private void VRATOPTR430090(List<ProcessMessageDocument> response)
        {
            var sumRS40 = RSChildDocuments.Sum(rp => rp.RS40.GetValueOrDefault());
            var sumRS41 = RSChildDocuments.Sum(rp => rp.RS41.GetValueOrDefault());
            var sumRS42 = RSChildDocuments.Sum(rp => rp.RS42.GetValueOrDefault());
            var sumRS43 = RSChildDocuments.Sum(rp => rp.RS43.GetValueOrDefault());
            var sumRS44 = RSChildDocuments.Sum(rp => rp.RS44.GetValueOrDefault());
            var sumRS45 = RSChildDocuments.Sum(rp => rp.RS45.GetValueOrDefault());
            var sumRS46 = RSChildDocuments.Sum(rp => rp.RS46.GetValueOrDefault());
            var sumRS47 = RSChildDocuments.Sum(rp => rp.RS47.GetValueOrDefault());
            var sumRS49 = RSChildDocuments.Sum(rp => rp.RS49.GetValueOrDefault());
            var sumRS50 = RSChildDocuments.Sum(rp => rp.RS50.GetValueOrDefault());
            var sumRS51 = RSChildDocuments.Sum(rp => rp.RS51.GetValueOrDefault());
            var sumRS52 = RSChildDocuments.Sum(rp => rp.RS52.GetValueOrDefault());
            var sumRS53 = RSChildDocuments.Sum(rp => rp.RS53.GetValueOrDefault());
            var sumRS55 = RSChildDocuments.Sum(rp => rp.RS55.GetValueOrDefault());
            var sumRS56 = RSChildDocuments.Sum(rp => rp.RS56.GetValueOrDefault());
            var sumRS57 = RSChildDocuments.Sum(rp => rp.RS57.GetValueOrDefault());
            var sumRS58 = RSChildDocuments.Sum(rp => rp.RS58.GetValueOrDefault());
            var PTR80 = ParentDocument.PTR80.GetValueOrDefault();


            var assertion = (RSChildDocuments.Count() > 0) &&
        (sumRS40 + sumRS41 + sumRS42 + sumRS43 + sumRS44 + sumRS45 + sumRS46 + sumRS47 +
        sumRS49 + sumRS50 + sumRS51 + sumRS52 + sumRS53 + sumRS55 + sumRS56 + sumRS57 + sumRS58 != PTR80);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430090",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Other rental deductions must equal the total other deductions in the Rental Schedules",
                    LongDescription = "The Other rental deductions amount in the main form must equal the total of all rental expenses in the Rental property, excluding Interest deductions and Capital works deductions",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430090" } }
                });
            }
        }
        #endregion


        #region VR.ATO.PTR.430081
        /*  IF ([PTR77] <> NULL OR [PTR78] > 0 OR [PTR79] > 0 OR [PTR80] > 0) AND (COUNT(SCHEDULE = "RS") = 0)
        RETURN VALIDATION MESSAGE
        ENDIF


        [PTR77] = PTR:RP.JAUS:bafpr2.02.00:Income.Operating.RentalIncomeGross.Amount
        [PTR78] = PTR:RP.JAUS:bafpr1.02.05:Expense.Interest.Amount
        [PTR79] = PTR:RP.JAUS:bafpr1.02.02:Expense.CapitalWorksDeduction.Amount
        [PTR80] = PTR:RP.JAUS:rvctc2.02.00:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
        */
        private void VRATOPTR430081(List<ProcessMessageDocument> response)
        {
            var PTR77 = ParentDocument.PTR77;
            var PTR78 = ParentDocument.PTR78.GetValueOrDefault();
            var PTR79 = ParentDocument.PTR79.GetValueOrDefault();
            var PTR80 = ParentDocument.PTR80.GetValueOrDefault();

            bool assertion = (PTR77 != null || PTR78 > 0 || PTR79 > 0 || PTR80 > 0) && (RSChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430081",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Rental schedule must be present",
                    LongDescription = "If there is a positive amount at Gross rent or at any Rental deductions item, then a Rental property schedule must be submitted for each rental property",
                    Location = "/xbrli:xbrl/tns:Income.Operating.RentalIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430081" } }
                });
            }
        }
        #endregion


        #region VR.ATO.PTR.430088
        /* IF  (COUNT(SCHEDULE = "RS") > 0) AND (SUM(ALL OCCURRENCES OF([RS54]))) <> [PTR79]
            RETURN VALIDATION MESSAGE
        ENDIF

        [PTR79] = PTR:RP.JAUS:bafpr1.02.02:Expense.CapitalWorksDeduction.Amount
        [RS54] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.CapitalWorksDeduction.Amount
        */
        private void VRATOPTR430088(List<ProcessMessageDocument> response)
        {
            var sumRS54 = RSChildDocuments.Sum(rp => rp.RS54.GetValueOrDefault());
            var PTR79 = ParentDocument.PTR79.GetValueOrDefault();

            bool assertion = (RSChildDocuments.Count() > 0) && (sumRS54) != PTR79;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430088",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Capital works deductions must equal the total Capital works deductions in the Rental Schedules",
                    LongDescription = "The Capital works deductions amount in the main form must equal the total of all Capital works deductions in the Rental property schedules",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalWorksDeduction.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430088" } }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430083

        //IF(COUNT(SCHEDULE = "RS") > 0) AND(SUM(ALL OCCURRENCES OF([RS38] + [RS39])) <> [PTR77])
        //  RETURN VALIDATION MESSAGE
        //ENDIF

        //[PTR77] = PTR:RP.JAUS:bafpr2.02.00:Income.Operating.RentalIncomeGross.Amount
        //[RS38] = RS:RP:bafpr2.02.04:Income.RealEstateProperty.Rental.Amount
        //[RS39] = RS:RP:bafpr2.02.04:Income.RealEstateProperty.RentalRelatedOther.Amount
        private void VRATOPTR430083(List<ProcessMessageDocument> response)
        {
            var sumRS38 = RSChildDocuments.Sum(rp => rp.RS38.GetValueOrDefault());
            var sumRS39 = RSChildDocuments.Sum(rp => rp.RS39.GetValueOrDefault());
            var PTR77 = ParentDocument.PTR77.GetValueOrDefault();

            bool assertion = (RSChildDocuments.Count() > 0) && (sumRS38 + sumRS39 != PTR77);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430083",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross rent must equal the total Gross rent in the Rental Schedules",
                    LongDescription = "The Gross rent amount in the main form must equal the total Gross rent present in all Rental property schedules",
                    Location = "/xbrli:xbrl/tns:Income.Operating.RentalIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430083" } }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430084

        //IF(COUNT(SCHEDULE = "RS") > 0) AND SUM(ALL OCCURRENCES OF([RS59])) <> ([PTR77] - [PTR78] - [PTR79] - [PTR80])
        //  RETURN VALIDATION MESSAGE
        //ENDIF

        //[PTR77] = PTR:RP.JAUS:bafpr2.02.00:Income.Operating.RentalIncomeGross.Amount
        //[PTR78] = PTR:RP.JAUS:bafpr1.02.05:Expense.Interest.Amount
        //[PTR79] = PTR:RP.JAUS:bafpr1.02.02:Expense.CapitalWorksDeduction.Amount
        //[PTR80] = PTR:RP.JAUS:rvctc2.02.00:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
        //[RS59] = RS:RP:bafpr2.02.04:Income.RealEstateProperty.RentalNet.Amount
        private void VRATOPTR430084(List<ProcessMessageDocument> response)
        {
            var sumRS59 = RSChildDocuments.Sum(rp => rp.RS59.GetValueOrDefault());
            var PTR77 = ParentDocument.PTR77.GetValueOrDefault();
            var PTR78 = ParentDocument.PTR78.GetValueOrDefault();
            var PTR79 = ParentDocument.PTR79.GetValueOrDefault();
            var PTR80 = ParentDocument.PTR80.GetValueOrDefault();

            bool assertion = (RSChildDocuments.Count() > 0) && sumRS59 != (PTR77 - PTR78 - PTR79 - PTR80);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430084",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Net rent must equal the total net rent in the Rental Schedules",
                    LongDescription = "The net rent amount in the main form (Gross rent minus rental deduction amounts) must equal the total Net rent present in all Rental property schedules",
                    Location = "/xbrli:xbrl/tns:Income.Operating.RentalIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430084" } }
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.430086

        //IF(COUNT(SCHEDULE = "RS") > 0) AND(SUM(ALL OCCURRENCES OF([RS48]))) <> [PTR78]
        //  RETURN VALIDATION MESSAGE
        //ENDIF

        //[PTR78] = PTR:RP.JAUS:bafpr1.02.05:Expense.Interest.Amount
        //[RS48] = RS:RP:bafpr2.02.04:Expense.RealEstateProperty.LoanInterest.Amount
        private void VRATOPTR430086(List<ProcessMessageDocument> response)
        {
            var sumRS48 = RSChildDocuments.Sum(rp => rp.RS48.GetValueOrDefault());
            var PTR78 = ParentDocument.PTR78.GetValueOrDefault();

            bool assertion = (RSChildDocuments.Count() > 0) && (sumRS48) != PTR78;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430086",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Rental Interest deductions must equal the total Interest deductions in the Rental Schedules",
                    LongDescription = "The Interest deductions amount in the main form must equal the total of all Interest deductions in the Rental property schedules",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430086" } }
                });
            }
        }
        #endregion

        #endregion

        #region DISTBENTRT CrossForm Validation Rules

        #region VR.ATO.PTR.440038

        /*  VR.ATO.PTR.440038
                    Distributions to Beneficiaries of Trust schedule must be present when Trust distributions - Income amounts or credits are provided.

                Legacy Rule Format:
                    IF COUNT(SCHEDULE = "DISTBENTRT") = 0 AND ([PTR66] > 0 OR [PTR70] > 0 OR [PTR322] > 0)
                       RETURN VALIDATION MESSAGE
                    ENDIF

            Technical Business Rule Format:
                    CountDocument('DISTBENTRT') = 0 AND (^PTR66 > 0 OR ^PTR70 > 0 OR ^PTR322 > 0)

        Data Elements:
    
            ^PTR66 = RP.JAUS.Prim:Income.TrustShareNet.Amount
    
            ^PTR70 = RP.JAUS.NonPrim:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR322 = RP.JAUS.NonPrim:Income.TrustDistributionFranked.Amount
        */
        protected void VRATOPTR440038(List<ProcessMessageDocument> response)
        {

            bool assertion = (ParentDocument.PTR66.GetValueOrDefault() > 0 || ParentDocument.PTR70.GetValueOrDefault() > 0 || ParentDocument.PTR322.GetValueOrDefault() > 0)
                  && (DISTBENTRTChildDocuments.Count() == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.PTR.440038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distributions to Beneficiaries of Trust schedule must be present",             
                    LongDescription = @"Distributions to Beneficiaries of Trust schedule must be present when Trust distributions - Income amounts or credits are provided",
                    Location = "/xbrli:xbrl/tns:Income.TrustShareNet.Amount",

                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440038" },
                        new ProcessMessageParameter { Name = "PTR66", Value = ParentDocument.PTR66.HasValue ? ParentDocument.PTR66.GetValueOrDefault().ToString(): "null" },
                        new ProcessMessageParameter { Name = "PTR70", Value = ParentDocument.PTR70.HasValue ? ParentDocument.PTR70.GetValueOrDefault().ToString(): "null" },
                        new ProcessMessageParameter { Name = "PTR322", Value = ParentDocument.PTR322.HasValue ? ParentDocument.PTR322.GetValueOrDefault().ToString(): "null" }
                    }
                });
              
            }
        }
        #endregion // VR.ATO.PTR.440038


        #endregion

    }
}
