using System.Linq;
using System.Collections.Generic;
using Ato.CD.Inbound.Shared;
using VaTS;
using DataContracts;
using Ato.EN.IntegrationServices.CodeGenerationIEE;
using Ato.EN.IntegrationServices.CodeGenerationPTR;

namespace Ato.CD.Inbound.PTR202402
{
    internal class CrossFormValidatorIEE : ICrossFormValidator
    {

        private PTR2024 ParentDocument { get; }
        private IEE2021 ChildDocument { get; }

        internal CrossFormValidatorIEE(PTR2024 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (IEE2021)childDocument.ConsumedReport;
        }

        public IEnumerable<ProcessMessageDocument> ValidateCrossFormRules()
        {
            var response = new List<ProcessMessageDocument>();
            
            VRATOGEN410115(response);
            VRATOGEN410119(response);
            VRATOIEE410396(response);
            VRATOIEE410193(response);
            VRATOIEE410069(response);
            VRATOIEE410084(response);
            VRATOIEE410051(response);
            VRATOPTR430176(response);
            VRATOGEN430008(response);
            VRATOPTR430178(response);
            VRATOIEE410324(response);

            return response;
        }

        #region VR.ATO.GEN.410115
        /*
        WHERE PARENT RETURN EXISTS 
        IF (IEE:RP:entity.identifier.TFN <> NULLORBLANK) 
        AND (IEE:RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
               RETURN VALIDATION MESSAGE
        ENDIF
        */
        private void VRATOGEN410115(List<ProcessMessageDocument> response)
        {
            bool assertion = !(string.IsNullOrWhiteSpace(ChildDocument.IEE3)) && ChildDocument.IEE3 != ParentDocument.RPIdentifierTFN;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.410115",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "TFN on Schedule must match TFN on main form",
                    Location = "/tns:IEE/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410115" } }
                });
            }
        }
        #endregion
        
        #region VR.ATO.GEN.410119
        /*
        WHERE PARENT RETURN EXISTS
        IF (IEE:INT:entity.identifier.ABN <> NULLORBLANK) AND (IEE:INT:entity.identifier.ABN <> 
        PARENT RETURN:INT:entity.identifier.ABN)
              RETURN VALIDATION MESSAGE
        ENDIF
        */
        private void VRATOGEN410119(List<ProcessMessageDocument> response)
        {
            bool assertion = !(string.IsNullOrWhiteSpace(ChildDocument.IEE200)) && ChildDocument.IEE200 != ParentDocument.INTIdentifierABN;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.410119",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "ABN of the Intermediary on the schedule must match the ABN on the main form",
                    Location = "/tns:IEE/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410119" } }
                });
            }
        }
        #endregion

        #region VR.ATO.IEE.410396
        /*
        IF (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) 
        AND (PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) 
        AND (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier 
        <> PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier)
        RETURN VALIDATION MESSAGE
        ENDIF
        */
        private void VRATOIEE410396(List<ProcessMessageDocument> response)
        {
            bool assertion = !(string.IsNullOrWhiteSpace(ChildDocument.IEE162))
                          && !(string.IsNullOrWhiteSpace(ParentDocument.PTR7))
                          && ChildDocument.IEE162 != ParentDocument.PTR7;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Your supplied ABN does not match the ABN supplied on the form it was submitted with",
                    LongDescription = "Your supplied ABN does not match the ABN supplied on the parent form it was submitted with",
                    Location = "/tns:IEE/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410396" } }
                });
            }
        }
        #endregion

        #region VR.ATO.IEE.410193
        /*
           WHERE PARENT RETURN EXISTS
           IF ([IEE116] = ""R"") AND ([IEE50] = NULLORBLANK) 
           RETURN VALIDATION MESSAGE
           ENDIF
           [IEE50] = IEE:RP:pyid.02.00:Identifiers.TaxFileNumber.Identifier IN TUPLE(iee.0003.lodge.req.02.00:FamilyTrust)
           [IEE116] = IEE:RP:rvctc3.02.02:Elections.InterposedEntityElectionOrRevocation.Code
        */
        private void VRATOIEE410193(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.IEE116 == "R" && string.IsNullOrWhiteSpace(ChildDocument.IEE50);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410193",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "For a revocation, Family Trust tax file number must be present",
                    Location = "/tns:IEE/tns:RP/tns:FamilyTrust/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410193" } }
                });
            }
        }
        #endregion

        #region VR.ATO.IEE.410069
        /*
            WHERE PARENT RETURN EXISTS
            IF ([IEE82] <> NULLORBLANK) AND ([IEE116] = ""E"")
                RETURN VALIDATION MESSAGE
            ENDIF

            [IEE82] = IEE:RP:rvctc3.02.00:Elections.Revocation.Year
            [IEE116] = IEE:RP:rvctc3.02.02:Elections.InterposedEntityElectionOrRevocation.Code
        */
        private void VRATOIEE410069(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.IEE82.HasValue && ChildDocument.IEE116.ToUpper().Equals("E");
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Questions about revocation must not be answered if the form is used for an election",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsY",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410069" },
                        new ProcessMessageParameter { Name = "IEE82", Value = ChildDocument.IEE82.Value.ToString() }
                    }
                });
            }
        }
        #endregion
        
        #region VR.ATO.IEE.410084
        //WHERE PARENT RETURN EXISTS
        //IF([IEE82] <> NULLORBLANK) AND([IEE82] < "2011")
        //    RETURN VALIDATION MESSAGE
        //ENDIF
        private void VRATOIEE410084(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.IEE82.HasValue && ChildDocument.IEE82 < 2011;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410238",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Year specified for IEE revocation must be 2011 or later",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsY",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410084" },
                        new ProcessMessageParameter { Name = "IEE82", Value = ChildDocument.IEE82.Value.ToString() }
                    }
                });
            }
        }
        #endregion
        
        #region VR.ATO.IEE.410051
        //WHERE PARENT RETURN EXISTS
        //IF([IEE49] <> NULL) AND([IEE45] <> PARENT RETURN:Report.TargetFinancial.Year)
        //  RETURN VALIDATION MESSAGE
        //ENDIF
        private void VRATOIEE410051(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.IEE49.HasValue && ChildDocument.IEE45 != ParentDocument.PTR1;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410235",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Election Commencement Time must not be present if Income Year Specified is different to the parent return target financial year",
                    Location = "/tns:IEE/tns:RP/tns:InterposedEntityElectionRevocationDetails/tns:ElectionsCommencementD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410051" },
                        new ProcessMessageParameter { Name = "IEE49", Value = ChildDocument.IEE49.Value.ToString() }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430176
        /*
                Rule ID: VR.ATO.PTR.430176

                IF(PTR:RP:pyin.xx.xx:Report.TargetFinancial.Year<> NULL)
                  AND(COUNT(SCHEDULE = ""IEE"") > 0) 
                  AND(ANY OCCURRENCE OF ([IEE116] = ""R"")
                  AND([IEE82] <> PTR:RP:pyin.xx.xx:Report.TargetFinancial.Year))
                      RETURN VALIDATION MESSAGE
                ENDIF

                [IEE82] = IEE:RP:rvctc3.xx.xx:Elections.Revocation.Year
                [IEE116] = IEE:RP:rvctc3.xx.xx:Elections.InterposedEntityElectionOrRevocation.Code
        */
        private void VRATOPTR430176(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.PTR1.HasValue && !string.IsNullOrWhiteSpace(ChildDocument.IEE116) && ChildDocument.IEE116 == "R"
                && ChildDocument.IEE82.HasValue && ChildDocument.IEE82 != ParentDocument.PTR1;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430176",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Year of revocation must be the same as the year of the return",
                    LongDescription = "The income year for which the details of the Interposed Entity Election are being revoked must equal the income year of the main return being lodged.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430176" },
                        new ProcessMessageParameter { Name = "IEE82", Value = ChildDocument.IEE82.Value.ToString() }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.430008
        /*
        Rule ID: VR.ATO.GEN.430008

        IF(COUNT(SCHEDULE = ""IEE"") > 0) AND RP:entity.identifier.TFN = (ANY OCCURRENCE OF([IEE50])) 
            RETURN VALIDATION MESSAGE
        ENDIF

        [IEE50] = IEE:RP:pyid.xx.xx:Identifiers.TaxFileNumber.Identifier IN TUPLE(iee.0002.lodge.req.xx.xx:FamilyTrust)
*/
        private void VRATOGEN430008(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.RPIdentifierTFN == ChildDocument.IEE50;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "The Family Trust TFN on the IEE must not be the same as the main entity TFN",
                    LongDescription = "An Entity cannot interpose itself; therefore the family trust tax file number quoted in Interposed Entity Election cannot be the same as the tax file number of the entity making the election",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430008" },
                        new ProcessMessageParameter { Name = "IEE50", Value = ChildDocument.IEE50.ToString() }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.PTR.430178
        /*
        Rule ID: VR.ATO.PTR.430178
        
        IF[PTR26] = NULL AND(COUNT(SCHEDULE = ""IEE"") > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        [PTR26] = PTR:RP:Elections.InterposedEntityElectionStatus.Year
            */
        private void VRATOPTR430178(List<ProcessMessageDocument> response)
        {
            bool assertion = !ParentDocument.PTR26.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.430178",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "IEE specified income year must be present",
                    LongDescription = "If an IEE schedule is present, the Interposed entity election status - Specified income year must be present",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionStatus.Year",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430178" }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.IEE.410324

        /* VR.ATO.IEE.410324
        Reporting period on Schedule must match reporting period on main form

        WHERE PARENT RETURN EXISTS
        IF [IEE201] <> PARENT RETURN:RP:period.startDate OR [IEE202] <> PARENT RETURN:RP:period.endDate 
            RETURN VALIDATION MESSAGE
        ENDIF
        
        IEE201 = tns:PeriodStartD

        IEE202 = tns:PeriodEndD

        */
        private void VRATOIEE410324(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.IEE201 != ParentDocument.RPStartDate || ChildDocument.IEE202 != ParentDocument.RPEndDate;
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.IEE.410324",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Reporting period does not match the reporting period supplied on the form it was submitted with",
                    Location = "//tns:IEE/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.IEE.410324" },
                        new ProcessMessageParameter { Name = "IEE201", Value = ChildDocument.IEE201.ToString() },
                        new ProcessMessageParameter { Name = "IEE202", Value = ChildDocument.IEE202.ToString() },
                        new ProcessMessageParameter { Name = "RP:period.startDate", Value = ParentDocument.RPStartDate.ToString() },
                        new ProcessMessageParameter { Name = "RP:period.endDate", Value = ParentDocument.RPEndDate.ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }

        #endregion VR.ATO.IEE.410324

    }
}