using System;
using System.Collections.Generic;
using Au.Gov.Sbr.Xbrl.Document;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationPTR
{
    public class PTR2024Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }
				
        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public PTR2024 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public PTR2024 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            PTR2024 report;
            report = new PTR2024();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrences being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrINTContext
            IEnumerable<SBRContext> sbrINTContexts = null;
            SBRContext sbrINTContext = null;
    
            sbrINTContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Intermediary")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedINTElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrINTContexts != null && sbrINTContexts.Count() > 0)
            {
                sbrINTContext = sbrINTContexts.First();
        
                report.INTId = sbrINTContext.Id;
                report.INTIdentifier = sbrINTContext.EntityIdentifier;
                report.INTIdentifierScheme = sbrINTContext.EntityScheme;
        
                report.INTCount = sbrINTContexts.Count();
                report.INTExists = true;
                report.INTOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContext) + 1;
                report.INTLastOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContexts.Last()) + 1;
        

                if (report.INTIdentifierScheme == "http://www.ato.gov.au/abn") report.INTIdentifierABN = report.INTIdentifier;
    
                report.INTStartDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationStartAsDateObject;
                report.INTEndDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedINTElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrINTContexts.Count(); i++)
                {
                    sbrINTContext = sbrINTContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrINTContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedINTElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedINTElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedINTElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Identifiers.TaxAgentClientReference.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR170 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Identifiers.TaxAgentNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR2 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;
            
                    #region sbrDeclarationCollection
            
                    if (sbrDeclarationCollection != null)
                    {
                        report.DeclarationCollectionExists = true;
                        report.DeclarationCollectionCount = sbrDeclarationCollection.Count();
                
                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTCollection;
                        sbrDeclarationCollection_TrueAndCorrect_INTCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_INTCollection != null && sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_INT;
                            sbrDeclarationCollection_TrueAndCorrect_INT = sbrDeclarationCollection_TrueAndCorrect_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.DeclarationCollection_TrueAndCorrect_INTExists = true;
                            report.DeclarationCollection_TrueAndCorrect_INTCount = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INT.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_INTLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR296 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR295 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR298 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR294 = DateTime.Parse(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR297 = currentValue.Value;
                            }
                
                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;
                    
                            #region sbrDeclaration_PersonUnstructuredNameCollection
                    
                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                                report.Declaration_PersonUnstructuredNameCollectionExists = true;
                                report.Declaration_PersonUnstructuredNameCollectionCount = sbrDeclaration_PersonUnstructuredNameCollection.Count();
                        
                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null && 
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrINTContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.ElementAt(0);
                        
                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                        
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Last().OccurrenceIndex + 1;
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.PTR313 = currentValue.Value;
                                    }
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.PTR314 = currentValue.Value;
                                    }
                                }  
                            } 
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateINTDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrINTContexts, report);
                    
                        }  
                    } 
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateINTDeclarationCollection(errors, sbrDeclarationCollection, sbrINTContexts, report);
            
        
                    IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName", out currentValue))
                        sbrPersonUnstructuredNameCollection = currentValue.Occurrences;
                    else
                        sbrPersonUnstructuredNameCollection = null;
            
                    #region sbrPersonUnstructuredNameCollection
            
                    if (sbrPersonUnstructuredNameCollection != null)
                    {
                        report.PersonUnstructuredNameCollectionExists = true;
                        report.PersonUnstructuredNameCollectionCount = sbrPersonUnstructuredNameCollection.Count();
                
                        IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection_Contact_INTCollection;
                        sbrPersonUnstructuredNameCollection_Contact_INTCollection = sbrPersonUnstructuredNameCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "Contact" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrPersonUnstructuredNameCollection_Contact_INTCollection != null && sbrPersonUnstructuredNameCollection_Contact_INTCollection.Count() > 0)
                        {
                            SBRElement sbrPersonUnstructuredNameCollection_Contact_INT;
                            sbrPersonUnstructuredNameCollection_Contact_INT = sbrPersonUnstructuredNameCollection_Contact_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap;
                            sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap = sbrPersonUnstructuredNameCollection_Contact_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.PersonUnstructuredNameCollection_Contact_INTExists = true;
                            report.PersonUnstructuredNameCollection_Contact_INTCount = sbrPersonUnstructuredNameCollection_Contact_INTCollection.Count();
                            report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_INT.OccurrenceIndex + 1;
                            report.PersonUnstructuredNameCollection_Contact_INTLastOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR171 = currentValue.Value;
                            }
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.FullName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR242 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrPersonUnstructuredNameCollection
                    if (validationMode) ValidateINTPersonUnstructuredNameCollection(errors, sbrPersonUnstructuredNameCollection, sbrINTContexts, report);
            
        
                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;
            
                    #region sbrElectronicContactTelephoneCollection
            
                    if (sbrElectronicContactTelephoneCollection != null)
                    {
                        report.ElectronicContactTelephoneCollectionExists = true;
                        report.ElectronicContactTelephoneCollectionCount = sbrElectronicContactTelephoneCollection.Count();
                
                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_INTCollection;
                        sbrElectronicContactTelephoneCollection_03_INTCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_INTCollection != null && sbrElectronicContactTelephoneCollection_03_INTCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_INT;
                            sbrElectronicContactTelephoneCollection_03_INT = sbrElectronicContactTelephoneCollection_03_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_INTTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_INTTupleElementMap = sbrElectronicContactTelephoneCollection_03_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ElectronicContactTelephoneCollection_03_INTExists = true;
                            report.ElectronicContactTelephoneCollection_03_INTCount = sbrElectronicContactTelephoneCollection_03_INTCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INT.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_INTLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR245 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR244 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR172 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR173 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateINTElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrINTContexts, report);
            
                } 
            } 
            #endregion End of Context sbrINTContext
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.PTR5 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR1 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR7 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Report.Amendment.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR316 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Report.AmendmentSequence.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR319 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Report.AmendmentType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR317 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Report.AmendmentReason.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR318 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                        report.OrganisationNameDetailsCollectionExists = true;
                        report.OrganisationNameDetailsCollectionCount = sbrOrganisationNameDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_C_RPCollection;
                        sbrOrganisationNameDetailsCollection_MN_C_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_C_RP;
                            sbrOrganisationNameDetailsCollection_MN_C_RP = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrOrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_MN_C_RPExists = true;
                            report.OrganisationNameDetailsCollection_MN_C_RPCount = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR229 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR261 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR6 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_P_MN_RPCollection;
                        sbrOrganisationNameDetailsCollection_P_MN_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "P") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_P_MN_RPCollection != null && sbrOrganisationNameDetailsCollection_P_MN_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_P_MN_RP;
                            sbrOrganisationNameDetailsCollection_P_MN_RP = sbrOrganisationNameDetailsCollection_P_MN_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_P_MN_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_P_MN_RPTupleElementMap = sbrOrganisationNameDetailsCollection_P_MN_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_P_MN_RPExists = true;
                            report.OrganisationNameDetailsCollection_P_MN_RPCount = sbrOrganisationNameDetailsCollection_P_MN_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_P_MN_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_P_MN_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_P_MN_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_P_MN_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_P_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR230 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_P_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR262 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_P_MN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR8 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection1(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrAddressDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails", out currentValue))
                        sbrAddressDetailsCollection = currentValue.Occurrences;
                    else
                        sbrAddressDetailsCollection = null;
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                        report.AddressDetailsCollectionExists = true;
                        report.AddressDetailsCollectionCount = sbrAddressDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_C_RPCollection;
                        sbrAddressDetailsCollection_POS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_C_RPCollection != null && sbrAddressDetailsCollection_POS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_C_RP;
                            sbrAddressDetailsCollection_POS_C_RP = sbrAddressDetailsCollection_POS_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_POS_C_RPTupleElementMap = sbrAddressDetailsCollection_POS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_POS_C_RPExists = true;
                            report.AddressDetailsCollection_POS_C_RPCount = sbrAddressDetailsCollection_POS_C_RPCollection.Count();
                            report.AddressDetailsCollection_POS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR216 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR223 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR226 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR9 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR211 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR214 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR215 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR10 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR12 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR11 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR302 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_P_RPCollection;
                        sbrAddressDetailsCollection_POS_P_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "P" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_P_RPCollection != null && sbrAddressDetailsCollection_POS_P_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_P_RP;
                            sbrAddressDetailsCollection_POS_P_RP = sbrAddressDetailsCollection_POS_P_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_P_RPTupleElementMap;
                            sbrAddressDetailsCollection_POS_P_RPTupleElementMap = sbrAddressDetailsCollection_POS_P_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_POS_P_RPExists = true;
                            report.AddressDetailsCollection_POS_P_RPCount = sbrAddressDetailsCollection_POS_P_RPCollection.Count();
                            report.AddressDetailsCollection_POS_P_RPOccurrenceIndex = sbrAddressDetailsCollection_POS_P_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_P_RPLastOccurrenceIndex = sbrAddressDetailsCollection_POS_P_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR219 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR224 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR227 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR14 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR212 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR217 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR218 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR15 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR17 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR16 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR303 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection1(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrMailRecipientCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}MailRecipient", out currentValue))
                        sbrMailRecipientCollection = currentValue.Occurrences;
                    else
                        sbrMailRecipientCollection = null;
            
                    #region sbrMailRecipientCollection
            
                    if (sbrMailRecipientCollection != null)
                    {
                        report.MailRecipientCollectionExists = true;
                        report.MailRecipientCollectionCount = sbrMailRecipientCollection.Count();
                        SBRElement sbrMailRecipient;
                        sbrMailRecipient = sbrMailRecipientCollection.ElementAt(0);
                
                        IDictionary<string, SBRElement> sbrMailRecipientTupleElementMap;
                        sbrMailRecipientTupleElementMap = sbrMailRecipient.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
            
                        IEnumerable<SBRElement> sbrMailRecipient_PersonNameDetailsCollection;
                        if (sbrMailRecipientTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails", out currentValue))
                            sbrMailRecipient_PersonNameDetailsCollection = currentValue.Occurrences;
                        else
                            sbrMailRecipient_PersonNameDetailsCollection = null;
                
                        #region sbrMailRecipient_PersonNameDetailsCollection
                
                        if (sbrMailRecipient_PersonNameDetailsCollection != null)
                        {
                            report.MailRecipient_PersonNameDetailsCollectionExists = true;
                            report.MailRecipient_PersonNameDetailsCollectionCount = sbrMailRecipient_PersonNameDetailsCollection.Count();
                    
                            IEnumerable<SBRElement> sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection;
                            sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection = sbrMailRecipient_PersonNameDetailsCollection.Where(
                                ce => ce.ChildElements != null && 
                                ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "MailRecipient") &&
                                ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection != null && sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection.Count() > 0)
                            {
                                SBRElement sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RP;
                                sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RP = sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection.ElementAt(0);
                    
                                IDictionary<string, SBRElement> sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap;
                                sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap = sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                    
                                report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPExists = true;
                                report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCount = sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection.Count();
                                report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPOccurrenceIndex = sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RP.OccurrenceIndex + 1;
                                report.MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPLastOccurrenceIndex = sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPCollection.Last().OccurrenceIndex + 1;
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.PersonNameType.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR231 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Usage.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR232 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Currency.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR233 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Title.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR19 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.NameSuffix.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR21 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.FamilyName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR20 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.GivenName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR22 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.OtherGivenName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR23 = currentValue.Value;
                                }
                            }  
                        } 
                        #endregion End of Tuple sbrMailRecipient_PersonNameDetailsCollection
                        if (validationMode) ValidateRPMailRecipient_PersonNameDetailsCollection(errors, sbrMailRecipient_PersonNameDetailsCollection, sbrRPContexts, report);
                
            
                        IEnumerable<SBRElement> sbrMailRecipient_OrganisationNameDetailsCollection;
                        if (sbrMailRecipientTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails", out currentValue))
                            sbrMailRecipient_OrganisationNameDetailsCollection = currentValue.Occurrences;
                        else
                            sbrMailRecipient_OrganisationNameDetailsCollection = null;
                
                        #region sbrMailRecipient_OrganisationNameDetailsCollection
                
                        if (sbrMailRecipient_OrganisationNameDetailsCollection != null)
                        {
                            report.MailRecipient_OrganisationNameDetailsCollectionExists = true;
                            report.MailRecipient_OrganisationNameDetailsCollectionCount = sbrMailRecipient_OrganisationNameDetailsCollection.Count();
                    
                            IEnumerable<SBRElement> sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection;
                            sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection = sbrMailRecipient_OrganisationNameDetailsCollection.Where(
                                ce => ce.ChildElements != null && 
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                            {
                                SBRElement sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RP;
                                sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RP = sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);
                    
                                IDictionary<string, SBRElement> sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                                sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                    
                                report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPExists = true;
                                report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPCount = sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                                report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                                report.MailRecipient_OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;
                        
                                if (sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR234 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR280 = currentValue.Value;
                                }
                        
                                if (sbrMailRecipient_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.PTR24 = currentValue.Value;
                                }
                            }  
                        } 
                        #endregion End of Tuple sbrMailRecipient_OrganisationNameDetailsCollection
                        if (validationMode) ValidateRPMailRecipient_OrganisationNameDetailsCollection(errors, sbrMailRecipient_OrganisationNameDetailsCollection, sbrRPContexts, report);
                
                
                        if (sbrMailRecipientTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                        {
                            if (!currentValue.IsNil) report.PTR25 = currentValue.Value;
                        }
                    } 
                    #endregion End of Tuple sbrMailRecipientCollection
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Elections.InterposedEntityElectionStatus.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR26 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Elections.InterposedEntityElectionRevocation.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR27 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Lodgment.FinalReturn.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR29 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationDetails.MainIncomeActivity.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR30 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationDetails.OrganisationIndustry2006Extended.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR31 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationDetails.ActivityEvent.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR32 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationDetails.TaxConsolidationStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR33 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationDetails.SignificantGlobalEntityStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR355 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}InternationalDealings.CountryByCountryReportingEntityStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR372 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR108 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR277 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}InternationalDealings.TransactionsWithTaxHavenCountries.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR109 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.PersonalServicesIncomeIncluded.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR110 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.PersonalServicesIncome.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR265 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.PersonalServicesIncome.Deduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR266 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR268 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR271 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR276 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR272 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR275 = ToBoolean(currentValues[0].Value);
                    }
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MTR_C_RPCollection;
                        sbrOrganisationNameDetailsCollection_MTR_C_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MTR") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MTR_C_RPCollection != null && sbrOrganisationNameDetailsCollection_MTR_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MTR_C_RP;
                            sbrOrganisationNameDetailsCollection_MTR_C_RP = sbrOrganisationNameDetailsCollection_MTR_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MTR_C_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MTR_C_RPTupleElementMap = sbrOrganisationNameDetailsCollection_MTR_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_MTR_C_RPExists = true;
                            report.OrganisationNameDetailsCollection_MTR_C_RPCount = sbrOrganisationNameDetailsCollection_MTR_C_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_MTR_C_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_MTR_C_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MTR_C_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MTR_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_MTR_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR273 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MTR_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR274 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MTR_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR116 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection2(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_BUS_C_RPCollection;
                        sbrAddressDetailsCollection_BUS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "BUS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_BUS_C_RPCollection != null && sbrAddressDetailsCollection_BUS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_BUS_C_RP;
                            sbrAddressDetailsCollection_BUS_C_RP = sbrAddressDetailsCollection_BUS_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_BUS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_BUS_C_RPTupleElementMap = sbrAddressDetailsCollection_BUS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_BUS_C_RPExists = true;
                            report.AddressDetailsCollection_BUS_C_RPCount = sbrAddressDetailsCollection_BUS_C_RPCollection.Count();
                            report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_BUS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_BUS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_BUS_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR222 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR225 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR228 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR117 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR213 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR220 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR221 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR118 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR120 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR119 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR304 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection2(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Purchases.AndOtherCosts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR122 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.WagesAndSalaries.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR127 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.WagesAndSalariesAction.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR128 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.PaymentToRelatedPartiesGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR129 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.FringeBenefits.EmployeeContribution.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR136 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Elections.TradingStock.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR140 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.AggregatedTurnoverRange.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR411 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.AggregatedTurnover.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR412 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DepreciatingAssets.IntangibleFirstDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR130 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DepreciatingAssets.OtherFirstDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR131 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.PropertyPlantAndEquipment.DepreciatingAssets.FirstDeductedSelfAssessedEffectiveLife.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR283 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.SubsequentYearAcceleratedDepreciationDeductionsForAssets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR401 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.PropertyPlantAndEquipment.DepreciatingAssets.EffectiveLifeRecalculation.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR278 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.DepreciatingAssets.AdjustableValuesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR279 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR282 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR281 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.DepreciatingAssets.IntangibleTerminationValue.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR132 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.DepreciatingAssets.OtherTerminationValue.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR133 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.ProjectPoolAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR134 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR135 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.EmployeeTrainingBonus.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR413 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.EnergyIncentive.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR415 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR248 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Investment.FinancialIncomeOrLossNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR254 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.RentalIncomeOrLossNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR255 = decimal.Parse(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrStatementOfDistributionCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}StatementOfDistribution", out currentValue))
                        sbrStatementOfDistributionCollection = currentValue.Occurrences;
                    else
                        sbrStatementOfDistributionCollection = null;
            
                    #region sbrStatementOfDistributionCollection
            
                    if (sbrStatementOfDistributionCollection != null)
                    {
                        report.StatementOfDistributionCollectionExists = true;
                        report.StatementOfDistributionCollectionCount = sbrStatementOfDistributionCollection.Count();
                        if (sbrStatementOfDistributionCollection != null && sbrStatementOfDistributionCollection.Count() > 0)
                        {
                            report.StatementOfDistributionCollection = new List<PTR2024.StatementOfDistribution>();
                            for (int tupleIndex = 0; tupleIndex < sbrStatementOfDistributionCollection.Count(); tupleIndex++)
                            {
                                SBRElement sbrStatementOfDistribution;
                                sbrStatementOfDistribution = sbrStatementOfDistributionCollection.ElementAt(tupleIndex);
                
                                IDictionary<string, SBRElement> sbrStatementOfDistributionTupleElementMap;
                                sbrStatementOfDistributionTupleElementMap = sbrStatementOfDistribution.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                                PTR2024.StatementOfDistribution statementOfDistribution = new PTR2024.StatementOfDistribution();
                                report.StatementOfDistributionCollection.Add(statementOfDistribution);
                                statementOfDistribution.OccurrenceIndex = tupleIndex + 1;
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonDemographicDetails.Birth.Date", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR284 = DateTime.Parse(currentValue.Value);
                                }
                    
                                IEnumerable<SBRElement> sbrStatementOfDistribution_PersonNameDetailsCollection;
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails", out currentValue))
                                    sbrStatementOfDistribution_PersonNameDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrStatementOfDistribution_PersonNameDetailsCollection = null;
                        
                                #region sbrStatementOfDistribution_PersonNameDetailsCollection
                        
                                if (sbrStatementOfDistribution_PersonNameDetailsCollection != null)
                                {
                                    statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollectionExists = true;
                                    statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollectionCount = sbrStatementOfDistribution_PersonNameDetailsCollection.Count();
                            
                                    IEnumerable<SBRElement> sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection;
                                    sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection = sbrStatementOfDistribution_PersonNameDetailsCollection.Where(
                                        ce => ce.ChildElements != null && 
                                        ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL") &&
                                        ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "Contact" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection != null && sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RP;
                                        sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RP = sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection.ElementAt(0);
                            
                                        IDictionary<string, SBRElement> sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap;
                                        sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap = sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                        statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPExists = true;
                                        statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCount = sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection.Count();
                                        statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPOccurrenceIndex = sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RP.OccurrenceIndex + 1;
                                        statementOfDistribution.StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPLastOccurrenceIndex = sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPCollection.Last().OccurrenceIndex + 1;
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.PersonNameType.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR235 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Usage.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR310 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Currency.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR267 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Title.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR311 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.NameSuffix.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR312 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.FamilyName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR146 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.GivenName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR147 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.OtherGivenName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR148 = currentValue.Value;
                                        }
                                    }  
                                } 
                                #endregion End of Tuple sbrStatementOfDistribution_PersonNameDetailsCollection
                                if (validationMode) ValidateRPStatementOfDistribution_PersonNameDetailsCollection(errors, sbrStatementOfDistribution_PersonNameDetailsCollection, sbrRPContexts, report);
                        
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.PartnerWithNoControlAndShareDisposalInPartnershipIncome.Indicator", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR246 = ToBoolean(currentValue.Value);
                                }
                    
                                IEnumerable<SBRElement> sbrStatementOfDistribution_AddressDetailsCollection;
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails", out currentValue))
                                    sbrStatementOfDistribution_AddressDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrStatementOfDistribution_AddressDetailsCollection = null;
                        
                                #region sbrStatementOfDistribution_AddressDetailsCollection
                        
                                if (sbrStatementOfDistribution_AddressDetailsCollection != null)
                                {
                                    statementOfDistribution.StatementOfDistribution_AddressDetailsCollectionExists = true;
                                    statementOfDistribution.StatementOfDistribution_AddressDetailsCollectionCount = sbrStatementOfDistribution_AddressDetailsCollection.Count();
                            
                                    IEnumerable<SBRElement> sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection;
                                    sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection = sbrStatementOfDistribution_AddressDetailsCollection.Where(
                                        ce => ce.ChildElements != null && 
                                        ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "RES") &&
                                        ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection != null && sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RP;
                                        sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RP = sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection.ElementAt(0);
                            
                                        IDictionary<string, SBRElement> sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap;
                                        sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap = sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPExists = true;
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPCount = sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection.Count();
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPOccurrenceIndex = sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RP.OccurrenceIndex + 1;
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_RES_C_RPLastOccurrenceIndex = sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPCollection.Last().OccurrenceIndex + 1;
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR329 = ToBoolean(currentValue.Value);
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR330 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR331 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR332 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR333 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR339 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR340 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR334 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR335 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR336 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_RES_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR337 = currentValue.Value;
                                        }
                                    }  
                                } 
                                #endregion End of Tuple sbrStatementOfDistribution_AddressDetailsCollection
                                if (validationMode) ValidateRPStatementOfDistribution_AddressDetailsCollection(errors, sbrStatementOfDistribution_AddressDetailsCollection, sbrRPContexts, report);
                        
                    
                                IEnumerable<SBRElement> sbrStatementOfDistribution_OrganisationNameDetailsCollection;
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails", out currentValue))
                                    sbrStatementOfDistribution_OrganisationNameDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrStatementOfDistribution_OrganisationNameDetailsCollection = null;
                        
                                #region sbrStatementOfDistribution_OrganisationNameDetailsCollection
                        
                                if (sbrStatementOfDistribution_OrganisationNameDetailsCollection != null)
                                {
                                    statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollectionExists = true;
                                    statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollectionCount = sbrStatementOfDistribution_OrganisationNameDetailsCollection.Count();
                            
                                    IEnumerable<SBRElement> sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection;
                                    sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection = sbrStatementOfDistribution_OrganisationNameDetailsCollection.Where(
                                        ce => ce.ChildElements != null && 
                                        ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                                        ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RP;
                                        sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RP = sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);
                            
                                        IDictionary<string, SBRElement> sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                                        sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                        statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPExists = true;
                                        statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCount = sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                                        statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                                        statementOfDistribution.StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;
                                
                                        if (sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR236 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR269 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR149 = currentValue.Value;
                                        }
                                    }  
                                } 
                                #endregion End of Tuple sbrStatementOfDistribution_OrganisationNameDetailsCollection
                                if (validationMode) ValidateRPStatementOfDistribution_OrganisationNameDetailsCollection(errors, sbrStatementOfDistribution_OrganisationNameDetailsCollection, sbrRPContexts, report);
                        
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR285 = currentValue.Value;
                                }
                        
                                #region sbrStatementOfDistribution_AddressDetailsCollection
                        
                                if (sbrStatementOfDistribution_AddressDetailsCollection != null)
                                {
                            
                                    IEnumerable<SBRElement> sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection;
                                    sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection = sbrStatementOfDistribution_AddressDetailsCollection.Where(
                                        ce => ce.ChildElements != null && 
                                        ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "BUS") &&
                                        ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection != null && sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RP;
                                        sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RP = sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection.ElementAt(0);
                            
                                        IDictionary<string, SBRElement> sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap;
                                        sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap = sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                            
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPExists = true;
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPCount = sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection.Count();
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPOccurrenceIndex = sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RP.OccurrenceIndex + 1;
                                        statementOfDistribution.StatementOfDistribution_AddressDetailsCollection_BUS_C_RPLastOccurrenceIndex = sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPCollection.Last().OccurrenceIndex + 1;
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR240 = ToBoolean(currentValue.Value);
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR237 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR270 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR151 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR152 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR238 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR239 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR153 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR155 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR154 = currentValue.Value;
                                        }
                                
                                        if (sbrStatementOfDistribution_AddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) statementOfDistribution.PTR308 = currentValue.Value;
                                        }
                                    }  
                                } 
                                #endregion End of Tuple sbrStatementOfDistribution_AddressDetailsCollection
                                if (validationMode) ValidateRPStatementOfDistribution_AddressDetailsCollection1(errors, sbrStatementOfDistribution_AddressDetailsCollection, sbrRPContexts, report);
                        
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Identifiers.TaxFileNumber.Identifier", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR157 = currentValue.Value;
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.BeneficiaryShare.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}PrimaryProduction") || p.Domain.EndsWith(":PrimaryProduction"))>-1)
                                                {
                                                    statementOfDistribution.PTR158 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}PrimaryProduction") || p.Domain.EndsWith(":PrimaryProduction"))>-1)
                                                {
                                                    statementOfDistribution.PTR368 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}PrimaryProduction") || p.Domain.EndsWith(":PrimaryProduction"))>-1)
                                                {
                                                    statementOfDistribution.PTR369 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.BeneficiaryShare.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}NonPrimaryProduction") || p.Domain.EndsWith(":NonPrimaryProduction"))>-1)
                                                {
                                                    statementOfDistribution.PTR159 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}NonPrimaryProduction") || p.Domain.EndsWith(":NonPrimaryProduction"))>-1)
                                                {
                                                    statementOfDistribution.PTR370 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil) 
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {                    
                                                if(sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p=>p.Domain.EndsWith("}NonPrimaryProduction") || p.Domain.EndsWith(":NonPrimaryProduction"))>-1)
                                                {
                                                    statementOfDistribution.PTR371 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR160 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.DistributionFranked.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR324 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR161 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR162 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR321 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR351 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.SmallBusinessEntityNet.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR353 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR164 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR249 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Investment.FinancialIncomeOrLossNet.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR256 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.RentalIncomeOrLossNet.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR257 = decimal.Parse(currentValue.Value);
                                }
                        
                                if (sbrStatementOfDistributionTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.ExplorationCreditsDistributed.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) statementOfDistribution.PTR342 = decimal.Parse(currentValue.Value);
                                }
                    } 
                    #endregion End of Tuple sbrStatementOfDistributionCollection
                        } 
                    } 
        
                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;
            
                    #region sbrDeclarationCollection
            
                    if (sbrDeclarationCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPCollection;
                        sbrDeclarationCollection_TrueAndCorrect_RPCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_RPCollection != null && sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_RP;
                            sbrDeclarationCollection_TrueAndCorrect_RP = sbrDeclarationCollection_TrueAndCorrect_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.DeclarationCollection_TrueAndCorrect_RPExists = true;
                            report.DeclarationCollection_TrueAndCorrect_RPCount = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RP.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_RPLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR291 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR290 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR293 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR289 = DateTime.Parse(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.PTR292 = currentValue.Value;
                            }
                
                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;
                    
                            #region sbrDeclaration_PersonUnstructuredNameCollection
                    
                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                        
                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null && 
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrRPContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.ElementAt(0);
                        
                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                        
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Last().OccurrenceIndex + 1;
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.PTR260 = currentValue.Value;
                                    }
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.PTR264 = currentValue.Value;
                                    }
                                }  
                            } 
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateRPDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrRPContexts, report);
                    
                        }  
                    } 
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateRPDeclarationCollection(errors, sbrDeclarationCollection, sbrRPContexts, report);
            
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Report.CompletionHours.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR169 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR315 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPPrimContext
            IEnumerable<SBRContext> sbrRPPrimContexts = null;
            SBRContext sbrRPPrimContext = null;
    
            sbrRPPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}PrimaryProduction")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPrimContexts != null && sbrRPPrimContexts.Count() > 0)
            {
                sbrRPPrimContext = sbrRPPrimContexts.First();
        
                report.RPPrimId = sbrRPPrimContext.Id;
                report.RPPrimIdentifier = sbrRPPrimContext.EntityIdentifier;
                report.RPPrimIdentifierScheme = sbrRPPrimContext.EntityScheme;
        
                report.RPPrimCount = sbrRPPrimContexts.Count();
                report.RPPrimExists = true;
                report.RPPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPrimContext) + 1;
                report.RPPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPrimContexts.Last()) + 1;
        

                if (report.RPPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPPrimIdentifierTFN = report.RPPrimIdentifier;
    
                report.RPPrimStartDate = ((SBRDurationPeriod)sbrRPPrimContext.Period).DurationStartAsDateObject;
                report.RPPrimEndDate = ((SBRDurationPeriod)sbrRPPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPPrimContexts.Count(); i++)
                {
                    sbrRPPrimContext = sbrRPPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR360 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR361 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPPrimContext
    
            #region sbrRPNonPrimContext
            IEnumerable<SBRContext> sbrRPNonPrimContexts = null;
            SBRContext sbrRPNonPrimContext = null;
    
            sbrRPNonPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonPrimaryProduction")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonPrimContexts != null && sbrRPNonPrimContexts.Count() > 0)
            {
                sbrRPNonPrimContext = sbrRPNonPrimContexts.First();
        
                report.RPNonPrimId = sbrRPNonPrimContext.Id;
                report.RPNonPrimIdentifier = sbrRPNonPrimContext.EntityIdentifier;
                report.RPNonPrimIdentifierScheme = sbrRPNonPrimContext.EntityScheme;
        
                report.RPNonPrimCount = sbrRPNonPrimContexts.Count();
                report.RPNonPrimExists = true;
                report.RPNonPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonPrimContext) + 1;
                report.RPNonPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonPrimContexts.Last()) + 1;
        

                if (report.RPNonPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonPrimIdentifierTFN = report.RPNonPrimIdentifier;
    
                report.RPNonPrimStartDate = ((SBRDurationPeriod)sbrRPNonPrimContext.Period).DurationStartAsDateObject;
                report.RPNonPrimEndDate = ((SBRDurationPeriod)sbrRPNonPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPNonPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonPrimContexts.Count(); i++)
                {
                    sbrRPNonPrimContext = sbrRPNonPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR364 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR365 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPNonPrimContext
    
            #region sbrRPJAUSContext
            IEnumerable<SBRContext> sbrRPJAUSContexts = null;
            SBRContext sbrRPJAUSContext = null;
    
            sbrRPJAUSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSContexts != null && sbrRPJAUSContexts.Count() > 0)
            {
                sbrRPJAUSContext = sbrRPJAUSContexts.First();
        
                report.RPJAUSId = sbrRPJAUSContext.Id;
                report.RPJAUSIdentifier = sbrRPJAUSContext.EntityIdentifier;
                report.RPJAUSIdentifierScheme = sbrRPJAUSContext.EntityScheme;
        
                report.RPJAUSCount = sbrRPJAUSContexts.Count();
                report.RPJAUSExists = true;
                report.RPJAUSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSContext) + 1;
                report.RPJAUSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSContexts.Last()) + 1;
        

                if (report.RPJAUSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSIdentifierTFN = report.RPJAUSIdentifier;
    
                report.RPJAUSStartDate = ((SBRDurationPeriod)sbrRPJAUSContext.Period).DurationStartAsDateObject;
                report.RPJAUSEndDate = ((SBRDurationPeriod)sbrRPJAUSContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJAUSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSContexts.Count(); i++)
                {
                    sbrRPJAUSContext = sbrRPJAUSContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.ForeignResidentWithholding.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR45 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR46 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}SuperannuationContribution.EmployerContributions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR47 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.CostOfSales.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR343 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.BadDebts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR344 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.LeaseTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR50 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.Rent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR345 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR346 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Royalties.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR347 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DepreciationAndAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR54 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.MotorVehicle.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR348 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.MotorVehicleTypeOrClaimMethod.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR56 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Operating.RepairsAndMaintenance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR349 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DeductibleOtherTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR58 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ReconciliationAdjustmentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR59 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.ReconciliationAdjustmentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR328 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.SmallBusinessEntityNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR352 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR63 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR350 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR73 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR74 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR75 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectly.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR320 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR76 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Operating.RentalIncomeGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR77 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR78 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.CapitalWorksDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR79 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR80 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ForestryManagedInvestmentScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR81 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Interest.Gross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR82 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR83 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.DividendsUnfranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR84 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.DividendsFranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR85 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.FrankingCredits.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR86 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.DividendsTFNAmountsWithheld.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR87 = decimal.Parse(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrOtherAustralianIncomeCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OtherAustralianIncome", out currentValue))
                        sbrOtherAustralianIncomeCollection = currentValue.Occurrences;
                    else
                        sbrOtherAustralianIncomeCollection = null;
            
                    #region sbrOtherAustralianIncomeCollection
            
                    if (sbrOtherAustralianIncomeCollection != null)
                    {
                        report.OtherAustralianIncomeCollectionExists = true;
                        report.OtherAustralianIncomeCollectionCount = sbrOtherAustralianIncomeCollection.Count();
                        if (sbrOtherAustralianIncomeCollection != null && sbrOtherAustralianIncomeCollection.Count() > 0)
                        {
                            report.OtherAustralianIncomeCollection = new List<PTR2024.OtherAustralianIncome>();
                            for (int tupleIndex = 0; tupleIndex < sbrOtherAustralianIncomeCollection.Count(); tupleIndex++)
                            {
                                SBRElement sbrOtherAustralianIncome;
                                sbrOtherAustralianIncome = sbrOtherAustralianIncomeCollection.ElementAt(tupleIndex);
                
                                IDictionary<string, SBRElement> sbrOtherAustralianIncomeTupleElementMap;
                                sbrOtherAustralianIncomeTupleElementMap = sbrOtherAustralianIncome.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                                PTR2024.OtherAustralianIncome otherAustralianIncome = new PTR2024.OtherAustralianIncome();
                                report.OtherAustralianIncomeCollection.Add(otherAustralianIncome);
                                otherAustralianIncome.OccurrenceIndex = tupleIndex + 1;
                        
                                if (sbrOtherAustralianIncomeTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Other.Description", out currentValue))
                                {
                                    if (!currentValue.IsNil) otherAustralianIncome.PTR88 = currentValue.Value;
                                }
                        
                                if (sbrOtherAustralianIncomeTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Other.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) otherAustralianIncome.PTR89 = decimal.Parse(currentValue.Value);
                                }
                    } 
                    #endregion End of Tuple sbrOtherAustralianIncomeCollection
                        } 
                    } 
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.AustralianInvestmentAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR90 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.ForestryManagedInvestmentSchemeDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR91 = decimal.Parse(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrOtherDeductionsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OtherDeductions", out currentValue))
                        sbrOtherDeductionsCollection = currentValue.Occurrences;
                    else
                        sbrOtherDeductionsCollection = null;
            
                    #region sbrOtherDeductionsCollection
            
                    if (sbrOtherDeductionsCollection != null)
                    {
                        report.OtherDeductionsCollectionExists = true;
                        report.OtherDeductionsCollectionCount = sbrOtherDeductionsCollection.Count();
                        if (sbrOtherDeductionsCollection != null && sbrOtherDeductionsCollection.Count() > 0)
                        {
                            report.OtherDeductionsCollection = new List<PTR2024.OtherDeductions>();
                            for (int tupleIndex = 0; tupleIndex < sbrOtherDeductionsCollection.Count(); tupleIndex++)
                            {
                                SBRElement sbrOtherDeductions;
                                sbrOtherDeductions = sbrOtherDeductionsCollection.ElementAt(tupleIndex);
                
                                IDictionary<string, SBRElement> sbrOtherDeductionsTupleElementMap;
                                sbrOtherDeductionsTupleElementMap = sbrOtherDeductions.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                                PTR2024.OtherDeductions otherDeductions = new PTR2024.OtherDeductions();
                                report.OtherDeductionsCollection.Add(otherDeductions);
                                otherDeductions.OccurrenceIndex = tupleIndex + 1;
                        
                                if (sbrOtherDeductionsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DeductibleOther.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) otherDeductions.PTR95 = currentValue.Value;
                                }
                        
                                if (sbrOtherDeductionsTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DeductibleOther.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) otherDeductions.PTR96 = decimal.Parse(currentValue.Value);
                                }
                    } 
                    #endregion End of Tuple sbrOtherDeductionsCollection
                        } 
                    } 
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR97 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJAUSContext
    
            #region sbrRPJFOREIGNContext
            IEnumerable<SBRContext> sbrRPJFOREIGNContexts = null;
            SBRContext sbrRPJFOREIGNContext = null;
    
            sbrRPJFOREIGNContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJFOREIGNElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJFOREIGNContexts != null && sbrRPJFOREIGNContexts.Count() > 0)
            {
                sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.First();
        
                report.RPJFOREIGNId = sbrRPJFOREIGNContext.Id;
                report.RPJFOREIGNIdentifier = sbrRPJFOREIGNContext.EntityIdentifier;
                report.RPJFOREIGNIdentifierScheme = sbrRPJFOREIGNContext.EntityScheme;
        
                report.RPJFOREIGNCount = sbrRPJFOREIGNContexts.Count();
                report.RPJFOREIGNExists = true;
                report.RPJFOREIGNOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContext) + 1;
                report.RPJFOREIGNLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContexts.Last()) + 1;
        

                if (report.RPJFOREIGNIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJFOREIGNIdentifierTFN = report.RPJFOREIGNIdentifier;
    
                report.RPJFOREIGNStartDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationStartAsDateObject;
                report.RPJFOREIGNEndDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJFOREIGNElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJFOREIGNContexts.Count(); i++)
                {
                    sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJFOREIGNContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJFOREIGNElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJFOREIGNElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJFOREIGNElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR102 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR104 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR105 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.InternationalDealings.TaxOffset.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR106 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR107 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR137 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.Royalties.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR138 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJFOREIGNContext
    
            #region sbrRPTOFAContext
            IEnumerable<SBRContext> sbrRPTOFAContexts = null;
            SBRContext sbrRPTOFAContext = null;
    
            sbrRPTOFAContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}FinancialArrangementTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}TOFA")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPTOFAElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPTOFAContexts != null && sbrRPTOFAContexts.Count() > 0)
            {
                sbrRPTOFAContext = sbrRPTOFAContexts.First();
        
                report.RPTOFAId = sbrRPTOFAContext.Id;
                report.RPTOFAIdentifier = sbrRPTOFAContext.EntityIdentifier;
                report.RPTOFAIdentifierScheme = sbrRPTOFAContext.EntityScheme;
        
                report.RPTOFACount = sbrRPTOFAContexts.Count();
                report.RPTOFAExists = true;
                report.RPTOFAOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTOFAContext) + 1;
                report.RPTOFALastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTOFAContexts.Last()) + 1;
        

                if (report.RPTOFAIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPTOFAIdentifierTFN = report.RPTOFAIdentifier;
    
                report.RPTOFAStartDate = ((SBRDurationPeriod)sbrRPTOFAContext.Period).DurationStartAsDateObject;
                report.RPTOFAEndDate = ((SBRDurationPeriod)sbrRPTOFAContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPTOFAElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPTOFAContexts.Count(); i++)
                {
                    sbrRPTOFAContext = sbrRPTOFAContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPTOFAContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPTOFAElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPTOFAElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPTOFAElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.GainsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR251 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.LossesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR252 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPTOFAContext
    
            #region sbrRPJFOREIGNCFCLCContext
            IEnumerable<SBRContext> sbrRPJFOREIGNCFCLCContexts = null;
            SBRContext sbrRPJFOREIGNCFCLCContext = null;
    
            sbrRPJFOREIGNCFCLCContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}CountryOfResidencyCFCClassificationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ListedCountry")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJFOREIGNCFCLCElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJFOREIGNCFCLCContexts != null && sbrRPJFOREIGNCFCLCContexts.Count() > 0)
            {
                sbrRPJFOREIGNCFCLCContext = sbrRPJFOREIGNCFCLCContexts.First();
        
                report.RPJFOREIGNCFCLCId = sbrRPJFOREIGNCFCLCContext.Id;
                report.RPJFOREIGNCFCLCIdentifier = sbrRPJFOREIGNCFCLCContext.EntityIdentifier;
                report.RPJFOREIGNCFCLCIdentifierScheme = sbrRPJFOREIGNCFCLCContext.EntityScheme;
        
                report.RPJFOREIGNCFCLCCount = sbrRPJFOREIGNCFCLCContexts.Count();
                report.RPJFOREIGNCFCLCExists = true;
                report.RPJFOREIGNCFCLCOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNCFCLCContext) + 1;
                report.RPJFOREIGNCFCLCLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNCFCLCContexts.Last()) + 1;
        

                if (report.RPJFOREIGNCFCLCIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJFOREIGNCFCLCIdentifierTFN = report.RPJFOREIGNCFCLCIdentifier;
    
                report.RPJFOREIGNCFCLCStartDate = ((SBRDurationPeriod)sbrRPJFOREIGNCFCLCContext.Period).DurationStartAsDateObject;
                report.RPJFOREIGNCFCLCEndDate = ((SBRDurationPeriod)sbrRPJFOREIGNCFCLCContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJFOREIGNCFCLCElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJFOREIGNCFCLCContexts.Count(); i++)
                {
                    sbrRPJFOREIGNCFCLCContext = sbrRPJFOREIGNCFCLCContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJFOREIGNCFCLCContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJFOREIGNCFCLCElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJFOREIGNCFCLCElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJFOREIGNCFCLCElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJFOREIGNCFCLCElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.InternationalDealings.Attributed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR98 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJFOREIGNCFCLCContext
    
            #region sbrRPJFOREIGNCFCUCContext
            IEnumerable<SBRContext> sbrRPJFOREIGNCFCUCContexts = null;
            SBRContext sbrRPJFOREIGNCFCUCContext = null;
    
            sbrRPJFOREIGNCFCUCContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}CountryOfResidencyCFCClassificationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}UnlistedCountry")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJFOREIGNCFCUCElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJFOREIGNCFCUCContexts != null && sbrRPJFOREIGNCFCUCContexts.Count() > 0)
            {
                sbrRPJFOREIGNCFCUCContext = sbrRPJFOREIGNCFCUCContexts.First();
        
                report.RPJFOREIGNCFCUCId = sbrRPJFOREIGNCFCUCContext.Id;
                report.RPJFOREIGNCFCUCIdentifier = sbrRPJFOREIGNCFCUCContext.EntityIdentifier;
                report.RPJFOREIGNCFCUCIdentifierScheme = sbrRPJFOREIGNCFCUCContext.EntityScheme;
        
                report.RPJFOREIGNCFCUCCount = sbrRPJFOREIGNCFCUCContexts.Count();
                report.RPJFOREIGNCFCUCExists = true;
                report.RPJFOREIGNCFCUCOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNCFCUCContext) + 1;
                report.RPJFOREIGNCFCUCLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNCFCUCContexts.Last()) + 1;
        

                if (report.RPJFOREIGNCFCUCIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJFOREIGNCFCUCIdentifierTFN = report.RPJFOREIGNCFCUCIdentifier;
    
                report.RPJFOREIGNCFCUCStartDate = ((SBRDurationPeriod)sbrRPJFOREIGNCFCUCContext.Period).DurationStartAsDateObject;
                report.RPJFOREIGNCFCUCEndDate = ((SBRDurationPeriod)sbrRPJFOREIGNCFCUCContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJFOREIGNCFCUCElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJFOREIGNCFCUCContexts.Count(); i++)
                {
                    sbrRPJFOREIGNCFCUCContext = sbrRPJFOREIGNCFCUCContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJFOREIGNCFCUCContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJFOREIGNCFCUCElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJFOREIGNCFCUCElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJFOREIGNCFCUCElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJFOREIGNCFCUCElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.InternationalDealings.Attributed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR100 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJFOREIGNCFCUCContext
    
            #region sbrRPClosingInstantContext
            IEnumerable<SBRContext> sbrRPClosingInstantContexts = null;
            SBRContext sbrRPClosingInstantContext = null;
    
            sbrRPClosingInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingInstantContexts != null && sbrRPClosingInstantContexts.Count() > 0)
            {
                sbrRPClosingInstantContext = sbrRPClosingInstantContexts.First();
        
                report.RPClosingInstantId = sbrRPClosingInstantContext.Id;
                report.RPClosingInstantIdentifier = sbrRPClosingInstantContext.EntityIdentifier;
                report.RPClosingInstantIdentifierScheme = sbrRPClosingInstantContext.EntityScheme;
        
                report.RPClosingInstantCount = sbrRPClosingInstantContexts.Count();
                report.RPClosingInstantExists = true;
                report.RPClosingInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingInstantContext) + 1;
                report.RPClosingInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingInstantContexts.Last()) + 1;
        

                if (report.RPClosingInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingInstantIdentifierTFN = report.RPClosingInstantIdentifier;
    
                report.RPClosingInstantInstantDate = ((SBRInstantPeriod)sbrRPClosingInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPClosingInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingInstantContexts.Count(); i++)
                {
                    sbrRPClosingInstantContext = sbrRPClosingInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.CurrentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR111 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR112 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Liabilities.CurrentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR113 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Liabilities.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR114 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.Inventories.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR123 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.Inventories.ValuationMethod.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR124 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.TradeandReceivablesOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR125 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Liabilities.TradeAndOtherPayablesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR126 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPClosingInstantContext
    
            #region sbrRPOpeningInstantContext
            IEnumerable<SBRContext> sbrRPOpeningInstantContexts = null;
            SBRContext sbrRPOpeningInstantContext = null;
    
            sbrRPOpeningInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Opening")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOpeningInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOpeningInstantContexts != null && sbrRPOpeningInstantContexts.Count() > 0)
            {
                sbrRPOpeningInstantContext = sbrRPOpeningInstantContexts.First();
        
                report.RPOpeningInstantId = sbrRPOpeningInstantContext.Id;
                report.RPOpeningInstantIdentifier = sbrRPOpeningInstantContext.EntityIdentifier;
                report.RPOpeningInstantIdentifierScheme = sbrRPOpeningInstantContext.EntityScheme;
        
                report.RPOpeningInstantCount = sbrRPOpeningInstantContexts.Count();
                report.RPOpeningInstantExists = true;
                report.RPOpeningInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOpeningInstantContext) + 1;
                report.RPOpeningInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOpeningInstantContexts.Last()) + 1;
        

                if (report.RPOpeningInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOpeningInstantIdentifierTFN = report.RPOpeningInstantIdentifier;
    
                report.RPOpeningInstantInstantDate = ((SBRInstantPeriod)sbrRPOpeningInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPOpeningInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOpeningInstantContexts.Count(); i++)
                {
                    sbrRPOpeningInstantContext = sbrRPOpeningInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOpeningInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOpeningInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOpeningInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOpeningInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOpeningInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Assets.Inventories.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR121 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOpeningInstantContext
    
            #region sbrRPLowCostContext
            IEnumerable<SBRContext> sbrRPLowCostContexts = null;
            SBRContext sbrRPLowCostContext = null;
    
            sbrRPLowCostContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SmallBusinessDepreciatingAssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LowCost")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLowCostElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLowCostContexts != null && sbrRPLowCostContexts.Count() > 0)
            {
                sbrRPLowCostContext = sbrRPLowCostContexts.First();
        
                report.RPLowCostId = sbrRPLowCostContext.Id;
                report.RPLowCostIdentifier = sbrRPLowCostContext.EntityIdentifier;
                report.RPLowCostIdentifierScheme = sbrRPLowCostContext.EntityScheme;
        
                report.RPLowCostCount = sbrRPLowCostContexts.Count();
                report.RPLowCostExists = true;
                report.RPLowCostOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLowCostContext) + 1;
                report.RPLowCostLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLowCostContexts.Last()) + 1;
        

                if (report.RPLowCostIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLowCostIdentifierTFN = report.RPLowCostIdentifier;
    
                report.RPLowCostStartDate = ((SBRDurationPeriod)sbrRPLowCostContext.Period).DurationStartAsDateObject;
                report.RPLowCostEndDate = ((SBRDurationPeriod)sbrRPLowCostContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPLowCostElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLowCostContexts.Count(); i++)
                {
                    sbrRPLowCostContext = sbrRPLowCostContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLowCostContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLowCostElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLowCostElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLowCostElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPLowCostElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DepreciatingAssets.SmallBusinessEntity.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR141 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPLowCostContext
    
            #region sbrRPGeneralPoolContext
            IEnumerable<SBRContext> sbrRPGeneralPoolContexts = null;
            SBRContext sbrRPGeneralPoolContext = null;
    
            sbrRPGeneralPoolContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SmallBusinessDepreciatingAssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}GeneralPool")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPGeneralPoolElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPGeneralPoolContexts != null && sbrRPGeneralPoolContexts.Count() > 0)
            {
                sbrRPGeneralPoolContext = sbrRPGeneralPoolContexts.First();
        
                report.RPGeneralPoolId = sbrRPGeneralPoolContext.Id;
                report.RPGeneralPoolIdentifier = sbrRPGeneralPoolContext.EntityIdentifier;
                report.RPGeneralPoolIdentifierScheme = sbrRPGeneralPoolContext.EntityScheme;
        
                report.RPGeneralPoolCount = sbrRPGeneralPoolContexts.Count();
                report.RPGeneralPoolExists = true;
                report.RPGeneralPoolOccurrenceIndex = Array.IndexOf(contextArray, sbrRPGeneralPoolContext) + 1;
                report.RPGeneralPoolLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPGeneralPoolContexts.Last()) + 1;
        

                if (report.RPGeneralPoolIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPGeneralPoolIdentifierTFN = report.RPGeneralPoolIdentifier;
    
                report.RPGeneralPoolStartDate = ((SBRDurationPeriod)sbrRPGeneralPoolContext.Period).DurationStartAsDateObject;
                report.RPGeneralPoolEndDate = ((SBRDurationPeriod)sbrRPGeneralPoolContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPGeneralPoolElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPGeneralPoolContexts.Count(); i++)
                {
                    sbrRPGeneralPoolContext = sbrRPGeneralPoolContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPGeneralPoolContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPGeneralPoolElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPGeneralPoolElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPGeneralPoolElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPGeneralPoolElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DepreciatingAssets.SmallBusinessEntity.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR142 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPGeneralPoolContext
    
            #region sbrRPJAUSPrimContext
            IEnumerable<SBRContext> sbrRPJAUSPrimContexts = null;
            SBRContext sbrRPJAUSPrimContext = null;
    
            sbrRPJAUSPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}PrimaryProduction")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSPrimContexts != null && sbrRPJAUSPrimContexts.Count() > 0)
            {
                sbrRPJAUSPrimContext = sbrRPJAUSPrimContexts.First();
        
                report.RPJAUSPrimId = sbrRPJAUSPrimContext.Id;
                report.RPJAUSPrimIdentifier = sbrRPJAUSPrimContext.EntityIdentifier;
                report.RPJAUSPrimIdentifierScheme = sbrRPJAUSPrimContext.EntityScheme;
        
                report.RPJAUSPrimCount = sbrRPJAUSPrimContexts.Count();
                report.RPJAUSPrimExists = true;
                report.RPJAUSPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSPrimContext) + 1;
                report.RPJAUSPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSPrimContexts.Last()) + 1;
        

                if (report.RPJAUSPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSPrimIdentifierTFN = report.RPJAUSPrimIdentifier;
    
                report.RPJAUSPrimStartDate = ((SBRDurationPeriod)sbrRPJAUSPrimContext.Period).DurationStartAsDateObject;
                report.RPJAUSPrimEndDate = ((SBRDurationPeriod)sbrRPJAUSPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJAUSPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSPrimContexts.Count(); i++)
                {
                    sbrRPJAUSPrimContext = sbrRPJAUSPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.ABNNotQuotedPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR35 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR39 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR258 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR43 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR61 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.PartnershipDistributionGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR65 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.TrustShareNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR66 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Party.TrustType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR299 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DistributionDeductionsOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR68 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR356 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR357 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJAUSPrimContext
    
            #region sbrRPJAUSNonPrimContext
            IEnumerable<SBRContext> sbrRPJAUSNonPrimContexts = null;
            SBRContext sbrRPJAUSNonPrimContext = null;
    
            sbrRPJAUSNonPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonPrimaryProduction")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSNonPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSNonPrimContexts != null && sbrRPJAUSNonPrimContexts.Count() > 0)
            {
                sbrRPJAUSNonPrimContext = sbrRPJAUSNonPrimContexts.First();
        
                report.RPJAUSNonPrimId = sbrRPJAUSNonPrimContext.Id;
                report.RPJAUSNonPrimIdentifier = sbrRPJAUSNonPrimContext.EntityIdentifier;
                report.RPJAUSNonPrimIdentifierScheme = sbrRPJAUSNonPrimContext.EntityScheme;
        
                report.RPJAUSNonPrimCount = sbrRPJAUSNonPrimContexts.Count();
                report.RPJAUSNonPrimExists = true;
                report.RPJAUSNonPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSNonPrimContext) + 1;
                report.RPJAUSNonPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSNonPrimContexts.Last()) + 1;
        

                if (report.RPJAUSNonPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSNonPrimIdentifierTFN = report.RPJAUSNonPrimIdentifier;
    
                report.RPJAUSNonPrimStartDate = ((SBRDurationPeriod)sbrRPJAUSNonPrimContext.Period).DurationStartAsDateObject;
                report.RPJAUSNonPrimEndDate = ((SBRDurationPeriod)sbrRPJAUSNonPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJAUSNonPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSNonPrimContexts.Count(); i++)
                {
                    sbrRPJAUSNonPrimContext = sbrRPJAUSNonPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSNonPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSNonPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSNonPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSNonPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.ABNNotQuotedPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR36 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Remuneration.PaymentToForeignResidentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR38 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR41 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR259 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR44 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR62 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.PartnershipDistributionLessForeignIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR69 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR70 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Party.TrustType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR327 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Expense.DistributionDeductionsOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR72 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.TrustDistributionFranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR322 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}IncomeTax.Deduction.TrustDistributionFranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR323 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR358 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR359 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJAUSNonPrimContext
    
            #region sbrRPSharePrimContext
            IEnumerable<SBRContext> sbrRPSharePrimContexts = null;
            SBRContext sbrRPSharePrimContext = null;
    
            sbrRPSharePrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ShareOfAmountTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Share"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}PrimaryProduction")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPSharePrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPSharePrimContexts != null && sbrRPSharePrimContexts.Count() > 0)
            {
                sbrRPSharePrimContext = sbrRPSharePrimContexts.First();
        
                report.RPSharePrimId = sbrRPSharePrimContext.Id;
                report.RPSharePrimIdentifier = sbrRPSharePrimContext.EntityIdentifier;
                report.RPSharePrimIdentifierScheme = sbrRPSharePrimContext.EntityScheme;
        
                report.RPSharePrimCount = sbrRPSharePrimContexts.Count();
                report.RPSharePrimExists = true;
                report.RPSharePrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSharePrimContext) + 1;
                report.RPSharePrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSharePrimContexts.Last()) + 1;
        

                if (report.RPSharePrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPSharePrimIdentifierTFN = report.RPSharePrimIdentifier;
    
                report.RPSharePrimStartDate = ((SBRDurationPeriod)sbrRPSharePrimContext.Period).DurationStartAsDateObject;
                report.RPSharePrimEndDate = ((SBRDurationPeriod)sbrRPSharePrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPSharePrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPSharePrimContexts.Count(); i++)
                {
                    sbrRPSharePrimContext = sbrRPSharePrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPSharePrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPSharePrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPSharePrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPSharePrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPSharePrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR362 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSharePrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR363 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPSharePrimContext
    
            #region sbrRPShareNonPrimContext
            IEnumerable<SBRContext> sbrRPShareNonPrimContexts = null;
            SBRContext sbrRPShareNonPrimContext = null;
    
            sbrRPShareNonPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ShareOfAmountTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Share"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonPrimaryProduction")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPShareNonPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPShareNonPrimContexts != null && sbrRPShareNonPrimContexts.Count() > 0)
            {
                sbrRPShareNonPrimContext = sbrRPShareNonPrimContexts.First();
        
                report.RPShareNonPrimId = sbrRPShareNonPrimContext.Id;
                report.RPShareNonPrimIdentifier = sbrRPShareNonPrimContext.EntityIdentifier;
                report.RPShareNonPrimIdentifierScheme = sbrRPShareNonPrimContext.EntityScheme;
        
                report.RPShareNonPrimCount = sbrRPShareNonPrimContexts.Count();
                report.RPShareNonPrimExists = true;
                report.RPShareNonPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPShareNonPrimContext) + 1;
                report.RPShareNonPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPShareNonPrimContexts.Last()) + 1;
        

                if (report.RPShareNonPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPShareNonPrimIdentifierTFN = report.RPShareNonPrimIdentifier;
    
                report.RPShareNonPrimStartDate = ((SBRDurationPeriod)sbrRPShareNonPrimContext.Period).DurationStartAsDateObject;
                report.RPShareNonPrimEndDate = ((SBRDurationPeriod)sbrRPShareNonPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPShareNonPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPShareNonPrimContexts.Count(); i++)
                {
                    sbrRPShareNonPrimContext = sbrRPShareNonPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPShareNonPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPShareNonPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPShareNonPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPShareNonPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPShareNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.NonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR366 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPShareNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.PTR367 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPShareNonPrimContext
            return report;

        } // Of Consume Method
        #region ValidateINTDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateINTDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;

                    SBRElement PTR313Value = null;
                    string PTR313 = null;
                    SBRElement PTR314Value = null;
                    string PTR314 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.Usage.Code", out PTR313Value))
                    {
                        if (!PTR313Value.IsNil) PTR313 = PTR313Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.FullName.Text", out PTR314Value))
                    {
                        if (!PTR314Value.IsNil) PTR314 = PTR314Value.Value;
                    }
        
                    #region VR.ATO.GEN.436279

        /*  VR.ATO.GEN.436279
            The unstructured person name in the declaration must include a Usage Code of 'DeclarationSignatory'.

        Legacy Rule Format:
            WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^PTR313 <> 'DeclarationSignatory')

        Data Elements:

        ^PTR313 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && PTR313 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = PTR313Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(PTR313Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR313", Value = PTR313 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTDeclarationCollection
        protected virtual void ValidateINTDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement PTR296Value = null;
                    string PTR296 = null;
                    SBRElement PTR295Value = null;
                    bool? PTR295 = null;
                    SBRElement PTR298Value = null;
                    string PTR298 = null;
                    SBRElement PTR294Value = null;
                    DateTime? PTR294 = null;
                    SBRElement PTR297Value = null;
                    string PTR297 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementType.Code", out PTR296Value))
                    {
                        if (!PTR296Value.IsNil) PTR296 = PTR296Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementAccepted.Indicator", out PTR295Value))
                    {
                        if (!PTR295Value.IsNil) PTR295 = ToBoolean(PTR295Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Statement.Text", out PTR298Value))
                    {
                        if (!PTR298Value.IsNil) PTR298 = PTR298Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Signature.Date", out PTR294Value))
                    {
                        if (!PTR294Value.IsNil) PTR294 = DateTime.Parse(PTR294Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.SignatoryIdentifier.Text", out PTR297Value))
                    {
                        if (!PTR297Value.IsNil) PTR297 = PTR297Value.Value;
                    }
        
                    #region VR.ATO.GEN.438061

        /*  VR.ATO.GEN.438061
            If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be 
            “TrueAndCorrect”.

        Legacy Rule Format:
            IF INT<>NULL AND INT:pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            ^INT <> NULL AND ^PTR296 <> 'TrueAndCorrect'

        Data Elements:

        ^PTR296 = INT:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^INT = INT
        */
                    assertion = (report.INTExists != false && PTR296 != @"TrueAndCorrect");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438061", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration statement type code is required.",
                            LongDescription = @"If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be ""TrueAndCorrect"".",
                            Location = PTR296Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementType.Code" : DeriveSbrElementLocation(PTR296Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438061"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR296", Value = PTR296 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTPersonUnstructuredNameCollection
        protected virtual void ValidateINTPersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement PTR171Value = null;
                    string PTR171 = null;
                    SBRElement PTR242Value = null;
                    string PTR242 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.Usage.Code", out PTR171Value))
                    {
                        if (!PTR171Value.IsNil) PTR171 = PTR171Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.FullName.Text", out PTR242Value))
                    {
                        if (!PTR242Value.IsNil) PTR242 = PTR242Value.Value;
                    }
        
                    #region VR.ATO.GEN.438080

        /*  VR.ATO.GEN.438080
            The unstructured person name for the intermediary contact must include a Usage Code of 'Contact'.

        Legacy Rule Format:
            WHERE IN TUPLE (xbrli\personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF INT:pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "Contact"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            ^PTR171 <> 'Contact'

        Data Elements:

        ^PTR171 = INT:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='Contact']
        */
                    assertion = (PTR171 != @"Contact");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438080", Severity = ProcessMessageSeverity.Error,
                            Description = @"Intermediary contact name must have a Usage code of 'Contact'",
                            LongDescription = @"The unstructured person name for the Intermediary contact must include a Usage Code of 'Contact'. Please contact your software vendor in order to rectify this error.",
                            Location = PTR171Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(PTR171Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438080"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR171", Value = PTR171 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTElectronicContactTelephoneCollection
        protected virtual void ValidateINTElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool ElectronicContactTelephoneCollection_03_INTExists = true;

                    SBRElement PTR245Value = null;
                    string PTR245 = null;

                    SBRElement PTR244Value = null;
                    string PTR244 = null;
                    SBRElement PTR172Value = null;
                    string PTR172 = null;
                    SBRElement PTR173Value = null;
                    string PTR173 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.Usage.Code", out PTR245Value))
                    {
                        if (!PTR245Value.IsNil) PTR245 = PTR245Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.ServiceLine.Code", out PTR244Value))
                    {
                        if (!PTR244Value.IsNil) PTR244 = PTR244Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.Area.Code", out PTR172Value))
                    {
                        if (!PTR172Value.IsNil) PTR172 = PTR172Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}ElectronicContact.Telephone.Minimal.Number", out PTR173Value))
                    {
                        if (!PTR173Value.IsNil) PTR173 = PTR173Value.Value;
                    }
        
                    #region VR.ATO.GEN.500131

        /*  VR.ATO.GEN.500131
            Telephone must include usage code '03' and service line code '01' or '02'.

        Legacy Rule Format:
            WHERE IN TUPLE(electroniccontacttelephone1.xx.xx:ElectronicContactTelephone)
            IF pyde.xx.xx:ElectronicContact.Telephone.Usage.Code <> "03" OR pyde.xx.xx:ElectronicContact.Telephone.ServiceLine.Code <> SET ("01", "02")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^ElectronicContactTelephone <> NULL) AND (^PTR245 <> '03' OR NotInSet(^PTR244, '"01", "02"'))

        Data Elements:

        ^PTR245 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Usage.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

        ^PTR244 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.ServiceLine.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

        ^ElectronicContactTelephone = INT:ElectronicContactTelephone
        */
                    assertion = (ElectronicContactTelephoneCollection_03_INTExists != false && (PTR245 != @"03" || !(IsMatch(PTR244, @"^(01|02)$",RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Telephone must include usage code '03' and service line code '01' or '02'.",
                            Location = PTR173Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.Telephone.Minimal.Number" : DeriveSbrElementLocation(PTR173Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500131"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR245", Value = PTR245 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR244", Value = PTR244 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPOrganisationNameDetailsCollection
        protected virtual void ValidateRPOrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool OrganisationNameDetailsCollection_MN_C_RPExists = true;

                    SBRElement PTR229Value = null;
                    string PTR229 = null;

                    SBRElement PTR261Value = null;
                    string PTR261 = null;
                    SBRElement PTR6Value = null;
                    string PTR6 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out PTR229Value))
                    {
                        if (!PTR229Value.IsNil) PTR229 = PTR229Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out PTR261Value))
                    {
                        if (!PTR261Value.IsNil) PTR261 = PTR261Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out PTR6Value))
                    {
                        if (!PTR6Value.IsNil) PTR6 = PTR6Value.Value;
                    }
        
                    #region VR.ATO.PTR.430281

        /*  VR.ATO.PTR.430281
            Organisation Name Type code must be 'MN' or 'MTR' and Currency Code must be 'C' or 'P'

        Legacy Rule Format:
            WHERE IN TUPLE (xbrli/Organisationname2.xx.xx:OrganisationNameDetails)
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> SET("MN", "MTR")) OR (pyde.xx.xx:OrganisationNameDetails.Currency.Code <> SET("C", "P"))
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND ((NotInSet(^PTR229, '"MN", "MTR"')) OR (NotInSet(^PTR261, '"C", "P"')))

        Data Elements:

        ^PTR229 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^PTR261 = RP:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^OrganisationNameDetails = RP:OrganisationNameDetails
        */
                    assertion = (OrganisationNameDetailsCollection_MN_C_RPExists != false && (!(IsMatch(PTR229, @"^(MN|MTR)$",RegexOptions.IgnoreCase)) || !(IsMatch(PTR261, @"^(C|P)$",RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430305", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type code must be 'MN' or 'MTR' and Currency Code must be 'C' or 'P'",
                            LongDescription = @"A non-individual name must have the Name Type code 'MN' (main) or 'MTR' (main trading name) and a Currency Code 'C' (current) or 'P' (previous). Please contact your software vendor in order to rectify this error.",
                            Location = PTR6Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalName.Text" : DeriveSbrElementLocation(PTR6Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430281"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPOrganisationNameDetailsCollection1
        protected virtual void ValidateRPOrganisationNameDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection
        protected virtual void ValidateRPAddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool AddressDetailsCollection_POS_C_RPExists = true;

                    SBRElement PTR223Value = null;
                    string PTR223 = null;

                    SBRElement PTR226Value = null;
                    string PTR226 = null;
                    SBRElement PTR216Value = null;
                    bool? PTR216 = null;
                    SBRElement PTR9Value = null;
                    string PTR9 = null;
                    SBRElement PTR211Value = null;
                    string PTR211 = null;
                    SBRElement PTR214Value = null;
                    string PTR214 = null;
                    SBRElement PTR215Value = null;
                    string PTR215 = null;
                    SBRElement PTR10Value = null;
                    string PTR10 = null;
                    SBRElement PTR12Value = null;
                    string PTR12 = null;
                    SBRElement PTR11Value = null;
                    string PTR11 = null;
                    SBRElement PTR302Value = null;
                    string PTR302 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out PTR223Value))
                    {
                        if (!PTR223Value.IsNil) PTR223 = PTR223Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out PTR226Value))
                    {
                        if (!PTR226Value.IsNil) PTR226 = PTR226Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out PTR216Value))
                    {
                        if (!PTR216Value.IsNil) PTR216 = ToBoolean(PTR216Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out PTR9Value))
                    {
                        if (!PTR9Value.IsNil) PTR9 = PTR9Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out PTR211Value))
                    {
                        if (!PTR211Value.IsNil) PTR211 = PTR211Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out PTR214Value))
                    {
                        if (!PTR214Value.IsNil) PTR214 = PTR214Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out PTR215Value))
                    {
                        if (!PTR215Value.IsNil) PTR215 = PTR215Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out PTR10Value))
                    {
                        if (!PTR10Value.IsNil) PTR10 = PTR10Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out PTR12Value))
                    {
                        if (!PTR12Value.IsNil) PTR12 = PTR12Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out PTR11Value))
                    {
                        if (!PTR11Value.IsNil) PTR11 = PTR11Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out PTR302Value))
                    {
                        if (!PTR302Value.IsNil) PTR302 = PTR302Value.Value;
                    }
        
                    #region VR.ATO.GEN.430307

        /*  VR.ATO.GEN.430307
            An address must have a Usage code 'POS' (postal) or 'BUS' (business) and a Currency Code 'C' (current) or 'P' (previous).

        Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Usage.Code <> SET("POS", "BUS")) OR (pyde.xx.xx:AddressDetails.Currency.Code <> SET("C", "P")) 
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^AddressDetails <> NULL) AND ((NotInSet(^PTR223, '"POS", "BUS"')) OR (NotInSet(^PTR226, '"C", "P"')))

        Data Elements:

        ^PTR223 = RP:AddressDetails:AddressDetails.Usage.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']

        ^PTR226 = RP:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']

        ^AddressDetails = RP:AddressDetails
        */
                    assertion = (AddressDetailsCollection_POS_C_RPExists != false && (!(IsMatch(PTR223, @"^(POS|BUS)$",RegexOptions.IgnoreCase)) || !(IsMatch(PTR226, @"^(C|P)$",RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430307", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address usage code must be 'POS' or 'BUS' and Currency Code must be 'C' or 'P'",
                            LongDescription = @"An address must have a Usage code 'POS' (postal) or 'BUS' (business) and a Currency Code 'C' (current) or 'P' (previous). Please contact your software vendor in order to rectify this error.",
                            Location = PTR9Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Line1.Text" : DeriveSbrElementLocation(PTR9Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430307"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection1
        protected virtual void ValidateRPAddressDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPMailRecipient_PersonNameDetailsCollection
        protected virtual void ValidateRPMailRecipient_PersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPExists = true;

                    SBRElement PTR231Value = null;
                    string PTR231 = null;

                    SBRElement PTR233Value = null;
                    string PTR233 = null;

                    SBRElement PTR232Value = null;
                    string PTR232 = null;
                    SBRElement PTR19Value = null;
                    string PTR19 = null;
                    SBRElement PTR21Value = null;
                    string PTR21 = null;
                    SBRElement PTR20Value = null;
                    string PTR20 = null;
                    SBRElement PTR22Value = null;
                    string PTR22 = null;
                    SBRElement PTR23Value = null;
                    string PTR23 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.PersonNameType.Code", out PTR231Value))
                    {
                        if (!PTR231Value.IsNil) PTR231 = PTR231Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Currency.Code", out PTR233Value))
                    {
                        if (!PTR233Value.IsNil) PTR233 = PTR233Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Usage.Code", out PTR232Value))
                    {
                        if (!PTR232Value.IsNil) PTR232 = PTR232Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Title.Text", out PTR19Value))
                    {
                        if (!PTR19Value.IsNil) PTR19 = PTR19Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.NameSuffix.Text", out PTR21Value))
                    {
                        if (!PTR21Value.IsNil) PTR21 = PTR21Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.FamilyName.Text", out PTR20Value))
                    {
                        if (!PTR20Value.IsNil) PTR20 = PTR20Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.GivenName.Text", out PTR22Value))
                    {
                        if (!PTR22Value.IsNil) PTR22 = PTR22Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.OtherGivenName.Text", out PTR23Value))
                    {
                        if (!PTR23Value.IsNil) PTR23 = PTR23Value.Value;
                    }
        
                    #region VR.ATO.PTR.430310

        /*  VR.ATO.PTR.430310
            The name of the partner to whom notices should be sent must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'MailRecipient'. Please contact your software vendor in order to rectify this error.

        Legacy Rule Format:
            WHERE IN TUPLE(prsnstrcnm3.xx.xx:PersonNameDetails) IN TUPLE (ptr.0012.lodge.req.xx.xx:MailRecipient)
            IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL") OR (pyde.xx.xx:PersonNameDetails.Currency.Code <> "C") OR (pyde.xx.xx:PersonNameDetails.Usage.Code <> "MailRecipient")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND ((^PTR231 <> 'LGL') OR (^PTR233 <> 'C') OR (^PTR232 <> 'MailRecipient'))

        Data Elements:

        ^PTR231 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']

        ^PTR233 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.Currency.Code WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']

        ^PTR232 = RP:MailRecipient:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.Usage.Code='MailRecipient'].[PersonNameDetails.PersonNameType.Code='LGL']

        ^PersonNameDetails = RP:MailRecipient:PersonNameDetails
        */
                    assertion = (MailRecipient_PersonNameDetailsCollection_MailRecipient_LGL_RPExists != false && (PTR231 != @"LGL" || PTR233 != @"C" || PTR232 != @"MailRecipient"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430310", Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name Type code, Currency code and Usage code must be present and correct",
                            LongDescription = @"The name of the person to whom notices should be sent must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'MailRecipient'. Please contact your software vendor in order to rectify this error.",
                            Location = PTR20Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.FamilyName.Text" : DeriveSbrElementLocation(PTR20Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430310"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR231", Value = PTR231 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR233", Value = PTR233 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR232", Value = PTR232 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPMailRecipient_OrganisationNameDetailsCollection
        protected virtual void ValidateRPMailRecipient_OrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool MailRecipient_OrganisationNameDetailsCollection_MN_C_RPExists = true;

                    SBRElement PTR234Value = null;
                    string PTR234 = null;

                    SBRElement PTR280Value = null;
                    string PTR280 = null;
                    SBRElement PTR24Value = null;
                    string PTR24 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out PTR234Value))
                    {
                        if (!PTR234Value.IsNil) PTR234 = PTR234Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out PTR280Value))
                    {
                        if (!PTR280Value.IsNil) PTR280 = PTR280Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out PTR24Value))
                    {
                        if (!PTR24Value.IsNil) PTR24 = PTR24Value.Value;
                    }
        
                    #region VR.ATO.PTR.430304

        /*  VR.ATO.PTR.430304
            Organisation Name Type code must be 'MN' and Currency Code must be 'C'

        Legacy Rule Format:
            WHERE IN TUPLE (Organisationname2.xx.xx:OrganisationNameDetails) IN TUPLE (ptr.0012.lodge.req.xx.xx:MailRecipient)
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN") OR (pyde.xx.xx:OrganisationNameDetails.Currency.Code <> "C")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND ((^PTR234 <> 'MN') OR (^PTR280 <> 'C'))

        Data Elements:

        ^PTR234 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^PTR280 = RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^OrganisationNameDetails = RP:MailRecipient:OrganisationNameDetails
        */
                    assertion = (MailRecipient_OrganisationNameDetailsCollection_MN_C_RPExists != false && (PTR234 != @"MN" || PTR280 != @"C"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430304", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type code must be 'MN' and Currency Code must be 'C'",
                            LongDescription = @"The name of the non-individual partner to whom notices should be sent must have the Name Type code 'MN' (main) and Currency Code 'C' (current). Please contact your software vendor in order to rectify this error.",
                            Location = PTR24Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalName.Text" : DeriveSbrElementLocation(PTR24Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430304"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR234", Value = PTR234 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR280", Value = PTR280 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPOrganisationNameDetailsCollection2
        protected virtual void ValidateRPOrganisationNameDetailsCollection2(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection2
        protected virtual void ValidateRPAddressDetailsCollection2(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool AddressDetailsCollection_BUS_C_RPExists = true;

                    SBRElement PTR225Value = null;
                    string PTR225 = null;

                    SBRElement PTR228Value = null;
                    string PTR228 = null;
                    SBRElement PTR222Value = null;
                    bool? PTR222 = null;
                    SBRElement PTR117Value = null;
                    string PTR117 = null;
                    SBRElement PTR213Value = null;
                    string PTR213 = null;
                    SBRElement PTR220Value = null;
                    string PTR220 = null;
                    SBRElement PTR221Value = null;
                    string PTR221 = null;
                    SBRElement PTR118Value = null;
                    string PTR118 = null;
                    SBRElement PTR120Value = null;
                    string PTR120 = null;
                    SBRElement PTR119Value = null;
                    string PTR119 = null;
                    SBRElement PTR304Value = null;
                    string PTR304 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out PTR225Value))
                    {
                        if (!PTR225Value.IsNil) PTR225 = PTR225Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out PTR228Value))
                    {
                        if (!PTR228Value.IsNil) PTR228 = PTR228Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out PTR222Value))
                    {
                        if (!PTR222Value.IsNil) PTR222 = ToBoolean(PTR222Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out PTR117Value))
                    {
                        if (!PTR117Value.IsNil) PTR117 = PTR117Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out PTR213Value))
                    {
                        if (!PTR213Value.IsNil) PTR213 = PTR213Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out PTR220Value))
                    {
                        if (!PTR220Value.IsNil) PTR220 = PTR220Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out PTR221Value))
                    {
                        if (!PTR221Value.IsNil) PTR221 = PTR221Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out PTR118Value))
                    {
                        if (!PTR118Value.IsNil) PTR118 = PTR118Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out PTR120Value))
                    {
                        if (!PTR120Value.IsNil) PTR120 = PTR120Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out PTR119Value))
                    {
                        if (!PTR119Value.IsNil) PTR119 = PTR119Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out PTR304Value))
                    {
                        if (!PTR304Value.IsNil) PTR304 = PTR304Value.Value;
                    }
        
                    #region VR.ATO.GEN.430306

        /*  VR.ATO.GEN.430306
            The Business address of main business must have Currency Code 'C' (current).

        Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Usage.Code = "BUS") AND (pyde.xx.xx:AddressDetails.Currency.Code = "P") 
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^AddressDetails <> NULL) AND ((^PTR225 = 'BUS') AND (^PTR228 = 'P'))

        Data Elements:

        ^PTR225 = RP:AddressDetails:AddressDetails.Usage.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']

        ^PTR228 = RP:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']

        ^AddressDetails = RP:AddressDetails
        */
                    assertion = (AddressDetailsCollection_BUS_C_RPExists != false && (PTR225 == @"BUS" && PTR228 == @"P"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430306", Severity = ProcessMessageSeverity.Error,
                            Description = @"Business address Currency code must be 'C'",
                            LongDescription = @"The Business address of main business must have Currency Code 'C' (current). Please contact your software vendor in order to rectify this error.",
                            Location = PTR117Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Line1.Text" : DeriveSbrElementLocation(PTR117Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430306"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPStatementOfDistribution_PersonNameDetailsCollection
        protected virtual void ValidateRPStatementOfDistribution_PersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPExists = true;

                    SBRElement PTR235Value = null;
                    string PTR235 = null;

                    SBRElement PTR267Value = null;
                    string PTR267 = null;

                    SBRElement PTR310Value = null;
                    string PTR310 = null;
                    SBRElement PTR311Value = null;
                    string PTR311 = null;
                    SBRElement PTR312Value = null;
                    string PTR312 = null;
                    SBRElement PTR146Value = null;
                    string PTR146 = null;
                    SBRElement PTR147Value = null;
                    string PTR147 = null;
                    SBRElement PTR148Value = null;
                    string PTR148 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.PersonNameType.Code", out PTR235Value))
                    {
                        if (!PTR235Value.IsNil) PTR235 = PTR235Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Currency.Code", out PTR267Value))
                    {
                        if (!PTR267Value.IsNil) PTR267 = PTR267Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Usage.Code", out PTR310Value))
                    {
                        if (!PTR310Value.IsNil) PTR310 = PTR310Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.Title.Text", out PTR311Value))
                    {
                        if (!PTR311Value.IsNil) PTR311 = PTR311Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.NameSuffix.Text", out PTR312Value))
                    {
                        if (!PTR312Value.IsNil) PTR312 = PTR312Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.FamilyName.Text", out PTR146Value))
                    {
                        if (!PTR146Value.IsNil) PTR146 = PTR146Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.GivenName.Text", out PTR147Value))
                    {
                        if (!PTR147Value.IsNil) PTR147 = PTR147Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonNameDetails.OtherGivenName.Text", out PTR148Value))
                    {
                        if (!PTR148Value.IsNil) PTR148 = PTR148Value.Value;
                    }
        
                    #region VR.ATO.PTR.430311

        /*  VR.ATO.PTR.430311
            The Individual name of each person in each Statement of Distribution must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'Contact'. Please contact your software vendor in order to rectify this error.

        Legacy Rule Format:
            WHERE IN TUPLE(prsnstrcnm3.xx.xx:PersonNameDetails) IN TUPLE (ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
            IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL") OR (pyde.xx.xx:PersonNameDetails.Currency.Code <> "C") OR (pyde.xx.xx:PersonNameDetails.Usage.Code <> "Contact")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND ((^PTR235 <> 'LGL') OR (^PTR267 <> 'C') OR (^PTR310 <> 'Contact'))

        Data Elements:

        ^PTR235 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']

        ^PTR267 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.Currency.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']

        ^PTR310 = RP:StatementOfDistribution:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact']

        ^PersonNameDetails = RP:StatementOfDistribution:PersonNameDetails
        */
                    assertion = (StatementOfDistribution_PersonNameDetailsCollection_LGL_Contact_RPExists != false && (PTR235 != @"LGL" || PTR267 != @"C" || PTR310 != @"Contact"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430311", Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name Type code, Currency code and Usage code must be present and correct",
                            LongDescription = @"The Individual name of each person in each Statement of Distribution must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'Contact'. Please contact your software vendor in order to rectify this error.",
                            Location = PTR146Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.FamilyName.Text" : DeriveSbrElementLocation(PTR146Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430311"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR235", Value = PTR235 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR267", Value = PTR267 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR310", Value = PTR310 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPStatementOfDistribution_AddressDetailsCollection
        protected virtual void ValidateRPStatementOfDistribution_AddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool StatementOfDistribution_AddressDetailsCollection_RES_C_RPExists = true;

                    SBRElement PTR330Value = null;
                    string PTR330 = null;

                    SBRElement PTR331Value = null;
                    string PTR331 = null;
                    SBRElement PTR329Value = null;
                    bool? PTR329 = null;
                    SBRElement PTR332Value = null;
                    string PTR332 = null;
                    SBRElement PTR333Value = null;
                    string PTR333 = null;
                    SBRElement PTR339Value = null;
                    string PTR339 = null;
                    SBRElement PTR340Value = null;
                    string PTR340 = null;
                    SBRElement PTR334Value = null;
                    string PTR334 = null;
                    SBRElement PTR335Value = null;
                    string PTR335 = null;
                    SBRElement PTR336Value = null;
                    string PTR336 = null;
                    SBRElement PTR337Value = null;
                    string PTR337 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Usage.Code", out PTR330Value))
                    {
                        if (!PTR330Value.IsNil) PTR330 = PTR330Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Currency.Code", out PTR331Value))
                    {
                        if (!PTR331Value.IsNil) PTR331 = PTR331Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.OverseasAddress.Indicator", out PTR329Value))
                    {
                        if (!PTR329Value.IsNil) PTR329 = ToBoolean(PTR329Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line1.Text", out PTR332Value))
                    {
                        if (!PTR332Value.IsNil) PTR332 = PTR332Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line2.Text", out PTR333Value))
                    {
                        if (!PTR333Value.IsNil) PTR333 = PTR333Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line3.Text", out PTR339Value))
                    {
                        if (!PTR339Value.IsNil) PTR339 = PTR339Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Line4.Text", out PTR340Value))
                    {
                        if (!PTR340Value.IsNil) PTR340 = PTR340Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.LocalityName.Text", out PTR334Value))
                    {
                        if (!PTR334Value.IsNil) PTR334 = PTR334Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Postcode.Text", out PTR335Value))
                    {
                        if (!PTR335Value.IsNil) PTR335 = PTR335Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.StateOrTerritory.Code", out PTR336Value))
                    {
                        if (!PTR336Value.IsNil) PTR336 = PTR336Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}AddressDetails.Country.Code", out PTR337Value))
                    {
                        if (!PTR337Value.IsNil) PTR337 = PTR337Value.Value;
                    }
        
                    #region VR.ATO.PTR.431052

        /*  VR.ATO.PTR.431052
            Address usage code must be 'RES' OR 'BUS' and Currency Code must be 'C'

        Legacy Rule Format:
            WHERE IN TUPLE(address2.xx.xx:AddressDetails) IN TUPLE (ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
            IF (pyde.xx.xx:AddressDetails.Usage.Code <> "BUS" OR "RES") OR (pyde.xx.xx:AddressDetails.Currency.Code <> "C") 
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (NotInSet(^PTR330, '"BUS", "RES"') OR (^PTR331 <> 'C'))

        Data Elements:

        ^PTR330 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Usage.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']

        ^PTR331 = RP:StatementOfDistribution:AddressDetails:AddressDetails.Currency.Code WHERE [AddressDetails.Usage.Code='RES'].[AddressDetails.Currency.Code='C']

        ^AddressDetails = RP:StatementOfDistribution:AddressDetails
        */
                    assertion = (StatementOfDistribution_AddressDetailsCollection_RES_C_RPExists != false && (!(IsMatch(PTR330, @"^(BUS|RES)$",RegexOptions.IgnoreCase)) || PTR331 != @"C"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PTR.431052", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address usage code must be 'BUS' OR 'RES' and Currency Code must be 'C'",
                            Location = PTR332Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Line1.Text" : DeriveSbrElementLocation(PTR332Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.431052"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPStatementOfDistribution_OrganisationNameDetailsCollection
        protected virtual void ValidateRPStatementOfDistribution_OrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPExists = true;

                    SBRElement PTR236Value = null;
                    string PTR236 = null;

                    SBRElement PTR269Value = null;
                    string PTR269 = null;
                    SBRElement PTR149Value = null;
                    string PTR149 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalNameType.Code", out PTR236Value))
                    {
                        if (!PTR236Value.IsNil) PTR236 = PTR236Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.Currency.Code", out PTR269Value))
                    {
                        if (!PTR269Value.IsNil) PTR269 = PTR269Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}OrganisationNameDetails.OrganisationalName.Text", out PTR149Value))
                    {
                        if (!PTR149Value.IsNil) PTR149 = PTR149Value.Value;
                    }
        
                    #region VR.ATO.PTR.430303

        /*  VR.ATO.PTR.430303
            Organisation Name Type code must be 'MN' and Currency Code must be 'C'

        Legacy Rule Format:
            WHERE IN TUPLE (Organisationname2.xx.xx:OrganisationNameDetails) IN TUPLE (ptr.0012.lodge.req.xx.xx:StatementOfDistribution)
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN") OR (pyde.xx.xx:OrganisationNameDetails.Currency.Code <> "C")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND ((^PTR236 <> 'MN') OR (^PTR269 <> 'C'))

        Data Elements:

        ^PTR236 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^PTR269 = RP:StatementOfDistribution:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^OrganisationNameDetails = RP:StatementOfDistribution:OrganisationNameDetails
        */
                    assertion = (StatementOfDistribution_OrganisationNameDetailsCollection_MN_C_RPExists != false && (PTR236 != @"MN" || PTR269 != @"C"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430303", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type code must be 'MN' and Currency Code must be 'C'",
                            LongDescription = @"Each non-individual name within the Statement of Distribution must have the Name Type code 'MN' (main) and Currency Code 'C' (current). Please contact your software vendor in order to rectify this error.",
                            Location = PTR149Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalName.Text" : DeriveSbrElementLocation(PTR149Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.PTR.430303"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR236", Value = PTR236 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR269", Value = PTR269 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPStatementOfDistribution_AddressDetailsCollection1
        protected virtual void ValidateRPStatementOfDistribution_AddressDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateRPDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists = true;

                    SBRElement PTR260Value = null;
                    string PTR260 = null;
                    SBRElement PTR264Value = null;
                    string PTR264 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.Usage.Code", out PTR260Value))
                    {
                        if (!PTR260Value.IsNil) PTR260 = PTR260Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}PersonUnstructuredName.FullName.Text", out PTR264Value))
                    {
                        if (!PTR264Value.IsNil) PTR264 = PTR264Value.Value;
                    }
        
                    #region VR.ATO.GEN.436279

        /*  VR.ATO.GEN.436279
            The unstructured person name in the declaration must include a Usage Code of 'DeclarationSignatory'.

        Legacy Rule Format:
            WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^PTR260 <> 'DeclarationSignatory')

        Data Elements:

        ^PTR260 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = RP:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists != false && PTR260 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = PTR260Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(PTR260Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR260", Value = PTR260 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPDeclarationCollection
        protected virtual void ValidateRPDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, PTR2024 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement PTR291Value = null;
                    string PTR291 = null;
                    SBRElement PTR290Value = null;
                    bool? PTR290 = null;
                    SBRElement PTR293Value = null;
                    string PTR293 = null;
                    SBRElement PTR289Value = null;
                    DateTime? PTR289 = null;
                    SBRElement PTR292Value = null;
                    string PTR292 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementType.Code", out PTR291Value))
                    {
                        if (!PTR291Value.IsNil) PTR291 = PTR291Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.StatementAccepted.Indicator", out PTR290Value))
                    {
                        if (!PTR290Value.IsNil) PTR290 = ToBoolean(PTR290Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Statement.Text", out PTR293Value))
                    {
                        if (!PTR293Value.IsNil) PTR293 = PTR293Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.Signature.Date", out PTR289Value))
                    {
                        if (!PTR289Value.IsNil) PTR289 = DateTime.Parse(PTR289Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ptr}Declaration.SignatoryIdentifier.Text", out PTR292Value))
                    {
                        if (!PTR292Value.IsNil) PTR292 = PTR292Value.Value;
                    }
        
                    #region VR.ATO.GEN.438062

        /*  VR.ATO.GEN.438062
            If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be “TrueAndCorrect”.

        Legacy Rule Format:
            IF INT=NULL AND RP:pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            ^INT = NULL AND ^PTR291 <> 'TrueAndCorrect'

        Data Elements:

        ^PTR291 = RP:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^INT = INT
        */
                    assertion = (report.INTExists == false && PTR291 != @"TrueAndCorrect");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438062", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration statement type code is required.",
                            LongDescription = @"If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be ""TrueAndCorrect"".",
                            Location = PTR291Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementType.Code" : DeriveSbrElementLocation(PTR291Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438062"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "PTR291", Value = PTR291 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

    } // Of Class
} // Of Namespace

