using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using Ato.EN.IntegrationServices.Document.DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationRDTIS
{
    /// <summary>
    /// XML Consumer for RDTIS
    /// </summary>
    public class RDTIS2023XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static RDTIS2023XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.rdtis.0006.2023.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;
  
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();        
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count-1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements) 
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
           return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory) 
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader) 
        {
            bool retval = false;

            if ( (reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if(_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if(_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
             }

             //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf , string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if(numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if(numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader) 
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }
      
                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public RDTIS2023 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            RDTIS2023 report = new RDTIS2023();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:RDTIS");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "RDTIS" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/rdtis")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

    
            #region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader,true) && reader.LocalName == "RP" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true; 
        
                #region AmendmentI
                _currentXPath.Push("/tns:AmendmentI");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "AmendmentI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.RDTIS1 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AmendmentI
        
                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.RDTIS70 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxFileNumberId
        
                #region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.RDTIS71 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodStartD
        
                #region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.RDTIS72 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodEndD
        
                #region InnovationRegistrationNumberId
                _currentXPath.Push("/tns:InnovationRegistrationNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "InnovationRegistrationNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.RDTIS2 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion InnovationRegistrationNumberId
        
                #region ExpenseResearchAndDevelopmentResearchAndDevelopmentAccountingExpenditureAddBackA
                _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentResearchAndDevelopmentAccountingExpenditureAddBackA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentResearchAndDevelopmentAccountingExpenditureAddBackA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.RDTIS3 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ExpenseResearchAndDevelopmentResearchAndDevelopmentAccountingExpenditureAddBackA
        
                #region NotionalDeductionsCalculation
                _currentXPath.Push("/tns:NotionalDeductionsCalculation");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "NotionalDeductionsCalculation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_NotionalDeductionsCalculationCollectionExists = true;
                    report.RP_NotionalDeductionsCalculationCollectionCount += 1;
                    _found = true; 
            
                    #region AustralianOwned
                    _currentXPath.Push("/tns:AustralianOwned");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianOwned" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_NotionalDeductionsCalculation_AustralianOwnedCollectionExists = true;
                        report.RP_NotionalDeductionsCalculation_AustralianOwnedCollectionCount += 1;
                        _found = true; 
                
                        #region ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS4 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA
                
                        #region ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS6 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA
                
                        #region RemunerationWagesAndSalariesSalaryA
                        _currentXPath.Push("/tns:RemunerationWagesAndSalariesSalaryA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationWagesAndSalariesSalaryA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS8 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion RemunerationWagesAndSalariesSalaryA
                
                        #region ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS10 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA
                
                        #region ExpenseResearchAndDevelopmentFeedstockA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentFeedstockA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentFeedstockA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS12 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentFeedstockA
                
                        #region ExpenseResearchAndDevelopmentAssociatesExpenditureA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesExpenditureA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesExpenditureA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS14 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAssociatesExpenditureA
                
                        #region ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS16 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA
                
                        #region ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS20 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA
                
                        #region ExpenseResearchAndDevelopmentAllocatedTotalA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAllocatedTotalA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAllocatedTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS22 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAllocatedTotalA
                    } // End of if AustralianOwned node exists
            
                    _currentXPath.Pop();
                    #endregion AustralianOwned
            
                    #region ForeignOwned
                    _currentXPath.Push("/tns:ForeignOwned");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ForeignOwned" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_NotionalDeductionsCalculation_ForeignOwnedCollectionExists = true;
                        report.RP_NotionalDeductionsCalculation_ForeignOwnedCollectionCount += 1;
                        _found = true; 
                
                        #region ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS5 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA
                
                        #region ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS7 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA
                
                        #region RemunerationWagesAndSalariesSalaryA
                        _currentXPath.Push("/tns:RemunerationWagesAndSalariesSalaryA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationWagesAndSalariesSalaryA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS9 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion RemunerationWagesAndSalariesSalaryA
                
                        #region ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS11 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA
                
                        #region ExpenseResearchAndDevelopmentFeedstockA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentFeedstockA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentFeedstockA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS13 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentFeedstockA
                
                        #region ExpenseResearchAndDevelopmentAssociatesExpenditureA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesExpenditureA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesExpenditureA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS15 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAssociatesExpenditureA
                
                        #region ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS17 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA
                
                        #region ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS21 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA
                
                        #region ExpenseResearchAndDevelopmentAllocatedTotalA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAllocatedTotalA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAllocatedTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS23 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAllocatedTotalA
                    } // End of if ForeignOwned node exists
            
                    _currentXPath.Pop();
                    #endregion ForeignOwned
            
                    #region ExpenseResearchAndDevelopmentAllocatedTotalA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAllocatedTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ExpenseResearchAndDevelopmentAllocatedTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS24 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentAllocatedTotalA
                } // End of if NotionalDeductionsCalculation node exists
        
                _currentXPath.Pop();
                #endregion NotionalDeductionsCalculation
        
                #region Feedstock
                _currentXPath.Push("/tns:Feedstock");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "Feedstock" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_FeedstockCollectionExists = true;
                    report.RP_FeedstockCollectionCount += 1;
                    _found = true; 
            
                    #region IncomeResearchAndDevelopmentRevenueTotalA
                    _currentXPath.Push("/tns:IncomeResearchAndDevelopmentRevenueTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeResearchAndDevelopmentRevenueTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS25 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeResearchAndDevelopmentRevenueTotalA
            
                    #region ExpenseResearchAndDevelopmentInputA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentInputA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentInputA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS26 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentInputA
            
                    #region ExpenseResearchAndDevelopmentAdjustmentBalanceA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAdjustmentBalanceA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAdjustmentBalanceA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS73 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentAdjustmentBalanceA
                } // End of if Feedstock node exists
        
                _currentXPath.Pop();
                #endregion Feedstock
        
                #region ResearchandDevelopmentrecoupment
                _currentXPath.Push("/tns:ResearchandDevelopmentrecoupment");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ResearchandDevelopmentrecoupment" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ResearchandDevelopmentrecoupmentCollectionExists = true;
                    report.RP_ResearchandDevelopmentrecoupmentCollectionCount += 1;
                    _found = true; 
            
                    #region ResearchAndDevelopmentGovernmentGrantsAndRecoupmentsA
                    _currentXPath.Push("/tns:ResearchAndDevelopmentGovernmentGrantsAndRecoupmentsA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ResearchAndDevelopmentGovernmentGrantsAndRecoupmentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS28 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ResearchAndDevelopmentGovernmentGrantsAndRecoupmentsA
            
                    #region ExpenseResearchAndDevelopmentRecoupmentRelatedA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentRecoupmentRelatedA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentRecoupmentRelatedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS29 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentRecoupmentRelatedA
            
                    #region ExpenseResearchAndDevelopmentGrantRelatedA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentGrantRelatedA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentGrantRelatedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS30 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentGrantRelatedA
            
                    #region ExpenseResearchAndDevelopmentRecoupmentNetA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentRecoupmentNetA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentRecoupmentNetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS74 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentRecoupmentNetA
                } // End of if ResearchandDevelopmentrecoupment node exists
        
                _currentXPath.Pop();
                #endregion ResearchandDevelopmentrecoupment
        
                #region AssessablebalancingadjustmentsIncomeResearchAndDevelopmentBalancingAdjustmentA
                _currentXPath.Push("/tns:AssessablebalancingadjustmentsIncomeResearchAndDevelopmentBalancingAdjustmentA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AssessablebalancingadjustmentsIncomeResearchAndDevelopmentBalancingAdjustmentA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.RDTIS75 = currentDecimalValue;
                            }
                            report.RP_AssessablebalancingadjustmentsCollectionExists = true;
                            report.RP_AssessablebalancingadjustmentsCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AssessablebalancingadjustmentsIncomeResearchAndDevelopmentBalancingAdjustmentA
        
                #region TotalclawbackIncomeResearchAndDevelopmentClawbackA
                _currentXPath.Push("/tns:TotalclawbackIncomeResearchAndDevelopmentClawbackA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TotalclawbackIncomeResearchAndDevelopmentClawbackA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.RDTIS76 = currentDecimalValue;
                            }
                            report.RP_TotalclawbackCollectionExists = true;
                            report.RP_TotalclawbackCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TotalclawbackIncomeResearchAndDevelopmentClawbackA
        
                #region DeductiblebalancingadjustmentsExpenseResearchAndDevelopmentAdjustmentBalanceA
                _currentXPath.Push("/tns:DeductiblebalancingadjustmentsExpenseResearchAndDevelopmentAdjustmentBalanceA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "DeductiblebalancingadjustmentsExpenseResearchAndDevelopmentAdjustmentBalanceA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.RDTIS77 = currentDecimalValue;
                            }
                            report.RP_DeductiblebalancingadjustmentsCollectionExists = true;
                            report.RP_DeductiblebalancingadjustmentsCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion DeductiblebalancingadjustmentsExpenseResearchAndDevelopmentAdjustmentBalanceA
        
                #region ExpenditureToAssociates
                _currentXPath.Push("/tns:ExpenditureToAssociates");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenditureToAssociates" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ExpenditureToAssociatesCollectionExists = true;
                    report.RP_ExpenditureToAssociatesCollectionCount += 1;
                    _found = true; 
            
                    #region IncurredOrPaid
                    _currentXPath.Push("/tns:IncurredOrPaid");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncurredOrPaid" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_ExpenditureToAssociates_IncurredOrPaidCollectionExists = true;
                        report.RP_ExpenditureToAssociates_IncurredOrPaidCollectionCount += 1;
                        _found = true; 
                
                        #region ExpenseResearchAndDevelopmentAssociatesCarriedForwardA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesCarriedForwardA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesCarriedForwardA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS51 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAssociatesCarriedForwardA
                
                        #region ExpenseResearchAndDevelopmentAssociatesUnpaidA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesUnpaidA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesUnpaidA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS32 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAssociatesUnpaidA
                
                        #region ExpenseResearchAndDevelopmentAssociatesClaimedA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesClaimedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesClaimedA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS33 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAssociatesClaimedA
                
                        #region ExpenseResearchAndDevelopmentAssociatesNotClaimedA
                        _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesNotClaimedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesNotClaimedA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS34 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseResearchAndDevelopmentAssociatesNotClaimedA
                    } // End of if IncurredOrPaid node exists
            
                    _currentXPath.Pop();
                    #endregion IncurredOrPaid
            
                    #region ExpenseResearchAndDevelopmentAssociatesCarriedForwardA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentAssociatesCarriedForwardA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentAssociatesCarriedForwardA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS35 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentAssociatesCarriedForwardA
                } // End of if ExpenditureToAssociates node exists
        
                _currentXPath.Pop();
                #endregion ExpenditureToAssociates
        
                #region AggregatedTurnover
                _currentXPath.Push("/tns:AggregatedTurnover");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "AggregatedTurnover" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_AggregatedTurnoverCollectionExists = true;
                    report.RP_AggregatedTurnoverCollectionCount += 1;
                    _found = true; 
            
                    #region TaxConcessionExemptEntityOwnershipI
                    _currentXPath.Push("/tns:TaxConcessionExemptEntityOwnershipI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "TaxConcessionExemptEntityOwnershipI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.RDTIS36 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionExemptEntityOwnershipI
            
                    #region TaxConcessionAtOrAboveThresholdI
                    _currentXPath.Push("/tns:TaxConcessionAtOrAboveThresholdI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionAtOrAboveThresholdI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.RDTIS37 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionAtOrAboveThresholdI
            
                    #region TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS38 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA
            
                    #region GroupMember
            
                    _currentXPath.Push("/tns:GroupMemberCollection");
                    // 4. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "GroupMemberCollection" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_AggregatedTurnover_GroupMemberCollection = new List<RDTIS2023.RP_AggregatedTurnover_GroupMember>();
                        report.RP_AggregatedTurnover_GroupMemberCollectionExists = true;
            
                        _found = true;
            
                        while ( ReadToNextElement(reader,false) && reader.LocalName == "GroupMember" ) 
                        {
                            _found = true;
            
                            RDTIS2023.RP_AggregatedTurnover_GroupMember groupMember = new RDTIS2023.RP_AggregatedTurnover_GroupMember();
                            report.RP_AggregatedTurnover_GroupMemberCollection.Add(groupMember);
                            report.RP_AggregatedTurnover_GroupMemberCollectionCount += 1;
                            groupMember.OccurrenceIndex = report.RP_AggregatedTurnover_GroupMemberCollectionCount;
            
                            _currentXPath.Push("/tns:GroupMember[" + report.RP_AggregatedTurnover_GroupMemberCollectionCount + "]");
                    
                            #region OrganisationNameDetailsOrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        groupMember.RDTIS41 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OrganisationNameDetailsOrganisationalNameT
                    
                            #region TaxFileNumberId
                            _currentXPath.Push("/tns:TaxFileNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        groupMember.RDTIS60 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxFileNumberId
                    
                            #region TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA
                            _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            groupMember.RDTIS42 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA
                        _currentXPath.Pop();
                        } // End of while loop for GroupMember
                    } // End of if GroupMemberCollection node exists
            
                    _currentXPath.Pop();
                    #endregion GroupMember
            
                    #region TaxConcessionResearchAndDevelopmentTaxOffsetAdditionalEntitiesTurnoverA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetAdditionalEntitiesTurnoverA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetAdditionalEntitiesTurnoverA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS52 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentTaxOffsetAdditionalEntitiesTurnoverA
            
                    #region TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverExclusionA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverExclusionA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverExclusionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS43 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverExclusionA
            
                    #region TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverTotalA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS44 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverTotalA
                } // End of if AggregatedTurnover node exists
        
                _currentXPath.Pop();
                #endregion AggregatedTurnover
        
                #region TaxRateDesignationR
                _currentXPath.Push("/tns:TaxRateDesignationR");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxRateDesignationR" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.RDTIS78 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxRateDesignationR
        
                #region RefundableTaxOffset
                _currentXPath.Push("/tns:RefundableTaxOffset");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "RefundableTaxOffset" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_RefundableTaxOffsetCollectionExists = true;
                    report.RP_RefundableTaxOffsetCollectionCount += 1;
                    _found = true; 
            
                    #region TaxConcessionResearchAndDevelopmentNotionalDeductionsRefundableA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentNotionalDeductionsRefundableA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentNotionalDeductionsRefundableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS45 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentNotionalDeductionsRefundableA
            
                    #region TaxConcessionResearchAndDevelopmentTaxOffsetRefundableA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetRefundableA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetRefundableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS46 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentTaxOffsetRefundableA
                } // End of if RefundableTaxOffset node exists
        
                _currentXPath.Pop();
                #endregion RefundableTaxOffset
        
                #region NonRefundableTaxOffset
                _currentXPath.Push("/tns:NonRefundableTaxOffset");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "NonRefundableTaxOffset" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_NonRefundableTaxOffsetCollectionExists = true;
                    report.RP_NonRefundableTaxOffsetCollectionCount += 1;
                    _found = true; 
            
                    #region ExpenseResearchAndDevelopmentTotalA
                    _currentXPath.Push("/tns:ExpenseResearchAndDevelopmentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseResearchAndDevelopmentTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS81 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseResearchAndDevelopmentTotalA
            
                    #region TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS47 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
            
                    #region MiscellaneousResearchAndDevelopmentIntensityR
                    _currentXPath.Push("/tns:MiscellaneousResearchAndDevelopmentIntensityR");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "MiscellaneousResearchAndDevelopmentIntensityR" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS82 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion MiscellaneousResearchAndDevelopmentIntensityR
            
                    #region Tier1
                    _currentXPath.Push("/tns:Tier1");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Tier1" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_NonRefundableTaxOffset_Tier1CollectionExists = true;
                        report.RP_NonRefundableTaxOffset_Tier1CollectionCount += 1;
                        _found = true; 
                
                        #region TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                        _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS83 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                
                        #region TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                        _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS84 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                    } // End of if Tier1 node exists
            
                    _currentXPath.Pop();
                    #endregion Tier1
            
                    #region Tier2
                    _currentXPath.Push("/tns:Tier2");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Tier2" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_NonRefundableTaxOffset_Tier2CollectionExists = true;
                        report.RP_NonRefundableTaxOffset_Tier2CollectionCount += 1;
                        _found = true; 
                
                        #region TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                        _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS85 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                
                        #region TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                        _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS86 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                    } // End of if Tier2 node exists
            
                    _currentXPath.Pop();
                    #endregion Tier2
            
                    #region NonTier
                    _currentXPath.Push("/tns:NonTier");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NonTier" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_NonRefundableTaxOffset_NonTierCollectionExists = true;
                        report.RP_NonRefundableTaxOffset_NonTierCollectionCount += 1;
                        _found = true; 
                
                        #region TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                        _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS87 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA
                
                        #region TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                        _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.RDTIS88 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                    } // End of if NonTier node exists
            
                    _currentXPath.Pop();
                    #endregion NonTier
            
                    #region TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                    _currentXPath.Push("/tns:TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.RDTIS48 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA
                } // End of if NonRefundableTaxOffset node exists
        
                _currentXPath.Pop();
                #endregion NonRefundableTaxOffset
            } // End of if RP node exists
    
            _currentXPath.Pop();
            #endregion RP

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "RP#AmendmentI,RP#TaxFileNumberId,RP#PeriodStartD,RP#PeriodEndD,RP#InnovationRegistrationNumberId,RP#ExpenseResearchAndDevelopmentResearchAndDevelopmentAccountingExpenditureAddBackA,AustralianOwned#ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA,AustralianOwned#ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA,AustralianOwned#RemunerationWagesAndSalariesSalaryA,AustralianOwned#ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA,AustralianOwned#ExpenseResearchAndDevelopmentFeedstockA,AustralianOwned#ExpenseResearchAndDevelopmentAssociatesExpenditureA,AustralianOwned#ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA,AustralianOwned#ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA,AustralianOwned#ExpenseResearchAndDevelopmentAllocatedTotalA,AustralianOwned,ForeignOwned#ExpenseResearchAndDevelopmentContractedExpenditureResearchServiceProviderA,ForeignOwned#ExpenseResearchAndDevelopmentContractedExpenditureNonResearchServiceProviderA,ForeignOwned#RemunerationWagesAndSalariesSalaryA,ForeignOwned#ExpenseResearchAndDevelopmentOtherExpenditureRegisteredResearchAgencyA,ForeignOwned#ExpenseResearchAndDevelopmentFeedstockA,ForeignOwned#ExpenseResearchAndDevelopmentAssociatesExpenditureA,ForeignOwned#ExpenseResearchAndDevelopmentDeclineInValuePlantAndDepreciatingAssetsA,ForeignOwned#ExpenseResearchAndDevelopmentCooperativeResearchCentreContributionsA,ForeignOwned#ExpenseResearchAndDevelopmentAllocatedTotalA,ForeignOwned,NotionalDeductionsCalculation#ExpenseResearchAndDevelopmentAllocatedTotalA,NotionalDeductionsCalculation,Feedstock#IncomeResearchAndDevelopmentRevenueTotalA,Feedstock#ExpenseResearchAndDevelopmentInputA,Feedstock#ExpenseResearchAndDevelopmentAdjustmentBalanceA,Feedstock,ResearchandDevelopmentrecoupment#ResearchAndDevelopmentGovernmentGrantsAndRecoupmentsA,ResearchandDevelopmentrecoupment#ExpenseResearchAndDevelopmentRecoupmentRelatedA,ResearchandDevelopmentrecoupment#ExpenseResearchAndDevelopmentGrantRelatedA,ResearchandDevelopmentrecoupment#ExpenseResearchAndDevelopmentRecoupmentNetA,ResearchandDevelopmentrecoupment,AssessablebalancingadjustmentsIncomeResearchAndDevelopmentBalancingAdjustmentA,TotalclawbackIncomeResearchAndDevelopmentClawbackA,DeductiblebalancingadjustmentsExpenseResearchAndDevelopmentAdjustmentBalanceA,IncurredOrPaid#ExpenseResearchAndDevelopmentAssociatesCarriedForwardA,IncurredOrPaid#ExpenseResearchAndDevelopmentAssociatesUnpaidA,IncurredOrPaid#ExpenseResearchAndDevelopmentAssociatesClaimedA,IncurredOrPaid#ExpenseResearchAndDevelopmentAssociatesNotClaimedA,IncurredOrPaid,ExpenditureToAssociates#ExpenseResearchAndDevelopmentAssociatesCarriedForwardA,ExpenditureToAssociates,AggregatedTurnover#TaxConcessionExemptEntityOwnershipI,AggregatedTurnover#TaxConcessionAtOrAboveThresholdI,AggregatedTurnover#TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA,GroupMember#OrganisationNameDetailsOrganisationalNameT,GroupMember#TaxFileNumberId,GroupMember#TaxConcessionResearchAndDevelopmentTaxOffsetEntityTurnoverA,GroupMember,GroupMemberCollection,AggregatedTurnover#TaxConcessionResearchAndDevelopmentTaxOffsetAdditionalEntitiesTurnoverA,AggregatedTurnover#TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverExclusionA,AggregatedTurnover#TaxConcessionResearchAndDevelopmentTaxOffsetTurnoverTotalA,AggregatedTurnover,RP#TaxRateDesignationR,RefundableTaxOffset#TaxConcessionResearchAndDevelopmentNotionalDeductionsRefundableA,RefundableTaxOffset#TaxConcessionResearchAndDevelopmentTaxOffsetRefundableA,RefundableTaxOffset,NonRefundableTaxOffset#ExpenseResearchAndDevelopmentTotalA,NonRefundableTaxOffset#TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA,NonRefundableTaxOffset#MiscellaneousResearchAndDevelopmentIntensityR,Tier1#TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA,Tier1#TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA,Tier1,Tier2#TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA,Tier2#TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA,Tier2,NonTier#TaxConcessionResearchAndDevelopmentNotionalDeductionsNonRefundableA,NonTier#TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA,NonTier,NonRefundableTaxOffset#TaxConcessionResearchAndDevelopmentTaxOffsetNonRefundableA,NonRefundableTaxOffset,RP";

    }
}

